xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
import module namespace ada     = "http://art-decor.org/ns/ada-common" at "ada-common.xqm";
declare option exist:serialize "method=xhtml media-type=text/html indent=yes";

let $acceptMediaType    := 'application/zip'
let $referrer           := if (request:exists()) then request:get-header('Referer') else ()
let $filename           := if (request:exists()) then request:get-uploaded-file-name('file') else ()
let $size               := if (request:exists()) then request:get-uploaded-file-size('file') else ()
let $zip                := if (request:exists()) then request:get-uploaded-file-data('file') else ()
let $mediatype          := if (ends-with($filename, '.zip') or ends-with($filename, '.xar')) then 'application/zip' else ()

let $extension          := tokenize(tokenize($filename, '/')[last()], '\.')[last()]
let $projectname        := substring-before(tokenize($filename, '/')[last()], '.' || $extension)

let $check              :=
    if (empty($filename)) then 
        error(QName('http://art-decor.org/ns/error', 'MissingParameter'), 'Missing filename')
    else
    if (empty($zip)) then
        error(QName('http://art-decor.org/ns/error', 'MissingParameter'), 'Missing file contents. Expected multipart/form-data with ''file'' part')
    else
    if ($mediatype = $acceptMediaType) then () else (
        error(QName('http://art-decor.org/ns/error', 'UnsupportedFileType'), concat('Filename must end in .zip or .xar. Found ', tokenize($filename, '\.')[last()]))
    )

let $storedzip          := xmldb:store('/db/apps/decor/tmp', $filename, xs:base64Binary($zip))

let $entryCb            := function($path as xs:anyURI, $type as xs:string, $params as item()*) { $path='repo.xml' or $path='expath-pkg.xml' }
let $dataCb             := function($path as xs:anyURI, $type as xs:string, $data as item()?, $params as item()*) { $data }
let $dataCb             :=
    function ($path as xs:string, $data-type as xs:string, $data as item()?, $param as item()*) as element() {
        if ($data-type = 'folder') then
            ()
        else (
        (: if ($data-type = 'resource') :)
            let $target := $data/*/*:target
            let $abbrev := $data/*/@abbrev
            let $uri    := $data/*/@name
            
            return
                if ($path = 'expath-pkg.xml') then 
                    if (not($abbrev = $projectname)) then
                        error(QName('http://art-decor.org/ns/error', 'UnsupportedAbbrev'), 'Illegal @abbrev value in ' || $path || '. Expected ' || $projectname || ', found: ' || $abbrev)
                    else
                    if (not($uri = 'http://art-decor.org/ns/ada/' || $projectname)) then
                        error(QName('http://art-decor.org/ns/error', 'UnsupportedUri'), 'Illegal @name value in ' || $path || '. Expected http://art-decor.org/ns/ada/' || $projectname || ', found: ' || $uri)
                    else (
                        $data/descendant-or-self::*[1]
                    )
                else
                if ($path = 'repo.xml') then
                    if ($target = 'ada-data/projects/' || $projectname) then $data/descendant-or-self::*[1] else (
                        error(QName('http://art-decor.org/ns/error', 'UnsupportedTarget'), 'Illegal target collection in ' || $path || '. Expected ada-data/projects/' || $projectname || ', found: ' || $target)
                    )
                else (
                    $data/descendant-or-self::*[1]
                )
        )
    }

let $install            := 
    try {
        let $f := compression:unzip(util:binary-doc($storedzip), $entryCb, (), $dataCb, ())
        
        return
            if (count($f) = 2) then () else (
                error(QName('http://art-decor.org/ns/error', 'IncorrectPackage'), 'Missing repo.xml and/or expath-pkg.xml in the package')
            )
        ,
        repo:install-and-deploy-from-db($storedzip)
    } 
    catch * {
        <fail>{$err:code, $err:description}</fail>
    }
let $delete             := xmldb:remove('/db/apps/decor/tmp', $filename)

return 
    <html>
        <head>
            <title>{local-name($install[1])}</title>
            <style type="text/css">* {{font-family: verdana; font-size: 1em;}}</style>
            <meta name="robots" content="noindex, nofollow" />
        {
            if ($install/@result = 'ok' and not(empty($referrer))) then 
                <meta http-equiv="refresh" content="5; URL={$referrer}"/> |
                <meta http-equiv="expires" content="0" />
            else ()
        }
        </head>
        <body>
        {
            if ($install/@result = 'ok') then (
                <h3>Install successful</h3>,
                if (empty($referrer)) then () else (
                    <p>Returning in five seconds to: <a href="{$referrer}">{$referrer}</a></p>
                )
            )
            else (
                <h3>Install failed</h3>,
                <p>{serialize($install)}</p>,
                <ul>
                    <li>Generic documentation: <a href="https://docs.art-decor.org/release2/specials/#art-decor-applications-aka-ada">ART-DECOR ADA</a></li>
                    <li>Expects top level <em>repo.xml</em> and <em>expath-pkg.xml</em> files - generated using release2package.xsl</li>
                    <li>-----</li>
                    <li>Zip the contents of the ADA project folder, so the zip does not contain the top level folder itself</li>
                    <li><b>macOS</b>: make sure the zip does not include macOS specific (hidden) files. In Terminal.app, go into the directory and use <pre>zip [folder name].zip *</pre></li>
                </ul>
            )
        }
        </body>
    </html>