xquery version "3.0";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright
    
    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.
    
    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.
    
    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
:)
module namespace md                 = "http://art-decor.org/ns/art/markdown";

declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";

declare
    %output:media-type("text/plain")
    %output:method("text")
function md:html2markdown($html as item()*) {
    md:typeswitcher($html)
};

declare %private function md:typeswitcher($node as item()*) {
string-join(
for $child in $node
return
    typeswitch ($child)
    case element() return
        switch (local-name($child))
        case 'div' return concat('&#xD;&#xA;&#xD;&#xA;', md:typeswitcher($child/node()), '&#xD;&#xA;&#xD;&#xA;')
        case 'p' return concat('&#xD;&#xA;&#xD;&#xA;', md:typeswitcher($child/node()), '&#xD;&#xA;&#xD;&#xA;')
        case 'span' return md:typeswitcher($child/node())
        case 'ul' return concat('&#xD;&#xA;', md:typeswitcher($child/(li | caption)), '&#xD;&#xA;')
        case 'ol' return concat('&#xD;&#xA;', md:typeswitcher($child/(li | caption)), '&#xD;&#xA;')
        case 'li' return concat(string-join(for $i in (1 to count($child/ancestor::li)) return '    ', ''), if ($child/parent::ol) then '1. ' else '* ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'em'
        case 'i' return concat('*', md:typeswitcher($child/node()), '*')
        case 'b' 
        case 'strong' return concat('**', md:typeswitcher($child/node()), '**')
        case 'a' return concat('[', md:typeswitcher($child/node()), '](', $child/@href,  if ($child[@title]) then concat(' "', replace($child/@title, '"', '&#39;'), '"') else (), ')')
        case 'br' return '&#xD;&#xA;'
        case 'h1' return concat('&#xD;&#xA;# ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'h2' return concat('&#xD;&#xA;## ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'h3' return concat('&#xD;&#xA;### ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'h4' return concat('&#xD;&#xA;#### ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'h5' return concat('&#xD;&#xA;##### ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'h6' return concat('&#xD;&#xA;###### ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'h7' return concat('&#xD;&#xA;###### ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'h8' return concat('&#xD;&#xA;###### ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'h9' return concat('&#xD;&#xA;###### ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'blockquote' return concat('&#xD;&#xA;> ', md:typeswitcher($child/node()), '&#xD;&#xA;')
        case 'hr' return '&#xD;&#xA;----------&#xD;&#xA;'
        case 'table' return concat('&#xD;&#xA;&#xD;&#xA;', md:typeswitcher($child/(caption, thead, tbody, tfoot, tr)), '&#xD;&#xA;&#xD;&#xA;')
        case 'caption' return concat('&#xD;&#xA;**', md:typeswitcher($child/node()), '**&#xD;&#xA;')
        case 'thead' return md:typeswitcher($child/tr)
        case 'tbody' return md:typeswitcher($child/tr)
        case 'tfoot' return md:typeswitcher($child/tr)
        case 'tr' return concat('| ', string-join(for $c in $child/(th | td) return md:typeswitcher($c), ' | '), ' |&#xD;&#xA;', 
                                        if ($child[th][empty(following-sibling::tr[th])]) then concat('|', string-join(for $i in (1 to count($child/(th | td))) return '---|', ''), '&#xD;&#xA;') else ())
        case 'th' return md:typeswitcher($child/node())
        case 'td' return md:typeswitcher($child/node())
        default return (
            let $startmarker     := if ($child[matches(parent::*/@style, 'font-weight:\s*bold')]) then '**' else ()
            let $startmarker     := if ($child[matches(parent::*/@style, 'font-style:\s*italic')]) then concat('*', $startmarker) else $startmarker
            let $startmarker     := if ($child[matches(parent::*/@style, 'text-decoration:\s*line-through')]) then concat('~~', $startmarker) else $startmarker
            
            return concat($startmarker, $child, string-join(codepoints-to-string(reverse(string-to-codepoints($startmarker))), ''))
        )
    case comment() return ()
    case text() return (
        let $startmarker     := if ($child[matches(parent::*/@style, 'font-weight:\s*bold')]) then '**' else ()
        let $startmarker     := if ($child[matches(parent::*/@style, 'font-style:\s*italic')]) then concat('*', $startmarker) else $startmarker
        let $startmarker     := if ($child[matches(parent::*/@style, 'text-decoration:\s*line-through')]) then concat('~~', $startmarker) else $startmarker
        
        return concat($startmarker, $child, string-join(codepoints-to-string(reverse(string-to-codepoints($startmarker))), ''))
    )
    default return md:typeswitcher($child/node())
, '')
};