xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Template API allows read, create, update of DECOR templates :)
module namespace tmapi              = "http://art-decor.org/ns/api/template";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace utiltemp    = "http://art-decor.org/ns/api/util-template" at "library/util-template-lib.xqm";
import module namespace utilhtml    = "http://art-decor.org/ns/api/util-html" at "library/util-html-lib.xqm";
import module namespace utilsvg     = "http://art-decor.org/ns/api/util-svg" at "library/util-svg-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";

declare namespace json              = "http://www.json.org";

(:~ Retrieves latest DECOR template based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function tmapi:getLatestTemplate($request as map(*)) {
    tmapi:getTemplate($request)
};

(:~ Retrieves latest DECOR template based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function tmapi:getTemplate($request as map(*)) {
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $id                     := $request?parameters?id[not(. = '')]
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    let $acceptTypes            := roaster:accepted-content-types()
    let $acceptedType           := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    let $format                 := tokenize($acceptedType, '/')[2]
    
    let $results                := utiltemp:getTemplate($id, ($effectiveDate, 'dynamic')[1], $projectPrefix, $projectVersion, $projectLanguage)
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple templates for id ''' || $id || '''. Expected 0..1. Alert your administrator as this should not be possible. Found in: ' || string-join($results/concat(@ident, ' / ', @url), ', '))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )
};

(:~ Retrieves latest DECOR template as a diagram based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $format                  - optional. overrides the accept-header for frontend purposes, default svg if not calculated
    @return as-is 
    @since 2025-03-07
:)
declare function tmapi:getLatestTemplateDiagram($request as map(*)) {
    tmapi:getTemplateDiagram($request)
};

(:~ Retrieves DECOR template as a diagram based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $format                  - optional. overrides the accept-header for frontend purposes, default svg if not calculated
    @return as-is 
    @since 2023-12-21
:)
declare function tmapi:getTemplateDiagram($request as map(*)) {

    let $project                := $request?parameters?project[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    (: no format given: default format is 'svg' :)
    let $format                 := $request?parameters?format[not(. = '')]
    
    let $id                     := $request?parameters?id[not(. = '')]
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
       
    (: no format given: default format is 'svg' :)
    let $format                 :=
        if (empty($format)) then 'svg' else $format
           
    (: 
       overwrite format in the backend is not always possible because of middleware behaviour
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
    :) 
    let $check                          :=
        if ($format = ('xml', 'hlist') and not((roaster:accepted-content-types()[. = ('image/svg+xml', 'application/xml')])[1] = 'application/xml')) then
            error($errors:BAD_REQUEST, 'In case of format parameter is xml or hlist the accept header should be application/xml')
        else ()
    
    (: no laguage given: default default language is art-language :)
    let $projectLanguage        := if (empty($projectLanguage)) then $setlib:strArtLanguage else $projectLanguage
    let $projectLanguage        := if ($projectLanguage = (utillib:getArtLanguages())) then $projectLanguage else 'en-US'
    
    let $projectPrefix          := if (empty($project)) then () else utillib:getDecor($project, (), $projectLanguage)/project/@prefix 
    
    (: Return zero or more extracted templates  :)
    let $templates              := if (empty($projectPrefix)) then utillib:getTemplateById($id, $effectiveDate)[@id] else utillib:getTemplateById($id, $effectiveDate, $projectPrefix, (), $projectLanguage)[@id]
    
    (: first the template scan used for the html (hgraph and wiki) payload and xml hlist :)
    let $templateScan           := if (count($templates)=1) then utiltemp:templateScan($templates, $projectPrefix, $projectLanguage, 'template', 1, 1, ()) else () 
    (: templateChain for svg and xml representation :)
    let $templateChain          := if(empty($templateScan)) then () else utiltemp:chainCopy2(utiltemp:chainCopy1(<tree>{$templateScan}</tree>, -1), 1)
    
    let $results                := 
        if (empty($templateScan)) then ()
        else if ($format = ('xml', 'hlist')) then (
            let $xml            :=  
                if ($format = 'xml') then (
                    <x project="{$projectPrefix}">
                         {
                             $templateChain,
                             <max>{max($templateChain//@len)}</max>
                         }
                    </x>
                )    
                else $templateScan
           return
           for $x in $xml
                return
                element {name($x)} {
                    $x/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($x/node(), $format)
                }
        )
        else if ($format=('hgraph', 'hgraphwiki', 'wikilist', 'transclusionwikilist')) then (
            response:set-header('Content-Type','text/html; charset=utf-8'),
            utiltemp:apply-hgraph-stylesheet(($templateScan ! (.)), $format)
        )
        else (
            response:set-header('Content-Type','image/svg+xml'),
            response:set-header('X-Robots-Tag', 'noindex'), 
            utilsvg:convertTemplate2Svg($templateChain)
        ) 

    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else $results
};

(:~ Retrieves latest DECOR template for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $language                - optional
    @param $tree                    - optional
    @return as-is or compiled as JSON
    @since 2025-03-07
:)
declare function tmapi:getLatestTemplateExtract($request as map(*)) {
    tmapi:getTemplateExtract($request)
};

(:~ Retrieves DECOR template for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $language                - optional
    @param $tree                    - optional
    @return as-is or compiled as JSON
    @since 2023-11-17
:)
declare function tmapi:getTemplateExtract($request as map(*)) {

    let $project                := $request?parameters?project[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $tree                   := $request?parameters?tree[not(. = '')]
    let $id                     := $request?parameters?id[not(. = '')]
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $acceptTypes            := roaster:accepted-content-types()
    let $acceptedType           := ($acceptTypes[. = ('application/xml', 'application/json')],'application/xml')[1]
    
    let $format                 := tokenize($acceptedType, '/')[2]
     
    let $projectPrefix          := utillib:getDecor($project, $projectVersion, $projectLanguage)/project/@prefix 
    
    let $check                  :=
        if (empty($projectPrefix)) then
            error($errors:BAD_REQUEST, 'Project ' || $project || ' not found.')
        else ()
    
    let $results                := 
        if (not($tree = 'expand')) 
            then utiltemp:getTemplateExtract($projectPrefix, $projectVersion, $projectLanguage, $id, $effectiveDate)
            else utiltemp:getExpandedTemplateExtract($projectPrefix, $projectVersion, $projectLanguage, $id, $effectiveDate)
   
    let $results                := if ($tree = ('wrap', 'expand')) then $results else ($results/template/template)[1] 

    return
        if (empty($results/*)) then (
            roaster:response(404, ())
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/node(), $format)
                } 
        )
};

(:~ Retrieves latest DECOR template view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language, also ui-language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2025-03-07
:)
declare function tmapi:getLatestTemplateView($request as map(*)) {
    tmapi:getTemplateView($request)
};

(:~ Retrieves DECOR template view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language, also ui-language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2024-10-14
:)
declare function tmapi:getTemplateView($request as map(*)) {

    let $project                := $request?parameters?project[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $seetype                := $request?parameters?seetype[not(. = '')]
    let $inline                 := $request?parameters?inline = true()
    let $collapsable            := not($request?parameters?collapsable = false())
    
    let $id                     := $request?parameters?id[not(. = '')]
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $decor                  := utillib:getDecor($project, $projectVersion, $projectLanguage)
    
    let $check                  :=
        if (empty($decor/project/@prefix)) then
            error($errors:BAD_REQUEST, 'Project ' || $project || ' not found.')
        else ()    
     
    let $results                := utiltemp:getExpandedTemplateExtract($decor/project/@prefix, $projectVersion, $projectLanguage, $id, $effectiveDate)            
                
    let $r-header               := response:set-header('Content-Type', 'text/html; charset=utf-8')
    
    return
        if (empty($results/*)) then (
            roaster:response(404, ())
        )
       else (
            (: prepare for Html :)
            let $language           := if ($decor/project/name[@language = $projectLanguage]) then $projectLanguage else $decor/project/@defaultLanguage
            let $header             := if ($inline) then false() else true()        
            
            return utilhtml:convertObject2Html($results, $language, $header, $collapsable, $projectVersion, $decor, $seetype)
        )
};

(:~ Returns a list of zero or more templates
    
    @param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only
    @param $projectVersion   - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
    @param $id               - optional. Identifier of the template to retrieve
    @param $name             - optional. Name of the template to retrieve (valueSet/@name)
    @param $effectiveDate    - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
    @param $treetype         - optional. Default $tmapi:TREETYPELIMITEDMARKED
    @param $resolve          - optional. Boolean. Default true if governanceGroupId is empty. If true retrieves all versions for references, otherwise just returns the references.
    @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
    @since 2013-06-14
:)
declare function tmapi:getTemplateList($request as map(*)) {
    let $governanceGroupId      := $request?parameters?governanceGroupId[not(. = '')]
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    
    let $searchTerms            := 
        typeswitch ($request?parameters?search) 
        case xs:string return 
            array:flatten(
                for $s in $request?parameters?search[string-length() gt 0]
                return tokenize(normalize-space(lower-case($s)),'\s')
            )
        default return array:flatten($request?parameters?search)
    
    let $id                     := $request?parameters?id[not(. = '')]
    let $nm                     := $request?parameters?name[not(. = '')]
    let $ed                     := $request?parameters?effectiveDate[not(. = '')]
    let $treetype               := $request?parameters?treetype
    let $resolve                :=  if (empty($governanceGroupId)) then not($request?parameters?resolve = false()) else $request?parameters?resolve = true()
    
    let $check                  :=
        if (count($governanceGroupId) + count($projectPrefix) = 0) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter governanceGroupId or prefix')
        else 
        if (count($governanceGroupId) + count($projectPrefix) gt 1) then 
            error($errors:BAD_REQUEST, 'Request SHALL have exactly one parameter governanceGroupId or prefix, not both or multiple')
        else
        if (count($governanceGroupId) gt 0 and not(empty($projectVersion))) then
            error($errors:BAD_REQUEST, 'Request SHALL NOT have both governanceGroupId and release of a single project')
        else
        if (count($governanceGroupId) gt 0 and not(empty($searchTerms))) then
            error($errors:BAD_REQUEST, 'Search only supported in a project')
        else ()

    let $results                := utiltemp:getTemplateList($governanceGroupId, $projectPrefix, $projectVersion, $projectLanguage, $searchTerms, $id, $ed, $treetype, $resolve)
        
    
    let $countTN                := count($results/*/template/version)
    
    return
        <list artifact="TM" current="{$countTN}" total="{$countTN}" all="{$countTN}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results/*)
        }
        </list>
};

(:~ Returns a list of zero or more templateAssociation
    
    @param $projectPrefix    - required. determines search scope. limits scope to this project only
    @param $projectVersion   - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
    @param $projectLanguage  - optional. works only in conjunction with $projectVersion. if empty defaults to *. if valued then the template will come explicitly from the compilation in that language. * is the "all languages" compilation
    @param $id               - optional. Identifier of the template to retrieve templateAssociations for
    @param $effectiveDate    - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
    @return list object with zero or more templateAssociation objects
    @since 2022-02-02
:)
declare function tmapi:getTemplateAssociationList($request as map(*)) {
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $id                     := $request?parameters?id[not(. = '')]
    let $ed                     := $request?parameters?effectiveDate[not(. = '')]
    
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have parameter prefix')
        else () 
        
   let $decor                   := utillib:getDecor($projectPrefix, $projectVersion, $projectLanguage)
    
    let $check                  :=
        if (count($decor) = 1) then () else
        if (count($decor) = 0) then
            if (empty($projectVersion)) then
                error($errors:BAD_REQUEST, 'Project with prefix ' || $projectPrefix || ' not found')
            else (
                error($errors:BAD_REQUEST, 'Project with prefix ' || $projectPrefix || ' release ' || $projectVersion || ' language ''' || $projectLanguage || ''' not found')
            )
        else (
            if (empty($projectVersion)) then
                error($errors:SERVER_ERROR, 'Found ' || count($decor) || ' instances of project with prefix ' || $projectPrefix || '. Expected 0..1. Alert your administrator as this should not be possible.')
            else (
                error($errors:SERVER_ERROR, 'Found ' || count($decor) || ' instances of project with prefix ' || $projectPrefix || ' release ' || $projectVersion || ' language ''' || $projectLanguage || '''. Expected 0..1. Alert your administrator as this should not be possible.')
            )
        )
    
    let $results                :=
        if (empty($id)) then $decor//templateAssociation
        else if (empty($ed)) then $decor//templateAssociation[@templateId = $id]
        else $decor//templateAssociation[@templateId = $id][@effectiveDate = $ed]
        
    
    return
        <list artifact="ASSOCIATION" current="{count($results)}" total="{count($results)}" all="{count($decor//templateAssociation)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Deletes a template reference based on $id (oid) and project id or prefix
    @param $id                      - required parameter denoting the id of the template
    @param $project                 - required. limits scope to this project only
:)
declare function tmapi:deleteTemplate($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0][not(. = '*')]
    let $id                     := $request?parameters?id[string-length() gt 0]
    
    let $check                  :=
        if (empty($project) or empty($id)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have both parameter project and id')
        else ()
    
    let $decor                  := utillib:getDecor($project, (), ())
    
    let $check                  :=
        if (empty($decor)) then
            error($errors:BAD_REQUEST, 'Project ' || $project || ' not found.')
        else ()
    let $check                  := utiltemp:checkTemplateAccess($authmap, $decor, (), (), false(), false())
       
    let $delete                 := update delete $decor/rules/template[@ref = $id]
    let $delete                 := update delete $decor/rules/templateAssociation[@templateId = $id][empty(*)]
    
    return roaster:response(204, ())
};

(:~ Returns template datatypes
    
    @param $format           - optional. defaults to hl7v3xml1
    @return datatypes or http 404 if not found
    @since 2022-02-07
:)
declare function tmapi:getTemplateDatatypes($request as map(*)) {
    
    let $format                 := ($request?parameters?format[not(. = '')], 'hl7v3xml1')[1]
        
    let $results                := utiltemp:getTemplateDatatypes($format)

    return
        if (empty($results)) then
            roaster:response(404, ())
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Retrieves a new template in edit mode to be saved later, or does initial save of a template
    @param $projectPrefix           - required. determines search scope. null is full server, pfx- limits scope to this project only
    @param $sourceId                - optional. parameter denoting the id of a template to use as a basis for creating the new template
    @param $sourceEffectiveDate     - optional. parameter denoting the effectiveDate of a template to use as a basis for creating the new template
    @param $targetDate              - optional. If true invokes effectiveDate of the new template as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
    @param $keepIds                 - optional. Only relevant if source template is specified. If true, the new template will keep the same id and only update the effectiveDate
    @param $lock                    - optional. relevant when saving a new template. does not set a lock if false and returns in read mode, sets a lock if true and leaves in edit mode, otherwise
    @param $request-body            - optional. json body containing new template structure
    @return template
    @since 2020-05-03
:)
declare function tmapi:postTemplate($request as map(*)) {
    let $authmap                := $request?user
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $sourceId               := $request?parameters?sourceId
    let $sourceEffectiveDate    := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?sourceEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?sourceEffectiveDate[string-length() gt 0]
        }
    let $refOnly                := $request?parameters?refOnly = true()
    let $targetDate             := $request?parameters?targetDate = true()
    let $keepIds                := $request?parameters?keepIds = true()
    let $lock                   := $request?parameters?lock = true()
    let $data                   := utillib:getBodyAsXml($request?body, 'template', ())
    
    let $check                  := 
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have prefix')
        else ()
    
    let $check                  := 
        if (empty($data) and empty($sourceId)) then
            error($errors:BAD_REQUEST, 'Request SHALL have source id if data is empty')
        else ()
    
    let $check                  := 
        if ($refOnly and empty($sourceId)) then
            error($errors:BAD_REQUEST, 'Request SHALL have source id if refOnly = true')
        else ()
    
    let $check                  := 
        if ($keepIds and empty($sourceId)) then
            error($errors:BAD_REQUEST, 'Request SHALL have source id if keepIds = true')
        else ()
    
    let $results                := utiltemp:createTemplate($authmap, $projectPrefix, $sourceId, $sourceEffectiveDate, $refOnly, $targetDate, $keepIds, $lock, $data)[@id][1]
    
    let $results                :=
        element {name($results)} {
                $results/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($results/*)
            }
    
    return
        if (empty($data)) then roaster:response(200, $results) else roaster:response(201, $results)
};

declare function tmapi:postTemplateExample($request as map(*)) {

    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $id                     := $request?parameters?id[not(. = '')]
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $elementId              := $request?parameters?elementId[not(. = '')]
    let $doSerialized           := $request?parameters?serialized = true()
    let $doRecursive            := $request?parameters?recursive = true()
    let $data                   := utillib:getBodyAsXml($request?body, 'template', ())
    
    let $projectPrefix          := ($data/@projectPrefix, $projectPrefix)[1]
    
    let $check                  :=
        if ($data) then
            if (empty($id)) then () else (
                error($errors:BAD_REQUEST, 'Request SHALL NOT have both request body and parameter id')
            )
        else 
        if (empty($id)) then 
            error($errors:BAD_REQUEST, 'Request SHALL at least have a request body or a parameter id')
        else ()
    
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have prefix')
        else ()
    
    let $result                 := utiltemp:createTemplateExample($projectPrefix, $projectVersion[1], $projectLanguage[1], $id, $effectiveDate, $elementId, $doSerialized, $doRecursive, $data)

    return  roaster:response(200, $result)
};

(:~ Update DECOR template
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the template to update
    @param $effectiveDate            - required. the effectiveDate for the template to update
    @param $request-body             - required. json body containing new template structure
    @return template structure 
    @since 2020-05-03
:)
declare function tmapi:putTemplate($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                   := utillib:getBodyAsXml($request?body, 'template', ())
    let $deletelock             := $request?parameters?deletelock = true()
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $results                := utiltemp:putTemplate($authmap, $id, $effectiveDate, $data, $deletelock)[@id][1]
    
    return 
        element {name($results)} {
            $results/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($results/*)
        }

};

(:~ Update DECOR template parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "e.g. [/statusCode|/expirationDate|/officialReleaseDate|/canonicalUri|/versionLabel|/name|/displayName|/experimental|/desc|/publishingAuthority|/copyright|/completeCodeSystem|/conceptList]", "value": "[string|object]" }

where

* op - add & replace (statusCode, expirationDate, officialReleaseDate, canonicalUri, versionLabel, name, displayName, experimental, desc, publishingAuthority, copyright, completeCodeSystem, conceptList) or remove (desc, publishingAuthority, copyright, completeCodeSystem, conceptList)

* path - see above

* value - string when the path is not /. object when path is /
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the issue to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return template structure
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function tmapi:patchTemplate($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $results                := utiltemp:patchTemplate($authmap, $id, $effectiveDate, $data)
    
    return 
        for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
};

(:~ Update DECOR templateAssociation parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "e.g. [/concept]", "value": "[object]" }

where

* op - add, replace or remove concept

* path - see above

* value - concept object
@param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
@param $id                       - required. the id for the issue to update 
@param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
@return template structure
@since 2020-05-03
@see http://tools.ietf.org/html/rfc6902
:)
declare function tmapi:patchTemplateAssociation($request as map(*)) {

    let $authmap                := $request?user
    let $project                := $request?parameters?project
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $check                  :=
        if (empty($project)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have parameter project')
        else () 
    
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $result                 := utiltemp:patchTemplateAssociation($authmap, $id, $effectiveDate, $project, $data)
    
    return 
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }

};

(:~ Update template statusCode, versionLabel, expirationDate and/or officialReleaseDate

    @param $authmap required. Map derived from token
    @param $id required. DECOR concept/@id to update
    @param $effectiveDate required. DECOR concept/@effectiveDate to update
    @param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function tmapi:putTemplateStatus($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $recurse                := $request?parameters?recurse = true()
    let $list                   := $request?parameters?list = true()
    let $statusCode             := $request?parameters?statusCode
    let $versionLabel           := $request?parameters?versionLabel
    let $expirationDate         := $request?parameters?expirationDate
    let $officialReleaseDate    := $request?parameters?officialReleaseDate
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                := utiltemp:setTemplateStatus($authmap, $id, $effectiveDate, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
        
    return
    if ($results[self::error] and not($list)) then
        error($errors:BAD_REQUEST, string-join(
            for $e in $results[self::error]
            return
                $e/@itemname || ' id ''' || $e/@id || ''' effectiveDate ''' || $e/@effectiveDate || ''' cannot be updated: ' || data($e)
            , ' ')
        )
    else (
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            (: max 2 arrays: one with error and one with success :)
            utillib:addJsonArrayToElements($results)
        }
        </list>   
    )
};

(:~ Retrieves DECOR template history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function tmapi:getTemplateHistory($request as map(*)) {
    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix          := ()
    let $results                := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-TEMPLATE, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-TEMPLATE}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Retrieves DECOR template usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2021-12-16
:)
declare function tmapi:getTemplateUsage($request as map(*)) {
    
    let $projectPrefix          := $request?parameters?prefix
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $results                := utiltemp:getDependenciesAndUsage($projectPrefix, (), (), $id, $effectiveDate)

    
    let $count                  := count($results)
    let $max                    := $count
    let $allcnt                 := $count
    
    return
        <list artifact="USAGE" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

declare function tmapi:postAddTemplateIds($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $templateId             := $request?parameters?id
    let $templateEffectiveDate  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    
    let $decorProjects          := utillib:getDecor($project)
    
    let $check                  :=
        if ($decorProjects) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
    
    let $update                 := 
        for $decor in $decorProjects
        let $check              := utiltemp:checkTemplateAccess($authmap, $decor, (), (), false(), false())
           
        return utiltemp:addTemplateElementAndAttributeIds($decor, $templateId, $templateEffectiveDate)
    
    return roaster:response(200, ())
};
