xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Dataset concept API allows read, create, update on DECOR concepts in DECOR datasets :)
module namespace mpapi              = "http://art-decor.org/ns/api/conceptmap";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "/db/apps/api/modules/library/util-lib.xqm";
import module namespace utilmp      = "http://art-decor.org/ns/api/util-conceptmap" at "library/util-conceptmap-lib.xqm";
import module namespace utilgg      = "http://art-decor.org/ns/api/util-governancegroup" at "library/util-governancegroup-lib.xqm";

import module namespace utilhtml    = "http://art-decor.org/ns/api/util-html" at "library/util-html-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "/db/apps/api/modules/library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "/db/apps/api/modules/library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";

declare namespace json              = "http://www.json.org";

(:~ Retrieves latest DECOR conceptMap based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the conceptMap
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function mpapi:getLatestConceptMap($request as map(*)) {
    mpapi:getConceptMap($request)
};

(:~ Retrieves DECOR conceptMap based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the conceptMap
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the conceptMap. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function mpapi:getConceptMap($request as map(*)) {
    
    let $projectPrefix                  := $request?parameters?prefix
    let $projectVersion                 := $request?parameters?release
    let $projectLanguage                := $request?parameters?language
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $withversions                   := $request?parameters?versions = true()
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]
    let $format                         := tokenize($acceptedType, '/')[2]
    
    let $results                        := utilmp:getConceptMap($projectPrefix, $projectVersion, $projectLanguage, $id, $effectiveDate, $withversions, true())
    let $results                        := 
        if ($withversions) then 
            <list artifact="VS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">{$results}</list> 
        else (
            head($results)
        )
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple conceptMaps for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )
};

(:~ Retrieves latest DECOR conceptmap for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the conceptmap
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @param $download                - optional as xs:boolean. Default: false. 
    @return as-is or as compiled as JSON
    @since 2025-03-07
:)
declare function mpapi:getLatestConceptMapExtract($request as map(*)) {
    mpapi:getConceptMapExtract($request)
};

(:~ Retrieves DECOR conceptmap for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the conceptmap
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the conceptmap. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @param $download                - optional as xs:boolean. Default: false. 
    @return as-is or as compiled as JSON
    @since 2024-12-12
:)
declare function mpapi:getConceptMapExtract($request as map(*)) {

    let $project                        := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $format                         := $request?parameters?format[not(. = '')]
    let $download                       := $request?parameters?download=true()
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    (: parameter format overrides accept-header as mime-type for using the api in the browser :)
    
    let $acceptTypes                    := roaster:accepted-content-types()
    
    (: when you do a right mouse click for downloading content in Edge or Chrome, you effectively do a GET without Accept header :)
    let $check                          := 
        if (empty($acceptTypes) and empty($format)) then
            error($errors:BAD_REQUEST, 'Your request does not have an Accept header, and is missing a format parameter. Don''t know what to do')
        else ()
    
    (: $format should never be empty coming from the api nor decor/services :)
    let $acceptedType                   :=
        switch ($format)
        (:case 'svs'
        case 'svsold'
        case 'mdibagch':)
        case 'xml'      return ($acceptTypes[. = ('application/xml', 'application/*', '*/*')], 'application/xml')[1] 
        case 'json'     return ($acceptTypes[. = ('application/json', 'application/*', '*/*')], 'application/json')[1]
        (:case 'csv'      return ($acceptTypes[. = ('text/csv', 'text/*', '*/*')], 'text/csv')[1]:)
        (:case 'fsh'      return ($acceptTypes[. = ('text/plain', 'text/fsh', 'text/*', '*/*')], 'text/plain')[1]:)
        (:case 'sql'      return ($acceptTypes[. = ('text/sql', 'text/*', '*/*')], 'text/sql')[1]:)
        default         return if (empty($format)) then 
            ($acceptTypes[. = ('application/xml', 'application/json', 'application/*', 'text/*', '*/*')])[1]
        else (
            error($errors:BAD_REQUEST, 'Requested format ''' || string-join($format, ', ') || ''' not supported. Supported are: xml, json')
        )
    let $responseType                    :=
        if (contains($acceptedType, '*') or empty($acceptedType)) then $utillib:formatacceptmap?($format) else $acceptedType
    
    (: 
       overwrite format in the backend is not always possible because of middleware behavior
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
    :) 
    let $check                          :=
        switch ($format)
        (:case 'svs'
        case 'svsold'
        case 'mdibach':)
        case 'xml' return if ($acceptedType = ('application/xml', 'application/*', '*/*')) then () else (
            error($errors:BAD_REQUEST, 'When parameter format=' || $format || ' the accept header SHALL contain application/xml. Found: ' || string-join($acceptTypes, ','))
        )
        case 'json' return if ($acceptedType = ('application/json', 'application/*', '*/*')) then () else (
            error($errors:BAD_REQUEST, 'When parameter format=' || $format || ' the accept header SHALL contain application/json. Found: ' || string-join($acceptTypes, ','))
        )
        default return if (empty($acceptedType)) then 
            error($errors:BAD_REQUEST, 'Unable to determine what format to return. Found Accept header: ''' || string-join($acceptTypes, ',') || ''' and format=''' || $format || '''')
        else 
        if ($acceptedType = 'application/json') then
            error($errors:BAD_REQUEST, 'When parameter format=' || $format || ' the accept header SHALL NOT contain application/json. Found: ' || string-join($acceptTypes, ','))
        else ()
    
    let $withVersions                   := not($effectiveDate[string-length() gt 0])

    let $projectPrefix                  := if (empty($project)) then () else utillib:getDecor($project, $projectVersion, $projectLanguage)/project/@prefix      
    
    let $conceptMaps                    := 
        if (empty($projectPrefix)) then utilmp:getConceptMap((), (), $projectLanguage, $id, $effectiveDate, $withVersions, true()) else utilmp:getConceptMap($projectPrefix, $projectVersion, $projectLanguage, $id, $effectiveDate, $withVersions, true())

    let $results                :=
         if (empty($conceptMaps/*)) then () 
         else if ($format = 'xml') then <conceptMaps>{$conceptMaps}</conceptMaps>
         else if ($format = 'json') then (
            for $x in <conceptMaps>{$conceptMaps}</conceptMaps>
            return
                element {name($x)} {
                    $x/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($x/node(), $format)
                }
         )   
         (:else if ($format = 'csv') then utilvs:convertValueSet2Csv($valueSets, $projectLanguage, $projectPrefix)
         else if ($format = 'fsh') then utilfsh:convertValueSets2Fsh($valueSets, $projectLanguage, $projectPrefix)/text()
         else if ($format = 'sql') then fn:string(transform:transform($valueSets, doc($setlib:strDecorServices ||'/resources/stylesheets/ToSql4ValueSets.xsl'), ())):)
         else error($errors:BAD_REQUEST, 'Requested format ' || $format || ' is not supported')
    
    let $filename                       := 'MP_' || ($conceptMaps//conceptMap/@displayName)[1] || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $format
    
    let $r-header               := 
        (response:set-header('Content-Type', $responseType || '; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
    return
        if (empty($results)) then roaster:response(404, ()) else roaster:response(200, $responseType, $results)

};

(:~ Retrieves latest DECOR conceptmap view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the conceptmap
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2024-12-12
:)
declare function mpapi:getLatestConceptMapView($request as map(*)) {
    mpapi:getConceptMapView($request)
};

(:~ Retrieves DECOR conceptmap view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the conceptmap
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2024-12-12
:)
declare function mpapi:getConceptMapView($request as map(*)) {

    let $project                        := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $seetype                        := $request?parameters?seetype[not(. = '')]
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    let $decor                          := 
        if (empty($project)) then () else utillib:getDecor($project, $projectVersion, $projectLanguage)
    
    let $results                        := 
        if (empty($decor/project/@prefix)) then utilmp:getConceptMap((), (), $projectLanguage, $id, $effectiveDate, false(), true()) else utilmp:getConceptMap($decor/project/@prefix, $projectVersion, $projectLanguage, $id, $effectiveDate, false(), true())

    let $filename                       := 'MP_' || $results/descendant-or-self::conceptMap/@displayName[1] || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html'
            
    let $r-header                       := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        response:set-header('Content-Disposition', 'filename='|| $filename))
    
    return
        if (empty($results/*)) then (
            roaster:response(404, ())
        )

        else 
        
        (: prepare for Html :)
        let $language                       := if (empty($projectLanguage)) then $setlib:strArtLanguage else $projectLanguage
        
        return utilhtml:convertObject2Html($results, $language, true(), false(), $projectVersion, $decor, $seetype)
       
};

(:~ Retrieves DECOR conceptMap history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function mpapi:getConceptMapHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-MAPPING, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-MAPPING}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Retrieves DECOR concept maps based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the concept
    @param $effectiveDate           - required parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $transactionId           - required parameter denoting the id of the transaction that the concept is in
    @param $transactionEffectiveDate- optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $associations            - optional boolean parameter relevant if $treeonly = 'false' to include associations: terminologyAssociation, identifierAssociation
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function mpapi:getConceptMapList($request as map(*)) {
    
    let $governanceGroupId      := $request?parameters?governanceGroupId
    let $projectPrefix          := $request?parameters?prefix
    let $projectVersion         := $request?parameters?release
    let $projectLanguage        := $request?parameters?language
    let $max                    := $request?parameters?max
    let $resolve                :=  if (empty($governanceGroupId)) then not($request?parameters?resolve = false()) else $request?parameters?resolve = true()
    
    let $searchTerms            := 
        typeswitch ($request?parameters?search) 
        case xs:string return 
            array:flatten(
                for $s in ($request?parameters?search, $request?parameters?id, $request?parameters?name)[string-length() gt 0]
                return tokenize(normalize-space(lower-case($s)),'\s')
            )
        default return array:flatten(($request?parameters?search, $request?parameters?id, $request?parameters?name))
        
    let $includebbr             := $request?parameters?includebbr = true()
    let $sort                   := $request?parameters?sort
    let $sortorder              := $request?parameters?sortorder
    
    let $check                  :=
        if (empty($governanceGroupId) and empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter governanceGroupId or prefix')
        else 
        if (not(empty($governanceGroupId)) and not(empty($projectPrefix))) then 
            error($errors:BAD_REQUEST, 'Request SHALL NOT have both parameter governanceGroupId or prefix, not both')
        else ()
    let $check                  :=
        if (empty($governanceGroupId)) then () else if ($resolve) then
            error($errors:BAD_REQUEST, 'Request SHALL NOT have governance group scope and resolve=true. This is too expensive to support')
        else ()
    
    let $result                 :=
        if (empty($governanceGroupId)) then
            utilmp:getConceptMapList($governanceGroupId, $projectPrefix, $projectVersion, $projectLanguage, $searchTerms, $includebbr, $sort, $sortorder, $max, $resolve, $request?parameters)
        else (
            for $project in utilgg:getLinkedProjects($governanceGroupId)/@ref
            return
                utilmp:getConceptMapList($governanceGroupId, $project, (), (), $searchTerms, $includebbr, $sort, $sortorder, $max, $resolve, $request?parameters)    
        )

    return
        <list artifact="MP" current="{count($result/*)}" total="{count($result/*)}" all="{count($result/*)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($result/*)
        }
        </list>
};

(:~ Update DECOR conceptMap. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "/", "value": "[terminologyAssociation|identifierAssociation]" }

where
* op - add (object associations, tracking, event) or remove (object associations only) or replace (displayName, priority, type, tracking, assignment)
* path - / 
* value - terminologyAssociation|identifierAssociation object

    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR concept/@id to update
    @param $effectiveDate           - required. DECOR concept/@effectiveDate to update
    @param $data                    - required. DECOR concept xml element containing everything that should be in the updated concept
    @return concept object as xml with json:array set on elements
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function mpapi:patchConceptMap($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $results                := utilmp:patchConceptMap($authmap, string($id), string($effectiveDate), $data)
    
    return
        element {name($results)} {
            $results/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($results/*)
        }
};

(: Create a conceptMap, either empty or based on another conceptMap

    @param $projectPrefix           - project to create this scenario in
    @param $targetDate              - optional. if true invokes effectiveDate of the new conceptMap and concepts as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
    @param $sourceId                - optional. parameter denoting the id of a dataset to use as a basis for creating the new conceptMap
    @param $sourceEffectiveDate     - optional. parameter denoting the effectiveDate of a dataset to use as a basis for creating the new conceptMap",
    @param $keepIds                 - optional. only relevant if source dataset is specified. If true, the new conceptMap will keep the same ids for the new conceptMap, and only update the effectiveDate
    @param $baseDatasetId           - optional. only relevant when a source dataset is specified and `keepIds` is false. This overrides the default base id for datasets in the project. The value SHALL match one of the projects base ids for datasets
    @param $data                    - optional. DECOR concept xml element containing everything that should be in the new concept
    @return (empty) dataset object as xml with json:array set on elements
:)
declare function mpapi:postConceptMap($request as map(*)) {

    let $authmap                := $request?user
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $targetDate             := $request?parameters?targetDate = true()
    let $sourceId               := $request?parameters?sourceId
    let $data                   := utillib:getBodyAsXml($request?body, 'conceptMap', ())
    let $sourceEffectiveDate    := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?sourceEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?sourceEffectiveDate[string-length() gt 0]
        }
    let $refOnly                := $request?parameters?refOnly = true()
    let $keepIds                := $request?parameters?keepIds = true()
    let $baseId                 := $request?parameters?baseId
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($projectPrefix)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter prefix')
        else ()
    
    let $results                := utilmp:createConceptMap($authmap, $projectPrefix, $targetDate, $sourceId, $sourceEffectiveDate, $refOnly, $keepIds, $baseId, $data)
    
    return
        roaster:response(201, 
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        
        )
};

(:~ Update DECOR conceptMap

    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR conceptMap/@id to update
    @param $data                    - required. DECOR conceptMap xml element containing everything that should be in the updated conceptMap
    @return conceptMap object as xml with json:array set on elements
    @since 2020-05-03
:)
declare function mpapi:putConceptMap($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                           := utillib:getBodyAsXml($request?body, 'conceptMap', ())
    let $deletelock                     := $request?parameters?deletelock = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $results                        := utilmp:putConceptMap($authmap, $id, $effectiveDate, $data, $deletelock)
    
    return (
        for $result in $results
        return
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
        )

};

(:~ Retrieves a template in edit mode
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @return template
    @since 2020-05-03
:)
declare function mpapi:getConceptMapForEdit($request as map(*)) {
    let $authmap            := $request?user
    let $id                 := $request?parameters?id[not(. = '')]
    let $effectiveDate      := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $breakLock          := $request?parameters?breakLock = true()
    
    let $acceptTypes        := roaster:accepted-content-types()
    let $acceptedType       := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]
    let $format             := tokenize($acceptedType, '/')[2]
           
    let $check              :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results            := utilmp:getConceptMapForEdit($authmap, $id, $effectiveDate, $breakLock)
    
    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*, $format)
        }    
};