xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Server API allows read, create, update of ART-DECOR Server properties :)
module namespace servermgmtapi             = "http://art-decor.org/ns/api/server-mgmt";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace permlib     = "http://art-decor.org/ns/api/permissions" at "library/permissions-lib.xqm";
import module namespace serverapi   = "http://art-decor.org/ns/api/server" at "server-api.xqm";
import module namespace claml       = "http://art-decor.org/ns/terminology/claml" at "../../terminology/claml/api/api-claml.xqm";
import module namespace decor-cache = "http://art-decor.org/ns/api/decor-cache" at "library/decor-cache.xqm";


(: TODO refactor references to art modules? :)
import module namespace adpfix      = "http://art-decor.org/ns/art-decor-permissions" at "../../art/api/api-permissions.xqm";

declare namespace http              = "http://expath.org/ns/http-client";
declare namespace json              = "http://www.json.org";
declare namespace expath            = "http://expath.org/ns/pkg";
declare namespace scheduler         = "http://exist-db.org/xquery/scheduler";

(:~ Update permissions related to ART and DECOR package (e.g. projects). Require dba authentication
@return TODO
@since 2020-05-03
@see https://docs.art-decor.org/administration/#fix-art-permissions
:)
declare function servermgmtapi:postUpdatePermissionsArt($request as map(*)) {
    (:let $dummy := util:log('INFO', 'servermgmtapi:postUpdatePermissionsArt()'):)
    let $authmap                        := $request?user
    
    let $check := servermgmtapi:checkAuthentication($authmap)

    let $fix    := adpfix:setArtPermissions()
    let $fix    := adpfix:setDecorPermissions()
    
    return <result dateTime='{current-dateTime()}'/>
};

(:~ Clean up /db/apps/decor/tmp and /db/apps/decor/releases/*/development so only the latest 3 compilations per project are left. Require dba authentication
@return TODO
@since 2020-05-03
@see https://docs.art-decor.org/administration/#clean-up-decor
:)
declare function servermgmtapi:postCleanUpDecor($request as map(*)) {
    (:let $dummy := util:log('INFO', 'servermgmtapi:postUpdatePermissionsArt()'):)
    let $authmap                        := $request?user
    
    let $check := servermgmtapi:checkAuthentication($authmap)

    let $fix    := servermgmtapi:cleanupDecorTmp()
    let $fix    := servermgmtapi:cleanupDevelopmentCompilations()
    
    return <result dateTime='{current-dateTime()}'/>
};

(:~ Clean up compilations in /db/apps/decor/tmp if they are not the latest compile for the project. Require dba authentication
@return TODO
@since 2023-11-08
@see https://docs.art-decor.org/administration/#clean-up-decor
:)
declare %private function servermgmtapi:cleanupDecorTmp() {
    let $cleanup    :=
        for $decors in collection($setlib:strDecorTemp)/compile[@compileDate castable as xs:dateTime][@compilationFinished castable as xs:dateTime]
        let $projectPrefix  := $decors/decor/project/@prefix
        group by $projectPrefix
        return (
            let $newest := max($decors/xs:dateTime(@compileDate))
            
            return (
                for $decor in $decors[not(@compileDate = string($newest))]
                let $dummy := util:log('INFO', 'cleanup decor tmp: removing out-dated temporary pre-compiled set for "' || $projectPrefix || '"...')
                return (
                    xmldb:remove(util:collection-name($decor), util:document-name($decor))
                )
            )
        )
    return $cleanup
};

(:~ Clean up all but the newest 3 project runtime compilations in /db/apps/decor/releases/*/development. Require dba authentication
@return TODO
@since 2023-11-08
@see https://docs.art-decor.org/administration/#clean-up-decor
:)
declare %private function servermgmtapi:cleanupDevelopmentCompilations() as element(compiled)* {
    let $pcount         := count(xmldb:get-child-collections($setlib:strDecorVersion))
    for $projectPrefix at $pc in xmldb:get-child-collections($setlib:strDecorVersion)
    let $compilations   := 
        for $compilation in $setlib:colDecorVersion//compiled[@for = concat($projectPrefix, '-')][ancestor::compilation]
        order by $compilation/@on descending
        return $compilation
    let $ocount     := count(subsequence($compilations, 4))
    let $dummy := if ($ocount > 0)
        then util:log('INFO', 'cleanup development compilations (' || $pc || '/' || $pcount || '): removing ' || $ocount || ' out-dated project runtime compilation(s) for "' || $projectPrefix || '"...')
        else util:log('INFO', 'cleanup development compilations (' || $pc || '/' || $pcount || '): no out-dated project runtime compilations for "' || $projectPrefix || '"...')
    order by lower-case($projectPrefix)
    return (
        for $ref in subsequence($compilations, 4)/@as
        let $c          := <compiled>{$ref/../@*}</compiled>
        let $del        := 
            try {
                xmldb:remove(concat(util:collection-name($ref), '/', $ref)),
                xmldb:remove(util:collection-name($ref), util:document-name($ref))
            }
            catch * {$c/@as, $err:code, $err:description}
        order by $c/@on
        return
            <compiled>{$c/@*}</compiled>
    )
};

(:~ Update permissions related to FHIR package. Note that FHIR package may not be installed and versions 1.0, 3.0, 4.0 and more may exist. Require dba authentication
@return TODO
@since 2020-05-03
:)
declare function servermgmtapi:postUpdatePermissionsFhir($request as map(*)) {
    (:let $dummy := util:log('INFO', 'servermgmtapi:postUpdatePermissionsFhir()'):)
    let $authmap := $request?user
    let $check := servermgmtapi:checkAuthentication($authmap)
    
    let $version := string($request?parameters?version)
    let $operationType := string($request?parameters?type)
    
    let $allFhirVersions as xs:string* := if ($version eq '') then serverapi:getInstalledFhirServices() else $version
    
    (: don't rewrap errors deeper down. They should be self-explanatory :)
    let $results :=
        for $fhirVersion in $allFhirVersions
        return servermgmtapi:performUpdatePermissionsFhir($fhirVersion, $operationType)
    
    return <result dateTime="{current-dateTime()}"/>
};

(:~ Update permissions related to Terminology package. Note that the package may not be installed. Require dba authentication
@return TODO
@since 2020-05-03
:)
declare function servermgmtapi:postUpdatePermissionsTerminology($request as map(*)) {
    (:let $dummy := util:log('INFO', 'servermgmtapi:postUpdatePermissionsTerminology()'):)
    
    let $authmap                        := $request?user
    
    let $check := servermgmtapi:checkAuthentication($authmap)
    
    let $moduleNS := xs:anyURI('http://art-decor.org/ns/terminology-permissions')
    let $modulePaths := (repo:get-root() || 'terminology/api/api-permissions.xqm')
    
    (:let $dummy := util:log('INFO', 'Dynamically loading module with namespace "' || $moduleNS || '" and paths ' || string-join($modulePaths, ', ')):)
    
    let $module := load-xquery-module($moduleNS, map{'location-hints': $modulePaths})
    
    (:let $dummy := util:log('INFO', 'Module with namespace "' || $moduleNS || '" loaded succesfully'):)
    
    let $function1 := servermgmtapi:function-lookup($module, QName($moduleNS, 'setTerminologyQueryPermissions'), 0)
    let $function2 := servermgmtapi:function-lookup($module, QName($moduleNS, 'setTerminologyAuthoringCollectionPermissions'), 0)
    
    (: Instead of checking if a function is available we catch the server error if it happens and inform the user like we do if the functions exists but throws an error. :)
    let $result := $function1()
    let $result := ($result, $function2())
    
    return <result dateTime="{current-dateTime()}"/>
};

(:~ Update permissions related to Oids package. Note that the package may not be installed. Require dba authentication
@return TODO
@since 2020-05-03
:)
declare function servermgmtapi:postUpdatePermissionsOids($request as map(*)) {
    (:let $dummy := util:log('INFO', 'servermgmtapi:postUpdatePermissionsOids()'):)
    
    let $authmap                        := $request?user
    
    let $check := servermgmtapi:checkAuthentication($authmap)
    
    let $moduleNS := xs:anyURI('http://art-decor.org/ns/oids/permissions')
    let $modulePaths := (repo:get-root() || 'tools/oids/api/api-permissions.xqm')
    
    (:let $dummy := util:log('INFO', 'Dynamically loading module with namespace "' || $moduleNS || '" and paths ' || string-join($modulePaths, ', ')):)
    
    let $module := load-xquery-module($moduleNS, map{'location-hints': $modulePaths})
    
    (:let $dummy := util:log('INFO', 'Module with namespace "' || $moduleNS || '" loaded succesfully'):)
    
    let $function := servermgmtapi:function-lookup($module, QName($moduleNS, 'setOidsPermissions'), 0)
    
    (: Instead of checking if a function is available we catch the server error if it happens and inform the user like we do if the functions exists but throws an error. :)
    let $result := $function()
    
    return <result dateTime="{current-dateTime()}"/>
};

(:~ Retrieve server-functions xml containing CRUD markers for UI-functionality:)
declare function servermgmtapi:getServerFunctions($request as map(*)) {
    
    let $results := doc($setlib:strServerFunctions)/server-functions

    return 
    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Update one or all OID Registries lookup files. Note that the package may not be installed. Note that this function may need to move to a separate set of OID functions. Require dba authentication
@return TODO
@since 2020-05-03
:)
declare function servermgmtapi:postUpdateOidLookups($request as map(*)) {
    let $moduleNS := xs:anyURI('http://art-decor.org/ns/tools/oids')
    let $modulePaths := (repo:get-root() || 'tools/oids/api/api-oids.xqm')
    
    let $authmap                        := $request?user
    let $statusonly                     := $request?parameters?statusonly = true()
    let $force                          := $request?parameters?force = true()
    let $registryname                   := 
        typeswitch ($request?parameters?registryname) 
        case xs:string return 
            array:flatten(
                for $s in $request?parameters?registryname[string-length() gt 0]
                return tokenize($s,'\s')
            )

        default return array:flatten($request?parameters?registryname)
    
    (:let $dummy := util:log('INFO', 'servermgmtapi:postUpdatePermissionsOids(), force="' || $force || '", registryname="' || $registryname || '"'):)
    
    let $check := servermgmtapi:checkAuthentication($authmap)
    
    (:let $dummy := util:log('INFO', 'Dynamically loading module with namespace "' || $moduleNS || '" and paths ' || string-join($modulePaths, ', ')):)
    
    let $module := load-xquery-module($moduleNS, map{'location-hints': $modulePaths})
    
    (:let $dummy := util:log('INFO', 'Module with namespace "' || $moduleNS || '" loaded succesfully'):)
    
    (: Instead of checking if a function is available we catch the server error if it happens and inform the user like we do if the functions exists but throws an error. :)
    let $function := servermgmtapi:function-lookup($module, QName($moduleNS, 'createOidRegistriesLookup'), 2)
    
    let $results := $function($registryname, $force)
    
    (: $result created by this call: <result dateTime="{current-dateTime()}">{$lookupFiles}</result> :)
    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Update classification index. Require dba authentication
@return result object with index, codeSystems count and last execution time
@since 2020-05-03
:)
declare function servermgmtapi:postUpdateClassificationIndex($request as map(*)) {
    (:let $dummy := util:log('INFO', 'servermgmtapi:postUpdateClassicationIndex()'):)
    let $authmap                        := $request?user
    let $statusonly                     := $request?parameters?statusonly = true()
    
    let $check                          := servermgmtapi:checkAuthentication($authmap)
    (: sample result:
      <result index="/db/apps/terminology/claml/classification-index.xml" codeSystems="0" time="2021-06-01T00:15:00+02:00"/>
    :)
    let $results                       := claml:createClaMLIndex($statusonly)
     
    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Update cache of DECOR projects. Require dba authentication
@return cachedBuildingBlockRepositories object
@since 2020-05-03
:)
declare function servermgmtapi:postUpdateDecorCache($request as map(*)) {
    let $authmap                        := $request?user
    
    let $statusonly                     := $request?parameters?statusonly = true()
    let $cacheformat                    := ($request?parameters?cacheformat[string-length() gt 0], 'decor')[1]
    
    (:let $dummy := util:log('INFO', 'servermgmtapi:postUpdateDecorCache(), user="' || $authmap?name || '", statusonly="' || $statusonly || '", cacheformat="' || $cacheformat || '"'):)
    
    let $check                          := servermgmtapi:checkAuthentication($authmap)
    let $results                        := decor-cache:updateDecorCache($authmap, $statusonly, $cacheformat, 'post')
    
    (: update the CADTS systems :)
    let $now                            := substring(string(current-dateTime()), 1, 19)
    let $stamp                          := util:uuid()
    let $projectcsconvert-request       :=  
        <projectcodesystemconvert-request uuid="{$stamp}" for="*cache*" on="{current-dateTime()}" as="{$stamp}" by="{$authmap?name}" progress="Added to process queue ..." progress-percentage="0"/>
    let $write                          := if ($statusonly) then () else xmldb:store($setlib:strDecorScheduledTasks, 'projectcodesystemconvert--cache--' || replace($now, '\D', '') || '.xml', $projectcsconvert-request)
    let $tt                             := if ($statusonly) then () else sm:chmod(xs:anyURI($write), 'rw-rw----')
    (:/ update the CADTS systems :)
    
    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Check for dba authentication.
@param $authmap map with user name, groups and isdba boolean
@return empty sequence if authentication OK, else the error() function is called.
@since 2021-05-19
:)
declare %private function servermgmtapi:checkAuthentication($authmap as map(*)?) {
    if (empty($authmap)) then
        error($errors:UNAUTHORIZED, 'You need to authenticate first')
    else
    if ($authmap?groups = 'dba') then () else (
        error($errors:FORBIDDEN, 'You need to be dba for this function')
    )
};

(:~ A wrapper around utillib:function-lookup() that throws a SERVER_ERROR, if the function is not found. :)
declare %private function servermgmtapi:function-lookup($module as map(*), $qname as xs:QName, $arity as xs:nonNegativeInteger) as function(*)? {
    let $function := utillib:function-lookup($module, $qname, $arity)
    return
       if (exists($function))
       then $function
       else error($errors:SERVER_ERROR, 'Dynamically loaded function not found, function "' || $qname || '", arity ' || $arity)
};

(:~ Auxiliary function that performs the required operation for the public calling function servermgmtapi:postUpdatePermissionsFhir().
    @return the result of the operation corresponding to the required operation
    @since 2021-05-20
:)
declare %private function servermgmtapi:performUpdatePermissionsFhir($fhirVersion as xs:string, $operationType as xs:string) {
    (:let $dummy := util:log('INFO', 'servermgmtapi:performUpdatePermissionsFhir(), version="' || $fhirVersion || '", type="' || $operationType || '"'):)
    
    let $allowedVersions as xs:string*    := serverapi:getInstalledFhirServices()
    let $check :=
        if ($fhirVersion = $allowedVersions) then () else
        if (empty($allowedVersions)) then 
            error($errors:BAD_REQUEST, 'This server currently has no FHIR servers installed, hence there are no permissions to update.')
        else (
            error($errors:BAD_REQUEST, 'This server currently does not have FHIR server version ' || $fhirVersion || ' installed. Installed version(s): ' || string-join($allowedVersions, ', '))
        )
        
    let $moduleNS := xs:anyURI('http://art-decor.org/ns/fhir-permissions')
    let $baseFhirModulePath := repo:get-root() || 'fhir/' || $fhirVersion || '/api/'
    let $modulePaths := ($baseFhirModulePath || 'api-permissions.xqm')
    
    (:let $dummy := util:log('INFO', 'Dynamically loading module with namespace "' || $moduleNS || '" and paths ' || string-join($modulePaths, ', ')):)
    
    let $module := load-xquery-module($moduleNS, map{'location-hints': $modulePaths})
    
    (:let $dummy := util:log('INFO', 'Module with namespace "' || $moduleNS || '" loaded succesfully'):)
    
    let $adfixNS := 'http://art-decor.org/ns/fhir-permissions'
    
    let $fhirFunction :=
        switch ($operationType)
        case ('capabilitystatement') return servermgmtapi:function-lookup($module, QName($adfixNS, 'refreshCapabilityStatement'), 0)
        case ('conformance') return servermgmtapi:function-lookup($module, QName($adfixNS, 'refreshCapabilityStatement'), 0)
        case ('cleanup') return servermgmtapi:function-lookup($module, QName($adfixNS, 'cleanupData'), 0)
        default return servermgmtapi:function-lookup($module, QName($adfixNS, 'setPermissions'), 0)
    
    return $fhirFunction()
};

(:~ Function to retrieve the details of all scheduled jobs of the eXist db. 
    Please note that you must be a dba to call this function. 
    The return structure is very similar to the origin, see 
    https://exist-db.org/exist/apps/fundocs/view.html?uri=http://exist-db.org/xquery/scheduler
    but has no namespace anymore, summarizes data and returns ART-DERCOR related scheduled-jobs only.
    @return scheduler structure as list of jobs focus on ART-DERCOR related scheduled-jobs
    @since 2022-07-04
:)
declare function servermgmtapi:getSchedulers($request as map(*)) {
    let $authmap  := $request?user
    let $check    := servermgmtapi:checkAuthentication($authmap)
    
    let $recognizedartdecortasks :=
        ('scheduled-notifier', 'scheduled-refreshs', 'scheduled-tasks', 'periodic-notifier')
        
    let $sj       := scheduler:get-scheduled-jobs()
    let $pj       := collection($setlib:strDecorScheduledTasks)/*[@for]
    
    let $results  :=
    (
        for $j in $sj/scheduler:jobs//scheduler:group/scheduler:job[@name = $recognizedartdecortasks]
            return
                <artdecorjob name="{$j/@name}" json:array="true">
                {   
                    for $t in $j/scheduler:trigger
                    return
                        <trigger>
                        {
                            attribute expression { $t/expression },
                            attribute state { $t/state },
                            attribute start { $t/start },
                            attribute end { $t/end },
                            attribute previous { $t/previous },
                            attribute next { $t/next },
                            attribute final { $t/final }
                        }
                        </trigger>
                }
                </artdecorjob>,
        for $p in $pj
            return
                <projectjob name="{$p/name()}" json:array="true">
                {
                    $p/(@* except @name)
                }
                </projectjob>
    )

    let $count := $results/*

    return
        <list artifact="ALLSCHEDULEDTASKS" current="{$count}" total="{$count}" all="{$count}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org"> 
        {
            $results
        }
        </list>
};

(:~ Check for server integrity.

    Checks server-wide the following topics
    * is there any dataset in a project that contains the same if and effectiveDate more than once (PRJ-DS-UNIQUE)
    * check on double code systems (same id) in the code system stable terminology branch (TERM-CS-UNIQUE)
    
    All checks are allowed only if authenticated as dba
    
@param $authmap map with user name, groups and isdba boolean
@return empty sequence if authentication OK, else the error() function is called.
@since 2025-10-16
:)
declare function servermgmtapi:serverIntegrityChecks($request as map(*)) {

    (:
    let $check :=
       if (empty($authmap)) then
           error($errors:UNAUTHORIZED, 'You need to authenticate first')
       else
       if ($authmap?groups = 'dba') then () else (
            error($errors:FORBIDDEN, 'You need to be dba for this function')
       )
    :)
    
    (:
        CHECK I: is there any dataset in a project that contains the same if and effectiveDate more than once (PRJ-DS-UNIQUE)
        --------
        @returns project id and name with problematic dataset(s) and the dataset information
        
        <problem type="PRJ-DS-UNIQUE" id=".." name="..">
            <dataset id=".." effectiveDate=".." name=".." count=".."/>
        </problem>
    :)
    
    let $scopedprojects := $setlib:colDecorData/decor
    
    let $tmp1 :=
        <result>
        {
            for $p in $scopedprojects
            let $projectid := $p/project/@id
            let $projectname := ($p/project/name/text())[1]
            let $dss := $p/datasets/dataset
            return
                <problem json:array="true" type="PRJ-DS-UNIQUE" id="{$projectid}" name="{$projectname}">
                {
                    for $ds in $dss
                    let $deid := $ds/@id
                    let $deed := $ds/@effectiveDate
                    let $dename := ($ds/name/text())[1]
                    return
                        if (count($dss[@id = $deid][@effectiveDate = $deed]) gt 1) 
                        then <dataset json:array="true" id="{$deid}" effectiveDate="{$deed}" name="{$dename}" count="{count($dss[@id = $deid][@effectiveDate = $deed])}"/>
                        else ()
                }
                </problem>
        }
        </result>
        
    let $results1 :=
        <result>
        {
            for $r in $tmp1/problem
            return 
                if (count($r/dataset) = 0) then ()
                else
                    <problem json:array="true" type="PRJ-DS-UNIQUE" id="{$r/@id}" name="{$r/@name}">
                    {
                        for $d in $r/dataset
                        let $cat := concat($d/@id, $d/@effectiveDate)
                        group by $cat
                        return $d[1]
                    }
                    </problem>
        }
        </result>
             
     (:
        CHECK II: check on double code systems (same id) in the code system stable terminology branch (TRM-CS-UNIQUE)
        ---------
        @returns code system id and name with double existence and database location path where the duplicates were detected
        
        <problem id="2.16.840.1.113883.3.1937.99.62.3.5.1" name="cs-measured-by">
           <codeSystem location="/db/apps/terminology-data/codesystem-stable-data/projects/demo1-/cs-measured-by-2.16.840.1.113883.3.1937.99.62.3.5.1"/>
           <codeSystem location="/db/apps/terminology-data/codesystem-stable-data/projects/test-"/>
        </problem>
    :)
    let $codeSystemIds := collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid
    let $distinctIds   := distinct-values($codeSystemIds)
    let $results2 :=
        <result>
        {
            if (count($codeSystemIds) = count($distinctIds)) then ()
            else
               for $id in $distinctIds
               let $codeSystems := collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid = $id]
               let $name :=
                  if ($codeSystems[1]/name[@language = 'en-US']) then
                     $codeSystems[1]/name[@language = 'en-US']
                  else
                     $codeSystems[1]/name[1]
               return
                  if (count($codeSystems) gt 1) then
                     <problem json:array="true" type="TERM-CS-UNIQUE" id="{$id}" name="{$name}">
                        {
                           for $system in $codeSystems
                           return
                           <codeSystem json:array="true" location="{util:collection-name($system)}"/>
                        }
                     </problem>
                  else
                     ()
         }
         </result>
     (:
        CHECK III: check on concept(s) that have contains + other element(s) than comment (PRJ-DE-CONTAINS)
        ---------
        @returns project id and name where concepts with a child contain element were detected that have other element(s) than comment,
                 in addition the offended contains element with ref an flexibility is given for better location

        <problem type="PRJ-DE-CONTAINS" count="1" id="2.16.840.1.113883.2.4.3.11.60.999.999.3" name=".." prefix="suitkomstgerichte-">
            <offendedContains ref="2.16.840.1.113883.3.1937.777.28.2.16747" flexibility="2021-11-11T13:01:34"/>
        </problem>
     :)
     let $allcontainswithconcepts := $setlib:colDecorData//contains[../concept]
     let $results3 :=
        <result>
        {
            for $p in $allcontainswithconcepts
            let $projectPrefix := $p/ancestor::decor/project/@prefix
            let $projectid := $p/ancestor::decor/project/@id
            let $projectname := ($p/ancestor::decor/project/name/text())[1]
            return
            <problem json:array="true" type="PRJ-DE-CONTAINS" count="{count($p)}" id="{$projectid}" name="{$projectname}" prefix="{$projectPrefix}">
            {
                <offendedContains json:array="true">{$p/@*}</offendedContains>
            }
            </problem>
        }
        </result>
         
     (:
        Put together everything in one integritycheck result 
        ---------
        A complete report example is shown here
        
        <list xmlns:json="http://www.json.org" artifact="SERVERINTEGRITY" current="3" total="3" all="3" lastModifiedDate="2025-11-11T11:01:52.8+01:00">
            <integritycheck>
                <problem json:array="true" type="PRJ-DS-UNIQUE" id="2.16.840.1.113883.3.1937.99.62.3" name="Demo 1: Measurements by patient">
                    <dataset json:array="true" id="2.16.840.1.113883.3.1937.99.62.3.1.1" effectiveDate="2012-05-30T11:32:36" name="Decor 1 Demo dataset" count="2"/>
                </problem>
                <problem json:array="true" type="TERM-CS-UNIQUE" id="2.16.840.1.113883.6.73" name="WC">
                    <codeSystem location="/db/apps/terminology-data/codesystem-stable-data/external/hl7intl/v3/v3-WC"/>
                    <codeSystem location="/db/apps/terminology-data/codesystem-stable-data/external/atc"/>
                </problem>
                <problem json:array="true" type="PRJ-DE-CONTAINS" count="1" id="2.16.840.1.113883.3.1937.777.28" name="PRSB Record structure and content" prefix="prsb03-">
                    <offendedContains ref="2.16.840.1.113883.3.1937.777.28.2.16747" flexibility="2021-11-11T13:01:34"/>
                </problem>
            </integritycheck>
        </list>
        
     :)
     let $results :=
        <integritycheck>
        {   
            $results1/problem,
            $results2/problem,
            $results3/problem
        }
        </integritycheck>
        
     let $count := count($results/problem)

     (:
        FINAL RESULTS
        -------------
    :)
     return
        <list artifact="SERVERINTEGRITY" current="{$count}" total="{$count}" all="{$count}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org"> 
        {
            $results
        }
        </list>
        
};
<<<<<<< HEAD

(: ======================= SCHEDULED-TASK START  ======================== :)

declare function servermgmtapi:getServerScheduledTasks($request as map(*)) {
    
    let $authmap                := $request?user
    let $taskId                 := $request?parameters?id[string-length() gt 0]
    let $taskType               := $request?parameters?type[string-length() gt 0]
    let $taskBy                 := $request?parameters?by[string-length() gt 0]
    let $sort                   := $request?parameters?sort[string-length() gt 0]
    let $sortOrder              := $request?parameters?sortorder[string-length() gt 0]
    
    let $results                := servermgmtapi:getServerScheduledTasks($authmap, $taskId, $taskType, $taskBy, $sort, $sortOrder)
    
    return
        $results
};

declare function servermgmtapi:getServerScheduledTasks($authmap as map(*)?, $taskId as xs:string*, $taskType as xs:string*, $taskBy as xs:string*, $sort as xs:string?, $sortorder as xs:string?) {
    
    (:if you are author in one, then you are author in all:)
    let $userIsAuthor           := if (empty($authmap)) then false() else false()
    
    let $sortorder              := $sortorder[. = 'descending']
    (: get all scheduled tasks :)
    let $allRequests            := collection($setlib:strDecorScheduledTasks)/*[name() = ('codesystem-report-request','integrity-check-request')]
    let $allCount               := count($allRequests)
    let $allRequests            := if (empty($taskId)) then $allRequests else $allRequests[@uuid = $taskId]
    let $allRequests            := if (empty($taskType)) then $allRequests else $allRequests[name() = $taskType]
    let $allRequests            := if (empty($taskBy)) then $allRequests else $allRequests[@by= $taskBy]
    
    let $count                  := count($allRequests)
    
    let $results                :=
        switch ($sort)
        case 'type' return
            if (empty($sortorder)) then 
                for $request in $allRequests order by $request/name() return $request
            else
                for $request in $allRequests order by $request/name() descending return $request
              
        case 'by' return
            if (empty($sortorder)) then 
                for $request in $allRequests order by $request/@by return $request
            else
                for $request in $allRequests order by $request/@by descending return $request   
              
        default return
            if (empty($sortorder)) then 
                for $request in $allRequests order by $request/@on return $request
            else
                for $request in $allRequests order by $request/@on descending return $request     

    let $results                :=
        if ($userIsAuthor) then $results else (
            for $request in $results return element {name($request)} {$request/(@* except @by), $request/node()}
        )
    
    return
        <list artifact="SCHEDULEDTASKS" current="{$count}" total="{$count}" all="{$allCount}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org"> 
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

declare function servermgmtapi:deleteServerScheduledTasks($request as map(*)) {
    
    let $authmap                := $request?user
    let $taskId                 := $request?parameters?id[string-length() gt 0]
    let $taskType               := $request?parameters?type[string-length() gt 0]
    let $taskBy                 := $request?parameters?by[string-length() gt 0]
    let $tasksInError           := $request?parameters?error[string-length() gt 0] = true()
    let $sort                   := $request?parameters?sort[string-length() gt 0]
    let $sortOrder              := $request?parameters?sortorder[string-length() gt 0]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                  :=
        for $d in $decor
        return
            if ($authmap?groups = ('dba'),'terminology') then true() else (
                error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to delete scheduled terminology tasks.'))
            )
    let $projectPrefixes        := if ($decor) then sort($decor/project/@prefix) else if ($project = '*cache*') then $project else ('*')
    
    let $allRequests            := collection($setlib:strDecorScheduledTasks)/*[@codeSystem]
    let $allRequests            := if (empty($taskId)) then $allRequests else $allRequests[@uuid = $taskId]
    let $allRequests            := if (empty($taskType)) then $allRequests else $allRequests[name() = $taskType]
    let $allRequests            := if (empty($taskBy)) then $allRequests else $allRequests[@by= $taskBy]
    let $allRequests            := if ($tasksInError) then $allRequests[@busy = 'error'] else $allRequests
    
    (: if a specific id was requested, give error if that is running. In all other cases just skip those without mention :)
    let $check                  :=
        if (empty($taskId)) then () else if ($allRequests[@busy = 'true']) then
            error($errors:BAD_REQUEST, 'You cannot delete a running task. Please wait until the task is either done or in error state, before trying again.')
        else ()
    
    let $delete                 := 
        for $r in $allRequests
        return
            if ($r[@busy = 'true']) then () else (
                xmldb:remove(util:collection-name($r), util:document-name($r))
            )
    
    let $results                := servermgmtapi:getServerScheduledTasks($authmap, (), $taskType, $taskBy, $sort, $sortOrder)
     
    return
        roaster:response(200, (), $results, map { "Last-Modified": $results/@lastModifiedDate })
};
=======
>>>>>>> e64dec383d855770a78c7707496b825429f4e826
