xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilq                  = "http://art-decor.org/ns/api/util-questionnaire";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace decorlib        = "http://art-decor.org/ns/api/decor" at "decor-lib.xqm";
import module namespace histlib         = "http://art-decor.org/ns/api/history" at "history-lib.xqm";
import module namespace ruleslib        = "http://art-decor.org/ns/api/rules" at "rules-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

declare namespace f                     = "http://hl7.org/fhir";

(:~ All functions support their own override, but this is the fallback for the maximum number of results returned on a search :)
declare %private variable $utilq:maxResults              := 50;
declare %private variable $utilq:STATUSCODES-FINAL       := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $utilq:RELATIONSHIP-TYPES      := utillib:getDecorTypes()/QuestionnaireRelationshipTypes;
declare %private variable $utilq:FHIROBJECT-FORMAT       := utillib:getDecorTypes()/FhirObjectFormat;
declare %private variable $utilq:FHIRRESOURCE-TYPE       := utillib:getDecorTypes()/FHIRResourceType;

(: =========================================================================================================================================== :)
(: ===============                                            QUESTIONNAIRE LOGIC                                              =============== :)
(: =========================================================================================================================================== :)

(:~ Central logic returning a list of zero or more questionnaires
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $objectId                - optional. Filter output to just those based on this questionnaire/relationship/@ref. For each questionnaire
    @param $objectEffectiveDate     - optional. Filter output to just those based on this questionnaire/relationship/@flexibility. relevant when objectId has a value.
    @param $sort                    - optional parameter to indicate sorting. Only option 'name'
    @param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
    @return List object with zero or more quesrionnaire
    @since 2020-05-03
:)
declare function utilq:getQuestionnaireList($projectPrefix as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $objectId as xs:string?, $objectEffectiveDate as xs:string?, $sort as xs:string?, $sortorder as xs:string?, $max as xs:string?) as element(list){
    
    let $decor                  := if (empty($projectPrefix)) then () else utillib:getDecor($projectPrefix, $projectVersion, $projectLanguage)
    
    let $allcnt                 := count($decor//questionnaire)
    
    let $allQuestionnaireResponses  := collection(util:collection-name($decor))//f:QuestionnaireResponse
    let $filteredQuestionnaires     := 
        if (empty($objectId)) then $decor//questionnaire else (
            if (empty($objectEffectiveDate)) then (
                let $object                     :=
                    if (empty($objectId)) then () else (
                        utillib:getScenario($objectId, $objectEffectiveDate, $decor, $projectVersion, $projectLanguage) |
                        utillib:getTransaction($objectId, $objectEffectiveDate, $decor, $projectVersion, $projectLanguage) |
                        utillib:getDataset($objectId, $objectEffectiveDate, $decor, $projectVersion, $projectLanguage)
                    )
                (: get questionnaires related to object :)
                return
                    if ($object) then $decor//questionnaire[relationship[@ref = $object/@id][@flexibility = $object/@effectiveDate]]
                    else $decor//questionnaire[relationship[@ref = $objectId]]
            )
            (: get questionnaires and questionnaireresponses related to object :)
            else $decor//questionnaire[relationship[@ref = $objectId][@flexibility = $objectEffectiveDate]]
        )
    
    let $projectLanguage        := 
        if (empty($projectLanguage)) then 
            if (empty($projectPrefix)) then $filteredQuestionnaires/ancestor::decor/project/@defaultLanguage
            else $decor/project/@defaultLanguage
        else $projectLanguage
    
    let $filteredQuestionnaires := $filteredQuestionnaires[self::questionnaire]
    
    (:can sort on indexed db content -- faster:)
    let $results                :=
        switch ($sort) 
        case 'effectiveDate'    return 
            if (empty($sortorder)) then 
                for $q in $filteredQuestionnaires order by $q/@effectiveDate return $q
            else
                for $q in $filteredQuestionnaires order by $q/@effectiveDate descending return $q
        case 'name'             return 
            if (empty($sortorder)) then 
                for $q in $filteredQuestionnaires order by lower-case($q/@name) return $q
            else
                for $q in $filteredQuestionnaires order by lower-case($q/@name) descending return $q
        default                 return $filteredQuestionnaires
    
    let $count              := count($results)
    let $max                := if ($max ge 1) then $max else $utilq:maxResults
    
    (: build map of oid names. is more costly when you do that deep down :)
    let $oidnamemap         := utilq:buildOidNameMap($results, $projectLanguage[1])
    let $questionnaires     :=
        for $q in $results
        return utilq:handleQuestionnaire($q, $projectPrefix, $projectLanguage[1], $oidnamemap, $allQuestionnaireResponses, true(), false(), true())
    
    return
        <list artifact="QQQR" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            subsequence($questionnaires, 1, $max)
        }
        </list>
};

(:~ Central logic for patching an existing questionnaire
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR questionnaire/@id to update
    @param $effectiveDate   - required. DECOR questionnaire/@effectiveDate to update
    @param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
    @return questionnaire object as xml with json:array set on elements
:)
declare function utilq:patchQuestionnaire($authmap as map(*), $doQuestionnaire as xs:boolean, $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) as element(){

    let $storedQuestionnaire    := utillib:getQuestionnaire($id, $effectiveDate)
    
    let $decor                  := $storedQuestionnaire/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    let $projectLanguage        := $decor/project/@defaultLanguage

    let $check                  :=
        if (empty($storedQuestionnaire)) then (
            error($errors:BAD_REQUEST, 'Questionnaire with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
        else
        if (count($storedQuestionnaire) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple results for id ' || $id || ' effectiveDate ' || $effectiveDate || '. Expected 1..1. Alert your administrator as this should not be possible.')
        )
        else ()
    
    let $lock                   := utilq:checkQuestionnaireAccess($authmap, $decor, $id, $effectiveDate, true())
    
    let $check                  :=
        if ($storedQuestionnaire[@statusCode = $utilq:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $utilq:STATUSCODES-FINAL)]) then () else 
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('The ', name($storedQuestionnaire), ' cannot be patched while it has one of status: ', string-join($utilq:STATUSCODES-FINAL, ', '), if ($storedQuestionnaire/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()

    let $check                  := utilq:checkQuestionnaireParameters($data, $storedQuestionnaire, $doQuestionnaire)
    
    let $intention              := if ($storedQuestionnaire[@statusCode = 'final']) then 'patch' else 'version'
    let $objectType             := $decorlib:OBJECTTYPE-QUESTIONNAIRE
    let $update                 := histlib:AddHistory($authmap?name, $objectType, $projectPrefix, $intention, $storedQuestionnaire)
    
    let $update                 := utilq:patchQuestionnaireParameters($data, $storedQuestionnaire)
    
    let $update                 :=
        if ($storedQuestionnaire[@lastModifiedDate]) 
        then update value $storedQuestionnaire/@lastModifiedDate with substring(string(current-dateTime()), 1, 19)
        else update insert attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)} into $storedQuestionnaire
    
    (: after all the updates, the XML Schema order is likely off. Restore order - TODO: externalize into separate function based on FHIR version :)
    
    let $return                 := utilq:handleQuestionnaire($storedQuestionnaire, $projectPrefix, $projectLanguage, map {}, (), false(), false(), false())
    
    let $update                 := update replace $storedQuestionnaire with $return
    let $update                 := update value $storedQuestionnaire/@lastModifiedDate with substring(string(current-dateTime()), 1, 19)
    let $update                 := update delete $storedQuestionnaire/@uuid
    
    let $update                 := update delete $lock
    
    let $oidnamemap             := utilq:buildOidNameMap($storedQuestionnaire, $projectLanguage[1])
    let $return                 := utilq:handleQuestionnaire($storedQuestionnaire, $projectPrefix, $projectLanguage, $oidnamemap, (), false(), false(), true())
    
    return
        element {name($return)} {
            $return/@*,
            attribute keys {string-join(for $k in map:keys($oidnamemap) return $k || '=' || map:get($oidnamemap, $k), ' ')},
            $return/*
        }
};

(:~ Central logic for creating a questionnaire, either empty or with input data or based on another questionnaire
    @param $projectPrefix project to create this questionnaire in
    @param $targetDate If true invokes effectiveDate of the new questionnaire as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
    @param $sourceId parameter denoting the id of a questionnaire to use as a basis for creating the new questionnaire
    @param $sourceEffectiveDate parameter denoting the effectiveDate of a questionnaire to use as a basis for creating the new questionnaire,
    @param $keepIds Only relevant if source questionnaire is specified. If true, the new questionnaire will keep the same ids for the new questionnaire, and only update the effectiveDate
    @param $baseId Only relevant when a source questionnaire is specified and `keepIds` is false. This overrides the default base id for questionnaire in the project. The value SHALL match one of the projects base ids for questionnaire
    @return questionnaire oblect
:)
declare function utilq:createQuestionnaire($authmap as map(*), $projectPrefix as xs:string, $data as element(questionnaire)?, $targetDate as xs:boolean, $sourceId as xs:string?, $sourceEffectiveDate as xs:string?, $keepIds as xs:boolean?, $baseId as xs:string?) as element() {

    let $decor                  := utillib:getDecor($projectPrefix)
    let $projectLanguage        := $decor/project/@defaultLanguage
    let $now                    := if ($targetDate) then substring(string(current-date()), 1, 10) || 'T00:00:00' else substring(string(current-dateTime()), 1, 19)
    
    let $sourceQuestionnaire    := if (empty($data)) then if (empty($sourceId)) then () else utillib:getQuestionnaire($sourceId, $sourceEffectiveDate) else $data
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, 'Project with prefix ' || $projectPrefix || ' does not exist')
        )
    
    let $check                  := utilq:checkQuestionnaireAccess($authmap, $decor)
    
    let $baseId                 :=
        if (empty($baseId)) then decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-QUESTIONNAIRE)[1]/@id else
        if (decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-QUESTIONNAIRE)/@id = $baseId) then $baseId else (
            error($errors:BAD_REQUEST, 'Questionnaire base id ' || $baseId || ' is not configured as baseId for questionnaires in this project. Expected one of: ' || string-join(decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-QUESTIONNAIRE)/@id, ' '))
        )
    
    let $check                  :=
        if (empty($sourceId)) then () else if ($sourceQuestionnaire) then () else (
            error($errors:BAD_REQUEST, 'Source questionnaire based on id ' || $sourceId || ' and effectiveDate ' || $sourceEffectiveDate || ' does not exist. Cannot use this as source')
        )
    
    let $keepIds                := $keepIds = true()
    
    (: decorlib:getNextAvailableIdP() returns <next base="1.2" max="2" next="{$max + 1}" id="1.2.3" type="DS"/> :)
    let $newQuestionnaireId     := if ($keepIds) then $sourceQuestionnaire/@id else (decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-QUESTIONNAIRE, $baseId)/@id)
    
    let $check                  :=
        if (string-length($newQuestionnaireId) > 0 and string-length($projectLanguage) > 0) then () else (
            error($errors:BAD_REQUEST, 'Default base ID for questionnaires not set or project default language not set')
        )
        
    let $result                 :=
        if ($sourceQuestionnaire) then 
            <questionnaire id="{$newQuestionnaireId}" effectiveDate="{$now}" statusCode="new" lastModifiedDate="{$now}">
            {
                $sourceQuestionnaire/name,
                $sourceQuestionnaire/desc,
                $sourceQuestionnaire/classification,
                $sourceQuestionnaire/(node() except (name | desc | classification))
            }
            </questionnaire>
        else (
            <questionnaire id="{$newQuestionnaireId}" effectiveDate="{$now}" statusCode="draft" lastModifiedDate="{$now}">
                <name language="{$projectLanguage}">Questionnaire</name>
                <desc language="{$projectLanguage}">Questionnaire</desc>
            </questionnaire>
        )
    let $result                 := utilq:handleQuestionnaire($result, $decor/project/@prefix, $projectLanguage, map {}, (), false(), true(), false())
    
    let $check                  := ruleslib:checkDecorSchema($result, 'create')
    
    let $result                 := (
        comment { ' ' || replace($result/name[1], '--', '-') || ' ' },
        <questionnaireAssociation questionnaireId="{$result/@id}" questionnaireEffectiveDate="{$result/@effectiveDate}"/>,
        $result
    )
    
    let $updateQuestionnaire    := 
        if ($decor/rules) then update insert $result into $decor/rules
        else if ($decor/issues) then update insert <rules>{$result}</rules> preceding $decor/issues
        else update insert <rules>{$result}</rules> into $decor
    
    return utilq:handleQuestionnaire(utillib:getQuestionnaire($result/@id, $result/@effectiveDate), $decor/project/@prefix, $projectLanguage, map {}, (), false(), true(), true())
};

(:~ Central logic for updating an existing questionnaire
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR questionnaire/@id to update
    @param $effectiveDate   - required. DECOR questionnaire/@effectiveDate to update
    @param $data            - required. DECOR questionnaire xml element containing everything that should be in the updated questionnaire or quesrtionnaire response
    @return concept object as xml with json:array set on elements
:)
declare function utilq:putQuestionnaire($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as item(), $fhirVersion as xs:string?, $deletelock as xs:boolean, $doQuestionnaire as xs:boolean) as element() {

    let $storedQuestionnaire    := utillib:getQuestionnaire($id, $effectiveDate)
    let $decor                  := $storedQuestionnaire/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    let $projectLanguage        := $decor/project/@defaultLanguage
    let $objectName             := 
        switch (lower-case(local-name($storedQuestionnaire)))
        case 'questionnaire' return 'Questionnaire'
        default return upper-case(substring(name($storedQuestionnaire), 1, 1)) || substring(name($storedQuestionnaire), 2)
    
    let $objectType             := $decorlib:OBJECTTYPE-QUESTIONNAIRE
     
    let $check                  :=
        if (empty($storedQuestionnaire)) then (
            error($errors:BAD_REQUEST, $objectName || ' with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
        else
        if (count($storedQuestionnaire) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple results for id ' || $id || ' effectiveDate ' || $effectiveDate || '. Expected 1..1. Alert your administrator as this should not be possible.')
        )
        else
        if ($storedQuestionnaire[@statusCode = $utilq:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, $objectName || ' with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' cannot be updated while it one of status: ' || string-join($utilq:STATUSCODES-FINAL, ', '))
        else ()
    
    let $lock                   := utilq:checkQuestionnaireAccess($authmap, $decor, $id, $effectiveDate, true())

    let $fhirVersion            := ($fhirVersion, $storedQuestionnaire/@fhirVersion)[1]
    
    let $check                  :=
        if ($fhirVersion = $utilq:FHIROBJECT-FORMAT/enumeration/@value) then () else (
            $objectName || '.fhirVersion has unsupported value ' || string-join($fhirVersion, ', ') || '. SHALL be one of: ' || string-join($utilq:FHIROBJECT-FORMAT/enumeration/@value, ', ')
        )
    
    let $forOutput              := false()
    (: build map of oid names. is more costly when you do that deep down :)
    let $oidnamemap             := map {}
    let $preparedData           := utilq:handleQuestionnaire($data, $projectPrefix, $projectLanguage, $oidnamemap, (), false(), true(), false())
    let $check                  := ruleslib:checkDecorSchema($preparedData, 'update')
    
    let $check                  :=
        if (lower-case(local-name($storedQuestionnaire)) = lower-case(local-name($data))) then () else (
            error($errors:BAD_REQUEST, 'You cannot update a ' || $objectName || ' with a ' || local-name($data))
        )
    
    let $intention              := if ($storedQuestionnaire[@statusCode='final']) then 'patch' else 'version'
    let $history                := histlib:AddHistory($authmap?name, $objectType, $projectPrefix, $intention, $storedQuestionnaire)
    
    let $instant                := current-dateTime()
    let $now                    := substring(string($instant), 1, 19)
    let $locks                  := if ($deletelock) then decorlib:getLocks($authmap, $storedQuestionnaire, true()) else ()
    let $update                 := update replace $storedQuestionnaire with $preparedData
    let $update                 := update value $storedQuestionnaire/@lastModifiedDate with substring(string(current-dateTime()), 1, 19)
    
    let $delete                 := if ($locks) then update delete $locks else ()
    
    (: build map of oid names. is more costly when you do that deep down :)
    let $oidnamemap             := utilq:buildOidNameMap($data, $projectLanguage[1])
    let $preparedData           := utilq:handleQuestionnaire($data, $projectPrefix, $projectLanguage, $oidnamemap, (), false(), true(), true())
    
    return
        element {name($preparedData)} {
            if ($data/@uuid) then $data/@uuid else attribute uuid {util:uuid()},
            $preparedData/(@* except @uuid),
            $preparedData/*
        }
};


(: ===============                                               HELPER FUNCTIONS                                              ================:)

(:~ Input is questionnaire. Returns just the DECOR wrapper info, not the actual Q. 
 
  @param $listMode as boolean, skips any parts irrelevant to a tree of questionnaires hence adds just meta data
  @param $decorMode as boolean, skips uuid attribute
:)
declare function utilq:handleQuestionnaire($qObjects as element()*, $projectPrefix as xs:string?, $projectLanguage as xs:string?, $oidnamemap as map(*)?, $questionnaireResponses as element(f:QuestionnaireResponse)*, $listMode as xs:boolean, $decorMode as xs:boolean, $forOutput as xs:boolean) as element()* {
    
    for $q in $qObjects 
    let $isQQ                   := lower-case(local-name($q)) = 'questionnaire'
    let $isQR                   := lower-case(local-name($q)) = 'questionnaireresponse'
    let $iddisplay              := if ($q/@id) then map:get($oidnamemap, $q/@id) else ()
    let $projectLanguage        := if (empty($projectLanguage)) then ($q/ancestor::decor/project/@defaultLanguage, 'en-US')[1] else $projectLanguage
    let $canonicalUri           := if ($isQQ) then attribute canonicalUri {($q/@canonicalUri[. castable as xs:anyURI], utillib:getServerURLFhirCanonicalBase() || 'Questionnaire/'|| $q/@id || '--' || replace($q/@effectiveDate, '\D', ''))[1]} else ()
    let $fhirVersion            := if (local-name($q) = 'QuestionnaireResponse') then attribute fhirVersion {tokenize(util:document-name($q), '-')[2]} else ()
    (: empty $oidnamemap unless we are working on output, otherwise we add codeSystemName attributes to the db that the user did not anticipate :)
    let $oidnamemap             := if ($forOutput) then $oidnamemap else map {}
    return
        element {lower-case(local-name($q))}
        {
            (: =========== attributes =========== :)
            if ($decorMode) then () else if ($q/@uuid) then $q/@uuid else attribute uuid {util:uuid()},
            (:Not in use yet: Reference to a Questionnaire Wrapper @id. Questionnaire Wrappers SHALL carry either @id or @ref:)
            if ($q/@ref) then $q/@ref else attribute id {($q/@id, $q/f:id/@value)[1]},
            if ($q/@effectiveDate) then $q/@effectiveDate else (
                let $d  := ($q/f:authored/@value, $q/f:meta/f:lastUpdated/@value, string(current-dateTime()))[1]
                return attribute effectiveDate {if ($d castable as xs:date) then substring($d, 1, 10) || 'T00:00:00' else substring($d, 1, 19)}
            ),
            if ($q/@statusCode) then $q/@statusCode else if ($q/f:status/@value) then attribute statusCode {$q/f:status/@value} else (),
            $q/@versionLabel,
            $q/@expirationDate[. castable as xs:dateTime],
            $q/@officialReleaseDate[. castable as xs:dateTime],
            $canonicalUri,
            if ($isQR and $q/f:questionnaire/@value) then attribute questionnaire {$q/f:questionnaire/@value} else (), 
            if ($q/@lastModifiedDate) then $q/@lastModifiedDate else attribute lastModifiedDate {substring(($q/f:meta/f:lastUpdated/@value, string(current-dateTime()))[1], 1, 19)},
            if (string-length($iddisplay) = 0) then () else attribute iddisplay {$iddisplay},
            if ($q/ancestor::decor/project[@prefix = $projectPrefix]) then () else
            if (empty($q/ancestor::decor/project/@prefix)) then () else attribute ident {$q/ancestor::decor/project/@prefix},
            $fhirVersion,
            if ($isQR) then attribute link {xs:anyURI(utillib:getServerURLFhirServices() || $fhirVersion || '/public/QuestionnaireResponse/' || $q/f:id/@value[1])} else (), 
            $q/ancestor::decor/@versionDate,
            $q/ancestor::decor/@language,
            
            (:Identifying the Questionnaire as Enhanced (type is QE), thus edited, modified, enriched.:)
            if ($isQQ) then attribute enhanced {$q/@enhanced = 'true'} else (),
            (: =========== elements =========== :)
            if ($q/name) then $q/name else (
                <name language="{($projectLanguage, 'en-US')[1]}"> 
                {
                    'QR authored at ' || substring(($q/f:authored/@value, $q/f:meta/f:lastUpdated/@value, string(current-dateTime()))[1], 1, 10)
                }
                </name>
            ),
            if ($listMode) then () else $q/desc,
            $q/classification,
            if ($listMode) then $q/relationship else (
                for $relationship in $q/relationship
                let $object     :=
                    if ($forOutput) then 
                        let $rid    := $relationship/@ref
                        let $red    := $relationship/@flexibility
                        let $rtp    := $relationship/@type
                        return
                            switch ($rtp)
                            case 'ANSW' return utillib:getQuestionnaire($rid, $red)
                            case 'DRIV' return utillib:getTransaction($rid, $red)
                            default return ( (: we don't know this type (yet?). 
                                Assume that questionnaires could be a specialization/version etc of another questionnaire, and questionnairereponse could be that of another questionnairerepsonse :)
                                if ($isQQ) then utillib:getQuestionnaire($rid, $red) else utillib:getQuestionnaireResponse($rid, $red)
                            )
                     else ()       
               
                let $artifact   := decorlib:getObjectDecorType($object[1])
                return
                    <relationship>
                    {
                        $relationship/@type[not(. = '')],
                        $relationship/@ref[not(. = '')],
                        $relationship/@flexibility[not(. = '')]
                        ,
                        if ($object) then (
                            $object/ancestor::decor/project/@prefix,
                            attribute iStatusCode {$object/@statusCode}, 
                            if ($object/@expirationDate) then attribute iExpirationDate {$object/@expirationDate} else (),
                            if ($object/@versionLabel) then attribute iVersionLabel {$object/@versionLabel} else (),
                            let $iddisplay  := 
                                if (map:contains($oidnamemap, $relationship/@ref)) then map:get($oidnamemap, $relationship/@ref) else (
                                    utillib:getNameForOID($relationship/@ref, $projectLanguage, $object/ancestor::decor)
                                )
                            return attribute refdisplay {$iddisplay},
                            attribute artifact {$artifact},
                            (:attribute localInherit {$object/ancestor::decor/project/@prefix = $projectPrefix},:)
                            if ($projectLanguage = '*') then ($object/name) else (
                                <name language="{$projectLanguage}">{data(($object/name[@language = $projectLanguage], $object/name)[1])}</name>
                            )
                        ) else ()
                    }
                    </relationship>
            )
            ,
            if ($listMode) then () else 
            if ($isQQ) then (
                $q/subjectType,
                if ($q/publishingAuthority) then 
                    (for $childnode in $q/publishingAuthority return utillib:preparePublishingAuthorityForUpdate($childnode))
                    else if ($forOutput) then utillib:inheritPublishingAuthority($q) else (),
                for $childnode in $q/code return utillib:prepareValueCodingTypeForUpdate($childnode, $oidnamemap),
                if ($q/copyright) then $q/copyright else if ($forOutput) then utillib:inheritCopyright($q) else (),
                for $childnode in $q/jurisdiction return utillib:prepareValueCodingTypeForUpdate($childnode, $oidnamemap),
                $q/contained,
                utilq:handleQuestionnaireItem($q/item, $oidnamemap, ())
            )
            else ()
            ,
            if ($decorMode) then () else (
                (: connect questionnaireresponse. Give empty parameter $filteredQuestionnaireResponses to avoid processing any further :)
                for $qr in $questionnaireResponses[f:questionnaire/@value = $canonicalUri]
                return utilq:handleQuestionnaire($qr, $projectPrefix, $projectLanguage, $oidnamemap, (), $listMode, $decorMode, $forOutput)
            )
        }
};

declare %private function utilq:handleQuestionnaireItem($items as element(item)*, $oidnamemap as map(*)?, $storedItems as element(item)*) as element(item)* {
    utilq:handleQuestionnaireItem($items, $oidnamemap, $storedItems, (), ())
};

declare %private function utilq:handleQuestionnaireItem($items as element(item)*, $oidnamemap as map(*)?, $storedItems as element(item)*, $readOnlyInherit as xs:string?, $hiddenItemInherit as xs:string?) as element(item)* {
    
    for $item in $items
    let $storedItem             := $storedItems[@linkId = $item/@linkId]
        
    (: AD30-1697 Switch on readOnly: if group item readOnly is true all children should inherit this value :)
    let $readOnlyInherit        :=
        if (not(empty($readOnlyInherit)) or empty($storedItem)) then $readOnlyInherit 
        else if ($item/@type = "group" and $item/@readOnly = true()) then true() 
        else ()

    let $readOnly               :=
        if (not(empty($readOnlyInherit))) then $readOnlyInherit else $item/@readOnly
    
    (: AD30-1697 Switch on hiddenItem: if group item hiddenItem is true all children should inherit this value :)    
    let $hiddenItemInherit      :=
        if (not(empty($hiddenItemInherit)) or empty($storedItem)) then $hiddenItemInherit 
        else if ($item/@type = "group" and $item/@hiddenItem = true()) then true() 
        else ()
    let $hiddenItem             := if (not(empty($hiddenItemInherit))) then $hiddenItemInherit else $item/@hiddenItem    
    
    return 
    <item>
    {
        for $att in $item/(@id | @linkId | @definition | @prefix | @type | @enableBehavior | @required | @repeats | @minLength | @maxLength | @minValue | @maxValue | @maxDecimalPlaces | @renderingStyle)
        order by name($att)
        return if (normalize-space($att) = '') then () else attribute {name($att)} {replace($att, '(^\s+)|(\s+$)', '')}
        , 
        if (not(empty($readOnly))) then attribute readOnly {$readOnly} else attribute readOnly {false()},
        if (not(empty($hiddenItem))) then attribute hiddenItem {$hiddenItem} else attribute hiddenItem {false()},
        $item/text,
        $item/synonym,
        $item/operationalization,
        for $childnode in $item/code return utillib:prepareValueCodingTypeForUpdate($childnode, $oidnamemap),
        $item/designNote,
        for $childnode in $item/itemControl return utillib:prepareValueCodingTypeForUpdate($childnode, $oidnamemap),
        for $childnode in $item/unitOption return utillib:prepareValueCodingTypeForUpdate($childnode, $oidnamemap),
        for $childnode in $item/enableWhen return utilq:prepareQuestionnaireEnableWhenConditionTypeForUpdate($childnode, $oidnamemap),
        for $childnode in $item/answerValueSet return utilq:prepareQuestionnaireAnswerValueSetForUpdate($childnode, $oidnamemap),
        for $childnode in $item/answerOption return utilq:handleQuestionnaireItemAnswerOptionType($childnode, $oidnamemap),
        for $childnode in $item/initial return utilq:handleQuestionnaireItemInitialOptionType($childnode, $oidnamemap),
        $item/terminologyServer[@ref[not(. = '')]],
        for $childnode in $item/itemExtractionContext return utilq:handleQuestionnaireItemExtractionContext($childnode),
        utilq:handleQuestionnaireItem($item/item, $oidnamemap, (if (empty($storedItem)) then () else $storedItem/item), $readOnlyInherit, $hiddenItemInherit)
    }
    </item>
};

declare %private function utilq:prepareQuestionnaireEnableWhenConditionTypeForUpdate($item as element(enableWhen), $oidnamemap as map(*)?) as element(enableWhen)? {
    
    for $node in $item[exists(*[starts-with(name(), 'answer')])]
    return
    <enableWhen>
    {
        for $att in $item/(@id | @question | @operator)
        order by name($att)
        return if (normalize-space($att) = '') then () else attribute {name($att)} {replace($att, '(^\s+)|(\s+$)', '')}
        ,
        switch (name($node/*))
        case 'answerBoolean' return $node/answerBoolean[1]
        case 'answerDecimal' return $node/answerDecimal[1]
        case 'answerDate' return $node/answerInteger[1]
        case 'answerDateTime' return $node/answerDateTime[1]
        case 'answerTime' return $node/answerTime[1]
        case 'answerString' return $node/answerString[1]
        case 'answerCoding' return utillib:prepareValueCodingTypeForUpdate($node/answerCoding[1], $oidnamemap)
        case 'answerQuantity' return utillib:prepareValueQuantityTypeForUpdate($node/answerQuantity[1], $oidnamemap)
        default return ()
    }
    </enableWhen>
};

declare %private function utilq:prepareQuestionnaireAnswerValueSetForUpdate($item as element(answerValueSet), $oidnamemap as map(*)?) as element(answerValueSet)? {
    
    for $node in $item
    return
    <answerValueSet>
    {
        for $att in $item/@*
        return if (normalize-space($att) = '') then () else attribute {name($att)} {replace($att, '(^\s+)|(\s+$)', '')}
        ,
        if ($node/name) then $node/name else ()
    }
    </answerValueSet>
};

declare %private function utilq:handleQuestionnaireItemAnswerOptionType($item as element(answerOption), $oidnamemap as map(*)?) as element(answerOption)? {
    
    for $node in $item[valueInteger | valueDate | valueTime | valueString | valueCoding | valueReference]
    return
    <answerOption>
    {
        switch (name($node/*))
        case 'valueInteger' return $node/valueInteger[1]
        case 'valueDate' return $node/valueDate[1]
        case 'valueTime' return $node/valueTime[1]
        case 'valueString' return $node/valueString
        case 'valueCoding' return 
            (
            $node/@ordinal[not(. = '')],
            utillib:prepareValueCodingTypeForUpdate($node/valueCoding[1], $oidnamemap)
            )
        case 'valueReference' return $node/valueReference[1]
        default return ()        
    }
    </answerOption>
};

declare %private function utilq:handleQuestionnaireItemInitialOptionType($item as element(initial), $oidnamemap as map(*)?) as element(initial)? {
    
    for $node in $item[valueBoolean | valueDecimal | valueInteger | valueDate | valueDateTime | valueTime | valueString | valueUri | valueCoding | valueQuantity]
    return
    <initial>
    {
        switch (name($node/*))
        case 'valueBoolean' return $node/valueBoolean[1]
        case 'valueDecimal' return $node/valueDecimal[1]
        case 'valueInteger' return $node/valueInteger[1]
        case 'valueDateTime' return $node/valueDateTime
        case 'valueTime' return $node/valueTime[1]
        case 'valueString' return $node/valueString[1]
        case 'valueUri' return $node/valueUri[1]
        case 'valueCoding' return utillib:prepareValueCodingTypeForUpdate($node/valueCoding[1], $oidnamemap)
        case 'valueQuantity' return utillib:prepareValueQuantityTypeForUpdate($node/valueQuantity[1], $oidnamemap)
        default return ()      
    }
    </initial>
};

(: Establishes mapping context for a Questionnaire item :)
declare %private function utilq:handleQuestionnaireItemExtractionContext($item as element(itemExtractionContext)) as element(itemExtractionContext)? {
    
    for $node in $item[(@code | expression/@*)[not(normalize-space() = '')]]
    return
    <itemExtractionContext>
    {
        for $att in $node/@code
        return if (normalize-space($att) = '') then () else attribute {name($att)} {replace($att, '(^\s+)|(\s+$)', '')}
        ,
        for $childnode in $node/expression
        return
            <expression>
            {
                for $att in $childnode/(@description | @name | @language | @expression | @reference)
                order by name($att)
                return if (normalize-space($att) = '') then () else attribute {name($att)} {replace($att, '(^\s+)|(\s+$)', '')}
            }
            </expression>
    }
    </itemExtractionContext>
};


(:~ Build a map that has oid as key and human readable version as value :)
declare function utilq:buildOidNameMap($questionnaires as element()*, $language as xs:string?) as map(*)? {
    map:merge(
        for $oid in $questionnaires/@id | $questionnaires//@codeSystem[utillib:isOid(.)][empty(../@codeSystemName)]
        let $grp := $oid
        group by $grp
        return map:entry($oid[1], utillib:getNameForOID($oid[1], $language, $oid[1]/ancestor::decor))
    )
};

declare %private function utilq:checkQuestionnaireAccess($authmap as map(*), $decor as element(decor)) as element()? {
    utilq:checkQuestionnaireAccess($authmap, $decor, (), (), false(), false())
};

declare %private function utilq:checkQuestionnaireAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean) as element()? {
    utilq:checkQuestionnaireAccess($authmap, $decor, $id, $effectiveDate, $checkLock, false())
};

declare %private function utilq:checkQuestionnaireAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean, $breakLock as xs:boolean) as element()? {

    let $check              :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-RULES)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify questionnaires in project ', $decor/project/@prefix, '. You have to be an active author in the project.'))
        )

    let $lock               := if ($checkLock) then (
        let $lock           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
        let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breakLock)
        return
            if ($lock) then () else (
                error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this questionnaire (anymore). Get a lock first.'))
            )    
        )
        else ()

    return $lock
};

declare %private function utilq:checkQuestionnaireParameters($parameters as element(parameters), $questionnaire as element(questionnaire)*, $doQuestionnaire as xs:boolean) {
    
    let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )

    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    let $checkn                 := 
        if ($parameters/parameter[@path = '/name']) then
            if (count($questionnaire/name) - count($parameters/parameter[@op = 'remove'][@path = '/name']) + count($parameters/parameter[@op = ('add', 'replace')][@path = '/name']) ge 1) then () else (
                'A ' || name($questionnaire) || ' SHALL have at least one name. You cannot remove every name.'
            )
        else ()
    
    let $check                  :=
        for $param in $parameters/parameter
        let $op                 := $param/@op
        let $path               := $param/@path
        let $value              := $param/@value
        let $pathpart           := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($questionnaire, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($questionnaire/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss'
                )
            )
            case '/canonicalUri' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:anyURI) then (
                    let $otherQuestionnnaires := $setlib:colDecorData//questionnaire[@canonicalUri = $value]
                    return 
                    if ($otherQuestionnnaires except $questionnaire) then
                        'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be unique on the server, otherwise you cannot reference from a QuestionnaireReponse. Found in ' || string-join(distinct-values($otherQuestionnnaires/ancestor::decor/project/@prefix), ', ')
                    else ()
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be a URI'
                )
            )
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/fhirVersion' return (
                if ($param[@value = $utilq:FHIROBJECT-FORMAT/enumeration/@value]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. fhirVersion has unsupported value ' || string-join($param/@value, ', ') || '. SHALL be one of: ' || string-join($utilq:FHIROBJECT-FORMAT/enumeration/@value, ', ')
                )
            )
            case '/enhanced' return (
                if ($op = 'remove') then () else 
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false'
                )
            )
            case '/name'
            case '/desc'
            case '/copyright' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $pathpart], $op, $path)
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/classification' return (
                if ($param[count(value/classification) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/classification/tag[not(string(string-join(., '')) = '')]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Classification SHALL have at least one non-empty tag'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/relationship' return (
                if ($param[count(value/relationship) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/relationship[@type = $utilq:RELATIONSHIP-TYPES/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Relationship has unsupported .type value ' || string-join($param/value/relationship/@type, ', ') || '. SHALL be one of: ' || string-join($utilq:RELATIONSHIP-TYPES/enumeration/@value, ', ')
                        ),
                        if ($param/value/relationship[@ref]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship SHALL have ref property'
                        ),
                       
                        if ($param/value/relationship[@type = 'ANSW'] and name($questionnaire) = 'questionnaire') then
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship type ' || ($param/value/relationship/@type)[1] || ' is not compatible with a ' || name($questionnaire)
                        else ()
                   )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/experimental' return (
                if ($op = 'remove') then () else 
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($questionnaire)
                else
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false'
                )
            )
            case '/subjectType' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($questionnaire)
                else
                if ($param[count(value/subjectType) = 1]) then
                    if ($param[value/subjectType/@code = $utilq:FHIRRESOURCE-TYPE/enumeration/@value]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $pathpart || ' has unsupported value ' || string-join($param/@value, ', ') || '. SHALL be one of: ' || string-join($utilq:FHIRRESOURCE-TYPE/enumeration/@value, ', ')
                    )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/endorsingAuthority'
            case '/publishingAuthority' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($questionnaire)
                else if ($param[count(value/*[name() = $pathpart]) = 1]) then ruleslib:checkPublishingAuthority($param/value/publishingAuthority, $op, $path)                  
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart])
                )
            )
            case '/code'
            case '/jurisdiction'
            return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($questionnaire)
                else
                if ($param[count(value/*[name() = $pathpart]) = 1]) then
                    if ($op = 'remove') then () else (
                        if ($param/value/*[name() = $pathpart][@code[matches(., '^\S+$')]][utillib:isOid(@codeSystem)][empty(@canonicalUri) or @canonicalUri castable as xs:anyURI]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have code without whitespace, a codeSystem as valid OID, and optionally a canonicalUri as URI. Found: ' || string-join(for $att in $param/value/code/@* return name($att) || ': "' || $att || '" ', ' ')
                        )
                    )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart])
                )
            )
            case '/item'
            return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($questionnaire)
                else
                if ($param[count(value/item) ge 1]) then (
                    let $data     := utilq:handleQuestionnaireItem($param/value/item, (), $questionnaire/item)
                    let $data     := 
                        element {name($questionnaire)} {
                            $questionnaire/@*,
                            $questionnaire/(node() except item),
                            $data
                        }
                   return ruleslib:checkDecorSchema($data, $op)
                )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have one or more ' || $pathpart || ' under value containing exactly one object. Found ' || count($param/value/*[name() = $pathpart]/*)
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported.'
            )
     return
        if (empty($checkn) and empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join (($checkn, $check), '. '))
        )
};

declare %private function utilq:patchQuestionnaireParameters($parameters as element(parameters), $questionnaire as element(questionnaire)*) {

    for $param in $parameters/parameter
    return
        switch ($param/@path)
        case '/statusCode'
        case '/expirationDate'
        case '/officialReleaseDate'
        case '/versionLabel'
        case '/experimental'
        case '/fhirVersion'
        case '/enhanced' return (
            let $attname        := substring-after($param/@path, '/')
            let $new            := attribute {$attname} {$param/@value}
            let $stored         := $questionnaire/@*[name() = $attname]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/canonicalUri' return (
            let $attname        := substring-after($param/@path, '/')
            let $new            := attribute {$attname} {$param/@value}
            let $stored         := $questionnaire/@canonicalUri
            let $storedUri      := if ($stored) then string($stored) else utillib:getCanonicalUri('Questionnaire', $questionnaire)
            let $storedqr       := $setlib:colDecorData//f:QuestionnaireResponse/f:questionnaire/@value[. = $storedUri]
            
            let $updateQuestionnaire          :=
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            let $updateQuestionnaireResponses :=
                if ($storedqr) then update value $storedqr with $new else ()
            
            return
                ()
        )
        case '/name'
        case '/copyright'
        case '/desc'
        return (
            (: only one per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
            let $stored         := $questionnaire/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/classification' 
        return (
            (: only one :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := 
                if ($param/value/*[name() = $elmname]) then
                    <classification>
                    {
                        $param/value/*[name() = $elmname]/tag[not(string(string-join(., '')) = '')]
                    }
                    </classification>
                else ()
            let $stored         := $questionnaire/*[name() = $elmname]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/relationship' return (
            (: only one per template or model, insert before others :)
            let $new            := utillib:prepareDatasetRelationshipForUpdate($param/value/relationship)
            let $stored         := 
                if ($new[@type = 'ANSW'] | $new[@type = 'DRIV']) then $questionnaire/relationship[@type = $new/@type] else $questionnaire/relationship[@ref = $new/@ref]
            
            return
            switch ($param/@op)
            case 'add' (: fall through :)
            case 'replace' return if ($stored) then (update insert $new preceding $stored[1], update delete $stored) else (update insert $new into $questionnaire)
            case 'remove' return update delete $stored
            default return ( (: unknown op :) )
        )
        (: datatype code :)
        case '/subjectType' return (
            let $new            := <subjectType code="{$param/value/subjectType/@code}"/>
            let $stored         := $questionnaire/subjectType[@code = $new/@code]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/endorsingAuthority' return (
            (: only one possible :)
            let $new            := utillib:prepareEndorsingAuthorityForUpdate($param/value/endorsingAuthority)
            let $stored         := $questionnaire/endorsingAuthority
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/publishingAuthority' return (
            (: only one possible :)
            let $new            := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
            let $stored         := $questionnaire/publishingAuthority
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        (: datatype Coding :)
        case '/code'
        case '/jurisdiction'
        return (
            let $attname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareValueCodingTypeForUpdate($param/value/*[name() = $attname])
            let $stored         := $questionnaire/*[name() = $attname][@code = $new/@code][@codeSystem = $new/@codeSystem]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/item'
        return (
            let $new            := utilq:handleQuestionnaireItem($param/value/item, (), $questionnaire/item)
            let $stored         := update delete $questionnaire/item
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $questionnaire
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        default return ( (: unknown path :) )
};