xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:    
    enumeration:
      noAssociation              No terminology association defined
      noValuesetAssociated       No valueset associatiated with conceptlist
      noValuesetItem             No valueset item associated with concept in conceptlist
      noUcumUnit                       The unit of the value domain is not a valid UCUM essence unit
      conceptNotFound            Concept not found in codesystem
      codesystemNotfound         Codesystem not found on server
      valuesetNotFound           Valueset not found
      valueSetSizeNotConceptListSize   The linked value set is not of the same size as the concept list
      noMatchingDesignation      Display name does not match any designation in concept
      designationCaseMismatch    Case of display name does not match designation case in concept
      conceptRetired             Concept is retired
      conceptDraft               Concept is in draft
      conceptExperimental        Concept is experimental
      cannotParseExpression      Postcoordinated expression cannot be parsed
      equalSignNotAtStart        equal sign is not allowed at start of a postcoordinated code
      uriNotValidCode            URI is not a valid code
      errorInExpression          Error in Snomed expression
      notCoreModule                    Concept is not part of Snomed core module
      ok                         OK
      
   enumeration severity:
      info
      warning
      error
:)
(:~ Terminology Report API creates Terminology Reports for a Project (CADTS-aware) :)
module namespace trepapi            = "http://art-decor.org/ns/api/terminology/report";


import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace treplib     = "http://art-decor.org/ns/api/terminology/reportlib" at "library/terminology-report-lib.xqm";
import module namespace utillib      = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib    = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib      = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";

declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";


declare %private variable $trepapi:PROGRESS-REPORT-10                := 'Checking request parameters ...';
declare %private variable $trepapi:PROGRESS-DATASET-20               := 'Checking datasets ...';
declare %private variable $trepapi:PROGRESS-SCENARIO-40              := 'Checking scenarios';
declare %private variable $trepapi:PROGRESS-VALUESET-60              := 'Checking valuesets ...';




(:~ Get a terminology report (list) for a given project
    @param $projectPrefix     - required path part which may be its prefix or its oid
    @return as JSON
    @since 2022-11-30
:)
declare function trepapi:getTerminologyReport($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $specificref            := $request?parameters?reportId[string-length() gt 0]
    let $retrieveReport       := $request?parameters?retrieve = true()
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
        
    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-PROJECT)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to create a runtime version of project ', $project, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if (empty($specificref)) then () else if (matches($specificref, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter reportId SHALL be an oid or uuid. Found: ' || $specificref)
        )
    
    let $results                := trepapi:getTerminologyReport($projectPrefix,$specificref,$retrieveReport)
     
    return
        if (empty($results)) then 
            roaster:response(404, ())
        else (
            $results
        )
};

(:~ Get a terminology report (list) for a given project
    @param $project                 - required path part which may be its prefix or its oid
    @param $projectLanguage         - optional parameter to select from a specific language
    @param $format                  - optional. if not given it is html
    @param $download                - optional as xs:boolean. Default: false. 
    @return as html
    @since 2025-06-19
:)
declare function trepapi:getTerminologyReportView($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $reportId               := $request?parameters?reportId[string-length() gt 0]
    let $format                 := $request?parameters?format[not(. = '')]
    let $download               := $request?parameters?download=true()
    
    let $format                         :=
    if (not($format = 'list')) then 'html' else 'list'
    
(:    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()   :)
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
        
    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $terminologyReport      := $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix][@as = $reportId]
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
(:    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-PROJECT)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to create a runtime version of project ', $project, '. You have to be an active author in the project.'))
        ) :)
    let $check                  :=
        if (empty($reportId)) then () else if (matches($reportId, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter reportId SHALL be an oid or uuid. Found: ' || $reportId)
        )
    
    
    let $filename                       := 'TR_' || $projectPrefix || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html' 
    let $r-header                       := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())

    return
        if (empty($terminologyReport)) then (
            roaster:response(404, ())
        )
        else 
        
        (: prepare for Html :)
        let $referenceUrl               := $decor/project/reference[@url castable as xs:anyURI]/@url
                    
        return 
            if ($format = 'list') 
                then treplib:convertReportSimple2Html($terminologyReport, $projectLanguage, $referenceUrl, $download) 
                else treplib:convertReport2Html($terminologyReport, $projectLanguage, $referenceUrl, $download)

};


(:~ Create new terminology report and return the listing of terminology reports

@param $project - required. project prefix or id of the project to create the environment for
@return xml listing of terminology reports
:)
declare function trepapi:createTerminologyReportRequest($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    let $check                  :=
        if (empty($request?body)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
        
    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-PROJECT)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to create a runtime version of project ', $project, '. You have to be an active author in the project.'))
        )
    
    let $now                    := substring(xs:string(current-dateTime()), 1, 19)
    let $stamp                  := util:uuid()
    let $author                 := $decor/project/author[@username = $authmap?name]
    let $author                 := if (empty($author)) then $authmap?name else $author
    let $language               := $decor/project/@defaultLanguage
    let $timeStamp              := 'development'
    
    let $check                  :=
        if (count($projectPrefix) = 1) then () else (
            error($errors:BAD_REQUEST, 'Project parameter ' || $project || ' SHALL refer to exactly one project. Found: ' || count($decor))
        )
        
    
    let $terminology-report-request   :=   <terminology-report-request uuid="{$stamp}" for="{$projectPrefix}" on="{$now}" as="{$stamp}" by="{$author}" language="{$language}" progress="Added to process queue ..." progress-percentage="0"/>

    let $write                  := xmldb:store($setlib:strDecorScheduledTasks, $projectPrefix || replace($now, '\D', '') || '.xml', $terminology-report-request)
    let $tt                     := sm:chmod(xs:anyURI($write), 'rw-rw----')
    
    return
        trepapi:getTerminologyReport($projectPrefix, (),())
};

(:~ 
   Get list of terminology reports or specific report

   @param $project - required. project prefix or id of the project
   @return list of terminology reports or specific report
:)
declare function trepapi:getTerminologyReport($projectPrefix as xs:string*, $specificref as xs:string?, $retrieve as xs:boolean?) {
   if ($retrieve) then (
        let $fullReport := $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix][@as = $specificref]
      return
      $fullReport
      )
    else (
        let $reportrequestF        :=
            if (string-length($specificref) > 0)
            then
                collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix][@as = $specificref]
            else
                collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix]
        let $terminologyReportF              :=
            if (string-length($specificref) > 0)
            then
                $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix][@as = $specificref]
            else
                $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix]
    
        let $count                  := count($terminologyReportF | $reportrequestF)
        
        let $terminologyReportF              :=
            for $c in $terminologyReportF
            let $as   := $c/@as
            group by $as
            order by $c/@on descending
            return $c[1]
        
        let $results                :=
            <list artifact="TERMINOLOGYREPORT" current="{$count}" total="{$count}" all="{$count}" lastModifiedDate="{current-dateTime()}">
            {
                for $c in $reportrequestF
                order by $c/@on descending
                return
                    element {name($c)}
                    {
                        $c/@*
                    }
                ,
                for $c at $count in $terminologyReportF
                return
                    <terminologyReport>
                    {
                        $c/@* except $c/@status,
                        if ($count < 3) then $c/@status else attribute {'status'} {'retired'},
                        $c/filter
                    }
                    </terminologyReport>
            }
            </list>
    
        for $result in $results
        return
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
    )
};



(:~ Process terminology report requests :)
declare function trepapi:process-terminology-report-request($threadId as xs:string, $request as element(terminology-report-request)) {   
    let $xsltParameters     :=  <parameters></parameters>
    let $logsignature           := 'trepapi:process-terminology-report-request'
    let $mark-busy              := update insert attribute busy {'true'} into $request
    let $progress               := 
        if ($request/@progress) then (
            update value $request/@progress with $trepapi:PROGRESS-REPORT-10 
        )
        else (
            update insert attribute progress {$trepapi:PROGRESS-REPORT-10} into $request
        )
    let $progress               := 
        if ($request/@progress-percentage) then (
            update value $request/@progress-percentage with round((100 div 9) * 1) 
        )
        else (
            update insert attribute progress-percentage {round((100 div 9) * 1)} into $request
        )
    let $timeStamp              := current-dateTime()
    let $projectPrefix          := $request/@for[not(. = '*')]
    let $decor                  := utillib:getDecor($projectPrefix)
    let $language               := $request/@language[not(. = '')]
    let $reportCollection       := concat($setlib:strDecorVersion,'/',substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development')

    return
        if (count($decor) = 1) then (
            
            update value $request/@progress with $trepapi:PROGRESS-DATASET-20,
            update value $request/@progress-percentage with round((100 div 9) * 2),
            (: check if CADTS codesystems are available :)
          (:  let $codesystemCheck :=
                  for $codesystem in $decor/terminology/codeSystem
                  return
                  if (collection(concat($setlib:strCodesystemStableData,'/projects'))//browsableCodeSystem[@oid=$codesystem/@id][@effectiveDate=$codesystem/@effectiveDate]) then
                     'true'
                  else 'false'
            let $convert :=
                  if ($codesystemCheck='false') then
                     'test'
                  else() :)
            let $filteredDecor  := treplib:getFilteredDecor($decor)
            let $terminologyReport := 
                <terminologyReport  for="{$request/@for}" on="{$request/@on}" as="{$request/@as}" by="{$request/@by}" status="active">
                    <project>
                     {$decor/project/name}
                     </project>
                    {
                     $filteredDecor/filter
                     }
                    <datasets datasetCount="{count($filteredDecor/datasets/dataset)}">
                    {
                        treplib:traverseDatasets($filteredDecor)
                    }
                    </datasets>
                {
                    update value $request/@progress with $trepapi:PROGRESS-SCENARIO-40,
                    update value $request/@progress-percentage with round((100 div 9) * 4)
                }
                    <transactions transactionCount="{count($filteredDecor/scenarios/scenario/transaction/transaction[representingTemplate][@statusCode=('draft','final','pending')])}">
                    {
                        (: datasets may have versions and concepts may have versions. we need to use both id and effectiveDate to get to the dataset. within a dataset there can be only 1 version of a concept :)
                        for $transaction in $filteredDecor/scenarios/scenario/transaction/transaction[representingTemplate][@statusCode=('draft','final','pending')]
                        let $ds  := if ($transaction/representingTemplate/@sourceDataset) then utillib:getDataset($transaction/representingTemplate/@sourceDataset, $transaction/representingTemplate/@sourceDatasetFlexibility) else ()
                        return
                        <transaction json:array="true">
                        {
                            $transaction/@*,
                            $transaction/name,
                            <representingTemplate conceptCount="{count($transaction/representingTemplate//concept)}">
                            {
                                $transaction/representingTemplate/@*
                            }
                                <dataset>
                                {
                                    $ds/name
                                }
                                </dataset>
                            {
                                for $concept in $transaction/representingTemplate/concept
                                return
                                    for $association in $concept/terminologyAssociation
                                        let $sourceconcept := ($ds//concept[@id=$concept/@ref])[1]
                                        return
                                            if ($association/@conceptId=$concept/@ref) then
                                    (: check concept association :)
                                      <concept json:array="true">
                                            {
                                                $sourceconcept/@*,
                                                $sourceconcept/name,
                                                treplib:handleTerminologyAssociation($filteredDecor, $concept/terminologyAssociation)
                                            }
                                            </concept>
                                    else if ($association/@conceptId=$sourceconcept/conceptList/@id) then
                                       (: check conceptList association :)
                                       treplib:handleConceptListAssociation($filteredDecor,$association)
                                    else if ($association/@conceptId=$sourceconcept/conceptList/concept/@id) then
                                       (: check conceptList concept associaation :)
                                       treplib:handleTerminologyAssociation($filteredDecor,$association)
                                    else ()
                            
                            }
                            </representingTemplate>
                        }
                        </transaction>
                    }
                    </transactions>
                {
                    update value $request/@progress with $trepapi:PROGRESS-VALUESET-60,
                    update value $request/@progress-percentage with round((100 div 9) * 6)
                }
                    <valueSets valueSetCount="{count($filteredDecor/terminology/valueSet[@id])}">
                    {
                        treplib:traverseValuesets($filteredDecor)   
                    }
                    </valueSets>
                    <ids>
                    {
                        let $codeSystemIds := distinct-values($decor/terminology/terminologyAssociation/@codeSystem | $decor/terminology/valueSet/conceptList/concept/@codeSystem)
                        let $idCheck :=
                            for $codeSystemId in $codeSystemIds
                            let $id := $decor/ids/id[@root = $codeSystemId]
                            return
                                <id>
                                {
                                    if ($id) then ($id/@*, $id/*) else (
                                        <message severity="warning" type="idRefMissing" codeSystem="{$codeSystemId}"></message>
                                    )
                                }
                                </id>
                        return
                            $idCheck
                    }
                    </ids>
                </terminologyReport>
            return (
                if (not(xmldb:collection-available($reportCollection))) then
                    try {
                        let $ttt    := xmldb:create-collection($setlib:strDecorVersion, concat(substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development'))
                        let $tt     := try { sm:chmod(xs:anyURI($ttt), 'rwxrwsr-x') } catch * {()}
                        return $ttt
                    } 
                    catch * {
                        <error>+++ Create collection failed '{concat(substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development')}': {$err:code}: {$err:description}</error>
                    }
                else()
                ,
                xmldb:store($reportCollection,concat($projectPrefix,'terminology-report-',$request/@as,'.xml'),transform:transform($terminologyReport ,xs:anyURI(concat('xmldb:exist://',$setlib:strApi ,'/resources/stylesheets/add-metadata-to-terminologyReport.xsl')), $xsltParameters))
                ,
                xmldb:remove(util:collection-name($request), util:document-name($request))
            )                
        )
        else (
            let $message                := 'ERROR. Found ' || count($decor) || ' projects for prefix ' || $projectPrefix || '. Expected: 1. Compile id: ' || $request/@as
            let $log                    := utillib:log-scheduler-event('error', $threadId, $logsignature, $message)
            let $progress               := update value $request/@progress with $message
            
            return ()
        )
    

};


(:~ Deletes terminology report

@param $project - required. project prefix or id of the project
@param $specificref - required. reference to the runtime environmnent
@return xml listing of environments
:)
declare function trepapi:deleteTerminologyReport($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $specificref            := $request?parameters?reportId[string-length() gt 0]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    let $check                  :=
        if (matches($specificref, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter compilationId SHALL be an oid or uuid. Found: ' || $specificref)
        )
    
    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($projectPrefix) = 1) then () else (
            error($errors:BAD_REQUEST, 'Project parameter ' || $project || ' SHALL refer to exactly one project. Found: ' || count($decor))
        )
    
    let $part1                  := concat('xmldb:exist://', $setlib:strDecorVersion, '/', substring($projectPrefix, 1, string-length($projectPrefix) - 1), '/development/')
    let $tmp1                   := 
        if (xmldb:collection-available($part1)) then (
            if (doc-available(concat($part1, $projectPrefix,'terminology-report-', $specificref, '.xml'))) then xmldb:remove($part1, concat($projectPrefix,'terminology-report-', $specificref, '.xml')) else ()
        )
        else () 
    
    let $part1                  := collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix][@as = $specificref]
    let $tmp3                   := if ($part1) then xmldb:remove(util:collection-name($part1), util:document-name($part1)) else ()
    
    return
        roaster:response(204, ())
};


(:~ Validate a given code against a codesystem
    @param $projectPrefix     - required path part which may be its prefix or its oid
    @return as JSON
    @since 2023-06-27
:)
declare function trepapi:validateCode($request as map(*)) {
    
    let $data                 := utillib:getBodyAsXml($request?body, (), ())
    let $codeSystemId         :=  $data/@codeSystemId
    let $code                 := $data/@code
    let $displayName          := $data/@displayName
   
    let $check                  :=
        if (empty($codeSystemId)) then () else if (matches($codeSystemId, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter codeSystemId SHALL be an oid or uuid. Found: ' || $codeSystemId)
        )
        
    let $check                  :=
        if (empty($code)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter code')
        else ()

    let $concept :=
         <concept code="{$code}" codeSystem="{$codeSystemId}" displayName="{$displayName}">
            {
            if (starts-with($code,'http')) then
               <message severity="error" type="uriNotValidCode" json:array="true">URI is not a valid code</message>
            else
            (: handle SNOMED post coordinated codes :)
                if ($codeSystemId='2.16.840.1.113883.6.96') then
                    if ($code castable as xs:integer) then
                        let $snomedConcept := collection($setlib:strCodesystemStableData)//concept[@code=$code][ancestor::browsableCodeSystem/@oid=$codeSystemId]
                        return
                        if ($snomedConcept) then
                            treplib:checkValueSetConcept($displayName,$snomedConcept)
                        else
                            (: check if code system is present :)
                            if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$codeSystemId) then
                                <message severity="error" type="conceptNotFound" json:array="true"><codeSystem oid="{$codeSystemId}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$codeSystemId]/name}</codeSystem></message>
                            else
                            <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
                    else 
                        treplib:checkSnomedExpression($code,$displayName)
                else
                     (: find code system first, local or cadts :)
                     if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$codeSystemId) then
                        let $codeSystemConcept := collection($setlib:strCodesystemStableData)//concept[@code=$code][ancestor::browsableCodeSystem/@oid=$codeSystemId]
                        return
                        if ($codeSystemConcept) then
                           treplib:checkValueSetConcept($displayName,$codeSystemConcept)
                        else
                           <message severity="warning" type="conceptNotFound" json:array="true"><codeSystem oid="{$codeSystemId}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$codeSystemId]/name}</codeSystem></message>
                     else if (collection($setlib:strDecorData)//codeSystem[@id=$codeSystemId]) then
                        let $localCodeSystemConcept := collection($setlib:strDecorData)//codedConcept[@code=$code]
                        return
                        if ($localCodeSystemConcept) then
                           let $localConcept :=
                              <concept>
                                 {
                                 $localCodeSystemConcept/@*,
                                 for $designation in $localCodeSystemConcept/designation
                                 return
                                 <designation lang="{$designation/@language}" type="{if ($designation/@type='preferred') then 'pref' else 'syn'}">{$designation/@displayName/string()}</designation>
                                 }
                              </concept>
                           return
                           treplib:checkValueSetConcept($displayName,$localConcept)
                        else
                           <message severity="error" type="conceptNotFound" json:array="true"><codeSystem oid="{$codeSystemId}">{collection($setlib:strDecorData)//codeSystem[@id=$codeSystemId]/@displayName}</codeSystem></message>
                     else
                        <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
            }
         </concept>
     
    return
        if (empty($concept)) then 
            roaster:response(404, ())
        else (
            $concept
        )
};

(:~ Report server wide usage of a codesystem concept in terminology associations, valuesets and templates
    @param  conceptCode          - required, code of the concept
    @param  codeSystemId         - required, id (OID) of the code system
    @param  projectprefix        - optional, limit report to a single project
    @param  governancegroupId    - optional, limit report to a specific governance group
    @return as JSON or XML
    @since 2025-10-10
:)
declare function trepapi:getConceptUsageReport($request as map(*)) {
   let $codeSystemId          := $request?parameters?codeSystemId
   let $codeToCeck            := $request?parameters?conceptId
   let $language              := $request?parameters?language
   let $governanceGroupId     := $request?parameters?governanceGroupId[string-length() gt 0]
   let $project               := $request?parameters?project[string-length() gt 0]

   let $check                 :=
     if (not(empty($governanceGroupId)) and not(empty($project))) then 
         error($errors:BAD_REQUEST, 'Request SHALL have parameter governanceGroupId or prefix, not both')
     else ()

   let $codeSystemConcepts    := collection($setlib:strCodesystemStableData)//concept[@code=$codeToCeck]
   let $codeSystemConcept     := $codeSystemConcepts[ancestor::browsableCodeSystem/@oid=$codeSystemId]
   
   let $decorCollection       := 
                                 if (empty($project) and empty($governanceGroupId)) then
                                   collection(concat($setlib:strDecorData,'/projects'))
                                 else if (empty($governanceGroupId) and not(empty($project))) then
                                    let $decor  := if (utillib:isOid($project)) then utillib:getDecorById($project) else utillib:getDecorByPrefix($project)
                                    return
                                    collection(util:collection-name($decor))
                                 else
                                    for $projectRef in collection($setlib:strDecorData)//governance-group-links[partOf/@ref=$governanceGroupId]/@ref/string()
                                    let $projectCollection := util:collection-name(collection($setlib:strDecorData)//project[@id=$projectRef])
                                    return
                                    collection($projectCollection)

   let $associations          := $decorCollection//terminologyAssociation[@codeSystem=$codeSystemId][@code=$codeToCeck]
   let $valueSetConcepts      := $decorCollection//terminology/valueSet/conceptList/concept[@codeSystem=$codeSystemId][@code=$codeToCeck]
   let $valueSetExceptions    := $decorCollection//terminology/valueSet/conceptList/exception[@codeSystem=$codeSystemId][@code=$codeToCeck]
   let $valueSetIncludes      := $decorCollection//terminology/valueSet/conceptList/include[@codeSystem=$codeSystemId][@code=$codeToCeck]
   let $valueSetExcludes      := $decorCollection//terminology/valueSet/conceptList/exclude[@codeSystem=$codeSystemId][@code=$codeToCeck]
   let $templateVocabulary    := $decorCollection//rules//vocabulary[@codeSystem=$codeSystemId][@code=$codeToCeck]
   
   let $designationUse :=
         if ($codeSystemId='2.16.840.1.113883.6.96') then
            'fsn'
         else if ($codeSystemId='2.16.840.1.113883.6.1') then
            'lfsn'
         else 'pref'
         
   let $resultAssociations :=
      for $association in $associations
            
            (: possible multiple versions with certain id :)
            let $concepts := 
 for $concept in $association
            let $project := $association/ancestor::decor
                              let $cpt := 
                                       if ($concept/@conceptFlexibility) then
                                          $project//concept[@id=$concept/@conceptId][@effectiveDate=$concept/@conceptFlexibility]
                                       else
                                          if ($project//concept[@id=$concept/@conceptId]/parent::conceptList) then
                              $project//concept[@id=$concept/@conceptId]/parent::conceptList/parent::valueDomain/parent::concept
                                 else
                                    $project//concept[@id=$concept/@conceptId]
                     return
                              $cpt[@statusCode=('draft','final')]
            
            let $inheritConcepts := 
                     for $concept in $association
                     let $project := $association/ancestor::decor
                                       let $cpt := 
                                 if ($concept/@conceptFlexibility) then
                                    $project//concept[inherit/@ref=$concept/@conceptId][inherit/@effectiveDate=$concept/@conceptFlexibility]
                                 else
                                    $project//concept[inherit/@ref=$concept/@conceptId]
                     return
                     $cpt[@statusCode=('draft','final')]
            
            return
                  for $cpt in  $concepts | $inheritConcepts
                  let $dataset := $cpt/ancestor::dataset
                  let $project := $association/ancestor::decor
                  let $projectName := if (string-length($project/project/name[@language=$language]) gt 0) then $project/project/name[@language=$language] else $project/project/name[1]
                  let $datasetName := if (string-length($dataset/name[@language=$language]) gt 0) then $dataset/name[@language=$language] else $dataset/name[1]
                  let $version := if (string-length($dataset/@versionLabel) gt 0) then concat(' (',$dataset/@versionLabel/string(),')') else ''
                  return
                  <concept id="{$cpt/@id}" effectiveDate="{$cpt/@effectiveDate}" statusCode="{$cpt/@statusCode}" dataset="{concat($datasetName,$version)}" datasetId="{$dataset/@id}" datasetEffectiveDate="{$dataset/@effectiveDate}" project="{$projectName}" prefix="{$project/project/@prefix}" json:array="true">
                  {
                  if ($cpt/inherit or $cpt/contains) then
                  let $original := utillib:getOriginalForConcept($cpt)
                  let $ref      := if ($cpt/inherit) then 'inherit' else 'contains'
                  return
                     (
                        attribute reference {$ref},
                        if ($original/name[@language=$language]) then $original/name[@language=$language] else $original/name[1]
                     )
                  else
                     if ($cpt/name[@language=$language]) then $cpt/name[@language=$language] else $cpt/name[1]
                  }
                  </concept>
               
         let $resultValueSets :=
      for $valueSetConcept in $valueSetConcepts | $valueSetExceptions | $valueSetIncludes | $valueSetExcludes
            let $valueSet := $valueSetConcept/ancestor::valueSet
            let $project := $valueSetConcept/ancestor::decor
            let $projectName := if (string-length($project/project/name[@language=$language]) gt 0) then $project/project/name[@language=$language] else $project/project/name[1]
            let $type := name($valueSetConcept)
            return
            if ($valueSet/@statusCode=('draft','pending','final')) then
               <valueSet project="{$projectName}" prefix="{$project/project/@prefix}" type="{$type}" json:array="true">
               {
                  $valueSet/@*
               }
               </valueSet>
            else ()
         
   let $resultTemplates :=
      for $vocab in $templateVocabulary
            let $template := $vocab/ancestor::template
            let $project := $vocab/ancestor::decor
            let $projectName := if (string-length($project/project/name[@language=$language]) gt 0) then $project/project/name[@language=$language] else $project/project/name[1]
            return
            if ($template/@statusCode=('draft','pending','review','active')) then
               <template project="{$projectName}" prefix="{$project/project/@prefix}" json:array="true">
                  {
                  $template/@*,
                  if ($template/desc[@language=$language]) then $template/desc[@language=$language] else $template/desc[1]
                  }
               </template>
            else ()
         
   
   let $response := 
                  <result>
                  <concept>
                     {
                     $codeSystemConcept/@*,
                     if ($codeSystemConcept/designation[@use=$designationUse]) then
                        $codeSystemConcept/designation[@use=$designationUse]
                     else
                        $codeSystemConcept/designation
                     }
                  </concept>
                  <associations count="{count($resultAssociations)}" projectCount="{count(distinct-values($resultAssociations/@prefix))}">
                     {
                     $resultAssociations
                     }
                  </associations>
                  <valueSets count="{count($resultValueSets)}" projectCount="{count(distinct-values($resultValueSets/@prefix))}">
                     {
                     $resultValueSets
                     }
                  </valueSets>
                  <templates count="{count($resultTemplates)}" projectCount="{count(distinct-values($resultTemplates/@prefix))}">
                     {
                  $resultTemplates
                     }
                  </templates>
                  </result>
   
   return
   $response
};


(:~ Get a codesystem usage report (list) for a given project
    @param $project                 - required path part which may be its prefix or its oid
    @param $projectLanguage         - optional parameter to select from a specific language
    @param $format                  - optional. if not given it is html
    @param $download                - optional as xs:boolean. Default: false. 
    @return as html
    @since 2025-06-19
:)
declare function trepapi:getCodeSystemUsageReportView($request as map(*)) {
    
    let $authmap                := $request?user
    let $language               := $request?parameters?language[not(. = '')]
    let $codeSystemId           := $request?parameters?codeSystemId[string-length() gt 0]
    let $reportId               := $request?parameters?reportId[string-length() gt 0]
    let $format                 := $request?parameters?format[not(. = '')]
    let $download               := $request?parameters?download=true()
    
    let $format                         :=
    if (not($format = 'list')) then 'html' else 'list'
    
(:    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()   :)

    
    let $codeSystemUsageReport      := collection($setlib:strDecorTemp)//codeSystemUsageReport[@codeSystem = $codeSystemId][@as = $reportId]
    

    let $check                  :=
        if (empty($reportId)) then () else if (matches($reportId, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter reportId SHALL be an oid or uuid. Found: ' || $reportId)
        )
    
    
    let $filename                       := 'CSR_' || $codeSystemId || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html' 
    let $r-header                       := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())

    return
        if (empty($codeSystemUsageReport)) then (
            roaster:response(404, ())
        )
        else                   
            treplib:convertCodeSystemUsageReport2Html($codeSystemUsageReport, $language, $download)

};


(:~ Create new codesystem usage report and return the listing of usage reports

@param $codeSystemId - required. code system id (oid) of the code system
@return xml listing of code system usage reports
:)
declare function trepapi:createCodeSystemReportRequest($request as map(*)) {
    
    let $authmap                := $request?user
    
    let $data                 := utillib:getBodyAsXml($request?body, (), ())
    let $codeSystemId           := $request?parameters?codeSystemId
    let $language              := $data/@language
    let $codeSystemName       := utillib:getNameForOID($codeSystemId,$language,())
    let $governanceGroupId     := $data/@governanceGroupId[not(. = '')]
    let $project               := $data/@project[not(. = '')]
    let $ancestor               := $data/@ancestor[not(. = '')]
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($codeSystemId)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter codeSytemId')
        else ()
    let $check                  :=
        if (empty($language)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter language')
        else ()
    let $check                 :=
        if (not(empty($governanceGroupId)) and not(empty($project))) then 
            error($errors:BAD_REQUEST, 'Request SHALL have parameter governanceGroupId or prefix, not both')
        else ()
     
    let $check                  :=
        if (empty($request?body)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()

    let $check                  :=
          if (not($authmap?groups = ('dba','terminology'))) then (
            error($errors:FORBIDDEN, 'User ' || $authmap?name || ' does not have sufficient permissions.')
          )
          else()
    
    let $ancestorName :=
         if ($ancestor) then
            let $concept := collection($setlib:strCodesystemStableData)//concept[@code=$ancestor][ancestor::browsableCodeSystem/@oid=$codeSystemId]
            return
            if ($concept/designation[@lang=$language]) then
               if ($concept/designation[@lang=$language][@use='fsn']) then
                  $concept/designation[@lang=$language][@use='fsn']/text()
               else
                  $concept/designation[@lang=$language][@use='pref']/text()
            else 
               if ($concept/designation[@use='fsn']) then
                  $concept/designation[@use='fsn'][1]/text()
               else
                  $concept/designation[@use='pref'][1]/text()
        else ()
        
    let $governanceGroupName  :=
         if ($governanceGroupId) then
            let $govGroup := collection($setlib:strArtData)//group[@id=$governanceGroupId]
            return
            if ($govGroup/name[@language=$language]) then
               $govGroup/name[@language=$language]/text()
            else
               $govGroup/name[1]/text()
         else ()
         
    let $projectName          :=
         if ($project) then
            let $decor  := if (utillib:isOid($project)) then utillib:getDecorById($project) else utillib:getDecorByPrefix($project)
            return
            if ($decor/project/name[@language=$language]) then
               $decor/project/name[@language=$language]/text()
            else
               $decor/project/name[1]/text()
         else ()
    
    let $now                    := substring(xs:string(current-dateTime()), 1, 19)
    let $stamp                  := util:uuid()
        
    
    let $codesystem-report-request   :=   <codesystem-report-request uuid="{$stamp}" codeSystem="{$codeSystemId}" codeSystemName="{$codeSystemName}" governanceGroup="{$governanceGroupId}" governanceGroupName="{$governanceGroupName}" project="{$project}" projectName="{$projectName}" ancestor="{$ancestor}" ancestorName="{$ancestorName}" on="{$now}" as="{$stamp}" by="{$authmap?name}" language="{$language}" progress="Added to process queue ..." progress-percentage="0"/>

    let $write                  := xmldb:store($setlib:strDecorScheduledTasks, $codeSystemId || '-' || replace($now, '\D', '') || '.xml', $codesystem-report-request)
    let $tt                     := sm:chmod(xs:anyURI($write), 'rw-rw----')
    
    return
        trepapi:getCodeSystemReport($codeSystemId, (),())
};


(:~    
   Get list of codesystem reports or specific report
   @param $codeSystemId - required. code system id (oid) of the code system
   @return list of codesystem reports or specific report
:)
declare function trepapi:getCodeSystemReport($request as map(*)) {
    
    let $authmap                := $request?user
    let $codeSystemId           := $request?parameters?codeSystemId
    let $specificref            := $request?parameters?reportId[string-length() gt 0]
    let $retrieveReport       := $request?parameters?retrieve = true()
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($codeSystemId)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter codeSytemId')
        else ()
        
    let $check                  :=
          if (not($authmap?groups = ('dba','terminology'))) then (
            error($errors:FORBIDDEN, 'User ' || $authmap?name || ' does not have sufficient permissions.')
          )
          else()

    let $check                  :=
        if (empty($specificref)) then () else if (matches($specificref, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter reportId SHALL be an oid or uuid. Found: ' || $specificref)
        )
    
    let $results                := trepapi:getCodeSystemReport($codeSystemId,$specificref,$retrieveReport)
     
    return
        if (empty($results)) then 
            roaster:response(404, ())
        else (
            $results
        )
};

(:~ 
   Get list of codesystem usage reports or specific report
   @param $codeSystemId - required. code system id (oid) of the code system
   @return list of codesystem reports or specific report
:)
declare function trepapi:getCodeSystemReport($codeSystemId as xs:string*, $specificref as xs:string?, $retrieve as xs:boolean?) {
   if ($retrieve) then (
        let $fullReport := collection($setlib:strDecorTemp)//codeSystemUsageReport[@codeSystem = $codeSystemId][@as = $specificref]
      return
      $fullReport
      )
    else (
        let $reportrequestF        :=
            if (string-length($specificref) > 0)
            then
                collection($setlib:strDecorScheduledTasks)/codesystem-report-request[@codeSystem = $codeSystemId][@as = $specificref]
            else
                collection($setlib:strDecorScheduledTasks)/codesystem-report-request[@codeSystem = $codeSystemId]
        let $codeSystemReportF              :=
            if (string-length($specificref) > 0)
            then
                collection($setlib:strDecorTemp)//codeSystemUsageReport[@codeSystem = $codeSystemId][@as = $specificref]
            else
                collection($setlib:strDecorTemp)//codeSystemUsageReport[@codeSystem= $codeSystemId]
    
        let $count                  := count($codeSystemReportF | $reportrequestF)
        
        let $codeSystemReportF              :=
            for $c in $codeSystemReportF
            let $as   := $c/@as
            group by $as
            order by $c/@on descending
            return $c[1]
        
        let $results                :=
            <list artifact="CODESYTEMREPORT" current="{$count}" total="{$count}" all="{$count}" lastModifiedDate="{current-dateTime()}">
            {
                for $c in $reportrequestF
                order by $c/@on descending
                return
                    element {name($c)}
                    {
                        $c/@*
                    }
                ,
                for $c at $count in $codeSystemReportF
                return
                    <codeSystemUsageReport>
                    {
                        $c/@*,
                        $c/filter
                    }
                    </codeSystemUsageReport>
            }
            </list>
    
        for $result in $results
        return
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
    )
};


(:~ Deletes codesystem usage report

@param $codeSystemId - required. id (oid) of the codesystem 
@param $specificref - required. reference to the runtime environmnent
@return xml listing of environments
:)
declare function trepapi:deleteCodeSystemUsageReport($request as map(*)) {
    
    let $authmap                := $request?user
    let $codeSystemId           := $request?parameters?codeSystemId[string-length() gt 0]
    let $specificref            := $request?parameters?reportId[string-length() gt 0]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (matches($specificref, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter compilationId SHALL be an oid or uuid. Found: ' || $specificref)
        )


    
    let $part1                  := collection($setlib:strDecorTemp)/codeSystemUsageReport[@codeSystem = $codeSystemId][@as = $specificref]
    let $tmp1                   := if ($part1) then xmldb:remove(util:collection-name($part1), util:document-name($part1)) else ()
    
    let $part1                  := collection($setlib:strDecorScheduledTasks)/codesystem-report-request[@codeSystem = $codeSystemId][@as = $specificref]
    let $tmp3                   := if ($part1) then xmldb:remove(util:collection-name($part1), util:document-name($part1)) else ()
    
    return
        roaster:response(204, ())
};



(:~ Process codesystem report requests :)
declare function trepapi:process-codesystem-report-request($threadId as xs:string, $request as element(codesystem-report-request)) {   
    let $logsignature           := 'trepapi:process-codesystem-report-request'
    let $mark-busy              := update insert attribute busy {'true'} into $request
    let $progress               := 
        if ($request/@progress) then (
            update value $request/@progress with $trepapi:PROGRESS-REPORT-10 
        )
        else (
            update insert attribute progress {$trepapi:PROGRESS-REPORT-10} into $request
        )
    let $progress               := 
        if ($request/@progress-percentage) then (
            update value $request/@progress-percentage with round((100 div 9) * 1) 
        )
        else (
            update insert attribute progress-percentage {round((100 div 9) * 1)} into $request
        )
    let $timeStamp              := current-dateTime()
    let $codeSystemId            := $request/@codeSystem
    let $language               := $request/@language[not(. = '')]
    let $governanceGroupId     := $request/@governanceGroup[not(. = '')]
    let $project               := $request/@project[not(. = '')]
    let $ancestor               := $request/@ancestor[not(. = '')]

    let $codeSystem              := collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$codeSystemId]
   
    let $toplevels                :=
    if ($codeSystemId='2.16.840.1.113883.6.96') then
                           ('373873005', '260787004', '78621006', '272379006', '419891008', '254291000', '404684003', '362981000', '123037004', '123038009', '308916002', '410607006', '243796009', '900000000000441003', '48176007', '370115009', '105590001', '71388002', '363787002')
                        else if ($codeSystemId='2.16.840.1.113883.6.1') then
                           ('LP29695-1','LP7787-7','LP29693-6','LP29696-9')
                        else ()
                        
   let $decorCollection       := 
                                 if (empty($project) and empty($governanceGroupId)) then
                                   collection(concat($setlib:strDecorData,'/projects'))
                                 else if (empty($governanceGroupId) and not(empty($project))) then
                                    let $decor  := if (utillib:isOid($project)) then utillib:getDecorById($project) else utillib:getDecorByPrefix($project)
                                    return
                                    collection(util:collection-name($decor))
                                 else
                                    for $projectRef in collection($setlib:strDecorData)//governance-group-links[partOf/@ref=$governanceGroupId]/@ref/string()
                                    let $projectCollection := util:collection-name(collection($setlib:strDecorData)//project[@id=$projectRef])
                                    return
                                    collection($projectCollection)
                                    
   (: ********** Terminology Associations ********** :)
    let $associations          := $decorCollection//terminologyAssociation[@codeSystem=$codeSystemId]
   
   let $progressUpdate        :=
                                 (
                                    update value $request/@progress with 'Checking terminology associations...',
                                    update value $request/@progress-percentage with round((100 div 9) * 2)
                                 )

   let $distinctAssociationCodes := if ($codeSystemId='2.16.840.1.113883.6.96') then
                                       distinct-values($associations/@code[. castable as xs:integer] )
   else
   distinct-values($associations/@code)
                                       
   let $postcoAssociationCodes   :=  if ($codeSystemId='2.16.840.1.113883.6.96') then
                                       distinct-values($associations/@code[not(. castable as xs:integer)] )
                                     else
                                       ()                                 
   
   let $associationResult     :=
                              for $code in $distinctAssociationCodes
                                 let $codeSystemConcepts    := collection($setlib:strCodesystemStableData)//concept[@code=$code]
                                 let $codeSystemConcept     := $codeSystemConcepts[ancestor::browsableCodeSystem/@oid=$codeSystemId]
                                 let $designations          := $codeSystemConcept/designation
                                 let $ancslf := $codeSystemConcept/ancSlf
                                 let $association := $associations[@code=$code]
                                 (: possible multiple versions with certain id :)
                                 let $concepts := 
                                                   for $concept in $association
                                                   let $project := $association/ancestor::decor
                                                   let $cpt := 
                                                            if ($concept/@conceptFlexibility) then
                                                               $project//concept[@id=$concept/@conceptId][@effectiveDate=$concept/@conceptFlexibility]
                                                            else
                                                               $project//concept[@id=$concept/@conceptId]
                                                   return
    $cpt[@statusCode=('draft','final')]
                                 let $inheritConcepts := 
                                          for $concept in $association
                                          let $project := $association/ancestor::decor
                                          let $cpt := 
                                                      if ($concept/@conceptFlexibility) then
                                                         $project//concept[inherit/@ref=$concept/@conceptId][inherit/@effectiveDate=$concept/@conceptFlexibility]
                                                      else
                                                         $project//concept[inherit/@ref=$concept/@conceptId]
                                          return
                                          $cpt[@statusCode=('draft','final')]
 return
                                 if (empty($ancestor) or $ancestor=$codeSystemConcept/ancestor) then
                                    <association json:array="true">
                                       {
                                       attribute conceptCount {count($concepts | $inheritConcepts)},
                                          if ($designations[@lang=$language]) then
                                          ()
                                       else
                                          attribute noLanguage {'true'}
                                       ,
                                       if ($codeSystemId= ('2.16.840.1.113883.6.96','2.16.840.1.113883.6.1')) then
                                          attribute toplevel {$ancslf[.=$toplevels]}
                                       else (),
                                       if ($codeSystemConcept) then
                                             <codeSystemConcept>
                                             {
                                             $codeSystemConcept/@*,
                                       if ($designations[@lang=$language]) then
                                          if ($designations[@lang=$language][@use='fsn']) then
                                       $designations[@lang=$language][@use='fsn']
                                             else
                                          $designations[@lang=$language][@use='pref']
                                       else
                                             if ($designations[@use='fsn']) then
                                                $designations[@use='fsn'][1]
                                             else
                                                $designations[@use='pref'][1]
                                          }  
                                          </codeSystemConcept>
                                          else
                                             <codeSystemConcept                                             notFound="" code="{$code}">                                    <designation>{$treplib:docHtmlNames[@key='not-found']/text[@language=$language]/text()}</designation>                                             </codeSystemConcept>
                                    ,
                                       for $cpt in  $concepts | $inheritConcepts
                                       let $dataset := $cpt/ancestor::dataset
                                       let $project := $cpt/ancestor::decor
                                       let $projectName := if ($project/project/name[@language=$language][string-length() gt 0]) then $project/project/name[@language=$language] else $project/project/name[1][string-length() gt 0]
                                       let $datasetName := if ($dataset/name[@language=$language][string-length() gt 0]) then $dataset/name[@language=$language] else $dataset/name[1][string-length() gt 0]
      let $version := if (string-length($dataset/@versionLabel) gt 0) then concat(' (',$dataset/@versionLabel/string(),')') else ''
      return
                                       <concept id="{$cpt/@id}" effectiveDate="{$cpt/@effectiveDate}" statusCode="{$cpt/@statusCode}" dataset="{concat($datasetName,$version)}" datasetId="{$dataset/@id}" datasetEffectiveDate="{$dataset/@effectiveDate}" project="{$projectName}" prefix="{$project/project/@prefix}" json:array="true">
                                       {
            if ($cpt/inherit or $cpt/contains) then
                                       let $original := utillib:getOriginalForConcept($cpt)
                                       let $ref      := if ($cpt/inherit) then 'inherit' else 'contains'
                                       return
                                          (
                                             attribute reference {$ref},
                                             if ($original/name[@language=$language][string-length() gt 0]) then $original/name[@language=$language] else $original/name[1][string-length() gt 0]
                                          )
                                       else
                                          if ($cpt/name[@language=$language][string-length() gt 0]) then $cpt/name[@language=$language] else $cpt/name[1][string-length() gt 0]
                                       }
                                       </concept>
                                    }
                                 </association>
   else ()
   
   (: ********** Postco Terminology Associations ********** :)     
   
   let $postcoAssociationResult     :=
                              for $code in $postcoAssociationCodes
                                 let $extractedCodes        := treplib:getCodesFromExpression($code)
                                 let $extractedConcepts     :=
                                             for $extractedCode in $extractedCodes
                                             let $codeSystemConcepts    := collection($setlib:strCodesystemStableData)//concept[@code=$extractedCode]
                                             let $codeSystemConcept     := $codeSystemConcepts[ancestor::browsableCodeSystem/@oid=$codeSystemId]
                                             return
                                             $codeSystemConcept
                                 let $association           := $associations[@code=$code]
                                 (: possible multiple versions with certain id :)
                                 let $concepts := 
                                                   for $concept in $association
                                                   let $project := $association/ancestor::decor
                                                   let $cpt := 
                                                            if ($concept/@conceptFlexibility) then
                                                               $project//concept[@id=$concept/@conceptId][@effectiveDate=$concept/@conceptFlexibility]
                                                            else
                                                               $project//concept[@id=$concept/@conceptId]
                                                   return
                                                   $cpt[@statusCode=('draft','final')]
                                 let $inheritConcepts := 
                                          for $concept in $association
                                          let $project := $association/ancestor::decor
                                          let $cpt := 
                                                      if ($concept/@conceptFlexibility) then
                                                         $project//concept[inherit/@ref=$concept/@conceptId][inherit/@effectiveDate=$concept/@conceptFlexibility]
                                                      else
                                                         $project//concept[inherit/@ref=$concept/@conceptId]
                                          return
                                          $cpt[@statusCode=('draft','final')]
                                 return
                                 if (empty($ancestor) or $ancestor=$extractedConcepts/ancestor) then
                                    <postcoAssociation json:array="true">
                                          {
                                          attribute conceptCount {count($concepts | $inheritConcepts)},
                                          attribute expression {$code}
                                          ,
                                          for $extractedConcept in $extractedConcepts
                                             let $designations          := $extractedConcept/designation
                                             return
                                              <codeSystemConcept json:array="true">
                                                   {
                                                   if ($extractedConcept) then
                                                      (
                                                      $extractedConcept/@*,
                                                      if ($designations[@lang=$language]) then
                                                         if ($designations[@lang=$language][@use='fsn']) then
                                                            $designations[@lang=$language][@use='fsn']
                                                         else
                                                            $designations[@lang=$language][@use='pref']
                                                      else
                                                         if ($designations[@use='fsn']) then
                                                            $designations[@use='fsn'][1]
                                                         else
                                                            $designations[@use='pref'][1]
                                                       )
                                                   else
                                                      (
                                                      attribute code {$extractedConcept/@code},
                                                      attribute notFound {''},
                                                      <designation>{$treplib:docHtmlNames[@key='not-found']/text[@language=$language]/text()}</designation>
                                                      )
                                                   }
                                                </codeSystemConcept>
                                          ,
                                          for $cpt in  $concepts | $inheritConcepts
                                          let $dataset := $cpt/ancestor::dataset
                                          let $project := $cpt/ancestor::decor
                                          let $projectName := if ($project/project/name[@language=$language][string-length() gt 0]) then $project/project/name[@language=$language] else $project/project/name[string-length() gt 0][1]
                                          let $datasetName := if ($dataset/name[@language=$language][string-length() gt 0]) then $dataset/name[@language=$language] else $dataset/name[string-length() gt 0][1]
                                          let $version := if (string-length($dataset/@versionLabel) gt 0) then concat(' (',$dataset/@versionLabel/string(),')') else ''
                                          return
                                          <concept id="{$cpt/@id}" effectiveDate="{$cpt/@effectiveDate}" statusCode="{$cpt/@statusCode}" dataset="{concat($datasetName,$version)}" datasetId="{$dataset/@id}" datasetEffectiveDate="{$dataset/@effectiveDate}" project="{$projectName}" prefix="{$project/project/@prefix}" json:array="true">
                                          {
                                          if ($cpt/inherit or $cpt/contains) then
                                          let $original := utillib:getOriginalForConcept($cpt)
                                          let $ref      := if ($cpt/inherit) then 'inherit' else 'contains'
                                          return
                                             (
                                                attribute reference {$ref},
                                                if ($original/name[@language=$language]) then $original/name[@language=$language] else $original/name[string-length() gt 0][1]
                                             )
                                          else
                                             if ($cpt/name[@language=$language]) then $cpt/name[@language=$language] else $cpt/name[string-length() gt 0][1]
                                          }
                                          </concept>
                                       }
                                    </postcoAssociation>
                                 else ()   
   
   (: ********** ValueSets ********** :)
   
   let $progressUpdate        :=
                                 (
                                    update value $request/@progress with 'Checking value sets...',
            update value $request/@progress-percentage with round((100 div 9) * 6)
            )

        let $valueSetConcepts      := $decorCollection//terminology/valueSet/conceptList/concept[@codeSystem=$codeSystemId]
   let $valueSetExceptions    := $decorCollection//terminology/valueSet/conceptList/exception[@codeSystem=$codeSystemId]
   let $valueSetIncludes      := $decorCollection//terminology/valueSet/conceptList/include[@codeSystem=$codeSystemId]
   let $valueSetExcludes      := $decorCollection//terminology/valueSet/conceptList/exclude[@codeSystem=$codeSystemId]
                                       
   let $allValueSetCodes      := if ($codeSystemId='2.16.840.1.113883.6.96') then
                                    distinct-values($valueSetConcepts/@code[. castable as xs:integer] | $valueSetExceptions/@code[. castable as xs:integer] | $valueSetIncludes/@code[. castable as xs:integer] | $valueSetExcludes/@code[. castable as xs:integer])
                else
   distinct-values($valueSetConcepts/@code | $valueSetExceptions/@code | $valueSetIncludes/@code | $valueSetExcludes/@code)

   let $postcoValueSetCodes   :=  if ($codeSystemId='2.16.840.1.113883.6.96') then
                                       distinct-values($valueSetConcepts/@code[not(. castable as xs:integer)] | $valueSetExceptions/@code[not(. castable as xs:integer)] | $valueSetIncludes/@code[not(. castable as xs:integer)] | $valueSetExcludes/@code[not(. castable as xs:integer)])
                                  else
                                    () 


   let $resultValueSets :=
                              for $code in $allValueSetCodes
                                 let $codeSystemConcepts    := collection($setlib:strCodesystemStableData)//concept[@code=$code]
                                 let $codeSystemConcept     := $codeSystemConcepts[ancestor::browsableCodeSystem/@oid=$codeSystemId]
                                 let $designations          := $codeSystemConcept/designation
                                 let $ancslf                := $codeSystemConcept/ancSlf
                                 let $vConcepts    := $valueSetConcepts[@code=$code]
                                 let $vExceptions  := $valueSetExceptions[@code=$code]
                                 let $vIncludes    := $valueSetIncludes[@code=$code]
                                 let $vExcludes    := $valueSetExcludes[@code=$code]
                                 return
                                 if (empty($ancestor) or $ancestor=$codeSystemConcept/ancestor) then
                                    <valueSetAssociation json:array="true">
                                    {
                                       attribute valueSetCount {count($vConcepts[ancestor::valueSet/@statusCode=('draft','pending','final')] | $vExceptions[ancestor::valueSet/@statusCode=('draft','pending','final')] | $vIncludes[ancestor::valueSet/@statusCode=('draft','pending','final')] | $vExcludes[ancestor::valueSet/@statusCode=('draft','pending','final')])},
                                       if ($codeSystemId= ('2.16.840.1.113883.6.96','2.16.840.1.113883.6.1')) then
                                          attribute toplevel {$ancslf[.=$toplevels]}
                                       else (),
                                       if                                       ($codeSystemConcept) then
                                       <codeSystemConcept>
                                       {
                                       $codeSystemConcept/@*,
                                       if ($designations[@lang=$language]) then
                                          if ($designations[@lang=$language][@use='fsn']) then
                                       $designations[@lang=$language][@use='fsn']
                                             else
                                          $designations[@lang=$language][@use='pref']
                                       else
                                             if ($designations[@use='fsn']) then
                                                $designations[@use='fsn'][1]
                                             else
                                                $designations[@use='pref'][1]
                                          }
                                    </codeSystemConcept>
                                    else 
                                       <codeSystemConcept notFound="" code="{$code}">
                                          <designation>{$treplib:docHtmlNames[@key='not-found']/text[@language=$language]/text()}</designation>
                                       </codeSystemConcept>
                                    ,
                                       for $valueSetConcept in $vConcepts | $vExceptions | $vIncludes | $vExcludes
                                       let $valueSets := $valueSetConcept/ancestor::valueSet
                                       for $valueSet in $valueSets
                                          let $project := $valueSet/ancestor::decor
                                          let $projectName := if ($project/project/name[@language=$language][string-length() gt 0]) then $project/project/name[@language=$language] else $project/project/name[string-length() gt 0][1]
                                          let $type := name($valueSetConcept)
                                          return
                                          if ($valueSet/@statusCode=('draft','pending','final')) then
                                             <valueSet project="{$projectName}" prefix="{$project/project/@prefix}" type="{$type}" json:array="true">
                                             {
                                                $valueSet/@*
                                             }
                                             </valueSet>
                                          else ()
                                       }
                                 </valueSetAssociation>
                              else()

        let $postcoValueSets :=
        for $code in $postcoValueSetCodes
                                 let $extractedCodes        := treplib:getCodesFromExpression($code)
                                 let $extractedConcepts     :=
                                                               for $extractedCode in $extractedCodes
                                                               let $codeSystemConcepts    := collection($setlib:strCodesystemStableData)//concept[@code=$extractedCode]
                                                               let $codeSystemConcept     := $codeSystemConcepts[ancestor::browsableCodeSystem/@oid=$codeSystemId]
                                                               return
                                                               $codeSystemConcept
                                 let $vConcepts             := $valueSetConcepts[@code=$code]
                                 let $vExceptions           := $valueSetExceptions[@code=$code]
                                 let $vIncludes             := $valueSetIncludes[@code=$code]
                                 let $vExcludes             := $valueSetExcludes[@code=$code]
                                 return
                                 if (empty($ancestor) or $ancestor=$extractedConcepts/ancestor) then
                                    <valueSetAssociation json:array="true">
                                       {
                                       attribute valueSetCount {count($vConcepts[ancestor::valueSet/@statusCode=('draft','pending','final')] | $vExceptions[ancestor::valueSet/@statusCode=('draft','pending','final')] | $vIncludes[ancestor::valueSet/@statusCode=('draft','pending','final')] | $vExcludes[ancestor::valueSet/@statusCode=('draft','pending','final')])},
                                       attribute expression {$code}
                                       ,
                                       for $extractedConcept in $extractedConcepts
                                          let $designations          := $extractedConcept/designation
                                          return
                                           <codeSystemConcept json:array="true">
                                                {
                                                if ($extractedConcept) then
                                                   (
                                                   $extractedConcept/@*,
                                                   if ($designations[@lang=$language]) then
                                                      if ($designations[@lang=$language][@use='fsn']) then
                                                         $designations[@lang=$language][@use='fsn']
                                                      else
                                                         $designations[@lang=$language][@use='pref']
                                                   else
                                                      if ($designations[@use='fsn']) then
                                                         $designations[@use='fsn'][1]
                                                      else
                                                         $designations[@use='pref'][1]
                                                    )
                                                else
                                                   (
                                                   attribute code {$extractedConcept/@code},
                                                   attribute notFound {''},
                                                   <designation>{$treplib:docHtmlNames[@key='not-found']/text[@language=$language]/text()}</designation>
                                                   )
                                                }
                                             </codeSystemConcept>
                                       ,
                                       for $valueSetConcept in $vConcepts | $vExceptions | $vIncludes | $vExcludes
                                          let $valueSets := $valueSetConcept/ancestor::valueSet
                                          for $valueSet in $valueSets
                                             let $project := $valueSet/ancestor::decor
                                             let $projectName := if ($project/project/name[@language=$language][string-length() gt 0]) then $project/project/name[@language=$language] else $project/project/name[string-length() gt 0][1]
                                             let $type := name($valueSetConcept)
                                             return
                                             if ($valueSet/@statusCode=('draft','pending','final')) then
                                                <valueSet project="{$projectName}" prefix="{$project/project/@prefix}" type="{$type}" json:array="true">
                                                {
                                                   $valueSet/@*
                                                }
                                                </valueSet>
                                             else ()
                                          }
                                    </valueSetAssociation>
                                 else()                              
        (: ********** Templates ********** :)                         
                              
   let $progressUpdate        :=
                                 (
                                    update value $request/@progress with 'Checking templates...',
                                    update value $request/@progress-percentage with round((100 div 9) * 8)
                                 )

   let $templateVocabularies  := $decorCollection//rules//vocabulary[@codeSystem=$codeSystemId]

   let $vocabCodes            := if ($codeSystemId='2.16.840.1.113883.6.96') then
                                    distinct-values($templateVocabularies/@code[. castable as xs:integer] )
      else
   distinct-values($templateVocabularies/@code)
                                       
   let $postcoVocabCodes      :=  if ($codeSystemId='2.16.840.1.113883.6.96') then
                                       distinct-values($templateVocabularies/@code[not(. castable as xs:integer)] )
                                     else
                                       ()                                   
                                 
      
   let $resultTemplates       :=
                                 for $code in $vocabCodes
                                    let $codeSystemConcepts    := collection($setlib:strCodesystemStableData)//concept[@code=$code]
                                    let $codeSystemConcept     := $codeSystemConcepts[ancestor::browsableCodeSystem/@oid=$codeSystemId]
                                    let $designations          := $codeSystemConcept/designation
                                    let $ancslf                := $codeSystemConcept/ancSlf
                                    let $templateVocabulary := $templateVocabularies[@code=$code]
                                    return
                                    if (empty($ancestor) or $ancestor=$codeSystemConcept/ancestor) then
                                       <templateAssociation json:array="true">
                                       {
                                          attribute templateCount {count($templateVocabulary)},
                                          if ($codeSystemId= ('2.16.840.1.113883.6.96','2.16.840.1.113883.6.1')) then
                                             attribute toplevel {$ancslf[.=$toplevels]}
                                          else (),
                                          if                                          ($codeSystemConcept) then
                                          <codeSystemConcept>
                                          {
                                          $codeSystemConcept/@*,
                                          if ($designations[@lang=$language]) then
                                             if ($designations[@lang=$language][@use='fsn']) then
                                          $designations[@lang=$language][@use='fsn']
                                                else
                                             $designations[@lang=$language][@use='pref']
                                          else
                                                if ($designations[@use='fsn']) then
                                                   $designations[@use='fsn'][1]
                                                else
                                                   $designations[@use='pref'][1]
                                             }
                                       </codeSystemConcept>
                                          else
                                          <codeSystemConcept notFound="" code="{$code}">
                                             <designation>{$treplib:docHtmlNames[@key='not-found']/text[@language=$language]/text()}</designation>
                                          </codeSystemConcept>
                                          ,
                                             for $vocab in $templateVocabulary
                                             let $template := $vocab/ancestor::template
                                             let $project := $vocab/ancestor::decor
                                             let $projectName := if ($project/project/name[@language=$language][string-length() gt 0]) then $project/project/name[@language=$language] else $project/project/name[string-length() gt 0][1]
                                             return
                                             if ($template/@statusCode=('draft','pending','review','active')) then
                                                <template project="{$projectName}" prefix="{$project/project/@prefix}" json:array="true">
                                                   {
                                                   $template/@*,
                                                   if ($template/desc[@language=$language]) then $template/desc[@language=$language] else $template/desc[1]
                                                   }
                                                </template>
                                             else ()
                                             }
                                     </templateAssociation>
                                 else()
   
let $postcoTemplates       :=
                                 for $code in $postcoVocabCodes
                                    let $extractedCodes        := treplib:getCodesFromExpression($code)
                                    let $extractedConcepts     :=
                                                                  for $extractedCode in $extractedCodes
                                                                  let $codeSystemConcepts    := collection($setlib:strCodesystemStableData)//concept[@code=$extractedCode]
                                                                  let $codeSystemConcept     := $codeSystemConcepts[ancestor::browsableCodeSystem/@oid=$codeSystemId]
                                                                  return
                                                                  $codeSystemConcept
                                    let $templateVocabulary    := $templateVocabularies[@code=$code]
                                    return
                                    if (empty($ancestor) or $ancestor=$extractedConcepts/ancestor) then
                                       <templateAssociation json:array="true">
                                          {
                                          attribute templateCount {count($templateVocabulary)},
                                          attribute expression {$code}
                                          ,
                                          for $extractedConcept in $extractedConcepts
                                             let $designations          := $extractedConcept/designation
                                             return
                                              <codeSystemConcept json:array="true">
                                                   {
                                                   if ($extractedConcept) then
                                                      (
                                                      $extractedConcept/@*,
                                                      if ($designations[@lang=$language]) then
                                                         if ($designations[@lang=$language][@use='fsn']) then
                                                            $designations[@lang=$language][@use='fsn']
                                                         else
                                                            $designations[@lang=$language][@use='pref']
                                                      else
                                                         if ($designations[@use='fsn']) then
                                                            $designations[@use='fsn'][1]
                                                         else
                                                            $designations[@use='pref'][1]
                                                       )
                                                   else
                                                      (
                                                      attribute code {$extractedConcept/@code},
                                                      attribute notFound {''},
                                                      <designation>{$treplib:docHtmlNames[@key='not-found']/text[@language=$language]/text()}</designation>
                                                      )
                                                   }
                                                </codeSystemConcept>
                                          ,
                                          for $vocab in $templateVocabulary
                                             let $template := $vocab/ancestor::template
                                             let $project := $vocab/ancestor::decor
                                             let $projectName := if ($project/project/name[@language=$language][string-length() gt 0]) then $project/project/name[@language=$language] else $project/project/name[string-length() gt 0][1]
                                             return
                                             if ($template/@statusCode=('draft','pending','review','active')) then
                                                <template project="{$projectName}" prefix="{$project/project/@prefix}" json:array="true">
                                                   {
                                                   $template/@*,
                                                   if ($template/desc[@language=$language]) then $template/desc[@language=$language] else $template/desc[1]
                                                   }
                                                </template>
                                             else ()
                                             }
                                        </templateAssociation>
                                    else()
             
                                    
                                    
   let $progressUpdate        :=
                                 (
                                    update value $request/@progress with 'Finished',
                                    update value $request/@progress-percentage with '100',
                                 update value $request/@busy with 'false'
                                 )
   let $fullReport         :=
                           <codeSystemUsageReport  status="active">
                           {
                           $request/@*
                           }
                              <postcoAssociations count="{count($postcoAssociationResult[concept])}" conceptCount="{count($postcoAssociationResult/concept)}" projectCount="{count(distinct-values($postcoAssociationResult/concept/@project))}">
                              {
                              $postcoAssociationResult[concept]
                              }
                              </postcoAssociations>
                              <postcoValueSets count="{count($postcoValueSets[valueSet])}" valueSetCount="{count($postcoValueSets//valueSet)}" projectCount="{count(distinct-values($postcoValueSets//valueSet/@project))}">
                              {
                              $postcoValueSets[valueSet]
                              }
                              </postcoValueSets>
                              <postcoTemplates count="{count($postcoTemplates[template])}" templateCount="{count($postcoTemplates//template)}" projectCount="{count(distinct-values($postcoTemplates//template/@project))}">
                              {
                              $postcoTemplates[template]
                              }
                              </postcoTemplates>
                              <associations count="{count($associationResult[concept])}" conceptCount="{count($associationResult/concept)}" projectCount="{count(distinct-values($associationResult/concept/@project))}">
                              {
                              $associationResult[concept]
 }
                              </associations>
                              <valueSets count="{count($resultValueSets[valueSet])}" valueSetCount="{count($resultValueSets//valueSet)}" projectCount="{count(distinct-values($resultValueSets//valueSet/@project))}">
                              {
                           $resultValueSets[valueSet]
 }
                              </valueSets>
                              <templates count="{count($resultTemplates[template])}" templateCount="{count($resultTemplates//template)}" projectCount="{count(distinct-values($resultTemplates//template/@project))}">
                              {
                           $resultTemplates[template]
                           }
                           </templates>
                           </codeSystemUsageReport>



    return
    if (count($codeSystem) = 1) then
      (
                xmldb:store($setlib:strDecorTemp,concat($codeSystemId,'-report-',replace($request/@on, '\D', ''),'.xml'),$fullReport)
                ,
                xmldb:remove(util:collection-name($request), util:document-name($request))
                )
    else (
            let $message                := 'ERROR. Found ' || count($codeSystem) || ' codesystems for id ' || $codeSystemId || '. Expected: 1.'
            let $log                    := utillib:log-scheduler-event('error', $threadId, $logsignature, $message)
            let $progress               := update value $request/@progress with $message
            
            return ()
        )
    
};

