xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Scenario API allows read, create, update of DECOR scenarios :)
module namespace scapi              = "http://art-decor.org/ns/api/scenario";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace utilsc      = "http://art-decor.org/ns/api/util-scenario" at "library/util-scenario-lib.xqm";
import module namespace utiltemp    = "http://art-decor.org/ns/api/util-template" at "library/util-template-lib.xqm";
import module namespace utilsvg     = "http://art-decor.org/ns/api/util-svg" at "library/util-svg-lib.xqm";
import module namespace utilhtml    = "http://art-decor.org/ns/api/util-html" at "library/util-html-lib.xqm";
import module namespace utilfsh     = "http://art-decor.org/ns/api/util-fsh" at "library/util-fsh-lib.xqm";

import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";
import module namespace serverapi   = "http://art-decor.org/ns/art-decor-server" at "server-api.xqm";
import module namespace qapi        = "http://art-decor.org/ns/api/questionnaire" at "questionnaire-api.xqm";

declare namespace json      = "http://www.json.org";
declare namespace f         = "http://hl7.org/fhir";

(: =========================================================================================================================================== :)
(: ===============                                              SCENARIO CALLS                                                 =============== :)
(: =========================================================================================================================================== :)

(:~ Retrieves latest DECOR scenario based on $id (oid)
    @param $id                      - required parameter denoting the id of the scenario
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function scapi:getLatestScenario($request as map(*)) {
    scapi:getScenario($request)
};

(:~ Retrieves DECOR scenario based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the scenario
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the scenario. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function scapi:getScenario($request as map(*)) {

    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := $request?parameters?prefix[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]
    
    let $results                        := utilsc:getScenario($id, $effectiveDate, $projectVersion, $projectLanguage)

    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple scenarios for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )

};

(:~ Retrieves DECOR scenario history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function scapi:getScenarioHistory($request as map(*)) {
    
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-SCENARIO, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-SCENARIO}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Returns a list of zero or more scenarios and child transactions
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $datasetId               - optional. Filter output to just those based on this dataset-id
    @param $datasetEffectiveDate    - optional. null filters to the newest version based on $datasetId, yyyy-mm-ddThh:mm:ss for this specific version
    @param $sort                    - optional parameter to indicate sorting. Only option 'name'
    @param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
    @return all live repository/non-private scenarios as JSON, all scenarios for the given $projectPrefix or nothing if not found
    @since 2020-05-03
:)
declare function scapi:getScenarioList($request as map(*)) {

    let $projectPrefix                  := $request?parameters?prefix[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $datasetId                      := $request?parameters?datasetId[not(. = '')]
    let $datasetEffectiveDate           := $request?parameters?datasetEffectiveDate[not(. = '')]
    let $sort                           := $request?parameters?sort[string-length() gt 0]
    let $sortorder                      := $request?parameters?sortorder[. = 'descending']
    let $max                            := $request?parameters?max
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]
    let $results                        := utilsc:getScenarioList($projectPrefix, $projectVersion, $projectLanguage, $datasetId, $datasetEffectiveDate, $sort, $sortorder, $max)

    return
        <list artifact="{$results[1]/@artifact}" current="{sum($results/xs:integer(@current))}" total="{sum($results/xs:integer(@total))}" all="{sum($results/xs:integer(@all))}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
            <actors json:array="true"/>
        {
            utillib:addJsonArrayToElements($results/*, $format)
        }
        </list>
};

(:~ Create scenario, new scenario with its own id and effectiveDate based on defaultBaseId for scenarios

    Situation 1: completely new scenario with its own id and effectiveDate
    - Input MAY be "empty" then we create a single scenario with id/effectiveDate/empty name and desc/single transaction group containing single transaction
    - Input MAY be a scenario without id/effectiveDate but with name/desc and other direct properties. If there are no transactions, we do the same as for empty input. 
    - Transaction handling is sent to transaction api where a similar function works in a similar way
    
    Situation 2: new scenario version based on existing scenario with new id and effectiveDate
    - Parameter sourceId and sourceEffectiveDate are required. Input SHALL be empty
    - Return a copy of existing scenario with new id and effectiveDates on scenario and child transactions 
    
    Note: if you want to update a transaction (group) from one dataset to the other, you can do that by adding 
          transaction.representingTemplate.targetDataset and optionally .targetDatasetFlexibility next to the original 
          transaction.representingTemplate.sourceDataset and .sourceDatasetFlexibility 
    
    The scenario creation process then tries to find matches in the new dataset for the concepts listed from the old dataset. This logic is handed off to the transaction api  
    
    @param $bearer-token        - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $projectPrefix       - required. what project to create into             
    @param $sourceId            - optional. the scenario id to create the new scenario from 
    @param $sourceEffectiveDate - optional. the scenario effectiveDate to create the scenario from 
    @param $request-body        - optional. body containing new scenario structure
    @return scenario structure
    @since 2020-05-03
:)
declare function scapi:postScenario($request as map(*)) {

    let $authmap                        := $request?user
    let $projectPrefix                  := $request?parameters?prefix[string-length() gt 0]
    let $sourceId                       := $request?parameters?sourceId
    let $sourceEffectiveDate            := $request?parameters?sourceEffectiveDate
    let $data                           := utillib:getBodyAsXml($request?body, 'scenario', ())
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                          :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Parameter prefix SHALL have a value')
        else ()
    
    let $check                          :=
        if (empty($sourceId) and empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter sourceId OR data')
        else
        if (not(empty($sourceId)) and not(empty($data))) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter sourceId OR data, not both')
        else
        if ($sourceEffectiveDate[not(. = '')] and empty($sourceId)) then
            error($errors:BAD_REQUEST, 'Request SHALL have non-empty parameter sourceId if parameter sourceEffectiveDate has a value')
        else ()
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]    
    
    let $result                         := utilsc:postScenario($authmap, $projectPrefix, $sourceId, $sourceEffectiveDate, $data)
    
    return
        if (empty($result)) then () else (
            roaster:response(201, 
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
            )
        )

};

(:~ Update scenario
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the scenario to update 
    @param $effectiveDate            - required. the effectiveDate for the scenario to update 
    @param $request-body             - required. body containing new scenario structure
    @return concept structure
    @since 2020-05-03
:)
declare function scapi:putScenario($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                           := utillib:getBodyAsXml($request?body, 'scenario', ())
    let $deletelock                     := $request?parameters?deletelock = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()

    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]

    let $result                         := utilsc:putScenario($authmap, string($id), $effectiveDate, $data, $deletelock)
    
    return
        if (empty($result)) then () else (
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*, $format)
            }
        )

};

(:~ Update DECOR scenario parts. Does not touch any transactions. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "/...", "value": "[string|object]" }

where
* op - add (object associations, tracking, event) or remove (object associations only) or replace (displayName, priority, type, tracking, assignment)
* path - / (object, tracking, assignment) or /displayName or /priority or /type
* value - string when the path is not /. object when path is /

    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the issue to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return dataset structure including generated meta data
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function scapi:patchScenario($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
     let $check                         :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
        
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )

    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]

    let $results                        := utilsc:patchScenario($authmap, string($id), $effectiveDate, $data)
    
    return 
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple scenarios for id '", $id, "' effectiveDate '", $effectiveDate, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )

};

(:~ Delete DECOR scenario
    @param $id              - required. DECOR scenario/@id to delete
    @param $effectiveDate   - optional. DECOR scenario/@effectiveDate to delete
    @return nothing
    @since 2021-02-19
:)
declare function scapi:deleteScenario($request as map(*)) {
    
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
            
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
        
    let $check                          :=
        if ($authmap?groups = 'dba') then () else (
            error($errors:FORBIDDEN, 'You need to be dba for this feature')
        )    
    
    let $storedScenario                 := utillib:getScenario($id, $effectiveDate)

    let $update                         := update delete $storedScenario
    
    return (
        roaster:response(204, ())
    )
};

(:~ Update scenario statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap required. Map derived from token
    @param $id required. DECOR concept/@id to update
    @param $effectiveDate required. DECOR concept/@effectiveDate to update
    @param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function scapi:putScenarioStatus($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $recurse                        := $request?parameters?recurse = true()
    let $list                           := $request?parameters?list = true()
    let $statusCode                     := $request?parameters?statusCode
    let $versionLabel                   := $request?parameters?versionLabel
    let $expirationDate                 := $request?parameters?expirationDate
    let $officialReleaseDate            := $request?parameters?officialReleaseDate
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                        := utilsc:setScenarioStatus($authmap, $id, $effectiveDate, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)

    return
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            (: max 2 arrays: one with error and one with success :)
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Returns an actors object with zero or more child actors from project
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $sort                    - optional parameter to indicate sorting. Only option 'name'
    @param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
    @return list with actors
    @since 2020-05-03
:)
declare function scapi:getScenarioActorList($request as map(*)) {

    let $projectPrefix                  := $request?parameters?prefix[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $sort                           := $request?parameters?sort[string-length() gt 0]
    let $sortorder                      := $request?parameters?sortorder[. = 'descending']
    let $max                            := $request?parameters?max
    
    let $check                          :=
        if (empty($projectPrefix)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter prefix')
        else ()
    
    let $results                        := utilsc:getScenarioActorList($projectPrefix, $projectVersion, $projectLanguage, $sort, $sortorder, $max)

    return
        <list artifact="{$results[1]/@artifact}" current="{sum($results/xs:integer(@current))}" total="{sum($results/xs:integer(@total))}" all="{sum($results/xs:integer(@allcnt))}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
            <actors json:array="true"/>
        {
            utillib:addJsonArrayToElements($results/*)
        }
        </list>
};

(:~ Retrieves DECOR scenario actor
    @param $projectPrefix            - optional. limits scope to this project only
    @param $id                       - required. id for the actor to retrieve
    @return one or zero actor object
    @since 2020-08-11
:)
declare function scapi:getScenarioActor($request as map(*)) {

    let $projectPrefix                  := $request?parameters?prefix[string-length() gt 0]
    let $id                             := $request?parameters?id
    
    let $results                        := utilsc:getScenarioActor($projectPrefix, $id)
    
    return
        if (empty($results)) then 
            roaster:response(404, ())
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple actors for id '", $id, "'. Expected 0..1. You should delete and recreate this actor. Consider alerting your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )

};

(:~ Create scenario actor, new scenario actor with its own id and effectiveDate based on defaultBaseId for actors
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $projectPrefix           - required. limits scope to this project only
    @param $request-body             - required. body containing new actor structure
    @return scenario structure
    @since 2020-05-03
:)
declare function scapi:postScenarioActor($request as map(*)) {

    let $authmap                := $request?user
    let $projectPrefix          := $request?parameters?prefix[string-length() gt 0]
    let $data                   := utillib:getBodyAsXml($request?body, 'actor', ())
    
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Parameter prefix SHALL have a value')
        else ()
    
    let $check                  :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $result                 := utilsc:postScenarioActor($authmap, $projectPrefix, $data)
    
    return
        if (empty($result)) then () else (
            roaster:response(201, 
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
            )
        )
};

(:~ Update DECOR scenario actor parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "[/type|/name|/desc]", "value": "[string|object]" }

where

* op - add (name|desc) or remove (name|desc) or replace (type|name|desc)

* path - /type, or /name or /desc 

* value - string when the path is /type, otherwise object
@param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
@param $id                       - required. the id for the scenario actor to update 
@param $request-body             - required. body containing array of parameter objects each containing RFC 6902 compliant contents
@return actor structure
@since 2020-05-03
@see http://tools.ietf.org/html/rfc6902
:)
declare function scapi:patchScenarioActor($request as map(*)) {

    let $authmap                        := $request?user
    let $projectPrefix                  := $request?parameters?prefix[string-length() gt 0]
    let $id                             := $request?parameters?id

    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())

    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $results                        := utilsc:patchScenarioActor($authmap, $projectPrefix, $id, $data)
    
    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }

};

(:~ Delete DECOR scenario actor
    @param $projectPrefix   - optional. DECOR project/@prefix to update
    @param $id              - required. DECOR actor/@id to update
    @return nothing
:)
declare function scapi:deleteScenarioActor($request as map(*)) {
    
    let $authmap                        := $request?user
    let $projectPrefix                  := $request?parameters?prefix[string-length() gt 0]
    let $id                             := $request?parameters?id
        
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                          :=
        if ($authmap?groups = 'dba') then () else (
            error($errors:FORBIDDEN, 'You need to be dba for this feature')
        )
    
    let $actor                          := utilsc:getScenarioActor($projectPrefix, $id)
    let $inUse                          := $setlib:colDecorData//actor[@id = $id][ancestor::transaction]
    
    let $check                          :=
        if ($inUse) then
            error($errors:BAD_REQUEST, 'Actor ' || $id || ' cannot be deleted while it is in use. Found usage in transaction(s): ' ||
                string-join(
                    for $a in $inUse
                    let $t  := $a/ancestor::transaction[1]
                    let $p  := $a/ancestor::decor[1]/project/@prefix
                    let $p  := if ($p = $projectPrefix) then () else (' project prefix=' || $a/ancestor::decor[1]/project/@prefix)
                    return '''' || $t/name[1] || ''' (id=' || $t/@id || $p || ')'''
                , ', ')
            )
        else ()
    
    let $update                         := update delete $actor
    
    return (
        roaster:response(204, ())
    )
};

(: =========================================================================================================================================== :)
(: ===============                                              TRANSACTION CALLS                                              =============== :)
(: =========================================================================================================================================== :)

(:~ Retrieves latest DECOR transaction based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the transaction
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $treeonly                - optional boolean parameter to get the tree structure only if true
    @param $fullTree                - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function scapi:getLatestTransaction($request as map(*)) {
    scapi:getTransaction($request)
};

(:~ Retrieves DECOR transaction based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the transaction
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $treeonly                - optional boolean parameter to get the tree structure only if true
    @param $fullTree                - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function scapi:getTransaction($request as map(*)) {

    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefixOrId              := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $treeOnly                       := $request?parameters?treeonly = true()
    let $fullTree                       := $request?parameters?fulltree = true()

    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]

    let $results                        := utilsc:getTransaction($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage, $treeOnly, $fullTree)

    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple transactions for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )

};

(:~ Retrieves DECOR transaction for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version

@param $id                      - required parameter denoting the id of the transaction
@param $effectiveDate           - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
@param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
@param $projectLanguage         - optional parameter to select from a specific compiled language
@param $format                  - optional. overrides the accept-header for frontend purposes
@param $download                - optional as xs:boolean. Default: false. 
@return as-is or as compiled as JSON
@since 2023-11-13
:)
declare function scapi:getTransactionExtract($request as map(*)) {

    let $project                        := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $communityName                  := 
        typeswitch ($request?parameters?community) 
        case xs:string return 
            array:flatten(
                for $s in $request?parameters?community[string-length() gt 0]
                return tokenize($s,'\s')
            )
        default return array:flatten($request?parameters?community)
    let $format                         := $request?parameters?format[not(. = '')]
    let $download                       := $request?parameters?download=true()
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    (: default behavior Accept header:
    - backend:   default Accept header is */* if accept header is omitted   
    - frontend:  default Accept header is application/xml :)
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json', 'text/plain', 'text/fsh', 'application/*', 'text/*', '*/*')])[1]
    
    (: when you do a right mouse click for downloading content in Edge or Chrome, you effectively do a GET without Accept header :)
    let $check                          := 
        if (empty($acceptTypes) and empty($format)) then
            error($errors:BAD_REQUEST, 'Your request does not have an Accept header, and is missing a format parameter. Don''t know what to do')
        else ()
    
    (: parameter format overrides Accept header  :)
    let $format                         := if ($format) then $format else tokenize($acceptedType, '/')[2]

    (: check valid formats :)
    let $check                          :=
        if ($format = ('xml', 'json', 'fsh')) then () else 
            error($errors:BAD_REQUEST, 'Value for parameter format ''' || string-join($format, ', ') || ''' not supported. Supported are: xml, json, fsh')
    
    let $transaction                    := utillib:getTransaction($id, $effectiveDate, $project, $projectVersion, $projectLanguage)
    
    
    let $results                        := if (empty($transaction)) then () else utillib:getDatasetExtract($transaction, $id, $effectiveDate, (), (), $projectVersion, $projectLanguage, $communityName)
 
    let $filename                       := 'DS_' || $results/@shortName || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $format
    
    let $results                        :=
        if ($format = 'fsh') then utilfsh:convertTransactionOrDataset2Fsh($results, $projectLanguage, $filename)/text() else ( 
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/node(), $format)
                }
         )
        
    (: determine response Content-Type/Media-type :)
    let $responseType                   := $utillib:formatacceptmap?($format) 
    
    let $responseheader                 := 
        (response:set-header('Content-Type', $responseType || '; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else 
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple transactions for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else roaster:response(200, $responseType, $results)      

};

(:~ Retrieves DECOR transaction view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $ui-language             - optional parameter to select from a specific language for the ui
    @param $hidecolumns             - optional parameter to hide columns in the view based on (hex) number
    @param $format                  - optional. if not given it is html
    @param $download                - optional as xs:boolean. Default: false. 
    @return as-is 
    @since 2024-09-24
:)
declare function scapi:getTransactionView($request as map(*)) {

    let $project                        := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $communityName                  := 
        typeswitch ($request?parameters?community) 
        case xs:string return 
            array:flatten(
                for $s in $request?parameters?community[string-length() gt 0]
                return tokenize($s,'\s')
            )
        default return array:flatten($request?parameters?community)
    let $ui-lang                        := $request?parameters?ui-language[not(. = '')]
    let $hidecolumns                    := $request?parameters?hidecolumns[not(. = '')]
    let $format                         := $request?parameters?format[not(. = '')]
    let $download                       := $request?parameters?download=true()
    let $inline                         := $request?parameters?inline = true()
    
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $format                         := if ($format = ('list', 'hlist')) then $format else 'html'
    
    let $transaction                    := utillib:getTransaction($id, $effectiveDate, $project, $projectVersion, $projectLanguage)
    
    let $decor                          := $transaction/ancestor::decor[1]

    let $results                        := if (empty($transaction)) then () else utillib:getDatasetExtract($transaction, $id, $effectiveDate, (), (), $projectVersion, $projectLanguage, $communityName)

    let $filename                       := 'DS_' || $results/@shortName || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html' 
        let $r-header                   := 
            (response:set-header('Content-Type', 'text/html; charset=utf-8'),
            if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple transactions for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else 
        
        (: prepare for Html :)
        let $referenceUrl               := $decor/project/reference[@url castable as xs:anyURI]/@url
        let $projectPrefix              := $decor/project/@prefix
        let $header                     := if ($inline) then false() else true()        

        (:get everything hanging off from the current template so we don't miss anything, but may have duplicates in the result:)
        let $templateChain              :=
            if ($format = ('list')) then () else (
                if ($results/@templateId) then (
                    let $transactionTemplate:= utillib:getTemplateById(string($results/@templateId), string($results/@templateEffectiveDate), $decor, $decor/@version, $projectLanguage)[@id]
                    let $decors             := utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL)
                    return 
                        $transactionTemplate | utiltemp:getTemplateChain($transactionTemplate, $decors, map:merge(map:entry(concat($transactionTemplate/@id, $transactionTemplate/@effectiveDate), '')))
                )
                else ()
            )
       return
           switch ($format)
           case 'list'
             return utilhtml:convertTransactionOrDataset2SimpleHtml($results, $projectLanguage, $ui-lang, $hidecolumns, true(), $projectVersion, $referenceUrl, $projectPrefix, false(), $download) 
           case 'hlist'
             return utilhtml:convertTransactionOrDataset2SimpleHierarchicalList($results, $projectLanguage, $ui-lang, $hidecolumns, true(), $projectVersion, $referenceUrl, $projectPrefix, false(), $download) 
           default
             return utilhtml:convertTransactionOrDataset2Html($results, $projectLanguage, $ui-lang, $hidecolumns, true(), true(), true(), $projectVersion, $referenceUrl, $projectPrefix, $templateChain, false(), $filename, $download, $header)
};

(:~ Retrieves DECOR transaction as a diagram based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $conceptId               - optional parameter denoting the id of the concept in the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the concept in the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $interactive             - boolean
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @return as-is 
    @since 2023-12-21
:)
declare function scapi:getTransactionDiagram($request as map(*)) {

    let $conceptId          := $request?parameters?conceptId[not(. = '')]
    let $conceptEffectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?conceptEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?conceptEffectiveDate[string-length() gt 0]
        }
    let $projectPrefixOrId              := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $interactive                    := not($request?parameters?interactive = false())
    let $format                         := $request?parameters?format[not(. = '')]
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $check                          :=
        if (empty($conceptId) and not(empty($conceptEffectiveDate))) then 
            error($errors:BAD_REQUEST, 'Request SHALL NOT have conceptEffectiveDate without conceptId') 
        else ()
    
    let $format                         := if (not($format = 'xml')) then 'svg' else 'xml'

    (: 
       overwrite format in the backend is not always possible because of middleware behaviour
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
    :) 
    let $check                          :=
        if ($format = 'xml' and not((roaster:accepted-content-types()[. = ('image/svg+xml', 'application/xml')])[1] = 'application/xml')) then 
            error($errors:BAD_REQUEST, 'In case of format parameter is xml the accept header should be application/xml')
        else ()

    let $results                        := utillib:getDatasetDiagram($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage, $conceptId, $conceptEffectiveDate, (), $interactive, $format)
 
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else if ($format = 'xml') then (
            for $result in $results
                return
                    element {name($result)} {
                        $result/@*,
                        namespace {"json"} {"http://www.json.org"},
                        utillib:addJsonArrayToElements($result/*, $format)
                    }
        ) else (
            let $responseheader      := (response:set-header('Content-Type','image/svg+xml'), response:set-header('X-Robots-Tag', 'noindex'))
            return $results
            )
};

(:~ Retrieves DECOR transaction history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function scapi:getTransactionHistory($request as map(*)) {
    
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-TRANSACTION, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-TRANSACTION}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Returns a list of zero or more transactions and child transactions
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $datasetId               - optional. Filter output to just those based on this dataset-id
    @param $datasetEffectiveDate    - optional. null filters to the newest version based on $datasetId, yyyy-mm-ddThh:mm:ss for this specific version
    @param $conceptId               - optional. Filter output to just those based on this concept-id
    @param $conceptEffectiveDate    - optional. null filters to the newest version based on $conceptId, yyyy-mm-ddThh:mm:ss for this specific version
    @param $sort                    - optional parameter to indicate sorting. Only option 'name'
    @param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
    @return all live repository/non-private transaction as JSON, all transactions for the given $projectPrefix or nothing if not found
    @since 2025-12-01
:)
declare function scapi:getTransactionList($request as map(*)) {

    let $project                        := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $datasetId                      := $request?parameters?datasetId[not(. = '')]
    let $datasetEffectiveDate           := $request?parameters?datasetEffectiveDate[not(. = '')]
    let $conceptId                      := $request?parameters?conceptId[not(. = '')]
    let $conceptEffectiveDate           := $request?parameters?conceptEffectiveDate[not(. = '')]
    let $sort                           := $request?parameters?sort[string-length() gt 0]
    let $sortorder                      := $request?parameters?sortorder[. = 'descending']
    let $max                            := $request?parameters?max
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]
    
    
    let $results                        := 
        if ($project or $datasetId or $conceptId) then
            utilsc:getTransactionList($project, $projectVersion, $projectLanguage, $datasetId, $datasetEffectiveDate, $conceptId, $conceptEffectiveDate, $sort, $sortorder, $max)
        else (
            error($errors:BAD_REQUEST, 'Request SHALL have either a prameter project OR a datasetId OR a conceptId.')
        )
    return
        <list artifact="{$results[1]/@artifact}" current="{sum($results/xs:integer(@current))}" total="{sum($results/xs:integer(@total))}" all="{sum($results/xs:integer(@all))}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results/*, $format)
        }
        </list>
};


(:~ Create transaction, new transaction with its own id and effectiveDate based on defaultBaseId for transactions

Situation 1: completely new transaction with its own id and effectiveDate
- Input MAY be "empty" then we create a single transaction of the requested type and if this is a group containing single transaction
- Input MAY be a transaction without id/effectiveDate but with name/desc and other direct properties. If there are no child transactions, we do the same as for empty input. 

Situation 2: new transaction version based on existing transaction with new id and effectiveDate
- Parameter sourceId and sourceEffectiveDate are required. Input SHALL be empty
- Return a copy of existing transaction with new id and effectiveDates on transaction and child transactions 

Note: if you want to update a transaction (group) from one dataset to the other, you can do that by adding 
      transaction.representingTemplate.targetDataset and optionally .targetDatasetFlexibility next to the original 
      transaction.representingTemplate.sourceDataset and .sourceDatasetFlexibility 

The transaction creation process then tries to find matches in the new dataset for the concepts listed from the old dataset. This logic is handed off to the transaction api  
    @param $scenarioId              - required. scenario id to insert transaction into.
    @param $scenarioEffectiveDate   - required. scenario effectiveDate to insert transaction into.
    @param $transactionBaseId       - optional. baseId to create the new transaction id out of. Defaults to defaultBaseId for type TR if omitted
    @param $transactionType         - optional if request-body is provided. 'group' or 'stationary' or 'initial' or 'back'
    @param $insertMode              - required. 'into' or 'preceding' or 'following'. For preceding and for following, $insertRef is required
    @param $insertRef               - optional. transaction/@id reference for insert. Inserts as new transaction in scenario if empty
    @param $insertFlexibility       - optional. transaction/@effectiveDate reference for insert. Only relevant when two versions of the same transaction are in the same scenario which is logically highly unlikely
    @param $sourceId                - optional. the transaction id to create the new transaction from 
    @param $sourceEffectiveDate     - optional. the transaction effectiveDate to create the transaction from 
    @param $request-body optional. body containing new transaction structure
    @return created transaction structure
    @since 2020-05-03
:)
declare function scapi:postTransaction($request as map(*)) {

    let $authmap                        := $request?user
    let $scenarioId                     := $request?parameters?id[not(. = '')]
    let $scenarioEffectiveDate          := $request?parameters?effectiveDate[not(. = '')]
    let $transactionBaseId              := $request?parameters?baseId[not(. = '')]
    let $transactionType                := $request?parameters?transactionType[not(. = '')]
    let $insertMode                     := $request?parameters?insertMode[not(. = '')]
    let $insertRef                      := $request?parameters?insertRef[not(. = '')]
    let $insertFlexibility              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?insertFlexibility)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?insertFlexibility[string-length() gt 0]
        }
    let $sourceId                       := $request?parameters?sourceId[not(. = '')]
    let $sourceEffectiveDate            := $request?parameters?sourceEffectiveDate[not(. = '')]
    let $data                           := utillib:getBodyAsXml($request?body, 'transaction', ())
    let $treeOnly                       := $request?parameters?treeonly = true()
    let $fullTree                       := $request?parameters?fulltree = true()
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else
        if (empty($scenarioId) or empty($scenarioEffectiveDate) or empty($insertMode)) then 
            error($errors:BAD_REQUEST, 'Parameter id, effectiveDate and insertMode SHALL have a value')
        else
        if (empty($transactionType) and empty($data/@type)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter transactionType OR data with a type')
        else ()
    
    let $return                 := utilsc:postTransaction($authmap, $scenarioId, $scenarioEffectiveDate, $transactionBaseId, $transactionType, $insertMode, $insertRef, $insertFlexibility, $sourceId, $sourceEffectiveDate, $data)
    let $result                 := utilsc:getTransaction($return/@id, $return/@effectiveDate, (), (), (), $treeOnly, $fullTree)
    
    return
        if (empty($result)) then
            error($errors:SERVER_ERROR, 'A transaction was supposed to be created with id ' || $return/@id || ' and effectiveDate ' || $return/@effectiveDate || ' but upon requesting for it, it turns out not to be found.')
        else (
            roaster:response(201, 
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
            )
        )
};

(:~ Update DECOR transaction parts. Does not touch any child transactions or representingTemplate contents. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "/...", "value": "[string|object]" }

where
* op - add or remove or replace
* path - e.g. /type, /model, /label, /canonicalUri, /name, /desc
* value - string when the path is not /. object when path is /

    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the issue to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return dataset structure including generated meta data
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function scapi:patchTransaction($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
 
    let $treeOnly                       := $request?parameters?treeonly = true()
    let $fullTree                       := $request?parameters?fulltree = true()
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                          :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                         := utilsc:patchTransaction($authmap, string($id), $effectiveDate, $data)
    let $results                        := utilsc:getTransaction($id, $effectiveDate, (), (), (), $treeOnly, $fullTree)
    return 
        if (empty($results)) then (
            error($errors:SERVER_ERROR, 'A transaction was supposed to be patched with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' but upon requesting for it, it turns out not to be found.')
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple transactions for id ''' || $id || ''' effectiveDate ''' || $effectiveDate || '''. Expected 0..1. Alert your administrator as this should not be possible.')
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Delete DECOR transaction
    @param $id              - required. DECOR transaction/@id to delete
    @param $effectiveDate   - optional. DECOR transaction/@effectiveDate to delete
    @return nothing
    @since 2021-02-19
:)
declare function scapi:deleteTransaction($request as map(*)) {
    
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
        
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
   
    let $check                          :=
        if ($authmap?groups = 'dba') then () else (
            error($errors:FORBIDDEN, 'You need to be dba for this feature')
        )    
    
    let $storedTransaction              := utillib:getTransaction($id, $effectiveDate)

    let $update                         := update delete $storedTransaction
    
    return (
        roaster:response(204, ())
    )
};

(:~ Create a scheduled task for a questionnaire-transform-request 
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $project                  - required. what project to create into
    @param $baseId                  - required. specific baseId to be used for the Questionnaires 
    @param $request-body             - required. body containing new scenario structure
    @return questionnaire-transform-request
    @since 2020-05-08
:)
declare function scapi:postLatestTransactionQuestionnaire($request as map(*)) {
    scapi:postTransactionQuestionnaire($request)
};

declare function scapi:postTransactionQuestionnaire($request as map(*)) {
    
    let $authmap                        := $request?user
    let $data                           := utillib:getBodyAsXml($request?body, 'transactions', ())             
    let $project                        := $request?parameters?project[string-length() gt 0]
    let $baseId                         := $request?parameters?baseId[string-length() gt 0]
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                          :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $request-files                  := utilsc:postTransactionQuestionnaire($authmap, $project, $baseId, $data)
    
    return 
        roaster:response(200,
            <list artifact="SCHEDULEDTASKS" current="{count($request-files)}" total="{count($request-files)}" all="{count($request-files)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
            {
                utillib:addJsonArrayToElements($request-files)
            }
            </list>
        )

};

(:~ Accepts a threadId (for logging purposes) and a questionnaire-transform-request element, which shall be dirently form the request file, containing all things required for doing transaction to questionnaire conversion. 
It processes all requested transactions into questionnaires and then commits them all to the project, or none if something fails.
:)
declare function scapi:process-questionnairetransform-request($threadId as xs:string, $request as element(questionnaire-transform-request)) {
    
    let $logsignature                   := 'scapi:process-questionnairetransform-request'
    let $mark-busy                      := update insert attribute busy {'true'} into $request
    
    let $progress                       := if ($request/@progress) then update value $request/@progress with 'Checking request parameters ...' else update insert attribute progress {'Checking request parameters ...'} into $request
        
    let $transactionCount               := count($request/transaction)
    let $progress                       := 5
    let $progress                       := if ($request/@progress-percentage) then update value $request/@progress-percentage with $progress else update insert attribute progress-percentage {$progress} into $request
        
        
    (: =================== PREPARE VARIABLES ==================== :)
        
    let $timeStamp                      := $request/@on
    let $projectPrefix                  := $request/@for[not(. = '*')]
    let $baseId                         := $request/@baseId[not(. = '')]
    let $transactions                   := $request/transaction
 
    (: =================== PREPARE VARIABLES ==================== :)
 
    let $decor                          := utillib:getDecor($projectPrefix)
 
    let $questionnaires                 := (
        for $transaction at $i in $transactions
        let $transactionId              := $transaction/@id
        let $transactionEffectiveDate   := $transaction/@effectiveDate

        let $progress                   := update value $request/@progress with 'Retrieving transaction ' || $i || ' of ' || $transactionCount || ' id ' || $transactionId || ' effectiveDate ' || $transactionEffectiveDate || ' ...'
        
        let $storedTransaction          := utillib:getTransaction($transactionId, $transactionEffectiveDate)
        let $check                      :=
            if (empty($storedTransaction)) then 
                error($errors:SERVER_ERROR, 'Transaction not found with id=' || $transactionId || ' effectiveDate=' || $transactionEffectiveDate)
            else ()
        
        let $progress                   := update value $request/@progress with 'Expanding dataset tree for transaction ' || $i || ' of ' || $transactionCount || ' id ' || $transactionId || ' effectiveDate ' || $transactionEffectiveDate || ' ...'
        
        (: getFullDatasetTree shall return name, desc etc in ALL languages (*) KH 20240531 :)
        let $fullDatasetTree            := utillib:getFullDatasetTree($storedTransaction, (), (), '*', (), false(), ())
        
        let $progress                   := update value $request/@progress-percentage with (5 + round((95 div ($transactionCount + 1)) * $i)) 
        let $progress                   := update value $request/@progress with 'Transforming transaction ' || $i || ' of ' || $transactionCount || ' id ' || $transactionId || ' effectiveDate ' || $transactionEffectiveDate || ' ...'
        let $newId                      := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-QUESTIONNAIRE, $baseId)/@max    + $i
        let $qqid                       := $baseId || '.' || $newId

        return utilsc:process-questionnairetransform($decor, $fullDatasetTree, $storedTransaction, $qqid)/*
        )
    
    let $updateQuestionnaire            := 
        if ($decor/rules) then update insert $questionnaires into $decor/rules
        else if ($decor/issues) then update insert <rules>{$questionnaires}</rules> preceding $decor/issues
        else update insert <rules>{$questionnaires}</rules> into $decor

    let $progress                       := update value $request/@progress-percentage with 100 
    let $progress                       := update value $request/@progress with 'Transforming ' || $transactionCount ||' transaction(s) done ...'
    
    let $remove                         := xmldb:remove(util:collection-name($request), util:document-name($request))

    return ()
};


(:~ Update representingTemplate
   @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
   @param $id                       - required. the id for the transaction to update 
   @param $effectiveDate            - required. the effectiveDate for the transaction to update 
   @param $request-body             - required. body containing new representingTemplate structure
   @return concept structure
   @since 2020-05-03
:)
declare function scapi:putRepresentingTemplate($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                           := utillib:getBodyAsXml($request?body, 'dataset', ())
    
    let $cloneFromTrxId                 := $request?parameters?cloneFromTransactionId[string-length() gt 0]
    let $cloneFromTrxEd                 := $request?parameters?cloneFromTransactionEffectiveDate[string-length() gt 0]
    let $cloneFromCptId                 := $request?parameters?cloneFromConceptId[string-length() gt 0]
    let $cloneFromCptEd                 := $request?parameters?cloneFromConceptEffectiveDate[string-length() gt 0]
    let $fullTree                       := $request?parameters?fulltree = true()
    let $deletelock                     := $request?parameters?deletelock = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                          :=
        if (empty($cloneFromTrxId) and empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either data or parameter cloneFromTransactionId that points to an existing transaction')
        else ()
        
    let $check                          :=
        if (not(empty($data)) and not(empty($cloneFromTrxId))) then
            error($errors:BAD_REQUEST, 'Request SHALL NOT have both data and parameter cloneFromTransactionId')
        else ()
    
    let $return                         := utilsc:putRepresentingTemplate($authmap, string($id), $effectiveDate, $data, $cloneFromTrxId, $cloneFromTrxEd, $cloneFromCptId, $cloneFromCptEd, $deletelock)
    let $result                         := utilsc:getTransaction($id, $effectiveDate, (), (), (), true(), $fullTree)
    
    return
        if (empty($result)) then () else (
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
        )

};

(:~ Update transaction statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap required. Map derived from token
    @param $id required. DECOR transaction/@id to update
    @param $effectiveDate required. DECOR transaction/@effectiveDate to update
    @param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function scapi:putTransactionStatus($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $recurse                        := $request?parameters?recurse = true()
    let $list                           := $request?parameters?list = true()
    let $statusCode                     := $request?parameters?statusCode
    let $versionLabel                   := $request?parameters?versionLabel
    let $expirationDate                 := $request?parameters?expirationDate
    let $officialReleaseDate            := $request?parameters?officialReleaseDate
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                        := utilsc:setTransactionStatus($authmap, $id, $effectiveDate, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
    
    return
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            (: max 2 arrays: one with error and one with success :)
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Clone a scenario transaction representingTemplate based on a new dataset version
    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR transaction/@id to update the contained representingTemplate
    @param $effectiveDate           - required. DECOR transaction/@effectiveDate to update
    @param $newDatasetId            - required. DECOR scenario xml element containing everything that should be in the updated scenario
    @param $newDatasetEffectiveDate - required. DECOR scenario xml element containing everything that should be in the updated scenario
    @return an intermediate representingTemplate element that might be checked by the user and then be used for an update of the existing scenario transaction representingTemplate.
:)
declare function scapi:smartCloneScenarioRepresentingTemplate($request as map(*)) {
    
    let $id                      := $request?parameters?id[string-length() gt 0]
    let $effectiveDate           := $request?parameters?effectiveDate[string-length() gt 0]
    let $newDatasetId            := $request?parameters?newDatasetId[string-length() gt 0]
    let $newDatasetEffectiveDate := $request?parameters?newDatasetEffectiveDate[string-length() gt 0]

    return utilsc:smartCloneRepresentingTemplate($id, $effectiveDate, $newDatasetId, $newDatasetEffectiveDate)
};

(:~ Retrieves DECOR transaction usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the transaction
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @return as-is or as compiled as JSON
    @since 2026-01-12
:)
declare function scapi:getTransactionUsage($request as map(*)) {
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $results                        := utilsc:getTransactionUsage($id, $effectiveDate)
    
    let $count                          := count($results)
    let $max                            := $count
    let $allcnt                         := $count
    
    return
        <list artifact="USAGE" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements(subsequence($results, 1, $max))
        }
        </list>
};