xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilds                 = "http://art-decor.org/ns/api/util-dataset";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace decorlib        = "http://art-decor.org/ns/api/decor" at "decor-lib.xqm";
import module namespace ruleslib        = "http://art-decor.org/ns/api/rules" at "rules-lib.xqm";
import module namespace histlib         = "http://art-decor.org/ns/api/history" at "history-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

declare %private variable $utilds:STATUSCODES-FINAL := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $utilds:VALUEDOMAIN-TYPE   := utillib:getDecorTypes()/DataSetValueType;
(:~ local debug 0 or 1 - or 2,... later :)
declare %private variable $utilds:DEBUG             := 0;

(:~ Central logic for updating an existing dataset
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR dataset/@id to update
    @param $effectiveDate   - required. DECOR dataset/@effectiveDate to update
    @param $data            - required. DECOR concept xml element containing everything that should be in the updated dataset
    @return concept object as xml with json:array set on elements
:)
declare function utilds:putDataset($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(dataset)) {

    let $editedDataset          := $data
    let $storedDataset          := utillib:getDataset($id, $effectiveDate)
    let $decor                  := $storedDataset/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($storedDataset) then () else (
            error($errors:BAD_REQUEST, 'Dataset with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    
    let $check                  := utilds:checkDatasetAccess($authmap, $decor)
    
    let $check                  :=
        if ($storedDataset/@statusCode = $utilds:STATUSCODES-FINAL) then
            if ($editedDataset/@statusCode = $utilds:STATUSCODES-FINAL) then    
                error($errors:BAD_REQUEST, concat('Dataset cannot be edited in status ', $storedDataset/@statusCode, '. ', if ($storedDataset/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            else ()
        else ()
    
    let $check                  :=
        if ($editedDataset[@id = $id] | $editedDataset[empty(@id)]) then () else (
            error($errors:BAD_REQUEST, concat('Dataset SHALL have the same id as the dataset id ''', $id, ''' used for updating. Found in request body: ', ($editedDataset/@id, 'null')[1]))
        )
    
    let $check                  :=
        if ($editedDataset[@effectiveDate = $effectiveDate] | $editedDataset[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Dataset SHALL have the same effectiveDate as the dataset effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($editedDataset/@effectiveDate, 'null')[1]))
        )
       
    let $errorRelationship      := if ($editedDataset/relationship) then ruleslib:checkRelationship($editedDataset/relationship, $storedDataset, 'update', ()) else ()
    let $check                  :=
        if ($errorRelationship) then 
            error($errors:BAD_REQUEST, string-join (($errorRelationship), ' '))
        else ()
    
    (: set and check a lock on every concept to be changed :)
    let $movelocks              :=
        for $concept in $editedDataset/descendant-or-self::concept[move] | $editedDataset/descendant-or-self::concept[edit/@mode='delete']
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        return utilds:checkDatasetConceptLock($authmap, $storedConcept)
    
    (: check every concept to be changed :)
    let $editlocks              := 
        for $concept in $editedDataset/descendant-or-self::concept[edit[not(@mode = 'delete')]]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        return 
        if ($storedConcept) then utilds:checkDatasetConceptForUpdate($authmap, $concept, $storedConcept) else (
            error($errors:SERVER_ERROR, 'Missing concept id=' || $concept/@id || ' effectiveDate=' || $concept/@effectiveDate)
        )
        
    (: save dataset history :)
    let $intention              := if ($storedDataset[@statusCode = 'final']) then 'patch' else 'version'
    let $history                := if ($storedDataset) then histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASET, $projectPrefix, $intention, $storedDataset) else ()
    
    (: update all changed concepts :)
    let $updateConcepts         := utilds:editDatasetConcepts($projectPrefix, $authmap, $editedDataset, $storedDataset)
    
    let $deleteLock             := update delete ($editlocks | $movelocks)
    
    (: update dataset (except already saved concepts) - and add associations and identifiers :)
    return utilds:editDataset($decor, $editedDataset, $storedDataset)
};

(:~ Central logic for retrieving DECOR dataset based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $treeonly                - optional boolean parameter to get the tree structure only if true
    @param $fullTree                - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @since 2020-05-03
:)
declare function utilds:getDataset($id as xs:string, $effectiveDate as xs:string?, $projectPrefixOrId as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?, $treeonly as xs:boolean?, $fullTree as xs:boolean?, $propertiesMap as map(*)?) {
        
    let $dataset                := 
        if ($treeonly) then 
            utillib:getDatasetTree($id, $effectiveDate, (), (), $fullTree, $propertiesMap) 
        else (
            utillib:getDataset($id, $effectiveDate, $projectPrefixOrId, $projectVersion[1], $projectLanguage[1])
        )
    
    for $ds in $dataset
    return
        element {name($ds)} {
            $ds/@*,
            $ds/name,
            $ds/desc,
            if ($ds[publishingAuthority] or $treeonly) then $ds/publishingAuthority else utillib:inheritPublishingAuthorityFromProject($ds/ancestor::decor)
            ,
            for $ocp in $ds/property
            return
                <property>
                {
                    $ocp/@*,
                    attribute { 'md5' } { utillib:getMD54nodeset($ocp) },
                    $ocp/node()
                }
                </property>
            ,
            if ($ds[copyright] or $treeonly) then $ds/copyright else utillib:inheritCopyright($ds)
            ,
            for $node in $ds/(* except (name | desc | publishingAuthority | property | copyright | issueAssociation | concept | recycle)) 
            return
                (: in case relationship already is good, no need to redo that :)
                if ($node[self::relationship][empty(@iStatusCode)]) then (
                    let $referredDataset    := if (string-length($node/@ref)=0) then () else utillib:getDataset($node/@ref, $node/@flexibility)
                    return
                        <relationship>
                        {
                            $node/@type, $node/@ref, $node/@flexibility,
                            if ($referredDataset) then (
                                $referredDataset/ancestor::decor/project/@prefix,
                                attribute iStatusCode {$referredDataset/@statusCode}, 
                                if ($referredDataset/@expirationDate) then attribute iExpirationDate {$referredDataset/@expirationDate} else (),
                                if ($referredDataset/@versionLabel) then attribute iVersionLabel {$referredDataset/@versionLabel} else (),
                                attribute iddisplay {utillib:getNameForOID($node/@ref, $referredDataset/ancestor::decor/project/@defaultLanguage, $referredDataset/ancestor::decor)},
                                attribute localInherit {$referredDataset/ancestor::decor/project/@prefix = $ds/ancestor::decor/project/@prefix},
                                $referredDataset/name
                            ) else ()
                        }
                        </relationship>
                )
                else $node
            ,
            (: we return just a count of issues and leave it up to the calling party to get those if required :)
            <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $ds/@id][@effectiveDate = $ds/@effectiveDate])}"/>,
            $dataset/concept,
            (: recycle bin if present after all concepts :)
            $dataset/recycle
        }
};

(:~ Central logic for retrieving DECOR dataset for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $communityName           - optional space separated list of community names. If omitted all communities are given
    @since 2023-11-10
:)
declare function utilds:getDatasetExtract($id as xs:string, $effectiveDate as xs:string?, $projectPrefixOrId as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?, $communityName as xs:string*) as element()* {
    
    let $dataset                := utillib:getDataset($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage)
    
    return
        if (empty($dataset)) then () else 
            let $datasetExtract := utillib:getDatasetExtract($dataset, $id, $effectiveDate, (), (), $projectVersion, $projectLanguage, $communityName)
            let $latestVersion  := utillib:getDataset($dataset/@id, ())/@effectiveDate = $dataset/@effectiveDate
            let $transactions   := $dataset/ancestor::decor//transaction[representingTemplate[@sourceDataset = $datasetExtract/@id][@sourceDatasetFlexibility = $datasetExtract/@effectiveDate]]
            let $transactions   := if ($latestVersion) 
                then $transactions | $dataset/ancestor::decor//transaction[representingTemplate[@sourceDataset = $datasetExtract/@id][not(@sourceDatasetFlexibility castable as xs:dateTime)]] 
                else $transactions
                
            return utillib:mergeDatasetWithUsage($datasetExtract, $transactions, $dataset/ancestor::decor/project/@prefix)    
};

(:~ Central logic for retrieving DECOR dataset usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function utilds:getDatasetUsage($id as xs:string, $effectiveDate as xs:string?) as element()* {

    let $datasetsAll            := utillib:getDataset($id, ())
    
    let $allTransactions        := utillib:getTransactionsByDataset($id, $effectiveDate)
    
    return
        for $item in $allTransactions
        let $trid               := $item/@id
        let $tred               := $item/@effectiveDate
        group by $trid, $tred
        return utillib:doTransactionAssociation($datasetsAll, $item[1])
};

(:~ Creates a list object with datasets
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $includeBBR              - optional. Include BBR datasets in the list. BBRs have to be declared in the project
    @param $scenariosonly           - optional boolean. If true only includes datasets bound in a scenario transaction
    @param $searchTerms             - optional array of string. Searches for datasets by name or ending with id
    @return all live repository/non-private datasets as JSON, all data sets for the given $projectPrefix or nothing if not found
    @since 2020-05-03
:)
declare function utilds:getDatasetList($projectPrefixOrId as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $includeBBR as xs:boolean, $scenariosonly as xs:boolean, $treetype as xs:string?, $searchTerms as xs:string*) as element(list) {
    
    let $datasets               := 
        if (empty($searchTerms)) then utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)/datasets/dataset else utilds:searchDataset($projectPrefixOrId, $searchTerms, $projectVersion, $projectLanguage, $includeBBR)
    
    let $check                  := utillib:checkDatasetConsistency($datasets)
    
    let $allcnt                 := count($datasets)
    
    let $datasets               := 
        (: if we aren't searching, and we are in 'current' project scope, and we are asked for inclusion of BBRs give also all live repository/non-private data sets :)
        if (empty($searchTerms) and not(empty($projectPrefixOrId)) and empty($projectVersion) and $includeBBR) then $datasets | utillib:getDecorByPrefix((), (), ())/datasets/dataset else $datasets
        
    let $datasets               :=
        if ($scenariosonly) then
            if (empty($projectPrefixOrId)) then $datasets[@id = ancestor::decor//representingTemplate/@sourceDataset]
            else (
                let $decor      := utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)
                return $datasets[@id = $decor//representingTemplate/@sourceDataset]
            )
        else $datasets
    
    let $datasets               :=
        for $ds in $datasets
        return
        element {name($ds)} {
            $ds/@*,
            if (empty($projectVersion)) then attribute ident {$ds/ancestor::decor/project/@prefix} else(),
            $ds/name,
            $ds/desc,
            if ($ds[@id]) then if ($ds[publishingAuthority]) then () else utillib:inheritPublishingAuthorityFromProject($ds/ancestor::decor) else (),
            $ds/(* except (name | desc))
         }
    
    let $datasetnames           := 
        for $name in $datasets/name
        let $versionLabel   := $name/../@versionLabel
        return string-join(($name, $versionLabel, $name/@language), '')
    
    let $datasetnameentries     :=
        for $datasetname in $datasetnames
        let $dsnm   := $datasetname
        group by $dsnm
        return map:entry($datasetname[1], count($datasetname))
    
    let $datasetnamemap         := map:merge($datasetnameentries)
    
    let $results                := 
        for $datasetById in $datasets
        let $ds-id              := $datasetById/@id
        group by $ds-id
        (:assumption is that newest comes after oldest. cheaper than date calculations:)
        order by ($datasetById[last()]/name[@language = $projectLanguage], $datasetById[last()]/name)[1], $ds-id descending
        return (
            for $dataset in $datasetById
            let $statusCode     := 
                if ($dataset/@statusCode) then $dataset/@statusCode else (
                    if ($dataset//concept[@statusCode = ('draft','new','review', 'pending')]) then 'draft' else 'final'
                )
            let $versionLabel   := $dataset/@versionLabel
            order by $dataset/@effectiveDate descending
            return
                <dataset>
                {
                    $dataset/@id,
                    $dataset/@effectiveDate,
                    attribute statusCode {$statusCode},
                    $dataset/@versionLabel,
                    $dataset/@expirationDate,
                    $dataset/@officialReleaseDate,
                    $dataset/@canonicalUri,
                    $dataset/@lastModifiedDate,
                    $dataset/@ident,
                    attribute uuid {util:uuid()}
                    ,
                    for $name in $dataset/name[exists(node())][.//text()[not(normalize-space() = '')]]
                    let $label      := if ($versionLabel) then concat(' (',$versionLabel,')') else ()
                    let $labeldate  := if ($dataset/@effectiveDate castable as xs:dateTime) then replace(format-dateTime(xs:dateTime($dataset/@effectiveDate),'[Y0001]-[M01]-[D01] [H01]:[m01]:[s01]', (), (), ()),' 00:00:00','') else ()
                    (: functional people find the date attached to the name in the selector/drop down list in the UI confusing
                        and it is not necessary when the name for a given language is unique enough. Often the effectiveDate
                        is not very helpful at all and rather random. So we generate a language dependent name specifically for 
                        the drop down selector that only concatenates the effectiveDate when the name for the given language is 
                        not unique in itself.
                    :)
                    let $selectorName   := string-join(($name, $versionLabel, $name/@language), '')
                    let $selectorName   := if (map:get($datasetnamemap, $selectorName) gt 1) then $name || $label || ' :: ' || $labeldate else $name || $label
                    
                    return
                        <name>
                        {
                            (: use this for the drop down selector :)
                            $name/(@* except @selectorName),
                            attribute selectorName {$selectorName},
                            $name/node()
                        }
                        </name>
                     ,
                     $dataset/(* except (name|concept|history|recycle))[exists(node())][.//text()[not(normalize-space() = '')]]
                }
                </dataset>
            )
    
    let $count                  := count($results)
        
    return
        <list artifact="DS" current="{$count}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            subsequence($results, 1, $count)
        }
        </list>
};

(: Central logic for creating a dataset, either empty or based on another dataset
    @param $projectPrefixOrId       - project to create this dataset in
    @param $targetDate              - If true invokes effectiveDate of the new dataset and concepts as [DATE]T00:00:00. 
                                    - If false invokes date + time [DATE]T[TIME] 
    @param $sourceId                - parameter denoting the id of a dataset to use as a basis for creating the new dataset
    @param $sourceEffectiveDate     - parameter denoting the effectiveDate of a dataset to use as a basis for creating the new dataset",
    @param $keepIds                 - Only relevant if source dataset is specified. If true, the new dataset will keep the same ids for the new dataset, and only update the effectiveDate
    @param $baseDatasetId           - Only relevant when a source dataset is specified and `keepIds` is false. This overrides the default base id for datasets in the project. The value SHALL match one of the projects base ids for datasets
    @param $baseConceptId           - Only relevant when a source dataset is specified and `keepIds` is false. This overrides the default base id for concepts in the project. The value SHALL match one of the projects base ids for concepts
    @param $templateAssociations    - Only relevant if source dataset is specified, and commit=true. 
                                    - If 'true' then template associations are generated as copy of existing template associations for the selected dataset concepts and are immediately committed in the project
                                        **Note that this immediately affects your project and that there is no undo possible!!**
                                    - If 'false' then template associations are generated and returned, but not committed to the project.
    @param $skipDeprecated          - Only relevant if source dataset is specified.
                                    - If 'true' then deprecated concepts in the base dataset/templateAssociations are included. This is recommended
                                    - If 'false' the deprecated concepts in the base dataset/templateAssociations are NOT included
                                        Note that cancelled, and rejected concept are never included as they are supposed to never have been published. 
                                        Deprecated concepts however have been published and (potentially) implemented.
    @param $testMode                - If you only want to try out what woould happen with all your selected parameters and don't want to commit the result just yet, this value should be true
    @return (empty) dataset object as xml with json:array set on elements
:)
declare function utilds:createDataset($authmap as map(*), $projectPrefixOrId as xs:string, $targetDate as xs:boolean, $sourceId as xs:string?, $sourceEffectiveDate as xs:string?, $keepIds as xs:boolean?, $baseDatasetId as xs:string?, $baseConceptId as xs:string?, $templateAssociations as xs:boolean?, $skipDeprecated as xs:boolean?, $testMode as xs:boolean?) {

    let $decor                  := utillib:getDecor($projectPrefixOrId)
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, 'Project ' || $projectPrefixOrId || ' does not exist')
        )
    
    let $projectPrefix          := ($decor/project/@prefix)[1]
    let $now                    := if ($targetDate) then substring(string(current-date()), 1, 10) || 'T00:00:00' else substring(string(current-dateTime()), 1, 19)
    let $sourceDataset          := if (empty($sourceId)) then () else utillib:getDataset($sourceId, $sourceEffectiveDate)
    
    let $check                  := utilds:checkDatasetAccess($authmap, $decor)
    
    let $baseDatasetId          :=
        if (empty($baseDatasetId)) then decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-DATASET)[1]/@id else
        if (decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-DATASET)/@id = $baseDatasetId) then $baseDatasetId else (
            error($errors:BAD_REQUEST, 'Dataset base id ' || $baseDatasetId || ' is not configured as baseId for datasets in this project. Expected one of: ' || string-join(decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-DATASET)/@id, ' '))
        )
    let $baseConceptId          :=
        if (empty($baseConceptId)) then decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPT)[1]/@id else
        if (decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPT)/@id = $baseConceptId) then $baseConceptId else (
            error($errors:BAD_REQUEST, 'Concept base id ' || $baseConceptId || ' is not configured as baseId for concepts in this project. Expected one of: ' || string-join(decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPT)/@id, ' '))
        )
    let $check                  :=
        if (empty($sourceId)) then () else if ($sourceDataset) then () else (
            error($errors:BAD_REQUEST, 'Source dataset based on id ' || $sourceId || ' and effectiveDate ' || $sourceEffectiveDate || ' does not exist. Cannot use this as source')
        )
    let $keepIds                := $keepIds = true()
    let $templateAssociations   := $templateAssociations = true() 
    let $skipDeprecated         := not($skipDeprecated = false())
    let $testMode               := $testMode = true()   
    
    (: decorlib:getNextAvailableIdP() returns <next base="1.2" max="2" next="{$max + 1}" id="1.2.3" type="DS"/> :)
    let $newDatasetId           := if ($keepIds) then $sourceDataset/@id else (decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-DATASET, $baseDatasetId)/@id)
    let $delmbaseId             := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPT, $baseConceptId)
    let $nextConceptId          := $delmbaseId/@id
    let $nextConceptId          := $delmbaseId/@id
    let $projectLanguage        := $decor/project/@defaultLanguage

    let $check                  :=
        if (string-length($newDatasetId) > 0 and string-length($nextConceptId) > 0 and string-length($projectLanguage) > 0) then () else (
            error($errors:BAD_REQUEST, 'Default base ID for datasets and/or concepts not set or project default language not set')
        )
    let $check                  :=
        if (utillib:getDataset($newDatasetId, $now)) then
            if ($keepIds and $targetDate) then
                error($errors:BAD_REQUEST, 'Cannot create new dataset. A dataset with id ''' || $newDatasetId || ''' and effectiveDate ''' || $now || ''' already exists. You cannot clone into a date based version more than once a day.')
            else (
                error($errors:BAD_REQUEST, 'Cannot create new dataset. A dataset with id ''' || $newDatasetId || ''' and effectiveDate ''' || $now || ''' already exists.')
            )
        else ()
        
    (: keep counter in db, so we make sure we don't issue the same id twice for sibling concepts :)
    let $insert                 := if ($decor/datasets[@nextConceptLeaf]) 
        then update value $decor/datasets/@nextConceptLeaf with xs:integer($delmbaseId/@max) + 1
        else update insert attribute nextConceptLeaf {xs:integer($delmbaseId/@max) + 1} into $decor/datasets
        
    let $result                 :=
        if ($sourceDataset) then 
            <dataset id="{$newDatasetId}" effectiveDate="{$now}" statusCode="new" lastModifiedDate="{$now}">
            {
                $sourceDataset/(* except concept)
                ,
                <relationship type="VERSION" ref="{$sourceDataset/@id}" flexibility="{$sourceDataset/@effectiveDate}"/>
                ,
                for $node in $sourceDataset/concept
                return
                    switch ($node/@statusCode) 
                    case 'pending'      return utilds:inheritConcept($node, $node/@statusCode, $baseConceptId, $decor, $now, $skipDeprecated, $keepIds)
                    case 'deprecated'   return if ($skipDeprecated) then () else utilds:inheritConcept($node, $node/@statusCode, $baseConceptId, $decor, $now, $skipDeprecated, $keepIds)
                    case 'cancelled'    return ()
                    case 'rejected'     return ()
                    default             return utilds:inheritConcept($node, 'draft', $baseConceptId, $decor, $now, $skipDeprecated, $keepIds)
            }
            </dataset>
        else (
            <dataset id="{$newDatasetId}" effectiveDate="{$now}" statusCode="draft" lastModifiedDate="{$now}">
                <name language="{$projectLanguage}">Dataset</name>
                <desc language="{$projectLanguage}">Dataset</desc>
                <concept id="{$nextConceptId}" effectiveDate="{$now}" type="item" statusCode="draft">
                    <name language="{$projectLanguage}">Concept</name>
                    <desc language="{$projectLanguage}">Concept</desc>
                    <valueDomain type="string"/>
                </concept>
            </dataset>
        )
        
    let $delete                 := update delete $decor/datasets/@nextConceptLeaf
        
    (: get rid of helper attribute //concept/(@oldId | @oldEffectiveDate) :)
    let $resultDataset          := utilds:copy-ds($result)
    let $updateDataset          := 
        if ($testMode) then () else 
        if ($decor/datasets) then
            update insert $resultDataset into $decor/datasets
        else (
            update insert <datasets>{$resultDataset}</datasets> following $decor/project
        )
            
    let $updateTemplateAssocs   := 
        if ($testMode or not($templateAssociations)) then () else (
            for $subconcept in $result/concept
            return utilds:add-template-assocs($decor, $subconcept, $templateAssociations)
        )

    return if ($testMode) then $resultDataset else utillib:getDataset($newDatasetId, $now)
};

(:~ Central logic for patching an existing dataset
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR dataset/@id to update
    @param $effectiveDate   - required. DECOR dataset/@effectiveDate to update
    @param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
    @return dataset object as xml with json:array set on elements
:)
declare function utilds:patchDataset($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) {

    let $storedDataset          := utillib:getDataset($id, $effectiveDate)
    let $decor                  := $storedDataset/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $check                  :=
        if ($storedDataset) then () else (
            error($errors:BAD_REQUEST, 'Dataset with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    
    let $lock                   := utilds:checkDatasetAccess($authmap, $decor, $id, $effectiveDate, true())
    
    let $check                  :=
        if ($storedDataset[@statusCode = $utilds:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $utilds:STATUSCODES-FINAL)]) then () else
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('Dataset cannot be patched while it has one of status: ', string-join($utilds:STATUSCODES-FINAL, ', '), if ($storedDataset/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
    
    let $intention              := if ($storedDataset[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASET, $projectPrefix, $intention, $storedDataset)
    
    let $update                 := utilds:patchDatasetParameters($data, $storedDataset)
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $preparedDataset        := utilds:correctDatasetElementSequence($storedDataset)
    
    let $update                 := update replace $storedDataset with $preparedDataset
    let $update                 := update delete $lock
    
    return ()
};

(:~ Central logic for patching an existing dataset statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR concept/@id to update
    @param $effectiveDate           - required. DECOR concept/@effectiveDate to update
    @param $recurse                 - optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $listOnly                - optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode           - optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel         - optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate       - optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate  - optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
:)
declare function utilds:setDatasetStatus($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $recurse as xs:boolean, $listOnly as xs:boolean, $newStatusCode as xs:string?, $newVersionLabel as xs:string?, $newExpirationDate as xs:string?, $newOfficialReleaseDate as xs:string?) as element()* {
    
    (:get object for reference:)
    let $object                 := utillib:getDataset($id, $effectiveDate)
    let $decor                  := $object/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($object) = 1) then () 
        else if ($object) then
            error($errors:SERVER_ERROR, 'Dataset id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($object) || ' were found in: ' || string-join(distinct-values($object/ancestor::decor/project/@prefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Dataset id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )
    
    let $check                  := utilds:checkDatasetAccess($authmap, $decor)
        
    let $testUpdate             := utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, true(), $projectPrefix)
    let $results                :=
        if ($testUpdate[self::error]) then $testUpdate 
        else if ($listOnly) then $testUpdate 
        else utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, false(), $projectPrefix)
        
    return
    if ($results[self::error] and not($listOnly)) then
        error($errors:BAD_REQUEST, string-join(
            for $e in $results[self::error]
            return
                $e/@itemname || ' id ''' || $e/@id || ''' effectiveDate ''' || $e/@effectiveDate || ''' cannot be updated: ' || data($e)
            , ' ')
        )
    else $results
};

declare function utilds:deleteDatasetLocks($authmap as map(*), $id as xs:string, $effectiveDate as xs:string) {
    
    let $dataset                := utillib:getDataset($id, $effectiveDate)
    let $decor                  := $dataset/ancestor::decor
    let $projectPrefix          := $dataset/ancestor::decor/project/@prefix
    
    let $check                  :=
        if (count($dataset) = 1) then () 
        else if ($dataset) then
            error($errors:SERVER_ERROR, 'Dataset id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($dataset) || ' were found in: ' || string-join(distinct-values($projectPrefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Dataset id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )
    
    let $check                  := utilds:checkDatasetAccess($authmap, $decor)

    let $clear                  := 
        for $lock in decorlib:getLocks($authmap, $dataset, true())
        let $rfid               := $lock/@ref
        let $rfed               := $lock/@effectiveDate
        return decorlib:deleteLock($authmap, $rfid, $rfed, ())
    
    (: delete everything thas has status new and does not have a lock on itself or one of its children :)
    return
        for $concept in $dataset//concept[@statusCode='new']
        let $locks              := decorlib:getLocks((), $concept, true())
        return if (empty($locks)) then update delete $concept else ()
};

declare %private function utilds:checkDatasetAccess($authmap as map(*), $decor as element(decor)) as element()? {
    utilds:checkDatasetAccess($authmap, $decor, (), (), false(), false())
};

declare %private function utilds:checkDatasetAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean) as element()? {
    utilds:checkDatasetAccess($authmap, $decor, $id, $effectiveDate, $checkLock, false())
};

declare %private function utilds:checkDatasetAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean, $breakLock as xs:boolean) as element()? {

    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-DATASETS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify datasets in project ', $decor/project/@prefix, '. You have to be an active author in the project.'))
        )

    return
        if ($checkLock) then (
            let $lock           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
            let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breakLock)
            return
                if ($lock) then $lock else (
                    error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this dataset (anymore). Get a lock first.'))
                )    
            )
            else ()
};

declare %private function utilds:checkDatasetConceptLock($authmap as map(*), $concept as element()?) as element()* {

    let $lock                   := decorlib:getLocks($authmap, $concept, false())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $concept, false())
    return
        if ($lock) then $lock else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this concept id=''' || $concept/@id || ''' and effectiveDate=''' || $concept/@effectiveDate || ''' (anymore). Get a lock first.'))
        )    
};

declare %private function utilds:checkDatasetConceptForUpdate($authmap as map(*), $editedConcept as element(concept), $storedConcept as element(concept)) as element()* {

    let $lock                   := utilds:checkDatasetConceptLock($authmap, $storedConcept)
    
    let $check                  :=
        if (empty($storedConcept)) then
            error($errors:BAD_REQUEST, 'Concept ' || $editedConcept/@id || ' effectiveDate ''' || $editedConcept/@effectiveDate || ''' SHALL exist in stored dataset on the server. Cannot update non-existent concept.')
        else if (count($storedConcept) = 1) then ()
        else (
            error($errors:SERVER_ERROR, 'Concept ' || $editedConcept/@id || ' effectiveDate ''' || $editedConcept/@effectiveDate || ''' gave ' || count($storedConcept) || ' stored concepts. Expected exactly 1. Found concept in projects:' || string-join(for $s in $storedConcept return $s/ancestor::decor/project/@prefix,' / '))
        )
    
    let $check                  :=
        if (utillib:isStatusChangeAllowable($storedConcept, $editedConcept/@statusCode)) then () else (
            error($errors:BAD_REQUEST, 'Concept ' || $editedConcept/@id || ' SHALL have the same statusCode ''' || $storedConcept/@statusCode || ''' or a supported status change. Found in request body: ''' || ($editedConcept/@statusCode, 'null')[1] || '''')
        )
    
    let $check                  := ruleslib:checkConcept($editedConcept, $storedConcept, 'update')
    
    return $lock
};

declare %private function utilds:editDatasetConcepts($projectPrefix as xs:string, $authmap as map(*), $dataset as element(dataset), $storedDataset as element(dataset)) {

    let $debug                  := if ($utilds:DEBUG > 0) then util:log('INFO', 'putDataset: delete concepts ' || count($dataset//concept[edit/@mode='delete'])) else ()
    
    let $deletes                :=
        for $concept in $dataset//concept[edit/@mode='delete']
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        let $history            := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, 'delete', $storedConcept)
        return utilds:deleteDatasetConcept($concept, $storedConcept, $dataset)
    
    let $debug                  := if ($utilds:DEBUG > 0) then util:log('INFO', 'putDataset: move concepts ' || count($dataset//concept[move])) else ()
    
    (: first do all the history of the moves before the database is changed :)
    let $moveHistory            :=
        for $concept in $dataset//concept[move]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        return utilds:historyOfMovedDatasetConcept($projectPrefix, $authmap, $concept, $storedConcept, $storedDataset)
    
    let $moves                  :=
        for $concept in $dataset//concept[move]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        return utilds:moveDatasetConcept($concept, $storedConcept, $storedDataset)
    
    let $debug                  := if ($utilds:DEBUG > 0) then util:log('INFO', 'putDataset: update concepts ' || count($dataset//concept[edit/@mode='edit'][not(move)])) else ()
    
    let $updates                :=
        for $concept in $dataset//concept[edit/@mode='edit'][not(move)]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]        
        let $intention          := if ($storedDataset[@statusCode = 'final']) then 'patch' else 'version'
        let $history            := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, $intention, $storedConcept)
        return update replace $storedConcept with utilds:prepareDatasetConceptForUpdate($concept, $storedConcept)
    
    return ()
};

(: deleting concepts:
       if statusCode=new     delete concept
       if statusCode!=new    set statusCode=deprecated
:)
declare %private function utilds:deleteDatasetConcept($concept as element(concept), $storedConcept as element(concept), $dataset as element(dataset)) {

    if ($dataset/@statusCode != 'new') then (
        if ($storedConcept/@statusCode = 'new') then update delete $storedConcept
        else if ($concept/@type='item') then update replace $storedConcept with utillib:prepareItemForUpdate($concept, $storedConcept)
        else if ($concept/@type='group') then update replace $storedConcept with utillib:prepareGroupForUpdate($concept, $storedConcept)
        else ()
    ) 
    else update delete $storedConcept
};

(: a moved concept does not go to history but the group of concepts is touched by a move (concepts in or out) they need to go to history as they changed :)
declare %private function utilds:historyOfMovedDatasetConcept($projectPrefix as xs:string, $authmap as map(*), $concept as element(concept), $storedConcept as element(concept), $storedDataset as element(dataset)) {

    let $sourceIntention        := if ($concept[edit/@mode = 'edit']) then 'move/version' else 'move'
    let $sourceHistory          := $storedConcept/parent::concept[not(@statusCode = 'new')]
        
    let $history                := if ($sourceHistory) then histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, $sourceIntention, $sourceHistory) else ()   
    
    let $destHistory            := 
        if ($concept[preceding-sibling::concept]) 
        then $storedDataset//concept[@id = $concept/@id]/parent::concept[not(@statusCode = 'new')]
        else if ($concept[parent::concept] and $storedDataset//concept[@id = $concept/parent::concept/@id]) 
        then $storedDataset//concept[@id = $concept/parent::concept/@id][not(@statusCode = 'new')]
        else ()
    
    let $history                := if ($destHistory) then histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, $sourceIntention, $destHistory) else ()

    return ()
};

declare %private function utilds:prepareDatasetConceptForUpdate($concept as element(concept), $storedConcept as element(concept)) as element(concept){
    
    let $preparedConcept        :=
        if ($concept[@type = 'item']) then
            if ($concept[edit/@mode='edit']) then utillib:prepareItemForUpdate($concept, $storedConcept) else utillib:prepareItemForUpdate($storedConcept, $storedConcept)
        else if ($concept[@type = 'group']) then
            if ($concept[edit/@mode='edit']) then utillib:prepareGroupForUpdate($concept, $storedConcept) else utillib:prepareGroupForUpdate($storedConcept, $storedConcept)
        else ()
    
    return
        if (count($preparedConcept) = 1) then $preparedConcept else (
            error($errors:SERVER_ERROR, concat('Concept id=''',$concept/@id,''' and effectiveDate=''',$concept/@effectiveDate,''' has unknown type ',$concept/@type,'. Expected item or group.'))
        )
};

(:  move concepts
    - local move
      - delete stored concept
      - insert moved concept after editedDataset preceding-sibling or into parent dataset/concept
    - move to other dataset (unsupported!!)
:)
declare %private function utilds:moveDatasetConcept($concept as element(concept), $storedConcept as element(concept), $storedDataset as element(dataset)) {
    
    let $preparedConcept        := utilds:prepareDatasetConceptForUpdate($concept, $storedConcept)
    
    (:  if move + edit: use what was sent to us
        if just move: use what we already had stored on the server
    :)
    
    return
        if ($concept[preceding-sibling::concept]) 
        then update insert $preparedConcept following $storedDataset//concept[@id = $concept/preceding-sibling::concept[1]/@id]
        else if ($concept[parent::dataset]) then (
            if ($storedDataset[concept | history]) 
            then update insert $preparedConcept preceding ($storedDataset/concept | $storedDataset/history)[1]
            else update insert $preparedConcept into $storedDataset
        )
        else if ($concept[parent::concept] and $storedDataset//concept[@id = $concept/parent::concept/@id]) then (
            let $storedParent   := $storedDataset//concept[@id = $concept/parent::concept/@id]
            return 
                if ($storedParent[concept | history]) then update insert $preparedConcept preceding ($storedParent/concept | $storedParent/history)[1] else update insert $preparedConcept into $storedParent
        ) else (
            error(QName('http://art-decor.org/ns/art/dataset', 'ContextNotFound'), concat('Could not determine context for saving concept with id ',$concept/@id,' and name ',$concept/name[1]))
        ),
        update delete $storedConcept
};

declare %private function utilds:prepareDatasetForUpdate($dataset as element(dataset), $storedDataset as element(dataset)) as element(dataset) { 

    <dataset id="{$dataset/@id}" effectiveDate="{$dataset/@effectiveDate}" statusCode="{$dataset/@statusCode}">
    {
        $dataset/@versionLabel[string-length() gt 0],
        $dataset/@expirationDate[string-length() gt 0],
        $dataset/@officialReleaseDate[string-length() gt 0],
        $dataset/@canonicalUri[string-length() gt 0],
        attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)},
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($dataset/name),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($dataset/desc),
        utillib:preparePublishingAuthorityForUpdate($dataset/publishingAuthority[empty(@inherited)]),
        utillib:prepareConceptPropertyForUpdate($dataset/property),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($dataset/copyright[empty(@inherited)]),
        utillib:prepareDatasetRelationshipForUpdate($dataset/relationship),
        $storedDataset/concept,
        $storedDataset/recycle
    }
    </dataset>
};

declare %private function utilds:editDataset($decor as element(decor), $dataset as element(dataset), $storedDataset as element(dataset)) {

    let $debug                  := if ($utilds:DEBUG > 0) then util:log('INFO', 'putDataset: update dataset' || count($storedDataset)) else ()
    
    let $preparedDataset        := utilds:prepareDatasetForUpdate($dataset, $storedDataset)
    
    (: check on scheme, if it fails the concepts are already changed ???   :)
    let $check                  := ruleslib:checkDecorSchema($preparedDataset, 'update')
    
    let $update                 := update replace $storedDataset with $preparedDataset
    
    (:store any terminologyAssociations that were saved in the dataset after deInherit
        only store stuff where the concept(List) still exists
    :)
    let $terminologyAssociations        := ($dataset//terminologyAssociation[ancestor::concept/edit/@mode='edit'] | $dataset//terminologyAssociation[ancestor::concept[move]])
    (:let $check                          := error($errors:SERVER_ERROR, 'Found ' || count($terminologyAssociations) || ' terminologyAssociations'):)
    let $terminologyAssociationUpdates  := utillib:addTerminologyAssociations($terminologyAssociations, $decor, false(), false())
    
    (:store any identifierAssociations that were saved in the dataset after deInherit
        only store stuff where the concept(List) still exists
    :)
    let $identifierAssociations         := ($dataset//identifierAssociation[ancestor::concept/edit/@mode='edit'] | $dataset//identifierAssociation[ancestor::concept[move]])
    (:let $check                          := error($errors:SERVER_ERROR, 'Found ' || count($identifierAssociations) || ' identifierAssociations'):)
    let $identifierAssociationUpdates   := utillib:addIdentifierAssociations($identifierAssociations, $decor, false(), false())
    
    return ()
};    

declare %private function utilds:searchDataset($projectPrefixOrId as xs:string?, $searchTerms as xs:string*, $projectVersion as xs:string?, $language as xs:string?, $includeBBR as xs:boolean) as element(dataset)* {
    
    let $queryOnId              := if (count($searchTerms)=1 and matches($searchTerms[1],'^\d+(\.\d+)*$')) then true() else false()
    
    let $results                := 
        if ($queryOnId) then (
            if ($projectPrefixOrId = '*') then 
                $setlib:colDecorData//dataset[ends-with(@id, $searchTerms[1])][ancestor::decor[not(@private='true')]]
            else 
            if (empty($projectPrefixOrId)) then 
                $setlib:colDecorData//dataset[ends-with(@id, $searchTerms[1])][ancestor::decor[not(@private='true')][@repository = 'true']]
            else (
                utillib:getDecor($projectPrefixOrId, $projectVersion, $language)//dataset[ends-with(@id, $searchTerms[1])]
            )
        ) 
        else
        if (empty($searchTerms)) then (
            let $datasets       := 
                if (empty($projectVersion)) then
                    if ($projectPrefixOrId = '*') then $setlib:colDecorData//dataset[ancestor::decor[not(@private='true')]]
                    else if ($includeBBR) then $setlib:colDecorData//dataset[ancestor::decor[not(@private='true')][@repository = 'true']]
                    else ()
                else ()
            
            return if ($projectPrefixOrId = '*' or empty($projectPrefixOrId)) then $datasets else $datasets | utillib:getDecor($projectPrefixOrId, $projectVersion, $language)//dataset
        ) 
        else (
            let $luceneQuery    := utillib:getSimpleLuceneQuery($searchTerms)
            let $luceneOptions  := utillib:getSimpleLuceneOptions()
            let $datasets       := 
                if (empty($projectVersion)) then
                    if ($projectPrefixOrId = '*') then $setlib:colDecorData//dataset[ft:query(name, $luceneQuery)][ancestor::decor[not(@private='true')]]
                    else if ($includeBBR) then $setlib:colDecorData//dataset[ft:query(name, $luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true']]
                    else ()
                else ()
            
            return if ($projectPrefixOrId = '*' or empty($projectPrefixOrId)) then $datasets else $datasets | utillib:getDecor($projectPrefixOrId, $projectVersion, $language)//dataset[ft:query(name, $luceneQuery)]
        )
        
    (:shortest match first:)
    return
        for $r in $results
        let $displayName        := if ($r/name[@language = $language]) then $r/name[@language = $language][1] else $r/name[1]
        order by string-length($displayName)
        return $r
};

(:~ Create new concept that inherits from input concept 
    if keep ids then only the effective date is set to now
    if not keep ids then 
        (1) either new ids are generated (later) based on base id for datalelements
        (2) or if the format of the id is datalement.version.itemid then the new id is datalement.version+1.itemid
:)
declare %private function utilds:inheritConcept($concept as element(), $statusCode as xs:string, $baseId as xs:string, $decor as element(decor), $now as xs:string, $skipDeprecated as xs:boolean, $keepIds as xs:boolean) as item()* {

    let $deid                   := $concept/@id
    let $deed                   := $concept/@effectiveDate
    
    let $nextdelmLeaf           := $decor/datasets/@nextConceptLeaf
    let $id                     := if ($keepIds) then $deid else concat($baseId, '.', $nextdelmLeaf)
    let $nextid                 := update value $nextdelmLeaf with xs:integer($nextdelmLeaf) + 1
    
    let $originalConcept        := utillib:getOriginalForConcept($concept)
    
    let $nme                    := $originalConcept/name[1]
    
    (: Use id of concept this concept inherited from if applicable, else use id :)
    (:let $inheritId          := if ($concept/inherit) then ($concept/inherit/@ref) else ($cid):)
    (: Use effectiveDate of concept this concept inherited from if applicable, else use effectiveDate :)
    (:let $inheritEff         := if ($concept/inherit) then ($concept/inherit/@effectiveDate) else ($eff):)
    return (
        comment {'Inherits from: ',$nme,' (status:',$concept/@statusCode/string(),', type:',$originalConcept/@type/string(),')'},
        <concept id="{$id}" statusCode="{$statusCode}" effectiveDate="{$now}" oldId="{$deid}" oldEffectiveDate="{$deed}">
            <inherit ref="{$deid}" effectiveDate="{$deed}"/>
        {
            for $subConcept in $concept/concept
            return
                switch ($subConcept/@statusCode) 
                case 'deprecated'   return if ($skipDeprecated) then () else utilds:inheritConcept($subConcept, $subConcept/@statusCode, $baseId, $decor, $now, $skipDeprecated, $keepIds)
                case 'cancelled'    return ()
                case 'rejected'     return ()
                default             return utilds:inheritConcept($subConcept, 'draft', $baseId, $decor, $now, $skipDeprecated, $keepIds)
        }
        </concept>
    )
};

(:~ Strip extra attributes oldId | oldEffectiveDate :)
declare %private function utilds:copy-ds($input as node()) {
    
    let $xslt := 
        <xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="2.0">
             <xsl:template match="concept">
                <xsl:copy>
                     <xsl:apply-templates select="@* except (@oldId | @oldEffectiveDate)"/>
                     <xsl:apply-templates select="node()"/>
                </xsl:copy>
             </xsl:template>
             <xsl:template match="@*|node()">
                 <xsl:copy>
                     <xsl:apply-templates select="@*|node()"/>
                 </xsl:copy>
             </xsl:template>
         </xsl:stylesheet>
    
    return transform:transform($input, $xslt, ())
};

(:
<templateAssociation templateId="2.16.840.1.113883.2.4.3.36.10.900804" effectiveDate="2012-07-04T00:00:00">
    <concept ref="2.16.840.1.113883.2.4.3.36.77.2.1.150050" effectiveDate="2012-04-10T00:00:00" elementId="2.16.840.1.113883.2.4.3.36.10.900804.1"/>
    <concept ref="2.16.840.1.113883.2.4.3.36.77.2.2.150050" effectiveDate="2013-11-03T00:00:00" elementId="2.16.840.1.113883.2.4.3.36.10.900804.1"/>
    <concept ref="2.16.840.1.113883.2.4.3.36.77.2.3.150050" effectiveDate="2014-07-12T00:00:00" elementId="2.16.840.1.113883.2.4.3.36.10.900804.1"/>
</templateAssociation>
:)
declare %private function utilds:add-template-assocs($decor as node(), $concept as node(), $commitUpdate as xs:boolean) as item()* {
    
    for $templateAssociation in $decor//templateAssociation[concept[@ref=$concept/@oldId][@effectiveDate=$concept/@oldEffectiveDate]]
    return (
        <templateAssociation>
        {
            $templateAssociation/@*,
            for $association in $templateAssociation/concept[@ref=$concept/@oldId][@effectiveDate=$concept/@oldEffectiveDate]
            let $newAssociation     :=
                <concept ref="{$concept/@id}" effectiveDate="{$concept/@effectiveDate}">
                {
                    $association/(@* except (@ref | @effectiveDate))
                }
                </concept>
            let $dummy1 := if ($commitUpdate) then (update insert $newAssociation following $templateAssociation/*[last()]) else ()
            return $newAssociation
        }
        </templateAssociation>
    )
    ,
    for $subconcept in $concept/concept
    return utilds:add-template-assocs($decor, $subconcept, $commitUpdate)
};

declare %private function utilds:checkDatasetParameters($parameters as element(parameters), $dataset as element(dataset)*) {

    let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    
    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $checkn                 := (
        if ($parameters/parameter[@path = '/name']) then
            if (count($dataset/name) - count($parameters/parameter[@op = 'remove'][@path = '/name']) + count($parameters/parameter[@op = ('add', 'replace')][@path = '/name']) ge 1) then () else (
                'A dataset SHALL have at least one name. You cannot remove every name.'
            )
        else (),
        if ($parameters/parameter[@path = '/desc']) then
            if (count($dataset/desc) - count($parameters/parameter[@op = 'remove'][@path = '/desc']) + count($parameters/parameter[@op = ('add', 'replace')][@path = '/desc']) ge 1) then () else (
                'A dataset SHALL have at least one description. You cannot remove every desc.'
            )
        else ()
    )
    
    let $check                  :=
        for $param in $parameters/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($dataset, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($dataset/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri'
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/name' 
            case '/desc' 
            case '/comment'
            case '/copyright' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $pathpart], $op, $path) else (
                    'Parameter ' || $op || ' of path ' || $path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/publishingAuthority' return (
                if ($param[count(value/publishingAuthority) = 1]) then ruleslib:checkPublishingAuthority($param/value/publishingAuthority, $op, $path) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single publishingAuthority under value. Found ' || count($param/value/publishingAuthority)
                )
            )
            case '/property' return (
                if ($param[count(value/property) = 1]) then ruleslib:checkProperty($param/value/property, $op, $path) else (
                    'Parameter ' || $op || ' of path ' || $path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/property) 
                )
            )
            case '/relationship' return (
                if ($op = 'remove') then () else ruleslib:checkRelationship($param/value/relationship, $dataset, $op, $path)
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     
     return
        if (empty($checkn) and empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join (($checkn, $check), ' '))
        )

};

declare %private function utilds:patchDatasetParameters($parameters as element(parameters), $dataset as element(dataset)*) {

    for $param in $parameters/parameter
    return
        switch ($param/@path)
        case '/statusCode'
        case '/expirationDate'
        case '/officialReleaseDate'
        case '/canonicalUri'
        case '/versionLabel' return (
            let $attname  := substring-after($param/@path, '/')
            let $new      := attribute {$attname} {$param/@value}
            let $stored   := $dataset/@*[name() = $attname]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $dataset
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/name' 
        case '/desc' return (
            (: only one per language :)
            let $elmname  := substring-after($param/@path, '/')
            let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
            let $stored   := $dataset/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $dataset
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/relationship' return (
            let $new      := utillib:prepareDatasetRelationshipForUpdate($param/value/relationship)
            let $stored   := $dataset/relationship[@ref = $new/@ref]
            let $stored   := if ($new/@flexibility) then $stored[@flexibility = $new/@flexibility] else $stored
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $dataset
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/publishingAuthority' return (
            (: only one possible :)
            let $new      := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
            let $stored   := $dataset/publishingAuthority
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $dataset
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/property' return (
            (: multiple possible :)
            (: prepare the new property element in parameter :)
            let $new            := utillib:prepareConceptPropertyForUpdate($param/value/property)
            (:
            let $check          := util:log('INFO', 'paramtr op ' || $param/@op)
            let $check          := util:log('INFO', 'concept: #properties=' || count($concept/property))
            :)
            (:
                find the correct stored concept/property elements by creating md5 for all already exsting concept/property 
                elements and add, replace or remove according to the parameters, build the return tree at the same time
            :)
            let $stored         := (
                for $c in $dataset/property
                let $md5        := utillib:getMD54nodeset($c)
                (:
                let $check      := util:log('INFO', 'dataset: property md5 ' || $md5 || ' paramtr md5 ' || $param/value/property/@md5)
                :)
                return
                    if ($md5 = $param/value/property/@md5)
                    then 
                        (: 
                            we found a matching element with what the front end has sent
                            if $c to be replaced use $new, 
                            if to be removed delete it, otherwise return unaltered $c 
                        :)
                        switch ($param/@op)
                        case ('replace') return ( update replace $c with $new )
                        case ('remove')  return update delete $c
                        default return $c                    
                    else
                        (:
                            no match in md5, this is a foreign element = simply copy what we have and leave it
                        :)
                        $c,
                 (: 
                    finally add a possible extra new element,
                    if @op = add add this as new element into the concept
                    if @op=replace + no md5 in parameter add this as new element into the concept
                 :)
                 switch ($param/@op)
                 case ('add') return ( update insert $new into $dataset )
                 case ('replace') return if ($param/value/property/@md5) then () else ( update insert $new into $dataset )
                 default return ()
            )
            (:
            let $check          := util:log('INFO', 'dataset stored: ' || string-join($stored/@md5, '-'))
            let $check          := util:log('INFO', 'concept: new #properties=' || count($stored))
            let $check          := util:log('INFO', 'paramtr: md5=' || $param/value/property/@md5 || ' att=' || string-join($param/value/property/@*, ', '))
            :)
            return $stored
        )
        case '/copyright' return (
            (: only one per language :)
            let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/copyright)
            let $stored   := $dataset/copyright[@language = $param/value/copyright/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $dataset
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        default return ( (: unknown path :) )
    
};

declare %private function utilds:correctDatasetElementSequence($dataset as element(dataset)) as element(dataset) {
    <dataset>
    {
        $dataset/@id,
        $dataset/@effectiveDate,
        $dataset/@statusCode,
        $dataset/@versionLabel[not(. = '')],
        $dataset/@expirationDate[not(. = '')],
        $dataset/@officialReleaseDate[not(. = '')],
        $dataset/@canonicalUri[not(. = '')],
        $dataset/@lastModifiedDate[not(. = '')], 
        $dataset/name,
        $dataset/desc,
        $dataset/publishingAuthority,
        $dataset/property,
        $dataset/copyright,
        $dataset/relationship,
        $dataset/concept,
        $dataset/recycle
    }
    </dataset>
};