xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Value set API allows read, create, update of DECOR codeSystems :)
module namespace csapi              = "http://art-decor.org/ns/api/codesystem";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utilcs      = "http://art-decor.org/ns/api/util-codesystem" at "library/util-codesystem-lib.xqm";
import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace utilhtml    = "http://art-decor.org/ns/api/util-html" at "library/util-html-lib.xqm";
import module namespace utilgg      = "http://art-decor.org/ns/api/util-governancegroup" at "library/util-governancegroup-lib.xqm";

import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";

declare namespace json              = "http://www.json.org";

declare %private variable $csapi:PROGRESS-CSCADTS-10    := 'Starting ...';
declare %private variable $csapi:PROGRESS-CSCADTS-20    := 'Working on ';

(:~ Retrieves all DECOR codeSystem for publication based on project $id (oid) or prefix
    @param $project                 - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $language                - optional parameter 
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @return as-is or as compiled as JSON
    @since 2024-11-14
:)
declare function csapi:getCodeSystemExtractList($request as map(*)) {
    
    let $projectPrefixOrId              := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $format                         := $request?parameters?format[not(. = '')]
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json', 'text/csv', 'text/sql', 'application/*', '*/*')])[1]
    
        (: when you do a right mouse click for downloading content in Edge or Chrome, you effectively do a GET without Accept header :)
    let $check                  := 
        if (empty($acceptTypes) and empty($format)) then
            error($errors:BAD_REQUEST, 'Your request does not have an Accept header, and is missing a format parameter. Don''t know what to do')
        else ()
    
    (: parameter format overrides Accept header  :)
    let $format                         := if ($format) then $format else tokenize($acceptedType, '/')[2]
    
    let $check                          :=
        if (empty($projectPrefixOrId)) then 
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    
    let $projectPrefix                  := utillib:getDecor($projectPrefixOrId)/project/@prefix
   
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Project ' || $projectPrefixOrId || ' not found.')
        else ()

    (: Return zero or more expanded codesystems wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element. :)
    
    let $codeSystems                    := 
        for $codeSystem in utilcs:getCodeSystemList((), $projectPrefix, $projectVersion, $projectLanguage, (), (), false(), (), (), (), false())/codeSystem/codeSystem
        return utilcs:getCodeSystemExtract($projectPrefix, $projectVersion, $projectLanguage, $codeSystem/(@id|@ref), $codeSystem/@effectiveDate, false())

    (: prepare response payload :)
    let $results                        :=
         if (empty($codeSystems/*)) then () 
         else (
            switch ($format)
            case 'xml'return <codeSystems>{$codeSystems/*}</codeSystems>
            case 'json' return (
                for $x in <codeSystems>{$codeSystems/*}</codeSystems>
                return
                 element {name($x)} {
                     $x/@*,
                     namespace {"json"} {"http://www.json.org"},
                     utillib:addJsonArrayToElements($x/node(), $format)
                 }
            )
            case 'csv' return utilcs:convertCodeSystem2Csv($codeSystems, $projectLanguage)
            case 'sql'return transform:transform($codeSystems, doc($setlib:strDecorServices ||'/resources/stylesheets/ToSql4ValueSets.xsl'), ()) 
            default return
                error($errors:BAD_REQUEST, 'Requested format ''' || string-join($format, ', ') || ''' not supported. Supported are: xml, json, csv, sql')
         )
            
    let $codeSystemNames                := distinct-values($codeSystems//codeSystem/@name)
    let $fileNamePart                   := if (count($codeSystemNames)>1) then 'project_' || $projectPrefix || '('|| count($codeSystems//codeSystem) || ')' else $codeSystemNames
    let $fileName                       := 'CS_' || $fileNamePart ||'_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $format
                
    (: determine response Content-Type/Media-type :)
    let $responseType                   := $utillib:formatacceptmap?($format)     

    let $r-header                       := 
        (response:set-header('Content-Type', $responseType || '; charset=utf-8'),
        response:set-header('Content-Disposition', 'filename='|| $fileName))

    return
        if (empty($results)) then roaster:response(404, ()) else (
            switch ($format)
            case 'csv' return $results
            default return roaster:response(200, $responseType, $results)
        )
};

(:~ Retrieves latest DECOR codeSystem for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the codeSystem
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $language                - optional
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @return as-is or as compiled as JSON
    @since 2023-11-14
:)
declare function csapi:getLatestCodeSystemExtract($request as map(*)) {
    csapi:getCodeSystemExtract($request)
};

(:~ Retrieves DECOR codeSystem for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the codeSystem
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the codeSystem. If not given assumes latest version for id
    @param $project                 - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $language                - optional parameter 
    @param $wrap                    - optional parameter to return a the codesystems wrapped in project - xml only
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @return as-is or as compiled as JSON
    @since 2023-11-17
:)
declare function csapi:getCodeSystemExtract($request as map(*)) {
    
    let $projectPrefixOrId              := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $withversions                   := $request?parameters?versions = true()
    let $format                         := $request?parameters?format[not(. = '')]
    let $download                       := $request?parameters?download=true()
    
    let $idOrName                       := $request?parameters?idOrName[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json', 'text/csv', 'text/sql', 'application/*', '*/*')])[1]
    
        (: when you do a right mouse click for downloading content in Edge or Chrome, you effectively do a GET without Accept header :)
    let $check                  := 
        if (empty($acceptTypes) and empty($format)) then
            error($errors:BAD_REQUEST, 'Your request does not have an Accept header, and is missing a format parameter. Don''t know what to do')
        else ()
    
    (: parameter format overrides Accept header  :)
    let $format                         := if ($format) then $format else tokenize($acceptedType, '/')[2]

    let $projectPrefix                  := 
        if (empty($projectPrefixOrId)) then () else utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)/project/@prefix            
    
    (: Return zero or more expanded codesystems wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element. :)
    let $codeSystems                    := if (empty($projectPrefix)) then utilcs:getCodeSystemExtract($idOrName, $effectiveDate) else utilcs:getCodeSystemExtract($projectPrefix, $projectVersion, $projectLanguage, $idOrName, $effectiveDate, $withversions)

    let $refInfo                        :=
                                          if ($codeSystems//codeSystem[@ref]) then
                                           let $info := concat($codeSystems//codeSystem[1]/@displayName,' (', $codeSystems//codeSystem[1]/@ref,')')
                                           return
                                              switch($projectLanguage)
                                                case "en-US" return concat('Code System ', $info,' is a reference, content is not available.')
                                                case "de-DE" return concat('Code System ',$info, ' ist eine Referenz, Inhalt ist nicht vorhanden.')
                                                case "nl-NL" return concat('Codesysteem ',$info,' is een referentie, inhoud is niet beschikbaar.')
                                                default return concat('Code System ', $info,' is a reference, content is not available.')
                                           else ()

    (: prepare response payload :)
    let $results                        :=
         if (empty($codeSystems/*)) then () 
         else (
            switch ($format)
            case 'xml'return <codeSystems>{$codeSystems/*}</codeSystems>
            case 'json' return (
                for $x in <codeSystems>{$codeSystems/*}</codeSystems>
                return
                 element {name($x)} {
                     $x/@*,
                     namespace {"json"} {"http://www.json.org"},
                     utillib:addJsonArrayToElements($x/node(), $format)
                 }
            )
            case 'csv' return 
                         if ($refInfo) then
                            $refInfo
                         else (utilcs:convertCodeSystem2Csv($codeSystems, $projectLanguage))
            case 'sql'return 
                         if ($refInfo) then
                            $refInfo
                         else (transform:transform($codeSystems, doc($setlib:strDecorServices ||'/resources/stylesheets/ToSql4ValueSets.xsl'), ()) )
            default return
                error($errors:BAD_REQUEST, 'Requested format ''' || string-join($format, ', ') || ''' not supported. Supported are: xml, json, csv, sql')
         )
         
    let $filename                       := 'CS_' || fn:distinct-values($codeSystems//codeSystem/@name)[1] ||'_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $format

    (: determine response Content-Type/Media-type :)
    let $responseType                   := $utillib:formatacceptmap?($format)     
                    
    let $r-header                       := 
        (response:set-header('Content-Type', $responseType || '; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
     return
        if (empty($results)) then roaster:response(404, ()) else (
            switch ($format)
            case 'csv' return $results
            default return roaster:response(200, $responseType, $results)
        )
};

(:~ Retrieves latest DECOR codeSystem view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the codeSystem
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language, also ui-language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2025-03-07
:)
declare function csapi:getLatestCodeSystemView($request as map(*)) {
    csapi:getCodeSystemView($request)
};

(:~ Retrieves DECOR codeSystem view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the codeSystem
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the codeSystem. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language, also ui-language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2024-10-14
:)
declare function csapi:getCodeSystemView($request as map(*)) {

    let $projectPrefixOrId              := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $seetype                        := $request?parameters?seetype[not(. = '')]
    let $inline                         := $request?parameters?inline = true()
    let $collapsable                    := not($request?parameters?collapsable = false())
    
    let $idOrName                       := $request?parameters?idOrName[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $decor                          := if (empty($projectPrefixOrId)) then () else utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)
            
    (: Return zero or more expanded codesystems wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element :)
    let $results                        := if (empty($decor/project/@prefix)) then utilcs:getCodeSystemExtract($idOrName, $effectiveDate) else utilcs:getCodeSystemExtract($decor/project/@prefix, $projectVersion, $projectLanguage, $idOrName, $effectiveDate, false())

    let $fileName                       := 'CS_' || fn:distinct-values($results//codeSystem/@name) ||'_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html'
            
    let $r-header                       := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        response:set-header('Content-Disposition', 'filename='|| $fileName))
    
    return
        if (empty($results/*)) then (
            roaster:response(404, ())
        )

        else 
        
        (: prepare for Html :)
        let $language                       := if (empty($projectLanguage)) then $setlib:strArtLanguage else $projectLanguage
        let $header                         := if ($inline) then false() else true()        
        
        return utilhtml:convertObject2Html($results, $language, $header, $collapsable, $projectVersion, $decor, $seetype)
       

};

(:~ Retrieves latest DECOR codeSystem based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the codeSystem
    @param $projectPrefix           - optional. limits scope to this project only
    @param $language                - optional
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function csapi:getLatestCodeSystem($request as map(*)) {
    csapi:getCodeSystem($request)
};

(:~ Retrieves DECOR codeSystem based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the codeSystem
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the codeSystem. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function csapi:getCodeSystem($request as map(*)) {
    
    let $projectPrefixOrId              := $request?parameters?prefix[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $withversions                   := $request?parameters?versions = true()
    
    let $results                        := utilcs:getCodeSystem($projectPrefixOrId, $projectVersion, $projectLanguage, $id, $effectiveDate, $withversions)
    let $results                        := 
        if ($withversions) then 
            <list artifact="CS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">{$results}</list> 
        else (
            head($results)
        )
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple codeSystems for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Retrieves DECOR codeSystem usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the codeSystem
    @param $effectiveDate           - required parameter denoting the effectiveDate of the codeSystem. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function csapi:getCodeSystemUsage($request as map(*)) {
    
    let $projectPrefixOrId              := $request?parameters?prefix
    let $projectVersion                 := $request?parameters?release
    let $projectLanguage                := $request?parameters?language
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $results                        := utilcs:getCodeSystemUsage($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage)
    
    let $count                          := count($results)
    let $max                            := $count
    let $allcnt                         := $count
    
    return
        <list artifact="USAGE" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements(subsequence($results, 1, $max))
        }
        </list>
};

(:~ Retrieves DECOR code system history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function csapi:getCodeSystemHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-CODESYSTEM, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-CODESYSTEM}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Returns a list of zero or more codeSystems
    @param $governanceGroupId - optional. determines search scope. null is full server, id limits scope to this projects under this governance group only. Does not mix with $projectPrefix
    @param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only. Does not mix with $governanceGroupId
    @param $projectVersion   - optional. if empty defaults to current version. if valued then the codeSystem will come explicitly from that archived project version which is expected to be a compiled version
    @param $projectLanguage  - optional. relevant in combination with $projectVersion if there are more than one languages the project is compiled in
    @param $id               - optional. Identifier of the codeSystem to retrieve
    @param $name             - optional. Name of the codeSystem to retrieve (codeSystem/@name)
    @param $flexibility      - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
    @param $treetype         - optional. Default $cs:TREETYPELIMITEDMARKED
    @param $doV2             - optional. Boolean. Default false. 
    @param $withversions     - optional. Boolean. Default true. If true and $doV2 is false, returns codeSystemList.codeSystem* where every codeSystem has @id|@ref and a descending by 
    effectiveDate list of matching codeSystems. If false and $doV2 is false, returns codeSystemList.codeSystem* containing only the latest version of the codeSystem (or ref if no versions 
    exist in $projectPrefix)<br/>If true and $doV2 is true, returns codeSystemList.codeSystem* where every codeSystem has @id|@ref and a descending by effectiveDate list of matching 
    codeSystems. If false and $doV2 is false, returns codeSystemList.codeSystem* containing only the latest version of the codeSystem (or ref if no versions exist in $projectPrefix)
    @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
    @since 2013-06-14
:)
declare function csapi:getCodeSystemList($request as map(*)) {

    let $governanceGroupId      := $request?parameters?governanceGroupId[not(. = '')]
    let $projectPrefixOrId      := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $max                    := $request?parameters?max
    let $resolve                := if (empty($governanceGroupId)) then not($request?parameters?resolve = false()) else $request?parameters?resolve = true()
    
    let $searchTerms            := 
        typeswitch ($request?parameters?search) 
        case xs:string return 
            array:flatten(
                for $s in ($request?parameters?search, $request?parameters?id, $request?parameters?name)[string-length() gt 0]
                return tokenize(normalize-space(lower-case($s)),'\s')
            )
        default return array:flatten(($request?parameters?search, $request?parameters?id, $request?parameters?name))
    
    let $ed                     := $request?parameters?effectiveDate[not(. = '')]
    let $includebbr             := $request?parameters?includebbr = true()
    let $sort                   := $request?parameters?sort[string-length() gt 0]
    let $sortorder              := $request?parameters?sortorder[. = 'descending']
    
    let $check                  :=
        if (empty($governanceGroupId) and empty($projectPrefixOrId)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter governanceGroupId or prefix')
        else 
        if (not(empty($governanceGroupId)) and not(empty($projectPrefixOrId))) then 
            error($errors:BAD_REQUEST, 'Request SHALL NOT have both parameter governanceGroupId or prefix, not both')
        else ()
    let $check                  :=
        if (empty($governanceGroupId)) then () else if ($resolve) then
            error($errors:BAD_REQUEST, 'Request SHALL NOT have governance group scope and resolve=true. This is too expensive to support')
        else ()
    
    let $startT                 := util:system-time()
    
    let $result                 :=
        if (empty($governanceGroupId)) then utilcs:getCodeSystemList($governanceGroupId, $projectPrefixOrId, $projectVersion, $projectLanguage, $searchTerms, $ed, $includebbr, $sort, $sortorder, $max, $resolve)
        else (
            for $projectId in utilgg:getLinkedProjects($governanceGroupId)/@ref
            return utilcs:getCodeSystemList($governanceGroupId, $projectId, (), (), $searchTerms, $ed, $includebbr, $sort, $sortorder, $max, $resolve)
        )
    
    let $durationT              := (util:system-time() - $startT) div xs:dayTimeDuration("PT0.001S")
    
    return
        <list artifact="{$result[1]/@artifact}" elapsed="{$durationT}" current="{sum($result/xs:integer(@current))}" total="{sum($result/xs:integer(@total))}" all="{sum($result/xs:integer(@all))}" resolve="{$resolve}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            if (empty($governanceGroupId)) then attribute project {$projectPrefixOrId} else attribute governanceGroupId {$governanceGroupId},
            utillib:addJsonArrayToElements($result/*)
        }
        </list>
};

(:~ Deletes a codeSystem reference based on $id (oid) and project id or prefix
    @param $id                      - required parameter denoting the id of the codeSystem
    @param $project                 - required. limits scope to this project only
:)
declare function csapi:deleteCodeSystem($request as map(*)) {
    
    let $authmap                := $request?user
    let $projectPrefixOrId      := $request?parameters?project[string-length() gt 0][not(. = '*')]
    let $id                     := $request?parameters?id[string-length() gt 0]
    
    let $check                  :=
        if (empty($projectPrefixOrId) or empty($id)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have both parameter project and id')
        else ()
    
    let $decor                  := utillib:getDecor($projectPrefixOrId)
    
    let $check                  :=
        if (empty($decor)) then
            error($errors:BAD_REQUEST, 'Project ' || $projectPrefixOrId || ' not found.')
        else ()
    
    let $check                  := utilcs:checkCodeSystemAccess($authmap, $decor)
    
    let $delete                 := update delete $decor/terminology/codeSystem[@ref = $id]
    
    return
        roaster:response(204, ())
};

(: Create a CodeSystem, either empty or based on another CodeSystem
    @param $authmap             - required. Map derived from token
    @param $projectPrefix       - project to create this codesystem in
    @param $targetDate          - if true invokes effectiveDate of the new codesystem and concepts as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
    @param $sourceId            - parameter denoting the id of a codesystem to use as a basis for creating the new codesystem
    @param $sourceEffectiveDate - parameter denoting the effectiveDate of a codesystem to use as a basis for creating the new codesystem",
    @param $keepIds             - only relevant if source codesystem is specified. If true, the new codesystem will keep the same ids for the new codesystem, and only update the effectiveDate
    @param $baseDatasetId       - only relevant when a source codesystem is specified and `keepIds` is false. This overrides the default base id for codesystem in the project. The value SHALL match one of the projects base ids for codesystem
    @return (empty) codesystem object as xml with json:array set on elements
:)
declare function csapi:postCodeSystem($request as map(*)) {

    let $authmap                := $request?user
    let $projectPrefixOrId      := $request?parameters?prefix[not(. = '')]
    let $targetDate             := $request?parameters?targetDate = true()
    let $sourceId               := $request?parameters?sourceId
    let $data                   := utillib:getBodyAsXml($request?body, 'codeSystem', ())
    let $sourceEffectiveDate    := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?sourceEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?sourceEffectiveDate[string-length() gt 0]
        }
    let $refOnly                := $request?parameters?refOnly = true()
    let $keepIds                := $request?parameters?keepIds = true()
    let $baseId                 := $request?parameters?baseId
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($projectPrefixOrId)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter prefix')
        else ()
    
    let $results                := utilcs:createCodeSystem($authmap, $projectPrefixOrId, $targetDate, $sourceId, $sourceEffectiveDate, $refOnly, $keepIds, $baseId, $data)
    
    return
        roaster:response(201, 
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        
        )
};

(:~ Update codeSystem statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR concept/@id to update
    @param $effectiveDate           - required. DECOR concept/@effectiveDate to update
    @param $listOnly                - optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode           - optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel         - optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate       - optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate  - optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function csapi:postCodeSystemStatus($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $list                           := $request?parameters?list = true()
    let $statusCode                     := $request?parameters?statusCode[not(. = '')]
    let $versionLabel                   := $request?parameters?versionLabel[not(. = '')]
    let $expirationDate                 := $request?parameters?expirationDate[not(. = '')]
    let $officialReleaseDate            := $request?parameters?officialReleaseDate[not(. = '')]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                        := utilcs:setCodeSystemStatus($authmap, $id, $effectiveDate, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
    
    return
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
            {
                utillib:addJsonArrayToElements($results)
            }
        </list>
};

(:~ Update codeSystem codedConcept statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR concept/@id to update
    @param $effectiveDate           - required. DECOR concept/@effectiveDate to update
    @param $listOnly                - optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode           - optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel         - optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate       - optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate  - optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function csapi:postCodeSystemCodedConceptStatus($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $code                           := $request?parameters?code[not(. = '')]
    let $recurse                        := $request?parameters?recurse = true()
    let $list                           := $request?parameters?list = true()
    let $statusCode                     := $request?parameters?statusCode[not(. = '')]
    let $versionLabel                   := $request?parameters?versionLabel[not(. = '')]
    let $expirationDate                 := $request?parameters?expirationDate[not(. = '')]
    let $officialReleaseDate            := $request?parameters?officialReleaseDate[not(. = '')]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    
    let $results                        := utilcs:setCodeSystemCodedConceptStatus($authmap, $id, $effectiveDate, $code, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
    
    return
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Update DECOR codeSystem parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

    { "op": "[add|remove|replace]", "path": "e.g. [/statusCode|/expirationDate|/officialReleaseDate|/canonicalUri|/versionLabel|/name|/displayName|/experimental|/desc|/publishingAuthority|/copyright|/completeCodeSystem|/conceptList]", "value": "[string|object]" }
    
    where
    
    * op - add & replace (statusCode, expirationDate, officialReleaseDate, canonicalUri, versionLabel, name, displayName, experimental, desc, publishingAuthority, copyright, completeCodeSystem, conceptList) or remove (desc, publishingAuthority, copyright, completeCodeSystem, conceptList)
    
    * path - see above
    
    * value - string when the path is not /. object when path is /
    
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the valueSet to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return codeSystem structure
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function csapi:patchCodeSystem($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()

    let $results                := utilcs:patchCodeSystem($authmap, string($id), string($effectiveDate), $data)
    
    return
        for $result in $results
        return
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
};

(: ======================= INTO CADTS ======================== :)
(:~ Convert all project code systems to Centralized ART-DECOR Terminology Services (CADTS), which makes them available for use in value set creation and expansions. Note that project code systems are left as-is. 
    The conversion only happens on the latest version of said code systems. Any code systems that are not the latest are skipped silently. Code systems that happen to match an externally managed system like SNOMED CT are skipped and reported as error
:)
declare function csapi:process-decorprojectcodesystemsincadts-request($threadId as xs:string, $request as element(projectcodesystemconvert-request)) {

    let $projectPrefixes        := tokenize($request/@for, '\s')[not(. = ('', '*'))]
    let $logsignature           := 'csapi:process-decorprojectcodesystemsincadts-request'
    let $progress-initial       := $csapi:PROGRESS-CSCADTS-10
    
    (: =================== PREPARE VARIABLES BEGIN ==================== :)
    let $codeSystems            := 
        if (empty($projectPrefixes)) then
            $setlib:colDecorCache//codeSystem[@id] | $setlib:colDecorData//codeSystem[@id]
        else 
        (: special case when you refresh the cache :)
        if ($projectPrefixes = '*cache*') then
            $setlib:colDecorCache//codeSystem[@id]
        else (
            $setlib:colDecorData//codeSystem[@id][ancestor::decor/project/@prefix = $projectPrefixes] | $setlib:colDecorData//codeSystem[@id][ancestor::decor/project/@id = $projectPrefixes]
        )
    let $codeSystems            :=
        for $csbyid at $i in $codeSystems[@statusCode = ('active', 'final', 'draft', 'pending', 'review')]
        let $csid               := $csbyid/@id
        group by $csid
        return
            head(
                for $cs in $csbyid 
                order by $cs/xs:dateTime(@effectiveDate) descending, util:collection-name($cs) descending 
                return $cs
            )
    let $csCount                := count($codeSystems)
    (: ===================  PREPARE VARIABLES END  ==================== :)
    
    let $mark-busy              := update insert attribute busy {'true'} into $request
    let $progress               := 
        if ($request/@progress) then (
            update value $request/@progress with $progress-initial 
        )
        else (
            update insert attribute progress {$progress-initial} into $request
        )
    let $progress               := 
        if ($request/@progress-percentage) then (
            update value $request/@progress-percentage with round((100 div ($csCount + 1)) * 1) 
        )
        else (
            update insert attribute progress-percentage {round((100 div ($csCount + 1)) * 1)} into $request
        )
    (: ========================= REMOVE EXISTING COLLECTION ============ :)
    
    let $removeExisting :=
        if (not(empty($projectPrefixes)) and not($projectPrefixes= '*cache*')) then
            for $prefix in $projectPrefixes
            return
                xmldb:remove($setlib:strCodesystemStableData || '/projects/' || $prefix)
        else()
    
    (: ========================= CONVERT BEGIN ========================= :)
    let $save               :=
        for $cs at $i in $codeSystems
        let $progress               := update value $request/@progress-percentage with round((100 div ($csCount + 1)) * $i) 
        let $progress               := update value $request/@progress with $csapi:PROGRESS-CSCADTS-20 || $i || ' of ' || $csCount 
        return (
            try { 
                let $r := utilcs:saveCodeSystemInCADTS($cs, $cs/ancestor::decor/project/@prefix, $cs/ancestor::decor/project/@defaultLanguage)
                return ()
            }
            catch * {
                let $cselement  := element {local-name($cs)} {$cs/@*, if ($cs[@ident]) then () else attribute ident {$cs/ancestor::decor[1]/project/@prefix}}
                let $save       := update insert $cselement into $request 
                
                return $err:description 
            }
        )
    (: ========================== CONVERT END ========================== :)
    
    return 
        if (empty($save)) then (
            xmldb:remove(util:collection-name($request), util:document-name($request))
        ) 
        else (
            error(xs:QName('csapi:process-decorprojectcodesystemsincadts-request'), 'One or more code systems were not converted successfully: ' || string-join($save, ' '))
        )
};

(:~ Retrieves the logo for a DECOR codeSystem based on $org denoting the Terminology Governance Group identifier string, such as "who" or "rsna" etc.
    @param $org                     - required parameter denoting the id string of the Terminology Governance Group
    @return governance group logo
    @since 2022-03-16
:)
declare function csapi:getTerminologyGoveranceGroupLogo($request as map(*)) {
    
    (: get the Terminology Governance Group identifier :)
    let $group         := $request?parameters?org[string-length() gt 0]
    
    let $check         :=
        if (empty($group[not(. = '')])) then
            error($errors:BAD_REQUEST, 'Missing required parameter org')
        else ()
    
    let $check         :=
        if ($group = ('adot', 'bfarm', 'hl7', 'hpo', 'iso', 'orphanet', 'regenstrief', 'rivm', 'rsna', 'snomed', 'who'))
        then ()
        else error($errors:BAD_REQUEST, 'Illegal value of parameter org')
        
    (: for now only default logos are returned with 40px in height :)
    let $defaultln     := "logo40.png"
    let $logosrc       := concat($setlib:strTerminologyLogo, '/', $group, '/', $defaultln)
    let $logofilename  := concat($group, '-', $defaultln)
    
    return
        if (util:binary-doc-available($logosrc)) then (
            (: instruct client to cache 7 days. which is 7 * 24 * 60 * 60 = 604800 seconds :)
            response:set-header("Cache-Control", "max-age=604800"),
            response:stream-binary(util:binary-doc($logosrc), 'image/png', $logofilename)
        ) else
            roaster:response(404, (), (), map { "Location": xs:anyURI($logosrc) })
};