(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Dataset concept API allows read, create, update on DECOR concepts in DECOR datasets :)
module namespace deapi              = "http://art-decor.org/ns/api/concept";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utilde      = "http://art-decor.org/ns/api/util-concept" at "library/util-concept-lib.xqm";
import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace utilhtml    = "http://art-decor.org/ns/api/util-html" at "library/util-html-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";

declare namespace json              = "http://www.json.org";

(:~ Retrieves latest DECOR concept based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                          - required parameter denoting the id of the concept
    @param $effectiveDate               - optional parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $transactionId               - required parameter denoting the id of the transaction that the concept is in
    @param $transactionEffectiveDate    - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion              - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage             - optional parameter to select from a specific compiled language
    @param $treeonly                    - optional boolean parameter to get the tree structure only if true
    @param $fullTree                    - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function deapi:getLatestConcept($request as map(*)) {
    deapi:getConcept($request)
};

(:~ Retrieves exact DECOR concept based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                          - required parameter denoting the id of the concept
    @param $effectiveDate               - required parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $transactionId               - required parameter denoting the id of the transaction that the concept is in
    @param $transactionEffectiveDate    - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion              - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage             - optional parameter to select from a specific compiled language
    @param $treeonly                    - optional boolean parameter to get the tree structure only if true
    @param $fullTree                    - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @param $associations                - optional boolean parameter relevant if $treeonly = 'false' to include associations: terminologyAssociation, identifierAssociation
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function deapi:getConcept($request as map(*)) {

    let $id                         := $request?parameters?id[not(. = '')]
    let $effectiveDate              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $transactionId              := $request?parameters?transactionId[not(. = '')]
    let $transactionEffectiveDate   := $request?parameters?transactionEffectiveDate[string-length() gt 0]
    let $project                    := $request?parameters?project[not(. = '')]
    let $projectVersion             := $request?parameters?release[not(. = '')]
    let $projectLanguage            := $request?parameters?language[not(. = '')]
    let $treeOnly                   := $request?parameters?treeonly = true()
    let $fullTree                   := $request?parameters?fulltree = true()
    let $associationMode            := $request?parameters?associations = true()
    
    let $acceptTypes                := roaster:accepted-content-types()
    let $acceptedType               := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]
    
    let $format                     := tokenize($acceptedType, '/')[2]
    
    let $propertiesMap              := 
        typeswitch ($request?parameters?conceptproperty) 
        case xs:string return 
            map:merge((
                for $s in $request?parameters?conceptproperty[string-length() gt 0]
                for $item in tokenize(lower-case($s),'\s') 
                return map:entry($item, true())
            ))
        default return map:merge(for $c in $request?parameters?conceptproperty?* return map:entry($c, true()))
    
    let $concept                    := 
        if ($transactionId) then utillib:getTransactionConcept($id, $effectiveDate, $transactionId, $transactionEffectiveDate, $project, $projectVersion, $projectLanguage) 
        else utillib:getConcept($id, $effectiveDate, $projectVersion, $projectLanguage)
        
    let $results                    :=
        if ($treeOnly) then utilde:getConceptTree($id, $effectiveDate, $transactionId[1], $transactionEffectiveDate[1], $fullTree, $propertiesMap)
        else utilde:getConceptExpanded($concept, $associationMode)

    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple concepts for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )

};

(:~ Retrieves DECOR dataset concept for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the concept
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $datasetOrtransactionId  - required parameter denoting the id of the dataset or transaction that the concept is in
    @param datasetOrtransactionE    - optional parameter denoting the effectiveDate of the dataset or transaction. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $communityName           - optional space separated list of community names. If omitted all communities are given
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @param $download                - optional as xs:boolean. Default: false. 
    @return as-is or as compiled as JSON
    @since 2024-11-08
:)
declare function deapi:getConceptExtract($request as map(*)) {

    let $datasetOrTransactionId     := $request?parameters?transactionId[not(. = '')]
    let $datasetOrTransactionEd     := $request?parameters?transactionEffectiveDate[string-length() gt 0]
    
    let $project                    := $request?parameters?project[not(. = '')]
    let $projectVersion             := $request?parameters?release[not(. = '')]
    let $projectLanguage            := $request?parameters?language[not(. = '')]
    
    let $communityName              := 
        typeswitch ($request?parameters?community) 
        case xs:string return 
            array:flatten(
                for $s in $request?parameters?community[string-length() gt 0]
                return tokenize($s,'\s')
            )
        default return array:flatten($request?parameters?community)
    
    let $format                     := $request?parameters?format[not(. = '')]
    let $download                   := $request?parameters?download = true()
    
    let $mergeDatasetWithUsage      := $request?parameters?mergeDatasetWithUsage = true()
    
    let $id                         := $request?parameters?id[not(. = '')]
    let $effectiveDate              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    (: parameter format overrides accept-header as mime-type for using the api in the browser :)
    
    (: 
        this call is used in backend and frontend (browser) - default behaviour mime-types:
        - backend:   default mime-type is json - response payload is json - if no accept type is given
        - frontend:  default mime-type is xml - response payload is xml - if no format is given
    :)
    
    let $acceptTypes                := roaster:accepted-content-types()
    let $acceptedType               := ($acceptTypes[. = ('application/xml', 'application/json', 'application/*', '*/*')], 'application/json')[1]
    
    let $format                     := if ($format) then $format else tokenize($acceptedType, '/')[2]

    (: 
       overwrite format in the backend is not always possible because of middleware behavior
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
    :) 
    let $check                          :=
        if ($format = 'xml' and not($acceptedType = ('application/xml', 'application/*', '*/*'))) then 
            error($errors:BAD_REQUEST, 'When parameter format=' || $format || ' the accept header should contain application/xml. Found: ' || string-join($acceptTypes, ','))
        else
        if (not($format = 'json') and $acceptedType = 'application/json') then
            error($errors:BAD_REQUEST, 'When parameter format=' || $format || ' the accept header can contain anything but application/json. Found: ' || string-join($acceptTypes, ','))
        else () 

    let $results                    := utilde:getConceptExtract($id, $effectiveDate, $datasetOrTransactionId, $datasetOrTransactionEd, $project, $projectVersion, $projectLanguage, $communityName, $mergeDatasetWithUsage)
        
    let $filePrefix                 := if (exists($results/@transactionId)) then 'TR_' else 'DS_'
    let $filename                   := $filePrefix || $results/@shortName || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $format
    
    let $results                    :=
        for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/node(), $format)
                }
    
    (: in this case the json format overrides header application/xml and works with a text header :)
    let $results                    :=
        if ($format = 'json' and not($acceptedType = 'application/json')) then
            fn:serialize($results, map{"method": $format , "indent": true()})
            else $results     
           
    let $r-header                   := 
        (response:set-header('Content-Type', 'application/' || $format || '; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple concepts for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else $results        

};

(:~ Retrieves DECOR dataset concept view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the concept
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $datasetOrtransactionId  - required parameter denoting the id of the dataset or transaction that the concept is in
    @param datasetOrtransactionE    - optional parameter denoting the effectiveDate of the dataset or transaction. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $ui-language             - optional parameter to select from a specific language for the ui
    @param $hidecolumns             - optional parameter to hide columns in the view based on (hex) number
    @param $format                  - optional. if not given it is html
    @param $download                - optional as xs:boolean. Default: false. 
    @return as-is 
    @since 2024-09-24
:)
declare function deapi:getConceptView($request as map(*)) {

    let $datasetOrTransactionId     := $request?parameters?transactionId[not(. = '')]
    let $datasetOrTransactionEd     := $request?parameters?transactionEffectiveDate[string-length() gt 0]

    let $project                    := $request?parameters?project[not(. = '')]
    let $projectVersion             := $request?parameters?release[not(. = '')]
    let $projectLanguage            := $request?parameters?language[not(. = '')]
    
    let $communityName              := 
        typeswitch ($request?parameters?community) 
        case xs:string return 
            array:flatten(
                for $s in $request?parameters?community[string-length() gt 0]
                return tokenize($s,'\s')
            )
        default return array:flatten($request?parameters?community)
    
    let $ui-lang                    := $request?parameters?ui-language[not(. = '')]
    let $hidecolumns                := $request?parameters?hidecolumns[not(. = '')]
    let $format                     := $request?parameters?format[not(. = '')]
    let $download                   := $request?parameters?download = true()
    let $inline                     := $request?parameters?inline = true()

    let $id                         := $request?parameters?id[not(. = '')]
    let $effectiveDate              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    let $format                     := if ($format = ('list', 'hlist')) then $format else 'html'

    
    let $dataset                    := if ($datasetOrTransactionId)
        then utillib:getDataset($datasetOrTransactionId, $datasetOrTransactionEd, $project, $projectVersion, $projectLanguage) | utillib:getTransaction($datasetOrTransactionId, $datasetOrTransactionEd, $project, $projectVersion, $projectLanguage)
        else utillib:getConcept($id, $effectiveDate, $project, $projectVersion, $projectLanguage)/ancestor::dataset

    let $decor                      := $dataset/ancestor::decor[1]

    let $results                    := 
        if (empty($dataset)) then () else 
        if ($datasetOrTransactionId) then 
            utillib:getDatasetExtract($dataset, $datasetOrTransactionId, $datasetOrTransactionEd, $id, $effectiveDate, $projectVersion, $projectLanguage, $communityName, (), false())
        else (
            utillib:getDatasetExtract($dataset, $id, $effectiveDate, $id, $effectiveDate, $projectVersion, $projectLanguage, $communityName, (), false())
        )

    let $filePrefix                 := if (exists($results/@transactionId)) then 'TR_' else 'DS_'
    let $filename                   := $filePrefix || $results/@shortName || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html' 
    let $r-header                   := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple datasets for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else 
        
        (: prepare for Html :)
        let $referenceUrl           := $decor/project/reference[@url castable as xs:anyURI]/@url
        let $projectPrefix          := $decor/project/@prefix
        let $header                 := if ($inline) then false() else true()        
         
     return
        switch ($format)
        case 'list'
            return utilhtml:convertTransactionOrDataset2SimpleHtml($results, $projectLanguage, $ui-lang, $hidecolumns, true(), $projectVersion, $referenceUrl, $projectPrefix, false(), $download)
        case 'hlist'
            return utilhtml:convertTransactionOrDataset2SimpleHierarchicalList($results, $projectLanguage, $ui-lang, $hidecolumns, true(), $projectVersion, $referenceUrl, $projectPrefix, false(), $download)
        default
            return utilhtml:convertTransactionOrDataset2Html($results, $projectLanguage, $ui-lang, $hidecolumns, true(), true(), true(), $projectVersion, $referenceUrl, $projectPrefix, (), false(), $filename, $download, $header)
};

(:~ Retrieves exact DECOR concept usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                          - required parameter denoting the id of the concept
    @param $effectiveDate               - required parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $transactionId               - required parameter denoting the id of the transaction that the concept is in
    @param $transactionEffectiveDate    - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion              - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage             - optional parameter to select from a specific compiled language
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function deapi:getConceptUsage($request as map(*)) {

    let $id                         := $request?parameters?id[not(. = '')]  
    let $effectiveDate              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $transactionId              := $request?parameters?transactionId[not(. = '')]
    let $transactionEffectiveDate   := $request?parameters?transactionEffectiveDate[string-length() gt 0]
    let $project                    := $request?parameters?project[not(. = '')]
    let $projectVersion             := $request?parameters?release[not(. = '')]
    let $projectLanguage            := $request?parameters?language[not(. = '')]
    
    let $concept                    :=
        if ($transactionId) then utillib:getTransactionConcept($id, $effectiveDate, $transactionId, $transactionEffectiveDate, $project, $projectVersion, $projectLanguage)
        else utillib:getConcept($id, $effectiveDate, $projectVersion, $projectLanguage)
    
    let $results                    := utillib:getConceptAssociations($concept, utillib:getOriginalForConcept($concept), true(), 'normal', true(), $projectLanguage)/(* except terminologyAssociation)
    
    let $count                      := count($results)
    let $max                        := $count
    let $allcnt                     := $count
    
    return
        <list artifact="USAGE" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements(subsequence($results, 1, $max))
        }
        </list>
};

(:~ Retrieves DECOR dataset concept history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function deapi:getConceptHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-DATASETCONCEPT, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-DATASETCONCEPT}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Retrieves an existing dataset concept for edit. This means the user needs to be an editing author of the project, and setting a lock
    @param $datasetId               - required. dataset id to insert concept into.
    @param $datasetEffectiveDate    - required. dataset effectiveDate to insert concept into.
    @param $conceptBaseId           - optional. baseId to create the new concept id out of. Defaults to defaultBaseId for type DE if omitted
    @param $conceptType             - required. 'item' or 'group'
    @param $insertMode              - required. 'into' or 'preceding' or 'following'. For preceding and for following, $insertRef is required
    @param $insertRef               - optional. concept/@id reference for insert. Inserts as new concept in dataset if empty
    @param $insertFlexibility       - optional. concept/@effectiveDate reference for insert. Only relevant when two versions of the same concept are in the same dataset which is logically highly unlikely
:)
declare function deapi:getConceptForEdit($request as map(*)) {

    let $authmap                    := $request?user
    let $id                         := $request?parameters?id[not(. = '')]
    let $effectiveDate              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $targetId                   := $request?parameters?targetId[not(. = '')]
    let $targetEffectiveDate        := $request?parameters?targetEffectiveDate[not(. = '')]
    let $targetType                 := $request?parameters?targetType[not(. = '')]
    let $generateConceptListIds     := $request?parameters?generateConceptListIds
    let $breakLock                  := $request?parameters?breakLock = true()
    let $associationMode            := $request?parameters?associations = true()
    
    let $acceptTypes                := roaster:accepted-content-types()
    let $acceptedType               := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]
    
    let $format                     := tokenize($acceptedType, '/')[2]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                    := utilde:getConceptForEdit($authmap, $id, $effectiveDate, $targetId, $targetEffectiveDate, $targetType, $generateConceptListIds, $breakLock, $associationMode)

    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple concepts for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )

};

(: deapi:getConceptList returns a list of zero or more concepts. Results are always sorted by the match length. Shortest match first and largest match last.
    @param $search                  - optional. Sequence of terms to look for. Returns every object if empty.
    @param $prefix                  - required. Required DECOR project prefix to search in.
    @param $datasetId               - optional. Returns a list concepts connected to a dataset with chosen id. This parameter works in conjunction with $datasetEffectiveDate.
    @param $datasetEffectiveDate    - optional. Returns a list concepts connected to an object with this effectiveDate. This parameter works in conjunction with $datasetId.
    @param $type                    - optional. Returns all types if empty. Or can choose type: 'item' or 'group'. 
    @param $status                  - optional. Returns a list of  most recent status code ("new","draft","pending","final","rejected","cancelled","deprecated") values that results should have. Returns all if empty.
    @param $max                     - optional. Maximum number of results to return, defaults to 50.
    @param $originalOnly            - required. Returns original concepts (that do not inherit) if true, otherwise returns every hit including concepts that inherit from matching concepts
    @param $localConceptsOnly       - required. Returns only concepts within the given prefix
:)
declare function deapi:getConceptList($request as map(*)) {
    
    let $searchTerms            := 
        typeswitch ($request?parameters?search) 
        case xs:string return 
            array:flatten(
                for $s in $request?parameters?search[string-length() gt 0]
                return tokenize(lower-case($s),'\s')
            )
        default return array:flatten($request?parameters?search)
    
    let $check                  :=
        if (empty($searchTerms)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter search or it is empty')
        else ()
    
    let $projectPrefix          := $request?parameters?prefix
    let $datasetId              := $request?parameters?datasetId[not(. = '')]
    let $datasetEffectiveDate   := $request?parameters?datasetEffectiveDate[not(. = '')]
    let $conceptId              := $request?parameters?conceptId[not(. = '')]
    let $conceptEffectiveDate   := $request?parameters?conceptEffectiveDate[not(. = '')]
    let $conceptType            := $request?parameters?type
    let $statusCodes            := 
        typeswitch ($request?parameters?status) 
        case xs:string return 
            array:flatten(
                for $s in $request?parameters?status[string-length() gt 0]
                return tokenize(lower-case($s),'\s')
            )
        default return array:flatten($request?parameters?status)
    
    (:mostly interesting when we're called from the similar concepts context in the dataset form when we're creating a new concept:)
    (:we inherit only from original concepts:)
    let $originalOnly           := $request?parameters?originalonly = true()
    
    (:we want to distinguish between local searches and searches that include the current project:)
    let $localConceptsOnly      := $request?parameters?localconceptsonly = true()
    
    let $maxResults             := $request?parameters?max
    let $maxResults             := if ($maxResults castable as xs:integer and xs:integer($maxResults)>0) then (xs:integer($maxResults)) else ()
    
    let $results                := utilde:getConceptList($projectPrefix, $searchTerms, $maxResults, (), (), $datasetId, $datasetEffectiveDate, $conceptId, $conceptEffectiveDate, $conceptType, $statusCodes, $originalOnly, $localConceptsOnly)

    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Creates new dataset concept with an empty name in each of the project languages
    @param $datasetId               - required. dataset id to insert concept into.
    @param $datasetEffectiveDate    - required. dataset effectiveDate to insert concept into.
    @param $conceptBaseId           - optional. baseId to create the new concept id out of. Defaults to defaultBaseId for type DE if omitted
    @param $conceptType             - required. 'item' or 'group'
    @param $insertMode              - required. 'into' or 'preceding' or 'following'. For preceding and for following, $insertRef is required
    @param $insertRef               - optional. concept/@id reference for insert. Inserts as new concept in dataset if empty
    @param $insertFlexibility       - optional. concept/@effectiveDate reference for insert. Only relevant when two versions of the same concept are in the same dataset which is logically highly unlikely
:)
declare function deapi:postConcept($request as map(*)) {

    let $authmap                    := $request?user
    let $datasetId                  := $request?parameters?id
    let $datasetEffectiveDate       := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $conceptBaseId              := $request?parameters?baseId
    let $conceptType                := $request?parameters?conceptType
    let $insertMode                 := $request?parameters?insertMode
    let $insertRef                  := $request?parameters?insertRef
    let $insertFlexibility          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?insertFlexibility)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?insertFlexibility[string-length() gt 0]
        }
    
    let $check                      :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                    :=  utilde:createConcept($authmap, $datasetId, $datasetEffectiveDate, $conceptBaseId, $conceptType, $insertMode, $insertRef, $insertFlexibility)
    
    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Update concept statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR concept/@id to update
    @param $effectiveDate           - required. DECOR concept/@effectiveDate to update
    @param $recurse                 - optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $list                    - optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $statusCode              - optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $versionLabel            - optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $expirationDate          - optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $officialReleaseDate     - optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return changed concept
:)
declare function deapi:putConceptStatus($request as map(*)) {

    let $authmap                    := $request?user
    let $id                         := $request?parameters?id
    let $effectiveDate              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $recurse                    := $request?parameters?recurse = true()
    let $list                       := $request?parameters?list = true()
    let $statusCode                 := $request?parameters?statusCode
    let $versionLabel               := $request?parameters?versionLabel
    let $expirationDate             := $request?parameters?expirationDate
    let $officialReleaseDate        := $request?parameters?officialReleaseDate
    
    let $check                      :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()

    let $result                     := utilde:setConceptStatus($authmap, $id, $effectiveDate, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
    
    return deapi:getConcept($request)
     
};

(:~ Update dataset concept
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR concept/@id to update
    @param $effectiveDate   - required. DECOR concept/@effectiveDate to update
    @param $data            - required. DECOR concept xml element containing everything that should be in the updated concept
    @return concept object as xml with json:array set on elements
    @since 2020-05-03
:)
declare function deapi:putConcept($request as map(*)) {

    let $authmap                    := $request?user
    let $id                         := $request?parameters?id
    let $effectiveDate              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $transactionId              := $request?parameters?transactionId
    let $transactionEffectiveDate   := $request?parameters?transactionEffectiveDate
    let $data                       := utillib:getBodyAsXml($request?body, 'concept', ())
    let $associationMode            := $request?parameters?associations = true()
    
    let $check                      :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                      :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $result                     := 
        if (empty($transactionId)) then utilde:putConcept($authmap, string($id), $effectiveDate, $data) else utilde:putTransactionConcept($authmap, string($id), $effectiveDate, $transactionId, $transactionEffectiveDate, $data)
    
    return deapi:getConcept($request)
};

(:~ Update DECOR concept parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

    { "op": "[add|remove|replace]", "path": "[/|/displayName|/priority|/type]", "value": "[string|object]" }
    
    where
    
    * op - add (object associations, tracking, event) or remove (object associations only) or replace (displayName, priority, type, tracking, assignment)
    
    * path - / (object, tracking, assignment) or /displayName or /priority or /type
    
    * value - string when the path is not /. object when path is /
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the concept to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return concept structure
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function deapi:patchConcept($request as map(*)) {

    let $authmap                    := $request?user
    let $id                         := $request?parameters?id
    let $effectiveDate                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                       := utillib:getBodyAsXml($request?body, 'parameters', ())
    let $associationMode            := $request?parameters?associations = true()
    
    let $check                      :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                      :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                 := utilde:patchConcept($authmap, string($id), $effectiveDate, $data)
    
    return deapi:getConcept($request)
};

(:~ Retrieves DECOR concept maps based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                          - required parameter denoting the id of the concept
    @param $effectiveDate               - required parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $transactionId               - required parameter denoting the id of the transaction that the concept is in
    @param $transactionEffectiveDate    - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion              - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage             - optional parameter to select from a specific compiled language
    @param $associations                - optional boolean parameter relevant if $treeonly = 'false' to include associations: terminologyAssociation, identifierAssociation
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function deapi:getConceptAssociationList($request as map(*)) {

    let $authmap                    := $request?user
    let $id                         := $request?parameters?id[not(. = '')]
    let $effectiveDate              := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $transactionId              := $request?parameters?transactionId[not(. = '')]
    let $transactionEffectiveDate   := $request?parameters?transactionEffectiveDate[string-length() gt 0]
    let $projectPrefixOrId          := $request?parameters?project[not(. = '')]
    let $projectVersion             := $request?parameters?release[not(. = '')]
    let $projectLanguage            := $request?parameters?language[not(. = '')]
    let $associationMode            := $request?parameters?mode[not(. = '')]
    
    let $result                     := utilde:getConceptAssociationList($id, $effectiveDate, $transactionId, $transactionEffectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage, $associationMode)

    return
        <list artifact="MP" current="{count($result/*)}" total="{count($result/*)}" all="{count($result/*)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($result/*)
        }
        </list>
};

(:~ Update DECOR dataset or transaction concept. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

    { "op": "[add|remove|replace]", "path": "/", "value": "[terminologyAssociation|identifierAssociation]" }
    
    where
    * op - add (object associations, tracking, event) or remove (object associations only) or replace (displayName, priority, type, tracking, assignment)
    * path - / 
    * value - terminologyAssociation|identifierAssociation object
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the issue to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return issue structure including generated meta data
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function deapi:patchConceptAssociation($request as map(*)) {

    let $authmap                        := $request?user
    let $deid                           := $request?parameters?id[not(. = '')]
    let $deed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $transactionId                  := $request?parameters?transactionId[not(. = '')]
    let $transactionEffectiveDate       := $request?parameters?transactionEffectiveDate[string-length() gt 0]
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    (:let $s                      := xmldb:store('/db/apps/decor/tmp', 'ttt.xml', $data) :)
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $result                         := utilde:patchConceptAssociation($authmap, string($deid), $deed, $transactionId, $transactionEffectiveDate, $data)
    
    return (
        roaster:response(200, 
            <list artifact="MP" current="{count($result/*)}" total="{count($result/*)}" all="{count($result/*)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
            {
                $result/(@* except (@artifact | @current | @total | @all)),
                utillib:addJsonArrayToElements($result/*)
            }
            </list>)
    )

};


