xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utiltemp               = "http://art-decor.org/ns/api/util-template";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace decorlib        = "http://art-decor.org/ns/api/decor" at "decor-lib.xqm";
import module namespace utilgg          = "http://art-decor.org/ns/api/util-governancegroup" at "util-governancegroup-lib.xqm";
import module namespace utiltcs         = "http://art-decor.org/ns/api/util-terminology-codesystem" at "util-terminology-codesystem-lib.xqm";
import module namespace histlib         = "http://art-decor.org/ns/api/history" at "history-lib.xqm";
import module namespace ruleslib        = "http://art-decor.org/ns/api/rules" at "rules-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

(:relevant for template list. If treetype is 'limited' and param id or name is valued then a list is built for just this id or name.:)
declare variable $utiltemp:TREETYPELIMITED              := 'limited';
(:relevant for template list. If treetype is 'marked' and param id or name is valued then a full list is built where every template hanging off this template is marked as such.:)
declare variable $utiltemp:TREETYPEMARKED               := 'marked';
(:relevant for template list. If treetype is 'marked' and param id or name is valued then a limited list is built containing only templates hanging off this template.:)
declare variable $utiltemp:TREETYPELIMITEDMARKED        := 'limitedmarked';

declare %private variable $utiltemp:STATUSCODES-FINAL   := ('active', 'cancelled', 'pending', 'review', 'rejected', 'retired');
declare %private variable $utiltemp:TEMPLATE-TYPES             := utillib:getDecorTypes()/TemplateTypes;
declare %private variable $utiltemp:TEMPLATE-FORMATS           := utillib:getDecorTypes()/TemplateFormats;
declare %private variable $utiltemp:RELATIONSHIP-TYPES         := utillib:getDecorTypes()/RelationshipTypes;
declare %private variable $utiltemp:EXAMPLE-TYPES              := utillib:getDecorTypes()/ExampleType;


(:~ Retrieves latest DECOR template based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @param $projectPrefixOrId       - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function utiltemp:getTemplate($id as xs:string, $effectiveDate as xs:string?, $projectPrefixOrId as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?) as element(template)* {
    
    let $id                     := $id[not(. = '')]
    let $effectiveDate          := $effectiveDate[not(. = '')]
    let $projectPrefixOrId      := $projectPrefixOrId[not(. = '')]
    let $projectVersion         := $projectVersion[not(. = '')]
    
    let $templates              :=
        if (empty($projectPrefixOrId)) then utiltemp:getExpandedTemplateById($id, $effectiveDate)
        else utiltemp:getExpandedTemplateById($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage)
        
     (: if multiple copies are found, prefer copies from this server of other copies. :)
    let $templates                := 
            if (count($templates) gt 1) then (
            let $t := $templates[@id][@url = $utillib:strDecorServicesURL] | $templates[@id][empty(@url)]
            let $t := if ($t) then $t else $templates[@id]
            return if ($t) then $t else $templates
        )
        else $templates
    
    return
        for $template in $templates
        return utiltemp:prepareTemplateForAPI($template)
};

(:~ Retrieves DECOR template for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @param $projectPrefixOrId       - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $language                - optional
    @param $tree                    - optional
    @return as-is or compiled as JSON
    @since 2023-11-17
:)
declare function utiltemp:getTemplateExtract($id as xs:string, $effectiveDate as xs:string?) as element(template)* {

    let $templates              := utillib:getTemplateById($id, $effectiveDate)
    
    return
    <return>
    {
        for $template in $templates
        let $prefix             := $template/ancestor::decor/project/@prefix
        let $url                := ($template/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
        group by $url, $prefix
        return
            <project ident="{$prefix}">
            {
                for $ns at $i in utillib:getDecorNamespaces($template/ancestor::decor)
                return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri},
                if ($template[ancestor::cacheme]) then attribute url {$template[1]/ancestor::cacheme/@bbrurl} else (),
                $template[1]/ancestor::decor/project/@defaultLanguage,
                $template
            }
            </project>
    }
    </return>
};

declare function utiltemp:getTemplateExtract($projectPrefixOrId as xs:string*, $projectVersion as xs:string*, $projectLanguage as xs:string?, $id as xs:string, $effectiveDate as xs:string?) as element(template)* {

    let $id                     := $id[not(. = '')]
    let $effectiveDate          := ($effectiveDate[not(. = '')], 'dynamic')[1]
    let $projectPrefixOrId      := $projectPrefixOrId[not(. = '')][1]
    let $projectVersion         := $projectVersion[not(. = '')][1]
    let $projectLanguage        := $projectLanguage[not(. = '')][1]

    (: retrieve all templates for the input $decor project(s) 
        - when compiled version  - as is - 
        - when live version - getting all (cached) decor projects in scope  
    :)
    let $decor                  := utillib:getDecor($projectPrefixOrId)
    let $projectPrefix          := ($decor/project/@prefix)[1]
    let $allTemplates           := utillib:getTemplateById($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage) | $decor//template[@ref = $id]
           
    return
    
    <return>
    {
        for $templates in $allTemplates
        let $id                 := $templates/@id | $templates/@ref
        group by $id
        return (
            <template id="{$id}" ident="{$projectPrefix}">
            {
                for $template in $templates
                let $tmed       := $template/@effectiveDate
                group by $tmed
                order by xs:dateTime($template[1]/@effectiveDate) descending
                return utiltemp:createTemplateExtract($template[1])
            }
            </template>
        )
    }
    </return>
};

declare %private function utiltemp:createTemplateExtract($template as element(template)) as element(template) {

    <template>
    {
        $template/(@* except (@*[contains(name(),'dummy-')] | @url | @ident)),
        attribute url {($template/@url, $template/ancestor::*/@bbrurl, $utillib:strDecorServicesURL)[1]},
        attribute ident {($template/@ident, $template/ancestor::*/@bbrident, $template/ancestor::decor/project/@prefix)[1]},
        if ($template/../@*[contains(name(),'dummy-')]) then $template/../@*[contains(name(),'dummy-')]
        else 
        if ($template/ancestor::decor) then (
            (: <ns uri="urn:hl7-org:v3" prefix="hl7" default="{$ns-default='hl7'}" readonly="true"/> :)
            for $ns at $i in utillib:getDecorNamespaces($template/ancestor::decor)
            return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri}
        )
        else (
            for $ns-prefix at $i in in-scope-prefixes($template)[not(.=('xml'))]
            let $ns-uri := namespace-uri-for-prefix($ns-prefix, $template[1])
            order by $ns-prefix
            return attribute {QName($ns-uri,concat($ns-prefix,':dummy-',$i))} {$ns-uri}
        )
        ,
        $template/node()
    }
    </template>

};

declare function utiltemp:getExpandedTemplateExtract($projectPrefixOrId as xs:string*, $projectVersion as xs:string*, $projectLanguage as xs:string?, $id as xs:string, $effectiveDate as xs:string?) as element(template)* {
    let $id                     := $id[not(. = '')]
    let $effectiveDate          := ($effectiveDate[not(. = '')], 'dynamic')[1]
    let $projectPrefixOrId      := $projectPrefixOrId[not(. = '')][1]
    let $projectVersion         := $projectVersion[not(. = '')][1]
    let $projectLanguage        := $projectLanguage[not(. = '')][1]
    
    (: retrieve all templates for the input $decor project(s) 
        - when compiled version  - as is - 
        - when live version - getting all (cached) decor projects in scope  
    :)
    let $decor                  := utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)
    let $projectPrefix          := ($decor/project/@prefix)[1]
    
    let $check                  :=
        if (empty($projectPrefix) or empty($decor)) then
            error($errors:BAD_REQUEST, 'Project ' || $projectPrefixOrId || ' not found.')
        else () 
    
    let $allTemplates           := utillib:getTemplateById($id, $effectiveDate, $decor) | $decor//template[@ref = $id]
    
    let $expndTemplates         :=
        for $templates in $allTemplates
        return
            if ($templates[@ref]) then $templates else ( 
                utiltemp:getExpandedTemplate($templates, $decor, $projectVersion, $projectLanguage)
            )
    
    return 
    <result>
    {
        for $templates in $expndTemplates
        let $id := $templates/@id | $templates/@ref
        group by $id 
        return
        <template id="{$id}" ident="{$projectPrefix}">
        {
            for $template in $templates
            return
            <template>
            {
                $template/(@id, @ref)[1],
                for $att in $templates/(@* except (@id | @ref))
                let $attname := name($att)
                group by $attname
                return
                    ($templates[@id]/@*[name() = $attname], $templates[@ref]/@*[name() = $attname])[1]
                ,
                $template/*,
                for $ur in utiltemp:getDependenciesAndUsage($projectPrefix, (), (), $template/(@id, @ref)[1], $template/@effectiveDate)
                let $if := $ur/(@id, @ref)[1] || $ur/@effectiveDate
                group by $if
                order by replace(replace(concat(($ur/@id)[1], '.'), '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1'), $if
                for $u in $ur
                return
                    if (name($u) = 'transactionAssociation') then () else $u
            }
            </template>
       }
       </template>
    }
    </result>
};

(:~ Copied from art/api/api-decor-template.xqm. Retrieves the ART-DECOR 3 compatible list of usages for as template
* Transaction associations
* Template associations inbound and outbound
:)
declare function utiltemp:getDependenciesAndUsage($projectPrefix as xs:string?, $projectVersion as xs:string?, $language as xs:string?, $id as xs:string, $effectiveDate as xs:string?) {

    let $templates              := if (empty($projectPrefix)) then utillib:getTemplateById($id, ())[@id] else utillib:getTemplateById($id, (), $projectPrefix, $projectVersion, $language)[@id]
       
    let $latest                 := string(max($templates/xs:dateTime(@effectiveDate)))
    let $template               := if ($effectiveDate castable as xs:dateTime) then $templates[@effectiveDate = $effectiveDate][1] else $templates[@effectiveDate = $latest][1]
    
    let $projectPrefix          := if (empty($projectPrefix)) then $template/ancestor::decor/project/@prefix else $projectPrefix
    let $decor                  := $template/ancestor::decor
    let $isLatestTm             := $latest = $template/@effectiveDate
    (: get dependencies in THIS project :)
    (: get dependencies in other projects than this one :)
    (: get uses :)
    let $result                 :=
        if ($template) then (
            utiltemp:getDependencies($template, $template/@id, $latest, 1, $setlib:colDecorData/decor | $setlib:colDecorCache/decor),
            utiltemp:templateUses($template, $template/@id, 1, $decor)
        ) 
        else ()
    
    (: uniquify result re/ type of ref element, type, id and effectiveDate, sort project ref's first :)
    let $templateReferences     :=
        for $d in $result
        let $id := concat(name($d), $d/@type, $d/@id, $d/@effectiveDate)
        group by $id
        order by (not($d[1]/@prefix=$projectPrefix))
        return $d[1]
    
    (: calculate transaction references for template itself and and any templates that refer to it :)
    let $t_templates            := $template | $templateReferences[self::ref]
    let $transactionReferences  := ($setlib:colDecorData//representingTemplate[@ref = $t_templates/@id] | $setlib:colDecorCache//representingTemplate[@ref = $t_templates/@id])
    
    (: calculate what the latest version is for templates that are being called dynamically. no use for static bindings. :)
    let $tmmap                  := 
        map:merge(
            for $ref in $transactionReferences
            let $tmid           := $ref/@ref
            group by $tmid
            return
                if ($ref/@flexibility[. castable as xs:dateTime]) then () else map:entry($tmid, utillib:getTemplateById($tmid, 'dynamic')[@id]/@effectiveDate)
        )
    
    (: transaction bindings count for static matches and dynamic matches :)
    let $transactionReferences  :=
        for $ref in $transactionReferences
        let $match              := if ($ref[@flexibility castable as xs:dateTime]) then $t_templates[@id = $ref/@ref][@effectiveDate = $ref/@flexibility] else $t_templates[@effectiveDate = map:get($tmmap, @id)]
        return if (empty($match)) then () else utillib:doTransactionAssociation($match[1], $ref/ancestor::transaction[1])
    
    return ($transactionReferences, $templateReferences)
};

declare %private function utiltemp:getDependencies($version as element(), $self as xs:string?, $latest as xs:string?, $level as xs:int, $decor as element()*) as element()* {
    
    let $ref                    := $version/@id/string()
    let $flexibility            := $version/@effectiveDate/string()
    let $lvtextinclude          := if ($level=1) then 'include' else 'dependency'
    let $lvtextcontains         := if ($level=1) then 'contains' else 'dependency'

    let $dep                    :=
        (: too deeply nested, raise error and give up :)
        if ($level > 5) then ()
        (: ref to myself in a level greater than 1, give up, you are done :)    
        else if (($ref = $self) and ($level > 1)) then ()
        (: all include statements anywhere with @ref and @flexibility :)
        (: get all referenced templates pointed to by @ref and @flexibility :)
        else (
            for $chain in $decor//include[@ref=$ref][@flexibility=$flexibility]/ancestor::template
            let $bbrurl         := $chain/ancestor::cacheme/@bbrurl
            let $bbrident       := ($chain/@ident, $chain/ancestor::cacheme/@bbrident, $chain/ancestor::decor/project/@prefix)[1] 
            return (
                <ref type="{$lvtextinclude}" prefix="{$bbrident}" flexibility="{$flexibility}">
                {
                    $chain/(@* except (@type|@prefix|@flexibility)),
                    if ($chain/@ident) then ()  else if ($bbrident) then attribute ident {$bbrident} else (),
                    if ($chain/@url) then () else if ($bbrurl) then attribute url {$bbrurl} else ()
                }
                </ref>,
                utiltemp:getDependencies($chain, $self, $latest, $level+1, $decor)
            )
            ,
        (: all elements anywhere with @contains and @flexibility :)
        (: get all referenced templates pointed to by @contains and @flexibility :)
            for $chain in $decor//element[@contains=$ref][@flexibility=$flexibility]/ancestor::template
            let $bbrurl         := $chain/ancestor::cacheme/@bbrurl
            let $bbrident       := ($chain/ancestor::cacheme/@bbrident, $chain/ancestor::decor/project/@prefix)[1] 
            return (
                <ref type="{$lvtextcontains}" prefix="{$bbrident}" flexibility="{$flexibility}">
                {
                    $chain/(@* except (@type|@prefix|@flexibility)),
                    if ($chain/@ident) then ()  else if ($bbrident) then attribute ident {$bbrident} else (),
                    if ($chain/@url) then () else if ($bbrurl) then attribute url {$bbrurl} else ()
                }
                </ref>,
                utiltemp:getDependencies($chain, $self, $latest, $level+1, $decor)
            )
            ,
        (: all 
               include statements anywhere with @ref but no @flexibility 
               or elements with @contains but no @flexibility 
               BUT ONLY in case $flexibility = $latest, i.e. dynamic binding 
        :)
            if ($flexibility = $latest) then (
                (: get all referenced templates pointed to by @ref :)
                let $ttis       := 
                    for $i in $decor//include[@ref = $ref]
                    let $rf     := $i/@ref
                    let $fl     := if ($i[@flexibility]) then $i/@flexibility else 'dynamic'
                    group by $rf, $fl
                    return if ($fl = 'dynamic') then $i else ()
                let $ttis       := $ttis/ancestor::template
                
                for $chain in $ttis
                let $bbrurl     := $chain/ancestor::cacheme/@bbrurl
                let $bbrident   := ($chain/ancestor::cacheme/@bbrident, $chain/ancestor::decor/project/@prefix)[1] 
                return (
                    <ref type="{$lvtextinclude}" prefix="{$bbrident}" flexibility="dynamic">
                    {
                        $chain/(@* except (@type|@prefix|@flexibility)),
                        if ($chain/@ident) then ()  else if ($bbrident) then attribute ident {$bbrident} else (),
                        if ($chain/@url) then () else if ($bbrurl) then attribute url {$bbrurl} else ()
                    }
                    </ref>,
                    utiltemp:getDependencies($chain, $self, $latest, $level+1, $decor)
                ),
                
            (: get all referenced templates pointed to by @contains :)
                let $ttis       := 
                    for $i in $decor//element[@contains = $ref]
                    let $rf     := $i/@contains
                    let $fl     := if ($i[@flexibility]) then $i/@flexibility else 'dynamic'
                    group by $rf, $fl
                    return if ($fl = 'dynamic') then $i else ()
                
                let $ttis       := $ttis/ancestor::template
                for $chain in $ttis
                let $bbrurl     := $chain/ancestor::cacheme/@bbrurl
                let $bbrident   := ($chain/ancestor::cacheme/@bbrident, $chain/ancestor::decor/project/@prefix)[1] 
                return (
                    <ref type="{$lvtextcontains}" prefix="{$bbrident}" flexibility="dynamic">
                    {
                        $chain/(@* except (@type|@prefix|@flexibility)),
                        if ($chain/@ident) then ()  else if ($bbrident) then attribute ident {$bbrident} else (),
                        if ($chain/@url) then () else if ($bbrurl) then attribute url {$bbrurl} else ()
                    }
                    </ref>,
                    utiltemp:getDependencies($chain, $self, $latest, $level+1, $decor)
                )
            )
            else ()
        )
            
     return $dep
     
};

declare %private function utiltemp:templateUses($version as element(), $self as xs:string?, $level as xs:int, $decor as element(decor)) as element(uses)* {
    
    (: too deeply nested, raise error and give up :)
    if ($level > 7) then ()
    (: ref to myself in a level greater than 1, give up, you are done :)
    else if (($version/@id = $self) and ($level > 1)) then ()

    else (
        for $lc in ($version//element[@contains] | $version//include[@ref])
        let $xid                := if ($lc/name() = 'element') then $lc/@contains else $lc/@ref
        let $flex               := ($lc/@flexibility, 'dynamic')[1]
        let $xtype              := if ($lc/name() = 'element') then 'contains' else 'include'
        let $templ              := utillib:getTemplateById($xid, $flex, $decor, $decor/@versionDate, $decor/@language)[@id]
        
        return
            if ($templ) then
                for $x in $templ
                
                return
                    <uses type="{$xtype}" prefix="{$decor/project/@prefix}" flexibility="{$flex}">
                        {
                         $x/(@* except (@type|@prefix|@flexibility|@*[contains(name(), 'dummy-')])),
                         if($x/@url) then () else attribute url {($x[1]/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]},
                         if($x/@ident) then () else attribute ident {($x/ancestor::decor/project/@prefix)[1]}
                        }
                    </uses>
            else (
                    <uses type="{$xtype}" prefix="{$decor/project/@prefix}" id="{$xid}" name="" displayName="" effectiveDate="{$flex}"/>
            )
    )
};

(:~ Returns a list of zero or more templates
   
   @param $projectPrefixOrId - optional. determines search scope. null is full server, pfx- limits scope to this project only
   @param $projectVersion    - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
   @param $id                - optional. Identifier of the template to retrieve
   @param $name              - optional. Name of the template to retrieve (valueSet/@name)
   @param $flexibility       - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
   @param $treetype          - optional. Default $utiltemp:TREETYPELIMITEDMARKED
   @param $resolve           - optional. Boolean. Default true. If true and $doV2 is false, returns valueSetList.valueSet* where every valueSet has @id|@ref and a descending by 
   effectiveDate list of matching valueSets. If false and $doV2 is false, returns valueSetList.valueSet* containing only the latest version of the valueSet (or ref if no versions 
   exist in $projectPrefixOrId)<br/>If true and $doV2 is true, returns valueSetList.valueSet* where every valueSet has @id|@ref and a descending by effectiveDate list of matching 
   valueSets. If false and $doV2 is false, returns valueSetList.valueSet* containing only the latest version of the valueSet (or ref if no versions exist in $projectPrefix)
   @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
   @since 2013-06-14
:)
declare function utiltemp:getTemplateList($governanceGroupId as xs:string?, $projectPrefixOrId as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $searchTerms as xs:string*, $objectid as xs:string?, $objected as xs:string?, $treetype as xs:string*, $resolve as xs:boolean) {

    let $governanceGroupId      := $governanceGroupId[not(. = '')]
    let $projectPrefixOrId      := $projectPrefixOrId[not(. = '')]
    let $projectVersion         := $projectVersion[not(. = '')]
    let $projectLanguage        := $projectLanguage[not(. = '')]
    
    let $objectid               := $objectid[not(. = '')]
    let $objected               := $objected[not(. = '')]
    
    (: $utiltemp:TREETYPELIMITED       If treetype is 'limited' and param id is valued then a list is built for just this id.:)
    (: $utiltemp:TREETYPEMARKED        If treetype is 'marked' and param id is valued then a full list is built where every template hanging off this template is marked as such.:)
    (: $utiltemp:TREETYPELIMITEDMARKED If treetype is 'limitedmarked' and param id is valued then a limited list is built containing only templates hanging off this template.:)

    let $treetype               := if (empty($objectid)) then ($utiltemp:TREETYPELIMITED) else ($treetype[. = ($utiltemp:TREETYPELIMITED, $utiltemp:TREETYPEMARKED, $utiltemp:TREETYPELIMITEDMARKED)], $utiltemp:TREETYPELIMITED)[1]

    let $decor                  := 
        if ($governanceGroupId) then
            for $projectId in utilgg:getLinkedProjects($governanceGroupId)/@ref return utillib:getDecorById($projectId)
        else (
            utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)
        )
    let $projectPrefix          := ($decor/project/@prefix)[1]   
    let $decors                 := utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL)
    
    let $results                := 
        if (empty($searchTerms)) then $decor//template
        else (
            let $luceneQuery                := utillib:getSimpleLuceneQuery($searchTerms, 'wildcard')
            let $luceneOptions              := utillib:getSimpleLuceneOptions()

            return 
                ($decors//template[@id = $searchTerms] |
                $decors//template[ends-with(@id, $searchTerms[1])] |
                $decors//template[@id][ft:query(@name | @displayName, $luceneQuery, $luceneOptions)])
        )
    
    let $objectsByRef           :=
        if ($resolve and empty($projectVersion)) then (
            for $tm in $results[@ref]
            let $tmref          := $tm/@ref
            group by $tmref
            return $tm[1] | $decors//template[@id = $tmref]
        )
        else (
            for $tm in $results[@ref]
            let $tmref          := $tm/@ref
            group by $tmref
            return $tm[1]
        )
        
    let $results                := $results[@id] | $objectsByRef
    
    
    let $results                :=
        if (empty($objectid) or $treetype=$utiltemp:TREETYPEMARKED) then $results else utillib:getTemplateById($objectid, $objected)
    
    let $starttemplate          :=
        if (empty($objectid)) then () 
        else if ($treetype=$utiltemp:TREETYPEMARKED) then utillib:getTemplateById($objectid, $objected)        
        else $results
        
    let $templateChain          := 
        if ($starttemplate) then 
            $starttemplate[1] | utiltemp:getTemplateChain($starttemplate[1], $decors, map:merge(map:entry(concat($starttemplate/@id, $starttemplate/@effectiveDate), ''))) 
        else ()
    
    let $templateSet            := if (empty($objectid)) then $results else $results | $templateChain
    
    let $result                 :=
        for $projectTemplatesById in $templateSet
        let $idref              := $projectTemplatesById/@id | $projectTemplatesById/@ref
        group by $idref
        return (
            let $representingTemplates          := $decor//representingTemplate[@ref = $idref]
            let $newestTemplateEffectiveDate    := string(max($projectTemplatesById/xs:dateTime(@effectiveDate)))
            let $newestTemplate                 := $projectTemplatesById[@effectiveDate = $newestTemplateEffectiveDate]
            let $templateSet                    :=
                for $template in $projectTemplatesById
                let $currentTemplateEffectiveDate   := $template/@effectiveDate
                let $lookingForNewest               := $template[@effectiveDate = $newestTemplateEffectiveDate]
                group by $currentTemplateEffectiveDate
                order by $template[1]/@effectiveDate descending
                return (
                    <version uuid="{util:uuid()}">
                    {
                        $template[1]/(@* except (@displayName|@ident|@url)),
                        attribute displayName {
                            if ($newestTemplate and $template[@ref]) then ($newestTemplate/@displayName, $newestTemplate/@name)[1] else ($template/@displayName, $template/@name)[1]
                        },
                        attribute url {($template[1]/ancestor-or-self::*/@url, $template[1]/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]},
                        attribute ident {($template[1]/ancestor::decor/project/@prefix)[1]},
                        if (empty($searchTerms)) then () else $template[1]/desc,
                        <classification>
                        {
                            ($template[1]/classification/@format)[1],
                            if ($template[1]/classification/@type) then
                                for $type in $template[1]/classification/@type
                                return <type>{$type}</type>
                            else <type>notype</type>
                            ,
                            for $tag in $template[1]/classification/tag
                            group by $tag
                            return $tag
                            ,
                            for $property in $template[1]/classification/property
                            group by $property
                            return $property
                        }
                        </classification>
                        ,
                        for $rtemp in $representingTemplates[@flexibility = $currentTemplateEffectiveDate]
                        return  <representingTemplate>{$rtemp/@ref, $rtemp/@flexibility}</representingTemplate>,
                        if ($lookingForNewest) then
                            for $rtemp in $representingTemplates[not(@flexibility castable as xs:dateTime)]
                            return <representingTemplate>{$rtemp/@ref}</representingTemplate>
                        else ()
                        ,
                        if ($starttemplate and not($treetype = $utiltemp:TREETYPELIMITED)) then
                            if ($template[@effectiveDate = $templateChain[@id=$template/@id]/@effectiveDate]) then
                            (
                                <ref type="template">{$starttemplate[1]/(@id | @name | @displayName | @effectiveDate)}</ref>
                            )
                            else()
                        else ()
                    }
                    </version>
                )
            
            return
            <template uuid="{util:uuid()}">
            {
                ($projectTemplatesById/@ref, $projectTemplatesById/@id)[1],
                ($templateSet[@id], $templateSet)[1]/(@* except (@uuid | @id | @ref | @class)),
                attribute class {($templateSet[@id][1]/classification/type/@type, 'notype')[1]},
                $templateSet
            }
            </template>
        )
    
    let $startDisplayName       := if ($starttemplate/@displayName) then $starttemplate/@displayName else $starttemplate/@name
    let $schemaTypes            := utillib:getDecorTypes()//TemplateTypes/enumeration
    let $classified             := true()
    return    
        <return>
        {
            if (empty($objectid)) then () else attribute id {$objectid},
            if (empty($objected)) then () else attribute effectiveDate {$objected},
            if (empty($objectid)) then () else attribute treetype {$treetype},
            attribute starttemplate {exists($starttemplate)},
            attribute prefix {$decor/project/@prefix},
            if ($classified) then (
                (: this is the code for a classification based hierarchical tree view :)
                for $schemaType in utillib:getDecorTypes()//TemplateTypes/enumeration
                let $templateSets := $result[@class = $schemaType/@value]
                return
                <category uuid="{util:uuid()}" type="{$schemaType/@value}">
                {
                    for $label in $schemaType/label
                    return <name language="{$label/@language}">{$label/text()}</name>
                    ,
                    for $templateSet in $templateSets
                    order by lower-case(($templateSet/@displayName, $templateSet/@name)[1])
                    return $templateSet
                }
                </category>
            )
            else (
                for $r in $result
                order by count($r//representingTemplate)=0, lower-case(($r/template[1]/@displayName, $r/template[1]/@name)[1])
                return $r
            )
        }
        </return>
};

(:~ Returns template datatypes
    
    @param $format           - optional. defaults to hl7v3xml1
    @return datatypes or http 404 if not found
    @since 2022-02-07
:)
declare function utiltemp:getTemplateDatatypes($format as xs:string) as element(supportedDataTypes)? {
    
    let $decorDatatypes         := $setlib:colDecorCore/supportedDataTypes[@type = $format]

    return
        if (empty($decorDatatypes)) then () else (
            <supportedDataTypes format="{$decorDatatypes/@type}">
            {
                $decorDatatypes/(@* except (@type | @xsi:*)),
                let $allTypes   := $decorDatatypes//dataType | $decorDatatypes//flavor | $decorDatatypes//atomicDataType
                (: assume that if there is a definition for the dataType at root level,  then that is complete, whereas a subtyped dataType does not need to be:)
                for $typeDef in $allTypes
                let $typeName    := $typeDef/@name 
                group by $typeName
                order by $typeName
                return
                    <type>
                    {
                        $typeDef[1]/@name, 
                        $typeDef[1]/@type, 
                        $typeDef[1]/@hasStrength, 
                        if ($typeDef[1]/self::flavor) then attribute isFlavor {'true'} else (),
                        if ($typeDef[1][self::atomicDataType]) then attribute type {'simpletype'} else (),
                        $typeDef[1]/@realm 
                    }
                        <item name="{$typeName}"/>
                    {
                        for $subtype in distinct-values(($typeDef[1]//dataType | $typeDef[1]//flavor | $typeDef[1]//atomicDataType)/@name)
                        order by $subtype
                        return <item name="{$subtype}"/>
                    }
                    {
                        $typeDef[1]/desc,
                        $typeDef[1]/attribute,
                        $typeDef[1]/element
                    }
                    </type>
            }
            </supportedDataTypes>
        )
};    

(:~ Central logic for creating an initial template

@param $authmap             - required. Map derived from token
@param $request-body        - required. json body containing new template structure
@param $lock                - required. true = create lock on result. false = do not create a lock on result
@return template structure
:)
declare function utiltemp:createTemplate($authmap as map(*), $projectPrefixOrId as xs:string, $sourceId as xs:string?, $sourceEffectiveDate as xs:string?, $refOnly as xs:boolean, $targetDate as xs:boolean, $keepIds as xs:boolean, $lock as xs:boolean, $data as element()?) as element(template) {

    let $decor                  := utillib:getDecor($projectPrefixOrId)
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project ''', $projectPrefixOrId, ''', does not exist. Cannot create in non-existent project.'))
        )
    
    let $projectPrefix          := ($decor/project/@prefix)[1]
    
    let $check                  := utiltemp:checkTemplateAccess($authmap, $decor, (), (), false(), false())
      
    let $sourceTemplate         := if ($sourceId) then utillib:getTemplateById($sourceId, $sourceEffectiveDate, $projectPrefix) else ()
    
    let $check                  :=  
        if ($sourceId) then (
            if ($sourceTemplate) then () else (
            error($errors:BAD_REQUEST, 'Prototype template with id ''' || $sourceId || ''' and effectiveDate ''' || $sourceEffectiveDate || ''' not found in the context of project ' || $projectPrefix)
            )
        )
        else ()
    
    let $data                   := 
        if ($refOnly) then <template ref="{$sourceId}" flexibility="{$sourceEffectiveDate}"/>
        else if (empty($data)) then 
            let $baseId         := decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)[1]/@id    
            return utiltemp:getTemplateForEdit($sourceTemplate, $decor, (), $targetDate, $keepIds, $baseId, 'new')    
        else $data

    let $storedTemplate         := if ($data/@id[not(. = '')]) then utillib:getTemplateById($data/@id, $data/@effectiveDate) else ()
    let $originalTemplate       := if ($data/@ref[not(. = '')]) then utillib:getTemplateById($data/@ref, if ($data/@flexibility[not(. = '')]) then $data/@flexibility else 'dynamic', $projectPrefix) else () 
    
    let $check                  :=
        if ($storedTemplate) then 
            error($errors:BAD_REQUEST, 'Template with id ''' || ($data/@id)[1] || ''' and effectiveDate ''' || $data/@effectiveDate || ''' already exists in project ' || string-join($storedTemplate/ancestor::decor/project/@prefix, ', '))
        else ()
    let $check                  :=
        if ($data/@ref[not(. = '')]) then 
            if ($originalTemplate) then () else (
                error($errors:BAD_REQUEST, 'Template with id ''' || ($data/@ref)[1] || ''' and effectiveDate ''' || $data/@flexibility || ''' not found in project ' || $projectPrefix)
            )
        else ()
    
    (:relevant only for mode=new and when a concept is connected somewhere, but doesn't hurt to generate for all cases:)
    let $newTemplateElementId   := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TEMPLATEELEMENT, ())[1]/@id
    
    let $preparedTemplate       := 
        if ($originalTemplate) then
            <template ref="{$data/@ref}" name="{$originalTemplate/@name}" displayName="{($originalTemplate/@displayName, $originalTemplate/@name)[not(. = '')][1]}"/>
        else
        if ($keepIds and $data/@id) then
            utiltemp:prepareIncomingTemplateForUpdate($decor, $data, $storedTemplate, 'version', $newTemplateElementId)
        else (
            utiltemp:prepareIncomingTemplateForUpdate($decor, $data, $storedTemplate, 'new', $newTemplateElementId)
        )
    
    let $check                  :=
        if ($preparedTemplate[@id]) then
            if (utillib:getDataset($preparedTemplate/@id, $preparedTemplate/@effectiveDate)) then
                if ($keepIds and $targetDate) then
                    error($errors:BAD_REQUEST, 'Cannot create new template. A template with id ''' || $preparedTemplate/@id || ''' and effectiveDate ''' || $preparedTemplate/@effectiveDate || ''' already exists. You cannot clone into a date based version more than once a day.')
                else (
                    error($errors:BAD_REQUEST, 'Cannot create new template. A template with id ''' || $preparedTemplate/@id || ''' and effectiveDate ''' || $preparedTemplate/@effectiveDate || ''' already exists.')
                )
            else ()
        else ()
    let $check                  := ruleslib:checkDecorSchema(<rules>{$preparedTemplate}</rules>, 'create')
    
    (: Note: this step also rolls up multiple templateAssociation elements into one. This is unlikely to occur unless manually. :)
    let $storedTemplateAssociations     := $decor//templateAssociation[@templateId = $data/@id][@effectiveDate = $data/@effectiveDate]
    let $preparedTemplateAssociation    := if ($preparedTemplate[@id]) then utiltemp:prepareTemplateAssociationForUpdate($storedTemplateAssociations, $preparedTemplate, $data, $newTemplateElementId, 'new') else ()
    
    (: now update the template :)
    let $storedReference        := if ($preparedTemplate[@id]) then () else $decor/rules/template[@ref = $preparedTemplate/@ref]
    let $update                 := if ($storedReference) then update replace $storedReference with $preparedTemplate else update insert $preparedTemplate into $decor/rules
    let $addids                 := if ($preparedTemplate[@id]) then utiltemp:addTemplateElementAndAttributeIds($decor, $preparedTemplate/@id, $preparedTemplate/@effectiveDate) else ()
    let $storedTemplate         := if ($preparedTemplate[@id]) then $decor/rules/template[@id = $preparedTemplate/@id][@effectiveDate = $preparedTemplate/@effectiveDate] else $decor/rules/template[@ref = $preparedTemplate/@ref]
    
    (: now rewrite any templateAssociations. We don't want broken links into elements or attributes that do not exist anymore :)
    let $update                 := 
        if ($storedTemplateAssociations) then (
            update delete $storedTemplateAssociations,
            update insert $preparedTemplateAssociation preceding $storedTemplate
        )
        else
        if ($preparedTemplateAssociation) then (
            update insert text {'&#x0a;        '} preceding $storedTemplate,
            update insert comment {concat(' ',$preparedTemplate/@displayName,' ')} preceding $storedTemplate,
            update insert $preparedTemplateAssociation preceding $storedTemplate
        ) else ()
    
    let $newlock                := if ($lock) then decorlib:setLock($authmap, $preparedTemplate/@id, $preparedTemplate/@effectiveDate, false()) else ()
    
    return
        if ($lock) then
            let $baseId         := decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)[1]/@id    
            return utiltemp:getTemplateForEdit($preparedTemplate, $decor, $newlock, true(), true(), $baseId, 'edit')   
        else if ($preparedTemplate[@id]) 
            then utiltemp:getTemplate($preparedTemplate/@id, $preparedTemplate/@effectiveDate, $projectPrefix, (), ())
            else utiltemp:getTemplate($data/@ref, $data/@flexibility, $projectPrefix, (), ())
};

declare function utiltemp:createTemplateExample($projectPrefixOrId as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $id as xs:string?, $effectiveDate as xs:string?, $elementId as xs:string?, $doSerialized as xs:boolean?, $doRecursive as xs:boolean?, $data as element()?) as element() {
    
    let $template               := if ($data) then $data else (utillib:getTemplateById($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage)[@id])[1]
    let $template               := if ($template) then utiltemp:prepareTemplateForUpdate($template) else ()
    let $format                 := ($template/classification/@format, 'hl7v3xml1')[not(. ='')][1]
    
    let $check                  :=
       if ($template) then 
           if (empty($elementId)) then () else if ($template//*[@id = $elementId]) then () else (
               error($errors:BAD_REQUEST,concat('Argument elementId ', $elementId, ' does not exist in template with id ', $template/@id, ' effectiveDate ', $template/@effectiveDate))
           )
       else (
           error($errors:BAD_REQUEST,concat('Argument id ', $id, ' effectiveDate ', $effectiveDate, ' did not lead to a template and no template provided.'))
       )
    
    let $templateChain          := 
        if ($template and $doRecursive) then (
            let $decor          := utillib:getDecor($projectPrefixOrId)
            let $decors         := utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL)
            return $template | utiltemp:getTemplateChain($template, $decors, map:merge(map:entry(concat($template/@id, $template/@effectiveDate), '')))
       )
       else (
           $template,
           for $elm in $template//element[@contains] | $template//include
           return (utillib:getTemplateById($elm/@contains | $elm/@ref, ($elm/@flexibility, 'dynamic')[1], $projectPrefixOrId, $projectVersion, $projectLanguage)[@id])[1]
       )
    
    let $templateChain          :=
        for $t in $templateChain
        let $ideff              := concat($t/@id, $t/@effectiveDate)
        group by $ideff
        order by $t[1]/@id
        return $t[1]
    
    let $vocabChain             :=
        for $v in $templateChain//vocabulary[@valueSet]
        let $vsid               := $v/@valueSet
        let $vsed               := $v/@flexibility[. castable as xs:dateTime]
        group by $vsid, $vsed
        return utillib:getValueSetByRef($vsid, ($vsed, 'dynamic')[1], $projectPrefixOrId, $projectVersion, $projectLanguage)
    
    let $vocabChain             :=
        for $vs in $vocabChain
        return
        try {
            let $expansion      := utiltcs:getValueSetExpansionSet($vs, map { "maxResults": 20, "expand": true(), "debug": true() })
            return
            <valueSet>
            {
                $vs/@*, 
                if ($vs/@ident) then () else attribute ident {($vs/ancestor::decor/project/@prefix)[1]},
                if ($vs/@url) then () else attribute url {($vs[1]/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]},
                $vs/(* except (completeCodeSystem | conceptList))
            }
                <conceptList>{$expansion/expansion/*}</conceptList>
            </valueSet>
        }
        catch * {
            $vs
        }
    
    let $decor                  := utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)
    let $projectPrefix          := ($decor/project/@prefix)[1]
    let $decorPackage           :=
       <decor>
           {$decor/@*}
           <project>{$decor/project/@*, $decor/project/defaultElementNamespace}</project>
           <terminology>{$vocabChain}</terminology>
           <rules>{$templateChain}</rules>
       </decor>
    
    
    let $xsltParameters         :=
       <parameters>
           <param name="tmid"                      value="{($template/@id, $id)[1]}"/>
           <param name="tmed"                      value="{($template/@effectiveDate, $effectiveDate)[1]}"/>
           <param name="elid"                      value="{$elementId}"/>
           <param name="doSelectedOnly"            value="{false()}"/>
           <param name="doRecursive"               value="{$doRecursive}"/>
           <param name="logLevel"                  value="'OFF'"/>
       </parameters>
    
    let $xslt                   := xs:anyURI('xmldb:exist://' || $setlib:strDecorCore || '/Template2Example.xsl')
    
    (:we need a root element, it's the way it is...:)
    let $tpath                  := replace($template/context/@path,'[/\[].*','')
    let $path                   := if (string-length($tpath)>0) then $tpath else ('art:placeholder')
    let $element                := <element name="{$path}" format="{$format}" xmlns:art="urn:art-decor:example" selected="">{$template/(attribute|element|include|choice)}</element>
    let $example                := if ($decor) then transform:transform($decorPackage, $xslt, $xsltParameters) else ()
    
    (:if we did not need our pseudo root-element, just leave it off:)
    let $example                := if ($example[count(@*)>0 or count(*)>1]) then $example else $example/node()
    
    return
       <example caption="" type="neutral" xmlns:json="http://www.json.org">
       {
           if ($doSerialized) then
               fn:serialize($example,
                   <output:serialization-parameters xmlns:output="http://www.w3.org/2010/xslt-xquery-serialization">
                       <output:method>xml</output:method>
                       <output:encoding>UTF-8</output:encoding>
                   </output:serialization-parameters>
               )
           else $example
       }
       </example>
       

};

(:~ Central logic for updating an existing template
    @param $authmap                 - required. Map derived from token
    @param $id                       - required. the id for the template to update
    @param $effectiveDate            - required. the effectiveDate for the template to update
    @param $request-body             - required. json body containing new template structure
    @return template structure
:)
declare function utiltemp:putTemplate($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(), $deletelock as xs:boolean) as element(template) {

    let $storedTemplate         := $setlib:colDecorData//template[@id = $data/@originalId][@effectiveDate = $effectiveDate] 
    let $decor                  := $storedTemplate/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($storedTemplate) then () else (
            error($errors:BAD_REQUEST, 'Template with id ''' || ($data/@originalId, $id)[1] || ''' and effectiveDate ''' || $effectiveDate || ''' does not exist')
        )
    
    let $lock                   := utiltemp:checkTemplateAccess($authmap, $decor, $id, $effectiveDate, true(), false())
    
    let $check                  :=
        if ($storedTemplate[@statusCode = $utiltemp:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, concat('Template cannot be updated while it has one of status: ', string-join($utiltemp:STATUSCODES-FINAL, ', '), if ($storedTemplate/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
        else ()
    
    let $check                  :=
        if ($data[@originalId = $id] | $data[empty(@originalId)][@id = $id]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no template id or the same template id as the template id ''', $id, ''' used for updating. Found in request body: ', ($data/@originalId, $data/@id, 'null')[1]))
        )
    
    let $check                  :=
        if ($data[@effectiveDate = $effectiveDate] | $data[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no template effectiveDate or the same template effectiveDate as the template effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($data/@effectiveDate, 'null')[1]))
        )
    
    (:relevant only for mode=new and when a concept is connected somewhere, but doesn't hurt to generate for all cases:)
    let $newTemplateElementId   := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TEMPLATEELEMENT, ())[1]/@id
    
    let $preparedTemplate       := utiltemp:prepareIncomingTemplateForUpdate($decor, $data, $storedTemplate, 'edit', $newTemplateElementId)
    
    let $check                  := ruleslib:checkDecorSchema(<rules>{$preparedTemplate}</rules>, 'update')

    (: Note: this step also rolls up multiple templateAssociation elements into one. This is unlikely to occur unless manually. :)
    let $storedTemplateAssociations     := $decor//templateAssociation[@templateId = $data/@originalId][@effectiveDate = $data/@effectiveDate]
    let $preparedTemplateAssociation    := utiltemp:prepareTemplateAssociationForUpdate($storedTemplateAssociations, $preparedTemplate, $data, $newTemplateElementId, 'edit')
        
    let $intention              := if ($storedTemplate[@statusCode = ('active', 'final')]) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-TEMPLATE, $projectPrefix, $intention, $storedTemplate)
    
    (: now update the template :)
    let $update                 := update replace $storedTemplate with $preparedTemplate
    let $addids                 := utiltemp:addTemplateElementAndAttributeIds($decor, $preparedTemplate/@id, ($preparedTemplate/@effectiveDate)[1])
    
    (: now rewrite any templateAssociations. We don't want broken links into elements or attributes that do not exist anymore :)
    let $update                 := 
        if ($storedTemplateAssociations) then (
            update delete $storedTemplateAssociations,
            update insert $preparedTemplateAssociation preceding $storedTemplate
        )
        else (
            update insert text {'&#x0a;        '} preceding $storedTemplate,
            update insert comment {concat(' ',$preparedTemplate/@displayName,' ')} preceding $storedTemplate,
            update insert $preparedTemplateAssociation preceding $storedTemplate
        )
    
    let $deleteLock             := if ($deletelock) then update delete $lock else ()
    
    return
        if ($deletelock) then utiltemp:getTemplate($preparedTemplate/@id, $preparedTemplate/@effectiveDate, $projectPrefix, (), ())
        else (
            let $baseId         := decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)[1]/@id    
            return utiltemp:getTemplateForEdit($preparedTemplate, $decor, $lock, true(), true(), $baseId, 'edit')   
        )
};

(:~ Central logic for patching an existing template
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR template/@id to update
    @param $effectiveDate   - required. DECOR template/@effectiveDate to update
    @param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
    @return template object
:)
declare function utiltemp:patchTemplate($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) as element(template) {

    let $storedTemplate         := $setlib:colDecorData//template[@id = $id][@effectiveDate = $effectiveDate]
    let $decor                  := $storedTemplate/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $check                  :=
        if ($storedTemplate) then () else (
            error($errors:BAD_REQUEST, 'Template with id ' || $id || ' and effectiveDate ' || $effectiveDate || ''' does not exist')
        )
    
    let $lock                   := utiltemp:checkTemplateAccess($authmap, $decor, $id, $effectiveDate, true(), false())

    let $check                  :=
        if ($storedTemplate[@statusCode = $utiltemp:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $utiltemp:STATUSCODES-FINAL)]) then () else
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('Template cannot be patched while it has one of status: ', string-join($utiltemp:STATUSCODES-FINAL, ', '), if ($storedTemplate/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
    
    let $check                  := utiltemp:checkTemplateParameters($data, $storedTemplate)
    
    let $intention              := if ($storedTemplate[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-TEMPLATE, $projectPrefix, $intention, $storedTemplate)
    
    let $patchedTemplate        := utiltemp:patchTemplateParameters($data, $storedTemplate)
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $preparedTemplate       := utiltemp:prepareTemplateForUpdate($patchedTemplate)
    
    let $update                 := update replace $storedTemplate with $preparedTemplate
    let $update                 := update delete $lock
    
    return utiltemp:getTemplate($preparedTemplate/@id, $preparedTemplate/@effectiveDate, $projectPrefix, (), ())
};

(:~ Central logic for patching an existing templateAssociation

@param $authmap         - required. Map derived from token
@param $id              - required. DECOR template/@id to update
@param $effectiveDate   - required. DECOR template/@effectiveDate to update
@param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
@return templateAssociation object
:)
declare function utiltemp:patchTemplateAssociation($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $decorOrPrefix as item(), $data as element(parameters)) as element(template) {
    
    let $decor                  := utillib:getDecor($decorOrPrefix, (), ())

    let $check                  :=
        if (empty($decor)) then
            error($errors:BAD_REQUEST, 'Project with prefix or id ' || $decorOrPrefix || ' not found.')
        else ()
        
    let $check                  := utiltemp:checkTemplateAccess($authmap, $decor, (), (), false(), false())
    
    let $storedTemplate         := utillib:getTemplateById($id, $effectiveDate, $decorOrPrefix, (), ())
    
    let $check                  :=
        if (count($storedTemplate) gt 1) then
            error($errors:SERVER_ERROR, 'Cannot patch template with id "' || $id || '" effectiveDate "' || $effectiveDate || '" project "' || ($decor/project/@prefix)[1] || '", because there are ' || count($storedTemplate) || ' matching templates. Please inform your administrator.')
        else ()
    
    let $updateTemplate         := $decor//template[@id = $id][@effectiveDate = $effectiveDate] | $decor//template[@ref = $id]
    
    let $check                  :=
        if ($storedTemplate) then () else (
            error($errors:BAD_REQUEST, 'Template with id ' || $id || ' and effectiveDate ' || $effectiveDate || ''' does not exist')
        )
 
    let $check                  :=
        if ($data[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $data/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $check                  :=
        for $param in $data/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/concept' return (
                if ($param[count(value/concept) = 1]) then (
                    if ($op = 'remove') then () else 
                    if ($param/value/concept[@ref][@effectiveDate][@elementId]) then (
                        let $datasetConcept := utillib:getConcept($param/value/concept/@ref, $param/value/concept/@effectiveDate)
                        return
                        if (empty($datasetConcept)) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. TemplateAssociation concept id ''' || $param/value/concept/@ref || ''' effectiveDate ''' || $param/value/concept/@effectiveDate || ''' not found.'
                        else 
                        if ($datasetConcept/ancestor::decor/project[not(@prefix = $decor/project/@prefix)]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. TemplateAssociation concept id ''' || $param/value/concept/@ref || ''' effectiveDate ''' || $param/value/concept/@effectiveDate || ''' SHALL be in the same project. Project indicated: ' || $decor/project/@prefix || ', concept in ' || ($datasetConcept/ancestor::decor/project/@prefix) 
                        else (),
                        if ($storedTemplate//attribute[@id = $param/value/concept/@elementId] | $storedTemplate//element[@id = $param/value/concept/@elementId]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. TemplateAssociation attribute or element id ''' || $param/value/concept/@elementId || ''' not found.'
                        )
                    ) else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. A templateAssociation concept SHALL have a ref, effectiveDate and elementId.'
                    )
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single concept under value. Found ' || count($param/value/concept)
                )  
            )
              
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     let $check                 :=
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )
    
    let $update                 :=
        for $param in $data/parameter
        return
            switch ($param/@path)
            case '/concept' return (
                let $paramValue := $param/value/concept
                let $preparedTemplateAssociation := <templateAssociation templateId="{$storedTemplate/@id}" effectiveDate="{$storedTemplate/@effectiveDate}"/>
                    let $new      := 
                    <concept>
                    {
                        $paramValue/@ref,
                        $paramValue/@effectiveDate,
                        ($paramValue/@elementId, $paramValue/@elementPath)[1]  
                    }
                    </concept>
                let $storedTemplateAssociation := $decor//templateAssociation[@templateId = $storedTemplate/@id][@effectiveDate = $storedTemplate/@effectiveDate] 
                let $storedTemplateAssociation := 
                    if ($storedTemplateAssociation) then ($storedTemplateAssociation) else (
                        let $update := update insert text {'&#x0a;        '} preceding $updateTemplate
                        let $update := update insert comment {concat(' ',$storedTemplate/@displayName,' ')} preceding $updateTemplate
                        let $update := update insert $preparedTemplateAssociation preceding $updateTemplate
                        
                        return
                        $decor//templateAssociation[@templateId = $storedTemplate/@id][@effectiveDate = $storedTemplate/@effectiveDate]
                    )
                let $stored                    := $storedTemplateAssociation/concept[@ref = $paramValue/@ref][@effectiveDate = $paramValue/@effectiveDate]
                let $stored                    := 
                    if ($paramValue/@elementId) then ($stored[@elementId = $paramValue/@elementId]) else 
                    if ($paramValue/@elementPath) then ($stored[@elementPath = $paramValue/@elementPath]) else ($stored)
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplateAssociation[1]
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
    
    return $decor//templateAssociation[@templateId = $storedTemplate/@id][@effectiveDate = $storedTemplate/@effectiveDate]
};

(:~ Central logic for patching an existing transaction statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap required. Map derived from token
    @param $id required. DECOR concept/@id to update
    @param $effectiveDate required. DECOR concept/@effectiveDate to update
    @param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function utiltemp:setTemplateStatus($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $recurse as xs:boolean, $listOnly as xs:boolean, $newStatusCode as xs:string?, $newVersionLabel as xs:string?, $newExpirationDate as xs:string?, $newOfficialReleaseDate as xs:string?) as element()* {
    
    (:get object for reference:)
    let $object                 := 
        if ($effectiveDate castable as xs:dateTime) then $setlib:colDecorData//template[@id = $id][@effectiveDate = $effectiveDate]
        else (
            let $set := $setlib:colDecorData//template[@id = $id]
            return
                $set[@effectiveDate = max($set/xs:dateTime(@effectiveDate))]
        )
    
    let $decor                  := $object/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($object) = 1) then () else
        if ($object) then
            error($errors:SERVER_ERROR, 'Template id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($object) || ' were found in: ' || string-join(distinct-values($object/ancestor::decor/project/@prefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Template id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )

    let $check                  := utiltemp:checkTemplateAccess($authmap, $decor, (), (), false(), false())
    
    let $templateChain          :=
        if ($recurse) then (
            let $templatechain  := utiltemp:getTemplateList((), $projectPrefix, (), (), (), $object/@id, $object/@effectiveDate, $utiltemp:TREETYPELIMITEDMARKED, false())
            for $t in $templatechain//template
            return $decor//template[@id = $t/@id][@effectiveDate = $t/@effectiveDate]
        ) 
        else $object

    let $testUpdate             :=
        for $object in $templateChain
        return utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, true(), $projectPrefix)
    
    return
        if ($testUpdate[self::error]) then $testUpdate else
        if ($listOnly) then $testUpdate else (
            for $object in $templateChain
            return utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, false(), $projectPrefix)
        )
    
    
};

(:~ Retrieves a template in edit mode
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @return template
    @since 2025-04-11
:)
declare function utiltemp:getTemplateForEdit($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $breaklock as xs:boolean) {

    let $template               := $setlib:colDecorData//template[@id = $id][@effectiveDate = $effectiveDate]
    
    let $decor                  := $template/ancestor::decor

    let $check                  :=
        if ($template) then () else (
            error($errors:BAD_REQUEST, 'Template with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' not found')
        )
    
    let $lock                   := utiltemp:checkTemplateAccess($authmap, $decor, $id, $effectiveDate, true(), $breaklock)
    
    let $check                  :=
        if ($template[@statusCode = $utiltemp:STATUSCODES-FINAL]) then (
            error($errors:BAD_REQUEST, 'Template cannot be updated while it has one of status: ' || string-join($utiltemp:STATUSCODES-FINAL, ', ') || (if ($template/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()

    let $baseId                 := decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)[1]/@id    
    
    return utiltemp:getTemplateForEdit($template, $decor, $lock, false(), true(), $baseId, 'edit')

};

(:~ Return zero or more expanded templates

    @param $id            - required. Identifier of the template to retrieve
    @param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
    @return Matching templates
    @since 2023-09-10

:)
declare function utiltemp:getExpandedTemplateById($id as xs:string, $flexibility as xs:string?) as element(template)* {

    for $template in utillib:getTemplateById($id, $flexibility)

    let $prefix                 := $template/ancestor::decor/project/@prefix
    let $url                    := ($template/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
    group by $url, $prefix
        return utiltemp:getExpandedTemplate($template, $template[1]/ancestor::decor, (), ())
};

(:~ Return zero or more expanded templates

    Inside the path /return/template[@id or @ref][@ident] is 1..* template element. This may is the expanded template (@id or @ref) with additions:
    - when the template originates from a different project than parent::template/@ident, the origin is added using template/@url and template/@ident.
    - the item element is added at every relevant level where the template itself doesn't explicitly define it
    - the include element will have the extra attributes tmid, tmname, tmdisplayName containing the id/name/displayName of the referred template (unless it cannot be found)
    - the include element will have an extra attribute linkedartefactmissing='true' when the referred template cannot be found
    - the include element will contain the template it refers (unless it cannot be found)
    - the element[@contains] element will have the extra attributes tmid, tmname, tmdisplayName containing the id/name/displayName of the contained template (unless it cannot be found)
    - the element[@contains] element will have an extra attribute linkedartefactmissing='true' when the contained template cannot be found
    - the vocabulary[@valueSet] element will have the extra attributes vsid, vsname, vsdisplayName containing the id/name/displayName of the referred value set (unless it cannot be found)
    - the vocabulary[@valueSet] element will have an extra attribute linkedartefactmissing='true' when the referred value set cannot be found
    - at the bottom of the template there may be extra elements:
      - If this is a template that is used in a transaction: &lt;representingTemplate ref="..." flexibility="..." model="..." sourceDataset="..." type="stationary" schematron="vacc-vacccda2"/>
      - For every include[@ref] / element[@contains]        : &lt;ref type="contains|include" id="..." name="..." displayName="..." effectiveDate="...."/>
      - For every project template that calls this template : &lt;ref type="dependency" id="..." name="..." displayName="..." effectiveDate="...."/>
        &lt;staticAssociations>
            
            - For every template association:
            &lt;origconcept datasetId="..." datasetEffectiveDate="..." ref="..." effectiveDate="..." elementId="..." path="...">
                &lt;concept id=".." effectiveDate="...">
                    &lt;name language="nl-NL">...</name>
                    &lt;desc language="nl-NL">...</desc>
                &lt;/concept>
            &lt;/origconcept>
            
        &lt;/staticAssociations>
    
    @param $id            - required. Identifier of the template to retrieve
    @param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
    @param $decorOrPrefix - required. determines search scope. pfx- limits scope to this project only
    @param $decorRelease  - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
    @param $decorLanguage - optional. Language compilation. Defaults to project/@defaultLanguage
    @return Matching templates
    @since 2014-06-20
:)
declare function utiltemp:getExpandedTemplateById($id as xs:string, $flexibility as xs:string?, $decorOrPrefix as item(), $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(template)* {
    
    for $template in utillib:getTemplateById($id, $flexibility, $decorOrPrefix, $decorRelease, $decorLanguage)
    let $prefix             := $template/ancestor::decor/project/@prefix
    let $url                := ($template/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
    group by $url, $prefix
    return utiltemp:getExpandedTemplate($template, $prefix, (), ())
};

(:~ Expands a template with @id by recursively resolving all includes. Use utiltemp:getExpandedTemplateById to resolve a template/@ref first. 
                            
    @param $template      - required. The template element with content to expand
    @param $decorOrPrefix - required. The origin of template. pfx- limits scope this project only
    @param $decorRelease  - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
    @param $decorLanguage - optional. Language compilation. Defaults to project/@defaultLanguage
    @return The expanded template
    @since 2013-06-14
:)
declare function utiltemp:getExpandedTemplate($template as element(), $decorOrPrefix as item(), $decorRelease as xs:string?, $decorLanguage as xs:string?) as element() {
    (: all rules and terminologies of this project for later use :)
    let $decor                  := utillib:getDecor($decorOrPrefix, $decorRelease, $decorLanguage)[1]
    let $decor                  := if ($decor) then $decor else ($setlib:colDecorCache//decor[project/@prefix = $decorOrPrefix])[1]
    let $prefix                 := $decor/project/@prefix
    let $language               := ($decor/@language, $decor/project/@defaultLanguage)[1]
    
    (: all transactions where this template is ref'ed :)
    let $decorRepresentingTemplates     := $setlib:colDecorData//representingTemplate[@ref = $template/@id]
    let $newestTemplateEffectiveDate    := utillib:getTemplateById($template/(@id | @ref), 'dynamic')/@effectiveDate
    let $currentTemplateIsNewest        := exists(
       $template[@effectiveDate = $newestTemplateEffectiveDate] |
       $template[@ref][empty(@flexibility)] |
       $template[@ref][@flexibility = 'dynamic'] |
       $template[@ref][@flexibility = $newestTemplateEffectiveDate]
    )
    
    let $theitem := if ($template/item) then $template/item[1] else <item original="false" label="{$template/@name}"/>
    
    return
        <template>
        {
            $template/@id,
            $template/@name,
            $template/@displayName,
            $template/@effectiveDate,
            $template/@statusCode,
            $template/@versionLabel,
            $template/@expirationDate,
            $template/@officialReleaseDate,
            $template/@canonicalUri,
            attribute isClosed {$template/@isClosed = 'true'},
            $template/@lastModifiedDate,
            attribute url {($template/@url, $template/ancestor::*/@bbrurl, $utillib:strDecorServicesURL)[1]},
            attribute ident {($template/@ident, $template/ancestor::*/@bbrident, $template/ancestor::decor/project/@prefix)[1]},
            for $ns at $i in utillib:getDecorNamespaces($template/ancestor::decor) 
            return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri},
            ($template/desc),
            for $node in $template/(* except desc)
            return utiltemp:copyNodes($node, $theitem, 1, $decor),
            (: get where the template is a representing template :)
            for $rtemp in $decorRepresentingTemplates
             (:starts with 4 digits, explicit dateTime else empty or dynamic :)
            let $rtempFlexibility   := if ($rtemp[@flexibility castable as xs:dateTime]) then string($rtemp/@flexibility) else $newestTemplateEffectiveDate
            let $rtempFlexValue     :=if (string-length($rtemp/@flexibility)>0) then $rtemp/@flexibility else 'dynamic'
            let $rtempTransaction   := $rtemp/parent::transaction
            let $ident              := $rtemp/ancestor::decor/project/@prefix
            return
            if ($rtempFlexibility = $template/@effectiveDate) then
                <representingTemplate ref="{$rtemp/@ref}" flexibility="{$rtempFlexValue}" schematron="{$prefix}{$rtempTransaction/@label}" transactionId="{$rtempTransaction/@id}" transactionEffectiveDate="{$rtempTransaction/@effectiveDate}">
                {
                    $rtemp/@sourceDataset,
                    $rtemp/@sourceDatasetFlexibility,
                    $rtempTransaction/@type,
                    $rtempTransaction/@model,
                    $rtempTransaction/@statusCode,
                    $rtempTransaction/@versionLabel,
                    attribute ident {$ident},
                    $rtempTransaction/name
                }
                </representingTemplate>
            else()
            ,
            utiltemp:createStaticAssociationElement($decor/rules/templateAssociation[@templateId = $template/@id][@effectiveDate = $template/@effectiveDate], $language)
        }
        </template>
};

declare %private function utiltemp:copyNodes($tnode as element(), $item as element(), $nesting as xs:integer, $decor as element(decor)) as element()* {
    let $elmname                := name($tnode)
    let $theitem                := if ($tnode/item) then $tnode/item[1] else <item original="false" label="{$item/@label}">{$item/desc}</item>
    let $language               := ($decor/@language, $decor/project/@defaultLanguage)[1]
    let $tnodedecor             := ($tnode/ancestor::decor, $decor)[1]
    
    return
        (: too deeply nested, raise error and give up :)
        if ($nesting > 30) then (
            element templateerror {
                attribute {'type'} {'nesting'}
            }
        ) 
        else (
            switch (name($tnode))
            case 'include' return utiltemp:copyNodesInclude($tnode, $theitem, $nesting, $decor)
            (: fall through, just copy node :)
            case 'desc'
            case 'constraint'
            case 'purpose'
            case 'copyright'
            case 'report'
            case 'assert'
            case 'let'
            case 'defineVariable'
            case 'classification'
            case 'publishingAuthority'
            case 'context'
            case 'text' return $tnode
            case 'example' return <example type="{($tnode/@type, 'neutral')[1]}">{$tnode/@caption, $tnode/node()}</example>
            case 'relationship' return utiltemp:copyNodesRelationship($tnode, $decor)
            case 'vocabulary' return utiltemp:copyNodesVocabulary($tnode, $decor)
            case 'attribute' return utiltemp:copyNodesAttribute($tnode, $theitem, $nesting, $decor)
            default return utiltemp:copyNodesDefault($tnode, $theitem, $nesting, $decor)
        )
};

declare %private function utiltemp:copyNodesInclude($tnode as element(), $theitem as element(), $nesting as xs:integer, $decor as element(decor)) as element()* {
    
    let $artefact               := if ($tnode/@ref) then utillib:getTemplateById($tnode/@ref, if ($tnode[@flexibility]) then $tnode/@flexibility else ('dynamic'), $decor/rules/ancestor::decor/project/@prefix, (), ())[@id][1] else ()
    let $tmpurl                 := ($artefact/@url, $artefact/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
    let $tmpident               := ($artefact/@ident, $artefact/ancestor::decor/project/@prefix)[1]
    
    return
        element include {
            $tnode/(@* except (@tmid|@tmname|@tmdisplayName|@tmeffectiveDate|@tmexpirationDate|@tmstatusCode|@tmversionLabel|@linkedartefactmissing|@url|@ident)),
            if ($tnode/@ref) then (
                for $att in $artefact/(@id, @name, @displayName, @effectiveDate, @expirationDate, @statusCode, @versionLabel)
                return attribute {'tm' || name($att)} {$att}
                ,
                if ($tmpurl) then attribute url {$tmpurl} else (),
                if ($tmpident) then attribute ident {$tmpident} else (),
                attribute linkedartefactmissing {empty($artefact)}
            ) else ()
            ,
            $tnode/text(),
            $tnode/desc,
            $theitem,
            for $s in $tnode/(* except (desc|item))
            return utiltemp:copyNodes($s, $theitem, $nesting+1, $decor)
            ,
            let $recentcardconf := utiltemp:cardconfs1element ($artefact, $tnode/@minimumMultiplicity, $tnode/@maximumMultiplicity, $tnode/@isMandatory, $tnode/@conformance)
            let $theitem        := if ($artefact/item) then $artefact/item[1] else <item original="false" label="{$artefact/@name}"/>
            for $t in $recentcardconf
            return utiltemp:copyNodes($t, $theitem, $nesting+1, $decor),
            if (empty($artefact)) then () else utiltemp:createStaticAssociationElement($decor/rules/templateAssociation[@templateId = $artefact/@id][@effectiveDate = $artefact/@effectiveDate], ($decor/@language, $decor/project/@defaultLanguage)[1])
        }
};

declare %private function utiltemp:copyNodesRelationship($tnode as element(), $decor as element(decor)) as element()* {

    let $artefact               := if ($tnode/@template) then utillib:getTemplateById($tnode/@template, if ($tnode[@flexibility]) then $tnode/@flexibility else ('dynamic'), $decor/rules/ancestor::decor/project/@prefix, (), ())[@id][1] else ()
    let $tmpurl                 := ($artefact/@url, $artefact/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
    let $tmpident               := ($artefact/@ident, $artefact/ancestor::decor/project/@prefix)[1]    
    
    return
        element relationship {
            $tnode/(@* except (@tmid|@tmname|@tmdisplayName|@tmeffectiveDate|@tmexpirationDate|@tmstatusCode|@tmversionLabel|@linkedartefactmissing|@url|@ident)),
            if ($tnode/@template) then (
                for $att in $artefact/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                return attribute {'tm' || name($att)} {$att}
                ,
                if ($tmpurl) then attribute url {$tmpurl} else (),
                if ($tmpident) then attribute ident {$tmpident} else (),
                attribute linkedartefactmissing {empty($artefact)}
            ) else ()
            ,
            $tnode/*
        }
};

declare %private function utiltemp:copyNodesVocabulary($tnode as element(), $decor as element(decor)) as element()* {

    let $tnodedecor             := ($tnode/ancestor::decor, $decor)[1]
    let $artefact               := utiltemp:artefactMissing('valueSet', $tnode/@valueSet, $tnode/@flexibility, $tnodedecor)
    
    return
        element vocabulary {
            $tnode/(@* except (@vsid|@vsname|@vsdisplayName|@vseffectiveDate|@vsexpirationDate|@vsstatusCode|@vsversionLabel|@linkedartefactmissing|@url|@ident)),
            if ($tnode/@valueSet) then (
                for $att in $artefact/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                return attribute {'vs' || name($att)} {$att}
                ,
                $artefact/@url,
                $artefact/@ident,
                attribute {'linkedartefactmissing'} {$artefact/@missing}
            ) else (),
            $tnode/*
        }
};

declare %private function utiltemp:copyNodesAttribute($tnode as element(), $theitem as element(), $nesting as xs:integer, $decor as element(decor)) as element()* {

    for $s in utiltemp:normalizeAttributes($tnode)
        return
        element attribute {
            $s/@*,
            for $t in $s/* 
            return utiltemp:copyNodes($t, $theitem, $nesting+1, $decor)
        }
};

declare %private function utiltemp:copyNodesDefault($tnode as element(), $theitem as element(), $nesting as xs:integer, $decor as element(decor)) as element()* {
    
    let $tnodedecor             := ($tnode/ancestor::decor, $decor)[1]
    let $artefact               := if ($tnode/@contains) then utiltemp:artefactMissing('template', $tnode/@contains, $tnode/@flexibility, $tnodedecor) else ()
    
    return
        element {name($tnode)} {
            $tnode/(@* except (@tmid|@tmname|@tmdisplayName|@tmeffectiveDate|@tmstatusCode|@linkedartefactmissing)),
            if ($tnode/@contains) then (
                for $att in $artefact/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                return attribute {'tm' || name($att)} {$att}
                ,
                $artefact/@url,
                $artefact/@ident,
                attribute {'linkedartefactmissing'} {$artefact/@missing}
            ) else ()
            ,
            $tnode/desc,
            if (name($tnode) = ('template', 'item')) then () else $theitem,
            for $s in $tnode/(* except (desc|item))
            return utiltemp:copyNodes($s, $theitem, $nesting+1, $decor)
        }
};

(:
    Accepts 1 or more templates, although typically only 1 (TODO: change $e to element(template)? instead of element()* ?), and will use parameters 
    $minimumMultiplicity, $maximumMultiplicity, $isMandatory and $conformance to override all top level element of those templates.
    
    This is applicable to template content that is called via <include/> where the card/conf of the include overrides corresponding properties of the 
    top level template element it is calling.
    
    Overrides are applicable to <element/>, <include/> and <choice/>. If any of the card/conf parameters has no value then the value (if any) of the 
    top level element applies.
:)
declare %private function utiltemp:cardconfs1element($e as element()*, $minimumMultiplicity as xs:string?, $maximumMultiplicity as xs:string?, $isMandatory as xs:string?, $conformance as xs:string? ) as element()* {
    
    for $child in $e/(element|include|choice|attribute|assert|report|let|defineVariable|constraint)
    let $minimumMultiplicity    := if (string-length($minimumMultiplicity)=0) then ($child/@minimumMultiplicity) else ($minimumMultiplicity)
    let $maximumMultiplicity    := if (string-length($maximumMultiplicity)=0) then ($child/@maximumMultiplicity) else ($maximumMultiplicity)
    let $isMandatory            := if (string-length($isMandatory)=0) then ($child/@isMandatory) else ($isMandatory)
    let $conformance            := if (string-length($conformance)=0) then ($child/@conformance) else ($conformance)
    return
        if ($child/self::element | $child/self::include) then (
            element {$child/name()} {
                $child/(@* except (@minimumMultiplicity|@maximumMultiplicity|@isMandatory|@conformance)),
                if (string-length($minimumMultiplicity)>0) then attribute minimumMultiplicity {$minimumMultiplicity} else (),
                if (string-length($maximumMultiplicity)>0) then attribute maximumMultiplicity {$maximumMultiplicity} else (),
                if (string-length($isMandatory)>0) then attribute isMandatory {$isMandatory} else (),
                if (string-length($conformance)>0) then attribute conformance {$conformance} else (),
                $child/node()
            }
        ) else if ($child/self::choice) then (
            element {$child/name()} {
                $child/(@* except (@minimumMultiplicity|@maximumMultiplicity|@isMandatory|@conformance)),
                if (string-length($minimumMultiplicity)>0) then attribute minimumMultiplicity {$minimumMultiplicity} else (),
                if (string-length($maximumMultiplicity)>0) then attribute maximumMultiplicity {$maximumMultiplicity} else (),
                $child/node()
            } 
        ) else (
            $child
        )
};

declare %private function utiltemp:artefactMissing($what as xs:string, $ref as xs:string?, $flexibility as xs:string?, $decor as element(decor)) as element()? {
    
    let $ref                    := $ref[string-length() gt 0]
    let $flexibility            := if ($flexibility castable as xs:dateTime) then $flexibility else 'dynamic'
    return
    (: returns <artefact missing="true"... if artefact cannot be found in decor project :)
    if (empty($decor)) then
        <artefact missing="false" id="{$ref}" name="" displayName="{$what}" effectiveDate="{$flexibility}" statusCode="oops"/>
    else 
    if (empty($ref)) then 
        <artefact missing="false" id="" name="" displayName="" effectiveDate="" statusCode=""/>
    else (
        switch ($what)
        case 'template' return (
            let $tmp            := (utillib:getTemplateById($ref, $flexibility, $decor, $decor/@versionDate, $decor/@language)[@id])
            let $tmpurl         := ($tmp/@url, $tmp/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
            let $tmpident       := ($tmp/@ident, $tmp/ancestor::decor/project/@prefix)[1]
            return
                <artefact missing="{empty($tmp)}" id="{$tmp[1]/(@id|@ref)}">
                {
                    $tmp[1]/(@* except (@id|@ref|@url|@ident)),
                    if ($tmpurl) then attribute url {$tmpurl} else (),
                    if ($tmpident) then attribute ident {$tmpident} else ()
                }
                </artefact>
        )
        case 'valueSet' return (
            (: find out effectiveDate for this value set :)
            let $tmp            := utillib:getValueSetByRef($ref, $flexibility, $decor, (), ())
            return
                <artefact missing="{empty($tmp)}" id="{$tmp[1]/(@id|@ref)}">{$tmp[1]/(@* except (@id|@ref))}</artefact>
        )
        default return ()
    )
};

(:recursive function to get all templates hanging off from the current template. has circular reference protection. Does not return the start templates :)
declare function utiltemp:getTemplateChain($startTemplates as element(template)*, $decorsInScope as element(decor)*, $resultsmap as map(*)) as element(template)* {

    for $ref in ($startTemplates//element/@contains | $startTemplates//include/@ref)
    let $flexibility            := if ($ref/../@flexibility) then $ref/../@flexibility else ('dynamic')
    group by $ref, $flexibility
    
    return (
        let $templates          := 
            if ($flexibility castable as xs:dateTime) then $decorsInScope//template[@id = $ref][@effectiveDate = $flexibility]
            else (
                let $t          := $decorsInScope//template[@id = $ref]
                return $t[@effectiveDate = string(max($t/xs:dateTime(@effectiveDate)))]
            )
         let $templates         :=
            for $template in $templates
            return
                element {name($template)}
                {
                    for $ns at $i in utillib:getDecorNamespaces($template/ancestor::decor)
                    return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri}
                    ,
                    if ($template[@url | @ident]) then $template/(@url | @ident) 
                    else if ($template[ancestor::cacheme]) then (
                        attribute url {$template[1]/ancestor::cacheme/@bbrurl},
                        attribute ident {$template[1]/ancestor::cacheme/@bbrident}
                    )
                    else ()
                    ,
                    $template/ancestor::decor/project/@defaultLanguage,
                    $template/(@* except (@url | @ident | @defaultLanguage)),
                    $template/*
                }
        
        let $newkey             := concat($templates[1]/@id, $templates[1]/@effectiveDate)
        
        return
            if ($templates) then
                if (map:contains($resultsmap, $newkey)) then () else (
                    let $newmap     := map:merge(($resultsmap, map:merge(map:entry($newkey, ''))))
                    return $templates | utiltemp:getTemplateChain($templates, $decorsInScope, $newmap)
                )
            else ()
    )
};

(:~ Rewrite an incoming template back into db format

    @param $decor - decor project to update results into
    @param $inputTemplate - incoming raw template
    @param $storedTemplate - template in db to update (if any)
    @param $mode - 'new' (completely new), 'edit' (edit of existing, same id/effectiveDate), 'version' (clone under same id, new effectiveDate), 'adapt' (clone under new id and effectiveDate)
:)
declare %private function utiltemp:prepareIncomingTemplateForUpdate($decor as element(decor), $inputTemplate as element(template), $storedTemplate as element(template)?, $mode as xs:string, $newTemplateElementId as xs:string?) as element(template) {

    let $originalId             := ($inputTemplate/@originalId, $inputTemplate/@id)[1]
    let $templateBaseIds        := decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)
    
    (:if a baseId is present use it, else use default template baseId:)
    let $templateRoot           :=
        if ($inputTemplate/@baseId) then
            if ($templateBaseIds[@id = $inputTemplate/@baseId]) then $inputTemplate/@baseId else (
                error($errors:BAD_REQUEST, 'Submitted data SHALL specify a known baseId in project ' || $decor/project/@prefix || '. BaseId ''' || $inputTemplate/@baseId || ''' not found')
            )
        else (
            decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)[1]/@id
        )
    
    (:generate new id if mode is 'new' or 'adapt', else use existing id:)
    let $newTemplateId          := if ($mode=('new','adapt')) then decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TEMPLATE, $templateRoot)[1]/@id else $inputTemplate/@id
    
    let $templateName           := attribute name {($inputTemplate/@name, utillib:getNameForOID($newTemplateId, (), $decor))[not(.='')][1]}
    let $templateDisplayName    := attribute displayName {($inputTemplate/@displayName, $templateName)[not(.='')][1]}

    (:keep effectiveDate for mode is 'edit', else generate new:)
    let $templateEffectiveTime  := 
        if ($mode = 'edit' and $storedTemplate[@effectiveDate castable as xs:dateTime]) then $storedTemplate/@effectiveDate
        else if ($inputTemplate[@effectiveDate castable as xs:dateTime][not(@effectiveDate = $storedTemplate/@effectiveDate)]) then $inputTemplate/@effectiveDate
        else attribute effectiveDate {format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')}
    (:set draft effectiveDate for mode is 'edit', else generate new:)
    let $templateStatusCode     := if ($inputTemplate/@statusCode[. = ('','new')]) then attribute statusCode {'draft'} else $inputTemplate/@statusCode
    
    return (
        <template>
        {
            $newTemplateId,
            $templateName,
            $templateDisplayName,
            $templateEffectiveTime,
            $templateStatusCode,
            $inputTemplate/@versionLabel[not(. = '')],
            $inputTemplate/@expirationDate[. castable as xs:dateTime],
            $inputTemplate/@officialReleaseDate[. castable as xs:dateTime],
            $inputTemplate/@canonicalUri[not(. = '')],
            $inputTemplate/@isClosed[. = ('true', 'false')],
            attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)},
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($inputTemplate/desc),
            utiltemp:prepareClassificationForUpdate($inputTemplate/classification),
            utiltemp:prepareRelationshipForUpdate($inputTemplate/relationship),
            utillib:preparePublishingAuthorityForUpdate($inputTemplate/publishingAuthority[empty(@inherited)][@name[not(. = '')]][1])
            ,
            for $node in $inputTemplate/purpose
            return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
            ,
            for $node in $inputTemplate/copyright[empty(@inherited)]
            return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
            ,
            utiltemp:prepareContextForUpdate($inputTemplate/context[1]),
            utiltemp:prepareItemForUpdate($inputTemplate/(item | items[@is = 'item'])[1]),
            utillib:prepareExampleForUpdate($inputTemplate/example)
            ,
            (: not yet necessary for patch as we don't patch these ... :)
            for $attribute in $inputTemplate/(attribute|items[@is = 'attribute'])
            return utiltemp:processAttribute($attribute, $newTemplateId, $newTemplateElementId, true())
            ,
            for $item in $inputTemplate/(element|include|choice|let|assert|report|defineVariable|constraint|items[@is = ('element','include','choice','let','assert','report','defineVariable','constraint')])
            return utiltemp:processElement($item, $newTemplateId, $newTemplateElementId, true())
        }
        </template>
    )
};

declare %private function utiltemp:prepareTemplateForUpdate($editedTemplate as element(template)) as element(template) {
    
    <template>
    {
        $editedTemplate/@id,
        $editedTemplate/@name,
        $editedTemplate/@displayName,
        $editedTemplate/@effectiveDate,
        $editedTemplate/@statusCode,
        $editedTemplate/@isClosed[not(.='')],
        $editedTemplate/@expirationDate[not(.='')],
        $editedTemplate/@officialReleaseDate[not(.='')],
        $editedTemplate/@versionLabel[not(.='')],
        $editedTemplate/@canonicalUri[not(.='')],
        attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)},
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedTemplate/desc),
        utiltemp:prepareClassificationForUpdate($editedTemplate/classification),
        utiltemp:prepareRelationshipForUpdate($editedTemplate/relationship),
        utillib:preparePublishingAuthorityForUpdate($editedTemplate/publishingAuthority[empty(@inherited)][@name[not(. = '')]][1])
        ,
        for $node in $editedTemplate/purpose
        return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
        ,
        for $node in $editedTemplate/copyright[empty(@inherited)]
        return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
        ,
        utiltemp:prepareContextForUpdate($editedTemplate/context[1]),
        utiltemp:prepareItemForUpdate($editedTemplate/item[1]),
        utillib:prepareExampleForUpdate($editedTemplate/example)
        ,
        (: not yet necessary for patch as we don't patch these ... :)
        for $attribute in $editedTemplate/(attribute|items[@is = 'attribute'])
        return utiltemp:processAttribute($attribute, '', '', false())
        ,
        for $item in $editedTemplate/(element|include|choice|let|assert|report|defineVariable|constraint|items[@is = ('element','include','choice','let','assert','report','defineVariable','constraint')])
        return (:$item :)utiltemp:processElement($item, '', '', false())
    }
    </template>
};

(: Input
      <classification format="hl7v3xml1">
        <type>clinicalstatementlevel</type>
        <type>cdaentrylevel</type>
        <tag>tag1</tag>
        <tag>tag2</tag>
        <property>property1</property>
        <property>property2</property>
      </classification>
   Output
      <classification type="clinicalstatementlevel" format="hl7v3xml1">
        <tag>tag1</tag>
        <tag>tag2</tag>
        <property>property1</property>
        <property>property2</property>
      </classification>
      <classification type="cdaentrylevel">
        <tag>tag1</tag>
        <tag>tag2</tag>
        <property>property1</property>
        <property>property2</property>
      </classification>
:)
declare %private function utiltemp:prepareClassificationForUpdate($nodes as element(classification)*) as element(classification)* {
    
    let $classbytype  :=
        for $type at $i in distinct-values(($nodes/@type, $nodes/type))
        return <classification type="{$type}"/>
    
    return
        if ($classbytype) then (
            <classification>
            {
                $classbytype[1]/@type,
                ($nodes/@format)[1],
                for $tag in $nodes/tag
                group by $tag
                return <tag>{utillib:parseNode($tag[1])/(node() except item)}</tag>
                ,
                for $property in $nodes/property
                group by $property
                return <property>{utillib:parseNode($property[1])/(node() except item)}</property>
            }
            </classification>
            ,
            $classbytype[position() gt 1]
        )
        else if ($nodes[@format | tag | property]) then (
            <classification>
            {
                $nodes[1]/@format,
                for $tag in $nodes/tag
                group by $tag
                return <tag>{utillib:parseNode($tag[1])/(node() except item)}</tag>
                ,
                for $property in $nodes/property
                group by $property
                return <property>{utillib:parseNode($property[1])/(node() except item)}</property>
            }
            </classification>
        )
        else ()
};

declare %private function utiltemp:prepareRelationshipForUpdate($nodes as element(relationship)*) as element(relationship)* {
    
    for $node in $nodes
    return
        element {name($node)} {
            $node/@type,
            ($node/@template, $node/@model)[1],
            $node/@flexibility[. = 'dynamic'] | $node/@flexibility[. castable as xs:dateTime]
        }
};

declare %private function utiltemp:prepareContextForUpdate($nodes as element(context)*) as element(context)* {
    
    for $node in $nodes
    let $att  :=
        switch ($node/@selected)
        case 'id'     return $node/@id[not(. = '')]
        case 'path'   return $node/@path[not(. = '')]
        default       return ($node/@id, $node/@path)[not(. = '')][1] 
    return if ($att) then <context>{$att}</context> else ()
};

declare %private function utiltemp:prepareItemForUpdate($nodes as element()*) as element(item)* {
    
    for $node in $nodes[self::items[@is = 'item'] | self::item][string-length(@label) gt 0][not(@original='false')]
    return
        <item>
        {
            $node/@label,
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($node/desc)
        }
        </item>
};

declare %private function utiltemp:processElement($element as element(), $templateId as xs:string, $elementId as xs:string, $selectedOnly as xs:boolean) as element()? {
    
    if (not($selectedOnly) or $element/@selected) then (
        let $elmName    := if (name($element) = 'items') then $element/@is else name($element)
        return
        switch ($elmName)
        case 'element' return
            <element>
            {
                (: attributes :)
                $element/@name[not(.='')],
                if ($element[@contains[not(.='')]]) then (
                    $element/@contains,
                    $element/@flexibility[. castable as xs:dateTime or . = 'dynamic']
                ) else ()
                ,
                if ($element/@datatype[not(.='')]) then (
                    $element/@datatype,
                    $element/@strength[not(.='')]
                ) else ()
                ,
                $element/@minimumMultiplicity[. = '0'] | $element/@minimumMultiplicity[. castable as xs:positiveInteger],
                $element/@maximumMultiplicity[. = '0'] | $element/@maximumMultiplicity[. castable as xs:positiveInteger] | $element/@maximumMultiplicity[. = '*'],
                $element/@conformance[. = ('R', 'C', 'NP')],
                $element/@isMandatory[. = 'true'],
                $element/@isClosed[.= ('true', 'false')],
                if ($element/@concept and $elementId[not(. = '')]) then
                   attribute id {$elementId}
                else(
                    $element/@id[not(.='')]
                )
                ,
                (: this part is in the schema, but not implemented (yet) :)
                $element/@include[not(.='')],
                $element/@useWhere[not(.='')],
                $element/@displayName[not(.='')],
                $element/@statusCode[not(.='')],
                $element/@versionLabel[not(.='')],
                $element/@effectiveDate[. castable as xs:dateTime],
                $element/@expirationDate[. castable as xs:dateTime],
                $element/@officialReleaseDate[. castable as xs:dateTime]
            }
            {
                (: elements :)
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($element/desc)
                ,
                for $item in $element/(item | items[@is = 'item'])[@label[not(. = '')]][not(@original='false')]
                return
                    <item>
                    {
                        $item/@label,
                        utillib:prepareFreeFormMarkupWithLanguageForUpdate($item/desc)
                    }
                    </item>
                ,
                utillib:prepareExampleForUpdate($element/(example | items[@is = 'example'])),
                utiltemp:prepareVocabulary($element/vocabulary),
                utiltemp:prepareElementProperty($element/(property | items[@is = 'property'])),
                $element/text
            }
            {
                for $attribute in $element/(attribute|items[@is = 'attribute'])
                return utiltemp:processAttribute($attribute, $templateId, $elementId, $selectedOnly)
                ,
                for $item in $element/(element|include|choice|let|assert|report|defineVariable|constraint|items[@is = ('element','include','choice','let','assert','report','defineVariable','constraint')])
                return utiltemp:processElement($item, $templateId, $elementId, $selectedOnly)
            }
            </element>
        
        case 'choice' return
            <choice>
            {
                (: attributes :)
                $element/@minimumMultiplicity[. = '0'] | $element/@minimumMultiplicity[. castable as xs:positiveInteger],
                $element/@maximumMultiplicity[. = '0'] | $element/@maximumMultiplicity[. castable as xs:positiveInteger] | $element/@maximumMultiplicity[. = '*'],
                (: elements :)
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($element/desc)
                ,
                for $item in $element/(item | items[@is = 'item'])[@label[not(. = '')]][not(@original='false')]
                return
                    <item>
                    {
                        $item/@label,
                        utillib:prepareFreeFormMarkupWithLanguageForUpdate($item/desc)
                    }
                    </item>
                ,
                for $item in $element/(element|include|constraint|items[@is = ('element','include','constraint')])
                return utiltemp:processElement($item, $templateId, $elementId, $selectedOnly)
            }
            </choice>
        
        case 'include' return
            <include>
            {
                (: attributes :)
                if ($element[@ref[not(. = '')]]) then (
                    $element/@ref,
                    $element/@flexibility[. castable as xs:dateTime] | $element/@flexibility[. = 'dynamic']
                ) else ()
                ,
                $element/@minimumMultiplicity[. = '0'] | $element/@minimumMultiplicity[. castable as xs:positiveInteger],
                $element/@maximumMultiplicity[. = '0'] | $element/@maximumMultiplicity[. castable as xs:positiveInteger] | $element/@maximumMultiplicity[. = '*'],
                $element/@conformance[. = ('R', 'C', 'NP')],
                $element/@isMandatory[. = 'true'],
                (: this part is in the schema, but not implemented (yet) :)
                $element/@scenario[not(.='')],
                $element/@effectiveDate[. castable as xs:dateTime],
                (: elements :)
                for $node in $element/desc
                return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
                ,
                for $item in $element/(item | items[@is = 'item'])[@label[not(. = '')]][not(@original='false')]
                return
                    <item>
                    {
                        $item/@label,
                        utillib:prepareFreeFormMarkupWithLanguageForUpdate($item/desc)
                    }
                    </item>
                ,
                utillib:prepareExampleForUpdate($element/(example | items[@is = 'example'])),
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($element/constraint)
            }
            </include>
        
        case 'let' return
            <let>
            {
                $element/@name,
                $element/@value,
                $element/node()
            }
            </let>
        
        case 'assert' return
            <assert>
            {
                $element/@role[not(.='')],
                $element/@test[not(.='')],
                $element/@flag[not(.='')],
                $element/@see[not(.='')],
                let $parsed := utillib:parseNode($element)
                return if ($parsed[count(*) = 1][*:p | *:div][normalize-space(string-join(text(), '')) = '']) then $parsed/*/node() else $parsed/node()
            }
            </assert>
        
        case 'report' return
            <report>
            {
                $element/@role[not(.='')],
                $element/@test[not(.='')],
                $element/@flag[not(.='')],
                $element/@see[not(.='')],
                let $parsed := utillib:parseNode($element)
                return if ($parsed[count(*) = 1][*:p | *:div][normalize-space(string-join(text(), '')) = '']) then $parsed/*/node() else $parsed/node()
            }
            </report>
        
        case 'defineVariable' return
            <defineVariable>
            {
                $element/@name[not(.='')],
                $element/@path[not(.='')],
                $element/node()
            }
            </defineVariable>
            
        case 'constraint' return
            <constraint>
            {
                $element/@language[not(.='')],
                $element/@lastTranslated[not(.='')],
                $element/@mimeType[not(.='')],
                utillib:parseNode($element)/node()
            }
            </constraint>
        
        default return ()
    
    )
    else ()
};

declare %private function utiltemp:processAttribute($attribute as element(), $templateId as xs:string, $elementId as xs:string, $selectedOnly as xs:boolean) as element()? {
    
    if (not($selectedOnly) or $attribute/@selected) then (
        <attribute>
        {
            $attribute/(@name|@classCode|@contextConductionInd|@contextControlCode|
                        @determinerCode|@extension|@independentInd|@institutionSpecified|
                        @inversionInd|@mediaType|@moodCode|@negationInd|
                        @nullFlavor|@operator|@qualifier|@representation|
                        @root|@typeCode|@unit|@use)[not(.='')]
            ,
            if ($attribute[@name = 'root'][matches(@value,'[a-zA-Z]')]/parent::element/@name[matches(., '^([^:]+:)?templateId')]) then
                if ($templateId[not(. = '')]) then attribute value {$templateId} else ()
            else $attribute/@value[not(. = '')]
            ,
            if ($attribute/@datatype[not(.='')]) then (
                $attribute/@datatype,
                $attribute/@strength[not(.='')]
            ) else ()
            ,
            if ($attribute/@prohibited = 'true' or $attribute/@conf = 'prohibited' or $attribute/@conformance = 'NP') then attribute prohibited {'true'}
            else if ($attribute/@isOptional = 'true' or $attribute/@conf = 'isOptional' or $attribute/@conformance = 'O') then attribute isOptional {'true'} else ()
            ,
            if ($attribute/@concept and $elementId[not(. = '')]) then attribute id {$elementId} else $attribute/@id[not(.='')]
            ,
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($attribute/desc)
            ,
            for $item in $attribute/(item | items[@is = 'item'])[@label[not(. = '')]][not(@original='false')]
            return
                <item>
                {
                    $item/@label,
                    utillib:prepareFreeFormMarkupWithLanguageForUpdate($item/desc)
                }
                </item>
            ,
            utiltemp:prepareVocabulary($attribute/vocabulary)
        }
        </attribute>
    )
    else ()
};

(:~ Process property element(s). Incoming property element may be named 'property' or items[@is='property']. If it contains any of the supported attributes with a value, 
    a property element is created with those attributes. Empty attributes and any other attributes than the supported attributes are ignored (like @is and @order). Supported 
    attributes with illegal contents raise an error :)
declare %private function utiltemp:prepareElementProperty($properties as element()*) as element(property)* {
    
    for $property in $properties[self::property | self::items[@is = 'property']][(@minInclude | @maxInclude | @minLength | @maxLength | @value | @fractionDigits | @unit | @currency)[not(. = '')]]
    return
        <property>
        {
            for $att in ($property/@minInclude, $property/@maxInclude)[not(. = '')]
            return
                if ($att castable as xs:decimal) then $att else (
                    error($errors:BAD_REQUEST, 'Property ' || name($att) || ' SHALL be a valid integer or decimal (with decimal point, not comma). Found ''' || $att || '''')
                )
            ,
            for $att in ($property/@minLength, $property/@maxLength)[not(. = '')]
            return
                if ($att castable as xs:positiveInteger) then $att else (
                    error($errors:BAD_REQUEST, 'Property ' || name($att) || ' SHALL be a valid positive integer. Found ''' || $att || '''')
                )
            ,
            $property/@value[not(. = '')]
            ,
            for $att in $property/@fractionDigits[not(. = '')]
            return
                if (matches($att, '^\d{1,4}[!\.]?$')) then $att else (
                    error($errors:BAD_REQUEST, 'Property ' || name($att) || ' SHALL match pattern of 1-4 digits followed by ! or . (period). Found ''' || $att || '''')
                )
            ,
            $property/@unit[not(. = '')],
            $property/@currency[not( . = '')]
        }
        </property>
};

declare %private function utiltemp:prepareVocabulary($vocabularies as element(vocabulary)*) as element()* {
    
    for $vocabulary in $vocabularies[(@valueSet | @code | @codeSystem | @codeSystemName | @codeSystemVersion | @displayName | @domain)[not(. = '')]]
    return
        <vocabulary>
        {
            if ($vocabulary/@valueSet[not(. = '')]) then (
                if ($vocabulary/@valueSet[utillib:isOid(.)]) then $vocabulary/@valueSet else (
                    error($errors:BAD_REQUEST, 'Vocabulary attribute valueSet SHALL be a valid OID. Found ''' || $vocabulary/@valueSet || '''')
                )
                ,
                if ($vocabulary/@flexibility[not(. = '')]) then 
                    if ($vocabulary/@flexibility[. castable as xs:dateTime] | $vocabulary/@flexibility[. = 'dynamic']) then $vocabulary/@flexibility else (
                        error($errors:BAD_REQUEST, 'Vocabulary attribute flexibility SHALL be a valid dateTime or ''dynamic''. Found ''' || $vocabulary/@flexibility || '''')
                    )
                else ()
            ) 
            else (
                if ($vocabulary/@code[not(. = '')]) then
                    if ($vocabulary/@code[matches(., '\s')]) then 
                        error($errors:BAD_REQUEST, 'Vocabulary attribute code SHALL NOT contain whitespace. Found ''' || $vocabulary/@code || '''')
                    else (
                        $vocabulary/@code
                    )
                else ()
                ,
                if ($vocabulary/@codeSystem[not(. = '')]) then
                    if ($vocabulary/@codeSystem[utillib:isOid(.)]) then $vocabulary/@codeSystem else (
                        error($errors:BAD_REQUEST, 'Vocabulary attribute codeSystem SHALL be a valid OID. Found ''' || $vocabulary/@codeSystem || '''')
                    )
                else ()
                ,
                $vocabulary/@codeSystemName[not(. = '')],
                $vocabulary/@codeSystemVersion[not(. = '')],
                $vocabulary/@displayName[not(. = '')]
            )
            ,
            $vocabulary/@domain[not(. = '')]
        }
        </vocabulary>
};

(:~ Collect all relevant templateAssociations into one element :)
declare %private function utiltemp:prepareTemplateAssociationForUpdate($storedTemplateAssociations as element(templateAssociation)*, $preparedTemplate as element(template), $inputTemplate as element(template)?, $newTemplateElementId as xs:string, $mode as xs:string) as element(templateAssociation) {
    
    <templateAssociation templateId="{$preparedTemplate/@id}" effectiveDate="{$preparedTemplate/@effectiveDate}">
    {
        (:  keep all concepts associated with element/@id | attribute/@id in the prepared template :)
        for $association in ($storedTemplateAssociations/concept[@elementId = ($preparedTemplate//element/@id | $preparedTemplate//attribute/@id)] |
                             $inputTemplate/staticAssociations/*[@elementId = ($preparedTemplate//element/@id | $preparedTemplate//attribute/@id)] | 
                             $storedTemplateAssociations/concept[@elementPath] | $inputTemplate/staticAssociations/*[@elementPath])
        let $deid           := $association/@ref
        let $deed           := $association/@effectiveDate
        let $elid           := $association/@elementId
        let $elpt           := $association/@elementPath
        group by $deid, $deed, $elid, $elpt
        return <concept>{$deid, $deed, $elid, $elpt}</concept>
        ,
        (:if mode is 'new' and conceptId is present, create new templateAssociation for concept:)
        if ($mode = 'new' and $inputTemplate[@conceptId]//@concept) then
            <concept ref="{$inputTemplate/@conceptId}" effectiveDate="{$inputTemplate/@conceptEffectiveDate}" elementId="{$newTemplateElementId}"/>
        else ()
    }
    </templateAssociation>
};

(:  Adds project unique ids to template elements and attributes where they do not exist yet, 
:   based on the defaultBaseId for type EL in the project
:   
:   @param $decor               - required. The <decor/> element for the project the template is in
:   @param $id                  - optional. Id of the template. Matches template/@id. If empty does all templates.
:   @param $effectiveDate       - optional. Considered only if $id is not empty. null does all versions, yyyy-mm-ddThh:mm:ss gets this specific version
:   @return nothing or error
:   @since 2016-11-30

:)
declare function utiltemp:addTemplateElementAndAttributeIds($decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?) {
    
    let $template               := utillib:getTemplateById($id, $effectiveDate)
        
    let $defaultElementBaseId   := decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATEELEMENT)/@id
    let $elementBaseId          := 
        if ($defaultElementBaseId) then
            if (ends-with($defaultElementBaseId,'.')) then $defaultElementBaseId[1]/string() else concat($defaultElementBaseId[1],'.') 
        else ()
    let $existingIds            := $decor//element[matches(@id,concat('^',$elementBaseId,'\d+$'))] | $decor//attribute[matches(@id,concat('^',$elementBaseId,'\d+$'))]
    let $elementIncr            := if ($existingIds) then max($existingIds/xs:integer(tokenize(@id,'\.')[last()])) else (0)
    (: 2014-12-16 AH Add support for ids on attributes without id and with no more than 1 attribute defined. Excludes things like:
        <attribute classCode="OBS" moodCode="EVN"/>
        Adding an id on this type of attribute would be ambiguous
    :)
    let $update                 :=
        if ($elementBaseId) then
            for $element at $pos in ($template//element[empty(@id)] | 
                                     $template//attribute[empty(@id)][count(@name|@classCode|@contextConductionInd|@contextControlCode|
                                                                          @determinerCode|@extension|@independentInd|@institutionSpecified|
                                                                          @inversionInd|@mediaType|@moodCode|@negationInd|
                                                                          @nullFlavor|@operator|@qualifier|@representation|
                                                                          @root|@typeCode|@unit|@use)=1])
            let $i              := $elementIncr + $pos
            let $u              := update insert attribute id {concat($elementBaseId, $i)} into $element
            return $i
        else ()
    
    return ()
};

declare %private function utiltemp:getTemplateForEdit($template as element(template), $decor as element(decor), $lock as element()?, $targetDate as xs:boolean, $keepIds as xs:boolean, $baseId as xs:string?, $mode as xs:string) as element(template) {

    let $templateAssociations   := $decor//templateAssociation[@templateId = $template/@id][@effectiveDate=$template/@effectiveDate]
    
    let $result                 := 
        <template> 
        {
            utiltemp:createTemplateAttributes($template, $decor, $targetDate, $keepIds, $baseId, $mode)/@*,
            <edit mode="edit"/>,
            $lock,
            if ($template/desc) then $template/desc else <desc language="{$decor/project/@defaultLanguage}"/>,
            $template/classification,
            utiltemp:createTemplateRelationships($template, $decor, $mode, $keepIds)/*, 
            for $node in $template/publishingAuthority return utiltemp:createElementWithAddress($node),
            for $node in $template/endorsingAuthority return utiltemp:createElementWithAddress($node),
            $template/purpose,
            $template/copyright, 
            utiltemp:createTemplateContext($template/context),
            if ($template/item) then utiltemp:createTemplateItem($template/item) else (),
            utiltemp:createTemplateExample($template/example),
            utiltemp:createTemplateElements($template, $decor, $mode),
            (: by adding them here, they stay in the clients XML until he closes the form :)
            if ($templateAssociations) then utiltemp:createStaticAssociationElement($templateAssociations, $decor/project/@defaultLanguage) else ()
        }
        </template>
    
    return 
        utiltemp:prepareTemplateForAPI($result)
};

(:~ Copy or create template attributes based on $mode. When $mode is 'edit', most things are copied. When $mode is 'new' we need
status code to be 'draft', and we have no use for expirationDate, officialReleaseDate and/or canonicalUri

    @param $template as element(template) - Template to copy attributes from. When in edit mode, you simply copy. When in new mode, 
           we copy most but not everything, e.g. statusCode
    @param $decor as element(decor) - Decor ancestor element of the $template, or of the project that the new template is to be created in. 
           When a prototype is in a different project than the to-be-created template then namespace declarations for the prototype template 
           are relevant to keep as they might conflict with the project that the new template is created in
    @param $targetDate as xs:boolean - Relevant for @effectiveDate only when $mode != 'edit. When true() uses yyyy-mm-ddT000:00:00. When false() uses yyyy-mm-ddThh:mm:ss. 
    @param $keepIds as xs:boolean - Boolean for keeping the same $template/@id for the output
    @param $baseId as xs:string? - Base OID for the template (everything but the last node in @id)
    @param $mode as xs:string - When editing an existing template set to 'edit'. When creating a new template (from prototype) set to 'new'
:)
declare %private function utiltemp:createTemplateAttributes($template as element(template), $decor as element(decor), $targetDate as xs:boolean, $keepIds as xs:boolean, $baseId as xs:string?, $mode as xs:string) as element() {

    let $projectPrefix                  := $decor/project/@prefix
    let $templateName                   := ($template/@name, utillib:getNameForOID($template/@id, (), $decor))[1]     

    return
    <attributes>
    {
        attribute projectPrefix {$projectPrefix},
        if ($baseId) then attribute baseId {$baseId} else (),
        attribute originalId {if ($keepIds) then $template/@id else ()},
        attribute id {if ($keepIds) then $template/@id else ()},
        attribute name {$templateName},
        attribute displayName {($template/@displayName, $templateName)[1]},
        attribute effectiveDate {
            if ($mode = 'edit') then $template/@effectiveDate else 
            if ($targetDate) then substring(string(current-date()), 1, 10) || 'T00:00:00' else substring(string(current-dateTime()), 1, 19)
        },
        if ($mode = 'new') then attribute statusCode {'draft'} else $template/@statusCode,
        $template/@versionLabel,
        $template/@isClosed,
        if ($mode = 'new') then () else $template/@expirationDate,
        if ($mode = 'new') then () else $template/@officialReleaseDate,
        (: note ... 
           - $mode new (from prototype) *shall not* copy the canonicalUri
           - $keepIds false *shall not* copy the canonicalUri
           - edit of an existing template *shall* copy the canonicalUri
           - it is debatable whether or not a version of an existing template *should* copy the canonicalUri. currently implemented: yes
        :)
        if ($mode = 'new' or not($keepIds)) then () else $template/@canonicalUri,
        for $ns at $i in utillib:getDecorNamespaces($decor) 
        return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri}
    }           
    </attributes>

};

declare %private function utiltemp:createTemplateRelationships($template as element(template), $decor as element(decor), $mode as xs:string, $keepIds as xs:boolean) as element(relationships) {

    <relationships>
    {
        if ($mode = 'edit') then () else (
            if ($template[@id]) then (
                (: note ... 
                    if $mode is 'new' then if $keepIds true the relationship shall be VERSION
                    (as the new thing is a natural version of the source) otherwise it shall be SPEC
                :)
                <relationship type="{
                    if ($mode = 'new') then (
                        if ($keepIds) then 'VERSION' else 'SPEC'
                    ) else upper-case($mode)}" template="{$template/@id}" selected="template" flexibility="{$template/@effectiveDate}">
                {
                    for $attr in $template/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                    return
                        attribute {'tm' || name($attr)} {$attr}
                    ,
                    attribute url {$utillib:strDecorServicesURL},
                    attribute ident {$decor/project/@prefix},
                    attribute linkedartefactmissing {'false'}
                }
                </relationship>
            ) else ()
        )
        ,
        for $relationship in $template/relationship
            let $artefact               := 
                if (empty($relationship/@template)) then () else (
                    (utillib:getTemplateById($relationship/@template, $relationship/@flexibility, $decor, (), ())[@id][@effectiveDate])[1]
                )
            let $tmpurl                 := ($artefact/@url, $artefact/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
            let $tmpident               := ($artefact/@ident, $artefact/ancestor::decor/project/@prefix)[1]
            return
            <relationship>
            {
                attribute type {$relationship/@type},
                if ($relationship/@template) then (
                    attribute template {$relationship/@template},
                    attribute selected {'template'},
                    attribute model {''},
                    attribute flexibility {$relationship/@flexibility},
                    if (empty($artefact)) then () else (
                        for $attr in $artefact/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                        return
                            attribute {'tm' || name($attr)} {$attr}
                        ,
                        attribute url {$tmpurl},
                        attribute ident {$tmpident}
                    ),
                    attribute linkedartefactmissing {empty($artefact)}
                )
                else (
                    attribute template {''},
                    attribute model {$relationship/@model},
                    attribute flexibility {$relationship/@flexibility},
                    attribute selected {'model'}
                )
            }
            </relationship>
        }
    </relationships>
    
};

declare %private function utiltemp:createElementWithAddress($node as element()) as element() {

    element {name($node)}
    {
        $node/(@* except @selected),
        attribute selected {''},
        $node/addrLine
    }
};

declare %private function utiltemp:createTemplateContext($context as element(context)?) as element(context) {

    <context>
    {
        if ($context/@id) then (
            $context/@id,
            attribute path {''},
            attribute selected {'id'}
        )
        else (
            attribute id {''},
            attribute path {$context/@path},
            attribute selected {'path'}
        )
    }
    </context>
};

declare %private function utiltemp:createTemplateItem($item as element(item)) as element(item) {

    <item label="{$item/@label}">
    {
        $item/desc
    }
    </item>
};

declare %private function utiltemp:createTemplateExample($examples as element(example)*) as element(example)* {

     if ($examples) then
        (: if multiple root nodes exist, it is never going to be valid xml in an editor, so we wrap it under one :)
        for $example in $examples
        return
            <example type="{$example/@type}" caption="{$example/@caption}">
            {
                if (count($example/*) gt 1) then <art:placeholder xmlns:art="urn:art-decor:example">{$example/node()}</art:placeholder> else $example/node()
            }
            </example>
    else 
        <example type="neutral" caption=""/>
};

declare %private function utiltemp:createTemplateElements($template as element(template), $decor as element(decor), $mode as xs:string) as element()* {
    
    let $projectPrefix                  := $decor/project/@prefix
    let $valueSetList                   := $decor/terminology/valueSet
    let $templateList                   := $decor/rules/template

    let $template                       := utiltemp:createTemplateExtract($template)

    (: TODO: why do we only pick SPEC? Could it not be "anything"? All it does it supply you with potential elements and attributes to activate :)
    let $specialization                 := $template/relationship[@type='SPEC'][@template][1]
    (:get prototype for editor with normalized attributes:)
    let $prototype                      := if ($specialization) then utillib:getTemplateById($specialization/@template, $specialization/@flexibility, $projectPrefix)[1] else ()

    return
    if ($prototype) then (
        (: if template is empty, check prototype for relevant parts :)
        if ($template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)) then (
            for $item in $template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
            return utiltemp:mergePrototypeTemplateForEdit($projectPrefix, $item, $prototype, $valueSetList, $templateList, $mode)
        ) else (
            for $item in $prototype/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
            return utiltemp:recurseItemForEdit($decor, $item, $valueSetList, $templateList, false(), $mode)
        )
    )
    else (
        for $item in $template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
        return utiltemp:recurseItemForEdit($decor, $item, $valueSetList, $templateList, true(), $mode)
    )
};

declare %private function utiltemp:createTemplateVocabulary($vocabulary as element(vocabulary), $projectPrefix as item(), $valueSetList as element()*) as element(vocabulary) {

    <vocabulary>
    {
        $vocabulary/(@valueSet|@flexibility|@code|@codeSystem|@codeSystemName|@codeSystemVersion|@displayName|@domain)[not(.='')],
        utiltemp:isValueSetInScope($projectPrefix, $vocabulary/@valueSet, $vocabulary/@flexibility, $valueSetList),
        $vocabulary/*
    }
    </vocabulary>
    
};

declare %private function utiltemp:mergePrototypeTemplateForEdit($projectPrefix as xs:string, $item as element(), $prototype as element()?, $valueSetList as element()*, $templateList as element()*, $mode as xs:string) as element()* {
    
    (: get corresponding node in prototype :)
    let $node                           := util:eval-inline($prototype, utiltemp:path-to-node($item))
    let $node                           := if ($node[@id = $item/@id]) then $node[@id = $item/@id][1] else $node[1]
    
    (: get the reversed sequence for previous siblings of the corresponding node in the prototype template, if any :)
    let $precedingNodes                 := reverse($node/preceding-sibling::*)
    
    let $results                        := 
    (
        (: if there are no preceding nodes in template, get all preceding nodes from prototype
            else get preceding nodes up to node with same name as preceding template node
        :)
        if (count($item/preceding-sibling::*) = 0 and count($node/preceding-sibling::*) gt 0) then (
            for $precedingNode in $node/preceding-sibling::*
            return utiltemp:recurseItemForEdit($node/ancestor::decor, $precedingNode, $valueSetList, $templateList, false(), $mode)
        )
        else (
            (: check if there are preceding nodes in prototype that are not in the template
                in order not retrieve every single node preceding the current one, we first 
                try to find a previous matching node, anything between the previous and the 
                current item should be merged into the current template in this context 
            :)
            let $indexNode  := 
                for $precedingItem in $item/preceding-sibling::*
                let $n      := utiltemp:index-node-in-set($precedingNodes, $precedingItem)
                return
                    if (count($n) le 1) then ($n) else (
                        error(QName('http://art-decor.org/ns/error', 'MultipleIndexNodes'), 
                            'Unexpected error. Found multiple (' || count($n) || ') index nodes so we do not know what to merge here.' || 
                            ' Current item: ' || utiltemp:path-to-node($item) || ' ' || string-join(for $i in $item return concat('''&lt;', $i/name(), string-join(for $att in $i/@* return concat(' ', $att/name(),'=''', data($att),''''), ''), '&gt;'''),' ') ||
                            ' Current preceding item: ' || utiltemp:path-to-node($precedingItem) || ' ' || string-join(for $i in $precedingItem return concat('''&lt;', $i/name(), string-join(for $att in $i/@* return concat(' ', $att/name(),'=''', data($att),''''), ''), '&gt;'''),' ') || 
                            ' index nodes: ' ||  string-join(for $i in $n return concat('''&lt;', $i/name(), string-join(for $att in $i/@* return concat(' ', $att/name(),'=''', data($att),''''), ''), '&gt;'''),' '))
                    )
            
            let $index      := if ($indexNode) then utiltemp:index-of-node($precedingNodes,$indexNode[last()]) else (count($precedingNodes))
            
            for $n in reverse($precedingNodes)
            let $lin := utiltemp:index-of-node($precedingNodes,$n) lt $index and 
                        $n[ self::attribute|
                            self::element|
                            self::choice|
                            self::include|
                            self::let|
                            self::defineVariable|
                            self::assert|
                            self::report|
                            self::constraint ]
            return if ($lin) then utiltemp:recurseItemForEdit($node/ancestor::decor, $n, $valueSetList, $templateList, false(), $mode) else ()
        )
        ,
        switch (name($item))
        case 'attribute' return utiltemp:createAttributeForEditMerge($item, $projectPrefix, $valueSetList, $node)
        case 'element' return utiltemp:createElementForEditMerge($projectPrefix, $item, $prototype, $valueSetList, $templateList, $mode, $node)
        case 'choice' return utiltemp:createChoiceForEditMerge($projectPrefix, $item, $prototype, $valueSetList, $templateList, $mode, $node)
        case 'include' return utiltemp:createIncludeForEditMerge($projectPrefix, $item, $prototype, $valueSetList, $templateList, $mode, $node)
        case 'let' 
        case 'defineVariable' return utiltemp:createLetOrDefineVariableForEdit($item, $node, true())
        case 'assert'
        case 'report' return utiltemp:createAssertOrReportForEdit($item, $node, true())
        case 'constraint' return utiltemp:createAssertOrReportForEdit($item, $node, true())
        default return ()
        ,
        (:
           if parent is not 'template' check if there are prototype nodes after the last template node
        :)
        if ($item/following-sibling::*) then () else 
        if ($node) then (
            (: our own current template had a match in the prototype :)
            for $n in ( $node/following-sibling::attribute|
                        $node/following-sibling::element|
                        $node/following-sibling::choice|
                        $node/following-sibling::include|
                        $node/following-sibling::let|
                        $node/following-sibling::defineVariable|
                        $node/following-sibling::assert|
                        $node/following-sibling::report|
                        $node/following-sibling::constraint)
            return utiltemp:recurseItemForEdit($node/ancestor::decor, $n, $valueSetList, $templateList, false(), $mode)
        )
        else (
            (: since our current node did not match, find any preceding node from our template that matched in the prototype :)
            let $precedingNodes :=
                for $n in ( $item/preceding-sibling::attribute|
                            $item/preceding-sibling::element|
                            $item/preceding-sibling::choice|
                            $item/preceding-sibling::include|
                            $item/preceding-sibling::let|
                            $item/preceding-sibling::defineVariable|
                            $item/preceding-sibling::assert|
                            $item/preceding-sibling::report|
                            $item/preceding-sibling::constraint)

                (: get corresponding node in prototype :)
                let $nodeInPrototype      := 
                    if ($n/@id) then $prototype/ancestor-or-self::template[1]//*[name() = name($n)][@id = $n/@id] 
                    else util:eval('$prototype/ancestor-or-self::template[1]/' || utiltemp:path-to-node($n))
                return $nodeInPrototype[1]
            
            (: add all nodes from the prototype template following the last match that we had, to the end of the current template :)
            for $n in ( $precedingNodes[last()]/following-sibling::attribute|
                        $precedingNodes[last()]/following-sibling::element|
                        $precedingNodes[last()]/following-sibling::choice|
                        $precedingNodes[last()]/following-sibling::include|
                        $precedingNodes[last()]/following-sibling::let|
                        $precedingNodes[last()]/following-sibling::defineVariable|
                        $precedingNodes[last()]/following-sibling::assert|
                        $precedingNodes[last()]/following-sibling::report|
                        $precedingNodes[last()]/following-sibling::constraint)
             return utiltemp:recurseItemForEdit($n/ancestor::decor, $n, $valueSetList, $templateList, false(), $mode)
        )
    )
    
    for $result in $results
    return
        if ($result/attribute) then (
            element {name($result)} {
                $result/@*,
                $result/attribute[last()] | $result/attribute[last()]/preceding-sibling::* except $result/element,
                $result/attribute[last()]/preceding-sibling::element | $result/attribute[last()]/following-sibling::node()
            }
        )
        else $result
};

declare %private function utiltemp:createAttributeForEditMerge($item as element(), $projectPrefix as item(), $valueSetList as element()*, $node as element()?) as element(attribute) {

    for $att in utiltemp:normalizeAttributes($item)
    return
    <attribute>
    {
        attribute selected {if ($node) then 'original' else ''},
        $att/@name,
        $att/@value,
        attribute conformance {
            if ($att[@prohibited='true']) then 'NP' else
            if ($att[@isOptional='true']) then 'O' else 'R'
        }
        ,
        $att/@datatype,
        $att/@id,
        if ($node/@datatype) then (
            attribute originalType {utiltemp:getDatatype($node/@datatype, $node/ancestor::template/classification/@format)},
            attribute originalStrength {utiltemp:rewriteStrength($node/@strength)}
        ) else 
        if ($att/@datatype) then (
            attribute originalType {utiltemp:getDatatype($att/@datatype, $att/ancestor::template/classification/@format)},
            attribute originalStrength {utiltemp:rewriteStrength($att/@strength)}
        )
        else ()
        ,
        attribute originalConformance {
            if ($node/@prohibited='true') then 'NP' else 
            if ($node/@isOptional='true') then 'O' else 
            if ($node) then 'R' else 'O'
        }
        ,
        $att/desc,      
        for $subitem in $att/(* except desc)
        return if ($subitem[name()='vocabulary']) then utiltemp:createTemplateVocabulary($subitem, $projectPrefix, $valueSetList) else $subitem

    }
    </attribute>

};

declare %private function utiltemp:createAttributeForEdit($item as element(), $projectPrefix as item(), $valueSetList as element()*, $mode as xs:string, $selected as xs:boolean) as element(attribute) {

    for $att in utiltemp:normalizeAttributes($item)
    return
    <attribute>
    {
        if ($selected) then (attribute selected {''}) else (),
        $att/@name,
        $att/@value,
        attribute conformance {
            if ($att[@prohibited='true']) then 'NP' else
            if ($att[@isOptional='true']) then 'O' else 'R'
        }
        ,
        $att/@datatype,
        $att/@id,
        if ($att/@strength) then attribute strength {utiltemp:rewriteStrength($att/@strength)} else (),
        if ($att/@datatype) then (
            attribute originalType {utiltemp:getDatatype($att/@datatype,$item/ancestor::template/classification/@format)},
            attribute originalStrength {utiltemp:rewriteStrength($att/@strength)}
        ) else ()
        ,
        attribute originalConformance {
            if ($selected and $mode = 'new') then 
                if ($att[@prohibited='true']) then 'NP' else
                if ($att[@isOptional='true']) then 'O' else 'R'
            else 'O'
        }
        ,
        $att/desc,
        for $subitem in $att/(* except desc)
        return if ($subitem[name()='vocabulary']) then utiltemp:createTemplateVocabulary($subitem, $projectPrefix, $valueSetList) else $subitem
        }
    </attribute>
};

declare %private function utiltemp:createElementForEditMerge($projectPrefix as xs:string, $item as element(), $prototype as element()?, $valueSetList as element()*, $templateList as element()*, $mode as xs:string, $node as element()?) as element(element) {

    <element>
    {
        attribute selected {if ($node) then 'original' else ''},
        $item/(@* except (@strength|@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
        utiltemp:isTemplateInScope($projectPrefix, $item/@contains, $item/@flexibility, $templateList),
        if ($item/@strength) then attribute strength {utiltemp:rewriteStrength($item/@strength)} else (),
        if ($node/@datatype) then (
            attribute originalType {utiltemp:getDatatype($node/@datatype, $node/ancestor::template/classification/@format)},
            attribute originalStrength {utiltemp:rewriteStrength($node/@strength)}
        ) else 
        if ($item/@datatype) then (
            attribute originalType {utiltemp:getDatatype($item/@datatype,$item/ancestor::template/classification/@format)},
            attribute originalStrength {utiltemp:rewriteStrength($item/@strength)}
        )
        else ()
        ,
        if ($item/@minimumMultiplicity) then () else attribute minimumMultiplicity {''},
        if ($item/@maximumMultiplicity) then () else attribute maximumMultiplicity {''},
        if ($node/@minimumMultiplicity) then attribute originalMin {$node/@minimumMultiplicity} else attribute originalMin {'0'},
        if ($node/@maximumMultiplicity) then attribute originalMax {$node/@maximumMultiplicity} else attribute originalMax {'*'},
        if ($item/@conformance) then () else attribute conformance {'O'},
        if ($item/@isMandatory) then () else attribute isMandatory {'false'},
        if ($item/@strength) then () else attribute strength {''},
        $item/desc,      
        for $vocabulary in $item/vocabulary
        return utiltemp:createTemplateVocabulary($vocabulary, $projectPrefix, $valueSetList)
        ,
        $item/property,
        $item/item[1],
        $item/text,
        utiltemp:createTemplateExample($item/example),
        for $subItem in $item/(attribute|element|choice|include|constraint|let|defineVariable|assert|report)
        return
            if ($node) then utiltemp:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $prototype, $valueSetList, $templateList, $mode)
            else utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, true(), $mode)
    }
    </element>
};

declare %private function utiltemp:createElementForEdit($projectPrefix as xs:string, $item as element(), $valueSetList as element()*, $templateList as element()*, $selected as xs:boolean, $mode as xs:string) as element(element) {

    <element>
    {
        if ($selected) then (attribute selected {''}) else (),
        $item/(@* except (@strength|@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
        utiltemp:isTemplateInScope($projectPrefix, $item/@contains, $item/@flexibility, $templateList),
        if ($item/@strength) then attribute strength {utiltemp:rewriteStrength($item/@strength)} else (),
        if ($item/@datatype) then (
            attribute originalType {utiltemp:getDatatype($item/@datatype,$item/ancestor::template/classification/@format)},
            attribute originalStrength {utiltemp:rewriteStrength($item/@strength)}
        )
        else ()
        ,
        if ($item/@minimumMultiplicity) then () else attribute minimumMultiplicity {''},
        if ($item/@maximumMultiplicity) then () else attribute maximumMultiplicity {''},
        attribute originalMin {'0'},
        attribute originalMax {'*'},
        if ($item/@conformance) then () else (
            let $conformance        := 
                if ($item[@isMandatory='true']) then 'R' else
                if ($item[@minimumMultiplicity castable as xs:integer]/@minimumMultiplicity > 0) then 'R' else 'O'
            return attribute conformance {$conformance}
        )
        ,
        if ($item/@isMandatory) then () else attribute isMandatory {'false'},
        if ($item/@strength) then () else attribute strength {''},
        $item/desc,
        for $vocabulary in $item/vocabulary
        return utiltemp:createTemplateVocabulary($vocabulary, $projectPrefix, $valueSetList)
        ,
        $item/property,
        $item/item[1],
        $item/text,
        utiltemp:createTemplateExample($item/example),
        for $subItem in $item/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
        return utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, $selected, $mode)
    }
    </element>
};

declare %private function utiltemp:createChoiceForEditMerge($projectPrefix as xs:string, $item as element(), $prototype as element()?, $valueSetList as element()*, $templateList as element()*, $mode as xs:string, $node as element()?) as element(choice) {

    <choice>
    {
        attribute selected {if ($node) then 'original' else ''},
        $item/@*,
        if (not($item/@minimumMultiplicity)) then attribute minimumMultiplicity {''} else (),
        if (not($item/@maximumMultiplicity)) then attribute maximumMultiplicity {''} else (),
        if ($node/@minimumMultiplicity) then attribute originalMin {$node/@minimumMultiplicity} else attribute originalMin {'0'},
        if ($node/@maximumMultiplicity) then attribute originalMax {$node/@maximumMultiplicity} else attribute originalMax {'*'},
        $item/desc,
        $item/item[1],
        for $subItem in $item/(element|choice|include|constraint)
        return
            if ($node) then utiltemp:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $prototype, $valueSetList, $templateList, $mode)
            else utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, true(), $mode)
    }
    </choice>

};

declare %private function utiltemp:createChoiceForEdit($projectPrefix as xs:string, $item as element(), $valueSetList as element()*, $templateList as element()*, $selected as xs:boolean, $mode as xs:string) as element(choice) {

    <choice>
    {
        if ($selected) then (attribute selected {''}) else (),
        $item/@*,
        if ($item/@minimumMultiplicity) then () else attribute minimumMultiplicity {''},
        if ($item/@maximumMultiplicity) then () else attribute maximumMultiplicity {''},
        attribute originalMin {'0'},
        attribute originalMax {'*'},
        $item/desc,
        $item/item[1],
        for $subItem in $item/(element|choice|include|constraint)
        return utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, $selected, $mode)
    }
    </choice>
};

declare %private function utiltemp:createIncludeForEditMerge($projectPrefix as xs:string, $item as element(), $prototype as element()?, $valueSetList as element()*, $templateList as element()*, $mode as xs:string, $node as element()?) as element(include) {

    <include>
    {
        attribute selected {if ($node) then 'original' else ''},
        $item/(@* except (@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
        utiltemp:isTemplateInScope($projectPrefix, $item/@ref, $item/@flexibility, $templateList),
        if (not($item/@minimumMultiplicity)) then attribute minimumMultiplicity {''} else (),
        if (not($item/@maximumMultiplicity)) then attribute maximumMultiplicity {''} else (),
        if ($node/@minimumMultiplicity) then attribute originalMin {$node/@minimumMultiplicity} else attribute originalMin {'0'},
        if ($node/@maximumMultiplicity) then attribute originalMax {$node/@maximumMultiplicity} else attribute originalMax {'*'},
        if (not($item/@conformance)) then attribute conformance {'O'} else (),
        if (not($item/@isMandatory)) then attribute isMandatory {'false'} else (),
        $item/desc,
        $item/item[1],
        utiltemp:createTemplateExample($item/example),
        for $subItem in $item/constraint
        return
            if ($node) then utiltemp:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $prototype, $valueSetList, $templateList, $mode)
            else utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, true(), $mode)
    }
    </include>

};

declare %private function utiltemp:createIncludeForEdit($projectPrefix as xs:string, $item as element(), $valueSetList as element()*, $templateList as element()*, $selected as xs:boolean, $mode as xs:string) as element(include) {

    <include>
    {
        if ($selected) then (attribute selected {''}) else (),
        $item/(@* except (@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
        utiltemp:isTemplateInScope($projectPrefix, $item/@ref, $item/@flexibility, $templateList),
        if ($item/@minimumMultiplicity) then () else attribute minimumMultiplicity {''},
        if ($item/@maximumMultiplicity) then () else attribute maximumMultiplicity {''},
        attribute originalMin {'0'},
        attribute originalMax {'*'},
        if ($item/@conformance) then () else (
            let $conformance        := if ($item/@minimumMultiplicity='1') then 'R' else ('O')
            return attribute conformance {$conformance}
        )
        ,
        if ($item/@isMandatory) then () else attribute isMandatory {'false'},
        if ($item/@flexibility) then () else attribute flexibility {'dynamic'},
        $item/desc,
        $item/item[1],
        utiltemp:createTemplateExample($item/example),
        for $subItem in $item/constraint
        return utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, $selected, $mode)
    }
    </include>

};

declare %private function utiltemp:createLetOrDefineVariableForEdit($item as element(), $node as element()?, $selected as xs:boolean) as element() {

    element {name($item)}
    {
        if ($node) then attribute selected {'original'} else if ($selected) then attribute selected {''} else (),
        attribute name {$item/@name},
        attribute value {$item/@value},
        $item/node()
    }
};

declare %private function utiltemp:createAssertOrReportForEdit($item as element(), $node as element()?, $selected as xs:boolean) as element() {

    element {name($item)}
    {
        if ($node) then attribute selected {'original'} else if ($selected) then attribute selected {''} else (),
        attribute role {$item/@role},
        attribute test {$item/@test},
        attribute see {$item/@see},
        attribute flag {$item/@flag},
        $item/node()
    }
};

declare %private function utiltemp:createConstraintForEdit($item as element(), $node as element()?, $selected as xs:boolean) as element(constraint) {

    <constraint>
    {
        if ($node) then attribute selected {'original'} else if ($selected) then attribute selected {''} else (),
        $item/(@* except @selected),
        $item/node()
    }
    </constraint>
};

declare %private function utiltemp:recurseItemForEdit($projectPrefix as item(), $item as element(), $valueSetList as element()*, $templateList as element()*, $selected as xs:boolean, $mode as xs:string) as element()* {
    
    switch (name($item))
    case 'attribute' return utiltemp:createAttributeForEdit($item, $projectPrefix, $valueSetList, $mode, $selected)
    case 'element' return utiltemp:createElementForEdit($projectPrefix, $item, $valueSetList, $templateList, $selected, $mode)
    case 'choice' return utiltemp:createChoiceForEdit($projectPrefix, $item, $valueSetList, $templateList, $selected, $mode)
    case 'include' return utiltemp:createIncludeForEdit($projectPrefix, $item, $valueSetList, $templateList, $selected, $mode)
    case 'let' return utiltemp:createLetOrDefineVariableForEdit($item, (), $selected)
    case 'assert'
    case 'report' return utiltemp:createAssertOrReportForEdit($item, (), $selected)
    case 'constraint' return utiltemp:createConstraintForEdit($item, (), $selected)
    default return ()
};

declare %private function utiltemp:createStaticAssociationElement($templateAssociations as element(templateAssociation)*, $language as xs:string?) as element(staticAssociations) {
    <staticAssociations>
    {
        for $association in $templateAssociations/concept
        let $concept            := utillib:getConcept($association/@ref, $association/@effectiveDate)
        let $dataset            := $concept/ancestor::dataset
        let $originalConcept    := utillib:getOriginalForConcept($concept)
        (: 2018-12-19 Disabled for performance reasons :)
        let $refdisplay         := if (1=1) then attribute refdisplay {utillib:getNameForOID($association/@ref, $language, $concept/ancestor::decor)} else ()
        let $path               := 
            for $c in $concept/ancestor::concept
            let $originalConceptName    := utillib:getOriginalForConcept($c)
            return 
                if ($originalConceptName/name[@language=$language]) then $originalConceptName/name[@language=$language] else ($originalConceptName/name[1])
        return
            <origconcept>
            {
                $association/@ref,
                $association/@effectiveDate,
                $refdisplay,
                attribute conceptStatusCode {$concept/@statusCode},
                if ($concept/@expirationDate) then attribute conceptExpirationDate {$concept/@expirationDate} else (),
                if ($concept/@versionLabel) then attribute conceptVersionLabel {$concept/@versionLabel} else (),
                attribute datasetId {$dataset/@id},
                attribute datasetEffectiveDate {$dataset/@effectiveDate},
                attribute datasetStatusCode {$dataset/@statusCode},
                if ($dataset/@expirationDate) then attribute datasetExpirationDate {$dataset/@expirationDate} else (),
                if ($dataset/@versionLabel) then attribute datasetVersionLabel {$dataset/@versionLabel} else (),
                attribute datasetName {$dataset/name[1]},
                attribute ident {$concept/ancestor::decor/project/@prefix},
                $association/@elementId,
                attribute templateId {$association/../@templateId},
                attribute templateEffectiveDate {$association/../@effectiveDate},
                $association/@elementPath,
                attribute path {string-join($path,' / ')}
                ,
                $originalConcept/name,
                $originalConcept/desc,
                for $name in $dataset/name
                return
                    <datasetName>{$name/@*, $name/node()}</datasetName>
                ,
                for $name in $concept/ancestor::decor/project/name
                return
                    <projectName>{$name/@*, $name/node()}</projectName>
            }
            </origconcept>
    }
    </staticAssociations>
};

declare %private function utiltemp:prepareTemplateForAPI($template as element(template)) as element(template) {
    let $ns-attributes  := 
        if ($template/@*[contains(name(), 'dummy')]) then () else ( 
            $template/../@*[contains(name(), 'dummy')]
        )
    return 
        <template>
        {
            $template/@*,
            if ($template/@ident) then () else (
                $template/../@ident,
                if ($template/@url) then () else $template/../@url
            ),
            $ns-attributes,
            for $ns-prefix at $i in in-scope-prefixes($template)
            let $ns-uri := namespace-uri-for-prefix($ns-prefix, $template)
            return
                (:hl7:dummy-default="urn:hl7-org:v3":)
                <ns uri="{$ns-uri}" prefix="{$ns-prefix}" default="{exists($template/@*[name() = $ns-prefix || ':dummy-default'])}"/>
            ,
            (: we return just a count of issues and leave it up to the calling party to get those if required :)
            <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $template/@id][@effectiveDate = $template/@effectiveDate] | $setlib:colDecorData//issue/object[@id = $template/@ref])}"/>
            ,
            (: we include all static association here that belong to our template. Note that there could be staticAssociation element under include. Those would come from a different template than our with potentially overlapping elementIds
               We do not want to include those because that would lead to false associations where a concept connected to a different, included template is assoicated to someting in the calling template.
            :)
            <staticAssociations>
            {
                for $node in $template//staticAssociations/*
                return
                    <concept>{$node/@*, $node/node()}</concept>
            }
            </staticAssociations>
            ,
            $template/desc
            ,
            utiltemp:collapseTemplateClassifications($template/classification)
            ,
            for $e in $template/(* except (ns|issueAssociation|staticAssociations|desc|classification|
                                                       element|attribute|include|choice|defineVariable|let|assert|report|property|item))
            return
                if ($e/self::example) then
                    <example>
                    {
                        if ($e/@type) then $e/@type else attribute type {'neutral'},
                        $e/@caption,
                        $e/node()
                    }
                    </example>
                else (
                    $e
                )
            ,
            utiltemp:prepareTemplateBodyForAPI($template/(element|attribute|include|choice|defineVariable|let|assert|report|property|item))
        }
        </template>
};

declare %private function utiltemp:prepareTemplateBodyForAPI($node as node()*) {
    for $e at $pos in $node
    let $elname := name($e)
    return
        <items>
        {
            attribute { 'is' } { $elname },
            attribute { 'order' } { $pos },
            if ($e/self::example) then (
                if ($e/@type) then $e/@type else attribute type {'neutral'},
                $e/@caption,
                $e/node()
            )
            else
            if ($e/self::assert | $e/self::report) then (
              $e/(@* except (@is | @order)),
              if ($e/@role) then () else attribute role {'error'},
              $e/node()
            )
            else (
                $e/(@* except (@is | @order))
                ,
                for $elm in $e/(node() except (ns|issueAssociation|classification|element|attribute|include|choice|defineVariable|let|assert|report|property|item|example))
                return
                    if ($elm/self::staticAssociations) then
                        <staticAssociations>
                        {
                            for $assoc in $elm/*
                            return
                                <concept>{$assoc/@*, $assoc/node()}</concept>
                        }
                        </staticAssociations>
                    else (
                        $elm
                    ) 
                ,
                utiltemp:prepareTemplateBodyForAPI($e/(element|attribute|include|choice|defineVariable|let|assert|report|property|item|example))
            )
        }
        </items>   
};

declare %private function utiltemp:collapseTemplateClassifications($classification as element(classification)*) {
    <classification>
    {
        ($classification/@format)[1]
        ,
        if ($classification/@type) then
            for $type in $classification/@type
            return
                <type>{data($type)}</type>
        else (
            <type>notype</type>
        ),
        for $tag in $classification/tag
        group by $tag
        return
            $tag
        ,
        for $property in $classification/property
        group by $property
        return
            $property
    }
    </classification>
};

declare %private function utiltemp:normalizeAttributes($attributes as element(attribute)*) as element(attribute)* {
    let $mapping    :=
        <attributes>
            <attribute id="1" name="classCode" datatype="cs"/>
            <attribute id="2" name="contextConductionInd" datatype="bl"/>
            <attribute id="3" name="contextControlCode" datatype="cs"/>
            <attribute id="4" name="determinerCode" datatype="cs"/>
            <attribute id="5" name="extension" datatype="st"/>
            <attribute id="6" name="independentInd" datatype="bl"/>
            <attribute id="7" name="institutionSpecified" datatype="bl"/>
            <attribute id="8" name="inversionInd" datatype="bl"/>
            <attribute id="9" name="mediaType" datatype="st"/>
            <attribute id="10" name="moodCode" datatype="cs"/>
            <attribute id="11" name="negationInd" datatype="bl"/>
            <attribute id="12" name="nullFlavor" datatype="cs"/>
            <attribute id="13" name="operator" datatype="cs"/>
            <attribute id="14" name="qualifier" datatype="set_cs"/>
            <attribute id="15" name="representation" datatype="cs"/>
            <attribute id="16" name="root" datatype="uid"/>
            <attribute id="17" name="typeCode" datatype="cs"/>
            <attribute id="18" name="unit" datatype="cs"/>
            <attribute id="19" name="use" datatype="set_cs"/>
        </attributes>
    
    for $attribute in $attributes
    for $att  at $i in $attribute/(@name|@classCode|@contextConductionInd|@contextControlCode|@determinerCode|
                                         @extension|@independentInd|@institutionSpecified|@inversionInd|
                                         @mediaType|@moodCode|@negationInd|@nullFlavor|
                                         @operator|@qualifier|@representation|@root|
                                         @typeCode|@unit|@use)
    let $anme := if ($att[name()='name']) then $att/string() else ($att/name())
    let $aval := if ($att[name()='name']) then $attribute/@value/string() else ($att/string())
    let $adt  := 
        if ($attribute/@datatype[not(.='')]) 
        then ($attribute/@datatype) 
        else if ($mapping/attribute/@name=$anme) 
        then (attribute datatype {$mapping/attribute[@name=$anme]/@datatype}) 
        else ()
    return
        <attribute name="{$anme}">
        {
            if ($adt=('bl','cs','set_cs') and contains($aval,'|')) then (
                (:boolean can only be true|false so no need to re-specify, cs/set_cs should be done in vocabulary:)
            ) else if (string-length($aval)>0) then (
                attribute value {$aval}
            ) else ()
            ,
            if ($attribute/@prohibited='true') then
                attribute prohibited {'true'}
            else (
                attribute isOptional {$attribute/@isOptional='true'}
            )
            ,
            $adt
            ,
            (:could lead to duplicates...:)
            if ($i=1) then ($attribute/@id) else ()
            ,
            $attribute/@selected
            ,
            $attribute/(desc|item|comment()),
            if ($adt=('cs','set_cs')) then (
                if (contains($aval,'|')) then (
                    for $v in tokenize($aval,'\|')
                    return <vocabulary code="{normalize-space($v)}"/>
                ) else ()
                ,
                $attribute/vocabulary
            ) else (
                $attribute/constraint
            )
        }
        </attribute>
};

(: Returns paths like this
    element[@name='hl7:informEvent'][empty(@contains)][@id='2.16.840.1.113883.2.4.3.111.3.7.9.16'][1]
:)
declare %private function utiltemp:path-to-node($nodes as node()*)  as xs:string* { 
    string-join(
        for $item in $nodes/ancestor-or-self::*
        return (
            if ($item[descendant-or-self::template]) then () else
            string-join(
                ($item/name(),
                if ($item[self::attribute][@name]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[self::element][@name][@contains]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']',
                    '[@contains=''',replace($item/@contains,'''',''''''),''']'
                )
                else
                if ($item[self::element][@name = ('hl7:templateId', 'cda:templateId')][attribute[@name = 'root'][@value]]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']','[attribute[@name = ''root''][@value = ''',$item/attribute[@name = 'root']/@value,''']]',
                    '[',count($item/preceding-sibling::element[@name = $item/@name][attribute[@name = 'root'][@value]]) + 1,']'
                )
                else
                if ($item[self::element][@name][empty(@contains)]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']','[empty(@contains)]',
                    '[',count($item/preceding-sibling::element[@name = $item/@name][empty(@contains)]) + 1,']'
                )
                else
                if ($item[self::choice]) then (
                    if ($item[element]) then (
                        '[element/@name=(',for $n in $item/element/@name return concat('''',replace($n,'''',''''''),''''),')]'
                    ) else
                    if ($item[include]) then (
                        '[include/@ref=(',for $n in $item/include/@ref return concat('''',replace($n,'''',''''''),''''),')]'
                    ) else
                    if ($item[choice]) then (
                        '[choice]'
                    ) else
                    if ($item[@minimumMultiplicity][@maximumMultiplicity]) then (
                        '[@minimumMultiplicity = ',$item/@minimumMultiplicity,']','[@maximumMultiplicity = ',$item/@maximumMultiplicity,']'
                    ) else
                    if ($item[@minimumMultiplicity]) then (
                        '[@minimumMultiplicity = ',$item/@minimumMultiplicity,']'
                    ) else
                    if ($item[@maximumMultiplicity]) then (
                        '[@maximumMultiplicity = ',$item/@maximumMultiplicity,']'
                    ) else (
                        ''
                    )
                )
                else
                if ($item[self::include]) then (
                    '[@ref=''',replace($item/@ref,'''',''''''),''']'
                )
                else
                if ($item[self::let]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[self::defineVariable]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[self::assert]) then (
                    '[@test=''',replace($item/@test,'''',''''''),''']'
                )
                else
                if ($item[self::report]) then (
                    '[@test=''',replace($item/@test,'''',''''''),''']'
                )
                else
                if ($item[self::constraint]) then (
                    '[@language=''',replace($item/@language,'''',''''''),''']'
                )
                else
                if ($item[self::example]) then (
                    '[',count(preceding-sibling::example) + 1,']'
                )
                else
                if ($item[self::text]) then (
                    '[. = ''',replace(., '''', ''''''),''']'
                )
                else
                if ($item[@name]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[@code][@codeSystem]) then (
                    '[@code=''',replace($item/@code,'''',''''''),''']',
                    '[@codeSystem=''',replace($item/@codeSystem,'''',''''''),''']'
                )
                else
                if ($item[@valueSet]) then (
                    '[@valueSet=''',replace($item/@valueSet,'''',''''''),''']'
                )
                else
                if ($item[@language]) then (
                    '[@language=''',replace($item/@language,'''',''''''),''']'
                )
                else (
                    let $n  := $item/(@* except (@isOptional|@prohibited|@datatype|@id))[1]
                    return (
                        if ($n) then concat('[@',name($n),'=''',replace($n,'''',''''''),''']') else ()
                    )
                )
            ), '')
        )
    , '/')
};

declare %private function utiltemp:index-of-node($nodes as node()* , $nodeToFind as node()) as xs:integer* {
    for $seq in (1 to count($nodes))
    return $seq[$nodes[$seq] is $nodeToFind]
};

declare %private function utiltemp:index-node-in-set($nodes as node()*, $nodeToFind as node()) as element()* {
    let $n      :=
        if ($nodeToFind[@name]/name()='attribute') then
            $nodes[self::attribute][@name = $nodeToFind/@name]
        else
        if ($nodeToFind[@contains]/name()='element') then
            $nodes[self::element][@name = $nodeToFind/@name][@contains = $nodeToFind/@contains]
        else
        (: exceptional case in template id="2.16.840.1.113883.10.20.29.1" name="USRealmHeaderforPatientGeneratedDocumentV2" where templateId is defined twice with same @root, different extension. first no @id, second with @id :)
        if ($nodeToFind[self::element][@name = ('hl7:templateId', 'cda:templateId')][attribute[@name = 'root'][@value]][attribute[@name = 'extension'][@value]]) then
            $nodes[self::element][@name = $nodeToFind/@name][attribute[@name = 'root'][@value = $nodeToFind/attribute[@name = 'root']/@value]][attribute[@name = 'extension'][@value = $nodeToFind/attribute[@name = 'extension']/@value]]
        else
        if ($nodeToFind[self::element][@name = ('hl7:templateId', 'cda:templateId')][attribute[@name = 'root'][@value]]) then
            $nodes[self::element][@name = $nodeToFind/@name][attribute[@name = 'root'][@value = $nodeToFind/attribute[@name = 'root']/@value]]
        else
        if ($nodeToFind/name()='element') then
            $nodes[self::element][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='choice') then
            if ($nodeToFind[*/@id]) then
                $nodes[self::choice][*/@id = $nodeToFind/*/@id]
            else
            if ($nodeToFind[element]) then
                $nodes[self::choice][element/@name = $nodeToFind/element/@name]
            else
            if ($nodeToFind[include]) then
                $nodes[self::choice][include/@ref = $nodeToFind/include/@ref]
            else
            if ($nodeToFind[choice]) then
                $nodes[self::choice][choice]
            else
            if ($nodeToFind[@minimumMultiplicity][@maximumMultiplicity]) then
                $nodes[self::choice][@minimumMultiplicity = $nodeToFind/@minimumMultiplicity][@maximumMultiplicity = $nodeToFind/@maximumMultiplicity]
            else
            if ($nodeToFind[@minimumMultiplicity]) then
                $nodes[self::choice][@minimumMultiplicity = $nodeToFind/@minimumMultiplicity]
            else
            if ($nodeToFind[@maximumMultiplicity]) then
                $nodes[self::choice][@maximumMultiplicity = $nodeToFind/@maximumMultiplicity]
            else (
                $nodes[self::choice]
            )
        else
        if ($nodeToFind/name()='include') then
            $nodes[self::include][@ref = $nodeToFind/@ref]
        else
        if ($nodeToFind/name()='let') then
            $nodes[self::let][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='defineVariable') then
            $nodes[self::defineVariable][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='assert') then
            $nodes[self::assert][@test = $nodeToFind/@test]
        else
        if ($nodeToFind/name()='report') then
            $nodes[self::report][@test = $nodeToFind/@test]
        else
        if ($nodeToFind/name()='text') then 
            $nodes[self::text][. = $nodeToFind]
        else
        if ($nodeToFind/name()='constraint') then
            $nodes[self::constraint][@language = $nodeToFind/@language]
        else
        if ($nodeToFind/name()='example') then
            $nodes[self::example][count($nodeToFind/preceding-sibling::example) + 1]
        else (
            $nodes[name()=$nodeToFind/name()][string-join(for $att in (@* except (@isOptional|@prohibited|@datatype|@id)) order by name($att) return $att,'')=$nodeToFind/string-join(for $att in (@* except (@isOptional|@prohibited|@datatype|@id)) order by name($att) return $att,'')]
        )
    
    return $n
};

(: keep it simple for performance. if the requested valueSet is in the project terminology by ref, don't look further :)
declare %private function utiltemp:isValueSetInScope($projectPrefix as item(), $ref as attribute()?,$flexibility as attribute()?,$valueSetList as element()*) as attribute()* {
    if (string-length($ref)=0) then ()
    else (
        let $vsElms := $valueSetList[(@id|@name|@ref)=$ref]
        let $vsEff  := if ($flexibility castable as xs:dateTime) then $flexibility else (max($vsElms[@effectiveDate]/xs:dateTime(@effectiveDate)))
       
        return
            if ($vsElms[@effectiveDate=$vsEff]) then (
                attribute vsid {$vsElms[@effectiveDate=$vsEff][1]/(@id|@ref)},
                attribute vsname {$vsElms[@effectiveDate=$vsEff][1]/@name},
                attribute vsdisplayName {$vsElms[@effectiveDate=$vsEff][1]/@displayName}
            )
            else if ($vsElms[@ref]) then (
                attribute vsid {$vsElms[1]/(@id|@ref)},
                attribute vsname {$vsElms[1]/@name},
                attribute vsdisplayName {$vsElms[1]/@displayName}
            )
            else (
                attribute linkedartefactmissing {'true'}
            )
    )
};

(: keep it simple for performance. if the requested template is in the project rules by ref, don't look further :)
declare %private function utiltemp:isTemplateInScope($projectPrefix as item(), $ref as attribute()?,$flexibility as attribute()?,$templateList as element()*) as attribute()* {
    if (string-length($ref)=0) then ()
    else (
        let $templates  := (utillib:getTemplateById($ref, ($flexibility, 'dynamic')[1], $projectPrefix, (), ())[@id])[1]
        
        return
            if ($templates) then (
                attribute tmid {$templates/@id},
                attribute tmdisplayName {if ($templates/@displayName) then $templates/@displayName else $templates/@name}
            ) else (
                attribute linkedartefactmissing {'true'}
            )
    )
};

declare %private function utiltemp:getDatatype($datatype as xs:string?, $classification-format as xs:string?) as xs:string? {
    let $classification-format  := if (empty($classification-format)) then 'hl7v3xml1' else $classification-format
    let $datatypes              := $setlib:colDecorCore//supportedDataTypes[@type = $classification-format]
    let $datatypeName           := if (contains($datatype,':')) then substring-after($datatype,':') else ($datatype)
    let $flavor                 := $datatypes//flavor[@name=$datatypeName]
    return
        if ($flavor) then ($flavor/ancestor::dataType[1]/@name) else ($datatype)
};

(: re-write attribute strength CNE is required and CWE is extensible :)
declare %private function utiltemp:rewriteStrength($os as xs:string?) as xs:string? {
    let $r := if ($os = 'CNE') then 'required' else if ($os = 'CWE') then 'extensible' else $os
    return $r
};

declare function utiltemp:checkTemplateAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean, $breaklock as xs:boolean) as element()?{

    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-RULES)) then () else (
            error($errors:FORBIDDEN, 'User '|| $authmap?name || ' does not have sufficient permissions to modify templates in project ' || $decor/project/@prefix || '. You have to be an active author in the project.')
        )
    
    return
        if ($checkLock) then (
            let $lock           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
            let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breaklock)
            return
                if ($lock) then $lock else (
                    error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this template (anymore). Get a lock first.'))
                )    
            )
            else ()
};

declare %private function utiltemp:checkTemplateParameters($parameters as element(parameters), $template as element(template)*) {

let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    
    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $check                  :=
        for $param in $parameters/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($template, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:templatestatusmap, string($template/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri'
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/name' 
            case '/displayName' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/isClosed' return (
                if ($op = 'remove') then () else 
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false.'
                )
            )
            case '/desc' 
            case '/copyright'
            case '/purpose' return (
                if ($param[count(value/*[name() = $pathpart]) = 1])
                then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $pathpart], $op, $path)
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart])
                )
            )
            case '/classification' return (
                if ($param[count(value/classification) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/classification[@format]) then
                            if ($param/value/classification[@format = $utiltemp:TEMPLATE-FORMATS/enumeration/@value]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Classification has unsupported .format value ' || string-join($param/value/classification/@format, ', ') || '. SHALL be one of: ' || string-join($utiltemp:TEMPLATE-FORMATS/enumeration/@value, ', ')
                            )
                        else (),
                        if ($param/value/classification[@type]) then
                            if ($param/value/classification[@type = $utiltemp:TEMPLATE-TYPES/enumeration/@value]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Classification has unsupported .type value ' || string-join($param/value/classification/@type, ', ') || '. SHALL be one of: ' || string-join($utiltemp:TEMPLATE-TYPES/enumeration/@value, ', ')
                            )
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/relationship' return (
                if ($param[count(value/relationship) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/relationship[@type = $utiltemp:RELATIONSHIP-TYPES/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Relationship has unsupported .type value ' || string-join($param/value/relationship/@type, ', ') || '. SHALL be one of: ' || string-join($utiltemp:RELATIONSHIP-TYPES/enumeration/@value, ', ')
                        ),
                        if ($param/value/relationship[count(@template | @model) = 1]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship SHALL have exactly one of template or model property'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/context' return (
                if ($param[count(value/context) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/context[count(@id | @path) = 1]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Context SHALL have exactly one of id or path property'
                        ),
                        if ($param/value/context[empty(@id)] | $param/value/context[@id = ('*', '**')]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Context SHALL be one of * or **'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/item' return (
                if ($param[count(value/item) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/item[@label]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Item SHALL have label property'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/publishingAuthority' return (
                if ($param[count(value/publishingAuthority) = 1]) 
                then ruleslib:checkPublishingAuthority($param/value/publishingAuthority, $op, $path)
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single publishingAuthority under value. Found ' || count($param/value/publishingAuthority)
                )
            )
            case '/example' return (
                let $unsupportedtypes := $param/value/example/@type[not(. = $utiltemp:EXAMPLE-TYPES/enumeration/@value)]
                return
                    if ($unsupportedtypes) then
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Example has unsupported type value(s) ' || string-join($unsupportedops, ', ') || '. SHALL be one of: ' || string-join($utiltemp:EXAMPLE-TYPES/enumeration/@value, ', ')
                    else ()
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
        
    return
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )

};

declare %private function utiltemp:patchTemplateParameters($parameters as element(parameters), $template as element(template)*) as element(template) {

    let $update                 :=
        for $param in $parameters/parameter
        return
            switch ($param/@path)
            case '/statusCode'
            case '/expirationDate'
            case '/officialReleaseDate'
            case '/canonicalUri'
            case '/versionLabel'
            case '/name'
            case '/isClosed' return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := attribute {$attname} {$param/@value}
                let $stored   := $template/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $template
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/displayName' return (
                let $attname  := substring-after($param/@path, '/')
                let $newdisp  := attribute displayName {$param/@value}
                let $newname  := attribute name {utillib:shortName($newdisp)}
                let $stored   := $template/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return (
                    if ($template/@*[name() = 'displayName']) then update replace $template/@*[name() = 'displayName'] with $newdisp else update insert $newdisp into $template,
                    if ($template/@*[name() = 'name']) then update replace $template/@*[name() = 'name'] with $newname else update insert $newname into $template
                )
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/desc' return (
                (: only one per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/desc)
                let $stored   := $template/*[name() = $elmname][@language = $param/value/desc/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $template
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/publishingAuthority' return (
                (: only one possible :)
                let $new      := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
                let $stored   := $template/publishingAuthority
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $template
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/purpose' return (
                (: only one possible per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/purpose)
                let $stored   := $template/*[name() = $elmname][@language = $param/value/purpose/@language]
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $template
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/copyright' return (
                (: only one per language :)
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/copyright)
                let $stored   := $template/copyright[@language = $param/value/copyright/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $template
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/classification' return (
                (: only one per type or format :)
                let $new      := utiltemp:prepareClassificationForUpdate($param/value/classification)
                let $stored   := $template/classification
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then (update insert $new preceding $stored[1], update delete $stored) else (update insert $new into $template)
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/relationship' return (
                (: only one per template or model, insert before others :)
                let $new      := utiltemp:prepareRelationshipForUpdate($param/value/relationship)
                let $stored   := $template/relationship[@template = $new/@template] | $template/relationship[@model = $new/@model]
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then (update insert $new preceding $stored[1], update delete $stored) else (update insert $new into $template)
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/context' return (
                (: only one :)
                let $new      := utiltemp:prepareContextForUpdate($param/value/context)
                let $stored   := $template/context
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $template
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/item' return (
                (: only one :)
                let $new      := utiltemp:prepareItemForUpdate($param/value/item)
                let $stored   := $template/item[1]
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $template
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/example' return (
                (: only one :)
                let $new      := 
                    for $example in $param/value/example
                    return
                        <example>
                        {
                            $example/@type[not(. = '')],
                            $example/@caption[not(. = '')],
                            $example/node()
                        }
                        </example>
                let $stored   := update delete $template/example
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($new) then update insert $new into $template else ()
                case 'remove' return ()
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
            
    return $template
};

declare function utiltemp:templateScan($template as element()?, $projectPrefix as xs:string?, $projectLanguage as xs:string, $by as xs:string, $indent as xs:int, $ylevel as xs:int, $templatesSoFar as xs:string*) as element()* {

(: get the templates, elements and include chain :)
    let $contains           :=   
        switch($projectLanguage)
            case "en-US" return "contains"
            case "de-DE" return "enthält"
            case "nl-NL" return "bevat"
            default return "contains"
 
    return        
    if (empty($template)) then ()  
    
    else if ($template[self::template]) then (
        <template>
        {
            $template/(@id | @name | @displayName),
            attribute effectiveDate {format-dateTime($template/@effectiveDate, '[Y0001]&#8209;[M01]&#8209;[D01]')},
            attribute by {$by},
            attribute len {
                max((string-length($template/@id),string-length($template/@name))) + 
                string-length($template/@effectiveDate) + 
                string-length($contains) + 
                1
            },
            attribute classification {$template/classification/@type/string()},
            for $lc at $step in ($template//element[@contains] | $template//include[@ref])
            let $xid        := if ($lc[self::element]) then $lc/@contains else $lc/@ref
            let $lcby       := if ($lc[self::element]) then $contains else $lc/name()
            let $xflx       := if ($lc[@flexibility]) then $lc/@flexibility else 'dynamic'
            let $min        := $lc/@minimumMultiplicity
            let $max        := $lc/@maximumMultiplicity
            let $lct        := if (empty($projectPrefix)) then utillib:getTemplateById($xid, $xflx)[@id][1] else utillib:getTemplateById($xid, $xflx, $projectPrefix, (), $projectLanguage)[@id][1]
            return 
                if ($templatesSoFar[.=$lct/concat(@id,@effectiveDate)]) then (
                    <template recurse="true">
                    {
                        $lct/(@id | @name | @displayName),
                        if ($min) then attribute minimumMultiplicity { $min } else (),
                        if ($max) then attribute maximumMultiplicity { $max } else (),
                        attribute effectiveDate {format-dateTime($lct/@effectiveDate, '[Y0001]&#8209;[M01]&#8209;[D01]')},
                        attribute by {$by},
                        attribute len {
                            max((string-length($lct/@id),string-length($lct/@name))) + 
                            string-length($lct/@effectiveDate) + 
                            string-length($contains) + 
                            1
                        }
                    }
                    </template>
                ) 
                else utiltemp:templateScan ($lct, $projectPrefix, $projectLanguage, $lcby, $indent + 1, $ylevel + $step, ($templatesSoFar, $lct/concat(@id,@effectiveDate)))
        }
        </template>
    ) 
    
    else if ($template[self::element][@contains]) then (
        <element>
        {
            $template/@contains,
            $template/@flexibility,
            $template/@minimumMultiplicity,
            $template/@maximumMultiplicity,
            attribute by {$by},
            attribute len { string-length($template/@contains) + string-length($template/@flexibility) },
            for $lc at $step in ($template//element[@contains] | $template//include[@ref])
            let $xid        := if ($lc[self::element]) then $lc/@contains else $lc/@ref
            let $lcby       := if ($lc[self::element]) then $contains else $lc/name()
            let $xflx       := if ($lc[@flexibility]) then $lc/@flexibility else 'dynamic'
            let $min        := $lc/@minimumMultiplicity
            let $max        := $lc/@maximumMultiplicity
            let $lct        := if (empty($projectPrefix)) then utillib:getTemplateById($xid, $xflx)[@id] else utillib:getTemplateById($xid, $xflx, $projectPrefix, (), $projectLanguage)[@id] 
            return 
                if ($templatesSoFar[.=$lct/concat(@id,@effectiveDate)]) then (
                    <template recurse="true">
                    {
                        $lct/(@id | @name | @displayName),
                        if ($min) then attribute minimumMultiplicity { $min } else (),
                        if ($max) then attribute maximumMultiplicity { $max } else (),
                        attribute effectiveDate {format-dateTime($lct/@effectiveDate, '[Y0001]&#8209;[M01]&#8209;[D01]')},
                        attribute by {$by},
                        attribute len {
                            max((string-length($lct/@id),string-length($lct/@name))) + 
                            string-length($lct/@effectiveDate) + 
                            string-length($contains) + 
                            1
                        }
                    }
                    </template>
                ) 
                else utiltemp:templateScan($lct, $projectPrefix, $projectLanguage, $lcby, $indent + 1, $ylevel + $step, ($templatesSoFar, $lct/concat(@id,@effectiveDate)))
        }
        </element>
    ) 
    
    else if ($template[self::include][@ref]) then (
        <include>
        {
            $template/@ref,
            $template/@flexibility,
            $template/@minimumMultiplicity,
            $template/@maximumMultiplicity,
            attribute by {$by},
            attribute len { string-length($template/@ref) + string-length($template/@flexibility) },
            for $lc at $step in ($template//element[@contains] | $template//include[@ref])
            let $xid        := if ($lc[self::element]) then $lc/@contains else $lc/@ref
            let $lcby       := if ($lc[self::element]) then $contains else $lc/name()
            let $xflx       := if ($lc[@flexibility]) then $lc/@flexibility else 'dynamic'
            let $min        := $lc/@minimumMultiplicity
            let $max        := $lc/@maximumMultiplicity
            let $lct        := if (empty($projectPrefix)) then utillib:getTemplateById($xid, $xflx)[@id] else utillib:getTemplateById($xid, $xflx, $projectPrefix, (), $projectLanguage)[@id] 
            return 
                if ($templatesSoFar[.=$lct/concat(@id,@effectiveDate)]) then (
                    <template recurse="true">
                    {
                        $lct/(@id | @name | @displayName),
                        if ($min) then attribute minimumMultiplicity { $min } else (),
                        if ($max) then attribute maximumMultiplicity { $max } else (),
                        attribute effectiveDate {format-dateTime($lct/@effectiveDate, '[Y0001]&#8209;[M01]&#8209;[D01]')},
                        attribute by {$by},
                        attribute len {
                            max((string-length($lct/@id),string-length($lct/@name))) + 
                            string-length($lct/@effectiveDate) + 
                            string-length($contains) + 
                            1
                        }
                    }
                    </template>
                ) 
                else utiltemp:templateScan ($lct, $projectPrefix, $projectLanguage, $lcby, $indent + 1, $ylevel + $step, ($templatesSoFar, $lct/concat(@id,@effectiveDate)))
        }
        </include>
    ) 
    else ()
};

declare function utiltemp:chainCopy1 ($template as element(), $indent as xs:int) as element() {
    
    element {$template/name()} {
        $template/@*,
        attribute indent {$indent},
        for $i at $step in $template/*
        return utiltemp:chainCopy1($i, $indent+1)
    }
};

declare function utiltemp:chainCopy2 ($template as element(), $pos as xs:int) as element() {
    
    element {$template/name()} {
        $template/@*,
        attribute pos {$pos},
        for $i at $step in $template//*
        let $oname := $i/name()
        return element {$oname} {
            $i/@*,
            attribute pos {$step},
            attribute connector {
                if ($template//*[($step - 1)]/@indent = $template//*[$step]/@indent) then 2 
                else if ($template//*[($step - 1)]/@indent < $template//*[$step]/@indent) then 1
                else if ($template//*[($step - 1)]/@indent > $template//*[$step]/@indent) then 4 + 1
                else 0
            }
        }
    }
};

(:
    hgraph format returns the hierarchical graph of the template chain including classification
    hgraphwiki is the same as hgraph but the OIDs of the templates are in Mediawiki link style [[1.2.3...]]
    transclusionwikilist returns the list of templates in Mediawiki transclusion style, sorted by classification
    wikilist returns the list of templates in Mediawiki list style, sorted by id
    ==========
:)
declare function utiltemp:apply-hgraph-stylesheet($input as node(), $format as xs:string) {
    
    let $xsltParameters     :=
        <parameters>
            <param name="outputformat" value="{$format}"/>
            <param name="coretableonly" value="false"/>
        </parameters>
    
    let $xslt               := xs:anyURI(concat('xmldb:exist://', $setlib:strDecorCore, '/Template2list8hgraph.xsl'))
    
    return transform:transform($input, $xslt, $xsltParameters)
};
