xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Questionnaire API allows read, create, update on DECOR links into HL7 FHIR Questionnaire :)
module namespace qapi              = "http://art-decor.org/ns/api/questionnaire";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "/db/apps/api/modules/library/util-lib.xqm";
import module namespace utilq       = "http://art-decor.org/ns/api/util-questionnaire" at "library/util-questionnaire-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";
import module namespace utilgg      = "http://art-decor.org/ns/api/util-governancegroup" at "library/util-governancegroup-lib.xqm";

declare namespace json      = "http://www.json.org";
declare namespace f         = "http://hl7.org/fhir";

(:~ Returns a list of zero or more questionnaires
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $objectId                - optional. Filter output to just those based on this questionnaire/relationship/@ref. For each questionnaire
    @param $objectEffectiveDate     - optional. Filter output to just those based on this questionnaire/relationship/@flexibility. relevant when objectId has a value.
    @param $sort                    - optional parameter to indicate sorting. Only option 'name'
    @param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
    @return all live repository/non-private questionnaire as JSON
    @since 2020-05-03
:)
declare function qapi:getQuestionnaireList($request as map(*)) {
    let $governanceGroupId              := $request?parameters?governanceGroupId[not(. = '')]
    let $projectPrefix                  := $request?parameters?prefix[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $objectId                       := $request?parameters?objectId[not(. = '')]
    let $objectEffectiveDate            := $request?parameters?objectEffectiveDate[not(. = '')]
    let $sort                           := $request?parameters?sort[string-length() gt 0]
    let $sortorder                      := $request?parameters?sortorder[. = 'descending']
    let $max                            := $request?parameters?max[not(. = '')]
    let $resolve                        := if (empty($governanceGroupId)) then not($request?parameters?resolve = false()) else $request?parameters?resolve = true()
    
    let $check                          :=
        if (empty($governanceGroupId) and empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter governanceGroupId or prefix')
        else 
        if (not(empty($governanceGroupId)) and not(empty($projectPrefix))) then 
            error($errors:BAD_REQUEST, 'Request SHALL have parameter governanceGroupId or prefix, not both')
        else ()
        
    let $startT                         := util:system-time()
    
    let $result                         :=
        if (empty($governanceGroupId)) then utilq:getQuestionnaireList($projectPrefix, $projectVersion, $projectLanguage, $objectId, $objectEffectiveDate, $sort, $sortorder, $max)
        else (
            for $projectId in utilgg:getLinkedProjects($governanceGroupId)/@ref
            return utilq:getQuestionnaireList($projectId, $projectVersion, $projectLanguage, $objectId, $objectEffectiveDate, $sort, $sortorder, $max)     
        )
        
    let $durationT              := (util:system-time() - $startT) div xs:dayTimeDuration("PT0.001S")
    
    return
        <list artifact="{$result[1]/@artifact}" elapsed="{$durationT}" current="{sum($result/xs:integer(@current))}" total="{sum($result/xs:integer(@total))}" all="{sum($result/xs:integer(@allcnt))}" resolve="{$resolve}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            if (empty($governanceGroupId)) then attribute project {$projectPrefix} else attribute governanceGroupId {$governanceGroupId},
            utillib:addJsonArrayToElements($result/*)
        }
        </list>
};

(:~  Return questionnaire based on id and optionally effectiveDate. If effectiveDate is empty, the latest version is returned

    @param $id                      - required questionnaire/@id
    @param $effectiveDate           - optional questionnaire/@effectiveDate
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @return exactly 1 questionnaire or nothing if not found
    @since 2015-04-27
:)
declare function qapi:getLatestQuestionnaire($request as map(*)) {
    qapi:getQuestionnaire($request)
};

declare function qapi:getQuestionnaire($request as map(*)) {

    let $projectPrefixOrId              := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]

    let $qs                             := utillib:getQuestionnaire($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage)
    let $oidnamemap                     := utilq:buildOidNameMap($qs, $projectLanguage[1])
    let $results                        := 
        for $q in $qs
        return utilq:handleQuestionnaire($q, (), $projectLanguage, $oidnamemap, collection(util:collection-name($q))//f:QuestionnaireResponse, false(), false(), true())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple questionnaires for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )
};


(:~  Return questionnaire based on id and optionally effectiveDate. If effectiveDate is empty, the latest version is returned

    @param $id                      - required questionnaire/@id
    @param $effectiveDate           - optional questionnaire/@effectiveDate
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @return exactly 1 questionnaire or nothing if not found
    @since 2025-08-26
:)
declare function qapi:getLatestQuestionnaireExtract($request as map(*)) {
    qapi:getQuestionnaireExtract($request)
};

declare function qapi:getQuestionnaireExtract($request as map(*)) {

    let $projectPrefixOrId              := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $format                         := $request?parameters?format[not(. = '')]
    let $download                       := $request?parameters?download=true()
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    (: default behavior Accept header:
    - backend:   default Accept header is */* if accept header is omitted   
    - frontend:  default Accept header is application/xml :)
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json', 'text/plain', 'text/fsh', 'application/*', 'text/*', '*/*')])[1]
    
    (: when you do a right mouse click for downloading content in Edge or Chrome, you effectively do a GET without Accept header :)
    let $check                          := 
        if (empty($acceptTypes) and empty($format)) then
            error($errors:BAD_REQUEST, 'Your request does not have an Accept header, and is missing a format parameter. Don''t know what to do')
        else ()
    
    (: parameter format overrides Accept header  :)
    let $format                         := if ($format) then $format else tokenize($acceptedType, '/')[2]

    (: check valid formats :)
    let $check                          :=
        if ($format = ('xml', 'json', 'fsh')) then () else 
            error($errors:BAD_REQUEST, 'Value for parameter format ''' || string-join($format, ', ') || ''' not supported. Supported are: xml, json, fsh')

    let $qs                             := utillib:getQuestionnaire($id, $effectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage)
    let $oidnamemap                     := utilq:buildOidNameMap($qs, $projectLanguage[1])
    let $results                        := 
        for $q in $qs
        return utilq:handleQuestionnaire($q, (), $projectLanguage, $oidnamemap, collection(util:collection-name($q))//f:QuestionnaireResponse, false(), false(), true())
    
        let $filename                   := 'DS_' || $results/@iddisplay || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $format
    
    let $results                        :=
        if ($format = 'fsh') then ((:utilfsh:convertquestionnaire2Fsh($results, $projectLanguage, $filename)/text():))
        else for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/node(), $format)
                }
    
    (: determine response Content-Type/Media-type :)
    let $responseType                   := $utillib:formatacceptmap?($format) 
           
    let $responseheader                 := response:set-header('Content-Type', $responseType || '; charset=utf-8')
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple questionnaires for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else roaster:response(200, $responseType, $results)
};


(:~ Retrieves DECOR questionnaire history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function qapi:getQuestionnaireHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-QUESTIONNAIRE, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-QUESTIONNAIRE}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Update DECOR questionnare parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

    { "op": "[add|remove|replace]", "path": "e.g. [/statusCode|/expirationDate|/officialReleaseDate|/canonicalUri|/versionLabel|/name|/displayName|/experimental|/desc|/publishingAuthority|/copyright|/completeCodeSystem|/conceptList]", "value": "[string|object]" }
    
    where
    
    * op - add & replace (statusCode, expirationDate, officialReleaseDate, canonicalUri, versionLabel, name, displayName, experimental, desc, publishingAuthority, copyright, completeCodeSystem, conceptList) or remove (desc, publishingAuthority, copyright, completeCodeSystem, conceptList)
    
    * path - see above
    
    * value - string when the path is not /. object when path is /
    
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the valueSet to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return questionnaire structure
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function qapi:patchQuestionnaire($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()  
    
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                          :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $results                        := utilq:patchQuestionnaire($authmap, true(), string($id), $effectiveDate, $data)
    
    return (
        for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
    )
};

(: Create a questionnaire, either empty or with input data or based on another questionnaire
    @param $projectPrefix project to create this questionnaire in
    @param $targetDate If true invokes effectiveDate of the new questionnaire as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
    @param $sourceId parameter denoting the id of a questionnaire to use as a basis for creating the new questionnaire
    @param $sourceEffectiveDate parameter denoting the effectiveDate of a questionnaire to use as a basis for creating the new questionnaire,
    @param $keepIds Only relevant if source questionnaire is specified. If true, the new questionnaire will keep the same ids for the new questionnaire, and only update the effectiveDate
    @param $baseId Only relevant when a source questionnaire is specified and `keepIds` is false. This overrides the default base id for questionnaire in the project. The value SHALL match one of the projects base ids for questionnaire
    @return questionnaire as xml with json:array set on elements
:)
declare function qapi:postQuestionnaire($request as map(*)) {

    let $authmap                        := $request?user
    let $projectPrefix                  := $request?parameters?project[not(. = '')]
    let $data                           := utillib:getBodyAsXml($request?body, 'questionnaire', ())
    let $targetDate                     := $request?parameters?targetDate = true()
    let $sourceId                       := $request?parameters?sourceId
    let $sourceEffectiveDate            := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?sourceEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?sourceEffectiveDate[string-length() gt 0]
        }
    let $keepIds                        := $request?parameters?keepIds = true()
    let $baseId                         := $request?parameters?baseId
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                          :=
        if (empty($projectPrefix)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    
    let $results                        := utilq:createQuestionnaire($authmap, $projectPrefix, $data, $targetDate, $sourceId, $sourceEffectiveDate, $keepIds, $baseId)
    
    return
        roaster:response(201, 
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

declare function qapi:putQuestionnaire($request as map(*)) {

    let $authmap                        := $request?user
    let $qid                            := $request?parameters?id
    let $qed                            := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $fhirVersion                    := $request?parameters?fhirVersion[string-length() gt 0]
    let $data                           := utillib:getBodyAsXml($request?body, 'questionnaire', ())
    let $deletelock                     := $request?parameters?deletelock = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $result                         := utilq:putQuestionnaire($authmap, string($qid), $qed, $data, $fhirVersion, $deletelock, true())
    
    return 
        if (empty($result)) then () else (
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
        )
};