xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
:)
(:~ Dataset concept API allows read, create, update on DECOR concepts in DECOR datasets :)
module namespace demapapi              = "http://art-decor.org/ns/api/conceptmap";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace vsapi       = "http://art-decor.org/ns/api/valueset" at "valueset-api.xqm";
import module namespace serverapi   = "http://art-decor.org/ns/api/server" at "server-api.xqm";

declare namespace json      = "http://www.json.org";

declare %private variable $demapapi:EQUIVALENCY-TYPE        := utillib:getDecorTypes()/EquivalencyType;
declare %private variable $demapapi:CODINGSTRENGTH-TYPE     := utillib:getDecorTypes()/CodingStrengthType;

(:~ Retrieves DECOR concept maps based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
@param $id                          - required parameter denoting the id of the concept
@param $effectiveDate               - required parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
@param $transactionId               - required parameter denoting the id of the transaction that the concept is in
@param $transactionEffectiveDate    - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
@param $projectVersion              - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
@param $projectLanguage             - optional parameter to select from a specific compiled language
@param $associations                - optional boolean parameter relevant if $treeonly = 'false' to include associations: terminologyAssociation, identifierAssociation
@return as-is or as compiled as JSON
@since 2020-05-03
:)
declare function demapapi:getConceptMapList($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $transactionId                  := $request?parameters?transactionId
    let $transactionEffectiveDate       := $request?parameters?transactionEffectiveDate[string-length() gt 0]
    let $projectVersion                 := $request?parameters?release
    let $projectLanguage                := $request?parameters?language
    let $associationMode                := $request?parameters?mode
    
    let $result                         := demapapi:getConceptMapList($id, $effectiveDate, $transactionId, $transactionEffectiveDate, $projectVersion, $projectLanguage, $associationMode)

    return
        <list artifact="MP" current="{count($result/*)}" total="{count($result/*)}" all="{count($result/*)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($result/*)
        }
        </list>
};

(:~ Update DECOR dataset or transaction concept. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "/", "value": "[terminologyAssociation|identifierAssociation]" }

where
* op - add (object associations, tracking, event) or remove (object associations only) or replace (displayName, priority, type, tracking, assignment)
* path - / 
* value - terminologyAssociation|identifierAssociation object
@param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
@param $id                       - required. the id for the issue to update 
@param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
@return issue structure including generated meta data
@since 2020-05-03
@see http://tools.ietf.org/html/rfc6902
:)
declare function demapapi:patchConceptMap($request as map(*)) {

    let $authmap                        := $request?user
    let $deid                           := $request?parameters?id
    let $deed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $transactionId                  := $request?parameters?transactionId
    let $transactionEffectiveDate       := $request?parameters?transactionEffectiveDate[string-length() gt 0]
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    (:let $s                      := xmldb:store('/db/apps/decor/tmp', 'ttt.xml', $data) :)
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $result                         := demapapi:patchConceptMap($authmap, string($deid), $deed, $transactionId, $transactionEffectiveDate, $data)
    return (
        roaster:response(200, 
            <list artifact="MP" current="{count($result/*)}" total="{count($result/*)}" all="{count($result/*)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
            {
                $result/(@* except (@artifact | @current | @total | @all)),
                utillib:addJsonArrayToElements($result/*)
            }
            </list>)
    )

};

(:~ Retrieve DECOR concept map list from a transaction or dataset :)
declare function demapapi:getConceptMapList($id as xs:string, $effectiveDate as xs:string?, $transactionId as xs:string?, $transactionEffectiveDate as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?, $associationMode as xs:string?) {
    let $id                         := $id[not(. = '')]
    let $effectiveDate              := $effectiveDate[not(. = '')]
    let $projectVersion             := $projectVersion[not(. = '')]
    let $projectLanguage            := $projectLanguage[not(. = '')]
    let $transactionId              := $transactionId[not(. = '')]
    let $transactionEffectiveDate   := $transactionEffectiveDate[not(. = '')]
    let $associationMode            := if ($associationMode) then $associationMode else if (empty($transactionId)) then 'normal' else 'all'
    
    let $storedConcept              :=
        if (empty($id)) then 
            () 
        else 
        if (empty($transactionId)) then 
            utillib:getConcept($id, $effectiveDate, $projectVersion, $projectLanguage) 
        else (
            utillib:getTransactionConcept($id, $effectiveDate, $transactionId, $transactionEffectiveDate, $projectVersion, $projectLanguage)
        )
    
    let $datasetConcept             := 
        if ($storedConcept) then
            if (empty($transactionId)) then
                $storedConcept
            else (
                let $dsid   := $storedConcept/ancestor::representingTemplate/@sourceDataset
                let $dsed   := $storedConcept/ancestor::representingTemplate/@sourceDatasetFlexibility
                
                return
                utillib:getDatasetConcept($dsid, $dsed, $storedConcept/@ref, $storedConcept/@flexibility, $projectVersion, $projectLanguage)
            )
        else ()
    let $originalConcept            := if ($datasetConcept) then utillib:getOriginalForConcept($datasetConcept) else ()
    
    return
        if ($storedConcept) then
            utillib:getConceptAssociations($storedConcept, $originalConcept, false(), $associationMode, true())
        else ()
};

(:~ Central logic for patching an existing dataset or transaction concept map

@param $authmap         - required. Map derived from token
@param $id              - required. DECOR concept/@id to update
@param $effectiveDate   - required. DECOR concept/@effectiveDate to update
@param $data            - required. DECOR concept xml element containing everything that should be in the updated concept
@return concept object as xml with json:array set on elements
:)
declare function demapapi:patchConceptMap($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $transactionId as xs:string?, $transactionEffectiveDate as xs:string?, $data as element(parameters)) {

    let $associationMode            := if ($transactionId) then 'all' else 'normal'
    let $projectVersion             := ()
    let $projectLanguage            := ()
    let $id                         := $id[not(. = '')]
    let $effectiveDate              := $effectiveDate[not(. = '')]
    let $transactionId              := $transactionId[not(. = '')]
    let $transactionEffectiveDate   := $transactionEffectiveDate[not(. = '')]
    
    let $storedConcept              :=
        if (empty($id)) then 
            () 
        else 
        if (empty($transactionId)) then 
            utillib:getConcept($id, $effectiveDate, $projectVersion, $projectLanguage) 
        else (
            utillib:getTransactionConcept($id, $effectiveDate, $transactionId, $transactionEffectiveDate, $projectVersion, $projectLanguage)
        )
    
    let $check                  :=
        if ($storedConcept) then () else
        if ($transactionId) then 
            error($errors:BAD_REQUEST, 'Concept with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist in transaction with id ' || $transactionId || ' and effectiveDate ' || $transactionEffectiveDate)
        else (
            error($errors:BAD_REQUEST, 'Concept with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    
    let $decor                      := $storedConcept/ancestor::decor
    let $projectPrefix              := $decor/project/@prefix
    
    let $check                  :=
        if (empty($transactionId)) then
            if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-DATASETS)) then () else (
                error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify datasets in project ', $projectPrefix, '. You have to be an active author in the project.'))
            )
        else (
            if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-SCENARIOS)) then () else (
                error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify scenarios in project ', $projectPrefix, '. You have to be an active author in the project.'))
            )
        )
    
    let $datasetConcept                 := 
        if ($storedConcept[ancestor::dataset]) then $storedConcept else
        if ($storedConcept[@id]) then (utillib:getConcept($storedConcept/@id, $storedConcept/@effectiveDate)) else
        if ($storedConcept[@ref]) then (utillib:getConcept($storedConcept/@ref, $storedConcept/@flexibility[. castable as xs:dateTime])) else ()
    let $originalConcept                := if ($datasetConcept) then utillib:getOriginalForConcept($datasetConcept) else ()
    let $originalConceptLists           :=
        for $ref in $originalConcept/valueDomain/conceptList
        return
            utillib:getOriginalConceptList($ref)
    
    let $associationIds         :=
        $storedConcept/(@id | @ref | valueDomain/conceptList/@id | valueDomain/conceptList/@ref | valueDomain/conceptList/concept/@id) |
        $datasetConcept/(@id | @ref | valueDomain/conceptList/@id | valueDomain/conceptList/@ref | valueDomain/conceptList/concept/@id) |
        $originalConcept/(@id | @ref | valueDomain/conceptList/@id | valueDomain/conceptList/@ref | valueDomain/conceptList/concept/@id) |
        $originalConceptLists/(@id | concept/@id)
    
    let $check                  :=
        if ($data[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $data/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    (: There are three types of associations;
    1. DatasetConcepts. Expect concept/@id=@conceptId, optionally concept/@effectiveDate=@conceptFlexibility, @code+@codeSystem+@displayName + optionally @equivalence
    2. Concept value domain conceptlist. Expect concept/@id=@conceptId, No @conceptFlexibility, @ref + optionally @flexibility and/or @strength
    3. Concept value domain conceptlist concept. Expect concept/@id=@conceptId, No @conceptFlexibility, @code+@codeSystem+@displayName + optionally @equivalence
    :)
    let $check                  :=
        for $param in $data/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $elmname    := substring-after($param/@path, '/')
        let $value      := $param/value/*[name() = $elmname]
        return
            switch ($path)
            case '/identifierAssociation' (:fall through :)
            case '/terminologyAssociation' return (
                if (count($value) = 1) then (
                    if ($value[@conceptId = $id]) then (
                        if ($value[@conceptFlexibility = $effectiveDate]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to the concept itself SHALL have a matching effectiveDate ' || $effectiveDate || '. Found  conceptFlexibility "' || $value/@conceptFlexibility || '"'
                        ),
                        if ($param[@valueSet | @flexibility]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a concept SHALL not bind to a value set. Found: "' || string-join(($value/@valueSet, $value/@flexibility), ' - ') || '"'
                        else ()
                    )
                    else
                    if ($value[@conceptId = $originalConceptLists/@id]) then (
                        if ($param[@conceptFlexibility]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a conceptList SHALL not have conceptFlexibility. Found: "' || $value/@conceptFlexibility || '"'
                        else (),
                        if ($param[@code | @codeSystem | @displayName]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a conceptList SHALL not bind to a single code. Found: "' || string-join(($value/@code, $value/@codeSystem, $value/@displayName), ' - ') || '"'
                        else ()
                    )
                    else
                    if ($value[@conceptId = $originalConceptLists/concept/@id]) then (
                        if ($param[@conceptFlexibility]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a conceptList.concept SHALL not have conceptFlexibility. Found: "' || $value/@conceptFlexibility || '"'
                        else (),
                        if ($param[@valueSet | @flexibility]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a conceptList.concept SHALL not bind to a value set. Found: "' || string-join(($value/@valueSet, $value/@flexibility), ' - ') || '"'
                        else ()
                    )
                    else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with conceptId "' || $value/@conceptId || '" SHALL be to the target concept (' || $id || '), or one of its conceptLists, or one of the concepts in such conceptList.'
                    ),
                    if ($value[@strength]) then
                        if ($value[@strength = $demapapi:CODINGSTRENGTH-TYPE/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association strength "' || $value/@strength || '" not supported. Supported are: ', string-join($demapapi:CODINGSTRENGTH-TYPE/enumeration/@value, ' ')
                        )
                    else (),
                    if ($value[@equivalence]) then 
                        if ($value[@equivalence = $demapapi:EQUIVALENCY-TYPE/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association equivalence "' || $value/@equivalence || '" not supported. Supported are: ', string-join($demapapi:EQUIVALENCY-TYPE/enumeration/@value, ' ')
                        )
                    else (),
                    if ($op = 'remove') then () else
                    if ($value[@conceptId]) then
                        if ($value[@valueSet]) then (
                            if ($value[@code | @codeSystem | @displayName | @ref]) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with valueSet SHALL NOT have code or codeSystem or displayName or ref.'
                            else (),
                            let $valueSet                   := 
                                if ($value[@valueSet]) then 
                                    vsapi:getValueSet(map { "parameters": map { "id": $value/@valueSet, "effectiveDate": $value/@flexibility } }) 
                                else ()
                            return
                            if (empty($valueSet)) then 
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with valueSet SHALL point to an existing value set. Could not find: ' || $param/@valueSet
                            else ()
                        )
                        else 
                        if ($value[@code]) then (
                            if ($value[@valueSet | @flexibility | @ref]) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with a code SHALL NOT have valueSet or flexibility or ref.'
                            else 
                            if ($value[@codeSystem][@displayName]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with a code SHALL have codeSystem and displayName.'
                            ),
                            if ($value[matches(@code, '^\S+')]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with a code SHALL NOT contain whitespace. Hint for SNOMED CT expressions: use concept ids only, e.g. 363679005:260686004=312250003,405813007=76752008'
                            ),
                            if ($value[utillib:isOid(@codeSystem)]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association codeSystem SHALL be a valid oid. Found: "' || $value/@codeSystem || '"'
                            )
                        )
                        else
                        if ($value[@ref]) then (
                            if ($value[@valueSet | @flexibility | @code | @codeSystem | @displayName]) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with a ref SHALL NOT have both valueSet or flexibility or code or codeSystem or displayName.'
                            else (),
                            if ($value[utillib:isOid(@ref)]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association ref SHALL be a valid oid. Found: "' || $value/@ref || '"'
                            )
                        )
                        else 
                        if ($transactionId) then (
                            (: this signals override of anything in the dataset :)
                        ) 
                        else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association SHALL have at least valueSet, code or ref'
                        )
                    else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association SHALL have conceptId.'
                    )
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. value SHALL contain a single ' || $elmname || ' element/object. Found: ' || count($value) 
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     let $check                 :=
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )
    
    let $update                 :=
        for $param in $data/parameter
        return
            switch ($param/@path)
            case '/terminologyAssociation' return (
                (: only one per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareTerminologyAssociationForUpdate($param/value/*[name() = $elmname], not(empty($transactionId)))
                let $valueSet := 
                    if ($new[@valueSet]) then 
                        vsapi:getValueSet(map { "parameters": map { "id": $new/@valueSet, "effectiveDate": $new/@flexibility } }) 
                    else ()
                (: should not have multiple connections to the same valueSet (regardless of version), so delete those first in case we're updating strength :)
                (: should not have multiple connections to the same code/codeSystem (regardless of version), so delete those first in case we're updating equivalence :)
                let $delete   := 
                    if ($new[@valueSet]) then 
                        if ($transactionId) then (
                            update delete $storedConcept/terminologyAssociation[@conceptId = $new/@conceptId][empty(@valueSet | @code)],
                            update delete $storedConcept/terminologyAssociation[@conceptId = $new/@conceptId][@valueSet = $new/@valueSet]
                        )
                        else (
                            update delete $decor/terminology/terminologyAssociation[@conceptId = $new/@conceptId][@valueSet = $new/@valueSet]
                        )
                    else
                    if ($new[@code]) then
                        if ($transactionId) then (
                            update delete $storedConcept/terminologyAssociation[@conceptId = $new/@conceptId][empty(@valueSet | @code)],
                            update delete $storedConcept/terminologyAssociation[@conceptId = $new/@conceptId][@code = $new/@code][@codeSystem = $new/@codeSystem]
                        )
                        else (
                            update delete $decor/terminology/terminologyAssociation[@conceptId = $new/@conceptId][@code = $new/@code][@codeSystem = $new/@codeSystem]
                        )
                    else
                    if ($transactionId) then
                        update delete $storedConcept/terminologyAssociation[@conceptId = $new/@conceptId]
                    else ()
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return (
                    let $updateTerminology          :=
                        if ($decor/terminology) then () else (
                            update insert <terminology/> following $decor/ids
                        )
                    let $updateAssociation          :=
                        if ($transactionId) then (
                            update insert $new into $storedConcept
                        )
                        else
                        if ($decor/terminology/terminologyAssociation) then (
                            update insert $new following $decor/terminology/terminologyAssociation[last()]
                        )
                        else 
                        if ($decor/terminology/*) then (
                            update insert $new preceding $decor/terminology/*[1]
                        )
                        else (
                            update insert $new into $decor/terminology
                        )
                    let $updateAssociation          :=
                        if (empty($valueSet)) then () else (
                            demapapi:addValueSetRef($decor, $valueSet/@ident, $valueSet/@url, $valueSet/@id, $valueSet/@name, ($valueSet/@displayName, $valueSet/@name)[1])
                        )
                    
                    return ()
                )
                case ('remove') return ()
                default return ( (: unknown op :) )
            )
            case '/identifierAssociation' return (
                (: only one per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareIdentifierAssociationForUpdate($param/value/*[name() = $elmname], not(empty($transactionId)))
                let $delete   := 
                    if ($new[@ref]) then 
                        if ($transactionId) then (
                            update delete $storedConcept/identifierAssociation[@conceptId = $new/@conceptId][empty(@ref)],
                            update delete $storedConcept/identifierAssociation[@conceptId = $new/@conceptId][@ref = $new/@ref]
                        )
                        else (
                            update delete $decor/ids/identifierAssociation[@conceptId = $new/@conceptId][@ref = $new/@ref]
                        )
                    else
                    if ($transactionId) then
                        update delete $storedConcept/identifierAssociation[@conceptId = $new/@conceptId]
                    else ()
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return (
                    if ($transactionId) then (
                        update insert $new into $storedConcept
                    )
                    else
                    if ($decor/ids/*) then (
                        update insert $new following $decor/ids/*[last()]
                    )
                    else (
                        update insert $new into $decor/ids
                    )
                )
                case ('remove') return ()
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
    
    return
        demapapi:getConceptMapList($id, $effectiveDate, $transactionId, $transactionEffectiveDate, $projectVersion, $projectLanguage, $associationMode)
};

declare %private function demapapi:addValueSetRef($decor as element(), $repoPrefix as xs:string, $repoUrl as xs:string, $valueSetId as xs:string, $valueSetName as xs:string, $valueSetDisplayName as xs:string) as item()* {
    
    let $valueSetRefElm     := <valueSet ref="{$valueSetId}" name="{$valueSetName}" displayName="{$valueSetDisplayName}"/>
    let $buildingBlockElm   := <buildingBlockRepository url="{$repoUrl}" ident="{$repoPrefix}"/>
    
    let $addValueSetRef      :=
        if ($decor//valueSet[@id = $valueSetId] | $decor//valueSet[@ref = $valueSetId]) then () else (
            let $dummy1 := update insert $valueSetRefElm following $decor/terminology/*[last()]
            let $dummy2 := 
                if ($decor/project/buildingBlockRepository[@url=$buildingBlockElm/@url][@ident=$buildingBlockElm/@ident][empty(@format)] |
                    $decor/project/buildingBlockRepository[@url=$buildingBlockElm/@url][@ident=$buildingBlockElm/@ident][@format='decor']) then ('false') else (
                    update insert $buildingBlockElm following $decor/project/(author|reference|restURI|defaultElementNamespace|contact)[last()]
                )
            
            return 
                if ($dummy2='false') then 'ref' else 'ref-and-bbr'
        )
    
    return ()
};