xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html 
:)
(:~ Code system API allows read, create, update on DECOR codeSystems :)
module namespace tcsapi             = "http://art-decor.org/ns/api/terminology/codesystem";


import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace vsapi       = "http://art-decor.org/ns/api/valueset" at "valueset-api.xqm";

import module namespace cs          = "http://art-decor.org/ns/decor/codesystem" at "/db/apps/art/api/api-decor-codesystem.xqm";
import module namespace adterm      = "http://art-decor.org/ns/terminology" at "/db/apps/terminology/api/api-terminology.xqm";

declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";

(:~ Search code systems for string in designation or search for code
    @param $context           - optional parameter denoting the context of the search
    @param $codeSystemId      - optional parameter denoting a list of codesytem ids to search
    @param $language          - optional parameter specifying search language, only relevant for string search ignored with code search
    @param $string            - optional parameter search string
    @param $code              - optional parameter code, either string or code must be present
    @param $ancestor          - optiona parameter denoting list of ancestor concept ids
    @return as JSON
    @since 2020-05-11
:)
declare function tcsapi:getCodeSystem($request as map(*)) {

   let $results := tcsapi:getCodeSystem($request?parameters?context, $request?parameters?codeSystemId, $request?parameters?language, $request?parameters?string, $request?parameters?code, $request?parameters?statusCode, $request?parameters?ancestor)
   return
     if (empty($results)) then (
         roaster:response(404, ())
     )
     else $results
};

(:~ Get information on the code system including version and language (if the system pertains to one language only)

@param $id       - required. Identification of the code system to get info for
@return information on the code system as json
@since 2020-05-11
:)
declare function tcsapi:getCodeSystemInfo($request as map(*)) {
   tcsapi:getCodeSystemVersionInfo($request?parameters?id)
};

(:~Get information on all distinct languages supported by all code systems combined

@return languages as json
@since 2020-06-19
:)
declare function tcsapi:getCodeSystemLanguages($request as map(*)) {
   tcsapi:getCodeSystemLanguages()
};

(:~Get code system concept

@return code system concept as json
@since 2020-05-11
:)
declare function tcsapi:getConcept($request as map(*)) {
   tcsapi:getConcept($request?parameters?codeSystemId, $request?parameters?conceptId, $request?parameters?preferred, $request?parameters?language, $request?parameters?children)
};

(:~Get child concepts of code system concept

@return concepts as json
@since 2020-05-13
:)
declare function tcsapi:getConceptChildren($request as map(*)) {
   tcsapi:getConceptChildren($request?parameters?codeSystemId, $request?parameters?conceptId, $request?parameters?preferred, $request?parameters?start, $request?parameters?children)
};

(:~ Expand a composition (with filters or concepts) of an intensional value set denoted by id / effectiveDate and returns preview count (default) or all expanded concepts including count (expand=true)

@return concepts as json
@since 2021-11-11
:)
declare function tcsapi:getExpandedComposition($request as map(*)) {
    tcsapi:postExpandedComposition($request)
};
declare function tcsapi:postExpandedComposition($request as map(*)) {

    let $debug              := false()

    let $id                 := $request?parameters?id[string-length() gt 0]
    let $effectiveDate      := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $expand             := $request?parameters?expand = true()
    let $project            := $request?parameters?project[string-length() gt 0]
    let $projectLanguage    := $request?parameters?language[string-length() gt 0]
    
    let $check              :=
        if (empty($id) or empty($effectiveDate)) 
        then error($errors:BAD_REQUEST, 'Parameter id and effectiveDate are required.') 
        else ()

    (: never return more that maxResults codes in expansion set :)
    let $maxResults         := xs:integer('100')
    
    (: get and check thevalue set specified :)
    let $valueSet           := tcsapi:getValueSetForExpansion($id, $effectiveDate)
    let $check              :=
        if (empty($valueSet)) 
        then error($errors:BAD_REQUEST, 'Value set with id ' || $id || ' as of ' || $effectiveDate || ' does not exist') 
        else ()
    
    let $composition        := 
        <composition>
        {
            $valueSet/completeCodeSystem | 
            $valueSet/conceptList/*
        }
        </composition>
        
    (: is the value set intensional at all? :)
    let $intensional        := count($composition/completeCodeSystem) + count($composition/include) + count($composition/exclude) gt 0
    
    (: get expansions as a query to execute :)
    let $expansion          := tcsapi:getExpansionQueryForCompose($composition)
    let $queryStringAll     := string-join($expansion/query/@q, '|')
    
    let $resultCount        := count(util:eval($queryStringAll))
    
    let $result             := 
        if ($expand) then (
            for $qq in $expansion/query
            let $qs := concat('subsequence(', $qq/@q, ', 1, ', $maxResults, ')')
            return
                <concepts codeSystem="{$qq/@codeSystem}" collection="{$qq/@collection}">
                {
                    util:eval($qs)
                }
                </concepts>
        ) else ()
    
    let $current            :=
        if ($expand) then if ($resultCount > $maxResults) then $maxResults else $resultCount else $resultCount
        
    return 
        <result current="{$current}" count="{$resultCount}" xmlns:json="http://www.json.org">
        {
            if ($debug) then attribute { 'query' } { $queryStringAll } else (),
            utillib:addJsonArrayToElements((
                <valueSet intensional="{$intensional}">
                {
                    $valueSet/@*,
                    $valueSet/(node() except conceptList)
                }
                </valueSet>,
                if ($debug) then $composition else (),
                <expansion expanded="{$expand}" max="{$maxResults}">
                {
                    if ($expand) then attribute { 'id' } { util:uuid() } else (),
                    if ($expand) then tcsapi:copyExpansionList(subsequence(tcsapi:cropc($result), 1, $maxResults)) else ()
                }
                </expansion>
            ))
        }
        </result>
};

(: private functions for expansion sets
   =================
:)

(: return value set with required ref and optional flexibility (default: DYNAMIC) in raw format :)
declare %private function tcsapi:getValueSetForExpansion($ref as xs:string, $flexibility as xs:string*) {
    let $valueSets      := $setlib:colDecorData//valueSet[@id = $ref] | $setlib:colDecorCache//valueSet[@id = $ref] | $setlib:colDecorExample//valueSet[@id = $ref]
    let $valueset       :=
       if ($flexibility castable as xs:dateTime) then (
           $valueSets[@effectiveDate = $flexibility]
       ) else (
           $valueSets[@effectiveDate = max($valueSets/xs:dateTime(@effectiveDate))][1]
       )
    return
        if (empty($valueset)) then
            error($errors:SERVER_ERROR, concat('Cannot expand. Missing valueSet inclusion ', $ref, ' ', ($flexibility[. castable as xs:dateTime], 'dynamic')[1], ' in central terminology services (CADTS). See your admin for solving this.'))
        else 
            <valueSet>
            {
                $valueset/@*,
                attribute { 'inode' } {
                    concat(
                        'collection(''', util:collection-name($valueset), ''')//terminology/valueSet[@id=''', $ref, ''']',  
                        if (empty($flexibility)) then '' else concat('[@effectiveDate=''', $flexibility, ''']')
                    )
                },
                $valueset/*
            }
            </valueSet>
};

(: resolve include/@ref value sets: all refs are replaced by their corresponding concepts, recursively :)
declare %private function tcsapi:resolveIncludeRefs ($cp as element()*, $inode as xs:string) {
    for $c in $cp/*
        return
        if ($c[name() = 'include'][@ref]) then (
            let $ref := $c/@ref
            let $flx := $c/@flexibility
            let $xvs := tcsapi:getValueSetForExpansion($ref, $flx)
            return 
                if (empty($xvs)) then <error type="valueset-not-found" ref="{$ref}" flexibility="{$flx}"/>
                else
                    tcsapi:resolveIncludeRefs(tcsapi:getValueSetForExpansion($ref, $flx)/conceptList, $xvs/@inode)
        )
        else
            element { name($c) } {
                $c/@*,
                if (string-length($inode) > 0 ) then attribute { 'inode' } { $inode } else (),
                $c/*
            }
};

declare %private function tcsapi:copyExpansionList($nodes as element()*) {
    for $node in $nodes
    let $elname := name($node)
    return
        <items>
        {
             attribute { 'is' } { $elname },
             $node/(@* except @is),
             $node/node()
        }
        </items>
};

(: return the collection of queries to be executed for counting or exapnsion, wrapped by code system :)
declare %private function tcsapi:getExpansionQueryForCompose($compose as element()*)  {
    (: 
        first, create a new composition with all include/@ref replaced by their corresponding concepts, recursively
    :)
    let $newc := 
        for $x in $compose
        return
            <composition>
            {
                $x/@*,
                tcsapi:resolveIncludeRefs ($x, '')
            }
            </composition>
    
    let $errors := $newc//error
    
    let $queries :=
        for $includes in $newc/(completeCodeSystem|include|concept)
        (: get code system OID :)
        let $codeSystemOid := $includes/@codeSystem
        group by $codeSystemOid
        return
            for $include in $includes
            (:  for CADTS codesystems: get the name of the collection where the code system data reside
                for coded concepts from value sets use the inode parameter, ie direct path to value set
            :)
            let $collectionPath := util:collection-name(collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid = $codeSystemOid])
            
            let $check          := 
                if (empty($collectionPath)) then
                    error($errors:SERVER_ERROR, 'Cannot expand. Missing codeSystem ' || $codeSystemOid || ' in central terminology services (CADTS). If this is an inline definition that has no formal code system yet, please create that. If this an externally defined code system, you might be able to get this installed through your server administrator depending on license and availability.')
                else ()
            
            (: if it is a codeSystem only (all of the codes) add that to the predicate :)
            let $codeSystemPredicate := 
                if (count(collection($collectionPath)/browsableCodeSystem) gt 1 and not($include/@op or $include/@code)) then
                    concat('browsableCodeSystem[@oid=''',$codeSystemOid,''']')
                else ()
            let $ancestorPredicate  :=
                if (count(collection($collectionPath)/browsableCodeSystem) gt 1 and $include/@op) then
                    concat('[ancestor::browsableCodeSystem/@oid=''',$codeSystemOid,''']')
                else()
            (: matching excludes i.e. same codesystem :)                       
            let $exclPredicates     := tcsapi:getPredicatesForExclude($compose/exclude[@codeSystem=$codeSystemOid])
            return
                <query codeSystem="{$codeSystemOid}" collection="{$collectionPath}">
                {
                    (: concepts :)
                    if ($include[name()='concept']) then
                        let $concepts   := $include/@code
                        let $conceptSelect := if ($concepts) then concat('[@code=(''',string-join($concepts,''','''),''')]') else()
                        return
                            attribute { 'q' } {
                                if ($include/@inode) then 
                                    $include/@inode || '//concept[@codeSystem=''' || $codeSystemOid || ''']' || $conceptSelect
                                else (
                                    'collection(''' || $collectionPath || ''')/' || $codeSystemPredicate || '/concept' || $conceptSelect || string-join($exclPredicates,'') || $ancestorPredicate
                                )
                            }
                    else (
                        (: filters :)
                        let $predicates := tcsapi:getPredicatesForInclude($include)
                        return
                            attribute { 'q' } { concat('collection(''',$collectionPath,''')/',$codeSystemPredicate,'/concept',string-join($predicates,''),
                            string-join($exclPredicates,''),$ancestorPredicate) }
                    )
                }
                </query>
    return
        <expansion>
        {
            $queries,
            $errors
        }
        </expansion>
};

(: function to get all predicates for include ops :)
declare %private function tcsapi:getPredicatesForInclude ($include as element()*) as xs:string*{
    for $filter in $include
    return
        if ($filter/@op='equal') then
            concat('[@code=(''',string-join($filter/@code,''','''),''')]')
        else
        if ($filter/@op='is-a') then
            concat('[ancSlf=''',$filter/@code,''']')
        else
        if ($filter/@op='descendent-of') then
            concat('[ancestor=''',$filter/@code,''']')
        else
        if (name($filter) = 'completeCodeSystem') then
            ()
        else ('[include-false]')
};

(: get all predicates for exclude ops :)
declare %private function tcsapi:getPredicatesForExclude ($exclude as element()*) as xs:string*{
    for $filter in $exclude
    return
        if ($filter/@op='equal') then
            concat('[not(@code=(''',string-join($filter/@code,''','''),'''))]')
        else if ($filter/@op='is-a') then
            concat('[not(ancSlf=''',$filter/@code,''')]')
        else if ($filter/@op='descendent-of') then
            concat('[not(ancestor=''',$filter/@code,''')]')
        else ('[exclude-false]')};

(: function to expose all expanded concepts found, with the important attributes per concept, and all concept children :)
declare %private function tcsapi:cropc ($c as element()*) {
    for $cc in $c
    let $codeSystem := $cc/@codeSystem
    return
        for $ccc in $cc/concept
        let $dp   := ($ccc/designation[@use = 'fsn'], $ccc/designation[@use = 'pref'], $ccc/@displayName)[1]
        return
        <concept>
        {
            $ccc/@code,
            attribute { 'codeSystem' } { $codeSystem },
            attribute { 'displayName' } { data($dp) },
            $ccc/(@level|@type)
            (:
            ,
            $ccc/* - what elements exactly?
            :)
        }
        </concept>
};

(: private functions for other tasks
   =================
:)

declare %private function tcsapi:getCodeSystem($context as xs:string*, $codeSystemId as xs:string*, $searchLanguage as xs:string*, $searchString as xs:string*, $searchCode as xs:string*, $statusCode as xs:string*, $ancestors as xs:string*) {

let $searchString       := if (not(empty($searchString))) then util:unescape-uri($searchString,'UTF-8') else ()
let $statusCodes        := tokenize(normalize-space($statusCode),'\s')
let $ancestors          := if (empty($ancestors)) then '' else tokenize(util:unescape-uri($ancestors,'UTF-8'),'\s')
let $refsets            := ''
let $maxResults         := xs:integer('20')

let $cs :=
   if (not(empty($searchString))) then
      adterm:searchDesignation($context,$codeSystemId, $searchLanguage, $searchString, $maxResults, $statusCodes, $ancestors, $refsets, ())
   else 
   if (not(empty($searchCode))) then
      adterm:searchCode($context,$codeSystemId, $searchCode,$maxResults, $statusCodes)
   else ()

return
    $cs
};

declare %private function tcsapi:getCodeSystemVersionInfo($codeSystemId as xs:string*) {

let $codeSystem := if (empty($codeSystemId)) then () else adterm:getCodeSystemInfo($codeSystemId)
let $license      := 
      <license>
      {
           (: for backward compatibility with older packages that have license.text instead of just license :)
           if ($codeSystem/license/text) then $codeSystem/license/text/node() else $codeSystem/license/node()
      }
      </license>
let $cs           := 
   <codeSystem>
   {
      $codeSystem/@*,
      $codeSystem/name,
      $codeSystem/language,
      $codeSystem/logo,
      $codeSystem/description,
      if ($license/node()) then utillib:serializeNode($license) else ()
   }
   </codeSystem>

return
    if (empty($cs)) then <rest:response><http:response status="404"/></rest:response> else $cs
};

declare %private function tcsapi:getConcept($codeSystemId as xs:string?, $code as xs:string?, $preferred as xs:string?, $language as xs:string?, $children as xs:string?) {

let $compose         := ()
let $cs := 
      if (not(empty($codeSystemId)) and not(empty($code))) then
         adterm:getConceptForLanguages($codeSystemId,$code,$preferred,$language,$children,$compose)
      else ()

return
    if (empty($cs)) then <rest:response><http:response status="404"/></rest:response> else $cs
};

declare %private function tcsapi:getConceptChildren($codeSystemId as xs:string?, $code as xs:string?, $preferred as xs:string?, $start as xs:string?, $length as xs:string?) {

let $compose         := ()
let $cs := 
      if (not(empty($codeSystemId)) and not(empty($code))) then
         adterm:getConceptChildrenForLanguage($codeSystemId,$code,$preferred,$start,$length,$compose)
      else ()

return
    if (empty($cs)) then <rest:response><http:response status="404"/></rest:response> else $cs
};

declare %private function tcsapi:getCodeSystemLanguages() {

let $languages   := distinct-values(collection($setlib:strCodesystemStableData)//browsableCodeSystem/language)

return
    <languages preferred="">
    {
        for $lang in $languages
        order by $lang
        return
        <language code="{$lang}">
            <preferred json:literal="true">false</preferred>
            <display json:literal="true">true</display>
            <search json:literal="true">false</search>
        </language>
    }
    </languages>
};