xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html 
:)
(:~ Terminology Report API creates Terminology Reports for a Project (CADTS-aware) :)
module namespace trepapi            = "http://art-decor.org/ns/api/terminology/report";


import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";

declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";

(:~ Get a terminology report for a given project
    @param $projectPrefix     - mandatory (by now). limits scope to this project only
    @param $language          - optional parameter specifying report language, defaults to en-US
    @return as JSON
    @since 2022-04-13
:)
declare function trepapi:getTerminologyReport($request as map(*)) {

    let $authmap                := $request?user
    let $projectPrefix          := $request?parameters?project[string-length() gt 0]

    let $check                  :=
        if (empty($projectPrefix)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
        
    let $decor                  := 
        if (utillib:isOid($projectPrefix)) then
            utillib:getDecorById($projectPrefix)
        else (
            utillib:getDecorByPrefix($projectPrefix)
        )
    
    let $projectPrefix          := ($decor/project/@prefix)[1]
       
    return
        if (empty($decor)) then (
            roaster:response(404, "application/json", ())
        )
        else
        if (count($decor) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple projects for project '", $projectPrefix, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            trepapi:doReport($decor)
        )
};

(:
    private functions
    =================
:)

declare %private function trepapi:doReport($decor as element()) as element() {
    (: do the terminology report :)
    let $projectId              := ($decor/project/@id)[1]
    let $projectPrefix          := ($decor/project/@prefix)[1]
    return
        <terminologyreport project="{$projectId}" prefix="{$projectPrefix}"/>
};