xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
:)
(:~ Value set API allows read, create, update of DECOR valueSets :)
module namespace vsapi              = "http://art-decor.org/ns/api/valueset";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace ggapi       = "http://art-decor.org/ns/api/governancegroups" at "governancegroups-api.xqm";
import module namespace serverapi   = "http://art-decor.org/ns/api/server" at "server-api.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";

import module namespace vs          = "http://art-decor.org/ns/decor/valueset" at "../../art/api/api-decor-valueset.xqm";
import module namespace cs          = "http://art-decor.org/ns/decor/codesystem" at "../../art/api/api-decor-codesystem.xqm";
import module namespace templ       = "http://art-decor.org/ns/decor/template" at "../../art/api/api-decor-template.xqm";

declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";

(:~ All functions support their own override, but this is the fallback for the maximum number of results returned on a search :)
declare %private variable $vsapi:maxResults                 := 50;
declare %private variable $vsapi:STATUSCODES-FINAL          := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $vsapi:ADDRLINE-TYPE              := utillib:getDecorTypes()/AddressLineType;
declare %private variable $vsapi:VOCAB-TYPE                 := utillib:getDecorTypes()/VocabType;
declare %private variable $vsapi:DESIGNATION-TYPE           := utillib:getDecorTypes()/DesignationType;
declare %private variable $vsapi:INTENSIONALOPERATORS-TYPE  := utillib:getDecorTypes()/IntensionalOperators;

(:~ Retrieves latest DECOR valueSet based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function vsapi:getLatestValueSet($request as map(*)) {
    vsapi:getValueSet($request)
};

(:~ Retrieves DECOR valueSet based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the valueSet. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function vsapi:getValueSet($request as map(*)) {
    
    let $projectPrefix                  := $request?parameters?prefix
    let $projectVersion                 := $request?parameters?release
    let $projectLanguage                := $request?parameters?language
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $withversions                   := $request?parameters?versions = true()
    
    let $results                        := vsapi:getValueSet($projectPrefix, $projectVersion, $projectLanguage, $id, $effectiveDate, $withversions, true())
    let $results                        := 
        if ($withversions) then 
            <list artifact="VS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">{$results}</list> 
        else (
            head($results)
        )
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple valueSets for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Retrieves DECOR valueSet history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function vsapi:getValueSetHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-VALUESET, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-VALUESET}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Retrieves DECOR valueSet usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the valueSet. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function vsapi:getValueSetUsage($request as map(*)) {
    
    let $projectPrefix                  := $request?parameters?prefix
    let $projectVersion                 := $request?parameters?release
    let $projectLanguage                := $request?parameters?language
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $valueSetsAll                   :=
        if (empty($id)) then () else (
            if (empty($projectPrefix)) then
                vs:getValueSetById($id, (), false())
            else (
                vs:getValueSetById($id, (), $projectPrefix, (), false())
            )
        )
    let $mostRecent                     := string(max($valueSetsAll/descendant-or-self::valueSet/xs:dateTime(@effectiveDate)))
    let $vs                             := 
        if (empty($effectiveDate)) then 
            $valueSetsAll/descendant-or-self::valueSet[@effectiveDate = $mostRecent] 
        else (
            $valueSetsAll/descendant-or-self::valueSet[@effectiveDate = $effectiveDate]
        )
    let $vs                             := $vs[1]
    
    (:let $check                          :=
        if (count($vs) le 1) then $vs else (
            error(xs:QName('vsapi:getValueSetUsage'), 'ValueSet ' || $id || ' effectiveDate ''' || $effectiveDate || ''' yields ' || count($vs) || ' results. Projects involved: ' || string-join(distinct-values($vs/ancestor-or-self::*/@ident), ', '))
        ):)
    
    let $isMostRecent                   := $mostRecent = $effectiveDate or empty($effectiveDate)
    
    let $allAssociations                := $setlib:colDecorData//terminologyAssociation[@valueSet = $id]
    let $allAssociations                :=
        if ($isMostRecent) then 
            $allAssociations[@flexibility = $effectiveDate] | $allAssociations[not(@flexibility castable as xs:dateTime)]
        else (
            $allAssociations[@flexibility = $effectiveDate]
        )
    let $allAssociations                :=
        if (empty($projectPrefix)) then $allAssociations else (
            for $ta in $allAssociations
            return
                if ($ta/ancestor::decor/project/@prefix = $projectPrefix) then $ta else ()
        )
        
    let $valueSetsAll                   := $setlib:colDecorData//terminology/valueSet//*[@ref = $id]
    let $valueSetsAll                   :=
        if ($isMostRecent) then 
            $valueSetsAll[@flexibility = $effectiveDate] | $valueSetsAll[not(@flexibility castable as xs:dateTime)]
        else (
            $valueSetsAll[@flexibility = $effectiveDate]
        )
    
    let $allTemplAssociations           := $setlib:colDecorData//template//vocabulary[@valueSet = $id]
    let $allTemplAssociations           :=
        if ($isMostRecent) then 
            $allTemplAssociations[@flexibility = $effectiveDate] | $allTemplAssociations[not(@flexibility castable as xs:dateTime)]
        else (
            $allTemplAssociations[@flexibility = $effectiveDate]
        )
    
    let $url                            := serverapi:getServerURLServices()
    
    let $results                        := (
        for $item in $allAssociations
        let $clpfx  := $item/ancestor::decor/project/@prefix
        let $clid   := $item/@conceptId
        group by $clpfx, $clid
        return (
            let $originalConcept            := utillib:getConceptList($item[1]/@conceptId, ())/ancestor::concept[1]
            return
                if ($originalConcept) then utillib:doConceptAssociation($vs[1], $originalConcept, $originalConcept/name) else ()
        )
        ,
        for $item in $valueSetsAll
        let $vsid := $item/ancestor::valueSet[1]/@id
        let $vsed := $item/ancestor::valueSet[1]/@effectiveDate
        group by $vsid, $vsed
        return
            utillib:doValueSetAssociation($vs, $item[1])
        ,
        for $item in $allTemplAssociations
        let $tmid := $item/ancestor::template[1]/@id
        let $tmed := $item/ancestor::template[1]/@effectiveDate
        group by $tmid, $tmed
        return
            utillib:doTemplateAssociation($vs, $item[1])
    )
    
    let $count                          := count($results)
    let $max                            := $count
    let $allcnt                         := $count
    
    return
        <list artifact="USAGE" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements(subsequence($results, 1, $max))
        }
        </list>
};

(:~ Returns a list of zero or more valuesets

@param $governanceGroupId - optional. determines search scope. null is full server, id limits scope to this projects under this governance group only. Does not mix with $projectPrefix
@param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only. Does not mix with $governanceGroupId
@param $projectVersion   - optional. if empty defaults to current version. if valued then the valueset will come explicitly from that archived project version which is expected to be a compiled version
@param $projectLanguage  - optional. relevant in combination with $projectVersion if there are more than one languages the project is compiled in
@param $id               - optional. Identifier of the valueset to retrieve
@param $name             - optional. Name of the valueset to retrieve (valueSet/@name)
@param $flexibility      - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
@param $treetype         - optional. Default $vs:TREETYPELIMITEDMARKED
@param $doV2             - optional. Boolean. Default false. 
@param $withversions     - optional. Boolean. Default true. If true and $doV2 is false, returns valueSetList.valueSet* where every valueSet has @id|@ref and a descending by 
effectiveDate list of matching valueSets. If false and $doV2 is false, returns valueSetList.valueSet* containing only the latest version of the valueSet (or ref if no versions 
exist in $projectPrefix)<br/>If true and $doV2 is true, returns valueSetList.valueSet* where every valueSet has @id|@ref and a descending by effectiveDate list of matching 
valueSets. If false and $doV2 is false, returns valueSetList.valueSet* containing only the latest version of the valueSet (or ref if no versions exist in $projectPrefix)
@return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
@since 2013-06-14
:)
declare function vsapi:getValueSetList($request as map(*)) {
    let $governanceGroupId      := $request?parameters?governanceGroupId
    let $projectPrefix          := $request?parameters?prefix
    let $projectVersion         := $request?parameters?release
    let $projectLanguage        := $request?parameters?language
    let $max                    := $request?parameters?max
    let $resolve                :=  if (empty($governanceGroupId)) then not($request?parameters?resolve = false()) else $request?parameters?resolve = true()
    let $searchTerms            := 
        array:flatten(
            for $s in ($request?parameters?search, $request?parameters?id, $request?parameters?name)[string-length() gt 0]
            return
                tokenize(normalize-space(lower-case($s)),'\s')
        )
    let $ed                     := $request?parameters?effectiveDate
    let $includebbr             := $request?parameters?includebbr = true()
    let $sort                   := $request?parameters?sort
    let $sortorder              := $request?parameters?sortorder
    
    let $check                  :=
        if (empty($governanceGroupId) and empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter governanceGroupId or prefix')
        else 
        if (not(empty($governanceGroupId)) and not(empty($projectPrefix))) then 
            error($errors:BAD_REQUEST, 'Request SHALL NOT have both parameter governanceGroupId or prefix, not both')
        else ()
    let $check                  :=
        if (empty($governanceGroupId)) then () else if ($resolve) then
            error($errors:BAD_REQUEST, 'Request SHALL NOT have governance group scope and resolve=true. This is too expensive to support')
        else ()
    
    let $startT                 := util:system-time()
    
    let $result                 :=
        if (empty($governanceGroupId)) then
            vsapi:getValueSetList($governanceGroupId, $projectPrefix, $projectVersion, $projectLanguage, $searchTerms, $ed, $includebbr, $sort, $sortorder, $max, $resolve)
        else (
            for $projectId in ggapi:getLinkedProjects($governanceGroupId)/@ref
            return
                vsapi:getValueSetList($governanceGroupId, $projectId, (), (), $searchTerms, $ed, $includebbr, $sort, $sortorder, $max, $resolve)
                
        )
    
    let $durationT              := (util:system-time() - $startT) div xs:dayTimeDuration("PT0.001S")
    
    return
        <list artifact="{$result[1]/@artifact}" elapsed="{$durationT}" current="{sum($result/xs:integer(@current))}" total="{sum($result/xs:integer(@total))}" all="{sum($result/xs:integer(@allcnt))}" resolve="{$resolve}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            if (empty($governanceGroupId)) then attribute project {$projectPrefix} else attribute governanceGroupId {$governanceGroupId},
            utillib:addJsonArrayToElements($result/*)
        }
        </list>
};

(:~ Deletes a valueSet reference based on $id (oid) and project id or prefix
    @param $id                      - required parameter denoting the id of the valueSet
    @param $project                 - required. limits scope to this project only
:)
declare function vsapi:deleteValueSet($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0][not(. = '*')]
    let $id                     := $request?parameters?id[string-length() gt 0]
    
    let $check                  :=
        if (empty($project) or empty($id)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have both parameter project and id')
        else ()
    
    let $decor                  := 
        if (utillib:isOid($project)) then
            utillib:getDecorById($project)
        else (
            utillib:getDecorByPrefix($project)
        )
    
    let $check                  :=
        if (empty($decor)) then
            error($errors:BAD_REQUEST, 'Project ' || $project || ' not found.')
        else ()
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify valueSets in project ', $project[1], '. You have to be an active author in the project.'))
        )
    
    let $delete                 := update delete $decor/terminology/valueSet[@ref = $id]
    
    return
        roaster:response(204, ())
};

(: Create a valueset, either empty or based on another valueSet

@param $projectPrefix project to create this scenario in
@param $targetDate If true invokes effectiveDate of the new valueSet and concepts as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
@param $sourceId parameter denoting the id of a dataset to use as a basis for creating the new valueSet
@param $sourceEffectiveDate parameter denoting the effectiveDate of a dataset to use as a basis for creating the new valueSet",
@param $keepIds Only relevant if source dataset is specified. If true, the new valueSet will keep the same ids for the new valueSet, and only update the effectiveDate
@param $baseDatasetId Only relevant when a source dataset is specified and `keepIds` is false. This overrides the default base id for datasets in the project. The value SHALL match one of the projects base ids for datasets
@return (empty) dataset object as xml with json:array set on elements
:)
declare function vsapi:postValueSet($request as map(*)) {

    let $authmap                := $request?user
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $targetDate             := $request?parameters?targetDate = true()
    let $sourceId               := $request?parameters?sourceId
    let $data                   := utillib:getBodyAsXml($request?body, 'valueSet', ())
    let $sourceEffectiveDate    := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?sourceEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?sourceEffectiveDate[string-length() gt 0]
        }
    let $refOnly                := $request?parameters?refOnly = true()
    let $keepIds                := $request?parameters?keepIds = true()
    let $baseId                 := $request?parameters?baseId
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($projectPrefix)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter prefix')
        else ()
    
    let $results                := vsapi:createValueSet($authmap, $projectPrefix, $targetDate, $sourceId, $sourceEffectiveDate, $refOnly, $keepIds, $baseId, $data)
    
    return
        roaster:response(201, 
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        
        )
};

(:~ Update valueSet statusCode, versionLabel, expirationDate and/or officialReleaseDate

@param $authmap required. Map derived from token
@param $id required. DECOR concept/@id to update
@param $effectiveDate required. DECOR concept/@effectiveDate to update
@param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates. Not implemented (yet)
@param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
@param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
@param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
@param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
@param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
@return list object with success and/or error elements or error
:)
declare function vsapi:postValueSetStatus($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    (: Note: we _could_ implement traversing into value set inclusions or even attached codesystems through this parameter. In AD2 however we never implemented this and we did not plan for introducing this in AD3 nor was it ever asked of us.
             Hence the provision is made should it ever be needed, but other than that, it is not implemented.
    :)
    let $recurse                        := $request?parameters?recurse = true()
    let $list                           := $request?parameters?list = true()
    let $statusCode                     := $request?parameters?statusCode
    let $versionLabel                   := $request?parameters?versionLabel
    let $expirationDate                 := $request?parameters?expirationDate
    let $officialReleaseDate            := $request?parameters?officialReleaseDate
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    return
        vsapi:setValueSetStatus($authmap, $id, $effectiveDate, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
};

(:~ Update DECOR valueSet

@param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
@param $id                       - required. the id for the valueSet to update 
@param $request-body             - required. json body containing new valueSet structure
@return valueSet structure including generated meta data
@since 2020-05-03
:)
declare function vsapi:putValueSet($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                           := utillib:getBodyAsXml($request?body, 'valueSet', ())
    let $deletelock                     := $request?parameters?deletelock = true()
    
    (:let $s                              := xmldb:store('/db/apps/decor/tmp', 'ttt.xml', $data):) 
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $return                         := vsapi:putValueSet($authmap, $id, $effectiveDate, $data, $deletelock)
    return (
        roaster:response(200, $return)
    )

};

(:~ Update DECOR valueSet parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "e.g. [/statusCode|/expirationDate|/officialReleaseDate|/canonicalUri|/versionLabel|/name|/displayName|/experimental|/desc|/publishingAuthority|/copyright|/completeCodeSystem|/conceptList]", "value": "[string|object]" }

where

* op - add & replace (statusCode, expirationDate, officialReleaseDate, canonicalUri, versionLabel, name, displayName, experimental, desc, publishingAuthority, copyright, completeCodeSystem, conceptList) or remove (desc, publishingAuthority, copyright, completeCodeSystem, conceptList)

* path - see above

* value - string when the path is not /. object when path is /
@param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
@param $id                       - required. the id for the valueSet to update 
@param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
@return valueSet structure
@since 2020-05-03
@see http://tools.ietf.org/html/rfc6902
:)
declare function vsapi:patchValueSet($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    (:let $s                      := xmldb:store('/db/apps/decor/tmp', 'ttt.xml', $data) :)
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                 := vsapi:patchValueSet($authmap, string($id), string($effectiveDate), $data)
    return (
        roaster:response(200, $return)
    )

};

(:~ Central logic for patching an existing valueSet statusCode, versionLabel, expirationDate and/or officialReleaseDate

@param $authmap required. Map derived from token
@param $id required. DECOR concept/@id to update
@param $effectiveDate required. DECOR concept/@effectiveDate to update
@param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
@param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
@param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
@param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
@param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
@param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
@return list object with success and/or error elements or error
:)
declare function vsapi:setValueSetStatus($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $recurse as xs:boolean, $listOnly as xs:boolean, $newStatusCode as xs:string?, $newVersionLabel as xs:string?, $newExpirationDate as xs:string?, $newOfficialReleaseDate as xs:string?) as element(list) {
    (:get object for reference:)
    let $object                 := 
        if (empty($effectiveDate)) then (
            let $ttt    := $setlib:colDecorData//valueSet[@id = $id]
            return
                $ttt[@effectiveDate = max($ttt/xs:dateTime(@effectiveDate))]
        )
        else (
            $setlib:colDecorData//valueSet[@id = $id][@effectiveDate = $effectiveDate]
        )
    let $decor                  := $object/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($object) = 1) then () else
        if ($object) then
            error($errors:SERVER_ERROR, 'ValueSet id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($object) || ' were found in: ' || string-join(distinct-values($object/ancestor::decor/project/@prefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'ValueSet id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify valueSets in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
        
    let $testUpdate             :=
        utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, true(), $projectPrefix)
    let $results                :=
        if ($testUpdate[self::error]) then $testUpdate else
        if ($listOnly) then $testUpdate else (
            utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, false(), $projectPrefix)
        )
    return
    if ($results[self::error] and not($listOnly)) then
        error($errors:BAD_REQUEST, string-join(
            for $e in $results[self::error]
            return
                $e/@itemname || ' id ''' || $e/@id || ''' effectiveDate ''' || $e/@effectiveDate || ''' cannot be updated: ' || data($e)
            , ' ')
        )
    else (
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            (: max 2 arrays: one with error and one with success :)
            utillib:addJsonArrayToElements($results)
        }
        </list>
    )
};

(:~ Central logic for updating an existing valueSet

@param $authmap         - required. Map derived from token
@param $id              - required. DECOR valueSet/@id to update
@param $request-body    - required. DECOR valueSet xml element containing everything that should be in the updated valueSet
@return valueSet object as xml with json:array set on elements
:)
declare function vsapi:putValueSet($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(), $deletelock as xs:boolean) as element(valueSet) {

    let $storedValueSet         := $setlib:colDecorData//valueSet[@id = $id][@effectiveDate = $effectiveDate] 
    let $decor                  := $storedValueSet/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $lock                   := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())
    
    let $check                  :=
        if ($storedValueSet) then () else (
            error($errors:BAD_REQUEST, 'ValueSet with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ' does not exist')
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify valueSets in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this valueSet (anymore). Get a lock first.'))
        )    
    let $check                  :=
        if ($storedValueSet[@statusCode = $vsapi:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, concat('ValueSet cannot be updated while it has one of status: ', string-join($vsapi:STATUSCODES-FINAL, ', '), if ($storedValueSet/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
        else ()
    let $check                  :=
        if ($data[@id = $id] | $data[empty(@id)]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no valueSet id or the same valueSet id as the valueSet id ''', $id, ''' used for updating. Found in request body: ', ($data/@id, 'null')[1]))
        )
    let $check                  :=
        if ($data[@effectiveDate = $effectiveDate] | $data[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no valueSet effectiveDate or the same valueSet effectiveDate as the valueSet effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($data/@effectiveDate, 'null')[1]))
        )
    
    let $baseValueset           := 
        <valueSet>
        {
            attribute id {$id} ,
            $data/@name[string-length()>0] ,
            $data/@displayName[string-length()>0] ,
            attribute effectiveDate {$effectiveDate} ,
            attribute statusCode {"draft"} ,
            $data/@versionLabel[string-length()>0] ,
            $data/@expirationDate[. castable as xs:dateTime] ,
            $data/@officialReleaseDate[. castable as xs:dateTime] ,
            $data/@experimental[. = ('true', 'false')] ,
            $data/@canonicalUri[string-length()>0],
            $data/@lastModifiedDate[string-length()>0]
        }
        </valueSet>
    let $newValueSet            := utillib:prepareValueSetForUpdate($data, $storedValueSet)
    
    (: save history:)
    let $intention                      := if ($storedValueSet[@statusCode = 'final']) then 'patch' else 'version'
    let $history                        := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-VALUESET, $projectPrefix, $intention, $storedValueSet)
    
    (: now update the value set :)
    let $valueSetUpdate         := update replace $storedValueSet with $newValueSet
    let $delete                 := update delete $storedValueSet//@json:array
    let $deleteLock             := if ($deletelock) then update delete $lock else ()
    
    let $result                 := vsapi:getValueSet($projectPrefix, (), (), $id, $effectiveDate, false(), true())
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Central logic for patching an existing valueSet

@param $authmap         - required. Map derived from token
@param $id              - required. DECOR valueSet/@id to update
@param $effectiveDate   - required. DECOR valueSet/@effectiveDate to update
@param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
@return dataset object as xml with json:array set on elements
:)
declare function vsapi:patchValueSet($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) as element(valueSet) {

    let $storedValueSet         := $setlib:colDecorData//valueSet[@id = $id][@effectiveDate = $effectiveDate]
    let $decor                  := $storedValueSet/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $lock                   := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())
    
    let $check                  :=
        if ($storedValueSet) then () else (
            error($errors:BAD_REQUEST, 'ValueSet with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify valueSets in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if ($storedValueSet[@statusCode = $vsapi:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $vsapi:STATUSCODES-FINAL)]) then () else
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('ValueSet cannot be patched while it has one of status: ', string-join($vsapi:STATUSCODES-FINAL, ', '), if ($storedValueSet/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
    let $check                  :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this valueSet (anymore). Get a lock first.'))
        )
    let $check                  :=
        if ($data[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $data/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $check                  :=
        for $param in $data/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($storedValueSet, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($storedValueSet/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri'
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/name' 
            case '/displayName' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/experimental' return (
                if ($op = 'remove') then () else 
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false.'
                )
            )
            case '/desc' 
            case '/copyright' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then
                    if ($param/value/*[name() = $pathpart][matches(@language, '[a-z]{2}-[A-Z]{2}')]) then
                        if ($param[@op = 'remove'] | $param/value/*[name() = $pathpart][.//text()[not(normalize-space() = '')]]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have contents.'
                        )
                    else (
                        'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have a language with pattern ll-CC.'
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/purpose' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then
                    if ($param/value/*[name() = $pathpart][matches(@language, '[a-z]{2}-[A-Z]{2}')]) then
                        if ($param[@op = 'remove'] | $param/value/*[name() = $pathpart][.//text()[not(normalize-space() = '')]]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have contents.'
                        )
                    else (
                        'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have a language with pattern ll-CC.'
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/publishingAuthority' return (
                if ($param[count(value/publishingAuthority) = 1]) then (
                    if ($param/value/publishingAuthority[@id]) then
                        if ($param/value/publishingAuthority[utillib:isOid(@id)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || ''' publishingAuthority.id SHALL be an OID is present. Found ' || string-join($param/value/publishingAuthority/@id, ', ')
                        )
                    else (),
                    if ($param/value/publishingAuthority[@name[not(. = '')]]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || ''' publishingAuthority.name SHALL be a non empty string.'
                    ),
                    let $unsupportedtypes := $param/value/publishingAuthority/addrLine/@type[not(. = $vsapi:ADDRLINE-TYPE/enumeration/@value)]
                    return
                        if ($unsupportedtypes) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Publishing authority has unsupported addrLine.type value(s) ' || string-join($unsupportedops, ', ') || '. SHALL be one of: ' || string-join($vsapi:ADDRLINE-TYPE/enumeration/@value, ', ')
                        else ()
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single publishingAuthority under value. Found ' || count($param/value/publishingAuthority)
                )
            )
            case '/completeCodeSystem' return (
                if ($param[count(value/completeCodeSystem) = 1]) then (
                    if ($param/value/completeCodeSystem[utillib:isOid(@codeSystem)]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. CompleteCodeSystem has unsupported value for .codeSystem ' || string-join($param/value/completeCodeSystem/@codeSystem, ', ') || '. SHALL a valid oid.'
                    ),
                    if ($param/value/completeCodeSystem/@type[not(. = 'D')]) then
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. CompleteCodeSystem has unsupported value for .type ' || string-join($param/value/completeCodeSystem/@type, ', ') || '. SHALL ''D'' if present.'
                    else ()
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single completeCodeSystem under value. Found ' || count($param/value/completeCodeSystem)
                )
            )
            case '/items' return (
                if ($op = 'remove') then () else
                if ($param[count(value/items) ge 1]) then (
                    for $concept in (   $param/value/items[@is='completeCodeSystem'])
                    let $is     := ($concept/@is, name($concept))[1]
                    return (
                        if ($concept/@type[not(. = 'D')]) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .type value ''' || $concept/@type || '''. SHALL be D'
                        else (),
                        if ($concept/@codeSystem[utillib:isOid(.)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .codeSystem value ''' || $concept/@codeSystem || '''. SHALL be a valid oid'
                        ),
                        if ($concept[empty(@flexibility)] | $concept[@flexibility = 'dynamic'] | $concept[@flexibility castable as xs:dateTime]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .codeSystem value ''' || $concept/@codeSystem || '''. SHALL be a valid dateTime or ''dynamic'''
                        )
                    ),
                    for $concept in (   $param/value/items[@is='concept'] | 
                                        $param/value/items[@is='exception'])
                    let $is     := ($concept/@is, name($concept))[1]
                    return (
                        if ($concept/@type[. = $vsapi:VOCAB-TYPE/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .type value ''' || $concept/@type || '''. SHALL be one of: ' || string-join($vsapi:VOCAB-TYPE/enumeration/@value, ', ')
                        ),
                        if ($concept/@level[. castable as xs:nonNegativeInteger]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .level value ''' || $concept/@level || '''. SHALL be 0 or higher'
                        ),
                        if ($concept/@codeSystem[utillib:isOid(.)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .codeSystem value ''' || $concept/@codeSystem || '''. SHALL be a valid oid'
                        ),
                        if ($concept/@code[matches(., '^\S+$')]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .code value ''' || $concept/@code || '''. SHALL be populated and SHALL NOT have whitespace'
                        ),
                        if ($concept/@displayName[not(. = '')]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .displayName value ''' || $concept/@displayName || '''. SHALL be populated'
                        )
                    ),
                    for $concept in (   $param/value/items[@is='include'] | 
                                        $param/value/items[@is='exclude'])
                    let $is     := ($concept/@is, name($concept))[1]
                    return (
                        if ($concept[empty(@op)] | $concept[@op = $vsapi:INTENSIONALOPERATORS-TYPE/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .op value ''' || $concept/@op || '''. SHALL be one of: ' || string-join($vsapi:INTENSIONALOPERATORS-TYPE/enumeration/@value, ', ')
                        ),
                        if ($concept[not((@ref | @flexibility | @exception[not(. = 'false')]) and (@op | @code | @codeSystem))]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported combination of attributes. SHALL have (.ref, .flexibility, .exception) or (.op, .code, .codeSystem)'
                        ),
                        if ($concept[empty(@codeSystem)] | $concept[utillib:isOid(@codeSystem)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .codeSystem value ''' || $concept/@codeSystem || '''. SHALL be a valid oid'
                        ),
                        if ($concept[empty(@code)] | $concept[matches(@code, '^\S+$')]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .codeSystem value ''' || $concept/@code || '''. SHALL NOT have whitespace'
                        )
                    )
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have one or more items under value. Found ' || count($param/value/items)
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     let $check                 :=
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )
    
    let $intention              := if ($storedValueSet[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-VALUESET, $projectPrefix, $intention, $storedValueSet)
    
    let $update                 :=
        for $param in $data/parameter
        return
            switch ($param/@path)
            case '/statusCode'
            case '/expirationDate'
            case '/officialReleaseDate'
            case '/canonicalUri'
            case '/versionLabel'
            case '/name'
            case '/displayName'
            case '/experimental' return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := attribute {$attname} {$param/@value}
                let $stored   := $storedValueSet/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedValueSet
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/desc' return (
                (: only one per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/desc)
                let $stored   := $storedValueSet/*[name() = $elmname][@language = $param/value/desc/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedValueSet
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/purpose' return (
                (: only one possible per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/purpose)
                let $stored   := $storedValueSet/*[name() = $elmname][@language = $param/value/purpose/@language]
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $storedValueSet
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/publishingAuthority' return (
                (: only one possible :)
                let $new      := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
                let $stored   := $storedValueSet/publishingAuthority
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedValueSet
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/copyright' return (
                (: only one per language :)
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/copyright)
                let $stored   := $storedValueSet/copyright[@language = $param/value/copyright/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedValueSet
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/completeCodeSystem' return (
                let $new      := 
                    <completeCodeSystem>
                    {
                        $param/value/completeCodeSystem/@codeSystem[not(. = '')],
                        $param/value/completeCodeSystem/@codeSystemName[not(. = '')],
                        $param/value/completeCodeSystem/@codeSystemVersion[not(. = '')],
                        $param/value/completeCodeSystem/@flexibility[not(. = '')],
                        $param/value/completeCodeSystem/@type[. = 'D']
                    }
                    </completeCodeSystem>
                let $stored   := $storedValueSet/completeCodeSystem[@codeSystem = $new/@codeSystem]
                
                return
                switch ($param/@op)
                case 'add'
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $storedValueSet
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/items' return (
                let $newcs    := 
                    for $completeCodeSystem in $param/value/items[@is = 'completeCodeSystem']
                    return
                        <completeCodeSystem>
                        {
                            $completeCodeSystem/@codeSystem, 
                            $completeCodeSystem/@codeSystemName[not(. = '')], 
                            $completeCodeSystem/@codeSystemVersion[not(. = '')], 
                            $completeCodeSystem/@flexibility[not(. = '')],
                            $completeCodeSystem/@type[. = 'D'] 
                        }
                        </completeCodeSystem>
                let $newcl    := utillib:prepareValueSetConceptListForUpdate($param/value/items[not(@is = 'completeCodeSystem')])
                let $stored   := $storedValueSet/conceptList | $storedValueSet/completeCodeSystem
                
                let $delete   := update delete $stored
                
                return
                    switch ($param/@op)
                    case 'add'
                    case 'replace' return update insert ($newcs | $newcl) into $storedValueSet
                    case 'remove' return ()
                    default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $preparedValueSet       := utillib:prepareValueSetForUpdate($storedValueSet, $storedValueSet)
    
    let $update                 := update replace $storedValueSet with $preparedValueSet
    let $update                 := update delete $lock
    
    let $result                 := vsapi:getValueSet($projectPrefix, (), (), $preparedValueSet/@id, $preparedValueSet/@effectiveDate, false(), true())
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ 
@param $doContents If we only need the valueSet for the list we can forego the contents
:)
declare %private function vsapi:getValueSet($projectPrefix as xs:string*, $projectVersion as xs:string*, $projectLanguage as xs:string?, $id as xs:string, $effectiveDate as xs:string?, $withversions as xs:boolean, $doContents as xs:boolean) as element(valueSet)* {
    let $id                     := $id[not(. = '')]
    let $effectiveDate          := $effectiveDate[not(. = '')]
    let $projectPrefix          := $projectPrefix[not(. = '')]
    let $projectVersion         := $projectVersion[not(. = '')]
    let $serialize              := true()
    
    let $results                :=
        if (empty($projectPrefix)) then (
            (: vs:getValueSetById($id, $flexibility, false()) will not return all versions, only latest. Don't want to break that and cause issues so we do it here :)
            let $valueSets      := $setlib:colDecorData//valueSet[@id = $id] | $setlib:colDecorCache//valueSet[@id = $id]
            let $valueSets      :=
                if (empty($valueSets)) then
                    $setlib:colDecorData//valueSet[@ref = $id] | $setlib:colDecorCache//valueSet[@ref = $id]
                else
                if ($withversions) then $valueSets else
                if ($effectiveDate castable as xs:dateTime) then (
                    $valueSets[@effectiveDate = $effectiveDate]
                ) else (
                    $valueSets[@effectiveDate = max($valueSets/xs:dateTime(@effectiveDate))][1]
                )
            let $projectPrefix      := $valueSets[1]/ancestor::decor/project/@prefix
            let $projectLanguage    := $valueSets[1]/ancestor::decor/project/@defaultLanguage
            let $valueSets      :=
                if ($valueSets) then 
                    <repository>
                    {
                        $valueSets[1]/ancestor::decor/project/(@* except (@url|@ident)),
                        attribute url {$vs:strDecorServicesURL},
                        attribute ident {$projectPrefix},
                        vs:getRawValueSet($valueSets[1], (), $projectLanguage, $projectPrefix, (), false(), $serialize)
                    }
                    </repository>
                else ()
            return
                $valueSets/valueSet
        )
        else (
            vs:getValueSetById($id, $effectiveDate, $projectPrefix[1], $projectVersion[1], $serialize)/(descendant-or-self::valueSet[@id][@effectiveDate], descendant-or-self::valueSet[@ref])[1] 
        )
    let $results                        :=
        if (empty($results) and not(empty($projectPrefix))) then
            if (utillib:isOid($projectPrefix)) then (
                let $decor              := utillib:getDecorById($projectPrefix, $projectVersion, $projectLanguage)
                return
                    ($decor//valueSet[@ref = $id])[last()]
            ) 
            else (
                let $decor              := utillib:getDecorByPrefix($projectPrefix, $projectVersion, $projectLanguage)
                return
                    ($decor//valueSet[@ref = $id])[last()]
            )
        else (
            $results
        )
    
    
    for $vs in $results
    let $id             := $vs/@id | $vs/@ref
    let $ed             := $vs/@effectiveDate
    let $projectPrefix  := ($vs/@ident, $vs/parent::*/@ident, $vs/ancestor::*/@bbrident, $vs/ancestor::decor/project/@prefix)[1]
    order by $id, $ed descending
    return
        element {name($vs)} {
            $vs/@*,
            if ($vs/@url) then () else attribute url {($vs/parent::*/@url, $vs/ancestor::*/@bbrurl, serverapi:getServerURLServices())[1]},
            if ($vs/@ident) then () else attribute ident {$projectPrefix}
            ,
            if ($doContents) then (
                $vs/desc,
                if ($vs/sourceCodeSystem) then $vs/sourceCodeSystem else (
                    vsapi:createSourceCodeSystems($vs, ($vs/ancestor::decor, $projectPrefix)[1], ($projectLanguage, $vs/ancestor::decor/project/@defaultLanguage)[1])
                ),
                if ($vs[@id]) then 
                    if ($vs[publishingAuthority]) then () else (
                        let $decor      := $vs/ancestor::decor
                        let $copyright  := ($decor/project/copyright[empty(@type)] | $decor/project/copyright[@type = 'author'])[1]
                        return
                        if ($copyright) then
                            <publishingAuthority name="{$copyright/@by}" inherited="project">
                            {
                                (: Currently there is no @id, but if it ever would be there ...:)
                                $copyright/@id,
                                $copyright/addrLine
                            }
                            </publishingAuthority>
                        else ()
                    )
                else ()
                ,
                $vs/(* except (desc | sourceCodeSystem | conceptList | completeCodeSystem))
                ,
                (: we return just a count of issues and leave it up to the calling party to get those if required :)
                <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $vs/@id][@effectiveDate = $vs/@effectiveDate] | $setlib:colDecorData//issue/object[@id = $vs/@ref])}"/>
                ,
                vsapi:copyConceptList($vs/conceptList/concept | $vs/conceptList/include | $vs/conceptList/exclude | $vs/conceptList/exception | $vs/completeCodeSystem)
            )
            else (
                (:$vs/classification:)
            )
        }
};

declare %private function vsapi:copyConceptList($nodes as element()*) {
    for $node in $nodes
    let $elname := name($node)
    return
        <items>
        {
             attribute { 'is' } { $elname },
             $node/(@* except @is),
             $node/node()
        }
        </items>
};

(: Central logic for creating an empty dataset

@param $authmap         - required. Map derived from token
@return (empty) dataset object as xml with json:array set on elements
:)
declare function vsapi:createValueSet($authmap as map(*), $projectPrefix as xs:string, $targetDate as xs:boolean, $sourceId as xs:string?, $sourceEffectiveDate as xs:string?, $refOnly as xs:boolean?, $keepIds as xs:boolean?, $baseId as xs:string?, $editedValueset as element(valueSet)?) {

    let $decor                      := utillib:getDecorByPrefix($projectPrefix)
    let $projectLanguage            := $decor/project/@defaultLanguage
    (: if the user sent us a valueSet with id, we should assume he intends to keep that id :)
    let $keepIds                    := if ($editedValueset/@id) then true() else $keepIds = true()
    (: if the user sent us a valueSet with effectiveDate, we should assume he intends to keep that effectiveDate :)
    let $now                        := 
        if ($editedValueset/@effectiveDate) then substring($editedValueset/@effectiveDate, 1, 19) else 
        if ($targetDate) then substring(string(current-date()), 1, 10) || 'T00:00:00' else substring(string(current-dateTime()), 1, 19)
    
    let $check                      :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, 'Project with prefix ' || $projectPrefix || ' does not exist')
        )
    let $check                      :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify terminology in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    
    let $sourceValueset             := if (empty($sourceId)) then () else vsapi:getValueSet($projectPrefix, (), (), $sourceId, $sourceEffectiveDate, false(), true())
    let $storedValueset             := 
        if ($keepIds and $editedValueset[@id]) then vsapi:getValueSet($projectPrefix, (), (), $editedValueset/@id, $now, false(), true()) else ()
    
    let $check                      :=
        if ($refOnly) then
            if (empty($sourceId)) then
                error($errors:BAD_REQUEST, 'Parameter sourceId SHALL be provided if a value set reference is to be created')
            else
            if (empty($sourceValueset)) then
                if (empty($sourceEffectiveDate)) then
                    error($errors:BAD_REQUEST, 'Parameter sourceId ' || $sourceId || ' SHALL lead to a value set in scope of this project')
                else (
                    error($errors:BAD_REQUEST, 'Parameters sourceId ' || $sourceId || ' and sourceEffectiveDate ' || $sourceEffectiveDate || ' SHALL lead to a value set in scope of this project')
                )
            else ()
        else
        if (empty($editedValueset) and empty($sourceId)) then 
            error($errors:BAD_REQUEST, 'ValueSet input data or a sourceId SHALL be provided')
        else
        if ($editedValueset) then
            if ($storedValueset) then
                error($errors:BAD_REQUEST, 'Cannot create new valueSet. The input valueSet with id ' || $editedValueset/@id || ' and effectiveDate ' || $now || ' already exists.')
            else ()
        else (
            if (empty($sourceValueset)) then 
                error($errors:BAD_REQUEST, 'Cannot create new valueSet. Source valueset based on id ' || $sourceId || ' and effectiveDate ' || $sourceEffectiveDate || ' does not exist.')
            else  ()
        )
    let $check                      :=
        if ($now castable as xs:dateTime) then () else (
            error($errors:BAD_REQUEST, 'Cannot create new valueSet. The provided effectiveDate ' || $now || ' is not a valid xs:dateTime. Expected yyyy-mm-ddThh:mm:ss.')
        )
    
    let $editedValueset             := ($editedValueset, $sourceValueset)[1]
    (: decorlib:getNextAvailableIdP() returns <next base="1.2" max="2" next="{$max + 1}" id="1.2.3" type="DS"/> :)
    let $newValuesetId              := if ($keepIds) then $editedValueset/@id else (decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-VALUESET, $baseId)/@id)
    
    let $check                      :=
        if (vsapi:getValueSet($projectPrefix, (), (), $newValuesetId, $now, false(), true())) then
            error($errors:BAD_REQUEST, 'Cannot create new valueSet. The to-be-created valueSet with id ' || $newValuesetId || ' and effectiveDate ' || $now || ' already exists.')
        else ()
    
    let $baseValueset               := 
        <valueSet>
        {
            attribute id {$newValuesetId} ,
            $editedValueset/@name[string-length()>0] ,
            $editedValueset/@displayName[string-length()>0] ,
            attribute effectiveDate {$now} ,
            attribute statusCode {"draft"} ,
            $editedValueset/@versionLabel[string-length()>0] ,
            $editedValueset/@expirationDate[string-length()>0] ,
            $editedValueset/@officialReleaseDate[string-length()>0] ,
            $editedValueset/@experimental[string-length()>0] ,
            $editedValueset/@canonicalUri[string-length()>0] ,
            $editedValueset/@lastModifiedDate[string-length()>0]
        }
        </valueSet>
    let $newValueSet                    := 
        if ($refOnly) then
            <valueSet ref="{$sourceValueset/@id}" name="{$sourceValueset/@name}" displayName="{($sourceValueset/@displayName, $sourceValueset/@name)[1]}"/>
        else (
            utillib:prepareValueSetForUpdate($editedValueset, $baseValueset)
        )
    
    let $valueSetAssociation            := 
        if ($editedValueset[string-length(@conceptId)>0]) then 
            <terminologyAssociation>
            {
                $editedValueset/@conceptId,
                attribute valueSet {$newValueSet/@id},
                if ($editedValueset[string-length(@flexibility)=0]) then () else (
                    attribute flexibility {$newValueSet/@effectiveDate}
                ),
                attribute effectiveDate {$now}
            }
            </terminologyAssociation>
        else ()
    let $terminologyAssociations        :=
        for $concept in $editedValueset//*[string-length(@conceptId)>0][string-length(@code)>0][string-length(@codeSystem)>0]
        return
            <terminologyAssociation>
            {
                $concept/@conceptId, $concept/@code, $concept/@codeSystem, $concept/@displayName, 
                attribute effectiveDate {$now}
            }
            </terminologyAssociation>
    
    (: prepare sub root elements if not existent :)
    let $valueSetUpdate                 :=
        if (not($decor/terminology) and $decor/codedConcepts) then
            update insert <terminology/> following $decor/codedConcepts
        else if (not($decor/terminology) and not($decor/codedConcepts)) then
            update insert <terminology/> following $decor/ids
        else ()
        (: now update the value set :)
    let $valueSetUpdate                 :=
        if ($refOnly and $decor//valueSet[@ref = $newValueSet/@ref]) then
            (: this could update the name/displayName if the source thing was updated since adding it previously :)
            update replace $decor//valueSet[@ref = $newValueSet/@ref] with $newValueSet
        else (
            update insert $newValueSet into $decor/terminology
        )
    
    let $terminologyAssociationUpdates  :=
        utillib:addTerminologyAssociations($valueSetAssociation | $terminologyAssociations, $decor, false(), false())
   
    (: return the regular valueSet that was created, or the requested version of the reference that was created, or the latest version of the reference that was created :)
    let $result                         := vsapi:getValueSet($projectPrefix, (), (), ($newValueSet/@id | $newValueSet/@ref), ($newValueSet/@effectiveDate, $sourceEffectiveDate, 'dynamic')[1], false(), true())
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:<sourceCodeSystem id="any-concept-or-exception-codesystem-in-valueSet" identifierName="codeSystemName" canonicalUri="xs:anyURI" canonicalUriDSTU2="xs:anyURI" canonicalUriSTU3="xs:anyURI" canonicalUriR4="xs:anyURI" />:)
declare %private function vsapi:createSourceCodeSystems($valueSet as element(valueSet), $decorOrPrefix as item(), $language as xs:string?) as element(sourceCodeSystem)* {
    for $sourceCodeSystem in distinct-values($valueSet//@codeSystem)
    let $codeSystemName := replace(normalize-space(utillib:getNameForOID($sourceCodeSystem, $language, $decorOrPrefix)),'\s','&#160;')
    return
        <sourceCodeSystem id="{$sourceCodeSystem}" identifierName="{$codeSystemName}" canonicalUri="{utillib:getCanonicalUriForOID($sourceCodeSystem, (), $decorOrPrefix, ())}">
        {
            for $fhirVersion in $setlib:arrKeyFHIRVersions
            return
                attribute {concat('canonicalUri', $fhirVersion)} {utillib:getCanonicalUriForOID($sourceCodeSystem, (), $decorOrPrefix, $fhirVersion)}
            ,
            (: ART-DECOR 3: get for example external/hl7 :)
            let $coll   := collection($setlib:strCodesystemStableData)//identifier[@id = 'urn:oid:' || $sourceCodeSystem]
            
            return
                if (empty($coll)) then () else attribute context {substring-after(util:collection-name($coll[1]), $setlib:strCodesystemStableData || '/')}
        }
        </sourceCodeSystem>
};

(:~ Returns a list of zero or more valuesets as listed in the terminology section. This function is useful e.g. to call from a ValueSetIndex. Parameter id, name or prefix is required.

@param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only
@param $projectVersion   - optional. if empty defaults to current version. if valued then the valueset will come explicitly from that archived project version which is expected to be a compiled version
@param $projectLanguage  - optional. defaults to project defaultLanguage or first available if multiple compilation exist 
@param $objectid         - optional. Identifier of the valueset to retrieve
@param $objectnm         - optional. Name of the valueset to retrieve (valueSet/@name)
@param $objected         - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
@param $max              - optional. Maximum number of results with minimum 1. Default is $vsapi:maxResults (50)
@param $resolve          - optional. Default = 'true' If true, resolves any references
@return List object with zero or more valueSet
@since 2013-06-14
:)
declare %private function vsapi:getValueSetList($governanceGroupdId as xs:string?, $projectPrefix as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $searchTerms as xs:string*, $objected as xs:string?, $includebbr as xs:boolean, $sort as xs:string?, $sortorder as xs:string?, $max as xs:integer?, $resolve as xs:boolean) as element(list) {
    
    let $objected               := $objected[not(. = '')]
    let $governanceGroupdId     := $governanceGroupdId[not(. = '')]
    let $projectPrefix          := $projectPrefix[not(. = '')]
    let $projectVersion         := $projectVersion[not(. = '')]
    let $projectLanguage        := ($projectLanguage[not(. = '')], '*')[1]
    let $sort                   := $sort[string-length() gt 0]
    let $sortorder              := $sortorder[. = 'descending']
    
    let $startT                 := util:system-time()
    let $strDecorServicesURL    := serverapi:getServerURLServices()
    
    let $check                  :=
        if (empty($projectPrefix[not(. = '')])) then
            error($errors:BAD_REQUEST, 'Missing required parameter prefix')
        else ()
    
    let $decor                  := 
        if (utillib:isOid($projectPrefix)) then
            utillib:getDecorById($projectPrefix, $projectVersion, $projectLanguage)
        else (
            utillib:getDecorByPrefix($projectPrefix, $projectVersion, $projectLanguage)
        )
    let $projectPrefix          := ($decor/project/@prefix)[1]
    
    let $results                := 
        if (empty($searchTerms)) then (
            $decor//valueSet
        )
        else (
            let $buildingBlockRepositories  := 
                if ($includebbr) then (
                    $decor/project/buildingBlockRepository[empty(@format)] | 
                    $decor/project/buildingBlockRepository[@format='decor'] | 
                    <buildingBlockRepository url="{serverapi:getServerURLServices()}" ident="{$projectPrefix}" format="decor"/>
                )
                else (
                    <buildingBlockRepository url="{serverapi:getServerURLServices()}" ident="{$projectPrefix}" format="decor"/>
                )
            let $luceneQuery                := utillib:getSimpleLuceneQuery($searchTerms, 'wildcard')
            let $luceneOptions              := utillib:getSimpleLuceneOptions()

            for $repository in $buildingBlockRepositories
            let $repourl                := $repository/@url
            let $repoident              := $repository/@ident
            let $cachedProject          := 
                if ($repourl = $strDecorServicesURL) then
                    utillib:getDecorByPrefix($repoident)
                else (
                    $setlib:colDecorCache//cacheme[@bbrurl = $repourl][@bbrident = $repoident]
                )
            let $objects                :=
                if ($cachedProject) then (
                    $cachedProject//valueSet[@id = $searchTerms] | 
                    $cachedProject//valueSet[@id][ft:query(@name, $luceneQuery, $luceneOptions)] |
                    $cachedProject//valueSet[@id][ft:query(@displayName, $luceneQuery, $luceneOptions)]
                )
                else (
                    let $service-uri    := xs:anyURI($repourl || '/SearchValueSet?searchString=' || encode-for-uri(string-join($searchTerms, ' ')) || '&amp;prefix=' || $repoident)
                    let $requestHeaders := 
                        <http:request method="GET" href="{$service-uri}">
                            <http:header name="Content-Type" value="text/xml"/>
                            <http:header name="Cache-Control" value="no-cache"/>
                            <http:header name="Max-Forwards" value="1"/>
                        </http:request>
                    let $server-response        := http:send-request($requestHeaders)
                    
                    return $server-response//valueSet[@id]
                )
            return (
                for $vs in $objects
                return
                    <valueSet>
                    {
                        $vs/(@* except (@url|@ident)),
                        attribute url {$repourl},
                        attribute ident {$repoident},
                        attribute cachedProject {exists($cachedProject)},
                        $vs/node()
                    }
                    </valueSet>
            )
        )
    
    let $allcnt                 := count($results)
    
    let $valueSetsByRef           :=
        if ($resolve and empty($projectVersion)) then
            for $vs in $results[@ref]
            let $id     := $vs/@ref
            let $vss    := vsapi:getValueSet($projectPrefix, $projectVersion, $projectLanguage, $vs/@ref, (), true(), false())
            let $vsbyid := $vss[@effectiveDate = max($vss/xs:dateTime(@effectiveDate))]
            return (
                (: rewrite name and displayName based on latest target codeSystem. These sometimes run out of sync when the original changes its name :)
                element {name($vs)} {
                    $vs/@ref, 
                    attribute name {($vsbyid/@name, $vs/@name)[1]}, 
                    attribute displayName {($vsbyid/@displayName, $vs/@displayName, $vsbyid/@name, $vs/@name)[1]}
                } | $vss
            )
        else (
            $results[@ref]
        )
    
    let $results            := $results[@id] | $valueSetsByRef
    
    (: now we can determine $objected :)
    let $results            :=
        if (empty($objected)) then 
            (: all :)
            $results
        else
        if ($objected castable as xs:dateTime) then
            (: match and ref :)
            $results[@ref] | $results[@effectiveDate = $objected]
        else (
            (: newest and ref :)
            $results[@ref] | $results[@effectiveDate = max($results[@effectiveDate]/xs:dateTime(@effectiveDate))]
        )
    
    let $results            :=
        for $vs in $results
        let $id             := $vs/@id | $vs/@ref
        group by $id
        return (
            let $subversions    :=
                for $vsv in $vs
                order by $vsv/@effectiveDate descending
                return
                    <valueSet uuid="{util:uuid()}">
                    {
                        $vsv/(@* except @uuid),
                        (: this element is not supported (yet?) :)
                        $vsv/classification
                    }
                    </valueSet>
            let $latest         := ($subversions[@id], $subversions)[1]
            return
            <valueSet uuid="{util:uuid()}" id="{$id}">
            {
                $latest/(@* except (@uuid | @id | @ref | @project)),
                ($vs/@ref)[1],
                (: there is no attribute @project, but better safe than sorry :)
                if (empty($governanceGroupdId)) then $latest/@project else attribute project {$projectPrefix},
                $subversions
            }
            </valueSet>
        )
    let $count              := count($results/valueSet)
    let $max                := if ($max ge 1) then $max else $count
    
    (: handle sorting. somehow reverse() does not do what I expect :)
    let $results            :=
        switch ($sort)
        case 'displayName' return 
            if ($sortorder = 'descending') then 
                for $r in $results order by $r/lower-case(@displayName) descending return $r
            else (
                for $r in $results order by $r/lower-case(@displayName)            return $r
            )
        case 'name'        return 
            if ($sortorder = 'descending') then 
                for $r in $results order by $r/lower-case(@name) descending return $r
            else (
                for $r in $results order by $r/lower-case(@name)            return $r
            )
        default            return 
            if ($sortorder = 'descending') then 
                for $r in $results order by replace(replace($r/@id, '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1') descending return $r
            else (
                for $r in $results order by replace(replace($r/@id, '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1')            return $r
            )
    
    let $durationT := (util:system-time() - $startT) div xs:dayTimeDuration("PT0.001S")
    
    return
        <list artifact="VS" elapsed="{$durationT}" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" resolve="{$resolve}" project="{$projectPrefix}" lastModifiedDate="{current-dateTime()}">
        {
            subsequence($results, 1, $max)
        }
        </list>
};