xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
:)
(:~ Template API allows read, create, update of DECOR templates :)
module namespace tmapi              = "http://art-decor.org/ns/api/template";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace ggapi       = "http://art-decor.org/ns/api/governancegroups" at "governancegroups-api.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";
import module namespace serverapi   = "http://art-decor.org/ns/api/server" at "server-api.xqm";
import module namespace tm          = "http://art-decor.org/ns/decor/template" at "../../art/api/api-decor-template.xqm";
import module namespace vs          = "http://art-decor.org/ns/decor/valueset" at "../../art/api/api-decor-valueset.xqm";
declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";

declare %private variable $tmapi:STATUSCODES-FINAL          := ('active', 'cancelled', 'pending', 'review', 'rejected', 'retired');
declare %private variable $tmapi:ADDRLINE-TYPE              := utillib:getDecorTypes()/AddressLineType;
declare %private variable $tmapi:TEMPLATE-TYPES             := utillib:getDecorTypes()/TemplateTypes;
declare %private variable $tmapi:TEMPLATE-FORMATS           := utillib:getDecorTypes()/TemplateFormats;
declare %private variable $tmapi:RELATIONSHIP-TYPES         := utillib:getDecorTypes()/RelationshipTypes;
declare %private variable $tmapi:EXAMPLE-TYPES              := utillib:getDecorTypes()/ExampleType;

(:~ Retrieves latest DECOR template based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function tmapi:getLatestTemplate($request as map(*)) {
    tmapi:getTemplate($request)
};

(:~ Retrieves latest DECOR template based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
@param $id                      - required parameter denoting the id of the template
@param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
@param $projectPrefix           - optional. limits scope to this project only
@param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
@return as-is or as compiled as JSON
@since 2020-05-03
:)
declare function tmapi:getTemplate($request as map(*)) {
    let $projectPrefix      := $request?parameters?prefix[not(. = '')]
    let $projectVersion     := $request?parameters?release[not(. = '')]
    let $projectLanguage    := $request?parameters?language[not(. = '')]
    let $id                 := $request?parameters?id[not(. = '')]
    let $effectiveDate      := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    let $results            := tmapi:getTemplate($id, ($effectiveDate, 'dynamic')[1], $projectPrefix, $projectVersion, $projectLanguage)
    
    (: if multiple copies are found, prefer copies from this server of other copies. :)
    let $results            :=
        if (count($results) gt 1) then (
            let $t := $results[@id][@url = $tm:strDecorServicesURL] | $results[@id][empty(@url)]
            let $t := if ($t) then $t else $results[@id]
            return
                if ($t) then $t else $results
        )
        else (
            $results
        )
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple templates for id ''' || $id || '''. Expected 0..1. Alert your administrator as this should not be possible. Found in: ' || string-join($results/concat(@ident, ' / ', @url), ', '))
            (:<list xmlns:json="http://www.json.org">
            {
                utillib:addJsonArrayToElements($results)
            }
            </list>:)
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Returns a list of zero or more templates
    
    @param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only
    @param $projectVersion   - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
    @param $id               - optional. Identifier of the template to retrieve
    @param $name             - optional. Name of the template to retrieve (valueSet/@name)
    @param $effectiveDate    - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
    @param $treetype         - optional. Default $tm:TREETYPELIMITEDMARKED
    @param $resolve          - optional. Boolean. Default true if governanceGroupId is empty. If true retrieves all versions for references, otherwise just returns the references.
    @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
    @since 2013-06-14
:)
declare function tmapi:getTemplateList($request as map(*)) {
    let $governanceGroupId      := $request?parameters?governanceGroupId[not(. = '')]
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $searchTerms            := 
        array:flatten(
            for $s in $request?parameters?search[string-length() gt 0]
            return
                tokenize(lower-case($s),'\s')
        )
    let $id                     := $request?parameters?id[not(. = '')]
    let $nm                     := $request?parameters?name[not(. = '')]
    let $ed                     := $request?parameters?effectiveDate[not(. = '')]
    let $treetype               := $request?parameters?treetype
    let $resolve                :=  if (empty($governanceGroupId)) then not($request?parameters?resolve = false()) else $request?parameters?resolve = true()
    
    let $check                  :=
        if (count($governanceGroupId) + count($projectPrefix) = 0) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter governanceGroupId or prefix')
        else 
        if (count($governanceGroupId) + count($projectPrefix) gt 1) then 
            error($errors:BAD_REQUEST, 'Request SHALL have exactly one parameter governanceGroupId or prefix, not both or multiple')
        else
        if (count($governanceGroupId) gt 0 and not(empty($projectVersion))) then
            error($errors:BAD_REQUEST, 'Request SHALL NOT have both governanceGroupId and release of a single project')
        else
        if (count($governanceGroupId) gt 0 and not(empty($searchTerms))) then
            error($errors:BAD_REQUEST, 'Search only supported in a project')
        else ()

    let $results                :=
        if (count($searchTerms) gt 0) then
            tmapi:searchTemplate($projectPrefix, $projectVersion, $projectLanguage, $searchTerms, $treetype, $resolve)
        else
        if (empty($governanceGroupId)) then
            tmapi:getTemplateList($projectPrefix, $projectVersion, $projectLanguage, $id, $nm, $ed, $treetype, $resolve)
        else (
            for $projectId in ggapi:getLinkedProjects($governanceGroupId)/@ref
            return
                tmapi:getTemplateList($projectPrefix, (), (), $id, $nm, $ed, $treetype, $resolve)
        )
    
    let $schemaTypes                := utillib:getDecorTypes()//TemplateTypes/enumeration
    let $results                :=
        for $class in $results/class[template]
        let $type   := $class/@type
        group by $type
        order by count($schemaTypes[@value = $type]/preceding-sibling::enumeration)
        return
            <category>
            {
                $class[1]/@*,
                for $name in $class[1]/label
                return
                    <name>
                    {
                        $name/@*,
                        $name/node()
                    }
                    </name>
                ,
                for $templateSet in $class/template
                let $tmid       := $templateSet/(@id | @ref)
                group by $tmid
                order by lower-case($templateSet[1]/@displayName)
                return
                    <template>
                    {
                        $templateSet[1]/@*,
                        for $template in $templateSet/template
                        let $tmed   := $template/@effectiveDate
                        group by $tmed
                        order by $tmed descending
                        return
                            <version>
                            {
                                $template[1]/@*,
                                $template[1]/desc
                                ,
                                <classification>
                                {
                                    ($template[1]/classification/@format)[1]
                                    ,
                                    if ($template[1]/classification/@type) then
                                        for $type in $template[1]/classification/@type
                                        return
                                            <type>{$type}</type>
                                    else (
                                        <type>notype</type>
                                    ),
                                    for $tag in $template[1]/classification/tag
                                    group by $tag
                                    return
                                        $tag
                                    ,
                                    for $property in $template[1]/classification/property
                                    group by $property
                                    return
                                        $property
                                }
                                </classification>
                                ,
                                $template[1]/(node() except (desc | classification))
                            }
                            </version>
                            
                    }
                    </template>
            }
            </category>
    
    let $countTN                := count($results/template/version)
    
    return
        <list artifact="TM" current="{$countTN}" total="{$countTN}" all="{$countTN}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Returns a list of zero or more templateAssociation
    
    @param $projectPrefix    - required. determines search scope. limits scope to this project only
    @param $projectVersion   - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
    @param $projectLanguage  - optional. works only in conjunction with $projectVersion. if empty defaults to *. if valued then the template will come explicitly from the compilation in that language. * is the "all languages" compilation
    @param $id               - optional. Identifier of the template to retrieve templateAssociations for
    @param $effectiveDate    - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
    @return list object with zero or more templateAssociation objects
    @since 2022-02-02
:)
declare function tmapi:getTemplateAssociationList($request as map(*)) {
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $id                     := $request?parameters?id[not(. = '')]
    let $ed                     := $request?parameters?effectiveDate[not(. = '')]
    
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have parameter prefix')
        else () 
        
    let $results                := tmapi:getTemplateAssociationList($projectPrefix, $projectVersion, $projectLanguage, $id, $ed)
    
    return
        $results
};

(:~ Deletes a template reference based on $id (oid) and project id or prefix
    @param $id                      - required parameter denoting the id of the template
    @param $project                 - required. limits scope to this project only
:)
declare function tmapi:deleteTemplate($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0][not(. = '*')]
    let $id                     := $request?parameters?id[string-length() gt 0]
    
    let $check                  :=
        if (empty($project) or empty($id)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have both parameter project and id')
        else ()
    
    let $decor                  := 
        if (utillib:isOid($project)) then
            utillib:getDecorById($project)
        else (
            utillib:getDecorByPrefix($project)
        )
    
    let $check                  :=
        if (empty($decor)) then
            error($errors:BAD_REQUEST, 'Project ' || $project || ' not found.')
        else ()
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-RULES)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify templates in project ', $project[1], '. You have to be an active author in the project.'))
        )
    
    let $delete                 := update delete $decor/rules/template[@ref = $id]
    let $delete                 := update delete $decor/rules/templateAssociation[@templateId = $id][empty(*)]
    
    return
        roaster:response(204, ())
};

(:~ Returns template datatypes
    
    @param $format           - optional. defaults to hl7v3xml1
    @return datatypes or http 404 if not found
    @since 2022-02-07
:)
declare function tmapi:getTemplateDatatypes($request as map(*)) {
    let $format                 := ($request?parameters?format[not(. = '')], 'hl7v3xml1')[1]
        
    let $decorDatatypes         := $setlib:colDecorCore/supportedDataTypes[@type = $format]

    let $results                :=
        if (empty($decorDatatypes)) then () else (
            <supportedDataTypes format="{$decorDatatypes/@type}">
            {
                $decorDatatypes/(@* except (@type | @xsi:*))
            }
            {
                let $allTypes   := $decorDatatypes//dataType | $decorDatatypes//flavor | $decorDatatypes//atomicDataType
                
                (: assume that if there is a definition for the dataType at root level,  then that is complete, whereas a subtyped dataType does not need to be:)
                for $typeDef in $allTypes
                let $typeName    := $typeDef/@name 
                group by $typeName
                order by $typeName
                return
                    <type>
                    {
                        $typeDef[1]/@name, 
                        $typeDef[1]/@type, 
                        $typeDef[1]/@hasStrength, 
                        if ($typeDef[1]/self::flavor) then attribute isFlavor {'true'} else (),
                        if ($typeDef[1][self::atomicDataType]) then attribute type {'simpletype'} else (),
                        $typeDef[1]/@realm 
                    }
                        <item name="{$typeName}"/>
                    {
                        for $subtype in distinct-values(($typeDef[1]//dataType | $typeDef[1]//flavor | $typeDef[1]//atomicDataType)/@name)
                        order by $subtype
                        return
                        <item name="{$subtype}"/>
                    }
                    {
                        $typeDef[1]/desc,
                        $typeDef[1]/attribute,
                        $typeDef[1]/element
                    }
                    </type>
            }
            </supportedDataTypes>
        )

    return
        if (empty($results)) then
            roaster:response(404, ())
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

declare function tmapi:getTemplate($id as xs:string, $effectiveDate as xs:string?, $projectPrefix as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?) as element(template)* {
    let $id                     := $id[not(. = '')]
    let $effectiveDate          := $effectiveDate[not(. = '')]
    let $projectPrefix          := $projectPrefix[not(. = '')]
    let $projectVersion         := $projectVersion[not(. = '')]
    
    let $templates              :=
        if (empty($projectPrefix)) then 
            (:
              <return>
                <project xmlns:ext="urn:hl7-org:v3/ext"...>
                  <template id="2.16.840.1.113883.3.1937.99.62.3.10.11" name="m-bsn" displayName="Meting met BSN" effectiveDate="2012-03-11T00:00:00" statusCode="draft">
            :)
            (:tm:getTemplateById($id, $effectiveDate):)
            tm:getExpandedTemplateById($id, $effectiveDate, false())
        else (
            (:
              <return>
                <template id="2.16.840.1.113883.3.1937.99.62.3.10.11" ident="demo1-">
                  <template xmlns:ext="urn:hl7-org:v3/ext" ...>
            :)
            (:tm:getTemplateById($id, $effectiveDate, $projectPrefix[1], $projectVersion[1]):)
            tm:getExpandedTemplateById($id, $effectiveDate, $projectPrefix, $projectVersion, (), false())
        )
    let $results                := 
        for $template in $templates/*/template
        return
            tmapi:prepareTemplateForAPI($template)

    return
        $results
};

declare function tmapi:prepareTemplateForAPI($template as element(template)) as element(template) {
    let $ns-attributes  := 
        if ($template/@*[contains(name(), 'dummy')]) then () else ( 
            $template/../@*[contains(name(), 'dummy')]
        )
    return 
        <template>
        {
            $template/@*,
            if ($template/@ident) then () else (
                $template/../@ident,
                if ($template/@url) then () else $template/../@url
            ),
            $ns-attributes,
            for $ns-prefix at $i in in-scope-prefixes($template)
            let $ns-uri := namespace-uri-for-prefix($ns-prefix, $template)
            return
                (:hl7:dummy-default="urn:hl7-org:v3":)
                <ns uri="{$ns-uri}" prefix="{$ns-prefix}" default="{exists($template/../@*[name() = $ns-prefix || ':dummy-default'])}"/>
            ,
            (: we return just a count of issues and leave it up to the calling party to get those if required :)
            <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $template/@id][@effectiveDate = $template/@effectiveDate] | $setlib:colDecorData//issue/object[@id = $template/@ref])}"/>
            ,
            (: we include all static association here that belong to our template. Note that there could be staticAssociation element under include. Those would come from a different template than our with potentially overlapping elementIds
               We do not want to include those because that would lead to false associations where a concept connected to a different, included template is assoicated to someting in the calling template.
            :)
            <staticAssociations>
            {
                for $node in $template//staticAssociations/*
                return
                    <concept>{$node/@*, $node/node()}</concept>
            }
            </staticAssociations>
            ,
            $template/desc
            ,
            tmapi:collapseTemplateClassifications($template/classification)
            ,
            for $e in $template/(* except (ns|issueAssociation|staticAssociations|desc|classification|element|attribute|include|choice|defineVariable|let|assert|report))
            return
                if ($e/self::example) then
                    <example>
                    {
                        if ($e/@type) then $e/@type else attribute type {'neutral'},
                        $e/@caption,
                        $e/node()
                    }
                    </example>
                else (
                    $e
                )
            ,
            tmapi:prepareTemplateBodyForAPI($template/(element|attribute|include|choice|defineVariable|let|assert|report|property|item))
        }
        </template>
};

declare %private function tmapi:prepareTemplateBodyForAPI($node as node()*) {
    for $e at $pos in $node
    let $elname := name($e)
    return
        <items>
        {
            attribute { 'is' } { $elname },
            attribute { 'order' } { $pos },
            if ($e/self::example) then (
                if ($e/@type) then $e/@type else attribute type {'neutral'},
                $e/@caption,
                utillib:serializeNode($e)/node()
            )
            else (
                $e/(@* except (@is | @order))
                ,
                for $elm in $e/(node() except (ns|issueAssociation|classification|element|attribute|include|choice|defineVariable|let|assert|report|property|item|example))
                return
                    if ($elm/self::staticAssociations) then
                        <staticAssociations>
                        {
                            for $assoc in $elm/*
                            return
                                <concept>{$assoc/@*, $assoc/node()}</concept>
                        }
                        </staticAssociations>
                    else (
                        $elm
                    ) 
                ,
                tmapi:prepareTemplateBodyForAPI($e/(element|attribute|include|choice|defineVariable|let|assert|report|property|item|example))
            )
        }
        </items>   
};

declare %private function tmapi:collapseTemplateClassifications($classification as element(classification)*) {
    <classification>
    {
        ($classification/@format)[1]
        ,
        if ($classification/@type) then
            for $type in $classification/@type
            return
                <type>{data($type)}</type>
        else (
            <type>notype</type>
        ),
        for $tag in $classification/tag
        group by $tag
        return
            $tag
        ,
        for $property in $classification/property
        group by $property
        return
            $property
    }
    </classification>
};

(:~ Retrieves a new template in edit mode to be saved later, or does initial save of a template
@param $projectPrefix           - required. determines search scope. null is full server, pfx- limits scope to this project only
@param $sourceId                - optional. parameter denoting the id of a template to use as a basis for creating the new template
@param $sourceEffectiveDate     - optional. parameter denoting the effectiveDate of a template to use as a basis for creating the new template
@param $targetDate              - optional. If true invokes effectiveDate of the new template as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
@param $keepIds                 - optional. Only relevant if source template is specified. If true, the new template will keep the same id and only update the effectiveDate
@param $lock                    - optional. relevant when saving a new template. does not set a lock if false and returns in read mode, sets a lock if true and leaves in edit mode, otherwise
@param $request-body            - optional. json body containing new template structure
@return template
@since 2020-05-03
:)
declare function tmapi:postTemplate($request as map(*)) {
    let $authmap                        := $request?user
    let $projectPrefix                  := $request?parameters?prefix[not(. = '')]
    let $sourceId                       := $request?parameters?sourceId
    let $sourceEffectiveDate            := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?sourceEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?sourceEffectiveDate[string-length() gt 0]
        }
    let $refOnly                        := $request?parameters?refOnly = true()
    let $targetDate                     := $request?parameters?targetDate = true()
    let $keepIds                        := $request?parameters?keepIds = true()
    let $lock                           := $request?parameters?lock = true()
    let $data                           := utillib:getBodyAsXml($request?body, 'template', ())
    
    let $check                          := 
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have prefix')
        else ()
    let $check                          := 
        if ($refOnly and empty($sourceId)) then
            error($errors:BAD_REQUEST, 'Request SHALL have source id if refOnly = true')
        else ()
    let $check                          := 
        if ($keepIds and empty($sourceId)) then
            error($errors:BAD_REQUEST, 'Request SHALL have source id if keepIds = true')
        else ()
    let $data                           := 
        if ($refOnly) then 
            <template ref="{$sourceId}" flexibility="{$sourceEffectiveDate}"/>
        else 
        if (empty($data)) then
            (:error($errors:SERVER_ERROR, 'Retrieval of new template not yet implemented'):)
            tmapi:getTemplateForEdit($authmap, $projectPrefix, 'new', $sourceId, $sourceEffectiveDate, $keepIds, $targetDate, false())
        else (
            $data
        )
    let $return                         := tmapi:postTemplate($authmap, $projectPrefix, $data, $keepIds, $lock)
            
    return
        if (empty($data)) then 
            roaster:response(200, $return)
        else (
            roaster:response(201, $return)
        )
};

declare function tmapi:postTemplateExample($request as map(*)) {
    let $authmap                        := $request?user
    let $projectPrefix                  := $request?parameters?prefix[not(. = '')]
    let $tmid                           := $request?parameters?id[not(. = '')]
    let $tmed                           := $request?parameters?effectiveDate[not(. = '')]
    let $release                        := $request?parameters?release[not(. = '')]
    let $language                       := $request?parameters?language[not(. = '')]
    let $elid                           := $request?parameters?elementId[not(. = '')]
    let $doSerialized                   := $request?parameters?serialized = true()
    (: tmapi:prepareTemplateForUpdate takes care of eliminating unselected items :)
    let $doSelectedOnly                 := false()
    let $doRecursive                    := $request?parameters?recursive = true()
    
    let $data                           := utillib:getBodyAsXml($request?body, 'template', ())
    let $projectPrefix                  := ($data/@projectPrefix, $projectPrefix)[1]
    let $check                          :=
        if ($data) then
            if (empty($tmid)) then () else (
                error($errors:BAD_REQUEST, 'Request SHALL NOT have both request body and parameter id')
            )
        else 
        if (empty($tmid)) then 
            error($errors:BAD_REQUEST, 'Request SHALL at least have a request body or a parameter id')
        else ()
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have prefix')
        else ()
    (: start processing ... :)
    let $template           := if ($data) then $data else (tm:getTemplateById($tmid, $tmed, $projectPrefix, $release)/template/template[@id])[1]
    let $prefix             := ($projectPrefix, $template/@ident)[1]
    let $template           := if ($template) then tmapi:prepareTemplateForUpdate($template) else ()
    let $format             := ($template/classification/@format, 'hl7v3xml1')[not(. ='')][1]
    
    let $check              :=
       if ($template) then 
           if (empty($elid)) then () else if ($template//*[@id = $elid]) then () else (
               error($errors:BAD_REQUEST,concat('Argument elementId ', $elid, ' does not exist in template with id ', $template/@id, ' effectiveDate ', $template/@effectiveDate))
           )
       else (
           error($errors:BAD_REQUEST,concat('Argument id ', $tmid, ' effectiveDate ', $tmed, ' did not lead to a template and no template provided.'))
       )
       
    let $templateChain      := 
       if ($template and $doRecursive) then $template | tm:getTemplateChain($prefix, $template, map:merge(map:entry(concat($template/@id, $template/@effectiveDate), ''))) else (
           $template,
           for $elm in $template//element[@contains] | $template//include
           return
               (tm:getTemplateById($elm/@contains | $elm/@ref, ($elm/@flexibility, 'dynamic')[1], $prefix)/template/template[@id])[1]
       )
    let $templateChain      :=
       for $t in $templateChain
       let $ideff  := concat($t/@id, $t/@effectiveDate)
       group by $ideff
       order by $t[1]/@id
       return $t[1]
    
    let $vocabChain         :=
       for $v in $templateChain//vocabulary[@valueSet]
       let $ideff  := concat($v/@valueSet, $v/@flexibility)
       group by $ideff
       return
           vs:getExpandedValueSetById($v[1]/@valueSet, if ($v[1]/@flexibility castable as xs:dateTime) then $v[1]/@flexibility else 'dynamic', $prefix, (), (), false())//valueSet[@id]
    
    let $decor              := utillib:getDecorByPrefix($prefix, $release, $language)
    let $decorPackage       :=
       <decor>
           {$decor/@*}
           <project>{$decor/project/@*, $decor/project/defaultElementNamespace}</project>
           <terminology>{$vocabChain}</terminology>
           <rules>{$templateChain}</rules>
       </decor>
    
    let $xsltParameters :=
       <parameters>
           <param name="tmid"                      value="{($template/@id, $tmid)[1]}"/>
           <param name="tmed"                      value="{($template/@effectiveDate, $tmed)[1]}"/>
           <param name="elid"                      value="{$elid}"/>
           <param name="doSelectedOnly"            value="{$doSelectedOnly}"/>
           <param name="doRecursive"               value="{$doRecursive}"/>
           <param name="logLevel"                  value="'OFF'"/>
       </parameters>
    
    let $xslt           := xs:anyURI('https://art-decor.org/ADAR/rv/Template2Example.xsl')
       (:if ($dodev) then 
           xs:anyURI('https://art-decor.org/ADAR-dev/rv/Template2Example.xsl')
       else (
           (\:xs:anyURI(concat('xmldb:exist://', $get:strDecorCore, '/Template2Example.xsl')):\)
           
       ):)
    let $dopackage          := false()
    (:we need a root element, it's the way it is...:)
    let $tpath              := replace($template/context/@path,'[/\[].*','')
    let $path               := if (string-length($tpath)>0) then $tpath else ('art:placeholder')
    let $element            := <element name="{$path}" format="{$format}" xmlns:art="urn:art-decor:example" selected="">{$template/(attribute|element|include|choice)}</element>
    let $example            := if ($dopackage) then () else if ($decor) then transform:transform($decorPackage, $xslt, $xsltParameters) else ()
    
    (:if we did not need our pseudo root-element, just leave it off:)
    let $example            := if ($example[count(@*)>0 or count(*)>1]) then $example else $example/node()
    
    let $return                         := 
       if ($dopackage) then $decorPackage else (
           <example caption="" type="neutral" xmlns:json="http://www.json.org">
           {
               if ($doSerialized) then
                   fn:serialize($example,
                       <output:serialization-parameters xmlns:output="http://www.w3.org/2010/xslt-xquery-serialization">
                           <output:method>xml</output:method>
                           <output:encoding>UTF-8</output:encoding>
                       </output:serialization-parameters>
                   )
               else (
                   $example
                   (:,
                   if ($dodev) then $decorPackage else ():)
               )
           }
           </example>
       )
    return
            roaster:response(200, $return)
};

(:~ Retrieves a template in edit mode
@param $id                      - required parameter denoting the id of the template
@param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
@return template
@since 2020-05-03
:)
declare function tmapi:getTemplateForEdit($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $breakLock                      := $request?parameters?breakLock = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $return                         := tmapi:getTemplateForEdit($authmap, (), 'edit', $id, $effectiveDate, true(), true(), $breakLock)
    return (
        roaster:response(200, $return)
    )
};

declare function tmapi:getTemplateForEdit($authmap as map(*), $projectPrefix as xs:string?, $mode as xs:string, $id as xs:string?, $effectiveDate as xs:string?, $keepIds as xs:boolean, $targetDate as xs:boolean, $breakLock as xs:boolean) as element(template) {
    
    let $template               := 
        if (empty($id)) then () else 
        if (empty($projectPrefix)) then 
            $setlib:colDecorData//template[@id = $id][@effectiveDate = $effectiveDate] 
        else (
            tmapi:getPrototype($id, $effectiveDate, $projectPrefix) 
        )
    let $decor                  := if (empty($projectPrefix)) then $template/ancestor::decor else utillib:getDecorByPrefix($projectPrefix)
    let $projectPrefix          := $decor/project/@prefix
    let $language               := $decor/project/@defaultLanguage
    
    let $lock                   := 
        if (empty($template) or not($mode = 'edit')) then () else decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := 
        if (empty($template) or not($mode = 'edit')) then () else if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breakLock)
    
    let $check                  :=
        if ($template) then () else if (empty($id)) then () else (
            error($errors:BAD_REQUEST, 'Template with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' not found in the context of project ' || $projectPrefix)
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-RULES)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify templates in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if ($mode = 'edit') then (
            if ($lock) then () else (
                error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this template (anymore). Get a lock first.'))
            ),
            if ($template[@statusCode = $tmapi:STATUSCODES-FINAL]) then
                error($errors:BAD_REQUEST, concat('Template cannot be updated while it has one of status: ', string-join($tmapi:STATUSCODES-FINAL, ', '), if ($template/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            else ()
        )
        else ()
    
    let $templateAssociations   := if (empty($id)) then () else $decor//templateAssociation[@templateId = $id][@effectiveDate=$effectiveDate]
    let $valueSetList           := $decor/terminology/valueSet
    let $templateList           := $decor/rules/template

    (: TODO: why do we only pick SPEC? Could it not be "anything"? All it does it supply you with potential elements and attributes to activate :)
    let $specialization         := $template/relationship[@type='SPEC'][@template][1]
    (:get prototype for editor with normalized attributes:)
    let $prototype              := if ($specialization) then (tmapi:getPrototype($specialization/@template, $specialization/@flexibility, $projectPrefix)) else ()
    
    let $useBaseId              := decorlib:getDefaultBaseIds($projectPrefix, $decorlib:OBJECTTYPE-TEMPLATE)[1]/@id
        
    let $namespaces-attrs       := 
        for $ns at $i in utillib:getDecorNamespaces($decor) 
        return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-',$i))} {$ns/@uri}
    
    let $templateName           := attribute name {($template/@name, utillib:getNameForOID($template/@id, (), $decor))[not(.='')][1]}
    let $templateDisplayName    := attribute displayName {($template/@displayName, $templateName)[not(. = '')][1]}
    let $result                 := 
        <template 
            projectPrefix="{$projectPrefix}" 
            baseId="{$useBaseId}">
        {
            attribute originalId {if ($mode = 'edit' or $keepIds) then $template/@id else ()},
            attribute id {if ($mode = 'edit' or $keepIds) then $template/@id else ()},
            $templateName,
            $templateDisplayName,
            attribute effectiveDate {
                if ($mode = 'edit') then $template/@effectiveDate else 
                if ($targetDate) then substring(string(current-date()), 1, 10) || 'T00:00:00' else substring(string(current-dateTime()), 1, 19)
            },
            attribute statusCode {if ($mode='edit') then $template/@statusCode else 'draft'},
            attribute versionLabel {$template/@versionLabel},
            attribute isClosed {$template/@isClosed},
            attribute expirationDate {$template/@expirationDate},
            attribute officialReleaseDate {$template/@officialReleaseDate},
            attribute canonicalUri {if ($mode='edit' or $keepIds) then $template/@canonicalUri else ()}
        }
        {
            $namespaces-attrs
        }
            <edit mode="{$mode}"/>
        {
            $lock/*
        }
        {
            if ($template/desc) then 
                for $desc in $template/desc
                return
                    utillib:serializeNode($desc)
            else (
                <desc language="{$language}"/>
            )
        }
        {
            $template/classification
        }
        {
            if ($mode = 'edit') then () else (
                if ($template[@id]) then (
                    <relationship type="{if ($mode = 'new') then 'SPEC' else upper-case($mode)}" template="{$template/@id}" selected="template" flexibility="{$template/@effectiveDate}">
                    {
                        for $attr in $template/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                        return
                            attribute {concat('tm', local-name($attr))} {$attr}
                        ,
                        attribute url {$tm:strDecorServicesURL},
                        attribute ident {$projectPrefix},
                        attribute linkedartefactmissing {'false'}
                    }
                    </relationship>
                ) else ()
            )
            ,
            for $relationship in $template/relationship
            let $addAttribute := if ($relationship/@template) then 'model' else 'template'
            let $searchTemplate := 
                if (empty($relationship/@template)) then () else (
                    tm:getTemplateById(
                        $relationship/@template,
                        ($relationship/@flexibility, 'dynamic')[1],
                        $decor)/template/template[@id][@effectiveDate]
                )
            let $recent         := $searchTemplate[1]
            return
            <relationship>
            {
                attribute type {$relationship/@type},
                if ($relationship/@template) then (
                    attribute template {$relationship/@template},
                    attribute model {''},
                    attribute flexibility {$relationship/@flexibility},
                    attribute selected {'template'},
                    if (empty($recent)) then () else (
                        for $attr in $searchTemplate/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                        return
                            attribute {concat('tm', local-name($attr))} {$attr}
                        ,
                        attribute url {$tm:strDecorServicesURL},
                        attribute ident {$projectPrefix}
                    ),
                    attribute linkedartefactmissing {empty($recent)}
                )
                else (
                    attribute template {''},
                    attribute model {$relationship/@model},
                    attribute flexibility {$relationship/@flexibility},
                    attribute selected {'model'}
                )
            }
            </relationship>
        }
        {
            for $node in $template/publishingAuthority
            return
                <publishingAuthority>
                {
                    $node/(@* except @selected),
                    attribute selected {''},
                    $node/addrLine
                }
                </publishingAuthority>
        }
        {
            for $node in $template/endorsingAuthority
            return
                <endorsingAuthority>
                {
                    $node/(@* except @selected),
                    attribute selected {''},
                    $node/addrLine
                }
                </endorsingAuthority>
        }
        {
            for $node in $template/purpose
            return
                utillib:serializeNode($node)
        }
        {
            for $node in $template/copyright
            return
                utillib:serializeNode($node)
        }
            <context>
            {
                if ($template/context/@id) then (
                    $template/context/@id,
                    attribute path {''},
                    attribute selected {'id'}
                )
                else (
                    attribute id {''},
                    attribute path {$template/context/@path},
                    attribute selected {'path'}
                )
            }
            </context>
            <item label="{$template/item/@label}">
            {
                for $desc in $template/item/desc
                return
                    utillib:serializeNode($desc)
            }
            </item>
        {
            if ($template/example) then
                (: if multiple root nodes exist, it is never going to be valid xml in an editor, so we wrap it under one :)
                for $example in $template/example
                return
                    <example type="{$example/@type}" caption="{$example/@caption}">
                    {
                        if (count($example/*) gt 1) then (
                            utillib:serializeNode(<art:placeholder xmlns:art="urn:art-decor:example">{$example/node()}</art:placeholder>)
                        )
                        else (
                            utillib:serializeNode($example)/node()
                        )
                    }
                    </example>
            else (
                <example type="neutral" caption=""/>
            )
        }
        {
            if (exists($specialization) and exists($prototype)) then (
                (: if template is empty, check prototype for relevant parts :)
                if ($template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)) then (
                    for $item in $template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
                    return
                        tmapi:mergePrototypeTemplateForEdit($projectPrefix,$item,$prototype,$language,$valueSetList,$templateList, $mode)
                ) else (
                    for $item in $prototype/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
                    return
                        tmapi:recurseItemForEdit($projectPrefix,$item,$language,$valueSetList,$templateList,false(), $mode)
                )
            )
            else (
                for $item in $template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
                return
                    tmapi:recurseItemForEdit($projectPrefix,$item,$language,$valueSetList,$templateList,true(), $mode)
            )
        }
        {
            (: by adding them here, they stay in the clients XML until he closes the form :)
            if ($templateAssociations) then tm:createStaticAssociationElement($templateAssociations, $language) else ()
        }
        </template>
    
    let $result                 := tmapi:prepareTemplateForAPI($result)
    
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(: keep it simple for performance. if the requested valueSet is in the project terminology by ref, don't look further :)
declare %private function tmapi:isValueSetInScope($projectPrefix as xs:string, $ref as attribute()?,$flexibility as attribute()?,$valueSetList as element()*) as attribute()* {
    if (string-length($ref)=0) then ()
    else (
        let $vsElms := $valueSetList[(@id|@name|@ref)=$ref]
        let $vsEff  := if ($flexibility castable as xs:dateTime) then $flexibility else (max($vsElms[@effectiveDate]/xs:dateTime(@effectiveDate)))
       
        return
            if ($vsElms[@effectiveDate=$vsEff]) then (
                attribute vsid {$vsElms[@effectiveDate=$vsEff][1]/(@id|@ref)},
                attribute vsname {$vsElms[@effectiveDate=$vsEff][1]/@name},
                attribute vsdisplayName {$vsElms[@effectiveDate=$vsEff][1]/@displayName}
            )
            else if ($vsElms[@ref]) then (
                attribute vsid {$vsElms[1]/(@id|@ref)},
                attribute vsname {$vsElms[1]/@name},
                attribute vsdisplayName {$vsElms[1]/@displayName}
            )
            else (
                attribute linkedartefactmissing {'true'}
            )
    )
};

(: keep it simple for performance. if the requested template is in the project rules by ref, don't look further :)
declare %private function tmapi:isTemplateInScope($projectPrefix as xs:string, $ref as attribute()?,$flexibility as attribute()?,$templateList as element()*) as attribute()* {
    if (string-length($ref)=0) then ()
    else (
        let $templates  := (tm:getTemplateByRef($ref,if (matches($flexibility,'^\d{4}')) then $flexibility else 'dynamic',$projectPrefix)/*/template[@id])[1]
        
        return
            if ($templates) then (
                attribute tmid {$templates/@id},
                attribute tmdisplayName {if ($templates/@displayName) then $templates/@displayName else $templates/@name}
            ) else (
                attribute linkedartefactmissing {'true'}
            )
    )
};

declare %private function tmapi:getDatatype($datatype as xs:string?, $classification-format as xs:string?) as xs:string? {
    let $classification-format  := if (empty($classification-format)) then 'hl7v3xml1' else $classification-format
    let $datatypes              := $setlib:colDecorCore//supportedDataTypes[@type = $classification-format]
    let $datatypeName           := if (contains($datatype,':')) then substring-after($datatype,':') else ($datatype)
    let $flavor                 := $datatypes//flavor[@name=$datatypeName]
    return
        if ($flavor) then ($flavor/ancestor::dataType[1]/@name) else ($datatype)
};

(: re-write attribute strength CNE is required and CWE is extensible :)
declare %private function tmapi:rewriteStrength($os as xs:string?) as xs:string? {
    let $r := if ($os = 'CNE') then 'required' else if ($os = 'CWE') then 'extensible' else $os
    return $r
};

declare %private function tmapi:recurseItemForEdit($projectPrefix as xs:string, $item as element(),$language as xs:string, $valueSetList as element()*, $templateList as element()*, $selected as xs:boolean, $mode as xs:string) as element()* {
if ($item/name()='attribute') then
    for $att in tm:normalizeAttributes($item)
    return
    <attribute>
    {
        $att/@name,
        $att/@value,
        attribute conformance {
            if ($att[@prohibited='true']) then 'NP' else
            if ($att[@isOptional='true']) then 'O' else
                'R'
        }
        ,
        $att/@datatype,
        $att/@id
        ,
        if ($att/@strength) then (
            attribute strength {tmapi:rewriteStrength($att/@strength)}
        ) else ()
        ,
        if ($att/@datatype) then (
            attribute originalType {tmapi:getDatatype($att/@datatype,$item/ancestor::template/classification/@format)},
            attribute originalStrength {tmapi:rewriteStrength($att/@strength)}
        ) else ()
        ,
        attribute originalConformance {
            if ($selected and $mode = 'new') then 
                if ($att[@prohibited='true']) then 'NP' else
                if ($att[@isOptional='true']) then 'O' else
                    'R'
            else ('O')
        }
        ,
        if ($selected) then (attribute selected {''}) else ()
        ,
        for $desc in $att/desc
        return
        utillib:serializeNode($desc)
        ,
        for $subitem in $att/(* except desc)
        return
            if ($subitem[name()='vocabulary']) then
                <vocabulary>
                {
                    $subitem/(@valueSet|@flexibility|@code|@codeSystem|@codeSystemName|@codeSystemVersion|@displayName|@domain)[not(.='')],
                    tmapi:isValueSetInScope($projectPrefix, $subitem/@valueSet,$subitem/@flexibility,$valueSetList),
                    $subitem/*
                }
                </vocabulary>
            else (
                $subitem
            )
    }
    </attribute>
else if ($item/name()='element') then
    <element>
    {
        $item/(@* except (@strength|@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
        tmapi:isTemplateInScope($projectPrefix, $item/@contains,$item/@flexibility,$templateList)
        ,
        if ($item/@strength) then (
            attribute strength {tmapi:rewriteStrength($item/@strength)}
        ) else ()
        ,
        if ($item/@datatype) then (
            attribute originalType {tmapi:getDatatype($item/@datatype,$item/ancestor::template/classification/@format)},
            attribute originalStrength {tmapi:rewriteStrength($item/@strength)}
        )
        else ()
        ,
        if ($item/@minimumMultiplicity) then () else (
            attribute minimumMultiplicity {''}
        )
        ,
        if ($item/@maximumMultiplicity) then () else (
            attribute maximumMultiplicity {''}
        )
        ,
        attribute originalMin {'0'},
        attribute originalMax {'*'}
        ,
        if ($item/@conformance) then () else (
            let $conformance := 
                if ($item[@isMandatory='true']) then 'R' else
                if ($item[@minimumMultiplicity castable as xs:integer]/@minimumMultiplicity > 0) then 'R' else ('O')
            return
            attribute conformance {$conformance}
        )
        ,
        if ($item/@isMandatory) then () else (
            attribute isMandatory {'false'}
        )
        ,
        if ($item/@strength) then () else (
            attribute strength {''}
        )
        ,
        if ($selected) then (attribute selected {''}) else ()
        ,
        for $desc in $item/desc
        return
            utillib:serializeNode($desc)
        ,
        for $vocabulary in $item/vocabulary
        return
            <vocabulary>
            {
                $vocabulary/(@valueSet|@flexibility|@code|@codeSystem|@codeSystemName|@codeSystemVersion|@displayName|@domain)[not(.='')],
                tmapi:isValueSetInScope($projectPrefix, $vocabulary/@valueSet,$vocabulary/@flexibility,$valueSetList),
                $vocabulary/*
            }
            </vocabulary>
        ,
        $item/property,
        $item/item,
        $item/text,
        for $example in $item/example
        return
            <example type="{$example/@type}" caption="{$example/@caption}">
            {
                if (count($example/*) gt 1) then (
                    utillib:serializeNode(<art:placeholder xmlns:art="urn:art-decor:example">{$example/node()}</art:placeholder>)
                )
                else (
                    utillib:serializeNode($example)/node()
                )
            }
            </example>
        ,
        for $subItem in $item/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
        return
            tmapi:recurseItemForEdit($projectPrefix, $subItem,$language,$valueSetList,$templateList,$selected, $mode)
    }
    </element>
else if ($item/name()='choice') then
    <choice>
    {
        $item/@*,
        if ($item/@minimumMultiplicity) then () else (
            attribute minimumMultiplicity {''}
        )
        ,
        if ($item/@maximumMultiplicity) then () else (
            attribute maximumMultiplicity {''}
        )
        ,
        attribute originalMin {'0'}
        ,
        attribute originalMax {'*'}
        ,
        if ($selected) then (attribute selected {''}) else ()
        ,
        for $desc in $item/desc
        return
        utillib:serializeNode($desc)
        ,
        $item/item,
        for $subItem in $item/(element|choice|include|constraint)
        return
        tmapi:recurseItemForEdit($projectPrefix, $subItem,$language,$valueSetList,$templateList,$selected, $mode)
    }
    </choice>
else if ($item/name()='include') then
    <include>
    {
        $item/(@* except (@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
        tmapi:isTemplateInScope($projectPrefix, $item/@ref,$item/@flexibility,$templateList)
        ,
        if ($item/@minimumMultiplicity) then () else (
            attribute minimumMultiplicity {''}
        )
        ,
        if ($item/@maximumMultiplicity) then () else (
            attribute maximumMultiplicity {''}
        )
        ,
        attribute originalMin {'0'}
        ,
        attribute originalMax {'*'}
        ,
        if ($item/@conformance) then () else (
            let $conformance := if ($item/@minimumMultiplicity='1') then 'R' else ('O')
            return
            attribute conformance {$conformance}
        )
        ,
        if ($item/@isMandatory) then () else (
            attribute isMandatory {'false'}
        )
        ,
        if ($item/@flexibility) then () else (
            attribute flexibility {'dynamic'}
        )
        ,
        if ($selected) then (attribute selected {''}) else ()
        ,
        for $desc in $item/desc
        return
        utillib:serializeNode($desc)
        ,
        $item/item,
        for $example in $item/example
        return
            <example type="{$example/@type}" caption="{$example/@caption}">
            {
                if (count($example/*) gt 1) then (
                    utillib:serializeNode(<art:placeholder xmlns:art="urn:art-decor:example">{$example/node()}</art:placeholder>)
                )
                else (
                    utillib:serializeNode($example)/node()
                )
            }
            </example>
        ,
        for $subItem in $item/constraint
        return
            tmapi:recurseItemForEdit($projectPrefix, $subItem, $language, $valueSetList, $templateList, $selected, $mode)
    }
    </include>
else if ($item/name()='let') then
    <let>
    {
        if ($selected) then (attribute selected {''}) else (),
        attribute name {$item/@name},
        attribute value {$item/@value},
        $item/node()
    }
    </let>
else if ($item/name()='assert') then
    <assert>
    {
        if ($selected) then (attribute selected {''}) else (),
        attribute role {$item/@role},
        attribute test {$item/@test},
        attribute see {$item/@see},
        attribute flag {$item/@flag},
        utillib:serializeNode($item)/node()
    }
    </assert>
else if ($item/name()='report') then
    <report>
    {
        if ($selected) then (attribute selected {''}) else (),
        attribute role {$item/@role},
        attribute test {$item/@test},
        attribute see {$item/@see},
        attribute flag {$item/@flag},
        utillib:serializeNode($item)/node()
    }
    </report>
else if ($item/name()='constraint') then
    <constraint>{if ($selected) then (attribute selected {''}) else (),$item/(@* except @selected),utillib:serializeNode($item)/node()}</constraint>
else()
};

declare %private function tmapi:index-of-node( $nodes as node()* , $nodeToFind as node() )  as xs:integer* {
    for $seq in (1 to count($nodes))
    return $seq[$nodes[$seq] is $nodeToFind]
};

declare %private function tmapi:index-node-in-set( $nodes as node()*, $nodeToFind as node() ) as element()* {
    let $n      :=
        if ($nodeToFind[@name]/name()='attribute') then
            $nodes[name()=$nodeToFind/name()][@name = $nodeToFind/@name]
        else
        if ($nodeToFind[@contains]/name()='element') then
            $nodes[name()=$nodeToFind/name()][@name = $nodeToFind/@name][@contains = $nodeToFind/@contains]
        else
        if ($nodeToFind[@name = ('hl7:templateId', 'cda:templateId')][attribute[@name = 'root'][@value]]/name()='element') then
            $nodes[name()=$nodeToFind/name()][@name = $nodeToFind/@name][attribute[@name = 'root'][@value = $nodeToFind/attribute[@name = 'root']/@value]]
        else
        if ($nodeToFind/name()='element') then
            $nodes[name()=$nodeToFind/name()][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='choice') then
            if ($nodeToFind[element]) then
                $nodes[self::choice][element/@name = $nodeToFind/element/@name]
            else
            if ($nodeToFind[include]) then
                $nodes[self::choice][include/@ref = $nodeToFind/include/@ref]
            else
            if ($nodeToFind[choice]) then
                $nodes[self::choice][choice]
            else
            if ($nodeToFind[@minimumMultiplicity][@maximumMultiplicity]) then
                $nodes[self::choice][@minimumMultiplicity = $nodeToFind/@minimumMultiplicity][@maximumMultiplicity = $nodeToFind/@maximumMultiplicity]
            else
            if ($nodeToFind[@minimumMultiplicity]) then
                $nodes[self::choice][@minimumMultiplicity = $nodeToFind/@minimumMultiplicity]
            else
            if ($nodeToFind[@maximumMultiplicity]) then
                $nodes[self::choice][@maximumMultiplicity = $nodeToFind/@maximumMultiplicity]
            else (
                $nodes[self::choice]
            )
        else
        if ($nodeToFind/name()='include') then
            $nodes[name()=$nodeToFind/name()][@ref = $nodeToFind/@ref]
        else
        if ($nodeToFind/name()='let') then
            $nodes[name()=$nodeToFind/name()][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='defineVariable') then
            $nodes[name()=$nodeToFind/name()][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='assert') then
            $nodes[name()=$nodeToFind/name()][@test = $nodeToFind/@test]
        else
        if ($nodeToFind/name()='report') then
            $nodes[name()=$nodeToFind/name()][@test = $nodeToFind/@test]
        else
        if ($nodeToFind/name()='text') then 
            $nodes[name()=$nodeToFind/name()][. = $nodeToFind]
        else
        if ($nodeToFind/name()='constraint') then
            $nodes[name()=$nodeToFind/name()][@language = $nodeToFind/@language]
        else
        if ($nodeToFind/name()='example') then
            $nodes[name()=$nodeToFind/name()][@language = $nodeToFind/@language][count(preceding-sibling::example) + 1]
        else (
            $nodes[name()=$nodeToFind/name()][string-join(for $att in (@* except (@isOptional|@prohibited|@datatype|@id)) order by name($att) return $att,'')=$nodeToFind/string-join(for $att in (@* except (@isOptional|@prohibited|@datatype|@id)) order by name($att) return $att,'')]
        )
    let $n      :=
        if ($n[2]) then 
            if ($n[@id = $nodeToFind/@id]) then 
                $n[@id = $nodeToFind/@id] 
            (:else 
            if ($n[deep-equal(self::node(), $nodeToFind)]) then 
                $n[deep-equal(self::node(), $nodeToFind)]:) 
            else $n
        else $n
        
    return $n
};

(: Returns paths like this
    element[@name='hl7:informEvent'][empty(@contains)][@id='2.16.840.1.113883.2.4.3.111.3.7.9.16'][1]
:)
declare %private function tmapi:path-to-node ( $nodes as node()* )  as xs:string* { 
    string-join(
        for $item in $nodes/ancestor-or-self::*[not(descendant-or-self::template)]
        return (
            string-join(
                ($item/name(),
                if ($item[@name]/name()='attribute') then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[@name][@contains]/name()='element') then (
                    '[@name=''',replace($item/@name,'''',''''''),''']',
                    '[@contains=''',replace($item/@contains,'''',''''''),''']'
                )
                else
                if ($item[@name = ('hl7:templateId', 'cda:templateId')][attribute[@name = 'root'][@value]]/name()='element') then (
                    '[@name=''',replace($item/@name,'''',''''''),''']','[attribute[@name = ''root''][@value = ''',$item/attribute[@name = 'root']/@value,''']]',
                    '[',count($item/preceding-sibling::element[@name = $item/@name][attribute[@name = 'root'][@value]]) + 1,']'
                )
                else
                if ($item[@name][empty(@contains)]/name()='element') then (
                    '[@name=''',replace($item/@name,'''',''''''),''']','[empty(@contains)]',
                    '[',count($item/preceding-sibling::element[@name = $item/@name][empty(@contains)]) + 1,']'
                )
                else
                if ($item/name()='choice') then (
                    if ($item[element]) then (
                        '[element/@name=(',for $n in $item/element/@name return concat('''',replace($n,'''',''''''),''''),')]'
                    ) else
                    if ($item[include]) then (
                        '[include/@ref=(',for $n in $item/include/@ref return concat('''',replace($n,'''',''''''),''''),')]'
                    ) else
                    if ($item[choice]) then (
                        '[choice]'
                    ) else
                    if ($item[@minimumMultiplicity][@maximumMultiplicity]) then (
                        '[@minimumMultiplicity = ',$item/@minimumMultiplicity,']','[@maximumMultiplicity = ',$item/@maximumMultiplicity,']'
                    ) else
                    if ($item[@minimumMultiplicity]) then (
                        '[@minimumMultiplicity = ',$item/@minimumMultiplicity,']'
                    ) else
                    if ($item[@maximumMultiplicity]) then (
                        '[@maximumMultiplicity = ',$item/@maximumMultiplicity,']'
                    ) else (
                        ''
                    )
                )
                else
                if ($item/name()='include') then (
                    '[@ref=''',replace($item/@ref,'''',''''''),''']'
                )
                else
                if ($item/name()='let') then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item/name()='defineVariable') then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item/name()='assert') then (
                    '[@test=''',replace($item/@test,'''',''''''),''']'
                )
                else
                if ($item/name()='report') then (
                    '[@test=''',replace($item/@test,'''',''''''),''']'
                )
                else
                if ($item/name()='constraint') then (
                    '[@language=''',replace($item/@language,'''',''''''),''']'
                )
                else
                if ($item/name()='example') then (
                    '[',count(preceding-sibling::example) + 1,']'
                )
                else
                if ($item/name()='text') then (
                    '[. = ''',replace(., '''', ''''''),''']'
                )
                else
                if ($item[@name]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[@code][@codeSystem]) then (
                    '[@code=''',replace($item/@code,'''',''''''),''']',
                    '[@codeSystem=''',replace($item/@codeSystem,'''',''''''),''']'
                )
                else
                if ($item[@valueSet]) then (
                    '[@valueSet=''',replace($item/@valueSet,'''',''''''),''']'
                )
                else
                if ($item[@language]) then (
                    '[@language=''',replace($item/@language,'''',''''''),''']'
                )
                else (
                    let $n  := $item/(@* except (@isOptional|@prohibited|@datatype|@id))[1]
                    return (
                        if ($n) then concat('[@',name($n),'=''',replace($n,'''',''''''),''']') else ()
                    )
                )
            ), '')
        )
    , '/')
};

declare %private function tmapi:mergePrototypeTemplateForEdit($projectPrefix as xs:string, $item as element(), $prototype as element(), $language as xs:string, $valueSetList as element()*, $templateList as element()*, $mode as xs:string) as element()* {
(: get corresponding node in prototype :)
let $node           := util:eval-inline($prototype,tmapi:path-to-node($item))
let $node           := if ($node[@id = $item/@id]) then $node[@id = $item/@id][1] else $node[1]
(: get the reversed sequence for previous siblings of the corresponding node in the prototype template, if any :)
let $precedingNodes := reverse($node/preceding-sibling::*)

let $results        := (
    (: if there are no preceding nodes in template, get all preceding nodes from prototype
        else get preceding nodes up to node with same name as preceding template node
    :)
    if (count($item/preceding-sibling::*)=0 and count($node/preceding-sibling::*)>0) then (
        for $precedingNode in $node/preceding-sibling::*
        return
            tmapi:recurseItemForEdit($projectPrefix,$precedingNode,$language,$valueSetList,$templateList,false(), $mode)
    )
    else (
        (: check if there are preceding nodes in prototype that are not in the template
            in order not get retrieve every single node preceding the current one, we first 
            try to find a previous matching node, anything between the previous and the 
            current item should be merged into the current template in this context 
        :)
        let $indexNode  := 
            for $precedingItem in $item/preceding-sibling::*
            let $n      := tmapi:index-node-in-set($precedingNodes, $precedingItem)
            return
                if (count($n) le 1) then ($n) else (
                    error(QName('http://art-decor.org/ns/error', 'MultipleIndexNodes'), concat('Unexpected error. Found multiple index nodes so we do not know what to merge here: ', tmapi:path-to-node($precedingItem), ', index nodes: ', string-join(for $i in $n return concat('''&lt;',$i/name(),string-join(for $att in $i/@* return concat(' ',$att/name(),'=''',data($att),''''),''),'&gt;'''),' ')))
                )
        
        let $index      := if ($indexNode) then tmapi:index-of-node($precedingNodes,$indexNode[last()]) else (count($precedingNodes))
        
        for $n in reverse($precedingNodes)
        let $lin := tmapi:index-of-node($precedingNodes,$n) lt $index and 
                    $n[ self::attribute|
                        self::element|
                        self::choice|
                        self::include|
                        self::let|
                        self::defineVariable|
                        self::assert|
                        self::report|
                        self::constraint ]
        return
            if ($lin) then tmapi:recurseItemForEdit($projectPrefix, $n, $language, $valueSetList, $templateList, false(), $mode) else ()
    )
    ,
    if ($item/name()='attribute') then
        for $att in tm:normalizeAttributes($item)
        return
        <attribute selected="{if ($node) then 'original' else ''}">
        {
            $att/@name,
            $att/@value,
            attribute conformance {
                if ($att[@prohibited='true']) then 'NP' else
                if ($att[@isOptional='true']) then 'O' else
                    'R'
            }
            ,
            $att/@datatype,
            $att/@id
            ,
            if ($node/@datatype) then (
                attribute originalType {tmapi:getDatatype($node/@datatype,$node/ancestor::template/classification/@format)},
                attribute originalStrength {tmapi:rewriteStrength($node/@strength)}
            ) else 
            if ($att/@datatype) then (
                attribute originalType {tmapi:getDatatype($att/@datatype,$att/ancestor::template/classification/@format)},
                attribute originalStrength {tmapi:rewriteStrength($att/@strength)}
            )
            else ()
            ,
            attribute originalConformance {
                if ($node/@prohibited='true') then 'NP' else 
                if ($node/@isOptional='true') then 'O' else 
                if ($node) then 'R' else 
                    'O'
            }
            ,
            for $desc in $att/desc
            return
            utillib:serializeNode($desc)
            ,
            for $subitem in $att/(* except desc)
            return
                if ($subitem[name()='vocabulary']) then
                    <vocabulary>
                    {
                        $subitem/(@valueSet|@flexibility|@code|@codeSystem|@codeSystemName|@codeSystemVersion|@displayName|@domain)[not(.='')],
                        tmapi:isValueSetInScope($projectPrefix, $subitem/@valueSet,$subitem/@flexibility,$valueSetList),
                        $subitem/*
                    }
                    </vocabulary>
                else (
                    $subitem
                )
        }
        </attribute>
    else if ($item/name()='element') then
        <element selected="{if ($node) then 'original' else ''}">
        {
            $item/(@* except (@strength|@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
            tmapi:isTemplateInScope($projectPrefix, $item/@contains,$item/@flexibility,$templateList)
            ,
            if ($item/@strength) then attribute strength {tmapi:rewriteStrength($item/@strength)} else ()
            ,
            if ($node/@datatype) then (
                attribute originalType {tmapi:getDatatype($node/@datatype,$node/ancestor::template/classification/@format)},
                attribute originalStrength {tmapi:rewriteStrength($node/@strength)}
            ) else 
            if ($item/@datatype) then (
                attribute originalType {tmapi:getDatatype($item/@datatype,$item/ancestor::template/classification/@format)},
                attribute originalStrength {tmapi:rewriteStrength($item/@strength)}
            )
            else ()
            ,
            if ($item/@minimumMultiplicity) then () else (
                attribute minimumMultiplicity {''}
            )
            ,
            if ($item/@maximumMultiplicity) then () else (
                attribute maximumMultiplicity {''}
            )
            ,
            if ($node/@minimumMultiplicity) then
                attribute originalMin {$node/@minimumMultiplicity}
            else (
                attribute originalMin {'0'}
            )
            ,
            if ($node/@maximumMultiplicity) then
                attribute originalMax {$node/@maximumMultiplicity}
            else (
                attribute originalMax {'*'}
            )
            ,
            if ($item/@conformance) then () else (
                attribute conformance {'O'}
            )
            ,
            if ($item/@isMandatory) then () else (
                attribute isMandatory {'false'}
            )
            ,
            if ($item/@strength) then () else (
                attribute strength {''}
            )
            ,
            for $desc in $item/desc
            return
                utillib:serializeNode($desc)
            ,
            for $vocabulary in $item/vocabulary
            return
                <vocabulary>
                {
                    $vocabulary/(@valueSet|@flexibility|@code|@codeSystem|@codeSystemName|@codeSystemVersion|@displayName|@domain)[not(.='')],
                    tmapi:isValueSetInScope($projectPrefix, $vocabulary/@valueSet,$vocabulary/@flexibility,$valueSetList),
                    $vocabulary/*
                }
                </vocabulary>
            ,
            $item/property,
            $item/item,
            $item/text,
            for $example in $item/example
            return
                <example type="{$example/@type}" caption="{$example/@caption}">
            {
                if (count($example/*) gt 1) then (
                    utillib:serializeNode(<art:placeholder xmlns:art="urn:art-decor:example">{$example/node()}</art:placeholder>)
                )
                else (
                    utillib:serializeNode($example)/node()
                )
            }
            </example>
            ,
            for $subItem in $item/(attribute|element|choice|include|constraint|let|defineVariable|assert|report)
            return
                tmapi:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $prototype, $language, $valueSetList, $templateList, $mode)
        }
        </element>
    else if ($item/name()='choice') then
        <choice selected="{if ($node) then 'original' else ''}">
        {
            $item/@*,
            if (not($item/@minimumMultiplicity)) then
                attribute minimumMultiplicity {''}
            else ()
            ,
            if (not($item/@maximumMultiplicity)) then
                attribute maximumMultiplicity {''}
            else ()
            ,
            if ($node/@minimumMultiplicity) then
                attribute originalMin {$node/@minimumMultiplicity}
            else (attribute originalMin {'0'})
            ,
            if ($node/@maximumMultiplicity) then
                attribute originalMax {$node/@maximumMultiplicity}
            else (attribute originalMax {'*'})
            ,
            for $desc in $item/desc
            return
            utillib:serializeNode($desc)
            ,
            $item/item,
            for $subItem in $item/(element|choice|include|constraint)
            return
            tmapi:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $prototype, $language, $valueSetList, $templateList, $mode)
        }
        </choice>
    else if ($item/name()='include') then
        <include selected="{if ($node) then 'original' else ''}">
        {
            $item/(@* except (@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
            tmapi:isTemplateInScope($projectPrefix, $item/@ref,$item/@flexibility,$templateList)
            ,
            if (not($item/@minimumMultiplicity)) then
                attribute minimumMultiplicity {''}
            else ()
            ,
            if (not($item/@maximumMultiplicity)) then
                attribute maximumMultiplicity {''}
            else ()
            ,
            if ($node/@minimumMultiplicity) then
                attribute originalMin {$node/@minimumMultiplicity}
            else (
                attribute originalMin {'0'}
            )
            ,
            if ($node/@maximumMultiplicity) then
                attribute originalMax {$node/@maximumMultiplicity}
            else (
                attribute originalMax {'*'}
            )
            ,
            if (not($item/@conformance)) then
                attribute conformance {'O'}
            else ()
            ,
            if (not($item/@isMandatory)) then
                attribute isMandatory {'false'}
            else ()
            ,
            for $desc in $item/desc
            return
            utillib:serializeNode($desc)
            ,
            $item/item
            ,
            for $example in $item/example
            return
                <example type="{$example/@type}" caption="{$example/@caption}">
                {
                    if (count($example/*) gt 1) then (
                        utillib:serializeNode(<art:placeholder xmlns:art="urn:art-decor:example">{$example/node()}</art:placeholder>)
                    )
                    else (
                        utillib:serializeNode($example)/node()
                    )
                }
                </example>
            ,
            for $subItem in $item/constraint
            return
                tmapi:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $prototype, $language, $valueSetList, $templateList, $mode)
        }
        </include>
    else if ($item/name()='let') then
        <let selected="{if ($node) then 'original' else ''}">
        {
            attribute name {$item/@name},
            attribute value {$item/@value},
            $item/node()
        }
        </let>
    else if ($item/name()='defineVariable') then
        <defineVariable selected="{if ($node) then 'original' else ''}">
        {
            attribute name {$item/@name},
            attribute value {$item/@value},
            $item/node()
        }
        </defineVariable>
    else if ($item/name()='assert') then
        <assert selected="{if ($node) then 'original' else ''}">
        {
            attribute role {$item/@role},
            attribute test {$item/@test},
            attribute see {$item/@see},
            attribute flag {$item/@flag},
            utillib:serializeNode($item)/node()
        }
        </assert>
    else if ($item/name()='report') then
        <report selected="{if ($node) then 'original' else ''}">
        {
            attribute role {$item/@role},
            attribute test {$item/@test},
            attribute see {$item/@see},
            attribute flag {$item/@flag},
            utillib:serializeNode($item)/node()
        }
        </report>
    else if ($item/name()='constraint') then
        <constraint selected="{if ($node) then 'original' else ''}">
        {
            $item/(@* except @selected),
            utillib:serializeNode($item)/node()
        }
        </constraint>
    else (),
    (:
       if parent is not 'template' check if there are prototype nodes after the last template node
    :)
    if ($item/following-sibling::*) then () else 
    if ($node) then (
        (: our own current template had a match in the prototype :)
        for $n in ( $node/following-sibling::attribute|
                    $node/following-sibling::element|
                    $node/following-sibling::choice|
                    $node/following-sibling::include|
                    $node/following-sibling::let|
                    $node/following-sibling::defineVariable|
                    $node/following-sibling::assert|
                    $node/following-sibling::report|
                    $node/following-sibling::constraint)
        return
            tmapi:recurseItemForEdit($projectPrefix, $n, $language, $valueSetList, $templateList, false(), $mode)
    )
    else (
        (: since our current node did not match, find any preceding node from our template that matched in the prototype :)
        let $precedingNodes :=
            for $n in ( $item/preceding-sibling::attribute|
                        $item/preceding-sibling::element|
                        $item/preceding-sibling::choice|
                        $item/preceding-sibling::include|
                        $item/preceding-sibling::let|
                        $item/preceding-sibling::defineVariable|
                        $item/preceding-sibling::assert|
                        $item/preceding-sibling::report|
                        $item/preceding-sibling::constraint)
            (: get corresponding node in prototype :)
            let $nodeInPrototype    := util:eval-inline($prototype,tmapi:path-to-node($n))
            let $nodeInPrototype    := if ($nodeInPrototype[@id = $n/@id]) then $nodeInPrototype[@id = $n/@id][1] else $nodeInPrototype[1]
            return
                $nodeInPrototype
        
        (: add all nodes from the prototype template following the last match that we had, to the end of the current template :)
        for $n in ( $precedingNodes[last()]/following-sibling::attribute|
                    $precedingNodes[last()]/following-sibling::element|
                    $precedingNodes[last()]/following-sibling::choice|
                    $precedingNodes[last()]/following-sibling::include|
                    $precedingNodes[last()]/following-sibling::let|
                    $precedingNodes[last()]/following-sibling::defineVariable|
                    $precedingNodes[last()]/following-sibling::assert|
                    $precedingNodes[last()]/following-sibling::report|
                    $precedingNodes[last()]/following-sibling::constraint)
         return
            tmapi:recurseItemForEdit($projectPrefix, $n, $language, $valueSetList, $templateList, false(), $mode)
    )
)

for $result in $results
return
    if ($result/attribute) then (
        element {name($result)} {
            $result/@*,
            $result/attribute[last()] | $result/attribute[last()]/preceding-sibling::* except $result/element,
            $result/attribute[last()]/preceding-sibling::element | $result/attribute[last()]/following-sibling::node()
        }
    )
    else (
        $result
    )
};

(:get prototype for editor with normalized attributes:)
declare %private function tmapi:getPrototype($id as xs:string?, $flexibility as xs:string?, $projectPrefix as xs:string?) as element(template)? {
    if (empty($id)) then
        ()
    else
    if (empty($projectPrefix)) then
        (tm:getTemplateById($id, if (matches($flexibility,'^\d{4}')) then $flexibility else 'dynamic')//template[@id][@effectiveDate])[1]
    else (
        (tm:getTemplateById($id, if (matches($flexibility,'^\d{4}')) then $flexibility else 'dynamic', $projectPrefix)//template[@id][@effectiveDate])[1]
    )
};

(:~ Update DECOR template

@param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
@param $id                       - required. the id for the template to update
@param $effectiveDate            - required. the effectiveDate for the template to update
@param $request-body             - required. json body containing new template structure
@return template structure 
@since 2020-05-03
:)
declare function tmapi:putTemplate($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                           := utillib:getBodyAsXml($request?body, 'template', ())
    let $deletelock                     := $request?parameters?deletelock = true()
    
    (:let $s                              := xmldb:store('/db/apps/decor/tmp', 'ttt.xml', $data):) 
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $return                         := tmapi:putTemplate($authmap, $id, $effectiveDate, $data, $deletelock)
    return (
        roaster:response(200, $return)
    )

};

(:~ Update DECOR template parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "e.g. [/statusCode|/expirationDate|/officialReleaseDate|/canonicalUri|/versionLabel|/name|/displayName|/experimental|/desc|/publishingAuthority|/copyright|/completeCodeSystem|/conceptList]", "value": "[string|object]" }

where

* op - add & replace (statusCode, expirationDate, officialReleaseDate, canonicalUri, versionLabel, name, displayName, experimental, desc, publishingAuthority, copyright, completeCodeSystem, conceptList) or remove (desc, publishingAuthority, copyright, completeCodeSystem, conceptList)

* path - see above

* value - string when the path is not /. object when path is /
@param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
@param $id                       - required. the id for the issue to update 
@param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
@return template structure
@since 2020-05-03
@see http://tools.ietf.org/html/rfc6902
:)
declare function tmapi:patchTemplate($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    (:let $s                      := xmldb:store('/db/apps/decor/tmp', 'ttt.xml', $data) :)
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                 := tmapi:patchTemplate($authmap, string($id), string($effectiveDate), $data)
    return (
        roaster:response(200, $return)
    )

};

(:~ Update DECOR templateAssociation parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "e.g. [/concept]", "value": "[object]" }

where

* op - add, replace or remove concept

* path - see above

* value - concept object
@param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
@param $id                       - required. the id for the issue to update 
@param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
@return template structure
@since 2020-05-03
@see http://tools.ietf.org/html/rfc6902
:)
declare function tmapi:patchTemplateAssociation($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    (:let $s                      := xmldb:store('/db/apps/decor/tmp', 'ttt.xml', $data) :)
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                 := tmapi:patchTemplateAssociation($authmap, string($id), string($effectiveDate), $data)
    return (
        roaster:response(200, $return)
    )

};

(:~ Update template statusCode, versionLabel, expirationDate and/or officialReleaseDate

@param $authmap required. Map derived from token
@param $id required. DECOR concept/@id to update
@param $effectiveDate required. DECOR concept/@effectiveDate to update
@param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
@param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
@param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
@param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
@param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
@param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
@return list object with success and/or error elements or error
:)
declare function tmapi:putTemplateStatus($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $recurse                        := $request?parameters?recurse = true()
    let $list                           := $request?parameters?list = true()
    let $statusCode                     := $request?parameters?statusCode
    let $versionLabel                   := $request?parameters?versionLabel
    let $expirationDate                 := $request?parameters?expirationDate
    let $officialReleaseDate            := $request?parameters?officialReleaseDate
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    return
        tmapi:setTemplateStatus($authmap, $id, $effectiveDate, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
};

(:~ Central logic for creating an initial template

@param $authmap             - required. Map derived from token
@param $request-body        - required. json body containing new template structure
@param $lock                - required. true = create lock on result. false = do not create a lock on result
@return template structure
:)
declare function tmapi:postTemplate($authmap as map(*), $projectPrefix as xs:string, $data as element(), $keepIds as xs:boolean, $lock as xs:boolean) as element(template) {

    let $storedTemplate                 := 
        if ($data/@id[not(. = '')]) then 
            $setlib:colDecorData//template[@id = $data/@id][@effectiveDate = $data/@effectiveDate] |
            $setlib:colDecorCache//template[@id = $data/@id][@effectiveDate = $data/@effectiveDate]
        else ()
    let $originalTemplate               :=
        if ($data/@ref[not(. = '')]) then
            tmapi:getPrototype($data/@ref, if ($data/@flexibility[not(. = '')]) then $data/@flexibility else 'dynamic', $projectPrefix)
        else () 
    let $decor                          := utillib:getDecorByPrefix($projectPrefix)
    
    let $check                          :=
        if ($storedTemplate) then 
            error($errors:BAD_REQUEST, 'Template with id ''' || ($data/@id)[1] || ''' and effectiveDate ''' || $data/@effectiveDate || ''' already exists in project ' || string-join($storedTemplate/ancestor::decor/project/@prefix, ', '))
        else ()
    let $check                          :=
        if ($data/@ref[not(. = '')]) then 
            if ($originalTemplate) then () else (
                error($errors:BAD_REQUEST, 'Template with id ''' || ($data/@ref)[1] || ''' and effectiveDate ''' || $data/@flexibility || ''' not found in project ' || $projectPrefix)
            )
        else ()
    let $check                          :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-RULES)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to add templates in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    
    (:relevant only for mode=new and when a concept is connected somewhere, but doesn't hurt to generate for all cases:)
    let $newTemplateElementId           := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TEMPLATEELEMENT, ())[1]/@id
    
    let $preparedTemplate               := 
        if ($originalTemplate) then
            <template ref="{$data/@ref}" name="{$originalTemplate/@name}" displayName="{($originalTemplate/@displayName, $originalTemplate/@name)[not(. = '')][1]}"/>
        else
        if ($keepIds and $data/@id) then
            tmapi:prepareIncomingTemplateForUpdate($decor, $data, $storedTemplate, 'version', $newTemplateElementId)
        else (
            tmapi:prepareIncomingTemplateForUpdate($decor, $data, $storedTemplate, 'new', $newTemplateElementId)
        )
    (: Note: this step also rolls up multiple templateAssociation elements into one. This is unlikely to occur unless manually. :)
    let $storedTemplateAssociations     := $decor//templateAssociation[@templateId = $data/@id][@effectiveDate = $data/@effectiveDate]
    let $preparedTemplateAssociation    := if ($preparedTemplate[@id]) then tmapi:prepareTemplateAssociationForUpdate($storedTemplateAssociations, $preparedTemplate, $data, $newTemplateElementId, 'new') else ()
    
    (:let $intention              := if ($storedTemplate[@statusCode = ('active', 'final')]) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-TEMPLATE, $projectPrefix, $intention, $storedTemplate):)
    
    (: now update the template :)
    let $storedReference                := if ($preparedTemplate[@id]) then () else $decor/rules/template[@ref = $preparedTemplate/@ref]
    let $update                         := if ($storedReference) then update replace $storedReference with $preparedTemplate else update insert $preparedTemplate into $decor/rules
    let $addids                         := if ($preparedTemplate[@id]) then tm:addTemplateElementAndAttributeIds($decor, $preparedTemplate/@id, $preparedTemplate/@effectiveDate) else ()
    let $storedTemplate                 := if ($preparedTemplate[@id]) then $decor/rules/template[@id = $preparedTemplate/@id][@effectiveDate = $preparedTemplate/@effectiveDate] else $decor/rules/template[@ref = $preparedTemplate/@ref]
    let $delete                         := update delete $storedTemplate//@json:array
    
    (: now rewrite any templateAssociations. We don't want broken links into elements or attributes that do not exist anymore :)
    let $update                         := 
        if ($storedTemplateAssociations) then (
            update delete $storedTemplateAssociations,
            update insert $preparedTemplateAssociation preceding $storedTemplate
        )
        else
        if ($preparedTemplateAssociation) then (
            update insert text {'&#x0a;        '} preceding $storedTemplate,
            update insert comment {concat(' ',$preparedTemplate/@displayName,' ')} preceding $storedTemplate,
            update insert $preparedTemplateAssociation preceding $storedTemplate
        ) else ()
    
    let $newlock                        := if ($lock) then decorlib:setLock($authmap, $preparedTemplate/@id, $preparedTemplate/@effectiveDate, false()) else ()
    
    let $result                         := 
        if ($lock) then
            tmapi:getTemplateForEdit($authmap, $projectPrefix, 'edit', $preparedTemplate/@id, $preparedTemplate/@effectiveDate, true(), true(), false())
        else
        if ($preparedTemplate[@id]) then
            tmapi:getTemplate($preparedTemplate/@id, $preparedTemplate/@effectiveDate, $projectPrefix, (), ())
        else (
            tmapi:getTemplate($data/@ref, $data/@flexibility, $projectPrefix, (), ())
        )
        
    return
        element {name($result[@id][1])} {
            $result[@id][1]/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result[@id][1]/*)
        }
};

(:~ Central logic for updating an existing template

@param $authmap                 - required. Map derived from token
@param $id                       - required. the id for the template to update
@param $effectiveDate            - required. the effectiveDate for the template to update
@param $request-body             - required. json body containing new template structure
@return template structure
:)
declare function tmapi:putTemplate($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(), $deletelock as xs:boolean) as element(template) {

    let $storedTemplate         := $setlib:colDecorData//template[@id = $data/@originalId][@effectiveDate = $effectiveDate] 
    let $decor                  := $storedTemplate/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $lock                   := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())
    
    let $check                  :=
        if ($storedTemplate) then () else (
            error($errors:BAD_REQUEST, 'Template with id ''' || ($data/@originalId, $id)[1] || ''' and effectiveDate ''' || $effectiveDate || ''' does not exist')
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-RULES)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify templates in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this template (anymore). Get a lock first.'))
        )
    let $check                  :=
        if ($storedTemplate[@statusCode = $tmapi:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, concat('Template cannot be updated while it has one of status: ', string-join($tmapi:STATUSCODES-FINAL, ', '), if ($storedTemplate/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
        else ()
    let $check                  :=
        if ($data[@originalId = $id] | $data[empty(@originalId)][@id = $id]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no template id or the same template id as the template id ''', $id, ''' used for updating. Found in request body: ', ($data/@originalId, $data/@id, 'null')[1]))
        )
    let $check                  :=
        if ($data[@effectiveDate = $effectiveDate] | $data[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no template effectiveDate or the same template effectiveDate as the template effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($data/@effectiveDate, 'null')[1]))
        )
    
    (:relevant only for mode=new and when a concept is connected somewhere, but doesn't hurt to generate for all cases:)
    let $newTemplateElementId   := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TEMPLATEELEMENT, ())[1]/@id
    
    let $preparedTemplate       := tmapi:prepareIncomingTemplateForUpdate($decor, $data, $storedTemplate, 'edit', $newTemplateElementId)
    (: Note: this step also rolls up multiple templateAssociation elements into one. This is unlikely to occur unless manually. :)
    let $storedTemplateAssociations     := $decor//templateAssociation[@templateId = $data/@originalId][@effectiveDate = $data/@effectiveDate]
    let $preparedTemplateAssociation    := tmapi:prepareTemplateAssociationForUpdate($storedTemplateAssociations, $preparedTemplate, $data, $newTemplateElementId, 'edit')
    
    let $intention              := if ($storedTemplate[@statusCode = ('active', 'final')]) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-TEMPLATE, $projectPrefix, $intention, $storedTemplate)
    
    (: now update the template :)
    let $update                 := update replace $storedTemplate with $preparedTemplate
    let $addids                 := tm:addTemplateElementAndAttributeIds($decor, $preparedTemplate/@id, ($preparedTemplate/@effectiveDate)[1])
    let $delete                 := update delete $storedTemplate//@json:array
    
    (: now rewrite any templateAssociations. We don't want broken links into elements or attributes that do not exist anymore :)
    let $update                 := 
        if ($storedTemplateAssociations) then (
            update delete $storedTemplateAssociations,
            update insert $preparedTemplateAssociation preceding $storedTemplate
        )
        else (
            update insert text {'&#x0a;        '} preceding $storedTemplate,
            update insert comment {concat(' ',$preparedTemplate/@displayName,' ')} preceding $storedTemplate,
            update insert $preparedTemplateAssociation preceding $storedTemplate
        )
    
    let $deleteLock             := if ($deletelock) then update delete $lock else ()
    let $result                 := 
        if ($deletelock) then
            tmapi:getTemplate($preparedTemplate/@id, $preparedTemplate/@effectiveDate, $projectPrefix, (), ())
        else (
            tmapi:getTemplateForEdit($authmap, $projectPrefix, 'edit', $preparedTemplate/@id, $preparedTemplate/@effectiveDate, true(), true(), false())
        )
    
    return
        element {name($result[@id][1])} {
            $result[@id][1]/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result[@id][1]/*)
        }
};

(:~ Retrieves DECOR template history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function tmapi:getTemplateHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-TEMPLATE, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-TEMPLATE}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Retrieves DECOR template usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2021-12-16
:)
declare function tmapi:getTemplateUsage($request as map(*)) {
    
    let $projectPrefix                  := $request?parameters?prefix
    (:let $projectVersion                 := $request?parameters?release
    let $projectLanguage                := $request?parameters?language:)
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $results                        := tmapi:getDependenciesAndUsage($projectPrefix, (), (), $id, $effectiveDate)
    (:let $mostRecent                     := tmapi:getTemplate($id, $effectiveDate, $projectPrefix, $projectVersion, $projectLanguage)/@effectiveDate
    let $isMostRecent                   := $mostRecent = $effectiveDate or empty($effectiveDate)
    
    let $allTransactions                := utillib:getTransactionsByTemplate($id, $effectiveDate)
    
    let $results                        := (
        for $item in $allTransactions
        let $trid := $item/@id
        let $tred := $item/@effectiveDate
        group by $trid, $tred
        return
            utillib:doTransactionAssociation($ds, $item[1])
    ):)
    
    let $count                          := count($results)
    let $max                            := $count
    let $allcnt                         := $count
    
    return
        <list artifact="USAGE" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(: ======== ADD TEMPLATE ELEMENT/ATTRIBUTE IDS ============ :)
declare function tmapi:postAddTemplateIds($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $templateId             := $request?parameters?id
    let $templateEffectiveDate  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    
    let $decorProjects          := if (utillib:isOid($project)) then utillib:getDecorById($project) else utillib:getDecorByPrefix($project)
    
    let $check                  :=
        if ($decorProjects) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
    
    let $update                 := 
        for $decor in $decorProjects
        let $check                  :=
            if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-RULES)) then () else (
                error($errors:FORBIDDEN, 'User ' || $authmap?name || ' does not have sufficient permissions to edit templates in project ' || $decor/project/@refix || '. You have to be an active author in the project.')
            )
        return tm:addTemplateElementAndAttributeIds($decor, $templateId, $templateEffectiveDate)
    
    return
        roaster:response(200, ())
};
(: ======== ADD TEMPLATE ELEMENT/ATTRIBUTE IDS ============ :)

(:~ Central logic for patching an existing template

@param $authmap         - required. Map derived from token
@param $id              - required. DECOR template/@id to update
@param $effectiveDate   - required. DECOR template/@effectiveDate to update
@param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
@return template object
:)
declare function tmapi:patchTemplate($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) as element(template) {

    let $storedTemplate         := $setlib:colDecorData//template[@id = $id][@effectiveDate = $effectiveDate]
    let $decor                  := $storedTemplate/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $lock                   := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())
    
    let $check                  :=
        if ($storedTemplate) then () else (
            error($errors:BAD_REQUEST, 'Template with id ' || $id || ' and effectiveDate ' || $effectiveDate || ''' does not exist')
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify templates in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if ($storedTemplate[@statusCode = $tmapi:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $tmapi:STATUSCODES-FINAL)]) then () else
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('Template cannot be patched while it has one of status: ', string-join($tmapi:STATUSCODES-FINAL, ', '), if ($storedTemplate/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
    let $check                  :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this template (anymore). Get a lock first.'))
        )
    let $check                  :=
        if ($data[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $data/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $check                  :=
        for $param in $data/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($storedTemplate, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:templatestatusmap, string($storedTemplate/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri'
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/name' 
            case '/displayName' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/isClosed' return (
                if ($op = 'remove') then () else 
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false.'
                )
            )
            case '/desc' 
            case '/copyright'
            case '/purpose' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then
                    if ($param/value/*[name() = $pathpart][matches(@language, '[a-z]{2}-[A-Z]{2}')]) then
                        if ($param[@op = 'remove'] | $param/value/*[name() = $pathpart][.//text()[not(normalize-space() = '')]]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have contents.'
                        )
                    else (
                        'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have a language with pattern ll-CC.'
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/classification' return (
                if ($param[count(value/classification) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/classification[@format]) then
                            if ($param/value/classification[@format = $tmapi:TEMPLATE-FORMATS/enumeration/@value]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Classification has unsupported .format value ' || string-join($param/value/classification/@format, ', ') || '. SHALL be one of: ' || string-join($tmapi:TEMPLATE-FORMATS/enumeration/@value, ', ')
                            )
                        else (),
                        if ($param/value/classification[@type]) then
                            if ($param/value/classification[@type = $tmapi:TEMPLATE-TYPES/enumeration/@value]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Classification has unsupported .type value ' || string-join($param/value/classification/@type, ', ') || '. SHALL be one of: ' || string-join($tmapi:TEMPLATE-TYPES/enumeration/@value, ', ')
                            )
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/relationship' return (
                if ($param[count(value/relationship) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/relationship[@type = $tmapi:RELATIONSHIP-TYPES/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Relationship has unsupported .type value ' || string-join($param/value/relationship/@type, ', ') || '. SHALL be one of: ' || string-join($tmapi:RELATIONSHIP-TYPES/enumeration/@value, ', ')
                        ),
                        if ($param/value/relationship[count(@template | @model) = 1]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship SHALL have exactly one of template or model property'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/context' return (
                if ($param[count(value/context) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/context[count(@id | @path) = 1]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Context SHALL have exactly one of id or path property'
                        ),
                        if ($param/value/context[empty(@id)] | $param/value/context[@id = ('*', '**')]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Context SHALL be one of * or **'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/item' return (
                if ($param[count(value/item) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/item[@label]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Item SHALL have label property'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/publishingAuthority' return (
                if ($param[count(value/publishingAuthority) = 1]) then (
                    if ($param/value/publishingAuthority[@id]) then
                        if ($param/value/publishingAuthority[utillib:isOid(@id)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || ''' publishingAuthority.id SHALL be an OID is present. Found ' || string-join($param/value/publishingAuthority/@id, ', ')
                        )
                    else (),
                    if ($param/value/publishingAuthority[@name[not(. = '')]]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || ''' publishingAuthority.name SHALL be a non empty string.'
                    ),
                    let $unsupportedtypes := $param/value/publishingAuthority/addrLine/@type[not(. = $tmapi:ADDRLINE-TYPE/enumeration/@value)]
                    return
                        if ($unsupportedtypes) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Publishing authority has unsupported addrLine.type value(s) ' || string-join($unsupportedops, ', ') || '. SHALL be one of: ' || string-join($tmapi:ADDRLINE-TYPE/enumeration/@value, ', ')
                        else ()
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single publishingAuthority under value. Found ' || count($param/value/publishingAuthority)
                )
            )
            case '/example' return (
                let $unsupportedtypes := $param/value/example/@type[not(. = $tmapi:EXAMPLE-TYPES/enumeration/@value)]
                return
                    if ($unsupportedtypes) then
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Example has unsupported type value(s) ' || string-join($unsupportedops, ', ') || '. SHALL be one of: ' || string-join($tmapi:EXAMPLE-TYPES/enumeration/@value, ', ')
                    else ()
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     let $check                 :=
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )
    
    let $intention              := if ($storedTemplate[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-TEMPLATE, $projectPrefix, $intention, $storedTemplate)
    
    let $update                 :=
        for $param in $data/parameter
        return
            switch ($param/@path)
            case '/statusCode'
            case '/expirationDate'
            case '/officialReleaseDate'
            case '/canonicalUri'
            case '/versionLabel'
            case '/name'
            case '/displayName'
            case '/isClosed' return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := attribute {$attname} {$param/@value}
                let $stored   := $storedTemplate/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplate
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/desc' return (
                (: only one per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/desc)
                let $stored   := $storedTemplate/*[name() = $elmname][@language = $param/value/desc/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplate
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/publishingAuthority' return (
                (: only one possible :)
                let $new      := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
                let $stored   := $storedTemplate/publishingAuthority
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplate
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/purpose' return (
                (: only one possible per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/purpose)
                let $stored   := $storedTemplate/*[name() = $elmname][@language = $param/value/purpose/@language]
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplate
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/copyright' return (
                (: only one per language :)
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/copyright)
                let $stored   := $storedTemplate/copyright[@language = $param/value/copyright/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplate
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/classification' return (
                (: only one per type or format :)
                let $new      := tmapi:prepareClassificationForUpdate($param/value/classification)
                let $stored   := $storedTemplate/classification
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then (update insert $new preceding $stored[1], update delete $stored) else (update insert $new into $storedTemplate)
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/relationship' return (
                (: only one per template or model, insert before others :)
                let $new      := tmapi:prepareRelationshipForUpdate($param/value/relationship)
                let $stored   := $storedTemplate/relationship[@template = $new/@template] | $storedTemplate/relationship[@model = $new/@model]
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then (update insert $new preceding $stored[1], update delete $stored) else (update insert $new into $storedTemplate)
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/context' return (
                (: only one :)
                let $new      := tmapi:prepareContextForUpdate($param/value/context)
                let $stored   := $storedTemplate/context
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplate
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/item' return (
                (: only one :)
                let $new      := tmapi:prepareItemForUpdate($param/value/item)
                let $stored   := $storedTemplate/item
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplate
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/example' return (
                (: only one :)
                let $new      := 
                    for $example in $param/value/example
                    return
                        <example>
                        {
                            $example/@type[not(. = '')],
                            $example/@caption[not(. = '')],
                            $example/node()
                        }
                        </example>
                let $stored   := update delete $storedTemplate/example
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($new) then update insert $new into $storedTemplate else ()
                case 'remove' return ()
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $preparedTemplate       := tmapi:prepareTemplateForUpdate($storedTemplate)
    
    let $update                 := update replace $storedTemplate with $preparedTemplate
    let $update                 := update delete $lock
    
    let $result                 := tmapi:getTemplate($preparedTemplate/@id, $preparedTemplate/@effectiveDate, $projectPrefix, (), ())
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Central logic for patching an existing templateAssociation

@param $authmap         - required. Map derived from token
@param $id              - required. DECOR template/@id to update
@param $effectiveDate   - required. DECOR template/@effectiveDate to update
@param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
@return templateAssociation object
:)
declare function tmapi:patchTemplateAssociation($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) as element(template) {

    let $storedTemplate         := $setlib:colDecorData//template[@id = $id][@effectiveDate = $effectiveDate]
    let $decor                  := $storedTemplate/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
        
    let $check                  :=
        if ($storedTemplate) then () else (
            error($errors:BAD_REQUEST, 'Template with id ' || $id || ' and effectiveDate ' || $effectiveDate || ''' does not exist')
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify templates in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if ($data[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $data/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $check                  :=
        for $param in $data/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/concept' return (
                if ($param[count(value/concept) = 1]) then (
                    if ($op = 'remove') then () else 
                    if ($param/value/concept[@ref][@effectiveDate][@elementId]) then (
                        if (utillib:getConcept($param/value/concept/@ref, $param/value/concept/@effectiveDate)) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. TemplateAssociation concept id ''' || $param/value/concept/@ref || ''' effectiveDate ''' || $param/value/concept/@effectiveDate || ''' not found.'
                        ),
                        if ($storedTemplate//attribute[@id = $param/value/concept/@elementId] | $storedTemplate//element[@id = $param/value/concept/@elementId]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. TemplateAssociation attribute or element id ''' || $param/value/concept/@elementId || ''' not found.'
                        )
                    ) else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. A templateAssociation concept SHALL have a ref, effectiveDate and elementId.'
                    )
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single concept under value. Found ' || count($param/value/concept)
                )  
            )
              
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     let $check                 :=
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )
    
    let $update                 :=
        for $param in $data/parameter
        return
            switch ($param/@path)
            case '/concept' return (
                let $paramValue := $param/value/concept
                let $preparedTemplateAssociation := <templateAssociation templateId="{$storedTemplate/@id}" effectiveDate="{$storedTemplate/@effectiveDate}"/>
                let $new      := 
                    <concept>
                    {
                        $paramValue/@ref,
                        $paramValue/@effectiveDate,
                        ($paramValue/@elementId, $paramValue/@elementPath)[1]  
                    }
                    </concept>
                let $storedTemplateAssociation := $decor//templateAssociation[@templateId = $storedTemplate/@id][@effectiveDate = $storedTemplate/@effectiveDate] 
                let $storedTemplateAssociation := 
                    if ($storedTemplateAssociation) then ($storedTemplateAssociation) else (
                        let $update := update insert text {'&#x0a;        '} preceding $storedTemplate
                        let $update := update insert comment {concat(' ',$storedTemplate/@displayName,' ')} preceding $storedTemplate
                        let $update := update insert $preparedTemplateAssociation preceding $storedTemplate
                        
                        return
                        $decor//templateAssociation[@templateId = $storedTemplate/@id][@effectiveDate = $storedTemplate/@effectiveDate]
                    )
                let $stored                    := $storedTemplateAssociation/concept[@ref = $paramValue/@ref][@effectiveDate = $paramValue/@effectiveDate]
                let $stored                    := 
                    if ($paramValue/@elementId) then ($stored[@elementId = $paramValue/@elementId]) else 
                    if ($paramValue/@elementPath) then ($stored[@elementPath = $paramValue/@elementPath]) else ($stored)
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $storedTemplateAssociation[1]
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
    
    let $result                 := $decor//templateAssociation[@templateId = $storedTemplate/@id][@effectiveDate = $storedTemplate/@effectiveDate]
    (:let $result                 :=
        for $staticAssociation in templ:createStaticAssociationElement($result, $decor/project/@defaultLanguage)
        return
            <templateAssociation templateId="{$storedTemplate/@id}" effectiveDate="{$storedTemplate/@effectiveDate}">
            {
                for $concept in $staticAssociation/*
                return
                    <concept>{$concept/@*, $concept/node()}</concept>
            }
            </templateAssociation>:)
    
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
};

(:~ Central logic for patching an existing transaction statusCode, versionLabel, expirationDate and/or officialReleaseDate

@param $authmap required. Map derived from token
@param $id required. DECOR concept/@id to update
@param $effectiveDate required. DECOR concept/@effectiveDate to update
@param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
@param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
@param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
@param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
@param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
@param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
@return list object with success and/or error elements or error
:)
declare function tmapi:setTemplateStatus($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $recurse as xs:boolean, $listOnly as xs:boolean, $newStatusCode as xs:string?, $newVersionLabel as xs:string?, $newExpirationDate as xs:string?, $newOfficialReleaseDate as xs:string?) as element(list) {
    (:get object for reference:)
    let $object                 := 
        if ($effectiveDate castable as xs:dateTime) then 
            $setlib:colDecorData//template[@id = $id][@effectiveDate = $effectiveDate]
        else (
            let $set := $setlib:colDecorData//template[@id = $id]
            return
                $set[@effectiveDate = max($set/xs:dateTime(@effectiveDate))]
        )
    let $decor                  := $object/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($object) = 1) then () else
        if ($object) then
            error($errors:SERVER_ERROR, 'Template id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($object) || ' were found in: ' || string-join(distinct-values($object/ancestor::decor/project/@prefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Template id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-SCENARIOS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify scenarios/transactions in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    
    let $templateChain          :=
        if ($recurse) then (
            let $templatechain          := tm:getTemplateList($object/@id, (), $object/@effectiveDate, $projectPrefix, (), false(), $tm:TREETYPELIMITEDMARKED)
                  
            for $t in $templatechain//ref/parent::template
            (:template/@ref will not match this. match only project local objects:)
            return 
                $decor//template[@id = $t/@id][@effectiveDate = $t/@effectiveDate]
        ) 
        else (
            $object
        )
    let $testUpdate             :=
        for $object in $templateChain
        return
            utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, true(), $projectPrefix)
    let $results                :=
        if ($testUpdate[self::error]) then $testUpdate else
        if ($listOnly) then $testUpdate else (
            for $object in $templateChain
            return
                utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, false(), $projectPrefix)
        )
    return
    if ($results[self::error] and not($listOnly)) then
        error($errors:BAD_REQUEST, string-join(
            for $e in $results[self::error]
            return
                $e/@itemname || ' id ''' || $e/@id || ''' effectiveDate ''' || $e/@effectiveDate || ''' cannot be updated: ' || data($e)
            , ' ')
        )
    else (
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            (: max 2 arrays: one with error and one with success :)
            utillib:addJsonArrayToElements($results)
        }
        </list>
    )
};

(:~ Returns a list of zero or more templates
   
   @param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only
   @param $projectVersion   - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
   @param $id               - optional. Identifier of the template to retrieve
   @param $name             - optional. Name of the template to retrieve (valueSet/@name)
   @param $flexibility      - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
   @param $treetype         - optional. Default $tm:TREETYPELIMITEDMARKED
   @param $resolve          - optional. Boolean. Default true. If true and $doV2 is false, returns valueSetList.valueSet* where every valueSet has @id|@ref and a descending by 
   effectiveDate list of matching valueSets. If false and $doV2 is false, returns valueSetList.valueSet* containing only the latest version of the valueSet (or ref if no versions 
   exist in $projectPrefix)<br/>If true and $doV2 is true, returns valueSetList.valueSet* where every valueSet has @id|@ref and a descending by effectiveDate list of matching 
   valueSets. If false and $doV2 is false, returns valueSetList.valueSet* containing only the latest version of the valueSet (or ref if no versions exist in $projectPrefix)
   @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
   @since 2013-06-14
:)
declare %private function tmapi:getTemplateList($projectPrefix as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $objectid as xs:string?, $objectnm as xs:string?, $objected as xs:string?, $treetype as xs:string*, $resolve as xs:boolean) {

    let $objectid               := $objectid[not(. = '')]
    let $objectnm               := $objectnm[not(. = '')]
    let $objected               := $objected[not(. = '')]
    let $projectPrefix          := $projectPrefix[not(. = '')]
    let $projectVersion         := $projectVersion[not(. = '')]
    let $treetype               := if (empty($treetype)) then $tm:TREETYPELIMITEDMARKED else $treetype
    
    return
        tm:getTemplateList($objectid, $objectnm, $objected, $projectPrefix, $projectVersion, $resolve, $treetype)
};

(:~ Returns a list of zero or more templateAssociation objects
   
   @param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only
   @param $projectVersion   - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
   @param $id               - optional. Identifier of the template to retrieve
   @param $effectiveDate    - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version
   @param $treetype         - optional. Default $tm:TREETYPELIMITEDMARKED
   @param $resolve          - optional. Boolean. Default true. If true and $doV2 is false, returns valueSetList.valueSet* where every valueSet has @id|@ref and a descending by 
   effectiveDate list of matching valueSets. If false and $doV2 is false, returns valueSetList.valueSet* containing only the latest version of the valueSet (or ref if no versions 
   exist in $projectPrefix)<br/>If true and $doV2 is true, returns valueSetList.valueSet* where every valueSet has @id|@ref and a descending by effectiveDate list of matching 
   valueSets. If false and $doV2 is false, returns valueSetList.valueSet* containing only the latest version of the valueSet (or ref if no versions exist in $projectPrefix)
   @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
   @since 2013-06-14
:)
declare %private function tmapi:getTemplateAssociationList($projectPrefix as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $objectid as xs:string?, $objected as xs:string?) as element(list) {
    let $objectid               := $objectid[not(. = '')]
    let $objected               := $objected[not(. = '')]
    let $projectPrefix          := $projectPrefix[not(. = '')]
    let $projectVersion         := $projectVersion[not(. = '')]
    
    let $decor                  :=
        if (utillib:isOid($projectPrefix)) then 
            utillib:getDecorById($projectPrefix, $projectVersion, $projectLanguage) 
        else (
            utillib:getDecorByPrefix($projectPrefix, $projectVersion, $projectLanguage)
        )
    
    let $check                  :=
        if (count($decor) = 1) then () else
        if (count($decor) = 0) then
            if (empty($projectVersion)) then
                error($errors:BAD_REQUEST, 'Project with prefix ' || $projectPrefix || ' not found')
            else (
                error($errors:BAD_REQUEST, 'Project with prefix ' || $projectPrefix || ' release ' || $projectVersion || ' language ''' || $projectLanguage || ''' not found')
            )
        else (
            if (empty($projectVersion)) then
                error($errors:SERVER_ERROR, 'Found ' || count($decor) || ' instances of project with prefix ' || $projectPrefix || '. Expected 0..1. Alert your administrator as this should not be possible.')
            else (
                error($errors:SERVER_ERROR, 'Found ' || count($decor) || ' instances of project with prefix ' || $projectPrefix || ' release ' || $projectVersion || ' language ''' || $projectLanguage || '''. Expected 0..1. Alert your administrator as this should not be possible.')
            )
        )
    
    let $results                :=
        if (empty($objectid)) then
            $decor//templateAssociation
        else
        if (empty($objected)) then
            $decor//templateAssociation[@templateId = $objectid]
        else (
            $decor//templateAssociation[@templateId = $objectid][@effectiveDate = $objected]
        )
    
    return
        <list artifact="ASSOCIATION" current="{count($results)}" total="{count($results)}" all="{count($decor//templateAssociation)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Creates a list object with templates

@param $projectPrefix           - required. limits scope to this project only
@param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
@param $projectLanguage         - optional parameter to select from a specific compiled language
@param $searchTerms             - optional array of string. Searches for datasets by name or ending with id
@return all live repository/non-private templates as JSON, all data sets for the given $projectPrefix or nothing if not found
@since 2020-05-03
:)
declare function tmapi:searchTemplate($projectPrefix as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $searchTerms as xs:string*, $treetype as xs:string*, $resolve as xs:boolean) {

    let $includeLocal               := true()
    let $decor                      := utillib:getDecorByPrefix($projectPrefix, $projectVersion, $projectLanguage)
    
    (:
        <repositoryTemplateList url="http://art-decor.org/decor/services/" ident="ad1bbr-">
            <rules>
                <result current="50" total="50">
                    <template id="2.16.840.1.113883.10.12.301">
                        <template name="CDAAct" displayName="CDA Act" isClosed="false" expirationDate="" officialReleaseDate="" id="2.16.840.1.113883.10.12.301" statusCode="active" versionLabel="" effectiveDate="2005-09-07T00:00:00" sortname="CDA Act">
                            <desc language="en-US">Template CDA Act (prototype, directly derived from POCD_RM000040 MIF)</desc>
                            <desc language="nl-NL">Template CDA Act (prototype, direct afgeleid uit POCD_RM000040 MIF)</desc>
                            <desc language="de-DE">Template CDA Act (Prototyp, direkt abgeleitet aus POCD_RM000040 MIF)</desc>
                            <desc language="pl-PL">&lt;font color="grey"&gt;&lt;i&gt;Description only available in other languages&lt;/i&gt;&lt;br /&gt;en-US: Template CDA Act (prototype, directly derived from POCD_RM000040 MIF)&lt;/font&gt;</desc>
                            <classification type="cdaentrylevel"/>
                        </template>
                    </template>
                </result>
            </rules>
        </repositoryTemplateList>
    :)
    let $buildingBlockRepositories  := $decor/project/buildingBlockRepository[empty(@format)] | $decor/project/buildingBlockRepository[@format='decor']
    let $buildingBlockRepositories  := 
        if ($includeLocal) then 
            $buildingBlockRepositories | <buildingBlockRepository url="{serverapi:getServerURLServices()}" ident="{$projectPrefix}" format="decor"/>
        else (
            $buildingBlockRepositories
        )
    (:
    <repositoryTemplateList url="http://art-decor.org/decor/services/" ident="ccda-">
    <rules>
      <template uuid="4fdd6b75-18fa-49c1-9d6b-7518fa79c1a6" id="2.16.840.1.113883.10.20.22.4.36" project="ccda-" projectName="Consolidated CDA" sortname="Admission Medication (V2)" name="AdmissionMedicationV2" effectiveDate="2014-06-09T00:00:00" statusCode="draft" versionLabel="2.1" displayName="Admission Medication (V2)" class="cdaentrylevel">
        <template uuid="8c5e0b82-ede2-4147-9e0b-82ede291475b" project="ccda-" projectName="Consolidated CDA" sortname="Admission Medication (V2)" id="2.16.840.1.113883.10.20.22.4.36" name="AdmissionMedicationV2" effectiveDate="2014-06-09T00:00:00" statusCode="draft" versionLabel="2.1" displayName="Admission Medication (V2)">
          <desc language="en-US">This template represents the medications taken by the patient prior to and at the time of admission. </desc>
          <desc language="nl-NL">&lt;span style="color: grey;"&gt;Geen beschrijving in het Nederlands (nl-NL) beschikbaar&lt;br/&gt;en-US : This template represents the medications taken by the patient prior to and at the time of admission. &lt;/span&gt;</desc>
          <desc language="de-DE">&lt;span style="color: grey;"&gt;Keine Beschreibung in Deutsch (de-DE) verfügbar&lt;br/&gt;en-US : This template represents the medications taken by the patient prior to and at the time of admission. &lt;/span&gt;</desc>
          <desc language="pl-PL">&lt;span style="color: grey;"&gt;Description only available in other languages&lt;br/&gt;en-US : This template represents the medications taken by the patient prior to and at the time of admission. &lt;/span&gt;</desc>
          <classification type="cdaentrylevel"/>
        </template>
        <template uuid="a81bb267-98e6-4d6a-9bb2-6798e62d6acd" project="ccda-" projectName="Consolidated CDA" sortname="Admission Medication" id="2.16.840.1.113883.10.20.22.4.36" name="AdmissionMedication" effectiveDate="2013-01-31T00:00:00" statusCode="draft" versionLabel="1.1" displayName="Admission Medication">
          <desc language="en-US">The Admission Medications entry codes medications that the patient took prior to admission.</desc>
          <desc language="nl-NL">&lt;span style="color: grey;"&gt;Geen beschrijving in het Nederlands (nl-NL) beschikbaar&lt;br/&gt;en-US : The Admission Medications entry codes medications that the patient took prior to admission.&lt;/span&gt;</desc>
          <desc language="de-DE">&lt;span style="color: grey;"&gt;Keine Beschreibung in Deutsch (de-DE) verfügbar&lt;br/&gt;en-US : The Admission Medications entry codes medications that the patient took prior to admission.&lt;/span&gt;</desc>
          <desc language="pl-PL">&lt;span style="color: grey;"&gt;Description only available in other languages&lt;br/&gt;en-US : The Admission Medications entry codes medications that the patient took prior to admission.&lt;/span&gt;</desc>
          <classification type="cdaentrylevel"/>
        </template>
      </template>
    </rules>
    </repositoryTemplateList>
    :)
    let $result := 
        for $repository in $buildingBlockRepositories
        return
        <repositoryTemplateList url="{$repository/@url}" ident="{$repository/@ident}">
        {
            tm:getRepositoryAndBBRTemplateList($searchTerms, $repository/@ident)/*
        }
        </repositoryTemplateList>

    let $schemaTypes                := utillib:getDecorTypes()//TemplateTypes/enumeration
    return
        <result count="{count($result//template[template])}" search="{$searchTerms}" includelocal="{$includeLocal}">
        {
            (:get templates with and without classification:)
            for $templateSet in $result/template
            let $class   := $templateSet/@class
            group by $class
            order by count($schemaTypes[@value = $class]/preceding-sibling::enumeration)
            return
                <class uuid="{util:uuid()}" type="{$class}">
                {
                    for $label in $schemaTypes[@value = $class]/label
                    return
                        <label language="{$label/@language}">{$label/text()}</label>
                }
                {
                    for $ts in $templateSet
                    order by lower-case($ts/@displayName)
                    return $ts
                }
                </class>
        }
        </result>
    };

(:~ Copied from art/api/api-decor-template.xqm. Retrieves the ART-DECOR 3 compatible list of usages for as template
* Transaction associations
* Template associations inbound and outbound
:)
declare function tmapi:getDependenciesAndUsage($projectPrefix as xs:string?, $projectVersion as xs:string?, $language as xs:string?, $id as xs:string, $effectiveDate as xs:string?) {

let $templateall    := 
    if (empty($projectPrefix)) then 
        tm:getTemplateById($id, '')/project/template[@id]
    else (
        tm:getTemplateById($id, '', $projectPrefix, $projectVersion)/template/template[@id]
    )
let $templatest     := max($templateall/xs:dateTime(@effectiveDate))
let $template       := 
    if (empty($effectiveDate) or string($templatest) = $effectiveDate) then   
        $templateall[1]
    else
    if (empty($projectPrefix)) then 
        tm:getTemplateById($id, $effectiveDate)/project/template[@id]
    else (
        tm:getTemplateById($id, $effectiveDate, $projectPrefix, $projectVersion)/template/template[@id]
    )

let $projectPrefix  := if (empty($projectPrefix)) then ($template[1]/ancestor-or-self::*/@ident)[1] else $projectPrefix
let $decor          := utillib:getDecorByPrefix($projectPrefix)
let $isLatestTm     := $templatest = $template/@effectiveDate

let $result                 :=
    if ($template) then (
        (: get dependencies in THIS project :)
        (: get dependencies in other projects than this one :)
        tm:getDependendies($template, $template/@id, $templatest, 1, $setlib:colDecorData//rules[ancestor::decor]),
        (: get dependencies in cache :)
        tm:getDependendies($template, $template/@id, $templatest, 1, $setlib:colDecorCache//rules[ancestor::decor]),
        (: get uses :)
        tm:templateUses($template, $template/@id, 1, $decor/rules, $decor)
    ) else ()

(: uniquify result re/ type of ref element, type, id and effectiveDate, sort project ref's first :)
let $templateReferences     :=
    for $d in $result
    let $id := concat(name($d), $d/@type, $d/@id, $d/@effectiveDate)
    group by $id
    order by (not($d[1]/@prefix=$projectPrefix))
    return
        $d[1]

(: calculate transaction references for template itself and and any templates that refer to it :)
let $t_templates            := $template | $templateReferences[self::ref]
let $transactionReferences  := ($setlib:colDecorData//representingTemplate[@ref = $t_templates/@id] | 
                                $setlib:colDecorCache//representingTemplate[@ref = $t_templates/@id])

(: calculate what the latest version is for templates that are being called dynamically. no use for static bindings. :)
let $tmmap                  := 
    map:merge(
        for $ref in $transactionReferences
        let $tmid           := $ref/@ref
        group by $tmid
        return
            if ($ref/@flexibility[. castable as xs:dateTime]) then () else (
                map:entry($tmid, tm:getTemplateById($tmid, '')/project/template[@id]/@effectiveDate)
            )
    )

(: transaction bindings count for static matches and dynamic matches :)
let $transactionReferences  :=
    for $ref in $transactionReferences
    let $match  := 
        if ($ref[@flexibility castable as xs:dateTime]) then
            $t_templates[@id = $ref/@ref][@effectiveDate = $ref/@flexibility]
        else (
            $t_templates[@effectiveDate = map:get($tmmap, @id)]
        )
    return
        if (empty($match)) then () else (
            utillib:doTransactionAssociation($match[1], $ref/ancestor::transaction[1])
        )

return (
    $transactionReferences,
    $templateReferences
)
};

(:~ Rewrite an incoming template back into db format

@param $decor - decor project to update results into
@param $inputTemplate - incoming raw template
@param $storedTemplate - template in db to update (if any)
@param $mode - 'new' (completely new), 'edit' (edit of existing, same id/effectiveDate), 'version' (clone under same id, new effectiveDate), 'adapt' (clone under new id and effectiveDate)
:)
declare %private function tmapi:prepareIncomingTemplateForUpdate($decor as element(decor), $inputTemplate as element(template), $storedTemplate as element(template)?, $mode as xs:string, $newTemplateElementId as xs:string?) as element(template) {

    let $originalId             := ($inputTemplate/@originalId, $inputTemplate/@id)[1]
    let $templateBaseIds        := decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)
    
    (:if a baseId is present use it, else use default template baseId:)
    let $templateRoot           :=
        if ($inputTemplate/@baseId) then
            if ($templateBaseIds[@id = $inputTemplate/@baseId]) then $inputTemplate/@baseId else (
                error($errors:BAD_REQUEST, 'Submitted data SHALL specify a known baseId in project ' || $decor/project/@prefix || '. BaseId ''' || $inputTemplate/@baseId || ''' not found')
            )
        else (
            decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)[1]/@id
        )
    
    (:generate new id if mode is 'new' or 'adapt', else use existing id:)
    let $newTemplateId              := 
        if ($mode=('new','adapt')) then
            decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TEMPLATE, $templateRoot)[1]/@id
        else (
            $inputTemplate/@id
        )
    
    let $templateName               := attribute name {($inputTemplate/@name, utillib:getNameForOID($newTemplateId, (), $decor))[not(.='')][1]}
    let $templateDisplayName        := attribute displayName {($inputTemplate/@displayName, $templateName)[not(.='')][1]}

    (:keep effectiveDate for mode is 'edit', else generate new:)
    let $templateEffectiveTime  := 
        if ($mode = 'edit' and $storedTemplate[@effectiveDate castable as xs:dateTime]) then
            $storedTemplate/@effectiveDate
        else
        if ($inputTemplate[@effectiveDate castable as xs:dateTime][not(@effectiveDate = $storedTemplate/@effectiveDate)]) then 
            $inputTemplate/@effectiveDate
        else (
            attribute effectiveDate {format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')}
        )
    (:set draft effectiveDate for mode is 'edit', else generate new:)
    let $templateStatusCode     := 
        if ($inputTemplate/@statusCode[. = ('','new')]) then
            attribute statusCode {'draft'}
        else (
            $inputTemplate/@statusCode
        )
    
    return (
        <template>
        {
            $newTemplateId,
            $templateName,
            $templateDisplayName,
            $templateEffectiveTime,
            $templateStatusCode,
            $inputTemplate/@versionLabel[not(. = '')],
            $inputTemplate/@expirationDate[. castable as xs:dateTime],
            $inputTemplate/@officialReleaseDate[. castable as xs:dateTime],
            $inputTemplate/@canonicalUri[not(. = '')],
            $inputTemplate/@isClosed[. = ('true', 'false')],
            attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)}
            ,
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($inputTemplate/desc)
            ,
            tmapi:prepareClassificationForUpdate($inputTemplate/classification)
            ,
            tmapi:prepareRelationshipForUpdate($inputTemplate/relationship)
            ,
            tmapi:prepareContextForUpdate($inputTemplate/context[1])
            ,
            tmapi:prepareItemForUpdate($inputTemplate/(item | items[@is = 'item'])[1])
            ,
            utillib:preparePublishingAuthorityForUpdate($inputTemplate/publishingAuthority[empty(@inherited)][@name[not(. = '')]][1])
            (:,
            utillib:preparePublishingAuthorityForUpdate($inputTemplate/endorsingAuthority[empty(@inherited)][@name[not(. = '')]][1])
            :)
            ,
            for $node in $inputTemplate/purpose
            return
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
            ,
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($inputTemplate/copyright[empty(@inherited)])
            (:,
            utillib:prepareObjectHistoryForUpdate($inputTemplate/revisionHistory):)
            ,
            utillib:prepareExampleForUpdate($inputTemplate/example)
            ,
            (: not yet necessary for patch as we don't patch these ... :)
            for $attribute in $inputTemplate/(attribute|items[@is = 'attribute'])
            return
                tmapi:processAttribute($attribute, $newTemplateId, $newTemplateElementId, true())
            ,
            for $item in $inputTemplate/(element|include|choice|let|assert|report|defineVariable|constraint|items[@is = ('element','include','choice','let','assert','report','defineVariable','constraint')])
            return
                tmapi:processElement($item, $newTemplateId, $newTemplateElementId, true())
        }
        </template>
    )
};

(:~ Collect all relevant templateAssociations into one element :)
declare function tmapi:prepareTemplateAssociationForUpdate($storedTemplateAssociations as element(templateAssociation)*, $preparedTemplate as element(template), $inputTemplate as element(template)?, $newTemplateElementId as xs:string, $mode as xs:string) as element(templateAssociation) {
    <templateAssociation templateId="{$preparedTemplate/@id}" effectiveDate="{$preparedTemplate/@effectiveDate}">
    {
        (:  keep all concepts associated with element/@id | attribute/@id in the prepared template :)
        for $association in ($storedTemplateAssociations/concept[@elementId = ($preparedTemplate//element/@id | $preparedTemplate//attribute/@id)] |
                             $inputTemplate/staticAssociations/*[@elementId = ($preparedTemplate//element/@id | $preparedTemplate//attribute/@id)] | 
                             $storedTemplateAssociations/concept[@elementPath] | $inputTemplate/staticAssociations/*[@elementPath])
        let $deid   := $association/@ref
        let $deed   := $association/@effectiveDate
        let $elid   := $association/@elementId
        let $elpt   := $association/@elementPath
        group by $deid, $deed, $elid, $elpt
        return
            <concept>{$deid, $deed, $elid, $elpt}</concept>
        ,
        (:if mode is 'new' and conceptId is present, create new templateAssociation for concept:)
        if ($mode = 'new' and $inputTemplate[@conceptId]//@concept) then
            <concept ref="{$inputTemplate/@conceptId}" effectiveDate="{$inputTemplate/@conceptEffectiveDate}" elementId="{$newTemplateElementId}"/>
        else ()
    }
    </templateAssociation>
};

declare function tmapi:prepareTemplateForUpdate($editedTemplate as element(template)) as element(template) {
    <template>
    {
        $editedTemplate/@id,
        $editedTemplate/@name,
        $editedTemplate/@displayName,
        $editedTemplate/@effectiveDate,
        $editedTemplate/@statusCode,
        $editedTemplate/@isClosed[not(.='')],
        $editedTemplate/@expirationDate[not(.='')],
        $editedTemplate/@officialReleaseDate[not(.='')],
        $editedTemplate/@versionLabel[not(.='')],
        $editedTemplate/@canonicalUri[not(.='')],
        attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)}
        ,
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedTemplate/desc)
        ,
        tmapi:prepareClassificationForUpdate($editedTemplate/classification)
        ,
        tmapi:prepareRelationshipForUpdate($editedTemplate/relationship)
        ,
        utillib:preparePublishingAuthorityForUpdate($editedTemplate/publishingAuthority[empty(@inherited)][@name[not(. = '')]][1])
        (:,
        utillib:preparePublishingAuthorityForUpdate($editedTemplate/endorsingAuthority[empty(@inherited)][@name[not(. = '')]][1])
        :)
        ,
        for $node in $editedTemplate/purpose
        return
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
        ,
        for $node in $editedTemplate/copyright
        return
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
        (:,
        utillib:prepareObjectHistoryForUpdate($editedTemplate/revisionHistory):)
        ,
        tmapi:prepareContextForUpdate($editedTemplate/context[1])
        ,
        tmapi:prepareItemForUpdate($editedTemplate/item[1])
        ,
        utillib:prepareExampleForUpdate($editedTemplate/example)
        ,
        (: not yet necessary for patch as we don't patch these ... :)
        for $attribute in $editedTemplate/(attribute|items[@is = 'attribute'])
        return
            (:$attribute :)tmapi:processAttribute($attribute, '', '', false())
        ,
        for $item in $editedTemplate/(element|include|choice|let|assert|report|defineVariable|constraint|items[@is = ('element','include','choice','let','assert','report','defineVariable','constraint')])
        return
            (:$item :)tmapi:processElement($item, '', '', false())
    }
    </template>
};

(: Input
      <classification format="hl7v3xml1">
        <type>clinicalstatementlevel</type>
        <type>cdaentrylevel</type>
        <tag>tag1</tag>
        <tag>tag2</tag>
        <property>property1</property>
        <property>property2</property>
      </classification>
   Output
      <classification type="clinicalstatementlevel" format="hl7v3xml1">
        <tag>tag1</tag>
        <tag>tag2</tag>
        <property>property1</property>
        <property>property2</property>
      </classification>
      <classification type="cdaentrylevel">
        <tag>tag1</tag>
        <tag>tag2</tag>
        <property>property1</property>
        <property>property2</property>
      </classification>
:)
declare function tmapi:prepareClassificationForUpdate($nodes as element(classification)*) as element(classification)* {
    let $classbytype  :=
        for $type at $i in distinct-values(($nodes/@type, $nodes/type))
        return
            <classification type="{$type}"/>
    
    return
        if ($classbytype) then (
            <classification>
            {
                $classbytype[1]/@type,
                ($nodes/@format)[1],
                for $tag in $nodes/tag
                group by $tag
                return
                    <tag>{utillib:parseNode($tag[1])/(node() except item)}</tag>
                ,
                for $property in $nodes/property
                group by $property
                return
                    <property>{utillib:parseNode($property[1])/(node() except item)}</property>
            }
            </classification>
            ,
            $classbytype[position() gt 1]
        )
        else 
        if ($nodes[@format | tag | property]) then (
            <classification>
            {
                $nodes[1]/@format,
                for $tag in $nodes/tag
                group by $tag
                return
                    <tag>{utillib:parseNode($tag[1])/(node() except item)}</tag>
                ,
                for $property in $nodes/property
                group by $property
                return
                    <property>{utillib:parseNode($property[1])/(node() except item)}</property>
            }
            </classification>
        )
        else (
        )
};

declare function tmapi:prepareRelationshipForUpdate($nodes as element(relationship)*) as element(relationship)* {
    for $node in $nodes
    return
        element {name($node)} {
            $node/@type,
            ($node/@template, $node/@model)[1],
            $node/@flexibility[. = 'dynamic'] | $node/@flexibility[. castable as xs:dateTime]
        }
};

declare function tmapi:prepareContextForUpdate($nodes as element(context)*) as element(context)* {
    for $node in $nodes
    let $att  :=
        switch ($node/@selected)
        case 'id'     return $node/@id[not(. = '')]
        case 'path'   return $node/@path[not(. = '')]
        default       return ($node/@id, $node/@path)[not(. = '')][1] 
    return
        if ($att) then <context>{$att}</context> else ()
};

declare function tmapi:prepareItemForUpdate($nodes as element()*) as element(item)* {
    for $node in $nodes[self::items[@is = 'item'] | self::item][string-length(@label) gt 0]
    return
        <item>
        {
            $node/@label,
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($node/desc)
        }
        </item>
};

declare %private function tmapi:processElement($element as element(), $templateId as xs:string, $elementId as xs:string, $selectedOnly as xs:boolean) as element()? {
    if (not($selectedOnly) or $element/@selected) then (
        let $elmName    := if (name($element) = 'items') then $element/@is else name($element)
        return
        switch ($elmName)
        case 'element' return
            <element>
            {
                (: attributes :)
                $element/@name[not(.='')],
                if ($element[@contains[not(.='')]]) then (
                    $element/@contains,
                    $element/@flexibility[. castable as xs:dateTime or . = 'dynamic']
                ) else ()
                ,
                if ($element/@datatype[not(.='')]) then (
                    $element/@datatype,
                    $element/@strength[not(.='')]
                ) else ()
                ,
                $element/@minimumMultiplicity[. = '0'] | $element/@minimumMultiplicity[. castable as xs:positiveInteger],
                $element/@maximumMultiplicity[. = '0'] | $element/@maximumMultiplicity[. castable as xs:positiveInteger] | $element/@maximumMultiplicity[. = '*'],
                $element/@conformance[. = ('R', 'C', 'NP')],
                $element/@isMandatory[. = 'true'],
                $element/@isClosed[.= ('true', 'false')],
                if ($element/@concept and $elementId[not(. = '')]) then
                   attribute id {$elementId}
                else(
                    $element/@id[not(.='')]
                )
                ,
                (: this part is in the schema, but not implemented (yet) :)
                $element/@include[not(.='')],
                $element/@useWhere[not(.='')],
                $element/@displayName[not(.='')],
                $element/@statusCode[not(.='')],
                $element/@versionLabel[not(.='')],
                $element/@effectiveDate[. castable as xs:dateTime],
                $element/@expirationDate[. castable as xs:dateTime],
                $element/@officialReleaseDate[. castable as xs:dateTime]
            }
            {
                (: elements :)
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($element/desc)
                ,
                for $item in $element/(item | items[@is = 'item'])[@label[not(. = '')]]
                return
                    <item>
                    {
                        $item/@label,
                        utillib:prepareFreeFormMarkupWithLanguageForUpdate($item/desc)
                    }
                    </item>
                ,
                utillib:prepareExampleForUpdate($element/(example | items[@is = 'example']))
                ,
                tmapi:prepareVocabulary($element/vocabulary)
                ,
                tmapi:prepareElementProperty($element/(property | items[@is = 'property']))
                ,
                $element/text
            }
            {
                for $attribute in $element/(attribute|items[@is = 'attribute'])
                return 
                    tmapi:processAttribute($attribute, $templateId, $elementId, $selectedOnly)
                ,
                for $item in $element/(element|include|choice|let|assert|report|defineVariable|constraint|items[@is = ('element','include','choice','let','assert','report','defineVariable','constraint')])
                return
                    tmapi:processElement($item, $templateId, $elementId, $selectedOnly)
            }
            </element>
        
        case 'choice' return
            <choice>
            {
                (: attributes :)
                $element/@minimumMultiplicity[. = '0'] | $element/@minimumMultiplicity[. castable as xs:positiveInteger],
                $element/@maximumMultiplicity[. = '0'] | $element/@maximumMultiplicity[. castable as xs:positiveInteger] | $element/@maximumMultiplicity[. = '*']
            }
            {
                (: elements :)
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($element/desc)
                ,
                for $item in $element/(item | items[@is = 'item'])[@label[not(. = '')]]
                return
                    <item>
                    {
                        $item/@label,
                        utillib:prepareFreeFormMarkupWithLanguageForUpdate($item/desc)
                    }
                    </item>
                ,
                for $item in $element/(element|include|constraint|items[@is = ('element','include','constraint')])
                return
                    tmapi:processElement($item, $templateId, $elementId, $selectedOnly)
            }
            </choice>
        
        case 'include' return
            <include>
            {
                (: attributes :)
                if ($element[@ref[not(. = '')]]) then (
                    $element/@ref,
                    $element/@flexibility[. castable as xs:dateTime] | $element/@flexibility[. = 'dynamic']
                ) else ()
                ,
                $element/@minimumMultiplicity[. = '0'] | $element/@minimumMultiplicity[. castable as xs:positiveInteger],
                $element/@maximumMultiplicity[. = '0'] | $element/@maximumMultiplicity[. castable as xs:positiveInteger] | $element/@maximumMultiplicity[. = '*'],
                $element/@conformance[. = ('R', 'C', 'NP')],
                $element/@isMandatory[. = 'true'],
                (: this part is in the schema, but not implemented (yet) :)
                $element/@scenario[not(.='')],
                $element/@effectiveDate[. castable as xs:dateTime]
           }
           {
                (: elements :)
                for $node in $element/desc
                return
                    utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
                ,
                for $item in $element/(item | items[@is = 'item'])[@label[not(. = '')]]
                return
                    <item>
                    {
                        $item/@label,
                        utillib:prepareFreeFormMarkupWithLanguageForUpdate($item/desc)
                    }
                    </item>
                ,
                utillib:prepareExampleForUpdate($element/(example | items[@is = 'example']))
                ,
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($element/constraint)
            }
            </include>
        
        case 'let' return
            <let>
            {
                $element/@name,
                $element/@value,
                $element/node()
            }
            </let>
        
        case 'assert' return
            <assert>
            {
                $element/@role[not(.='')],
                $element/@test[not(.='')],
                $element/@flag[not(.='')],
                $element/@see[not(.='')],
                let $parsed := utillib:parseNode($element)
                
                return
                    if ($parsed[count(*) = 1][*:p | *:div][normalize-space(string-join(text(), '')) = '']) then 
                        $parsed/*/node() 
                    else (
                        $parsed/node()
                    )
            }
            </assert>
        
        case 'report' return
            <report>
            {
                $element/@role[not(.='')],
                $element/@test[not(.='')],
                $element/@flag[not(.='')],
                $element/@see[not(.='')],
                let $parsed := utillib:parseNode($element)
                
                return
                    if ($parsed[count(*) = 1][*:p | *:div][normalize-space(string-join(text(), '')) = '']) then 
                        $parsed/*/node() 
                    else (
                        $parsed/node()
                    )
            }
            </report>
        
        case 'defineVariable' return
            <defineVariable>
            {
                $element/@name[not(.='')],
                $element/@path[not(.='')],
                $element/node()
            }
            </defineVariable>
            
        case 'constraint' return
            <constraint>
            {
                $element/@language[not(.='')],
                $element/@lastTranslated[not(.='')],
                $element/@mimeType[not(.='')],
                utillib:parseNode($element)/node()
            }
            </constraint>
        
        default return ()
    
    )
    else ()
};

declare %private function tmapi:processAttribute($attribute as element(), $templateId as xs:string, $elementId as xs:string, $selectedOnly as xs:boolean) as element()? {
    if (not($selectedOnly) or $attribute/@selected) then (
        <attribute>
        {
            $attribute/(@name|@classCode|@contextConductionInd|@contextControlCode|
                        @determinerCode|@extension|@independentInd|@institutionSpecified|
                        @inversionInd|@mediaType|@moodCode|@negationInd|
                        @nullFlavor|@operator|@qualifier|@representation|
                        @root|@typeCode|@unit|@use)[not(.='')]
            ,
            if ($attribute[@name = 'root'][matches(@value,'[a-zA-Z]')]/parent::element/@name[matches(., '^([^:]+:)?templateId')]) then
                if ($templateId[not(. = '')]) then 
                    attribute value {$templateId}
                else ()
            else (
                $attribute/@value[not(. = '')]
            )
            ,
            if ($attribute/@datatype[not(.='')]) then (
                $attribute/@datatype,
                $attribute/@strength[not(.='')]
            ) else ()
            ,
            if ($attribute/@prohibited = 'true' or $attribute/@conf = 'prohibited' or $attribute/@conformance = 'NP') then
                attribute prohibited {'true'}
            else 
            if ($attribute/@isOptional = 'true' or $attribute/@conf = 'isOptional' or $attribute/@conformance = 'O') then
                attribute isOptional {'true'}
            else ()
            ,
            if ($attribute/@concept and $elementId[not(. = '')]) then
                attribute id {$elementId}
            else(
                $attribute/@id[not(.='')]
            )
            ,
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($attribute/desc)
            ,
            for $item in $attribute/(item | items[@is = 'item'])[@label[not(. = '')]]
            return
                <item>
                {
                    $item/@label,
                    utillib:prepareFreeFormMarkupWithLanguageForUpdate($item/desc)
                }
                </item>
            ,
            tmapi:prepareVocabulary($attribute/vocabulary)
        }
        </attribute>
    )
    else ()
};

(:~ Process property element(s). Incoming property element may be named 'property' or items[@is='property']. If it contains any of the supported attributes with a value, 
    a property element is created with those attributes. Empty attributes and any other attributes than the supported attributes are ignored (like @is and @order). Supported 
    attributes with illegal contents raise an error :)
declare %private function tmapi:prepareElementProperty($properties as element()*) as element(property)* {
    for $property in $properties[self::property | self::items[@is = 'property']][(@minInclude | @maxInclude | @minLength | @maxLength | @value | @fractionDigits | @unit | @currency)[not(. = '')]]
    return
        <property>
        {
            for $att in ($property/@minInclude, $property/@maxInclude)[not(. = '')]
            return
                if ($att castable as xs:decimal) then $att else (
                    error($errors:BAD_REQUEST, 'Property ' || name($att) || ' SHALL be a valid integer or decimal (with decimal point, not comma). Found ''' || $att || '''')
                )
            ,
            for $att in ($property/@minLength, $property/@maxLength)[not(. = '')]
            return
                if ($att castable as xs:positiveInteger) then $att else (
                    error($errors:BAD_REQUEST, 'Property ' || name($att) || ' SHALL be a valid positive integer. Found ''' || $att || '''')
                )
            ,
            $property/@value[not(. = '')]
            ,
            for $att in $property/@fractionDigits[not(. = '')]
            return
                if (matches($att, '^\d{1,4}[!\.]?$')) then $att else (
                    error($errors:BAD_REQUEST, 'Property ' || name($att) || ' SHALL match pattern of 1-4 digits followed by ! or . (period). Found ''' || $att || '''')
                )
            ,
            $property/@unit[not(. = '')],
            $property/@currency[not( . = '')]
        }
        </property>
};

declare %private function tmapi:prepareVocabulary($vocabularies as element(vocabulary)*) as element()* {
    for $vocabulary in $vocabularies[(@valueSet | @code | @codeSystem | @codeSystemName | @codeSystemVersion | @displayName | @domain)[not(. = '')]]
    return
        <vocabulary>
        {
            if ($vocabulary/@valueSet[not(. = '')]) then (
                if ($vocabulary/@valueSet[utillib:isOid(.)]) then $vocabulary/@valueSet else (
                    error($errors:BAD_REQUEST, 'Vocabulary attribute valueSet SHALL be a valid OID. Found ''' || $vocabulary/@valueSet || '''')
                )
                ,
                if ($vocabulary/@flexibility[not(. = '')]) then 
                    if ($vocabulary/@flexibility[. castable as xs:dateTime] | $vocabulary/@flexibility[. = 'dynamic']) then $vocabulary/@flexibility else (
                        error($errors:BAD_REQUEST, 'Vocabulary attribute flexibility SHALL be a valid dateTime or ''dynamic''. Found ''' || $vocabulary/@flexibility || '''')
                    )
                else ()
            ) 
            else (
                if ($vocabulary/@code[not(. = '')]) then
                    if ($vocabulary/@code[matches(., '\s')]) then 
                        error($errors:BAD_REQUEST, 'Vocabulary attribute code SHALL NOT contain whitespace. Found ''' || $vocabulary/@code || '''')
                    else (
                        $vocabulary/@code
                    )
                else ()
                ,
                if ($vocabulary/@codeSystem[not(. = '')]) then
                    if ($vocabulary/@codeSystem[utillib:isOid(.)]) then $vocabulary/@codeSystem else (
                        error($errors:BAD_REQUEST, 'Vocabulary attribute codeSystem SHALL be a valid OID. Found ''' || $vocabulary/@codeSystem || '''')
                    )
                else ()
                ,
                $vocabulary/@codeSystemName[not(. = '')],
                $vocabulary/@codeSystemVersion[not(. = '')],
                $vocabulary/@displayName[not(. = '')]
            )
            ,
            $vocabulary/@domain[not(. = '')]
        }
        </vocabulary>
};

declare %private function tmapi:getTemplateHistoryTEST($projectPrefix as xs:string*, $id as xs:string*, $effectiveDate as xs:string*) {
let $result := ()
    (: shall retrun a list of history items of the artefact identified by id and effectiveDate, returns these items
    <history
        type="TM"
        id="{$h/@artefactId}"
        effectiveDate="{$h/@artefactEffectiveDate}"
        statusCode="{$h/@artefactStatusCode}"
        performer="{data($h/@author)}"
        intention="{$h/@intention}"
        when="{format-dateTime($h/@date,'[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]','en',(),())}"/>
    :)
return
    $result
};