xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
:)
(:~ Questionnaire and QuestionnaireResponse API allows read, create, update on DECOR links into HL7 FHIR Questionnaire and QuestionnaireResponse :)
module namespace qapi              = "http://art-decor.org/ns/api/questionnaire";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "/db/apps/api/modules/library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "/db/apps/api/modules/library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";
import module namespace jx          = "http://joewiz.org/ns/xquery/json-xml" at "library/json-xml.xqm";
import module namespace serverapi   = "http://art-decor.org/ns/art-decor-server" at "server-api.xqm";
import module namespace router      = "http://e-editiones.org/roaster/router";

declare namespace json      = "http://www.json.org";
declare namespace f         = "http://hl7.org/fhir";

(:~ All functions support their own override, but this is the fallback for the maximum number of results returned on a search :)
declare %private variable $qapi:maxResults             := 50;
declare %private variable $qapi:ITEM-STATUS            := utillib:getDecorTypes()/ItemStatusCodeLifeCycle;
declare %private variable $qapi:STATUSCODES-FINAL      := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $qapi:RELATIONSHIP-TYPES     := utillib:getDecorTypes()/QuestionnaireRelationshipTypes;
declare %private variable $qapi:FHIROBJECT-FORMAT      := utillib:getDecorTypes()/FhirObjectFormat;

(:~ Returns a list of zero or more questionnaire and questionnaireresponses
    
@param $projectPrefix           - required. limits scope to this project only
@param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
@param $projectLanguage         - optional parameter to select from a specific compiled language
@param $objectId                - optional. Filter output to just those based on this questionnaire/relationship/@ref. For each questionnaire, also all questionnaireResponses are returned unless omitResponses=true
@param $objectEffectiveDate     - optional. Filter output to just those based on this questionnaire/relationship/@flexibility. relevant when objectId has a value.
@param $sort                    - optional parameter to indicate sorting. Only option 'name'
@param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
@return all live repository/non-private questionnaire and questionnaireResponse as JSON
@since 2020-05-03
:)
declare function qapi:getQuestionnaireList($request as map(*)) {

    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $objectId               := $request?parameters?objectId[not(. = '')]
    let $objectEffectiveDate    := $request?parameters?objectEffectiveDate[not(. = '')]
    let $sort                   := $request?parameters?sort[not(. = '')]
    let $sortorder              := $request?parameters?sortorder[not(. = '')]
    let $max                    := $request?parameters?max[not(. = '')]
    
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have parameter prefix')
        else ()
    return
        qapi:getQuestionnaireList($projectPrefix, $projectVersion, $projectLanguage, $objectId, $objectEffectiveDate, $sort, $sortorder, $max)

};

declare function qapi:getQuestionnaireList($projectPrefix as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $objectId as xs:string?, $objectEffectiveDate as xs:string?, $sort as xs:string?, $sortorder as xs:string?, $max as xs:string?) {
    let $sort                       := $sort[string-length() gt 0]
    let $sortorder                  := $sortorder[. = 'descending']
    
    let $projectPrefix              := $projectPrefix[not(. = '')]
    let $projectVersion             := $projectVersion[not(. = '')]
    let $decor                      := if (empty($projectPrefix)) then () else utillib:getDecorByPrefix($projectPrefix, $projectVersion, $projectLanguage)
    
    let $allcnt                     := count($decor/scenarios/questionnaire)
    
    let $objectId                   := $objectId[not(. = '')]
    let $objectEffectiveDate        := $objectEffectiveDate[not(. = '')]
    
    let $filteredQuestionnaires     := 
        if (empty($objectId)) then $decor/scenarios/questionnaire | $decor/scenarios/questionnaireresponse else (
            if (empty($objectEffectiveDate)) then (
                let $object                     :=
                    if (empty($objectId)) then () else (
                        utillib:getScenario($objectId, $objectEffectiveDate, $projectVersion, $projectLanguage) |
                        utillib:getTransaction($objectId, $objectEffectiveDate, $projectVersion, $projectLanguage) |
                        utillib:getDataset($objectId, $objectEffectiveDate, $projectVersion, $projectLanguage)
                    )
                (: get questionnaires and questionnaireresponses related to object :)
                let $qs          := 
                    if ($object) then (
                        $decor/scenarios/questionnaire[relationship[@ref = $object/@id][@flexibility = $object/@effectiveDate]] |
                        $decor/scenarios/questionnaireresponse[relationship[@ref = $object/@id][@flexibility = $object/@effectiveDate]]
                    )
                    else ( 
                        $decor/scenarios/questionnaire[relationship[@ref = $objectId]] | 
                        $decor/scenarios/questionnaireresponse[relationship[@ref = $objectId]]
                    )
                (: get questionnaireresponses related to questionnaires :)
                let $qr         := 
                    for $q in $qs
                    return
                        $decor/scenarios/questionnaireresponse[relationship[@ref = $q/@id][@flexibility = $q/@effectiveDate]]
                
                return $qs | $qr
            )
            else (
                (: get questionnaires and questionnaireresponses related to object :)
                let $qs          := 
                    $decor/scenarios/questionnaire[relationship[@ref = $objectId][@flexibility = $objectEffectiveDate]] |
                    $decor/scenarios/questionnaireresponse[relationship[@ref = $objectId][@flexibility = $objectEffectiveDate]]
                (: get questionnaireresponses related to questionnaires :)
                let $qr         := 
                    for $q in $qs
                    return
                        $decor/scenarios/questionnaireresponse[relationship[@ref = $q/@id][@flexibility = $q/@effectiveDate]]
                
                return $qs | $qr
            )
        )
    
    let $projectLanguage      := 
        if (empty($projectLanguage)) 
        then 
            if (empty($projectPrefix)) then
                $filteredQuestionnaires/ancestor::decor/project/@defaultLanguage
            else
                $decor/project/@defaultLanguage
        else $projectLanguage
    
    (: build map of oid names. is more costly when you do that deep down :)
    let $oidnamemap             := map {}(:qapi:buildOidNameMap($filteredQuestionnaires, $projectLanguage[1]):)
    
    let $filteredQuestionnaireResponses := $filteredQuestionnaires[self::questionnaireresponse]
    let $filteredQuestionnaires         := $filteredQuestionnaires[self::questionnaire]
    
    (:can sort on indexed db content -- faster:)
    let $results                :=
        switch ($sort) 
        case 'effectiveDate'    return 
            if (empty($sortorder)) then 
                for $q in $filteredQuestionnaires order by $q/@effectiveDate return $q
            else
                for $q in $filteredQuestionnaires order by $q/@effectiveDate descending return $q
        case 'name'    return 
            if (empty($sortorder)) then 
                for $q in $filteredQuestionnaires order by lower-case($q/@name) return $q
            else
                for $q in $filteredQuestionnaires order by lower-case($q/@name) descending return $q
        default                 return $filteredQuestionnaires
    
    let $count              := count($results)
    let $max                := if ($max ge 1) then $max else $qapi:maxResults
    
    let $matchedqr          :=
        for $q in $results
        return
            qapi:questionnaireBasics($q, $projectPrefix, $projectLanguage[1], $oidnamemap, $filteredQuestionnaireResponses)
    let $unmatchedqr        :=
        for $qr in $filteredQuestionnaireResponses
        let $relationship := $qr/relationship[@type = 'ANSW'][1]
        return
            if ($matchedqr[@id = $relationship/@ref][@effectiveDate = $relationship/@flexibility]) then () else qapi:questionnaireBasics($qr, $projectPrefix, $projectLanguage[1], $oidnamemap, ())
    let $pendingRequests    := ()
        (:for $pendingRequest in collection($setlib:strDecorScheduledTasks)/questionnaire-transform-request
        order by $pendingRequest/@on
        return
            $pendingRequest:)
        
    
    return
    <list artifact="QQQR" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
    {
        utillib:addJsonArrayToElements( ($matchedqr, $unmatchedqr, $pendingRequests) )
    }
    </list>
};

(:~  Return questionnaire or questionnaireresponse based on id and optionally effectiveDate. If effectiveDate is empty, the latest version is returned
:
:   @param $q-id   - required dataset/@id
:   @param $q-ed   - optional dataset/@effectiveDate
:   @return exactly 1 questionnaire or questionnaireresponse or nothing if not found
:   @since 2015-04-27
:)
declare function qapi:getLatestQuestionnaire($request as map(*)) {
    qapi:getQuestionnaire($request)
};

declare function qapi:getLatestQuestionnaireResponse($request as map(*)) {
    qapi:getQuestionnaireResponse($request)
};

declare function qapi:getQuestionnaire($request as map(*)) {

    let $q-id                           := $request?parameters?id
    let $q-ed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    
    let $oidnamemap                   := map {}
    let $projectPrefix                := ()
    let $decorVersion                 := $request?parameters?release
    let $language                     := $request?parameters?language

    let $q                            := utillib:getQuestionnaire($q-id, $q-ed, $decorVersion, $language)
    let $fhirVersion                  := if (empty($q/@fhirVersion)) then '4.0' else $q/@fhirVersion
    let $acceptTypes                  := router:accepted-content-types()
    let $acceptedType                 := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]

    let $results                      := qapi:questionnaireBasics($q, $projectPrefix, $language, $oidnamemap, $q/../questionnaireresponse)
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple questionnaires for id '", $q-id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Retrieves DECOR questionnaire history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function qapi:getQuestionnaireHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-QUESTIONNAIRE, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-QUESTIONNAIRE}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

declare function qapi:getQuestionnaireResponse($request as map(*)) {

    let $qr-id                           := $request?parameters?id
    let $qr-ed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $oidnamemap                   := map {}
    let $projectPrefix                := ()
    let $decorVersion                 := $request?parameters?release
    let $language                     := $request?parameters?language

    let $qr                           := utillib:getQuestionnaireResponse($qr-id, $qr-ed, $decorVersion, $language)
    let $fhirVersion                  := if (empty($qr/@fhirVersion)) then '4.0' else $qr/@fhirVersion
    let $acceptTypes                  := router:accepted-content-types()
    let $acceptedType                 := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]
    
    let $results                      := qapi:questionnaireBasics($qr, $projectPrefix, $language, $oidnamemap, ())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple questionnaireresponses for id '", $qr-id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Retrieves DECOR questionnaire response history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function qapi:getQuestionnaireResponseHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:declare function qapi:getQuestionnaire($q-id as xs:string, $q-ed as xs:string?) as element()* {
    qapi:getQuestionnaire($q-id, $q-ed,(),())
};:)

(:~  Return the live questionnaire if param decorVersion is not a dateTime OR 
:   compiled, archived/released DECOR questionnaire or questionnaireresponse based on @id|@effectiveDate and decor/@versionDate.
:   Released/compiled matches could be present in multiple languages. Use the parameters @language to select the right one.
:
:   @param $q-id           - required questionnaire or questionnaireresponse/@id
:   @param $q-ed           - optional questionnaire or questionnaireresponse/@effectiveDate
:   @param $decorVersion    - optional decor/@versionDate
:   @param $language        - optional decor/@language. Only relevant with parameter $decorVersion. Empty uses decor/project/@defaultLanguage, '*' selects all versions, e.g. 'en-US' selects the US English version (if available)
:   @return exactly 1 live questionnaire or questionnaireresponse, or 1 or more archived DECOR version instances or nothing if not found
:   @since 2015-04-29
:)
(:declare function qapi:getQuestionnaire($q-id as xs:string, $q-ed as xs:string?, $decorVersion as xs:string?) as element()* {
    qapi:getQuestionnaire($q-id, $q-ed,$decorVersion,())
};
:)
(:declare function qapi:getQuestionnaire($q-id as xs:string, $q-ed as xs:string?, $decorVersion as xs:string?, $language as xs:string?) as element()* {
    let $decor          :=
        if ($decorVersion[string-length()>0]) then
            let $d      := $setlib:colDecorVersion/decor[@versionDate=$decorVersion]
            return
            if (empty($language)) then
                $d[@language=$d/project/@defaultLanguage]
            else if ($language='*') then
                $d
            else
                $d[@language=$language]
        else (
            $setlib:colDecorData/decor
        )
    let $quest          := $decor//questionnaire[@id = $q-id][ancestor::scenarios] | $decor//questionnaireresponse[@id = $q-id][ancestor::scenarios]
    
    return
        if ($q-ed castable as xs:dateTime) then 
            $quest[@effectiveDate = $q-ed] 
        else 
        if ($quest[2]) then 
            $quest[@effectiveDate = max($quest/xs:dateTime(@effectiveDate))]
        else (
            $quest
        )
};
:)
declare function qapi:patchQuestionnaire($request as map(*)) {

    let $authmap                       := $request?user
    let $qid                           := $request?parameters?id
    let $qed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $projectVersion                 := ()
    let $projectLanguage                := ()
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                         := qapi:patchQuestionnaire($authmap, true(), string($qid), $qed, $data)
   
    return 
      
        $return

};

declare function qapi:patchQuestionnaireResponse($request as map(*)) {

    let $authmap                       := $request?user
    let $qid                           := $request?parameters?id
    let $qed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $projectVersion                 := ()
    let $projectLanguage                := ()
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                         := qapi:patchQuestionnaire($authmap, false(), string($qid), $qed, $data)
   
    return 
      
        $return

};

declare function qapi:patchQuestionnaire($authmap as map(*), $doQuestionnaire as xs:boolean, $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) {

    let $label                  := if ($doQuestionnaire) then 'Questionnaire' else 'QuestionnaireResponse'
    let $storedQuestionnaire    := 
        if ($doQuestionnaire) then 
            utillib:getQuestionnaire($id, $effectiveDate)
        else (
            utillib:getQuestionnaireResponse($id, $effectiveDate)
        )
    
    let $decor                  := $storedQuestionnaire/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    let $projectLanguage        := $decor/project/@defaultLanguage

    let $check                  :=
        if (empty($storedQuestionnaire)) then (
            error($errors:BAD_REQUEST, $label || ' with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
        else
        if (count($storedQuestionnaire) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple results for id ' || $id || ' effectiveDate ' || $effectiveDate || '. Expected 1..1. Alert your administrator as this should not be possible.')
        )
        else ()
    
    let $lock                   := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())
    
    let $check                  :=
        if ($storedQuestionnaire[@statusCode = $qapi:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $qapi:STATUSCODES-FINAL)]) then () else 
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('The ', name($storedQuestionnaire), ' cannot be patched while it has one of status: ', string-join($qapi:STATUSCODES-FINAL, ', '), if ($storedQuestionnaire/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-SCENARIOS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify questionnaires in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this ' || name($storedQuestionnaire) || ' (anymore). Get a lock first.'))
        )
    let $check                  :=
        if ($data[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $data/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    let $checkn                 := 
        if ($data/parameter[@path = '/name']) then
            if (count($storedQuestionnaire/name) - count($data/parameter[@op = 'remove'][@path = '/name']) + count($data/parameter[@op = ('add', 'replace')][@path = '/name']) ge 1) then () else (
                'A ' || name($storedQuestionnaire) || ' SHALL have at least one name. You cannot remove every name.'
            )
        else ()
    
    let $fhirVersion            := ($data/parameter[@path = '/fhirVersion']/@value, $storedQuestionnaire/@fhirVersion)[1]
    let $fhirSchema             := concat('xmldb:exist:///db/apps/fhir/',$fhirVersion,'/resources/schemas/fhir-single.xsd')
    let $xsd                    := 
        <xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns="http://hl7.org/fhir" xmlns:xhtml="http://www.w3.org/1999/xhtml" xmlns:xml="http://www.w3.org/XML/1998/namespace" targetNamespace="http://hl7.org/fhir" elementFormDefault="qualified">
            <xs:include schemaLocation="{$fhirSchema}"/>
            
            <xs:element name="experimental" type="boolean"/>
            <xs:element name="subjectType" type="code"/>
            <xs:element name="publisher" type="string"/>
            <xs:element name="contact" type="ContactDetail"/>
            <xs:element name="useContext" type="UsageContext"/>
            <xs:element name="jurisdiction" type="CodeableConcept"/>
            <xs:element name="purpose" type="markdown"/>
            <xs:element name="copyright" type="markdown"/>
            <xs:element name="code" type="Coding"/>
        </xs:schema>
    let $check                  :=
        if (doc-available($fhirSchema)) then () else (
            error($errors:SERVER_ERROR, 'Unsupported FHIR version ' || $fhirVersion || ' found on result or FHIR server is lacking ' || substring-after($fhirSchema, 'apps/') || '. Alert your administrator as this should not be possible.')
        )
    
    let $check                  :=
        for $param in $data/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($storedQuestionnaire, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($storedQuestionnaire/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri' return (
                if ($storedQuestionnaire/f:*/f:url/@value) then
                    if ($storedQuestionnaire/f:*/f:url[@value = $value]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. You cannot set a different value for this attribute than the content has: ' || string-join($storedQuestionnaire/f:*/f:url/@value, ', ')
                    )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. You cannot set a value for this attribute if the content does not have .url value'
                )
            )
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/fhirVersion'
            return (
                if ($param[@value = $qapi:FHIROBJECT-FORMAT/enumeration/@value]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. fhirVersion has unsupported value ' || string-join($param/@value, ', ') || '. SHALL be one of: ' || string-join($qapi:FHIROBJECT-FORMAT/enumeration/@value, ', ')
                )
            )
            case '/enhanced'
            return (
                if ($op = 'remove') then () else 
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false.'
                )
            )
            case '/name'
            case '/desc'
            return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then
                    if ($param/value/*[name() = $pathpart][matches(@language, '[a-z]{2}-[A-Z]{2}')]) then 
                        if ($param[@op = 'remove'] | $param/value/*[name() = $pathpart][.//text()[not(normalize-space() = '')]]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have contents.'
                        )
                    else (
                        'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have a language with pattern ll-CC.'
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/classification' return (
                if ($param[count(value/classification) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/classification/tag[not(string(string-join(., '')) = '')]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Classification SHALL have at least one non-empty tag'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/relationship' return (
                if ($param[count(value/relationship) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/relationship[@type = $qapi:RELATIONSHIP-TYPES/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Relationship has unsupported .type value ' || string-join($param/value/relationship/@type, ', ') || '. SHALL be one of: ' || string-join($qapi:RELATIONSHIP-TYPES/enumeration/@value, ', ')
                        ),
                        if ($param/value/relationship[@ref]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship SHALL have ref property'
                        ),
                        if (name($storedQuestionnaire) = 'questionnaireresponse') then
                            if ($param/value/relationship[@type = 'DRIV']) then
                                'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship type ' || $param/value/relationship/@type || ' is not compatible with a ' || name($storedQuestionnaire)
                            else (
                                let $target := utillib:getQuestionnaire($param/value/relationship/@ref, $param/value/relationship/@flexibility)
                                return
                                if (empty($target)) then 
                                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship ' || $param/value/relationship/@ref || ' flexibility ' || $param/value/relationship/@flexibility || ' is not found'
                                else
                                if ($target[name() = 'questionnaire']) then () else (
                                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship ' || $param/value/relationship/@ref || ' flexibility ' || $param/value/relationship/@flexibility || ' is not a questionnaire but a ' || name($target[1])
                                )
                            )
                        else
                        if ($param/value/relationship[@type = 'ANSW'] and name($storedQuestionnaire) = 'questionnaire') then
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship type ' || ($param/value/relationship/@type)[1] || ' is not compatible with a ' || name($storedQuestionnaire)
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/Questionnaire/experimental' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*:experimental) = 1]) then
                    if ($op = 'remove') then () else (
                        let $data   := qapi:addFhirNamespace($param/value/*:experimental)
                        let $r      := validation:jaxv-report($data, $xsd)
                        return
                        if ($r/status='invalid') then
                            error($errors:BAD_REQUEST, 'FHIR contents are not schema compliant: ' || serialize($r))
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one experimental under value. Found ' || count($param/value/*:experimental) 
                )
            )
            case '/Questionnaire/subjectType' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*:subjectType) = 1]) then
                    if ($op = 'remove') then () else (
                        let $data   := qapi:addFhirNamespace($param/value/*:subjectType)
                        let $r      := validation:jaxv-report($data, $xsd)
                        return
                        if ($r/status='invalid') then
                            error($errors:BAD_REQUEST, 'FHIR contents are not schema compliant: ' || serialize($r))
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one subjectType under value. Found ' || count($param/value/*:subjectType) 
                )
            )
            case '/Questionnaire/publisher' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*:publisher) = 1]) then
                    if ($op = 'remove') then () else (
                        let $data   := qapi:addFhirNamespace($param/value/*:publisher)
                        let $r      := validation:jaxv-report($data, $xsd)
                        return
                        if ($r/status='invalid') then
                            error($errors:BAD_REQUEST, 'FHIR contents are not schema compliant: ' || serialize($r))
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one publisher under value. Found ' || count($param/value/*:publisher) 
                )
            )
            case '/Questionnaire/contact' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*:contact) = 1]) then
                    if ($op = 'remove') then () else (
                        let $data   := qapi:addFhirNamespaceToContactDetail($param/value/*:contact)
                        let $r      := validation:jaxv-report($data, $xsd)
                        return
                        if ($r/status='invalid') then
                            error($errors:BAD_REQUEST, 'FHIR contents are not schema compliant: ' || serialize($r))
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one contact under value. Found ' || count($param/value/*:contact) 
                )
            )
            case '/Questionnaire/copyright' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*:copyright) = 1]) then
                    if ($op = 'remove') then () else (
                        let $data   := qapi:addFhirNamespace($param/value/*:copyright)
                        let $r      := validation:jaxv-report($data, $xsd)
                        return
                        if ($r/status='invalid') then
                            error($errors:BAD_REQUEST, 'FHIR contents are not schema compliant: ' || serialize($r))
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one copyright under value. Found ' || count($param/value/*:copyright) 
                )
            )
            case '/Questionnaire/code' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*:code) = 1]) then
                    if ($op = 'remove') then () else (
                        let $data   := qapi:addFhirNamespaceToCoding($param/value/*:code)
                        let $r      := validation:jaxv-report($data, $xsd)
                        return
                        if ($r/status='invalid') then
                            error($errors:BAD_REQUEST, 'FHIR contents are not schema compliant: ' || serialize($r))
                        else ()
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one code under value. Found ' || count($param/value/*:code) 
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     let $check                 :=
        if (empty($checkn) and empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join (($checkn, $check), ' '))
        )
    
    let $intention              := if ($storedQuestionnaire[@statusCode = 'final']) then 'patch' else 'version'
    let $objectType             := 
      switch (name($storedQuestionnaire))
      case 'questionnaireresponse' return $decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE
      default return $decorlib:OBJECTTYPE-QUESTIONNAIRE
    let $update                 := histlib:AddHistory($authmap?name, $objectType, $projectPrefix, $intention, $storedQuestionnaire)
    
    let $update                 :=
        for $param in $data/parameter
        return
            switch ($param/@path)
            case '/statusCode'
            case '/expirationDate'
            case '/officialReleaseDate'
            case '/canonicalUri'
            case '/versionLabel' 
            case '/fhirVersion'
            case '/enhanced' return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := attribute {$attname} {$param/@value}
                let $stored   := $storedQuestionnaire/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/name'
            case '/desc'
            return (
                (: only one per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
                let $stored   := $storedQuestionnaire/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/classification' 
            return (
                (: only one :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := 
                    if ($param/value/*[name() = $elmname]) then
                        <classification>
                        {
                            $param/value/*[name() = $elmname]/tag[not(string(string-join(., '')) = '')]
                        }
                        </classification>
                    else ()
                let $stored   := $storedQuestionnaire/*[name() = $elmname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/relationship' return (
                (: only one per template or model, insert before others :)
                let $new      := utillib:prepareDatasetRelationshipForUpdate($param/value/relationship)
                let $stored   := 
                    if ($new[@type = 'ANSW'] | $new[@type = 'DRIV']) then 
                        $storedQuestionnaire/relationship[@type = $new/@type]
                    else (
                        $storedQuestionnaire/relationship[@ref = $new/@ref]
                    )
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then (update insert $new preceding $stored[1], update delete $stored) else (update insert $new into $storedQuestionnaire)
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            (: datatype boolean :)
            case '/Questionnaire/experimental' 
            return (
                let $new      := qapi:addFhirNamespace($param/value/*:experimental)
                let $stored   := $storedQuestionnaire/f:Questionnaire/f:experimental
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire/f:Questionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            (: datatype code :)
            case '/Questionnaire/subjectType' return (
                let $new      := qapi:addFhirNamespace($param/value/*:subjectType)
                let $stored   := $storedQuestionnaire/f:Questionnaire/f:subjectType[@value = $param/value/*:subjectType/@value]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire/f:Questionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            (: datatype string :)
            case '/Questionnaire/publisher' return (
                let $new      := qapi:addFhirNamespace($param/value/*:publisher)
                let $stored   := $storedQuestionnaire/f:Questionnaire/f:publisher
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire/f:Questionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            (: datatype ContactDetail :)
            case '/Questionnaire/contact' return (
                let $new      := qapi:addFhirNamespaceToContactDetail($param/value/*:contact)
                let $stored   := $storedQuestionnaire/f:Questionnaire/f:contact[f:name[@value = $new/f:name/@value] | .[empty(f:name)][f:telecom/f:value/@value = $new[empty(f:name)]/f:telecom/f:value/@value]]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire/f:Questionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            (: datatype markdown :)
            case '/Questionnaire/copyright' return (
                let $new      := qapi:addFhirNamespace($param/value/*:copyright)
                let $stored   := $storedQuestionnaire/f:Questionnaire/f:copyright
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire/f:Questionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            (: datatype Coding :)
            case '/Questionnaire/code' return (
                let $new      := qapi:addFhirNamespaceToCoding($param/value/*:code)
                let $stored   := $storedQuestionnaire/f:Questionnaire/f:code[f:system/@value = $new/*:system/@value][f:code/@value = $new/*:code/@value]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire/f:Questionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
    
    let $update                 :=
        if ($storedQuestionnaire[@lastModifiedDate]) then 
            update value $storedQuestionnaire/@lastModifiedDate with substring(string(current-dateTime()), 1, 19)
        else (
            update insert attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)} into $storedQuestionnaire
        )
    
    (: after all the updates, the XML Schema order is likely off. Restore order - TODO: externalize into separate function based on FHIR version :)
    let $preparedQuestionnaire  := 
        element {name($storedQuestionnaire)} {
            $storedQuestionnaire/(@* except @experimental),
            $storedQuestionnaire/name[@language = $projectLanguage],
            $storedQuestionnaire/name[not(@language = $projectLanguage)],
            $storedQuestionnaire/desc[@language = $projectLanguage],
            $storedQuestionnaire/desc[not(@language = $projectLanguage)],
            $storedQuestionnaire/classification,
            $storedQuestionnaire/relationship,
            if ($doQuestionnaire and $storedQuestionnaire/f:Questionnaire) then (
                for $q in $storedQuestionnaire/f:Questionnaire
                return
                    switch (tokenize($fhirVersion, '\.')[1])
                    case '4' return qapi:rebuildAndSyncQuestionnaireR4($q, $storedQuestionnaire, $projectLanguage)
                    case '5' return qapi:rebuildAndSyncQuestionnaireR5($q, $storedQuestionnaire, $projectLanguage)
                    default return ()
            )
            else (
                $storedQuestionnaire/f:*
            )
        }
    
    let $update                 := update replace $storedQuestionnaire with $preparedQuestionnaire
    let $update                 := update delete $lock
    
    (: build map of oid names. is more costly when you do that deep down :)
    let $oidnamemap             := map {}(:qapi:buildOidNameMap($filteredQuestionnaires, $projectLanguage[1]):)
    let $questionnaireResponses := if (name($storedQuestionnaire) = 'questionnaire') then $decor/scenarios/questionnaireresponse else ()
    let $return                 := qapi:questionnaireBasics($preparedQuestionnaire, $projectPrefix, $projectLanguage, $oidnamemap, $questionnaireResponses)
    
    return
        element {name($return)} {
            $return/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($return/*)
        }
};

declare %private function qapi:rebuildAndSyncQuestionnaireR4($q as element(f:Questionnaire), $storedQuestionnaire as element(questionnaire), $projectLanguage as xs:string) as element(f:Questionnaire) {
    <Questionnaire xmlns="http://hl7.org/fhir">
    {
        (: Resource :)
        $q/f:id,
        <meta>
        {
            $storedQuestionnaire/f:versionId,
            <lastUpdated value="{utillib:add-Amsterdam-timezone(xs:dateTime($storedQuestionnaire/@lastModifiedDate))}"/>,
            $storedQuestionnaire/f:source,
            $storedQuestionnaire/f:profile,
            $storedQuestionnaire/f:security,
            $storedQuestionnaire/f:tag
        }
        </meta>,
        $q/f:implicitRules,
        $q/f:language,
        (: DomainResource :)
        $q/f:text,
        $q/f:contained,
        $q/f:extension,
        $q/f:modifierExtension,
        (: Questionnaire :)
        if ($storedQuestionnaire/@canonicalUri) then
            <url value="{$storedQuestionnaire/@canonicalUri}"/>
        else ()
        ,
        (:copy id from DECOR wrapper:)
        <identifier>
            <system value="urn:ietf:rfc:3986"/>
            <value value="urn:oid:{$storedQuestionnaire/@id}"/>
        </identifier>
        ,
        (: other identifiers if any:)
        $q/f:identifier[not(f:value/@value = 'urn:oid:' || $storedQuestionnaire/@id)]
        ,
        if ($storedQuestionnaire/@versionLabel) then
            <version value="{$storedQuestionnaire/@versionLabel}"/>
        else ()
        ,
        (: >= STU5 :)
        (:$q/f:versionAlgorithm,:)
        if ($storedQuestionnaire/*:name) then (
            <name value="{utillib:shortName(($storedQuestionnaire/*:name[@language = $projectLanguage], $storedQuestionnaire/*:name)[1])}">
            {
                for $n in $storedQuestionnaire/*:name[not(@language = $projectLanguage)]
                return
                    <extension url="http://hl7.org/fhir/StructureDefinition/translation">
                        <extension url="lang">
                            <valueCode value="{$n/@language}"/>
                        </extension>
                        <extension url="content">
                            <valueString value="{utillib:shortName(data($n))}"/>
                        </extension>
                    </extension>
            }
            </name>,
            <title value="{($storedQuestionnaire/*:name[@language = $projectLanguage], $storedQuestionnaire/*:name)[1]}">
            {
                for $n in $storedQuestionnaire/*:name[not(@language = $projectLanguage)]
                return
                    <extension url="http://hl7.org/fhir/StructureDefinition/translation">
                        <extension url="lang">
                            <valueCode value="{$n/@language}"/>
                        </extension>
                        <extension url="content">
                            <valueString value="{data($n)}"/>
                        </extension>
                    </extension>
            }
            </title>
        )
        else (
            (: Cardinality R4/STU5 0..1: it would actually be out of sync with the wrapper :)
            (:$q/f:name,
            $q/f:title:)
        )
        ,
        $q/f:derivedFrom,
        $q/f:status,
        $q/f:experimental,
        $q/f:subjectType,
        <date value="{utillib:add-Amsterdam-timezone(xs:dateTime($storedQuestionnaire/@effectiveDate))}"/>,
        $q/f:publisher,
        $q/f:contact,
        if ($storedQuestionnaire/*:desc) then
            <description value="{($storedQuestionnaire/*:desc[@language = $projectLanguage], $storedQuestionnaire/*:desc)[1]}">
            {
                for $n in $storedQuestionnaire/*:desc[not(@language = $projectLanguage)]
                return
                    <extension url="http://hl7.org/fhir/StructureDefinition/translation">
                        <extension url="lang">
                            <valueCode value="{$n/@language}"/>
                        </extension>
                        <extension url="content">
                            <valueString value="{data($n)}"/>
                        </extension>
                    </extension>
            }
            </description>
        else (
            (: Cardinality R4/STU5 0..1: it would actually be out of sync with the wrapper :)
            (:$q/f:description:)
        ),
        $q/f:useContext,
        $q/f:jurisdiction,
        $q/f:purpose,
        $q/f:copyright,
        (: >= STU5 :)
        (:$q/f:copyrightLabel,:)
        if ($storedQuestionnaire/@officialReleaseDate) then
            <approvalDate value="{utillib:add-Amsterdam-timezone(xs:dateTime($storedQuestionnaire/@officialReleaseDate))}"/>
        else (
            (: Cardinality R4/STU5 0..1: it would actually be out of sync with the wrapper :)
            (:$q/f:approvalDate:)
        ),
        $q/f:lastReviewDate,
        $q/f:effectivePeriod,
        for $n in $q/f:code order by lower-case($n/@value) return $n,
        $q/f:item
    }
    </Questionnaire>
};
declare %private function qapi:rebuildAndSyncQuestionnaireR5($q as element(f:Questionnaire), $storedQuestionnaire as element(questionnaire), $projectLanguage as xs:string) as element(f:Questionnaire) {
    <Questionnaire xmlns="http://hl7.org/fhir">
    {
        (: Resource :)
        $q/f:id,
        <meta>
        {
            $storedQuestionnaire/f:versionId,
            <lastUpdated value="{utillib:add-Amsterdam-timezone(xs:dateTime($storedQuestionnaire/@lastModifiedDate))}"/>,
            $storedQuestionnaire/f:source,
            $storedQuestionnaire/f:profile,
            $storedQuestionnaire/f:security,
            $storedQuestionnaire/f:tag
        }
        </meta>,
        $q/f:implicitRules,
        $q/f:language,
        (: DomainResource :)
        $q/f:text,
        $q/f:contained,
        $q/f:extension,
        $q/f:modifierExtension,
        (: Questionnaire :)
        if ($storedQuestionnaire/@canonicalUri) then
            <url value="{$storedQuestionnaire/@canonicalUri}"/>
        else ()
        ,
        (:copy id from DECOR wrapper:)
        <identifier>
            <system value="urn:ietf:rfc:3986"/>
            <value value="urn:oid:{$storedQuestionnaire/@id}"/>
        </identifier>
        ,
        (: other identifiers if any:)
        $q/f:identifier[not(f:value/@value = 'urn:oid:' || $storedQuestionnaire/@id)]
        ,
        if ($storedQuestionnaire/@versionLabel) then
            <version value="{$storedQuestionnaire/@versionLabel}"/>
        else ()
        ,
        (: >= STU5 :)
        $q/f:versionAlgorithm,
        if ($storedQuestionnaire/*:name) then (
            <name value="{utillib:shortName(($storedQuestionnaire/*:name[@language = $projectLanguage], $storedQuestionnaire/*:name)[1])}">
            {
                for $n in $storedQuestionnaire/*:name[not(@language = $projectLanguage)]
                return
                    <extension url="http://hl7.org/fhir/StructureDefinition/translation">
                        <extension url="lang">
                            <valueCode value="{$n/@language}"/>
                        </extension>
                        <extension url="content">
                            <valueString value="{utillib:shortName(data($n))}"/>
                        </extension>
                    </extension>
            }
            </name>,
            <title value="{($storedQuestionnaire/*:name[@language = $projectLanguage], $storedQuestionnaire/*:name)[1]}">
            {
                for $n in $storedQuestionnaire/*:name[not(@language = $projectLanguage)]
                return
                    <extension url="http://hl7.org/fhir/StructureDefinition/translation">
                        <extension url="lang">
                            <valueCode value="{$n/@language}"/>
                        </extension>
                        <extension url="content">
                            <valueString value="{data($n)}"/>
                        </extension>
                    </extension>
            }
            </title>
        )
        else (
            (: Cardinality R4/STU5 0..1: it would actually be out of sync with the wrapper :)
            (:$q/f:name,
            $q/f:title:)
        )
        ,
        $q/f:derivedFrom,
        $q/f:status,
        $q/f:experimental,
        $q/f:subjectType,
        <date value="{utillib:add-Amsterdam-timezone(xs:dateTime($storedQuestionnaire/@effectiveDate))}"/>,
        $q/f:publisher,
        $q/f:contact,
        if ($storedQuestionnaire/*:desc) then
            <description value="{($storedQuestionnaire/*:desc[@language = $projectLanguage], $storedQuestionnaire/*:desc)[1]}">
            {
                for $n in $storedQuestionnaire/*:desc[not(@language = $projectLanguage)]
                return
                    <extension url="http://hl7.org/fhir/StructureDefinition/translation">
                        <extension url="lang">
                            <valueCode value="{$n/@language}"/>
                        </extension>
                        <extension url="content">
                            <valueString value="{data($n)}"/>
                        </extension>
                    </extension>
            }
            </description>
        else (
            (: Cardinality R4/STU5 0..1: it would actually be out of sync with the wrapper :)
            (:$q/f:description:)
        ),
        $q/f:useContext,
        $q/f:jurisdiction,
        $q/f:purpose,
        $q/f:copyright,
        (: >= STU5 :)
        $q/f:copyrightLabel,
        if ($storedQuestionnaire/@officialReleaseDate) then
            <approvalDate value="{utillib:add-Amsterdam-timezone(xs:dateTime($storedQuestionnaire/@officialReleaseDate))}"/>
        else (
            (: Cardinality R4/STU5 0..1: it would actually be out of sync with the wrapper :)
            (:$q/f:approvalDate:)
        ),
        $q/f:lastReviewDate,
        $q/f:effectivePeriod,
        for $n in $q/f:code order by lower-case($n/@value) return $n,
        $q/f:item
    }
    </Questionnaire>
};
declare %private function qapi:addFhirNamespace($ns as node()*) as node()* {
    for $n in $ns
    return
        element {QName('http://hl7.org/fhir', local-name($n))} {
            $n/@id,
            if ($n instance of attribute()) then attribute value {$n} else $n/@value,
            if ($n[self::*:extension | self::*:modifierExtension]) then $n/@url else (),
            qapi:addFhirNamespaceToExtension($n/*:extension | $n/*:modifierExtension),
            qapi:addFhirNamespace($n/(* except ($n/*:extension | $n/*:modifierExtension)))
        }
};
declare %private function qapi:addFhirNamespaceToAddress($ns as node()*) as node()* {
    for $n in $ns
    return
        element {QName('http://hl7.org/fhir', local-name($n))} {
            $n/@id,
            qapi:addFhirNamespaceToExtension($n/*:extension),
            qapi:addFhirNamespace($n/*:use | $n/@use),
            qapi:addFhirNamespace($n/*:type | $n/@type),
            qapi:addFhirNamespace($n/*:text | $n/@text),
            qapi:addFhirNamespace($n/*:line | $n/@line),
            qapi:addFhirNamespace($n/*:city | $n/@city),
            qapi:addFhirNamespace($n/*:district | $n/@district),
            qapi:addFhirNamespace($n/*:state | $n/@state),
            qapi:addFhirNamespace($n/*:postalCode | $n/@postalCode),
            qapi:addFhirNamespace($n/*:country | $n/@country),
            qapi:addFhirNamespaceToPeriod($n/*:period)
        }
};
declare %private function qapi:addFhirNamespaceToCodeableConcept($ns as node()*) as node()* {
    for $n in $ns
    return
        element {QName('http://hl7.org/fhir', local-name($n))} {
            $n/@id,
            qapi:addFhirNamespaceToExtension($n/*:extension),
            qapi:addFhirNamespaceToCoding($n/*:coding),
            qapi:addFhirNamespace($n/*:text | $n/@text)
        }
};
declare %private function qapi:addFhirNamespaceToCoding($ns as node()*) as node()* {
    for $n in $ns
    return
        element {QName('http://hl7.org/fhir', local-name($n))} {
            $n/@id,
            qapi:addFhirNamespaceToExtension($n/*:extension),
            qapi:addFhirNamespace($n/*:system | $n/@system),
            qapi:addFhirNamespace($n/*:version | $n/@version),
            qapi:addFhirNamespace($n/*:code | $n/@code),
            qapi:addFhirNamespace($n/*:display | $n/@display),
            qapi:addFhirNamespace($n/*:userSelected | $n/@userSelected)
        }
};
declare %private function qapi:addFhirNamespaceToContactDetail($ns as node()*) as node()* {
    for $n in $ns
    return
        element {QName('http://hl7.org/fhir', local-name($n))} {
            $n/@id,
            qapi:addFhirNamespaceToExtension($n/*:extension),
            qapi:addFhirNamespace($n/*:name | $n/@name),
            qapi:addFhirNamespaceToContactPoint($n/*:telecom)
        }
};
declare %private function qapi:addFhirNamespaceToContactPoint($ns as node()*) as node()* {
    for $n in $ns
    return
        element {QName('http://hl7.org/fhir', local-name($n))} {
            $n/@id,
            qapi:addFhirNamespaceToExtension($n/*:extension),
            qapi:addFhirNamespace($n/*:system | $n/@system),
            qapi:addFhirNamespace($n/*:value | $n/@value),
            qapi:addFhirNamespace($n/*:use | $n/@use),
            qapi:addFhirNamespace($n/*:rank | $n/@rank),
            qapi:addFhirNamespaceToPeriod($n/*:period)
        }
};
declare %private function qapi:addFhirNamespaceToPeriod($ns as node()*) as node()* {
    for $n in $ns
    return
        element {QName('http://hl7.org/fhir', local-name($n))} {
            $n/@id,
            qapi:addFhirNamespaceToExtension($n/*:extension),
            qapi:addFhirNamespace($n/*:start | $n/@start),
            qapi:addFhirNamespace($n/*:end | $n/@end)
        }
};
declare %private function qapi:addFhirNamespaceToQuantity($ns as node()*) as node()* {
    for $n in $ns
    return
        element {QName('http://hl7.org/fhir', local-name($n))} {
            $n/@id,
            qapi:addFhirNamespaceToExtension($n/*:extension),
            qapi:addFhirNamespace($n/*:value | $n/@value),
            qapi:addFhirNamespace($n/*:comparator | $n/@comparator),
            qapi:addFhirNamespace($n/*:unit | $n/@unit),
            qapi:addFhirNamespace($n/*:system | $n/@system),
            qapi:addFhirNamespace($n/*:code | $n/@code)
        }
};
declare %private function qapi:addFhirNamespaceToExtension($ns as element()*) as node()* {
    for $n in $ns
    return
        if (local-name($n) = ('extension', 'modifierExtension')) then
            element {QName('http://hl7.org/fhir', local-name($n))} {
                $n/@id,
                $n/@url,
                for $sn in $n/*:extension
                return
                    qapi:addFhirNamespaceToExtension($sn)
                ,
                for $sn in $n/(* except $n/*:extension) | $n/@*[starts-with(local-name(), 'value')]
                return
                    switch (local-name($sn))
                    case 'valueBase64Binary'
                    case 'valueBoolean'
                    case 'valueCanonical'
                    case 'valueCode'
                    case 'valueDate'
                    case 'valueDateTime'
                    case 'valueDecimal'
                    case 'valueId'
                    case 'valueInstant'
                    case 'valueInteger'
                    case 'valueInteger64'
                    case 'valueMarkdown'
                    case 'valueOid'
                    case 'valuePositiveInt'
                    case 'valueString'
                    case 'valueTime'
                    case 'valueUnsignedInt'
                    case 'valueUri'
                    case 'valueUrl'
                    case 'valueUuid' return (
                        element {QName('http://hl7.org/fhir', local-name($sn))} {
                            attribute value {($sn/@value, $sn)[1]}
                        }
                    )
                    case 'valueCodeableConcept' return qapi:addFhirNamespaceToCodeableConcept($ns)
                    case 'valueCoding' return qapi:addFhirNamespaceToCoding($ns)
                    case 'valueAge'
                    case 'valueCount'
                    case 'valueDistance'
                    case 'valueDuration'
                    case 'valueMoney'
                    case 'valueQuantity' return qapi:addFhirNamespaceToQuantity($ns)
                    case 'valueAddress' return qapi:addFhirNamespaceToAddress($ns)
                    case 'valuePeriod' return qapi:addFhirNamespaceToPeriod($ns)
                    case 'valueContactPoint' return qapi:addFhirNamespaceToContactPoint($ns)
                    (:case 'valueAnnotation'
                    case 'valueAttachment'
                    case 'valueCodeableReference'
                    case 'valueHumanName'
                    case 'valueIdentifier'
                    case 'valueRange'
                    case 'valueRatio'
                    case 'valueRatioRange'
                    case 'valueReference'
                    case 'valueSampledData'
                    case 'valueSignature'
                    case 'valueTiming'
                    case 'valueContactDetail'
                    case 'valueDataRequirement'
                    case 'valueExpression'
                    case 'valueParameterDefinition'
                    case 'valueRelatedArtifact'
                    case 'valueTriggerDefinition'
                    case 'valueUsageContext'
                    case 'valueAvailability'
                    case 'valueExtendedContactDetail'
                    case 'valueDosage'
                    case 'valueMeta':)
                    default return (
                        error($errors:SERVER_ERROR, 'Unsupported datatype in extension: ' || local-name($sn))
                    )
            }
        else (
            (: emit error? Someone obviously sent us the wrong thing :)
        )
};

(:~ Build a map that has oid as key and human readable version as value :)
declare %private function qapi:buildOidNameMap($questionnaires as element()*, $language as xs:string?) as map(*)? {
    map:merge(
        for $oid in $questionnaires/@id
        let $grp := $oid
        group by $grp
        return
            map:entry($oid[1], utillib:getNameForOID($oid[1], $language, $oid[1]/ancestor::decor))
    )
};

declare function qapi:putQuestionnaire($request as map(*)) {

    let $authmap                        := $request?user
    let $qid                            := $request?parameters?id
    let $qed                            := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $fhirVersion                    := $request?parameters?fhirVersion[string-length() gt 0]
    let $data                           := $request?body
    let $deletelock                     := $request?parameters?deletelock = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $return                         := qapi:putQuestionnaire($authmap, string($qid), $qed, $data, $fhirVersion, $deletelock)
    
    (:let $doQuestionnaire                := utillib:getQuestionnaire($qid, $qed, (), ())
    let $result                         := if ($doQuestionnaire) then (utillib:getQuestionnaire($qid, $qed, (), ()))
    
                                              else(utillib:getQuestionnaireResponse($qid, $qed, (), ())):)
                                              
    
    let $result                         := utillib:getQuestionnaire($qid, $qed, (), ())
   
    return 
        (:qapi:getQuestionnaire($request):)
        if (empty($result)) then () else (
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
        )
};

(:~ Central logic for updating an existing questionnaire or questionnaireresponse

@param $authmap         - required. Map derived from token
@param $id              - required. DECOR questionnaire/@id to update
@param $effectiveDate   - required. DECOR questionnaire/@effectiveDate to update
@param $data            - required. DECOR questionnaire xml element containing everything that should be in the updated questionnaire or quesrtionnaire response
@return concept object as xml with json:array set on elements
:)
declare function qapi:putQuestionnaire($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as item(), $fhirVersion as xs:string?, $deletelock as xs:boolean) {

    let $editedQuestionnaire    := $data
   
    let $storedQuestionnaire    := utillib:getQuestionnaire($id, $effectiveDate)
    let $decor                  := $storedQuestionnaire/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    let $objectName             := 
        if ($storedQuestionnaire) then 
            upper-case(substring(name($storedQuestionnaire), 1, 1)) || substring(name($storedQuestionnaire), 2)
        else (
          'Questionnaire or QuestionnaireResponse'
        )
    
    let $objectType             := 
        if ($storedQuestionnaire[name()= 'questionnaire']) then $decorlib:OBJECTTYPE-QUESTIONNAIRE else 
        if ($storedQuestionnaire[name()= 'questionnaireresponse']) then $decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE else()
    
    let $check                  :=
        if (empty($storedQuestionnaire)) then (
            error($errors:BAD_REQUEST, $objectName || ' with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
        else
        if (count($storedQuestionnaire) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple results for id ' || $id || ' effectiveDate ' || $effectiveDate || '. Expected 1..1. Alert your administrator as this should not be possible.')
        )
         else
        if ($storedQuestionnaire[@statusCode = $qapi:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, $objectName || ' with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' cannot be updated while it one of status: ' || string-join($qapi:STATUSCODES-FINAL, ', '))
        else ()

    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-SCENARIOS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify scenarios in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    
    let $fhirVersion            := if ($fhirVersion) then $fhirVersion else $storedQuestionnaire/@fhirVersion
    let $data                   :=
        typeswitch ($data)
        case document-node() return $data/*
        case element() return $data
        default return (
            let $xsltParameters := ()
            let $xslt   := concat('/db/apps/fhir/',$fhirVersion,'/resources/stylesheets/fhir-json2xml.xsl')
            let $check  :=
                if (doc-available($xslt)) then () else (
                    error($errors:SERVER_ERROR, 'Unsupported FHIR version ' || $fhirVersion || ' found on result or FHIR server is lacking ' || substring-after($xslt, 'apps/') || '. Alert your administrator as this should not be possible.')
                )
            return
                fn:serialize($data, map{'method':'json', 'indent': true(), 'json-ignore-whitespace-text-nodes':'yes'}) =>
                jx:json-to-xml() => transform:transform(doc($xslt), $xsltParameters)
        )
    let $xsd                    := concat('/db/apps/fhir/',$fhirVersion,'/resources/schemas/fhir-single.xsd')
    let $check                  :=
        if (doc-available($xsd)) then (
            let $r  := validation:jaxv-report($data, doc($xsd))
            return
                if ($r/status='invalid') then
                    error($errors:BAD_REQUEST, 'FHIR contents are not schema compliant: ' || serialize($r))
                else ()
        ) 
        else (
            error($errors:SERVER_ERROR, 'Unsupported FHIR version ' || $fhirVersion || ' found on result or FHIR server is lacking ' || substring-after($xsd, 'apps/') || '. Alert your administrator as this should not be possible.')
        )

    let $lock                   := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())

    let $check                  :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this ' || name($storedQuestionnaire) || ' (anymore). Get a lock first.'))
        )                   
    
    let $intention                      :=
        if ($storedQuestionnaire[@statusCode='final']) then 'patch' else 'version'
    let $history                        := histlib:AddHistory($authmap?name, $objectType, $projectPrefix, $intention, $storedQuestionnaire)
        
    let $now                        := substring(string(current-dateTime()), 1, 19)
    let $locks                      := if ($deletelock) then decorlib:getLocks($authmap, $storedQuestionnaire, true()) else ()
    let $update                     := update replace $storedQuestionnaire/f:* with $data
    let $update                     := 
        if ($storedQuestionnaire[@lastModifiedDate]) then
            update replace $storedQuestionnaire/@lastModifiedDate with $now
        else (
            update insert attribute lastModifiedDate {$now} into $storedQuestionnaire
        )
    let $update                     := 
        if ($storedQuestionnaire[@fhirVersion]) then
            update replace $storedQuestionnaire/@fhirVersion with $fhirVersion
        else (
            update insert attribute fhirVersion {$fhirVersion} into $storedQuestionnaire
        )
    let $update                     :=
        if ($data/f:url/@value) then
            if ($storedQuestionnaire[@canonicalUri]) then
                update replace $storedQuestionnaire/@canonicalUri with $data/f:url/@value
            else (
                update insert attribute canonicalUri {$data/f:url/@value} into $storedQuestionnaire
            )  
        else ()
    let $update                     := 
        if ($storedQuestionnaire[@enhanced]) then
            update replace $storedQuestionnaire/@enhanced with 'true'
        else (
            update insert attribute enhanced {'true'} into $storedQuestionnaire
        )
    
    let $delete                     := if ($locks) then update delete $locks else ()
    
    return ()

};
(:~ Input is questionnaire or questionnaireresponse (through recursion). Returns just the DECOR wrapper info, not the actual Q or QR, 
with as children any questionnaireresponse elements that match the parent questionnaire with their first relationship to a questionnaire :)
declare function qapi:questionnaireBasics($qObjects as element()*, $projectPrefix as xs:string?, $projectLanguage as xs:string?, $oidnamemap as item(), $filteredQuestionnaireResponses as element(questionnaireresponse)*) as element()* {
    for $q in $qObjects 
    let $iddisplay              := map:get($oidnamemap, $q/@id)
    let $projectLanguage        := if (empty($projectLanguage)) then ($q/ancestor::decor/project/@defaultLanguage, 'en-US')[1] else $projectLanguage
    return
        element {name($q)}
        {
            attribute uuid {util:uuid()},
            (:Not in use yet: Reference to a Questionnaire Wrapper @id. Questionnaire Wrappers SHALL carry either @id or @ref:)
            $q/@id,
            $q/@ref,
            $q/@effectiveDate,
            $q/@statusCode,
            attribute versionLabel {($q/f:*/f:version/@value, $q/@versionLabel)[not(. = '')][1]},
            attribute expirationDate {$q/@expirationDate},
            attribute officialReleaseDate {$q/@officialReleaseDate},
            if (name($q) = 'questionnaire') then
                attribute canonicalUri {($q/f:*/f:url/@value, $q/@canonicalUri)[not(. = '')][1]}
            else (),
            $q/@lastModifiedDate,
            if (string-length($iddisplay) = 0) then () else attribute iddisplay {$iddisplay},
            if ($q/ancestor::decor/project[@prefix = $projectPrefix]) then () else
            if (empty($q/ancestor::decor/project/@prefix)) then () else (
                attribute ident {$q/ancestor::decor/project/@prefix}
            )
            ,
            $q/@fhirVersion,
            $q/ancestor::decor/@versionDate,
            $q/ancestor::decor/@language,
            (:If the FHIR Questionnaire is contained as a reference, this link points to the source:)
            let $fhirId := concat($q/@id,'--',replace($q/@effectiveDate,'\D',''))
            let $fhirCanonicalBase := serverapi:getServerURLFhirServices()
            return
       
            if ($q/@link) then $q/@link else
            
            if (name($q) = 'questionnaire') then
                attribute link {$fhirCanonicalBase || 'Questionnaire/'|| $fhirId}
            else 
            if (name($q) = 'questionnaireresponse') then
                attribute link {$fhirCanonicalBase || 'QuestionnaireResponse/'|| $fhirId}
            else (),      
            
            (:Identifying the Questionnaire as Enhanced (type is QE), thus edited, modified, enriched.:)
            if (name($q) = 'questionnaire') then
                attribute enhanced {$q/@enhanced = 'true'} | attribute experimental {$q/@experimental = 'true'}
            else ()
            ,
            (: Support http://hl7.org/fhir/StructureDefinition/translation? :)
            if ($q/name) then $q/name else (
                let $n  := ($q/f:*[1]/f:title/@value, $q/f:*[1]/f:name/@value)[not(. = '')][1]
                return
                    <name language="{$projectLanguage}">{data($n)}</name>
            )
            ,
            if ($q/desc) then $q/desc else (
                let $n  := ($q/f:*[1]/f:description/@value)[not(. = '')][1]
                return
                    if ($n) then <desc language="{$projectLanguage}">{data($n)}</desc> else ()
            )
            ,
            $q/classification,
            $q/relationship
            ,
            (: connect questionnaireresponse. Give empty parameter $filteredQuestionnaireResponses to avoid processing any further :)
            for $qr in $filteredQuestionnaireResponses[relationship[@type = 'ANSW'][1][@ref = $q/@id][@flexibility = $q/@effectiveDate]]
            return
                qapi:questionnaireBasics($qr, $projectPrefix, $projectLanguage, $oidnamemap, ())
        }
};