xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html 
:)
(:~ Terminology Report API creates Terminology Reports for a Project (CADTS-aware) :)
module namespace trepapi            = "http://art-decor.org/ns/api/terminology/report";


import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace adterm      = "http://art-decor.org/ns/terminology" at "../../terminology/api/api-terminology.xqm";

declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";


declare %private variable $trepapi:PROGRESS-REPORT-10                := 'Checking request parameters ...';
declare %private variable $trepapi:PROGRESS-DATASET-20               := 'Checking datasets ...';
declare %private variable $trepapi:PROGRESS-SCENARIO-40              := 'Checking scenarios';
declare %private variable $trepapi:PROGRESS-VALUESET-60              := 'Checking valuesets ...';


(:~ Get a terminology report (list) for a given project
    @param $projectPrefix     - required path part which may be its prefix or its oid
    @return as JSON
    @since 2022-11-30
:)
declare function trepapi:getTerminologyReport($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $specificref            := $request?parameters?reportId[string-length() gt 0]
    let $retrieveReport       := $request?parameters?retrieve = true()
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
        
    let $decor                  := if (utillib:isOid($project)) then utillib:getDecorById($project) else utillib:getDecorByPrefix($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-PROJECT)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to create a runtime version of project ', $project, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if (empty($specificref)) then () else if (matches($specificref, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter reportId SHALL be an oid or uuid. Found: ' || $specificref)
        )
    
    let $results                := trepapi:getTerminologyReport($projectPrefix,$specificref,$retrieveReport)
     
    return
        if (empty($results)) then 
            roaster:response(404, ())
        else (
            $results
        )
};

(:~ Create new terminology report and return the listing of terminology reports

@param $project - required. project prefix or id of the project to create the environment for
@return xml listing of terminology reports
:)
declare function trepapi:createTerminologyReportRequest($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    let $check                  :=
        if (empty($request?body)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
        
    let $decor                  := if (utillib:isOid($project)) then utillib:getDecorById($project) else utillib:getDecorByPrefix($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-PROJECT)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to create a runtime version of project ', $project, '. You have to be an active author in the project.'))
        )
    
    let $now                    := substring(xs:string(current-dateTime()), 1, 19)
    let $stamp                  := util:uuid()
    let $author                 := $decor/project/author[@username = $authmap?name]
    let $author                 := if (empty($author)) then $authmap?name else $author
    let $language               := $decor/project/@defaultLanguage
    let $timeStamp              := 'development'
    
    let $check                  :=
        if (count($projectPrefix) = 1) then () else (
            error($errors:BAD_REQUEST, 'Project parameter ' || $project || ' SHALL refer to exactly one project. Found: ' || count($decor))
        )
        
    
    let $terminology-report-request   :=   <terminology-report-request uuid="{$stamp}" for="{$projectPrefix}" on="{$now}" as="{$stamp}" by="{$author}" language="{$language}" progress="Added to process queue ..." progress-percentage="0"/>

    let $write                  := xmldb:store($setlib:strDecorScheduledTasks, $projectPrefix || replace($now, '\D', '') || '.xml', $terminology-report-request)
    let $tt                     := sm:chmod($write, 'rw-rw----')
    
    return
        trepapi:getTerminologyReport($projectPrefix, (),())
};

(:~ 
   Get list of terminology reports or specific report

   @param $project - required. project prefix or id of the project
   @return list of terminology reports or specific report
:)
declare function trepapi:getTerminologyReport($projectPrefix as xs:string*, $specificref as xs:string?, $retrieve as xs:boolean?) {
   if ($retrieve) then (
        let $fullReport := $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix][@as = $specificref]
      return
      $fullReport
      )
    else (
        let $reportrequestF        :=
            if (string-length($specificref) > 0)
            then
                collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix][@as = $specificref]
            else
                collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix]
        let $terminologyReportF              :=
            if (string-length($specificref) > 0)
            then
                $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix][@as = $specificref]
            else
                $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix]
    
        let $count                  := count($terminologyReportF | $reportrequestF)
        
        let $terminologyReportF              :=
            for $c in $terminologyReportF
            let $as   := $c/@as
            group by $as
            order by $c/@on descending
            return $c[1]
        
        let $results                :=
            <list artifact="TERMINOLOGYREPORT" current="{$count}" total="{$count}" all="{$count}" lastModifiedDate="{current-dateTime()}">
            {
                for $c in $reportrequestF
                order by $c/@on descending
                return
                    element {name($c)}
                    {
                        $c/@*
                    }
                ,
                for $c at $count in $terminologyReportF
                return
                    <terminologyReport>
                    {
                        $c/@* except $c/@status,
                        if ($count < 3) then $c/@status else attribute {'status'} {'retired'},
                        $c/dir
                    }
                    </terminologyReport>
            }
            </list>
    
        for $result in $results
        return
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
    )
};


(:~ Process terminology report requests :)
declare function trepapi:process-terminology-report-request($threadId as xs:string, $request as element(terminology-report-request)) {   
    let $logsignature           := 'trepapi:process-terminology-report-request'
    let $mark-busy              := update insert attribute busy {'true'} into $request
    let $progress               := 
        if ($request/@progress) then (
            update value $request/@progress with $trepapi:PROGRESS-REPORT-10 
        )
        else (
            update insert attribute progress {$trepapi:PROGRESS-REPORT-10} into $request
        )
    let $progress               := 
        if ($request/@progress-percentage) then (
            update value $request/@progress-percentage with round((100 div 9) * 1) 
        )
        else (
            update insert attribute progress-percentage {round((100 div 9) * 1)} into $request
        )
    let $timeStamp              := current-dateTime()
    let $projectPrefix          := $request/@for[not(. = '*')]
    let $decor                  := if (empty($projectPrefix)) then () else utillib:getDecorByPrefix($projectPrefix)
    let $language               := $request/@language[not(. = '')]
    let $reportCollection       := concat($setlib:strDecorVersion,'/',substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development')

    return
       
        if (count($decor) = 1) then (
            update value $request/@progress with $trepapi:PROGRESS-DATASET-20,
            update value $request/@progress-percentage with round((100 div 9) * 2),
            let $terminologyReport := 
             <terminologyReport  for="{$request/@for}" on="{$request/@on}" as="{$request/@as}" by="{$request/@by}" status="active">
                  
                  <datasets datasetCount="{count($decor/datasets/dataset)}">
                     {
                     trepapi:traverseDatasets($decor)
                     }
                  </datasets>
                  {
                  update value $request/@progress with $trepapi:PROGRESS-SCENARIO-40,
                  update value $request/@progress-percentage with round((100 div 9) * 4)
                  }
                  <scenarios scenarioCount="{count($decor/scenarios/scenario)}">
                     {
                     for $scenario in $decor/scenarios/scenario
                     return
                     <scenario transactionCount="{count($scenario/transaction)}" json:array="true">
                        {
                        $scenario/name,
                        for $transaction in $scenario/transaction
                        return
                        <transaction json:array="true">
                           {
                           $transaction/name,
                           for $subTransaction in $transaction/transaction
                           return
                           <transaction json:array="true">
                              {
                              $subTransaction/name,
                              for $representingTemplate in $subTransaction/representingTemplate
                              return
                              <representingTemplate conceptCount="{count($representingTemplate//concept)}" json:array="true">
                                 <sourceDataset>
                                    {
                                    $decor//dataset[@id=$representingTemplate/@sourceDataset]/name
                                    }
                                 </sourceDataset>
                                 {
                                 for $concept in $representingTemplate/concept
                                 return
                                 if ($concept/terminologyAssociation)then
                                    trepapi:handleTerminologyAssociation($decor,$concept/terminologyAssociation)
                                 else
                                    <message type="noAssociation" json:array="true">No terminology association defined</message>
                                 }
                              </representingTemplate>
                              }
                           </transaction>
                           }
                        </transaction>
                        }
                     </scenario>
                     }
                  </scenarios>
                  {
                  update value $request/@progress with $trepapi:PROGRESS-VALUESET-60,
                  update value $request/@progress-percentage with round((100 div 9) * 6)
                  }
                  <valueSets valueSetCount="{count($decor/terminology/valueSet[@id])}">
                     {
                      trepapi:traverseValuesets($decor)   
                     }
                  </valueSets>
                </terminologyReport>
                return
                (
                if (not(xmldb:collection-available($reportCollection))) then
                  try {
                      let $ttt    := xmldb:create-collection($setlib:strDecorVersion, concat(substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development'))
                      let $tt     := try { sm:chmod($ttt, 'rwxrwsr-x') } catch * {()}
                      
                      return $ttt
                  } catch * {
                      <error>+++ Create collection failed '{concat(substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development')}': {$err:code}: {$err:description}</error>
                  }
                  else()
                  ,
                  xmldb:store($reportCollection,concat($projectPrefix,'terminology-report-',$request/@as,'.xml'),$terminologyReport)
                  ,
                  xmldb:remove(util:collection-name($request), util:document-name($request))
                  )
                
        )
        else (
            let $message                := 'ERROR. Found ' || count($decor) || ' projects for prefix ' || $projectPrefix || '. Expected: 1. Compile id: ' || $request/@as
            let $log                    := utillib:log-scheduler-event('error', $threadId, $logsignature, $message)
            let $progress               := update value $request/@progress with $message
            
            return ()
        )
    

};


(:~ Deletes terminology report

@param $project - required. project prefix or id of the project
@param $specificref - required. reference to the runtime environmnent
@return xml listing of environments
:)
declare function trepapi:deleteTerminologyReport($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $specificref            := $request?parameters?reportId[string-length() gt 0]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    let $check                  :=
        if (matches($specificref, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter compilationId SHALL be an oid or uuid. Found: ' || $specificref)
        )
    
    let $decor                  := if (utillib:isOid($project)) then utillib:getDecorById($project) else utillib:getDecorByPrefix($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($projectPrefix) = 1) then () else (
            error($errors:BAD_REQUEST, 'Project parameter ' || $project || ' SHALL refer to exactly one project. Found: ' || count($decor))
        )
    
    let $part1                  := concat('xmldb:exist://', $setlib:strDecorVersion, '/', substring($projectPrefix, 1, string-length($projectPrefix) - 1), '/development/')
    let $tmp1                   := 
        if (xmldb:collection-available($part1)) then (
            if (doc-available(concat($part1, $projectPrefix,'terminology-report-', $specificref, '.xml'))) then xmldb:remove($part1, concat($projectPrefix,'terminology-report-', $specificref, '.xml')) else ()
        )
        else () 
    
    let $part1                  := collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix][@as = $specificref]
    let $tmp3                   := if ($part1) then xmldb:remove(util:collection-name($part1), util:document-name($part1)) else ()
    
    return
        roaster:response(204, ())
};

(:
    private functions
    =================
:)

declare %private function trepapi:doReport($decor as element()) as element() {
    (: do the terminology report :)
    let $projectId              := ($decor/project/@id)[1]
    let $projectPrefix          := ($decor/project/@prefix)[1]
    return
        <terminologyreport project="{$projectId}" prefix="{$projectPrefix}"/>
};


(: Traverse datasets :)
declare %private function trepapi:traverseDatasets($decor as element()) as element()* {
   for $dataset in $decor/datasets/dataset
   let $concepts := $dataset//concept[not(parent::conceptList)][@statusCode=('draft','final')]
   return
   <dataset id="{$dataset/@id}" conceptCount="{count($concepts)}" json:array="true">
   {
      $dataset/name,
      for $concept in $concepts
      let $terminologyAssociations := $decor/terminology/terminologyAssociation[@conceptId=$concept/@id]
      return
         <concept json:array="true">
   {
      $concept/@*,
      $concept/name,
      if ($terminologyAssociations) then
         for $association in $terminologyAssociations
         return
         trepapi:handleTerminologyAssociation($decor,$association)
      else
      <message type="noAssociation">No terminology association defined</message>
       ,
       if ($concept/valueDomain/conceptList) then
         let $conceptListAssociation:= $decor/terminology/terminologyAssociation[@conceptId=$concept/valueDomain/conceptList/@id]
         return
         <conceptList>
            {
            (
            if ($conceptListAssociation) then
               for $association in $conceptListAssociation
               return
               trepapi:handleConceptListAssociation($decor,$association)
            else
            <message type="noValuesetAssociated">No valueset associated with conceptlist</message>
            ,
            for $listConcept in $concept/valueDomain/conceptList/concept
               let $listConceptAssociations := $decor/terminology/terminologyAssociation[@conceptId=$listConcept/@id]
               return
               <listConcept>
               {
               $listConcept/@id,
               $listConcept/name,
               if($listConceptAssociations) then
                  for $association in $listConceptAssociations
                  return
                  trepapi:handleTerminologyAssociation($decor,$association)
               
               else
               <message type="noValuesetItem">No valueset item associated with concept in list</message>
             }
               </listConcept>
             )
             }
          </conceptList>
       else()

    }  
   </concept>
    }  
   </dataset>
};


(: handle terminology association :)
declare %private function trepapi:handleTerminologyAssociation($decor as element(),$terminologyAssociation as element()) as element(){
   (: check if codesystem is in project :)
   if (starts-with($terminologyAssociation/@codeSystem,$decor/project/@id)) then
      let $localConcept :=  $decor//codedConcept[@code=$terminologyAssociation/@code][ancestor::codeSystem/@id=$terminologyAssociation/@codeSystem]
      return
      if ($localConcept) then
         trepapi:checkConcept($terminologyAssociation/@displayName,$localConcept)
      else
         (: check if code system is present :)
         if ($decor//codeSystem[@id=$terminologyAssociation/@codeSystem]) then
            <message type="conceptNotFound">Concept not found</message>
         else
            let $codeSystemName := if ($terminologyAssociation/@codeSystemName) then $terminologyAssociation/@codeSystemName else '?'
            return
            <message type="codesystemNotfound">Codesystem not found</message>
   else
      (: handle SNOMED post coordinated codes :)
      if ($terminologyAssociation/@codeSystem='2.16.840.1.113883.6.96') then
         if ($terminologyAssociation/@code castable as xs:integer) then
            let $snomedConcept := collection($setlib:strCodesystemStableData)//concept[@code=$terminologyAssociation/@code][ancestor::browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem]
            return
            if ($snomedConcept) then
               trepapi:checkConcept($terminologyAssociation/@displayName,$snomedConcept)
            else
               (: check if code system is present :)
               if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem) then
                  <message type="conceptNotFound">Concept not found</message>
               else
                  <message type="codesystemNotfound">Codesystem not found</message>
         else<nop/>
         
      else
         let $codeSystemConcept := collection($setlib:strCodesystemStableData)//concept[@code=$terminologyAssociation/@code][ancestor::browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem]
         return
         if ($codeSystemConcept) then
            trepapi:checkConcept($terminologyAssociation/@displayName,$codeSystemConcept)
         else
            (: check if code system is present :)
            if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem) then
               <message type="conceptNotFound">Concept not found</message>
            else
               let $codeSystemName := if ($terminologyAssociation/@codeSystemName) then $terminologyAssociation/@codeSystemName else '?'
               return
               <message type="codesystemNotfound">Codesystem not found</message>
};


(: handle conceptlist association :)
declare %private function trepapi:handleConceptListAssociation($decor as element(), $conceptListAssociation as element()) as element(){
   (: check if valueset is in project :)
   if (starts-with($conceptListAssociation/@valueSet,$decor/project/@id)) then
      let $valueSet := $decor/terminology/valueSet[@id=$conceptListAssociation/@valueSet]
      return
      if ($valueSet) then
         <message type="ok" json:array="true">OK</message>
      else
         <message type="valuesetNotFound" json:array="true">Valueset not found</message>
   else()

};

declare %private function trepapi:checkConcept($displayName as xs:string, $concept as element()) as element(){
   let $message :=
      if ($concept/@statusCode='retired') then
         <message type="conceptRetired">Concept retired</message>
      else if ($concept/@statusCode='draft') then
         <message type="conceptDraft">Concept is in draft</message>
      else if ($concept/@statusCode='experimental') then
         <message type="conceptExperimental">Concept is experimental</message>
      else
         if ($displayName=$concept/designation) then
            <message type="ok">OK</message>
         else 
            let $designations := $concept/designation
            return
            <message type="noMatchingDesignation">Display name does not match concept designation
            {
            for $designation in $designations
            return
            <designation json:array="true">
            {$designation/@*,$designation/text()}
            </designation>
            }
            </message>
            

   return
   $message
};


(: Traverse valuesets :)
declare %private function trepapi:traverseValuesets($decor as element()) as element()*{
   for $valueSet in $decor/terminology/valueSet[@id]
   return
   <valueSet conceptCount="{count($valueSet/conceptList/concept)}" json:array="true">
      {
      $valueSet/@displayName,
      for $concept in $valueSet/conceptList/concept
         return
         <concept>
         {
         $concept/@*,
         (: check if codesystem is in project :)
         if (starts-with($concept/@codeSystem,$decor/project/@id)) then
            let $localConcept :=  $decor//codedConcept[@code=$concept/@code][ancestor::codeSystem/@id=$concept/@codeSystem]
            return
            if ($localConcept) then
               trepapi:checkConcept($concept/@displayName,$localConcept)
            else
               (: check if code system is present :)
               if ($decor//codeSystem[@id=$concept/@codeSystem]) then
                  <message type="conceptNotFound" json:array="true">Concept not found</message>
               else
                  let $codeSystemName := if ($concept/@codeSystemName) then $concept/@codeSystemName else '?'
                  return
                  <message type="codesystemNotfound" json:array="true">Codesystem not found</message>
         else
            (: handle SNOMED post coordinated codes :)
            if ($concept/@codeSystem='2.16.840.1.113883.6.96') then
               if ($concept/@code castable as xs:integer) then
                  let $snomedConcept := collection($setlib:strCodesystemStableData)//concept[@code=$concept/@code][ancestor::browsableCodeSystem/@oid=$concept/@codeSystem]
                  return
                  if ($snomedConcept) then
                     trepapi:checkConcept($concept/@displayName,$snomedConcept)
                  else
                     (: check if code system is present :)
                     if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$concept/@codeSystem) then
                        <message type="conceptNotFound" json:array="true">Concept not found</message>
                     else
                        <message type="codesystemNotfound" json:array="true">Codesystem not found</message>
               else 
                  let $expressionTest := trepapi:parseSnomedExpression(replace($concept/@code,'\|(.*?)\|',''))
               return
            if ($expressionTest//message) then
                     $expressionTest
                   else
               (
                     <message type="ok">OK</message>,
                     $expressionTest
                     )
               
            else
               let $codeSystemConcept := collection($setlib:strCodesystemStableData)//concept[@code=$concept/@code][ancestor::browsableCodeSystem/@oid=$concept/@codeSystem]
               return
               if ($codeSystemConcept) then
                  trepapi:checkConcept($concept/@displayName,$codeSystemConcept)
               else
                  (: check if code system is present :)
                  if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$concept/@codeSystem) then
                     <message type="conceptNotFound" json:array="true">Concept not found</message>
                  else
                     let $codeSystemName := if ($concept/@codeSystemName) then $concept/@codeSystemName else '?'
                     return
                     <message type="codesystemNotfound" json:array="true">Codesystem not found</message>
             }
            </concept>         
            }
   </valueSet>

};

declare function trepapi:parseSnomedExpression ($expression as xs:string) {
   (: check for multiple focus concepts :)
   if (normalize-space(substring-before($expression,'+')) castable as xs:integer) then
      let $focusCode := normalize-space(substring-before($expression,'+'))
      return
      (
      <concept code="{$focusCode}">{trepapi:getSnomedConceptDesignations($focusCode)}</concept>,
      <focus/>,
      trepapi:parseSnomedExpression(normalize-space(substring-after($expression,'+')))
      )
   else if (normalize-space(substring-before($expression,':')) castable as xs:integer) then
      let $conceptCode := normalize-space(substring-before($expression,':'))
      let $refinements := tokenize(normalize-space(substring-after($expression,':')),',')
      return
      (
      <concept code="{$conceptCode}">{trepapi:getSnomedConceptDesignations($conceptCode)}</concept>,
         
         for $refinement in $refinements
         return
         if (normalize-space(substring-before($refinement,'=')) castable as xs:integer) then
         let $refinementCode := normalize-space(substring-before($refinement,'='))
         return
         <refinement>
         <concept code="{$refinementCode}">{trepapi:getSnomedConceptDesignations($refinementCode)}</concept>
            {
            if (normalize-space(substring-after($refinement,'=')) castable as xs:integer) then
               let $equalsCode :=normalize-space(substring-after($refinement,'='))
               return
               <equals code="{$equalsCode}">{trepapi:getSnomedConceptDesignations($equalsCode)}</equals>
            else
            <message type="cannotParseExpression">Expression cannot be parsed</message>
            }
         </refinement>
         else <message type="cannotParseExpression">Expression cannot be parsed</message>
         
      
      )
   else 
      <message type="cannotParseExpression">Expression cannot be parsed</message>
};

declare function trepapi:getSnomedConceptDesignations($code as xs:integer) as element()*{
   let $concept := collection(concat($setlib:strCodesystemStableData,'/external/snomed'))//concept[@code=$code]
   return
   if ($concept) then
       if ($concept/@statusCode='retired') then
         <message type="conceptRetired">Concept retired</message>
      else if ($concept/@statusCode='draft') then
         <message type="conceptDraft">Concept is in draft</message>
      else if ($concept/@statusCode='experimental') then
         <message type="conceptExperimental">Concept is experimental</message>
      else  
         let $designations := $concept/designation
         return
         $designations[@use='pref']
   else
      <message type="conceptNotFound">Concept not found</message>
};
