xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
:)
(:~ Questionnaire and QuestionnaireResponse API allows read, create, update on DECOR links into HL7 FHIR Questionnaire and QuestionnaireResponse :)
module namespace qapi              = "http://art-decor.org/ns/api/questionnaire";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "/db/apps/api/modules/library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "/db/apps/api/modules/library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";
import module namespace jx          = "http://joewiz.org/ns/xquery/json-xml" at "library/json-xml.xqm";
import module namespace serverapi   = "http://art-decor.org/ns/art-decor-server" at "server-api.xqm";
import module namespace router      = "http://e-editiones.org/roaster/router";

declare namespace json      = "http://www.json.org";
declare namespace f         = "http://hl7.org/fhir";

(:~ All functions support their own override, but this is the fallback for the maximum number of results returned on a search :)
declare %private variable $qapi:maxResults              := 50;
declare %private variable $qapi:ITEM-STATUS             := utillib:getDecorTypes()/ItemStatusCodeLifeCycle;
declare %private variable $qapi:STATUSCODES-FINAL       := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $qapi:RELATIONSHIP-TYPES      := utillib:getDecorTypes()/QuestionnaireRelationshipTypes;
declare %private variable $qapi:FHIROBJECT-FORMAT       := utillib:getDecorTypes()/FhirObjectFormat;
declare %private variable $qapi:FHIRRESOURCE-TYPE       := utillib:getDecorTypes()/FHIRResourceType;
declare %private variable $qapi:ADDRLINE-TYPE           := utillib:getDecorTypes()/AddressLineType;

(:~ Returns a list of zero or more questionnaire and questionnaireresponses
    
@param $projectPrefix           - required. limits scope to this project only
@param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
@param $projectLanguage         - optional parameter to select from a specific compiled language
@param $objectId                - optional. Filter output to just those based on this questionnaire/relationship/@ref. For each questionnaire, also all questionnaireResponses are returned unless omitResponses=true
@param $objectEffectiveDate     - optional. Filter output to just those based on this questionnaire/relationship/@flexibility. relevant when objectId has a value.
@param $sort                    - optional parameter to indicate sorting. Only option 'name'
@param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
@return all live repository/non-private questionnaire and questionnaireResponse as JSON
@since 2020-05-03
:)
declare function qapi:getQuestionnaireList($request as map(*)) {

    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $objectId               := $request?parameters?objectId[not(. = '')]
    let $objectEffectiveDate    := $request?parameters?objectEffectiveDate[not(. = '')]
    let $sort                   := $request?parameters?sort[not(. = '')]
    let $sortorder              := $request?parameters?sortorder[not(. = '')]
    let $max                    := $request?parameters?max[not(. = '')]
    
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have parameter prefix')
        else ()
    return
        qapi:getQuestionnaireList($projectPrefix, $projectVersion, $projectLanguage, $objectId, $objectEffectiveDate, $sort, $sortorder, $max)

};

declare function qapi:getQuestionnaireList($projectPrefix as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $objectId as xs:string?, $objectEffectiveDate as xs:string?, $sort as xs:string?, $sortorder as xs:string?, $max as xs:string?) {
    let $sort                       := $sort[string-length() gt 0]
    let $sortorder                  := $sortorder[. = 'descending']
    
    let $projectPrefix              := $projectPrefix[not(. = '')]
    let $projectVersion             := $projectVersion[not(. = '')]
    let $decor                      := if (empty($projectPrefix)) then () else utillib:getDecorByPrefix($projectPrefix, $projectVersion, $projectLanguage)
    
    let $allcnt                     := count($decor/scenarios/questionnaire)
    
    let $objectId                   := $objectId[not(. = '')]
    let $objectEffectiveDate        := $objectEffectiveDate[not(. = '')]
    
    let $filteredQuestionnaires     := 
        if (empty($objectId)) then $decor/scenarios/questionnaire | $decor/scenarios/questionnaireresponse else (
            if (empty($objectEffectiveDate)) then (
                let $object                     :=
                    if (empty($objectId)) then () else (
                        utillib:getScenario($objectId, $objectEffectiveDate, $projectVersion, $projectLanguage) |
                        utillib:getTransaction($objectId, $objectEffectiveDate, $projectVersion, $projectLanguage) |
                        utillib:getDataset($objectId, $objectEffectiveDate, $projectVersion, $projectLanguage)
                    )
                (: get questionnaires and questionnaireresponses related to object :)
                let $qs          := 
                    if ($object) then (
                        $decor/scenarios/questionnaire[relationship[@ref = $object/@id][@flexibility = $object/@effectiveDate]] |
                        $decor/scenarios/questionnaireresponse[relationship[@ref = $object/@id][@flexibility = $object/@effectiveDate]]
                    )
                    else ( 
                        $decor/scenarios/questionnaire[relationship[@ref = $objectId]] | 
                        $decor/scenarios/questionnaireresponse[relationship[@ref = $objectId]]
                    )
                (: get questionnaireresponses related to questionnaires :)
                let $qr         := 
                    for $q in $qs
                    return
                        $decor/scenarios/questionnaireresponse[relationship[@ref = $q/@id][@flexibility = $q/@effectiveDate]]
                
                return $qs | $qr
            )
            else (
                (: get questionnaires and questionnaireresponses related to object :)
                let $qs          := 
                    $decor/scenarios/questionnaire[relationship[@ref = $objectId][@flexibility = $objectEffectiveDate]] |
                    $decor/scenarios/questionnaireresponse[relationship[@ref = $objectId][@flexibility = $objectEffectiveDate]]
                (: get questionnaireresponses related to questionnaires :)
                let $qr         := 
                    for $q in $qs
                    return
                        $decor/scenarios/questionnaireresponse[relationship[@ref = $q/@id][@flexibility = $q/@effectiveDate]]
                
                return $qs | $qr
            )
        )
    
    let $projectLanguage      := 
        if (empty($projectLanguage)) 
        then 
            if (empty($projectPrefix)) then
                $filteredQuestionnaires/ancestor::decor/project/@defaultLanguage
            else
                $decor/project/@defaultLanguage
        else $projectLanguage
    
    (: build map of oid names. is more costly when you do that deep down :)
    let $oidnamemap             := map {}(:qapi:buildOidNameMap($filteredQuestionnaires, $projectLanguage[1]):)
    
    let $filteredQuestionnaireResponses := $filteredQuestionnaires[self::questionnaireresponse]
    let $filteredQuestionnaires         := $filteredQuestionnaires[self::questionnaire]
    
    (:can sort on indexed db content -- faster:)
    let $results                :=
        switch ($sort) 
        case 'effectiveDate'    return 
            if (empty($sortorder)) then 
                for $q in $filteredQuestionnaires order by $q/@effectiveDate return $q
            else
                for $q in $filteredQuestionnaires order by $q/@effectiveDate descending return $q
        case 'name'    return 
            if (empty($sortorder)) then 
                for $q in $filteredQuestionnaires order by lower-case($q/@name) return $q
            else
                for $q in $filteredQuestionnaires order by lower-case($q/@name) descending return $q
        default                 return $filteredQuestionnaires
    
    let $count              := count($results)
    let $max                := if ($max ge 1) then $max else $qapi:maxResults
    
    let $matchedqr          :=
        for $q in $results
        return
            qapi:handleQuestionnaire($q, $projectPrefix, $projectLanguage[1], $oidnamemap, $filteredQuestionnaireResponses, true(), false())
    let $unmatchedqr        :=
        for $qr in $filteredQuestionnaireResponses
        let $relationship := $qr/relationship[@type = 'ANSW'][1]
        return
            if ($matchedqr[@id = $relationship/@ref][@effectiveDate = $relationship/@flexibility]) then () else qapi:handleQuestionnaire($qr, $projectPrefix, $projectLanguage[1], $oidnamemap, (), true(), false())
    let $pendingRequests    := ()
        (:for $pendingRequest in collection($setlib:strDecorScheduledTasks)/questionnaire-transform-request
        order by $pendingRequest/@on
        return
            $pendingRequest:)
        
    
    return
    <list artifact="QQQR" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
    {
        utillib:addJsonArrayToElements( ($matchedqr, $unmatchedqr, $pendingRequests) )
    }
    </list>
};

(:~  Return questionnaire or questionnaireresponse based on id and optionally effectiveDate. If effectiveDate is empty, the latest version is returned
:
:   @param $q-id   - required dataset/@id
:   @param $q-ed   - optional dataset/@effectiveDate
:   @return exactly 1 questionnaire or questionnaireresponse or nothing if not found
:   @since 2015-04-27
:)
declare function qapi:getLatestQuestionnaire($request as map(*)) {
    qapi:getQuestionnaire($request)
};

declare function qapi:getLatestQuestionnaireResponse($request as map(*)) {
    qapi:getQuestionnaireResponse($request)
};

declare function qapi:getQuestionnaire($request as map(*)) {

    let $q-id                           := $request?parameters?id
    let $q-ed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    
    let $oidnamemap                   := map {}
    let $projectPrefix                := ()
    let $decorVersion                 := $request?parameters?release
    let $language                     := $request?parameters?language

    let $q                            := utillib:getQuestionnaire($q-id, $q-ed, $decorVersion, $language)
    let $fhirVersion                  := if (empty($q/@fhirVersion)) then '4.0' else $q/@fhirVersion
    let $acceptTypes                  := router:accepted-content-types()
    let $acceptedType                 := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]

    let $results                      := qapi:handleQuestionnaire($q, $projectPrefix, $language, $oidnamemap, $q/../questionnaireresponse, false(), false())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple questionnaires for id '", $q-id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Retrieves DECOR questionnaire history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function qapi:getQuestionnaireHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-QUESTIONNAIRE, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-QUESTIONNAIRE}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

declare function qapi:getQuestionnaireResponse($request as map(*)) {

    let $qr-id                           := $request?parameters?id
    let $qr-ed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $oidnamemap                   := map {}
    let $projectPrefix                := ()
    let $decorVersion                 := $request?parameters?release
    let $language                     := $request?parameters?language

    let $qr                           := utillib:getQuestionnaireResponse($qr-id, $qr-ed, $decorVersion, $language)
    let $fhirVersion                  := if (empty($qr/@fhirVersion)) then '4.0' else $qr/@fhirVersion
    let $acceptTypes                  := router:accepted-content-types()
    let $acceptedType                 := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]
    
    let $results                      := qapi:handleQuestionnaire($qr, $projectPrefix, $language, $oidnamemap, (), false(), false())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple questionnaireresponses for id '", $qr-id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Retrieves DECOR questionnaire response history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function qapi:getQuestionnaireResponseHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:declare function qapi:getQuestionnaire($q-id as xs:string, $q-ed as xs:string?) as element()* {
    qapi:getQuestionnaire($q-id, $q-ed,(),())
};:)

(:~  Return the live questionnaire if param decorVersion is not a dateTime OR 
:   compiled, archived/released DECOR questionnaire or questionnaireresponse based on @id|@effectiveDate and decor/@versionDate.
:   Released/compiled matches could be present in multiple languages. Use the parameters @language to select the right one.
:
:   @param $q-id           - required questionnaire or questionnaireresponse/@id
:   @param $q-ed           - optional questionnaire or questionnaireresponse/@effectiveDate
:   @param $decorVersion    - optional decor/@versionDate
:   @param $language        - optional decor/@language. Only relevant with parameter $decorVersion. Empty uses decor/project/@defaultLanguage, '*' selects all versions, e.g. 'en-US' selects the US English version (if available)
:   @return exactly 1 live questionnaire or questionnaireresponse, or 1 or more archived DECOR version instances or nothing if not found
:   @since 2015-04-29
:)
(:declare function qapi:getQuestionnaire($q-id as xs:string, $q-ed as xs:string?, $decorVersion as xs:string?) as element()* {
    qapi:getQuestionnaire($q-id, $q-ed,$decorVersion,())
};
:)
(:declare function qapi:getQuestionnaire($q-id as xs:string, $q-ed as xs:string?, $decorVersion as xs:string?, $language as xs:string?) as element()* {
    let $decor          :=
        if ($decorVersion[string-length()>0]) then
            let $d      := $setlib:colDecorVersion/decor[@versionDate=$decorVersion]
            return
            if (empty($language)) then
                $d[@language=$d/project/@defaultLanguage]
            else if ($language='*') then
                $d
            else
                $d[@language=$language]
        else (
            $setlib:colDecorData/decor
        )
    let $quest          := $decor//questionnaire[@id = $q-id][ancestor::scenarios] | $decor//questionnaireresponse[@id = $q-id][ancestor::scenarios]
    
    return
        if ($q-ed castable as xs:dateTime) then 
            $quest[@effectiveDate = $q-ed] 
        else 
        if ($quest[2]) then 
            $quest[@effectiveDate = max($quest/xs:dateTime(@effectiveDate))]
        else (
            $quest
        )
};
:)
declare function qapi:patchQuestionnaire($request as map(*)) {

    let $authmap                       := $request?user
    let $qid                           := $request?parameters?id
    let $qed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $projectVersion                 := ()
    let $projectLanguage                := ()
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                         := qapi:patchQuestionnaire($authmap, true(), string($qid), $qed, $data)
   
    return 
      
        $return

};

declare function qapi:patchQuestionnaireResponse($request as map(*)) {

    let $authmap                       := $request?user
    let $qid                           := $request?parameters?id
    let $qed                           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $projectVersion                 := ()
    let $projectLanguage                := ()
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $return                         := qapi:patchQuestionnaire($authmap, false(), string($qid), $qed, $data)
   
    return 
      
        $return

};

declare function qapi:patchQuestionnaire($authmap as map(*), $doQuestionnaire as xs:boolean, $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) {

    let $label                  := if ($doQuestionnaire) then 'Questionnaire' else 'QuestionnaireResponse'
    let $storedQuestionnaire    := 
        if ($doQuestionnaire) then 
            utillib:getQuestionnaire($id, $effectiveDate)
        else (
            utillib:getQuestionnaireResponse($id, $effectiveDate)
        )
    
    let $decor                  := $storedQuestionnaire/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    let $projectLanguage        := $decor/project/@defaultLanguage

    let $check                  :=
        if (empty($storedQuestionnaire)) then (
            error($errors:BAD_REQUEST, $label || ' with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
        else
        if (count($storedQuestionnaire) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple results for id ' || $id || ' effectiveDate ' || $effectiveDate || '. Expected 1..1. Alert your administrator as this should not be possible.')
        )
        else ()
    
    let $lock                   := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())
    
    let $check                  :=
        if ($storedQuestionnaire[@statusCode = $qapi:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $qapi:STATUSCODES-FINAL)]) then () else 
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('The ', name($storedQuestionnaire), ' cannot be patched while it has one of status: ', string-join($qapi:STATUSCODES-FINAL, ', '), if ($storedQuestionnaire/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-SCENARIOS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify questionnaires in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this ' || name($storedQuestionnaire) || ' (anymore). Get a lock first.'))
        )
    let $check                  :=
        if ($data[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $data/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    let $checkn                 := 
        if ($data/parameter[@path = '/name']) then
            if (count($storedQuestionnaire/name) - count($data/parameter[@op = 'remove'][@path = '/name']) + count($data/parameter[@op = ('add', 'replace')][@path = '/name']) ge 1) then () else (
                'A ' || name($storedQuestionnaire) || ' SHALL have at least one name. You cannot remove every name.'
            )
        else ()
    
    let $check                  :=
        for $param in $data/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($storedQuestionnaire, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($storedQuestionnaire/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss'
                )
            )
            case '/canonicalUri' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:anyURI) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be a URI'
                )
            )
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/fhirVersion'
            return (
                if ($param[@value = $qapi:FHIROBJECT-FORMAT/enumeration/@value]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. fhirVersion has unsupported value ' || string-join($param/@value, ', ') || '. SHALL be one of: ' || string-join($qapi:FHIROBJECT-FORMAT/enumeration/@value, ', ')
                )
            )
            case '/enhanced'
            return (
                if ($op = 'remove') then () else 
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false'
                )
            )
            case '/name'
            case '/desc'
            case '/copyright'
            return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then
                    if ($param/value/*[name() = $pathpart][matches(@language, '[a-z]{2}-[A-Z]{2}')]) then 
                        if ($param[@op = 'remove'] | $param/value/*[name() = $pathpart][.//text()[not(normalize-space() = '')]]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have contents.'
                        )
                    else (
                        'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. A ' || $pathpart || ' SHALL have a language with pattern ll-CC'
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/classification' return (
                if ($param[count(value/classification) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/classification/tag[not(string(string-join(., '')) = '')]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Classification SHALL have at least one non-empty tag'
                        )
                    )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/relationship' return (
                if ($param[count(value/relationship) = 1]) then 
                    if ($op = 'remove') then () else (
                        if ($param/value/relationship[@type = $qapi:RELATIONSHIP-TYPES/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Relationship has unsupported .type value ' || string-join($param/value/relationship/@type, ', ') || '. SHALL be one of: ' || string-join($qapi:RELATIONSHIP-TYPES/enumeration/@value, ', ')
                        ),
                        if ($param/value/relationship[@ref]) then () else (
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship SHALL have ref property'
                        ),
                        if (name($storedQuestionnaire) = 'questionnaireresponse') then
                            if ($param/value/relationship[@type = 'DRIV']) then
                                'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship type ' || $param/value/relationship/@type || ' is not compatible with a ' || name($storedQuestionnaire)
                            else (
                                let $target := utillib:getQuestionnaire($param/value/relationship/@ref, $param/value/relationship/@flexibility)
                                return
                                if (empty($target)) then 
                                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship ' || $param/value/relationship/@ref || ' flexibility ' || $param/value/relationship/@flexibility || ' is not found'
                                else
                                if ($target[name() = 'questionnaire']) then () else (
                                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship ' || $param/value/relationship/@ref || ' flexibility ' || $param/value/relationship/@flexibility || ' is not a questionnaire but a ' || name($target[1])
                                )
                            )
                        else
                        if ($param/value/relationship[@type = 'ANSW'] and name($storedQuestionnaire) = 'questionnaire') then
                            'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Relationship type ' || ($param/value/relationship/@type)[1] || ' is not compatible with a ' || name($storedQuestionnaire)
                        else ()
                    )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/experimental' return (
                if ($op = 'remove') then () else 
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false'
                )
            )
            case '/subjectType' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/subjectType) = 1]) then
                    if ($param[value/subjectType/@code = $qapi:FHIRRESOURCE-TYPE/enumeration/@value]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $pathpart || ' has unsupported value ' || string-join($param/@value, ', ') || '. SHALL be one of: ' || string-join($qapi:FHIRRESOURCE-TYPE/enumeration/@value, ', ')
                    )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/endorsingAuthority'
            case '/publishingAuthority' return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*[name() = $pathpart]) = 1]) then (
                    if ($param/value/*[name() = $pathpart][@id]) then
                        if ($param/value/*[name() = $pathpart][utillib:isOid(@id)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || ''' ' || $pathpart || '.id SHALL be an OID is present. Found ' || string-join($param/value/*[name() = $pathpart]/@id, ', ')
                        )
                    else (),
                    if ($param/value/*[name() = $pathpart][@name[not(. = '')]]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || ''' ' || $pathpart || '.name SHALL be a non empty string.'
                    ),
                    let $unsupportedtypes := $param/value/*[name() = $pathpart]/addrLine/@type[not(. = $qapi:ADDRLINE-TYPE/enumeration/@value)]
                    return
                        if ($unsupportedtypes) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $pathpart || ' has unsupported addrLine.type value(s) ' || string-join($unsupportedops, ', ') || '. SHALL be one of: ' || string-join($qapi:ADDRLINE-TYPE/enumeration/@value, ', ')
                        else ()
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart])
                )
            )
            case '/code'
            case '/jurisdiction'
            return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*[name() = $pathpart]) = 1]) then
                    if ($op = 'remove') then () else (
                        if ($param/value/*[name() = $pathpart][@code[matches(., '^\S+$')]][utillib:isOid(@codeSystem)][empty(@canonicalUri) or @canonicalUri castable as xs:anyURI]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have code without whitespace, a codeSystem as valid OID, and optionally a canonicalUri as URI. Found: ' || string-join(for $att in $param/value/code/@* return name($att) || ': "' || $att || '" ', ' ')
                        )
                    )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart])
                )
            )
            case '/item'
            return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/item) = 1]) then (
                    let $data     := qapi:handleQuestionnaireItem($param/value/item)
                    let $data     := 
                        element {name($storedQuestionnaire)} {
                            $storedQuestionnaire/@*,
                            $storedQuestionnaire/(node() except item),
                            $data
                        }
                    let $r        := validation:jaxv-report($data, $setlib:docDecorSchema)
                    return
                    if ($r/status='invalid') then
                        error($errors:BAD_REQUEST, 'Questionnaire contents are not schema compliant: ' || serialize($param/value/item) || ' ----- ' || serialize($data) || ' ----- ' || serialize($r))
                    else ()
                )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value containing exactly one object. Found ' || count($param/value/*[name() = $pathpart]/*)
                )
            )
            (:case '/contained'
            return (
                if (not($doQuestionnaire)) then
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported on ' || name($storedQuestionnaire)
                else
                if ($param[count(value/*[name() = $pathpart]/*) = 1]) then (
                    (\: object shall have id that is referred to :\)
                    let $reference    := '#' || value/*[name() = $pathpart]/*/id/@value
                    let $referrers    := $storedQuestionnaire/item//@ref | $data/parameter[not(@op = 'remove')][@path = '/item']/value/item//@ref
                    return
                    if ($op = 'remove') then
                        if ($referrers[. = $ref]) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Contained object SHALL NOT still be referred to in the questionnaire at removal time. Found references ' || $reference
                        else ()
                    else
                    if ($referrers[. = $ref]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Contained object SHALL have a reference to it somewhere in the questionnaire. Expected reference ' || $reference
                    )
                )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value containing exactly one object. Found ' || count($param/value/*[name() = $pathpart]/*) 
                )
            ):)
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported.'
            )
     let $check                 :=
        if (empty($checkn) and empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join (($checkn, $check), '. '))
        )
    
    let $intention              := if ($storedQuestionnaire[@statusCode = 'final']) then 'patch' else 'version'
    let $objectType             := 
        switch (name($storedQuestionnaire))
        case 'questionnaireresponse' return $decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE
        default return $decorlib:OBJECTTYPE-QUESTIONNAIRE
    let $update                 := histlib:AddHistory($authmap?name, $objectType, $projectPrefix, $intention, $storedQuestionnaire)
    
    let $update                 :=
        for $param in $data/parameter
        return
            switch ($param/@path)
            case '/statusCode'
            case '/expirationDate'
            case '/officialReleaseDate'
            case '/canonicalUri'
            case '/versionLabel'
            case '/experimental'
            case '/fhirVersion'
            case '/enhanced' return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := attribute {$attname} {$param/@value}
                let $stored   := $storedQuestionnaire/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/name'
            case '/copyright'
            case '/desc'
            return (
                (: only one per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
                let $stored   := $storedQuestionnaire/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/classification' 
            return (
                (: only one :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := 
                    if ($param/value/*[name() = $elmname]) then
                        <classification>
                        {
                            $param/value/*[name() = $elmname]/tag[not(string(string-join(., '')) = '')]
                        }
                        </classification>
                    else ()
                let $stored   := $storedQuestionnaire/*[name() = $elmname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/relationship' return (
                (: only one per template or model, insert before others :)
                let $new      := utillib:prepareDatasetRelationshipForUpdate($param/value/relationship)
                let $stored   := 
                    if ($new[@type = 'ANSW'] | $new[@type = 'DRIV']) then 
                        $storedQuestionnaire/relationship[@type = $new/@type]
                    else (
                        $storedQuestionnaire/relationship[@ref = $new/@ref]
                    )
                
                return
                switch ($param/@op)
                case 'add' (: fall through :)
                case 'replace' return if ($stored) then (update insert $new preceding $stored[1], update delete $stored) else (update insert $new into $storedQuestionnaire)
                case 'remove' return update delete $stored
                default return ( (: unknown op :) )
            )
            (: datatype code :)
            case '/subjectType' return (
                let $new      := <subjectType code="{$param/value/subjectType/@code}"/>
                let $stored   := $storedQuestionnaire/subjectType[@code = $new/@code]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/endorsingAuthority' return (
                (: only one possible :)
                let $new      := utillib:prepareEndorsingAuthorityForUpdate($param/value/endorsingAuthority)
                let $stored   := $storedQuestionnaire/endorsingAuthority
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/publishingAuthority' return (
                (: only one possible :)
                let $new      := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
                let $stored   := $storedQuestionnaire/publishingAuthority
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            (: datatype Coding :)
            case '/code'
            case '/jurisdiction'
            return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareValueCodingTypeForUpdate($param/value/*[name() = $attname])
                let $stored   := $storedQuestionnaire/*[name() = $attname][@code = $new/@code][@codeSystem = $new/@codeSystem]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/item'
            return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := qapi:handleQuestionnaireItem($param/value/item)
                let $stored   := $storedQuestionnaire/item
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            (:case '/contained'
            return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := $param/value/*[name() = $attname]
                (\: Assumption: should be unique regardless of resource type
                    https://chat.fhir.org/#narrow/stream/179166-implementers/topic/Contained.20Resource.2Eid.3A.20unique.20or.20unique.20per.20type.3F
                :\)
                let $objid    := $new/id/@value
                let $stored   := $storedQuestionnaire/*[name() = $attname]/*/id[@value = $objid]
                
                return
                switch ($param/@op)
                case ('add') (\: fall through :\)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedQuestionnaire
                case ('remove') return update delete $stored
                default return ( (\: unknown op :\) )
            ):)
            default return ( (: unknown path :) )
    
    let $update                 :=
        if ($storedQuestionnaire[@lastModifiedDate]) then 
            update value $storedQuestionnaire/@lastModifiedDate with substring(string(current-dateTime()), 1, 19)
        else (
            update insert attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)} into $storedQuestionnaire
        )
    
    (: after all the updates, the XML Schema order is likely off. Restore order - TODO: externalize into separate function based on FHIR version :)
    
    (: build map of oid names. is more costly when you do that deep down :)
    let $oidnamemap             := map {}(:qapi:buildOidNameMap($filteredQuestionnaires, $projectLanguage[1]):)
    let $questionnaireResponses := if (name($storedQuestionnaire) = 'questionnaire') then $decor/scenarios/questionnaireresponse else ()
    let $return                 := qapi:handleQuestionnaire($storedQuestionnaire, $projectPrefix, $projectLanguage, $oidnamemap, $questionnaireResponses, false(), false())
    
    let $update                 := update replace $storedQuestionnaire with $return
    let $update                 := update value $storedQuestionnaire/@lastModifiedDate with substring(string(current-dateTime()), 1, 19)
    let $update                 := update delete $storedQuestionnaire/@uuid
    let $update                 := update delete $storedQuestionnaire/questionnaireresponse
    
    let $update                 := update delete $lock
    
    return
        element {name($return)} {
            $return/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($return/*)
        }
};

(:~ Build a map that has oid as key and human readable version as value :)
declare %private function qapi:buildOidNameMap($questionnaires as element()*, $language as xs:string?) as map(*)? {
    map:merge(
        for $oid in $questionnaires/@id
        let $grp := $oid
        group by $grp
        return
            map:entry($oid[1], utillib:getNameForOID($oid[1], $language, $oid[1]/ancestor::decor))
    )
};

declare function qapi:putQuestionnaireResponse($request as map(*)) {

    let $authmap                        := $request?user
    let $qid                            := $request?parameters?id
    let $qed                            := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $fhirVersion                    := $request?parameters?fhirVersion[string-length() gt 0]
    let $data                           := utillib:getBodyAsXml($request?body, 'questionnaire', ())
    let $deletelock                     := $request?parameters?deletelock = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $result                         := qapi:putQuestionnaire($authmap, string($qid), $qed, $data, $fhirVersion, $deletelock, false())
    return 
        if (empty($result)) then () else (
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
        )
        
    (:let $accept-type                    := head(router:accepted-content-types()[contains(., 'json') or contains(., 'xml')])
    
    return
        if (empty($result)) then roaster:response(404, ()) else (
            response:redirect-to(xs:anyURI(serverapi:getServerURLFhirServices() || $result/@fhirVersion || '/public/QuestionnaireResponse/' || $result/@id || '--' || replace($result/@effectiveDate, '\D', '') || '/?_format=' || $accept-type))
        ):)
};

declare function qapi:putQuestionnaire($request as map(*)) {

    let $authmap                        := $request?user
    let $qid                            := $request?parameters?id
    let $qed                            := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $fhirVersion                    := $request?parameters?fhirVersion[string-length() gt 0]
    let $data                           := utillib:getBodyAsXml($request?body, 'questionnaire', ())
    let $deletelock                     := $request?parameters?deletelock = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $result                         := qapi:putQuestionnaire($authmap, string($qid), $qed, $data, $fhirVersion, $deletelock, true())
    return 
        if (empty($result)) then () else (
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
        )
        
    (:let $accept-type                    := head(router:accepted-content-types()[contains(., 'json') or contains(., 'xml')])
    
    return
        if (empty($result)) then roaster:response(404, ()) else (
            response:redirect-to(xs:anyURI(serverapi:getServerURLFhirServices() || $result/@fhirVersion || '/public/Questionnaire/' || $result/@id || '--' || replace($result/@effectiveDate, '\D', '') || '/?_format=' || $accept-type))
        ):)
};

(:~ Central logic for updating an existing questionnaire or questionnaireresponse

@param $authmap         - required. Map derived from token
@param $id              - required. DECOR questionnaire/@id to update
@param $effectiveDate   - required. DECOR questionnaire/@effectiveDate to update
@param $data            - required. DECOR questionnaire xml element containing everything that should be in the updated questionnaire or quesrtionnaire response
@return concept object as xml with json:array set on elements
:)
declare function qapi:putQuestionnaire($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as item(), $fhirVersion as xs:string?, $deletelock as xs:boolean, $doQuestionnaire as xs:boolean) {

    let $storedQuestionnaire    := if ($doQuestionnaire) then utillib:getQuestionnaire($id, $effectiveDate) else utillib:getQuestionnaireResponse($id, $effectiveDate)
    let $decor                  := $storedQuestionnaire/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    let $projectLanguage        := $decor/project/@defaultLanguage
    let $objectName             := 
        switch (lower-case(local-name($storedQuestionnaire)))
        case 'questionnaire' return 'Questionnaire'
        case 'questionnaireresponse' return 'QuestionnaireResponse'
        default return upper-case(substring(name($storedQuestionnaire), 1, 1)) || substring(name($storedQuestionnaire), 2)
    
    let $objectType             := 
        if ($storedQuestionnaire[name()= 'questionnaire']) then $decorlib:OBJECTTYPE-QUESTIONNAIRE else 
        if ($storedQuestionnaire[name()= 'questionnaireresponse']) then $decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE else()
    
    let $check                  :=
        if (empty($storedQuestionnaire)) then (
            error($errors:BAD_REQUEST, $objectName || ' with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
        else
        if (count($storedQuestionnaire) gt 1) then (
            error($errors:SERVER_ERROR, 'Found multiple results for id ' || $id || ' effectiveDate ' || $effectiveDate || '. Expected 1..1. Alert your administrator as this should not be possible.')
        )
        else
        if ($storedQuestionnaire[@statusCode = $qapi:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, $objectName || ' with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' cannot be updated while it one of status: ' || string-join($qapi:STATUSCODES-FINAL, ', '))
        else ()
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-SCENARIOS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify scenarios in project ', $projectPrefix, '. You have to be an active author in the project.'))
        )
    
    let $fhirVersion            := ($fhirVersion, $storedQuestionnaire/@fhirVersion)[1]
    
    let $check                  :=
        if ($fhirVersion = $qapi:FHIROBJECT-FORMAT/enumeration/@value) then () else (
            $objectName || '.fhirVersion has unsupported value ' || string-join($fhirVersion, ', ') || '. SHALL be one of: ' || string-join($qapi:FHIROBJECT-FORMAT/enumeration/@value, ', ')
        )
    
    (: build map of oid names. is more costly when you do that deep down :)
    let $oidnamemap             := map {}(:qapi:buildOidNameMap($filteredQuestionnaires, $projectLanguage[1]):)
    let $preparedData           := qapi:handleQuestionnaire($data, $projectPrefix, $projectLanguage, $oidnamemap, (), false(), true())
    let $check                  := (
            let $r  := validation:jaxv-report($preparedData, $setlib:docDecorSchema)
            return
                if ($r/status='invalid') then
                    error($errors:BAD_REQUEST, 'Contents are not schema compliant: ' || serialize($r))
                else ()
        )
    let $check                  :=
        if (lower-case(local-name($storedQuestionnaire)) = lower-case(local-name($data))) then () else (
            error($errors:BAD_REQUEST, 'You cannot update a ' || $objectName || ' with a ' || local-name($data))
        )
    
    let $lock                   := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())

    let $check                  :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this ' || name($storedQuestionnaire) || ' (anymore). Get a lock first.'))
        )                   
    
    let $intention              :=
        if ($storedQuestionnaire[@statusCode='final']) then 'patch' else 'version'
    let $history                := histlib:AddHistory($authmap?name, $objectType, $projectPrefix, $intention, $storedQuestionnaire)
    
    let $instant                := current-dateTime()
    let $now                    := substring(string($instant), 1, 19)
    let $locks                  := if ($deletelock) then decorlib:getLocks($authmap, $storedQuestionnaire, true()) else ()
    let $update                 := update replace $storedQuestionnaire with $preparedData
    let $update                 := update value $storedQuestionnaire/@lastModifiedDate with substring(string(current-dateTime()), 1, 19)
    
    let $delete                 := if ($locks) then update delete $locks else ()
    
    return
        element {name($preparedData)} {
            if ($data/@uuid) then $data/@uuid else attribute uuid {util:uuid()},
            $preparedData/(@* except @uuid),
            $preparedData/*,
            $data/questionnaireresponse
        }

};
(:~ Input is questionnaire or questionnaireresponse (through recursion). Returns just the DECOR wrapper info, not the actual Q or QR, 
with as children any questionnaireresponse elements that match the parent questionnaire with their first relationship to a questionnaire 
  @param $listMode as boolean, skips any parts irrelevant to a tree of questionnaires/questionnaireresponse hence adds just meta data
  @param $decorMode as boolean, skips uuid attribute and listing of questionnaireresponses
:)
declare function qapi:handleQuestionnaire($qObjects as element()*, $projectPrefix as xs:string?, $projectLanguage as xs:string?, $oidnamemap as item(), $filteredQuestionnaireResponses as element(questionnaireresponse)*, $listMode as xs:boolean, $decorMode as xs:boolean) as element()* {
    for $q in $qObjects 
    let $iddisplay              := map:get($oidnamemap, $q/@id)
    let $projectLanguage        := if (empty($projectLanguage)) then ($q/ancestor::decor/project/@defaultLanguage, 'en-US')[1] else $projectLanguage
    return
        element {name($q)}
        {
            (: =========== attributes =========== :)
            if ($decorMode) then () else (
                if ($q/@uuid) then $q/@uuid else attribute uuid {util:uuid()}
            ),
            (:Not in use yet: Reference to a Questionnaire Wrapper @id. Questionnaire Wrappers SHALL carry either @id or @ref:)
            $q/@id,
            $q/@ref,
            $q/@effectiveDate,
            $q/@statusCode,
            $q/@versionLabel,
            $q/@expirationDate[. castable as xs:dateTime],
            $q/@officialReleaseDate[. castable as xs:dateTime],
            if (name($q) = 'questionnaire') then
                $q/@canonicalUri[. castable as xs:anyURI]
            else (),
            if ($q/@lastModifiedDate) then $q/@lastModifiedDate else attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)},
            if (string-length($iddisplay) = 0) then () else attribute iddisplay {$iddisplay},
            if ($q/ancestor::decor/project[@prefix = $projectPrefix]) then () else
            if (empty($q/ancestor::decor/project/@prefix)) then () else (
                attribute ident {$q/ancestor::decor/project/@prefix}
            )
            ,
            (: 2022-12-31 AH Disabled after leaving the FHIR format for DECOR :)
            if (name($q) = 'questionnaire') then () else $q/@fhirVersion,
            $q/ancestor::decor/@versionDate,
            $q/ancestor::decor/@language,
            
            (:If the FHIR Questionnaire is contained as a reference, this link points to the source:)
       
            (: 2022-12-31 AH Disabled after leaving the FHIR format for DECOR :)
            (:if ($q/@link) then $q/@link else
            
            if (name($q) = 'questionnaire') then
                attribute link {$fhirCanonicalBase || 'Questionnaire/'|| $fhirId}
            else :)
            if (name($q) = 'questionnaireresponse') then
                if ($q/@link) then $q/@link else (
                    let $fhirId             := concat($q/@id,'--',replace($q/@effectiveDate,'\D',''))
                    let $fhirCanonicalBase  := serverapi:getServerURLFhirServices()
                    return
                        attribute link {$fhirCanonicalBase || 'QuestionnaireResponse/'|| $fhirId}
                )
            else (),    
            
            (:Identifying the Questionnaire as Enhanced (type is QE), thus edited, modified, enriched.:)
            if (name($q) = 'questionnaire') then (
                attribute enhanced {$q/@enhanced = 'true'}, 
                if ($q/f:Questionnaire) then
                    attribute experimental {$q/f:Questionnaire/f:experimental/@value = 'true'}
                else ()
            ) 
            else ()
            ,
            (: =========== elements =========== :)
            $q/name,
            if (name($q) = 'questionnaire') then
                $q/desc
            else (),
            $q/classification,
            $q/relationship,
            if ($listMode) then () else 
            if (name($q) = 'questionnaire') then (
                $q/subjectType,
                $q/publishingAuthority,
                $q/endorsingAuthority,
                $q/code,
                $q/copyright,
                $q/jurisdiction,
                $q/contained,
                $q/item
            )
            else ()
            ,
            if ($decorMode) then () else (
                (: connect questionnaireresponse. Give empty parameter $filteredQuestionnaireResponses to avoid processing any further :)
                for $qr in $filteredQuestionnaireResponses[relationship[@type = 'ANSW'][1][@ref = $q/@id][@flexibility = $q/@effectiveDate]]
                return
                    qapi:handleQuestionnaire($qr, $projectPrefix, $projectLanguage, $oidnamemap, (), $listMode, $decorMode)
            )
        }
};
declare function qapi:handleQuestionnaireItem($item as element(item)) as element(item)? {
    <item>
    {
        for $att in $item/(@linkId | @definition | @prefix | @type | @required | @repeats | @readOnly | @maxLength | @minValue | @maxValue)
        order by name($att)
        return
            if (normalize-space($att) = '') then () else (
                attribute {name($att)} {replace($att, '(^\s+)|(\s+$)', '')}
            )
    }
    {
        $item/text,
        for $childnode in $item/code return utillib:prepareValueCodingTypeForUpdate($childnode),
        $item/designNote,
        for $childnode in $item/itemControl return utillib:prepareValueCodingTypeForUpdate($childnode),
        for $childnode in $item/unitOption return utillib:prepareValueCodingTypeForUpdate($childnode),
        $item/answerValueSet[@ref],
        for $childnode in $item/answerOption return qapi:handleQuestionnaireItemAnswerOptionType($childnode),
        $item/terminologyServer[@ref[not(. = '')]],
        for $childnode in $item/itemExtractionContext return qapi:handleQuestionnaireItemExtractionContext($childnode),
        $item/itemExtractionContext,
        $item/text,
        for $childitem in $item/item return qapi:handleQuestionnaireItem($childitem)
    }
    </item>
};
declare function qapi:handleQuestionnaireItemAnswerOptionType($item as element(answerOption)) as element(answerOption)? {
    for $node in $item[valueInteger | valueDate | valueTime | valueString | valueCoding | valueReference]
    return
    <answerOption>
    {
        if ($node/valueInteger) then 
            $node/valueInteger[1]
        else 
        if ($node/valueDate) then 
            $node/valueDate[1]
        else
        if ($node/valueString) then 
            $node/valueString[1]
        else
        if ($node/valueCoding) then 
            utillib:prepareValueCodingTypeForUpdate($node/valueCoding[1])
        else ()
    }
    </answerOption>
};
(: Establishes mapping context for a Questionnaire item :)
declare function qapi:handleQuestionnaireItemExtractionContext($item as element(itemExtractionContext)) as element(itemExtractionContext)? {
    for $node in $item[(@code | expression/@*)[not(normalize-space() = '')]]
    return
    <itemExtractionContext>
    {
        for $att in $node/@code
        return
            if (normalize-space($att) = '') then () else (
                attribute {name($att)} {replace($att, '(^\s+)|(\s+$)', '')}
            )
        ,
        for $childnode in $node/expression
        return
            <expression>
            {
                for $att in $childnode/(@description | @name | @language | @expression | @reference)
                order by name($att)
                return
                    if (normalize-space($att) = '') then () else (
                        attribute {name($att)} {replace($att, '(^\s+)|(\s+$)', '')}
                    )
            }
            </expression>
    }
    </itemExtractionContext>
};
