xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilmp                 = "http://art-decor.org/ns/api/util-conceptmap";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

import module namespace errors      = "http://e-editiones.org/roaster/errors";

(:~ Return zero or more conceptMaps as-is wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element. This repository element
holds at least the attribute @ident with the originating project prefix and optionally the attribute @url with the repository URL in case of an external
repository. Id based references can match both conceptMap/@id and conceptMap/@ref. The latter is resolved. Note that duplicate conceptMap matches may be 
returned. Example output:
<return>
  <repository url="http://art-decor.org/decor/services/" ident="ad2bbr-">
    <conceptMap id="2.16.840.1.113883.1.11.10282" displayName="ParticipationSignature" effectiveDate="2013-03-11T00:00:00" statusCode="final" versionLabel="DEFN=UV=VO=1206-20130318">
    ...
    </conceptMap>
  </repository>
</return>

@param $id           - required. Identifier of the conceptMap to retrieve
@param $flexibility  - optional. null gets all versions, 'dynamic' gets the newest version based on id, yyyy-mm-ddThh:mm:ss gets this specific version
@return Zero concept maps in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
@since 2013-06-14
:)
declare function utilmp:getConceptMapById($id as xs:string, $flexibility as xs:string?, $serialize as xs:boolean) as element(return) {
    
    let $conceptMaps        :=
        if ($flexibility castable as xs:dateTime) then (
        let $decorSets      := $setlib:colDecorData//conceptMap[@id = $id][@effectiveDate = $flexibility]
        let $cacheSets      := $setlib:colDecorCache//conceptMap[@id = $id][@effectiveDate = $flexibility]
        return ($decorSets, $cacheSets)
    )
    else if ($flexibility) then (
        let $decorSets      := $setlib:colDecorData//conceptMap[@id = $id]
        let $cacheSets      := $setlib:colDecorCache//conceptMap[@id = $id]
        let $conceptMaps    := ($decorSets, $cacheSets)
        return $conceptMaps[@effectiveDate = string(max($conceptMaps/xs:dateTime(@effectiveDate)))]
    )
    else (
        let $decorSets      := $setlib:colDecorData//conceptMap[@id = $id]
        let $cacheSets      := $setlib:colDecorCache//conceptMap[@id = $id]
        return ($decorSets, $cacheSets)
    )
    
   
    return
    <return>
    {
        for $repository in $conceptMaps
        let $prefix         := $repository/ancestor::decor/project/@prefix
        let $urlident       := concat($repository/ancestor::cacheme/@bbrurl, $prefix)
        group by $urlident
        return
            <project ident="{$prefix}">
            {
                if ($repository[ancestor::cacheme]) then attribute url {$repository[1]/ancestor::cacheme/@bbrurl} else (),
                $repository[1]/ancestor::decor/project/@defaultLanguage,
                for $conceptMap in $repository
                let $ideff  := concat($conceptMap/@id, $conceptMap/@ref, $conceptMap/@effectiveDate)
                group by $ideff
                order by $conceptMap[1]/@effectiveDate descending
                return utilmp:getRawConceptMap($conceptMap[1], $repository[1]/ancestor::decor/project/@defaultLanguage, $prefix, (), $serialize)
            }
            </project>
    }
    </return>
};

(:~ Return zero or more conceptMaps as-is wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element. This repository element
holds at least the attribute @ident with the originating project prefix and optionally the attribute @url with the repository URL in case of an external
repository. Id based references can match both conceptMap/@id and conceptMap/@ref. The latter is resolved. Note that duplicate conceptMap matches may be 
returned. Example output for conceptMap/@ref:<br/>

<return>
  <project ident="epsos-">
    <conceptMap ref="2.16.840.1.113883.1.11.10282" displayName="ParticipationSignature">
    ...
    </conceptMap>
  </project>
  <repository url="http://art-decor.org/decor/services/" ident="ad2bbr-" referencedFrom="epsos-">
    <conceptMap id="2.16.840.1.113883.1.11.10282" displayName="ParticipationSignature" effectiveDate="2013-03-11T00:00:00" statusCode="final" versionLabel="DEFN=UV=VO=1206-20130318">
    ...
    </conceptMap>
  </repository>
</return>
   
@param $id           - required. Identifier of the conceptMap to retrieve
@param $flexibility  - optional. null gets all versions, 'dynamic' gets the newest version based on id (regardless of name), yyyy-mm-ddThh:mm:ss gets this specific version
@param $prefix       - required. determines search scope. pfx- limits scope to this project only
@param $version      - optional. if empty defaults to current version. if valued then the conceptMap will come explicitly from that archived project version which is expected to be a compiled version
@return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
@since 2013-06-14
:)
declare function utilmp:getConceptMapById($id as xs:string, $flexibility as xs:string?, $prefix as xs:string, $version as xs:string?, $serialize as xs:boolean) as element(return) {

    let $internalrepositories       := utillib:getDecorByPrefix($prefix, $version, ())[1]
    
    let $internalconceptMaps        :=
        if (empty($version)) then
            $setlib:colDecorData//conceptMap[@id = $id][ancestor::decor/project/@prefix = $prefix] | $setlib:colDecorData//conceptMap[@ref = $id][ancestor::decor/project/@prefix = $prefix]
        else 
            $setlib:colDecorVersion//decor[@versionDate = $version][project/@prefix = $prefix][1]/terminology/conceptMap[@id = $id] | $setlib:colDecorVersion//decor[@versionDate = $version][project/@prefix = $prefix][1]/terminology/conceptMap[@ref = $id]
           
    let $repositoryConceptMapLists  :=
        <repositoryConceptMapLists>
        {
            (:  don't go looking in repositories when this is an archived project version. the project should be compiled already and 
                be self contained. Repositories in their current state would give a false picture of the status quo when the project 
                was archived.
            :)
            if (empty($version)) then
                let $buildingBlockRepositories  := $internalrepositories/project/buildingBlockRepository[empty(@format)] | 
                                                   $internalrepositories/project/buildingBlockRepository[@format='decor']
                (:this is the starting point for the list of servers we already visited to avoid circular reference problems:)
                return utilmp:getConceptMapByIdFromBBR($prefix, $id, $prefix, $buildingBlockRepositories, (), (), ())/descendant-or-self::*[conceptMap[@id]]
            else ()
        }
        {
            (:from the requested project, return conceptMap/(@id and @ref):)
            (: when retrieving value sets from a compiled project, the @url/@ident they came from are on the conceptMap element
               reinstate that info on the repositoryConceptMapList element so downstream logic works as if it really came from 
               the repository again.
            :)
            for $repository in $internalrepositories
            for $conceptMaps in $internalconceptMaps
            let $source := concat($conceptMaps/@url,$conceptMaps/@ident)
            group by $source
            return
                if (string-length($conceptMaps[1]/@url)=0) then
                    <repositoryConceptMapList ident="{$repository/project/@prefix}">
                    {
                        $conceptMaps
                    }
                    </repositoryConceptMapList>
                else (
                    <repositoryConceptMapList url="{$conceptMaps[1]/@url}" ident="{$conceptMaps[1]/@ident}" referencedFrom="{$prefix}">
                    {
                        for $conceptMap in $conceptMaps
                        return <conceptMap>{$conceptMap/(@* except (@url|@ident|@referencedFrom)), $conceptMap/node()}</conceptMap>
                    }
                    </repositoryConceptMapList>
                )
        }
        </repositoryConceptMapLists>
    
    let $max                        :=
        if ($flexibility castable as xs:dateTime) then $flexibility else 
        if (empty($flexibility)) then () else max($repositoryConceptMapLists//conceptMap/xs:dateTime(@effectiveDate))
    
    return
        <return>
        {
            if (empty($max)) then
                for $segment in $repositoryConceptMapLists/*
                let $elmname        := if ($segment[empty(@url)][@ident=$prefix]) then 'project' else 'repository'
                return
                    element {$elmname} {
                        $segment/@*,
                        for $conceptMap in $segment/conceptMap
                        let $ideff  := concat($conceptMap/@id, $conceptMap/@ref, $conceptMap/@effectiveDate)
                        group by $ideff
                        order by $conceptMap[1]/@effectiveDate descending
                        return utilmp:getRawConceptMap($conceptMap[1], (), $prefix, $version, $serialize)
                        
                    }
            else (
                for $segment in $repositoryConceptMapLists/*[conceptMap[@effectiveDate = $max]]
                let $elmname        := if ($segment[empty(@url)][@ident=$prefix]) then 'project' else 'repository'
                return
                    element {$elmname} {
                        $segment/@*, 
                        for $conceptMap in $segment/conceptMap[@ref] | $segment/conceptMap[@effectiveDate = $max]
                        let $ideff  := concat($conceptMap/@id, $conceptMap/@ref, $conceptMap/@effectiveDate)
                        group by $ideff
                        order by $conceptMap[1]/@effectiveDate descending
                        return utilmp:getRawConceptMap($conceptMap[1], (), $prefix, $version, $serialize)
                    }
            )
        }
        </return>
};

(:~ Look for conceptMap[@id] and recurse if conceptMap[@ref] is returned based on the buildingBlockRepositories in the project that returned it.
If we get a conceptMap[@ref] from an external repository (through RetrieveConceptMap), then tough luck, nothing can help us. The returned 
data is a nested repositoryConceptMapList element allowing you to see the full trail should you need that. Includes duplicate protection 
so every project is checked once only.

Example below reads:
- We checked hwg- and found BBR hg-
- We checked hg- and found BBR nictz2bbr-
- We checked nictiz2bbr- and found the requested conceptMap

<repositoryConceptMapList url="http://decor.nictiz.nl/decor/services/" ident="hg-" referencedFrom="hwg-">
  <repositoryConceptMapList url="http://decor.nictiz.nl/decor/services/" ident="nictiz2bbr-" referencedFrom="hg-">
    <conceptMap id="2.16.840.1.113883.2.4.3.11.60.1.11.2" name="RoleCodeNLZorgverlenertypen" displayName="RoleCodeNL - zorgverlenertype (personen)" effectiveDate="2011-10-01T00:00:00" statusCode="final">
    ...
    </conceptMap>
  </repositoryConceptMapList>
</repositoryConceptMapList>
:)
declare %private function utilmp:getConceptMapByIdFromBBR($basePrefix as xs:string, $id as xs:string, $prefix as xs:string, $externalrepositorylist as element()*, $bbrmap as map(*)?, $localbyid as element(conceptMap)*, $localbyref as element(conceptMap)*) as element()* {

    let $newmap                 := 
        map:merge(
            for $bbr in $externalrepositorylist return map:entry(concat($bbr/@ident, $bbr/@url), '')
        )
    
    let $conceptMapsById        := if ($localbyid) then $localbyid else $setlib:colDecorData//conceptMap[@id = $id]
    let $conceptMapsByRef       := if ($localbyref) then $localbyref else $setlib:colDecorData//conceptMap[@ref = $id]
    
    let $return                 := 
        for $repository in $externalrepositorylist
        let $repourl            := $repository/@url
        let $repoident          := $repository/@ident
        let $hasBeenProcessedBefore := 
            if (empty($bbrmap)) then false() else map:contains($bbrmap, concat($repoident, $repourl)) or map:contains($bbrmap, concat($basePrefix, $utillib:strDecorServicesURL))
          
        let $newmap             :=
            if (empty($bbrmap)) then $newmap else map:merge((
                for $k in map:keys($bbrmap) return map:entry($k, ''),
                for $k in map:keys($newmap) return if (map:contains($bbrmap, $k)) then () else map:entry($k, '')
            ))
        return
            if ($hasBeenProcessedBefore) then () else (
                <repositoryConceptMapList url="{$repourl}" ident="{$repoident}" referencedFrom="{$prefix}">
                {
                    (: if this buildingBlockRepository resolves to our own server, then get it directly from the db. :)
                    if ($repository[@url = $utillib:strDecorServicesURL]) then (
                        let $conceptMaps    := $conceptMapsById[ancestor::decor/project[@prefix = $repoident]]
                        let $conceptMapsRef := $conceptMapsByRef[ancestor::decor/project[@prefix = $repoident]]
                        
                        let $project        := ($conceptMaps/ancestor::decor | $conceptMapsRef/ancestor::decor)[1]
                        let $bbrs           :=
                            $project//buildingBlockRepository[empty(@format)] |
                            $project//buildingBlockRepository[@format='decor']
                        return 
                        if ($project) then (
                            attribute projecttype {'local'},
                            $conceptMaps,
                            if (not($conceptMaps) or $conceptMapsRef) then (
                                utilmp:getConceptMapByIdFromBBR($basePrefix, $id, $repoident, $bbrs, $newmap, $conceptMapsById, $conceptMapsByRef)
                            ) else ()
                        ) else ()
                    )
                    (: check cache first and do a server call as last resort. :)
                    else (
                        let $cachedProject          := $setlib:colDecorCache//cacheme[@bbrurl = $repourl][@bbrident = $repoident]
                        let $cachedConceptMaps        := $cachedProject//conceptMap[@id = $id] | $cachedProject//conceptMap[@ref = $id]
                        let $bbrs               :=
                            $cachedProject//buildingBlockRepository[empty(@format)] |
                            $cachedProject//buildingBlockRepository[@format='decor']
                        return
                        if ($cachedProject) then (
                            attribute projecttype {'cached'},
                            $cachedConceptMaps[@id],
                            if (not($cachedConceptMaps) or $cachedConceptMaps[@ref]) then (
                                utilmp:getConceptMapByIdFromBBR($basePrefix, $id, $repoident, $bbrs, $newmap, $conceptMapsById, $conceptMapsByRef)
                            ) else ()
                        )
                       
                        else ()                   
                    )
                }
                </repositoryConceptMapList>
            )
    
    return $return
};

(:~ Get contents of a conceptMap :)
declare function utilmp:getRawConceptMap($conceptMap as element(conceptMap), $language as xs:string?, $prefix as xs:string, $version as xs:string?, $serialize as xs:boolean) as element(conceptMap) {
    <conceptMap>
    {
        $conceptMap/@*,
        if ($serialize) then for $node in $conceptMap/desc return utillib:serializeNode($node) else ($conceptMap/desc),
        $conceptMap/publishingAuthority,
        if ($conceptMap[@id]) then 
            if ($conceptMap[publishingAuthority]) then () else (
                let $decor      := utillib:getDecorByPrefix($prefix, $version, $language)
                let $copyright  := ($decor/project/copyright[empty(@type)] | $decor/project/copyright[@type = 'author'])[1]
                return
                if ($copyright) then
                    <publishingAuthority name="{$copyright/@by}" inherited="project">
                    {
                        (: Currently there is no @id, but if it ever would be there ...:)
                        $copyright/@id,
                        $copyright/addrLine
                    }
                    </publishingAuthority>
                else ()
            )
        else ()
        ,
        $conceptMap/endorsingAuthority,
        if ($serialize) then for $node in $conceptMap/purpose return utillib:serializeNode($node) else ($conceptMap/purpose)
        ,
        let $copyrights     := if ($conceptMap[@id]) then utillib:handleCodeSystemCopyrights($conceptMap/copyright, distinct-values($conceptMap//@codeSystem)) else $conceptMap/copyright
        return if ($serialize) then for $node in $copyrights return utillib:serializeNode($node) else ($copyrights)
        ,
        $conceptMap/jurisdiction,
        $conceptMap/sourceScope,
        $conceptMap/targetScope,
        for $group in $conceptMap/group
        return
            <group>
            {
                $group/source,
                $group/target,
                $group/element
            }
            </group>
    }
    </conceptMap>
};