xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilcs                 = "http://art-decor.org/ns/api/util-codesystem";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

import module namespace errors      = "http://e-editiones.org/roaster/errors";

(:~ Return zero or more codesystems as-is

@param $id           - required. Identifier of the codesystem to retrieve
@param $flexibility  - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@return Matching value sets
@since 2024-02-13
:)
declare function utilcs:getCodeSystemById ($id as xs:string, $flexibility as xs:string?) as element(codeSystem)* {

    if ($flexibility castable as xs:dateTime) then (
        let $decorSets      := $setlib:colDecorData//codeSystem[@id = $id][@effectiveDate = $flexibility]
        let $cacheSets      := $setlib:colDecorCache//codeSystem[@id = $id][@effectiveDate = $flexibility]
        return ($decorSets, $cacheSets)
    )
    else if ($flexibility) then (
        let $decorSets      := $setlib:colDecorData//codeSystem[@id = $id]
        let $cacheSets      := $setlib:colDecorCache//codeSystem[@id = $id]
        let $codeSystems      := ($decorSets, $cacheSets)
        return $codeSystems[@effectiveDate = string(max($codeSystems/xs:dateTime(@effectiveDate)))]
    )
    else (
        let $decorSets      := $setlib:colDecorData//codeSystem[@id = $id]
        let $cacheSets      := $setlib:colDecorCache//codeSystem[@id = $id]
        return ($decorSets, $cacheSets)
    )
    
};

declare function utilcs:getCodeSystemById($id as xs:string, $flexibility as xs:string?, $decorOrPrefix as item(), $decorRelease as xs:string?) as element(codeSystem)* {
    utilcs:getCodeSystemById($id, $flexibility, $decorOrPrefix, $decorRelease, ())
};

(:~ Return zero or more codesystems as-is

@param $id            - required. Identifier of the value set to retrieve
@param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@param $decorOrPrefix - required. determines search scope. pfx- limits scope to this project only
@param $decorRelease  - optional. if empty defaults to current version. if valued then the value set will come explicitly from that archived project version which is expected to be a compiled version
@param $decorLanguage - optional. Language compilation. Defaults to project/@defaultLanguage.
@return Matching value sets
@since 2024-02-13
:)

declare function utilcs:getCodeSystemById($id as xs:string, $flexibility as xs:string?, $decorOrPrefix as item(), $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(codeSystem)* {
    
    let $decor                  := utillib:getDecor($decorOrPrefix, $decorRelease, $decorLanguage)
    let $decors                 := if (empty($decorRelease)) then utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL) else $decor
    
    return
        if ($flexibility castable as xs:dateTime) then $decors//codeSystem[@id = $id][@effectiveDate = $flexibility]
        else if ($flexibility) then (
            let $codeSystems    := $decors//codeSystem[@id = $id]
            return $codeSystems[@effectiveDate = string(max($codeSystems/xs:dateTime(@effectiveDate)))]
        )
        else $decors//codeSystem[@id = $id]
};

(:~ Gets the content of a conceptlist by recursively resolving all codedconcepts. Use utilvs:getValueSetExtracted to resolve a codedConcepts. :)
declare function utilcs:getCodedConcepts($codedConcepts as element(codedConcept)*, $codeSystemId as xs:string, $level as xs:integer, $conceptList as element(conceptList), $sofar as xs:string*, $checkParentChild as xs:boolean, $language as xs:string) {
    
    for $codedConcept in $codedConcepts[not(@statusCode = ('cancelled', 'rejected'))]
    let $children               := 
        if ($checkParentChild) then
            $conceptList/codedConcept[parent/@code = $codedConcept/@code][not(@code = $codedConcept/@code)]
        else (
            let $nextFirstOtherHierarchy            := $codedConcept/following-sibling::codedConcept[@level castable as xs:integer][xs:integer(@level) le $level]
            return $codedConcept/following-sibling::codedConcept[@level castable as xs:integer][xs:integer(@level) = ($level + 1)][not(preceding-sibling::codedConcept[@code = $nextFirstOtherHierarchy/@code])]
        )
    let $type                   := 
        if ($codedConcept[@statusCode = ('deprecated', 'retired', 'cancelled', 'rejected')]) then 'D' 
        else if ($codedConcept[@type = 'A'] | $codedConcept[@abstract = 'true'] | $codedConcept/property[@code = 'notSelectable']/valueBoolean[@value = 'true']) then 'A' else
        if ($children) then 'S' else ('L')
    return (
        <concept code="{$codedConcept/@code}" 
                 codeSystem="{$codeSystemId}" 
                 codeSystemName="{$codedConcept/ancestor::codeSystem/@displayName}" 
                 codeSystemVersion="{$codedConcept/ancestor::codeSystem/@effectiveDate}" 
                 displayName="{utilcs:getDesignationFromCodedConcept($codedConcept, $language)}" 
                 level="{$level}" type="{$type}">
        {
            for $designation in $codedConcept/designation
            let $designationtype    := if ($designation[@type]) then $designation/@type else 'preferred'
            return
                <designation>
                {
                    $designation/@displayName, 
                    attribute type {$designationtype}, 
                    $designation/@language, 
                    $designation/@lastTranslated, 
                    $designation/@mimeType
                }
                </designation>
        }
        {
            $codedConcept/desc[.//text()[not(normalize-space() = '')]]
        }
        </concept>
        ,
        utilcs:getCodedConcepts($children[not(@code = $sofar)], $codeSystemId, $level + 1, $conceptList, distinct-values(($sofar, $children/@code)), $checkParentChild, $language)
    )
};

declare %private function utilcs:getDesignationFromCodedConcept($codedConcept as element(codedConcept), $language as xs:string?) as xs:string? {
    let $designation            := $codedConcept/designation[@language = $language]
    let $designation            := if ($designation) then $designation else $codedConcept/designation
    let $designation            :=
        if ($designation[@type = 'fsn']) then $designation[@type = 'fsn'][1] 
        else if ($designation[@type = 'preferred']) then $designation[@type = 'preferred'][1] 
        else if ($designation[empty(@type)]) then $designation[empty(@type)][1] else $designation[1]
    
    return $designation/@displayName
};

