xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utiltemp               = "http://art-decor.org/ns/api/util-template";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

import module namespace errors      = "http://e-editiones.org/roaster/errors";

(:~ Return zero or more templates as-is

@param $id           - required. Identifier of the template to retrieve
@param $flexibility  - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@return Matching templates
@since 2025-01-24
:)
declare function utiltemp:getTemplateById($id as xs:string, $flexibility as xs:string?) as element(template)* {
    
    if ($flexibility castable as xs:dateTime) then (
        let $t1                 := $setlib:colDecorData//template[@id = $id][@effectiveDate = $flexibility]
        let $t2                 := $setlib:colDecorCache//template[@id = $id][@effectiveDate = $flexibility]
        return $t1 | $t2
    )
    else
    if ($flexibility) then (
        let $t1                 := $setlib:colDecorData//template[@id = $id]
        let $t2                 := $setlib:colDecorCache//template[@id = $id]
        let $tall               := $t1 | $t2
        let $tmax               := string(max($tall/xs:dateTime(@effectiveDate)))
        return $tall[@effectiveDate = $tmax]
    )
    else (
        let $t1                 := $setlib:colDecorData//template[@id = $id]
        let $t2                 := $setlib:colDecorCache//template[@id = $id]
        return $t1 | $t2
    )
};

(:~ Return zero or more templates as-is

@param $id            - required. Identifier of the template to retrieve
@param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@param $decorOrPrefix - required. determines search scope. pfx- limits scope to this project only
@return Matching templates
@since 2025-01-25
:)
declare function utiltemp:getTemplateById($id as xs:string, $flexibility as xs:string?, $decorOrPrefix as item()) as element(template)* {
    utiltemp:getTemplateById($id, $flexibility, $decorOrPrefix, (), ())
};

(:~ Return zero or more templates as-is

@param $id            - required. Identifier of the template to retrieve
@param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@param $decorOrPrefix - required. determines search scope. pfx- limits scope to this project only
@param $decorVersion  - optional. if empty defaults to current version. if valued then the template will come explicitly from that archived project version which is expected to be a compiled version
@param $language      - optional. Language compilation. Defaults to project/@defaultLanguage.
@return Matching templates
@since 2025-01-25
:)
declare function utiltemp:getTemplateById($id as xs:string, $flexibility as xs:string?, $decorOrPrefix as item(), $decorVersion as xs:string?, $language as xs:string?) as element(template)* {

    let $decor                  := utillib:getDecor($decorOrPrefix, $decorVersion, $language)
    let $decors                 := if (empty($decorVersion)) then utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL) else $decor
    
    return
        if ($flexibility castable as xs:dateTime) then $decors//template[@id = $id][@effectiveDate = $flexibility]
        else
        if ($flexibility) then (
            let $templates := $decors//template[@id = $id]
            return $templates[@effectiveDate = string(max($templates/xs:dateTime(@effectiveDate)))]
        )
        else $decors//template[@id = $id]
};

declare function utiltemp:getTemplateExtract($id as xs:string, $effectiveDate as xs:string?) as element(template)* {

    let $templates              := utiltemp:getTemplateById($id, $effectiveDate)
    
    return
    <return>
    {
        for $template in $templates
        let $prefix             := $template/ancestor::decor/project/@prefix
        let $url                := ($template/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
        group by $url, $prefix
        return
            <project ident="{$prefix}">
            {
                for $ns at $i in utillib:getDecorNamespaces($template/ancestor::decor)
                return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri},
                if ($template[ancestor::cacheme]) then attribute url {$template[1]/ancestor::cacheme/@bbrurl} else (),
                $template[1]/ancestor::decor/project/@defaultLanguage,
                $template
            }
            </project>
    }
    </return>
};

declare function utiltemp:getTemplateExtract($projectPrefix as xs:string*, $projectVersion as xs:string*, $projectLanguage as xs:string?, $id as xs:string, $effectiveDate as xs:string?) as element(template)* {

    let $id                     := $id[not(. = '')]
    let $effectiveDate          := ($effectiveDate[not(. = '')], 'dynamic')[1]
    let $projectPrefix          := $projectPrefix[not(. = '')][1]
    let $projectVersion         := $projectVersion[not(. = '')][1]
    let $projectLanguage        := $projectLanguage[not(. = '')][1]

    (: retrieve all templates for the input $decor project(s) 
        - when compiled version  - as is - 
        - when live version - getting all (cached) decor projects in scope  
    :)
    
    let $allTemplates           := utiltemp:getTemplateById($id, $effectiveDate, $projectPrefix, $projectVersion, $projectLanguage) | utillib:getDecor($projectPrefix, $projectVersion, $projectLanguage)//template[@ref = $id]
           
    return
    
    <return>
    {
        for $templates in $allTemplates
        let $id                 := $templates/@id | $templates/@ref
        group by $id
        return (
            <template id="{$id}" ident="{$projectPrefix}">
            {
                for $template in $templates
                let $tmed       := $template/@effectiveDate
                group by $tmed
                order by xs:dateTime($template[1]/@effectiveDate) descending
                return 
                    <template>
                    {
                        $template[1]/(@* except @*[contains(name(),'dummy-')]),
                        if ($template/@url) then $template/@url else if ($template/ancestor::*/@bbrurl) then attribute url {$template/ancestor::*/@bbrur} else (),
                        attribute ident {($template/@ident, $template/ancestor::decor/project/@prefix)[1]},
                        if ($template[1]/../@*[contains(name(),'dummy-')]) then $template[1]/../@*[contains(name(),'dummy-')]
                        else if ($template[1]/ancestor::decor) then (
                            (: <ns uri="urn:hl7-org:v3" prefix="hl7" default="{$ns-default='hl7'}" readonly="true"/> :)
                            for $ns at $i in utillib:getDecorNamespaces($templates[1]/ancestor::decor)
                            return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri}
                        )
                        else (
                            for $ns-prefix at $i in in-scope-prefixes($template[1])[not(.=('xml'))]
                            let $ns-uri := namespace-uri-for-prefix($ns-prefix, $template[1])
                            order by $ns-prefix
                            return attribute {QName($ns-uri,concat($ns-prefix,':dummy-',$i))} {$ns-uri}
                        )
                        ,
                        $template[1]/node()
                    }
                    </template>
            }
            </template>
        )
    }
    </return>
};

(:recursive function to get all templates hanging off from the current template. has circular reference protection. Does not return the start templates :)
declare function utiltemp:getTemplateChain($startTemplates as element(template)*, $decorsInScope as element(decor)*, $resultsmap as map(*)) as element(template)* {

    for $ref in ($startTemplates//element/@contains | $startTemplates//include/@ref)
    let $flexibility            := if ($ref/../@flexibility) then $ref/../@flexibility else ('dynamic')
    group by $ref, $flexibility
    
    return (
        let $templates          := 
            if ($flexibility castable as xs:dateTime) then $decorsInScope//template[@id = $ref][@effectiveDate = $flexibility]
            else (
                let $t          := $decorsInScope//template[@id = $ref]
                return $t[@effectiveDate = string(max($t/xs:dateTime(@effectiveDate)))]
            )
         let $templates         :=
            for $template in $templates
            return
            element {name($template)}
            {
                for $ns at $i in utillib:getDecorNamespaces($template/ancestor::decor)
                return attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri},
                if ($template[ancestor::cacheme]) then attribute url {$template[1]/ancestor::cacheme/@bbrurl} else (),
                $template[1]/ancestor::decor/project/@defaultLanguage,
                $template/@*,
                $template/*
            }
        
        let $newkey             := concat($templates[1]/@id, $templates[1]/@effectiveDate)
        
        return
        if ($templates) then
            if (map:contains($resultsmap, $newkey)) then () else (
                let $newmap     := map:merge(($resultsmap, map:merge(map:entry($newkey, ''))))
                
                return $templates | utiltemp:getTemplateChain($templates, $decorsInScope, $newmap)
            )
        else ()
    )
};

declare function utiltemp:getTemplateForEdit($template as element(template), $decor as element(decor), $lock as element()?, $targetDate as xs:boolean, $keepIds as xs:boolean, $baseId as xs:string?, $mode as xs:string) as element(template) {

    let $templateAssociations   := $decor//templateAssociation[@templateId = $template/@id][@effectiveDate=$template/@effectiveDate]
    
    let $result                 := 
        <template> 
        {
            utiltemp:createTemplateAttributes($template, $decor, $targetDate, $keepIds, $baseId, $mode)/@*,
            <edit mode="edit"/>,
            $lock,
            if ($template/desc) then $template/desc else <desc language="{$decor/project/@defaultLanguage}"/>,
            $template/classification,
            utiltemp:createTemplateRelationships($template, $decor, $mode, $keepIds)/*, 
            for $node in $template/publishingAuthority return utiltemp:createElementWithAddress($node),
            for $node in $template/endorsingAuthority return utiltemp:createElementWithAddress($node),
            $template/purpose,
            $template/copyright, 
            utiltemp:createTemplateContext($template/context),
            if ($template/item) then utiltemp:createTemplateItem($template/item) else (),
            utiltemp:createTemplateExample($template/example),
            utiltemp:createTemplateElements($template, $decor, $mode),
            (: by adding them here, they stay in the clients XML until he closes the form :)
            if ($templateAssociations) then utiltemp:createStaticAssociationElement($templateAssociations, $decor/project/@defaultLanguage) else ()
        }
        </template>
    
    return 
        utiltemp:prepareTemplateForAPI($result)
};

(:~ Copy or create template attributes based on $mode. When $mode is 'edit', most things are copied. When $mode is 'new' we need
status code to be 'draft', and we have no use for expirationDate, officialReleaseDate and/or canonicalUri

@param $template as element(template) - Template to copy attributes from. When in edit mode, you simply copy. When in new mode, 
       we copy most but not everything, e.g. statusCode
@param $decor as element(decor) - Decor ancestor element of the $template, or of the project that the new template is to be created in. 
       When a prototype is in a different project than the to-be-created template then namespace declarations for the prototype template 
       are relevant to keep as they might conflict with the project that the new template is created in
@param $targetDate as xs:boolean - Relevant for @effectiveDate only when $mode != 'edit. When true() uses yyyy-mm-ddT000:00:00. When false() uses yyyy-mm-ddThh:mm:ss. 
@param $keepIds as xs:boolean - Boolean for keeping the same $template/@id for the output
@param $baseId as xs:string? - Base OID for the template (everything but the last node in @id)
@param $mode as xs:string - When editing an existing template set to 'edit'. When creating a new template (from prototype) set to 'new'
:)
declare %private function utiltemp:createTemplateAttributes($template as element(template), $decor as element(decor), $targetDate as xs:boolean, $keepIds as xs:boolean, $baseId as xs:string?, $mode as xs:string) as element() {

    let $projectPrefix                  := $decor/project/@prefix
    let $templateName                   := ($template/@name, utillib:getNameForOID($template/@id, (), $decor))[1]     

    return
    <attributes>
    {
        attribute projectPrefix {$projectPrefix},
        if ($baseId) then attribute baseId {$baseId} else (),
        attribute originalId {if ($keepIds) then $template/@id else ()},
        attribute id {if ($keepIds) then $template/@id else ()},
        attribute name {$templateName},
        attribute displayName {($template/@displayName, $templateName)[1]},
        attribute effectiveDate {
            if ($mode = 'edit') then $template/@effectiveDate else 
            if ($targetDate) then substring(string(current-date()), 1, 10) || 'T00:00:00' else substring(string(current-dateTime()), 1, 19)
        },
        if ($mode = 'new') then attribute statusCode {'draft'} else $template/@statusCode,
        $template/@versionLabel,
        $template/@isClosed,
        if ($mode = 'new') then () else $template/@expirationDate,
        if ($mode = 'new') then () else $template/@officialReleaseDate,
        (: note ... 
           - $mode new (from prototype) *shall not* copy the canonicalUri
           - $keepIds false *shall not* copy the canonicalUri
           - edit of an existing template *shall* copy the canonicalUri
           - it is debatable whether or not a version of an existing template *should* copy the canonicalUri. currently implemented: yes
        :)
        if ($mode = 'new' or not($keepIds)) then () else $template/@canonicalUri,
        for $ns at $i in utillib:getDecorNamespaces($decor) 
        return 
            attribute {QName($ns/@uri,concat($ns/@prefix,':dummy-', if ($ns/@default = 'true') then 'default' else $i))} {$ns/@uri}
    }           
    </attributes>

};

declare %private function utiltemp:createTemplateRelationships($template as element(template), $decor as element(decor), $mode as xs:string, $keepIds as xs:boolean) as element(relationships) {

    <relationships>
    {
        if ($mode = 'edit') then () else (
            if ($template[@id]) then (
                (: note ... 
                    if $mode is 'new' then if $keepIds true the relationship shall be VERSION
                    (as the new thing is a natural version of the source) otherwise it shall be SPEC
                :)
                <relationship type="{
                    if ($mode = 'new') then (
                        if ($keepIds) then 'VERSION' else 'SPEC'
                    ) else upper-case($mode)}" template="{$template/@id}" selected="template" flexibility="{$template/@effectiveDate}">
                {
                    for $attr in $template/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                    return
                        attribute {'tm' || name($attr)} {$attr}
                    ,
                    attribute url {$utillib:strDecorServicesURL},
                    attribute ident {$decor/project/@prefix},
                    attribute linkedartefactmissing {'false'}
                }
                </relationship>
            ) else ()
        )
        ,
        for $relationship in $template/relationship
            let $artefact               := 
                if (empty($relationship/@template)) then () else (
                    (utiltemp:getTemplateById($relationship/@template, $relationship/@flexibility, $decor, (), ())[@id][@effectiveDate])[1]
                )
            let $tmpurl                 := ($artefact/@url, $artefact/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]
            let $tmpident               := ($artefact/@ident, $artefact/ancestor::decor/project/@prefix)[1]
            return
            <relationship>
            {
                attribute type {$relationship/@type},
                if ($relationship/@template) then (
                    attribute template {$relationship/@template},
                    attribute selected {'template'},
                    attribute model {''},
                    attribute flexibility {$relationship/@flexibility},
                    if (empty($artefact)) then () else (
                        for $attr in $artefact/(@id | @name | @displayName | @effectiveDate | @expirationDate | @statusCode | @versionLabel)
                        return
                            attribute {'tm' || name($attr)} {$attr}
                        ,
                        attribute url {$tmpurl},
                        attribute ident {$tmpident}
                    ),
                    attribute linkedartefactmissing {empty($artefact)}
                )
                else (
                    attribute template {''},
                    attribute model {$relationship/@model},
                    attribute flexibility {$relationship/@flexibility},
                    attribute selected {'model'}
                )
            }
            </relationship>
        }
    </relationships>
    
};

declare %private function utiltemp:createElementWithAddress($node as element()) as element() {

    element {name($node)}
    {
        $node/(@* except @selected),
        attribute selected {''},
        $node/addrLine
    }
};

declare %private function utiltemp:createTemplateContext($context as element(context)?) as element(context) {

    <context>
    {
        if ($context/@id) then (
            $context/@id,
            attribute path {''},
            attribute selected {'id'}
        )
        else (
            attribute id {''},
            attribute path {$context/@path},
            attribute selected {'path'}
        )
    }
    </context>
};

declare %private function utiltemp:createTemplateItem($item as element(item)) as element(item) {

    <item label="{$item/@label}">
    {
        $item/desc
    }
    </item>
};

declare %private function utiltemp:createTemplateExample($examples as element(example)*) as element(example)* {

     if ($examples) then
        (: if multiple root nodes exist, it is never going to be valid xml in an editor, so we wrap it under one :)
        for $example in $examples
        return
            <example type="{$example/@type}" caption="{$example/@caption}">
            {
                if (count($example/*) gt 1) then <art:placeholder xmlns:art="urn:art-decor:example">{$example/node()}</art:placeholder> else $example/node()
            }
            </example>
    else 
        <example type="neutral" caption=""/>
};

declare %private function utiltemp:createTemplateElements($template as element(template), $decor as element(decor), $mode as xs:string) as element()* {
    
    let $projectPrefix                  := $decor/project/@prefix
    let $valueSetList                   := $decor/terminology/valueSet
    let $templateList                   := $decor/rules/template

    (: TODO: why do we only pick SPEC? Could it not be "anything"? All it does it supply you with potential elements and attributes to activate :)
    let $specialization                 := $template/relationship[@type='SPEC'][@template][1]
    (:get prototype for editor with normalized attributes:)
    let $prototype                      := if ($specialization) then utiltemp:getTemplateById($specialization/@template, $specialization/@flexibility, $projectPrefix)[1] else ()

    return
    if ($prototype) then (
        (: if template is empty, check prototype for relevant parts :)
        if ($template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)) then (
            for $item in $template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
            return utiltemp:mergePrototypeTemplateForEdit($projectPrefix, $item, $prototype, $valueSetList, $templateList, $mode)
        ) else (
            for $item in $prototype/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
            return utiltemp:recurseItemForEdit($decor, $item, $valueSetList, $templateList, false(), $mode)
        )
    )
    else (
        for $item in $template/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
        return utiltemp:recurseItemForEdit($decor, $item, $valueSetList, $templateList, true(), $mode)
    )
};

declare %private function utiltemp:createTemplateVocabulary($vocabulary as element(vocabulary), $projectPrefix as item(), $valueSetList as element()*) as element(vocabulary) {

    <vocabulary>
    {
        $vocabulary/(@valueSet|@flexibility|@code|@codeSystem|@codeSystemName|@codeSystemVersion|@displayName|@domain)[not(.='')],
        utiltemp:isValueSetInScope($projectPrefix, $vocabulary/@valueSet, $vocabulary/@flexibility, $valueSetList),
        $vocabulary/*
    }
    </vocabulary>
    
};

(: TODO - deconstruct in smaller functions :)
declare %private function utiltemp:mergePrototypeTemplateForEdit($projectPrefix as xs:string, $item as element(), $prototype as element()?, $valueSetList as element()*, $templateList as element()*, $mode as xs:string) as element()* {
(: get corresponding node in prototype :)
    let $node             := if ($item/@id) then $prototype//*[name() = name($item)][@id = $item/@id] else util:eval('$prototype/' || utiltemp:path-to-node($item))
    let $node             := $node[1]
    (: get the reversed sequence for previous siblings of the corresponding node in the prototype template, if any :)
    let $precedingNodes := reverse($node/preceding-sibling::*)
    
    let $results        := (
        (: if there are no preceding nodes in template, get all preceding nodes from prototype
            else get preceding nodes up to node with same name as preceding template node
        :)
        if (count($item/preceding-sibling::*) = 0 and count($node/preceding-sibling::*) gt 0) then (
            for $precedingNode in $node/preceding-sibling::*
            return utiltemp:recurseItemForEdit($node/ancestor::decor, $precedingNode, $valueSetList, $templateList, false(), $mode)
        )
        else (
            (: check if there are preceding nodes in prototype that are not in the template
                in order not retrieve every single node preceding the current one, we first 
                try to find a previous matching node, anything between the previous and the 
                current item should be merged into the current template in this context 
            :)
            (:
            {
                "code": "MultipleIndexNodes",
                "description": "Unexpected error. Found multiple (2) index nodes so we do not know what to merge here. 
                    Current item path: element[@id='2.16.840.1.113883.3.1937.99.3.9.5723']/element[@id='2.16.840.1.113883.3.1937.99.3.9.5727'] 
                    Current preceding item path: element[@id='2.16.840.1.113883.3.1937.99.3.9.5723']/element[@id='2.16.840.1.113883.3.1937.99.99.906.9.4'] 
                    index nodes: '<element name='cda:templateId' datatype='II' minimumMultiplicity='1' maximumMultiplicity='1' isMandatory='true'>' '<element name='cda:templateId' datatype='II' minimumMultiplicity='1' maximumMultiplicity='1' isMandatory='true' id='2.16.840.1.113883.3.1937.99.3.9.5724'>'",
                "module": "/db/apps/api/modules/library/util-template-lib.xqm",
                "line": "357",
                "column": "21"
            }
            :)
            
            let $indexNode  := 
                for $precedingItem in $item/preceding-sibling::*
                let $n      := utiltemp:index-node-in-set($precedingNodes, $precedingItem)
                return
                    if (count($n) le 1) then ($n) else (
                        error(QName('http://art-decor.org/ns/error', 'MultipleIndexNodes'), 
                            'Unexpected error. Found multiple (' || count($n) || ') index nodes so we do not know what to merge here.' || 
                            ' Current item: ' || utiltemp:path-to-node($item) || ' ' || string-join(for $i in $item return concat('''&lt;', $i/name(), string-join(for $att in $i/@* return concat(' ', $att/name(),'=''', data($att),''''), ''), '&gt;'''),' ') ||
                            ' Current preceding item: ' || utiltemp:path-to-node($precedingItem) || ' ' || string-join(for $i in $precedingItem return concat('''&lt;', $i/name(), string-join(for $att in $i/@* return concat(' ', $att/name(),'=''', data($att),''''), ''), '&gt;'''),' ') || 
                            ' index nodes: ' ||  string-join(for $i in $n return concat('''&lt;', $i/name(), string-join(for $att in $i/@* return concat(' ', $att/name(),'=''', data($att),''''), ''), '&gt;'''),' '))
                    )
            
            let $index      := if ($indexNode) then utiltemp:index-of-node($precedingNodes,$indexNode[last()]) else (count($precedingNodes))
            
            for $n in reverse($precedingNodes)
            let $lin := utiltemp:index-of-node($precedingNodes,$n) lt $index and 
                        $n[ self::attribute|
                            self::element|
                            self::choice|
                            self::include|
                            self::let|
                            self::defineVariable|
                            self::assert|
                            self::report|
                            self::constraint ]
            return
                if ($lin) then utiltemp:recurseItemForEdit($node/ancestor::decor, $n, $valueSetList, $templateList, false(), $mode) else ()
        )
        ,
        if ($item/name()='attribute') then
            for $att in utiltemp:normalizeAttributes($item)
            return
            <attribute selected="{if ($node) then 'original' else ''}">
            {
                $att/@name,
                $att/@value,
                attribute conformance {
                    if ($att[@prohibited='true']) then 'NP' else
                    if ($att[@isOptional='true']) then 'O' else
                        'R'
                }
                ,
                $att/@datatype,
                $att/@id
                ,
                if ($node/@datatype) then (
                    attribute originalType {utiltemp:getDatatype($node/@datatype,$node/ancestor::template/classification/@format)},
                    attribute originalStrength {utiltemp:rewriteStrength($node/@strength)}
                ) else 
                if ($att/@datatype) then (
                    attribute originalType {utiltemp:getDatatype($att/@datatype,$att/ancestor::template/classification/@format)},
                    attribute originalStrength {utiltemp:rewriteStrength($att/@strength)}
                )
                else ()
                ,
                attribute originalConformance {
                    if ($node/@prohibited='true') then 'NP' else 
                    if ($node/@isOptional='true') then 'O' else 
                    if ($node) then 'R' else 
                        'O'
                }
                ,
                $att/desc            
                ,
                for $subitem in $att/(* except desc)
                return
                    if ($subitem[name()='vocabulary']) then utiltemp:createTemplateVocabulary($subitem, $projectPrefix, $valueSetList) else $subitem
    
            }
            </attribute>
        
        else if ($item/name()='element') then
            <element selected="{if ($node) then 'original' else ''}">
            {
                $item/(@* except (@strength|@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
                utiltemp:isTemplateInScope($projectPrefix, $item/@contains, $item/@flexibility, $templateList)
                ,
                if ($item/@strength) then attribute strength {utiltemp:rewriteStrength($item/@strength)} else ()
                ,
                if ($node/@datatype) then (
                    attribute originalType {utiltemp:getDatatype($node/@datatype,$node/ancestor::template/classification/@format)},
                    attribute originalStrength {utiltemp:rewriteStrength($node/@strength)}
                ) else 
                if ($item/@datatype) then (
                    attribute originalType {utiltemp:getDatatype($item/@datatype,$item/ancestor::template/classification/@format)},
                    attribute originalStrength {utiltemp:rewriteStrength($item/@strength)}
                )
                else ()
                ,
                if ($item/@minimumMultiplicity) then () else (
                    attribute minimumMultiplicity {''}
                )
                ,
                if ($item/@maximumMultiplicity) then () else (
                    attribute maximumMultiplicity {''}
                )
                ,
                if ($node/@minimumMultiplicity) then
                    attribute originalMin {$node/@minimumMultiplicity}
                else (
                    attribute originalMin {'0'}
                )
                ,
                if ($node/@maximumMultiplicity) then
                    attribute originalMax {$node/@maximumMultiplicity}
                else (
                    attribute originalMax {'*'}
                )
                ,
                if ($item/@conformance) then () else (
                    attribute conformance {'O'}
                )
                ,
                if ($item/@isMandatory) then () else (
                    attribute isMandatory {'false'}
                )
                ,
                if ($item/@strength) then () else (
                    attribute strength {''}
                )
                ,
                $item/desc            
                ,
                for $vocabulary in $item/vocabulary
                return 
                    utiltemp:createTemplateVocabulary($vocabulary, $projectPrefix, $valueSetList)
                ,
                $item/property,
                $item/item[1],
                $item/text,
                utiltemp:createTemplateExample($item/example)
                ,
                for $subItem in $item/(attribute|element|choice|include|constraint|let|defineVariable|assert|report)
                return
                    if ($node) then
                        utiltemp:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $node, $valueSetList, $templateList, $mode)
                    else (
                        utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, true(), $mode)
                    )
            }
            </element>
        
        else if ($item/name()='choice') then
            <choice selected="{if ($node) then 'original' else ''}">
            {
                $item/@*,
                if (not($item/@minimumMultiplicity)) then
                    attribute minimumMultiplicity {''}
                else ()
                ,
                if (not($item/@maximumMultiplicity)) then
                    attribute maximumMultiplicity {''}
                else ()
                ,
                if ($node/@minimumMultiplicity) then
                    attribute originalMin {$node/@minimumMultiplicity}
                else (attribute originalMin {'0'})
                ,
                if ($node/@maximumMultiplicity) then
                    attribute originalMax {$node/@maximumMultiplicity}
                else (attribute originalMax {'*'})
                ,
                $item/desc
                ,
                $item/item[1]
                ,
                for $subItem in $item/(element|choice|include|constraint)
                return
                    if ($node) then
                        utiltemp:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $node, $valueSetList, $templateList, $mode)
                    else (
                        utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, true(), $mode)
                    )
            }
            </choice>
        
        else if ($item/name()='include') then
            <include selected="{if ($node) then 'original' else ''}">
            {
                $item/(@* except (@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
                utiltemp:isTemplateInScope($projectPrefix, $item/@ref, $item/@flexibility, $templateList)
                ,
                if (not($item/@minimumMultiplicity)) then
                    attribute minimumMultiplicity {''}
                else ()
                ,
                if (not($item/@maximumMultiplicity)) then
                    attribute maximumMultiplicity {''}
                else ()
                ,
                if ($node/@minimumMultiplicity) then
                    attribute originalMin {$node/@minimumMultiplicity}
                else (
                    attribute originalMin {'0'}
                )
                ,
                if ($node/@maximumMultiplicity) then
                    attribute originalMax {$node/@maximumMultiplicity}
                else (
                    attribute originalMax {'*'}
                )
                ,
                if (not($item/@conformance)) then
                    attribute conformance {'O'}
                else ()
                ,
                if (not($item/@isMandatory)) then
                    attribute isMandatory {'false'}
                else ()
                ,
                $item/desc
                ,
                $item/item[1]
                ,
                utiltemp:createTemplateExample($item/example)
                ,
                for $subItem in $item/constraint
                return
                    if ($node) then
                        utiltemp:mergePrototypeTemplateForEdit($projectPrefix, $subItem, $node, $valueSetList, $templateList, $mode)
                    else (
                        utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, true(), $mode)
                    )
            }
            </include>
        
        else if ($item/name()='let') then
            <let selected="{if ($node) then 'original' else ''}">
            {
                attribute name {$item/@name},
                attribute value {$item/@value},
                $item/node()
            }
            </let>
        
        else if ($item/name()='defineVariable') then
            <defineVariable selected="{if ($node) then 'original' else ''}">
            {
                attribute name {$item/@name},
                attribute value {$item/@value},
                $item/node()
            }
            </defineVariable>
        
        else if ($item/name()=('assert','report')) then
            element { name($item) }
            {
                attribute selected {if ($node) then 'original' else ''},
                attribute role {$item/@role},
                attribute test {$item/@test},
                attribute see {$item/@see},
                attribute flag {$item/@flag},
                $item/node()
            }
       
       else if ($item/name()='constraint') then
            <constraint selected="{if ($node) then 'original' else ''}">
            {
                $item/(@* except @selected),
                $item/node()
            }
            </constraint>
        else (),
        (:
           if parent is not 'template' check if there are prototype nodes after the last template node
        :)
        if ($item/following-sibling::*) then () else 
        if ($node) then (
            (: our own current template had a match in the prototype :)
            for $n in ( $node/following-sibling::attribute|
                        $node/following-sibling::element|
                        $node/following-sibling::choice|
                        $node/following-sibling::include|
                        $node/following-sibling::let|
                        $node/following-sibling::defineVariable|
                        $node/following-sibling::assert|
                        $node/following-sibling::report|
                        $node/following-sibling::constraint)
            return
                utiltemp:recurseItemForEdit($node/ancestor::decor, $n, $valueSetList, $templateList, false(), $mode)
        )
        else (
            (: since our current node did not match, find any preceding node from our template that matched in the prototype :)
            let $precedingNodes :=
                for $n in ( $item/preceding-sibling::attribute|
                            $item/preceding-sibling::element|
                            $item/preceding-sibling::choice|
                            $item/preceding-sibling::include|
                            $item/preceding-sibling::let|
                            $item/preceding-sibling::defineVariable|
                            $item/preceding-sibling::assert|
                            $item/preceding-sibling::report|
                            $item/preceding-sibling::constraint)
                (: get corresponding node in prototype :)
                
                let $nodeInPrototype      := 
                    if ($n/@id) then 
                        $prototype/ancestor-or-self::template[1]//*[name() = name($n)][@id = $n/@id] 
                    else (
                        util:eval('$prototype/ancestor-or-self::template[1]/' || utiltemp:path-to-node($n))
                    )
                return
                    $nodeInPrototype[1]
            
            (: add all nodes from the prototype template following the last match that we had, to the end of the current template :)
            for $n in ( $precedingNodes[last()]/following-sibling::attribute|
                        $precedingNodes[last()]/following-sibling::element|
                        $precedingNodes[last()]/following-sibling::choice|
                        $precedingNodes[last()]/following-sibling::include|
                        $precedingNodes[last()]/following-sibling::let|
                        $precedingNodes[last()]/following-sibling::defineVariable|
                        $precedingNodes[last()]/following-sibling::assert|
                        $precedingNodes[last()]/following-sibling::report|
                        $precedingNodes[last()]/following-sibling::constraint)
             return
                utiltemp:recurseItemForEdit($n/ancestor::decor, $n, $valueSetList, $templateList, false(), $mode)
        )
    )
    
    for $result in $results
    return
        if ($result/attribute) then (
            element {name($result)} {
                $result/@*,
                $result/attribute[last()] | $result/attribute[last()]/preceding-sibling::* except $result/element,
                $result/attribute[last()]/preceding-sibling::element | $result/attribute[last()]/following-sibling::node()
            }
        )
        else (
            $result
        )
};

(: TODO - deconstruct in smaller functions :)
declare %private function utiltemp:recurseItemForEdit($projectPrefix as item(), $item as element(), $valueSetList as element()*, $templateList as element()*, $selected as xs:boolean, $mode as xs:string) as element()* {

    switch (name($item))
    case 'attribute' return
        for $att in utiltemp:normalizeAttributes($item)
        return
        <attribute>
        {
            if ($selected) then (attribute selected {''}) else ()
            ,
            $att/@name,
            $att/@value,
            attribute conformance {
                if ($att[@prohibited='true']) then 'NP' else
                if ($att[@isOptional='true']) then 'O' else
                    'R'
            }
            ,
            $att/@datatype,
            $att/@id
            ,
            if ($att/@strength) then (
                attribute strength {utiltemp:rewriteStrength($att/@strength)}
            ) else ()
            ,
            if ($att/@datatype) then (
                attribute originalType {utiltemp:getDatatype($att/@datatype,$item/ancestor::template/classification/@format)},
                attribute originalStrength {utiltemp:rewriteStrength($att/@strength)}
            ) else ()
            ,
            attribute originalConformance {
                if ($selected and $mode = 'new') then 
                    if ($att[@prohibited='true']) then 'NP' else
                    if ($att[@isOptional='true']) then 'O' else
                        'R'
                else ('O')
            }
            ,
            $att/desc
            ,
            for $subitem in $att/(* except desc)
            return
                if ($subitem[name()='vocabulary']) then utiltemp:createTemplateVocabulary($subitem, $projectPrefix, $valueSetList) else $subitem
        }
        </attribute>
    
    case 'element' return
        <element>
        {
            if ($selected) then (attribute selected {''}) else ()
            ,
            $item/(@* except (@strength|@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
            utiltemp:isTemplateInScope($projectPrefix, $item/@contains, $item/@flexibility, $templateList)
            ,
            if ($item/@strength) then (
                attribute strength {utiltemp:rewriteStrength($item/@strength)}
            ) else ()
            ,
            if ($item/@datatype) then (
                attribute originalType {utiltemp:getDatatype($item/@datatype,$item/ancestor::template/classification/@format)},
                attribute originalStrength {utiltemp:rewriteStrength($item/@strength)}
            )
            else ()
            ,
            if ($item/@minimumMultiplicity) then () else (
                attribute minimumMultiplicity {''}
            )
            ,
            if ($item/@maximumMultiplicity) then () else (
                attribute maximumMultiplicity {''}
            )
            ,
            attribute originalMin {'0'},
            attribute originalMax {'*'}
            ,
            if ($item/@conformance) then () else (
                let $conformance := 
                    if ($item[@isMandatory='true']) then 'R' else
                    if ($item[@minimumMultiplicity castable as xs:integer]/@minimumMultiplicity > 0) then 'R' else ('O')
                return
                attribute conformance {$conformance}
            )
            ,
            if ($item/@isMandatory) then () else (
                attribute isMandatory {'false'}
            )
            ,
            if ($item/@strength) then () else (
                attribute strength {''}
            )
            ,
            $item/desc
            ,
            for $vocabulary in $item/vocabulary
            return
                utiltemp:createTemplateVocabulary($vocabulary, $projectPrefix, $valueSetList)
            ,
            $item/property,
            $item/item[1],
            $item/text,
            utiltemp:createTemplateExample($item/example)
            ,
            for $subItem in $item/(attribute|element|choice|include|let|defineVariable|assert|report|constraint)
            return
                utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, $selected, $mode)
        }
        </element>
    
    case 'choice' return
        <choice>
        {
            if ($selected) then (attribute selected {''}) else ()
            ,
            $item/@*,
            if ($item/@minimumMultiplicity) then () else (
                attribute minimumMultiplicity {''}
            )
            ,
            if ($item/@maximumMultiplicity) then () else (
                attribute maximumMultiplicity {''}
            )
            ,
            attribute originalMin {'0'}
            ,
            attribute originalMax {'*'}
            ,
            $item/desc
            ,
            $item/item[1],
            for $subItem in $item/(element|choice|include|constraint)
            return
                utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, $selected, $mode)
        }
        </choice>
    
    case 'include' return
        <include>
        {
            if ($selected) then (attribute selected {''}) else ()
            ,
            $item/(@* except (@tmid|@tmname|@tmdisplayName|@linkedartefactmissing)),
            utiltemp:isTemplateInScope($projectPrefix, $item/@ref, $item/@flexibility, $templateList)
            ,
            if ($item/@minimumMultiplicity) then () else (
                attribute minimumMultiplicity {''}
            )
            ,
            if ($item/@maximumMultiplicity) then () else (
                attribute maximumMultiplicity {''}
            )
            ,
            attribute originalMin {'0'}
            ,
            attribute originalMax {'*'}
            ,
            if ($item/@conformance) then () else (
                let $conformance := if ($item/@minimumMultiplicity='1') then 'R' else ('O')
                return
                attribute conformance {$conformance}
            )
            ,
            if ($item/@isMandatory) then () else (
                attribute isMandatory {'false'}
            )
            ,
            if ($item/@flexibility) then () else (
                attribute flexibility {'dynamic'}
            )
            ,
            $item/desc
            ,
            $item/item[1]
            ,
            utiltemp:createTemplateExample($item/example)
            ,
            for $subItem in $item/constraint
            return
                utiltemp:recurseItemForEdit($projectPrefix, $subItem, $valueSetList, $templateList, $selected, $mode)
        }
        </include>
    
    case 'let' return
        <let>
        {
            if ($selected) then (attribute selected {''}) else (),
            attribute name {$item/@name},
            attribute value {$item/@value},
            $item/node()
        }
        </let>
    
    case 'assert'
    case 'report' return
        element { name($item) } 
        {
            if ($selected) then (attribute selected {''}) else (),
            attribute role {$item/@role},
            attribute test {$item/@test},
            attribute see {$item/@see},
            attribute flag {$item/@flag},
            $item/node()
        }
    
    case 'constraint' return
        <constraint>
        {
            if ($selected) then (attribute selected {''}) else (),
            $item/(@* except @selected),
            $item/node()
        }
        </constraint>
    
    default return ()
};

declare function utiltemp:createStaticAssociationElement($templateAssociations as element(templateAssociation)*, $language as xs:string?) as element(staticAssociations) {
    <staticAssociations>
    {
        for $association in $templateAssociations/concept
        let $concept            := utillib:getConcept($association/@ref, $association/@effectiveDate)
        let $dataset            := $concept/ancestor::dataset
        let $originalConcept    := utillib:getOriginalForConcept($concept)
        (: 2018-12-19 Disabled for performance reasons :)
        let $refdisplay         := if (1=1) then attribute refdisplay {utillib:getNameForOID($association/@ref, $language, $concept/ancestor::decor)} else ()
        let $path               := 
            for $c in $concept/ancestor::concept
            let $originalConceptName    := utillib:getOriginalForConcept($c)
            return 
                if ($originalConceptName/name[@language=$language]) then $originalConceptName/name[@language=$language] else ($originalConceptName/name[1])
        return
            <origconcept>
            {
                $association/@ref,
                $association/@effectiveDate,
                $refdisplay,
                attribute conceptStatusCode {$concept/@statusCode},
                if ($concept/@expirationDate) then attribute conceptExpirationDate {$concept/@expirationDate} else (),
                if ($concept/@versionLabel) then attribute conceptVersionLabel {$concept/@versionLabel} else (),
                attribute datasetId {$dataset/@id},
                attribute datasetEffectiveDate {$dataset/@effectiveDate},
                attribute datasetStatusCode {$dataset/@statusCode},
                if ($dataset/@expirationDate) then attribute datasetExpirationDate {$dataset/@expirationDate} else (),
                if ($dataset/@versionLabel) then attribute datasetVersionLabel {$dataset/@versionLabel} else (),
                attribute datasetName {$dataset/name[1]},
                attribute ident {$concept/ancestor::decor/project/@prefix},
                $association/@elementId,
                attribute templateId {$association/../@templateId},
                attribute templateEffectiveDate {$association/../@effectiveDate},
                $association/@elementPath,
                attribute path {string-join($path,' / ')}
                ,
                $originalConcept/name,
                $originalConcept/desc,
                for $name in $dataset/name
                return
                    <datasetName>{$name/@*, $name/node()}</datasetName>
                ,
                for $name in $concept/ancestor::decor/project/name
                return
                    <projectName>{$name/@*, $name/node()}</projectName>
            }
            </origconcept>
    }
    </staticAssociations>
};

declare function utiltemp:prepareTemplateForAPI($template as element(template)) as element(template) {
    let $ns-attributes  := 
        if ($template/@*[contains(name(), 'dummy')]) then () else ( 
            $template/../@*[contains(name(), 'dummy')]
        )
    return 
        <template>
        {
            $template/@*,
            if ($template/@ident) then () else (
                $template/../@ident,
                if ($template/@url) then () else $template/../@url
            ),
            $ns-attributes,
            for $ns-prefix at $i in in-scope-prefixes($template)
            let $ns-uri := namespace-uri-for-prefix($ns-prefix, $template)
            return
                (:hl7:dummy-default="urn:hl7-org:v3":)
                <ns uri="{$ns-uri}" prefix="{$ns-prefix}" default="{exists($template/@*[name() = $ns-prefix || ':dummy-default'])}"/>
            ,
            (: we return just a count of issues and leave it up to the calling party to get those if required :)
            <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $template/@id][@effectiveDate = $template/@effectiveDate] | $setlib:colDecorData//issue/object[@id = $template/@ref])}"/>
            ,
            (: we include all static association here that belong to our template. Note that there could be staticAssociation element under include. Those would come from a different template than our with potentially overlapping elementIds
               We do not want to include those because that would lead to false associations where a concept connected to a different, included template is assoicated to someting in the calling template.
            :)
            <staticAssociations>
            {
                for $node in $template//staticAssociations/*
                return
                    <concept>{$node/@*, $node/node()}</concept>
            }
            </staticAssociations>
            ,
            $template/desc
            ,
            utiltemp:collapseTemplateClassifications($template/classification)
            ,
            for $e in $template/(* except (ns|issueAssociation|staticAssociations|desc|classification|
                                                       element|attribute|include|choice|defineVariable|let|assert|report|property|item))
            return
                if ($e/self::example) then
                    <example>
                    {
                        if ($e/@type) then $e/@type else attribute type {'neutral'},
                        $e/@caption,
                        $e/node()
                    }
                    </example>
                else (
                    $e
                )
            ,
            utiltemp:prepareTemplateBodyForAPI($template/(element|attribute|include|choice|defineVariable|let|assert|report|property|item))
        }
        </template>
};

declare %private function utiltemp:prepareTemplateBodyForAPI($node as node()*) {
    for $e at $pos in $node
    let $elname := name($e)
    return
        <items>
        {
            attribute { 'is' } { $elname },
            attribute { 'order' } { $pos },
            if ($e/self::example) then (
                if ($e/@type) then $e/@type else attribute type {'neutral'},
                $e/@caption,
                $e/node()
            )
            else
            if ($e/self::assert | $e/self::report) then (
              $e/(@* except (@is | @order)),
              if ($e/@role) then () else attribute role {'error'},
              $e/node()
            )
            else (
                $e/(@* except (@is | @order))
                ,
                for $elm in $e/(node() except (ns|issueAssociation|classification|element|attribute|include|choice|defineVariable|let|assert|report|property|item|example))
                return
                    if ($elm/self::staticAssociations) then
                        <staticAssociations>
                        {
                            for $assoc in $elm/*
                            return
                                <concept>{$assoc/@*, $assoc/node()}</concept>
                        }
                        </staticAssociations>
                    else (
                        $elm
                    ) 
                ,
                utiltemp:prepareTemplateBodyForAPI($e/(element|attribute|include|choice|defineVariable|let|assert|report|property|item|example))
            )
        }
        </items>   
};

declare %private function utiltemp:collapseTemplateClassifications($classification as element(classification)*) {
    <classification>
    {
        ($classification/@format)[1]
        ,
        if ($classification/@type) then
            for $type in $classification/@type
            return
                <type>{data($type)}</type>
        else (
            <type>notype</type>
        ),
        for $tag in $classification/tag
        group by $tag
        return
            $tag
        ,
        for $property in $classification/property
        group by $property
        return
            $property
    }
    </classification>
};

declare function utiltemp:normalizeAttributes($attributes as element(attribute)*) as element(attribute)* {
    let $mapping    :=
        <attributes>
            <attribute id="1" name="classCode" datatype="cs"/>
            <attribute id="2" name="contextConductionInd" datatype="bl"/>
            <attribute id="3" name="contextControlCode" datatype="cs"/>
            <attribute id="4" name="determinerCode" datatype="cs"/>
            <attribute id="5" name="extension" datatype="st"/>
            <attribute id="6" name="independentInd" datatype="bl"/>
            <attribute id="7" name="institutionSpecified" datatype="bl"/>
            <attribute id="8" name="inversionInd" datatype="bl"/>
            <attribute id="9" name="mediaType" datatype="st"/>
            <attribute id="10" name="moodCode" datatype="cs"/>
            <attribute id="11" name="negationInd" datatype="bl"/>
            <attribute id="12" name="nullFlavor" datatype="cs"/>
            <attribute id="13" name="operator" datatype="cs"/>
            <attribute id="14" name="qualifier" datatype="set_cs"/>
            <attribute id="15" name="representation" datatype="cs"/>
            <attribute id="16" name="root" datatype="uid"/>
            <attribute id="17" name="typeCode" datatype="cs"/>
            <attribute id="18" name="unit" datatype="cs"/>
            <attribute id="19" name="use" datatype="set_cs"/>
        </attributes>
    
    for $attribute in $attributes
    for $att  at $i in $attribute/(@name|@classCode|@contextConductionInd|@contextControlCode|@determinerCode|
                                         @extension|@independentInd|@institutionSpecified|@inversionInd|
                                         @mediaType|@moodCode|@negationInd|@nullFlavor|
                                         @operator|@qualifier|@representation|@root|
                                         @typeCode|@unit|@use)
    let $anme := if ($att[name()='name']) then $att/string() else ($att/name())
    let $aval := if ($att[name()='name']) then $attribute/@value/string() else ($att/string())
    let $adt  := 
        if ($attribute/@datatype[not(.='')]) 
        then ($attribute/@datatype) 
        else if ($mapping/attribute/@name=$anme) 
        then (attribute datatype {$mapping/attribute[@name=$anme]/@datatype}) 
        else ()
    return
        <attribute name="{$anme}">
        {
            if ($adt=('bl','cs','set_cs') and contains($aval,'|')) then (
                (:boolean can only be true|false so no need to re-specify, cs/set_cs should be done in vocabulary:)
            ) else if (string-length($aval)>0) then (
                attribute value {$aval}
            ) else ()
            ,
            if ($attribute/@prohibited='true') then
                attribute prohibited {'true'}
            else (
                attribute isOptional {$attribute/@isOptional='true'}
            )
            ,
            $adt
            ,
            (:could lead to duplicates...:)
            if ($i=1) then ($attribute/@id) else ()
            ,
            $attribute/@selected
            ,
            $attribute/(desc|item|comment()),
            if ($adt=('cs','set_cs')) then (
                if (contains($aval,'|')) then (
                    for $v in tokenize($aval,'\|')
                    return <vocabulary code="{normalize-space($v)}"/>
                ) else ()
                ,
                $attribute/vocabulary
            ) else (
                $attribute/constraint
            )
        }
        </attribute>
};

(: Returns paths like this
    element[@name='hl7:informEvent'][empty(@contains)][@id='2.16.840.1.113883.2.4.3.111.3.7.9.16'][1]
:)
declare %private function utiltemp:path-to-node($nodes as node()*)  as xs:string* { 
    string-join(
        for $item in $nodes/ancestor-or-self::*
        return (
            if ($item[descendant-or-self::template]) then () else
            string-join(
                ($item/name(),
                if ($item[@id]) then (
                    '[@id=''',replace($item/@id,'''',''''''),''']'
                )
                else
                if ($item[self::attribute][@name]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[self::element][@name][@contains]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']',
                    '[@contains=''',replace($item/@contains,'''',''''''),''']'
                )
                else
                if ($item[self::element][@name = ('hl7:templateId', 'cda:templateId')][attribute[@name = 'root'][@value]]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']','[attribute[@name = ''root''][@value = ''',$item/attribute[@name = 'root']/@value,''']]',
                    '[',count($item/preceding-sibling::element[@name = $item/@name][attribute[@name = 'root'][@value]]) + 1,']'
                )
                else
                if ($item[self::element][@name][empty(@contains)]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']','[empty(@contains)]',
                    '[',count($item/preceding-sibling::element[@name = $item/@name][empty(@contains)]) + 1,']'
                )
                else
                if ($item[self::choice]) then (
                    if ($item[element]) then (
                        '[element/@name=(',for $n in $item/element/@name return concat('''',replace($n,'''',''''''),''''),')]'
                    ) else
                    if ($item[include]) then (
                        '[include/@ref=(',for $n in $item/include/@ref return concat('''',replace($n,'''',''''''),''''),')]'
                    ) else
                    if ($item[choice]) then (
                        '[choice]'
                    ) else
                    if ($item[@minimumMultiplicity][@maximumMultiplicity]) then (
                        '[@minimumMultiplicity = ',$item/@minimumMultiplicity,']','[@maximumMultiplicity = ',$item/@maximumMultiplicity,']'
                    ) else
                    if ($item[@minimumMultiplicity]) then (
                        '[@minimumMultiplicity = ',$item/@minimumMultiplicity,']'
                    ) else
                    if ($item[@maximumMultiplicity]) then (
                        '[@maximumMultiplicity = ',$item/@maximumMultiplicity,']'
                    ) else (
                        ''
                    )
                )
                else
                if ($item[self::include]) then (
                    '[@ref=''',replace($item/@ref,'''',''''''),''']'
                )
                else
                if ($item[self::let]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[self::defineVariable]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[self::assert]) then (
                    '[@test=''',replace($item/@test,'''',''''''),''']'
                )
                else
                if ($item[self::report]) then (
                    '[@test=''',replace($item/@test,'''',''''''),''']'
                )
                else
                if ($item[self::constraint]) then (
                    '[@language=''',replace($item/@language,'''',''''''),''']'
                )
                else
                if ($item[self::example]) then (
                    '[',count(preceding-sibling::example) + 1,']'
                )
                else
                if ($item[self::text]) then (
                    '[. = ''',replace(., '''', ''''''),''']'
                )
                else
                if ($item[@name]) then (
                    '[@name=''',replace($item/@name,'''',''''''),''']'
                )
                else
                if ($item[@code][@codeSystem]) then (
                    '[@code=''',replace($item/@code,'''',''''''),''']',
                    '[@codeSystem=''',replace($item/@codeSystem,'''',''''''),''']'
                )
                else
                if ($item[@valueSet]) then (
                    '[@valueSet=''',replace($item/@valueSet,'''',''''''),''']'
                )
                else
                if ($item[@language]) then (
                    '[@language=''',replace($item/@language,'''',''''''),''']'
                )
                else (
                    let $n  := $item/(@* except (@isOptional|@prohibited|@datatype|@id))[1]
                    return (
                        if ($n) then concat('[@',name($n),'=''',replace($n,'''',''''''),''']') else ()
                    )
                )
            ), '')
        )
    , '/')
};

declare %private function utiltemp:index-of-node($nodes as node()* , $nodeToFind as node()) as xs:integer* {
    for $seq in (1 to count($nodes))
    return $seq[$nodes[$seq] is $nodeToFind]
};

declare %private function utiltemp:index-node-in-set($nodes as node()*, $nodeToFind as node()) as element()* {
    let $n      :=
        if ($nodeToFind[@id]) then
            $nodes[@id = $nodeToFind/@id]
        else
        if ($nodeToFind[@name]/name()='attribute') then
            $nodes[self::attribute][@name = $nodeToFind/@name]
        else
        if ($nodeToFind[@contains]/name()='element') then
            $nodes[self::element][@name = $nodeToFind/@name][@contains = $nodeToFind/@contains]
        else
        (: exceptional case in template id="2.16.840.1.113883.10.20.29.1" name="USRealmHeaderforPatientGeneratedDocumentV2" where templateId is defined twice with same @root, different extension. first no @id, second with @id :)
        if ($nodeToFind[self::element][@name = ('hl7:templateId', 'cda:templateId')][attribute[@name = 'root'][@value]][attribute[@name = 'extension'][@value]]) then
            $nodes[self::element][@name = $nodeToFind/@name][attribute[@name = 'root'][@value = $nodeToFind/attribute[@name = 'root']/@value]][attribute[@name = 'extension'][@value = $nodeToFind/attribute[@name = 'extension']/@value]]
        else
        if ($nodeToFind[self::element][@name = ('hl7:templateId', 'cda:templateId')][attribute[@name = 'root'][@value]]) then
            $nodes[self::element][@name = $nodeToFind/@name][attribute[@name = 'root'][@value = $nodeToFind/attribute[@name = 'root']/@value]]
        else
        if ($nodeToFind/name()='element') then
            $nodes[self::element][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='choice') then
            if ($nodeToFind[*/@id]) then
                $nodes[self::choice][*/@id = $nodeToFind/*/@id]
            else
            if ($nodeToFind[element]) then
                $nodes[self::choice][element/@name = $nodeToFind/element/@name]
            else
            if ($nodeToFind[include]) then
                $nodes[self::choice][include/@ref = $nodeToFind/include/@ref]
            else
            if ($nodeToFind[choice]) then
                $nodes[self::choice][choice]
            else
            if ($nodeToFind[@minimumMultiplicity][@maximumMultiplicity]) then
                $nodes[self::choice][@minimumMultiplicity = $nodeToFind/@minimumMultiplicity][@maximumMultiplicity = $nodeToFind/@maximumMultiplicity]
            else
            if ($nodeToFind[@minimumMultiplicity]) then
                $nodes[self::choice][@minimumMultiplicity = $nodeToFind/@minimumMultiplicity]
            else
            if ($nodeToFind[@maximumMultiplicity]) then
                $nodes[self::choice][@maximumMultiplicity = $nodeToFind/@maximumMultiplicity]
            else (
                $nodes[self::choice]
            )
        else
        if ($nodeToFind/name()='include') then
            $nodes[self::include][@ref = $nodeToFind/@ref]
        else
        if ($nodeToFind/name()='let') then
            $nodes[self::let][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='defineVariable') then
            $nodes[self::defineVariable][@name = $nodeToFind/@name]
        else
        if ($nodeToFind/name()='assert') then
            $nodes[self::assert][@test = $nodeToFind/@test]
        else
        if ($nodeToFind/name()='report') then
            $nodes[self::report][@test = $nodeToFind/@test]
        else
        if ($nodeToFind/name()='text') then 
            $nodes[self::text][. = $nodeToFind]
        else
        if ($nodeToFind/name()='constraint') then
            $nodes[self::constraint][@language = $nodeToFind/@language]
        else
        if ($nodeToFind/name()='example') then
            $nodes[self::example][count($nodeToFind/preceding-sibling::example) + 1]
        else (
            $nodes[name()=$nodeToFind/name()][string-join(for $att in (@* except (@isOptional|@prohibited|@datatype|@id)) order by name($att) return $att,'')=$nodeToFind/string-join(for $att in (@* except (@isOptional|@prohibited|@datatype|@id)) order by name($att) return $att,'')]
        )
    
    return $n
};

(: keep it simple for performance. if the requested valueSet is in the project terminology by ref, don't look further :)
declare %private function utiltemp:isValueSetInScope($projectPrefix as item(), $ref as attribute()?,$flexibility as attribute()?,$valueSetList as element()*) as attribute()* {
    if (string-length($ref)=0) then ()
    else (
        let $vsElms := $valueSetList[(@id|@name|@ref)=$ref]
        let $vsEff  := if ($flexibility castable as xs:dateTime) then $flexibility else (max($vsElms[@effectiveDate]/xs:dateTime(@effectiveDate)))
       
        return
            if ($vsElms[@effectiveDate=$vsEff]) then (
                attribute vsid {$vsElms[@effectiveDate=$vsEff][1]/(@id|@ref)},
                attribute vsname {$vsElms[@effectiveDate=$vsEff][1]/@name},
                attribute vsdisplayName {$vsElms[@effectiveDate=$vsEff][1]/@displayName}
            )
            else if ($vsElms[@ref]) then (
                attribute vsid {$vsElms[1]/(@id|@ref)},
                attribute vsname {$vsElms[1]/@name},
                attribute vsdisplayName {$vsElms[1]/@displayName}
            )
            else (
                attribute linkedartefactmissing {'true'}
            )
    )
};

(: keep it simple for performance. if the requested template is in the project rules by ref, don't look further :)
declare %private function utiltemp:isTemplateInScope($projectPrefix as item(), $ref as attribute()?,$flexibility as attribute()?,$templateList as element()*) as attribute()* {
    if (string-length($ref)=0) then ()
    else (
        let $templates  := (utiltemp:getTemplateById($ref, ($flexibility, 'dynamic')[1], $projectPrefix, (), ())[@id])[1]
        
        return
            if ($templates) then (
                attribute tmid {$templates/@id},
                attribute tmdisplayName {if ($templates/@displayName) then $templates/@displayName else $templates/@name}
            ) else (
                attribute linkedartefactmissing {'true'}
            )
    )
};

declare %private function utiltemp:getDatatype($datatype as xs:string?, $classification-format as xs:string?) as xs:string? {
    let $classification-format  := if (empty($classification-format)) then 'hl7v3xml1' else $classification-format
    let $datatypes              := $setlib:colDecorCore//supportedDataTypes[@type = $classification-format]
    let $datatypeName           := if (contains($datatype,':')) then substring-after($datatype,':') else ($datatype)
    let $flavor                 := $datatypes//flavor[@name=$datatypeName]
    return
        if ($flavor) then ($flavor/ancestor::dataType[1]/@name) else ($datatype)
};

(: re-write attribute strength CNE is required and CWE is extensible :)
declare %private function utiltemp:rewriteStrength($os as xs:string?) as xs:string? {
    let $r := if ($os = 'CNE') then 'required' else if ($os = 'CWE') then 'extensible' else $os
    return $r
};

