xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
module namespace treplib            = "http://art-decor.org/ns/api/terminology/reportlib";

import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";

declare %private variable $treplib:supportedLanguages   := ('en-US','en-GB','nl-NL','de-DE','fr-FR','it-IT','pl-PL');

(:
   Get a representation of the decor file with minimal info based on the active filter
   If filter is on
      - Get transactions in filter
      - Walk trough source datasets and select concepts
      - Walk trough concepts and get associated valuesets
      
:)
declare function treplib:getFilteredDecor($decor as element()) as element()* {
   let $filters := collection(util:collection-name($decor))/filters[util:document-name(.) = 'filters.xml']
   let $filterActive := empty($filters[@filter = 'off']) and $filters[*] and (empty($filters/@filterId) or $filters/@filterId = $filters/filters/@id)
   let $filterId := $filters/@filterId/string()
   let $filterLabel := $filters/filters[@id=$filters/@filterId]/@label
   let $filteredTransactions := 
      if ($filterActive) then
         for $transaction in $filters/filters[@id=$filters/@filterId]/transaction
         let $trans := $decor//transaction[@id=$transaction/@ref][@effectiveDate=$transaction/@flexibility]
         return
         <scenario>
         {
         $trans
         }
         </scenario>
      else
         let $trans := $decor//transaction
         return
         <scenario>
         {
         $trans
         }
         </scenario>
   
   let $filteredDatasets      :=
         if ($filterActive) then 
            for $representingTemplate in $filteredTransactions//representingTemplate[@sourceDataset]
            let $sourceDataset := 
                              if ($decor//dataset[@id=$representingTemplate/@sourceDataset][@sourceDatasetFlexibility] castable as xs:dateTime) then
                                 $decor//dataset[@id=$representingTemplate/@sourceDataset][@effectiveDate=$representingTemplate/@sourceDatasetFlexibility]
                              else
                                let $dSet := $decor//dataset[@id=$representingTemplate/@sourceDataset]
                                let $latestDate := max($dSet/xs:dateTime(@effectiveDate))
                                return
                                $dSet[@effectiveDate=$latestDate]
            return
            <dataset>
               {
               $sourceDataset/@*,
               $sourceDataset/name
               }
               <transaction>
               {
                $representingTemplate/parent::transaction/name
                }
               </transaction>
               {
               for $concept in $sourceDataset/concept[@id=$representingTemplate/concept/@ref][@statusCode=('draft','final','pending')][not(inherit)]
               return
               treplib:copyConcept($concept, $filterActive, $representingTemplate)
               }
            </dataset>
         else
         for $dataset in $decor/datasets/dataset[@statusCode=('draft','final','pending')]
         return
         <dataset>
               {
               $dataset/@*,
               $dataset/name,
               for $concept in $dataset/concept[@statusCode=('draft','final','pending')][not(inherit)]
               return
               treplib:copyConcept($concept, $filterActive, ())
               }
         </dataset>
         
   let $filteredValuesetIds :=
         for $conceptList in $filteredDatasets//conceptList
            let $conceptListAssociations:= $decor/terminology/terminologyAssociation[@conceptId=$conceptList/@id]
            for $association in $conceptListAssociations
               let $valueSet := 
                  if ($association/@flexibility='dynamic') then
                     let $vSet := $decor/terminology/valueSet[@id=$association/@valueSet]
                     let $latestDate := max($vSet/xs:dateTime(@effectiveDate))
                     return
                     $vSet[@effectiveDate=$latestDate]
                  else if ($association/@flexibility castable as xs:dateTime) then
                     $decor/terminology/valueSet[@id=$association/@valueSet][@effectiveDate=$association/@flexibility]
                  else ()
               return
               concat($valueSet/@id,'--',$valueSet/@effectiveDate)
   
   let $filteredValuesets :=
      for $id in distinct-values($filteredValuesetIds)
      let $valueset := $decor/terminology/valueSet[@id=substring-before($id,'--')][@effectiveDate=substring-after($id,'--')]
      return
      $valueset
         
   return
      <filteredDecor>
      {
      if ($filterActive) then
      <filter  filterId="{$filterId}" filterLabel="{$filterLabel}"/>
      else ()
      }
      <project>
      {$decor/project/@*}
      </project>
      <datasets>
         {
         $filteredDatasets
         }
      </datasets>
      <scenarios>
         {
         $filteredTransactions
         }
      </scenarios>
      <ids>
      {
         $decor/ids/id
      }
      </ids>
      <terminology>
      {
         $decor/terminology/terminologyAssociation,
         $decor/terminology/codeSystem,
         $filteredValuesets
      }
      </terminology>
      </filteredDecor>
};

declare function treplib:copyConcept($concept as element(), $filterActive as xs:boolean, $representingTemplate as element()*) as element()*{
   <concept>
      {
      $concept/@*,
      $concept/*[not(name()='concept')],
      if ($filterActive) then
         for $child in  $concept/concept[@id=$representingTemplate/concept/@ref][@statusCode=('draft','final','pending')][not(inherit)]
         return
         treplib:copyConcept($child, $filterActive, $representingTemplate)
      else
         for $child in  $concept/concept[@statusCode=('draft','final','pending')][not(inherit)]
         return
         treplib:copyConcept($child, $filterActive, $representingTemplate)
      }
   </concept>
};

(: Traverse datasets :)
declare function treplib:traverseDatasets($decor as element()) as element()* {
   for $dataset in $decor/datasets/dataset
   let $concepts := $dataset//concept[not(parent::conceptList)]
   return
   <dataset conceptCount="{count($concepts)}" json:array="true">
   {
      $dataset/@*,
      $dataset/name,
      $dataset/transaction,
      for $concept in $concepts
      let $terminologyAssociations := $decor/terminology/terminologyAssociation[@conceptId=$concept/@id]
      let $checkedConcept :=
         <concept json:array="true">
   {
      $concept/@*,
      $concept/name,
      if ($terminologyAssociations) then
         for $association in $terminologyAssociations
         return
         treplib:handleTerminologyAssociation($decor,$association)
      else
      <message severity="info" type="noAssociation" json:array="true">No terminology association defined</message>
       ,
       if ($concept/valueDomain/conceptList) then
         let $conceptListAssociation:= $decor/terminology/terminologyAssociation[@conceptId=$concept/valueDomain/conceptList/@id]
         return
         <conceptList>
            {
            (
            if ($conceptListAssociation) then
               for $association in $conceptListAssociation
               return
               treplib:handleConceptListAssociation($decor,$association)
            else
            <message severity="warning" type="noValuesetAssociated" json:array="true">No valueset associated with conceptlist</message>
            ,
            for $listConcept in $concept/valueDomain/conceptList/concept
               let $listConceptAssociations := $decor/terminology/terminologyAssociation[@conceptId=$listConcept/@id]
               return
               <listConcept>
               {
               $listConcept/@id,
               $listConcept/name,
               if($listConceptAssociations) then
                  for $association in $listConceptAssociations
                  return
                  treplib:handleTerminologyAssociation($decor,$association)
               
               else
               <message severity="warning" type="noValuesetItem" json:array="true">No valueset item associated with concept in list</message>
             }
               </listConcept>
             )
             }
          </conceptList>
       else if ($concept/valueDomain/@type='quantity') then
               let $ucum := $setlib:colDecorCore/ucums/ucum
               for $valDomain in $concept/valueDomain[@type='quantity']
                  for $property in $valDomain/property
                  let $ucumCheck := $ucum[@unit=$property/@unit]
                  return
                  if (empty($ucumCheck)) then
                     <message severity="info" type="noUcumUnit" json:array="true">{$property/@unit}</message>
                  else ()
             else ()
    }  
   </concept>
    return
         <concept>
         {
         $checkedConcept/@*,
         $checkedConcept/name,
         if ($checkedConcept//message[not(@type='ok')]) then
            $checkedConcept/message[not(@type='ok')]
         else
            <message severity="info" type="ok" json:array="true">OK</message>
         ,
         $checkedConcept/conceptList
         }  
   </concept>
    }  
   </dataset>
};


(: handle terminology association :)
declare function treplib:handleTerminologyAssociation($decor as element(),$terminologyAssociation as element()){
   (: check if codesystem is in project :)
   if (starts-with($terminologyAssociation/@codeSystem,$decor/project/@id)) then
      let $localConcept :=  $decor//codedConcept[@code=$terminologyAssociation/@code][ancestor::codeSystem/@id=$terminologyAssociation/@codeSystem]
      return
      if ($localConcept) then
         treplib:checkConcept($terminologyAssociation,$localConcept)
      else
         (: check if code system is present :)
         if ($decor//codeSystem[@id=$terminologyAssociation/@codeSystem]) then
            <message severity="error" type="conceptNotFound" json:array="true">
            {
               $terminologyAssociation
            }
            </message>
         else
            let $codeSystemName := if ($terminologyAssociation/@codeSystemName) then $terminologyAssociation/@codeSystemName else '?'
            return
            <message severity="warning" type="codesystemNotfound" json:array="true">
            {
               $terminologyAssociation
            }
            </message>
   else
      (: handle SNOMED post coordinated codes :)
      if ($terminologyAssociation/@codeSystem='2.16.840.1.113883.6.96') then
         if ($terminologyAssociation/@code castable as xs:integer) then
            let $snomedConcept := collection($setlib:strCodesystemStableData)//concept[@code=$terminologyAssociation/@code][ancestor::browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem]
            let $log := util:log('info',$terminologyAssociation)
            return
            if ($snomedConcept) then
               treplib:checkConcept($terminologyAssociation,$snomedConcept)
            else
               (: check if code system is present :)
               if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem) then
                  <message severity="error" type="conceptNotFound" json:array="true"><codeSystem                  oid="{$terminologyAssociation/@codeSystem}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$terminologyAssociation/@codeSystem]/name}</codeSystem>
                  {
                     $terminologyAssociation
                  }
                  </message>
               else
                  <message severity="warning" type="codesystemNotfound" json:array="true">
                  {
                     $terminologyAssociation
                  }</message>
         else treplib:checkAssociationSnomedExpression($terminologyAssociation/@code, $terminologyAssociation/@displayName)
         
      else
         let $codeSystemConcept := collection($setlib:strCodesystemStableData)//concept[@code=$terminologyAssociation/@code][ancestor::browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem]
         return
         if ($codeSystemConcept) then
            treplib:checkConcept($terminologyAssociation,$codeSystemConcept)
         else
            (: check if code system is present :)
            if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem) then
               <message severity="error" type="conceptNotFound" json:array="true"><codeSystem               oid="{$terminologyAssociation/@codeSystem}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$terminologyAssociation/@codeSystem]/name}</codeSystem>
               {
                  $terminologyAssociation
               }
               </message>
            else
               let $codeSystemName := if ($terminologyAssociation/@codeSystemName) then $terminologyAssociation/@codeSystemName else '?'
               return
               <message severity="warning" type="codesystemNotfound" json:array="true">
               {
                  $terminologyAssociation
               }
               </message>
};


(: handle conceptlist association :)
declare function treplib:handleConceptListAssociation($decor as element(), $conceptListAssociation as element()) {
   (: check if valueset is in project :)
   if (starts-with($conceptListAssociation/@valueSet,$decor/project/@id)) then
      let $valueSet := 
               if ($conceptListAssociation/@flexibility='dynamic') then
                  let $vSet := $decor/terminology/valueSet[@id=$conceptListAssociation/@valueSet]
      let $latestDate := max($vSet/xs:dateTime(@effectiveDate))
                  return
      $vSet[@effectiveDate=$latestDate]
               else if ($conceptListAssociation/@flexibility castable as xs:dateTime) then
                  $decor/terminology/valueSet[@id=$conceptListAssociation/@valueSet][@effectiveDate=$conceptListAssociation/@flexibility]
               else ()
      return
      if ($valueSet) then
         (: check if conceptlist contains concepts and is of the same size as the valueset :)
         let $conceptList := $decor//conceptList[@id=$conceptListAssociation/@conceptId]
         return
         if ($conceptList/concept) then
            if (count($conceptList/concept) = count($valueSet//concept | $valueSet//exception)) then
               <message severity="info" type="ok" json:array="true"><valueSet displayName="{$valueSet/@displayName}" size="{count($valueSet//concept | $valueSet//exception)}"/></message>
      else
         <message severity="info" type="valueSetSizeNotConceptListSize" json:array="true"><conceptList size="{count($conceptList/concept)}"/><valueSet displayName="{$valueSet/@displayName}" size="{count($valueSet//concept | $valueSet//exception)}"/></message>
         else
            <message severity="info" type="ok" json:array="true">OK</message>
      else
         <message severity="error" type="valuesetNotFound" json:array="true">Valueset not found</message>
   else()

};


declare %private function treplib:checkConcept($terminologyAssociation as element(), $concept as element()) as element()*{
   let $displayName := $terminologyAssociation/@displayName
   let $designations :=
         for $designation in $concept/designation[not(@use='rel')][@lang=$treplib:supportedLanguages]
               order by $designation/@lang
               return
               <designation json:array="true">
                  {
                  $designation/@*,
                  $designation/text(),
                  if ($designation/*) then
                     let $parts := 
                     for $part in $designation/*
                     return
                     $part/text()
                     return
                     string-join($parts,' ')
                  else ()
                  }
               </designation>
   let $statusMessage :=
      if ($concept/@statusCode='retired') then
         <message severity="warning" type="conceptRetired" json:array="true">
            <concept code="{$concept/@code}">
               {
            for $designation in $concept/designation[@use='pref']
               return
               <designation json:array="true">
                  {
                  $designation/@*,
                  $designation/text()
                  }
               </designation>
               }
            </concept>
            {
            $terminologyAssociation,
            for $association in $concept/association
            return
            <association json:array="true">
               {
               $association/@*,
               $association/*
               }
            </association>
            }
         </message>
      else if ($concept/@statusCode='draft') then
         <message severity="warning" type="conceptDraft" json:array="true">
            {$terminologyAssociation}
         </message>
      else if ($concept/@statusCode='experimental') then
         <message severity="warning" type="conceptExperimental" json:array="true">
            {$terminologyAssociation}
         </message>
      else
         <message severity="info" type="ok" json:array="true">OK</message>
      
   let $designationMessage :=
      if ($displayName=$designations) then
            <message severity="info" type="ok" json:array="true">OK</message>
         else 
            let $lowerCase    := 
                                for $designation in $designations
                                return
            lower-case($designation)
            return
            if (lower-case($displayName)=$lowerCase) then
               <message severity="warning" type="designationCaseMismatch" json:array="true" displayName="{$displayName}">Case of display name does not match designation case in concept
               {
            $designations,
            $terminologyAssociation
 } </message>
               else
            <message severity="warning" type="noMatchingDesignation" json:array="true" displayName="{$displayName}">Display name does not match concept designation
               {
               $designations,
            $terminologyAssociation
            }
            </message>
            let $moduleMessage :=
      if ($concept/@moduleId) then
         if ($concept/@moduleId=('900000000000207008','900000000000012004')) then
            <message severity="info" type="ok" json:array="true">OK</message>
         else
            <message severity="info" type="notCoreModule" json:array="true">Concept is not part of core module</message>
      else ()
            
   return
               if ($statusMessage/@type='ok' and $designationMessage/@type='ok' and $moduleMessage/@type='ok') then
         <message severity="info" type="ok" json:array="true">OK</message>
      else
         ($statusMessage[not(@type='ok')],$designationMessage[not(@type='ok')],$moduleMessage[not(@type='ok')])
};

declare function treplib:checkValueSetConcept($displayName as xs:string, $concept as element()) as element()*{
   let $designations :=
         for $designation in $concept/designation[not(@use='rel')][@lang=$treplib:supportedLanguages]
               order by $designation/@lang
               return
   <designation json:array="true">
               {
                  $designation/@*,
                  $designation/text(),
               if               ($designation/*)   then
   let $parts := 
 for $part in $designation/*
 return
 $part/text()
 return
 string-join($parts,' ')
 else ()
   }
               </designation>
   let $statusMessage :=
      if ($concept/@statusCode='retired') then
         <message severity="warning" type="conceptRetired" json:array="true">
            <concept code="{$concept/@code}">
               {
            for $designation in $concept/designation[@use='pref']
               return
               <designation json:array="true">
                  {
                  $designation/@*,
                  $designation/text()
                  }
               </designation>
               }
            </concept>
            {
            for $association in $concept/association
            return
            <association json:array="true">
               {
               $association/@*,
               $association/*
               }
            </association>
            }
         </message>
      else if ($concept/@statusCode='draft') then
         <message severity="warning" type="conceptDraft" json:array="true">Concept is in draft</message>
      else if ($concept/@statusCode='experimental') then
         <message severity="warning" type="conceptExperimental" json:array="true">Concept is experimental</message>
      else
         <message severity="info" type="ok" json:array="true">OK</message>
   let $designationMessage :=
      if ($displayName=$concept/designation) then
            <message severity="info" type="ok" json:array="true">OK</message>
      else
         let $lowerCase    := 
                                for $designation in $designations
                                return
                                lower-case($designation)
            return
            if (lower-case($displayName)=$lowerCase) then
               <message severity="warning" type="designationCaseMismatch" displayName="{$displayName}" json:array="true">Case of display name does not match designation case in concept
               {
               $designations
               }
               </message>
            else
               <message severity="warning" type="noMatchingDesignation" displayName="{$displayName}" json:array="true">Display name does not match concept designation
               {
               $designations
               }
               </message>
   let $moduleMessage :=
      if ($concept/@moduleId) then
         if ($concept/@moduleId=('900000000000207008','900000000000012004')) then
            <message severity="info" type="ok" json:array="true">OK</message>
         else
            <message severity="info" type="notCoreModule" json:array="true">Concept is not part of core module</message>
      else()      
   return
 if ($statusMessage/@type='ok' and         $designationMessage/@type='ok' and $moduleMessage/@type='ok') then
         <message severity="info" type="ok"      json:array="true">OK</message>
   else
   ($statusMessage[not(@type='ok')],$designationMessage[not(@type='ok')],$moduleMessage[not(@type='ok')])
};


(: Traverse valuesets :)
declare function treplib:traverseValuesets($decor as element()) as element()*{
   for $valueSet in $decor/terminology/valueSet[@id][@statusCode=('draft','final','pending')]
   let $sourceCodeSystems := 
            for $system in distinct-values($valueSet/conceptList/concept/@codeSystem)
            return
   <sourceCodeSystem id="{$system}">
               {
               (: check if codesystem is local :)
               if (starts-with($system,$decor/project/@id)) then
                  if ($decor//codeSystem[@id=$system]) then
                     <message severity="info" type="ok">OK</message>
                  else
                     <message severity="warning" type="codesystemNotfound">Code stystem not found</message>
               else
                  if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$system) then
                     <message severity="info" type="ok">OK</message>
                  else
                     <message severity="warning" type="codesystemNotfound">Code stystem not found</message>
               }
            </sourceCodeSystem>
   order by $valueSet/@displayName
   return
   <valueSet conceptCount="{count($valueSet/conceptList/concept)}" json:array="true">
      {
      $valueSet/@*,
      $sourceCodeSystems,
      for $concept in $valueSet/conceptList/concept
         return
         <concept json:array="true">
         {
         $concept/@*,
         $concept/desc,
         if (starts-with($concept/@code,'http')) then
            <message severity="error" type="uriNotValidCode" json:array="true">URI is not a valid code</message>
         (: check if codesystem is in project :)
         else if (starts-with($concept/@codeSystem,$decor/project/@id)) then
            let $localConcept :=  $decor//codedConcept[@code=$concept/@code][ancestor::codeSystem/@id=$concept/@codeSystem]
            return
            if ($localConcept) then
               treplib:checkValueSetConcept($concept/@displayName,$localConcept)
            else
               (: check if code system is present :)
               if ($decor//codeSystem[@id=$concept/@codeSystem]) then
                  <message severity="error" type="conceptNotFound" json:array="true">Concept not found</message>
               else
                  let $codeSystemName := if ($concept/@codeSystemName) then $concept/@codeSystemName else '?'
                  return
                  <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
         else
            (: handle SNOMED post coordinated codes :)
            if ($concept/@codeSystem='2.16.840.1.113883.6.96') then
               if ($concept/@code castable as xs:integer) then
                  let $snomedConcept := collection($setlib:strCodesystemStableData)//concept[@code=$concept/@code][ancestor::browsableCodeSystem/@oid=$concept/@codeSystem]
                  return
                  if ($snomedConcept) then
                     treplib:checkValueSetConcept($concept/@displayName,$snomedConcept)
                  else
                     (: check if code system is present :)
                     if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$concept/@codeSystem) then
                        <message severity="error" type="conceptNotFound" json:array="true"><codeSystem oid="{$concept/@codeSystem}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$concept/@codeSystem]/name}</codeSystem></message>
                     else
                        <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
               else 
                  treplib:checkSnomedExpression($concept/@code,$concept/@displayName)
               
            
                     else
               let $codeSystemConcept := collection($setlib:strCodesystemStableData)//concept[@code=$concept/@code][ancestor::browsableCodeSystem/@oid=$concept/@codeSystem]
               return
               if ($codeSystemConcept) then
                  treplib:checkValueSetConcept($concept/@displayName,$codeSystemConcept)
               else
                  (: check if code system is present :)
                  if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$concept/@codeSystem) then
                     <message severity="error" type="conceptNotFound" json:array="true"><codeSystem oid="{$concept/@codeSystem}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$concept/@codeSystem]/name}</codeSystem></message>
                  else
                     let $codeSystemName := if ($concept/@codeSystemName) then $concept/@codeSystemName else '?'
                     return
                     <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
             }
            </concept>         
            }
   </valueSet>

};

(:
   Check SNOMED postcoordinated expression
   Removes terms from expression and calls parseSnomedExpression
:)

declare function treplib:checkAssociationSnomedExpression ($expression as xs:string, $displayName as xs:string) {
   if (starts-with($expression,'=')) then
      <message severity="error" type="equalSignNotAtStart" json:array="true">equal sign is not allowed at start of a postcoordinated code</message>
   else if (starts-with($expression,'http')) then
      <message severity="error" type="uriNotValidCode" json:array="true">URI is not a valid code</message>
   else
      let $result := treplib:parseSnomedExpression(replace($expression,'\|(.*?)\|',''))
   return
   if ($result//message or name($result[1])='message') then
      let $errorType :=
            if ($result//message) then
               $result//message[1]/@type
            else 
               $result[1]/@type
          return
         <message severity="error" type="errorInExpression" json:array="true">
         <expression string="{$expression}" displayName="{$displayName}">
            {
            $result
            }
         </expression>
      </message>
   else
      <message severity="info" type="ok" json:array="true">
         <expression string="{$expression}" displayName="{$displayName}">
            {
            $result
            }
         </expression>
      </message>
};

declare function treplib:checkSnomedExpression ($expression as xs:string,$displayName as xs:string) {
   if (starts-with($expression,'=')) then
      <message severity="error" type="equalSignNotAtStart" json:array="true">equal sign is not allowed at start of a postcoordinated code</message>
   else if (starts-with($expression,'http')) then
      <message severity="error" type="uriNotValidCode" json:array="true">URI is not a valid code</message>
   else
      let $result := treplib:parseSnomedExpression(replace($expression,'\|(.*?)\|',''))
   return
   if ($result//message or name($result[1])='message') then
      let $errorType :=
            if ($result//message) then
               $result//message[1]/@type
            else 
               $result[1]/@type
          return
         <message severity="error" type="errorInExpression" json:array="true">
      <expression string="{$expression}" displayName="{$displayName}">
            {
      $result
      }
      </expression>
      </message>
   else
      <message severity="info" type="ok" json:array="true">
      <expression string="{$expression}" displayName="{$displayName}">
            {
         $result
      }
      </expression>
      </message>
};

(:
   Recursive function for parsing Snomed expression.
   Returns xml structure of the expression with detected errors
:)

declare %private function treplib:parseSnomedExpression ($expression as xs:string) {
   (: check if refset member :)
   if (starts-with(normalize-space($expression),'^')) then 
      if (normalize-space(substring-after($expression,'^')) castable as      xs:integer) then
         let $memberCode := normalize-space(substring-after($expression,'^'))
         return
         <memberOf>
            <concept code="{$memberCode}">{treplib:getSnomedConceptDesignations($memberCode)}</concept>
         </memberOf>
      else
      <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
   (: check for multiple focus concepts :)
   else if (normalize-space(substring-before($expression,'+')) castable as      xs:integer) then
   let $focusCode := normalize-space(substring-before($expression,'+'))
 return
      (
      <concept code="{$focusCode}" json:array="true">{treplib:getSnomedConceptDesignations($focusCode)}</concept>,
   <additionalFocus/>,
      treplib:parseSnomedExpression(normalize-space(substring-after($expression,'+')))
   )
   else if (normalize-space(substring-before($expression,':')) castable as xs:integer) then
      let $conceptCode := normalize-space(substring-before($expression,':'))
      return
      (: check for attribute groups :)
      if (contains(substring-after($expression,':'),'{')) then
      let $attributeGroups := tokenize(substring-after($expression,':'),'\{')
         return
         (
         <concept code="{$conceptCode}" json:array="true">{treplib:getSnomedConceptDesignations($conceptCode)}</concept>,
      for $group in $attributeGroups
      let $groupRefinements:= tokenize(substring-before($group,'}'),',')
            return
            if (string-length($group) > 3) then
      if (contains($group,'}')) then
   <group json:array="true">
                   {
                   for $refinement in $groupRefinements
                   return
                   if (normalize-space(substring-before($refinement,'=')) castable as xs:integer) then
      let $refinementCode := normalize-space(substring-before($refinement,'='))
      return
                   <refinement json:array="true">
                   <concept code="{$refinementCode}">{treplib:getSnomedConceptDesignations($refinementCode)}</concept>
                      {
                      if (normalize-space(substring-after($refinement,'=')) castable as xs:integer) then
                         let $equalsCode :=normalize-space(substring-after($refinement,'='))
                         return
                         <equals code="{$equalsCode}">{treplib:getSnomedConceptDesignations($equalsCode)}</equals>
                      else
                      <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
                      }
                   </refinement>
                   else <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
                   }
                   </group>
                else <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
             else ()
          )
      else
         let $refinements := tokenize(normalize-space(substring-after($expression,':')),',')
      return
      (
      <concept code="{$conceptCode}" json:array="true">{treplib:getSnomedConceptDesignations($conceptCode)}</concept>,
         
         for $refinement in $refinements
         return
         if (normalize-space(substring-before($refinement,'=')) castable as xs:integer) then
         let $refinementCode := normalize-space(substring-before($refinement,'='))
         return
         <refinement json:array="true">
         <concept code="{$refinementCode}">{treplib:getSnomedConceptDesignations($refinementCode)}</concept>
            {
            if (normalize-space(substring-after($refinement,'=')) castable as xs:integer) then
               let $equalsCode :=normalize-space(substring-after($refinement,'='))
               return
               <equals code="{$equalsCode}">{treplib:getSnomedConceptDesignations($equalsCode)}</equals>
            else
            <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
            }
         </refinement>
         else <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
         )
   else if (normalize-space($expression) castable as xs:integer) then
      <concept code="{normalize-space($expression)}" json:array="true">{treplib:getSnomedConceptDesignations($expression)}</concept>
      else
         <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
};

declare function treplib:getSnomedConceptDesignations($code as xs:integer) as element()*{
   let $concept := collection(concat($setlib:strCodesystemStableData,'/external/snomed'))//concept[@code=$code]
   return
   if ($concept) then
       if ($concept/@statusCode='retired') then
         let $designations := $concept/designation
         return
         <message severity="warning" type="conceptRetired" json:array="true">
            <concept code="{$concept/@code}">
               {
            for $designation in $designations[@use='pref']
               return
               <designation json:array="true">
                  {
                  $designation/@*,
                  $designation/text()
                  }
               </designation>
               }
            </concept>
            {
            for $association in $concept/association
            return
            <association json:array="true">
               {
               $association/@*,
               $association/*
               }
            </association>
            }
         </message>
      else if ($concept/@statusCode='draft') then
         <message severity="warning" type="conceptDraft" json:array="true">Concept is in draft</message>
      else if ($concept/@statusCode='experimental') then
         <message severity="warning" type="conceptExperimental" json:array="true">Concept is experimental</message>
      else
  let $designations := $concept/designation
         return
         for $designation in $designations[@use='pref']
   order by $designation/@lang
         return
         <designation json:array="true">
            {
            $designation/@*,
            $designation/text()
            }
         </designation>
   else
      <message severity="error" type="conceptNotFound" json:array="true">Concept not found</message>
};
