xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilvs                 = "http://art-decor.org/ns/api/util-valueset";

import module namespace utilcs          = "http://art-decor.org/ns/api/util-codesystem" at "util-codesystem-lib.xqm";
import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";

import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

(:~ Return zero or more value sets as-is

@param $id           - required. Identifier of the value set to retrieve
@param $flexibility  - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@return Matching value sets
@since 2023-09-10
:)
declare function utilvs:getValueSetById($id as xs:string, $flexibility as xs:string?) as element(valueSet)* {

    if ($flexibility castable as xs:dateTime) then (
        let $decorSets      := $setlib:colDecorData//valueSet[@id = $id][@effectiveDate = $flexibility]
        let $cacheSets      := $setlib:colDecorCache//valueSet[@id = $id][@effectiveDate = $flexibility]
        return ($decorSets, $cacheSets)
    )
    else if ($flexibility) then (
        let $decorSets      := $setlib:colDecorData//valueSet[@id = $id]
        let $cacheSets      := $setlib:colDecorCache//valueSet[@id = $id]
        let $valuesets      := ($decorSets, $cacheSets)
        return $valuesets[@effectiveDate = string(max($valuesets/xs:dateTime(@effectiveDate)))]
    )
    else (
        let $decorSets      := $setlib:colDecorData//valueSet[@id = $id]
        let $cacheSets      := $setlib:colDecorCache//valueSet[@id = $id]
        return ($decorSets, $cacheSets)
    )
};

declare function utilvs:getValueSetByName($name as xs:string, $flexibility as xs:string?) as element(valueSet)* {

    if ($flexibility castable as xs:dateTime) then (
        let $decorSets      := $setlib:colDecorData//valueSet[@name = $name][@effectiveDate = $flexibility]
        let $cacheSets      := $setlib:colDecorCache//valueSet[@name = $name][@effectiveDate = $flexibility]
        return ($decorSets, $cacheSets)
    )
    else if ($flexibility) then (
        let $decorSets      := $setlib:colDecorData//valueSet[@name = $name]
        let $cacheSets      := $setlib:colDecorCache//valueSet[@name = $name]
        let $valuesets      := ($decorSets, $cacheSets)
        return $valuesets[@effectiveDate = string(max($valuesets/xs:dateTime(@effectiveDate)))]
    )
    else (
        let $decorSets      := $setlib:colDecorData//valueSet[@name = $name]
        let $cacheSets      := $setlib:colDecorCache//valueSet[@name = $name]
        return ($decorSets, $cacheSets)
    )
};

(:~ Return zero or more value sets as-is

@param $id            - required. Identifier of the value set to retrieve
@param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@param $decorOrPrefix - required. determines search scope. pfx- limits scope to this project only
@param $decorRelease  - optional. if empty defaults to current version. if valued then the value set will come explicitly from that archived project version which is expected to be a compiled version
@param $decorLanguage - optional. Language compilation. Defaults to project/@defaultLanguage.
@return Matching value sets
@since 2023-09-10
:)
declare function utilvs:getValueSetById($id as xs:string, $flexibility as xs:string?, $decorOrPrefix as item(), $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(valueSet)* {

    let $decor              := utillib:getDecor($decorOrPrefix, $decorRelease, $decorLanguage)
    let $decors             := if (empty($decorRelease)) then utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL) else $decor
    
    return
        if ($flexibility castable as xs:dateTime) then $decors//terminology/valueSet[@id = $id][@effectiveDate = $flexibility]
        else if ($flexibility) then (
            let $valuesets  := $decors//terminology/valueSet[@id = $id]
            return $valuesets[@effectiveDate = string(max($valuesets/xs:dateTime(@effectiveDate)))]
        )
        else $decors//terminology/valueSet[@id = $id]
};

declare function utilvs:getValueSetByName($name as xs:string, $flexibility as xs:string?, $decorOrPrefix as item(), $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(valueSet)* {
    
    let $decor              := utillib:getDecor($decorOrPrefix, $decorRelease, $decorLanguage)
    let $decors             := if (empty($decorRelease)) then utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL) else $decor
    
    return
        if ($flexibility castable as xs:dateTime) then $decors//valueSet[@name = $name][@effectiveDate = $flexibility]
        else if ($flexibility) then (
            let $valuesets := $decors//valueSet[@name = $name]
            return $valuesets[@effectiveDate = string(max($valuesets/xs:dateTime(@effectiveDate)))]
        )
        else $decors//valueSet[@name = $name]
};

declare function utilvs:getValueSetByRef($ref as xs:string, $flexibility as xs:string?, $decorOrPrefix as item(), $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(valueSet)* {

    if (utillib:isOid($ref)) 
        then utilvs:getValueSetById($ref, $flexibility, $decorOrPrefix, $decorRelease, $decorLanguage)
        else utilvs:getValueSetByName($ref, $flexibility, $decorOrPrefix, $decorRelease, $decorLanguage) 
};

declare function utilvs:getValueSetByRef($ref as xs:string, $flexibility as xs:string?) as element(valueSet)* {
    if (utillib:isOid($ref)) 
        then utilvs:getValueSetById($ref, $flexibility)
        else utilvs:getValueSetByName($ref, $flexibility)
};

(:~ Get contents of a valueSet and return like this:
 :   <valueSet id|ref="oid" ...>
 :       <completeCodeSystem .../>         -- if applicable
 :       <conceptList>
 :           <concept .../>
 :           <include ...>                 -- handled by utilvs:getValueSetInclude() 
 :               <valueSet ...>
 :                   ...
 :               </valueSet>
 :           </include>
 :           <exception .../>
 :       </conceptList>
 :   </valueSet>
 :)
declare function utilvs:getRawValueSet($valueSet as element(valueSet), $includetrail as element()*, $language as xs:string?, $getincludes as xs:boolean) as element(valueSet) {
    <valueSet>
    {
        $valueSet/@*,
        if ($valueSet[@url])   then () else attribute url {($valueSet/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]},
        if ($valueSet[@ident]) then () else attribute ident {($valueSet/ancestor::decor/project/@prefix)[1]},
        $valueSet/desc,
        $valueSet/publishingAuthority,
        if ($valueSet[@id]) then 
            if ($valueSet[publishingAuthority]) then () else (
                let $decor      := $valueSet/ancestor::decor
                let $copyright  := ($decor/project/copyright[empty(@type)] | $decor/project/copyright[@type = 'author'])[1]
                return
                if ($copyright) then
                    <publishingAuthority name="{$copyright/@by}" inherited="project">
                    {
                        (: Currently there is no @id, but if it ever would be there ...:)
                        $copyright/@id,
                        $copyright/addrLine
                    }
                    </publishingAuthority>
                else ()
            )
        else ()
        ,
        $valueSet/endorsingAuthority,
        $valueSet/purpose,
        if ($valueSet[@id]) then utilvs:handleValueSetCopyrights($valueSet/copyright, distinct-values($valueSet//@codeSystem)) else $valueSet/copyright,
        for $revisionHistory in $valueSet/revisionHistory
        return
            <revisionHistory>
            {
                $revisionHistory/@*,
                $revisionHistory/desc
            }
            </revisionHistory>
        ,
        if ($valueSet/completeCodeSystem | $valueSet/conceptList[*]) then (
            <conceptList>
            {
                for $codeSystem in $valueSet/completeCodeSystem
                return 
                    <include>
                    {
                        $codeSystem/@*, 
                        $codeSystem/filter
                    }
                    </include>
                ,
                for $node in $valueSet/conceptList/concept | 
                             $valueSet/conceptList/include | 
                             $valueSet/conceptList/exclude | 
                             $valueSet/conceptList/exception
                return (
                    if ($node[self::include[@ref]]) then if ($getincludes) then utilvs:getValueSetInclude($node, $includetrail, $language) else $node
                    else if ($node[self::include[@op]] | $node[self::exclude[@op]]) then $node
                    else (
                        element {$node/name()} 
                        {
                            $node/@*,
                            $node/designation,
                            $node/desc,
                            $node/filter
                        }
                    )
                )
            }
            </conceptList>
        ) else ()
    }
    </valueSet>
};

(:~ Extract a valueSet with @id by recursively resolving all includes. Use utilvs:getValueSetExtract to resolve a valueSet/@ref first. 
<valueSet all-attributes of the input>
  <desc all />
  <sourceCodeSystem id="any-concept-or-exception-codesystem-in-valueSet" identifierName="codeSystemName" canonicalUri="xs:anyURI" canonicalUriDSTU2="xs:anyURI" canonicalUriSTU3="xs:anyURI" canonicalUriR4="xs:anyURI" />
  <completeCodeSystem all-attributes of the input>
  <conceptList>
    <concept all-attributes-and-desc-elements />
    <exception all-attributes-and-desc-elements exceptions-contain-no-duplicates />
  </conceptList>
</valueSet>
                                
@param $valueSet     - required. The valueSet element with content to expand
@param $prefix       - required. The origin of valueSet. pfx- limits scope this project only
@param $version      - optional. if empty defaults to current version. if valued then the valueset will come explicitly from that archived project version which is expected to be a compiled version
@return The expanded valueSet
@since 2024-05-02
:)
declare function utilvs:getValueSetExtracted($valueSet as element(), $projectPrefix as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?) as element(valueSet) {
    
    let $decor                  := utillib:getDecorByPrefix($projectPrefix, $projectVersion, $projectLanguage)[1]
    let $language               := if (empty($projectLanguage)) then ($decor/@language, $decor/project/@defaultLanguage)[1] else $projectLanguage
    let $language               := if (empty($language)) then ($valueSet/../@defaultLanguage, utillib:getServerLanguage()) else $language
    let $rawValueSet            := utilvs:getRawValueSet($valueSet, <include ref="{$valueSet/(@id|@ref)}" flexibility="{$valueSet/@effectiveDate}"/>, $language, true())
    let $orgPrefix              := if ($valueSet/../@ident) then $valueSet/../@ident else $projectPrefix

    let $codesystemmap   :=
        map:merge(
            for $sourceCodeSystem in $rawValueSet/completeCodeSystem[empty(filter)] | $rawValueSet/conceptList/include[@codeSystem][empty(@op | @code | @ref | filter)]
            let $csid           := $sourceCodeSystem/@codeSystem
            let $csed           := if ($sourceCodeSystem[@flexibility castable as xs:dateTime]) then $sourceCodeSystem/@flexibility else 'dynamic'
            let $csided         := $csid || $csed
            group by $csided
            return (
                let $codeSystemContents := utilcs:getCodeSystemById($csid[1], $csed[1], $orgPrefix, $projectVersion)[@id][1]
                return if ($codeSystemContents) then map:entry($csided, $codeSystemContents) else ()
            )
        )
     
     let $completeCodeSystems   := $valueSet//completeCodeSystem | $valueSet/conceptList/include[@codeSystem][empty(@op | @code | @ref | filter | @exception[. = 'true'])]

     
     return
        <valueSet>
        {
            $rawValueSet/(@* except (@url|@ident|@referencedFrom)),
            $rawValueSet/desc
        }
        {
            (: when a value set is pulled from a compiled project version, the sourceCodeSystem should already be present :)
            if ($rawValueSet/sourceCodeSystem) then $rawValueSet/sourceCodeSystem else (
                utilvs:createSourceCodeSystems($rawValueSet, $completeCodeSystems, ($rawValueSet/ancestor::decor, $projectPrefix)[1], $language)
            )
        }
        {
            $rawValueSet/publishingAuthority,
            $rawValueSet/endorsingAuthority,
            $rawValueSet/revisionHistory,
            $rawValueSet/purpose,
            $rawValueSet/copyright
        }
        {
            if ($rawValueSet/conceptList[include | exclude | duplicate] or count(map:keys($codesystemmap)) gt 0) 
                then utilvs:getValueSetConceptList($rawValueSet, $completeCodeSystems, $codesystemmap, $language)             
                else $rawValueSet/conceptList
        }    
        </valueSet>
};

(:~ Return zero or more extracted valuesets wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element. This repository element
:   holds at least the attribute @ident with the originating project prefix and optionally the attribute @url with the repository URL in case of an external
:   repository. Id based references can match both valueSet/@id and valueSet/@ref. The latter is resolved. Note that duplicate valueSet matches may be 
:   returned. Example output:
:   &lt;return>
:       &lt;repository url="http://art-decor.org/decor/services/" ident="ad2bbr-">
:           &lt;valueSet id="2.16.840.1.113883.1.11.10282" name="ParticipationSignature" displayName="ParticipationSignature" effectiveDate="2013-03-11T00:00:00" statusCode="final" versionLabel="DEFN=UV=VO=1206-20130318">
:               ...
:           &lt;/valueSet>
:       &lt;/repository>
:   &lt;/return>
:   <ul><li> Valuesets get an extra attribute @fromRepository containing the originating project prefix/ident</li> 
:   <li>valueSet/@ref is treated as if it were the referenced valueset</li>
:   <li>If parameter prefix is not used then only valueSet/@id is matched. If that does not give results, then 
:     valueSet/@ref is matched, potentially expanding into a buildingBlockRepository</li>
:   <li>Any (nested) include @refs in the valueset are resolved, if applicable using the buildingBlockRepository 
:     configuration in the project that the valueSet/@id was found in</li>
:                                
:   @param $id           - required. Identifier of the valueset to retrieve
:   @param $flexibility  - optional. null gets all versions, 'dynamic' gets the newest version based on id, yyyy-mm-ddThh:mm:ss gets this specific version
:   @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
:   @since 2024-02-06
:)
declare function utilvs:getValueSetExtract($idOrName as xs:string, $effectiveDate as xs:string?) as element(valueSet)* {
    
    let $idOrName               := $idOrName[not(. = '')]
    let $effectiveDate          := ($effectiveDate[not(. = '')], 'dynamic')[1]
    
    let $valueSets              := utilvs:getValueSetByRef($idOrName, $effectiveDate)
    
    return
        <return>
        {
            for $vs in $valueSets
            let $prefix         := $vs/ancestor::decor/project/@prefix
            let $urlident       := $vs/ancestor::cacheme/@bbrurl || $prefix
            group by $urlident
            return
                <project ident="{$prefix}">
                {
                    if ($vs[ancestor::cacheme]) then
                        attribute url {$vs[1]/ancestor::cacheme/@bbrurl}
                    else (),
                    $vs[1]/ancestor::decor/project/@defaultLanguage,
                    for $valueSet in $vs[@id]
                    return
                        utilvs:getValueSetExtracted($valueSet, $prefix, (), ())
                }
                </project>
        }
        </return>

};

(:~ Return zero or more expanded valuesets wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element. This repository element
:   holds at least the attribute @ident with the originating project prefix and optionally the attribute @url with the repository URL in case of an external
:   repository. Id based references can match both valueSet/@id and valueSet/@ref. The latter is resolved. Note that duplicate valueSet matches may be 
:   returned. Example output:
:   &lt;return>
:       &lt;repository url="http://art-decor.org/decor/services/" ident="ad2bbr-">
:           &lt;valueSet id="2.16.840.1.113883.1.11.10282" name="ParticipationSignature" displayName="ParticipationSignature" effectiveDate="2013-03-11T00:00:00" statusCode="final" versionLabel="DEFN=UV=VO=1206-20130318">
:               ...
:           &lt;/valueSet>
:       &lt;/repository>
:   &lt;/return>
:   <ul><li> Valuesets get an extra attribute @fromRepository containing the originating project prefix/ident</li> 
:   <li>valueSet/@ref is treated as if it were the referenced valueset</li>
:   <li>If parameter prefix is not used then only valueSet/@id is matched. If that does not give results, then 
:     valueSet/@ref is matched, potentially expanding into a buildingBlockRepository</li>
:   <li>Any (nested) include @refs in the valueset are resolved, if applicable using the buildingBlockRepository 
:     configuration in the project that the valueSet/@id was found in</li>
:                                
:   @param $id           - required. Identifier of the valueset to retrieve
:   @param $flexibility  - optional. null gets all versions, 'dynamic' gets the newest version based on id, yyyy-mm-ddThh:mm:ss gets this specific version
:   @param $prefix       - required. determines search scope. pfx- limits scope to this project only
:   @param $version      - optional. if empty defaults to current version. if valued then the valueset will come explicitly from that archived project version which is expected to be a compiled version
:   @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
:   @since 2024-02-06
:)
declare function utilvs:getValueSetExtract($projectPrefix as xs:string*, $projectVersion as xs:string*, $projectLanguage as xs:string?, $idOrName as xs:string, $effectiveDate as xs:string?, $withversions as xs:boolean?) as element(valueSet)* {
    let $idOrName               := $idOrName[not(. = '')]
    let $flexibility            := if ($withversions) then () else ($effectiveDate[not(. = '')], 'dynamic')[1]
    let $projectPrefix          := $projectPrefix[not(. = '')][1]
    let $projectVersion         := $projectVersion[not(. = '')][1]
    let $projectLanguage        := $projectLanguage[not(. = '')][1]

    (: retrieve all valuesets for the input $decor project(s) 
        - when compiled version  - as is - 
        - when live version - getting all (cached) decor projects in scope  
    :)
    let $decor                  := utillib:getDecor($projectPrefix, $projectVersion, $projectLanguage)
    let $decors                 := if (empty($projectVersion)) then utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL) else $decor[1]
    
    let $valueSets              := 
        if (utillib:isOid($idOrName)) then (
        
            if ($flexibility castable as xs:dateTime) then $decors//terminology/valueSet[@id = $idOrName][@effectiveDate = $flexibility]
            else if ($flexibility) then (
                let $valuesets  := $decors//terminology/valueSet[@id = $idOrName]
                return $valuesets[@effectiveDate = string(max($valuesets/xs:dateTime(@effectiveDate)))]
            )
            else $decors//terminology/valueSet[@id = $idOrName]
        )
        else (
            if ($flexibility castable as xs:dateTime) then $decors//valueSet[@name = $idOrName][@effectiveDate = $flexibility]
            else if ($flexibility) then (
                let $valuesets  := $decors//valueSet[@name = $idOrName]
                return $valuesets[@effectiveDate = string(max($valuesets/xs:dateTime(@effectiveDate)))]
            )
            else $decors//valueSet[@name = $idOrName]
       )    
    
    let $valueSets              :=  $valueSets | $decor//terminology/valueSet[@ref = $idOrName]
    
    return
    <return>
    {
        (:from the requested project, return valueSet/(@id and @ref):)
        (: when retrieving value sets from a compiled project, the @url/@ident they came from are on the valueSet element
           reinstate that info so downstream logic works as if it really came from the repository again.
        :)
        for $repository in $decor
        for $vs in $valueSets
        let $url                := if (empty($vs/@url)) then ($vs/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1] else $vs/@url
        let $ident              := if (empty($vs/@ident)) then $vs/ancestor::decor/project/@prefix else $vs/@ident
        let $repo               := if ($ident=$projectPrefix) then 'project' else 'repository'
        let $refFrom            := if (empty($vs/@referencedFrom)) then ($decors//project[buildingBlockRepository/@ident = $ident][buildingBlockRepository/@url = $url]/@prefix)[1] else $vs/@referencedFrom
        let $projectType        := if (exists($vs/ancestor::cacheme)) then 'cached' else 'local'
        let $source             := $url || $ident
        group by $source
        order by $repo[1]
        return
            element {$repo[1]} {
                
                if ($ident=$projectPrefix) then attribute ident{$ident[1]}
                else
                ( 
                    attribute url {$url[1]},
                    attribute ident{$ident[1]},
                    attribute referencedFrom {$refFrom[1]},
                    attribute projecttype {$projectType[1]}
                ),
                for $valueSet in $vs
                    order by $valueSet/@effectiveDate descending
                    return utilvs:getValueSetExtracted($valueSet, $projectPrefix, $projectVersion, $projectLanguage)
                }
    }
    </return>
};

declare %private function utilvs:getValueSetConceptList($valueSet as element(valueSet), $completeCodeSystems as element()*, $codesystemmap as map(*)?, $language as xs:string) as element(conceptList) {

    let $conceptList           :=
    <conceptList>
        {
            (:get defined and included codeSystems:)
            for $sourceCodeSystem in $completeCodeSystems
                let $csid                       := $sourceCodeSystem/@codeSystem
                let $csed                       := if ($sourceCodeSystem[@flexibility castable as xs:dateTime]) then $sourceCodeSystem/@flexibility else 'dynamic'
                let $csided                     := $csid || $csed
                let $codeSystemName             := $sourceCodeSystem/@codeSystemName
                return
                    if (map:contains($codesystemmap, $csided)) then (
                        let $conceptList        := map:get($codesystemmap, $csided)//conceptList
                        let $checkParentChild   := exists($conceptList/codedConcept[parent | child])
                        let $topLevelConcepts   := if ($checkParentChild) then $conceptList/codedConcept[empty(parent)] else $conceptList/codedConcept[@level = '0']
                        
                        return utilcs:getCodedConcepts($topLevelConcepts, $csid, 0, $conceptList, $topLevelConcepts/@code, $checkParentChild, $language)
                    ) else (
                        <include codeSystem="{$csid}">
                        {
                            if ($codeSystemName = '') then () else attribute codeSystemName {$codeSystemName}, 
                            $sourceCodeSystem/@codeSystemVersion, 
                            $sourceCodeSystem/@flexibility,
                            (: 2020-12 Added for deprecation of a codeSystem in the context of a valueSet :)
                            $sourceCodeSystem/@type[. = 'D'],
                            for $f in $sourceCodeSystem/filter
                            return <filter>{$f/@property, $f/@op, $f/@value}</filter>
                        }
                        </include>
                    )
        }
        {
            for $source in $valueSet/conceptList/include[@ref][valueSet]
            return comment { ' include ref=' || $source/@ref || ' flexibility=' || ($source/@flexibility[not(. = '')], 'dynamic')[1] || ' displayName=' || replace(($source/valueSet/@displayName)[1], '--', '-\\-') || ' effectiveDate=' || ($source/valueSet/@effectiveDate)[1] || ' '}
        }
        {
            for $source in $valueSet//conceptList/concept[not(ancestor::include[@exception = 'true'])] | 
                           $valueSet//conceptList/include[not(.[@ref][valueSet] | .[@exception = 'true'])] except $completeCodeSystems |
                           $valueSet//conceptList/exclude
            let $csid           := $source/@codeSystem
            let $csed           := if ($source[@flexibility castable as xs:dateTime]) then $source/@flexibility else 'dynamic'
            let $csided         := $csid || $csed
            return
                if ($source[self::concept | self::exclude]) then $source 
                else if (map:contains($codesystemmap, $csided)) then () else $source
        }
        {
            for $node in $valueSet//conceptList/exception |
                         $valueSet//conceptList/concept[ancestor::include[@exception='true']]
            let $code           := $node/@code
            let $codeSystem     := $node/@codeSystem
            group by $code, $codeSystem
            return
                (:<exception>{$exceptions/@*, $exceptions/*}</exception>:)
                <exception>{$node[1]/@*, $node[1]/*}</exception>
        }
        {
            $valueSet//conceptList/duplicate
        }
        </conceptList>
    
    return $conceptList
};

(: Returns appropriate copyright texts based on $usedCodeSystems
   <copyright language="en-US">Licensing note: .......</copyright>
:)
declare %private function utilvs:handleValueSetCopyrights($copyrightsource as element(copyright)*, $usedCodeSystems as item()*) as element(copyright)* {

    let $extracopyrights        :=
        <extracopyright oid="2.16.840.1.113883.6.96" urn="http://snomed.info/sct"> This artefact includes content from SNOMED Clinical Terms® (SNOMED CT®) which is copyright of the International Health Terminology Standards Development Organisation (IHTSDO). Implementers of these artefacts must have the appropriate SNOMED CT Affiliate license - for more information contact http://www.snomed.org/snomed-ct/getsnomed-ct or info@snomed.org.
        </extracopyright>
        
    let $extra                  :=
        for $t in $extracopyrights
        return
            (: if codesystem is used :) 
            if ($t/@oid = $usedCodeSystems) then (
                (: if text is not yet in the copyright texts :)
                if ($copyrightsource[@language = 'en-US'][empty(@inherited)]) then 
                    <copyright language="en-US">{$copyrightsource[@language='en-US']/(node() except *:div[@data-source='inherited'])}<div data-source="inherited" style="border-top: 1px solid black">{$t/text()}</div></copyright>
                else 
                    <copyright language="en-US" inherited="true">{$t/text()}</copyright>
            )
            else ()
    
    return if (empty($extra)) then $copyrightsource else ($copyrightsource[not(@language='en-US')], $extra)
    
};

declare %private function utilvs:getValueSetInclude($include as element(), $includetrail as element()*, $language as xs:string?) as element()* {
    let $valuesetId         := $include/@ref
    let $valuesetFlex       := $include/@flexibility
    let $decor              := $include/ancestor::decor
    let $valueSet           := utilvs:getValueSetById($valuesetId, $valuesetFlex, $decor, (), ())
    
    return
        if ($includetrail[@ref = $include/@ref][@flexibility = $valueSet/@effectiveDate]) then <duplicate>{$include/@*}</duplicate>
        else (
            let $includetrail := $includetrail | <include ref="{$include/@ref}" flexibility="{$valueSet/@effectiveDate}"/>
            return
            <include>
            {
                $include/@ref,
                $include/@flexibility,
                $include/@exception
            }
            {
                if (exists($valueSet)) then utilvs:getRawValueSet($valueSet[1], $includetrail, $language, true()) else ()
            }
            </include>
        )
};

(:<sourceCodeSystem id="any-concept-or-exception-codesystem-in-valueSet" identifierName="codeSystemName" canonicalUri="xs:anyURI" canonicalUriDSTU2="xs:anyURI" canonicalUriSTU3="xs:anyURI" canonicalUriR4="xs:anyURI" />:)
declare %private function utilvs:createSourceCodeSystems($valueSet as element(valueSet), $completeCodeSystems as element()*, $decorOrPrefix as item(), $language as xs:string?) as element(sourceCodeSystem)* {
    for $sourceCodeSystem in $valueSet//@codeSystem
    let $csid               := $sourceCodeSystem
    let $csed               := $sourceCodeSystem/../@codeSystemVersion
    let $identifierName     := ($completeCodeSystems[@codeSystemName][@codeSystem = $csid]/@codeSystemName, utillib:getNameForOID($csid, $language, $decorOrPrefix))[1]
    group by $csid, $csed
    order by $csid[1], $csed[1]
    return
        <sourceCodeSystem>
        {
            attribute id {$csid},
            attribute identifierName {replace(normalize-space(($identifierName)[1]),'\s','&#160;')},
            attribute canonicalUri {utillib:getCanonicalUriForOID('CodeSystem', $csid, $csed, $decorOrPrefix, ())},
            for $fhirVersion in $setlib:arrKeyFHIRVersions
                return attribute {concat('canonicalUri', $fhirVersion)} {utillib:getCanonicalUriForOID('CodeSystem', $csid, $csed, $decorOrPrefix, $fhirVersion)},
            (: ART-DECOR 3: get for example external/hl7 :)
            let $coll       := collection($setlib:strCodesystemStableData)//identifier[@id = 'urn:oid:' || $csid]
            return if (empty($coll)) then () else attribute context {substring-after(util:collection-name($coll[1]), $setlib:strCodesystemStableData || '/')}
        }
        </sourceCodeSystem>
};

