xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Template API allows read, create, update of DECOR templates :)
module namespace tmgetfe            = "http://art-decor.org/ns/api/template-get-foredit";

import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace utillib     = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace utiltemp    = "http://art-decor.org/ns/api/util-template" at "library/util-template-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

declare %private variable $tmgetfe:STATUSCODES-FINAL          := ('active', 'cancelled', 'pending', 'review', 'rejected', 'retired');

(:~ Retrieves a template in edit mode
@param $id                      - required parameter denoting the id of the template
@param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
@return template
@since 2020-05-03
:)
declare function tmgetfe:getTemplateForEdit($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $breakLock                      := $request?parameters?breakLock = true()
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := tokenize($acceptedType, '/')[2]
           
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $template                       := $setlib:colDecorData//template[@id = $id][@effectiveDate = $effectiveDate]
    
    let $decor                          := $template/ancestor::decor
    let $projectPrefix                  := $decor/project/@prefix
    
    let $check                          :=
        if ($template) then () else (
            error($errors:BAD_REQUEST, 'Template with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' not found in the context of project ' || $projectPrefix)
        )
    let $check                          :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-RULES)) then () else (
            error($errors:FORBIDDEN, 'User '|| $authmap?name || ' does not have sufficient permissions to modify templates in project ' || $projectPrefix || '. You have to be an active author in the project.')
        )
    
    let $check                          :=
        if ($template[@statusCode = $tmgetfe:STATUSCODES-FINAL]) then (
            error($errors:BAD_REQUEST, 'Template cannot be updated while it has one of status: ' || string-join($tmgetfe:STATUSCODES-FINAL, ', ') || (if ($template/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
    
    let $lock                           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
    let $lock                           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breakLock)
    
    let $check                          :=
        if ($lock) then () else (
            error($errors:FORBIDDEN, 'User ' || $authmap?name || ' does not have a lock for this template (anymore). Get a lock first.')
        )

    let $baseId                         := decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TEMPLATE)[1]/@id    
    let $results                        := utiltemp:getTemplateForEdit($template, $decor, $lock, false(), true(), $baseId, 'edit')
    
    for $result in $results
    return
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*, $format)
        }    
};


