xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace ruleslib               = "http://art-decor.org/ns/api/rules";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace utilvs          = "http://art-decor.org/ns/api/util-valueset" at "util-valueset-lib.xqm";
import module namespace utilcs          = "http://art-decor.org/ns/api/util-codesystem" at "util-codesystem-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

declare %private variable $ruleslib:ADDRLINE-TYPE   := utillib:getDecorTypes()/AddressLineType;

(: first we decide how to start te error message :)
declare %private function ruleslib:getErrorString($elmnt as element(), $operation as xs:string, $path as xs:string?) as xs:string {

    if ($path)
        then 'Parameter ' || $operation || ' not allowed for ''' || $path || ''':' 
        else 'Cannot '|| $operation || ' '''|| name($elmnt) ||''':'};

(: ==================================== GENERAL RULES ================================================:)

(: rule - every create or update of a DECOR Artefact has to be schema compiant :)
declare function ruleslib:checkDecorSchema($preparedData as item(), $operation as xs:string) {

    let $artefact           := name($preparedData)
    let $error              := 'Cannot '|| $operation || ' ' || $artefact || ': the ' || $artefact || ' is not schema compliant against ' || substring-after(util:collection-name($setlib:docDecorSchema), 'db/apps') || '/' || util:document-name($setlib:docDecorSchema) || ' on this server.'
    
    let $check              := validation:jaxv-report($preparedData, $setlib:docDecorSchema)
    
    return
        if ($check/status='invalid') then
            error($errors:BAD_REQUEST, $error || ' === SEE REPORT === ' || serialize($check))
        else ()
};

(: rule - every create or update or patch of a publishing authority has to be checked on
        * cannot be marked inherited for update 
        * id has to be an OID
        * name cannot be empty
        * address line types has to be supported
:)
declare function ruleslib:checkPublishingAuthority($publishingAuthority as element(publishingAuthority), $operation as xs:string, $path as xs:string?) as xs:string* {
    
    let $check              := () 
    
    (: cannot be marked inherited for update :)
    let $error              := $check || ' Contents SHALL NOT be marked ''inherited''.' 
    let $check              := if ($publishingAuthority[@inherited]) then $error else $check    
    
    (: id has to be an OID :)
    let $error              := $check || ' PublishingAuthority.id SHALL be an OID, found ' || string-join($publishingAuthority/@id, ', ')
    let $check              := if ($publishingAuthority[@id]) then if ($publishingAuthority[utillib:isOid(@id)]) then $check else $error else $check
    
    (: name cannot be empty :)
    let $error              := $check || ' PublishingAuthority.name SHALL be a non empty string.'
    let $check              := if ($publishingAuthority[@name[not(. = '')]]) then $check else $error

    (: address line types has to be supported :)
    let $unsupportedtypes   := $publishingAuthority/addrLine/@type[not(. = $ruleslib:ADDRLINE-TYPE/enumeration/@value)]
    let $error              := $check || ' Publishing authority has unsupported addrLine.type value(s) ' || string-join($unsupportedtypes, ', ') || '. SHALL be one of: ' || string-join($ruleslib:ADDRLINE-TYPE/enumeration/@value, ', ')
    let $check              := if ($unsupportedtypes) then $error else $check
    
    return if ($check) then ruleslib:getErrorString($publishingAuthority, $operation, $path) || $check  else ()  
};

(: rule - every create or update or patch of a FreeFormMarkupWithLanguage type element has to be checked on
        * cannot be marked inherited for update 
        * element must have contents
        * language is has regEx [a-z]{2}-[A-Z]{2}
:)
declare function ruleslib:checkFreeFormMarkupWithLanguage($elmnt as element()*, $operation as xs:string, $path as xs:string?) as xs:string* {

    let $elmname            := name($elmnt)
    let $check              := ()
     
    (: cannot be marked inherited for update :)
    let $error              := $check || ' Contents SHALL NOT be marked ''inherited''.' 
    let $check              := if ($elmnt[@inherited]) then $error else $check    
    
    (: element has contents :)
    let $error              := $check || ' A ''' || $elmname || ''' SHALL have contents.'
    let $check              := if ($operation = 'remove' or $elmnt[.//text()[not(normalize-space() = '')]]) then $check else $error
    
    (: language is has regEx [a-z]{2}-[A-Z]{2} :)
    let $error              := $check || ' A ''' || $elmname || ''' SHALL have a language with pattern ll-CC.'
    let $check              := if ($elmnt[matches(@language, '[a-z]{2}-[A-Z]{2}')]) then $check else $error
    
    return if ($check) then ruleslib:getErrorString($elmnt, $operation, $path) || $check  else ()  
};

(: rule - every create or update or patch of a ValueCodingType type element shall be code without whitespace, a codeSystem as valid OID, and optionally a canonicalUri as URI :)
declare function ruleslib:checkValueCodingType($elmnt as element(), $operation as xs:string, $path as xs:string?) as xs:string* {

    let $error              := ' Input SHALL have code without whitespace, a codeSystem as valid OID, and optionally a canonicalUri as URI. Found: ' || string-join(for $att in $elmnt/@* return name($att) || ': "' || $att || '" ', ' ')
    
    return 
        if ($elmnt[@code[matches(., '^\S+$')]][utillib:isOid(@codeSystem)][empty(@canonicalUri) or @canonicalUri castable as xs:anyURI]) then () else ruleslib:getErrorString($elmnt, $operation, $path) || $error
    
};

(: ==================================== CONCEPTMAP RULES ============================================ :)

(: rule - check on conceptMap sourceScope or targetScope
        * source and target valueset can not have the same value
        * valueSet @ref should be an OID
        * valueSet is found in decor project
        * canonicalUri is empty or the same as valueSet canoncialUri 
:)
(: expect: element(sourceScope) or element(targetScope) :)
declare function ruleslib:checkConceptMapScope($elmnt as element(), $sourceOrTargetScope as element(), $decor as element(decor), $operation as xs:string, $path as xs:string?) as xs:string* {

    let $elmname            := name($elmnt)
    let $check              := ()

    (: source and target valueset can not have the same value :)
    let $error              := $check || ' SOURCESCOPE.ref and TARGETSCOPE.ref have the same value. Found: for both valueSets ''' || $elmnt/@ref || '''.' 
    let $check              := if ($elmnt/@ref eq $sourceOrTargetScope/@ref) then $error else $check

    let $check              :=
    
        (: valueSet @ref should be an OID :)
        if (not(utillib:isOid($elmnt/@ref))) then ' Input SHALL have ref as an OID on ' || $elmname || ' under value. Found: ''' || $elmnt/@ref || '''.'
        else (
            let $valueSet   := utilvs:getValueSetById($elmnt/@ref, $elmnt/@flexibility, $decor, (), ())[1]
            return
                (: valueSet is found in decor project :)
                if (empty($valueSet)) then  $check || ' ' || upper-case($elmname) || '.ref ''' || $elmnt/@ref || ''' does not resolve to a value set.' else $check
            )
            
    return if ($check) then ruleslib:getErrorString($elmnt, $operation, $path) || $check else ()            
};

(: rule - check on conceptMap group
        * target and element are both there or both not
        * source and target codesystem can not have the same value
        * for source and target codesystem: @ref should be an OID
        * for source and target codesystem: codeSystem is found as a sourceCodeSystem in scope valueset
        * for source and target codesystem: canonicalUri is empty or the same as codesystem canoncialUri 
:)
declare function ruleslib:checkConceptMapGroup($conceptMap as element(conceptMap), $decor as element(decor), $operation as xs:string, $path as xs:string?) as xs:string* {
    
    (: first check the DECOR.xsd and stop if not compliant :)
    let $check              := ruleslib:checkDecorSchema($conceptMap, $operation)

    let $check              :=
    for $group in $conceptMap/group 
        let $check          := ()
        let $elmname        := name($group)
        
        (: target and element are both there or both not :)
        let $error          := $check || ' Input SHALL have both ''target'' and ''element'' OR ''target'' and ''element'' are both empty.'  
        let $check          := if (($group/target and $group/element) or (empty($group/target) and empty($group/element))) then $check else $error
        
        (: source and target codesystem can not have the same value :)
        let $error          := $check || ' SOURCE.codeSystem and TARGET.codeSystem cannot have the same value. Found: for both codeSystems ''' || $group/source/@codeSystem || '''.' 
        let $check          := if ($group/source/@codeSystem eq $group/target/@codeSystem) then $error else $check
        
        (: check source and target codesystem :)
        let $check          := $check || ruleslib:checkConceptMapGroupCodeSystem($group/source, $conceptMap/sourceScope, $decor, $operation)
        let $check          := if ($group/target) then $check || ruleslib:checkConceptMapGroupCodeSystem($group/target, $conceptMap/targetScope, $decor, $operation) else ()
        
        return if($check) then ruleslib:getErrorString($group, $operation, $path) || ' following errors occur on codeSystem ''' || $group/source/@codeSystem || ''':' || $check else ()
        
    return if (empty($check)) then () else string-join($check)
};

declare %private function ruleslib:checkConceptMapGroupCodeSystem($sourceOrTarget as element(), $scope as element(), $decor as element(), $operation as xs:string) as xs:string* {

    let $csid               := $sourceOrTarget/@codeSystem
    let $csed               := $sourceOrTarget/@codeSystemVersion
    let $check              :=
        if (utillib:isOid($csid)) then 
        (   
            let $codeSystem := utilvs:getValueSet($decor/@prefix, (), (), $scope/@ref, $scope/@flexibility, false(), true())/sourceCodeSystem[@id = $csid]
            return    
                (: codeSystem is found as a sourceCodeSystem in scope valueset :)
                if (empty($codeSystem)) then ' ' || upper-case(name($sourceOrTarget)) || '.codeSystem ''' || $sourceOrTarget/@codeSystem || ''' does not resolve to a sourceCcodeSystem of valueSet ''' || $scope/@ref || '''.' else ()
        )
        
        else ' Input SHALL have ' || name($sourceOrTarget) || '.codeSystem as an OID on group under value. Found: ''' || $csid || '''.'
    
    return $check

};

(: the complete check of a conceptMap before creating or updating :)
declare function ruleslib:checkConceptMap($conceptMap as element(conceptMap), $decor as element(decor), $operation as xs:string) as xs:string* {

    (: first check the DECOR.xsd and stop if not compliant :)
    let $check              := ruleslib:checkDecorSchema($conceptMap, $operation)
    
    return
    for $node in $conceptMap/*
        return
            switch (name($node))
            case 'desc' 
            case 'purpose'
            case 'copyright' return ruleslib:checkFreeFormMarkupWithLanguage($node, $operation, ())
            case 'publishingAuthority' return ruleslib:checkPublishingAuthority($node, $operation, ())
            case 'jurisdiction' return ruleslib:checkValueCodingType($node, $operation, ())
            case 'sourceScope' return ruleslib:checkConceptMapScope($node, $conceptMap/targetScope, $decor, $operation, ())
            case 'targetScope' return ruleslib:checkConceptMapScope($node, $conceptMap/sourceScope, $decor, $operation, ())
            case 'group' return ruleslib:checkConceptMapGroup($conceptMap, $decor, $operation, ())
            default return () 

};

(: ==================================== VALUESET RULES ============================================ :)

(: check is before preparing items to conceptlist - for now only on put call :)
declare function ruleslib:checkValueSetItems($id as xs:string, $valueSet as element(valueSet), $operation as xs:string) {

    let $check                  := 
        if ($valueSet//exclude[@codeSystem][empty(@code)] | $valueSet//items[@is = 'exclude'][@codeSystem][empty(@code)]) then (
            error($errors:BAD_REQUEST, 'Value set with exclude on complete code system not supported. Exclude SHALL have both @code and @codeSystem')
        ) else ()
    
    let $css                    := $valueSet//(include | exclude |items[@is=('include', 'exclude')])[@codeSystem][empty(@code)]/@codeSystem
    let $check                  := 
        if (count($css) = count(distinct-values($css))) then () else (
             error($errors:BAD_REQUEST, 'Value set SHALL contain max 1 include per complete code system.')
        )
    
    let $css                    := $valueSet//(include | exclude |items[@is=('include', 'exclude')])/@ref
    let $check                  := 
        if (count($css) = count(distinct-values($css))) then () else (
            error($errors:BAD_REQUEST, 'Value set SHALL contain max 1 include/exclude per value set regardless of its version.')
        )
    let $check                  := 
        if ($css = $id) then 
            error($errors:BAD_REQUEST, 'Value set SHALL NOT contain itself as inclusion.')
        else ()
    
    return ()

};

(: ==================================== TEMPLATE RULES ============================================ :)


