xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
module namespace treplib            = "http://art-decor.org/ns/api/terminology/reportlib";

import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";

declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";

declare variable $treplib:docHtmlNames                   := treplib:getHtmlNames();
declare variable $treplib:strAD3ServerURL                := $utillib:strAD3ServerURL || '#/';
declare %private variable $treplib:supportedLanguages   := ('en-US','en-GB','nl-NL','de-DE','fr-FR','it-IT','pl-PL');

(:
   Get a representation of the decor file with minimal info based on the active filter
   If filter is on
      - Get transactions in filter
      - Walk trough source datasets and select concepts
      - Walk trough concepts and get associated valuesets
      
:)
declare function treplib:getFilteredDecor($decor as element()) as element()* {
   let $filters := collection(util:collection-name($decor))/filters[util:document-name(.) = 'filters.xml']
   let $filterActive := empty($filters[@filter = 'off']) and $filters[*] and (empty($filters/@filterId) or $filters/@filterId = $filters/filters/@id)
   let $filterId := $filters/@filterId/string()
   let $filterLabel := $filters/filters[@id=$filters/@filterId]/@label
   let $filteredTransactions := 
      if ($filterActive) then
         for $transaction in $filters/filters[@id=$filters/@filterId]/transaction
         let $trans := $decor//transaction[@id=$transaction/@ref][@effectiveDate=$transaction/@flexibility]
         return
         <scenario>
         {
         $trans
         }
         </scenario>
      else
         let $trans := $decor//transaction
         return
         <scenario>
         {
         $trans
         }
         </scenario>
   (: get list of unique dataset ids :)
   let $filteredDatasetIds      :=
         for $representingTemplate in $filteredTransactions//representingTemplate[@sourceDataset]
            let $sourceDataset := 
                              if ($decor//dataset[@id=$representingTemplate/@sourceDataset][@sourceDatasetFlexibility] castable as xs:dateTime) then
                                 $decor//dataset[@id=$representingTemplate/@sourceDataset][@effectiveDate=$representingTemplate/@sourceDatasetFlexibility]
                              else
                                let $dSet := $decor//dataset[@id=$representingTemplate/@sourceDataset]
                                let $latestDate := max($dSet/xs:dateTime(@effectiveDate))
                                return
                                $dSet[@effectiveDate=$latestDate]
            return
            concat($sourceDataset/@id,'--',$sourceDataset/@effectiveDate)
               
               let $filteredDatasets      :=
         if ($filterActive) then
            for $datasetId in distinct-values($filteredDatasetIds)
               let $dset := $decor/datasets/dataset[@id=substring-before($datasetId,'--')][@effectiveDate=substring-after($datasetId,'--')]
               return
               <dataset>
               {
                $dset/@*,
               $dset/name
                }
               {
               for $concept in $dset/concept[@statusCode=('draft','final','pending')][not((inherit,contains))]
               let $refConcept := $filteredTransactions//concept[@ref=$concept/@id]
               return
               if ($refConcept) then
               treplib:copyConcept($concept, $filterActive, $filteredTransactions)
               else ()
               }
            </dataset>
         else
         for $dataset in $decor/datasets/dataset[@statusCode=('draft','final','pending')]
         return
         <dataset>
               {
               $dataset/@*,
               $dataset/name,
               for $concept in $dataset/concept[@statusCode=('draft','final','pending')][not((inherit,contains))]
               return
               treplib:copyConcept($concept, $filterActive, ())
               }
         </dataset>
         
   let $filteredValuesetIds :=
         for $conceptList in $filteredDatasets//conceptList
            let $conceptListAssociations:= $decor/terminology/terminologyAssociation[@conceptId=$conceptList/@id]
            for $association in $conceptListAssociations
               let $valueSet := 
                  if ($association/@flexibility='dynamic') then
                     let $vSet := $decor/terminology/valueSet[@id=$association/@valueSet]
                     let $latestDate := max($vSet/xs:dateTime(@effectiveDate))
                     return
                     $vSet[@effectiveDate=$latestDate]
                  else if ($association/@flexibility castable as xs:dateTime) then
                     $decor/terminology/valueSet[@id=$association/@valueSet][@effectiveDate=$association/@flexibility]
                  else ()
               return
               concat($valueSet/@id,'--',$valueSet/@effectiveDate)
   
   let $filteredValuesets :=
      for $id in distinct-values($filteredValuesetIds)
      let $valueset := $decor/terminology/valueSet[@id=substring-before($id,'--')][@effectiveDate=substring-after($id,'--')]
      return
      $valueset
         
   return
      <filteredDecor>
      {
      if ($filterActive) then
      <filter  filterId="{$filterId}" filterLabel="{$filterLabel}"/>
      else ()
      }
      <project>
      {$decor/project/@*}
      </project>
      <datasets>
         {
         $filteredDatasets
         }
      </datasets>
      <scenarios>
         {
         $filteredTransactions
         }
      </scenarios>
      <ids>
      {
         $decor/ids/id
      }
      </ids>
      <terminology>
      {
         $decor/terminology/terminologyAssociation,
         $decor/terminology/codeSystem,
         $filteredValuesets
      }
      </terminology>
      </filteredDecor>
};

declare function treplib:copyConcept($concept as element(), $filterActive as xs:boolean, $filteredTransactions as element()*) as element()*{
   <concept>
      {
      $concept/@*,
      $concept/*[not(name()='concept')],
      if ($filterActive) then
         for $child in  $concept/concept[@statusCode=('draft','final','pending')][not((inherit,contains))]
         let $refConcept := $filteredTransactions//concept[@ref=$concept/@id]
         return
         if ($refConcept) then
            treplib:copyConcept($child, $filterActive, $filteredTransactions)
      else ()
      else
         for $child in  $concept/concept[@statusCode=('draft','final','pending')][not((inherit,contains))]
         return
         treplib:copyConcept($child, $filterActive, $filteredTransactions)
      }
   </concept>
};

(: Traverse datasets :)
declare function treplib:traverseDatasets($decor as element()) as element()* {
   for $dataset in $decor/datasets/dataset
   let $concepts := $dataset//concept[not(parent::conceptList)]
   return
   <dataset conceptCount="{count($concepts)}" json:array="true">
   {
      $dataset/@*,
      $dataset/name,
      $dataset/transaction,
      for $concept in $concepts
      let $terminologyAssociations := $decor/terminology/terminologyAssociation[@conceptId=$concept/@id]
      let $checkedConcept :=
         <concept json:array="true" level="{count($concept/ancestor::concept)}">
   {
      $concept/@*,
      $concept/name,
      if ($terminologyAssociations) then
         for $association in $terminologyAssociations
         return
         treplib:handleTerminologyAssociation($decor,$association)
      else
      <message severity="info" type="noAssociation" json:array="true">No terminology association defined</message>
       ,
       if ($concept/valueDomain/conceptList) then
         let $conceptListAssociation:= $decor/terminology/terminologyAssociation[@conceptId=$concept/valueDomain/conceptList/@id]
         return
         <conceptList>
            {
            (
            if ($conceptListAssociation) then
               for $association in $conceptListAssociation
               return
               treplib:handleConceptListAssociation($decor,$association)
            else
            <message severity="warning" type="noValuesetAssociated" json:array="true">No valueset associated with conceptlist</message>
            ,
            for $listConcept in $concept/valueDomain/conceptList/concept
               let $listConceptAssociations := $decor/terminology/terminologyAssociation[@conceptId=$listConcept/@id]
               return
               <listConcept>
               {
               $listConcept/@id,
               $listConcept/name,
               if($listConceptAssociations) then
                  for $association in $listConceptAssociations
                  return
                  treplib:handleTerminologyAssociation($decor,$association)
               
               else
               <message severity="warning" type="noValuesetItem" json:array="true">No valueset item associated with concept in list</message>
             }
               </listConcept>
             )
             }
          </conceptList>
       else if ($concept/valueDomain/@type='quantity') then
               let $ucum := $setlib:colDecorCore/ucums/ucum
               for $valDomain in $concept/valueDomain[@type='quantity']
                  for $property in $valDomain/property
                  let $ucumCheck := $ucum[@unit=$property/@unit]
                  return
                  if (empty($ucumCheck)) then
                     <message severity="info" type="noUcumUnit" json:array="true">{$property/@unit}</message>
                  else ()
             else ()
    }  
   </concept>
    return
         <concept>
         {
         $checkedConcept/@*,
         $checkedConcept/name,
         if ($checkedConcept//message[not(@type='ok')]) then
            $checkedConcept/message[not(@type='ok')]
         else
            <message severity="info" type="ok" json:array="true">OK</message>
         ,
         $checkedConcept/conceptList
         }  
   </concept>
    }  
   </dataset>
};


(: handle terminology association :)
declare function treplib:handleTerminologyAssociation($decor as element(),$terminologyAssociation as element()){
   (: check if codesystem is in project :)
   if (starts-with($terminologyAssociation/@codeSystem,$decor/project/@id)) then
      let $localConcept :=  $decor//codedConcept[@code=$terminologyAssociation/@code][ancestor::codeSystem/@id=$terminologyAssociation/@codeSystem]
      return
      if ($localConcept) then
         treplib:checkConcept($terminologyAssociation,$localConcept)
      else
         (: check if code system is present :)
         if ($decor//codeSystem[@id=$terminologyAssociation/@codeSystem]) then
            <message severity="error" type="conceptNotFound" json:array="true">
            {
               $terminologyAssociation
            }
            </message>
         else
            let $codeSystemName := if ($terminologyAssociation/@codeSystemName) then $terminologyAssociation/@codeSystemName else '?'
            return
            <message severity="warning" type="codesystemNotfound" json:array="true">
            {
               $terminologyAssociation
            }
            </message>
   else
      (: handle SNOMED post coordinated codes :)
      if ($terminologyAssociation/@codeSystem='2.16.840.1.113883.6.96') then
         if ($terminologyAssociation/@code castable as xs:integer) then
            let $snomedConcept := collection($setlib:strCodesystemStableData)//concept[@code=$terminologyAssociation/@code][ancestor::browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem]
            return
            if ($snomedConcept) then
               treplib:checkConcept($terminologyAssociation,$snomedConcept)
            else
               (: check if code system is present :)
               if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem) then
                  <message severity="error" type="conceptNotFound" json:array="true"><codeSystem                  oid="{$terminologyAssociation/@codeSystem}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$terminologyAssociation/@codeSystem]/name}</codeSystem>
                  {
                     $terminologyAssociation
                  }
                  </message>
               else
                  <message severity="warning" type="codesystemNotfound" json:array="true">
                  {
                     $terminologyAssociation
                  }</message>
         else treplib:checkAssociationSnomedExpression($terminologyAssociation/@code, $terminologyAssociation/@displayName)
         
      else
         let $codeSystemConcept := collection($setlib:strCodesystemStableData)//concept[@code=$terminologyAssociation/@code][ancestor::browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem]
         return
         if ($codeSystemConcept) then
            treplib:checkConcept($terminologyAssociation,$codeSystemConcept)
         else
            (: check if code system is present :)
            if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$terminologyAssociation/@codeSystem) then
               <message severity="error" type="conceptNotFound" json:array="true"><codeSystem               oid="{$terminologyAssociation/@codeSystem}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$terminologyAssociation/@codeSystem]/name}</codeSystem>
               {
                  $terminologyAssociation
               }
               </message>
            else
               let $codeSystemName := if ($terminologyAssociation/@codeSystemName) then $terminologyAssociation/@codeSystemName else '?'
               return
               <message severity="warning" type="codesystemNotfound" json:array="true">
               {
                  $terminologyAssociation
               }
               </message>
};


(: handle conceptlist association :)
declare function treplib:handleConceptListAssociation($decor as element(), $conceptListAssociation as element()) {
   (: check if valueset is in project :)
   if (starts-with($conceptListAssociation/@valueSet,$decor/project/@id)) then
      let $valueSet := 
               if ($conceptListAssociation/@flexibility='dynamic') then
                  let $vSet := $decor/terminology/valueSet[@id=$conceptListAssociation/@valueSet]
      let $latestDate := max($vSet/xs:dateTime(@effectiveDate))
                  return
      $vSet[@effectiveDate=$latestDate]
               else if ($conceptListAssociation/@flexibility castable as xs:dateTime) then
                  $decor/terminology/valueSet[@id=$conceptListAssociation/@valueSet][@effectiveDate=$conceptListAssociation/@flexibility]
               else ()
      return
      if ($valueSet) then
         (: check if conceptlist contains concepts and is of the same size as the valueset :)
         let $conceptList := $decor//conceptList[@id=$conceptListAssociation/@conceptId]
         return
         if ($conceptList/concept) then
            if (count($conceptList/concept) = count($valueSet//concept | $valueSet//exception)) then
               <message severity="info" type="ok" json:array="true"><valueSet displayName="{$valueSet/@displayName}" size="{count($valueSet//concept | $valueSet//exception)}"/></message>
      else
         <message severity="info" type="valueSetSizeNotConceptListSize" json:array="true"><conceptList size="{count($conceptList/concept)}"/><valueSet displayName="{$valueSet/@displayName}" size="{count($valueSet//concept | $valueSet//exception)}"/></message>
         else
            <message severity="info" type="ok" json:array="true">OK</message>
      else
         <message severity="error" type="valuesetNotFound" json:array="true">Valueset not found</message>
   else()

};


declare %private function treplib:checkConcept($terminologyAssociation as element(), $concept as element()) as element()*{
   let $displayName := $terminologyAssociation/@displayName
   let $designations :=
         for $designation in $concept/designation[not(@use='rel')][@lang=$treplib:supportedLanguages]
               order by $designation/@lang
               return
               <designation json:array="true">
                  {
                  $designation/@*,
                  $designation/text(),
                  if ($designation/*) then
                     let $parts := 
                     for $part in $designation/*
                     return
                     $part/text()
                     return
                     string-join($parts,' ')
                  else ()
                  }
               </designation>
   let $statusMessage :=
      if ($concept/@statusCode='retired') then
         <message severity="warning" type="conceptRetired" json:array="true">
            <concept code="{$concept/@code}">
               {
            for $designation in $concept/designation[@use='pref']
               return
               <designation json:array="true">
                  {
                  $designation/@*,
                  $designation/text()
                  }
               </designation>
               }
            </concept>
            {
            $terminologyAssociation,
            for $association in $concept/association
            return
            <association json:array="true">
               {
               $association/@*,
               $association/*
               }
            </association>
            }
         </message>
      else if ($concept/@statusCode='draft') then
         <message severity="warning" type="conceptDraft" json:array="true">
            {$terminologyAssociation}
         </message>
      else if ($concept/@statusCode='experimental') then
         <message severity="warning" type="conceptExperimental" json:array="true">
            {$terminologyAssociation}
         </message>
      else
         <message severity="info" type="ok" json:array="true">OK</message>
      
   let $designationMessage :=
      if ($displayName=$designations) then
            <message severity="info" type="ok" json:array="true">OK</message>
         else 
            let $lowerCase    := 
                                for $designation in $designations
                                return
            lower-case($designation)
            return
            if (lower-case($displayName)=$lowerCase) then
               <message severity="warning" type="designationCaseMismatch" json:array="true" displayName="{$displayName}">Case of display name does not match designation case in concept
               {
            $designations,
            $terminologyAssociation
 } </message>
               else
            <message severity="warning" type="noMatchingDesignation" json:array="true" displayName="{$displayName}">Display name does not match concept designation
               {
               $designations,
            $terminologyAssociation
            }
            </message>
            let $moduleMessage :=
      if ($concept/@moduleId) then
         if ($concept/@moduleId=('900000000000207008','900000000000012004')) then
            <message severity="info" type="ok" json:array="true">OK</message>
         else
            <message severity="info" type="notCoreModule" json:array="true">Concept is not part of core module</message>
      else ()
            
   return
               if ($statusMessage/@type='ok' and $designationMessage/@type='ok' and $moduleMessage/@type='ok') then
         <message severity="info" type="ok" json:array="true">OK</message>
      else
         ($statusMessage[not(@type='ok')],$designationMessage[not(@type='ok')],$moduleMessage[not(@type='ok')])
};

declare function treplib:checkValueSetConcept($displayName as xs:string, $concept as element()) as element()*{
   let $designations :=
         for $designation in $concept/designation[not(@use='rel')][@lang=$treplib:supportedLanguages]
               order by $designation/@lang
               return
   <designation json:array="true">
               {
                  $designation/@*,
                  $designation/text(),
               if               ($designation/*)   then
   let $parts := 
 for $part in $designation/*
 return
 $part/text()
 return
 string-join($parts,' ')
 else ()
   }
               </designation>
   let $statusMessage :=
      if ($concept/@statusCode='retired') then
         <message severity="warning" type="conceptRetired" json:array="true">
            <concept code="{$concept/@code}">
               {
            for $designation in $concept/designation[@use='pref']
               return
               <designation json:array="true">
                  {
                  $designation/@*,
                  $designation/text()
                  }
               </designation>
               }
            </concept>
            {
            for $association in $concept/association
            return
            <association json:array="true">
               {
               $association/@*,
               $association/*
               }
            </association>
            }
         </message>
      else if ($concept/@statusCode='draft') then
         <message severity="warning" type="conceptDraft" json:array="true">Concept is in draft</message>
      else if ($concept/@statusCode='experimental') then
         <message severity="warning" type="conceptExperimental" json:array="true">Concept is experimental</message>
      else
         <message severity="info" type="ok" json:array="true">OK</message>
   let $designationMessage :=
      if ($displayName=$concept/designation or $displayName=$concept/designation/@displayName) then
            <message severity="info" type="ok" json:array="true">OK</message>
      else
         let $lowerCase    := 
                                for $designation in $designations
                                return
                                lower-case($designation)
            return
            if (lower-case($displayName)=$lowerCase) then
               <message severity="warning" type="designationCaseMismatch" displayName="{$displayName}" json:array="true">Case of display name does not match designation case in concept
               {
               $designations
               }
               </message>
            else
               <message severity="warning" type="noMatchingDesignation" displayName="{$displayName}" json:array="true">Display name does not match concept designation
               {
               $designations
               }
               </message>
   let $moduleMessage :=
      if ($concept/@moduleId) then
         if ($concept/@moduleId=('900000000000207008','900000000000012004')) then
            <message severity="info" type="ok" json:array="true">OK</message>
         else
            <message severity="info" type="notCoreModule" json:array="true">Concept is not part of core module</message>
      else()      
   return
 if ($statusMessage/@type='ok' and         $designationMessage/@type='ok' and $moduleMessage/@type='ok') then
         <message severity="info" type="ok"      json:array="true">OK</message>
   else
   ($statusMessage[not(@type='ok')],$designationMessage[not(@type='ok')],$moduleMessage[not(@type='ok')])
};


(: Traverse valuesets :)
declare function treplib:traverseValuesets($decor as element()) as element()*{
   for $valueSet in $decor/terminology/valueSet[@id][@statusCode=('draft','final','pending')]
   let $sourceCodeSystems := 
            for $system in distinct-values($valueSet/conceptList/concept/@codeSystem | $valueSet/conceptList/exception/@codeSystem)
            
            return
   <sourceCodeSystem id="{$system}">
               {
               (: check if codesystem is local :)
               if (starts-with($system,$decor/project/@id)) then
                  if ($decor//codeSystem[@id=$system]) then
                     <message severity="info" type="ok">OK</message>
                  else
                     <message severity="warning" type="codesystemNotfound">Code stystem not found</message>
               else
                  if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$system) then
                     <message severity="info" type="ok">OK</message>
                  else
                     <message severity="warning" type="codesystemNotfound">Code stystem not found</message>
               }
            </sourceCodeSystem>
   order by $valueSet/@displayName
   return
   <valueSet conceptCount="{count($valueSet/conceptList/concept)}" json:array="true">
      {
      $valueSet/@*,
      $sourceCodeSystems,
      for $concept in $valueSet/conceptList/concept|$valueSet/conceptList/exception
         return
         <concept json:array="true">
         {
         $concept/@*,
         if ($concept/name()='exception') then
            attribute exception {''}
         else (),
         $concept/desc,
         if (starts-with($concept/@code,'http')) then
            <message severity="error" type="uriNotValidCode" json:array="true">URI is not a valid code</message>
         (: check if codesystem is in project :)
         else if (starts-with($concept/@codeSystem,$decor/project/@id)) then
            let $localConcept :=  $decor//codedConcept[@code=$concept/@code][ancestor::codeSystem/@id=$concept/@codeSystem]
            return
            if ($localConcept) then
               treplib:checkValueSetConcept($concept/@displayName,$localConcept)
            else
               (: check if code system is present :)
               if ($decor//codeSystem[@id=$concept/@codeSystem]) then
                  <message severity="error" type="conceptNotFound" json:array="true">Concept not found</message>
               else
                  let $codeSystemName := if ($concept/@codeSystemName) then $concept/@codeSystemName else '?'
                  return
                  <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
         else
            (: handle SNOMED post coordinated codes :)
            if ($concept/@codeSystem='2.16.840.1.113883.6.96') then
               if ($concept/@code castable as xs:integer) then
                  let $snomedConcept := collection($setlib:strCodesystemStableData)//concept[@code=$concept/@code][ancestor::browsableCodeSystem/@oid=$concept/@codeSystem]
                  return
                  if ($snomedConcept) then
                     treplib:checkValueSetConcept($concept/@displayName,$snomedConcept)
                  else
                     (: check if code system is present :)
                     if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$concept/@codeSystem) then
                        <message severity="error" type="conceptNotFound" json:array="true"><codeSystem oid="{$concept/@codeSystem}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$concept/@codeSystem]/name}</codeSystem></message>
                     else
                        <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
               else 
                  treplib:checkSnomedExpression($concept/@code,$concept/@displayName)
               
            
                     else
               let $codeSystemConcept := collection($setlib:strCodesystemStableData)//concept[@code=$concept/@code][ancestor::browsableCodeSystem/@oid=$concept/@codeSystem]
               return
               if ($codeSystemConcept) then
                  treplib:checkValueSetConcept($concept/@displayName,$codeSystemConcept)
               else
                  (: check if code system is present :)
                  if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$concept/@codeSystem) then
                     <message severity="error" type="conceptNotFound" json:array="true"><codeSystem oid="{$concept/@codeSystem}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$concept/@codeSystem]/name}</codeSystem></message>
                  else
                     let $codeSystemName := if ($concept/@codeSystemName) then $concept/@codeSystemName else '?'
                     return
                     <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
             }
            </concept>         
            }
   </valueSet>

};

(:
   Check SNOMED postcoordinated expression
   Removes terms from expression and calls parseSnomedExpression
:)

declare function treplib:checkAssociationSnomedExpression ($expression as xs:string, $displayName as xs:string) {
   if (starts-with($expression,'=')) then
      <message severity="error" type="equalSignNotAtStart" json:array="true">equal sign is not allowed at start of a postcoordinated code</message>
   else if (starts-with($expression,'http')) then
      <message severity="error" type="uriNotValidCode" json:array="true">URI is not a valid code</message>
   else
      let $result := treplib:parseSnomedExpression(replace($expression,'\|(.*?)\|',''))
   return
   if ($result//message or name($result[1])='message') then
      let $errorType :=
            if ($result//message) then
               $result//message[1]/@type
            else 
               $result[1]/@type
          return
         <message severity="error" type="errorInExpression" json:array="true">
         <expression string="{$expression}" displayName="{$displayName}">
            {
            $result
            }
         </expression>
      </message>
   else
      <message severity="info" type="ok" json:array="true">
         <expression string="{$expression}" displayName="{$displayName}">
            {
            $result
            }
         </expression>
      </message>
};

declare function treplib:checkSnomedExpression ($expression as xs:string,$displayName as xs:string) {
   if (starts-with($expression,'=')) then
      <message severity="error" type="equalSignNotAtStart" json:array="true">equal sign is not allowed at start of a postcoordinated code</message>
   else if (starts-with($expression,'http')) then
      <message severity="error" type="uriNotValidCode" json:array="true">URI is not a valid code</message>
   else
      let $result := treplib:parseSnomedExpression(replace($expression,'\|(.*?)\|',''))
   return
   if ($result//message or name($result[1])='message') then
      let $errorType :=
            if ($result//message) then
               $result//message[1]/@type
            else 
               $result[1]/@type
          return
         <message severity="error" type="errorInExpression" json:array="true">
      <expression string="{$expression}" displayName="{$displayName}">
            {
      $result
      }
      </expression>
      </message>
   else
      <message severity="info" type="ok" json:array="true">
      <expression string="{$expression}" displayName="{$displayName}">
            {
         $result
      }
      </expression>
      </message>
};

(:
   Recursive function for parsing Snomed expression.
   Returns xml structure of the expression with detected errors
:)

declare %private function treplib:parseSnomedExpression ($expression as xs:string) {
   (: check if refset member :)
   if (starts-with(normalize-space($expression),'^')) then 
      if (normalize-space(substring-after($expression,'^')) castable as      xs:integer) then
         let $memberCode := normalize-space(substring-after($expression,'^'))
         return
         <memberOf>
            <concept code="{$memberCode}">{treplib:getSnomedConceptDesignations($memberCode)}</concept>
         </memberOf>
      else
      <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
   (: check for multiple focus concepts :)
   else if (normalize-space(substring-before($expression,'+')) castable as      xs:integer) then
   let $focusCode := normalize-space(substring-before($expression,'+'))
 return
      (
      <concept code="{$focusCode}" json:array="true">{treplib:getSnomedConceptDesignations($focusCode)}</concept>,
   <additionalFocus/>,
      treplib:parseSnomedExpression(normalize-space(substring-after($expression,'+')))
   )
   else if (normalize-space(substring-before($expression,':')) castable as xs:integer) then
      let $conceptCode := normalize-space(substring-before($expression,':'))
      return
      (: check for attribute groups :)
      if (contains(substring-after($expression,':'),'{')) then
      let $attributeGroups := tokenize(substring-after($expression,':'),'\{')
         return
         (
         <concept code="{$conceptCode}" json:array="true">{treplib:getSnomedConceptDesignations($conceptCode)}</concept>,
      for $group in $attributeGroups
      let $groupRefinements:= tokenize(substring-before($group,'}'),',')
            return
            if (string-length($group) > 3) then
      if (contains($group,'}')) then
   <group json:array="true">
                   {
                   for $refinement in $groupRefinements
                   return
                   if (normalize-space(substring-before($refinement,'=')) castable as xs:integer) then
      let $refinementCode := normalize-space(substring-before($refinement,'='))
      return
                   <refinement json:array="true">
                   <concept code="{$refinementCode}">{treplib:getSnomedConceptDesignations($refinementCode)}</concept>
                      {
                      if (normalize-space(substring-after($refinement,'=')) castable as xs:integer) then
                         let $equalsCode :=normalize-space(substring-after($refinement,'='))
                         return
                         <equals code="{$equalsCode}">{treplib:getSnomedConceptDesignations($equalsCode)}</equals>
                      else
                      <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
                      }
                   </refinement>
                   else <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
                   }
                   </group>
                else <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
             else ()
          )
      else
         let $refinements := tokenize(normalize-space(substring-after($expression,':')),',')
      return
      (
      <concept code="{$conceptCode}" json:array="true">{treplib:getSnomedConceptDesignations($conceptCode)}</concept>,
         
         for $refinement in $refinements
         return
         if (normalize-space(substring-before($refinement,'=')) castable as xs:integer) then
         let $refinementCode := normalize-space(substring-before($refinement,'='))
         return
         <refinement json:array="true">
         <concept code="{$refinementCode}">{treplib:getSnomedConceptDesignations($refinementCode)}</concept>
            {
            if (normalize-space(substring-after($refinement,'=')) castable as xs:integer) then
               let $equalsCode :=normalize-space(substring-after($refinement,'='))
               return
               <equals code="{$equalsCode}">{treplib:getSnomedConceptDesignations($equalsCode)}</equals>
            else
            <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
            }
         </refinement>
         else <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
         )
   else if (normalize-space($expression) castable as xs:integer) then
      <concept code="{normalize-space($expression)}" json:array="true">{treplib:getSnomedConceptDesignations($expression)}</concept>
      else
         <message severity="warning" type="cannotParseExpression" json:array="true">Expression cannot be parsed</message>
};

declare function treplib:getSnomedConceptDesignations($code as xs:integer) as element()*{
   let $concept := collection(concat($setlib:strCodesystemStableData,'/external/snomed'))//concept[@code=$code]
   return
   if ($concept) then
       if ($concept/@statusCode='retired') then
         let $designations := $concept/designation
         return
         <message severity="warning" type="conceptRetired" json:array="true">
            <concept code="{$concept/@code}">
               {
            for $designation in $designations[@use='pref']
               return
               <designation json:array="true">
                  {
                  $designation/@*,
                  $designation/text()
                  }
               </designation>
               }
            </concept>
            {
            for $association in $concept/association
            return
            <association json:array="true">
               {
               $association/@*,
               $association/*
               }
            </association>
            }
         </message>
      else if ($concept/@statusCode='draft') then
         <message severity="warning" type="conceptDraft" json:array="true">Concept is in draft</message>
      else if ($concept/@statusCode='experimental') then
         <message severity="warning" type="conceptExperimental" json:array="true">Concept is experimental</message>
      else
  let $designations := $concept/designation
         return
         for $designation in $designations[@use='pref']
   order by $designation/@lang
         return
         <designation json:array="true">
            {
            $designation/@*,
            $designation/text()
            }
         </designation>
   else
      <message severity="error" type="conceptNotFound" json:array="true">Concept not found</message>
};

declare function treplib:convertReport2Html($terminologyReport as node(), $language as xs:string, $url as xs:anyURI?, $download as xs:boolean) as node() {


    let $title                          := $treplib:docHtmlNames[@key='terminologyReport']/text[@language=$language]

    let $oidList := for $oid in distinct-values($terminologyReport//sourceCodeSystem/@id)
                     let $displayName := utillib:getNameForOID($oid,$language,$terminologyReport/@for)
                     return
                     <codeSystem oid="{$oid}" displayName="{$displayName}"/>
 
    let $resourcePath                   :=  
        if ($download) then 'https://assets.art-decor.org/ADAR/rv/assets' else if(contains($utillib:strDecorServicesURL,'localhost')) then '/exist/apps/decor/core/assets' else '/exist/apps/decor/core/assets'
    let $html := 
    <html>
        <head>
            <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
            <title>{$title/text()}</title>
            <style type="text/css">
                <![CDATA[
                  tr:nth-child(even).zebra { background-color: #f2f2f2;}
                  td.warning {color: #FF8C00;}
                  td.error {color: red;}
                  td.exception {font-style: italic;}
                  body,form,table,tr,td,th,p{font-family: "Verdana", "Arial", sans-serif;font-size: 12px;font-weight: normal;color: #333333;}
                  h1{font-size: 20px;font-weight: bold;margin-left: 0px;margin-right: 0px;margin-top: 10px;margin-bottom: 10px;color: #e16e22;}
                  h2{font-size: 18px;font-weight: bold;margin-left: 0px;margin-right: 0px;margin-top: 4px;margin-bottom: 8px;background-color: #ece9e4;color: #e16e22;width: auto;}
                  th{background-color: #ece9e4;color: #7a6e62;font-weight: bold;text-align: left;vertical-align: top;padding-left:0.3em;}
                  a:link{color: inherit;}
                  a:visited{color: inherit;}
                  a:hover{color: red;}
                  a:active{color: #e16e22;}
                  .tooltip-wrap {position: relative;margin-right:0.3em;float:right;}
                  .tooltip-wrap .tooltip-content {display: none; position: absolute; bottom: 5%; left: 5%; right: 5%; background-color: #fff; padding: .5em; min-width: 10rem;}
                  .tooltip-wrap:hover .tooltip-content {display: block;cursor: pointer;width: 280px; background-color: #555; color: #fff; text-align: center; border-radius: 4px; padding: 2px;}
                 ]]>
            </style>
        </head>
        <body>
            <table style="width: 100%">
                <tbody class="list">
                    <tr>
                        <td style="text-align: left;">
                            <h1>{concat($title,':')}</h1>
                            <h1>{concat($terminologyReport/project/name[@language=$language]/text(),' (',$terminologyReport/@on,')')}</h1>
                        </td>
                    </tr>
                    <tr>
                    <td>
                     {
                     if ($terminologyReport/filter) then
                        concat($treplib:docHtmlNames[@key='filter']/text[@language=$language],': ',$terminologyReport/filter/@filterLabel)
                     else
                        $treplib:docHtmlNames[@key='noFilter']/text[@language=$language]
                     }
                    </td>
                    </tr>
                </tbody>
            </table>
            <h1>{$treplib:docHtmlNames[@key='datasets']/text[@language=$language]}</h1>
               {
               for $dataset in $terminologyReport/datasets/dataset
                  let $dsid   := $dataset/@id/string()
                  let $dsef   := $dataset/@effectiveDate/string()
               return
               <table style="width: 100%;">
                <thead>
                  <tr>
                  <th colspan="4"><h2>{ $dataset/name[@language=$language] } ({$dataset/@effectiveDate/string()}) <span style="float: right;padding-right:1em;">{$treplib:docHtmlNames[@key=$dataset/@statusCode]/text[@language=$language]/text()}</span></h2></th>
                  </tr>
                  <tr>
                  <th width="3%">Id</th>
                  <th width="50%">{$treplib:docHtmlNames[@key='concept']/text[@language=$language]}</th>
                  <th width="25%">{$treplib:docHtmlNames[@key='columnMessage']/text[@language=$language]}</th>
                  <th width="22%">{$treplib:docHtmlNames[@key='columnMessage']/text[@language=$language]} {$treplib:docHtmlNames[@key='conceptList']/text[@language=$language]}</th>
                  </tr>
                </thead>
                <tbody>
                {
                for $concept in $dataset/concept
                  let $indent    := xs:integer($concept/@level/string()) * 2
                  let $cptid     := $concept/@id/string()
                  let $cptef     := $concept/@effectiveDate/string()
                  let $cpthref   := $treplib:strAD3ServerURL || $terminologyReport/@for/string() || '/datasets/dataset/' || $dsid || '/' || encode-for-uri($dsef)  || '/concept/' ||  $cptid || '/' || encode-for-uri($cptef)
                return
                <tr class="zebra">
                  <td style="padding-right:0.5em;">
                     <span class="tooltip-wrap">
                       <span>{tokenize($concept/@id/string(),'\.')[last()]}</span>
                       <span class="tooltip-content">
                         {$concept/@id/string()}
                       </span> 
                     </span>
                      </td>
                  <td style="padding-left:{$indent+0.5}em;">
                      <a href="{$cpthref}" target="_blank">{$concept/name[@language=$language]}</a>
                  </td>
                  <td>
                  <table width="100%">
                     {
                     for $message in $concept/message
                     return
                     <tr>
                     {
                     if ($message/@unit) then
                        <td class="{$message/@severity/string()}">{concat($treplib:docHtmlNames[@key=$message/@type]/text[@language=$language]/text(),': ',$message/@unit/string())}</td>
                     else
                        <td class="{$message/@severity/string()}">{$treplib:docHtmlNames[@key=$message/@type]/text[@language=$language]/text()}</td>
                     }
                     </tr>
                     }
                     </table>
                  </td>
                  <td>
                  <table width="100%">
                     {
                     for $message in $concept/conceptList/message
                     return
                     <tr>
                        <td class="{$message/@severity/string()}">{$treplib:docHtmlNames[@key=$message/@type]/text[@language=$language]/text()}</td>
                     </tr>
                     }
                     </table>
                  </td>
                </tr>
                }
                </tbody>
               </table>
               }
               <p/>
               <h1>{$treplib:docHtmlNames[@key='ValueSets']/text[@language=$language]}</h1>
               {
               for $valueset in $terminologyReport/valueSets/valueSet
                 let $vsid                       := $valueset/@id/string()
                 let $vsef                       := $valueset/@effectiveDate/string()
                 let $vshref                     := $treplib:strAD3ServerURL || $terminologyReport/@for/string() || '/terminology/valueset/' || $vsid || '/' || encode-for-uri($vsef)
               return
               (
               <table width="100%">
                <thead>
                  <tr>
                  <th colspan="4"><h2><a href="{$vshref}" target="_blank">{ $valueset/@displayName/string() }</a> <span style="float: right;padding-right:1em;">{$treplib:docHtmlNames[@key=$valueset/@statusCode]/text[@language=$language]/text()}</span></h2></th>
                  </tr>
                  {
                   for $sourceCodeSystem in $valueset/sourceCodeSystem
                   return
                  <tr>
                  <th width="10%">
                     {$treplib:docHtmlNames[@key='columnCodeSystem']/text[@language=$language]}
                  </th>
                  <td width="40%">
                     { $oidList[@oid=$sourceCodeSystem/@id]/@displayName/string()} </td>
                  <td width="20%">
                     {$sourceCodeSystem/@id/string()}
                  </td>
                  <td width="30%">
                     {$treplib:docHtmlNames[@key=$sourceCodeSystem/message/@type]/text[@language=$language]}
                  </td>
                  </tr>
                  }
                  <tr>
                  <th width="10%">
                     {$treplib:docHtmlNames[@key='columnCode']/text[@language=$language]}
                  </th>
                  <th width="60%" colspan="2">
                     {$treplib:docHtmlNames[@key='columnDisplayName']/text[@language=$language]}
                  </th>
                  <th width="30%">
                     {$treplib:docHtmlNames[@key='columnMessage']/text[@language=$language]}
                  </th>
                  </tr>
                </thead>
                <tbody>
                  {
                  for $concept in $valueset/concept
                  return
                  <tr class="zebra">
                  <td class="{if ($concept/@exception) then 'exception' else ()}">
                     {$concept/@code/string()}
                  </td>
                  <td class="{if ($concept/@exception) then 'exception' else ()}" colspan="2">
                     {$concept/@displayName/string()}
                  </td>
                  <td>
                     <table>
                        {
                        for $message in $concept/message
                        return
                        <tr>
                        <td class="{$message/@severity/string()} {if ($concept/@exception) then 'exception' else ()}">
                        {$treplib:docHtmlNames[@key=$message/@type]/text[@language=$language]}
                        </td>
                        </tr>
                        }
                     </table>
                  </td>
                  </tr>
                  }
                </tbody>
               </table>
               ,
               <p/>
               )
               }
        </body>
    </html>
    return $html

};

declare function treplib:convertReportSimple2Html($terminologyReport as node(), $language as xs:string, $url as xs:anyURI?, $download as xs:boolean) as node() {


    let $title                          := 'Test'


 
    let $resourcePath                   :=  
        if ($download) then 'https://assets.art-decor.org/ADAR/rv/assets' else if(contains($utillib:strDecorServicesURL,'localhost')) then '/exist/apps/decor/core/assets' else '/decor/core/assets'
    let $html := 
    <html>
        <head>
            <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
            <title>{$title}</title>
            <link href="{$resourcePath}/css/default.css" rel="stylesheet" type="text/css"></link>
            <style type="text/css">
                <![CDATA[
                 ol.ad-dataset-group { list-style-type: none;}
                 li.ad-dataset-group { font-weight: bold; padding: 7px 0 0 0;}
                 li.ad-dataset-item, li.ad-template { font-weight: normal; list-style-type: none; padding: 7px 0 0 0; }
                 ul.ad-terminology-code { list-style-position: inside; font-weight: normal; padding: 7px 0 3px 20px; border: 0px; list-style-type: disc;  }
                 ul.ad-transaction-condition { padding: 7px 0 3px 20px; list-style-position: inside;}
                 li.ad-transaction-condition { font-style: italic; font-weight: normal; list-style-type: circle; padding: 0 3px 0 5px; border-left: 5px solid #ddd; }
                 div.ad-dataset-itemnumber, div.ad-templatetype { margin: -2px 0 5px 6px; display: inline-block; border: 1px solid #c0c0c0; 
                    background-color: #eee; border-radius: 3px 3px 3px 3px; -moz-border-radius: 3px 3px 3px 3px; -webkit-border-radius: 3px 3px 3px 3px; 
                    width: auto !important; padding: 1px 5px 1px 5px;}
                 div.ad-dataset-level1 { font-size: 2ex; font-weight: bold; border-bottom: 2px solid #ddd;}
                 table.ad-transaction-table, table.ad-template-table {border: 1px solid #888; border-collapse: collapse; width:100%;}
                 div.cdadocumentlevel { background-color: #eef; }
                 div.cdaheaderlevel { background-color: #ffe; }
                 div.cdasectionlevel { background-color: #efe; }
                 div.cdaentrylevel { background-color: #fef; }
                 div.ad-itemnumber-green { background-color: #efe; }
                 div.ad-itemnumber-blue { background-color: #cef; }
                 div.ad-itemnumber-yellow { background-color: #ffe; }
                 div.description, div.context {font-weight: normal; font-size: 12px; color: #aaa;}
                 ul.tmap {list-style-type: none; padding: 0; margin: 0; color:#8b0000; font-weight: normal; font-size: 10px; }
                 ]]>
            </style>
        </head>
        <body>
            <table style="width: 100%;">
                <tbody class="list">
                    <tr>
                        <td style="text-align: left;">
                            <h1>{$title}</h1>
                        </td>
                        <td style="text-align: right;">
                        Title
                        </td>
                    </tr>
                </tbody>
            </table>
               {
               for $dataset in $terminologyReport/datasets/dataset
               return
               <table>
                <thead>
                  <tr>
                  <th>{ $dataset/name[@language=$language] }</th>
                  <th></th>
                  </tr>
                </thead>
               </table>
               }
        </body>
    </html>
    
    return $html
};


declare %private function treplib:getHtmlNames() as element(entry)* {
    let $names                      :=
    <names>
            <entry key="noAssociation">
            <text language="en-US">No terminology association</text>
            <text language="de-DE">Keine Terminologieassoziation</text>
            <text language="nl-NL">Geen terminologiekoppeling</text>
        </entry>
        <entry key="noValuesetAssociated">
            <text language="en-US">No value set association</text>
            <text language="de-DE">Keine Valueset-Zuordnung</text>
            <text language="nl-NL">Geen waardelijst associatie</text>
        </entry>
        <entry key="noValuesetItem">
            <text language="en-US">No value set item association</text>
            <text language="de-DE">Keine Value Set-Element-Zuordnung</text>
            <text language="nl-NL">Geen waardelijst concept associatie</text>
        </entry>
        <entry key="noUcumUnit">
            <text language="en-US">Unit is not in UCUM</text>
            <text language="de-DE">Einheit ist nicht in UCUM</text>
            <text language="nl-NL">Eenheid staat niet in UCUM</text>
        </entry>
        <entry key="conceptNotFound">
            <text language="en-US">Concept not found</text>
            <text language="de-DE">Konzept nicht gefunden</text>
            <text language="nl-NL">Concept niet gevonden</text>
        </entry>
        <entry key="codesystemNotfound">
            <text language="en-US">Code system not found</text>
            <text language="de-DE">Codesystem nicht gefunden</text>
            <text language="nl-NL">Codesysteem niet gevonden</text>
        </entry>
        <entry key="valuesetNotFound">
            <text language="en-US">Value set not found</text>
            <text language="de-DE">Valueset nicht gefunden</text>
            <text language="nl-NL">Waardelijst niet gevonden</text>
        </entry>
        <entry key="valueSetSizeNotConceptListSize">
            <text language="en-US">Value set size ≠ concept list size</text>
            <text language="de-DE">Wertesatzgröße ≠ Konzeptlistengröße</text>
            <text language="nl-NL">Grootte van waardelijst ≠ grootte van conceptlijst</text>
        </entry>
        <entry key="noMatchingDesignation">
            <text language="en-US">No matching designation</text>
            <text language="de-DE">Keine passende Bezeichnung</text>
            <text language="nl-NL">Geen overeenkomende weergavenaam</text>
        </entry>
        <entry key="designationCaseMismatch">
            <text language="en-US">Designation case mismatch</text>
            <text language="de-DE">Nichtübereinstimmung der Bezeichnungs-Groß-/Kleinschreibung</text>
            <text language="nl-NL">Hoofdlettergebruik komt niet overeen</text>
        </entry>
        <entry key="conceptRetired">
            <text language="en-US">Concept retired</text>
            <text language="de-DE">Konzept obsolet</text>
            <text language="nl-NL">Concept obsoleet</text>
        </entry>
        <entry key="conceptDraft">
            <text language="en-US">Concept in draft</text>
            <text language="de-DE">Konzept im Entwurf</text>
            <text language="nl-NL">Concept is voorlopig</text>
        </entry>
        <entry key="conceptExperimental">
            <text language="en-US">Concept is experimental</text>
            <text language="de-DE">Konzept ist experimentell</text>
            <text language="nl-NL">Concept is experimenteel</text>
        </entry>
        <entry key="cannotParseExpression">
            <text language="en-US">Cannot parse expression</text>
            <text language="de-DE">Ausdruck kann nicht analysiert werden</text>
            <text language="nl-NL">Expressie kan niet verwerkt worden</text>
        </entry>
        <entry key="equalSignNotAtStart">
            <text language="en-US">Equal sign is not allowed at start of expression</text>
            <text language="de-DE">Gleichheitszeichen sind am Anfang des Ausdrucks nicht zulässig</text>
            <text language="nl-NL">Is-gelijk-teken is niet toegestaan aan het begin</text>
        </entry>
        <entry key="uriNotValidCode">
            <text language="en-US">URI is not valid as a code</text>
            <text language="de-DE">URI ist als Code nicht gültig</text>
            <text language="nl-NL">URI is toegestaan als code</text>
        </entry>
        <entry key="errorInExpression">
            <text language="en-US">Error in expression</text>
            <text language="de-DE">Fehler im Ausdruck</text>
            <text language="nl-NL">Fout in expressie</text>
        </entry>
        <entry key="notCoreModule">
            <text language="en-US">Concept not in core module</text>
            <text language="de-DE">Konzept nicht im Kernmodul</text>
            <text language="nl-NL">Concept niet in internationale module</text>
        </entry>
        <entry key="ok">
            <text language="en-US">OK</text>
            <text language="de-DE">OK</text>
            <text language="nl-NL">OK</text>
        </entry>       
        <entry key="columnCode">
            <text language="en-US">Code</text>
            <text language="de-DE">Code</text>
            <text language="nl-NL">Code</text>
        </entry>
        <entry key="columnCodeSystem">
            <text language="en-US">Code System</text>
            <text language="de-DE">Codesystem</text>
            <text language="nl-NL">Codesysteem</text>
        </entry>
        <entry key="columnDisplayName">
            <text language="en-US">Display Name</text>
            <text language="de-DE">Anzeigename</text>
            <text language="nl-NL">Weergavenaam</text>
        </entry>
        <entry key="columnMessage">
            <text language="en-US">Message</text>
            <text language="de-DE">Nachricht</text>
            <text language="nl-NL">Bericht</text>
        </entry>
        <entry key="terminologyReport">
            <text language="en-US">Terminology Report</text>
            <text language="de-DE">Terminologiebericht</text>
            <text language="nl-NL">Terminologie rapport</text>
        </entry>
        <entry key="ValueSets">
            <text language="en-US">Value Sets</text>
            <text language="de-DE">Value Sets</text>
            <text language="nl-NL">Waardelijsten</text>
        </entry>
        <entry key="datasets">
            <text language="en-US">Datasets</text>
            <text language="de-DE">Datensätze</text>
            <text language="nl-NL">Datasets</text>
        </entry>
         <entry key="concept">
             <text language="en-US">Concept</text>
             <text language="de-DE">Konzept</text>
             <text language="nl-NL">Concept</text>
         </entry>
         <entry key="filter">
             <text language="en-US">Filter</text>
             <text language="de-DE">Filter</text>
             <text language="nl-NL">Filter</text>
         </entry>
         <entry key="noFilter">
             <text language="en-US">No filter</text>
             <text language="de-DE">Kein Filter</text>
             <text language="nl-NL">Geen filter</text>
         </entry>
        <entry key="draft">
            <text language="en-US">Draft</text>
            <text language="de-DE">Entwurf</text>
            <text language="nl-NL">Ontwerp</text>
        </entry>
        <entry key="pending">
            <text language="en-US">Under pre-publication review</text>
            <text language="de-DE">Unter Revision vor der Publikation</text>
            <text language="nl-NL">Onder revisie vóór de publicatie</text>
        </entry>
        <entry key="final">
            <text language="en-US">Final</text>
            <text language="de-DE">Definitiv</text>
            <text language="nl-NL">Definitief</text>
        </entry>
        <entry key="conceptList">
            <text language="en-US">Concept list</text>
            <text language="de-DE">Konzeptliste</text>
            <text language="nl-NL">Conceptenlijst</text>
        </entry>
        <entry key="terminologyAssociation">
            <text language="en-US">Terminology Association</text>
            <text language="de-DE">Terminologie-Assoziation</text>
            <text language="nl-NL">Terminologiekoppeling</text>
        </entry>
        
        
        
        
        
        <entry key="columnName">
            <text language="en-US">Name</text>
            <text language="de-DE">Name</text>
            <text language="nl-NL">Naam</text>
        </entry>
        <entry key="columnID">
            <text language="en-US">ID</text>
            <text language="de-DE">ID</text>
            <text language="nl-NL">ID</text>
        </entry>
        <entry key="columnMandatory">
            <text language="en-US">Mandatory</text>
            <text language="de-DE">Mandatory</text>
            <text language="nl-NL">Mandatory</text>
        </entry>
        <entry key="columnCardinality">
            <text language="en-US">Card</text>
            <text language="de-DE">Kard</text>
            <text language="nl-NL">Card</text>
        </entry>
        <entry key="columnMax">
            <text language="en-US">Max</text>
            <text language="de-DE">Max</text>
            <text language="nl-NL">Max</text>
        </entry>
        <entry key="columnCCD">
            <text language="en-US">Datatype CC</text>
            <text language="de-DE">Datentyp CC</text>
            <text language="nl-NL">Datatype CC</text>
        </entry>
        <entry key="columnDatatype">
            <text language="en-US">Datatype</text>
            <text language="de-DE">Datentyp</text>
            <text language="nl-NL">Datatype</text>
        </entry>
        <entry key="columnProperty">
            <text language="en-US">Property</text>
            <text language="de-DE">Eigenschaft</text>
            <text language="nl-NL">Eigenschap</text>
        </entry>
        <entry key="columnExample">
            <text language="en-US">Example</text>
            <text language="de-DE">Beispiel</text>
            <text language="nl-NL">Voorbeeld</text>
        </entry>
        <entry key="columnDescription">
            <text language="en-US">Description</text>
            <text language="de-DE">Beschreibung</text>
            <text language="nl-NL">Omschrijving</text>
        </entry>
        <entry key="columnContext">
            <text language="en-US">Context</text>
            <text language="de-DE">Kontext</text>
            <text language="nl-NL">Context</text>
        </entry>
        <entry key="columnOperationalization">
            <text language="en-US">Operationalizations</text>
            <text language="de-DE">Operationalisierungen</text>
            <text language="nl-NL">Operationalisaties</text>
        </entry>
        <entry key="columnRationale">
            <text language="en-US">Rationale</text>
            <text language="de-DE">Rationale</text>
            <text language="nl-NL">Rationale</text>
        </entry>
        <entry key="columnSource">
            <text language="en-US">Source</text>
            <text language="de-DE">Quelle</text>
            <text language="nl-NL">Bron</text>
        </entry>
        <entry key="columnComment">
            <text language="en-US">Comment</text>
            <text language="de-DE">Kommentar</text>
            <text language="nl-NL">Opmerking</text>
        </entry>
        <entry key="columnCondition">
            <text language="en-US">Condition</text>
            <text language="de-DE">Condition</text>
            <text language="nl-NL">Conditie</text>
        </entry>
        <entry key="columnEnableWhen">
            <text language="en-US">Enable when</text>
            <text language="de-DE">Aktiviert wann</text>
            <text language="nl-NL">Actief als</text>
        </entry>
        <entry key="columnMapping">
            <text language="en-US">Mapping</text>
            <text language="de-DE">Mapping</text>
            <text language="nl-NL">Mapping</text>
        </entry>
        <entry key="columnStatus">
            <text language="en-US">Status</text>
            <text language="de-DE">Status</text>
            <text language="nl-NL">Status</text>
        </entry>
        <entry key="columnCommunity">
            <text language="en-US">Community</text>
            <text language="de-DE">Community</text>
            <text language="nl-NL">Community</text>
        </entry>
        <entry key="columnTerminology">
            <text language="en-US">Terminology</text>
            <text language="de-DE">Terminologie</text>
            <text language="nl-NL">Terminologie</text>
        </entry>
        <entry key="columnValueSet">
            <text language="en-US">Value Set</text>
            <text language="de-DE">Value Set</text>
            <text language="nl-NL">Waardelijst</text>
        </entry>
        <entry key="columnType">
            <text language="en-US">Type</text>
            <text language="de-DE">Type</text>
            <text language="nl-NL">Type</text>
        </entry>
        <entry key="columnParent">
            <text language="en-US">Parent concept</text>
            <text language="de-DE">Elternkonzept</text>
            <text language="nl-NL">Ouderconcept</text>
        </entry>
        <entry key="columnInherit">
            <text language="en-US">Inherit from</text>
            <text language="de-DE">Erbt von</text>
            <text language="nl-NL">Erft van</text>
        </entry>
        <entry key="columnCodeSystemWithFilters">
            <text language="en-US">Code System with Filters</text>
            <text language="de-DE">Codesystem met Filters</text>
            <text language="nl-NL">Codesysteem met filters</text>
        </entry>
        <entry key="dynamic">
            <text language="en-US">dynamic</text>
            <text language="de-DE">dynamisch</text>
            <text language="nl-NL">dynamisch</text>
        </entry>
        <entry key="static">
            <text language="en-US">static</text>
            <text language="de-DE">statisch</text>
            <text language="nl-NL">statisch</text>
        </entry>
        <entry key="length">
            <text language="en-US">Length</text>
            <text language="nl-NL">Lengte</text>
            <text language="de-DE">Länge</text>
        </entry>
        <entry key="range">
            <text language="en-US">Min/max</text>
            <text language="nl-NL">Min/max</text>
            <text language="de-DE">Min/max</text>
        </entry>
        <entry key="fractionDigits">
            <text language="en-US">Decimals</text>
            <text language="nl-NL">Decimalen</text>
            <text language="de-DE">Ziffern</text>
        </entry>
        <entry key="defaultValue">
            <text language="en-US">Default value</text>
            <text language="nl-NL">Standaardwaarde</text>
            <text language="de-DE">Standard Wert</text>
        </entry>
        <entry key="fixedValue">
            <text language="en-US">Fixed value</text>
            <text language="nl-NL">Vaste waarde</text>
            <text language="de-DE">Fixen Wert</text>
        </entry>
        <entry key="currency">
            <text language="en-US">Currency</text>
            <text language="nl-NL">Valuta</text>
            <text language="de-DE">Währung</text>
        </entry>
        <entry key="timeStampPrecision">
            <text language="en-US">Timestamp precision</text>
            <text language="nl-NL">Tijdstempelprecisie</text>
            <text language="de-DE">Genauigkeit Zeitangabe</text>
        </entry>
        <entry key="unit">
            <text language="en-US">Unit</text>
            <text language="nl-NL">Eenheid</text>
            <text language="de-DE">Einheit</text>
        </entry>
        <entry key="group">
            <text language="en-US">Group</text>
            <text language="de-DE">Gruppe</text>
            <text language="nl-NL">Groep</text>
        </entry>
        <entry key="item">
            <text language="en-US">Item</text>
            <text language="de-DE">Item</text>
            <text language="nl-NL">Item</text>
        </entry>
        <entry key="groups">
            <text language="en-US">Groups</text>
            <text language="de-DE">Gruppen</text>
            <text language="nl-NL">Groepen</text>
        </entry>
        <entry key="items">
            <text language="en-US">Items</text>
            <text language="de-DE">Items</text>
            <text language="nl-NL">Items</text>
        </entry>
        <entry key="ValueSet">
            <text language="en-US">Value Set</text>
            <text language="de-DE">Value Set</text>
            <text language="nl-NL">Waardelijst</text>
        </entry>
        <entry key="CodeSystem">
            <text language="en-US">Code System</text>
            <text language="de-DE">Code System</text>
            <text language="nl-NL">Codesysteem</text>
        </entry>
        <entry key="CodeSystems">
            <text language="en-US">Code Systems</text>
            <text language="de-DE">Code Systeme</text>
            <text language="nl-NL">Codesystemen</text>
        </entry>
        <entry key="Template">
            <text language="en-US">Template</text>
            <text language="de-DE">Template</text>
            <text language="nl-NL">Template</text>
        </entry>
        <entry key="ConceptMaps">
            <text language="en-US">Concept Maps</text>
            <text language="de-DE">Concept Maps</text>
            <text language="nl-NL">Conceptmaps</text>
        </entry>   
        <entry key="orWord">
            <text language="en-US">or</text>
            <text language="de-DE">oder</text>
            <text language="nl-NL">of</text>
        </entry>
        <entry key="else">
            <text language="en-US">Else</text>
            <text language="de-DE">Sonst</text>
            <text language="nl-NL">Anders</text>
         </entry>
         <entry key="undefined">
             <text language="en-US">All/any undefined</text>
             <text language="de-DE">Alles/irgendein undefiniert</text>
             <text language="nl-NL">Alles/tenminste één niet gedefinieerd</text>
         </entry>
         <entry key="dataset">
            <text language="en-US">Data Set</text>
            <text language="de-DE">Datensatz</text>
            <text language="nl-NL">Dataset</text>
         </entry>
         <entry key="concepts">
             <text language="en-US">Concepts</text>
             <text language="de-DE">Konzepte</text>
             <text language="nl-NL">Concepten</text>
         </entry>
         <entry key="scenario">
             <text language="en-US">Scenario</text>
             <text language="de-DE">Szenario</text>
             <text language="nl-NL">Scenario</text>
         </entry>
         <entry key="transaction">
             <text language="en-US">Transaction</text>
             <text language="de-DE">Transaktion</text>
             <text language="nl-NL">Transactie</text>
         </entry>
         <entry key="textSearch">
            <text language="en-US">Search by name</text>
            <text language="de-DE">Suche auf Name</text>
            <text language="nl-NL">Zoek op naam</text>
        </entry>
        <entry key="buttonListView">
            <text language="en-US">List/tree view</text>
            <text language="de-DE">Liste/Baum Ansicht</text>
            <text language="nl-NL">Lijst/Tree view</text>
        </entry>
        <entry key="resetToDefault">
            <text language="en-US">Reset View</text>
            <text language="de-DE">Originalansicht</text>
            <text language="nl-NL">Reset view</text>
        </entry>
        <entry key="buttonSortByName">
            <text language="en-US">Sort by name</text>
            <text language="de-DE">Sortiere auf Name</text>
            <text language="nl-NL">Sorteer op naam</text>
        </entry>
        <entry key="buttonCollapseAll">
            <text language="en-US">Collapse All</text>
            <text language="de-DE">Alles einklappen</text>
            <text language="nl-NL">Alles inklappen</text>
        </entry>
        <entry key="buttonDownload">
            <text language="en-US">Download</text>
            <text language="de-DE">Download</text>
            <text language="nl-NL">Downloaden</text>
        </entry>
        <entry key="buttonDownloadHelp">
            <text language="en-US">Downloads the full page as HTML. This sends a new request to the server to make it suitable for opening in Excel. Tries to retain markup.</text>
            <text language="de-DE">Lädt die ganze Seite als HTML herunter. Dadurch wird eine neue Anforderung an den Server gesendet, damit dieser zum Öffnen in Excel geeignet ist. Versucht, das Markup beizubehalten.</text>
            <text language="nl-NL">Downloadt de hele pagina als HTML. Hiervoor wordt een nieuwe verzoek naar de server gestuurd om deze geschikt te maken voor openen in Excel. Probeert alle opmaak te behouden.</text>
        </entry>
        <entry key="buttonExportTableToExcel">
            <text language="en-US">Export Table to Excel</text>
            <text language="de-DE">Tabelle nach Excel exportieren</text>
            <text language="nl-NL">Exporteer tabel naar Excel</text>
        </entry>
        <entry key="buttonExportTableToExcelHelp">
            <text language="en-US">Downloads the table as currently visible as Excel (xls). Looses some of the markup.</text>
            <text language="de-DE">Lädt die Tabelle so aktuell wie Excel (xls) herunter. Verliert einen Teil des Markups.</text>
            <text language="nl-NL">Downloadt de tabel zoals weergegeven als Excel (xls). Verliest een deel van de opmaak.</text>
        </entry>
        <entry key="collapse">
            <text language="en-US">Collapse</text>
            <text language="de-DE">Einklappen</text>
            <text language="nl-NL">Inklappen</text>
        </entry>
        <entry key="expand">
            <text language="en-US">Expand</text>
            <text language="de-DE">Ausklappen</text>
            <text language="nl-NL">Uitklappen</text>
        </entry>
        <entry key="buttonExpandAll">
            <text language="en-US">Expand All</text>
            <text language="de-DE">Alles ausklappen</text>
            <text language="nl-NL">Alles uitklappen</text>
        </entry>
        <entry key="buttonCollapseAllCodes">
            <text language="en-US">Collapse All Codes</text>
            <text language="de-DE">Alle Codes einklappen</text>
            <text language="nl-NL">Alle codes inklappen</text>
        </entry>
        <entry key="buttonShow">
            <text language="en-US">HTML</text>
            <text language="de-DE">HTML</text>
            <text language="nl-NL">HTML</text>
        </entry>
        <entry key="buttonConvert">
            <text language="en-US">CDAr3</text>
            <text language="de-DE">CDAr3</text>
            <text language="nl-NL">CDAr3</text>
        </entry>
        <entry key="buttonGo">
            <text language="en-US">Show</text>
            <text language="de-DE">Zeigen</text>
            <text language="nl-NL">Tonen</text>
        </entry>
        <entry key="helpText">
            <text language="en-US">Click on triangles in the first column to open/close groups. Click the [-] sign to hide a column. Choose a column with  'Show column' to show it again. You can drag 
                columns to re-arrange the view.</text>
            <text language="nl-NL">Klik op de driehoeken in de eerste kolom om groepen te openen of te sluiten. Klik op [-] om een kolom te verbergen. Kies een kolom in 'Toon kolom' om deze weer te tonen. 
                Kolommen kunnen versleept worden naar een nieuwe positie.</text>
            <text language="de-DE">Klicken Sie auf die Dreiecke in der ersten Spalte zum Öffnen oder Schließen von Gruppen. Klicken Sie auf das [-] Zeichen, um eine Spalte zu verbergen. Wählen Sie eine 
                Spalte mit "Zeige Spalte", um diese wieder zu zeigen. Man kann die Spalten ziehen, um diese neu einzuordnen.</text>
        </entry>
        <entry key="goTo">
            <text language="en-US">Show </text>
            <text language="de-DE">Zeige </text>
            <text language="nl-NL">Toon </text>
        </entry>
    </names>
    
    return $names/entry
};