xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace ruleslib               = "http://art-decor.org/ns/api/rules";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace utilvs          = "http://art-decor.org/ns/api/util-valueset" at "util-valueset-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

declare %private variable $ruleslib:ADDRLINE-TYPE       := utillib:getDecorTypes()/AddressLineType;
declare %private variable $ruleslib:CONCEPT-TYPE        := utillib:getDecorTypes()/DataSetConceptType;
declare %private variable $ruleslib:VALUEDOMAIN-TYPE    := utillib:getDecorTypes()/DataSetValueType;
declare %private variable $ruleslib:EXAMPLE-TYPE        := utillib:getDecorTypes()/ExampleType;
declare %private variable $ruleslib:CONFORMANCE-TYPE    := utillib:getDecorTypes()/ConformanceType;
declare %private variable $ruleslib:RELATIONSHIP-TYPE   := utillib:getDecorTypes()/RelationshipTypes;
declare %private variable $ruleslib:VALUEDOMAIN-PROPS   := utillib:getDecorTypes()/DataSetValueProperty/attribute;

(: first we decide how to start te error message :)
declare %private function ruleslib:getErrorString($elmnt as element(), $operation as xs:string, $path as xs:string?) as xs:string {

    if ($path)
        then 'Parameter ' || $operation || ' not allowed for ''' || $path || ''':' 
        else 'Cannot '|| $operation || ' '''|| name($elmnt) ||''':'};

(: ==================================== GENERAL RULES ================================================:)

(: rule - every create or update of a DECOR Artefact has to be schema compiant :)
declare function ruleslib:checkDecorSchema($preparedData as item(), $operation as xs:string) {

    let $artefact           := name($preparedData)
    let $error              := 'Cannot '|| $operation || ' ' || $artefact || ': the ' || $artefact || ' is not schema compliant against ' || substring-after(util:collection-name($setlib:docDecorSchema), 'db/apps') || '/' || util:document-name($setlib:docDecorSchema) || ' on this server.'
    
    let $check              := validation:jaxv-report($preparedData, $setlib:docDecorSchema)
    
    return
        if ($check/status='invalid') then
            error($errors:BAD_REQUEST, $error || ' === SEE REPORT === ' || serialize($check))
        else ()
};

(: rule - every create or update or patch of a publishing authority has to be checked on
        * cannot be marked inherited for update 
        * id has to be an OID
        * name cannot be empty
        * address line types has to be supported
:)
declare function ruleslib:checkPublishingAuthority($publishingAuthority as element(publishingAuthority), $operation as xs:string, $path as xs:string?) as xs:string* {
    
    let $check              := () 
    
    (: cannot be marked inherited for update :)
    let $error              := $check || ' Contents SHALL NOT be marked ''inherited''.' 
    let $check              := if ($publishingAuthority[@inherited]) then $error else $check    
    
    (: id has to be an OID :)
    let $error              := $check || ' PublishingAuthority.id SHALL be an OID, found ' || string-join($publishingAuthority/@id, ', ')
    let $check              := if ($publishingAuthority[@id]) then if ($publishingAuthority[utillib:isOid(@id)]) then $check else $error else $check
    
    (: name cannot be empty :)
    let $error              := $check || ' PublishingAuthority.name SHALL be a non empty string.'
    let $check              := if ($publishingAuthority[@name[not(. = '')]]) then $check else $error

    (: address line types has to be supported :)
    let $unsupportedtypes   := $publishingAuthority/addrLine/@type[not(. = $ruleslib:ADDRLINE-TYPE/enumeration/@value)]
    let $error              := $check || ' Publishing authority has unsupported addrLine.type value(s) ' || string-join($unsupportedtypes, ', ') || '. SHALL be one of: ' || string-join($ruleslib:ADDRLINE-TYPE/enumeration/@value, ', ')
    let $check              := if ($unsupportedtypes) then $error else $check
    
    return if ($check) then ruleslib:getErrorString($publishingAuthority, $operation, $path) || $check  else ()  
};

(: rule - every create or update or patch of a FreeFormMarkupWithLanguage type element has to be checked on
        * cannot be marked inherited for update 
        * element must have contents
        * language is has regEx [a-z]{2}-[A-Z]{2}
:)
declare function ruleslib:checkFreeFormMarkupWithLanguage($elmnt as element()*, $operation as xs:string, $path as xs:string?) as xs:string* {

    let $elmname            := name($elmnt)
    let $check              := ()
     
    (: cannot be marked inherited for update :)
    let $error              := $check || ' Contents SHALL NOT be marked ''inherited''.' 
    let $check              := if ($elmnt[@inherited]) then $error else $check    
    
    (: element has contents :)
    let $error              := $check || ' A ''' || $elmname || ''' SHALL have contents.'
    let $check              := if ($operation = 'remove' or $elmnt[.//text()[not(normalize-space() = '')]]) then $check else $error
    
    (: language is has regEx [a-z]{2}-[A-Z]{2} :)
    let $error              := $check || ' A ''' || $elmname || ''' SHALL have a language with pattern ll-CC.'
    let $check              := if ($elmnt[matches(@language, '[a-z]{2}-[A-Z]{2}')]) then $check else $error
    
    return if ($check) then ruleslib:getErrorString($elmnt, $operation, $path) || $check  else ()  
};

(: rule - every create or update or patch of a ValueCodingType type element shall be code without whitespace, a codeSystem as valid OID, and optionally a canonicalUri as URI :)
declare function ruleslib:checkValueCodingType($elmnt as element(), $operation as xs:string, $path as xs:string?) as xs:string* {

    let $error              := ' Input SHALL have code without whitespace, a codeSystem as valid OID, and optionally a canonicalUri as URI. Found: ' || string-join(for $att in $elmnt/@* return name($att) || ': "' || $att || '" ', ' ')
    
    return 
        if ($elmnt[@code[matches(., '^\S+$')]][utillib:isOid(@codeSystem)][empty(@canonicalUri) or @canonicalUri castable as xs:anyURI]) then () else ruleslib:getErrorString($elmnt, $operation, $path) || $error
    
};

(: rule - every create or update or patch of a dataset (concept) relationship has to be checked on
        * relationship type has to be supported 
        * ref has to be an OID
        * flexibility has to be omitted, yyyy-MM-DDThh:mm:ss or 'dynamic'
:)
declare function ruleslib:checkRelationship($relationship as element(relationship), $operation as xs:string, $path as xs:string?) as xs:string* {
    
    let $check              := () 
    
    (: relationship types has to be supported :)
    let $unsupportedtypes   := $relationship/@type[not(. = $ruleslib:RELATIONSHIP-TYPE/enumeration/@value)]
    let $error              := $check || ' Relationship.type has unsupported value ' || string-join($unsupportedtypes, ', ') || '. SHALL be one of: ' || string-join($ruleslib:RELATIONSHIP-TYPE/enumeration/@value, ', ')
    let $check              := if ($unsupportedtypes) then $error else $check

    (: ref has to be an OID :)
    let $error              := $check || ' Relationship.ref SHALL be an OID, found ' || string-join($relationship/@ref, ', ')
    let $check              := if ($relationship[@ref]) then if ($relationship[utillib:isOid(@ref)]) then $check else $error else $check
    
    (: flexibility has to be omitted, yyyy-MM-DDThh:mm:ss or 'dynamic' :)
    let $error              := $check || ' Relationship.flexibility SHALL  omitted, yyyy-MM-DDThh:mm:ss or ''dynamic'', found ' || string-join($relationship/@flexibility, ', ')
    let $check              := if ($relationship[empty(@flexibility) or @flexibility castable as xs:dateTime or @flexibility = 'dynamic']) then $check else $error

    return if ($check) then ruleslib:getErrorString($relationship, $operation, $path) || $check  else ()  
};

(: ==================================== CONCEPTMAP RULES ============================================ :)

(: rule - check on conceptMap sourceScope or targetScope
        * source and target valueset can not have the same value
        * valueSet @ref should be an OID
        * valueSet is found in decor project
        * canonicalUri is empty or the same as valueSet canoncialUri 
:)
(: expect: element(sourceScope) or element(targetScope) :)
declare function ruleslib:checkConceptMapScope($elmnt as element(), $sourceOrTargetScope as element(), $decor as element(decor), $operation as xs:string, $path as xs:string?) as xs:string* {

    let $elmname            := name($elmnt)
    let $check              := ()

    (: source and target valueset can not have the same value :)
    let $error              := $check || ' SOURCESCOPE.ref and TARGETSCOPE.ref have the same value. Found: for both valueSets ''' || $elmnt/@ref || '''.' 
    let $check              := if ($elmnt/@ref eq $sourceOrTargetScope/@ref) then $error else $check

    let $check              :=
    
        (: valueSet @ref should be an OID :)
        if (not(utillib:isOid($elmnt/@ref))) then ' Input SHALL have ref as an OID on ' || $elmname || ' under value. Found: ''' || $elmnt/@ref || '''.'
        else (
            let $valueSet   := utilvs:getValueSetById($elmnt/@ref, $elmnt/@flexibility, $decor, (), ())[1]
            return
                (: valueSet is found in decor project :)
                if (empty($valueSet)) then  $check || ' ' || upper-case($elmname) || '.ref ''' || $elmnt/@ref || ''' does not resolve to a value set.' else $check
            )
            
    return if ($check) then ruleslib:getErrorString($elmnt, $operation, $path) || $check else ()            
};

(: rule - check on conceptMap group
        * target and element are both there or the unmatched group has no target element and the element/target have no code attribute
        * source and target codesystem can not have the same value
        * for source and target codesystem: @ref should be an OID
        * for source and target codesystem: codeSystem is found as a sourceCodeSystem in scope valueset
        * for source and target codesystem: canonicalUri is empty or the same as codesystem canoncialUri 
:)
declare function ruleslib:checkConceptMapGroup($conceptMap as element(conceptMap), $decor as element(decor), $operation as xs:string, $path as xs:string?) as xs:string* {
    
    (: first check the DECOR.xsd and stop if not compliant :)
    let $check              := ruleslib:checkDecorSchema($conceptMap, $operation)

    let $check              :=
    for $group in $conceptMap/group 
        let $check          := ()
        let $elmname        := name($group)
        
        (: target and element are both there or the unmatched group has no target element and the element/target have no code attribute :) 
        let $error          := $check || ' Input SHALL have both ''target'' and ''element'' OR ''target'' and ''element/target/@code'' are both empty. Found: target ' || $group/target/@codeSystem || ' and element ' || $group/element/@code[@code][1]  
        let $check          := if (($group/target and $group/element) or (empty($group/target) and ($group/element[@code]))) then $check else $error
        
        (: source and target codesystem can not have the same value :)
        let $error          := $check || ' SOURCE.codeSystem and TARGET.codeSystem cannot have the same value. Found: for both codeSystems ''' || $group/source/@codeSystem || '''.' 
        let $check          := if ($group/source/@codeSystem eq $group/target/@codeSystem) then $error else $check
        
        (: check source and target codesystem :)
        let $check          := $check || ruleslib:checkConceptMapGroupCodeSystem($group/source, $conceptMap/sourceScope, $decor, $operation)
        let $check          := if ($group/target) then $check || ruleslib:checkConceptMapGroupCodeSystem($group/target, $conceptMap/targetScope, $decor, $operation) else ()
        
        return if($check) then ruleslib:getErrorString($group, $operation, $path) || ' following errors occur on codeSystem ''' || $group/source/@codeSystem || ''':' || $check else ()
        
    return if (empty($check)) then () else string-join($check)
};

declare %private function ruleslib:checkConceptMapGroupCodeSystem($sourceOrTarget as element(), $scope as element(), $decor as element(), $operation as xs:string) as xs:string* {

    let $csid               := $sourceOrTarget/@codeSystem
    let $csed               := $sourceOrTarget/@codeSystemVersion
    let $check              :=
        if (utillib:isOid($csid)) then 
        (   
            let $codeSystem := utilvs:getValueSet($decor/@prefix, (), (), $scope/@ref, $scope/@flexibility, false(), true())/sourceCodeSystem[@id = $csid]
            return    
                (: codeSystem is found as a sourceCodeSystem in scope valueset :)
                if (empty($codeSystem)) then ' ' || upper-case(name($sourceOrTarget)) || '.codeSystem ''' || $sourceOrTarget/@codeSystem || ''' does not resolve to a sourceCcodeSystem of valueSet ''' || $scope/@ref || '''.' else ()
        )
        
        else ' Input SHALL have ' || name($sourceOrTarget) || '.codeSystem as an OID on group under value. Found: ''' || $csid || '''.'
    
    return $check

};

(: the complete check of a conceptMap before creating or updating :)
declare function ruleslib:checkConceptMap($conceptMap as element(conceptMap), $decor as element(decor), $operation as xs:string) as xs:string* {

    (: first check the DECOR.xsd and stop if not compliant :)
    let $check              := ruleslib:checkDecorSchema($conceptMap, $operation)
    
    return
    for $node in $conceptMap/*
        return
            switch (name($node))
            case 'desc' 
            case 'purpose'
            case 'copyright' return ruleslib:checkFreeFormMarkupWithLanguage($node, $operation, ())
            case 'publishingAuthority' return ruleslib:checkPublishingAuthority($node, $operation, ())
            case 'jurisdiction' return ruleslib:checkValueCodingType($node, $operation, ())
            case 'sourceScope' return ruleslib:checkConceptMapScope($node, $conceptMap/targetScope, $decor, $operation, ())
            case 'targetScope' return ruleslib:checkConceptMapScope($node, $conceptMap/sourceScope, $decor, $operation, ())
            case 'group' return ruleslib:checkConceptMapGroup($conceptMap, $decor, $operation, ())
            default return () 

};

(: ==================================== VALUESET RULES ============================================ :)

(: check is before preparing items to conceptlist - for now only on put call :)
declare function ruleslib:checkValueSetItems($id as xs:string, $valueSet as element(valueSet), $operation as xs:string) {

    let $check                  := 
        if ($valueSet//exclude[@codeSystem][empty(@code)] | $valueSet//items[@is = 'exclude'][@codeSystem][empty(@code)]) then (
            error($errors:BAD_REQUEST, 'Value set with exclude on complete code system not supported. Exclude SHALL have both @code and @codeSystem')
        ) else ()
    
    let $css                    := $valueSet//(include | exclude |items[@is=('include', 'exclude')])[@codeSystem][empty(@code)]/@codeSystem
    let $check                  := 
        if (count($css) = count(distinct-values($css))) then () else (
             error($errors:BAD_REQUEST, 'Value set SHALL contain max 1 include per complete code system.')
        )
    
    let $css                    := $valueSet//(include | exclude |items[@is=('include', 'exclude')])/@ref
    let $check                  := 
        if (count($css) = count(distinct-values($css))) then () else (
            error($errors:BAD_REQUEST, 'Value set SHALL contain max 1 include/exclude per value set regardless of its version.')
        )
    let $check                  := 
        if ($css = $id) then 
            error($errors:BAD_REQUEST, 'Value set SHALL NOT contain itself as inclusion.')
        else ()
    
    return ()

};

(: ==================================== TEMPLATE RULES ============================================ :)

(: ==================================== CONCEPT RULES ============================================= :)

(: check is before preparing concept items or groups - for now only on put call :)
declare function ruleslib:checkConcept($concept as element(concept), $storedConcept as element(concept), $operation as xs:string) {

    let $projectPrefix          := $storedConcept/ancestor::decor/project/@prefix
    
    let $check                  :=
        for $n in $concept/descendant-or-self::concept[not(name | inherit | contains)]
        return
            error($errors:BAD_REQUEST, 'Concept and conceptList.concept SHALL have at least one name if it does not inherit or contains')
    
    let $check                  :=
        if ($concept//conceptList/concept[inherit | contains]) then 
            error($errors:BAD_REQUEST, 'ConceptList.concept SHALL NOT have inherit or contains')
        else ()
    
    let $checkVal               := $concept/descendant-or-self::concept[@type = $ruleslib:CONCEPT-TYPE/enumeration/@value][not(ancestor::conceptList)]
    let $check                  :=
        if ($checkVal) then () else (
            error($errors:BAD_REQUEST, 'Concept type ' || $concept/@type || ' SHALL be one of ' || string-join($ruleslib:CONCEPT-TYPE/enumeration/@value, ', '))
        )
    
    let $check                  :=
        if ($concept[not(@type = 'item')] | $concept[@type = 'item'][inherit | contains | valueDomain]) then () else (
            error($errors:BAD_REQUEST, 'Concept type ' || $concept/@type || ' SHALL have an inherit, a contains or valueDomain.')
        )
    
    let $check                  :=
        for $n in $concept//inherit[not(@ref = $storedConcept//inherit/@ref)]
        return
            if ($storedConcept/ancestor-or-self::concept[@id = $n/@ref][@effectiveDate = $n/@effectiveDate]) then
                error($errors:BAD_REQUEST, 'Concept ' || local-name($n) || ' SHALL NOT point to one of its parents (circular reference). Found: ref=''' || $n/@ref || ''' effectiveDate=''' || $n/@effectiveDate || '''')
            else
            if ($n[utillib:isOid(@ref)][@effectiveDate castable as xs:dateTime]) then
                if ($setlib:colDecorData//concept[@id = $n/@ref][@effectiveDate = $n/@effectiveDate]) then () else (
                    error($errors:BAD_REQUEST, 'Concept ' || local-name($n) || ' SHALL point to an existing concept. Found: ref=''' || $n/@ref || ''' effectiveDate=''' || $n/@effectiveDate || '''')
                )
            else (
                error($errors:BAD_REQUEST, 'Concept ' || local-name($n) || ' SHALL have both ref as oid and effectiveDate as dateTime. Found: ref=''' || $n/@ref || ''' effectiveDate=''' || $n/@effectiveDate || '''')
            )
    
    let $check                  :=
        for $n in $concept//contains[not(@ref = $storedConcept//contains/@ref)]
        return
            if ($n[utillib:isOid(@ref)][@flexibility castable as xs:dateTime]) then
                if ($setlib:colDecorData//concept[@id = $n/@ref][@effectiveDate = $n/@effectiveDate]) then () else (
                    error($errors:BAD_REQUEST, 'Concept ' || local-name($n) || ' SHALL point to an existing concept. Found: ref=''' || $n/@ref || ''' flexibility=''' || $n/@flexibility || '''')
                )
            else (
                error($errors:BAD_REQUEST, 'Concept ' || local-name($n) || ' SHALL have both ref as oid and flexibility as dateTime. Found: ref=''' || $n/@ref || ''' flexibility=''' || $n/@flexibility || '''')
            )
    
    let $valueDomain            := $concept//valueDomain[not(ancestor::concept[1][inherit | contains])]
    
    let $check                  := if ($valueDomain) then ruleslib:checkConceptValueDomain($valueDomain, $concept, 'update', ()) else ()
    
    let $check                  :=
        for $clid in $storedConcept//valueDomain/conceptList[not(@id = $concept//valueDomain/conceptList/@id)]/@id
        let $referred           := $setlib:colDecorData//conceptList[@ref = $clid][ancestor::datasets][not(ancestor::history)][ancestor::decor[project[@prefix=$projectPrefix]]]
        return
            if ($referred) then
                error($errors:BAD_REQUEST, 'Concept uses a new valueDomain conceptList id ' || $clid ||' but this is still being referred to by ' || count($referred) || ' concepts. Referring concepts: ' || string-join($referred/concat('concept ', @id, ' / ', @effectiveDate), ', '))
            else ()
    
    let $check                  :=
        for $clid in $valueDomain/conceptList[not(@ref = $storedConcept//valueDomain/conceptList/@ref)]/@ref
        return
            if ($setlib:colDecorData//conceptList/concept[@id = $clid][ancestor::datasets][not(ancestor::history)][ancestor::decor[project[@prefix=$projectPrefix]]]) then () else (
                error($errors:BAD_REQUEST, 'Concept valueDomain conceptList with ref ' || $clid || ' SHALL refer to a conceptList in this project ' || $projectPrefix)
            )
    
    return ()
};

declare function ruleslib:checkConceptValueDomain($valueDomain as element(valueDomain), $concept as element(concept), $operation as xs:string, $path as xs:string?) {

    let $check              := () 
    
    (: valuedomain types has to be supported :)
    let $error              := $check || ' Concept.valueDomain.type has unsupported value ' || string-join($valueDomain/@type, ', ') || '. SHALL be one of: ' || string-join($ruleslib:VALUEDOMAIN-TYPE/enumeration/@value, ', ')
    let $check              := if ($valueDomain/@type = $ruleslib:VALUEDOMAIN-TYPE/enumeration/@value) then $check else $error

    (: valuedomain example types has to be supported :)
    let $error              := $check || ' Concept.valueDomain.example.type has unsupported value ' || string-join($valueDomain/example[not(@type = $ruleslib:EXAMPLE-TYPE)]/@type, ' or ') || '. SHALL be one of: ' || string-join($ruleslib:EXAMPLE-TYPE/enumeration/@value, ', ')
    let $check              := if (count($valueDomain/example/@type) = count($valueDomain/example[@type = $ruleslib:EXAMPLE-TYPE/enumeration/@value])) then $check else $error

    (: valuedomain properties to be supported :)
    let $error              := $check || ' Concept.valueDomain.property SHALL NOT have unsupported values. Found: ' || string-join($valueDomain/property/@*[not(name() = $ruleslib:VALUEDOMAIN-PROPS/@name)]/name(), ', ')
    let $check              := if ($valueDomain/property[@*[not(name() = $ruleslib:VALUEDOMAIN-PROPS/@name)]]) then $error else $check
    
    let $check              := if ($valueDomain[@type = 'code']) then (
        (: coded valueDomain shall have a conceptList :)
        let $error          := $check || ' Coded valueDomain SHALL have a conceptList.'
        return if ($valueDomain[@type = 'code']/conceptList[@id | @ref]) then $check else if ($concept/@statusCode = 'new') then $check else $error
        ,
        (: if there are no conceptList/@ids assume fine, if they are the same as the current conceptList/@ids, assume fine, else check unique :)
        for $clid in $valueDomain/conceptList[not(@id = $concept/valueDomain/conceptList/@id)]/@id
        let $error          := $check || ' Concept valueDomain conceptList SHALL have a unique oid. Recommended pattern: [concept id].[concept effectiveDate as yyyyMMddhhmmss].0. Found: ' || $clid || '.'
        return if (utillib:isOid($clid) and empty($setlib:colDecorData//conceptList/concept[@id = $clid][ancestor::datasets][not(ancestor::history)])) then $check else $error
        ,
        (: if there are no conceptList/concept/@ids assume fine, if they are the same as the current conceptList/@ids, assume fine, else check unique :)
        for $clid in $valueDomain/conceptList/concept[not(@id = $concept/valueDomain/conceptList/concept/@id)]/@id
        let $error              := $check || ' Concept valueDomain conceptList concept SHALL have a unique oid. Recommended pattern: [concept id].[concept effectiveDate as yyyyMMddhhmmss].[n]. Found: ' || $clid || '.'
        return if (utillib:isOid($clid) and empty($setlib:colDecorData//@id[. = $clid] | $setlib:colDecorData//@codeSystem[. = $clid])) then $check else $error
    )
    else $check
    
    (: concept valueDomain shall not have more than 1 conceptList :)
    let $error              := $check || 'Concept valueDomain SHALL NOT have more than 1 conceptList'
    let $check              := if ($valueDomain[count(conceptList) le 1]) then $check else $error
    
    (: concept valueDomain shall have unique ids on all its concepts :)
    let $checkVal           := $valueDomain/conceptList/concept/@id
    let $error              := $check || 'Concept valueDomain SHALL HAVE unique ids on all its concepts. Duplicate(s) found: ' || string-join(distinct-values($checkVal[count(index-of($checkVal, .)) gt 1]), ', ')
    let $check              := if (count($checkVal) = count(distinct-values($checkVal))) then $check else $error

    return if ($check) then ruleslib:getErrorString($valueDomain, $operation, $path) || $check  else ()  

};

(: ================================TRANSACTION CONCEPT RULES ====================================== :)

declare function ruleslib:checkTransactionConcept($concept as element(concept), $operation as xs:string) {
    let $check                  :=
        if ($concept[@minimumMultiplicity | @maximumMultiplicity | @conformance]) then () else (
            error($errors:BAD_REQUEST, 'Concept SHALL have one of minimumMultiplicity | maximumMultiplicity | conformance')
        )
    
    let $check                  :=
        if ($concept//@conformance[not(. = ($ruleslib:CONFORMANCE-TYPE/enumeration/@value, 'M'))]) then
            error($errors:BAD_REQUEST, 'Concept conformance ' || string-join($concept//@conformance[not(. = $ruleslib:CONFORMANCE-TYPE/enumeration/@value)], ' and ') || ' SHALL be one of ' || string-join($ruleslib:CONFORMANCE-TYPE/enumeration/@value, ', '))
        else ()
    
    let $check                  :=
        switch ($concept/@conformance)
        case 'C' return (
            if ($concept[@minimumMultiplicity | @maximumMultiplicity | @isMandatory]) then
                error($errors:BAD_REQUEST, 'Concept SHALL NOT have conformance C(onditional) and specify generic minimumMultiplicity, maximumMultiplicity or isMandatory')
            else (),
            if ($concept[condition]) then () else (
                error($errors:BAD_REQUEST, 'Concept with conformance C(onditional) SHALL have condition elements specifying the conditions')
            ),
            if ($concept/condition[@conformance = 'C']) then
                error($errors:BAD_REQUEST, 'Concept condition SHALL NOT have conformance C(onditional)')
            else (),
            if ($concept/condition[not(desc)]) then
                error($errors:BAD_REQUEST, 'Concept condition SHALL have a description outlining the circumstances')
            else (),
            if ($concept/condition[not(count(desc) = count(distinct-values(desc/@language)))]) then
                error($errors:BAD_REQUEST, 'Concept condition descriptions SHALL each have a unique language')
            else ()
        )
        case 'NP' return (
            if ($concept[condition | @minimumMultiplicity[not(. = '0')] | @maximumMultiplicity[not(. = '0')] | @isMandatory[not(. = 'false')]]) then
                error($errors:BAD_REQUEST, 'Concept with conformance Not Present (NP) SHALL NOT have conditions or any of minimumMultiplicity | maximumMultiplicity | isMandatory')
            else ()
        )
        default return (
            if ($concept[condition]) then
                error($errors:BAD_REQUEST, 'Concept SHALL NOT have condition elements unless it has conformance C(onditional)')
            else ()
        )
    
    let $check                  :=
        if ($concept[count(context) = count(distinct-values(context/@language))]) then () else (
            error($errors:BAD_REQUEST, 'Concept contexts SHALL each have a unique language')
        )
    
    let $check                  :=
        if ($concept//@minimumMultiplicity[not(. castable as xs:integer or . = '?')]) then
            error($errors:BAD_REQUEST, 'Concept minimumMultiplicity SHALL be an integer. Found: ' || string-join($concept//@minimumMultiplicity[not(. castable as xs:integer or . = '?')], ' and '))
        else ()
    
    let $check                  :=
        if ($concept//@maximumMultiplicity[not(. castable as xs:integer or . = ('*', '?'))]) then
            error($errors:BAD_REQUEST, 'Concept maximumMultiplicity SHALL be an integer or '*'. Found: ' || string-join($concept//@maximumMultiplicity[not(. castable as xs:integer or . = ('*', '?'))], ' and '))
        else ()
    
    let $minmax                 := $concept/descendant-or-self::*[@minimumMultiplicity castable as xs:integer][@maximumMultiplicity castable as xs:integer][xs:integer(@minimumMultiplicity) gt xs:integer(@maximumMultiplicity)]
    let $check                  :=
        if ($minmax) then
            error($errors:BAD_REQUEST, 'Concept minimumMultiplicity SHALL be &lt;= maximumMultiplicity. Found: ' || string-join($minmax/concat(@minimumMultiplicity, '..', @maximumMultiplicity), ' and '))
        else ()
    
    let $check                  := 
        for $man in $concept/descendant-or-self::*[@isMandatory = 'true'] | $concept/descendant-or-self::*[@conformance = 'M']
        return
            if ($man[condition] | $man[empty(@minimumMultiplicity)] | $man[@minimumMultiplicity = '0']) then
                error($errors:BAD_REQUEST, 'Concept that is mandatory SHALL NOT have conditions and SHALL have minimumMultiplicity != 0')
            else ()
    (: terminologyAssociation :)
    let $check                  :=
        for $assoc in $concept/terminologyAssociation
        return (
            if ($assoc[@conceptId]) then () else (
                error($errors:BAD_REQUEST, 'Concept terminologyAssociation SHALL have a conceptId')
            ),
            if ($assoc[@conceptFlexibility]) then
                if ($assoc/@conceptFlexibility[. castable as xs:dateTime]) then () else (
                    error($errors:BAD_REQUEST, 'Concept terminologyAssociation conceptFlexibility SHALL be yyyy-mm-ddThh:mm:ss. Found: ' || $assoc/@conceptFlexibility)
                )
            else (),
            if ($assoc[@valueSet][@conceptFlexibility]) then 
                error($errors:BAD_REQUEST, 'Concept terminologyAssociation with a valueSet SHALL NOT have conceptFlexibility as valueDomain.conceptList does not have an effectiveDate to point to')
            else (),
            if ($assoc[@valueSet][@code | @codeSystem | @codeSystemName | @codeSystemVersion]) then
                error($errors:BAD_REQUEST, 'Concept terminologyAssociation SHALL NOT have a valueSet and one of code | codeSystem | codeSystemName | codeSystemVersion')
            else (),
            if ($assoc[@valueSet][@flexibility]) then () else if ($assoc[@flexibility]) then 
                error($errors:BAD_REQUEST, 'Concept terminologyAssociation SHALL NOT have a flexibility without valueSet')
            else (),
            if ($assoc[@flexibility]) then
                if ($assoc/@flexibility[. castable as xs:dateTime] | $assoc/@flexibility[. = 'dynamic']) then () else (
                    error($errors:BAD_REQUEST, 'Concept terminologyAssociation flexibility SHALL be yyyy-mm-ddThh:mm:ss or be dynamic. Found: ' || $assoc/@flexibility)
                )
            else (),
            if ($assoc[@effectiveDate]) then
                if ($assoc/@effectiveDate[. castable as xs:dateTime]) then () else (
                    error($errors:BAD_REQUEST, 'Concept terminologyAssociation effectiveDate SHALL be yyyy-mm-ddThh:mm:ss. Found: ' || $assoc/@effectiveDate)
                )
            else ()
        )
    (: identifierAssociation :)
    let $check                  :=
        for $assoc in $concept/identifierAssociation
        return (
            if ($assoc[@conceptId]) then () else (
                error($errors:BAD_REQUEST, 'Concept identifierAssociation SHALL have a conceptId')
            ),
            if ($assoc[@conceptFlexibility]) then
                if ($assoc/@conceptFlexibility[. castable as xs:dateTime]) then () else (
                    error($errors:BAD_REQUEST, 'Concept identifierAssociation conceptFlexibility SHALL be yyyy-mm-ddThh:mm:ss. Found: ' || $assoc/@conceptFlexibility)
                )
            else (),
            if ($assoc[@effectiveDate]) then
                if ($assoc/@effectiveDate[. castable as xs:dateTime]) then () else (
                    error($errors:BAD_REQUEST, 'Concept identifierAssociation effectiveDate SHALL be yyyy-mm-ddThh:mm:ss. Found: ' || $assoc/@effectiveDate)
                )
            else ()
        )
    return ()
};


