xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilde                 = "http://art-decor.org/ns/api/util-concept";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace decorlib        = "http://art-decor.org/ns/api/decor" at "decor-lib.xqm";
import module namespace histlib         = "http://art-decor.org/ns/api/history" at "history-lib.xqm";
import module namespace ruleslib        = "http://art-decor.org/ns/api/rules" at "rules-lib.xqm";
import module namespace utiltcs         = "http://art-decor.org/ns/api/util-terminology-codesystem" at "util-terminology-codesystem-lib.xqm";
import module namespace utilvs          = "http://art-decor.org/ns/api/util-valueset" at "util-valuest-lib.xqm";
import module namespace utiluser        = "http://art-decor.org/ns/api/util-user" at "util-user-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

(:~ All functions support their own override, but this is the fallback for the maximum number of results returned on a search :)
declare %private variable $utilde:maxResults           := 50;
declare %private variable $utilde:STATUSCODES-FINAL    := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $utilde:STATUSCODES-SKIPS    := ('rejected', 'cancelled');
declare %private variable $utilde:CONCEPT-TYPE         := utillib:getDecorTypes()/DataSetConceptType;
declare %private variable $utilde:VOCAB-TYPE           := utillib:getDecorTypes()/VocabType;
declare %private variable $utilde:VALUEDOMAIN-TYPE     := utillib:getDecorTypes()/DataSetValueType;
declare %private variable $utilde:EXAMPLE-TYPE         := utillib:getDecorTypes()/ExampleType;
declare %private variable $utilde:CODINGSTRENGTH-TYPE  := utillib:getDecorTypes()/CodingStrengthType;
declare %private variable $utilde:EQUIVALENCY-TYPE     := utillib:getDecorTypes()/EquivalencyType;

declare %private variable $utilde:DEBUG                 := false();

(:~ Retrieves exact DECOR concept based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                          - required parameter denoting the id of the concept
    @param $effectiveDate               - required parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $transactionId               - required parameter denoting the id of the transaction that the concept is in
    @param $transactionEffectiveDate    - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion              - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage             - optional parameter to select from a specific compiled language
    @param $treeonly                    - optional boolean parameter to get the tree structure only if true
    @param $fullTree                    - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function utilde:getConceptTree($conceptId as xs:string?, $conceptEffectiveDate as xs:string?, $transactionId as xs:string?, $transactionEffectiveDate as xs:string?, $fullTree as xs:boolean, $propertiesMap as map(*)?) as element(dataset)? {
    
    let $concept                := utilde:getConcept($conceptId, $conceptEffectiveDate)[1]
    let $representingTemplate   := if ($transactionId) then utillib:getTransaction($transactionId, $transactionEffectiveDate)/representingTemplate else ()
    
    let $projectPrefix          := $concept/ancestor::decor/project/@prefix
    let $defaultLanguage        := $concept/ancestor::decor/project/@defaultLanguage
    let $projectLanguages       := $concept/ancestor::decor/project/name/@language
    
    let $conceptBasic           :=
        if ($concept) then 
            if ($representingTemplate) then utilde:transactionConceptBasics($concept, $representingTemplate, $fullTree, $defaultLanguage, $propertiesMap) else utilde:conceptBasics($concept, $propertiesMap)
        else ()
        
    return utilde:copy2concept($conceptBasic)    
};

declare %private function utilde:copy2concept($node as node()*) as node()* {
    for $e in $node
    return
        <concept>
        {
            $e/@*,
            $e/name,
            utilde:copy2concept($e/concept)
        }
        </concept>
};

(:~ Function for getting all details of a concept without its children if any. In the case of a concept that inherits/contains, 
    the whole concept is in fact readonly, and the only property that is actually of this concept, might be comment. To distinguish 
    between this concepts comments and those of the original, the original comments are added as "inheritedComment".
        
    Originally this code was part of package art in modules/get-decor-concept.xq
    @param $concept required. dataset or transaction concept
    @param $associationMode. required. Include associations or not
:)
declare function utilde:getConceptExpanded($concept as element(concept)?, $associationMode as xs:boolean) as element(concept)? {
    let $projectVersion         := $concept/ancestor::decor/@versionDate
    let $projectId              := $concept/ancestor::decor/project/@id
    let $projectPrefix          := $concept/ancestor::decor/project/@prefix
    let $language               := $concept/ancestor::decor/project/@defaultLanguage
    
    let $trid                   := $concept/ancestor::transaction[1]/@id
    let $tred                   := $concept/ancestor::transaction[1]/@effectiveDate
        
    let $transactionConcept     := if ($concept/ancestor::transaction) then $concept else ()
    let $concept                := if ($concept/ancestor::transaction) then utilde:getConcept($concept/@ref, $concept/@flexibility, $projectVersion, $concept/ancestor::decor/@language) else $concept

    let $associations           :=
        if ($associationMode) then 
            if ($transactionConcept) then utilde:getConceptAssociations($transactionConcept, utilde:getOriginalForConcept($concept), false(), 'all', true(), $language) 
            else if ($concept) then utilde:getConceptAssociations($concept, utilde:getOriginalForConcept($concept), false(), 'normal', true(), $language) 
            else ()
        else ()
    
    (:  usually you contain/inherit from the original, but if you contain/inherit from something that inherits, then these two will differ
        originalConcept has the type and the associations. The rest comes from the inheritConcept/containConcept :)
    let $inheritConcept         := if ($concept[inherit]) then utilde:getConcept($concept/inherit/@ref, $concept/inherit/@effectiveDate) else ()
    let $containConcept         := if ($concept[contains]) then utilde:getConcept($concept/contains/@ref, $concept/contains/@flexibility) else ()
    let $originalConcept        := utilde:getOriginalForConcept($concept)

    let $communityInfo          := if ($projectId) then decorlib:getDecorCommunity((), $projectId, $projectVersion)//association[object[@ref = ($concept/@id, $originalConcept/@id)][@flexibility = ($concept/@effectiveDate, $originalConcept/@effectiveDate)]] else ()

    let $iddisplay              := utillib:getNameForOID($concept/@id, $language, $concept/ancestor::decor)

    return
    if (empty($concept)) then () else
    if (empty($projectVersion)) then
        <concept>
        {
            $concept/@id,
            $concept/@effectiveDate,
            $concept/@statusCode,
            $originalConcept/@type,
            $concept/@expirationDate,
            $concept/@officialReleaseDate,
            $concept/@versionLabel,
            $concept/@canonicalUri,
            $concept/@lastModifiedDate,
            if (string-length($iddisplay) = 0) then () else attribute iddisplay {$iddisplay},
            if ($transactionConcept) then (
                attribute minimumMultiplicity {if ($transactionConcept) then utillib:getMinimumMultiplicity($transactionConcept) else ('0')},
                attribute maximumMultiplicity {if ($transactionConcept) then utillib:getMaximumMultiplicity($transactionConcept) else ('*')},
                if ($transactionConcept/@isMandatory='true') then attribute conformance {'M'} else $transactionConcept/@conformance[not(.='')],
                attribute isMandatory {$transactionConcept/@isMandatory='true'},
                $transactionConcept/@enableBehavior[not(.='')]
            )
            else ()
            ,
            if ($transactionConcept) then (
                 $transactionConcept/context,
                (: element children, not concept &amp; history children, filtered for language :)
                for $condition in $transactionConcept/condition
                return
                <condition>
                {
                    attribute minimumMultiplicity {utillib:getMinimumMultiplicity($condition)},
                    attribute maximumMultiplicity {utillib:getMaximumMultiplicity($condition)},
                    if ($condition/@isMandatory='true') then attribute conformance {'M'} else $condition/@conformance[not(.='')],
                    attribute isMandatory {$condition/@isMandatory='true'},
                    $condition/desc,
                    if ($condition[desc]) then () else if ($condition[normalize-space() = '']) then () else <desc language="{$language}">{$condition/text()}</desc>
                }
                </condition>
                ,
                $transactionConcept/enableWhen
                )
            else ()
            ,
            $associations/*[@conceptId = ($concept/@id | $concept/inherit/@ref | $concept/contains/@ref | $originalConcept/@id)],
            (: we return just a count of issues and leave it up to the calling party to get those if required :)
            <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate])}"/>       
            ,
            if ($concept[inherit]) then utilde:conceptExpandedInherit($concept/inherit, $inheritConcept, $originalConcept) else ()
            ,
            if ($concept[contains]) then utilde:conceptExpandedContains($concept/contains, $containConcept, $originalConcept)
            else if ($inheritConcept[contains]) then utilde:conceptExpandedContains($inheritConcept/contains, $inheritConcept, utilde:getOriginalForConcept($inheritConcept)) else ()
            ,
            $originalConcept/name,
            $originalConcept/synonym,
            $originalConcept/desc,
            $originalConcept/source,
            $originalConcept/rationale
            ,
            if ($concept[inherit | contains]) then 
                for $comment in $originalConcept/comment
                return 
                    <inheritedComment>
                    {
                        $comment/@*, 
                        $comment/*
                    }
                    </inheritedComment>
            else ()
            ,
            $concept/comment,
            $originalConcept/property
            ,
            for $relationship in $originalConcept/relationship
            let $referredConcept        := utilde:getConcept($relationship/@ref, $relationship/@flexibility)
            let $originalReferredConcept:= utilde:getOriginalForConcept($referredConcept)[1]
            return utilde:conceptExpandedRelationship($relationship, $referredConcept, $originalReferredConcept)
            ,
            $originalConcept/operationalization,
            for $valueDomain in $originalConcept/valueDomain
            return
                <valueDomain>
                {
                    $valueDomain/@*,
                    for $conceptList in $valueDomain/conceptList 
                    let $originalConceptList := utilde:getOriginalConceptList($conceptList)
                    return
                        <conceptList>
                        {
                            (:note that this will retain conceptList[@ref] if applicable:)
                            $conceptList/(@* except @conceptId),
                            if ($conceptList[@ref] and $originalConceptList) then (
                                attribute prefix {$originalConceptList/ancestor::decor/project/@prefix},
                                attribute conceptId {$originalConceptList/ancestor::concept[1]/@id},
                                attribute conceptEffectiveDate {$originalConceptList/ancestor::concept[1]/@effectiveDate},
                                attribute datasetId {$originalConceptList/ancestor::dataset[1]/@id},
                                attribute datasetEffectiveDate {$originalConceptList/ancestor::dataset[1]/@effectiveDate}
                            )
                            else ()
                            ,
                            $associations/*[@conceptId = $conceptList/(@ref | @id)]
                            ,
                            for $conceptListNode in $originalConceptList/*
                            return
                                if ($conceptListNode/self::concept) then (
                                    <concept>
                                    {
                                        $conceptListNode/@*,
                                        $associations/*[@conceptId = $conceptListNode/@id],
                                        $conceptListNode/name,
                                        $conceptListNode/synonym,
                                        $conceptListNode/desc
                                    }
                                    </concept>
                                )
                                else $conceptListNode
                        }
                        </conceptList>
                    ,
                    $valueDomain/property[@*[string-length() gt 0]]
                    ,
                    for $ex in $valueDomain/example
                    return
                        <example type="{($ex/@type, 'neutral')[1]}">{$ex/@caption, $ex/node()}</example>
                }
                </valueDomain>
            ,
            utilde:conceptExpandedCommunity($communityInfo),
            $concept/history
        }
        </concept>
    else (
        (: already compiled, don't redo :)
        <concept>
        {
            $concept/@*, 
            $concept/node(),
            if ($concept/community) then () else utilde:conceptExpandedCommunity($communityInfo)
        }
        </concept>
    )
};

(:~ Retrieves DECOR dataset concept for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the concept
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $datasetOrtransactionId  - required parameter denoting the id of the dataset or transaction that the concept is in
    @param datasetOrtransactionE    - optional parameter denoting the effectiveDate of the dataset or transaction. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @return as-is
    @since 2024-11-08
:)
declare function utilde:getConceptForExtract($id as xs:string, $effectiveDate as xs:string?, $datasetOrTransactionId as xs:string?, $datasetOrTransactionEd as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?, $communityName as xs:string*) as element()* {
    
    let $dataset                := 
        if ($datasetOrTransactionId) then 
            utillib:getDataset($datasetOrTransactionId, $datasetOrTransactionEd, $projectVersion) | 
            utillib:getTransaction($datasetOrTransactionId, $datasetOrTransactionEd, $projectVersion)
        else utilde:getConcept($id, $effectiveDate, $projectVersion)/ancestor::dataset
    
    return
        if (empty($dataset)) then () else 
 
            let $datasetExtract := 
                if ($datasetOrTransactionId) then 
                    utillib:getDatasetExtract($dataset, $datasetOrTransactionId, $datasetOrTransactionEd, $id, $effectiveDate,$projectVersion, $projectLanguage, $communityName)
                else utillib:getDatasetExtract($dataset, $id, $effectiveDate, $id, $effectiveDate,$projectVersion, $projectLanguage, $communityName)
            let $latestVersion  := utillib:getDataset($dataset/@id, ())/@effectiveDate = $dataset/@effectiveDate
            let $transactions   := $dataset/ancestor::decor//transaction[representingTemplate[@sourceDataset = $datasetExtract/@id][@sourceDatasetFlexibility = $datasetExtract/@effectiveDate]]
            let $transactions   := 
                if ($latestVersion) then 
                    $transactions | $dataset/ancestor::decor//transaction[representingTemplate[@sourceDataset = $datasetExtract/@id][not(@sourceDatasetFlexibility castable as xs:dateTime)]] 
                else $transactions
                
            return
                utillib:mergeDatasetWithUsage($datasetExtract, $transactions, $dataset/ancestor::decor/project/@prefix)    
};

(:~ Retrieves an existing dataset concept for edit. This means the user needs to be an editing author of the project, and setting a lock
    @param $datasetId               - required. dataset id to insert concept into.
    @param $datasetEffectiveDate    - required. dataset effectiveDate to insert concept into.
    @param $conceptBaseId           - optional. baseId to create the new concept id out of. Defaults to defaultBaseId for type DE if omitted
    @param $conceptType             - required. 'item' or 'group'
    @param $insertMode              - required. 'into' or 'preceding' or 'following'. For preceding and for following, $insertRef is required
    @param $insertRef               - optional. concept/@id reference for insert. Inserts as new concept in dataset if empty
    @param $insertFlexibility       - optional. concept/@effectiveDate reference for insert. Only relevant when two versions of the same concept are in the same dataset which is logically highly unlikely
:)
declare function utilde:getConceptForEdit($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $targetId as xs:string?, $targetEffectiveDate as xs:string?, $targetType as xs:string?, $generateConceptListIds as xs:boolean?, $breaklock as xs:boolean?, $associationMode as xs:boolean?) {
    
    let $concept                := utilde:getConcept($id, $effectiveDate, (), ())
    let $decor                  := $concept/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($concept) then () else (
            error($errors:BAD_REQUEST, 'Concept with id ' || $id || ' and effectiveDate ' || $effectiveDate || ', does not exist. Cannot edit non-existent concept.')
        )
    
    let $lock                   := utilde:checkConceptAccess($authmap, $decor, $id, $effectiveDate, true(), $breaklock)
    
    let $check                  :=
        if ($concept/@statusCode = $utilde:STATUSCODES-FINAL) then
            error($errors:BAD_REQUEST, concat('Concept cannot be edited in status ', $concept/@statusCode, '. ', if ($concept/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
        else ()
    
    let $targetConcept          := 
        if ($targetType = 'to-designcopy') then if ($concept[contains]) then utilde:getConcept($concept/contains/@ref, $concept/contains/@flexibility, (), ()) else () else
        if ($targetType = 'to-copy')       then if ($concept[inherit]) then utilde:getConcept($concept/inherit/@ref, $concept/inherit/@effectiveDate, (), ()) else 
                                                if ($concept[contains]) then utilde:getConcept($concept/contains/@ref, $concept/contains/@flexibility, (), ()) else ()  else
        if (empty($targetId)) then () else utilde:getConcept($targetId, $targetEffectiveDate, (), ())
    
    let $check                  :=
        if ($targetConcept) then
            if ($targetConcept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]) then
                error($errors:BAD_REQUEST, 'Concept SHALL NOT be the same as the target concept.')
            else ()
        else ()
   
    let $check                  :=
        switch ($targetType)
        (: new inherit :)
        case 'designcopy' return (
            if (empty($targetConcept)) then
                error($errors:BAD_REQUEST, 'Parameter targetType ''' || $targetType || ''' SHALL have a targetId/targetEffectiveDate to an existing concept. Otherwise we cannot know what to target.')
            else
            if ($concept/ancestor-or-self::concept[@id = $targetConcept/@id][@effectiveDate = $targetConcept/@effectiveDate]) then
                error($errors:BAD_REQUEST, 'Target concept for inherit id=''' || $targetConcept/@id || ''' effectiveDate=''' || $targetConcept/@effectiveDate || ''' SHALL NOT point to one of the parents of the source concept (circular reference).')
            else ()
        )
        (: new contains :)
        case 'containment' return (
            if (empty($targetConcept)) then
                error($errors:BAD_REQUEST, 'Parameter targetType ''' || $targetType || ''' SHALL have a targetId/targetEffectiveDate to an existing concept. Otherwise we cannot know what to target.')
            else ()
        )
        (: from contains to inherit :)
        case 'to-designcopy' return (
            if (empty($concept/contains/@ref)) then
                error($errors:BAD_REQUEST, 'Parameter targetType ''' || $targetType || ''' SHALL be used on a concept that has a containment. Otherwise we cannot convert anything.')
            else
            if (empty($targetConcept)) then
                error($errors:BAD_REQUEST, 'Parameter targetType ''' || $targetType || ''' SHALL be used on a concept that has a containment we can find. Otherwise we cannot convert anything.')
            else ()
        )
        (: from inherit to relationship :)
        case 'to-copy' return (
            if (empty($concept/inherit/@ref)) then
                error($errors:BAD_REQUEST, 'Parameter targetType ''' || $targetType || ''' SHALL be used on a concept that has an inherit/is a design copy. Otherwise we cannot convert anything.')
            else
            if (empty($targetConcept)) then
                error($errors:BAD_REQUEST, 'Parameter targetType ''' || $targetType || ''' SHALL be used on a concept that has an inherit/is a design copy we can find. Otherwise we cannot convert anything.')
            else ()
        )
        default return (
            if (empty($targetConcept)) then () else 
            if (empty($targetType)) then (
                error($errors:BAD_REQUEST, 'Parameter targetType SHALL NOT be empty when a target concept is requested. Expected one of "designcopy", "containment"')
            )
            else (
                error($errors:BAD_REQUEST, 'Parameter targetType ''' || $targetType || ''' is not supported. Expected one of "designcopy", "containment", "to-designcopy", or "to-copy"')
            )
        )
    
    let $userDisplayName        := utiluser:getUserDisplayName($authmap?name)
    
    let $language               := $decor/project/@defaultLanguage
    
    let $level                  := 0
    let $baseId                 := string-join(tokenize($concept/@id,'\.')[position()!=last()],'.') || '.'
    
    let $debug := if($utilde:DEBUG) then util:log('INFO', 'getConceptForEdit: update for type ' || $targetType) else ()
    let $debug := if($utilde:DEBUG) then util:log('INFO', 'getConceptForEdit: concept status ' || $concept/@statusCode) else ()
    let $debug := if($utilde:DEBUG) then util:log('INFO', 'getConceptForEdit: target status ' || $targetConcept/@statusCode) else ()

    (: delete our temp counters for concepts and conceptLists :)
    let $delete                 := update delete $concept/ancestor::datasets/@maxcounter
    let $delete                 := update delete $concept/ancestor::datasets/@maxcounter-cl

    (: 
        KH 20241009: handle concepts but skip cancelled or rejected = $utilde:STATUSCODES-SKIPS concepts
        Also dsapi:createDataset en dsapi:inheritConcept handles these skips for other calls
    :)
    let $response               := if ($targetConcept/@statusCode = $utilde:STATUSCODES-SKIPS) then () 
        else utilde:getConceptForEdit($authmap, $userDisplayName, $decor, $concept, $targetConcept, $targetType, $generateConceptListIds, $level, $baseId, $projectPrefix, $language, $associationMode)
    
    (: delete our temp counters for concepts and conceptLists :)
    let $delete                 := update delete $concept/ancestor::datasets/@maxcounter
    let $delete                 := update delete $concept/ancestor::datasets/@maxcounter-cl
    
    let $update                 :=
        if ($targetConcept and $targetType = ('designcopy', 'containment', 'to-designcopy')) then (
            (: storing the result one more time ensures that the top level concept is written correctly too. 
                without this we will return more info than we have saved in the project. :)
            if ($response[concept]) then (
                (: first delete the stuff that was saved during utilde:getConceptForEdit() :)
                let $delete     := update delete $concept/concept[@statusCode = 'new'][inherit | contains]
                (: now resave using the intended structure but respect existing concepts from before the inherit. This way we can still cancel :)
                let $insert     := if ($concept[concept | history]) then update insert $response/concept preceding ($concept/concept | $concept/history)[1] else update insert $response/concept into $concept
                
                let $storedConcept  := utilde:getConcept($response/@id, $response/@effectiveDate)
                let $debug          := if($utilde:DEBUG) then util:log('INFO', 'getConceptForEdit: before cleanup ' || count($storedConcept) ) else ()
                let $cleanConcept   := utilde:cleanupAfterCopyForEdit($storedConcept)
                let $debug          := if($utilde:DEBUG) then util:log('INFO', 'getConceptForEdit: after cleanup ' || count($cleanConcept) ) else ()
                
                let $replace        := update replace $storedConcept with $cleanConcept
                 
                let $debug          := if($utilde:DEBUG) then util:log('INFO', 'getConceptForEdit: after update replace') else ()
                 
                return ()
            ) else ()
        ) else ()
            
    return $response
};
(:
    containment   - create new contains            - use associations from B (target)
    Input:  <concept id="A"> ??? </concept>
          + <target  id="B"> ??? </target>
    Output: <concept id="A"> <contains ref="B"/> </concept>
    
    designcopy    - create new inherit             - use associations from B (target)
    Input:  <concept id="A"> ??? </concept>
          + <target  id="B"> ??? </target>
    Output: <concept id="A"> <inherit ref="B"/> </concept>
    
    to-designcopy - convert contains to inherit    - use associations from A
    Input:  <concept id="A"> <contains ref="B"/> </concept>
          + <target  id="B"> ??? </target>
    Output: <concept id="A"> <inherit ref="B"/> </concept>
    
    to-copy       - convert inherit to new concept - use associations from A
    Input:  <concept id="A"> <inherit ref="B"/> </concept>
          + <target  id="B"> ??? </target>
    Output: <concept id="A"> ??? </concept>
    
    "default"     - leave concept as-is            - use associations from A
    Input:  <concept id="A"> ??? </concept>
          + <target  id="B"> ??? </target>
    Output: <concept id="A"> ??? </concept>
:)
declare %private function utilde:getConceptForEdit($authmap as map(*), $userDisplayName as xs:string, $decor as element(decor), $concept as element(concept), $targetConcept as element(concept)?, $targetType as xs:string?, $generateConceptListIds as xs:boolean?, $level as xs:integer, $baseId as xs:string, $projectPrefix as xs:string, $language as xs:string, $associationMode as xs:boolean?) as element(concept) {
    
    let $projectLanguages       := $decor/project/name/@language
    
    let $editMode               := if ($concept/@statusCode=('new','draft','pending')) then 'edit' else if ($concept/@statusCode='final') then 'move' else ()
    let $deInherit              := $targetType = 'to-copy'
    
    (:usually you inherit from the original, but if you inherit from something that inherits, then these two will differ
        originalConcept has the type and the associations. The rest comes from the inheritConcept
    :)
    let $inheritConcept         := if ($concept[inherit]) then utilde:getConcept($concept/inherit/@ref, $concept/inherit/@effectiveDate) else ()
    let $containConcept         := if ($concept[contains]) then utilde:getConcept($concept/contains/@ref, $concept/contains/@flexibility) else ()
    let $originalConcept        := if (empty($targetConcept)) then utilde:getOriginalForConcept($concept) else utilde:getOriginalForConcept($targetConcept)
    
    let $copyFromConcept        := if ($targetConcept | $concept[inherit | contains]) then $originalConcept else $concept
    
    let $baseIdConceptList      := $concept/@id || '.' || replace($concept/@effectiveDate,'\D','') || '.'
    
    let $debug                  := if($utilde:DEBUG) then util:log('INFO', 'getConceptForEdit: orig ' || $originalConcept/@id || ' ' || $originalConcept/@statusCode) else ()

    let $associations           :=
        if ($associationMode or $targetType = ('containment','designcopy','to-designcopy','to-copy')) then 
            switch ($targetType)
            case 'containment' 
            case 'designcopy' return utilde:getConceptAssociations($targetConcept, $originalConcept, false(), 'normal', true(), $language)
            default           return utilde:getConceptAssociations($concept, $originalConcept, false(), 'normal', true(), $language)
        else ()

    let $debug                  :=
        if($utilde:DEBUG) then util:log('INFO', 'getConceptForEdit: to ... -> ' || $concept//@id || ' ' || $concept/@statusCode) else ()
    
    return
        <concept>
        {
            utilde:createConceptAttributesForEdit($concept, $originalConcept, $decor, $language)/@*,
            (:  When an item/group is new, we need a pseudo move so upon save we move to either the location it
                already is, or where the user moved it after creation with potential other new items trailing it.
            :)
            if ($concept/@statusCode='new') then <move move="true"/> else (),
            <edit mode="{$editMode}" deinherit="{$deInherit}"/>,
            decorlib:getLocks($authmap, $concept/@id, $concept/@effectiveDate, ()),
            if ($associations) then utilde:createConceptAssociationsForEdit($associations, $concept, $targetConcept, $targetType) else ()
            ,
            (: we return just a count of issues and leave it up to the calling party to get those if required :)
            <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate])}"/>
            ,     
            utilde:createInheritContainsRelationshipForEdit($concept, $targetConcept, $targetType, $inheritConcept, $containConcept, $originalConcept, $language, $level)
            ,
            for $lang in distinct-values(($projectLanguages, $language, $copyFromConcept/name/@language)) return $copyFromConcept/name[@language=$lang][1],
            $copyFromConcept/synonym,
            for $lang in distinct-values(($projectLanguages, $language, $copyFromConcept/desc/@language)) return $copyFromConcept/desc[@language=$lang][1],
            for $lang in distinct-values(($projectLanguages, $language, $copyFromConcept/source/@language)) return $copyFromConcept/source[@language=$lang][1],
            for $lang in distinct-values(($projectLanguages, $language, $copyFromConcept/rationale/@language)) return $copyFromConcept/rationale[@language=$lang][1]
            ,
            if ($concept[inherit | contains]) then utilde:createInheritContainsCommentForEdit($originalConcept, $deInherit) else (),
            $concept/comment,           
            $copyFromConcept/property
            ,
            if ($concept[inherit | contains] and $targetType = 'to-copy') then (
                let $relationship       := <relationship type="SPEC" ref="{$concept/inherit/@ref | $concept/contains/@ref}" flexibility="{$concept/inherit/@effectiveDate | $concept/contains/@flexibility}"/>
                let $referredConcept    := $inheritConcept | $containConcept
                return utilde:expandedInheritContainsRelationship($relationship, $referredConcept, $originalConcept, $language)
            ) else ()
            ,
            for $relationship in $copyFromConcept/relationship
            let $referredConcept            := utilde:getConcept($relationship/@ref, $relationship/@flexibility)
            let $originalReferredConcept    := utilde:getOriginalForConcept($referredConcept)[1]
            return utilde:expandedInheritContainsRelationship($relationship, $referredConcept, $originalReferredConcept, $language)
            ,
            for $lang in distinct-values(($projectLanguages, $language, $copyFromConcept/operationalization/@language)) return $copyFromConcept/operationalization[@language=$lang][1]
            ,
            if ((not($concept[inherit | contains]) or $deInherit) and $copyFromConcept[@type='item'][not(valueDomain)]) then
                <valueDomain type="count">
                    <conceptList id="{$baseIdConceptList || '0'}"/>
                </valueDomain>
            else ()
            ,
            utilde:createConceptValueDomainForEdit($decor, $copyFromConcept, $deInherit, $generateConceptListIds, $language, $associations, $baseIdConceptList)
            ,
            if ($targetConcept[contains] or $targetType = ('containment', 'to-copy')) then () else 
                utilde:createSubConceptForEdit($authmap, $userDisplayName, $decor, $concept, $targetConcept, $targetType, $generateConceptListIds, $level, $baseId, $projectPrefix, $language, $associationMode)
        }
        </concept>
};

declare %private function utilde:createConceptAttributesForEdit($concept as element(concept), $originalConcept as element(concept), $decor as element(decor), $language as xs:string?) as element() {

    let $iddisplay              := utillib:getNameForOID($concept/@id, $language, $decor)

    return
    <attributes>
    {
        $concept/@id,
        $concept/@effectiveDate,
        $concept/@statusCode,
        $originalConcept/@type,
        $concept/@expirationDate,
        $concept/@officialReleaseDate,
        $concept/@versionLabel,
        $concept/@canonicalUri,
        $concept/@lastModifiedDate,
        if (string-length($iddisplay) = 0) then () else attribute iddisplay {$iddisplay}
    }           
    </attributes>

};

declare %private function utilde:createConceptAssociationsForEdit($associations as element()*, $concept as element(concept), $targetConcept as element(concept)?, $targetType as xs:string?) as element()* {
    
    switch($targetType)
    case 'containment'
    case 'designcopy' return $associations/*[@conceptId = ($targetConcept/@id | $targetConcept/contains/@ref | $targetConcept/inherit/@ref)]
    case 'to-copy' return (
    (: create identifier/terminology association for concept if present for inherited concept:)
        for $association in $associations/*[@conceptId = ($targetConcept/@id | $targetConcept/contains/@ref | $targetConcept/inherit/@ref)]
        return
            element {name($association)}
            {
                attribute conceptId {$concept/@id},
                attribute conceptFlexibility {$concept/@effectiveDate},
                attribute effectiveDate {format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')},
                $association/(@*[string-length()>0] except (@conceptId | @conceptFlexibility | @effectiveDate))
            }
    )
    default return $associations/*[@conceptId = ($concept/@id | $concept/contains/@ref | $concept/inherit/@ref)]
    
};

declare %private function utilde:createInheritContainsRelationshipForEdit($concept as element(concept), $targetConcept as element(concept)?, $targetType as xs:string?, $inheritConcept as element(concept)?, $containConcept as element(concept)?, $originalConcept as element(concept)?, $language as xs:string?, $level as xs:integer) as element()? {

    let $targetLanguage         := $targetConcept/ancestor::decor/project/@defaultLanguage
    
    let $relationShip           :=
        switch ($targetType)
        case 'containment' return utilde:expandedInheritContainsRelationship(<contains ref="{$targetConcept/@id}" flexibility="{$targetConcept/@effectiveDate}"/>, $targetConcept, $originalConcept, $targetLanguage)
        case 'to-designcopy' return (
            (: for the top level to-designcopy concept we simply want to convert to inherit as request, but we do not want to recurse into circular reference so we 're-contain' those. :)
            if ($concept/ancestor-or-self::concept[@id = $targetConcept/@id] and $level gt 0) then
                utilde:expandedInheritContainsRelationship(<contains ref="{$targetConcept/@id}" flexibility="{$targetConcept/@effectiveDate}" />, $targetConcept, $originalConcept, $targetLanguage)
            else utilde:expandedInheritContainsRelationship(<inherit ref="{$targetConcept/@id}" effectiveDate="{$targetConcept/@effectiveDate}"/>, $targetConcept, $originalConcept, $targetLanguage)
        )
        case 'designcopy' return (
            utilde:expandedInheritContainsRelationship(<inherit ref="{$targetConcept/@id}" effectiveDate="{$targetConcept/@effectiveDate}"/>, $targetConcept, $originalConcept, $targetLanguage)
            ,
            for $e in $containConcept/contains | $containConcept/inherit
            return utilde:expandedInheritContainsRelationship($e, $targetConcept, $originalConcept, $targetLanguage)
        )
        case 'to-copy' return (
            for $e in $inheritConcept/contains | $inheritConcept/inherit
            return utilde:expandedInheritContainsRelationship($e, $targetConcept, $originalConcept, $targetLanguage)
        )
        default return (
            if ($concept[inherit]) then utilde:expandedInheritContainsRelationship($concept/inherit, $inheritConcept, $originalConcept, $language) else (),
            if ($concept[contains]) then utilde:expandedInheritContainsRelationship($concept/contains, $containConcept, $originalConcept, $language) else ()
        )

    return $relationShip
};

declare %private function utilde:createInheritContainsCommentForEdit($originalConcept as element(concept)?, $deInherit as xs:boolean) as element()? {

    for $comment in $originalConcept/comment
        return
            if ($deInherit) then $comment
            else (
                <inheritedComment>
                {
                    $comment/@*, 
                    $comment/node()
                }
                </inheritedComment>
            )
};

declare %private function utilde:createConceptValueDomainForEdit($decor as element(decor), $copyFromConcept as element(concept), $deInherit as xs:boolean, $generateConceptListIds as xs:boolean?, $language as xs:string, $associations as element()*, $baseIdConceptList as xs:string) as element(valueDomain)? {

    let $datasets               := $decor/datasets
    let $setmaxcounter          := if ($datasets[@maxcounter-cl]) then () else update insert attribute maxcounter-cl {()} into $datasets
    
    let $projectLanguages       := $decor/project/name/@language
    
    let $multipleConceptLists   := count($copyFromConcept[@type='item'][empty(valueDomain/conceptList)] | $copyFromConcept[@type='item']/valueDomain/conceptList) gt 1
    for $valueDomain in $copyFromConcept[@type='item']/valueDomain
    return
        <valueDomain type="{$valueDomain/@type}">
        {
            for $property in $valueDomain/property[@*[string-length() gt 0]]
            return
                <property>
                {
                    $property/@*[not(. = '')]
                }
                </property>
        ,
        (: valueDomain/conceptList and valueDomain/conceptList/concept id logic:
            - parent::concept/@id . parent::concept/@effectiveDate is base
                [the effectiveDate makes versioned concepts possible. without this timestamp, the 
                    conceptList ids from different concept versions would be the same]
            - conceptList/@id and conceptList/concept/@id are max new item
            - for historic reasons first conceptList/@id is '0' (instead of '1')
        :)
        (:
            Please note that the same numbering logic is applied when adding new conceptList/concepts in the dataset form!
        :)
        let $allConceptListConcepts := ($copyFromConcept/valueDomain/conceptList[@id] | $copyFromConcept/valueDomain/conceptList[@id]/concept[@id])
        return
        if (empty($valueDomain/conceptList)) then (
            (: This valueDomain does not have a conceptList (yet), maybe due to its type, add a pseudo conceptList :)
            let $nextclid           := 
                if ($multipleConceptLists) then (
                    let $newCurrentId   := 
                        if ($datasets[@maxcounter-cl castable as xs:integer]) then $datasets/@maxcounter-cl + 1 else decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPTLIST, $baseIdConceptList)/@next
                    let $setmaxcounter  := update value $datasets/@maxcounter-cl with $newCurrentId
                    return $baseIdConceptList || $newCurrentId
                )
                else $baseIdConceptList || '0'

            return <conceptList id="{$nextclid}"/>
        )
        else (
            (: 
                This valueDomain has conceptList[@id] or conceptList[@ref]. 
                - conceptList[@id]
                    - Normal case is to copy @id and concepts
                    - Deinherit case generates new ids
                - conceptList[@ref]
                    - Normal case is to copy @ref and concepts from original
                    - Deinherit case generates new ids
            :)
            for $conceptList in $valueDomain/conceptList
            let $doNewIds       := ($deInherit and $generateConceptListIds)
            let $nextclid       := 
                if ($doNewIds) then
                    if ($multipleConceptLists) then (
                        let $newCurrentId   := if ($datasets[@maxcounter-cl castable as xs:integer]) then $datasets/@maxcounter-cl + 1 else decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPTLIST, $baseIdConceptList)/@next
                        let $setmaxcounter  := update value $datasets/@maxcounter-cl with $newCurrentId
                        return $baseIdConceptList || $newCurrentId
                    )
                    else $baseIdConceptList || '0'
                else ()
            (: could be @ref :)
            let $originalConceptList        := utilde:getOriginalConceptList($conceptList)
            return (
                <conceptList>
                {
                    if ($doNewIds) then (
                        attribute id {$nextclid},
                        attribute oldid {$conceptList/(@id|@ref)},
                        (: create terminology association for concept if present for inherited concept:)
                        for $terminologyAssociation in $associations/*[@conceptId = ($conceptList/@id | $conceptList/@ref | $originalConceptList/@id)]
                        return
                            <terminologyAssociation conceptId="{$nextclid}" effectiveDate="{format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')}">
                            {
                                $terminologyAssociation/(@*[not(. = '')] except (@conceptId | @effectiveDate))
                            }
                            </terminologyAssociation>
                    ) 
                    else 
                    if ($deInherit) then (
                        attribute ref {$conceptList/(@id|@ref)},
                        $associations/*[@conceptId = $conceptList/(@ref | @id)]
                    )
                    else (
                        $conceptList/(@id|@ref),
                        $associations/*[@conceptId = $conceptList/(@ref | @id)]
                    )
                }
                {
                    for $conceptListConcept at $clcpos in $originalConceptList/concept
                    let $newclcid := $nextclid || '.' || $clcpos
                    
                    return
                        <concept>
                        {
                            if ($doNewIds) then attribute id {$newclcid} else ($conceptListConcept/(@id|@ref)),
                            if ($valueDomain[@type = 'ordinal'] | $conceptListConcept/@ordinal) then attribute ordinal {$conceptListConcept/@ordinal} else (),
                            attribute exception {$conceptListConcept/@exception='true'},
                            $conceptListConcept/@level[not(. = '')],
                            $conceptListConcept/@type[not(. = '')]
                            ,
                            if ($doNewIds) then
                                (: create terminology association for concept if present for inherited concept:)
                                for $terminologyAssociation in $associations/*[@conceptId = $conceptListConcept/@id]
                                return
                                    <terminologyAssociation conceptId="{$newclcid}" effectiveDate="{format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')}">
                                    {
                                        $terminologyAssociation/(@*[not(. = '')] except (@conceptId | @effectiveDate))
                                    }
                                    </terminologyAssociation>
                            else $associations/*[@conceptId = $conceptListConcept/@id]
                            ,
                            for $lang in distinct-values(($projectLanguages, $language, $conceptListConcept/name/@language)) return $conceptListConcept/name[@language=$lang][1],
                            $conceptListConcept/synonym, 
                            for $lang in distinct-values(($projectLanguages, $language, $copyFromConcept/desc/@language)) return $conceptListConcept/desc[@language=$lang][1]
                        }
                        </concept>
                }
                </conceptList>
            )
        )
    }
    {
        for $ex in $valueDomain/example
        return
            <example type="{($ex/@type, 'neutral')[not(. = '')][1]}">
            {
                $ex/@caption[not(. = '')],
                $ex/node()
            }
            </example>
        ,
        $valueDomain/(* except (property|conceptList|example))
    }
    </valueDomain>
};

declare %private function utilde:createSubConceptForEdit($authmap as map(*), $userDisplayName as xs:string, $decor as element(decor), $concept as element(concept), $targetConcept as element(concept)?, $targetType as xs:string?, $generateConceptListIds as xs:boolean?, $level as xs:integer, $baseId as xs:string, $projectPrefix as xs:string, $language as xs:string, $associationMode as xs:boolean?) as element(concept)* { 

    let $datasets               := $decor/datasets
    let $setmaxcounter          := if ($datasets[@maxcounter]) then () else update insert attribute maxcounter {()} into $datasets
(: 
    KH 20241009: handle all subconcepts but skip cancelled or rejected = $utilde:STATUSCODES-SKIPS concepts
    Also dsapi:createDataset en dsapi:inheritConcept handles these skips for other calls
:)
    for $subConcept in $targetConcept/concept[not(@statusCode = $utilde:STATUSCODES-SKIPS)]
    let $newCurrentId       := 
        if ($datasets[@maxcounter castable as xs:integer]) then $datasets/@maxcounter + 1 else (
            max($datasets//concept[matches(@id,concat($baseId,'\d+$'))]/xs:integer(tokenize(@id,'\.')[last()]))+1
        )
    let $newCurrentId       := if ($newCurrentId) then $newCurrentId else (1)
    let $newId              := $baseId || $newCurrentId
    let $newEffectiveDate   := format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')
    let $newLock            := 
        <lock type="{$decorlib:OBJECTTYPE-DATASETCONCEPT}" ref="{$newId}" effectiveDate="{$newEffectiveDate}" 
              user="{$authmap?name}" userName="{$userDisplayName}" since="{current-dateTime()}" prefix="{$projectPrefix}"/>
    let $insertLock         := update insert $newLock into $setlib:docDecorLocks/decorLocks
    let $newConcept         :=
        <concept id="{$newId}" type="{$subConcept/@type}" statusCode="new" effectiveDate="{$newEffectiveDate}" lastModifiedDate="{$newEffectiveDate}">
            <edit mode="edit"/>
            {$newLock}
            <inherit ref="{$subConcept/@id}" effectiveDate="{$subConcept/@effectiveDate}"/>
        </concept>
    (: inserting it as we go ensures the right id count :)
    let $insertNewConcept   := update insert $newConcept into $concept
    let $setmaxcounter      := update value $datasets/@maxcounter with $newCurrentId
    
    return utilde:getConceptForEdit($authmap, $userDisplayName, $decor, $concept/concept[@id = $newId], $subConcept, $targetType, $generateConceptListIds, $level + 1, $baseId, $projectPrefix, $language, $associationMode)
};

(:~ Adds properties of the referredConcept to the inherit | contains | relationship element. It also adds properties of the originalConcept to the inherit | contains | relationship, if the the referredConcept is not an originalConcept :)
declare %private function utilde:expandedInheritContainsRelationship($in as element(), $referred as element()?, $original as element()?, $language as xs:string?) as element() {

    element{name($in)}
    {
        $in/@type, $in/@ref, $in/@effectiveDate, $in/@flexibility,
        $referred/ancestor::decor/project/@prefix,
        attribute datasetId {$referred/ancestor-or-self::dataset/@id},
        attribute datasetEffectiveDate {$referred/ancestor-or-self::dataset/@effectiveDate},
        attribute datasetStatusCode {$referred/ancestor-or-self::dataset/@statusCode},
        attribute iType {$original/@type}, 
        attribute iStatusCode {$referred/@statusCode}, 
        attribute iEffectiveDate {$referred/@effectiveDate},
        if ($referred/@expirationDate) then attribute iExpirationDate {$referred/@expirationDate} else (),
        if ($referred/@versionLabel) then attribute iVersionLabel {$referred/@versionLabel} else (),
        attribute refdisplay {utillib:getNameForOID($in/@ref, $language, $referred/ancestor::decor)}
        ,
        if ($referred[@id = $original/@id][@effectiveDate = $original/@effectiveDate]) then () else (
            attribute originalId {$original/@id}, 
            attribute originalEffectiveDate {$original/@effectiveDate}, 
            attribute originalStatusCode {$original/@statusCode}, 
            if ($original[@expirationDate]) then attribute originalExpirationDate {$original/@expirationDate} else (),
            if ($original[@versionLabel]) then attribute originalVersionLabel {$original/@versionLabel} else (),
            attribute originalPrefix {$original/ancestor::decor/project/@prefix}
        )
        ,
        if (name($in) = 'relationship') then (
            let $n      := $original/name[@language=$language]
            return if ($n) then $n else 
                <name language="{$language}">
                {
                    $original/name[1]/node()
                }
                </name>
        )
        else ()
    }
};

declare %private function utilde:cleanupAfterCopyForEdit ($concepts as element()*) as element()* {

    let $results :=
        for $concept in $concepts
        return
            if ($concept[@statusCode = 'new'][inherit | contains])
            then
                <concept>
                {
                    $concept/(@* except (@navkey | @type | @iddisplay | @refdisplay | @*[. = ''])),
                    for $inherit in $concept/inherit
                    return <inherit>{$inherit/@ref, $inherit/@effectiveDate}</inherit>,
                    for $contain in $concept/contains
                    return <contains>{$contain/@ref, $contain/@flexibility}</contains>,
                    utilde:cleanupAfterCopyForEdit($concept/concept)
                }
                </concept>
             else
                (: simply copy node, special handle if a concept with sub concepts :)
                element { $concept/name() }
                {
                    $concept/@*,
                    $concept/(* except concept),
                    utilde:cleanupAfterCopyForEdit($concept/concept)
                }
                
    return $results

};

(: Retrieve conceptList returns a list of zero or more concepts. Results are always sorted by the match length. Shortest match first and largest match last.
    @param $search                  - optional. Sequence of terms to look for. Returns every object if empty.
    @param $prefix                  - required. Required DECOR project prefix to search in.
    @param $datasetId               - optional. Returns a list concepts connected to a dataset with chosen id. This parameter works in conjunction with $datasetEffectiveDate.
    @param $datasetEffectiveDate    - optional. Returns a list concepts connected to an object with this effectiveDate. This parameter works in conjunction with $datasetId.
    @param $type                    - optional. Returns all types if empty. Or can choose type: 'item'or 'group'. 
    @param $status                  - optional. Returns a list of  most recent status code ("new","draft","pending","final","rejected","cancelled","deprecated") values that results should have. Returns all if empty.
    @param $max                     - optional. Maximum number of results to return, defaults to 50.
    @param $originalOnly            - required. Returns original concepts (that do not inherit) if true, otherwise returns every hit including concepts that inherit from matching concepts
    @param $localConceptsOnly       - required. Returns only concepts within the given prefix
:)
declare function utilde:getConceptList($projectPrefix as xs:string?, $searchTerms as xs:string+, $maxResults as xs:integer?, $version as xs:string?, $language as xs:string?, $datasetOrTransactionId as xs:string?, $datasetOrTransactionEffectiveDate as xs:string?, $conceptId as xs:string?, $conceptEffectiveDate as xs:string?, $conceptType as xs:string?, $statusCodes as xs:string*, $originalConceptsOnly as xs:boolean, $localConceptsOnly as xs:boolean) as element(result) {
    
    let $contextConcept         := if (empty($conceptId)) then () else utilde:getConcept($conceptId, $conceptEffectiveDate)
    
    let $queryOnId              := if (count($searchTerms)=1 and matches($searchTerms[1],'^\d+(\.\d+)*$')) then true() else false()
    
    let $resultsOnId            := 
        if ($queryOnId) then (
            if ($projectPrefix = '*') then 
                if (empty($conceptType))
                then $setlib:colDecorData//concept[ends-with(@id, $searchTerms[1])][ancestor::decor[not(@private='true')]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)]
                else $setlib:colDecorData//concept[ends-with(@id, $searchTerms[1])][ancestor::decor[not(@private='true')]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType]
            else if (empty($projectPrefix)) then
                if (empty($conceptType)) 
                then $setlib:colDecorData//concept[ends-with(@id, $searchTerms[1])][ancestor::decor[not(@private='true')][@repository = 'true']][ancestor::datasets][not(ancestor::conceptList | ancestor::history)]
                else $setlib:colDecorData//concept[ends-with(@id, $searchTerms[1])][ancestor::decor[not(@private='true')][@repository = 'true']][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType]
            else if (empty($version)) then
                if (empty($conceptType)) 
                then $setlib:colDecorData//concept[ends-with(@id, $searchTerms[1])][ancestor::decor/project[@prefix=$projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)]
                else $setlib:colDecorData//concept[ends-with(@id, $searchTerms[1])][ancestor::decor/project[@prefix=$projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType]
            else if ($setlib:colDecorVersion/decor[@versionDate = $version][@language = $language]/project[@prefix=$projectPrefix]) then
                if (empty($conceptType)) 
                then $setlib:colDecorVersion//concept[ends-with(@id, $searchTerms[1])][ancestor::decor[@versionDate = $version][@language = $language]/project[@prefix=$projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)]
                else $setlib:colDecorVersion//concept[ends-with(@id, $searchTerms[1])][ancestor::decor[@versionDate = $version][@language = $language]/project[@prefix=$projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType]
            else if (empty($conceptType)) 
                then utillib:getDecorByPrefix($projectPrefix, $version, $language)//concept[ends-with(@id, $searchTerms[1])][ancestor::datasets]
                else utillib:getDecorByPrefix($projectPrefix, $version, $language)//concept[ends-with(@id, $searchTerms[1])][ancestor::datasets][@type = $conceptType]
        ) 
        else ()
        
    let $resultsOnName          :=
        if ($queryOnId) then () 
        else (
            let $luceneQuery    := utillib:getSimpleLuceneQuery($searchTerms)
            let $luceneOptions  := utillib:getSimpleLuceneOptions()
            return
            
            if ($projectPrefix = '*') then
                if (empty($conceptType)) then
                    ($setlib:colDecorData//concept[ft:query(name,$luceneQuery)][ancestor::decor[not(@private='true')]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)] |
                     $setlib:colDecorData//concept[ft:query(synonym,$luceneQuery)][ancestor::decor[not(@private='true')]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)])
                else 
                    ($setlib:colDecorData//concept[ft:query(name,$luceneQuery)][ancestor::decor[not(@private='true')]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType] |
                     $setlib:colDecorData//concept[ft:query(synonym,$luceneQuery)][ancestor::decor[not(@private='true')]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType])
            else if (empty($projectPrefix)) then
                if (empty($conceptType)) then
                    ($setlib:colDecorData//concept[ft:query(name,$luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true']][ancestor::datasets][not(ancestor::conceptList | ancestor::history)] |
                     $setlib:colDecorData//concept[ft:query(synonym,$luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true']][ancestor::datasets][not(ancestor::conceptList | ancestor::history)])
                else 
                    ($setlib:colDecorData//concept[ft:query(name,$luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true']][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType] |
                     $setlib:colDecorData//concept[ft:query(synonym,$luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true']][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType])
            else if (empty($version)) then
                if (empty($conceptType)) then
                    ($setlib:colDecorData//concept[ft:query(name,$luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true'] | ancestor::decor/project[@prefix = $projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)] |
                     $setlib:colDecorData//concept[ft:query(synonym,$luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true'] | ancestor::decor/project[@prefix = $projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)])
                else 
                    ($setlib:colDecorData//concept[ft:query(name,$luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true'] | ancestor::decor/project[@prefix = $projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType] |
                     $setlib:colDecorData//concept[ft:query(synonym,$luceneQuery)][ancestor::decor[not(@private='true')][@repository = 'true'] | ancestor::decor/project[@prefix = $projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType])
                
            else if ($setlib:colDecorVersion/decor[@versionDate = $version][@language = $language]/project[@prefix=$projectPrefix]) then
                    if (empty($conceptType)) then
                        ($setlib:colDecorVersion//concept[ft:query(name,$luceneQuery)][ancestor::decor[@versionDate = $version][@language = $language]/project[@prefix = $projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)] |
                         $setlib:colDecorVersion//concept[ft:query(synonym,$luceneQuery)][ancestor::decor[@versionDate = $version][@language = $language]/project[@prefix = $projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)])
                    else 
                        ($setlib:colDecorVersion//concept[ft:query(name,$luceneQuery)][ancestor::decor[@versionDate = $version][@language = $language]/project[@prefix = $projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType] |
                         $setlib:colDecorVersion//concept[ft:query(synonym,$luceneQuery)][ancestor::decor[@versionDate = $version][@language = $language]/project[@prefix = $projectPrefix]][ancestor::datasets][not(ancestor::conceptList | ancestor::history)][@type = $conceptType])
                else if (empty($conceptType)) then
                        (utillib:getDecorByPrefix($projectPrefix, $version, $language)//concept[ft:query(name,$luceneQuery)][ancestor::datasets] |
                         utillib:getDecorByPrefix($projectPrefix, $version, $language)//concept[ft:query(synonym,$luceneQuery)][ancestor::datasets])
                    else 
                        (utillib:getDecorByPrefix($projectPrefix, $version, $language)//concept[ft:query(name,$luceneQuery)][ancestor::datasets][@type = $conceptType] |
                         utillib:getDecorByPrefix($projectPrefix, $version, $language)//concept[ft:query(synonym,$luceneQuery)][ancestor::datasets][@type = $conceptType])
        )
    
    let $allResults             := 
        if ($originalConceptsOnly) then $resultsOnName[not(inherit | contains)] | $resultsOnId[not(inherit | contains)]
        else if ($queryOnId) then $resultsOnId
        else if (empty($version)) then 
            for $concept in ($resultsOnName | $resultsOnId) 
            return utilde:getConceptsThatInheritFromConcept($concept, ())
        (: everything should already match in compiled projects. No need to re-look-up :)
        else $resultsOnName | $resultsOnId
    
    let $transaction            := if ($datasetOrTransactionId) then (utillib:getTransaction($datasetOrTransactionId, $datasetOrTransactionEffectiveDate, $version, $language)) else ()
    
    let $resultsFiltered        := 
        if ($transaction) then $allResults[@id = $transaction//concept/@ref]
        else if ($datasetOrTransactionId) then 
            if ($datasetOrTransactionEffectiveDate castable as xs:dateTime) then 
                $allResults[ancestor::dataset[@id = $datasetOrTransactionId][@effectiveDate = $datasetOrTransactionEffectiveDate]] | $allResults[@datasetId = $datasetOrTransactionId][@datasetEffectiveDate = $datasetOrTransactionEffectiveDate]
            else 
                $allResults[ancestor::dataset[@id = $datasetOrTransactionId]] | $allResults[@datasetId = $datasetOrTransactionId]
        else $allResults
    let $resultsFiltered        := if ($localConceptsOnly) then $resultsFiltered[(ancestor::decor/project/@prefix | @ident) = $projectPrefix] else $resultsFiltered
    let $resultsFiltered        := if (empty($statusCodes)) then $resultsFiltered else $resultsFiltered[@statusCode = $statusCodes]
    
    (:deduplicate and sort by shortest match first:)
    let $resultsFiltered        :=
        for $concepts in $resultsFiltered
        let $conceptId          := concat($concepts/@id, $concepts/@effectiveDate)
        group by $conceptId
        order by string-length(($concepts/name/text())[1])
        return $concepts[1]
    
    let $resultsFiltered        :=
        for $concepts in $resultsFiltered
        let $conceptLength      := string-length(($concepts/name/text())[1])
        group by $conceptLength 
        order by $conceptLength
        return (
            for $concept in $concepts[ancestor::decor/project[@prefix=$projectPrefix] | @ident[.=$projectPrefix]]
            order by $concept/ancestor::dataset/@id, $concept/@datasetId
            return $concept
            ,
            for $concept in $concepts[not(ancestor::decor/project[@prefix=$projectPrefix] | @ident[.=$projectPrefix])]
            order by $concept/ancestor::dataset/@id, $concept/@datasetId
            return $concept
        )
    
    let $count                  := count($resultsFiltered)
    let $maxResults             := if ($maxResults) then $maxResults else $count
    
    return
        <list artifact="DE" current="{if ($count le $maxResults) then $count else $maxResults}" total="{$count}" all="{$count}" lastModifiedDate="{current-dateTime()}" q="{if (request:exists()) then request:get-query-string() else ()}">
        {
            for $concepts in subsequence($resultsFiltered,1,$maxResults)
            let $ancestorProject    := $concepts[1]/ancestor::decor
            let $displayName        := $concepts[1]/name[1]
            order by string-length($displayName)
            return
                element {$concepts[1]/local-name()}
                {
                    attribute uuid {util:uuid()},
                    $concepts[1]/(@* except (@uuid|@conceptId|@datasetId|@datasetEffectiveDate|@datasetStatusCode|@datasetVersionLabel|@datasetExpirationDate|@ident|@repository|@private|@experimental)),
                    attribute conceptId {$concepts[1]/@id},
                    if ($ancestorProject) then (
                        attribute datasetId             {$concepts[1]/ancestor::dataset/@id},
                        attribute datasetEffectiveDate  {$concepts[1]/ancestor::dataset/@effectiveDate},
                        attribute datasetStatusCode     {$concepts[1]/ancestor::dataset/@statusCode},
                        if ($concepts[1]/ancestor::dataset/@versionLabel) then   attribute datasetVersionLabel {$concepts[1]/ancestor::dataset/@versionLabel} else (),
                        if ($concepts[1]/ancestor::dataset/@expirationDate) then attribute datasetExpirationDate {$concepts[1]/ancestor::dataset/@expirationDate} else (),
                        attribute ident                 {$ancestorProject/project/@prefix},
                        attribute repository            {$ancestorProject/@repository='true'},
                        attribute private               {$ancestorProject/@private='true'},
                        attribute experimental          {$ancestorProject/@experimental='true'}
                    )
                    else (
                        $concepts[1]/@datasetId,
                        $concepts[1]/@datasetEffectiveDate,
                        $concepts[1]/@datasetStatusCode,
                        $concepts[1]/@datasetVersionLabel,
                        $concepts[1]/@datasetExpirationDate,
                        $concepts[1]/@ident,
                        $concepts[1]/@repository,
                        $concepts[1]/@private,
                        $concepts[1]/@expiremental
                    ),
                    (: mark search result as being one of the parents or self of context concept :)
                    if ($contextConcept) then 
                        attribute contextParent {exists($contextConcept/ancestor-or-self::concept[@id = $concepts[1]/@id][@effectiveDate = $concepts[1]/@effectiveDate])}
                    else (),
                    $concepts[1]/name,
                    $concepts[1]/inherit,
                    $concepts[1]/contains,
                    if ($ancestorProject) then (
                        for $name in $concepts[1]/ancestor::dataset/name
                        return
                            <datasetName>{$name/@*, $name/node()}</datasetName>
                        ,
                        for $name in $concepts[1]/ancestor::decor/project/name
                        return
                            <projectName>{$name/@*, $name/node()}</projectName>
                    )
                    else (
                        $concepts[1]/datasetName,
                        $concepts[1]/projectName
                    ),
                    
                    let $dbconcept  := $setlib:colDecorData//concept[@id = $concepts[1]/@id][not(ancestor::history)]
                    let $dbconcept  := $dbconcept[@effectiveDate = $concepts[1]/@effectiveDate]
                    let $languages  := $dbconcept/ancestor::decor/project/name/@language
                    
                    for $language in $languages
                    return
                    <path language="{$language}">
                    {
                        for $ancestor in $dbconcept/ancestor::concept
                        let $originalConcept    := utilde:getOriginalForConcept($ancestor)
                        let $conceptName        := if ($originalConcept/name[@language=$language]) then $originalConcept/name[@language=$language] else ($originalConcept/name[1])
                        return
                            concat(($conceptName/text())[1], ' / ')
                    }
                    </path>
                }
        }
        </list>
};

(:~ Internal helper function that recursively finds concepts inheriting from the current concept 
    When you search on id, it might find inherited concepts. 
    This does not happen for search by name as inherited concepts do not have a name
:)
declare %private function utilde:getConceptsThatInheritFromConcept($concept as element(concept), $results as element(concept)*) as item()* {

    let $concept                :=
        if ($concept[not(name)]) then (
            let $originalConcept    := utilde:getOriginalForConcept($concept)
            return
            element {$concept/name()} 
            {
                if ($concept[@ref]) then ($concept/(@ref|@flexibility), $originalConcept/@statusCode) else $concept/(@id|@effectiveDate|@statusCode),
                attribute type                  {$originalConcept/@type},
                attribute datasetId             {$concept/ancestor::dataset/@id},
                attribute datasetEffectiveDate  {$concept/ancestor::dataset/@effectiveDate},
                attribute datasetStatusCode     {$concept/ancestor::dataset/@statusCode},
                if ($concept/ancestor::dataset/@versionLabel) then attribute datasetVersionLabel {$concept/ancestor::dataset/@versionLabel} else (),
                if ($concept/ancestor::dataset/@expirationDate) then attribute datasetExpirationDate {$concept/ancestor::dataset/@expirationDate} else (),
                attribute ident                 {$concept/ancestor::decor/project/@prefix},
                attribute repository            {$concept/ancestor::decor/@repository='true'},
                attribute private               {$concept/ancestor::decor/@private='true'},
                attribute experimental          {$concept/ancestor::decor/project/@experimental='true'},
                $originalConcept/name,
                $concept/inherit,
                $concept/contains,
                for $name in $concept/ancestor::dataset/name
                return <datasetName>{$name/@*, $name/node()}</datasetName>
                ,
                for $name in $concept/ancestor::decor/project/name
                return <projectName>{$name/@*, $name/node()}</projectName>
                
            }
        ) else ($concept)
    
    let $inheritingConcepts     := $setlib:colDecorData//inherit[@ref = $concept/@id][not(ancestor::history)]
    let $inheritingConcepts     := $inheritingConcepts[@effectiveDate=$concept/@effectiveDate]/parent::concept
    
    let $containingConcepts     := $setlib:colDecorData//contains[@ref = $concept/@id][not(ancestor::history)]
    let $containingConcepts     := $containingConcepts[@flexibility = $concept/@effectiveDate]/parent::concept
    
    let $currentResult          :=
        for $currentResultConcept in ($inheritingConcepts | $containingConcepts)
        let $currentResultConceptWithName :=
            element {$currentResultConcept/name()} 
            {
                attribute id                    {$currentResultConcept/@id}, 
                attribute effectiveDate         {$currentResultConcept/@effectiveDate},
                attribute statusCode            {$currentResultConcept/@statusCode},
                attribute type                  {$concept/@type},
                attribute datasetId             {$currentResultConcept/ancestor::dataset/@id},
                attribute datasetEffectiveDate  {$currentResultConcept/ancestor::dataset/@effectiveDate},
                attribute datasetStatusCode     {$concept/ancestor::dataset/@statusCode},
                if ($currentResultConcept/ancestor::dataset/@versionLabel) then   attribute datasetVersionLabel {$currentResultConcept/ancestor::dataset/@versionLabel} else (),
                if ($currentResultConcept/ancestor::dataset/@expirationDate) then attribute datasetExpirationDate {$currentResultConcept/ancestor::dataset/@expirationDate} else (),
                attribute ident                 {$currentResultConcept/ancestor::decor/project/@prefix},
                attribute repository            {$currentResultConcept/ancestor::decor/@repository='true'},
                attribute private               {$currentResultConcept/ancestor::decor/@private='true'},
                attribute experimental          {$currentResultConcept/ancestor::decor/project/@experimental='true'},
                $concept/name,
                $currentResultConcept/inherit | $currentResultConcept/contains,
                for $name in $currentResultConcept/ancestor::dataset/name
                return <datasetName>{$name/@*, $name/node()}</datasetName>
                ,
                for $name in $currentResultConcept/ancestor::decor/project/name
                return <projectName>{$name/@*, $name/node()}</projectName>
            }
        
        return utilde:getConceptsThatInheritFromConcept($currentResultConceptWithName,$results)
    
    return $concept | $results | $currentResult
};

(:~ Central logic for creating empty dataset concept
    @param $datasetId               - required. dataset id to insert concept into.
    @param $datasetEffectiveDate    - required. dataset effectiveDate to insert concept into.
    @param $conceptBaseId           - optional. baseId to create the new concept id out of. Defaults to defaultBaseId for type DE if omitted
    @param $conceptType             - required. 'item' or 'group'
    @param $insertMode              - required. 'into' or 'preceding' or 'following'. For preceding and for following, $insertRef is required
    @param $insertRef               - optional. concept/@id reference for insert. Inserts as new concept in dataset if empty
    @param $insertFlexibility       - optional. concept/@effectiveDate reference for insert. Only relevant when two versions of the same concept are in the same dataset which is logically highly unlikely
:)
declare function utilde:createConcept($authmap as map(*), $datasetId as xs:string, $datasetEffectiveDate as xs:string, $conceptBaseId as xs:string?, $conceptType as xs:string, $insertMode as xs:string, $insertRef as xs:string?, $insertFlexibility as xs:string?) {

    (: check if dataset not final or deprecated ? (security):)
    let $dataset                := utillib:getDataset($datasetId, $datasetEffectiveDate)
    let $decor                  := $dataset/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $storedConcept          := if ($insertFlexibility castable as xs:dateTime) then $dataset//concept[@id = $insertRef][@effectiveDate = $insertFlexibility][not(ancestor::history)] else $dataset//concept[@id = $insertRef]
    
    let $check                  :=
        if ($dataset) then () else (
            error($errors:BAD_REQUEST, 'Dataset with id ' || $datasetId || ' and effectiveDate ' || $datasetEffectiveDate || ' , does not exist. Cannot create in non-existent dataset.')
        )
    
    let $check                  := utilde:checkConceptAccess($authmap, $decor)
    
    let $check                  :=
        if ($storedConcept/ancestor-or-self::concept/@statusCode = $utilde:STATUSCODES-FINAL) then
            error($errors:BAD_REQUEST, concat('Concept cannot be added while it or any of its parents or the dataset itself have one of status: ', string-join($utilde:STATUSCODES-FINAL, ', ')))
        else ()
    
    let $check                  :=  
        if ($conceptType = $utilde:CONCEPT-TYPE/enumeration/@value) then () else (
            error($errors:BAD_REQUEST, 'Concept type ' || $conceptType || ' SHALL be one of ' || string-join($utilde:CONCEPT-TYPE/enumeration/@value, ', '))
        )
    
    let $check                  :=
        switch ($insertMode)
        case 'into' 
        case 'following'
        case 'preceding' return ()
        default return (
            error($errors:BAD_REQUEST, 'Parameter insertMode ' || $insertMode || ' SHALL be one of ''into'', ''following'', or ''preceding''')
        )
    
    let $check                  :=
        if (empty($insertRef)) then 
            if ($insertMode = 'into') then () else (
                error($errors:BAD_REQUEST, 'Parameter insertRef SHALL have a value when insertMode is ' || $insertMode || '. Otherwise we don''t know where to insert')
            )
        else ()
    
    let $check                  :=
        if (empty($conceptBaseId)) then () else (
            let $conceptBaseIds         := decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPT)
            return
                if ($conceptBaseIds[@id = $conceptBaseId]) then () else (
                    error($errors:BAD_REQUEST, 'Parameter conceptBaseId SHALL have a value that is declared as baseId with type DE (dataset concepts) in the project. Allowable are ' || string-join($conceptBaseIds/@id, ', '))
                )
        )

    (: Note: not protected from missing defaultBaseId for given type ... :)
    let $baseId           := if (string-length($conceptBaseId)=0) then decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPT)/@id else ($conceptBaseId)
    let $newId            := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-DATASETCONCEPT, $baseId)/@id
    let $newEffectiveDate := format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')
    
    let $concept          :=
        <concept id="{$newId}" type="{$conceptType}" statusCode="new" effectiveDate="{$newEffectiveDate}" lastModifiedDate="{$newEffectiveDate}">
        {
            for $language in distinct-values($decor/project/name/@language)
            return <name language="{$language}"/>
            ,
            if ($conceptType = 'item') then <valueDomain type="{$utilde:VALUEDOMAIN-TYPE/enumeration/@value[not(. = 'code')][1]}"/> else ()
        }
        </concept>
    
    let $insert           :=
        if ($storedConcept) then
            if ($insertMode='following') then update insert $concept following $storedConcept[last()]
            else if ($insertMode='preceding') then update insert $concept preceding $storedConcept[1]
            else if ($storedConcept[history | concept]) then update insert $concept preceding ($storedConcept/history | $storedConcept/concept)[1]
            else update insert $concept into $storedConcept
        else if ($dataset[concept]) then update insert $concept following $dataset/concept[last()]
        else update insert $concept into $dataset

    let $lock       := decorlib:setLock($authmap, $concept/@id, $concept/@effectiveDate, false())

    return $concept
};

(:~ Central logic for patching an existing dataset concept statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR concept/@id to update
    @param $effectiveDate           - required. DECOR concept/@effectiveDate to update
    @param $recurse                 - optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $list                    - optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $statusCode              - optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $versionLabel            - optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $expirationDate          - optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $officialReleaseDate     - optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
:)
declare function utilde:setConceptStatus($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $recurse as xs:boolean, $listOnly as xs:boolean, $newStatusCode as xs:string?, $newVersionLabel as xs:string?, $newExpirationDate as xs:string?, $newOfficialReleaseDate as xs:string?)  {
    (:get object for reference:)
    let $object                 := utilde:getConcept($id, $effectiveDate)
    let $decor                  := $object/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($object) = 1) then () else
        if ($object) then
            error($errors:SERVER_ERROR, 'Concept id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($object) || ' were found in: ' || string-join(distinct-values($object/ancestor::decor/project/@prefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Concept id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )
    let $check                  := utilde:checkConceptAccess($authmap, $decor)
       
    (: https://art-decor.atlassian.net/browse/AD30-259 :)
    let $check                  :=
        if ($object[@statusCode = 'cancelled'] and $object[not(@statusCode = $newStatusCode)]) then
            if ($object/ancestor::dataset[@statusCode = 'draft']) then () else (
                error($errors:BAD_REQUEST, 'Status transition not allowed from ' || $object/@statusCode || ' to ''' || $newStatusCode || ''', because the dataset is not in status draft but in status ''' || $object/ancestor::dataset/@statusCode || '''')
            )
        else ()
    
    let $testUpdate             := utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, true(), $projectPrefix)
    
    let $results                :=
        if ($testUpdate[self::error]) then $testUpdate 
        else if ($listOnly) then $testUpdate 
        else utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, false(), $projectPrefix)
    
    let $check                  :=
        if ($results[self::error] and not($listOnly)) then
         error($errors:BAD_REQUEST, string-join(
             for $e in $results[self::error]
             return $e/@itemname || ' id ''' || $e/@id || ''' effectiveDate ''' || $e/@effectiveDate || ''' cannot be updated: ' || data($e)
             , ' ')
         )
        else ()
    
    return ()
};

(:~ Central logic for updating an existing dataset concept
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR concept/@id to update
    @param $effectiveDate   - required. DECOR concept/@effectiveDate to update
    @param $data            - required. DECOR concept xml element containing everything that should be in the updated concept
:)
declare function utilde:putConcept($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(concept)) {

    let $storedConcept          := utilde:getConcept($id, $effectiveDate)
    let $decor                  := $storedConcept/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    
    let $check                  :=
        if ($storedConcept) then () else (
            error($errors:BAD_REQUEST, 'Concept with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    
    let $lock                   := utilde:checkConceptAccess($authmap, $decor, $id, $effectiveDate, true())
    
    let $check                  :=
        if ($storedConcept/@statusCode = $utilde:STATUSCODES-FINAL) then
            error($errors:BAD_REQUEST, concat('Concept cannot be edited in status ', $storedConcept/@statusCode, '. ', if ($storedConcept/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
        else ()
    
    let $check                  :=
        if ($data[@id = $id] | $data[empty(@id)]) then () else (
            error($errors:BAD_REQUEST, concat('Concept SHALL have the same id as the concept id ''', $id, ''' used for updating. Found in request body: ', ($data/@id, 'null')[1]))
        )
    
    let $check                  :=
        if ($data[@effectiveDate = $effectiveDate] | $data[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Concept SHALL have the same effectiveDate as the concept effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($data/@effectiveDate, 'null')[1]))
        )
    
    let $check                  :=
        if ($data[@statusCode = $storedConcept/@statusCode] | $data[empty(@statusCode)]) then () else (
            error($errors:BAD_REQUEST, concat('Concept SHALL have the same statusCode as the concept statusCode ''', $storedConcept/@statusCode, ''' used for updating. Found in request body: ', ($data/@statusCode, 'null')[1]))
        )
    
    let $check                  := ruleslib:checkConcept($data, $storedConcept, 'update')
    
    let $preparedConcept        :=
        if ($data[valueDomain | @type[. = 'item']]) then utilde:prepareItemForUpdate($data, $storedConcept)
        else if ($data[concept | @type[. = 'group']]) then utilde:prepareGroupForUpdate($data, $storedConcept)
        (: already handled above :)
        else ()           
    
    let $intention              := if ($storedConcept[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, $intention, $storedConcept)
    let $update                 := update replace $storedConcept with $preparedConcept
    let $update                 := update delete $lock
    
    return ()
};

(:~ Central logic for patching an existing transaction concept
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR concept/@id to update
    @param $effectiveDate   - required. DECOR concept/@effectiveDate to update
    @param $transactionId   - required. DECOR transaction/@id this concept is in
    @param $transactionEffectiveDate   - required. DECOR transaction/@effectiveDate this concept is in
    @param $data            - required. DECOR concept xml element containing everything that should be in the updated concept
:)
declare function utilde:putTransactionConcept($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $transactionId as xs:string, $transactionEffectiveDate as xs:string, $data as element(concept)) {
    
    let $storedConcept          := utilde:getTransactionConcept($id, $effectiveDate, $transactionId, $transactionEffectiveDate)
    let $decor                  := $storedConcept/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($storedConcept) then () else (
            error($errors:BAD_REQUEST, 'Transaction concept with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist in transaction with id ' || ($storedConcept/ancestor::transaction)[1]/@id || ' and effectiveDate ' || ($storedConcept/ancestor::transaction)[1]/@effectiveDate)
        )
    
    let $lock                   := utilde:checkTransactionConceptAccess($authmap, $decor, $id, $effectiveDate, true(), $storedConcept)
    
    let $check                  :=
        if ($storedConcept/ancestor-or-self::concept/@statusCode = $utilde:STATUSCODES-FINAL) then
            error($errors:BAD_REQUEST, concat('Concept cannot be edited while it or any of its parents have one of status: ', string-join($utilde:STATUSCODES-FINAL, ', ')))
        else ()
    
    let $check                  :=
        if ($data[@id = $id]) then () else (
            error($errors:BAD_REQUEST, concat('Concept SHALL have the same id as the concept id ''', $id, ''' used for updating. Found in request body: ', ($data/@id, 'null')[1]))
        )
    
    let $check                  :=
        if ($data[@effectiveDate = $effectiveDate]) then () else (
            error($errors:BAD_REQUEST, concat('Concept SHALL have the same effectiveDate as the concept id ''', $effectiveDate, ''' used for updating. Found in request body: ', ($data/@effectiveDate, 'null')[1]))
        )
    
    let $check                  := ruleslib:checkTransactionConcept($data, 'update')    
    
    let $preparedConcept        := utilde:prepareTransactionItemForUpdate($data, $storedConcept)
    
    let $intention              := if ($storedConcept/ancestor::transaction[1][@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-TRANSACTION, $projectPrefix, $intention, $storedConcept/ancestor::transaction[1])
    let $update                 := update replace $storedConcept with $preparedConcept
    let $update                 := update delete $lock
    
    return ()
};

declare function utilde:prepareItemForUpdate($concept as element(),$storedItem as element()?) as element(concept) {

    let $status                 := if ($concept[@statusCode=('new','')] | $concept[empty(@statusCode)]) then 'draft' else ($concept/@statusCode)

    return
        <concept id="{$concept/@id}" effectiveDate="{$concept/@effectiveDate}" statusCode="{$status}">
        {
            if ($concept[inherit | contains]) then () else attribute type {'item'},
            $concept/@versionLabel[string-length() gt 0],
            $concept/@expirationDate[string-length() gt 0],
            $concept/@officialReleaseDate[string-length() gt 0],
            $concept/@canonicalUri[string-length() gt 0],
            attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)}
            ,
            if ($concept[inherit]) then (
                <inherit>{$concept/inherit/@ref, $concept/inherit/@effectiveDate}</inherit>
                ,
                for $node in $concept/comment
                return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
            )
            else if ($concept[contains]) then (
                <contains>{$concept/contains/@ref, $concept/contains/@flexibility}</contains>
                ,
                for $node in $concept/comment
                return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
            )
            else (
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/name)
                ,
                for $node in $concept/synonym
                return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
                ,
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/desc),
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/source),
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/rationale)
                ,
                for $node in $concept/comment
                return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
                ,
                utillib:prepareConceptPropertyForUpdate($concept/property),
                utillib:prepareDatasetRelationshipForUpdate($concept/relationship),
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/operationalization)
                ,
                for $valueDomain in $concept/valueDomain
                return utilde:handleConceptValueDomain($valueDomain)
            )
       }
       </concept>
};
declare function utilde:prepareTransactionItemForUpdate($concept as element(),$storedItem as element()?) as element(concept) {

    <concept ref="{$concept/@id}" flexibility="{$concept/@effectiveDate}">
    {
         (: keep empty min/max. this happens for incomplete transactions where not all is known yet :)
        if ($concept[@conformance = ('C', 'NP')]) then () else ($concept/@minimumMultiplicity, $concept/@maximumMultiplicity),
        if ($concept[@conformance = ('C', 'NP')]) then $concept/@conformance[not(.='')] else $concept/@isMandatory[. = 'true'],
        $concept/context[exists(node())][.//text()[not(normalize-space() = '')]]
        ,
        if ($concept[@conformance = 'C']) then (
            for $condition in $concept/condition
            return
            <condition>
            {
                (: keep empty min/max. this happens for incomplete transactions where not all is known yet :)
                if ($condition/@conformance='NP') then () else ($condition/@minimumMultiplicity, $condition/@maximumMultiplicity),
                if ($condition[@conformance = ('C', 'NP')]) then $concept/@conformance[not(.='')] else $condition/@isMandatory[. = 'true']
                ,
                if ($condition[desc]) then utillib:prepareFreeFormMarkupWithLanguageForUpdate($condition/desc)
                else if ($condition[text()[not(. = '')]]) then () else <desc language="{$storedItem/ancestor::decor/project/@defaultLanguage}">{$condition/text()}</desc>
            }
            </condition>
        )
        else ()
        ,
        (: prepare them while allowing for 'empty' associations, that don't bind to anything, signalling deletion in the context of this transaction :)
        for $node in $concept/terminologyAssociation
        return utillib:prepareTerminologyAssociationForUpdate($node, true())
        ,
        for $node in $concept/identifierAssociation
        return utillib:prepareIdentifierAssociationForUpdate($node, true())
    }
    </concept>
};

declare function utilde:prepareGroupForUpdate($concept as element(),$storedGroup as element()?) as element() {
    
    let $status                 := if ($concept[@statusCode=('new','')] | $concept[empty(@statusCode)]) then 'draft' else ($concept/@statusCode)

    return
    <concept id="{$concept/@id}" effectiveDate="{$concept/@effectiveDate}" statusCode="{$status}">
    {
        if ($concept[inherit | contains]) then () else attribute type {'group'},
        $concept/@versionLabel[string-length() gt 0],
        $concept/@expirationDate[string-length() gt 0],
        $concept/@officialReleaseDate[string-length() gt 0],
        $concept/@canonicalUri[string-length() gt 0],
        attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)}
        ,
        if ($concept[inherit]) then (
            <inherit>{$concept/inherit/@ref, $concept/inherit/@effectiveDate}</inherit>
            ,
            for $node in $concept/comment
            return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
        )
        else if ($concept[contains]) then (
            <contains>{$concept/contains/@ref, $concept/contains/@flexibility}</contains>
            ,
            for $node in $concept/comment
            return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
        )
        else (
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/name)
            ,
            for $node in $concept/synonym
            return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
            ,
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/desc),
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/source),
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/rationale)
            ,
            for $node in $concept/comment
            return utillib:prepareFreeFormMarkupWithLanguageForUpdate($node)
            ,
            utillib:prepareConceptPropertyForUpdate($concept/property),
            utillib:prepareDatasetRelationshipForUpdate($concept/relationship),
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/operationalization)
        )
        ,
        if ($concept[contains]) then () 
        else if ($concept[inherit][concept]) then (
            (: when you select edit for a group, only the group level comes to the client. If you hit save 
            (e.g. after adding a comment somewhere) you would delete the stored children. So we check if the thing
            coming in has children of its own. When it does, it must have been for a new inherit in which case we 
            purposely want to overwrite the stored version of the group :)
            for $child in $concept/concept
            return 
                if ($child[concept]) 
                then utilde:prepareGroupForUpdate($child, utilde:getConcept($child/@id, $child/@effectiveDate)) 
                else utilde:prepareItemForUpdate($child, utilde:getConcept($child/@id, $child/@effectiveDate))
        ) 
        (: just get whatever was stored before under this group :)
        else $storedGroup/concept
    }
    </concept>
};

declare function utilde:handleConceptValueDomain($in as element(valueDomain)?) as element(valueDomain)? {
    
    if ($in[@type = $utilde:VALUEDOMAIN-TYPE/enumeration/@value]) then
        element {name($in)} {
            $in/@type,
            for $n in $in/property
            let $prop   :=
                <property>
                {
                    if ($in/@type=('count')) then $n/(@minInclude, @maxInclude, @default, @fixed)[not(. = '')]
                    else if ($in/@type=('decimal')) then $n/(@minInclude, @maxInclude, @fractionDigits, @default, @fixed)[not(. = '')]
                    else if ($in/@type=('duration','quantity')) then $n/(@minInclude, @maxInclude, @fractionDigits, @default, @fixed, @unit, @currency)[not(. = '')]
                    else if ($in/@type=('date','datetime','time')) then $n/(@timeStampPrecision, @default, @fixed)[not(. = '')]
                    else if ($in/@type=('code','score','ordinal','boolean')) then $n/(@fractionDigits, @default, @fixed)[not(. = '')]
                    else if ($in/@type=('string','text','identifier','blob')) then $n/(@minLength, @maxLength, @default, @fixed)[not(. = '')]
                    else ()
                }
                </property>
            return
                $prop[@*]
            ,
            for $n in $in[@type = ('code', 'ordinal')]/conceptList[utillib:isOid(@id) or utillib:isOid(@ref)]
            return
                <conceptList>
                {
                    $n/@id | $n/@ref,
                    for $sn in $n/concept[utillib:isOid(@id)]
                    return
                        <concept>
                        {
                            $sn/@id,
                            $sn/@exception[. = ('true', 'false')],
                            $sn/@type[. = $utilde:VOCAB-TYPE/enumeration/@value],
                            $sn/@level[. castable as xs:integer],
                            $sn/@ordinal[not(. = '')],
                            utillib:prepareFreeFormMarkupWithLanguageForUpdate($sn/name),
                            (: don't deduplicate by language so we do for loop here :)
                            for $ssn in $sn/synonym
                            return utillib:prepareFreeFormMarkupWithLanguageForUpdate($ssn)
                            ,
                            utillib:prepareFreeFormMarkupWithLanguageForUpdate($sn/desc)
                        }
                        </concept>
                }
                </conceptList>
            ,
            for $n in $in/example[exists(node())][.//text()[not(normalize-space() = '')]]
            return
                <example>
                {
                    $n/@type[. = $utilde:EXAMPLE-TYPE/enumeration/@value],
                    $n/@caption[not(. = '')],
                    $n/node()
                }
                </example>
        }
    else ()
};

(:~ Central logic for patching an existing dataset concept
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR concept/@id to update
    @param $effectiveDate   - required. DECOR concept/@effectiveDate to update
    @param $data            - required. DECOR concept xml element containing everything that should be in the updated concept
    @return concept object as xml with json:array set on elements
:)
declare function utilde:patchConcept($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) {

    let $storedConcept          := utilde:getConcept($id, $effectiveDate)
    let $decor                  := $storedConcept/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $check                  :=
        if ($storedConcept) then () else (
            error($errors:BAD_REQUEST, 'Concept with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    
    let $lock                   := utilde:checkConceptAccess($authmap, $decor, $id, $effectiveDate, true())
    
    let $check                  :=
        if ($storedConcept/ancestor::concept/@statusCode = $utilde:STATUSCODES-FINAL) then
            error($errors:BAD_REQUEST, concat('Concept patched be added while any of its parents has one of status: ', string-join($utilde:STATUSCODES-FINAL, ', ')))
        else 
        if ($storedConcept[@statusCode = $utilde:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $utilde:STATUSCODES-FINAL)]) then () else 
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('Concept cannot be patched while it has one of status: ', string-join($utilde:STATUSCODES-FINAL, ', ')))
            )
        else ()
    
    let $check                  := utilde:checkConceptParameters($data, $storedConcept)
    
    let $intention              := if ($storedConcept[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, $intention, $storedConcept)
    
    let $patch                  := utilde:patchConceptParameters($data, $storedConcept)
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $originalConcept        := utilde:getOriginalForConcept($storedConcept)
    let $preparedConcept        := if ($storedConcept[@type[. = 'group'] | concept] | $originalConcept[@type = 'group']) 
        then utilde:prepareGroupForUpdate($storedConcept, $storedConcept)
        else utilde:prepareItemForUpdate($storedConcept, $storedConcept)
    
    let $update                 := update replace $storedConcept with $preparedConcept
    let $update                 := update delete $lock
    
    return ()
};

declare %private function utilde:checkConceptParameters($parameters as element(parameters), $concept as element(concept)*) {

    let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    
    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $allowableInheritPaths  := ('/statusCode', '/canonicalUri', '/expirationDate', '/officialReleaseDate', '/versionLabel', '/comment')
    let $checkn                 :=
        if ($concept[inherit | contains]) then
            if (count($parameters/parameter/@path) = count($parameters/parameter[@path = $allowableInheritPaths])) then () else (
                'A concept that inherits or contains, does not allow patching of ' || string-join($parameters/parameter[not(@path = $allowableInheritPaths)], ', ')
            )
        else (
            if (count($concept/name) - count($parameters/parameter[@op = 'remove'][@path = '/name']) + count($parameters/parameter[@op = 'add'][@path = '/name']) ge 1) then () else (
                'A concept SHALL have at least one name. You cannot remove every name.'
            )
            ,
            let $currentConceptType   := utilde:getOriginalForConcept($concept)/@type
            let $newConceptType       := ($parameters/parameter[@path = '/type'][not(@op = 'remove')]/@value)[last()]
            return
            if (($newConceptType, $currentConceptType)[1] = 'group') then
                if (($parameters/parameter[starts-with(@path, '/valueDomain')][last()])[not(@op = 'remove')]) then
                    'You are adding/replacing a valueDomain and switching to type group. A concept group SHALL NOT have a valueDomain.'
                else
                if (count($concept/valueDomain) - count($parameters/parameter[@op = 'remove'][@path = '/valueDomain']) + count($parameters/parameter[not(@op = 'remove')][@path = '/valueDomain']) lt 1) then () else (
                    if ($currentConceptType = 'group') then
                        'The stored concept is a group but has a valueDomain that you are not removing. A concept group SHALL NOT have a valueDomain.'
                    else (
                        'You are switching to concept type group and the current concept has a valueDomain that you are not removing. A concept group SHALL NOT have a valueDomain.'
                    )
                )
            else (
                if ($concept[concept]) then
                    'A concept item SHALL NOT have child concepts. You cannot switch the type to ''item'' while the group is non-empty'
                else (),
                if (count($concept/valueDomain) - count($parameters/parameter[@op = 'remove'][@path = '/valueDomain']) + count($parameters/parameter[not(@op = 'remove')][@path = '/valueDomain']) ge 1) then () else (
                    'A concept SHALL have a valueDomain. You cannot remove every valueDomain.'
                )
            )
        )

    let $check                  :=
        for $param in $parameters/parameter
        let $op                 := $param/@op
        let $path               := $param/@path
        let $value              := $param/@value
        let $pathpart           := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($concept, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($concept/@statusCode)), ', ')
                )
            )
            case '/type' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if ($value = $utilde:CONCEPT-TYPE/enumeration/@value) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join($utilde:CONCEPT-TYPE/enumeration/@value, ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri'
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/name' 
            case '/synonym'
            case '/desc' 
            case '/source'
            case '/operationalization' 
            case '/rationale' 
            case '/comment' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $pathpart], $op, $path)
                else (
                    'Parameter ' || $op || ' of path ' || $path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/property' return (
                if ($param[count(value/property) = 1]) then
                    if ($param/value/property/@name[not(. = '')]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Property.name SHALL be a non empty string.'
                    )
                else (
                    'Parameter ' || $op || ' of path ' || $path || ' not supported. Input SHALL have exactly one property under value. Found ' || count($param/value/property) 
                ),
                if ($param/value/property[@datatype]) then 
                    if ($param/value/property[@datatype = $utilde:VALUEDOMAIN-TYPE/enumeration/@value]) then () else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || ''' with type ''' || string-join($param/value/property/@datatype, ' ') || '''. Supported are: ' || string-join($utilde:VALUEDOMAIN-TYPE/enumeration/@value, ', ')
                    )
                else ()
            )
            case '/relationship' return ruleslib:checkRelationship($param/value/relationship, $op, $path)
            case '/contains'
            case '/inherit' return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Use /concept/{id}/{effectiveDate}/$inherit to manipulate inherit and contains'
            )
            case '/valueDomain' return (
                if ($op = 'remove') then () else (
                    if ($op = 'replace') then
                        if (count($concept/valueDomain) le 1 or count($concept/valueDomain[@type = $param/value/valueDomain/@type]) le 1) then () else (
                            'Parameter ' || $op || ' ambiguous for ''' || $path || '''. Multiple valueDomains exist. Don''t know what to replace.'
                        )
                    else (),
                    if ($op = 'add') then
                        if ($concept/valueDomain) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. There already is a valueDomain and multiple are not allowed.'
                        else ()
                    else (),
                    
                    ruleslib:checkConceptValueDomain($param/value/valueDomain, $concept, $op, $path)
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     return
        if (empty($checkn) and empty($check)) then () else (
            error($errors:BAD_REQUEST, string-join (($checkn, $check), ' '))
        )

};

declare %private function utilde:patchConceptParameters($parameters as element(parameters), $concept as element(concept)*) {
   
    for $param in $parameters/parameter
    return
        switch ($param/@path)
        case '/statusCode'
        case '/type' return (
            let $attname        := substring-after($param/@path, '/')
            let $new            := attribute {$attname} {$param/@value}
            let $stored         := $concept/@*[name() = $attname]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $concept
            (:case ('remove') return update delete $stored:)
            default return ( (: unknown op :) )
        )
        case '/expirationDate'
        case '/officialReleaseDate'
        case '/canonicalUri'
        case '/versionLabel' return (
            let $attname        := substring-after($param/@path, '/')
            let $new            := attribute {$attname} {$param/@value}
            let $stored         := $concept/@*[name() = $attname]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $concept
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/name' 
        case '/desc' 
        case '/source'
        case '/operationalization' 
        case '/rationale' return (
            (: only one per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
            let $stored         := $concept/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $concept
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/synonym'
        case '/comment' return ( 
            (: multiple possible per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
            let $stored         := $concept/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language][utillib:canonicalizeString(.) = utillib:canonicalizeString($new)]
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $concept
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/relationship' return (
            let $new            := utillib:prepareDatasetRelationshipForUpdate($param/value/relationship)
            let $stored         := $concept/relationship[@ref = $new/@ref]
            let $stored         := if ($new/@flexibility) then $stored[@flexibility = $new/@flexibility] else $stored
            
            return
            switch ($param/@op)
            case ('add') return update insert $new into $concept
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $concept
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/property' return (
            (: multiple possible per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareConceptPropertyForUpdate($param/value/property)
            let $stored         := $concept/property[@name = $new/@name][lower-case(normalize-space(string-join(.//text(), ''))) = lower-case(normalize-space(string-join($new//text(), '')))]
            
            return
            switch ($param/@op)
            case ('add') return update insert $new into $concept
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $concept
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/valueDomain' return (
            (: multiple possible :)
            let $elmname        := substring-after($param/@path, '/')
            
            (: build a temp valueDomain element, if needed with additions :)
            let $checkedValueDomain      :=
                if ( ($param/value/valueDomain[@type = ('code', 'ordinal')] and not($param/value/valueDomain/conceptList[@id | @ref]))
                      and $concept/@statusCode = 'new') then (
                    (: no concept list on a new coded concept, add one now, see AD30-1453 :)
                    <valueDomain>
                    {
                        $param/value/valueDomain/@*,
                        $param/value/valueDomain/node(),
                        <conceptList id="{concat($concept/@id, '.', replace($concept/@effectiveDate,'\D', ''), '.0')}"/>
                    }
                    </valueDomain>
                ) 
                else $param/value/valueDomain
                
            let $new            := utilde:handleConceptValueDomain($checkedValueDomain)
            let $stored         := ($concept/valueDomain[@type = $new/@type] | $concept/valueDomain)[1]
            
            return
            switch ($param/@op)
            case ('add') return update insert $new into $concept
            case ('replace') return if ($stored) then update replace $stored with $new else ()
            case ('remove') return update delete $concept/valueDomain[@type = $new/@type][count(property) = count($new/property)][count(example) = count($new/example)][empty(conceptList | $new/conceptList) or conceptList/(@id | @ref) = $new/conceptList/(@id | @ref)]
            default return ( (: unknown op :) )
        )
        default return ( (: unknown path :) )
};

(:~ Retrieve DECOR concept map list from a transaction or dataset :)
declare function utilde:getConceptAssociationList($id as xs:string, $effectiveDate as xs:string?, $transactionId as xs:string?, $transactionEffectiveDate as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?, $associationMode as xs:string?) {
    
    let $associationMode        := if ($associationMode) then $associationMode else if (empty($transactionId)) then 'normal' else 'all'
    
    let $storedConcept          :=
        if (empty($id)) then () 
        else if (empty($transactionId)) 
        then utilde:getConcept($id, $effectiveDate, $projectVersion, $projectLanguage) 
        else utilde:getTransactionConcept($id, $effectiveDate, $transactionId, $transactionEffectiveDate, $projectVersion, $projectLanguage)
    
    let $datasetConcept         := 
        if ($storedConcept) then
            if (empty($transactionId)) then $storedConcept
            else (
                let $dsid       := $storedConcept/ancestor::representingTemplate/@sourceDataset
                let $dsed       := $storedConcept/ancestor::representingTemplate/@sourceDatasetFlexibility
                
                return utilde:getDatasetConcept($dsid, $dsed, $storedConcept/@ref, $storedConcept/@flexibility, $projectVersion, $projectLanguage)
            )
        else ()
    let $originalConcept        := if ($datasetConcept) then utilde:getOriginalForConcept($datasetConcept) else ()
    
    return if ($storedConcept) then utilde:getConceptAssociations($storedConcept, $originalConcept, false(), $associationMode, true(), $projectLanguage) else ()
};

(:~ Central logic for patching an existing dataset or transaction concept map
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR concept/@id to update
    @param $effectiveDate   - required. DECOR concept/@effectiveDate to update
    @param $data            - required. DECOR concept xml element containing everything that should be in the updated concept
    @return concept object as xml with json:array set on elements
:)
declare function utilde:patchConceptAssociation($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $transactionId as xs:string?, $transactionEffectiveDate as xs:string?, $data as element(parameters)) {

    let $associationMode        := if ($transactionId) then 'all' else 'normal'
        
    let $storedConcept          :=
        if (empty($id)) then () 
        else if (empty($transactionId)) 
        then utilde:getConcept($id, $effectiveDate, (), ()) 
        else utilde:getTransactionConcept($id, $effectiveDate, $transactionId, $transactionEffectiveDate, (), ())
    
    let $check                  :=
        if ($storedConcept) then () 
        else if ($transactionId) then 
            error($errors:BAD_REQUEST, 'Concept with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist in transaction with id ' || $transactionId || ' and effectiveDate ' || $transactionEffectiveDate)
        else (
            error($errors:BAD_REQUEST, 'Concept with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    
    let $decor                  := $storedConcept/ancestor::decor
    
    let $check                  := if (empty($transactionId)) then  utilde:checkConceptAccess($authmap, $decor) else  utilde:checkTransactionConceptAccess($authmap, $decor)
    
    let $datasetConcept         := 
        if ($storedConcept[ancestor::dataset]) then $storedConcept 
        else if ($storedConcept[@id]) then (utilde:getConcept($storedConcept/@id, $storedConcept/@effectiveDate)) 
        else if ($storedConcept[@ref]) then (utilde:getConcept($storedConcept/@ref, $storedConcept/@flexibility[. castable as xs:dateTime])) else ()
    
    let $check                  := utilde:checkConceptAssociationParameters($data, $datasetConcept, $id, $effectiveDate, $transactionId)
    
    let $update                 := utilde:patchConceptAssociationParameters($data, $storedConcept, $datasetConcept, $transactionId)
    
    return utilde:getConceptAssociationList($id, $effectiveDate, $transactionId, $transactionEffectiveDate, (), (), $associationMode)
};

declare %private function utilde:checkConceptAssociationParameters($parameters as element(parameters), $concept as element(concept)*, $id as xs:string, $effectiveDate as xs:string, $transactionId as xs:string?) {

    let $originalConcept        := if ($concept) then utilde:getOriginalForConcept($concept) else ()
    let $originalConceptLists   :=
        for $ref in $originalConcept/valueDomain/conceptList
        return utilde:getOriginalConceptList($ref)
    
    let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    
    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    (: There are three types of associations;
    1. DatasetConcepts. Expect concept/@id=@conceptId, optionally concept/@effectiveDate=@conceptFlexibility, @code+@codeSystem+@displayName + optionally @equivalence
    2. Concept value domain conceptlist. Expect concept/@id=@conceptId, No @conceptFlexibility, @ref + optionally @flexibility and/or @strength
    3. Concept value domain conceptlist concept. Expect concept/@id=@conceptId, No @conceptFlexibility, @code+@codeSystem+@displayName + optionally @equivalence
    :)
    let $check                  :=
        for $param in $parameters/parameter
        let $op                 := $param/@op
        let $path               := $param/@path
        let $elmname            := substring-after($param/@path, '/')
        let $value              := $param/value/*[name() = $elmname]
        return
            switch ($path)
            case '/identifierAssociation' (:fall through :)
            case '/terminologyAssociation' return (
                if (count($value) = 1) then (
                    if ($value[@conceptId = $id]) then (
                        if ($value[@conceptFlexibility = $effectiveDate]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to the concept itself SHALL have a matching effectiveDate ' || $effectiveDate || '. Found  conceptFlexibility "' || $value/@conceptFlexibility || '"'
                        ),
                        if ($param[@valueSet | @flexibility]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a concept SHALL not bind to a value set. Found: "' || string-join(($value/@valueSet, $value/@flexibility), ' - ') || '"'
                        else ()
                    )
                    else
                    if ($value[@conceptId = $originalConceptLists/@id]) then (
                        if ($param[@conceptFlexibility]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a conceptList SHALL not have conceptFlexibility. Found: "' || $value/@conceptFlexibility || '"'
                        else (),
                        if ($param[@code | @codeSystem | @displayName]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a conceptList SHALL not bind to a single code. Found: "' || string-join(($value/@code, $value/@codeSystem, $value/@displayName), ' - ') || '"'
                        else ()
                    )
                    else
                    if ($value[@conceptId = $originalConceptLists/concept/@id]) then (
                        if ($param[@conceptFlexibility]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a conceptList.concept SHALL not have conceptFlexibility. Found: "' || $value/@conceptFlexibility || '"'
                        else (),
                        if ($param[@valueSet | @flexibility]) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association to a conceptList.concept SHALL not bind to a value set. Found: "' || string-join(($value/@valueSet, $value/@flexibility), ' - ') || '"'
                        else ()
                    )
                    else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with conceptId "' || $value/@conceptId || '" SHALL be to the target concept (' || $id || '), or one of its conceptLists, or one of the concepts in such conceptList.'
                    ),
                    if ($value[@strength]) then
                        if ($value[@strength = $utilde:CODINGSTRENGTH-TYPE/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association strength "' || $value/@strength || '" not supported. Supported are: ', string-join($utilde:CODINGSTRENGTH-TYPE/enumeration/@value, ' ')
                        )
                    else (),
                    if ($value[@equivalence]) then 
                        if ($value[@equivalence = $utilde:EQUIVALENCY-TYPE/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association equivalence "' || $value/@equivalence || '" not supported. Supported are: ', string-join($utilde:EQUIVALENCY-TYPE/enumeration/@value, ' ')
                        )
                    else (),
                    if ($op = 'remove') then () else
                    if ($value[@conceptId]) then
                        if ($value[@valueSet]) then (
                            if ($value[@code | @codeSystem | @displayName | @ref]) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with valueSet SHALL NOT have code or codeSystem or displayName or ref.'
                            else (),
                            let $valueSet                   := 
                                if ($value[@valueSet]) then utilvs:getValueSet((), (), (), $value/@valueSet, $value/@flexibility, false(), true()) else ()
                            return
                            if (empty($valueSet)) then 
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with valueSet SHALL point to an existing value set. Could not find: ' || $param/@valueSet
                            else ()
                        )
                        else 
                        if ($value[@code]) then (
                            if ($value[@valueSet | @flexibility | @ref]) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with a code SHALL NOT have valueSet or flexibility or ref.'
                            else 
                            if ($value[@codeSystem][@displayName]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with a code SHALL have codeSystem and displayName.'
                            ),
                            if ($value[matches(@code, '^\S+')]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with a code SHALL NOT contain whitespace. Hint for SNOMED CT expressions: use concept ids only, e.g. 363679005:260686004=312250003,405813007=76752008'
                            ),
                            if ($value[utillib:isOid(@codeSystem)]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association codeSystem SHALL be a valid oid. Found: "' || $value/@codeSystem || '"'
                            )
                        )
                        else
                        if ($value[@ref]) then (
                            if ($value[@valueSet | @flexibility | @code | @codeSystem | @displayName]) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association with a ref SHALL NOT have both valueSet or flexibility or code or codeSystem or displayName.'
                            else (),
                            if ($value[utillib:isOid(@ref)]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association ref SHALL be a valid oid. Found: "' || $value/@ref || '"'
                            )
                        )
                        else 
                        if ($transactionId) then (
                            (: this signals override of anything in the dataset :)
                        ) 
                        else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association SHALL have at least valueSet, code or ref'
                        )
                    else (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Association SHALL have conceptId.'
                    )
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. value SHALL contain a single ' || $elmname || ' element/object. Found: ' || count($value) 
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     
     return
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )
};

declare %private function utilde:patchConceptAssociationParameters($parameters as element(parameters), $concept as element(concept)*, $datasetConcept as element(concept)*, $transactionId as xs:string?) {

    let $decor                  := $concept/ancestor::decor
    
    for $param in $parameters/parameter
    return
        switch ($param/@path)
        case '/terminologyAssociation' return (
            (: only one per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareTerminologyAssociationForUpdate($param/value/*[name() = $elmname], not(empty($transactionId)))
            let $valueSet       := if ($new[@valueSet]) then utilvs:getValueSet((), (), (), $new/@valueSet, $new/@flexibility, false(), true()) else ()
            
            (: if the transaction concept does not have any listings yet for this id, we copy what the dataset concept has first so we can work from there :)
            let $add            :=
                if ($concept[ancestor::transaction][empty(*[@conceptId = $new/@conceptId])]) then (
                    let $datasetConceptAssociations := 
                        for $assoc in utilde:getConceptAssociations($datasetConcept)
                        return
                            if ($assoc[self::terminologyAssociation]) then utillib:prepareTerminologyAssociationForUpdate($assoc, not(empty($transactionId)))
                            else if ($assoc[self::identifierAssociation]) then utillib:prepareIdentifierAssociationForUpdate($assoc, not(empty($transactionId)))
                            else $assoc
                    
                    return if ($datasetConceptAssociations[@conceptId = $new/@conceptId]) then update insert $datasetConceptAssociations[@conceptId = $new/@conceptId] into $concept else () 
                ) 
                else ()
            
            (: should not have multiple connections to the same valueSet (regardless of version), so delete those first in case we're updating strength :)
            (: should not have multiple connections to the same code/codeSystem (regardless of version), so delete those first in case we're updating equivalence :)
            let $delete         := 
                if ($new[@valueSet]) then 
                    if ($transactionId) then (
                        update delete $concept/terminologyAssociation[@conceptId = $new/@conceptId][empty(@valueSet | @code)],
                        update delete $concept/terminologyAssociation[@conceptId = $new/@conceptId][@valueSet = $new/@valueSet]
                    )
                    else (
                        update delete $decor/terminology/terminologyAssociation[@conceptId = $new/@conceptId][@valueSet = $new/@valueSet]
                    )
                else
                if ($new[@code]) then
                    if ($transactionId) then (
                        update delete $concept/terminologyAssociation[@conceptId = $new/@conceptId][empty(@valueSet | @code)],
                        update delete $concept/terminologyAssociation[@conceptId = $new/@conceptId][@code = $new/@code][@codeSystem = $new/@codeSystem]
                    )
                    else (
                        update delete $decor/terminology/terminologyAssociation[@conceptId = $new/@conceptId][@code = $new/@code][@codeSystem = $new/@codeSystem]
                    )
                else if ($transactionId) then  update delete $concept/terminologyAssociation[@conceptId = $new/@conceptId] else ()
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return (
                let $updateTerminology  := if ($decor/terminology) then () else update insert <terminology/> following $decor/ids
                let $updateAssociation  :=
                    if ($transactionId) then update insert $new into $concept
                    else if ($decor/terminology/terminologyAssociation) then update insert $new following $decor/terminology/terminologyAssociation[last()]
                    else if ($decor/terminology/*) then update insert $new preceding $decor/terminology/*[1]
                    else update insert $new into $decor/terminology
                    
                let $updateAssociation          :=
                    if (empty($valueSet)) then () else utilde:addValueSetRef($decor, $valueSet/@ident, $valueSet/@url, $valueSet/@id, $valueSet/@name, ($valueSet/@displayName, $valueSet/@name)[1])
                
                return ()
            )
            case ('remove') return (
                (: if we removed the last terminologyAssociation from a transaction concept then we need to 
                add a marker that makes the 'non-binding' explicit, otherwise the dataset bindings would be reinstated :)
                if ($transactionId) then
                    if ($new[empty(@code | @codeSystem | @valueSet)]) then ((: we were asked to remove an empty terminologyAssociation. this was already done before this line, so nothing left to do :)) else
                    if ($concept[empty(terminologyAssociation[@conceptId = $new/@conceptId])]) then (
                        let $new    := <terminologyAssociation>{$new/@conceptId | $new/@conceptFlexibility | $new/@effectiveDate | $new/@lastModifiedDate | $new/@expirationDate | $new/@officialReleaseDate}</terminologyAssociation>
                        return update insert utillib:prepareTerminologyAssociationForUpdate($new, true()) into $concept
                    ) else ()
                else ()
            )
            default return ( (: unknown op :) )
        )
        case '/identifierAssociation' return (
            (: only one per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareIdentifierAssociationForUpdate($param/value/*[name() = $elmname], not(empty($transactionId)))
            
            (: if the transaction concept does not have any listings yet for this id, we copy what the dataset concept has first so we can work from there :)
            let $add            :=
                if ($concept[ancestor::transaction][empty(*[@conceptId = $new/@conceptId])]) then (
                    let $datasetConceptAssociations := 
                        for $assoc in utilde:getConceptAssociations($datasetConcept)
                        return
                            if ($assoc[self::terminologyAssociation]) then utillib:prepareTerminologyAssociationForUpdate($assoc, not(empty($transactionId)))
                            else if ($assoc[self::identifierAssociation]) then utillib:prepareIdentifierAssociationForUpdate($assoc, not(empty($transactionId)))
                            else $assoc
                    
                    return if ($datasetConceptAssociations) then update insert $datasetConceptAssociations[@conceptId = $new/@conceptId] into $concept else () 
                ) 
                else ()
            
            let $delete         := 
                if ($new[@ref]) then 
                    if ($transactionId) then (
                        update delete $concept/identifierAssociation[@conceptId = $new/@conceptId][empty(@ref)],
                        update delete $concept/identifierAssociation[@conceptId = $new/@conceptId][@ref = $new/@ref]
                    )
                    else (
                        update delete $decor/ids/identifierAssociation[@conceptId = $new/@conceptId][@ref = $new/@ref]
                    )
                else
                if ($transactionId) then update delete $concept/identifierAssociation[@conceptId = $new/@conceptId] else ()
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return (
                if ($transactionId) then update insert $new into $concept
                else if ($decor/ids/*) then update insert $new following $decor/ids/*[last()]
                else update insert $new into $decor/ids
            )
            case ('remove') return (
                (: if we removed the last identifierAssociation from a transaction concept then we need to 
                add a marker that makes the 'non-binding' explicit, otherwise the dataset bindings would be reinstated :)
                if ($transactionId) then
                    if ($new[empty(@ref)]) then ((: we were asked to remove an empty identifierAssociation. this was already done before this line, so nothing left to do :)) else
                    if ($concept[empty(identifierAssociation[@conceptId = $new/@conceptId])]) then (
                        let $new    := <identifierAssociation>{$new/@conceptId | $new/@conceptFlexibility | $new/@effectiveDate | $new/@lastModifiedDate | $new/@expirationDate | $new/@officialReleaseDate}</identifierAssociation>
                        return update insert utillib:prepareIdentifierAssociationForUpdate($new, true()) into $concept
                    ) 
                    else ()
                else ()
            )
            default return ( (: unknown op :) )
        )
        default return ( (: unknown path :) )
};

(:~ Return zero or more dataset concepts as-is

    @param $id           - required. Identifier of the concept to retrieve
    @param $flexibility  - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
    @return Matching dataset concepts
    @since 2025-05-01
:)
declare function utilde:getConcept($id as xs:string, $flexibility as xs:string?) as element(concept)* {
    utilde:getConcept($id, $flexibility, (), ())
};

(:~ Return zero or more dataset concepts as-is

    @param $id            - required. Identifier of the value set to retrieve
    @param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
    @param $decorRelease  - optional. if empty defaults to current version. if valued then the value set will come explicitly from that archived project version which is expected to be a compiled version
    @return Matching dataset concepts
    @since 2025-05-01
:)
declare function utilde:getConcept($id as xs:string, $flexibility as xs:string?, $decorRelease as xs:string?) as element(concept)* {
    utilde:getConcept($id, $flexibility, $decorRelease, ())
};

(:~ Return zero or more dataset concepts as-is

    @param $id            - required. Identifier of the value set to retrieve
    @param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
    @param $decorRelease  - optional. if empty defaults to current version. if valued then the value set will come explicitly from that archived project version which is expected to be a compiled version
    @param $decorLanguage - optional. Language compilation. Defaults to project/@defaultLanguage.
    @return Matching dataset concepts
    @since 2025-05-01
:)
declare function utilde:getConcept($id as xs:string, $flexibility as xs:string?, $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(concept)* {

    let $concepts               :=    
        (: get concept in a release icm with language :)
        if ($decorRelease castable as xs:dateTime) then (
            for $concept in $setlib:colDecorVersion//concept[@id = $id]
            let $versionDate    := $concept/ancestor::decor/@versionDate
            group by $versionDate 
            return (
                if ($versionDate = $decorRelease) then (
                let $concept    :=
                    if ($flexibility castable as xs:dateTime) then $concept[@effectiveDate = $flexibility] 
                    else if ($flexibility) then $concept[@effectiveDate = string(max($concept/xs:dateTime(@effectiveDate)))] 
                    else $concept
                    return (
                        if ($decorLanguage = '*') then $concept
                        else if (empty($decorLanguage)) then ($concept[ancestor::decor[@language = project/@defaultLanguage]], $concept)[1]
                        else $concept[ancestor::decor[@language = $decorLanguage]]
                    )
                )
                else ()
            )
        )
        (: get concept in live version :)
        else (
            if ($flexibility castable as xs:dateTime) then $setlib:colDecorData//concept[@id = $id][@effectiveDate = $flexibility]
            else if ($flexibility) then (
                let $concepts   := $setlib:colDecorData//concept[@id = $id]
                return $concepts[@effectiveDate = string(max($concepts/xs:dateTime(@effectiveDate)))]
            )
            else $setlib:colDecorData//concept[@id = $id]
        )

    return $concepts[ancestor::datasets][not(ancestor::history)]
};

(:~ Return zero or more concepts as is from a specific dataset

    @param $datasetId       - required dataset/@id
    @param $datasetEd       - optional dataset/@effectiveDate
    @param $id              - required concept/@id
    @param $flexibility     - optional concept/@effectiveDate
    @param $decorRelease    - optional decor/@versionDate
    @param $decorLanguage   - optional decor/@language. Only relevant with parameter $decorVersion. Empty uses decor/project/@defaultLanguage, '*' selects all versions, e.g. 'en-US' selects the US English version (if available)
    Released/compiled matches could be present in multiple languages. Use the parameters @language to select the right one.
    @return exactly 1 live concept, or 1 or more archived DECOR version instances or nothing if not found
    @since 2015-04-29
:)
declare function utilde:getDatasetConcept($datasetId as xs:string, $datasetEd as xs:string?, $id as xs:string, $flexibility as xs:string?, $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(concept)* {
    
    let $dataset                := utillib:getDataset($datasetId , $datasetEd, $decorRelease, $decorLanguage)
    
    let $concepts               := 
        if ($flexibility castable as xs:dateTime) then $dataset//concept[@id = $id][@effectiveDate = $flexibility]
            else if ($flexibility) then (
                let $concepts   := $dataset//concept[@id = $id]
                return $concepts[@effectiveDate = string(max($concepts/xs:dateTime(@effectiveDate)))]
            )
            else $dataset//concept[@id = $id]
    
    return $concepts[not(ancestor::history)]
};

(:~ Return live transaction concept based on id and optionally effectiveDate + transaction id and optionally effectiveDate. If an effectiveDate is empty, the latest version is returned
    
    @param $id              - required concept/@id
    @param $flexibility     - optional concept/@effectiveDate
    @param $transactionId   - required transaction/@id
    @param $transactionEd   - optional transaction/@effectiveDate
    @return exactly 1 concept or nothing if not found
    @since 2018-06-21
:)
declare function utilde:getTransactionConcept($id as xs:string, $flexibility as xs:string?, $transactionId as xs:string?, $transactionEd as xs:string?) as element(concept)* {
    utilde:getTransactionConcept($id, $flexibility, $transactionId, $transactionEd, (), ())
};

(:~ Return live transaction concept if param decorVersion is not a dateTime OR 
    compiled, archived/released DECOR transaction concept based on concept/@id|@effectiveDate and decor/@versionDate.
    Released/compiled matches could be present in multiple languages. Use the parameters @language to select the right one.
    
    @param $id              - required concept/@id
    @param $flexibility     - optional concept/@effectiveDate
    @param $transactionId   - required transaction/@id
    @param $transactionEd   - optional transaction/@effectiveDate
    @param $decorRelease    - optional decor/@versionDate
    @param $decorLanguage   - optional decor/@language. Only relevant with parameter $decorVersion. Empty uses decor/project/@defaultLanguage, '*' selects all versions, e.g. 'en-US' selects the US English version (if available)
    Released/compiled matches could be present in multiple languages. Use the parameters @language to select the right one.
    @return exactly 1 live concept, or 1 or more archived DECOR version instances or nothing if not found
    @since 2015-04-29
:)
declare function utilde:getTransactionConcept($id as xs:string, $flexibility as xs:string?, $transactionId as xs:string?, $transactionEd as xs:string?, $decorRelease as xs:string?) as element(concept)* {
    utilde:getTransactionConcept($id, $flexibility, $transactionId, $transactionEd, $decorRelease, ())
};

declare function utilde:getTransactionConcept($id as xs:string, $flexibility as xs:string?, $transactionId as xs:string?, $transactionEd as xs:string?, $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(concept)* {
    let $concepts               := 
        if (empty($transactionId)) then $setlib:colDecorData//concept[@ref = $id][ancestor::transaction]
        else utillib:getTransaction($transactionId, $transactionEd, $decorRelease, $decorLanguage)//concept[@ref = $id]
    
    (: Note that we currently do not implement doing much with the concept effectivedate. In a normal dataset and hence in a normal transaction there will be only 1 version of a given concept so we should not really worry about getting 'the right version' :)
    return
        if ($concepts[@flexibility = $flexibility]) then $concepts[@flexibility = $flexibility][1] else
        if ($flexibility castable as xs:dateTime or $concepts[2]) then 
            for $concept in $concepts
            let $dsid           := $concept/ancestor::representingTemplate/@sourceDataset
            let $dsed           := $concept/ancestor::representingTemplate/@sourceDatasetFlexibility
            let $trdeed         := $concept/@flexibility
            group by $dsid, $dsed, $trdeed
            return (
                let $datasetConcept := utilde:getDatasetConcept($dsid, $dsed, $id, $trdeed, $decorRelease, $decorLanguage)
                return
                    if ($flexibility castable as xs:dateTime) then if ($datasetConcept[@effectiveDate = $flexibility]) then $concept[1] else ()
                    else $concepts[not(@flexibility castable as xs:dateTime)] | $concepts[@flexibility = max($datasetConcept/xs:dateTime(@effectiveDate))]
            )
        else $concepts
};

(:~ Return the original conceptList. If input has @id, return input. If input has @ref, find original with @id :)
declare function utilde:getOriginalConceptList($conceptList as element(conceptList)?) as element(conceptList)? {

    if ($conceptList[@ref]) then (
        let $projectPrefix      := 
            if ($conceptList/ancestor::decor) then $conceptList/ancestor::decor/project/@prefix
            (:when called through getOriginalConcept the prefix is on a parent inherit or contains element:)
            else($conceptList/ancestor::inherit/@prefix | $conceptList/ancestor::contains/@prefix)[1]
        
        (:get original from same project:)
        let $originalConceptList := $setlib:colDecorData//conceptList[@id = $conceptList/@ref][ancestor::datasets][not(ancestor::history)][ancestor::decor[project[@prefix=$projectPrefix]]]
        (:got original, return it:) (:could not find original, just return reference:)
        return if ($originalConceptList) then $originalConceptList[1] else $conceptList           
    )
    (: original or empty :)
    else $conceptList
};

(:~  retrieve original concept, no added elements, no preservation of comments that may have been added in inheritance trail. 
:   Just comments of original. Caller needs to keep track of current added comments
:   input: decor concept
:   returns: the original concept
:)
declare function utilde:getOriginalForConcept($concept as element(concept)?) as element(concept)? {
    
    if ($concept[inherit | contains]) then (
        let $ref                := if ($concept[inherit]) then $concept/inherit/@ref           else $concept/contains/@ref
        let $eff                := if ($concept[inherit]) then $concept/inherit/@effectiveDate else $concept/contains/@flexibility 
        
        let $concepts           := utilde:getConcept($ref, $eff, (), ())
        let $error              := 
            if (count($concepts) le 1) then () else (
                error($errors:SERVER_ERROR, 'Concept id="' || $ref || '" effectiveDate="' || $eff || '" gave multiple hits. Expected exactly 1. Found in: ' || string-join(distinct-values($concepts/ancestor::decor/project/@prefix), ' '))
            )
        return utilde:getOriginalForConcept($concepts)
    )
    else $concept
};

(:~ Recursive function for retrieving the basic concept info for a concept hierarchy.
   Used for creating dataset navigation.
   
   TODO A concept can be in any of these states:
    1. concept[@added]              -- a concept that was willfully added under an inherited group, 
                                        concept actually lives in the dataset
    2. concept[@absent]             -- a concept that was willfully omitted under an inherited group, 
                                        concept actually lives in the dataset and could be activated by deleting @absent at any time
    3. concept[@missing-in-source]  -- a concept exists in the current dataset, but not in the original concept group that the parent of the current concept inherits from, 
                                        concept does not live in the current dataset, but is created runtime to flag this exceptional state. 
                                        There are two possible reasons for this situation, both can only occur post inheriting:
                                        - Someone hand-edited the origin in the DECOR project file (you cannot do this using ART)
                                        - Someone inherited from a concept that inherits and this concept was updated to inherit from a different concept group
    
    A concept could have both @absent and @missing-in-source.
    
    4. concept[@missing-in-target]  -- a concept exists in the original concept group that the parent of the current concept inherits from, but not in the current dataset, 
                                        concept does not live in the current dataset, but is created runtime to flag this exceptional state. 
                                        There are two possible reasons for this situation, both can only occur post inheriting:
                                        - Someone edited the origin in the DECOR project file and added the concept (group)
                                        - Someone inherited from a concept that inherits and this concept was updated to inherit from a different concept group
    
    5. If the concept does not have any of the markers above, it should be considered a normal concept (group)
:)
declare function utilde:conceptBasics($concept as element(), $propertiesMap as map(*)?) as element(concept) {

    let $originalConcept        := if ($concept[name]) then $concept else utilde:getOriginalForConcept($concept)
    let $inheritConcept         := if ($concept[inherit]) then utilde:getConcept($concept/inherit/@ref, $concept/inherit/@effectiveDate) else ()
    
    return
        <concept>
        {
            (:useful for targeted nagivation in the tree. alternative is keeping @id/@effectiveDate together all the time:)
            attribute navkey {util:uuid()},
            $originalConcept/@type,
            $concept/(@* except (@type | @navkey)),
            if ($concept/inherit) then utilde:conceptExpandedInherit($concept/inherit, $inheritConcept, $originalConcept) else (),
            $concept/contains,
            $originalConcept/name,
            $originalConcept/synonym,
            if ($propertiesMap?desc) then $originalConcept/desc else (),
            if ($propertiesMap?source) then $originalConcept/source else (),
            if ($propertiesMap?rationale) then $originalConcept/source else (),
            if ($propertiesMap?comment) then $concept[inherit | contains]/comment[.//text()] | $originalConcept/comment else (),
            if ($propertiesMap?property) then $originalConcept/property else $originalConcept/property[@name = 'Keyword'],
            if ($propertiesMap?inheritedcontains) then (
                for $ic in $inheritConcept/contains
                return
                    <inheritedContains>
                    {
                        $ic/@*,
                        $ic/*
                    }
                    </inheritedContains>
            ) else (),
            if ($propertiesMap?relationship) then (
                (:new since 2015-04-21:)
                for $node in $originalConcept/relationship
                let $relatedConcept           := utilde:getConcept($node/@ref, $node/@flexibility)
                let $relatedOriginalConcept   := utilde:getOriginalForConcept($relatedConcept)
                return
                    <relationship>
                    {
                        $node/@type, $node/@ref, $node/@flexibility, $node/@refdisplay,
                        $relatedConcept/ancestor::decor/project/@prefix,
                        if ($relatedConcept) then (
                            attribute datasetId {$relatedConcept/ancestor::dataset/@id},
                            attribute datasetEffectiveDate {$relatedConcept/ancestor::dataset/@effectiveDate},
                            attribute datasetStatusCode {$relatedConcept/ancestor::dataset/@statusCode},
                            if ($relatedConcept/ancestor::dataset[@expirationDate]) then attribute datasetExpirationDate {$relatedConcept/ancestor::dataset/@expirationDate} else (),
                            if ($relatedConcept/ancestor::dataset[@versionLabel]) then attribute datasetVersionLabel {$relatedConcept/ancestor::dataset/@versionLabel} else (),
                            attribute iType {$relatedOriginalConcept/@type}, 
                            attribute iStatusCode {$relatedConcept/@statusCode}, 
                            attribute iEffectiveDate {$relatedConcept/@effectiveDate},
                            if ($relatedConcept[@expirationDate]) then attribute iExpirationDate {$relatedConcept/@expirationDate} else (),
                            if ($relatedConcept[@versionLabel]) then attribute iVersionLabel {$relatedConcept/@versionLabel} else ()
                        ) else ()
                        ,
                        $relatedOriginalConcept/name
                    }
                    </relationship>
            ) else (),
            if ($propertiesMap?valuedomain) then 
                for $valueDomain in $originalConcept/valueDomain
                return
                    <valueDomain>
                    {
                        $valueDomain/@*, 
                        $valueDomain/conceptList,
                        $valueDomain/property[@*[string-length() gt 0]]
                        ,
                        for $ex in $valueDomain/example
                        return
                            <example type="{($ex/@type, 'neutral')[1]}">{$ex/@caption, $ex/node()}</example>
                    }
                    </valueDomain>
             else ()
            ,
            if ($concept[contains]) then ( (: don't venture into potential circular references :) ) else (
                for $c in $concept/concept
                return utilde:conceptBasics($c, $propertiesMap)
            )
        }
        </concept>
};

(:~ Recursive function for retrieving the basic concept info for a concept hierarchy in the context of a representingTemplate.
   Adds @absent='true' to concepts not in representingTemplate.
   Used for creating dataset navigation and  by transaction editor
   In a number of cases like RetrieveTransaction and template mapping we do not care about absent concepts to just omit those when $fullTree=false()
   Note: only active concepts are traversed. A concept with an absent parent is not retrieved. This situation should never occur.
:)
declare function utilde:transactionConceptBasics($concept as element(concept), $representingTemplate as element(representingTemplate), $fullTree as xs:boolean, $defaultLanguage as xs:string, $propertiesMap as map(*)?) as element(concept)? {

    let $matchingConcept        := $representingTemplate/concept[@ref=$concept/@id] 
    let $matchingConcept        := if ($matchingConcept[@flexibility]) then $matchingConcept[@flexibility=$concept/@effectiveDate][1] else $matchingConcept[1]
    
    return
        if ($matchingConcept or $fullTree) then (
            let $originalConcept:= if ($concept[name]) then $concept else utilde:getOriginalForConcept($concept)
            let $isMandatory    := $matchingConcept/@isMandatory='true'
            let $conditions     := 
                for $condition in $matchingConcept/condition
                let $isMandatory:= $condition/@isMandatory='true'
                return
                    <condition>
                    {
                        attribute minimumMultiplicity {utillib:getMinimumMultiplicity($condition)},
                        attribute maximumMultiplicity {utillib:getMaximumMultiplicity($condition)},
                        attribute conformance {if ($isMandatory) then 'M' else ($condition/@conformance)},
                        attribute isMandatory {$isMandatory},
                        $condition/desc,
                        if ($condition[desc]) then () else if ($condition[string-length(.)=0]) then () else <desc language="{$defaultLanguage}">{$condition/text()}</desc>
                    }
                    </condition>
            
            return 
                <concept>
                {
                    (:useful for targeted nagivation in the tree. alternative is keeping @id/@effectiveDate together all the time:)
                    attribute navkey {util:uuid()},
                    $originalConcept/@type,
                    $concept/(@* except (@type | @ref | @flexibility | @navkey))
                    ,
                    if ($matchingConcept) then ((:purposefully do not add anything here!!:)) else (attribute absent {'true'}),
                    attribute minimumMultiplicity {if ($matchingConcept) then utillib:getMinimumMultiplicity($matchingConcept) else ('0')},
                    attribute maximumMultiplicity {if ($matchingConcept) then utillib:getMaximumMultiplicity($matchingConcept) else ('*')},
                    attribute conformance {if ($isMandatory) then 'M' else ($matchingConcept/@conformance)},
                    attribute isMandatory {$isMandatory},
                    $matchingConcept/@enableBehavior,
                    $concept/inherit,
                    $concept/contains,
                    $originalConcept/name,
                    $originalConcept/synonym,
                    if ($propertiesMap?desc) then $originalConcept/desc else (),
                    if ($propertiesMap?source) then $originalConcept/source else (),
                    if ($propertiesMap?rationale) then $originalConcept/source else (),
                    if ($propertiesMap?comment) then $concept[inherit | contains]/comment[.//text()] | $originalConcept/comment else (),
                    if ($propertiesMap?property) then $originalConcept/property else $originalConcept/property[@name = 'Keyword'],
                    if ($propertiesMap?relationship) then (
                        for $node in $originalConcept/relationship
                        let $relatedConcept           := utilde:getConcept($node/@ref, $node/@flexibility)
                        let $relatedOriginalConcept   := utilde:getOriginalForConcept($relatedConcept)
                        return
                            <relationship>
                            {
                                $node/@type, $node/@ref, $node/@flexibility, $node/@refdisplay,
                                $relatedConcept/ancestor::decor/project/@prefix,
                                if ($relatedConcept) then (
                                    attribute datasetId {$relatedConcept/ancestor::dataset/@id},
                                    attribute datasetEffectiveDate {$relatedConcept/ancestor::dataset/@effectiveDate},
                                    attribute datasetStatusCode {$relatedConcept/ancestor::dataset/@statusCode},
                                    if ($relatedConcept/ancestor::dataset[@expirationDate]) then attribute datasetExpirationDate {$relatedConcept/ancestor::dataset/@expirationDate} else (),
                                    if ($relatedConcept/ancestor::dataset[@versionLabel]) then attribute datasetVersionLabel {$relatedConcept/ancestor::dataset/@versionLabel} else (),
                                    attribute iType {$relatedOriginalConcept/@type}, 
                                    attribute iStatusCode {$relatedConcept/@statusCode}, 
                                    attribute iEffectiveDate {$relatedConcept/@effectiveDate},
                                    if ($relatedConcept[@expirationDate]) then attribute iExpirationDate {$relatedConcept/@expirationDate} else (),
                                    if ($relatedConcept[@versionLabel]) then attribute iVersionLabel {$relatedConcept/@versionLabel} else ()
                                ) else ()
                                ,
                                $relatedOriginalConcept/name
                            }
                            </relationship>
                    ) else (),
                    if ($propertiesMap?valuedomain) then
                        for $valueDomain in $originalConcept/valueDomain
                        return
                            <valueDomain>
                            {
                                $valueDomain/@*, 
                                $valueDomain/conceptList,
                                $valueDomain/property[@*[string-length() gt 0]]
                                ,
                                for $ex in $valueDomain/example
                                return <example type="{($ex/@type, 'neutral')[1]}">{$ex/@caption, $ex/node()}</example>
                            }
                            </valueDomain>
                    else (),
                    $matchingConcept/context,
                    $conditions,
                    $matchingConcept/enableWhen,
                    $matchingConcept/terminologyAssociation,
                    $matchingConcept/identifierAssociation,
                    if ($concept[contains]) then ( (: don't venture into potential circular references :) ) else (
                        for $c in $concept/concept
                        return utilde:transactionConceptBasics($c, $representingTemplate, $fullTree, $defaultLanguage, $propertiesMap)
                    )
                }
                </concept>
        ) else ()
};

(:~ Get .//terminologyAssociation and identifierAssociation element directly from the $concept, or if none exist connected to the dataset concept. 
    Note that an incoming $concept might be a compiled $concept, a regular dataset concept or a transaction concept.
:)
declare function utilde:getConceptAssociations($concept as element(concept)?) as element()* {
    let $datasetConcept         :=
        if ($concept[ancestor::dataset]) then $concept else
        if ($concept[ancestor::transaction]) then (utillib:getDataset($concept/ancestor::representingTemplate/@sourceDataset, $concept/ancestor::representingTemplate/@sourceDatasetFlexibility)//concept[@id = $concept/@ref]) else
        if ($concept[@id]) then (utilde:getConcept($concept/@id, $concept/@effectiveDate)) else
        if ($concept[@ref]) then (utilde:getConcept($concept/@ref, $concept/@flexibility)) else ()
    return utilde:getConceptAssociations($concept, utilde:getOriginalForConcept($datasetConcept), false())
};

(:~ Get .//terminologyAssociation and identifierAssociation element directly from the $concept, or if none exist connected to the dataset concept. 
   Note that an incoming $concept might be a compiled $concept, a regular dataset concept or a transaction concept.
   @param $concept DECOR dataset concept
   @param $originalConcept the original dataset concept
   @param $allAssociations only looks for terminologyAssociation and identifierAssociation if false. Otherwise also includes associations to transaction, issue, template, and other concepts
:)
declare function utilde:getConceptAssociations($concept as element(concept)?, $originalConcept as element(concept)?, $allAssociations as xs:boolean) as element()* {
    if (empty($concept)) then () 
    
    else if ($concept/valueSet/terminologyAssociation | $concept/terminologyAssociation | $concept/identifierAssociation) then (
        (: if for some reason sent us a concept with contained terminologyAssociation then we should assume this is the result of utilde:getFullConcept and just return what was 'calculated' there.
            caveat: utilde:getFullConcept does not keep the conceptList/concept associations, only concept and valueSet.
        :)
        let $decor              := $concept/ancestor::decor
        let $language           := ($decor/project/@defaultLanguage, 'en-US')[1]
        let $associations       := $concept/valueSet/terminologyAssociation | $concept/terminologyAssociation | $concept/identifierAssociation
        (: build map of oid names. is more costly when you do that deep down :)
        let $oidnamemap         := utilde:buildOidNameMap($associations/(@ref, @codeSystem), $language, $decor)

        return utilde:conceptExpandedAssociation($concept, $associations, $oidnamemap, $language)
    )
    
    else if ($concept[ancestor::transaction]) then ()
    
    else (
        let $decor              := $concept/ancestor::decor
        (: if for some reason someone sent us a concept from memory and not from the db, then we will not have a decor ancestor... :)
        let $language           := ($decor/project/@defaultLanguage, 'en-US')[1]
        let $datasetConcept     :=
            if ($concept[ancestor::dataset]) then $concept else
            if ($concept[ancestor::transaction]) then (utillib:getDataset($concept/ancestor::representingTemplate/@sourceDataset, $concept/ancestor::representingTemplate/@sourceDatasetFlexibility)//concept[@id = $concept/@ref]) else
            if ($concept[@id]) then (utilde:getConcept($concept/@id, $concept/@effectiveDate)) else
            if ($concept[@ref]) then (utilde:getConcept($concept/@ref, $concept/@flexibility)) else ()
        let $originalConcept    := if ($originalConcept) then $originalConcept else utilde:getOriginalForConcept($datasetConcept)
        
        let $decor              :=  if ($decor) then $decor else $datasetConcept/ancestor::decor
        
        let $conceptLists       :=
            for $conceptList in $originalConcept/valueDomain/conceptList
            return utilde:getOriginalConceptList($conceptList)
        
        (: map for all concept(List) ids, without any entry value :)
        let $oidset             := distinct-values($concept/@id | $datasetConcept/@id | $originalConcept/@id | $conceptLists//@id)
        
        let $terminologyAssociations    :=
            for $association in (   $decor/terminology/terminologyAssociation[@conceptId = $oidset][not(@conceptFlexibility castable as xs:dateTime)] | 
                                    $decor/terminology/terminologyAssociation[@conceptId = $datasetConcept/@id][@conceptFlexibility = $datasetConcept/@effectiveDate] | 
                                    $decor/terminology/terminologyAssociation[@conceptId = $originalConcept/@id][@conceptFlexibility = $originalConcept/@effectiveDate]
                                    )
            let $associationType := $association/concat(@conceptId,@valueSet,@flexibility,@code,@codeSystem)
            group by $associationType
            return $association[1]
        
        let $identifierAssociations     :=
            if ($originalConcept[valueDomain[@type= 'identifier']]) then (
                for $association in $decor/ids/identifierAssociation[@conceptId = $oidset][not(@conceptFlexibility castable as xs:dateTime)] | 
                                    $decor/ids/identifierAssociation[@conceptId = $datasetConcept/@id][@conceptFlexibility = $datasetConcept/@effectiveDate] | 
                                    $decor/ids/identifierAssociation[@conceptId = $originalConcept/@id][@conceptFlexibility = $originalConcept/@effectiveDate]
                let $associationType := $association/concat(@conceptId,@ref)
                group by $associationType
                return $association[1]
            ) else ()
        (: build map of oid names. is more costly when you do that deep down :)
        let $oidnamemap         := utilde:buildOidNameMap(($terminologyAssociations/@codeSystem | $identifierAssociations/@ref), $language, $decor)
    
        return (
            utilde:conceptExpandedAssociation($concept, $terminologyAssociations | $identifierAssociations, $oidnamemap, $decor/project/@defaultLanguage)
            ,
            (: associations of a concept with a concept, transaction, template and issue would be part of different calls before. As of ART-DECOR 3 we now regard them as associations of the concept :)
            if ($allAssociations) then (
                let $isLatestDataset    := utillib:getDataset($datasetConcept/ancestor::dataset/@id, ())/@effectiveDate = $datasetConcept/ancestor::dataset/@effectiveDate
                let $transactions       :=
                    $setlib:colDecorData//concept[@ref = $datasetConcept/@id][ancestor::representingTemplate[@sourceDataset = $datasetConcept/ancestor::dataset/@id][@sourceDatasetFlexibility = $datasetConcept/ancestor::dataset/@effectiveDate]]
                let $transactions       :=
                    if ($isLatestDataset) then
                        $transactions | $setlib:colDecorData//concept[@ref = $datasetConcept/@id][ancestor::representingTemplate[@sourceDataset = $datasetConcept/ancestor::dataset/@id][empty(@sourceDatasetFlexibility)]]
                    else $transactions
                
                (: returns a concept associated with a transaction :)
                for $association in     $transactions
                let $associationType    := $association/concat($association/ancestor::transaction[1]/@id, $association/ancestor::transaction[1]/@effectiveDate, $association/@ref)
                group by $associationType
                return utillib:doTransactionAssociation($concept, $association[1])
                ,
                (: returns a concept associated with a template :)
                for $association in     $setlib:colDecorData//templateAssociation/concept[@ref = $datasetConcept/@id][@effectiveDate = $datasetConcept/@effectiveDate]
                let $associationType    := $association/concat($association/parent::templateAssociation/@templateId, $association/parent::templateAssociation/@effectiveDate, $association/@elementId, $association/@elementPath)
                group by $associationType
                return utillib:doTemplateAssociation($concept, $association[1])
                ,
                (: returns a concept associated with a questionnaire :)
                for $association in     $setlib:colDecorData//questionnaireAssociation/concept[@ref = $datasetConcept/@id][@effectiveDate = $datasetConcept/@effectiveDate]
                let $associationType    := $association/concat($association/parent::questionnaireAssociation/@templateId, $association/parent::questionnaireAssociation/@effectiveDate, $association/@elementId, $association/@elementPath)
                group by $associationType
                return utillib:doQuestionnaireAssociation($concept, $association[1])
                ,
                (: returns a concept associated with another concept through an inherit, contains, or relationship :)
                for $association in     (   $setlib:colDecorData//inherit[@ref = $datasetConcept/@id][@effectiveDate = $datasetConcept/@effectiveDate] |
                                            $setlib:colDecorData//contains[@ref = $datasetConcept/@id][@flexibility = $datasetConcept/@effectiveDate] |
                                            $setlib:colDecorData//relationship[@ref = $datasetConcept/@id][@flexibility = $datasetConcept/@effectiveDate]
                                        )
                let $associationType    := $association/concat($association/parent::concept[1]/@id, $association/parent::concept[1]/@effectiveDate)
                group by $associationType
                let $targetObjectNames  := if ($association[1]/parent::concept[1]/name) then $association[1]/parent::concept[1]/name else if ($datasetConcept/name) then $datasetConcept/name else $originalConcept/name
                return utillib:doConceptAssociation($concept, $association[1], $targetObjectNames)
                    
            ) else ()
        )
    )
};

(:~ Function for retrieving list of terminology associations for (transaction) concept and contained conceptList/concept Modes: all, diff, normal (default)
    - all: retrieve any dataset and transaction terminologyAssociations and identifierAssociations for this concept
    - diff: as 'all', but trimmed only to those 
        - in any transactions if this a dataset concept ($trid is empty)
        - at dataset level if this a transaction concept ($trid is not empty)
    - normal (default): retrieve any terminologyAssociations and identifierAssociations for this concept at the appropriate level
        - in transaction ($trid is not empty)
        - at dataset level ($trid is empty)
:)
declare function utilde:getConceptAssociations($concept as element(concept)?, $originalConcept as element(concept)?, $allAssociations as xs:boolean, $mode as xs:string?, $doNameForOID as xs:boolean, $language as xs:string?) as element(associations) {

    let $mode                   := if ($mode = ('all', 'diff')) then $mode else 'normal'
    let $transactionConcept     := $concept[ancestor::transaction]
    let $datasetConcept         := 
        if ($concept[ancestor::dataset]) then $concept else
        if ($concept[@id]) then (utilde:getConcept($concept/@id, $concept/@effectiveDate)) else
        if ($concept[@ref]) then (utilde:getConcept($concept/@ref, $concept/@flexibility[. castable as xs:dateTime])) else ()
    let $originalConcept        := if ($datasetConcept) then utilde:getOriginalForConcept($datasetConcept) else ()
    
    let $conceptId              := $datasetConcept/@id
    let $conceptEffectiveDate   := $datasetConcept/@effectiveDate
    let $transactionId          := $transactionConcept/ancestor::transaction[1]/@id
    let $transactionEffectiveDate:= $transactionConcept/ancestor::transaction[1]/@effectiveDate
    
    let $decor                  := $concept/ancestor::decor
    let $language               := if ($language) then $language else $decor/project/@defaultLanguage
    
    let $datasetConceptAssociations     := utilde:getConceptAssociations($datasetConcept, $originalConcept, $allAssociations)
    
    let $transactionConceptAssocations  := 
        if (empty($concept))        then () else
        if ($transactionConcept)    then (
            (: this is a transaction concept :)
            switch ($mode)
            case 'all' return utilde:getConceptAssociations($transactionConcept, $originalConcept, false())
            default return utilde:getConceptAssociations($concept, $originalConcept, false())
        ) else (
            (: this is a dataset concept :)
            switch ($mode)
            case 'diff' return (
                for $tc in utilde:getTransactionConcept($conceptId, $conceptEffectiveDate, (), (), (), ())
                return utilde:getConceptAssociations($tc, $originalConcept, false())
            )
            default return ()
        )
    
        (: logic for transaction concepts: 
            - get associations from transaction concept if they exist else get from dataset concept
            - note that a transaction concept association might be 'empty', i.e. have no code/valueSet/ref. If that's the case then this counts as 'do not use dataset association(s), if any'
                This is relevant e.g. when you have a dataset conceptList with concepts, but in the transaction you do not support some of the concepts. In this case there's no real overriding association possible
        :)
        (: TODO: smaller functions :)
        let $associations       :=
            switch ($mode)
            (: 
            - all: retrieve any dataset and transaction terminologyAssociations and identifierAssociations for this concept
            :)
            case 'all' return (
                (: mark active if this is transaction concept only :)
                if ($transactionConcept) then 
                    for $tca in $datasetConceptAssociations
                    let $tcaId  : = 
                        if (not($originalConcept/@id = $transactionConcept/@ref) and $originalConcept/@id = $tca/@conceptId) then $transactionConcept/@ref else $tca/@conceptId
                    return
                        element {name($tca)} {
                            (: a dataset association is active in absence of an overriding transaction association :)
                            attribute active {empty($transactionConceptAssocations[name() = name($tca)][@conceptId = $tcaId])},
                            $tca/(@* except (@active | @*[starts-with(name(), 'transaction')]))
                        }
                else $datasetConceptAssociations
                ,
                (: Mark the thing with the transaction id/effectiveDate to smooth downstream processing :)
                for $tca in $transactionConceptAssocations
                return
                    element {name($tca)} {
                        (: a transaction association is always active and overrides and dataset association :)
                        attribute active {'true'},
                        $tca/(@* except (@active | @*[starts-with(name(), 'transaction')])),
                        for $attr in $transactionConcept/ancestor::transaction[1]/(@id, @effectiveDate, @versionLabel, @statusCode, @expirationDate)
                        return attribute {'transaction' || upper-case(substring(local-name($attr), 1, 1)) || substring(local-name($attr), 2)} {$attr}
                    }
            )
            (:
            - diff: as 'all', but trimmed only to those 
                - in any transactions if this a dataset concept ($trid is empty)
                - at dataset level if this a transaction concept ($trid is not empty)
            :)
            case 'diff' return (
                if ($concept[ancestor::transaction]) then (
                    for $dsa in $datasetConceptAssociations[self::terminologyAssociation]
                    return $dsa[@conceptId = $transactionConceptAssocations[self::terminologyAssociation]/@conceptId]
                    ,
                    for $dsa in $datasetConceptAssociations[self::identifierAssociation]
                    return $dsa[@conceptId = $transactionConceptAssocations[self::identifierAssociation]/@conceptId]
                    ,
                    $datasetConceptAssociations[not(self::terminologyAssociation | self::identifierAssociation)]
                )
                else (
                    $datasetConceptAssociations
                    ,
                    (: Mark the thing with the transaction id/effectiveDate to smooth downstream processing :)
                    for $tca in $transactionConceptAssocations
                    return
                        element {name($tca)} {
                            $tca/(@* except (@*[starts-with(name(), 'transaction')])),
                            for $attr in $transactionConcept/ancestor::transaction[1]/(@id, @effectiveDate, @versionLabel, @statusCode, @expirationDate)
                            return attribute {'transaction' || upper-case(substring(local-name($attr), 1, 1)) || substring(local-name($attr), 2)} {$attr}
                        }
                )
            )
            (:
            - normal (default): retrieve any terminologyAssociations and identifierAssociations for this concept at the appropriate level
                - in transaction ($trid is not empty)
                - at dataset level ($trid is empty)
            
            :)
            default return (
                if ($concept[ancestor::transaction]) then
                    for $conceptId in $transactionConceptAssocations/@conceptId | $datasetConceptAssociations/@conceptId
                    let $id := $conceptId
                    group by $id
                    return (
                        let $trassoc    := $transactionConceptAssocations[self::terminologyAssociation][@conceptId = $conceptId[1]]
                        return if ($trassoc) then $trassoc else ($datasetConceptAssociations[self::terminologyAssociation][@conceptId = $conceptId[1]])
                        ,
                        let $trassoc    := $transactionConceptAssocations[self::identifierAssociation][@conceptId = $conceptId[1]]
                        return if ($trassoc) then $trassoc else ($datasetConceptAssociations[self::identifierAssociation][@conceptId = $conceptId[1]])
                        ,
                        $datasetConceptAssociations[not(self::terminologyAssociation | self::identifierAssociation)]
                    )
                else (
                    $datasetConceptAssociations
                    , 
                    (: Mark the thing with the transaction id/effectiveDate to smooth downstream processing :)
                    for $tca in $transactionConceptAssocations
                    return
                        element {name($tca)} {
                            $tca/(@* except (@*[starts-with(name(), 'transaction')])),
                            for $attr in $transactionConcept/ancestor::transaction[1]/(@id, @effectiveDate, @versionLabel, @statusCode, @expirationDate)
                            return attribute {'transaction' || upper-case(substring(local-name($attr), 1, 1)) || substring(local-name($attr), 2)} {$attr}
                        }
                )
            )
        
        return <associations mode="{$mode}">{$associations}</associations>
};

(:~ Function for getting designations for a terminologyAssociation :)
declare %private function utilde:conceptExpandedAssociation($concept as element()?, $associations as element()*, $oidnamemap as map(*)?, $language as xs:string?) as element()* {
    
    for $n in $associations
    let $decor                  := $n/ancestor::decor
    let $codeSystemName         := if (empty($oidnamemap) or empty($n/@codeSystem)) then () else map:get($oidnamemap, $n/@codeSystem)
    let $identifierName         := if (empty($oidnamemap) or empty($n/@ref)) then () else map:get($oidnamemap, $n/@ref)
    let $codeSystemConcept      := if ($n[@code][@codeSystem]) then utiltcs:getConcept($n/@codeSystem,$n/@code, (), $language, '0') else ()
    let $vsref                  := $n/@valueSet
    let $vsflex                 := $n/@flexibility
    let $vs                     := if (empty($vsref)) then () else utilvs:getValueSetById($vsref, ($vsflex, 'dynamic')[1])[@id]
    let $valueSetName           := ($vs/@displayName, $vs/@name)[1]
    let $valueSetStatusCode     := $vs[1]/@statusCode
    let $valueSetVersionLabel   := $vs[1]/@versionLabel
    order by $n/@conceptId, $n/@codeSystem, $n/@expirationDate descending
    return
        element {name($n)} {
            $n/(@* except (@codeSystemName|@valueSetName|@valueSetStatusCode|@valueSetVersionLabel|@refdisplay|@designation|@transactionId|@transactionEffectiveDate|@transactionVersionLabel|@transactionStatusCode|@transactionExpirationDate|@transactionName)),
            if (empty($codeSystemName))         then () else attribute codeSystemName {$codeSystemName},
            if (empty($valueSetName))           then () else attribute valueSetName {$valueSetName},
            if (empty($valueSetStatusCode))     then () else attribute valueSetStatusCode {$valueSetStatusCode},
            if (empty($valueSetVersionLabel))   then () else attribute valueSetVersionLabel {$valueSetVersionLabel},
            if (empty($identifierName))         then () else attribute refdisplay {$identifierName},
            
            (: Mark the thing with the transaction id/effectiveDate to smooth downstream processing :)
            for $attr in $concept/ancestor::transaction[1]/(@id, @effectiveDate, @versionLabel, @statusCode, @expirationDate)
            return attribute {'transaction' || upper-case(substring(local-name($attr), 1, 1)) || substring(local-name($attr), 2)} {$attr}
            ,
            if ($codeSystemConcept/designation/@languageCode) then
                for $l in distinct-values($codeSystemConcept/designation/@languageCode)
                let $d          := $codeSystemConcept/designation[@languageCode = $l][1]
                return
                    <designation language="{$l}" displayName="{$d/node()}">
                    {
                        switch ($d/@use = 'fsn') 
                        case 'fsn' return attribute type {'fsn'} 
                        case 'pref'
                        case 'lfsn' return attribute type {'preferred'}
                        default return ()
                        
                    }
                    </designation>
            else 
            if ($n/@displayName) then <designation language="{$language}">{$n/@displayName}</designation> else ()
        }
};

(:~ Function to get a number of additional properties for a concept/inherit that helps a user understand what the inheritance is pointing to. 
    
    @param $inherit concept/inherit element expected to carry ref and in most cases effectiveDate
    @param $inheritConcept target concept of $inherit. If you already got that concept in the calling function, you may supply it, otherwise this function looks up the concept
    @param $originalConcept original concept of $inheritConcept. If you already got that concept in the calling function, you may supply it, otherwise this function looks up the concept, which MAY be the very same concept as $inheritConcept
    @return At the very least the properties already present on $contains are returned. If the target is found, additional properties are also included
:)
declare function utilde:conceptExpandedInherit($inherit as element(inherit), $inheritConcept as element(concept)?, $originalConcept as element(concept)?) as element(inherit) {
    utilde:conceptExpandedInherit($inherit, $inheritConcept, $originalConcept, map {})
};
declare function utilde:conceptExpandedInherit($inherit as element(inherit), $inheritConcept as element(concept)?, $originalConcept as element(concept)?, $oidnamemap as map(*)) as element(inherit) {
    
    let $inheritConcept         := if ($inheritConcept) then $inheritConcept else utilde:getConcept($inherit/@ref, $inherit/@effectiveDate)
    let $originalConcept        := if ($originalConcept) then $originalConcept else utilde:getOriginalForConcept($inheritConcept)
    
    return
    <inherit>
    {
        $inherit/@ref,
        $inherit/@effectiveDate
    }
    {
        if ($inheritConcept) then (
            $inheritConcept/ancestor::decor/project/@prefix
            ,
            (: datasetId, datasetStatusCode, datasetEffectiveDate, datasetExpirationDate, datasetVersionLabel :)
            for $att in $inheritConcept/ancestor::dataset/(@id, @effectiveDate, @statusCode, @expirationDate, @versionLabel)
            return attribute {'dataset' || upper-case(substring(name($att), 1, 1)) || substring(name($att), 2)} {$att}
            ,
            attribute iType {$originalConcept/@type}
            ,
            (: iStatusCode, iEffectiveDate, iExpirationDate, iVersionLabel :)
            for $att in $inheritConcept/(@statusCode, @effectiveDate, @expirationDate, @versionLabel)
            return attribute {'i' || upper-case(substring(name($att), 1, 1)) || substring(name($att), 2)} {$att}
            ,
            (: originalStatusCode, originalEffectiveDate, originalExpirationDate, originalVersionLabel :)
            if ($inheritConcept[@id = $originalConcept/@id][@effectiveDate = $originalConcept/@effectiveDate]) then () else (
                for $att in $originalConcept/(@id, @statusCode, @effectiveDate, @expirationDate, @versionLabel)
                return attribute {'original' || upper-case(substring(name($att), 1, 1)) || substring(name($att), 2)} {$att}
                ,
                attribute originalPrefix {$originalConcept/ancestor::decor/project/@prefix}
            )
            ,
            (: awkward inconsistency ... originally some places had @iddisplay and others @refdisplay ... add both to avoid breaking stuff :)
            let $iddisplay      := map:get($oidnamemap, $inherit/@ref)
            let $iddisplay      := if (empty($iddisplay)) then utillib:getNameForOID($inherit/@ref, $inherit/ancestor::decor/project/@defaultLanguage, $inheritConcept/ancestor::decor) else ($iddisplay)
            return attribute iddisplay {$iddisplay} | attribute refdisplay {$iddisplay}
            ,
            attribute localInherit {$inheritConcept/ancestor::decor/project/@prefix = $inherit/ancestor::decor/project/@prefix}
            ,
            (: check if we have any additional immediate child concepts that the original does not ... 
               these might also be flagged linkedartefactmissing if they are really gone but more likely they are moved to somewhere else :)
            let $currentExtra       :=
                for $currentChild in $inherit/../concept
                let $inheritChild   := $inheritConcept/concept[@id = $currentChild/inherit/@ref][@effectiveDate = $currentChild/inherit/@effectiveDate]
                let $inheritChild   := if ($inheritChild) then $inheritChild else $originalConcept/concept[@id = $currentChild/inherit/@ref][@effectiveDate = $currentChild/inherit/@effectiveDate]
                return if ($inheritChild) then () else $currentChild
             
            (: check if our original has any additional immediate child concepts that the current does not ... :)
            let $inheritExtra       :=
                for $inheritChild in $inheritConcept/concept | $originalConcept/concept
                let $currentChild   := $inherit/../concept/inherit[@ref = $inheritChild/@id][@effectiveDate = $inheritChild/@effectiveDate]
                return if ($currentChild) then () else $inheritChild
            
            return         
                if ($currentExtra | $inheritExtra) then (
                    attribute inheritanceIssues {true()},
                    attribute inheritExtraConcepts {count($inheritExtra)},
                    attribute currentExtraConcepts {count($currentExtra)}
                )
                else ()
        ) 
        else attribute linkedartefactmissing {true()}
    }
    </inherit>
};

(:~ Function to get a number of additional properties for a concept/contains that helps a user understand what the containment is pointing to. 
    @param $contains concept/contains element expected to carry ref and in most cases flexibility
    @param $containConcept target concept of $contains. If you already got that concept in the calling function, you may supply it, otherwise this function looks up the concept
    @param $originalConcept original concept of $containConcept. If you already got that concept in the calling function, you may supply it, otherwise this function looks up the concept, which MAY be the very same concept as $containConcept
    @return At the very least the properties already present on $contains are returned. If the target is found, additional properties are also included
:)
declare %private function utilde:conceptExpandedContains($contains as element(contains), $containConcept as element(concept)?, $originalConcept as element(concept)?) as element(contains) {
    
    let $containConcept         := if ($containConcept) then $containConcept else utilde:getConcept($contains/@ref, $contains/@flexibility)
    let $originalConcept        := if ($originalConcept) then $originalConcept else utilde:getOriginalForConcept($containConcept)
    
    return
    <contains>
    {
        $contains/@ref,
        $contains/@flexibility
        ,
        if ($containConcept) then (
            $containConcept/ancestor::decor/project/@prefix,
            attribute datasetId {$containConcept/ancestor::dataset/@id},
            attribute datasetEffectiveDate {$containConcept/ancestor::dataset/@effectiveDate},
            attribute iType {$originalConcept/@type}, 
            attribute iStatusCode {$containConcept/@statusCode}, 
            attribute iEffectiveDate {$containConcept/@effectiveDate},
            if ($containConcept[@expirationDate]) then attribute iExpirationDate {$containConcept/@expirationDate} else (),
            if ($containConcept[@versionLabel]) then attribute iVersionLabel {$containConcept/@versionLabel} else (),
            attribute iddisplay {utillib:getNameForOID($contains/@ref, (), $containConcept/ancestor::decor)},
            attribute localInherit {$containConcept/ancestor::decor/project/@prefix = $contains/ancestor::decor/project/@prefix},
            if ($containConcept[@id = $originalConcept/@id][@effectiveDate = $originalConcept/@effectiveDate]) then () else (
                attribute originalId {$originalConcept/@id}, 
                attribute originalEffectiveDate {$originalConcept/@effectiveDate}, 
                attribute originalStatusCode {$originalConcept/@statusCode}, 
                if ($originalConcept[@expirationDate]) then attribute originalExpirationDate {$originalConcept/@expirationDate} else (),
                if ($originalConcept[@versionLabel]) then attribute originalVersionLabel {$originalConcept/@versionLabel} else (),
                attribute originalPrefix {$originalConcept/ancestor::decor/project/@prefix}
            )
        ) 
        else attribute linkedartefactmissing {true()}
    }
    </contains>
};

(:~ Function to get a number of additional properties for a concept/relationship that helps a user understand what the relationship is pointing to. 
    @param $relationship concept/relationship element expected to carry type and ref and in most cases flexibility
    @param $referredConcept target concept of $relationship. If you already got that concept in the calling function, you may supply it, otherwise this function looks up the concept
    @param $originalConcept original concept of $referredConcept. If you already got that concept in the calling function, you may supply it, otherwise this function looks up the concept, which MAY be the very same concept as $referredConcept
    @return At the very least the properties already present on $relationship are returned. If the target is found, additional properties are also included
:)
declare %private function utilde:conceptExpandedRelationship($relationship as element(relationship), $referredConcept as element(concept)?, $originalConcept as element(concept)?) as element(relationship) {
    
    let $language               := $relationship/ancestor::decor/project/@defaultLanguage
    let $referredConcept        := if ($referredConcept) then $referredConcept else utilde:getConcept($relationship/@ref, $relationship/@flexibility)
    let $originalReferredConcept:= if ($originalConcept) then $originalConcept else utilde:getOriginalForConcept($referredConcept)[1]
    
    return
        <relationship>
        {
            $relationship/@type,
            $relationship/@ref,
            $relationship/@flexibility
            ,
            if ($referredConcept) then (
                $referredConcept/ancestor::decor/project/@prefix,
                attribute datasetId {$referredConcept/ancestor::dataset/@id},
                attribute datasetEffectiveDate {$referredConcept/ancestor::dataset/@effectiveDate},
                attribute iType {$originalConcept/@type}, 
                attribute iStatusCode {$referredConcept/@statusCode}, 
                if ($referredConcept[@expirationDate]) then attribute iExpirationDate {$referredConcept/@expirationDate} else (),
                if ($referredConcept[@versionLabel]) then attribute iVersionLabel {$referredConcept/@versionLabel} else (),
                attribute iddisplay {utillib:getNameForOID($relationship/@ref, (), $referredConcept/ancestor::decor)},
                attribute localInherit {$referredConcept/ancestor::decor/project/@prefix = $relationship/ancestor::decor/project/@prefix},
                $originalConcept/name,
                if ($originalConcept/name[@language = $language]) then () else (
                    <name language="{$language}">{data($originalConcept/name[1])}</name>
                )
            ) else ()
        }
        </relationship>
};

(:~ Function to get community info on a concept (if any) 
    @param $associationset Set of associations from one or more communities
    @return community element carrying its name, displayName, desc, relevant prototypes, and relevant associations
:)
declare %private function utilde:conceptExpandedCommunity($associationset as element(association)*) as element(community)* {
    
    for $associations in $associationset
    let $communityPrefix        := $associations/ancestor::community/@name
    group by $communityPrefix
    return
        <community name="{$communityPrefix}">
        {
            $associations[1]/ancestor::community/@displayName,
            $associations[1]/ancestor::community/desc,
            <prototype>
            {
                let $prototypes     :=
                    if ($associations[1]/ancestor::community/prototype/@ref) then 
                        doc(xs:anyURI($associations[1]/ancestor::community/prototype/@ref))/prototype
                    else
                        $associations[1]/ancestor::community/prototype
                
                return $prototypes/data[@type = $associations//data/@type]
            }
            </prototype>,
            <associations>
            {
                for $association in $associations
                return
                    <association>
                    {
                        $association/@*,
                        $association/data
                    }
                    </association>
            }
            </associations>
        }
        </community>
};

declare %private function utilde:checkConceptAccess($authmap as map(*), $decor as element(decor)) as element()? {
    utilde:checkConceptAccess($authmap, $decor, (), (), false(), false())
};

declare %private function utilde:checkConceptAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean) as element()? {
    utilde:checkConceptAccess($authmap, $decor, $id, $effectiveDate, $checkLock, false())
};

declare %private function utilde:checkConceptAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean, $breakLock as xs:boolean) as element()? {

    let $check              :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-DATASETS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify concepts in project ', $decor/project/@prefix, '. You have to be an active author in the project.'))
        )

    let $lock               := if ($checkLock) then (
        let $lock           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
        let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breakLock)
        return
            if ($lock) then () else (
                error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this concept (anymore). Get a lock first.'))
            )    
        )
        else ()

    return $lock
};

declare %private function utilde:checkTransactionConceptAccess($authmap as map(*), $decor as element(decor)) as element()? {
    utilde:checkTransactionConceptAccess($authmap, $decor, (), (), false(), ())
};

declare %private function utilde:checkTransactionConceptAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean, $storedConcept as element(concept)?) as element()? {

    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-SCENARIOS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify concepts in project ', $decor/project/@prefix, '. You have to be an active author in the project.'))
        )

    return
        if ($checkLock) then (
            let $lock           := decorlib:getLocks($authmap, ($storedConcept/ancestor::transaction)[1]/@id, ($storedConcept/ancestor::transaction)[1]/@effectiveDate, ())
            let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())
            return
                if ($lock) then $lock else (
                    error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this concept (anymore). Get a lock first.'))
                )    
            )
        else ()
};

declare %private function utilde:buildOidNameMap($oids as xs:string*, $language as xs:string?, $decorOrPrefix as item()*) as map(*)? {
    map:merge(
        for $oid in distinct-values($oids)
        let $oidName    := utillib:getNameForOID($oid, $language, $decorOrPrefix)
        return if (string-length($oidName) = 0) then () else map:entry($oid, $oidName)
    )
};

(:~ Return the conceptList by this id :)
declare function utilde:getConceptListById($id as xs:string, $projectPrefix as xs:string?) as element(conceptList)* {
    let $results  :=
        if (empty($projectPrefix)) then 
            $setlib:colDecorData//conceptList[@id = $id][ancestor::datasets][not(ancestor::history)]
        else 
            $setlib:colDecorData//conceptList[@id = $id][ancestor::datasets][not(ancestor::history)][ancestor::decor[project[@prefix=$projectPrefix]]]

    return
        if (count($results) le 1) then $results else (
            error(xs:QName('utillib:getConceptList'), 'ConceptList ' || $id || ' with prefix ''' || $projectPrefix || ''' yields ' || count($results) || ' results. Projects involved: ' || string-join(distinct-values($results/ancestor::decor/project/@prefix), ', '))
        )
};
(:~ Return the conceptList by this id :)
declare function utilde:getConceptListConceptById($id as xs:string, $projectPrefix as xs:string?) as element(concept)? {
    let $results  :=
        if (empty($projectPrefix)) then 
            $setlib:colDecorData//conceptList/concept[@id = $id][ancestor::datasets][not(ancestor::history)]
        else 
            $setlib:colDecorData//conceptList/concept[@id = $id][ancestor::datasets][not(ancestor::history)][ancestor::decor[project[@prefix=$projectPrefix]]]

    return
        if (count($results) le 1) then $results else (
            error(xs:QName('utillib:getConceptListConcept'), 'ConceptList concept ' || $id || ' with prefix ''' || $projectPrefix || ''' yields ' || count($results) || ' results. Projects involved: ' || string-join(distinct-values($results/ancestor::decor/project/@prefix), ', '))
        )
};
(:~ Return the conceptLists that refer to this conceptList id. Principally you may only refer to a conceptList from within the same project, but 
    inconsistencies might have occurred that we don't know about. We'll leave that for a calling function to decide :)
declare function utilde:getConceptListByRef($ref as xs:string, $projectPrefix as xs:string) as element(conceptList)* {
    if (empty($projectPrefix)) then 
        $setlib:colDecorData//conceptList[@ref = $ref][ancestor::datasets][not(ancestor::history)]
    else 
        $setlib:colDecorData//conceptList[@ref = $ref][ancestor::datasets][not(ancestor::history)][ancestor::decor[project[@prefix=$projectPrefix]]]
};

declare %private function utilde:addValueSetRef($decor as element(), $repoPrefix as xs:string, $repoUrl as xs:string, $valueSetId as xs:string, $valueSetName as xs:string, $valueSetDisplayName as xs:string) as item()* {
    
    let $valueSetRefElm     := <valueSet ref="{$valueSetId}" name="{$valueSetName}" displayName="{$valueSetDisplayName}"/>
    let $buildingBlockElm   := <buildingBlockRepository url="{$repoUrl}" ident="{$repoPrefix}"/>
    
    let $addValueSetRef      :=
        if ($decor//valueSet[@id = $valueSetId] | $decor//valueSet[@ref = $valueSetId]) then () else (
            let $dummy1 := update insert $valueSetRefElm following $decor/terminology/*[last()]
            let $dummy2 := 
                if ($decor/project/buildingBlockRepository[@url=$buildingBlockElm/@url][@ident=$buildingBlockElm/@ident][empty(@format)] |
                    $decor/project/buildingBlockRepository[@url=$buildingBlockElm/@url][@ident=$buildingBlockElm/@ident][@format='decor']) then ('false') else (
                    update insert $buildingBlockElm following $decor/project/(author|reference|restURI|defaultElementNamespace|contact)[last()]
                )
            
            return 
                if ($dummy2='false') then 'ref' else 'ref-and-bbr'
        )
    
    return ()
};