xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilfsh                = "http://art-decor.org/ns/api/util-fsh";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace utiltemp        = "http://art-decor.org/ns/api/util-template" at "util-template-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace markdown        = "http://art-decor.org/ns/api/markdown" at "markdown-lib.xqm";
import module namespace utilde          = "http://art-decor.org/ns/api/util-concept" at "util-concept-lib.xqm";

import module namespace adterm          = "http://art-decor.org/ns/terminology" at "/db/apps/terminology/api/api-terminology.xqm";

declare %private variable $utilfsh:inactiveStatusCodes := ('cancelled','rejected','deprecated');

(:
    Build the FSH representation for dataset or transaction
:)
declare function utilfsh:convertTransactionOrDataset2Fsh(
    $fullDatasetTree as node(),
    $language as xs:string?,
    $fileNameforDownload as xs:string?
    (:
    $ui-lang as xs:string?,
    $hidecolumns as xs:string?, 
    $showonlyactiveconcepts as xs:boolean, 
    $collapsed as xs:boolean, 
    $draggable as xs:boolean, 
    $version as xs:string?,
    $url as xs:anyURI?,
    $projectPrefix as xs:string,
    $templateChain as element()*,
    $doSubConcept as xs:boolean,
    $download as xs:boolean,
    $header as xs:boolean
    :)
    ) as node() {
    
    let $language            := if (empty($language)) then $fullDatasetTree/name/@language[1] else $language
    let $isTransaction       := exists($fullDatasetTree/@transactionId)
    let $displayTitle        := ' ' || $fullDatasetTree/name[@language=$language][1]/string() || ' ' || $fullDatasetTree/@versionLabel || ' '
    
    let $name                := 
        if ($isTransaction and $fullDatasetTree/*:name[@language = $language][node()][. = 'Registration']) then
            ($fullDatasetTree/*:name[@language = $language], $fullDatasetTree/*:name)[node()][1]
        else (
            ($fullDatasetTree/*:name[@language = $language], $fullDatasetTree/*:name)[.//text()[string-length()>0]][1]
        )
    let $desc                := 
        if ($isTransaction and $fullDatasetTree/*:desc[@language = $language][node()][. = '-']) then
            ($fullDatasetTree/*:desc[@language = $language], $fullDatasetTree/*:desc)[node()][1]
        else (
            ($fullDatasetTree/*:desc[@language = $language], $fullDatasetTree/*:desc)[.//text()[string-length()>0]][1]
        )

    let $desc   := if (empty($desc)) then '-' else markdown:html2markdown(utillib:parseNode($desc)/node())
    
    (: fullDatasetTree contains shortName based on project default language, while we want a language that matches the $language which may be different :)
    let $shortname           := utillib:shortName($name)
    let $shortname           := concat(upper-case(substring($shortname, 1, 1)), substring($shortname, 2))
    let $shortname4id        := lower-case(replace($shortname, '_', '-'))

    let $rootPath            := () (: if ($fullDatasetTree/*:concept[2]) then $shortname else () :)
    let $prefix              := ($fullDatasetTree/@prefix | $fullDatasetTree/ancestor::decor/project/@prefix)[1]

    (: init array :)
    let $fsh := []
    
    (: header :)
    let $fsh := array:append($fsh, "// -------------------------------------------------------------------------------")
    let $fsh := array:append($fsh, "//  Logical Model " || $displayTitle)
    let $fsh := if (string-length($fileNameforDownload) > 0) then array:append($fsh, "//  Filename " || $fileNameforDownload) else ()
    let $fsh := array:append($fsh, "// -------------------------------------------------------------------------------")
    let $fsh := array:append($fsh, "Logical:     " || $shortname)
    let $fsh := array:append($fsh, "Parent:      " || "Element")
    let $fsh := array:append($fsh, "Id:          " || $shortname4id)
    let $fsh := array:append($fsh, "Title:       " || '"' || normalize-space($displayTitle) || '"')
    let $fsh := array:append($fsh, "Description: " || '"""' || normalize-space($desc) || '"""')
    
    (: go through concepts :)
    let $cfsh := []
    let $cfsh := array:append($cfsh,
        for $concept in $fullDatasetTree/*:concept
        return utilfsh:concept2fsh ($concept, $rootPath, $prefix, $language, $isTransaction)
        )
                
    return 
        <fsh>
        {
            concat(string-join(data($fsh), '&#10;'), '&#10;', string-join($cfsh, ''), '&#10;')
        }
        </fsh>
};

declare function utilfsh:concept2fsh (
    $concept as element(concept),
    $parentPath as xs:string?,
    $prefix as xs:string?,
    $language as xs:string,
    $isTransaction as xs:boolean
    ) as xs:string {
    
    let $originalConcept    := utilde:getOriginalForConcept($concept)

    let $name               := 
        if ($originalConcept/*:name[@language = $language]) then 
            $originalConcept/*:name[@language = $language][1]
        else 
            $originalConcept/*:name[node()][1]
    let $desc               := 
        if ($originalConcept/*:desc[@language = $language]) then 
            $originalConcept/*:desc[@language = $language][1]
        else 
            $originalConcept/*:desc[node()][1]
    
    let $shortName          := utilfsh:fhirName($name)
    let $id                 := 
        if ($concept[@id]) then
            $concept/concat(@id,'--', replace(@effectiveDate,'[^\d]',''))
        else
            $concept/concat(@ref,'--', replace(@flexibility,'[^\d]',''))
    let $shortId            := tokenize(tokenize($id, '\.')[last()], '-')[1]
            
    let $name               := $name || " (" || $shortId || ")"
    let $longId             :=  "ID: " || $id
    let $mddesc             := markdown:html2markdown(utillib:parseNode($desc)/node())
    let $desc               := 
        if (string-length($mddesc) = 0) then $longId
        else $mddesc || "&#10;&#10;" || $longId
    
    let $path               := string-join(($parentPath, $shortName), '.')
    let $minCard            := utillib:getMinimumMultiplicity($concept)
    let $maxCard            := utillib:getMaximumMultiplicity($concept)
    let $isRequired         := $concept[@conformance='R']
    let $isMandatory        := $concept[@isMandatory='true']
    
    let $valueDomain        := if ($originalConcept/*:valueDomain[2]) then () else $originalConcept/*:valueDomain[1]
    let $dataType           := if ($originalConcept/@type='group') then 'BackboneElement' else utilfsh:decorValueDomainType2FHIRtype($valueDomain/@type)
    
    let $concept            :=
        if ($concept[count(*:concept) = 1]/*:concept/*:contains) then (
            let $actualConceptName          := 
                if ($concept/*:concept/*:name[@language = $language]) then 
                    $concept/*:concept/*:name[@language = $language][1]
                else 
                    $concept/*:concept/*:name[node()][1]
            (: fullDatasetTree contains shortName based on project default language, while we want a language that matches the $language which may be different :)
            let $actualShortName            := 
                (:if (matches($actualConceptName, $adfhirsd:ncnamePattern)) then $actualConceptName else:)
                (:if ($concept/*:concept[@shortName]) then $concept/*:concept/@shortName else adfhir:shortName($actualConceptName):)
                utilfsh:fhirName($actualConceptName)
            
            return
                if ($shortName = $actualShortName) then $concept/*:concept else $concept
        )
        else ($concept)
     
     (:
        one FSH line, must have a valid element name (shortname)
        |* dateOfArrivalAtCenter 1..* dateTime "Date of arrival to our institution" """Date of arrival to our institution"""				
     :)
     let $fsh1 :=
        if (string-length($shortName) > 0) then (
            "* " || $path || " " || $minCard || ".." || $maxCard || " " || $dataType || ' "' || $name || '"',
            if (string-length($desc) > 0) then ' """' || $desc || '"""' else ()
        ) else ()
     
     let $fsh2 :=
        if ($isTransaction) then (
               for $subconcept in $concept/*:concept
               return 
                   utilfsh:concept2fsh($subconcept, $path, $prefix, $language, $isTransaction)
           )
           else (
               for $subconcept in $concept/*:concept
               return (
                   if ($subconcept[@statusCode = $utilfsh:inactiveStatusCodes]) then () else
                       utilfsh:concept2fsh($subconcept, $path, $prefix, $language, $isTransaction)
               )
           )
           
      return concat(string-join($fsh1, ''), '&#10;', string-join(data($fsh2), ''))
};

declare function utilfsh:decorValueDomainType2FHIRtype($decorType as item()?) as xs:string {
    switch ($decorType)
    case 'count'        return 'Count'      (:could have chosen integer of decimal. not sure why to pick what :)
    case 'code'         return 'CodeableConcept'
    case 'ordinal'      return 'CodeableConcept'
    case 'identifier'   return 'Identifier'
    case 'string'       return 'string'
    case 'text'         return 'markdown'
    case 'date'         return 'date'       (:need work with properties:)
    case 'datetime'     return 'dateTime'   (:need work with properties:)
    case 'time'         return 'time'       (:need work with properties:)
    case 'complex'      return 'string'     (:this is a problem. this is lazy dataset behavior that is unimplementable:)
    case 'quantity'     return 'Quantity'
    case 'duration'     return 'Duration'
    case 'boolean'      return 'boolean'
    case 'blob'         return 'base64Binary'
    case 'decimal'      return 'decimal'
    default             return 'string'
};

(:~ Returns a FHIR "machine processing" name, 
    start with an upper-case ASCII letter ('A'..'Z') followed
    by any combination of upper- or lower-case ASCII letters
    ('A'..'Z', and 'a'..'z'), numerals ('0'..'9') and '_',
    with a length limit of 255 characters.
    
    Most common diacritics are replaced
    
    Input:  xs:string, example: "UnderScored Lowercase ë"
    Output: xs:string, example: "underscored_lowercase_e"
    
    uses utillib shortName function and adds refinements
    
    @author Kai Heitmann
    @since 2025
:)
declare function utilfsh:fhirName($name as xs:string?) as xs:string? {
    let $shortname := utillib:shortName($name)
    let $shortname :=
        if ($shortname) then (

            (: 
                CamelCase short name, e.g. "find_matching_alternatives" => "FindMatchingAlternatives"
            :)
            let $r1 := string-join(
                for $part in tokenize($shortname, "_")
                return concat(
                  upper-case(substring($part, 1, 1)),
                  substring($part, 2)
                )
              , '')

            (: make sure we do not start with a digit :)
            let $r2 := replace($r1, '^(\d)' , 'X$1')
            
            return $r2
            
        ) else ()
    
    return if (matches($shortname, '^[a-zA-Z_][a-zA-Z\d_]+$')) then $shortname else ()
};