xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilds                 = "http://art-decor.org/ns/api/util-dataset";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace decorlib        = "http://art-decor.org/ns/api/decor" at "decor-lib.xqm";
import module namespace ruleslib        = "http://art-decor.org/ns/api/rules" at "rules-lib.xqm";
import module namespace histlib         = "http://art-decor.org/ns/api/history" at "history-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

declare %private variable $utilds:STATUSCODES-FINAL := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
(:~ local debug 0 or 1 - or 2,... later :)
declare %private variable $utilds:DEBUG             := 0;

(:~ Central logic for updating an existing dataset
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR dataset/@id to update
    @param $effectiveDate   - required. DECOR dataset/@effectiveDate to update
    @param $data            - required. DECOR concept xml element containing everything that should be in the updated dataset
    @return concept object as xml with json:array set on elements
:)
declare function utilds:putDataset($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(dataset)) {

    let $editedDataset          := $data
    let $storedDataset          := utilds:getDataset($id, $effectiveDate)
    let $decor                  := $storedDataset/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($storedDataset) then () else (
            error($errors:BAD_REQUEST, 'Dataset with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    
    let $check                  := utilds:checkDatasetAccess($authmap, $decor)
    
    let $check                  :=
        if ($storedDataset/@statusCode = $utilds:STATUSCODES-FINAL) then
            if ($editedDataset/@statusCode = $utilds:STATUSCODES-FINAL) then    
                error($errors:BAD_REQUEST, concat('Dataset cannot be edited in status ', $storedDataset/@statusCode, '. ', if ($storedDataset/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            else ()
        else ()
    
    let $check                  :=
        if ($editedDataset[@id = $id] | $editedDataset[empty(@id)]) then () else (
            error($errors:BAD_REQUEST, concat('Dataset SHALL have the same id as the dataset id ''', $id, ''' used for updating. Found in request body: ', ($editedDataset/@id, 'null')[1]))
        )
    
    let $check                  :=
        if ($editedDataset[@effectiveDate = $effectiveDate] | $editedDataset[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Dataset SHALL have the same effectiveDate as the dataset effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($editedDataset/@effectiveDate, 'null')[1]))
        )
       
    let $errorRelationship      := if ($editedDataset/relationship) then ruleslib:checkRelationship($editedDataset/relationship, $storedDataset, 'update', ()) else ()
    let $check                  :=
        if ($errorRelationship) then 
            error($errors:BAD_REQUEST, string-join (($errorRelationship), ' '))
        else ()
    
    (: set and check a lock on every concept to be changed :)
    let $lock                   :=
        for $concept in $editedDataset/descendant-or-self::concept[move] | $editedDataset/descendant-or-self::concept[edit]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        return utilds:checkDatasetConceptLock($authmap, $storedConcept)
    
    (: check every concept to be changed :)
    let $checkConcepts          := 
        for $concept in $editedDataset/descendant-or-self::concept[edit[not(@mode = 'delete')]]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        return utilds:checkDatasetConceptForUpdate($concept, $storedConcept)
        
    (: save dataset history :)
    let $intention              := if ($storedDataset[@statusCode = 'final']) then 'patch' else 'version'
    let $history                := if ($storedDataset) then histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASET, $projectPrefix, $intention, $storedDataset) else ()
    
    (: update all changed concepts :)
    let $updateConcepts         := utilds:editDatasetConcepts($projectPrefix, $authmap, $editedDataset, $storedDataset)
    
    let $deleteLock             := update delete $lock
    
    (: update dataset (except already saved concepts) - and add associations and identifiers :)
    return utilds:editDataset($decor, $editedDataset, $storedDataset)
};

(:~ Return the live data set based on id and optionally effectiveDate. If effectiveDate is empty, the latest version is returned
    @param $id                  - required dataset/@id
    @param $flexibility         - optional dataset/@effectiveDate
    @return exactly 1 dataset or nothing if not found
    @since 2015-04-27
:)
declare function utilds:getDataset($id as xs:string, $flexibility as xs:string?) as element(dataset)? {
    utilds:getDataset($id, $flexibility, (), ())
};
(:~ Return the live data set if param decorRelease is not a dateTime OR 
    compiled, archived/released DECOR data set based on dataset/@id|@effectiveDate and decor/@versionDate.
    Released/compiled matches could be present in multiple languages. Use the parameters @language to select the right one.
    
    @param $id                  - required dataset/@id
    @param $flexibility         - optional dataset/@effectiveDate
    @param $decorRelease        - optional decor/@versionDate
    @param $decorLanguage       - optional decor/@language. Only relevant with parameter $decorVersion. Empty uses decor/project/@defaultLanguage, '*' selects all versions, e.g. 'en-US' selects the US English version (if available)
    @return exactly 1 live data set, or 1 or more archived DECOR version instances or nothing if not found
    @since 2015-04-29
:)
declare function utilds:getDataset($id as xs:string, $flexibility as xs:string?, $decorRelease as xs:string?) as element(dataset)* {
    utilds:getDataset($id, $flexibility, $decorRelease, ())
};
declare function utilds:getDataset($id as xs:string, $flexibility as xs:string?, $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(dataset)* {
    
    let $decor                  :=
        if ($decorRelease[string-length() gt 0]) then (
            let $d              := $setlib:colDecorVersion/decor[@versionDate = $decorRelease]
            return
            if (empty($decorLanguage)) then ($d[@language = $d/project/@defaultLanguage], $d)[1]
            else if ($decorLanguage='*') then $d
            else $d[@language = $decorLanguage]
        ) else $setlib:colDecorData/decor
    
    return
        if ($flexibility castable as xs:dateTime) then $decor//dataset[@id = $id][@effectiveDate = $flexibility][ancestor::datasets]
        else (
            let $datasets       := $decor//dataset[@id = $id]
            let $datasets       := $datasets[ancestor::datasets]
            return $datasets[@effectiveDate = max($datasets/xs:dateTime(@effectiveDate))]
        )
};

declare %private function utilds:checkDatasetAccess($authmap as map(*), $decor as element(decor)) as element()? {
    utilds:checkDatasetAccess($authmap, $decor, (), (), false(), false())
};

declare %private function utilds:checkDatasetAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean) as element()? {
    utilds:checkDatasetAccess($authmap, $decor, $id, $effectiveDate, $checkLock, false())
};

declare %private function utilds:checkDatasetAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean, $breakLock as xs:boolean) as element()? {

    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-DATASETS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify datasets in project ', $decor/project/@prefix, '. You have to be an active author in the project.'))
        )

    return
        if ($checkLock) then (
            let $lock           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
            let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breakLock)
            return
                if ($lock) then $lock else (
                    error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this dataset (anymore). Get a lock first.'))
                )    
            )
            else ()
};

declare %private function utilds:checkDatasetConceptLock($authmap as map(*), $concept as element()) as element()? {

    let $lock                   := decorlib:getLocks($authmap, $concept, false())
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $concept, false())
    return
        if ($lock) then $lock else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this concept id=''' || $concept/@id || ''' and effectiveDate=''' || $concept/@effectiveDate || ''' (anymore). Get a lock first.'))
        )    
};

declare %private function utilds:checkDatasetConceptForUpdate($editedConcept as element(concept), $storedConcept as element(concept)) {

    let $check                  :=
        if (empty($storedConcept)) then
            error($errors:BAD_REQUEST, 'Concept ' || $editedConcept/@id || ' effectiveDate ''' || $editedConcept/@effectiveDate || ''' SHALL exist in stored dataset on the server. Cannot update non-existent concept.')
        else if (count($storedConcept) = 1) then ()
        else (
            error($errors:SERVER_ERROR, 'Concept ' || $editedConcept/@id || ' effectiveDate ''' || $editedConcept/@effectiveDate || ''' gave ' || count($storedConcept) || ' stored concepts. Expected exactly 1. Found concept in projects:' || string-join(for $s in $storedConcept return $s/ancestor::decor/project/@prefix,' / '))
        )
    
    let $check                  :=
        if (utillib:isStatusChangeAllowable($storedConcept, $editedConcept/@statusCode)) then () else (
            error($errors:BAD_REQUEST, 'Concept ' || $editedConcept/@id || ' SHALL have the same statusCode ''' || $storedConcept/@statusCode || ''' or a supported status change. Found in request body: ''' || ($editedConcept/@statusCode, 'null')[1] || '''')
        )
    
    let $check                  := ruleslib:checkConcept($editedConcept, $storedConcept, 'update')
    
    return ()
};

declare %private function utilds:editDatasetConcepts($projectPrefix as xs:string, $authmap as map(*), $dataset as element(dataset), $storedDataset as element(dataset)) {

    let $debug                  := if ($utilds:DEBUG > 0) then util:log('INFO', 'putDataset: delete concepts ' || count($dataset//concept[edit/@mode='delete'])) else ()
    
    let $deletes                :=
        for $concept in $dataset//concept[edit/@mode='delete']
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        let $history            := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, 'delete', $storedConcept)
        return utilds:deleteDatasetConcept($concept, $storedConcept, $dataset)
    
    let $debug                  := if ($utilds:DEBUG > 0) then util:log('INFO', 'putDataset: move concepts ' || count($dataset//concept[move])) else ()
    
    (: first do all the history of the moves before the database is changed :)
    let $moveHistory            :=
        for $concept in $dataset//concept[move]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        return utilds:historyOfMovedDatasetConcept($projectPrefix, $authmap, $concept, $storedConcept, $storedDataset)
    
    let $moves                  :=
        for $concept in $dataset//concept[move]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]
        return utilds:moveDatasetConcept($concept, $storedConcept, $storedDataset)
    
    let $debug                  := if ($utilds:DEBUG > 0) then util:log('INFO', 'putDataset: update concepts ' || count($dataset//concept[edit/@mode='edit'][not(move)])) else ()
    
    let $updates                :=
        for $concept in $dataset//concept[edit/@mode='edit'][not(move)]
        let $storedConcept      := $storedDataset//concept[@id = $concept/@id][@effectiveDate = $concept/@effectiveDate]        
        let $intention          := if ($storedDataset[@statusCode = 'final']) then 'patch' else 'version'
        let $history            := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, $intention, $storedConcept)
        return update replace $storedConcept with utilds:prepareDatasetConceptForUpdate($concept, $storedConcept)
    
    return ()
};

(: deleting concepts:
       if statusCode=new     delete concept
       if statusCode!=new    set statusCode=deprecated
:)
declare %private function utilds:deleteDatasetConcept($concept as element(concept), $storedConcept as element(concept), $dataset as element(dataset)) {

    if ($dataset/@statusCode != 'new') then (
        if ($storedConcept/@statusCode = 'new') then update delete $storedConcept
        else if ($concept/@type='item') then update replace $storedConcept with utillib:prepareItemForUpdate($concept, $storedConcept)
        else if ($concept/@type='group') then update replace $storedConcept with utillib:prepareGroupForUpdate($concept, $storedConcept)
        else ()
    ) 
    else update delete $storedConcept
};

(: a moved concept does not go to history but the group of concepts is touched by a move (concepts in or out) they need to go to history as they changed :)
declare %private function utilds:historyOfMovedDatasetConcept($projectPrefix as xs:string, $authmap as map(*), $concept as element(concept), $storedConcept as element(concept), $storedDataset as element(dataset)) {

    let $sourceIntention        := if ($concept[edit/@mode = 'edit']) then 'move/version' else 'move'
    let $sourceHistory          := $storedConcept/parent::concept[not(@statusCode = 'new')]
        
    let $history                := if ($sourceHistory) then histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, $sourceIntention, $sourceHistory) else ()   
    
    let $destHistory            := 
        if ($concept[preceding-sibling::concept]) 
        then $storedDataset//concept[@id = $concept/@id]/parent::concept[not(@statusCode = 'new')]
        else if ($concept[parent::concept] and $storedDataset//concept[@id = $concept/parent::concept/@id]) 
        then $storedDataset//concept[@id = $concept/parent::concept/@id][not(@statusCode = 'new')]
        else ()
    
    let $history                := if ($destHistory) then histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-DATASETCONCEPT, $projectPrefix, $sourceIntention, $destHistory) else ()

    return ()
};

declare %private function utilds:prepareDatasetConceptForUpdate($concept as element(concept), $storedConcept as element(concept)) as element(concept){
    
    let $preparedConcept        :=
        if ($concept[@type = 'item']) then
            if ($concept[edit/@mode='edit']) then utillib:prepareItemForUpdate($concept, $storedConcept) else utillib:prepareItemForUpdate($storedConcept, $storedConcept)
        else if ($concept[@type = 'group']) then
            if ($concept[edit/@mode='edit']) then utillib:prepareGroupForUpdate($concept, $storedConcept) else utillib:prepareGroupForUpdate($storedConcept, $storedConcept)
        else ()
    
    return
        if (count($preparedConcept) = 1) then $preparedConcept else (
            error($errors:SERVER_ERROR, concat('Concept id=''',$concept/@id,''' and effectiveDate=''',$concept/@effectiveDate,''' has unknown type ',$concept/@type,'. Expected item or group.'))
        )
};

(:  move concepts
    - local move
      - delete stored concept
      - insert moved concept after editedDataset preceding-sibling or into parent dataset/concept
    - move to other dataset (unsupported!!)
:)
declare %private function utilds:moveDatasetConcept($concept as element(concept), $storedConcept as element(concept), $storedDataset as element(dataset)) {
    
    let $preparedConcept        := utilds:prepareDatasetConceptForUpdate($concept, $storedConcept)
    
    (:  if move + edit: use what was sent to us
        if just move: use what we already had stored on the server
    :)
    
    return
        if ($concept[preceding-sibling::concept]) 
        then update insert $preparedConcept following $storedDataset//concept[@id = $concept/preceding-sibling::concept[1]/@id]
        else if ($concept[parent::dataset]) then (
            if ($storedDataset[concept | history]) 
            then update insert $preparedConcept preceding ($storedDataset/concept | $storedDataset/history)[1]
            else update insert $preparedConcept into $storedDataset
        )
        else if ($concept[parent::concept] and $storedDataset//concept[@id = $concept/parent::concept/@id]) then (
            let $storedParent   := $storedDataset//concept[@id = $concept/parent::concept/@id]
            return 
                if ($storedParent[concept | history]) then update insert $preparedConcept preceding ($storedParent/concept | $storedParent/history)[1] else update insert $preparedConcept into $storedParent
        ) else (
            error(QName('http://art-decor.org/ns/art/dataset', 'ContextNotFound'), concat('Could not determine context for saving concept with id ',$concept/@id,' and name ',$concept/name[1]))
        ),
        update delete $storedConcept
};

declare %private function utilds:prepareDatasetForUpdate($dataset as element(dataset), $storedDataset as element(dataset)) as element(dataset) { 

    <dataset id="{$dataset/@id}" effectiveDate="{$dataset/@effectiveDate}" statusCode="{$dataset/@statusCode}">
    {
        $dataset/@versionLabel[string-length() gt 0],
        $dataset/@expirationDate[string-length() gt 0],
        $dataset/@officialReleaseDate[string-length() gt 0],
        $dataset/@canonicalUri[string-length() gt 0],
        attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)},
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($dataset/name),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($dataset/desc),
        utillib:preparePublishingAuthorityForUpdate($dataset/publishingAuthority[empty(@inherited)]),
        utillib:prepareConceptPropertyForUpdate($dataset/property),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($dataset/copyright[empty(@inherited)]),
        utillib:prepareDatasetRelationshipForUpdate($dataset/relationship),
        $storedDataset/concept,
        $storedDataset/recycle
    }
    </dataset>
};

declare %private function utilds:editDataset($decor as element(decor), $dataset as element(dataset), $storedDataset as element(dataset)) {

    let $debug                  := if ($utilds:DEBUG > 0) then util:log('INFO', 'putDataset: update dataset' || count($storedDataset)) else ()
    
    let $preparedDataset        := utilds:prepareDatasetForUpdate($dataset, $storedDataset)
    
    (: check on scheme, if it fails the concepts are already changed ???   :)
    let $check                  := ruleslib:checkDecorSchema($preparedDataset, 'update')
    
    let $update                 := update replace $storedDataset with $preparedDataset
    
    (:store any terminologyAssociations that were saved in the dataset after deInherit
        only store stuff where the concept(List) still exists
    :)
    let $terminologyAssociations        := ($dataset//terminologyAssociation[ancestor::concept/edit/@mode='edit'] | $dataset//terminologyAssociation[ancestor::concept[move]])
    let $terminologyAssociationUpdates  := utillib:addTerminologyAssociations($terminologyAssociations, $decor, false(), false())
    
    (:store any identifierAssociations that were saved in the dataset after deInherit
        only store stuff where the concept(List) still exists
    :)
    let $identifierAssociations         := ($dataset//identifierAssociation[ancestor::concept/edit/@mode='edit'] | $dataset//identifierAssociation[ancestor::concept[move]])
    let $identifierAssociationUpdates   := utillib:addIdentifierAssociations($identifierAssociations, $decor, false(), false())
    
    return ()
};    
    