xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilvs                 = "http://art-decor.org/ns/api/util-valueset";

import module namespace utilgg          = "http://art-decor.org/ns/api/util-governancegroup" at "util-governancegroup-lib.xqm";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace decorlib        = "http://art-decor.org/ns/api/decor" at "decor-lib.xqm";
import module namespace histlib         = "http://art-decor.org/ns/api/history" at "history-lib.xqm";
import module namespace ruleslib        = "http://art-decor.org/ns/api/rules" at "rules-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

declare namespace json                  = "http://www.json.org";

declare %private variable $utilvs:STATUSCODES-FINAL          := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $utilvs:ADDRLINE-TYPE              := utillib:getDecorTypes()/AddressLineType;
declare %private variable $utilvs:VOCAB-TYPE                 := utillib:getDecorTypes()/VocabType;
declare %private variable $utilvs:DESIGNATION-TYPE           := utillib:getDecorTypes()/DesignationType;
declare %private variable $utilvs:INTENSIONALOPERATORS-TYPE  := utillib:getDecorTypes()/IntensionalOperators;



(:~ Central logic for updating an existing valueSet
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR valueSet/@id to update
    @param $request-body    - required. DECOR valueSet xml element containing everything that should be in the updated valueSet
    @return valueSet object as xml with json:array set on elements
:)
declare function utilvs:putValueSet($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(), $deletelock as xs:boolean) as element(valueSet) {

    let $storedValueSet         := $setlib:colDecorData//valueSet[@id = $id][@effectiveDate = $effectiveDate] 
    let $decor                  := $storedValueSet/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($storedValueSet) then () else (
            error($errors:BAD_REQUEST, 'ValueSet with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ' does not exist')
        )
      
    let $lock                   := utilvs:checkValueSetAccess($authmap, $decor, $id, $effectiveDate, true())
    
    let $check                  :=
        if ($storedValueSet[@statusCode = $utilvs:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, concat('ValueSet cannot be updated while it has one of status: ', string-join($utilvs:STATUSCODES-FINAL, ', '), if ($storedValueSet/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
        else ()
    let $check                  :=
        if ($data[@id = $id] | $data[empty(@id)]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no valueSet id or the same valueSet id as the valueSet id ''', $id, ''' used for updating. Found in request body: ', ($data/@id, 'null')[1]))
        )
    let $check                  :=
        if ($data[@effectiveDate = $effectiveDate] | $data[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no valueSet effectiveDate or the same valueSet effectiveDate as the valueSet effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($data/@effectiveDate, 'null')[1]))
        )
    
    let $check                  := ruleslib:checkValueSetItems($id, $data, 'update')
    
    let $newValueSet            := utilvs:prepareValueSetForUpdate($data, $storedValueSet)
    
    let $check                  := ruleslib:checkDecorSchema($newValueSet, 'update')
    
    (: save history:)
    let $intention              := if ($storedValueSet[@statusCode = 'final']) then 'patch' else 'version'
    let $history                := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-VALUESET, $projectPrefix, $intention, $storedValueSet)
    
    (: now update the value set :)
    let $valueSetUpdate         := update replace $storedValueSet with $newValueSet
    let $deleteLock             := if ($deletelock) then update delete $lock else ()
    
    return utillib:getValueSet($projectPrefix, (), (), $id, $effectiveDate, false(), true())
    
};

(:~ Central logic for patching an existing valueSet
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR valueSet/@id to update
    @param $effectiveDate   - required. DECOR valueSet/@effectiveDate to update
    @param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
    @return dataset object as xml with json:array set on elements
:)
declare function utilvs:patchValueSet($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) as element(valueSet) {

    let $storedValueSet         := $setlib:colDecorData//valueSet[@id = $id][@effectiveDate = $effectiveDate]
    let $decor                  := $storedValueSet/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $check                  :=
        if ($storedValueSet) then () else (
            error($errors:BAD_REQUEST, 'ValueSet with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )

    let $lock                   := utilvs:checkValueSetAccess($authmap, $decor, $id, $effectiveDate, true())
    
    let $check                  :=
        if ($storedValueSet[@statusCode = $utilvs:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $utilvs:STATUSCODES-FINAL)]) then () else
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('ValueSet cannot be patched while it has one of status: ', string-join($utilvs:STATUSCODES-FINAL, ', '), if ($storedValueSet/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()

    let $check                  := utilvs:checkValueSetParameters($data, $storedValueSet)
    
    let $intention              := if ($storedValueSet[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-VALUESET, $projectPrefix, $intention, $storedValueSet)
    
    let $patch                  := utilvs:patchValueSetParameters($data, $storedValueSet)
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $preparedValueSet       := utilvs:prepareValueSetForUpdate($storedValueSet, $storedValueSet)
    
    let $update                 := update replace $storedValueSet with $preparedValueSet
    let $update                 := update delete $lock
    
    return utillib:getValueSet($projectPrefix, (), (), $preparedValueSet/@id, $preparedValueSet/@effectiveDate, false(), true())
};

(: Central logic for creating an empty dataset
    @param $authmap         - required. Map derived from token
    @return (empty) dataset object as xml with json:array set on elements
:)
declare function utilvs:createValueSet($authmap as map(*), $projectPrefix as xs:string, $targetDate as xs:boolean, $sourceId as xs:string?, $sourceEffectiveDate as xs:string?, $refOnly as xs:boolean?, $keepIds as xs:boolean?, $baseId as xs:string?, $editedValueset as element(valueSet)?) {

    let $decor                  := utillib:getDecorByPrefix($projectPrefix)
    let $projectLanguage        := $decor/project/@defaultLanguage
    
    let $check                  :=
        if (empty($editedValueset/@id)) then ()  
            else if (matches($editedValueset/@id, '^[0-2](\.(0|[1-9][0-9]*)){0,3}$')) then
                error($errors:BAD_REQUEST, 'Value set id ''' || $editedValueset/@id || ''' is reserved. You cannot reuse a reserved identifier. See http://oid-info.com/get/' || $editedValueset/@id)
            else if (utillib:isOid($editedValueset/@id)) then ()  
            else error($errors:BAD_REQUEST, 'Value set id ''' || $editedValueset/@id || ''' SHALL be a valid OID')
    
    (: if the user sent us a valueSet with id, we should assume he intends to keep that id :)
    let $keepIds                := if ($editedValueset/@id) then true() else $keepIds = true()
    (: if the user sent us a valueSet with effectiveDate, we should assume he intends to keep that effectiveDate :)
    let $now                    := 
        if ($editedValueset/@effectiveDate) then substring($editedValueset/@effectiveDate, 1, 19) else 
        if ($targetDate) then substring(string(current-date()), 1, 10) || 'T00:00:00' else substring(string(current-dateTime()), 1, 19)
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, 'Project with prefix ' || $projectPrefix || ' does not exist')
        )
    
    let $check                  := utilvs:checkValueSetAccess($authmap, $decor, (), (), false())
    
    let $sourceValueset         := if (empty($sourceId)) then () else utillib:getValueSet($projectPrefix, (), (), $sourceId, $sourceEffectiveDate, false(), true())
    let $storedValueset         := 
        if ($keepIds and $editedValueset[@id]) then utillib:getValueSet($projectPrefix, (), (), $editedValueset/@id, $now, false(), true()) else ()
    
    let $check                  :=
        if ($refOnly) then
            if (empty($sourceId)) then
                error($errors:BAD_REQUEST, 'Parameter sourceId SHALL be provided if a value set reference is to be created')
            else
            if (empty($sourceValueset)) then
                if (empty($sourceEffectiveDate)) then
                    error($errors:BAD_REQUEST, 'Parameter sourceId ' || $sourceId || ' SHALL lead to a value set in scope of this project')
                else (
                    error($errors:BAD_REQUEST, 'Parameters sourceId ' || $sourceId || ' and sourceEffectiveDate ' || $sourceEffectiveDate || ' SHALL lead to a value set in scope of this project')
                )
            else ()
        else
        if (empty($editedValueset) and empty($sourceId)) then 
            error($errors:BAD_REQUEST, 'ValueSet input data or a sourceId SHALL be provided')
        else
        if ($editedValueset) then
            if ($storedValueset) then
                error($errors:BAD_REQUEST, 'Cannot create new valueSet. The input valueSet with id ' || $editedValueset/@id || ' and effectiveDate ' || $now || ' already exists.')
            else ()
        else (
            if (empty($sourceValueset)) then 
                error($errors:BAD_REQUEST, 'Cannot create new valueSet. Source valueset based on id ' || $sourceId || ' and effectiveDate ' || $sourceEffectiveDate || ' does not exist.')
            else  ()
        )
    let $check                  :=
        if ($now castable as xs:dateTime) then () else (
            error($errors:BAD_REQUEST, 'Cannot create new valueSet. The provided effectiveDate ' || $now || ' is not a valid xs:dateTime. Expected yyyy-mm-ddThh:mm:ss.')
        )
    
    let $editedValueset         := ($editedValueset, $sourceValueset)[1]
    (: decorlib:getNextAvailableIdP() returns <next base="1.2" max="2" next="{$max + 1}" id="1.2.3" type="DS"/> :)
    let $newValuesetId          := if ($keepIds) then $editedValueset/@id else (decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-VALUESET, $baseId)/@id)
    
    let $check                  :=
        if (utillib:getValueSet($projectPrefix, (), (), $newValuesetId, $now, false(), false())[@id]) then
            error($errors:BAD_REQUEST, 'Cannot create new valueSet. The to-be-created valueSet with id ' || $newValuesetId || ' and effectiveDate ' || $now || ' already exists.')
        else ()
    
    let $baseValueset           := 
        <valueSet>
        {
            attribute id {$newValuesetId} ,
            $editedValueset/@name[string-length()>0] ,
            $editedValueset/@displayName[string-length()>0] ,
            attribute effectiveDate {$now} ,
            attribute statusCode {"draft"} ,
            $editedValueset/@versionLabel[string-length()>0] ,
            $editedValueset/@expirationDate[string-length()>0] ,
            $editedValueset/@officialReleaseDate[string-length()>0] ,
            $editedValueset/@experimental[string-length()>0] ,
            $editedValueset/@canonicalUri[string-length()>0] ,
            $editedValueset/@lastModifiedDate[string-length()>0]
        }
        </valueSet>
    
    let $newValueSet            := 
        if ($refOnly) then <valueSet ref="{$sourceValueset/@id}" name="{$sourceValueset/@name}" displayName="{($sourceValueset/@displayName, $sourceValueset/@name)[1]}"/>
        else (
            let $check          := ruleslib:checkValueSetItems($newValuesetId, $editedValueset, 'create')
            return utilvs:prepareValueSetForUpdate($editedValueset, $baseValueset)
        )
    
    let $check                  := ruleslib:checkDecorSchema($newValueSet, 'create')         
    
    let $valueSetAssociation    := 
        if ($editedValueset[string-length(@conceptId)>0]) then 
            <terminologyAssociation>
            {
                $editedValueset/@conceptId,
                attribute valueSet {$newValueSet/@id},
                if ($editedValueset[string-length(@flexibility)=0]) then () else (
                    attribute flexibility {$newValueSet/@effectiveDate}
                ),
                attribute effectiveDate {$now}
            }
            </terminologyAssociation>
        else ()
    
    let $terminologyAssociations    :=
        for $concept in $editedValueset//*[string-length(@conceptId)>0][string-length(@code)>0][string-length(@codeSystem)>0]
        return
            <terminologyAssociation>
            {
                $concept/@conceptId, $concept/@code, $concept/@codeSystem, $concept/@displayName, 
                attribute effectiveDate {$now}
            }
            </terminologyAssociation>
    
    (: prepare sub root elements if not existent :)
    let $valueSetUpdate         :=
        if (not($decor/terminology) and $decor/codedConcepts) then update insert <terminology/> following $decor/codedConcepts
        else if (not($decor/terminology) and not($decor/codedConcepts)) then update insert <terminology/> following $decor/ids
        else ()
    
    (: now update the value set :)
    let $valueSetUpdate         :=
        (: this could update the name/displayName if the source thing was updated since adding it previously :)
        if ($refOnly and $decor//valueSet[@ref = $newValueSet/@ref]) then update replace $decor//valueSet[@ref = $newValueSet/@ref] with $newValueSet
        else if ($decor/terminology/conceptMap) then update insert $newValueSet preceding $decor/terminology/conceptMap[1] 
        else update insert $newValueSet into $decor/terminology
    
    let $terminologyAssociationUpdates  :=
        utillib:addTerminologyAssociations($valueSetAssociation | $terminologyAssociations, $decor, false(), false())
   
    (: return the regular valueSet that was created, or the requested version of the reference that was created, or the latest version of the reference that was created :)
    return utillib:getValueSet($projectPrefix, (), (), ($newValueSet/@id | $newValueSet/@ref), ($newValueSet/@effectiveDate, $sourceEffectiveDate, 'dynamic')[1], false(), true())
};

(:~ Returns a list of zero or more valuesets as listed in the terminology section. This function is useful e.g. to call from a ValueSetIndex. Parameter id, name or prefix is required.
    @param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only
    @param $projectVersion   - optional. if empty defaults to current version. if valued then the valueset will come explicitly from that archived project version which is expected to be a compiled version
    @param $projectLanguage  - optional. defaults to project defaultLanguage or first available if multiple compilation exist 
    @param $objectid         - optional. Identifier of the valueset to retrieve
    @param $objectnm         - optional. Name of the valueset to retrieve (valueSet/@name)
    @param $objected         - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
    @param $max              - optional. Maximum number of results with minimum 1.
    @param $resolve          - optional. Default = 'true' If true, resolves any references
    @return List object with zero or more valueSet
    @since 2013-06-14
:)
declare function utilvs:getValueSetList($governanceGroupId as xs:string?, $projectPrefix as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?, $searchTerms as xs:string*, $objected as xs:string?, $includebbr as xs:boolean, $sort as xs:string?, $sortorder as xs:string?, $max as xs:integer?, $resolve as xs:boolean) as element(list) {
    
    let $projectLanguage        := ($projectLanguage[not(. = '')], '*')[1]
    
    let $startT                 := util:system-time()
    
    let $decor                  := 
        if ($governanceGroupId) then
            for $projectId in utilgg:getLinkedProjects($governanceGroupId)/@ref
            return utillib:getDecorById($projectId)
        else utillib:getDecor($projectPrefix, $projectVersion, $projectLanguage)
 
    let $projectPrefix          := ($decor/project/@prefix)[1]
    
    let $results                := 
        if (empty($searchTerms)) then $decor//valueSet
        else (
            let $buildingBlockRepositories  := if ($includebbr) then utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL) else $decor
            let $luceneQuery    := utillib:getSimpleLuceneQuery($searchTerms, 'wildcard')
            let $luceneOptions  := utillib:getSimpleLuceneOptions() 

            for $ob in ($buildingBlockRepositories//valueSet[@id = $searchTerms] | 
                        $buildingBlockRepositories//valueSet[@id][ft:query(@name, $luceneQuery, $luceneOptions)] |
                        $buildingBlockRepositories//valueSet[@id][ft:query(@displayName, $luceneQuery, $luceneOptions)])
            return
                <valueSet>
                {
                    $ob/(@* except (@url|@ident)),
                    attribute url {($ob/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]},
                    attribute ident {$ob/ancestor::decor/project/@prefix},
                    attribute cachedProject {exists($ob/ancestor::cacheme)},
                    $ob/node()
                }
                </valueSet>
        )
    
    let $allcnt                 := count($results)
    
    let $objectsByRef           :=
        if ($resolve and empty($projectVersion)) then (
            let $decors         := if (empty($projectVersion)) then utillib:getBuildingBlockRepositories($decor, map {}, $utillib:strDecorServicesURL) else $decor
            
            for $csref in $results/@ref
            group by $csref
            return (
                let $css        := $decors//valueSet[@id = $csref]
                let $csbyid := $css[@effectiveDate = max($css[@effectiveDate castable as xs:dateTime]/xs:dateTime(@effectiveDate))]
                (: rewrite name and displayName based on latest target codeSystem. These sometimes run out of sync when the original changes its name :)
                return (
                    element {name($csref[1])} {
                        $csref[1], 
                        attribute name {($csbyid/@name, $csref/../@name)[1]}, 
                        attribute displayName {($csbyid/@displayName, $csref/../@displayName, $csbyid/@name, $csref/../@name)[1]}
                    }
                    ,
                    for $csid in $css
                    return
                    element {name($csid)} {
                        $csid/@*,
                        if ($csid/ancestor::decor) then (
                            if ($csid/@url) then () else   attribute url {($csid/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]},
                            if ($csid/@ident) then () else attribute ident {($csid/ancestor::decor/project/@prefix)[1]}
                        )
                        else (),
                        (: this element is not supported (yet?) :)
                        $csid/classification
                    }
                )
            )
        )
        else $results[@ref]
    
    let $results                := $results[@id] | $objectsByRef
    
    (: now we can determine $objected :)
    let $results                :=
        if (empty($objected)) then $results
        else if ($objected castable as xs:dateTime) then  $results[@ref] | $results[@effectiveDate = $objected]
        else $results[@ref] | $results[@effectiveDate = max($results[@effectiveDate]/xs:dateTime(@effectiveDate))]
    
    let $results                :=
        for $res in $results
        let $id                 := $res/@id | $res/@ref
        group by $id
        return (
            let $subversions    :=
                for $resv in $res
                order by $resv/@effectiveDate descending
                return
                    <valueSet uuid="{util:uuid()}">
                    {
                        $resv/(@* except @uuid),
                        (: this element is not supported (yet?) :)
                        $resv/classification
                    }
                    </valueSet>
            let $latest         := ($subversions[@id], $subversions)[1]
            let $rproject       := $res/ancestor::decor/project/@prefix
            return
            <valueSet uuid="{util:uuid()}" id="{$id}">
            {
                $latest/(@* except (@uuid | @id | @ref | @project)),
                ($res/@ref)[1],
                (: there is no attribute @project, but better safe than sorry :)
                (: wrong setting for governance group search: if (empty($governanceGroupId)) then $latest/@project else attribute project {$projectPrefix}, :)
                if (empty($governanceGroupId)) then $latest/@project else attribute project { $rproject[1] },
                $subversions
            }
            </valueSet>
        )
    let $count                  := count($results/valueSet)
    let $max                    := if ($max ge 1) then $max else $count
    
    (: handle sorting. somehow reverse() does not do what I expect :)
    let $results                :=
        switch ($sort)
        case 'displayName' return 
            if ($sortorder = 'descending') then 
                for $r in $results order by $r/lower-case(@displayName) descending return $r
            else (
                for $r in $results order by $r/lower-case(@displayName)            return $r
            )
        case 'name'        return 
            if ($sortorder = 'descending') then 
                for $r in $results order by $r/lower-case(@name) descending return $r
            else (
                for $r in $results order by $r/lower-case(@name)            return $r
            )
        default            return 
            if ($sortorder = 'descending') then 
                for $r in $results order by replace(replace($r/@id, '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1') descending return $r
            else (
                for $r in $results order by replace(replace($r/@id, '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1')            return $r
            )
    
    let $durationT              := (util:system-time() - $startT) div xs:dayTimeDuration("PT0.001S")
    
    return
        <list artifact="VS" elapsed="{$durationT}" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" resolve="{$resolve}" project="{$projectPrefix}" lastModifiedDate="{current-dateTime()}">
        {
            subsequence($results, 1, $max)
        }
        </list>
};

(:~ Returns a valueset for publication in csv format 
    @param $cvalueSetss required, is retrieved from request id/effectiveDate
    @param $language optional from request parameter or derived from artLanguage 
    @param $projectPrefix optional from parameter request 
    @returns a valueSet in csv format
:)
declare function utilvs:convertValueSet2Csv($valueSets as element()*, $language as xs:string?, $projectPrefix as xs:string?) as xs:string* {

    let $language           := if (empty($language)) then $setlib:strArtLanguage else $language
    
    let $mapentries :=
        for $d in distinct-values($valueSets//designation/string-join((@type,@language),' / '))
        return map:entry($d, max($valueSets//*/count(designation[$d = string-join((@type,@language),' / ')])))
    let $designationmap     := map:merge($mapentries)
    let $header             := 
        concat('Level;Type;Code;DisplayName;CodeSystem;CodeSystemName;CodeSystemVersion;Flexibility;Exception',
            if (count($mapentries)=0) then () else (
                concat(';',string-join(
                    for $d in map:keys($designationmap)
                    for $i in (1 to map:get($designationmap, $d))
                    return concat('&quot;Designation ',$d,'&quot;')
                ,';'))
            )
        ,'&#13;&#10;')
    let $columncount        := count(tokenize($header,';'))
    let $keyValueSet        :=
        switch($language)
            case "en-US" return "Value Set"
            case "de-DE" return "Value Set"
            case "nl-NL" return "Waardelijst"
            default return "Value Set"
    return (
    (:<concept code="xxxxxx" codeSystem="2.16.840.1.113883.2.4.15.4" displayName="Medicatie" level="0" type="A"/> :)
    (: Replace double quotes with single quotes in the CSV values, except in the code itself, 
    and place in between double quotes if there a white space character in a string
    Note that in the exceptional event that a code contains a double quote, the CSV renders invalid :)
    for $valueset at $i in $valueSets//valueSet[@id]
    return (
        if ($i > 1) then ('&#13;&#10;') else (),
        concat('&quot;',$keyValueSet,' ',$valueset/@displayName,' - ',$valueset/(@id|@ref),' ',$valueset/@effectiveDate,'&quot;',string-join(for $i in (1 to ($columncount - 1)) return ';',''),'&#13;&#10;'),
        $header,
        for $concept in ($valueset/completeCodeSystem, $valueset/conceptList/include[@codeSystem][not(@code)], $valueset/conceptList/concept, $valueset/conceptList/exception)
            let $conceptCode                    := data($concept/@code)
            let $quotedConceptCode              := if (matches($conceptCode,'\s+')) then (concat('&quot;',$conceptCode,'&quot;')) else ($conceptCode)
            let $conceptDisplayName             := replace(data($concept/@displayName),'"','&apos;')
            let $quotedConceptDisplayName       := if (matches($conceptDisplayName,'\s+')) then (concat('&quot;',$conceptDisplayName,'&quot;')) else ($conceptDisplayName)
            let $conceptCodeSystem              := replace(data($concept/@codeSystem),'"','&apos;')
            let $quotedConceptCodeSystem        := if (matches($conceptCodeSystem,'\s+')) then (concat('&quot;',$conceptCodeSystem,'&quot;')) else ($conceptCodeSystem)
            let $generatedCodeSystemName        := utillib:getNameForOID($concept/@codeSystem,$language,$projectPrefix)
            let $conceptCodeSystemName          := replace(if ($concept/@codeSystemName) then (data($concept/@codeSystemName)) else if (replace($generatedCodeSystemName,'[0-9\.]','')!='') then ($generatedCodeSystemName) else (''),'"','&apos;')
            let $quotedConceptCodeSystemName    := if (matches($conceptCodeSystemName,'\s+')) then (concat('&quot;',$conceptCodeSystemName,'&quot;')) else ($conceptCodeSystemName)
            let $conceptCodeSystemVersion       := replace(data($concept/@codeSystemVersion),'"','&apos;')
            let $quotedConceptCodeSystemVersion := if (matches($conceptCodeSystemVersion,'\s+')) then (concat('&quot;',$conceptCodeSystemVersion,'&quot;')) else ($conceptCodeSystemVersion)
            let $conceptFlexibility             := if ($concept[self::completeCodeSystem | self::include]) then (replace(if ($concept/@flexibility) then (data($concept/@flexibility)) else ('dynamic'),'"','&apos;')) else ('')
            let $quotedconceptFlexibility       := if (matches($conceptFlexibility,'\s+')) then (concat('&quot;',$conceptFlexibility,'&quot;')) else ($conceptFlexibility)
            let $quotedException                := $concept/name()='exception'
        return (
            concat(data($concept/@level),';',data($concept/@type),';',$quotedConceptCode,';',$quotedConceptDisplayName,';',$quotedConceptCodeSystem,';',$quotedConceptCodeSystemName,';',$quotedConceptCodeSystemVersion,';',$quotedconceptFlexibility,';',$quotedException,
                concat(';',string-join(
                    for $d in map:keys($designationmap)
                    for $i in (1 to map:get($designationmap, $d))
                    let $designation    := $concept/designation[$d=string-join((@type,@language),' / ')][1]
                    return 
                        if ($designation[@displayName]) then
                        if (matches($designation/@displayName,'\s+')) then (
                            concat('&quot;',$designation/@displayName,'&quot;')
                        ) else (
                            $designation/@displayName
                        )
                        else ('')
                ,';'))
            ,'&#13;&#10;')
        )
    ))
    
};    

(:~ Returns a valueset for publication in xml-svs format 
    @param $valueSets required, is retrieved from request id/effectiveDate
    @param $language optional from request parameter or derived from artLanguage 
    @param $projectPrefix optional from parameter request 
    @returns a valueSet in xml-svs format
    
    KH 20190815
    format mdibagch is a proprietay special type of svs in a different namespace "urn:ch:admin:bag:epr:2017:MdiImport"
    for customer Bundesamt für Gesundheits, Bern, Schweiz
:)
declare function utilvs:convertValueSet2Svs($valueSets as element()*, $language as xs:string?, $projectVersion as xs:string?, $format as xs:string) as element()* {

    let $valuesetcount          := count($valueSets//*:valueSet[@id])

    (: KH 20190815 speacial handling of value set element name for mdibagch :)
    let $valuesetelementname    := if (($valuesetcount > 1) and ($format!='mdibagch')) then 'DescribedValueSet' else 'ValueSet'

    let $decor                  := utillib:getDecorByPrefix(($valueSets//project/@ident)[1], $projectVersion)[1]
    let $url                    := if ($decor) then $decor/project else ()
    let $projectname            := if ($url) then ($url/name)[1] else ()
    let $svs1language           := if (empty($language)) then '' else $language 
    let $language               := if (empty($language)) then $setlib:strArtLanguage else $language
    
    let $svsresult              :=  
        for $valueset in $valueSets//*:valueSet[@id]
        let $intensional        := $valueset/*:completeCodeSystem or $valueset/*:conceptList/*:include or $valueset/*:conceptList/*:exclude
        let $includerefonly     := not($intensional) or ($valueset/*:conceptList/*:include[@ref] and not($valueset/*:completeCodeSystem or $valueset/*:conceptList/*:include[empty(@ref)] or $valueset/*:conceptList/*:exclude))
        (: KH 20190815 special handling of namespaces for mdibagch :)
        let $defaultNamespace-uri   :=  if ($format='mdibagch') then "urn:ch:admin:bag:epr:2017:MdiImport" else "urn:ihe:iti:svs:2008"
        let $otherLanguages     := $valueset/*:conceptList/*:concept/*:designation[@language != $svs1language] | $valueset/*:conceptList/*:exception/*:designation[@language != $svs1language]
        let $tmpLang            := $valueset/*:conceptList/*:concept/*:designation/@language
        let $hasnodesignations  := count($tmpLang) = 0
        let $allLanguages       := if ($hasnodesignations) then $setlib:strArtLanguage else $tmpLang

        return
        (:<result xmlns="urn:ihe:iti:svs:2008" xmlns:ihesvs="urn:ihe:iti:svs:2008" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">:)
        element {QName($defaultNamespace-uri,"result")} {
            element {QName($defaultNamespace-uri, $valuesetelementname)} {
                attribute { "id" } {$valueset/@id},
                attribute { "displayName" } {if ($valueset/@displayName) then $valueset/@displayName else $valueset/@name},
                attribute { "version" } {if ($valueset/@versionLabel) then $valueset/@versionLabel else $valueset/@effectiveDate},
                
                (: Source: This is the source of the value set, identifying the originator or publisher of the information :)
                let $pa      := if ($valueset/*:publishingAuthority/@name) then data($valueset/*:publishingAuthority/@name) else ()
                let $cr      := if ($valueset/*:copyright) then $valueset/*:copyright else ()
                return if ($pa or $cr) then 
                   <Source>
                   {
                       $pa,
                       if ($pa) then ' - ' else (),
                       string-join($cr/text(), ' - ')
                   }
                   </Source> 
                   else (),
                
                (: Purpose: Brief description about the general purpose of the value set :)
                if ($valueset/*:desc[@language="en-US"]) then <Purpose>{$valueset/*:desc[@language="en-US"]/text()}</Purpose> 
                else if ($valueset/*:desc[not(@language="en-US")][1]) then <Purpose>{concat(data($valueset/*:desc/@language), ': ')}{$valueset/*:desc[not(@language="en-US")][1]/text()}</Purpose> 
                else (),
            
                (: Active, Inactive, local extensions :)
                if ($valueset/@statusCode=("final", "draft")) then <Status>Active</Status>
                else if ($valueset/@statusCode=("rejected", "cancelled", "deprecated")) then <Status>Inactive</Status>
                else <Status>{data($valueset/@statusCode)}</Status>,
                
                (: Type: This describes the type of the value set. It shall be Intensional, Extensional, or Expanded.
                        Note: This is the type of the value set in the repository. The ConceptList that will also be returned is the current expansion of the value set.
                :)
                if ($intensional) then <Type>Intensional</Type> else <Type>Extensional</Type>,
                
                (: Effective Date The date when the value set is expected to be effective, defaults to effectiveDate :)
                if ($valueset/@officialReleaseDate) then <EffectiveDate>{data(substring($valueset/@officialReleaseDate,1, 10))}</EffectiveDate> else <EffectiveDate>{data(substring($valueset/@effectiveDate,1, 10))}</EffectiveDate>,
                
                (: Expiration Date: The date when the value set is no longer expected to be used :)
                if ($valueset/@expirationDate) then <ExpirationDate>{data(substring($valueset/@expirationDate,1, 10))}</ExpirationDate> else (),

                (: Creation Date: The date of creation of the value set :)
                if ($valueset/@effectiveDate) then <CreationDate>{data(substring($valueset/@effectiveDate,1, 10))}</CreationDate> else (),
                if (not($intensional) or $includerefonly) then (
                    if ( (string-length($svs1language) = 0) or ($hasnodesignations) ) then (
                        for $lang in distinct-values($allLanguages)
                        return
                        <ConceptList xml:lang="{$lang}">
                        {
                            for $concept in ($valueset/*:conceptList/*:concept | $valueset/*:conceptList/*:exception)
                            let $d := if ($concept/designation[@language=$lang]) then $concept/designation[@language=$lang] else $concept
                            let $conceptCode              := data($concept/@code)
                            let $conceptCodeSystem        := data($concept/@codeSystem)
                            let $conceptDisplayName       := data($d/@displayName)
                            let $conceptCodeSystemVersion := data($concept/@codeSystemVersion)
                            return
                            <Concept code="{$conceptCode}" codeSystem="{$conceptCodeSystem}" displayName="{$conceptDisplayName}">
                            {
                                if (string-length($conceptCodeSystemVersion) > 0) then $conceptCodeSystemVersion else ()
                            }
                            </Concept>
                        }
                        </ConceptList>
                    )
                    else (
                        (: a specific language is specified, find the designations :)
                        <ConceptList xml:lang="{$svs1language}">
                        {
                            for $concept in ($valueset/*:conceptList/*:concept | $valueset/*:conceptList/*:exception)
                            let $d := if ($concept/designation[@language=$svs1language]) then $concept/designation[@language=$svs1language] else $concept
                            let $conceptCode              := data($concept/@code)
                            let $conceptCodeSystem        := data($concept/@codeSystem)
                            let $conceptDisplayName       := data($d/@displayName)
                            let $conceptCodeSystemVersion := data($concept/@codeSystemVersion)
                            return
                            <Concept code="{$conceptCode}" codeSystem="{$conceptCodeSystem}" displayName="{$conceptDisplayName}">
                            {
                                if (string-length($conceptCodeSystemVersion) > 0) then $conceptCodeSystemVersion else ()
                            }
                            </Concept>
                        
                        }
                        </ConceptList>
                    )
                )
                else (<!-- Intensional Value Set not expanded -->),
                (:
                   Place Project OID if available in Group
                   <Group id="2.16.840.1.113883.3.1937.777.13" displayName="International Patient Summary (IPS)" sourceOrganization="ART-DECOR Expert Group">
                :)
                if ($url) then
                       <Group id="{$url/@id}" displayName="{$projectname}" sourceOrganization="ART-DECOR Expert Group">
                       {
                           <Keyword>ART-DECOR Project</Keyword>
                       }
                       </Group>
                else () 
                }
            }
        
    return
        (: KH 20190815 special handling for mdibagch, no cacheExpirationHint="{format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')}" :)
        if ($format='mdibagch') then
            <RetrieveMultipleValueSetsResponse 
                    xmlns="urn:ch:admin:bag:epr:2017:MdiImport" xmlns:ihesvs="urn:ihe:iti:svs:2008" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
            {
                    $svsresult/*
            }
            </RetrieveMultipleValueSetsResponse>
        (: standard svs format :)
        else if ($valuesetcount > 1) then
            <RetrieveMultipleValueSetsResponse 
                xmlns="urn:ihe:iti:svs:2008" xmlns:ihesvs="urn:ihe:iti:svs:2008" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                cacheExpirationHint="{format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')}">
            {
                $svsresult/*
            }
            </RetrieveMultipleValueSetsResponse>
        else
            <RetrieveValueSetResponse 
                xmlns="urn:ihe:iti:svs:2008" xmlns:ihesvs="urn:ihe:iti:svs:2008" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                cacheExpirationHint="{format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')}">
            {
                $svsresult/*
            }
            </RetrieveValueSetResponse>

};

(:~ Returns a valueset for publication in xml-svs format (old)
    @param $valueSets required, is retrieved from request id/effectiveDate
    @param $language optional from request parameter 
    @returns a valueSet in xml-svs format
:)
declare function utilvs:convertValueSet2Svsold($valueSets as element()*, $language as xs:string?) as element()* {

    <RetrieveValueSetResponse xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="urn:ihe:iti:svs:2008" cacheExpirationHint="{format-dateTime(current-dateTime(), '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01]')}">
    {
        for $valueset in $valueSets//*:valueSet[@id]
        return
            <ValueSet id="{$valueset/@id}" displayName="{if ($valueset/@displayName) then $valueset/@displayName else $valueset/@name}" version="{if ($valueset/@versionLabel) then $valueset/@versionLabel else $valueset/@effectiveDate}">
            {
                <ConceptList xml:lang="en-US">
                {
                    for $concept in ($valueset/*:conceptList/*:concept | $valueset/*:conceptList/*:exception)
                    let $conceptCode        := data($concept/@code)
                    let $conceptCodeSystem  := data($concept/@codeSystem)
                    let $conceptDisplayName := data($concept/@displayName)
                    return
                    <Concept code="{$conceptCode}" codeSystem="{$conceptCodeSystem}" displayName="{$conceptDisplayName}" />
                }
                </ConceptList>,
                let $otherLanguages := ($valueset/*:conceptList/*:concept/*:designation[@language != 'en-US'] | 
                                        $valueset/*:conceptList/*:exception/*:designation[@language != 'en-US'])
                return
                if ($otherLanguages) then
                    for $lang in distinct-values($otherLanguages/@language)
                    return
                        if (string-length($language) = 0 or (string-length($language) > 0 and $language=$lang))
                        then
                            <ConceptList xml:lang="{$lang}">
                            {
                                for $d in ($otherLanguages[@language=$lang])
                                let $designationDisplayName := data($d/@displayName)
                                let $conceptCode            := data($d/parent::*/@code)
                                let $conceptCodeSystem      := data($d/parent::*/@codeSystem)
                                return
                                <Concept code="{$conceptCode}" codeSystem="{$conceptCodeSystem}" displayName="{$designationDisplayName}" />
                            }
                            </ConceptList>
                        else ()
                else ()
            }
            </ValueSet>
    }
    </RetrieveValueSetResponse>
};


(:~ Central logic for patching an existing valueSet statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap     - required. Map derived from token
    @param $id required. DECOR concept/@id to update
    @param $effectiveDate required. DECOR concept/@effectiveDate to update
    @param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function utilvs:setValueSetStatus($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $recurse as xs:boolean, $listOnly as xs:boolean, $newStatusCode as xs:string?, $newVersionLabel as xs:string?, $newExpirationDate as xs:string?, $newOfficialReleaseDate as xs:string?) as element(list) {
    (:get object for reference:)
    let $object                 := if (empty($effectiveDate)) then (
            let $ttt            := $setlib:colDecorData//valueSet[@id = $id]
            return $ttt[@effectiveDate = max($ttt/xs:dateTime(@effectiveDate))]
        )
        else $setlib:colDecorData//valueSet[@id = $id][@effectiveDate = $effectiveDate]

    let $decor                  := $object/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($object) = 1) then () else
        if ($object) then
            error($errors:SERVER_ERROR, 'ValueSet id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($object) || ' were found in: ' || string-join(distinct-values($object/ancestor::decor/project/@prefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'ValueSet id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )
    let $check                  := utilvs:checkValueSetAccess($authmap, $decor, (), (), false())
        
    let $testUpdate             :=
        utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, true(), $projectPrefix)
    let $results                :=
        if ($testUpdate[self::error]) then $testUpdate else
        if ($listOnly) then $testUpdate else (
            utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, false(), $projectPrefix)
        )
    return
    if ($results[self::error] and not($listOnly)) then
        error($errors:BAD_REQUEST, string-join(
            for $e in $results[self::error]
            return
                $e/@itemname || ' id ''' || $e/@id || ''' effectiveDate ''' || $e/@effectiveDate || ''' cannot be updated: ' || data($e)
            , ' ')
        )
    else $results
};

(:~ Central logic for patching an existing valueSet expansion statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap required. Map derived from token
    @param $id required. valueSetExpansionset/@id
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @return list object with success and/or error elements or error
:)
declare function utilvs:setValueSetExpansionSetStatus($authmap as map(*), $id as xs:string, $newStatusCode as xs:string?, $newExpirationDate as xs:string?) as element(list) {
    (:get object for reference:)
    let $object                 := collection($setlib:strValuesetExpansionData)/valueSetExpansionSet[@id = $id]
    
    let $check                  :=
        if (count($object) = 1) then () else
        if ($object) then
            error($errors:SERVER_ERROR, 'Value set expansion id ''' || $id || ''' cannot be updated because multiple ' || count($object) || ' were found. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Value set expansion id ''' || $id || ''' cannot be updated because it cannot be found.')
        )
    let $check                  :=
        if (empty($newStatusCode)) then () else
        if (utillib:isStatusChangeAllowable($object, $newStatusCode)) then () 
        else (
            error($errors:BAD_REQUEST, 'Status transition not allowed from ' || $object/@statusCode || ' to ' || $newStatusCode || '. Allowed: ' || string-join(map:get($utillib:valuesetexpansionstatusmap, string($object/@statusCode)), ', or '))
        )
    let $check                  :=
        if (empty($newExpirationDate)) then () else if ($newExpirationDate castable as xs:dateTime or $newExpirationDate = '') then () 
        else (
            error($errors:BAD_REQUEST, 'New value set expansion expirationDate ''' || $newExpirationDate || ''' is not supported. Please support a valid dateTime or empty string')
        )
        
    let $check                  := utilvs:checkValueSetExpansionAccess($authmap, $object, false())
        
    let $update                 :=
        if ($newStatusCode) then
            update value $object/@statusCode with $newStatusCode
        else ()
    let $update                 :=
        if ($newExpirationDate) then
            if ($newExpirationDate = '') then
                update delete $object/@expirationDate
            else
            if ($object/@expirationDate) then
                update value $object/@expirationDate with $newExpirationDate
            else (
                update insert attribute expirationDate {$newExpirationDate} into $object 
            )
        else ()

    return $object
        
};

(:~ Central logic for patching an existing valueSet
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR valueSet/@id to update
    @param $effectiveDate   - required. DECOR valueSet/@effectiveDate to update
    @param $data            - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
    @return dataset object as xml with json:array set on elements
:)
declare function utilvs:patchValueSetExpansionSet($authmap as map(*), $id as xs:string, $data as element(parameters)) as element(valueSetExpansionSet) {

    let $storedExpansion        := collection($setlib:strValuesetExpansionData)/valueSetExpansionSet[@id = $id]
    
    let $check                  :=
        if (count($storedExpansion) = 1) then () else
        if ($storedExpansion) then
            error($errors:SERVER_ERROR, 'Value set expansion id ''' || $id || ''' cannot be updated because multiple (' || count($storedExpansion) || ') were found. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Value set expansion id ' || $id || ' does not exist')
        )
        
    let $check                  := utilvs:checkValueSetExpansionAccess($authmap, $storedExpansion, false())
    
    let $check                  :=
        if ($data[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $data/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $check                  :=
        for $param in $data/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($storedExpansion, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:valuesetexpansionstatusmap, string($storedExpansion/@statusCode)), ', ')
                )
            )
            case '/expirationDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     let $check                 :=
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )
    
    let $update                 :=
        for $param in $data/parameter
        return
            switch ($param/@path)
            case '/statusCode'
            case '/expirationDate' return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := 
                    if ($attname = 'expirationDate') then
                        attribute {$attname} {substring($param/@value, 1, 19)}
                    else (
                        attribute {$attname} {$param/@value}
                    )
                let $stored   := $storedExpansion/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $storedExpansion
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
    
    return $storedExpansion
};

declare function utilvs:checkValueSetAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean) as element()?{

    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify valueSets in project ', $decor/project/@prefix, '. You have to be an active author in the project.'))
        )

    return
        if ($checkLock) then (
            let $lock           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
            let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, false())
            return
                if ($lock) then $lock else (
                    error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this valueSet (anymore). Get a lock first.'))
                )    
            )
            else ()
};

declare %private function utilvs:checkValueSetParameters($parameters as element(parameters), $valueSet as element(valueSet)*) {

    let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $check                  :=
        for $param in $parameters/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($valueSet, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($valueSet/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri'
            case '/versionLabel' return (
                if ($op = 'remove') then () else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/name' 
            case '/displayName' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/experimental' return (
                if ($op = 'remove') then () else 
                if ($value[. = ('true', 'false')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be true or false.'
                )
            )
            case '/desc' 
            case '/purpose'
            case '/copyright' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $pathpart], $op, $path)
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart])
                )
            )
            case '/publishingAuthority' return (
                if ($param[count(value/publishingAuthority) = 1]) then ruleslib:checkPublishingAuthority($param/value/publishingAuthority, $op, $path)
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single publishingAuthority under value. Found ' || count($param/value/publishingAuthority)
                )
            )
            case '/items' return (
                if ($op = 'remove') then () else
                if ($param[count(value/items) ge 1]) then (
                    for $concept in (   $param/value/items[@is='concept'] | 
                                        $param/value/items[@is='exception'])
                    let $is     := ($concept/@is, name($concept))[1]
                    return (
                        if ($concept/@type[. = $utilvs:VOCAB-TYPE/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .type value ''' || $concept/@type || '''. SHALL be one of: ' || string-join($utilvs:VOCAB-TYPE/enumeration/@value, ', ')
                        ),
                        if ($concept/@level[. castable as xs:nonNegativeInteger]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .level value ''' || $concept/@level || '''. SHALL be 0 or higher'
                        ),
                        if ($concept/@codeSystem[utillib:isOid(.)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .codeSystem value ''' || $concept/@codeSystem || '''. SHALL be a valid oid'
                        ),
                        if ($concept/@code[matches(., '^\S+$')]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .code value ''' || $concept/@code || '''. SHALL be populated and SHALL NOT have whitespace'
                        ),
                        if ($concept/@displayName[not(. = '')]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .displayName value ''' || $concept/@displayName || '''. SHALL be populated'
                        )
                    ),
                    for $concept in (   $param/value/items[@is='include'] | 
                                        $param/value/items[@is='exclude'])
                    let $is     := ($concept/@is, name($concept))[1]
                    return (
                        if ($concept[empty(@op)] | $concept[@op = $utilvs:INTENSIONALOPERATORS-TYPE/enumeration/@value]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .op value ''' || $concept/@op || '''. SHALL be one of: ' || string-join($utilvs:INTENSIONALOPERATORS-TYPE/enumeration/@value, ', ')
                        ),
                        if ($concept[@code][@flexibility]) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported combination of attributes. SHALL NOT have .code and .flexibility'
                        else
                        if ($concept[empty(@ref | @codeSystem)][@flexibility]) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported combination of attributes. SHALL NOT have .flexibility without .codeSystem or .ref'
                        else
                        if ($concept[empty(@code)][@op]) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported combination of attributes. SHALL NOT have .op without .code'
                        else
                        if ($concept[@ref][@op]) then
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported combination of attributes. SHALL NOT have .op and .ref'
                        else
                        
                        
                        if ($concept[empty(@codeSystem)] | $concept[utillib:isOid(@codeSystem)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .codeSystem value ''' || $concept/@codeSystem || '''. SHALL be a valid oid'
                        ),
                        if ($concept[empty(@code)] | $concept[matches(@code, '^\S+$')]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' has unsupported .codeSystem value ''' || $concept/@code || '''. SHALL NOT have whitespace'
                        ),
                        if ($concept/@ref = $valueSet/(@id | @ref)) then 
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. ValueSet ' || $is || ' SHALL NOT include itself'
                        else (),
                        if ($concept/filter) then (
                            if ($concept/filter[.[empty(@property)] | .[empty(@op)] | .[empty(@value)]]) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || string-join($concept/@codeSystem, ', ') || ' has at least one filter that is missing property, op or value.'
                            else (),
                            let $unsupops   := $concept/filter[not(@op = $utilvs:INTENSIONALOPERATORS-TYPE/enumeration/@value)]
                            return
                            if ($unsupops) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || string-join($concept/@codeSystem, ', ') || ' has at least one filter with an unsupported value for ''op''. Found: ''' || string-join($unsupops/@op, ', ') || '''. SHALL be one of: ' || string-join($utilvs:INTENSIONALOPERATORS-TYPE/enumeration/@value, ', ')
                            else ()
                        )
                        else (),
                        if ($is = 'exclude') then 
                            if ($concept[@codeSystem][empty(@code)]) then
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. ' || $is || ' SHALL have both .code and .codeSystem. Exclusion of complete code system not supported'
                            else ()
                        else ()
                    ),
                    let $css                := $param/value/items[@is=('include', 'exclude')][@codeSystem][empty(@code)]/@codeSystem
                    return
                        if (count($css) = count(distinct-values($css))) then () else (
                             'Parameter ' || $op || ' not allowed for ''' || $path || '''. Value set SHALL contain max 1 include per complete code system.'
                        )
                    ,
                    let $css                := $param/value/items[@is=('include', 'exclude')]/@ref
                    return
                        if (count($css) = count(distinct-values($css))) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Value set SHALL contain max 1 include/exclude per value set regardless of its version.'
                        )
                ) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have one or more items under value. Found ' || count($param/value/items)
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     return
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join ($check, ' '))
        )

};

(:~  Return only valid, non empty attributes on a valueSet as defined in the DECOR format.
:   Example baseValueset:
:       <valueSet id="..." name="..." displayName="..." effectiveDate="yyyy-mm-ddThh:mm:ss" statusCode="draft" versionLabel="..."/>
:
:   @param $editedValueSet     - 1..1 valueSet as edited
:   @param $baseValueset       - 1..1 valueSet element containing all attributes that should be set on the prepared valueSet, like new id/status etc.
:   @return the pruned valueSet element for the given project in $decor, or nothing input was empty
:   @since 2015-09-22
:)
declare %private function utilvs:prepareValueSetForUpdate($editedValueset as element(valueSet), $baseValueset as element(valueSet)) as element(valueSet) {
    
    <valueSet>
    {
        $baseValueset/(@*[not(. = '')] except @lastModifiedDate),
        attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)},
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedValueset/desc),
        utilvs:prepareRelationshipForUpdate($editedValueset/relationship),
        utillib:preparePublishingAuthorityForUpdate($editedValueset/publishingAuthority[empty(@inherited)]),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedValueset/purpose),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedValueset/copyright[empty(@inherited)]),
        utilvs:prepareValueSetConceptListForUpdate(
            $editedValueset/completeCodeSystem | 
            $editedValueset/items[@is = 'completeCodeSystem'] | 
            $editedValueset/conceptList/* | 
            $editedValueset/items[not(@is = 'completeCodeSystem')]
        )
    }
    </valueSet>
};

(: Yikes I was copied and then adapted from the util-template-lib.xqm ... :)
declare %private function utilvs:prepareRelationshipForUpdate($nodes as element(relationship)*) as element(relationship)* {
    
    for $node in $nodes
    return
        element {name($node)} {
            $node/@type,
            $node/@ref,
            $node/@flexibility[. = 'dynamic'] | $node/@flexibility[. castable as xs:dateTime]
        }
};

declare %private function utilvs:prepareValueSetConceptListForUpdate($editedConceptList as element()*) as element(conceptList)? {
    
    if ($editedConceptList[self::items[@is = ('completeCodeSystem','concept', 'include', 'exclude', 'exception')]|self::completeCodeSystem|self::concept|self::include|self::exclude|self::exception]) then
        <conceptList>
        {
            for $completeCodeSystem in  $editedConceptList[self::completeCodeSystem][string-length(@codeSystem) gt 0] | 
                                        $editedConceptList[self::items[@is = 'completeCodeSystem']][string-length(@codeSystem) gt 0]
            return
                <include>
                {
                    $completeCodeSystem/@codeSystem, 
                    $completeCodeSystem/@codeSystemName[not(. = '')], 
                    $completeCodeSystem/@codeSystemVersion[not(. = '')], 
                    $completeCodeSystem/@flexibility[. castable as xs:dateTime], 
                    $completeCodeSystem/@type[. = 'D'],
                    for $f in $completeCodeSystem/filter[(@property|@op|@value)[not(. = '')]]
                    return
                        <filter>{$f/@property[not(. = '')], $f/@op[not(. = '')], $f/@value[not(. = '')]}</filter>
                }
                </include>
            ,
            for $concept in (   $editedConceptList[self::items][@is='concept'][string-length(@code)>0][string-length(@codeSystem)>0] |
                                $editedConceptList[self::items][@is='include'][(@ref|@op|@codeSystem)[string-length()>0]][not(@exception='true')] |
                                $editedConceptList[self::items][@is='exclude'][(@op|@codeSystem)[string-length()>0]][not(@exception='true')] |
            
                                $editedConceptList[self::concept][string-length(@code)>0][string-length(@codeSystem)>0] |
                                $editedConceptList[self::include][(@ref|@op|@codeSystem)[string-length()>0]][not(@exception='true')] |
                                $editedConceptList[self::exclude][(@op|@codeSystem)[string-length()>0]][not(@exception='true')])
            let $elmname    := ($concept/@is, $concept/name())[1]
            return
            element {$elmname}
            {
                switch ($elmname)
                case 'concept' return ($concept/@code, $concept/@codeSystem, $concept/@codeSystemName, $concept/@codeSystemVersion, $concept/@displayName, $concept/@ordinal[. castable as xs:decimal], $concept/@level, $concept/@type)
                default return (
                    $concept/@op, $concept/@code, $concept/@codeSystem, $concept/@codeSystemName, $concept/@codeSystemVersion, $concept/@displayName, $concept/@ref, $concept/@flexibility[. castable as xs:dateTime] | $concept/@flexibility[. = 'dynamic'], $concept/@exception[. = ('true', 'false')], $concept/@type[. = 'D']
                )
                ,
                for $desc in $concept/designation[@displayName[string-length()>0]] return utillib:parseNode($desc),
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/desc),
                for $f in $concept/filter[(@property|@op|@value)[not(. = '')]]
                return <filter>{$f/@property[not(. = '')], $f/@op[not(. = '')], $f/@value[not(. = '')]}</filter>
            }
            ,
            for $concept in (   $editedConceptList[self::items][@is='exception'][string-length(@code) gt 0][string-length(@codeSystem) gt 0] |
                                $editedConceptList[self::items][@is='include'][(@ref|@op|@codeSystem)[string-length()>0]][@exception='true'] |
                                $editedConceptList[self::items][@is='exclude'][(@op|@codeSystem)[string-length() gt 0]][@exception='true'] |
            
                                $editedConceptList[self::exception][string-length(@code) gt 0][string-length(@codeSystem) gt 0] |
                                $editedConceptList[self::include][(@ref|@op|@codeSystem)[string-length() gt 0]][@exception='true'] |
                                $editedConceptList[self::exclude][(@op|@codeSystem)[string-length() gt 0]][@exception='true'])
            let $elmname  := ($concept/@is, $concept/name())[1]
            return
            element {$elmname}
            {
                switch ($elmname)
                case 'exception' return ($concept/@code, $concept/@codeSystem, $concept/@codeSystemName, $concept/@codeSystemVersion, $concept/@displayName, $concept/@ordinal[. castable as xs:decimal], $concept/@level, $concept/@type)
                default return (
                    $concept/@op, $concept/@code, $concept/@codeSystem, $concept/@codeSystemName, $concept/@codeSystemVersion, $concept/@displayName, $concept/@ref, $concept/@flexibility[. castable as xs:dateTime] | $concept/@flexibility[ . = 'dynamic'], $concept/@exception[. = ('true', 'false')], $concept/@type[. = 'D']
                )
                ,
                for $desc in $concept/designation[@displayName[string-length() gt 0]] return utillib:parseNode($desc),
                utillib:prepareFreeFormMarkupWithLanguageForUpdate($concept/desc),
                for $f in $concept/filter[(@property|@op|@value)[not(. = '')]]
                return <filter>{$f/@property[not(. = '')], $f/@op[not(. = '')], $f/@value[not(. = '')]}</filter>
            }
        }
        </conceptList>
    else ()
};

declare function utilvs:checkValueSetExpansionAccess($authmap as map(*), $object as element(), $delete as xs:boolean) {

    (: don't want to update cache or external expansions :)
    let $location1              := tokenize(util:collection-name($object), '/')[last() - 1]
    let $projectPrefix          := tokenize(util:collection-name($object), '/')[last()]
    let $location2              := string-join(($location1, $projectPrefix), '/')
    let $decor                  := if ($location1 = 'projects') then utillib:getDecorByPrefix($projectPrefix) else ()
    
    let $check                  :=
        if (empty($decor)) then (
            if ($delete) then (
                if ($authmap?dba) then () else 
                    error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to delete valueSet expansions in ', $location1, '. You need to be a dba'))
            ) 
            else 
                error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify valueSet expansions in ', $location1, '. These expansions can only be updated through server admin install'))
        )
        else if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else 
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify valueSet expansions in project ', $projectPrefix, '. You have to be an active author in the project.'))

    return ()

};

declare function utilvs:getValueSetUsage($id as xs:string, $effectiveDate as xs:string, $projectPrefix as item(), $projectVersion as xs:string?, $projectLanguage as xs:string?) as element()* {

    let $valueSetsAll           := if (empty($projectPrefix)) then utillib:getValueSetById($id, ()) else utillib:getValueSetById($id, (), $projectPrefix, $projectVersion, $projectLanguage)
     
    let $isMostRecent           := $effectiveDate = (string(max($valueSetsAll/descendant-or-self::valueSet/xs:dateTime(@effectiveDate)))) 
    
    let $vs                     := $valueSetsAll/descendant-or-self::valueSet[@effectiveDate = $effectiveDate][1]
    
    let $allAssociations        := $setlib:colDecorData//terminologyAssociation[@valueSet = $id]
    let $allAssociations        := if ($isMostRecent) then $allAssociations[@flexibility = $effectiveDate] | $allAssociations[not(@flexibility castable as xs:dateTime)] else $allAssociations[@flexibility = $effectiveDate]
    let $allAssociations        := if (empty($projectPrefix)) then $allAssociations else (
            for $ta in $allAssociations
            return if ($ta/ancestor::decor/project/@prefix = $projectPrefix) then $ta else ()
        )
        
    let $valueSetsAll           := $setlib:colDecorData//terminology/valueSet//*[@ref = $id]
    let $valueSetsAll           := if ($isMostRecent) then $valueSetsAll[@flexibility = $effectiveDate] | $valueSetsAll[not(@flexibility castable as xs:dateTime)] else $valueSetsAll[@flexibility = $effectiveDate]
    
    let $valueSetExpansionsAll  := collection($setlib:strValuesetExpansionData)/valueSetExpansionSet[valueSet[@id = $id][@effectiveDate = $effectiveDate]]
    
    let $conceptMapsAllSource   := $setlib:colDecorData//conceptMap/sourceScope[@ref = $id]
    let $conceptMapsAllSource   := if ($isMostRecent) then $conceptMapsAllSource[@flexibility = $effectiveDate] | $conceptMapsAllSource[not(@flexibility castable as xs:dateTime)] else $conceptMapsAllSource[@flexibility = $effectiveDate]
    
    let $conceptMapsAllTarget   := $setlib:colDecorData//conceptMap/targetScope[@ref = $id]
    let $conceptMapsAllTarget   := if ($isMostRecent) then $conceptMapsAllTarget[@flexibility = $effectiveDate] | $conceptMapsAllTarget[not(@flexibility castable as xs:dateTime)] else $conceptMapsAllTarget[@flexibility = $effectiveDate]
        
    let $allTemplAssociations   := $setlib:colDecorData//template//vocabulary[@valueSet = $id]
    let $allTemplAssociations   := if ($isMostRecent) then $allTemplAssociations[@flexibility = $effectiveDate] | $allTemplAssociations[not(@flexibility castable as xs:dateTime)] else $allTemplAssociations[@flexibility = $effectiveDate]
    
    let $url                    := $utillib:strDecorServicesURL
    
    return
    (
        for $item in $allAssociations
        let $clpfx              := $item/ancestor::decor/project/@prefix
        let $clid               := $item/@conceptId
        group by $clpfx, $clid
        return (
            let $originalConcept:= $setlib:colDecorData//conceptList[@id = $id][ancestor::datasets]/ancestor::concept[1]
            return if ($originalConcept) then utillib:doConceptAssociation($vs[1], $originalConcept, $originalConcept/name) else ()
        )
        ,
        for $item in $valueSetsAll
        let $vsid               := $item/ancestor::valueSet[1]/@id
        let $vsed               := $item/ancestor::valueSet[1]/@effectiveDate
        group by $vsid, $vsed
        return utillib:doValueSetAssociation($vs, $item[1])
        ,
        let $decorMap           := map:merge(for $p in distinct-values($vs/ancestor-or-self::*/@ident) return map:entry($p, utillib:getDecorByPrefix($p)))
        for $item in $valueSetExpansionsAll
        return utillib:doValueSetExpansionAssociation($vs, $item[1], try { map:get($decorMap, ($vs/ancestor-or-self::*/@ident)[1]) } catch * {()})
        ,
        for $item in $conceptMapsAllSource | $conceptMapsAllTarget
        let $type               := local-name($item)
        let $vsid               := $item/ancestor::conceptMap[1]/@id
        let $vsed               := $item/ancestor::conceptMap[1]/@effectiveDate
        (: group by type of association and then by valueSet. This 'may' list a 
        single conceptMap twice if valueSet is on source and target :)
        group by $type, $vsid, $vsed
        return utillib:doConceptMapAssociation($vs, $item[1], 
            for $l in distinct-values($item[1]/ancestor::decor/project/name/@language) 
            return <name language="{$l}">{data($item[1]/ancestor::conceptMap/@displayName)}</name>)
        ,
        for $item in $allTemplAssociations
        let $tmid               := $item/ancestor::template[1]/@id
        let $tmed               := $item/ancestor::template[1]/@effectiveDate
        group by $tmid, $tmed
        return utillib:doTemplateAssociation($vs, $item[1])
    )

};

declare %private function utilvs:patchValueSetParameters($parameters as element(parameters), $valueSet as element(valueSet)*) {

    for $param in $parameters/parameter
    return
        switch ($param/@path)
        case '/statusCode'
        case '/expirationDate'
        case '/officialReleaseDate'
        case '/canonicalUri'
        case '/versionLabel'
        case '/name'
        case '/displayName'
        case '/experimental' return (
            let $attname  := substring-after($param/@path, '/')
            let $new      := attribute {$attname} {$param/@value}
            let $stored   := $valueSet/@*[name() = $attname]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $valueSet
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/desc' return (
            (: only one per language :)
            let $elmname  := substring-after($param/@path, '/')
            let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/desc)
            let $stored   := $valueSet/*[name() = $elmname][@language = $param/value/desc/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $valueSet
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/purpose' return (
            (: only one possible per language :)
            let $elmname  := substring-after($param/@path, '/')
            let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/purpose)
            let $stored   := $valueSet/*[name() = $elmname][@language = $param/value/purpose/@language]
            
            return
            switch ($param/@op)
            case 'add' (: fall through :)
            case 'replace' return if ($stored) then update replace $stored with $new else update insert $new into $valueSet
            case 'remove' return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/publishingAuthority' return (
            (: only one possible :)
            let $new      := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
            let $stored   := $valueSet/publishingAuthority
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $valueSet
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/copyright' return (
            (: only one per language :)
            let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/copyright)
            let $stored   := $valueSet/copyright[@language = $param/value/copyright/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $valueSet
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
    
        case '/items' return (
            let $newcl    := utilvs:prepareValueSetConceptListForUpdate($param/value/items)
            let $stored   := $valueSet/conceptList | $valueSet/completeCodeSystem
            let $delete   := update delete $stored
            
            return
                switch ($param/@op)
                case 'add'
                case 'replace' return update insert $newcl into $valueSet
                case 'remove' return ()
                default return ( (: unknown op :) )
        )
        default return ( (: unknown path :) )
};