xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utiluser               = "http://art-decor.org/ns/api/util-user";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

(:~ uri that the user full name is under in eXist-db :)
declare variable $utiluser:uriPropFullName      := xs:anyURI('http://axschema.org/namePerson');

(:~
:   Return display name for the given username, usually his full name. If this username is not equal to the currently logged in user, the 
:   currently logged in user needs to be part of a group with permissions. If he is not, an error is returned
:   
:   @param $username The username to get the info for
:   @return The configured display name e.g. 'John Doe', null or error()
:   @since 2013-11-07
:)
declare function utiluser:getUserDisplayName($username as xs:string) as xs:string? {
    try {
        let $displayName    := if (sm:user-exists($username)) then sm:get-account-metadata($username, $utiluser:uriPropFullName) else ()
        
        return
            if (empty($displayName)) then error() else ($displayName)
    }    
    catch * {
        (utiluser:getUserInfo($username)/displayName[string-length() gt 0], $username)[1]
    }
};

(:~
:   Return full userInfo for the given username. If this username is not equal to the currently logged in user, the 
:   currently logged in user needs to be part of a group with permissions. If he is not, an error is returned
:   
:   @param $username The username to get the info for
:   @return The configured user info e.g. 
:       <user name="john">
:           <defaultLanguage/>
:           <displayName/>
:           <description/>
:           <email/>
:           <organization/>
:           <logins/>
:           <lastissuenotify/>
:       </user>, null or error()
:   @since 2013-11-07
:)
declare function utiluser:getUserInfo($username as xs:string) as element(user)? {
    let $userInfo   := collection($setlib:strUserInfo)//user[@name = $username]
    
    return
        if (count($userInfo) gt 1) then
            error($errors:SERVER_ERROR, 'User info file contains multiple mentions of user ' || $username || '. Alert your administrator as this should not be possible.')
        else (
            $userInfo
        )
};