xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Template API allows read, create, update of DECOR templates :)
module namespace prextapi           = "http://art-decor.org/ns/api/project-extract";

import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace utillib     = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace complib     = "http://art-decor.org/ns/api/compile" at "library/compile-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

declare namespace xmldb             = "http://exist-db.org/xquery/xmldb";
declare namespace util              = "http://exist-db.org/xquery/util";
declare namespace json              = "http://www.json.org";
declare namespace output            = "http://www.w3.org/2010/xslt-xquery-serialization";
declare namespace sch               = "http://purl.oclc.org/dsdl/schematron";
declare namespace sm                = "http://exist-db.org/xquery/securitymanager";
declare namespace svrl              = "http://purl.oclc.org/dsdl/svrl";
declare namespace xs                = "http://www.w3.org/2001/XMLSchema";
declare namespace xsi               = "http://www.w3.org/2001/XMLSchema-instance";
declare namespace xsl               = "http://www.w3.org/1999/XSL/Transform";

declare variable $prextapi:VERBATIM     := 'verbatim';
declare variable $prextapi:COMPILED     := 'compiled';
declare variable $prextapi:FILTER         := 'finalfilter';

(:~ Retrieves DECOR project for check and publication purpose based on $project (prefix or oid) denoting its version
    @param $project                 - required, limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $format                  - optional, overrides the accept-header for frontend purposes
                                        Default: xml
    @param $download                - optional as xs:boolean. 
                                        Default: false. 
    @param $mode                    - required parameter for the transformation of the project, possible values:
                                       'verbatim' gets the project as-is (for a release the only possible choice)
                                       'compiled' tries to make this project self-contained by getting referenced content from other projects/repositories into the current project
                                       'test' retrieves the filters that would be applied when compiled
                                        Default: verbatim 
    Next parameters are only suitable for the current version
    $param $runtimeonly             - optional as xs:boolean, only relevant for 'compiled' mode and the current version.
                                        Returns a compile limited to scenarios, terminology and templates which is enough for creating a runtime enviroment
                                        Default: false.
    $param $forceRecompile          - optional as xs:boolean, only relevant for 'compiled' mode. 
                                        Forces a new (limited runtimeonly) compile if no one exist. 
                                        Default: false.
    $param $filterId                - optional, only relevant if not 'verbatim' mode 
                                        Use a defined filter for the extraction, if not given do a full extraction.
    @return as-is or as compiled as XML
    @since 2025-03-03
:)

declare function prextapi:getProjectExtract($request as map(*)) {

    let $project                        := $request?parameters?project[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $format                         := $request?parameters?format[not(. = '')]
    let $download                       := $request?parameters?download=true()
    let $mode                           := $request?parameters?mode[not(. = '')]
    (: only needed for the current version :)
    let $runtimeOnly                    := $request?parameters?runtimeOnly=true()
    let $forceRecompile                 := $request?parameters?forceRecompile=true()
    let $filterId                       := $request?parameters?filterId
    
    (: CHECK PARAMETERS :)
    (: parameter format overrides accept-header as mime-type for using the api in the browser :)
    
    (: 
        this call is used in backend and frontend (browser) - default behaviour mime-types:
        - backend:   default mime-type is json - response payload is json - if no accept type is given
        - frontend:  default mime-type is xml - response payload is xml - if no format is given
    :)
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format                         := if ($format) then $format else tokenize($acceptedType, '/')[2]

    (: 
       overwrite format in the backend is not always possible because of middleware behaviour
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
    :) 
    let $check                          :=
        if ($format = 'xml' and not($acceptedType = 'application/xml')) then 
            error($errors:BAD_REQUEST, 'In case of format parameter is xml the accept header should be application/xml')
        else ()   

    (: check project :)
    let $decor                  := utillib:getDecor($project, $projectVersion, $projectLanguage)
    
    let $check                  :=
        if (empty($decor)) then
            error($errors:BAD_REQUEST, 'Parameter project ' || $project || ' does not lead to a DECOR project.')
        else
        if (count($decor) gt 1 and empty($projectVersion)) then
            error($errors:SERVER_ERROR, 'Parameter project ' || $project || ' leads to multiple (' || count($decor) || ') DECOR projects. Contact your system administrator.')
        else ()
    
    (: TODO: do we just overrule the mode and filterId or are we checking it when a release is given :)
    let $mode                   := if ($projectVersion) then $prextapi:VERBATIM else $mode
    let $filterId               := if ($projectVersion) then () else $filterId

    let $check                  :=
        if ($projectVersion and ($mode = not($prextapi:VERBATIM))) then
            error($errors:BAD_REQUEST, 'Parameter mode and filterId should be empty in case of retrieving release ' || $projectVersion || ' for for project ' || $project ||'. The project  is already compiled.')
        else ()    
    
    (: check filterId :)
    let $filterSet              := if ($filterId) then complib:getCompilationFilterSet($decor)[@filterId = $filterId] else ()    
    
    let $check                  := 
        if ($filterId) then (
            if ($filterSet) then () else
                error($errors:BAD_REQUEST, 'Parameter filterId ' || $filterId || ' id not found for project '|| $project || '.')
        )
        else ()
  
    (: GET RESULTS :)
    let $now                    := substring(string(current-dateTime()), 1, 19)
    let $language               := if ($decor/project/name[@language = $projectLanguage] or $projectLanguage = '*') then $projectLanguage else ($decor/project/@defaultLanguage, utillib:getServerLanguage())[1]
    let $results                := prextapi:getProjectExtract($decor, $now, $language, $mode, $filterSet, $runtimeOnly, $forceRecompile)
    
    let $results                := 
        if ($format = 'xml') then 
        (
            if (not($mode = $prextapi:FILTER)) then processing-instruction {'xml-model'} {' href="https://assets.art-decor.org/ADAR/rv/DECOR.sch" type="application/xml" schematypens="http://purl.oclc.org/dsdl/schematron"'} else (),
            if ($download) then processing-instruction {'xml-stylesheet'} {' type="text/xsl" href="https://assets.art-decor.org/ADAR/rv/DECOR2schematron.xsl"'} else (),
            $results
        )    
        else
        (
            let $results                :=
                for $result in $results
                    return
                    element {name($result)} {
                        $result/@*,
                        namespace {"json"} {"http://www.json.org"},
                        utillib:addJsonArrayToElements($result/*) 
                        }
                (: in this case the json format overrides header application/xml and works with a text header :)
                return if (not($acceptedType = 'application/json')) then fn:serialize($results, map{"method": $format , "indent": true()}) else $results
        )
    
    (: RETURN RESULTS :)
    let $projectPrefix          := $decor/project/@prefix
    
    let $filename               := 
        if ($projectVersion) then util:document-name($decor[@language = $language])
        else if ($mode = $prextapi:VERBATIM) then $projectPrefix || replace($now,'[:\-]','') || '-decor.xml'
        else $projectPrefix || replace($now,'[:\-]','') || '-' || (if ($language = '*') then 'all' else $language[1]) || '-decor-compiled.xml'
    
    let $r-header                       := 
        (response:set-header('Content-Type', 'application/' || $format || '; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
    return
        if (empty($results)) then roaster:response(404, ()) 
        else  $results
};

declare function prextapi:getProjectExtract($decor as element(decor), $now as xs:dateTime, $language as xs:string, $mode as xs:string, $filters as element(filters)?, $runtimeOnly as xs:boolean?, $forceRecompile as xs:boolean?) as element()? {

    let $filters                :=
        if ($filters) then
            <filters>
            {
                $filters/(@* except @filter),
                attribute filter {'on'},
                $filters/node()
            }
            </filters>
        else ()
    
    return
        switch ($mode)
        case $prextapi:COMPILED return complib:getCompiledResults($decor, $now, $language, complib:getFinalCompilationFilters($decor, $filters), $runtimeOnly, $forceRecompile)/descendant-or-self::decor[1]
        case $prextapi:FILTER return complib:getFinalCompilationFilters($decor, $filters)
        
        (: default return project as-is in case of 
           - mode verbatim for current version gives as-is and 
           - no matter the mode - projectVersion gives always as-is 
        :)
        default return if ($decor[@compilationDate]) then $decor[@language = $language][1] else $decor[1]
};
