xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilfsh                = "http://art-decor.org/ns/api/util-fsh";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace markdown        = "http://art-decor.org/ns/api/markdown" at "markdown-lib.xqm";

declare namespace util                  = "http://exist-db.org/xquery/util";

declare %private variable $utilfsh:inactiveStatusCodes := ('cancelled','rejected','deprecated');
declare %private variable $utilfsh:ucum                := if (doc-available(concat($setlib:strDecorCore, '/DECOR-ucum.xml'))) then doc(concat($setlib:strDecorCore, '/DECOR-ucum.xml'))/*/ucum else ();
declare %private variable $utilfsh:CR                  := "&#10;";

(:
    MAIN ENTRY functions
    ====================
:)

(:
    Build the FSH representation for dataset or transaction
:)
declare function utilfsh:convertTransactionOrDataset2Fsh(
    $fullDatasetTree as node(),
    $language as xs:string?,
    $fileNameforDownload as xs:string?
    (:
    $ui-lang as xs:string?,
    $hidecolumns as xs:string?, 
    $showonlyactiveconcepts as xs:boolean, 
    $collapsed as xs:boolean, 
    $draggable as xs:boolean, 
    $version as xs:string?,
    $url as xs:anyURI?,
    $projectPrefix as xs:string,
    $templateChain as element()*,
    $doSubConcept as xs:boolean,
    $download as xs:boolean,
    $header as xs:boolean
    :)
    ) as node() {
    
    let $isTransaction       := exists($fullDatasetTree/@transactionId)
    let $decorId             := if ($isTransaction) then $fullDatasetTree/@transactionId else $fullDatasetTree/@id
    let $decorEffectiveDate  := if ($isTransaction) then $fullDatasetTree/@transactionEffectiveDate else $fullDatasetTree/@effectiveDate
    let $decorStatusCode     := if ($isTransaction) then $fullDatasetTree/@transactionStatusCode else $fullDatasetTree/@statusCode
    let $decorExpirationDate := if ($isTransaction) then $fullDatasetTree/@transactionExpirationDate else $fullDatasetTree/@expirationDate
    let $decorVersionLabel   := if ($isTransaction) then $fullDatasetTree/@transactionVersionLabel else $fullDatasetTree/@versionLabel
    let $fhirId              := concat($decorId,'--',replace($decorEffectiveDate,'[^\d]',''))
        
    let $maps                := 
        for $codeSystem in $fullDatasetTree//@codeSystem 
        let $csid := $codeSystem
        let $csed := $codeSystem/../@codeSystemVersion
        group by $csid, $csed
        return 
            map:entry($csid || $csed, utillib:getCanonicalUriForOID('CodeSystem', $csid, $csed, $fullDatasetTree/@prefix, $setlib:strKeyFHIRR4))
    let $oidfhirmap          := map:merge($maps)
    
    let $language            := if (empty($language)) then $fullDatasetTree/name/@language[1] else $language
    let $displayTitle        := $fullDatasetTree/name[@language=$language][1]/string() || ' ' || $fullDatasetTree/@versionLabel
    
    let $name                := 
        if ($isTransaction and $fullDatasetTree/*:name[@language = $language][node()][. = 'Registration']) then
            ($fullDatasetTree/*:name[@language = $language], $fullDatasetTree/*:name)[node()][1]
        else (
            ($fullDatasetTree/*:name[@language = $language], $fullDatasetTree/*:name)[.//text()[string-length()>0]][1]
        )
    let $desc                := 
        if ($isTransaction and $fullDatasetTree/*:desc[@language = $language][node()][. = '-']) then
            ($fullDatasetTree/*:desc[@language = $language], $fullDatasetTree/*:desc)[node()][1]
        else (
            ($fullDatasetTree/*:desc[@language = $language], $fullDatasetTree/*:desc)[.//text()[string-length()>0]][1]
        )

    let $desc   := if (empty($desc)) then '-' else markdown:html2markdown(utillib:parseNode($desc)/node())
    
    (: fullDatasetTree contains shortName based on project default language, while we want a language that matches the $language which may be different :)
    let $shortname           := utillib:shortName($name)
    let $shortname           := concat(upper-case(substring($shortname, 1, 1)), substring($shortname, 2))
    
    (: determine id, if canonicalUri is given use basename of it, otherwise the calculated FHIR id :)
    let $url                 :=
        if ($fullDatasetTree[@transactionCanonicalUri]) then $fullDatasetTree/@transactionCanonicalUri else 
        if ($fullDatasetTree[@canonicalUri]) then $fullDatasetTree/@canonicalUri
        else ()
    let $shortname4id        := 
        if (empty($url)) then $fhirId
        else tokenize($url, '/')[last()]
        
    let $rootPath            := if ($fullDatasetTree/*:concept[2]) then lower-case(replace($shortname, '_', '-')) else () (: not used :)
    
    let $prefix              := ($fullDatasetTree/@prefix | $fullDatasetTree/ancestor::decor/project/@prefix)[1]

    (: init array :)
    let $fsh := []
    
    (: 
        header
        ======
    :)
    let $fsh := array:append($fsh, "// -------------------------------------------------------------------------------")
    let $fsh := array:append($fsh, "// Logical Model " || $displayTitle)
    let $aif := if ($isTransaction) then "Transaction" else "Dataset"
    let $fsh := array:append($fsh, "// derived from ART-DECOR " || $aif || " " || $decorId || " as of " || $decorEffectiveDate)
    let $fsh := if (string-length($fileNameforDownload) > 0) then array:append($fsh, "// Filename " || $fileNameforDownload) else $fsh
    let $fsh := array:append($fsh, "// -------------------------------------------------------------------------------")
    let $fsh := array:append($fsh, "Logical:     " || $shortname)
    let $fsh := array:append($fsh, "Parent:      " || "Element")
    let $fsh := array:append($fsh, "Id:          " || $shortname4id)
    let $fsh := array:append($fsh, "Title:       " || '"' || $displayTitle || '"')
    let $fsh := array:append($fsh, "Description: " || '"""' || normalize-space($desc) || '"""')
    
    let $fsh := array:append($fsh, "")
    
    (: handle multilingual name / desc :)
    let $fsh := utilfsh:addMultilingual($fsh, "* ^name" || $utilfsh:CR || "  * ", $fullDatasetTree/*:name[not(@language = $language)])
    let $fsh := utilfsh:addMultilingual($fsh, "* ^desc" || $utilfsh:CR || "  * ", $fullDatasetTree/*:desc[not(@language = $language)])
    
    let $fsh := array:append($fsh, "* ^language = #" || $language)
    let $fsh := array:append($fsh, "* ^status = #" || utilfsh:decorStatus2fhirStatus($decorStatusCode))
    (: version :)
    let $semver := utilfsh:getSemverString($decorVersionLabel)
    let $fsh    := array:append($fsh, "* ^version = " || '"' || (if (empty($semver)) then $decorEffectiveDate else $semver) || '"')    
    (: url :)
    let $fsh := if ($url) then array:append($fsh, "* ^url = " || '"' || $url || '"') else $fsh
    
    let $fsh := utilfsh:addFSHmetadates($fsh, "* ", $fullDatasetTree/@effectiveDate, $fullDatasetTree/@expirationDate)
    let $fsh := utilfsh:addFSHidentifierOid($fsh, "* ", $decorId)
    let $fsh := utilfsh:addFSHpublishingAuthorityContactCopyright($fsh, "* ", $fullDatasetTree/publishingAuthority, $fullDatasetTree/copyright)
    let $fsh := utilfsh:addPurpose($fsh, "* ", $fullDatasetTree/purpose, $language)
    
    let $fsh := array:append($fsh, "")
    
    (:
        concepts
        ========
    :)    
    let $cfsh := []
    (: check for root element :)
    (: not used
    let $cfsh := 
        if (empty($rootPath)) then $cfsh
        else array:append($cfsh, "* " || $rootPath || " 1..1 BackboneElement " || '"' || $displayTitle || '"&#10;')
    :)
    (: go through concepts :)
    let $cfsh := array:append($cfsh,
        for $concept in $fullDatasetTree/*:concept
        return utilfsh:concept2fsh ($concept, $rootPath, $prefix, $language, $isTransaction, $oidfhirmap)
        )
                
    return 
        <fsh>
        {
            concat(string-join(data($fsh), '&#10;'), '&#10;', string-join($cfsh, ''), '&#10;')
        }
        </fsh>
};


declare function utilfsh:convertValueSets2Fsh(
    $valueSets as node(),
    $language as xs:string?,
    $projectPrefix as xs:string
    ) as node() {
    
    let $maps                 := 
        for $codeSystem in $valueSets//@codeSystem
        let $csid := $codeSystem
        let $csed := $codeSystem/../@codeSystemVersion
        group by $csid, $csed
        return 
            map:entry($csid || $csed, utillib:getCanonicalUriForOID('CodeSystem', $csid, $csed, $projectPrefix, $setlib:strKeyFHIRR4))
    let $csmaps     := map:merge($maps)
    
    (: go through list of value sets :)
    let $fsh := []
    let $fsh := array:append($fsh,
        for $valueSet in $valueSets//valueSet[@id]
        return utilfsh:convertValueSet2Fsh($valueSet, $language, $projectPrefix, $csmaps)
    )
        
    return
        <fsh>
        {
            concat(string-join(data($fsh), '&#10;'), '&#10;&#10;&#10;&#10;')
        }
        </fsh>
};

declare function utilfsh:convertValueSet2Fsh(
    $valueSet as element(valueSet),
    $language as xs:string?,
    $projectPrefix as xs:string,
    $oidfhirmap as map(*)?
    ) as node() {

    let $vsmap               := map:entry($valueSet/@id, utillib:getCanonicalUriForOID('ValueSet', $valueSet, $projectPrefix, $setlib:strKeyFHIRR4))
    let $oidfhirmap          := map:merge(($oidfhirmap, $vsmap))
    
    let $decorId             := $valueSet/@id
    let $decorEffectiveDate  := $valueSet/@effectiveDate
    let $decorStatusCode     := $valueSet/@statusCode
    let $decorExpirationDate := $valueSet/@expirationDate
    let $decorVersionLabel   := $valueSet/@versionLabel
    
    let $fhirId              := concat($decorId,'--',replace($decorEffectiveDate,'[^\d]',''))
    
    let $language            := if (empty($language)) then $valueSet/name/@language[1] else $language
    
    let $displayTitle        := $valueSet/@displayName || ' ' || $valueSet/@versionLabel
    
    let $name                := $valueSet/@displayName
    
    let $desc                :=  ($valueSet/*:desc[@language = $language], $valueSet/*:desc)[.//text()[string-length()>0]][1]
    let $desc                := if (empty($desc)) then () else markdown:html2markdown(utillib:parseNode($desc)/node())
    
    let $shortname           := utillib:shortName($name)
    let $shortname           := concat(upper-case(substring($shortname, 1, 1)), substring($shortname, 2))
    
    (: determine id, if canonicalUri is given use basename of it, otherwise the calculated FHIR id :)
    let $url                 := if ($valueSet/@canonicalUri) then $valueSet/@canonicalUri else ()
    
    let $shortname4id        := 
        if (empty($url)) then $fhirId
        else tokenize($url, '/')[last()]
        
    (: init array :)
    let $fsh := []
    
    (: 
        header
        ======
    :)
    let $fsh := array:append($fsh, "// -------------------------------------------------------------------------------")
    let $fsh := array:append($fsh, "// ValueSet " || $displayTitle)
    let $aif := "ValueSet"
    let $fsh := array:append($fsh, "// derived from ART-DECOR " || $aif || " " || $decorId || " as of " || $decorEffectiveDate)
    (:
    let $fsh := if (string-length($fileNameforDownload) > 0) then array:append($fsh, "// Filename " || $fileNameforDownload) else $fsh
    :)
    let $fsh := array:append($fsh, "// -------------------------------------------------------------------------------")
    let $fsh := array:append($fsh, "ValueSet:    " || $shortname)
    let $fsh := array:append($fsh, "Id:          " || $shortname4id)
    let $fsh := array:append($fsh, "Title:       " || '"' || $displayTitle || '"')
    let $fsh := if (empty($desc)) then $fsh else array:append($fsh, "Description: " || '"""' || normalize-space($desc) || '"""')
    
    let $fsh := array:append($fsh, "")
    
    let $fsh := array:append($fsh, "* ^language = #" || $language)
    let $fsh := array:append($fsh, "* ^status = #" || utilfsh:decorStatus2fhirStatus($decorStatusCode))
    (: version :)
    let $semver := utilfsh:getSemverString($decorVersionLabel)
    let $fsh    := array:append($fsh, "* ^version = " || '"' || (if (empty($semver)) then $decorEffectiveDate else $semver) || '"')
    (: date :)
    let $fsh    := if ($valueSet[@lastModifiedDate castable as xs:dateTime]) then
        array:append($fsh, "* ^date = " || '"' || adjust-dateTime-to-timezone(xs:dateTime($valueSet/@lastModifiedDate)) || '"') else $fsh
    (: url :)
    let $fsh := if ($url) then array:append($fsh, "* ^url = " || '"' || $url || '"') else $fsh
    (: experimental :)
    let $fsh := if ($valueSet/@experimental = 'true') then array:append($fsh, "* ^experimental = true") else array:append($fsh, "* ^experimental = false")
        
    let $fsh := utilfsh:addFSHmetadates($fsh, "* ", $valueSet/@effectiveDate, $valueSet/@expirationDate)
    let $fsh := utilfsh:addFSHidentifierOid($fsh, "* ", $decorId)
    let $fsh := utilfsh:addFSHpublishingAuthorityContactCopyright($fsh, "* ", $valueSet/publishingAuthority, $valueSet/copyright)
    let $fsh := utilfsh:addPurpose($fsh, "* ", $valueSet/purpose, $language)

    let $fsh := array:append($fsh, "")
    
    (: run through valueset content :)
    
    (: skip abstract concepts. FHIR ValueSet does not support those :)
    (: https://jira.hl7.org/browse/FHIR-17277 :)
    let $conceptList            := 
        for $c in ($valueSet/*:conceptList/*:concept | $valueSet/*:conceptList/*:exception)[not(@type = 'A')]
        return $c

    (: note that this boolean signifies expanding inactive codes. we do not formally know if the Deprecated concept
       in our ValueSet definition is actually deprecated in the CodeSystem or just in the ValueSet. Hence we cannot say 
       for sure that inactive=true. However ... deprecation in a value set is more likely to occur based on deprecation in
       the CodeSystem than just the ValueSet and for all sense and purposes: the implementer probably would not care 
       about the diferrence, so we signal inactive=true ... if only because the deprecated extension does not support
       expand as context so we can only mark inactive=true there and the distinction is gone then anyway
    :)
    let $fsh        := if ($conceptList[@type = 'D']) then array:append($fsh,  "* ^inactive = true") else $fsh
        
    let $fsh        := array:append($fsh,
        for $clc in $valueSet/conceptList/concept
        return
            "* " || map:get($oidfhirmap, $clc/@codeSystem || $clc/@codeSystemVersion) || "#" || $clc/@code ||
             (if ($clc/@displayName) then concat(' "', $clc/@displayName, '"') else ())
    )
    
    return
        <fsh>
        {
            concat(string-join(data($fsh), '&#10;'), '&#10;&#10;')
        }
        </fsh>
};
    
(:
    PRIVATE functions
    =================
:)

declare %private function utilfsh:concept2fsh (
    $concept as element(concept),
    $parentPath as xs:string?,
    $prefix as xs:string?,
    $language as xs:string,
    $isTransaction as xs:boolean,
    $oidfhirmap as map(*)
    ) as xs:string {
    
    let $originalConcept    := utillib:getOriginalForConcept($concept)

    let $name               := 
        if ($originalConcept/*:name[@language = $language]) then 
            $originalConcept/*:name[@language = $language][1]
        else 
            $originalConcept/*:name[node()][1]
    let $desctmp            := 
        if ($originalConcept/*:desc[@language = $language]) then 
            $originalConcept/*:desc[@language = $language][1]
        else 
            $originalConcept/*:desc[node()][1]
    
    let $shortName          := utilfsh:fhirName($name)
    let $id                 := 
        if ($concept[@id]) then
            $concept/concat(@id,'--', replace(@effectiveDate,'[^\d]',''))
        else
            $concept/concat(@ref,'--', replace(@flexibility,'[^\d]',''))
    let $shortId            := tokenize(tokenize($id, '\.')[last()], '-')[1]
            
    let $name               := $name || " (" || $shortId || ")"
    let $longId             :=  "ART-DECOR ID: " || $id
    let $mddesc             := markdown:html2markdown(markdown:minifyHTML(utillib:parseNode($desctmp)/node()))
    let $desc               := 
        if (string-length($mddesc) = 0) then $longId
        else $mddesc || "&#10;" || $longId
    
    let $path               := string-join(($parentPath, $shortName), '.')
    let $minCard            := utillib:getMinimumMultiplicity($concept)
    let $maxCard            := utillib:getMaximumMultiplicity($concept)
    let $isRequired         := $concept[@conformance='R']
    let $isMandatory        := $concept[@isMandatory='true']
        
    let $concept            :=
        if ($concept[count(*:concept) = 1]/*:concept/*:contains) then (
            let $actualConceptName          := 
                if ($concept/*:concept/*:name[@language = $language]) then 
                    $concept/*:concept/*:name[@language = $language][1]
                else 
                    $concept/*:concept/*:name[node()][1]
            (: fullDatasetTree contains shortName based on project default language, while we want a language that matches the $language which may be different :)
            let $actualShortName            := 
                (:if (matches($actualConceptName, $adfhirsd:ncnamePattern)) then $actualConceptName else:)
                (:if ($concept/*:concept[@shortName]) then $concept/*:concept/@shortName else adfhir:shortName($actualConceptName):)
                utilfsh:fhirName($actualConceptName)
            
            return
                if ($shortName = $actualShortName) then $concept/*:concept else $concept
        )
        else ($concept)
     
    (: presence of @shortName is an indicator of fullDatasetTree instead of dataset :)
    let $assocs             := 
        if ($concept[@shortName]) then
            $concept/*:terminologyAssociation | 
            $concept/*:valueDomain//*:terminologyAssociation
        else
            utillib:getConceptAssociations($concept, $originalConcept, false())/*
            
    (: utillib:getGetEnhancedValueset does the heavy lifting, but could produce an empty valueSet element, so check that before relying on it. :)
    let $valueSets          := 
        if ($concept[*:valueSet[@id | @ref]]) then
            $concept/*:valueSet
        else (
            for $assoc in $assocs[@valueSet]
            return utillib:getValueSetById($assoc/@valueSet, if ($assoc/@flexibility) then $assoc/@flexibility else 'dynamic')//*:valueSet[@id]
        )
        
    let $valueDomain        := if ($originalConcept/*:valueDomain[2]) then () else $originalConcept/*:valueDomain[1]
    let $dataType           := if ($originalConcept/@type='group') then 'BackboneElement' else utilfsh:decorValueDomainType2FHIRtype($valueDomain/@type)
    let $properties         := $valueDomain/*:property[@unit]
    
    (: if only one value set binding is specified emit it here (without slicing) - can probably be done in one run in fsh :)
    let $bindings           :=
        if ($valueSets[2]) then () else (
            for $valueSet in $valueSets
            let $strength   := utilfsh:decorBindingStrength2fhirBindingStrength($assocs[@valueSet = $valueSet/@id][1]/@strength)
            return
                utilfsh:decorVocabulary2fhirBinding($valueSet, $prefix, $strength, $path)
         )
         
     let $punits             :=
        if ($dataType = ('Quantity', 'Duration', 'Count') and count($properties) = 1) then 
            (: elementdefinition-allowedUnits has a required binding to UCUM so if @unit is not a UCUM unit, then alas, no extension :)
            (: TODO
            for $property in $properties
            let $isUcumUnit := $getf:CS_UCUMCOMMONUNITS[@unit = $property/@unit][@message = 'OK']
            return
                if ($isUcumUnit) then
                    <EXTENSIONTODO url="http://hl7.org/fhir/StructureDefinition/elementdefinition-allowedUnits">
                        <valueCodeableConcept>
                            <coding>
                                <system value="http://unitsofmeasure.org/" />
                                <code value="{$property/@unit}" />
                            </coding>
                        </valueCodeableConcept>
                    </EXTENSIONTODO>
                else ()
            :)
         () else ()
     
     let $conditionText      := 'One concept must be selected'
     let $conditions         :=
        for $conditions at $i in $concept/../*:concept/*:condition[*:desc[starts-with(., $conditionText)]]
        let $grp             := $conditions/*:desc[starts-with(., $conditionText)]
        group by $grp
        return
            "&#10;  * ^condition[+] = " || concat(($conditions[1]/../@iddisplay, 'cond')[1], '-', $i[1])
            
     let $requirements       :=
        if ($originalConcept[*:rationale[@language = $language][node()]]) then
           "&#10;  * ^requirements = " || '"""' || markdown:html2markdown(utillib:parseNode($originalConcept/*:rationale[@language = $language][1])/node()) || '"""'
        else ()
     
     let $examples           :=
        utilfsh:decorExample2FHIRexample($concept, $valueDomain/*:example, $dataType, $properties, $oidfhirmap)
    
     (: http://hl7.org/fhir/R4/structuredefinition.html#invs sdf-9 disallows this on base :)
     (: |  * ^code[+] = http://snomed.info/sct#424144002 "Current chronological age (observable entity)" :)
     let $cassocs             :=
        if (contains($path, '.')) then
            for $assoc in $assocs[@conceptId=$concept/@id][@code][empty(@expirationDate)] | $assocs[@conceptId=$originalConcept/@id][@code][empty(@expirationDate)]
            let $adisplay := if ($assoc/@displayName) then concat(' "', $assoc/@displayName, '"') else ""
            return
                "&#10;  * ^code[+] = " || utillib:getCanonicalUriForOID('CodeSystem', $assoc/@codeSystem, $assoc/@codeSystemVersion, $prefix, $setlib:strKeyFHIRR4) ||
                "#" || $assoc/@code || $adisplay
        else ()
        
    let $synonyms            :=
        for $synonym in $originalConcept/*:synonym[text()]
        return
            "&#10;  * ^alias[+] = " || '"' || $synonym || '"'
     
     (:
        one FSH line, must have a valid element name (shortname)
        |* dateOfArrivalAtCenter 1..* dateTime "Date of arrival to our institution" """Date of arrival to our institution"""				
     :)
     let $fsh1 :=
        if (string-length($shortName) > 0) then (
            (: line :)
            "* " || $path || " " || $minCard || ".." || $maxCard || " " || $dataType || ' "' || $name || '"',
            if (string-length($desc) > 0) then ' """' || $desc || '"""' else (),
            (: lines :)
            $punits,
            $cassocs,
            $synonyms,
            $bindings,
            $conditions,
            $requirements,
            $examples
        ) else ()
        
     (: hush through sub concepts :)
     let $fsh2 :=
        if ($isTransaction) then (
               for $subconcept in $concept/*:concept
               return 
                   utilfsh:concept2fsh($subconcept, $path, $prefix, $language, $isTransaction, $oidfhirmap)
           )
           else (
               for $subconcept in $concept/*:concept
               return (
                   if ($subconcept[@statusCode = $utilfsh:inactiveStatusCodes]) then () else
                       utilfsh:concept2fsh($subconcept, $path, $prefix, $language, $isTransaction, $oidfhirmap)
               )
           )
           
      return concat(string-join($fsh1, ''), '&#10;', string-join(data($fsh2), ''))
};

declare function utilfsh:decorValueDomainType2FHIRtype($decorType as item()?) as xs:string {
    switch ($decorType)
    case 'count'        return 'Count'      (:could have chosen integer of decimal. not sure why to pick what :)
    case 'code'         return 'CodeableConcept'
    case 'ordinal'      return 'CodeableConcept'
    case 'identifier'   return 'Identifier'
    case 'string'       return 'string'
    case 'text'         return 'markdown'
    case 'date'         return 'date'       (:need work with properties:)
    case 'datetime'     return 'dateTime'   (:need work with properties:)
    case 'time'         return 'time'       (:need work with properties:)
    case 'complex'      return 'string'     (:this is a problem. this is lazy dataset behavior that is unimplementable:)
    case 'quantity'     return 'Quantity'
    case 'duration'     return 'Duration'
    case 'boolean'      return 'boolean'
    case 'blob'         return 'base64Binary'
    case 'decimal'      return 'decimal'
    default             return 'string'
};

(:~ Returns a FHIR "machine processing" name, 
    start with an upper-case ASCII letter ('A'..'Z') followed
    by any combination of upper- or lower-case ASCII letters
    ('A'..'Z', and 'a'..'z'), numerals ('0'..'9') and '_',
    with a length limit of 255 characters.
    
    Most common diacritics are replaced
    
    Input:  xs:string, example: "UnderScored Lowercase ë"
    Output: xs:string, example: "underscored_lowercase_e"
    
    uses utillib shortName function and adds refinements
    
    @author Kai Heitmann
    @since 2025
:)
declare %private function utilfsh:fhirName($name as xs:string?) as xs:string? {
    let $shortname := utillib:shortName($name)
    let $shortname :=
        if ($shortname) then (

            (: 
                CamelCase short name, e.g. "find_matching_alternatives" => "FindMatchingAlternatives"
            :)
            let $r1 := string-join(
                for $part in tokenize($shortname, "_")
                return concat(
                  upper-case(substring($part, 1, 1)),
                  substring($part, 2)
                )
              , '')

            (: make sure we do not start with a digit :)
            let $r2 := replace($r1, '^(\d)' , 'X$1')
            
            (: make sure we are <= 64 characters, if so add a random number 10..99 :)
            let $r2 := if (string-length($r2) > 64) then concat(substring($r2, 1, 62), utilfsh:randomSuffix2()) else $r2
            
            return $r2
            
        ) else ()
        
    return if (matches($shortname, '^[a-zA-Z_][a-zA-Z\d_]+$')) then $shortname else ()
};

(:  Legal in FHIR:          draft | active | retired
    Legal in Template ITS:  draft | active | retired | new | rejected | cancelled | pending | review
:)
declare %private function utilfsh:decorStatus2fhirStatus($status as xs:string?) as xs:string? {
    switch ($status)
    case 'new'          return 'draft'
    case 'draft'        return 'draft'
    case 'pending'      return 'draft'
    case 'active'       return 'active'
    case 'final'        return 'active' (: not technically a template status, but an item status :)
    case 'review'       return 'draft'
    case 'cancelled'    return 'retired'
    case 'rejected'     return 'retired'
    case 'deprecated'   return 'retired'
    default             return 'draft'
};

declare %private function utilfsh:decorBindingStrength2fhirBindingStrength($strength as xs:string?) as xs:string {
    switch ($strength)
    case 'CNE'          return 'required'
    case 'CWE'          return 'extensible'
    case 'required'     return 'required'
    case 'extensible'   return 'extensible'
    case 'preferred'    return 'preferred'
    case 'example'      return 'example'
    default             return 'required'
};

declare %private function utilfsh:decorVocabulary2fhirBinding(
    $vocabularies as element()*,
    $projectPrefix as xs:string,
    $strength as xs:string?,
    $path as xs:string
    ) as xs:string* {
    
    for $vocabulary in $vocabularies[@valueSet] | $vocabularies[local-name()='valueSet']
    let $valueSet   := if ($vocabulary[local-name()='valueSet']) then $vocabulary else (utillib:getValueSetByRef($vocabulary/@valueSet, string($vocabulary/@flexibility), $projectPrefix, (), ())//*:valueSet[@id])[1]
    let $canonical  := utillib:getCanonicalUriForOID('ValueSet', $valueSet, $projectPrefix, $setlib:strKeyFHIRR4)
    return
        "&#10;* " || $path || " from " || $canonical || ' (' || $strength || ")" || "  // " || ($valueSet/@displayName, $valueSet/@name)[1]

};

(:~ Returns a random two character sequence for unification AA..ZZ
    
    @author Kai Heitmann
    @since 2025
:)
declare %private function utilfsh:randomSuffix2() as xs:string { 
    concat(
        substring('0123456789', util:random(9)+1, 1),
        substring('abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', util:random(51)+1, 1)
    )
};

declare %private function utilfsh:decorExample2FHIRexample(
    $concept as element(concept),
    $examples as element(example)*,
    $fhirType as xs:string?,
    $properties as element()*,
    $oidfhirmap as map(*)) as xs:string* {
    
    let $exampleElementName     := 
        switch ($fhirType)
        case 'Age'              return 'valueQuantity'
        case 'Distance'         return 'valueQuantity'
        case 'SimpleQuantity'   return 'valueQuantity'
        case 'Duration'         return 'valueQuantity'
        case 'Count'            return 'valueQuantity'
        case 'Quantity'         return 'valueQuantity'
        default                 return concat('value',upper-case(substring($fhirType, 1, 1)), substring($fhirType, 2))
    
    for $example at $i in $examples[not(@type = 'error')]
    let $value  := normalize-space(string-join($example//text(), ''))
    return
        switch ($fhirType)
        case 'CodeableConcept' return (
            let $exampleConcept     := $example/ancestor::valueDomain/conceptList/concept[name = $value]
            let $assocs             :=
                if (count($concept) = 1) then 
                    $concept/terminologyAssociation[@conceptId = $concept/@id] 
                else (
                    $concept/terminologyAssociation[@displayName = $value]
                )
            let $assocs             := 
                if ($assocs) then $assocs else (
                    $concept/valueSet/conceptList/*[@displayName = $value]
                )
            return (
                "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"',
                for $ass in $assocs[@code][@codeSystem]
                    return (
                        "&#10;  * ^example[=]." || $exampleElementName || ".coding = " ||
                            map:get($oidfhirmap, $ass/@codeSystem || $ass/@codeSystemVersion) || "#" || $ass/@code ||
                            (if ($ass/@displayName) then concat(' "', $ass/@displayName, '"') else ()),
                        "&#10;  * ^example[=]." || $exampleElementName || ".text = " || '"""' || $value || '"""'
                    )
            )
        )
        case 'Identifier' return (
            let $assocs   := $example/ancestor::concept[1]/identifierAssociation
            return (
                "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                (if (count($assocs) = 1) then "&#10;  * ^example[=]." || $exampleElementName || ".system = " || '"' ||
                    utillib:getCanonicalUriForOID('NamingSystem', $assocs/@ref, (), $example/ancestor::dataset/@prefix, $setlib:strKeyFHIRR4) || '"'
                else ()),
                "&#10;  * ^example[=]." || $exampleElementName || ".value =  " || '"' || $value || '"'
            )
        )
        case 'string' return
                "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                "&#10;  * ^example[=]." || $exampleElementName || " = " || '"' || $example/text() || '"'
        case 'markdown' return
                "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                "&#10;  * ^example[=]." || $exampleElementName || ".value =  " || '"' || markdown:html2markdown($example/node()) || '"'
        case 'date' return (
            let $valid  :=
                if ($value castable as xs:date) then $value else
                if (matches($value, '^\d{2}/\d{2}/\d{4}')) then (
                    (: US style mm/dd/yyyy :)
                    concat(substring($value, 7, 4), '-', substring($value, 1, 2), '-', substring($value, 4, 2))[. castable as xs:date]
                )
                else
                if (matches($value, '^\d{2}.\d{2}.\d{4}')) then (
                    (: Rest of the world style dd-mm-yyyy :)
                    concat(substring($value, 7, 4), '-', substring($value, 4, 2), '-', substring($value, 1, 2))[. castable as xs:date]
                ) else ()
                
            return
                if (string-length($valid) > 0) then (
                    "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                    "&#10;  * ^example[=]." || $exampleElementName || ".value =  " || '"' || $valid || '"'
                ) else ()
        )
        case 'dateTime' return (
            let $valid  :=
                if ($value castable as xs:dateTime) then $value else
                if ($value castable as xs:date) then $value else
                if (matches($value, '^\d{2}/\d{2}/\d{4}')) then (
                    (: US style mm/dd/yyyy :)
                    let $d        := concat(substring($value, 7, 4), '-', substring($value, 1, 2), '-', substring($value, 4, 2))
                    let $t        := replace(normalize-space(substring($value, 11)), '[^\d:Z+\-]', '')
                    let $t        := if (matches($t, '^\d{2}(:\d{2})?$')) then substring($t || ':00:00', 1, 8) else $t
                    let $dt       := $d || 'T' || $t
                    
                    return
                    if ($dt castable as xs:dateTime) then
                        if (empty(timezone-from-time(xs:time($t)))) then
                            string(adjust-dateTime-to-timezone(xs:dateTime($dt)))
                        else (
                            $dt
                        )
                    else 
                    if ($d castable as xs:date) then $d else ()
                )
                else
                if (matches($value, '^\d{2}.\d{2}.\d{4}')) then (
                    (: Rest of the world style dd-mm-yyyy :)
                    let $d        := concat(substring($value, 7, 4), '-', substring($value, 1, 2), '-', substring($value, 4, 2))
                    let $t        := replace(normalize-space(substring($value, 11)), '[^\d:Z+\-]', '')
                    let $t        := if (matches($t, '^\d{2}(:\d{2})?$')) then substring($t || ':00:00', 1, 8) else $t
                    let $dt       := $d || 'T' || $t
                    
                    return
                    if ($dt castable as xs:dateTime) then
                        if (empty(timezone-from-time(xs:time($t)))) then
                            string(adjust-dateTime-to-timezone(xs:dateTime($dt)))
                        else (
                            $dt
                        )
                    else 
                    if ($d castable as xs:date) then $d else ()
                ) else ()
            return
                if (string-length($valid) > 0) then (
                    "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                    "&#10;  * ^example[=]." || $exampleElementName || ".value =  " || '"' || $valid || '"'
                ) else ()
        )
        case 'time' return (
            let $valid  := if ($value castable as xs:time) then $value else ()
            return
                if (string-length($valid) > 0) then (
                    "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                    "&#10;  * ^example[=]." || $exampleElementName || ".value =  " || '"' || $valid || '"'
                ) else ()
        )
        case 'decimal' return (
            let $value  := if ($value castable as xs:decimal) then $value else replace($value, ',', '.')
            return
                if ($value castable as xs:decimal) then (
                    "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                    "&#10;  * ^example[=]." || $exampleElementName || ".value =  " || $value
                ) else ()
        )
        case 'boolean' return (
            let $value  :=
                if ($value castable as xs:boolean) then xs:boolean($value) else 
                if (lower-case($value) = ('yes','ja','oui','si')) then true() else
                if (lower-case($value) = ('no','nee','non','nein')) then false() else (true())
            return
                if ($value castable as xs:decimal) then (
                    "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                    "&#10;  * ^example[=]." || $exampleElementName || ".value =  " || '"' || $value || '"'
                ) else ()
        )
        case 'base64Binary' return ()
        default return (
            if ($fhirType = ('Age','Count','Duration','Quantity','SimpleQuantity')) then (
                let $valuepart  := replace($value,'^([\d,\.]+).*','$1')
                (: fix decimals with comma (non US) if possible/necessary :)
                let $valuepart  := 
                    if ($valuepart castable as xs:decimal) then 
                        $valuepart 
                    else
                    if (replace($valuepart, ',', '.') castable as xs:decimal) then 
                        replace($valuepart, ',', '.')
                    else (
                        (: cannot be fixed. someone needs to fix this in the dataset example :)
                        $valuepart
                    )
                (: unit is required :)
                let $unitpart   := (normalize-space(substring-after($value, $valuepart)), $properties/@unit, '1')[string-length() > 0][1]
                return
                if ($valuepart castable as xs:decimal) then (
                    "&#10;  * ^example[+].label = " || '"' || "# " || $i || '"' ||
                    "&#10;  * ^example[=]." || $exampleElementName || ".value =  " || '"' || $valuepart || '"' ||
                    (if (empty($unitpart) or $unitpart = '1') then () else "&#10;  * ^example[=]." || $exampleElementName || ".unit = " || '"' || $unitpart || '"') ||
                    (if ($unitpart = '1' or $utilfsh:ucum[@unit = $unitpart]) then (
                        "&#10;  * ^example[=]." || $exampleElementName || ".system = " || '"http://unitsofmeasure.org"' ||
                        "&#10;  * ^example[=]." || $exampleElementName || ".code =  " || '"' || $unitpart || '"'
                    ) else ())
                ) else ()
            )
            else ()
        )
};

declare %private function utilfsh:getSemverString($versionlabel as xs:string?) as xs:string? {
    if (utilfsh:isSemverString($versionlabel)) then ($versionlabel) else
    if (utilfsh:isSemverString(concat($versionlabel, '.0'))) then (concat($versionlabel, '.0')) else
    if (utilfsh:isSemverString(substring($versionlabel, 2))) then substring($versionlabel, 2) else
    if (utilfsh:isSemverString(concat(substring($versionlabel, 2), '.0'))) then concat(substring($versionlabel, 2), '.0')
    else ()
};

(:~ A normal version number MUST take the form X.Y.Z where X, Y, and Z are non-negative integers, and MUST NOT contain 
    leading zeroes. X is the major version, Y is the minor version, and Z is the patch version. Each element MUST 
    increase numerically. For instance: 1.9.0 -> 1.10.0 -> 1.11.0.
    
    Build metadata MAY be denoted by appending a plus sign and a series of dot separated identifiers 
    immediately following the patch or pre-release version. Identifiers MUST comprise only ASCII 
    alphanumerics and hyphens [0-9A-Za-z-]. Identifiers MUST NOT be empty. Build metadata MUST be 
    ignored when determining version precedence. Thus two versions that differ only in the build metadata, 
    have the same precedence. 
    Examples: 1.0.0-alpha+001, 1.0.0+20130313144700, 1.0.0-beta+exp.sha.5114f85, 1.0.0+21AF26D3----117B344092BD.
    
    Previous used regex was: ^\d+\.\d+\.\d+(-[A-Za-z\d+\-])?$
    
    NEW: ^(0|[1-9]\d*)\.(0|[1-9]\d*)\.(0|[1-9]\d*)(?:-((?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*)(?:\.(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*))*))?(?:\+([0-9a-zA-Z-]+(?:\.[0-9a-zA-Z-]+)*))?$

    https://semver.org :)
declare function utilfsh:isSemverString($versionlabel as xs:string?) as xs:boolean {
    if (empty($versionlabel)) then false() else matches($versionlabel, '^(0|[1-9]\d*)\.(0|[1-9]\d*)\.(0|[1-9]\d*)(?:-((?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*)(?:\.(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*))*))?(?:\+([0-9a-zA-Z-]+(?:\.[0-9a-zA-Z-]+)*))?$')
};

(:~ Append official identifier OID

    @author Kai Heitmann
    @since 2025
:)
declare %private function utilfsh:addFSHidentifierOid($fsh, $prefix, $decorId) {
    array:append ($fsh,
        $prefix || "^identifier.use = #official"                     || $utilfsh:CR ||
        $prefix || "^identifier.system = " || '"urn:ietf:rfc:3986"'  || $utilfsh:CR ||
        $prefix || "^identifier.value = " || '"urn:oid:' || $decorId || '"'
    )
};

(:~ Append meta dates effectiveDate and expirationDate

    @author Kai Heitmann
    @since 2025
:)
declare %private function utilfsh:addFSHmetadates(
    $fsh as array(*),
    $prefix as xs:string,
    $effectiveDate as xs:string?,
    $expirationDate as xs:string?
) as array(*) {
    let $ed := $effectiveDate castable as xs:dateTime
    let $ex := $expirationDate castable as xs:dateTime
    return if ($ed or $ex) then
        array:append ($fsh,
            "* ^extension.url = " || '"http://hl7.org/fhir/StructureDefinition/resource-effectivePeriod"' || $utilfsh:CR ||
            (if ($ed) then
                $prefix || "^extension.valuePeriod.start = " || '"' || adjust-dateTime-to-timezone(xs:dateTime($effectiveDate)) || '"' || $utilfsh:CR else "") ||
            (if ($ex) then
                $prefix || "^extension.valuePeriod.end = " || '"' || adjust-dateTime-to-timezone(xs:dateTime($expirationDate)) || '"' else "")
        ) else $fsh
};

(:~ Append meta publishingAuthority and contact and copyright

    @author Kai Heitmann
    @since 2025
:)
declare %private function utilfsh:addFSHpublishingAuthorityContactCopyright(
    $fsh as array(*),
    $prefix as xs:string,
    $publishingAuthority as element(publishingAuthority)*,
    $copyright as element(copyright)*
) as array(*) {
   
    (: publishingAuthority: FHIR publisher is only a string, catenate all :)
    let $fsh        := if ($publishingAuthority) then
        array:append($fsh,  $prefix || "^publisher = " || '"' || string-join($publishingAuthority/@name, ", ") || '"') else $fsh
        
    (: make use of possibly more information in publishingAuthority with addrLines etc :)
    let $crstring   :=
        for $publs in $publishingAuthority[@name]
        return (
            $prefix || "^contact[+].name = " || '"' || $publs/@name || '"',
            for $addrLine in $publs/*:addrLine
            let $system := if ($addrLine/@type = ('phone', 'email', 'fax', 'uri')) then $addrLine/@type else 'other'
            return (
                $prefix || "^contact[=].telecom[+].system = #" || $system,
                $prefix || "^contact[=].telecom[=].value  = " || '"' || $addrLine || '"'
            )
        )
    let $fsh := if (not(empty($crstring))) then array:append($fsh, concat($utilfsh:CR, string-join($crstring, $utilfsh:CR), $utilfsh:CR)) else $fsh
    
    (: copyright in FHIR is only markdown, catenate all texts :)
    let $fsh        := if ($copyright) then array:append($fsh,  $prefix || "^copyright = " || '"' || string-join($copyright/text(), ", ") || '"') else $fsh
    
    return $fsh
};

declare %private function utilfsh:addPurpose(
    $fsh as array(*),
    $prefix as xs:string,
    $in as element(purpose)*,
    $language as xs:string
)  as array(*) {
    let $default := ($in[@language = $language], $in)[1]
    return
        if ($default) then (
            array:append($fsh, $prefix || "^purpose = " || '"' || $default || '"'),
            for $n in $in[@language][not(@language = $default/@language)]
            return utilfsh:addMultilingual($fsh, $prefix, $in[@language][not(@language = $default/@language)])
        ) else $fsh
};

declare %private function utilfsh:addMultilingual(
    $fsh as array(*),
    $prefix as xs:string,
    $mlelm as node()*
) as array(*) {
    let $result := 
        for $n in $mlelm
        return
                $prefix || "^extension.url = " || '"http://hl7.org/fhir/StructureDefinition/translation"' || $utilfsh:CR ||
                "  " || $prefix || "^extension.url = " || '"lang"' || $utilfsh:CR ||
                "  " || $prefix || "^extension.valueCode = " || '"' || $n/@language || '"' || $utilfsh:CR ||
                "  " || $prefix || "^extension.url = " || '"content"' || $utilfsh:CR ||
                "  " || $prefix || "^extension.valueMarkdown = " || '"' || markdown:html2markdown(utillib:parseNode($n)/node()) || '"'
     return
        if (count($mlelm) > 0) then array:append($fsh, string-join($result, $utilfsh:CR))
        else $fsh
};