xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Dataset API allows read, create, update on DECOR datasets :)
module namespace dsapi              = "http://art-decor.org/ns/api/dataset";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utilds      = "http://art-decor.org/ns/api/util-dataset" at "library/util-dataset-lib.xqm";
import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace utilhtml    = "http://art-decor.org/ns/api/util-html" at "library/util-html-lib.xqm";
import module namespace utilfsh     = "http://art-decor.org/ns/api/util-fsh" at "library/util-fsh-lib.xqm";

import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";

declare namespace json              = "http://www.json.org";

(:~ Retrieves latest DECOR dataset based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $treeonly                - optional boolean parameter to get the tree structure only if true
    @param $fullTree                - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function dsapi:getLatestDataset($request as map(*)) {
    dsapi:getDataset($request)
};

(:~ Retrieves DECOR dataset based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $treeonly                - optional boolean parameter to get the tree structure only if true
    @param $fullTree                - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function dsapi:getDataset($request as map(*)) {

    let $id                 := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectVersion     := $request?parameters?release[not(. = '')]
    let $projectLanguage    := $request?parameters?language[not(. = '')]
    let $treeOnly           := $request?parameters?treeonly = true()
    let $fullTree           := $request?parameters?fulltree = true()
    let $propertiesMap      := 
        map:merge((
            for $s in $request?parameters?conceptproperty[string-length() gt 0]
            for $item in tokenize(lower-case($s),'\s') 
            return map:entry($item, true())
        ))
    
    let $acceptTypes        := roaster:accepted-content-types()
    let $acceptedType       := ($acceptTypes[. = ('application/xml', 'application/json')],'application/json')[1]
    
    let $format             := tokenize($acceptedType, '/')[2]
    
    let $results            := utilds:getDataset($id, $effectiveDate, $projectVersion, $projectLanguage, $treeOnly, $fullTree, $propertiesMap)

    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple datasets for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*, $format)
                }
        )
};

(:~ Retrieves DECOR dataset as a diagram based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $conceptId               - optional parameter denoting the id of the concept in the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the concept in the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $filter                  - optional parameter to exclude statusses from the selection
    @param $interactive             - boolean
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @return as-is 
    @since 2023-12-21
:)

declare function dsapi:getDatasetDiagram($request as map(*)) {

    let $conceptId                      := $request?parameters?conceptId[not(. = '')]
    let $conceptEffectiveDate           := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?conceptEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?conceptEffectiveDate[string-length() gt 0]
        }
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $filter                         := $request?parameters?filter[not(. = '')]
    let $interactive                    := not($request?parameters?interactive = false())
    let $format                         := $request?parameters?format[not(. = '')]
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $check                          :=
        if (empty($conceptId) and not(empty($conceptEffectiveDate))) then 
            error($errors:BAD_REQUEST, 'Request SHALL NOT have conceptEffectiveDate without conceptId') 
        else ()
    
    let $format                         := if (not($format = 'xml')) then 'svg' else 'xml'
    (: 
       overwrite format in the backend is not always possible because of middleware behaviour
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
    :) 
    let $check                          :=
        if ($format = 'xml' and not((roaster:accepted-content-types()[. = ('image/svg+xml', 'application/xml')])[1] = 'application/xml')) then 
            error($errors:BAD_REQUEST, 'In case of format parameter is xml the accept header should be application/xml')
        else ()   
    
    let $results                        := utillib:getDatasetDiagram($id, $effectiveDate, $projectVersion, $projectLanguage, $conceptId, $conceptEffectiveDate, $filter, $interactive, $format)
 
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else if ($format = 'xml') then (
            for $result in $results
                return
                    element {name($result)} {
                        $result/@*,
                        namespace {"json"} {"http://www.json.org"},
                        utillib:addJsonArrayToElements($result/*, $format)
                    }
        ) else (
            let $responseheader      := (response:set-header('Content-Type','image/svg+xml'), response:set-header('X-Robots-Tag', 'noindex'))
            return $results
            )
};

(:~ Retrieves DECOR dataset for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $communityName           - optional space separated list of community names. If omitted all communities are given
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @param $download                - optional as xs:boolean. Default: false. 
    @return as-is or as compiled as JSON
    @since 2023-11-10
:)
declare function dsapi:getDatasetExtract($request as map(*)) {

    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $communityName                  := $request?parameters?community[not(. = '')]
    let $format                         := $request?parameters?format[not(. = '')]
    let $download                       := $request?parameters?download=true()
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    (: parameter format overrides accept-header as mime-type for using the api in the browser :)
    
    (: 
        this call is used in backend and frontend (browser) - default behaviour mime-types:
        - backend:   default mime-type is json - response payload is json - if no accept type is given
        - frontend:  default mime-type is xml - response payload is xml - if no format is given
    :)
    
    let $acceptTypes                    := roaster:accepted-content-types()
    let $acceptedType                   := ($acceptTypes[. = ('application/xml', 'application/json')], 'application/json')[1]
    
    let $format                         := if ($format) then $format else tokenize($acceptedType, '/')[2]

    (: 
       overwrite format in the backend is not always possible because of middleware behaviour
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
    :) 
    let $check                          :=
        if ($format = 'xml' and not($acceptedType = 'application/xml')) then 
            error($errors:BAD_REQUEST, "In case of format parameter is xml the accept header should be application/xml")
        else ()

    (: check valid formats :)
    let $check                          :=
        if ($format = 'xml' or $format = 'json' or $format = 'fsh') then ()
        else error($errors:BAD_REQUEST, "The format parameter shall be one of 'xml' or 'json' or 'fsh', found: " || $format)

    let $results                        := utilds:getDatasetExtract($id, $effectiveDate, $projectVersion, $projectLanguage, $communityName)
    
    let $filename                       := 'DS_' || $results/@shortName || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $format
    
    let $results                        :=
        if ($format = 'fsh') then utilfsh:convertTransactionOrDataset2Fsh($results, $projectLanguage, $filename)/text()
        else for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/node(), $format)
                }
    
    (: in this case the json format overrides header application/xml and works with a text header :)
    let $results                        := if ($format = 'json' and not($acceptedType = 'application/json')) then fn:serialize($results, map{"method": $format , "indent": true()}) else $results
    
    (: determine repsonse format :)
    let $responseformat                 := if ($format = 'fsh') then 'text' else $format
           
    let $responseheader                 := 
        (response:set-header('Content-Type', 'application/' || $responseformat || '; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple datasets for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else $results        

};

(:~ Retrieves DECOR dataset view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $ui-language             - optional parameter to select from a specific language for the ui
    @param $hidecolumns             - optional parameter to hide columns in the view based on (hex) number
    @param $format                  - optional. if not given it is html
    @param $download                - optional as xs:boolean. Default: false. 
    @return as-is 
    @since 2024-09-24
:)
declare function dsapi:getDatasetView($request as map(*)) {

    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $communityName                  := $request?parameters?community[not(. = '')]
    let $ui-lang                        := $request?parameters?ui-language[not(. = '')]
    let $hidecolumns                    := $request?parameters?hidecolumns[not(. = '')]
    let $format                         := $request?parameters?format[not(. = '')]
    let $download                       := $request?parameters?download=true()
    let $inline                         := $request?parameters?inline = true()

    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }

    let $format                         := if (not($format = 'list')) then 'html' else 'list'
    
    let $dataset                        := utillib:getDataset($id, $effectiveDate, $projectVersion, $projectLanguage)
    let $decor                          := $dataset/ancestor::decor[1]

    let $results                        := if (empty($dataset)) then () else utillib:getDatasetExtract($dataset, $id, $effectiveDate, (), (), $projectVersion, $projectLanguage, $communityName)

    let $filename                       := 'DS_' || $results/@shortName || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html' 
    let $r-header                       := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple datasets for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else 
        
        (: prepare for Html :)
        let $referenceUrl               := $decor/project/reference[@url castable as xs:anyURI]/@url
        let $projectPrefix              := $decor/project/@prefix
        let $header                     := if ($inline) then false() else true()        
             
        return 
            if ($format = 'list') 
                then utilhtml:convertTransactionOrDataset2SimpleHtml($results, $projectLanguage, $ui-lang, $hidecolumns, true(), $projectVersion, $referenceUrl, $projectPrefix, false(), $download) 
                else utilhtml:convertTransactionOrDataset2Html($results, $projectLanguage, $ui-lang, $hidecolumns, true(), true(), true(), $projectVersion, $referenceUrl, $projectPrefix, (), false(), $filename, $download, $header)
};

(:~ Retrieves DECOR dataset usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the dataset
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the dataset. If not given assumes latest version for id
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function dsapi:getDatasetUsage($request as map(*)) {
    
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $results                        := utilds:getDatasetUsage($id, $effectiveDate)
    
    let $count                          := count($results)
    let $max                            := $count
    let $allcnt                         := $count
    
    return
        <list artifact="USAGE" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements(subsequence($results, 1, $max))
        }
        </list>
};

(:~ Retrieves DECOR dataset history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return list
    @since 2022-08-16
:)
declare function dsapi:getDatasetHistory($request as map(*)) {
    let $authmap                        := $request?user
    let $id                             := $request?parameters?id
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix                  := ()
    let $results                        := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-DATASET, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-DATASET}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Returns a list of zero or more datasets
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $includeBBR              - optional. Include BBR datasets in the list. BBRs have to be declared in the project
    @param $scenariosonly           - optional boolean. If true only includes datasets bound in a scenario transaction
    @return all live repository/non-private datasets as JSON, all data sets for the given $projectPrefix or nothing if not found
    @since 2020-05-03
:)
declare function dsapi:getDatasetList($request as map(*)) {

    let $searchTerms                    := 
        array:flatten(
            for $s in $request?parameters?search[string-length() gt 0]
            return
                tokenize(lower-case($s),'\s')
        )
    let $projectPrefix                  := $request?parameters?prefix[not(. = '')]
    let $projectVersion                 := $request?parameters?release[not(. = '')]
    let $projectLanguage                := $request?parameters?language[not(. = '')]
    let $includeBbr                     := $request?parameters?includebbr = true()
    let $scenariosOnly                  := $request?parameters?scenariosonly = true()
    let $treeType                       := $request?parameters?treetype[not(. = '')]
    
   
    let $results                        := utilds:getDatasetList($projectPrefix, $projectVersion, $projectLanguage, $includeBbr, $scenariosOnly, $treeType, $searchTerms)
    
    return
        <list artifact="{$results[1]/@artifact}" current="{sum($results/xs:integer(@current))}" total="{sum($results/xs:integer(@total))}" all="{sum($results/xs:integer(@allcnt))}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements($results/*)
        }
        </list>
};

(: Create a dataset, either empty or based on another dataset
    @param $projectPrefix           - project to create this dataset in
    @param $targetDate              - If true invokes effectiveDate of the new dataset and concepts as [DATE]T00:00:00. 
                                    - If false invokes date + time [DATE]T[TIME] 
    @param $sourceId                - parameter denoting the id of a dataset to use as a basis for creating the new dataset
    @param $sourceEffectiveDate     - parameter denoting the effectiveDate of a dataset to use as a basis for creating the new dataset",
    @param $keepIds                 - Only relevant if source dataset is specified. If true, the new dataset will keep the same ids for the new dataset, and only update the effectiveDate
    @param $baseDatasetId           - Only relevant when a source dataset is specified and `keepIds` is false. This overrides the default base id for datasets in the project. The value SHALL match one of the projects base ids for datasets
    @param $baseConceptId           - Only relevant when a source dataset is specified and `keepIds` is false. This overrides the default base id for concepts in the project. The value SHALL match one of the projects base ids for concepts
    @param $templateAssociations    - Only relevant if source dataset is specified, and commit=true. 
                                    - If 'true' then template associations are generated as copy of existing template associations for the selected dataset concepts and are immediately committed in the project
                                        **Note that this immediately affects your project and that there is no undo possible!!**
                                    - If 'false' then template associations are generated and returned, but not committed to the project.
    @param $skipDeprecated          - Only relevant if source dataset is specified.
                                    - If 'true' then deprecated concepts in the base dataset/templateAssociations are included. This is recommended
                                    - If 'false' the deprecated concepts in the base dataset/templateAssociations are NOT included
                                        Note that cancelled, and rejected concept are never included as they are supposed to never have been published. 
                                        Deprecated concepts however have been published and (potentially) implemented.
    @param $testMode                - If you only want to try out what woould happen with all your selected parameters and don't want to commit the result just yet, this value should be true
    @return (empty) dataset object as xml with json:array set on elements
:)
declare function dsapi:postDataset($request as map(*)) {

    let $authmap                        := $request?user
    let $projectPrefix                  := $request?parameters?prefix[not(. = '')]
    let $targetDate                     := $request?parameters?targetDate = true()
    let $sourceId                       := $request?parameters?sourceId[not(. = '')]
    let $sourceEffectiveDate            := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?sourceEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?sourceEffectiveDate[string-length() gt 0]
        }
    let $keepIds                        := $request?parameters?keepIds = true()
    let $baseDatasetId                  := $request?parameters?baseDatasetId[not(. = '')]
    let $baseConceptId                  := $request?parameters?baseConceptId[not(. = '')]
    let $templateAssociations           := $request?parameters?templateAssociations = true()
    let $skipDeprecated                 := $request?parameters?skipDeprecated = true()
    let $testMode                       := $request?parameters?testMode = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                          :=
        if (empty($projectPrefix)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter prefix')
        else ()
    
    let $results                        := utilds:createDataset($authmap, $projectPrefix, $targetDate, $sourceId, $sourceEffectiveDate, $keepIds, $baseDatasetId, $baseConceptId, $templateAssociations, $skipDeprecated, $testMode)
    
    return
        roaster:response(if ($testMode) then 200 else 201, 
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Update dataset
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the dataset to update 
    @param $effectiveDate            - required. the effectiveDate for the dataset to update 
    @param $request-body             - required. body containing new concept structure
    @return concept structure
    @since 2020-05-03
:)
declare function dsapi:putDataset($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                           := utillib:getBodyAsXml($request?body, 'dataset', ())
    let $treeOnly                       := $request?parameters?treeonly = true()
    let $fullTree                       := $request?parameters?fulltree = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()

    let $check                          :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $update                         := utilds:putDataset($authmap, string($id), $effectiveDate, $data)
    let $results                        := utilds:getDataset($id, $effectiveDate, (), (), $treeOnly, $fullTree, ())
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple datasets for id '", $id, "' effectiveDate '", $effectiveDate, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Update DECOR dataset parts. Does not touch any concepts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

{ "op": "[add|remove|replace]", "path": "[/|/displayName|/priority|/type]", "value": "[string|object]" }

where

* op - add (object associations, tracking, event) or remove (object associations only) or replace (displayName, priority, type, tracking, assignment)

* path - / (object, tracking, assignment) or /displayName or /priority or /type

* value - string when the path is not /. object when path is /
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the issue to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return dataset structure including generated meta data
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function dsapi:patchDataset($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  :=
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $treeOnly                       := $request?parameters?treeonly = true()
    let $fullTree                       := $request?parameters?fulltree = true()
    let $data                           := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
        
    let $check                          :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $update                         := utilds:patchDataset($authmap, string($id), $effectiveDate, $data)
    let $results                        := utilds:getDataset($id, $effectiveDate, (), (), $treeOnly, $fullTree, ())
    
    return 
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple datasets for id '", $id, "' effectiveDate '", $effectiveDate, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )

};

(:~ Update dataset statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap                 - required. Map derived from token
    @param $id                      - required. DECOR concept/@id to update
    @param $effectiveDate           - required. DECOR concept/@effectiveDate to update
    @param $recurse                 - optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $listOnly                - optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode           - optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel         - optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate       - optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate  - optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function dsapi:putDatasetStatus($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $recurse                        := $request?parameters?recurse = true()
    let $list                           := $request?parameters?list = true()
    let $statusCode                     := $request?parameters?statusCode[not(. = '')]
    let $versionLabel                   := $request?parameters?versionLabel[not(. = '')]
    let $expirationDate                 := $request?parameters?expirationDate[not(. = '')]
    let $officialReleaseDate            := $request?parameters?officialReleaseDate[not(. = '')]
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                        := utilds:setDatasetStatus($authmap, $id, $effectiveDate, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
    
    return
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            (: max 2 arrays: one with error and one with success :)
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Clears all lock on a dataset and it concepts, and deletes all concepts with status 'new'. This is effectively like hitting cancel on editing in progress :)
declare function dsapi:postDatasetClearLocks($request as map(*)) {

    let $authmap                        := $request?user
    let $id                             := $request?parameters?id[not(. = '')]
    let $effectiveDate                  := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $treeOnly                       := $request?parameters?treeonly = true()
    let $fullTree                       := $request?parameters?fulltree = true()
    
    let $check                          :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $delete                         := utilds:deleteDatasetLocks($authmap, $id, $effectiveDate)
    let $results                        := utilds:getDataset($id, $effectiveDate, (), (), $treeOnly, $fullTree, ())
    
    return 
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple datasets for id '", $id, "' effectiveDate '", $effectiveDate, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};