xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utiltcs                = "http://art-decor.org/ns/api/util-terminology-codesystem";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

declare function utiltcs:getValueSetExpansionSet($valueSet as element(valueSet), $params as map(*)) {
    let $maxResults           := ($params?maxResults[. instance of xs:positiveInteger][. ge 1], 1000)[1]
    let $expand               := $params?expand[. instance of xs:boolean] = true()
    let $debug                := $params?debug[. instance of xs:boolean] = true()
    let $searchString         := $params?searchString[not(. = '')]
    let $valueSet             := utiltcs:getValueSetForExpansion($valueSet)
    let $composition          := 
        <composition>
        {
            $valueSet/@*
        }
        {
            for $ccs in $valueSet/completeCodeSystem
            return
                <include>{$ccs/@*, $ccs/*}</include>
            ,
            $valueSet/conceptList/*
        }
        </composition>
        
    (: is the value set intensional at all? :)
    let $intensional        := count($composition/completeCodeSystem) + count($composition/include) + count($composition/exclude) gt 0
    
    (: get expansions as a query to execute :)
    let $luceneQuery        := if (empty($searchString)) then () else utillib:getSimpleLuceneQuery($searchString)
    let $luceneOptions      := if (empty($searchString)) then () else utillib:getSimpleLuceneOptions()
    let $expansion          := utiltcs:getExpansionQueryForCompose($composition)
    let $queryStringAll     := string-join($expansion/query/@q, '|')
    
    (: first concepts, then exceptions :)
    let $result             := 
        for $qq in $expansion/query
        let $type           := if ($qq[@exception='true']) then 'exception' else 'concept'
        order by $type
        return
            <concepts codeSystem="{$qq/@codeSystem}" collection="{$qq/@collection}" type="{$type}">
            {
                if (empty($searchString)) then 
                    util:eval($qq/@q)
                else (
                    util:eval($qq/@q)[ft:query(designation, $luceneQuery, $luceneOptions)]
                )
            }
            </concepts>
    let $resultCount        := count($result/*)
    
    (: count any code in include or exclude construct that is NOT active, for warnings on incomplete expansions :)
    let $nonActiveCount     := count(util:eval(string-join($expansion/query/@nonactive, '|')))
    
    let $current            :=
        if ($expand) then if ($resultCount > $maxResults) then $maxResults else $resultCount else $resultCount
    
    let $timestamp          := current-dateTime()
    
    return
        <valueSetExpansionSet current="{$current}" count="{$resultCount}" nonactive="{$nonActiveCount}" search="{string-length($searchString) gt 0}">
        {
            
            if ($debug) then attribute query { $queryStringAll } else ()
        }
        {
            if ($expand) then (
                attribute id {util:uuid()},
                attribute effectiveDate {substring(string($timestamp), 1, 19)},
                attribute statusCode {'final'},
                for $vs in $valueSet
                return
                    element {name($vs)} {
                        $vs/(@* except @inode),
                        $vs/*
                    }
                ,
                if ($debug) then $composition else ()
                ,
                <expansion timestamp="{$timestamp}" total="{$resultCount}">
                {
                    utiltcs:cropc(subsequence($result/*, 1, $maxResults))
                }
                </expansion>
            ) else ()
        }
        </valueSetExpansionSet>
};

(: return the collection of queries to be executed for counting or exapnsion, wrapped by code system :)
declare %private function utiltcs:getExpansionQueryForCompose($compose as element()*)  {
    (: 
        first, create a new composition with all include/@ref replaced by their corresponding concepts, recursively
    :)
    let $newc := 
        for $x in $compose
        return
            <composition>
            {
                $x/@*,
                utiltcs:resolveIncludeRefs ($x, '', <valueSet ref="{$compose/@id}" flexibility="{$compose/@effectiveDate}"/>)
            }
            </composition>
    
    let $errors := $newc//error
    
    (: Question: why don't we include exception in this for loop? It means exceptions are never part of an expansion :)
    let $queries :=
        for $includes in $newc/(completeCodeSystem|include|concept|exception)
        (: get code system OID :)
        let $codeSystemOid    := $includes/@codeSystem
        let $codeSystemFlex   := $includes/@flexibility
        group by $codeSystemOid
        return (
            (:  for CADTS codesystems: get the name of the collection where the code system data reside
                for coded concepts from value sets use the inode parameter, ie direct path to value set
            :)
            let $browsableCodeSystem  := 
                if (empty($codeSystemFlex)) then 
                    collection($setlib:strCodesystemStableData)/browsableCodeSystem[@oid = $codeSystemOid]
                else if ($codeSystemFlex='dynamic') then
                    let $bCodeSys := collection($setlib:strCodesystemStableData)/browsableCodeSystem[@oid = $codeSystemOid]
                    let $latestDate := max($bCodeSys/xs:dateTime(@effectiveDate))
                    return
                    $bCodeSys[@effectiveDate=$latestDate]                 
                else (
                    collection($setlib:strCodesystemStableData)/browsableCodeSystem[@oid = $codeSystemOid][@effectiveDate = $codeSystemFlex]
                )
            let $check                := 
                switch (count($browsableCodeSystem))
                case 1 return ()
                case 0 return (
                    let $otherVersion := collection($setlib:strCodesystemStableData)/browsableCodeSystem[@oid = $codeSystemOid]
                    let $r            :=
                        if (empty($codeSystemFlex)) then
                            'Cannot expand. Missing codeSystem ' || $codeSystemOid
                        else (
                            'Cannot expand. Missing codeSystem ' || $codeSystemOid || ' flexibility ' || $codeSystemFlex
                        )
                    return
                        if ($otherVersion) then
                            error($errors:SERVER_ERROR, $r || ' in central terminology services (CADTS). We currently have version ' || string-join($otherVersion/@effectiveDate, ', ') || '. CADTS supports 1 version at a time.')
                        else (
                            error($errors:SERVER_ERROR, $r || ' in central terminology services (CADTS). If this is an inline definition that has no formal code system yet, please create that. If this an externally defined code system, you might be able to get this installed through your server administrator depending on license and availability.')
                        )
                )
                default return
                    error($errors:SERVER_ERROR, 'Cannot expand. Multiple codeSystems for ' || $codeSystemOid || ' in central terminology services (CADTS). This is a db error for an administrator to fix.')
                    
            let $collectionPath       := util:collection-name($browsableCodeSystem)
            (: Now that we write every codeSystem in its own collection, this is no longer needed :)
            (:let $codeSystemCount      := count(collection($collectionPath)/browsableCodeSystem):)
            
            for $include in $includes
            let $isException          := $include/name()='exception' or $include/@exception='true'
            (: matching excludes i.e. same codesystem, and optimize if current include is a simple code:
               only excludes that are not also a simple code, unless the exclude is for exactly the same include code
                
                This prevents things like [@code=('195967001')][not(@code=('54070000'))][not(@code=('54837006'))][not(@code=('726738003'))] based on
                <concept code="195967001" codeSystem="2.16.840.1.113883.6.96" displayName="Asthma (disorder)" level="0" type="S"/>
                <exclude code="54070000" codeSystem="2.16.840.1.113883.6.96" displayName="Postpartum education (procedure)"/>
                <exclude code="54837006" codeSystem="2.16.840.1.113883.6.96" displayName="Straight back syndrome (disorder)"/>
                <exclude code="726738003" codeSystem="2.16.840.1.113883.6.96" displayName="Cytology report (record artifact)"/>
                
            :)                       
            let $exclPredicates     := 
                  if ($include[name()=('concept', 'exception')][@code] | $include[name()=('include')][@code][@op = 'equal' or empty(@op)]) then
                      utiltcs:getPredicatesForIncludeExclude(
                          $compose/exclude/(.[@codeSystem=$codeSystemOid] except .[@code][@op = 'equal' or empty(@op)][not(@code = $include/@code)])
                      )
                  else (
                      utiltcs:getPredicatesForIncludeExclude($compose/exclude[@codeSystem=$codeSystemOid])
                  )
            return
                <query codeSystem="{$codeSystemOid}" collection="{$collectionPath}">
                {
                    if ($codeSystemFlex) then attribute codeSystemVersion {$codeSystemFlex} else ()
                }
                {
                    (: concepts with a @code, eg <concept code="112144000" codeSystem="2.16.840.1.113883.6.96" displayName="Blood group A (finding)" level="0" type="L"/> :)
                    if ($include[name()=('concept', 'exception')][@code = $newc/exclude[@codeSystem=$codeSystemOid]/@code]) then ()
                    else
                    if ($include[name()=('concept', 'exception')][@code]) then (
                        let $concepts         := $include/@code
                        let $conceptSelect    := if ($concepts) then '[@code=(''' || string-join($concepts,''',''') || ''')]' else()
                        let $ieconcept        := ($include/@code | $compose/exclude[@codeSystem=$codeSystemOid]/@code)
                        let $ieconceptSelect  := if ($ieconcept) then '[@code=(''' || string-join($ieconcept,''',''') || ''')]' else()
                        return
                            (
                                attribute exception { $isException }
                                ,
                                attribute { 'q' } {
                                    if ($include/@inode) then 
                                        $include/@inode || '//concept[@codeSystem=''' || $codeSystemOid || ''']' || $conceptSelect
                                    else (
                                        'collection(''' || $collectionPath || ''')//concept' || $conceptSelect || string-join($exclPredicates,'')
                                    )
                                },
                                (:
                                    query to count any code in include or exclude construct that is NOT active, for warnings on incomplete expansions.
                                    For "stable" CADTS codesystems only "draft", "active", "retired" and "experimental" are valid values anyway,
                                    add query for all nonactive ie not "retired"
                                :)
                                attribute { 'nonactive' } {
                                    if ($include/@inode) then 
                                        $include/@inode || '//concept[@statusCode=''' || 'retired' || '''][@codeSystem=''' || $codeSystemOid || ''']' || $conceptSelect
                                    else (
                                        'collection(''' || $collectionPath || ''')//concept[@statusCode=''' || 'retired' || ''']' || $conceptSelect
                                    )
                                }
                            )
                    )
                    (: includes with no @code and no @op but @codeSystem, eg <include codeSystem="1.2.276.0.76.5.533" flexibility="2022-04-01T00:00:00" displayName="KDL"/> or with filter
                       <include codeSystem="1.2.276.0.76.5.533" flexibility="2022-04-01T00:00:00" displayName="KDL">
                         <filter property="child" op="exist" value="false"/>
                       </include>
                    :)
                    else if ($include[name()='include'][empty(@op | @code)][@codeSystem]) then (
                        (: filters :)
                        let $predicates := utiltcs:getPredicatesForIncludeExclude($include)
                        return (
                            attribute exception { $isException }
                            ,
                            attribute { 'q' } { 'collection(''' || $collectionPath || ''')//concept' || string-join($predicates,'') },
                            (:
                                query to count any code in include or exclude construct that is NOT active, for warnings on incomplete expansions.
                                For "stable" CADTS codesystems only "draft", "active", "retired" and "experimental" are valid values anyway,
                                add query for all nonactive ie not "retired"
                            :)
                            attribute { 'nonactive' } { 'collection(''' || $collectionPath || ''')//concept' || string-join($predicates,'') || '[@statusCode=''' || 'retired' || ''']' }
                        )
                    (: includes with @op, @code and @codeSystem, eg <include op="is-a" code="49062001" codeSystem="2.16.840.1.113883.6.96" displayName="Device (physical object)"/> :)
                    )
                    else (
                        (: filters :)
                        let $predicates       := utiltcs:getPredicatesForIncludeExclude($include)
                        let $ieconcept        := ($include/@code | $compose/exclude[@codeSystem=$codeSystemOid]/@code)
                        let $ieconceptSelect  := if ($ieconcept) then '[@code=(''' || string-join($ieconcept,''',''') || ''')]' else()
                        return (
                            attribute exception { $isException }
                            ,
                            attribute { 'q' } { 
                                'collection(''' || $collectionPath || ''')//concept' || string-join($predicates,'') || string-join($exclPredicates,'')
                            },
                            (:  query to count any code in include or exclude construct that is NOT active, for warnings on incomplete expansions.
                                For "stable" CADTS codesystems only "draft", "active", "retired" and "experimental" are valid values anyway,
                                add query for all nonactive ie not "retired"
                            :)
                            attribute { 'nonactive' } { 
                                'collection(''' || $collectionPath || ''')//concept[@statusCode=''' || 'retired' || ''']' || $ieconceptSelect }
                        )
                    )
                }
                </query>
        )
    return
        <expansion>
        {
            $queries[@q],
            $errors
        }
        </expansion>
};

(: return value set with required ref and optional flexibility (default: DYNAMIC) in raw format :)
declare %private function utiltcs:getValueSetForExpansion($ref as xs:string, $flexibility as xs:string?) as element(valueSet) {
    let $valueSets      := utillib:getValueSetByRef($ref, ($flexibility[string-length() gt 0], 'dynamic')[1])
    return
        switch (count($valueSets))
        case 0 return 
            error($errors:BAD_REQUEST, 'Cannot expand. Missing valueSet (inclusion) ''' || $ref || ''' ''' || ($flexibility[. castable as xs:dateTime], 'dynamic')[1] || '''')
        case 1 return
            utiltcs:getValueSetForExpansion($valueSets)
        default return
            error($errors:SERVER_ERROR, 'Cannot expand. Multiple (included) valueSets (' || count($valueSets) || ') found  for ''' || $ref || ''' ''' || ($flexibility[. castable as xs:dateTime], 'dynamic')[1] || '''. See your admin for solving this')
};
declare %private function utiltcs:getValueSetForExpansion($valueSet as element(valueSet)) as element(valueSet) {
    <valueSet>
    {
        $valueSet/(@* except @inode),
        if ($valueSet/@ident) then () else (
            attribute ident {($valueSet/ancestor::decor/project/@prefix)[1]}
        ),
        if ($valueSet/@url) then () else if ($valueSet/ancestor::*/@bbrurl) then
            attribute url   {($valueSet/ancestor::cacheme/@bbrurl)[1]}
        else (),
        attribute inode {
            concat(
                'collection(''', util:collection-name($valueSet), ''')//terminology/valueSet[@id=''', $valueSet/@id, '''][@effectiveDate=''', $valueSet/@effectiveDate, ''']'
            )
        },
        $valueSet/*
    }
    </valueSet>
};

(: function to expose all expanded concepts found, with the important attributes per concept, and all concept children :)
declare %private function utiltcs:cropc ($c as element()*) {
    for $cc in $c
    let $elmname  := $cc/../@type 
    let $csid     := $cc/../@codeSystem
    let $dp       := ($cc/designation[@use = 'fsn'], $cc/designation[@use = 'pref'], $cc/@displayName)[1]
    return
        element {$elmname}
        {
            $cc/@code,
            $csid,
            attribute displayName {$dp},
            attribute level {($cc/@level, 0)[1]},
            attribute type {($cc/@type, 'L')[1]},
            $cc/designation,
            $cc/desc
        }
};

(: resolve include/@ref value sets: all refs are replaced by their corresponding concepts, recursively :)
declare %private function utiltcs:resolveIncludeRefs($cp as element()*, $inode as xs:string, $sofar as element(valueSet)*) {
    for $c in $cp/*
    return
    if ($c[name() = 'include'][@ref]) then (
        let $ref  := $c/@ref
        let $flx  := $c/@flexibility
        let $xvs  := $sofar[@ref = $ref][@flexibility = string($flx)]
        
        let $xvs2 := if (empty($xvs)) then utiltcs:getValueSetForExpansion($ref, $flx) else ()
        return 
            if ($xvs)         then  <error type="valueset-recursion" ref="{$ref}" flexibility="{$flx}"/> else
            if (empty($xvs2)) then  <error type="valueset-not-found" ref="{$ref}" flexibility="{$flx}"/>
            else (
                utiltcs:resolveIncludeRefs($xvs2/conceptList, $xvs2/@inode, $sofar | <valueSet ref="{$ref}" flexibility="{$flx}"/>)
            )
    )
    else
        element { name($c) } {
            $c/@*,
            if (string-length($inode) > 0 ) then attribute { 'inode' } { $inode } else (),
            $c/*
        }
};

(: function to get all predicates for include and exclude ops :)
declare %private function utiltcs:getPredicatesForIncludeExclude($include as element()*) as xs:string* {
    let $doExclude  := $include/name() = 'exclude'
(:
    get predicates for following constructs
    include op code codeSystem
        with op = equal | is-a | descendent-of
        => results in [@code=, [ancSlf=, [ancestor=
    completeCodeSystem without filter => ()
    completeCodeSystem with filter property op value
        with property = child | parent (so far, any property of the code system in the future)
        with op = exist (so far, any valid op in the future)
        with value = true | false (so far, any valid string in the future)
        
        <property code="X">
            <valueString value="B"/>
        </property>
:)
(: <include codeSystem="2.16.840.1.113883.5.22" codeSystemName="v3-CodeSystem" flexibility="2019-03-20T00:00:00"/> :)
(: collection('/db/apps/terminology-data/codesystem-stable-data/external/hl7/CodeSystem')//concept[include--codeSystem-codeSystemName-flexibility-not-supported] :)
    for $filter in $include
    return (
        if ($filter/@op='equal' or $filter[empty(@op)][@code]) then
            if ($doExclude) then
                concat('[not(@code=(''',string-join($filter/@code,''','''),'''))]')
            else (
                concat('[@code=(''',string-join($filter/@code,''','''),''')]')
            )
        else
        if ($filter/@op='is-a') then
            if ($doExclude) then
                concat('[not(ancSlf=''',$filter/@code,''')]')
            else (
                concat('[ancSlf=''',$filter/@code,''']')
            )
        else
        if ($filter/@op='descendent-of') then
            if ($doExclude) then
                concat('[not(ancestor=''',$filter/@code,''')]')
            else (
                concat('[ancestor=''',$filter/@code,''']')
            )
        else
        if ($filter/@op='in') then
            if ($doExclude) then
                concat('[not(refsets/refset/@refsetId=''',$filter/@code,''')]')
            else (
                concat('[refsets/refset/@refsetId=''',$filter/@code,''']')
            )
        else
        if ($filter[empty(@op | @code)][@codeSystem]) then
            ()
        else
        if ($filter[filter]) then 
            ()
        else (
            '['|| $filter/name() ||'-' || $filter/@op || '-' || string-join(for $att in $filter/(@* except @op) order by name($att) return name($att), '-') || '-not-supported]'
        )
        ,
        for $ccsf in $filter/filter
        let $ccc := concat($ccsf/@property , '-', $ccsf/@op)
        return
            if ($ccc = 'child-exist' and $ccsf/@value = 'true') then
                if ($doExclude) then '[not(child)]' else '[child]'
            else
            if ($ccc = 'child-exist' and $ccsf/@value = 'false') then
                if ($doExclude) then '[child]' else '[not(child)]'
            else
            if ($ccc = 'parent-exist' and $ccsf/@value = 'true') then
                if ($doExclude) then '[not(parent)]' else '[parent]'
            else
            if ($ccc = 'parent-exist' and $ccsf/@value = 'false') then
                if ($doExclude) then '[parent]' else '[not(parent)]'
            else
            if ($ccsf/@op = 'equal') then
                if ($doExclude) then
                    concat('[not(property[@code=''',$ccsf/@property,'''][valueString/@value=''',$ccsf/@value,'''])]')
                else (
                    concat('[property[@code=''',$ccsf/@property,'''][valueString/@value=''',$ccsf/@value,''']]')
                )
            else ()
    )
};