xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Server API allows read, create, update of ART-DECOR Server properties :)
module namespace oidregistry        = "http://art-decor.org/ns/api/oidregistry";

import module namespace errors      = "http://e-editiones.org/roaster/errors";
import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";

declare namespace json              = "http://www.json.org";

declare function oidregistry:getRegistry($request as map(*)) {
    let $allRegistries              :=
        for $registry in collection($setlib:strOidsData)/myoidregistry
        let $displayName := ($registry/registry/name/@value)[1]
        order by lower-case($displayName)
        return 
            <registry name="{$registry/@name}" displayName="{$displayName}" json:array="true"/>
    
    let $count                      := count($allRegistries)
    
    return
    <list xmlns:json="http://www.json.org" current="{$count}" total="{$count}" all="{$count}" lastModifiedDate="{current-dateTime()}">
    {
        $allRegistries
    }
    </list>
};

declare function oidregistry:getOid($request as map(*)) {
    (: TODO In the HTML generated by the original script OIDIndex.xquery, the <input> with id "registry" has name "prefix".
       I think that retrieving the "registry" will always give no result. :)
    let $parameters       := $request?parameters

    let $searchRegistry   := (  $parameters?prefix[. ne ''][1],
                                $parameters?registry[. ne ''][1]
                             )[1]
    let $searchId         := $parameters?id[. ne ''][1]
    let $searchName       := $parameters?name[. ne ''] => lower-case() => replace('-',' ') => tokenize('\s')
    let $searchAuthority  := $parameters?responsibleAuthority[. ne ''][1] => lower-case() => tokenize('\s')
    let $hasFhirProperty  := $parameters?hasFhirProperty
    let $effectiveDate    := $parameters?effectiveDate[. ne ''][1]
    
    let $maxResults       := 100 (: TODO maintain as constant? :)
    let $oidList          := oidregistry:getOidList($searchRegistry, $searchId, $searchName, $searchAuthority, $hasFhirProperty, $effectiveDate)
    
    (: TODO we need/want a root element for the XML result; the schema iso-13582-2015.xsd defines the element <registry>
       for that, but it also requires other elements preceding the sequence of <oid> elements.
       The current elements in the example below have been copied from file art-decor-backend/utilities/hl7nl-oid-registry/hl7nl-oids.xml, but
       that is of course not appropriate.
       Maybe just have a dummy root element, such as <result>, and accept invalidity?
       Or maybe use this dummy XML in a function that is part of a validation API call? 
    :)
    
    let $total            := count($oidList)
    (:
        <registry>
            <validTime><low value="20040308"/></validTime>
            <scopedOID value="2.16.840.1.113883.2.4"/>
            <name value="The HL7 Netherlands Registry"/>
            <description language="en-US" mediaType="text/plain" value="This is the HL7 Netherlands registry"/>
            <description language="nl-NL" mediaType="text/plain" value="Dit is het HL7 Nederland register"/>
            <person><name><part type="GIV" value="Helen"/><part type="FAM" value="Drijfhout-Wisse"/></name></person>
            <hostingOrganization><name><part value="ART-DECOR"/></name></hostingOrganization>
            {  subsequence($oidList, 1, $maxResults) } 
        </registry>
    :)
    let $result           := subsequence($oidList, 1, $maxResults)
        
    return
        <list xmlns:json="http://www.json.org" current="{count($result)}" total="{$total}" all="{$total}" lastModifiedDate="{current-dateTime()}">
        {for $term in $searchName return <wildcard>{$term}</wildcard>}
        {
            utillib:addJsonArrayToElements($result)
        }
        </list>
        
};

declare %private function oidregistry:isWithinDateRange($input as xs:string,$lowBoundary as xs:string?,$highBoundary as xs:string?) as xs:boolean {
    let $i := number(replace($input,'^\d',''))
    let $l := number(replace($lowBoundary,'^\d',''))
    let $h := number(replace($highBoundary,'^\d',''))
    return
    if (empty($input) or string-length($i) < 8) then (
        true()
    ) else if (empty($lowBoundary) and empty($highBoundary)) then (
        true()
    ) else if (not(empty($lowBoundary)) and empty($highBoundary) and $i >= $l) then (
        true()
    ) else if (not(empty($highBoundary)) and empty($lowBoundary) and $i <= $h) then (
        true()
    ) else if (not(empty($lowBoundary) or empty($highBoundary)) and $i >= $l and $i <= $h) then (
        true()
    ) else (
        false()
    )
};

(: Sample of oid XML format:
    <oid>
        <dotNotation value="2.16.840.1.113883"/>
        <symbolicName value="hL7"/>
        <category code="LNS"/>
        <status code="completed"/>
        <creationDate nullFlavor="NI"/>
        <realm code="UV"/>
        <description language="en-US" mediaType="text/plain" value="Root of Health Level Seven's Object Identifier subtree."/>
        <registrationAuthority>
            <code code="PRI"/>
            <scopingOrganization><name><part value="HL7 International"/></name></scopingOrganization>
        </registrationAuthority>
        <responsibleAuthority>
            <code code="PRI"/>
            <statusCode code="active"/>
            <validTime nullFlavor="NI"/>
            <scopingOrganization><name><part value="HL7 International"/></name></scopingOrganization>
        </responsibleAuthority>
        <additionalProperty>
            <attribute value="Oid_Type"/>
            <value value="0"/>
        </additionalProperty>
        <additionalProperty>
            <attribute value="External_OID_flag"/>
            <value value="int"/>
        </additionalProperty>
    </oid>
:)

(:~ Retrieve a list with OIDs in a format that, apart from root element structure, in its XML form, validates with iso-13582-2015.xsd :)
declare %private function oidregistry:getOidList($searchRegistry as xs:string*, $searchId as xs:string?, $searchName as xs:string*, $searchAuthority as xs:string*, $hasFhirProperty as xs:boolean?, $effectiveDate as xs:string?) as element(oid)* {
    
    let $queryName                  := if (empty($searchName)) then () else utillib:getSimpleLuceneQuery($searchName, 'wildcard')
    let $queryAuthority             := if (empty($searchAuthority)) then () else utillib:getSimpleLuceneQuery($searchAuthority, 'wildcard')
    
    let $q                          :=
        if (empty($searchRegistry)) then
            'collection($setlib:strOidsData)/myoidregistry//oid'
        else (
            'collection($setlib:strOidsData)/myoidregistry[@name = $searchRegistry]//oid'
        )
   let $q                           := 
        if (empty($searchId)) then (
            $q
        )
        else (
            $q || '[dotNotation[@value = $searchId]] | ' || $q || '[dotNotation[starts-with(@value, concat($searchId, ''.''))]]'
        )
    let $q                           := 
        if (empty($searchName[string-length()>0])) then (
            $q
        )
        else (
            $q || '[ft:query(description/@value, $queryName, utillib:getSimpleLuceneOptions())] | ' || $q || '[ft:query(symbolicName/@value, $queryName, utillib:getSimpleLuceneOptions())]' 
        )
    let $q                           := 
        if (empty($searchAuthority[string-length()>0])) then (
            $q
        )
        else (
            $q || '[ft:query(responsibleAuthority/scopingOrganization/name/part/@value, $queryAuthority, utillib:getSimpleLuceneOptions())]'
        )
    let $q                           :=
        if (empty($hasFhirProperty)) then
            (: give me all :)
            $q
        else
        if ($hasFhirProperty) then
            (: give me all with property FHIR :)
            $q || '[additionalProperty/attribute[contains(@value, ''FHIR'')]]'
        else (
            (: give me all except with property FHIR :)
            $q || '[not(additionalProperty/attribute[contains(@value, ''FHIR'')])]'
        )
    let $q                           :=
        if (empty($effectiveDate)) then (
            util:eval($q)
        ) else (
            for $oid in util:eval($q)
            let $isWithinRange := oidregistry:isWithinDateRange($effectiveDate, $oid/responsibleAuthority/validTime/low/@value, $oid/responsibleAuthority/validTime/high/@value)
            return if ($isWithinRange) then $oid else ()
        )
    
    return $q
};
