xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:~ Value set API allows read, create, update of DECOR valueSets :)
module namespace vsapi              = "http://art-decor.org/ns/api/valueset";

import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace utillib     = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace utilvs      = "http://art-decor.org/ns/api/util-valueset" at "library/util-valueset-lib.xqm";

import module namespace utilhtml    = "http://art-decor.org/ns/api/util-html" at "library/util-html-lib.xqm";
import module namespace utilfsh     = "http://art-decor.org/ns/api/util-fsh" at "library/util-fsh-lib.xqm";
import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib    = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";
import module namespace utilgg      = "http://art-decor.org/ns/api/util-governancegroup" at "library/util-governancegroup-lib.xqm";
import module namespace histlib     = "http://art-decor.org/ns/api/history" at "library/history-lib.xqm";

declare namespace json              = "http://www.json.org";

(:~ Retrieves list of all DECOR valueSets of a project denoted by project prefix or oid for publication
    @param $project                 - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $language                - optional
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @return as-is or as compiled as JSON
    @since 2024-11-15
:)
declare function vsapi:getValueSetExtractList($request as map(*)) { 

    let $project                := $request?parameters?project[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $format                 := $request?parameters?format[not(. = '')]
           
    (: parameter format overrides accept-header as mime-type for using the api in the browser :)
    
    (: 
        this call is used in backend and frontend (browser) - default behaviour mime-types:
        - backend:   default mime-type is json - response payload is json - if no accept type is given and format is not csv or sql  
        - frontend:  default mime-type is xml - response payload is xml - format is not csv or sql 
    :)
    
    let $acceptTypes            := roaster:accepted-content-types()
    let $acceptedType           := ($acceptTypes[. = ('application/xml', 'application/json', 'text/csv', 'text/sql')],'application/json')[1]
    
    let $format                 := if ($format) then $format else tokenize($acceptedType, '/')[2]
    let $isXml                  := $format = ('xml', 'svs', 'mdibagch', 'svsold')

    (: 
       overwrite format in the backend is not always possible because of middleware behaviour
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
       - if in backend the format is not json and accept is application/json, roaster always gives a json payload
    :) 
    let $check                  :=
        if ($isXml and not($acceptedType = 'application/xml')) then 
            error($errors:BAD_REQUEST, 'In case of format parameter is xml or svs(old) or mdibagch the accept header should be application/xml')
        else if (not($format = 'json') and $acceptedType = 'application/json') then
            error($errors:BAD_REQUEST, 'In case of format parameter is ' || $format || ' the accept header can be anything but application/json')
        else ()   

    let $check                  :=
        if (empty($project)) then 
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    
    let $projectPrefix          := utillib:getDecor($project, $projectVersion, $projectLanguage)/project/@prefix
   
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Project ' || $project || ' not found.')
        else ()
    
    (: Return zero or more expanded valuesets wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element :)
    let $valueSets              := 
        for $valueSet in utilvs:getValueSetList((), $projectPrefix, $projectVersion, $projectLanguage, (), (), false(), (), (), (), false())/valueSet/valueSet
        return utillib:getValueSetExtract($projectPrefix, $projectVersion, $projectLanguage, $valueSet/(@id|@ref), $valueSet/@effectiveDate, false())

    (: prepare response payload :)
    let $results                :=
         if (empty($valueSets/*)) then () 
         else if($format = ('xml', 'json')) then (
            let $xmljson        := <valueSets>{$valueSets/*}</valueSets>                   
            let $xmljson        := 
            for $x in $xmljson
                return
                element {name($x)} {
                    $x/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($x/node(), $format)
                }
             (: in this case the json format overrides header application/xml and works with a text header :)
             return if ($format = 'json' and not($acceptedType = 'application/json')) then fn:serialize($xmljson, map{"method": $format , "indent": true()}) else $xmljson
         )   
         else if ($format = ('svs', 'mdibagch')) then utilvs:convertValueSet2Svs($valueSets, $projectLanguage, $projectVersion, $format) 
         else if ($format = 'csv') then utilvs:convertValueSet2Csv($valueSets, $projectLanguage, $projectPrefix)
         else if ($format = 'sql') then fn:string(transform:transform($valueSets, doc($setlib:strDecorServices ||'/resources/stylesheets/ToSql4ValueSets.xsl'), ()))
         else error($errors:BAD_REQUEST, 'Requested mime-type ' || $format || ' is not supported')
    
    (: prepare response header content-type and content-disposition :)
    let $contentType            :=
        if ($isXml) then 'application/xml'
        else if ($format = ('csv', 'sql')) then 'text/' || $format
        else 'application/' || $format
                
    let $fileExt                := if ($isXml) then 'xml' else $format
    let $valueSetNames          := distinct-values($valueSets//valueSet/@name)
    let $fileNamePart           := if (count($valueSetNames)>1) then 'project_' || $projectPrefix || '('|| count($valueSets//valueSet) || ')' else $valueSetNames
    let $fileName               := 'VS_' || $fileNamePart ||'_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $fileExt
        
    let $r-header               := 
            (response:set-header('Content-Type', $contentType || '; charset=utf-8'),
            response:set-header('Content-Disposition', 'filename='|| $fileName))
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else $results

};

(:~ Retrieves latest DECOR valueSet for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $projectPrefix           - optional limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $language                - optional
    @param $format                  - optional overrides the accept-header for frontend purposes
    @return as-is or as compiled as JSON
    @since 2023-11-14
:)
declare function vsapi:getLatestValueSetExtract($request as map(*)) {
    vsapi:getValueSetExtract($request)
};

(:~ Retrieves one or more DECOR valueSet for publication based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the valueSet. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $project                 - optional. limits scope to this project only
    @param $language                - optional
    @param $wrap                    - optional parameter to return the valuesets wrapped in repository and with project data - xml only
    @param $format                  - optional. overrides the accept-header for frontend purposes
    @return as-is or as compiled as JSON
    @since 2023-11-14
:)
declare function vsapi:getValueSetExtract($request as map(*)) { 

    let $project                := $request?parameters?project[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $withversions           := $request?parameters?versions = true()
    let $format                 := $request?parameters?format[not(. = '')]
    
    let $idOrName               := $request?parameters?idOrName[not(. = '')]
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
           
    (: parameter format overrides accept-header as mime-type for using the api in the browser :)
    
    (: 
        this call is used in backend and frontend (browser) - default behaviour mime-types:
        - backend:   default mime-type is json - response payload is json - if no accept type is given and format is not csv or sql  
        - frontend:  default mime-type is xml - response payload is xml - format is not csv or sql 
    :)
    
    let $acceptTypes            := roaster:accepted-content-types()
    let $acceptedType           := ($acceptTypes[. = ('application/xml', 'application/json', 'text/csv', 'text/sql')],'application/json')[1]
    
    let $format                 := if ($format) then $format else tokenize($acceptedType, '/')[2]
    let $isXml                  := $format = ('xml', 'svs', 'mdibagch', 'svsold') 

    (: 
       overwrite format in the backend is not always possible because of middleware behaviour
       - if in backend the format is xml and accept is not application/xml, roaster always gives a json payload 
       - if in backend the format is not json and accept is application/json, roaster always gives a json payload
    :) 
    let $check                  :=
        if ($isXml and not($acceptedType = 'application/xml')) then 
            error($errors:BAD_REQUEST, 'In case of format parameter is xml or svs(old) or mdibagch the accept header should be application/xml')
        else if (not($format = 'json') and $acceptedType = 'application/json') then
            error($errors:BAD_REQUEST, 'In case of format parameter is ' || $format || ' the accept header can be anything but application/json')
        else ()   

    let $projectPrefix          := if (empty($project)) then () else utillib:getDecor($project, $projectVersion, $projectLanguage)/project/@prefix 
    
    (: Return zero or more expanded valuesets wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element :)
    let $valueSets              := if (empty($projectPrefix)) then utillib:getValueSetExtract($idOrName, $effectiveDate) else utillib:getValueSetExtract($projectPrefix, $projectVersion, $projectLanguage, $idOrName, $effectiveDate, $withversions)

    (: prepare response payload :)
    let $results                :=
         if (empty($valueSets/*)) then () 
         else if($format = ('xml', 'json')) then (
            let $xmljson        := <valueSets>{$valueSets/*}</valueSets>                   
            let $xmljson        := 
            for $x in $xmljson
                return
                element {name($x)} {
                    $x/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($x/node(), $format)
                }
             (: in this case the json format overrides header application/xml and works with a text header :)
             return if ($format = 'json' and not($acceptedType = 'application/json')) then fn:serialize($xmljson, map{"method": $format , "indent": true()}) else $xmljson
         )   
         else if ($format = ('svs', 'mdibagch')) then utilvs:convertValueSet2Svs($valueSets, $projectLanguage, $projectVersion, $format) 
         else if ($format = 'svsold') then utilvs:convertValueSet2Svsold($valueSets, $projectLanguage) 
         else if ($format = 'csv') then utilvs:convertValueSet2Csv($valueSets, $projectLanguage, $projectPrefix)
         else if ($format = 'fsh') then utilfsh:convertValueSets2Fsh($valueSets, $projectLanguage, $projectPrefix)
         else if ($format = 'sql') then fn:string(transform:transform($valueSets, doc($setlib:strDecorServices ||'/resources/stylesheets/ToSql4ValueSets.xsl'), ()))
         else error($errors:BAD_REQUEST, 'Requested mime-type ' || $format || ' is not supported')
    
    (: prepare response header content-type and content-disposition :)
    let $contentType            :=
        if ($isXml) then 'application/xml'
        else if ($format = ('csv', 'sql')) then 'text/' || $format
        else 'application/' || $format
            
    let $fileExt                := if ($isXml) then 'xml' else $format
    let $fileName               := 'VS_' || fn:distinct-values($valueSets//valueSet/@name) ||'_(download_' || substring(fn:string(current-dateTime()),1,19) || ').' || $fileExt
        
    let $r-header               := 
        (response:set-header('Content-Type', $contentType || '; charset=utf-8'),
        response:set-header('Content-Disposition', 'filename='|| $fileName))
    
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if ($format = 'fsh') then (
            $results/text()
        )
        else $results

};

(:~ Retrieves DECOR valueSet view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the valueSet. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language, also ui-language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2024-10-14
:)
declare function vsapi:getValueSetViewList($request as map(*)) {

    let $project                := $request?parameters?project[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $inline                 := $request?parameters?inline = true()
    let $collapsable            := not($request?parameters?collapsable = false())
   
    let $check                  :=
        if (empty($project)) then 
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    
    let $decor                  := utillib:getDecor($project, $projectVersion, $projectLanguage)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Project ' || $project || ' not found.')
        else ()
    
    (: Return zero or more expanded valuesets wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element :)
    let $results                      := 
        for $valueSet in utilvs:getValueSetList((), $projectPrefix, $projectVersion, $projectLanguage, (), (), false(), (), (), (), false())/valueSet/valueSet
        return utillib:getValueSetExtract($projectPrefix, $projectVersion, $projectLanguage, $valueSet/(@id|@ref), $valueSet/@effectiveDate, false())

    let $valueSetNames              := distinct-values($results//valueSet/@name)
    let $fileNamePart               := if (count($valueSetNames)>1) then 'project_' || $projectPrefix || '('|| count($results//valueSet) || ')' else $valueSetNames
    let $fileName                   := 'VS_' || $fileNamePart ||'_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html'
            
    let $r-header                       := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        response:set-header('Content-Disposition', 'filename='|| $fileName))
    
    return
        if (empty($results/*)) then (
            roaster:response(404, ())
        )
        else 
            (: prepare for Html :)
            let $language                       := if (empty($projectLanguage)) then $setlib:strArtLanguage else $projectLanguage
            let $header                         := if ($inline) then false() else true()        
            
            return utilhtml:convertObject2Html($results, $language, $header, $collapsable, $projectVersion, $decor)

};

(:~ Retrieves latest DECOR valueSet view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language, also ui-language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2025-03-07
:)
declare function vsapi:getLatestValueSetView($request as map(*)) {
    vsapi:getValueSetView($request)
};

(:~ Retrieves DECOR valueSet view based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the valueSet. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language, also ui-language
    @param $inline                  - optional parameter to omit HTML header info. Useful for inclusion of HTML in other pages.
    @param $collapsable             - optional parameter the valueset is collapable.
    @return as-is 
    @since 2024-10-14
:)
declare function vsapi:getValueSetView($request as map(*)) {

    let $project                := $request?parameters?project[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $inline                 := $request?parameters?inline = true()
    let $collapsable            := not($request?parameters?collapsable = false())
    
    let $idOrName               := $request?parameters?idOrName[not(. = '')]
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $decor                  := if (empty($project)) then () else utillib:getDecor($project, $projectVersion, $projectLanguage)
            
    (: Return zero or more expanded valuesets wrapped in a &lt;return/&gt; element, and subsequently inside a &lt;repository&gt; element :)
    let $results                := if (empty($decor/project/@prefix)) then utillib:getValueSetExtract($idOrName, $effectiveDate) else utillib:getValueSetExtract($decor/project/@prefix, $projectVersion, $projectLanguage, $idOrName, $effectiveDate, false()) 

    let $fileName               := 'VS_' || fn:distinct-values($results//valueSet/@name) ||'_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html'
            
    let $r-header               := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        response:set-header('Content-Disposition', 'filename='|| $fileName))
    
    return
        if (empty($results/*)) then (
            roaster:response(404, ())
        )
        else 
            (: prepare for Html :)
            let $language                       := if (empty($projectLanguage)) then $setlib:strArtLanguage else $projectLanguage
            let $header                         := if ($inline) then false() else true()        
            
            return utilhtml:convertObject2Html($results, $language, $header, $collapsable, $projectVersion, $decor)
};

(:~ Retrieves latest DECOR valueSet based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function vsapi:getLatestValueSet($request as map(*)) {
    vsapi:getValueSet($request)
};

(:~ Retrieves DECOR valueSet based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the valueSet. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function vsapi:getValueSet($request as map(*)) {
    
    let $projectPrefix          := $request?parameters?prefix
    let $projectVersion         := $request?parameters?release
    let $projectLanguage        := $request?parameters?language
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    (: only applicable when no effective date is given, if true all versions are given :)
    let $withversions           := $request?parameters?versions = true()
    
    let $results                := utillib:getValueSet($projectPrefix, $projectVersion, $projectLanguage, $id, $effectiveDate, $withversions, true())
    let $results                        := 
        if ($withversions) then 
            <list artifact="VS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">{$results}</list> 
        else (
            head($results)
        )
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple valueSets for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Retrieves DECOR valueSet history based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id
    @param $effectiveDate           - optional parameter denoting the effectiveDate. If not given assumes latest version for id
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function vsapi:getValueSetHistory($request as map(*)) {
    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $projectPrefix          := ()
    let $results                := histlib:ListHistory($authmap, $decorlib:OBJECTTYPE-VALUESET, (), $id, $effectiveDate, 0)
    
    return
        <list artifact="{$decorlib:OBJECTTYPE-VALUESET}" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}">
        {
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Retrieves DECOR valueSet usage based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the valueSet
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the valueSet. If not given assumes latest version for id
    @param $projectPrefix           - optional. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function vsapi:getValueSetUsage($request as map(*)) {
    
    let $projectPrefix          := $request?parameters?prefix
    let $projectVersion         := $request?parameters?release
    let $projectLanguage        := $request?parameters?language
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $results                := utilvs:getValueSetUsage($id, $effectiveDate, $projectPrefix, $projectVersion, $projectLanguage)
    
    let $count                  := count($results)
    let $max                    := $count
    let $allcnt                 := $count
    
    return
        <list artifact="USAGE" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements(subsequence($results, 1, $max))
        }
        </list>
};

(:~ Returns a list of zero or more valuesets
    @param $governanceGroupId - optional. determines search scope. null is full server, id limits scope to this projects under this governance group only. Does not mix with $projectPrefix
    @param $projectPrefix    - optional. determines search scope. null is full server, pfx- limits scope to this project only. Does not mix with $governanceGroupId
    @param $projectVersion   - optional. if empty defaults to current version. if valued then the valueset will come explicitly from that archived project version which is expected to be a compiled version
    @param $projectLanguage  - optional. relevant in combination with $projectVersion if there are more than one languages the project is compiled in
    @param $id               - optional. Identifier of the valueset to retrieve
    @param $name             - optional. Name of the valueset to retrieve (valueSet/@name)
    @param $flexibility      - optional. null gets all versions, 'dynamic' gets the newest version based on id or name, yyyy-mm-ddThh:mm:ss gets this specific version
    @return Zero value sets in case no matches are found, one if only one exists or if a specific version was requested, or more if more versions exist and no specific version was requested
    @since 2013-06-14
:)
declare function vsapi:getValueSetList($request as map(*)) {

    let $governanceGroupId      := $request?parameters?governanceGroupId[not(. = '')]
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $projectVersion         := $request?parameters?release[not(. = '')]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $max                    := $request?parameters?max
    let $resolve                :=  if (empty($governanceGroupId)) then not($request?parameters?resolve = false()) else $request?parameters?resolve = true()
    let $searchTerms            := 
        array:flatten(
            for $s in ($request?parameters?search, $request?parameters?id, $request?parameters?name)[string-length() gt 0]
            return tokenize(normalize-space(lower-case($s)),'\s')
        )
    let $ed                     := $request?parameters?effectiveDate[not(. = '')]
    let $includebbr             := $request?parameters?includebbr = true()
    let $sort                   := $request?parameters?sort[string-length() gt 0]
    let $sortorder              := $request?parameters?sortorder[. = 'descending']
    
    let $check                  :=
        if (empty($governanceGroupId) and empty($projectPrefix)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have either parameter governanceGroupId or prefix')
        else 
        if (not(empty($governanceGroupId)) and not(empty($projectPrefix))) then 
            error($errors:BAD_REQUEST, 'Request SHALL have parameter governanceGroupId or prefix, not both')
        else ()
    let $check                  :=
        if (empty($governanceGroupId)) then () else if ($resolve) then
            error($errors:BAD_REQUEST, 'Request SHALL NOT have governance group scope and resolve=true. This is too expensive to support')
        else ()
    
    let $startT                 := util:system-time()
    
    let $result                 :=
        if (empty($governanceGroupId)) then
            utilvs:getValueSetList((), $projectPrefix, $projectVersion, $projectLanguage, $searchTerms, $ed, $includebbr, $sort, $sortorder, $max, $resolve)
        else (
            for $projectId in utilgg:getLinkedProjects($governanceGroupId)/@ref
            return utilvs:getValueSetList($governanceGroupId, $projectId, (), (), $searchTerms, $ed, $includebbr, $sort, $sortorder, $max, $resolve)    
        )
    
    let $durationT              := (util:system-time() - $startT) div xs:dayTimeDuration("PT0.001S")
    
    return
        <list artifact="{$result[1]/@artifact}" elapsed="{$durationT}" current="{sum($result/xs:integer(@current))}" total="{sum($result/xs:integer(@total))}" all="{sum($result/xs:integer(@allcnt))}" resolve="{$resolve}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            if (empty($governanceGroupId)) then attribute project {$projectPrefix} else attribute governanceGroupId {$governanceGroupId},
            utillib:addJsonArrayToElements($result/*)
        }
        </list>
};

(:~ Deletes a valueSet reference based on $id (oid) and project id or prefix
    @param $id                      - required parameter denoting the id of the valueSet
    @param $project                 - required. limits scope to this project only
:)
declare function vsapi:deleteValueSet($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0][not(. = '*')]
    let $id                     := $request?parameters?id[string-length() gt 0]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                  :=
        if (empty($project) or empty($id)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have both parameter project and id')
        else ()
    
    let $decor                  := utillib:getDecor($project, (), ())
    
    let $check                  :=
        if (empty($decor)) then
            error($errors:BAD_REQUEST, 'Project ' || $project || ' not found.')
        else ()
    
    let $check                  := utilvs:checkValueSetAccess($authmap, $decor, (), (), false())
    
    let $delete                 := update delete $decor/terminology/valueSet[@ref = $id]
    
    return
        roaster:response(204, ())
};

(: Create a valueset, either empty or based on another valueSet
    @param $projectPrefix project to create this scenario in
    @param $targetDate If true invokes effectiveDate of the new valueSet and concepts as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
    @param $sourceId parameter denoting the id of a dataset to use as a basis for creating the new valueSet
    @param $sourceEffectiveDate parameter denoting the effectiveDate of a dataset to use as a basis for creating the new valueSet",
    @param $keepIds Only relevant if source dataset is specified. If true, the new valueSet will keep the same ids for the new valueSet, and only update the effectiveDate
    @param $baseDatasetId Only relevant when a source dataset is specified and `keepIds` is false. This overrides the default base id for datasets in the project. The value SHALL match one of the projects base ids for datasets
    @return (empty) dataset object as xml with json:array set on elements
:)
declare function vsapi:postValueSet($request as map(*)) {

    let $authmap                := $request?user
    let $projectPrefix          := $request?parameters?prefix[not(. = '')]
    let $targetDate             := $request?parameters?targetDate = true()
    let $sourceId               := $request?parameters?sourceId
    let $data                   := utillib:getBodyAsXml($request?body, 'valueSet', ())
    let $sourceEffectiveDate    := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?sourceEffectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?sourceEffectiveDate[string-length() gt 0]
        }
    let $refOnly                := $request?parameters?refOnly = true()
    let $keepIds                := $request?parameters?keepIds = true()
    let $baseId                 := $request?parameters?baseId
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                  :=
        if (empty($projectPrefix)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter prefix')
        else ()
    
    let $results                := utilvs:createValueSet($authmap, $projectPrefix, $targetDate, $sourceId, $sourceEffectiveDate, $refOnly, $keepIds, $baseId, $data)
    
    return
        roaster:response(201, 
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        
        )
};

(:~ Update valueSet statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap required. Map derived from token
    @param $id required. DECOR concept/@id to update
    @param $effectiveDate required. DECOR concept/@effectiveDate to update
    @param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates. Not implemented (yet)
    @param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function vsapi:postValueSetStatus($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    (: Note: we _could_ implement traversing into value set inclusions or even attached codesystems through this parameter. In AD2 however we never implemented this and we did not plan for introducing this in AD3 nor was it ever asked of us.
             Hence the provision is made should it ever be needed, but other than that, it is not implemented.
    :)
    let $recurse                := $request?parameters?recurse = true()
    let $list                   := $request?parameters?list = true()
    let $statusCode             := $request?parameters?statusCode
    let $versionLabel           := $request?parameters?versionLabel
    let $expirationDate         := $request?parameters?expirationDate
    let $officialReleaseDate    := $request?parameters?officialReleaseDate
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                := utilvs:setValueSetStatus($authmap, $id, $effectiveDate, $recurse, $list, $statusCode, $versionLabel, $expirationDate, $officialReleaseDate)
    
    return 
        <list artifact="STATUS" current="{count($results)}" total="{count($results)}" all="{count($results)}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            (: max 2 arrays: one with error and one with success :)
            utillib:addJsonArrayToElements($results)
        }
        </list>
};

(:~ Update DECOR valueSet
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the valueSet to update 
    @param $request-body             - required. json body containing new valueSet structure
    @return valueSet structure including generated meta data
    @since 2020-05-03
:)
declare function vsapi:putValueSet($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    let $data                   := utillib:getBodyAsXml($request?body, 'valueSet', ())
    let $deletelock             := $request?parameters?deletelock = true()
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($data)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
    
    let $result                 := utilvs:putValueSet($authmap, $id, $effectiveDate, $data, $deletelock)
    
    return 
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }

};

(:~ Update DECOR valueSet parts. Expect array of parameter objects, each containing RFC 6902 compliant contents. Note: RestXQ does not do PATCH (yet), but that would be the preferred option.

    { "op": "[add|remove|replace]", "path": "e.g. [/statusCode|/expirationDate|/officialReleaseDate|/canonicalUri|/versionLabel|/name|/displayName|/experimental|/desc|/publishingAuthority|/copyright|/completeCodeSystem|/conceptList]", "value": "[string|object]" }
    
    where
    
    * op - add & replace (statusCode, expirationDate, officialReleaseDate, canonicalUri, versionLabel, name, displayName, experimental, desc, publishingAuthority, copyright, completeCodeSystem, conceptList) or remove (desc, publishingAuthority, copyright, completeCodeSystem, conceptList)
    
    * path - see above
    
    * value - string when the path is not /. object when path is /
    
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the valueSet to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return valueSet structure
    @since 2020-05-03
    @see http://tools.ietf.org/html/rfc6902
:)
declare function vsapi:patchValueSet($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $effectiveDate          := 
        try {
            xmldb:decode-uri(xs:anyURI(string($request?parameters?effectiveDate)))[string-length() gt 0]
        }
        catch * {
            $request?parameters?effectiveDate[string-length() gt 0]
        }
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $result                 := utilvs:patchValueSet($authmap, string($id), string($effectiveDate), $data)
    
    return (
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
    )

};

(: ================================================== :)
(: ============== VALUE SET EXPANSIONS ============== :)
(: ================================================== :)

(:~ Retrieves DECOR valueSet expansion based on $id (oid)
    @param $id                      - required parameter denoting the id of the valueSet
    @return as-is or as compiled as JSON
    @since 2023-08-010
:)
declare function vsapi:getValueSetExpansionSet($request as map(*)) {
    
    let $id                     := $request?parameters?id
    
    let $results                := collection($setlib:strValuesetExpansionData)/valueSetExpansionSet[@id = $id]
    return
        if (empty($results)) then (
            roaster:response(404, ())
        )
        else
        if (count($results) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple valueSet expansions for id '", $id, "'. Expected 0..1. Alert your administrator as this should not be possible."))
        )
        else (
            for $result in $results
            return
                element {name($result)} {
                    $result/@*,
                    namespace {"json"} {"http://www.json.org"},
                    utillib:addJsonArrayToElements($result/*)
                }
        )
};

(:~ Update valueSet expansion statusCode, and/or expirationDate
    @param $authmap required. Map derived from token
    @param $id required. valueSetExpansionset/@id
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @return list object with success and/or error elements or error
:)
declare function vsapi:postValueSetExpansionSetStatus($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $statusCode             := $request?parameters?statusCode
    let $expirationDate         := $request?parameters?expirationDate
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $results                := utilvs:setValueSetExpansionSetStatus($authmap, $id, $statusCode, $expirationDate)
    
    return
    <list artifact="STATUS" current="1" total="1" all="1" lastModifiedDate="1" xmlns:json="http://www.json.org">
        {
            utillib:addJsonArrayToElements(
                <success itemcode="VSE" itemname="{local-name($results)}">
                {
                    $results/(@* except (@itemcode | @itemname))
                }
                </success>
            )
        }
        </list>
        
};

(:~ Delete valueSet expansion
    @param $authmap required. Map derived from token
    @param $id required. alueSetExpansionset/@id
    @return nothing
:)
declare function vsapi:deleteValueSetExpansionSet($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    (: don't want to update cache or external expansions :)
    let $check                  :=
        for $expansion in collection($setlib:strValuesetExpansionData)/valueSetExpansionSet[@id = $id]
        return utilvs:checkValueSetExpansionAccess($authmap, $expansion, true())
        
    let $delete                 := 
        for $expansion in collection($setlib:strValuesetExpansionData)/valueSetExpansionSet[@id = $id]
        return xmldb:remove(util:collection-name($expansion), util:document-name($expansion))
    
    return
        roaster:response(204, ())
};

(:~ Update DECOR valueSet expansion set parts. Expect array of parameter objects, each containing RFC 6902 compliant contents.

    { "op": "[add|remove|replace]", "path": "e.g. [/statusCode|/expirationDate]", "value": "[string]" }
    
    where
    
    * op - add & replace (statusCode, expirationDate) or remove (expirationDate)
    
    * path - see above
    
    * value - (empty) string
    
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $id                       - required. the id for the valueSet to update 
    @param $request-body             - required. json body containing array of parameter objects each containing RFC 6902 compliant contents
    @return valueSet expansion set structure
    @since 2023-08-14
    @see http://tools.ietf.org/html/rfc6902
:)
declare function vsapi:patchValueSetExpansionSet($request as map(*)) {

    let $authmap                := $request?user
    let $id                     := $request?parameters?id
    let $data                   := utillib:getBodyAsXml($request?body, 'parameters', ())
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    
    let $check                  :=
        if ($data) then () else (
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        )
    
    let $result                 := utilvs:patchValueSetExpansionSet($authmap, string($id), $data)
    
    return (        
        element {name($result)} {
            $result/@*,
            namespace {"json"} {"http://www.json.org"},
            utillib:addJsonArrayToElements($result/*)
        }
    )
};
