xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilmp                 = "http://art-decor.org/ns/api/util-conceptmap";

import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace decorlib        = "http://art-decor.org/ns/api/decor" at "decor-lib.xqm";
import module namespace histlib         = "http://art-decor.org/ns/api/history" at "history-lib.xqm";
import module namespace ruleslib        = "http://art-decor.org/ns/api/rules" at "rules-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";
declare namespace json                  = "http://www.json.org";

declare %private variable $utilmp:STATUSCODES-FINAL      := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $utilmp:ADDRLINE-TYPE          := utillib:getDecorTypes()/AddressLineType;

(:~ Central logic for retrieving a DECOR conceptMap based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                          - required parameter denoting the id of the conceptMap
    @param $effectiveDate               - optional parameter denoting the effectiveDate of the conceptMap. If not given assumes latest version for id
    @param $projectPrefix               - optional. limits scope to this project only
    @param $projectVersion              - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
(:~ @param $doContents If we only need the conceptmapt for the list we can forego the contents :)
declare function utilmp:getConceptMap($projectPrefix as xs:string*, $projectVersion as xs:string*, $projectLanguage as xs:string?, $id as xs:string, $effectiveDate as xs:string?, $withversions as xs:boolean, $doContents as xs:boolean) as element(conceptMap)* {
    let $id                 := $id[not(. = '')]
    let $effectiveDate      := if ($withversions) then () else ($effectiveDate[not(. = '')], 'dynamic')[1]
    let $projectPrefix      := $projectPrefix[not(. = '')][1]
    let $projectVersion     := $projectVersion[not(. = '')][1]
    let $projectLanguage    := $projectLanguage[not(. = '')][1]
    
    let $results            := if (empty($projectPrefix)) then utilmp:getConceptMapById($id, $effectiveDate) else utilmp:getConceptMapById($id, $effectiveDate, $projectPrefix, $projectVersion, $projectLanguage)
    
    let $results            :=
        if ($results) then $results else (
            if (empty($projectPrefix)) then (
                let $decorSets  := $setlib:colDecorData//conceptMap[@ref = $id]
                let $cacheSets  := $setlib:colDecorCache//conceptMap[@ref = $id]
                return ($decorSets, $cacheSets)
            )
            else utillib:getDecor($projectPrefix, $projectVersion, $projectLanguage)//conceptMap[@ref = $id]
        )[last()]
    
    for $conceptMap in $results
    let $id                 := $conceptMap/@id | $conceptMap/@ref
    let $ed                 := $conceptMap/@effectiveDate
    let $projectPrefix      := ($conceptMap/@ident, $conceptMap/parent::*/@ident, $conceptMap/ancestor::*/@bbrident, $conceptMap/ancestor::decor/project/@prefix)[1]
    order by $id, $ed descending
    let $cm                 := utilmp:getRawConceptMap($conceptMap)
    return
        element {name($cm)} {
            $cm/@*,
            if ($cm/@url) then () else attribute url {($cm/parent::*/@url, $cm/ancestor::*/@bbrurl, $utillib:strDecorServicesURL)[1]},
            if ($cm/@ident) then () else attribute ident {$projectPrefix},
            if ($doContents) then (
                $cm/(* except group),
                (: we return just a count of issues and leave it up to the calling party to get those if required :)
                <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $cm/@id][@effectiveDate = $cm/@effectiveDate] | $setlib:colDecorData//issue/object[@id = $cm/@ref])}"/>,
                $cm/group
            )
            else ()
        }
};

(:~ Central logic for retrieving DECOR concept maps based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                          - required parameter denoting the id of the concept
    @param $effectiveDate               - required parameter denoting the effectiveDate of the concept. If not given assumes latest version for id
    @param $transactionId               - required parameter denoting the id of the transaction that the concept is in
    @param $transactionEffectiveDate    - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion              - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage             - optional parameter to select from a specific compiled language
    @param $associations                - optional boolean parameter relevant if $treeonly = 'false' to include associations: terminologyAssociation, identifierAssociation
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function utilmp:getConceptMapList($governanceGroupId as xs:string?, $project as item(), $projectVersion as xs:string?, $projectLanguage as xs:string?, $searchTerms as xs:string*, $includebbr as xs:boolean, $sort as xs:string?, $sortorder as xs:string?, $max as xs:integer?, $resolve as xs:boolean, $otherparams as map(*)?) as element(list) {
    
    let $governanceGroupId  := $governanceGroupId[not(. = '')]
    let $project            := $project[not(. = '')]
    let $projectVersion     := $projectVersion[not(. = '')]
    let $projectLanguage    := ($projectLanguage[not(. = '')], '*')[1]
    let $sort               := $sort[string-length() gt 0]
    let $sortorder          := $sortorder[. = 'descending']
    
    let $cmid               := $otherparams?id[not(. = '')]
    let $cmed               := $otherparams?effectiveDate[not(. = '')]
    let $status             := $otherparams?status[not(. = '')]
    let $vsid               := $otherparams?valueSetId[not(. = '')]
    let $vsed               := $otherparams?valueSetEffectiveDate[not(. = '')]
    let $vssourceid         := $otherparams('valueSetId:source')[not(. = '')]
    let $vssourceed         := $otherparams('valueSetEffectiveDate:source')[not(. = '')]
    let $vstargetid         := $otherparams('valueSetId:target')[not(. = '')]
    let $vstargeted         := $otherparams('valueSetEffectiveDate:target')[not(. = '')]
    
    let $csid               := $otherparams?codeSystemId[not(. = '')]
    let $csed               := $otherparams?codeSystemEffectiveDate[not(. = '')]
    let $cssourceid         := $otherparams('codeSystemId:source')[not(. = '')]
    let $cssourceed         := $otherparams('codeSystemEffectiveDate:source')[not(. = '')]
    let $cstargetid         := $otherparams('codeSystemId:target')[not(. = '')]
    let $cstargeted         := $otherparams('codeSystemEffectiveDate:target')[not(. = '')]
    
    let $startT             := util:system-time()
    
    let $check              :=
        if (empty($project[not(. = '')])) then
            error($errors:BAD_REQUEST, 'Missing required parameter prefix')
        else ()
    
    let $decor              := utillib:getDecor($project, $projectVersion, $projectLanguage)
       
    let $projectPrefix      := ($decor/project/@prefix)[1]
    
    (: get all conceptMaps for $decor / $id / $ed :)
    let $results            :=
        if (empty($cmid)) then $decor//conceptMap 
        else if ($cmed castable as xs:dateTime) then $decor//conceptMap[@id = $cmid][@effectiveDate = $cmed]
        else $decor//conceptMap[@id = $cmid]

    (: if search: get all conceptMaps within search :)
    let $results            := if (empty($searchTerms)) then $results else utilmp:searchConceptMaps($decor, $searchTerms, $includebbr)          
    
    let $allcnt             := count($results)
        
    (: if resolve: add all referenced conceptMaps to conceptMaps :)
    let $conceptMapsByRef   := if ($resolve and empty($projectVersion)) then utilmp:getConceptMapsByRef($results, $projectPrefix, $projectLanguage) else $results[@ref]
    let $results            := $results[@id] | $conceptMapsByRef
    
    (: if status: get all conceptMaps within this status :)
    let $results            := if (empty($status)) then $results else $results[@statusCode = $status]
        
    (: handle all other parameters :)
    let $results            :=
        if (empty($vsid)) then $results 
        else if (empty($vsed)) then $results[sourceScope[@ref = $vsid] | targetScope[@ref = $vsed]]
        else $results[sourceScope[@ref = $vsid][@flexibility = $vsed] | targetScope[@ref = $vsed][@flexibility = $vsed]]
        
    let $results            :=
        if (empty($vssourceid)) then $results  
        else if (empty($vssourceed)) then $results[sourceScope[@ref = $vssourceid]]
        else $results[sourceScope[@ref = $vssourceid][@flexibility = $vssourceed]]
        
    let $results            :=
        if (empty($vstargetid)) then $results 
        else if (empty($vstargeted)) then $results[targetScope[@ref = $vstargetid]]
        else $results[tagretScope[@ref = $vstargetid][@flexibility = $vstargeted]]

    let $results            :=
        if (empty($csid)) then $results
        else if (empty($csed)) then $results[group[source[@codeSystem = $csid] | target[@codeSystem = $csed]]]
        else $results[group[source[@codeSystem = $csid][@codeSystemVersion = $csed] | target[@codeSystem = $csed][@codeSystemVersion = $csed]]]

    let $results            :=
        if (empty($cssourceid)) then $results 
        else if (empty($cssourceed)) then $results[group[source[@codeSystem = $cssourceid]]]
        else $results[group[source[@codeSystem = $cssourceid][@codeSystemVersion = $cssourceed]]]

    let $results            :=
        if (empty($cstargetid)) then $results 
        else if (empty($cstargeted)) then $results[group[target[@codeSystem = $cstargetid]]]
        else $results[group[target[@codeSystem = $cstargetid][@codeSystemVersion = $cstargeted]]]
        
    (: create payload for conceptMapList :)
    let $results            :=
        for $cm in $results
        let $id             := $cm/@id | $cm/@ref
        group by $id
        return (
            let $subversions:=
                for $cmv in $cm
                order by $cmv/@effectiveDate descending
                return
                    <conceptMap uuid="{util:uuid()}">
                    {
                        $cmv/(@* except @uuid),
                        (: this element is not supported (yet?) :)
                        $cmv/classification
                    }
                    </conceptMap>
            let $latest         := ($subversions[@id], $subversions)[1]
            let $rproject       := $cm/ancestor::decor/project/@prefix
            return
            <conceptMap uuid="{util:uuid()}" id="{$id}">
            {
                $latest/(@* except (@uuid | @id | @ref | @project)),
                ($cm/@ref)[1],
                (: there is no attribute @project, but better safe than sorry :)
                (: wrong setting for governance group search: if (empty($governanceGroupId)) then $latest/@project else attribute project {$projectPrefix}, :)
                if (empty($governanceGroupId)) then $latest/@project else attribute project { $rproject[1] },
                $subversions
            }
            </conceptMap>
        )

    let $count              := count($results/conceptMap)
    let $max                := if ($max ge 1) then $max else $count
    
    (: sort payload :)
    let $results            := utilmp:sortConceptMaps($results, $sort, $sortorder)
    
    let $durationT          := (util:system-time() - $startT) div xs:dayTimeDuration("PT0.001S")
    
    return
        <list artifact="MP" elapsed="{$durationT}" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" resolve="{$resolve}" project="{$projectPrefix}" lastModifiedDate="{current-dateTime()}">
        {
            subsequence($results, 1, $max)
        }
        </list>
};


(:~ Central logic for patching an existing concept map
{ "op": "[add|remove|replace]", "path": "/", "value": "[terminologyAssociation|identifierAssociation]" }

where
* op - add (object associations, tracking, event) or remove (object associations only) or replace (displayName, priority, type, tracking, assignment)
* path - / 
* value - terminologyAssociation|identifierAssociation object

    @param $authmap                     - required. Map derived from token
    @param $id                          - required. DECOR concept/@id to update
    @param $effectiveDate               - required. DECOR concept/@effectiveDate to update
    @param $data                        - required. DECOR concept xml element containing everything that should be in the updated concept
    @return concept object as xml with json:array set on elements
    @since 2020-05-03
:)
declare function utilmp:patchConceptMap($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) {
    
    let $storedConceptMap   := $setlib:colDecorData//conceptMap[@id = $id][@effectiveDate = $effectiveDate] 
    let $decor              := $storedConceptMap/ancestor::decor
    let $projectPrefix      := $decor/project/@prefix
    let $projectLanguage    := $decor/project/@defaultLanguage
            
    let $lock               := utilmp:checkConceptMapAccess($authmap, $decor, $id, $effectiveDate, true())

    let $check              :=
        if ($storedConceptMap[@statusCode = $utilmp:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $utilmp:STATUSCODES-FINAL)]) then () else
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('ConceptMap cannot be patched while it has one of status: ', string-join($utilmp:STATUSCODES-FINAL, ', '), if ($storedConceptMap/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
        
    let $check              := utilmp:checkConceptMapParameters($data, $storedConceptMap, $decor)
    
    let $intention          := if ($storedConceptMap[@statusCode = 'final']) then 'patch' else 'version'
    let $update             := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-MAPPING, $projectPrefix, $intention, $storedConceptMap)
    
    let $storedConceptMap   := utilmp:patchConceptMapParameters($data, $storedConceptMap)
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $preparedConceptMap := utilmp:prepareConceptMapForUpdate($storedConceptMap, $storedConceptMap)
    
    (: now update the concept map :)
    let $conceptMapUpdate   := update replace $storedConceptMap with $preparedConceptMap 
    let $update             := update delete $lock

    return utilmp:getConceptMap($projectPrefix, (), $projectLanguage, $id, $effectiveDate, false(), true())
};

(: Central logic for creating a new conceptMap

    @param $projectPrefix           - project to create this scenario in
    @param $targetDate              - optional. if true invokes effectiveDate of the new conceptMap and concepts as [DATE]T00:00:00. If false invokes date + time [DATE]T[TIME] 
    @param $sourceId                - optional. parameter denoting the id of a dataset to use as a basis for creating the new conceptMap
    @param $sourceEffectiveDate     - optional. parameter denoting the effectiveDate of a dataset to use as a basis for creating the new conceptMap",
    @param $keepIds                 - optional. only relevant if source dataset is specified. If true, the new conceptMap will keep the same ids for the new conceptMap, and only update the effectiveDate
    @param $baseDatasetId           - optional. only relevant when a source dataset is specified and `keepIds` is false. This overrides the default base id for datasets in the project. The value SHALL match one of the projects base ids for datasets
    @param $data                    - optional. DECOR concept xml element containing everything that should be in the new concept
    @return (empty) dataset object as xml with json:array set on elements
:)
declare function utilmp:createConceptMap($authmap as map(*), $projectPrefixOrId as xs:string, $targetDate as xs:boolean, $sourceId as xs:string?, $sourceEffectiveDate as xs:string?, $refOnly as xs:boolean?, $keepIds as xs:boolean?, $baseId as xs:string?, $editedConceptMap as element(conceptMap)?) {

    let $decor              := utillib:getDecor($projectPrefixOrId)
    let $check              :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, 'Project ''' || $projectPrefixOrId || ''' does not exist')
        )
        
    let $projectPrefix      := ($decor/project/@prefix)[1]
    let $projectLanguage    := $decor/project/@defaultLanguage
    (: if the user sent us a conceptMap with id, we should assume he intends to keep that id :)
    let $keepIds            := if ($editedConceptMap/@id) then true() else $keepIds = true()
    (: if the user sent us a conceptMap with effectiveDate, we should assume he intends to keep that effectiveDate :)
    let $now                := 
        if ($editedConceptMap/@effectiveDate) then substring($editedConceptMap/@effectiveDate, 1, 19) else 
        if ($targetDate) then substring(string(current-date()), 1, 10) || 'T00:00:00' else substring(string(current-dateTime()), 1, 19)
    
    let $check              := utilmp:checkConceptMapAccess($authmap, $decor)
    
    let $check              :=
        if (empty($editedConceptMap/@id)) then () else (
            if (matches($editedConceptMap/@id, '^[0-2](\.(0|[1-9][0-9]*)){0,3}$')) then
                error($errors:BAD_REQUEST, 'Concept map id ''' || $editedConceptMap/@id || ''' is reserved. You cannot reuse a reserved identifier. See http://oid-info.com/get/' || $editedConceptMap/@id)
            else
            if (utillib:isOid($editedConceptMap/@id)) then () else (
                error($errors:BAD_REQUEST, 'Concept map id ''' || $editedConceptMap/@id || ''' SHALL be a valid OID')
            )
        )
    
    let $sourceConceptMap   := if (empty($sourceId)) then () else utilmp:getConceptMap($projectPrefix, (), (), $sourceId, $sourceEffectiveDate, false(), true())
    let $storedConceptMap   := if ($keepIds and $editedConceptMap[@id]) then utilmp:getConceptMap((), (), (), $editedConceptMap/@id, $now, false(), true()) else ()
    
    let $check              :=
        if ($refOnly) then
            if (empty($sourceId)) then
                error($errors:BAD_REQUEST, 'Parameter sourceId SHALL be provided if a value set reference is to be created')
            else
            if (empty($sourceConceptMap)) then
                if (empty($sourceEffectiveDate)) then
                    error($errors:BAD_REQUEST, 'Parameter sourceId ' || $sourceId || ' SHALL lead to a conceptMap in scope of this project')
                else (
                    error($errors:BAD_REQUEST, 'Parameters sourceId ' || $sourceId || ' and sourceEffectiveDate ' || $sourceEffectiveDate || ' SHALL lead to a conceptMap in scope of this project')
                )
            else ()
        else
        if (empty($editedConceptMap) and empty($sourceId)) then 
            error($errors:BAD_REQUEST, 'ConceptMap input data or a sourceId SHALL be provided')
        else
        if ($editedConceptMap) then
            if ($storedConceptMap) then
                error($errors:BAD_REQUEST, 'Cannot create new conceptMap. A conceptMap with id ' || $editedConceptMap/@id || ' and effectiveDate ' || $now || ' already exists.')
            else ()
        else (
            if (empty($sourceConceptMap)) then 
                error($errors:BAD_REQUEST, 'Cannot create new conceptMap. Source valueset based on id ' || $sourceId || ' and effectiveDate ' || $sourceEffectiveDate || ' does not exist.')
            else  ()
        )
    
    let $check              :=
        if ($now castable as xs:dateTime) then () else (
            error($errors:BAD_REQUEST, 'Cannot create new conceptMap. The provided effectiveDate ' || $now || ' is not a valid xs:dateTime. Expected yyyy-mm-ddThh:mm:ss.')
        )
    
    let $editedConceptMap   := ($editedConceptMap, $sourceConceptMap)[1]
    (: decorlib:getNextAvailableIdP() returns <next base="1.2" max="2" next="{$max + 1}" id="1.2.3" type="DS"/> :)
    let $newConceptMapId    := if ($keepIds) then $editedConceptMap/@id else (decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-MAPPING, $baseId)/@id)
    
    let $check              :=
        if (utilmp:getConceptMap($projectPrefix, (), (), $newConceptMapId, $now, false(), true())) then
            error($errors:BAD_REQUEST, 'Cannot create new conceptMap. The to-be-created conceptMap with id ' || $newConceptMapId || ' and effectiveDate ' || $now || ' already exists.')
        else ()
    
    let $baseConceptMap     := 
        <conceptMap>
        {
            attribute id {$newConceptMapId} ,
            $editedConceptMap/@name[string-length()>0] ,
            $editedConceptMap/@displayName[string-length()>0] ,
            attribute effectiveDate {$now} ,
            attribute statusCode {"draft"} ,
            $editedConceptMap/@versionLabel[string-length()>0] ,
            $editedConceptMap/@expirationDate[string-length()>0] ,
            $editedConceptMap/@officialReleaseDate[string-length()>0] ,
            $editedConceptMap/@experimental[string-length()>0] ,
            $editedConceptMap/@canonicalUri[string-length()>0] ,
            $editedConceptMap/@lastModifiedDate[string-length()>0]
        }
        </conceptMap>
    
    let $newConceptMap      := 
        if ($refOnly) then <conceptMap ref="{$sourceConceptMap/@id}" name="{utillib:shortName($sourceConceptMap/@displayName)}" displayName="{$sourceConceptMap/@displayName}"/>
        else utilmp:prepareConceptMapForUpdate($editedConceptMap, $baseConceptMap) 
 
    let $check              := if ($refOnly) then () else ruleslib:checkConceptMap($newConceptMap, $decor, 'create')
    
    let $check              := if (empty($check)) then () else (
            error($errors:BAD_REQUEST, string-join ($check, ' '))
        )
    
    (: prepare sub root elements if not existent :)
    let $conceptMapUpdate   :=
        if (not($decor/terminology) and $decor/codedConcepts) then update insert <terminology/> following $decor/codedConcepts
        else if (not($decor/terminology) and not($decor/codedConcepts)) then update insert <terminology/> following $decor/ids
        else ()
    
    (: now update the conceptmap :)
    let $conceptMapUpdate   :=
        (: this could update the name/displayName if the source thing was updated since adding it previously :)
        if ($refOnly and $decor//conceptMap[@ref = $newConceptMap/@ref]) then update replace $decor//conceptMap[@ref = $newConceptMap/@ref] with $newConceptMap
        else update insert $newConceptMap into $decor/terminology
    
    (: return the regular conceptMap that was created, or the requested version of the reference that was created, or the latest version of the reference that was created :)
    return utilmp:getConceptMap($projectPrefix, (), (), ($newConceptMap/@id | $newConceptMap/@ref), ($newConceptMap/@effectiveDate, $sourceEffectiveDate, 'dynamic')[1], false(), true())
};

(:~ Central logic for updating an existing conceptMap

    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR conceptMap/@id to update
    @param $request-body    - required. DECOR conceptMap xml element containing everything that should be in the updated conceptMap
    @return conceptMap object as xml with json:array set on elements
:)
declare function utilmp:putConceptMap($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(), $deletelock as xs:boolean) as element(conceptMap) {

    let $storedConceptMap   := $setlib:colDecorData//conceptMap[@id = $id][@effectiveDate = $effectiveDate] 
    let $decor              := $storedConceptMap/ancestor::decor
    let $projectPrefix      := $decor/project/@prefix
    let $projectLanguage    := $decor/project/@defaultLanguage
    
    let $check              :=
        if ($storedConceptMap) then () else (
            error($errors:BAD_REQUEST, 'ConceptMap with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ' does not exist')
        )
      
    let $lock               := utilmp:checkConceptMapAccess($authmap, $decor, $id, $effectiveDate, true())                  
    
    let $check              :=
        if ($storedConceptMap[@statusCode = $utilmp:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, concat('ConceptMap cannot be updated while it has one of status: ', string-join($utilmp:STATUSCODES-FINAL, ', '), if ($storedConceptMap/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
        else ()
    
    let $check              :=
        if ($data[@id = $id] | $data[empty(@id)]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no conceptMap id or the same conceptMap id as the conceptMap id ''', $id, ''' used for updating. Found in request body: ', ($data/@id, 'null')[1]))
        )
    
    let $check              :=
        if ($data[@effectiveDate = $effectiveDate] | $data[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Submitted data shall have no conceptMap effectiveDate or the same conceptMap effectiveDate as the conceptMap effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($data/@effectiveDate, 'null')[1]))
        )
    
    let $check              :=
        if (count($data/group) gt 1 and $data/group[not(source)][not(target)]) then (
            error($errors:BAD_REQUEST, 'Group with no source and target is only allowed for ONE group. Found: ' || count($data/group) || ' groups.')
        ) else ()
    
    let $check              :=
        if ($data/value/group[not(source)][not(target)][not(element/@codeSystem)]) then (
             error($errors:BAD_REQUEST, 'Input group with no source and target SHALL have a codesystem in all element childs.')
        ) else ()
    
    let $newConceptMap      := utilmp:prepareConceptMapForUpdate($data, $storedConceptMap)
    
    (: check the conceptmap before update :)
    let $check              := ruleslib:checkConceptMap($newConceptMap, $decor, 'update')
    
    let $check              := if (empty($check)) then () else (
            error($errors:BAD_REQUEST, string-join ($check, ' '))
        )
    
    (: save history:)
    let $intention          := if ($storedConceptMap[@statusCode = 'final']) then 'patch' else 'version'
    let $history            := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-MAPPING, $projectPrefix, $intention, $storedConceptMap)
    
    (: now update the conceptmap :)
    let $conceptMapUpdate   := update replace $storedConceptMap with $newConceptMap
    let $delete             := update delete $storedConceptMap//@json:array
    let $deleteLock         := if ($deletelock) then update delete $lock else ()
    
    return utilmp:getConceptMap($projectPrefix, (), $projectLanguage, $id, $effectiveDate, false(), true())
};

(:~ Retrieves a conceptMap in edit mode
    @param $id                      - required parameter denoting the id of the template
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the template. If not given assumes latest version for id
    @return template
    @since 2025-04-29
:)
declare function utilmp:getConceptMapForEdit($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $breaklock as xs:boolean) {

    let $conceptMap         := $setlib:colDecorData//conceptMap[@id = $id][@effectiveDate = $effectiveDate] 
    
    let $decor              := $conceptMap/ancestor::decor
    let $projectPrefix      := $decor/project/@prefix
    let $projectLanguage    := $decor/project/@defaultLanguage

    let $check                  :=
        if ($conceptMap) then () else (
            error($errors:BAD_REQUEST, 'Conceptmap with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' not found')
        )
    
    let $lock                   := utilmp:checkConceptMapAccess($authmap, $decor, $id, $effectiveDate, true(), $breaklock)
    
    let $check                  :=
        if ($template[@statusCode = $utilmp:STATUSCODES-FINAL]) then (
            error($errors:BAD_REQUEST, 'Conceptmap cannot be updated while it has one of status: ' || string-join($utilmp:STATUSCODES-FINAL, ', ') || (if ($conceptMap/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()

    return utilmp:getConceptMap($projectPrefix, (), $projectLanguage, $id, $effectiveDate, false(), true())
};

(:~ Return zero or more conceptmaps as-is
@param $id           - required. Identifier of the conceptmap to retrieve
@param $flexibility  - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@return Matching conceptmaps
@since 2025-03-25
:)
declare %private function utilmp:getConceptMapById($id as xs:string, $flexibility as xs:string?) as element(conceptMap)* {

    if ($flexibility castable as xs:dateTime) then (
        let $decorSets      := $setlib:colDecorData//conceptMap[@id = $id][@effectiveDate = $flexibility]
        let $cacheSets      := $setlib:colDecorCache//conceptMap[@id = $id][@effectiveDate = $flexibility]
        return ($decorSets, $cacheSets)
    )
    else if ($flexibility) then (
        let $decorSets      := $setlib:colDecorData//conceptMap[@id = $id]
        let $cacheSets      := $setlib:colDecorCache//conceptMap[@id = $id]
        let $conceptMaps      := ($decorSets, $cacheSets)
        return $conceptMaps[@effectiveDate = string(max($conceptMaps/xs:dateTime(@effectiveDate)))]
    )
    else (
        let $decorSets      := $setlib:colDecorData//conceptMap[@id = $id]
        let $cacheSets      := $setlib:colDecorCache//conceptMap[@id = $id]
        return ($decorSets, $cacheSets)
    )
};

(:~ Return zero or more conceptmaps as-is
@param $id            - required. Identifier of the conceptmap to retrieve
@param $flexibility   - optional. null gets all versions, yyyy-mm-ddThh:mm:ss gets this specific version, anything that doesn't cast to xs:dateTime gets latest version
@param $decorOrPrefix - required. determines search scope. pfx- limits scope to this project only
@param $decorRelease  - optional. if empty defaults to current version. if valued then the conceptmap will come explicitly from that archived project version which is expected to be a compiled version
@param $decorLanguage - optional. Language compilation. Defaults to project/@defaultLanguage.
@return Matching conceptmaps
@since 2025-03-25
:)
declare %private function utilmp:getConceptMapById($id as xs:string, $flexibility as xs:string?, $decorOrPrefix as item(), $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(conceptMap)* {

    let $decor              := utillib:getDecor($decorOrPrefix, $decorRelease, $decorLanguage)
    let $decors             := if (empty($decorRelease)) then utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL) else $decor
    
    return
        if ($flexibility castable as xs:dateTime) then $decors//terminology/conceptMap[@id = $id][@effectiveDate = $flexibility]
        else if ($flexibility) then (
            let $conceptMaps  := $decors//terminology/conceptMap[@id = $id]
            return $conceptMaps[@effectiveDate = string(max($conceptMaps/xs:dateTime(@effectiveDate)))]
        )
        else $decors//terminology/conceptMap[@id = $id]
};

(:~ Get contents of a conceptMap :)
declare function utilmp:getRawConceptMap($conceptMap as element(conceptMap)) as element(conceptMap) {
    <conceptMap>
    {
        $conceptMap/@*,
        $conceptMap/desc,
        $conceptMap/publishingAuthority,
        if ($conceptMap[@id]) then if ($conceptMap[publishingAuthority]) then () else utillib:inheritPublishingAuthorityFromProject($conceptMap/ancestor::decor) else (),
        $conceptMap/endorsingAuthority,
        $conceptMap/purpose,
        if ($conceptMap[@id]) then utillib:handleCodeSystemCopyrights($conceptMap/copyright, distinct-values($conceptMap//@codeSystem)) else $conceptMap/copyright,
        $conceptMap/jurisdiction,
        utilmp:createSourceOrTargetScope($conceptMap/sourceScope),
        utilmp:createSourceOrTargetScope($conceptMap/targetScope),
        utilmp:getRawConceptMapGroup($conceptMap/group)
    }
    </conceptMap>
};

declare %private function utilmp:getRawConceptMapGroup($groups as element(group)*) as element(group)* {
    
    for $group in $groups
    return 
        <group>
        {
            if ($group/source) then utilmp:createSourceOrTargetCodeSystem($group/source) else (),
            if ($group/target) then utilmp:createSourceOrTargetCodeSystem($group/target) else (),
            $group/(* except (source | target))
        }
        </group>
};

declare %private function utilmp:createSourceOrTargetCodeSystem($sourceOrTarget as element()) as element() {
    
    (: keeping the displayName synchronized: displayname of a codeSystem is default codeSystemName with fallback the OIDName and the displayName :)
    let $displayName        := (utillib:getCodeSystemById($sourceOrTarget/@codeSystem, ())/@codeSystemName, utillib:getNameForOID($sourceOrTarget/@codeSystem, (), ()), $sourceOrTarget/@displayName)[1][not(. = '')]
    let $canonicalUri       := (utillib:getCanonicalUriForOID('CodeSystem', $sourceOrTarget/@codeSystem, (), (), ()), $sourceOrTarget/@canonicalUri)[1][not(. = '')] 
    
    return
        element {name($sourceOrTarget)}
        {           
            $sourceOrTarget/@codeSystem[not(. = '')],
            $sourceOrTarget/@codeSystemVersion[not(. = '')],
            if ($displayName) then attribute displayName {replace(normalize-space($displayName[1]), '\s', '&#160;')} else (),
            if ($canonicalUri) then attribute canonicalUri {$canonicalUri} else ()
        }
};

(: expect: element(sourceScope) or element(targetScope) :)
declare %private function utilmp:createSourceOrTargetScope($scope as element()) as element() {
    
    (: keeping the displayName synchronized: displayname of a valueSet is default displayNamea with fallback the OIDName and the displayName :)
    let $valueSet           := utillib:getValueSetById($scope/@ref, ())
    let $displayName        := ($valueSet/@displayName, $scope/@displayName)[1][not(. = '')]
    let $canonicalUri       := ($valueSet/@canonicalUri, $scope/@canonicalUri)[1][not(. = '')] 
    
    return
        element {name($scope)}
        {           
            $scope/@ref[not(. = '')],
            $scope/@flexibility[not(. = '')],
            if ($displayName) then attribute displayName {$displayName} else (),
            if ($canonicalUri) then attribute canonicalUri {$canonicalUri} else ()
        }
};

declare %private function utilmp:searchConceptMaps($decor as element(decor), $searchTerms as xs:string*, $includebbr as xs:boolean) as element(conceptMap)* {
    
    let $buildingBlockRepositories  := if ($includebbr) then utillib:getBuildingBlockRepositories($decor, (), $utillib:strDecorServicesURL) else $decor
    let $luceneQuery                := utillib:getSimpleLuceneQuery($searchTerms, 'wildcard')
    let $luceneOptions              := utillib:getSimpleLuceneOptions() 

    for $ob in ($buildingBlockRepositories//conceptMap[@id = $searchTerms] | 
                $buildingBlockRepositories//conceptMap[@id][ft:query(@name, $luceneQuery, $luceneOptions)] |
                $buildingBlockRepositories//conceptMap[@id][ft:query(@displayName, $luceneQuery, $luceneOptions)])
    return
        <conceptMap>
        {
            $ob/(@* except (@url|@ident)),
            attribute url {($ob/ancestor::cacheme/@bbrurl, $utillib:strDecorServicesURL)[1]},
            attribute ident {$ob/ancestor::decor/project/@prefix},
            attribute cachedProject {exists($ob/ancestor::cacheme)},
            $ob/node()
        }
        </conceptMap>
};

declare %private function utilmp:getConceptMapsByRef($conceptMaps as element(conceptMap)*, $projectPrefix as item(), $projectLanguage as xs:string?) as element()* {

    for $cm in $conceptMaps[@ref]
        let $id             := $cm/@ref
        let $cms            := utilmp:getConceptMap($projectPrefix, (), $projectLanguage, $cm/@ref, (), true(), false())
        let $cmbyid         := $cms[@effectiveDate = max($cms/xs:dateTime(@effectiveDate))]
        return (
            (: rewrite name and displayName based on latest target codeSystem. These sometimes run out of sync when the original changes its name :)
            element {name($cm)} {
                $cm/@ref, 
                attribute name {utillib:shortName(($cmbyid/@displayName, $cm/@displayName)[1])}, 
                attribute displayName {($cmbyid/@displayName, $cm/@displayName, $cmbyid/@name, $cm/@name)[1]}
            } | $cms
        )


};

(: handle sorting. somehow reverse() does not do what I expect :)
declare %private function utilmp:sortConceptMaps($conceptMaps as element(conceptMap)*, $sort as xs:string?, $sortorder as xs:string?) as element (conceptMap)* {

    switch ($sort)
    case 'displayName' return 
        if ($sortorder = 'descending') then 
            for $r in $conceptMaps order by $r/lower-case(@displayName) descending return $r
        else (
            for $r in $conceptMaps order by $r/lower-case(@displayName)            return $r
        )
    case 'name'        return 
        if ($sortorder = 'descending') then 
            for $r in $conceptMaps order by $r/lower-case(@name) descending return $r
        else (
            for $r in $conceptMaps order by $r/lower-case(@name)            return $r
        )
    default            return 
        if ($sortorder = 'descending') then 
            for $r in $conceptMaps order by replace(replace($r/@id, '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1') descending return $r
        else (
            for $r in $conceptMaps order by replace(replace($r/@id, '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1')            return $r
        )

};

declare %private function utilmp:checkConceptMapAccess($authmap as map(*), $decor as element(decor)) as element()? {
    utilmp:checkConceptMapAccess($authmap, $decor, (), (), false(), false())
};

declare %private function utilmp:checkConceptMapAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean) as element()? {
    utilmp:checkConceptMapAccess($authmap, $decor, $id, $effectiveDate, $checkLock, false())
};

declare %private function utilmp:checkConceptMapAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean, $breakLock as xs:boolean) as element()? {

    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-TERMINOLOGY)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify conceptMaps in project ', $decor/project/@prefix, '. You have to be an active author in the project.'))
        )

    return
        if ($checkLock) then (
            let $lock           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
            let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breakLock)
            return
                if ($lock) then $lock else (
                    error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this conceptMap (anymore). Get a lock first.'))
                )    
            )
            else ()
};

declare %private function utilmp:checkConceptMapParameters($parameters as element(parameters), $conceptMap as element(conceptMap)*, $decor as element(decor)) {

    let $check              :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops     := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check              :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
        
    let $check              :=
        for $param in $parameters/parameter
        let $op             := $param/@op
        let $path           := $param/@path
        let $elmname        := substring-after($param/@path, '/')
        let $value          := ($param/@value, $param/value/*[name() = $elmname])[1]
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then 'Parameter path ''' || $path || ''' does not support ' || $op
                else if (utillib:isStatusChangeAllowable($conceptMap, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($conceptMap/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove') then () 
                else if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/displayName' return (
                if ($op = 'remove') then 'Parameter path ''' || $path || ''' does not support ' || $op
                else if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/canonicalUri'
            case '/versionLabel' return (
                if ($op = 'remove') then ()
                else if ($param/@value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/desc' 
            case '/purpose'
            case '/copyright' return (
                if ($param[count(value/*[name() = $elmname]) = 1])
                then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $elmname], $op, $path)
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $elmname || ' under value. Found ' || count($param/value/*[name() = $elmname])
                )
            )
            case '/publishingAuthority' return (
                if ($param[count(value/publishingAuthority) = 1]) 
                then ruleslib:checkPublishingAuthority($param/value/publishingAuthority, $op, $path)
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single publishingAuthority under value. Found ' || count($param/value/publishingAuthority)
                )
            )
            case '/jurisdiction' return (
                if ($param[count(value/jurisdiction) = 1]) 
                then if ($op = 'remove') then () 
                else ruleslib:checkValueCodingType($param/value/jurisdiction, $op, $path)
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $elmname || ' under value. Found ' || count($param/value/jurisdiction)
                )
            )
            case '/sourceScope' return (
                if ($param[count(value/sourceScope) = 1]) then (
                    if ($op = 'remove') then 'Parameter ' || $op || ' not allowed for ''' || $path || ''''
                    else 
                    let $sourceOrTargetScope    := ($param/value/targetScope, $conceptMap/targetScope)[1]
                    return ruleslib:checkConceptMapScope($param/value/sourceScope, $sourceOrTargetScope, $decor, $op, $path)
                )       
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $elmname || ' under value. Found: ' || count($param/value/*[name() = $elmname])
                )
            )
            case '/targetScope' return (
                if ($param[count(value/targetScope) = 1]) then (
                    if ($op = 'remove') then 'Parameter ' || $op || ' not allowed for ''' || $path || ''''
                    
                    else ruleslib:checkConceptMapScope($param/value/targetScope, ($param/value/sourceScope | $conceptMap/sourceScope)[1], $decor, $op, $path)
                )       
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $elmname || ' under value. Found: ' || count($param/value/*[name() = $elmname])
                )
            )
            case '/group' return (
                if ($param[count(value/group) = 1]) then (
                    if ($op = 'remove') then () 
                    else if ($conceptMap/group and $parameters/parameter/value/group[not(source)][not(target)]) then (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input group with no source and target is only allowed for concept map with NO existing groups. Found: ' || count($conceptMap/group) || ' existing groups.'
                    )
                    else if ($parameters/parameter/value/group[not(source)][not(target)][not(element/@codeSystem)]) then (
                        'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input group with no source and target SHALL have a codesystem in all element childs.'
                    )
                    else (
                    let $preparedConceptMap :=
                        element {name($conceptMap)}
                        {
                            $conceptMap/@*,
                            $conceptMap/(* except (sourceScope | targetScope | group)),
                            ($param/value/sourceScope, $conceptMap/sourceScope)[1],
                            ($param/value/targetScope, $conceptMap/targetScope)[1],
                            utilmp:prepareConceptMapGroupForUpdate($parameters/parameter/value/group)
                        }
                    return ruleslib:checkConceptMapGroup($preparedConceptMap, $decor, $op, $path)
                    )
                )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Input SHALL have exactly one ' || $elmname || ' under value. Found: ' || count($param/value/*[name() = $elmname])
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     
     return
        if (empty($check)) then () else (
            error($errors:BAD_REQUEST, string-join ($check, ' '))
        )

};

declare %private function utilmp:patchConceptMapParameters($parameters as element(parameters), $conceptMap as element(conceptMap)*) as element(conceptMap) {

    let $update             :=
        for $param in $parameters/parameter
        let $elmname        := substring-after($param/@path, '/')
        return
            switch ($param/@path)
            case '/statusCode'
            case '/displayName'
            case '/expirationDate'
            case '/officialReleaseDate'
            case '/canonicalUri'
            case '/versionLabel' return (
                let $attname:= substring-after($param/@path, '/')
                let $new    := attribute {$attname} {$param/@value}
                let $stored := $conceptMap/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $conceptMap
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/desc'
            case '/purpose'
            case '/copyright' return (
                (: only one per language :)
                let $new    := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
                let $stored := $conceptMap/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $conceptMap
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/publishingAuthority' return (
                (: only one possible :)
                let $new    := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
                let $stored := $conceptMap/publishingAuthority
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $conceptMap
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/jurisdiction'
            return (
                let $attname:= substring-after($param/@path, '/')
                let $new    := utillib:prepareValueCodingTypeForUpdate($param/value/*[name() = $attname])
                let $stored := $conceptMap/*[name() = $attname][@code = $new/@code][@codeSystem = $new/@codeSystem]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $conceptMap
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/sourceScope'
            case '/targetScope' return (
                (: only one per language :)
                let $new    := utilmp:prepareConceptMapScopeForUpdate($param/value/*[name() = $elmname])
                let $stored := $conceptMap/*[name() = $elmname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $conceptMap
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/group' return (
                (: only one per language :)
                (: TODO: need of re-calculate whole group :)
                let $new    := utilmp:getRawConceptMapGroup($param/value/group)
                let $stored := $conceptMap/group[source/@codeSystem = $new/source/@codeSystem][target/@codeSystem = $new/target/@codeSystem]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $conceptMap
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )
            
    return $conceptMap
};

(:~  Return only valid, non empty attributes on a conceptMap as defined in the DECOR format.
:   Example baseConceptMap:
:       <conceptMap id="..." name="..." displayName="..." effectiveDate="yyyy-mm-ddThh:mm:ss" statusCode="draft" versionLabel="..."/>
:
:   @param $editedConceptMap   - 1..1 conceptMap as edited
:   @param $baseConceptMap     - 1..1 conceptMap element containing all attributes that should be set on the prepared conceptMap, like new id/status etc.
:   @return the pruned conceptMap element for the given project in $decor, or nothing input was empty
:   @since 2015-09-22
:)
declare %private function utilmp:prepareConceptMapForUpdate($editedConceptMap as element(conceptMap), $baseConceptMap as element(conceptMap)) as element(conceptMap) {
    <conceptMap>
    {
        $baseConceptMap/(@*[not(. = '')] except @lastModifiedDate),
        attribute lastModifiedDate {substring(string(current-dateTime()), 1, 19)}
    }
    {
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedConceptMap/desc),
        utillib:preparePublishingAuthorityForUpdate($editedConceptMap/publishingAuthority[empty(@inherited)]),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedConceptMap/purpose),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedConceptMap/copyright[empty(@inherited)]),
        utilmp:prepareConceptMapScopeForUpdate($editedConceptMap/sourceScope),
        utilmp:prepareConceptMapScopeForUpdate($editedConceptMap/targetScope),
        utilmp:prepareConceptMapGroupForUpdate($editedConceptMap/group)
    }
    </conceptMap>
};

(: expect: element(sourceScope) or element(targetScope) :)
declare %private function utilmp:prepareConceptMapScopeForUpdate($scope as element()) as element()? {
    for $node in $scope
    return utilmp:createSourceOrTargetScope($scope)
       
};

declare %private function utilmp:prepareConceptMapGroupForUpdate($groups as element(group)*) as element(group)* {
    
    (: check whether we have to re-calcuate the group source and target objects,
        see https://art-decor.atlassian.net/browse/AD30-1812
    :)
    let $groups             := if (count($groups) = 1 and $groups[not(source)][not(target)]) then utilmp:recalculateConceptMapGroupForUpdate($groups) else $groups
    
    for $group in $groups
    return
        <group>
        {
            utilmp:prepareConceptMapSourceTargetForUpdate($group/source),
            utilmp:prepareConceptMapSourceTargetForUpdate($group/target),
            utilmp:prepareConceptMapElementForUpdate($group/element)
        }
        </group>
};

(: expect: element(source) or element(target) :)
declare %private function utilmp:prepareConceptMapSourceTargetForUpdate($sourceOrTarget as element()*) as element()* {

    for $node in $sourceOrTarget
    let $key := $node/@codeSystem[not(. = '')] || $node/@codeSystemVersion[not(. = '')] || $node/@displayName[not(. = '')] || $node/@canonicalUri[not(. = '')]
    group by $key
    return utilmp:createSourceOrTargetCodeSystem($node[1])
};

declare %private function utilmp:prepareConceptMapElementForUpdate($elmnt as element(element)*) as element(element)* {
    for $node in $elmnt
    return
        <element>
        {
            $node/@code[not(. = '')], $node/@displayName[not(. = '')],
            $node/target
        }
        </element>
};

declare %private function utilmp:recalculateConceptMapGroupForUpdate($groups as element(group)) as element(group)* {

    (: If there is only one group element that has no source and target,
        re-calculate the groups with their respective source and target items and 
        all elements in it that belongs to that group
    :)   

    (: the unmatched group has no target element and the element/target have no code attribute :) 

    let $groupMap               :=
    map:merge(
        for $elmnt in $groups/element
        let $source             := $elmnt/@codeSystem
        let $target             := $elmnt//target/@codeSystem
        let $sourceTarget       := $source || $target
        group by $sourceTarget
        return (
            let $groupEntry     :=
                <group>
                {
                    <source codeSystem="{$source[1]}"/>,
                    if ($target) then <target codeSystem="{$target[1]}"/> else ()
                }
                </group>
            return map:entry($sourceTarget, $groupEntry)
        )
    )
    
    for $groupKey in map:keys($groupMap)
    let $group                  := map:get($groupMap, $groupKey)
    let $elmnts                 :=
        if ($group/target) then
            for $elmnt in $groups/element 
            return $elmnt[@codeSystem = $group/source/@codeSystem][target/@codeSystem = $group/target/@codeSystem]
        else
            for $elmnt in $groups/element 
            return $elmnt[@codeSystem = $group/source/@codeSystem][not(target/@codeSystem)]
    return 
        <group>
            {
            $group/*,
            for $elmnt in $elmnts return 
            <element>
            {
                $elmnt/(@* except @codeSystem),
                $elmnt/(* except target),
                for $target in $elmnt/target return
                <target>
                {
                    $target/(@* except @codeSystem),
                    $target/*
                }
                </target>
            }
            </element>
        }
        </group>    
};
