xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright

    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.

    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
:)
(:
    Module: list / hoover history and releases
    
    Takes a couple of arguments
        project - a project prefix parameter, if not specified all projects are processed
        action  - the action to be performed
                  one of the following
                    list_history: list statistics about history of project or all projects
                    list_releases: list statistics about releases of project or all projects
                    hoover_history: hoover (clean up) history of project or all projects
                    hoover_releases: hoover (clean up) releases of project or all projects
                    
                    if no action is specified, nothing happens and a succesful result is returned
                    
                    hint: on hoover the history is not deleted but a skeletton is created
                    to be still in place and all body data of the artefacts is written to
                    the history.art-decor.org database

        secret  - the secret, actually only something to prevent use by accident
        
        user    - username and passowrd as login credentials, to allow the actions to be performed
        password
    
    history
    =======
    has per project a collection where several XML files may reside such as TM.xml or VS.xml
    for collection the history of changes of templates, value sets and other artefacts.
    Every time an artefact is changed and will be saved, the old version is stored in the history
    along with some meta data.
    
    An example
    
    <history
        date="2021-06-24T10:03:24.524+02:00" 
        authorid="kai" author="Kai Heitmann" 
        id="693a099d-c095-4264-ba09-9dc095b26408"    <-- id to find it in the history.art-decor.org database
        intention="version" 
        artefactId="2.16.840.1.113883.2.6.60.3.2.6.3" artefactEffectiveDate="2018-01-25T00:00:00" artefactStatusCode="draft" 
        hoovered="true"    <-- hoovered means not body of the artefact, only the skeletton
    >
        <concept id="2.16.840.1.113883.2.6.60.3.2.6.3" effectiveDate="2018-01-25T00:00:00" statusCode="draft" type="item">
            <name language="de-DE">Patienten ID</name>
            ... all the body of the artefact if not hoovered, 
            ... or only the root element and attributes (on cocnepts and datasets
            ... also with the name to allow proper display on hoovered versions.
        </concept>
    </history>
:)

import module namespace art = "http://art-decor.org/ns/art" at "../../art/modules/art-decor.xqm";
import module namespace setlib = "http://art-decor.org/ns/art-decor-settings" at "../../art/api/art-decor-settings.xqm";
declare namespace request = "http://exist-db.org/xquery/request";

(:
    copy the results from the hoovering into a clean skeleton history to be stored
    in the history branch of the project as a replacement for the full history
    flag every histro as hoovered
:)
declare function local:copy-hoovered-history($hh as node()) {
    let $r :=
    <histories>
        {
            $hh/@*,
            for $xh in $hh/history
            return
                <history>
                    {
                        $xh/@*,
                        if ($xh/@hoovered) then () else attribute {'hoovered'} {'true'},
                        $xh/(node() except zip)
                    }
                </history>
        }
    </histories>
    return
        $r
};

declare function local:hoover-history-collections($collection as xs:string) {
    for $histories in (doc($collection)/histories)
    let $p := $histories/@projectPrefix
    let $a := $histories/@artefactType
    return
        <histories>
            {
                $histories/@*,
                for $history in $histories/history
                let $hid := $history/@id
                return
                    if ($history/@hoovered='true')
                    then 
                        $history (: copy unchanged :)
                    else
                        for $x in $history
                        let $elm := $x/*[1]
                        let $elmn := $elm/name()
                        let $comp := compression:deflate(util:string-to-binary(serialize($elm)))
                        return
                            <history>
                                {
                                    $history/@*,
                                    element {$elmn} {
                                        $elm/@*,
                                        if ($a = 'DE' or $a = 'DS')
                                        then
                                            $elm/name
                                        else
                                            ()
                                    },
                                    <zip
                                        id="{$hid}"
                                        date="{$history/@date}"
                                        compressed="{string-length($comp)}">
                                        {
                                            $comp
                                        }
                                    </zip>
                                    ,
                                    if (false())
                                    then
                                        <zap
                                            uncompressed="{string-length(util:binary-to-string(compression:inflate($comp)))}">
                                            {
                                                util:binary-to-string(compression:inflate($comp))
                                            }
                                        </zap>
                                    else
                                        ()
                                }
                            </history>
            }
        </histories>
};


(: a project prefix parameter :)
let $projectPrefix := if (request:exists()) then
    request:get-parameter('project', '')
else
    ('')
    
    (: a action parameter :)
let $action := if (request:exists()) then
    request:get-parameter('action', '')
else
    ('')
    
    (: a secret parameter :)
let $secret := if (request:exists()) then
    request:get-parameter('secret', '')
else
    ('')
    
    (: get login credentials :)
let $theactinghooverusername := if (request:exists()) then
    request:get-parameter('user', '')
else
    ('')
let $theactinghooverpassword := if (request:exists()) then
    request:get-parameter('password', '')
else
    ('')


let $result :=
    if ($action = ('list_history', 'hoover_history'))
    then
        if ($secret = '61fgs756.s9' and (xmldb:login('/db', $theactinghooverusername, $theactinghooverpassword)))
        then
            <result outcome="success">
                {
                    (: 
                        history variables
                    :)
                    let $tmp1 := substring($projectPrefix, 1, string-length($projectPrefix) - 1)
                    let $tmp2 := concat($setlib:strDecorHistory, '/', $tmp1)
                    let $historyDirs :=
                        if (string-length($tmp1) = 0)
                        then
                            (: if no project is given, act on all projects available :)
                            xmldb:get-child-collections($setlib:strDecorHistory)
                        else
                            if (xmldb:collection-available($tmp2))
                            then
                                (: if a project is given and a collection is found work with it :)
                                $tmp1
                            else
                                (: project given but not found, return that :)
                                ()        

                    for $hdir in $historyDirs
                    let $hpath := concat($setlib:strDecorHistory, '/', $hdir)
                    let $c := xmldb:get-child-resources($hpath)
                    return
                        (
                        <history
                            project="{$hdir}"
                            path="{$hpath}"
                            files="{count($c)}"
                            nodes="{count(collection($hpath)//history)}"
                        >
                            {
                                for $hf in $c
                                let $hp := concat($hpath, '/', $hf)
                                let $uuid := util:uuid()
                                let $tmpf := concat('_', $uuid, '.xml')
                                let $res :=
                                    switch ($action) 
                                        case "hoover_history"
                                            return local:hoover-history-collections($hp)
                                        default
                                            return ()
                                let $success :=
                                    if (empty($res))
                                    then
                                        ()
                                    else
                                        try {
                                            (: store the newly hoovered history file as temp :)
                                            let $store1 :=
                                                xmldb:store($hpath, $tmpf, local:copy-hoovered-history($res))
                                            (: remove the old history file :)
                                            let $store2 :=
                                                xmldb:remove($hpath, $hf)
                                            (: set in place the newly hoovered history file :)
                                            let $store3 :=
                                                xmldb:rename($hpath, $tmpf, $hf)
                                            return
                                                true()
                                        }
                                        catch * {
                                            <error>Caught error {$err:code}: {$err:description}. Data: {$err:value}</error>
                                        }
                                return
                                    $res
                            }
                        </history>
                        )
                }
            </result>
        else
            <result outcome="unauthorized"/>
    else if ($action = ('list_releases', 'hoover_releases')) 
    then 
        if ($secret = '61fgs756.s9' and (xmldb:login('/db', $theactinghooverusername, $theactinghooverpassword)))
        then
            <result outcome="success">
            {
                (:
                    releases variables
                :)
                let $tmp1 := substring($projectPrefix, 1, string-length($projectPrefix) - 1)
                let $tmp2 := concat($setlib:strDecorVersion, '/', $tmp1)
                let $releaseDirs :=
                    if (string-length($tmp1) = 0)
                    then
                        (: if no project is given, act on all projects available :)
                        xmldb:get-child-collections($setlib:strDecorVersion)
                    else
                        if (xmldb:collection-available($tmp2))
                        then
                            (: if a project is given and a collection is found work with it :)
                            $tmp1
                        else
                            (: project given but not found, return that :)
                            ()
                 for $rdir in $releaseDirs
                    let $rpath := concat($setlib:strDecorVersion, '/', $rdir)
                    let $c := xmldb:get-child-collections($rpath)
                    return
                        (
                        <version
                            project="{$rdir}"
                            count="{count(collection($rpath)//publication[@projectId])}"
                        >
                        {
                            for $rf in $c
                            let $rp := concat($rpath, '/', $rf)
                            let $uuid := util:uuid()
                            let $pubrequest := collection($rp)//publication[@projectId]
                            let $res :=
                                switch ($action) 
                                    case "hoover_releases"
                                        return ()
                                    default
                                        return ()
                            return
                                <item 
                                    name="{$rf}"
                                    path="{$rp}"
                                    versionDate="{$pubrequest/@versionDate}">
                                {
                                    <publication>
                                    {
                                        attribute { 'type' } { name($pubrequest/(release|version)) },
                                        $pubrequest/@*,
                                        <filters>
                                        {
                                            $pubrequest/filters/@*
                                        }
                                        </filters>,
                                        $pubrequest/request,
                                        $pubrequest/processed
                                    }
                                    </publication>
                                }
                                </item>
                        }
                        </version>
                        )
            }
            </result>
        else
             <result outcome="unauthorized"/>
    else <result outcome="success"/>
    
return
    $result