<!-- 
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.    
-->
<!DOCTYPE stylesheet [
<!ENTITY omitted "&#xa0;">
]>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:xforms="http://www.w3.org/2002/xforms" xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns:uuid="java:java.util.UUID" xmlns:diff="http://art-decor.org/ns/decor/diff" xmlns:f="http://hl7.org/fhir" xmlns:local="http://art-decor.org/functions" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" version="2.0" exclude-result-prefixes="#all">
    <!--
        This stylesheet converts the rule set from a DECOR file into an appropriate HTML representation
        Note: this file cannot be used alone, it's called by the schematron conversion stylesheet
    -->
    <xsl:include href="v3mXML2HTMLGenerator2.xsl"/>
    <!--
        This stylesheet renders transaction groups into SVG for the Scenarios tab
    -->
    <xsl:include href="DECOR2svg.xsl"/>
    <!-- This stylesheet renders structure definitions and other FHIR artefacts -->
    <xsl:include href="DECOR4fhir.xsl"/>
    <!-- make a list of tab names (keys for messages) and their corresponding file names -->
    <xsl:variable name="tabnameslist" as="element()">
        <tabs xmlns="">
            <tab key="tabFrontPage" filename="index.html"/>
            <tab key="tabProjectInformation" filename="project.html"/>
            <xsl:if test="//datasets/dataset/*">
                <tab key="tabDataSet" filename="dataset.html"/>
            </xsl:if>
            <xsl:if test="//scenarios/*">
                <tab key="tabScenarios" filename="scenarios.html"/>
            </xsl:if>
            <xsl:if test="//ids/id">
                <tab key="tabIdentifiers" filename="identifiers.html"/>
            </xsl:if>
            <xsl:if test="//terminology[valueSet[@id] | codeSystem[@id] | conceptMap[@id]]">
                <tab key="tabTerminology" filename="terminology.html"/>
            </xsl:if>
            <xsl:if test="//rules/template">
                <tab key="tabRules" filename="rules.html"/>
            </xsl:if>
            <xsl:if test="$allQuestionnaires">
                <tab key="tabQuestionnaires" filename="questionnaires.html"/>
            </xsl:if>
            <xsl:if test="//issues/issue">
                <tab key="tabIssues" filename="issues.html"/>
            </xsl:if>
            <tab key="tabCompileTime" filename="compiletime.html"/>
            <tab key="tabLegal" filename="legal.html"/>
        </tabs>
    </xsl:variable>
    <!-- Do SVG -->
    <xsl:variable name="allSvg">
        <xsl:if test="$switchCreateDocSVG = true()">
            <!-- Need at least 1 initial and 1 back for a webservice -->
            <xsl:for-each select="//scenarios/scenario//transaction[@type = 'group'][transaction[@type = ('initial', 'stationary')]]">
                <transaction xmlns="">
                    <xsl:copy-of select="@id | @effectiveDate"/>
                    <xsl:apply-templates select="self::node()" mode="transactionGroupToSVGFunctional"/>
                    <xsl:apply-templates select="self::node()" mode="transactionGroupToSVGTechnical"/>
                </transaction>
            </xsl:for-each>
        </xsl:if>
    </xsl:variable>
    <!-- Calculate missing issues/labels/label definitions. Needed on Issues and Compile tab -->
    <xsl:variable name="allMissingLabels">
        <wrap xmlns="">
            <xsl:for-each select="$allIssues/issue/(tracking | assignment)[@labels]">
                <xsl:variable name="issueId" select="parent::issue/@id"/>
                <xsl:variable name="issuename">
                    <xsl:choose>
                        <xsl:when test="string-length(parent::issue/@displayName) &gt; 0">
                            <xsl:value-of select="parent::issue/@displayName"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'notLabeled'"/>
                            </xsl:call-template>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <xsl:variable name="labelCodes" select="@labels"/>
                <xsl:for-each select="distinct-values(tokenize($labelCodes, ' '))">
                    <xsl:variable name="labelCode" select="."/>
                    <xsl:if test="not($allIssues/labels/label[@code = $labelCode])">
                        <missingCode ref="{$issueId}" refName="{$issuename}" labelCode="{$labelCode}"/>
                    </xsl:if>
                </xsl:for-each>
            </xsl:for-each>
        </wrap>
    </xsl:variable>
    <xsl:template name="doTabs4ThisTab">
        <!-- the name of the actual tab to build tabs for, see variable tabnames -->
        <xsl:param name="actualtab"/>
        <!-- the actual content -->
        <xsl:param name="actualcontent"/>
        <xsl:result-document href="{$theHtmlDir}{local:doHtmlName($actualtab, (), (),'.html','true')}" format="xhtml">
            <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                <head>
                    <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                    <title>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="$actualtab"/>
                        </xsl:call-template>
                        <xsl:text>: </xsl:text>
                        <xsl:value-of select="//project/@prefix"/>
                        <xsl:text> </xsl:text>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'decorTitleString'"/>
                        </xsl:call-template>
                    </title>
                    <xsl:call-template name="addAssetsHeader">
                        <xsl:with-param name="doTreeTable" select="$actualtab = ('tabDataSet', 'tabScenarios', 'tabRules', 'tabQuestionnaires', 'tabTerminology')"/>
                        <xsl:with-param name="treeTableColumnIndex" as="xs:integer">
                            <xsl:choose>
                                <xsl:when test="$actualtab = ('tabDataSet')">3</xsl:when>
                                <xsl:when test="$actualtab = ('tabScenarios') and $theV2Scenarios">5</xsl:when>
                                <xsl:when test="$actualtab = ('tabScenarios')">4</xsl:when>
                                <xsl:when test="$actualtab = ('tabRules')">1</xsl:when>
                                <xsl:when test="$actualtab = ('tabTerminology')">1</xsl:when>
                                <xsl:when test="$actualtab = ('tabQuestionnaires')">1</xsl:when>
                                <xsl:otherwise>0</xsl:otherwise>
                            </xsl:choose>
                        </xsl:with-param>
                    </xsl:call-template>
                </head>
                <body>
                    <table class="title">
                        <tr>
                            <td style="text-align: left;">
                                <h1 style="margin-left: 5px; margin-bottom: 0px;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'decorProjectInfoString'"/>
                                        <xsl:with-param name="p1">
                                            <xsl:choose>
                                                <xsl:when test="string-length(//project/name[@language = $defaultLanguage or not(@language)]) = 0">
                                                    <!-- fallback -->
                                                    <xsl:value-of select="(//project/name)[1]"/>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:value-of select="//project/name[@language = $defaultLanguage or not(@language)][1]"/>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:with-param>
                                        <xsl:with-param name="p2" select="//project/@prefix"/>
                                    </xsl:call-template>
                                </h1>
                                <table style="margin-left: 2px; margin-bottom: 10px;">
                                    <tr>
                                        <td style="font-weight: bold; color: #665a4e;">
                                            <xsl:if test="//project/(version | release)">
                                                <!-- is current publication a release or just a version? -->
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key">
                                                        <xsl:choose>
                                                            <xsl:when test="$publicationIsRelease">decorProjectReleaseAsOf</xsl:when>
                                                            <xsl:otherwise>decorProjectVersionAsOf</xsl:otherwise>
                                                        </xsl:choose>
                                                    </xsl:with-param>
                                                    <xsl:with-param name="p1">
                                                        <xsl:call-template name="showDate">
                                                            <xsl:with-param name="date" select="$latestVersionOrRelease/@date"/>
                                                        </xsl:call-template>
                                                    </xsl:with-param>
                                                    <xsl:with-param name="p2" select="($latestVersionOrRelease/@versionLabel[not(. = '')], ' ')[1]"/>
                                                </xsl:call-template>
                                            </xsl:if>
                                        </td>
                                        <td>
                                            <xsl:text>|</xsl:text>
                                        </td>
                                        <td style="font-weight: bold; color: #665a4e;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'htmlExtractAsOf'"/>
                                            </xsl:call-template>
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$currentDateTime"/>
                                            </xsl:call-template>
                                        </td>
                                        <td>
                                            <xsl:text>|</xsl:text>
                                        </td>
                                        <td style="font-weight: bold; color: #665a4e;">
                                            <xsl:if test="$inDevelopment">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'developmentVersion'"/>
                                                </xsl:call-template>
                                            </xsl:if>
                                        </td>
                                    </tr>
                                </table>
                            </td>
                            <xsl:choose>
                                <xsl:when test="$useCustomLogo = true()">
                                    <td style="text-align: right;">
                                        <xsl:choose>
                                            <xsl:when test="string-length($useCustomLogoHREF) &gt; 0">
                                                <a href="{$useCustomLogoHREF}">
                                                    <img class="title" src="{$useCustomLogoSRC}" alt=""/>
                                                </a>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <img class="title" src="{$useCustomLogoSRC}" alt=""/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </td>
                                </xsl:when>
                                <xsl:otherwise>
                                    <!--<td style="text-align: right;">
                                        <img class="title" src="{$theAssetsDir}art-decor-logo-small.jpg" alt=""/>
                                    </td>-->
                                </xsl:otherwise>
                            </xsl:choose>
                        </tr>
                    </table>
                    <ul id="TabbedPanel">
                        <xsl:for-each select="$tabnameslist/tab">
                            <xsl:variable name="thistab" select="@key"/>
                            <xsl:variable name="thisfn" select="@filename"/>
                            <xsl:choose>
                                <xsl:when test="$actualtab = $thistab">
                                    <li class="TabbedPanelsTabSelected">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="$thistab"/>
                                        </xsl:call-template>
                                    </li>
                                </xsl:when>
                                <xsl:otherwise>
                                    <li class="TabbedPanelsTabNotSelected">
                                        <a href="{$thisfn}">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="$thistab"/>
                                            </xsl:call-template>
                                        </a>
                                    </li>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:for-each>
                    </ul>
                    <xsl:copy-of select="$actualcontent"/>
                    <p>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'decorFooterText'"/>
                        </xsl:call-template>
                    </p>
                    <xsl:call-template name="addAssetsBottom">
                        <xsl:with-param name="doTreeTable" select="$actualtab = ('tabDataSet', 'tabScenarios', 'tabRules', 'tabQuestionnaires', 'tabTerminology')"/>
                    </xsl:call-template>
                </body>
            </html>
        </xsl:result-document>
    </xsl:template>
    <xsl:template name="convertDECOR2HTML">
        <!-- 
            build html documentation, consists of
            - front page (index)
            - project information page (project)
            - data sets page (datasets)
            - scenarios page (scenarios)
            - identifiers page (identifiers)
            - terminology page (terminology)
            - rules page (rules)
            - issues page (issues)
            - compile time page (compiletime)
            - legal page (legal)
        -->
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Front Page Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <!-- RELEASE.txt or VERSION.txt
            This file helps publication on a website
        -->
        <xsl:variable name="releaseType" select="
                if ($publicationIsRelease = true()) then
                    ('RELEASE')
                else
                    ('VERSION')"/>
        <xsl:result-document href="{$theHtmlDir}{$releaseType}.txt" method="text">
            <xsl:value-of select="$latestVersionOrRelease/@versionLabel"/>
        </xsl:result-document>
        <!-- The index.php pulls this in without escaping so we escape this part -->
        <xsl:result-document href="{$theHtmlDir}aaacomments" method="xml" omit-xml-declaration="yes">
            <xsl:value-of select="$latestVersionOrRelease/@versionLabel"/>
        </xsl:result-document>
        
        <!-- Front Page
            ============
        -->
        <xsl:call-template name="doFrontPageTab"/>
        <!-- Project Information
            =====================
        -->
        <xsl:call-template name="doProjectTab"/>
        <!-- Data Sets
            ===========
        -->
        <xsl:if test="$tabnameslist//tab[@key = 'tabDataSet']">
            <xsl:call-template name="doDatasetsTab"/>
        </xsl:if>
        <!-- Scenarios
            ===========
        -->
        <xsl:if test="$tabnameslist//tab[@key = 'tabScenarios']">
            <xsl:call-template name="doScenarioTab"/>
        </xsl:if>
        <!-- Identifiers
            =============
        -->
        <xsl:if test="$tabnameslist//tab[@key = 'tabIdentifiers']">
            <xsl:call-template name="doIdentificationTab"/>
        </xsl:if>
        <!-- Terminology
            =============
        -->
        <xsl:if test="$tabnameslist//tab[@key = 'tabTerminology']">
            <xsl:call-template name="doTerminologyTab"/>
        </xsl:if>
        <!-- Rules
            =======
        -->
        <xsl:if test="$tabnameslist//tab[@key = 'tabRules']">
            <xsl:call-template name="doRulesTab"/>
        </xsl:if>
        <!-- Questionnaires
            =======
        -->
        <xsl:if test="$tabnameslist//tab[@key = 'tabQuestionnaires']">
            <xsl:call-template name="doQuestionnairesTab"/>
        </xsl:if>
        <!-- Issues
            ========
        -->
        <xsl:if test="$tabnameslist//tab[@key = 'tabIssues']">
            <xsl:call-template name="doIssuesTab"/>
        </xsl:if>
        <!-- Compile Time
            ==============
        -->
        <xsl:call-template name="doCompileTimeTab"/>
        <!-- Legal
            =======
        -->
        <xsl:call-template name="doLegalTab"/>
        <!-- 
            Filtered? Add the filtered file
        -->
        <xsl:if test="$filtersfileavailable = true()">
            <xsl:variable name="filterindf" select="concat($outputBaseUriPrefix, $projectPrefix, 'filter-', $theTimeStamp, '.txt')"/>
            <xsl:result-document href="{$filterindf}" method="text">
                <xsl:call-template name="getMessage">
                    <xsl:with-param name="key" select="'partialPublicationInfolist'"/>
                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                    <xsl:with-param name="p1" select="doc($filtersfile)/*/@label"/>
                </xsl:call-template>
            </xsl:result-document>
        </xsl:if>
    </xsl:template>
    <xsl:template name="doFrontPageTab">
        <xsl:variable name="content1">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <!-- 
                    this is a table
                    two or more columns
                    left colum is 2 rows
                      top: logo 
                      bottom: project info + copyright
                    right column is n rows (as much as copyrights)
                -->
                <table cellspacing="3" cellpadding="15" style="width: 100%; border: 0; background-color: #FFFFFF;">
                    <tr style="vertical-align: top;">
                        <td style="max-width: 50%;">
                            <img src="{$theAssetsDir}3dartdecor2016.jpg" style="max-width: 200px;" alt="ART-DECOR logo"/>
                        </td>
                        <td rowspan="3">
                            <xsl:if test="//project/copyright[@logo | *]">
                                <table border="0" cellspacing="7" cellpadding="11" style="background-color: #FFFFFF;">
                                    <xsl:for-each select="//project/copyright[@logo | *]">
                                        <tr>
                                            <td style="width: 3px; background-color: #cecbc6;">
                                                <!-- place a grey line before each logo/address -->
                                            </td>
                                            <td style="width: 1%;">
                                                <!-- place a logo if specified, check projectprefix-logos/@logo -->
                                                <xsl:if test="@logo">
                                                    <xsl:variable name="theLogo">
                                                        <xsl:value-of select="$theLogosDir"/>
                                                        <xsl:value-of select="@logo"/>
                                                    </xsl:variable>
                                                    <img src="{$theLogo}" style="max-width:200px; max-height:70px;" alt="logo"/>
                                                </xsl:if>
                                            </td>
                                            <td style="vertical-align: top;">
                                                <!-- show all adrress lines -->
                                                <xsl:for-each select="addrLine">
                                                    <xsl:choose>
                                                        <xsl:when test="@type = 'uri'">
                                                            <a href="{.}">
                                                                <xsl:value-of select="."/>
                                                            </a>
                                                        </xsl:when>
                                                        <xsl:when test="@type = ('twitter', 'linkedin', 'facebook')">
                                                            <a href="{.}" title="{@type}">
                                                                <xsl:call-template name="showIcon">
                                                                    <xsl:with-param name="which" select="@type"/>
                                                                </xsl:call-template>
                                                            </a>
                                                        </xsl:when>
                                                        <xsl:otherwise>
                                                            <xsl:value-of select="."/>
                                                        </xsl:otherwise>
                                                    </xsl:choose>
                                                    <br/>
                                                </xsl:for-each>
                                            </td>
                                        </tr>
                                    </xsl:for-each>
                                </table>
                            </xsl:if>
                        </td>
                    </tr>
                    <tr style="vertical-align: top;">
                        <td>
                            <h1>
                                <xsl:choose>
                                    <xsl:when test="string-length(//project/name[@language = $defaultLanguage or not(@language)]) = 0">
                                        <!-- fallback -->
                                        <xsl:value-of select="(//project/name)[1]"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="//project/name[@language = $defaultLanguage or not(@language)][1]"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </h1>
                        </td>
                    </tr>
                    <tr style="vertical-align: top;">
                        <td>
                            <h4>
                                <xsl:if test="//project/(version | release)">
                                    <!-- is current publication a release or just a version? -->
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key">
                                            <xsl:choose>
                                                <xsl:when test="$publicationIsRelease">decorProjectReleaseAsOf</xsl:when>
                                                <xsl:otherwise>decorProjectVersionAsOf</xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:with-param>
                                        <xsl:with-param name="p1">
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$latestVersionOrRelease/@date"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                        <xsl:with-param name="p2" select="($latestVersionOrRelease/@versionLabel[not(. = '')], ' ')[1]"/>
                                    </xsl:call-template>
                                    <br/>
                                </xsl:if>
                                <xsl:if test="$projectIsRepository or $projectIsPrivate">
                                    <xsl:call-template name="doPrivateRepositoryNotice"/>
                                    <br/>
                                </xsl:if>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'htmlExtractAsOf'"/>
                                </xsl:call-template>
                                <xsl:call-template name="showDate">
                                    <xsl:with-param name="date" select="$currentDateTime"/>
                                </xsl:call-template>
                                <xsl:if test="$inDevelopment">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'developmentVersion'"/>
                                    </xsl:call-template>
                                </xsl:if>
                            </h4>
                            <p/>
                            <a href="..">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'IndexOfAllPublications'"/>
                                </xsl:call-template>
                            </a>
                            <p/>
                            <xsl:if test="$filtersfileavailable = true()">
                                <div style="display:block; background-color: #f0fceb; padding: 10px;">
                                    <xsl:call-template name="showIcon">
                                        <xsl:with-param name="which">partialpublication</xsl:with-param>
                                    </xsl:call-template>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:text> </xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'partialPublication'"/>
                                    </xsl:call-template>
                                    <hr/>
                                    <xsl:value-of select="doc($filtersfile)/*/normalize-space(@label)"/>
                                </div>
                                <p/>
                            </xsl:if>
                            <xsl:for-each select="//project/license[string-length(@code) > 0]">
                                <xsl:variable name="licenseCode" select="@code"/>
                                <xsl:variable name="licenseDisplayName" select="@displayName[not(. = '')]"/>
                                <p>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'decorProjectLicense'"/>
                                        <xsl:with-param name="p1" select="@code"/>
                                        <xsl:with-param name="p2">
                                            <xsl:choose>
                                                <xsl:when test="$licenseDisplayName">
                                                    <xsl:value-of select="$licenseDisplayName"/>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="getXFormsLabel">
                                                        <xsl:with-param name="simpleTypeKey" select="'LicenseTypes'"/>
                                                        <xsl:with-param name="simpleTypeValue" select="$licenseCode"/>
                                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </p>
                            </xsl:for-each>
                            <xsl:for-each select="//project/copyright[string-length(@years) > 0][string-length(@by) > 0]">
                                <xsl:sort select="(@type, 'author')[1]"/>
                                <p>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'decorProjectCopyrightBy'"/>
                                        <xsl:with-param name="p1">
                                            <xsl:variable name="thisYear" select="format-date(current-date(), '[Y0001]')"/>
                                            <xsl:variable name="years" as="xs:string*">
                                                <xsl:for-each select="tokenize(@years, '\s')">
                                                    <xsl:choose>
                                                        <!-- thisYear- will be thisYear -->
                                                        <xsl:when test=". = concat($thisYear, '-')">
                                                            <xsl:value-of select="substring-before(., '-')"/>
                                                        </xsl:when>
                                                        <!-- 2012- will be 2012-thisYear -->
                                                        <xsl:when test="matches(., '^\d{4}-$')">
                                                            <xsl:value-of select="concat(., $thisYear)"/>
                                                        </xsl:when>
                                                        <!-- else leave as-is -->
                                                        <xsl:otherwise>
                                                            <xsl:value-of select="."/>
                                                        </xsl:otherwise>
                                                    </xsl:choose>
                                                </xsl:for-each>
                                            </xsl:variable>
                                            <xsl:value-of select="string-join($years, ' ')"/>
                                        </xsl:with-param>
                                        <xsl:with-param name="p2" select="@by"/>
                                        <xsl:with-param name="p3">
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'CopyrightType'"/>
                                                <xsl:with-param name="simpleTypeValue" select="(@type, 'author')[1]"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </p>
                            </xsl:for-each>
                        </td>
                    </tr>
                    <!-- 
                        PATCH ZONE KH 20220718
                        For the projects 'hl7itig' and 'demp5' the scenarios and transactions are listed if it is a scenario based filter.
                        Such a page acts as a "index" page summarizing the filtered artefacts with all descriptions displayed.
                        In the future this shall be a switch (most likely).
                     -->
                    <xsl:if test="$filtersfileavailable = true() and $projectPrefix = ('demo5-', 'hl7itig')">
                        <tr>
                            <td>
                                <h1>
                                    <xsl:choose>
                                        <xsl:when test="string-length(//project/name[@language = $defaultLanguage or not(@language)]) = 0">
                                            <!-- fallback -->
                                            <xsl:value-of select="(//project/name)[1]"/>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:value-of select="//project/name[@language = $defaultLanguage or not(@language)][1]"/>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text> – </xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'partialPublication'"/>
                                    </xsl:call-template>
                                    <xsl:text> </xsl:text>
                                    <i>
                                        <xsl:value-of select="doc($filtersfile)/*/normalize-space(@label)"/>
                                    </i>
                                </h1>
                                <xsl:for-each select="doc($filtersfile)/*/(scenario|transaction)">
                                    <xsl:variable name="refid" select="@ref"/>
                                    <xsl:variable name="refeff" select="@flexibility"/>
                                    <xsl:variable name="stobj" select="$allScenarios/scenarios/scenario[@id=$refid][@effectiveDate=$refeff] | $allScenarios//scenarios/scenario//transaction[@id=$refid][@effectiveDate=$refeff]"/>
                                    <xsl:variable name="objname" select="name(.)"/>
                                    <xsl:variable name="trgroup" select="$stobj/@type = 'group'"/>
                                    <!-- heading with a little bit of style tweaks towards our standards CSS here -->
                                    <xsl:choose>
                                        <xsl:when test="$objname = 'scenario'">
                                            <h2 style="background-color: inherit;">
                                                <xsl:value-of select="@name"/>
                                                <xsl:text> (</xsl:text>
                                                <xsl:value-of select="$objname"/>
                                                <xsl:text>)</xsl:text>
                                            </h2>
                                        </xsl:when>
                                        <xsl:when test="$trgroup">
                                            <h3 style="background-color: inherit; padding-left: 0; color: #665a4e;">
                                                <xsl:value-of select="@name"/>
                                                <xsl:text> (</xsl:text>
                                                <xsl:value-of select="$objname"/>
                                                <xsl:text> group)</xsl:text>
                                            </h3>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <h4 style="background-color: inherit; padding-left: 0; color: #9a9a9a;">
                                                <xsl:value-of select="@name"/>
                                                <xsl:text> (</xsl:text>
                                                <xsl:value-of select="$objname"/>
                                                <xsl:text>)</xsl:text>
                                            </h4>
                                            <xsl:for-each select="@*"><xsl:value-of select="name(.)"/></xsl:for-each>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Id'"/>
                                    </xsl:call-template>
                                    <xsl:text> </xsl:text>
                                    <xsl:value-of select="$refid"/>
                                    <xsl:text> (</xsl:text>
                                    <xsl:call-template name="showDate">
                                        <xsl:with-param name="date" select="$refeff"/>
                                    </xsl:call-template>
                                    <xsl:text>)</xsl:text>
                                    <xsl:if test="$objname = 'scenario'">
                                        <xsl:text> </xsl:text>
                                        <a href="{local:doHtmlName('SC', $projectPrefix, $refid, $refeff, $refid, $refeff, (), (), '.html', 'false')}" onclick="target='_blank';">
                                            <xsl:attribute name="title">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'Details'"/>
                                                </xsl:call-template>
                                            </xsl:attribute>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Details'"/>
                                            </xsl:call-template>
                                        </a>
                                    </xsl:if>
                                    <p/>
                                    <xsl:call-template name="doDescription">
                                        <xsl:with-param name="ns" select="$stobj/desc"/>
                                    </xsl:call-template>
                                </xsl:for-each>
                            </td>
                        </tr>
                    </xsl:if>
                </table>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabFrontPage'"/>
            <xsl:with-param name="actualcontent" select="$content1"/>
        </xsl:call-template>
    </xsl:template>
    <xsl:template name="doProjectTab">
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Project Page Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="content2">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <h2>
                    <a name="labelProject"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabProjectInfoTitleString'"/>
                    </xsl:call-template>
                </h2>
                <xsl:call-template name="doProjectInfo"/>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabProjectInformation'"/>
            <xsl:with-param name="actualcontent" select="$content2"/>
        </xsl:call-template>
    </xsl:template>
    <xsl:template name="doScenarioTab">
        <!-- certain columns are hidden initially by default when treetable is active.
            That doesn't work with classic so we don't hide them
        -->
        <xsl:variable name="coldisplay">
            <xsl:if test="$switchCreateTreeTableHtml = 'true'">'display: none;'</xsl:if>
        </xsl:variable>
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Scenarios Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="content4">
            <xsl:variable name="doDatasets" select="exists(//representingTemplate/@sourceDataset)" as="xs:boolean"/>
            <xsl:variable name="doTemplates" select="exists(//representingTemplate/@ref)" as="xs:boolean"/>
            <xsl:variable name="doModels" select="exists(//transaction[$doTemplates]/@model)" as="xs:boolean"/>
            <xsl:variable name="doQuestionnaires" select="exists(//representingTemplate/@representingQuestionnaire)" as="xs:boolean"/>
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <h2>
                    <a name="labelScenarios"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabScenariosTitleString'"/>
                    </xsl:call-template>
                </h2>
                <div class="tabtab">
                    <xsl:choose>
                        <xsl:when test="count(//scenarios/scenario) = 0">
                            <table style="border: 0;">
                                <xsl:call-template name="doMessage">
                                    <xsl:with-param name="level" select="'info'"/>
                                    <xsl:with-param name="msg">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueNoScenariosDefinedYet'"/>
                                        </xsl:call-template>
                                    </xsl:with-param>
                                </xsl:call-template>
                            </table>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                <div>
                                    <button id="expandAll" type="button">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                        </xsl:call-template>
                                    </button>
                                    <button id="collapseAll" type="button">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                        </xsl:call-template>
                                    </button>
                                    <input id="nameSearch">
                                        <xsl:attribute name="placeholder">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'textSearch'"/>
                                            </xsl:call-template>
                                        </xsl:attribute>
                                    </input>
                                    <select id="hiddenColumns">
                                        <option value="title">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'showColumn'"/>
                                            </xsl:call-template>
                                        </option>
                                        <option value="columnDescription">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Description'"/>
                                            </xsl:call-template>
                                        </option>
                                        <option value="columnEffectiveDate">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'effectiveDate'"/>
                                            </xsl:call-template>
                                        </option>
                                        <xsl:if test="//scenarios/scenario[@expirationDate[not(. = '')]]">
                                            <option value="columnExpirationDate">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'expirationDate'"/>
                                                </xsl:call-template>
                                            </option>
                                        </xsl:if>
                                        <option value="columnVersionLabel" disabled="disabled">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                            </xsl:call-template>
                                        </option>
                                        <option value="columnModel">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Model'"/>
                                            </xsl:call-template>
                                        </option>
                                        <option value="columnDataset" disabled="disabled">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnDataset'"/>
                                            </xsl:call-template>
                                        </option>
                                        <option value="columnTemplate" disabled="disabled">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'template'"/>
                                            </xsl:call-template>
                                        </option>
                                    </select>
                                </div>
                            </xsl:if>
                            <table id="transactionTable" class="treetable highlight" width="100%" border="0" cellspacing="0" cellpadding="5">
                                <thead>
                                    <tr>
                                        <th class="columnDiagram" style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Diagram'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th class="columnXML" style="width: 50px;">
                                            <xsl:text>XML</xsl:text>
                                        </th>
                                        <th class="columnAllView" style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnAllView'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th class="columnCareView" style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnCareView'"/>
                                            </xsl:call-template>
                                        </th>
                                        <xsl:if test="$theV2Scenarios">
                                            <th class="columnV2IGView" style="width: 50px;">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'columnV2IGView'"/>
                                                </xsl:call-template>
                                            </th>
                                        </xsl:if>
                                        <th style="min-width: 40%; max-width: 50%;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnName'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th class="columnDescription" style="{$coldisplay}">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Description'"/>
                                            </xsl:call-template>
                                            <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                                <xsl:text>&#160;</xsl:text>
                                                <span class="hideMe" type="button">[‑]</span>
                                            </xsl:if>
                                        </th>
                                        <th class="columnEffectiveDate" style="width: 150px; {$coldisplay}">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'effectiveDate'"/>
                                            </xsl:call-template>
                                            <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                                <xsl:text>&#160;</xsl:text>
                                                <span class="hideMe" type="button">[‑]</span>
                                            </xsl:if>
                                        </th>
                                        <xsl:if test="//scenarios/scenario[@expirationDate[not(. = '')]]">
                                            <th class="columnExpirationDate" style="width: 150px; {$coldisplay}">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'expirationDate'"/>
                                                </xsl:call-template>
                                                <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                                    <xsl:text>&#160;</xsl:text>
                                                    <span class="hideMe" type="button">[‑]</span>
                                                </xsl:if>
                                            </th>
                                        </xsl:if>
                                        <th class="columnVersionLabel" style="width: 150px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                            </xsl:call-template>
                                            <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                                <xsl:text>&#160;</xsl:text>
                                                <span class="hideMe" type="button">[‑]</span>
                                            </xsl:if>
                                        </th>
                                        <xsl:if test="$doDatasets">
                                            <th class="columnDataset">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'columnDataset'"/>
                                                </xsl:call-template>
                                                <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                                    <xsl:text>&#160;</xsl:text>
                                                    <span class="hideMe" type="button">[‑]</span>
                                                </xsl:if>
                                            </th>
                                        </xsl:if>
                                        <xsl:if test="$doModels">
                                            <th class="columnModel" style="{$coldisplay}">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'Model'"/>
                                                </xsl:call-template>
                                                <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                                    <xsl:text>&#160;</xsl:text>
                                                    <span class="hideMe" type="button">[‑]</span>
                                                </xsl:if>
                                            </th>
                                        </xsl:if>
                                        <xsl:if test="$doTemplates">
                                            <th class="columnTemplate">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'template'"/>
                                                </xsl:call-template>
                                                <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                                    <xsl:text>&#160;</xsl:text>
                                                    <span class="hideMe" type="button">[‑]</span>
                                                </xsl:if>
                                            </th>
                                        </xsl:if>
                                        <xsl:if test="$doQuestionnaires">
                                            <th class="columnQuestionnaire">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'Questionnaire'"/>
                                                </xsl:call-template>
                                                <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                                    <xsl:text>&#160;</xsl:text>
                                                    <span class="hideMe" type="button">[‑]</span>
                                                </xsl:if>
                                            </th>
                                        </xsl:if>
                                    </tr>
                                </thead>
                                <tbody>
                                    <xsl:for-each select="//scenarios/scenario | //scenarios/scenario//transaction">
                                        <xsl:sort select="ancestor-or-self::scenario/@effectiveDate" order="descending"/>
                                        <xsl:variable name="level" select="count(ancestor::scenario | ancestor::transaction) + 1"/>
                                        <xsl:variable name="itemAnchor" select="local:doHtmlAnchor(@id, @effectiveDate)"/>
                                        <xsl:variable name="scid" select="ancestor-or-self::scenario/@id"/>
                                        <xsl:variable name="sceff" select="ancestor-or-self::scenario/@effectiveDate"/>
                                        <!-- cache transaction/@effectiveDate. This is relatively new so might not be present -->
                                        <xsl:variable name="trid" select="self::transaction/@id"/>
                                        <xsl:variable name="tred" select="self::transaction/@effectiveDate"/>
                                        <xsl:variable name="transactionAnchor">
                                            <xsl:if test="$trid">
                                                <xsl:text>#</xsl:text>
                                                <xsl:value-of select="local:doHtmlAnchor($trid, $tred)"/>
                                            </xsl:if>
                                        </xsl:variable>
                                        <xsl:variable name="node-id" select="generate-id(.)"/>
                                        <xsl:variable name="node-class" as="xs:string">
                                            <xsl:choose>
                                                <xsl:when test="self::scenario">scline</xsl:when>
                                                <xsl:when test="self::transaction[@type = 'group']">trgroupline</xsl:when>
                                                <xsl:otherwise>trline</xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:variable>
                                        <xsl:variable name="sc-style" select="
                                                if (self::scenario) then
                                                    'border-top: 6px solid white;'
                                                else
                                                    ()"/>
                                        <tr data-tt-id="{$node-id}" class="{$node-class}">
                                            <xsl:if test="parent::scenario | parent::transaction">
                                                <xsl:attribute name="data-tt-parent-id" select="generate-id(parent::*)"/>
                                            </xsl:if>
                                            <!-- Diagram -->
                                            <td class="columnDiagram" style="{$sc-style}">
                                                <xsl:if test="count(representingTemplate/concept) &gt; 0 and string-length($projectRestUriDSDiagram) &gt; 0">
                                                    <a href="{$projectRestUriDSDiagram}&amp;transactionId={@id}&amp;transactionEffectiveDate={encode-for-uri(@effectiveDate)}&amp;format=svg" onclick="target='_blank';">link</a>
                                                </xsl:if>
                                            </td>
                                            <!-- XML -->
                                            <td class="columnXML" style="{$sc-style}">
                                                <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                <xsl:if test="count(representingTemplate/concept) &gt; 0">
                                                    <xsl:if test="string-length($projectRestUriDS) &gt; 0">
                                                        <a href="{$projectRestUriDS}&amp;id={$trid}&amp;effectiveDate={encode-for-uri($tred)}&amp;format=xml" onclick="target='_blank';">xml</a>
                                                    </xsl:if>
                                                </xsl:if>
                                            </td>
                                            <!-- columnAllView -->
                                            <!-- FIXME? the ProjectIndex has a different understanding of "All" than this link has... -->
                                            <td class="columnAllView" style="{$sc-style}">
                                                <xsl:choose>
                                                    <xsl:when test="self::scenario | self::transaction[@type[. = 'group'] | transaction]">
                                                        <a href="{local:doHtmlName('SC', $projectPrefix, $scid, $sceff, $trid, $tred, (), (), '.html', 'false')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Details'"/>
                                                                </xsl:call-template>
                                                            </xsl:attribute>
                                                            <xsl:text>html</xsl:text>
                                                        </a>
                                                    </xsl:when>
                                                    <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                    <xsl:when test="count(representingTemplate/concept) &gt; 0">
                                                        <a href="{local:doHtmlName('TR',$trid,$tred,'.html')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'columnAllView'"/>
                                                                </xsl:call-template>
                                                            </xsl:attribute>
                                                            <xsl:text>html</xsl:text>
                                                        </a>
                                                    </xsl:when>
                                                </xsl:choose>
                                            </td>
                                            <!-- columnCareViewView -->
                                            <!-- FIXME? the ProjectIndex offers 2 flavors of care view. We only offer one version here... -->
                                            <td class="columnCareView" style="{$sc-style}">
                                                <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                <xsl:if test="representingTemplate/concept">
                                                    <xsl:if test="string-length($projectRestUriDS) &gt; 0">
                                                        <a href="{$projectRestUriDS}&amp;id={$trid}&amp;effectiveDate={encode-for-uri($tred)}&amp;format=html" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'columnCareView'"/>
                                                                </xsl:call-template>
                                                            </xsl:attribute>
                                                            <xsl:text>html</xsl:text>
                                                        </a>
                                                    </xsl:if>
                                                </xsl:if>
                                            </td>
                                            <!-- HL7 V2 IG -->
                                            <xsl:if test="$theV2Scenarios">
                                                <td class="columnV2IGView" style="{$sc-style}">
                                                    <xsl:if test="self::scenario">
                                                        <xsl:variable name="thisScenarioIG" select="$theV2Scenarios[@id = $scid][@effectiveDate = $sceff]" as="element(scenario)*"/>
                                                        <xsl:for-each select="$thisScenarioIG">
                                                            <div>
                                                                <a href="{local:doHtmlName('SC', $projectPrefix, $scid, $sceff, (), (), (), (), '_HL7V2-IG.html', 'false')}" onclick="target='_blank';">
                                                                    <xsl:attribute name="title">
                                                                        <xsl:call-template name="getMessage">
                                                                            <xsl:with-param name="key" select="'DetailsHL7V2IG'"/>
                                                                        </xsl:call-template>
                                                                    </xsl:attribute>
                                                                    <xsl:text>html</xsl:text>
                                                                </a>
                                                            </div>
                                                            <xsl:if test="$switchCreateSchematron">
                                                                <div>
                                                                    <a href="{local:doHtmlName('SC', $projectPrefix, $scid, $sceff, (), (), (), (), '_HL7V2-CP.xml', 'false')}" onclick="target='_blank';">
                                                                        <xsl:attribute name="title">
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'DetailsHL7V2CP'"/>
                                                                            </xsl:call-template>
                                                                        </xsl:attribute>
                                                                        <xsl:text>xml</xsl:text>
                                                                    </a>
                                                                </div>
                                                            </xsl:if>
                                                        </xsl:for-each>
                                                    </xsl:if>
                                                </td>
                                            </xsl:if>
                                            <!-- Name -->
                                            <td class="columnName" style="{$sc-style}">
                                                <a name="{$itemAnchor}"/>
                                                <xsl:variable name="columnName">
                                                    <xsl:call-template name="showStatusDot">
                                                        <xsl:with-param name="status" select="@statusCode"/>
                                                        <xsl:with-param name="title">
                                                            <xsl:call-template name="getXFormsLabel">
                                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                            </xsl:call-template>
                                                        </xsl:with-param>
                                                    </xsl:call-template>
                                                    <xsl:text>&#160;</xsl:text>
                                                    <xsl:choose>
                                                        <xsl:when test="self::scenario"/>
                                                        <xsl:when test="self::transaction[@type[. = 'group'] | transaction]">
                                                            <xsl:call-template name="showIcon">
                                                                <xsl:with-param name="which">doublearrow</xsl:with-param>
                                                                <xsl:with-param name="tooltip">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'Group'"/>
                                                                    </xsl:call-template>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                        </xsl:when>
                                                        <xsl:when test="self::transaction">
                                                            <xsl:call-template name="showDirection">
                                                                <xsl:with-param name="dir" select="@type"/>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                        </xsl:when>
                                                    </xsl:choose>
                                                    <xsl:call-template name="doName">
                                                        <xsl:with-param name="ns" select="name"/>
                                                    </xsl:call-template>
                                                </xsl:variable>
                                                <xsl:choose>
                                                    <xsl:when test="not($switchCreateTreeTableHtml = 'true')">
                                                        <table>
                                                            <tr>
                                                                <xsl:call-template name="doIndentLevel">
                                                                    <xsl:with-param name="level" select="$level"/>
                                                                    <xsl:with-param name="icon" select="false()"/>
                                                                </xsl:call-template>
                                                                <td>
                                                                    <xsl:copy-of select="$columnName"/>
                                                                </td>
                                                            </tr>
                                                        </table>
                                                    </xsl:when>
                                                    <xsl:otherwise>
                                                        <xsl:copy-of select="$columnName"/>
                                                    </xsl:otherwise>
                                                </xsl:choose>
                                            </td>
                                            <!-- Desc -->
                                            <td class="columnDescription" style="{$sc-style} {$coldisplay}">
                                                <xsl:call-template name="doDescription">
                                                    <xsl:with-param name="ns" select="desc"/>
                                                    <xsl:with-param name="shortDesc" select="true()"/>
                                                </xsl:call-template>
                                            </td>
                                            <!-- effectiveDate -->
                                            <td class="columnEffectiveDate" style="{$sc-style} {$coldisplay}">
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="@effectiveDate"/>
                                                </xsl:call-template>
                                            </td>
                                            <!-- expirationDate -->
                                            <xsl:if test="//scenarios/scenario[@expirationDate[not(. = '')]]">
                                                <td class="columnExpirationDate" style="{$coldisplay} {$sc-style}">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="@expirationDate"/>
                                                    </xsl:call-template>
                                                </td>
                                            </xsl:if>
                                            <!-- versionLabel -->
                                            <td class="columnVersionLabel" style="{$sc-style}">
                                                <xsl:value-of select="@versionLabel"/>
                                            </td>
                                            <!-- Dataset -->
                                            <xsl:if test="$doDatasets">
                                                <td class="columnDataset" style="{$sc-style}">
                                                    <xsl:for-each select="representingTemplate[@sourceDataset]">
                                                        <xsl:variable name="dsid" select="@sourceDataset"/>
                                                        <xsl:variable name="dsed" select="@sourceDatasetFlexibility"/>
                                                        <xsl:variable name="dataset" select="
                                                                if ($dsid) then
                                                                    local:getDataset($dsid, $dsed)
                                                                else
                                                                    ()" as="element()?"/>
                                                        <xsl:if test="$dsid and not($dataset)">
                                                            <xsl:call-template name="logMessage">
                                                                <xsl:with-param name="level" select="$logERROR"/>
                                                                <xsl:with-param name="msg">
                                                                    <xsl:text>+++ Could not find referenced dataset id='</xsl:text>
                                                                    <xsl:value-of select="$dsid"/>
                                                                    <xsl:text>' referenced from transaction '</xsl:text>
                                                                    <xsl:call-template name="doDescription">
                                                                        <xsl:with-param name="ns" select="../name"/>
                                                                    </xsl:call-template>
                                                                    <xsl:text>' (</xsl:text>
                                                                    <xsl:value-of select="../@id"/>
                                                                    <xsl:text> - </xsl:text>
                                                                    <xsl:value-of select="../@effectiveDate"/>
                                                                    <xsl:text>). This might be due to it being in a different project/BBR or because the sourceDataset/sourceDatasetFlexibility is broken.</xsl:text>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                        </xsl:if>
                                                        <a href="{local:doHtmlName('TR',$trid,$tred,'.html')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:call-template name="doName">
                                                                    <xsl:with-param name="ns" select="$dataset/name"/>
                                                                </xsl:call-template>
                                                                <xsl:if test="$dataset[@versionLabel]">
                                                                    <xsl:text> </xsl:text>
                                                                    <xsl:value-of select="$dataset/@versionLabel"/>
                                                                </xsl:if>
                                                                <xsl:text> </xsl:text>
                                                                <i>
                                                                    <xsl:text>(</xsl:text>
                                                                    <xsl:value-of select="$dsid"/>
                                                                    <xsl:text> / </xsl:text>
                                                                    <xsl:choose>
                                                                        <xsl:when test="$dsed castable as xs:dateTime">
                                                                            <xsl:call-template name="showDate">
                                                                                <xsl:with-param name="date" select="$dsed"/>
                                                                            </xsl:call-template>
                                                                        </xsl:when>
                                                                        <xsl:otherwise>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                            </xsl:call-template>
                                                                        </xsl:otherwise>
                                                                    </xsl:choose>
                                                                    <xsl:text>)</xsl:text>
                                                                </i>
                                                            </xsl:attribute>
                                                            <xsl:call-template name="doName">
                                                                <xsl:with-param name="ns" select="$dataset/name"/>
                                                            </xsl:call-template>
                                                            <xsl:if test="$dataset[@versionLabel]">
                                                                <xsl:text> </xsl:text>
                                                                <xsl:value-of select="$dataset/@versionLabel"/>
                                                            </xsl:if>
                                                        </a>
                                                    </xsl:for-each>
                                                </td>
                                            </xsl:if>
                                            <!-- Model -->
                                            <xsl:if test="$doModels">
                                                <td class="columnModel" style="{$sc-style} {$coldisplay}">
                                                    <xsl:value-of select="@model"/>
                                                </td>
                                            </xsl:if>
                                            <!-- Template -->
                                            <xsl:if test="$doTemplates">
                                                <td class="columnTemplate" style="{$sc-style}">
                                                    <xsl:for-each select="representingTemplate[@ref]">
                                                        <xsl:variable name="tmid" select="@ref"/>
                                                        <xsl:variable name="tmed" select="
                                                                if (@flexibility castable as xs:dateTime) then
                                                                    @flexibility
                                                                else
                                                                    ('dynamic')"/>
                                                        <xsl:variable name="rccontent" as="element()?">
                                                            <xsl:call-template name="getRulesetContent">
                                                                <xsl:with-param name="ruleset" select="$tmid"/>
                                                                <xsl:with-param name="flexibility" select="$tmed"/>
                                                                <xsl:with-param name="sofar" select="()"/>
                                                            </xsl:call-template>
                                                        </xsl:variable>
                                                        <xsl:variable name="tmname">
                                                            <xsl:choose>
                                                                <xsl:when test="$rccontent[@displayName]">
                                                                    <xsl:value-of select="$rccontent/@displayName"/>
                                                                </xsl:when>
                                                                <xsl:when test="$rccontent[@name]">
                                                                    <xsl:value-of select="$rccontent/@name"/>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:value-of select="$tmid"/>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </xsl:variable>
                                                        <a href="{local:doHtmlName('TM',$rccontent/@id,$tmed,'.html')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:value-of select="$tmname"/>
                                                                <xsl:text> </xsl:text>
                                                                <i>
                                                                    <xsl:text>(</xsl:text>
                                                                    <xsl:if test="$rccontent">
                                                                        <xsl:value-of select="$rccontent/(@id | @ref)"/>
                                                                        <xsl:text> / </xsl:text>
                                                                    </xsl:if>
                                                                    <xsl:choose>
                                                                        <xsl:when test="$tmed castable as xs:dateTime">
                                                                            <xsl:call-template name="showDate">
                                                                                <xsl:with-param name="date" select="$tmed"/>
                                                                            </xsl:call-template>
                                                                        </xsl:when>
                                                                        <xsl:otherwise>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                            </xsl:call-template>
                                                                        </xsl:otherwise>
                                                                    </xsl:choose>
                                                                    <xsl:text>)</xsl:text>
                                                                </i>
                                                            </xsl:attribute>
                                                            <xsl:value-of select="$tmname"/>
                                                        </a>
                                                    </xsl:for-each>
                                                </td>
                                            </xsl:if>
                                            <!-- Questionnaire -->
                                            <xsl:if test="$doQuestionnaires">
                                                <td class="columnQuestionnaire" style="{$sc-style}">
                                                    <xsl:for-each select="representingTemplate[@representingQuestionnaire]">
                                                        <xsl:variable name="qqid" select="@representingQuestionnaire"/>
                                                        <xsl:variable name="qqed" select="@representingQuestionnaireFlexibility"/>
                                                        <xsl:variable name="qq" as="element()?">
                                                            <xsl:call-template name="getQuestionnaireContent">
                                                                <xsl:with-param name="qqid" select="$qqid"/>
                                                                <xsl:with-param name="qqed" select="$qqed"/>
                                                            </xsl:call-template>
                                                        </xsl:variable>
                                                        <xsl:variable name="qqname">
                                                            <xsl:call-template name="doName">
                                                                <xsl:with-param name="ns" select="$qq/name"/>
                                                            </xsl:call-template>
                                                        </xsl:variable>
                                                        <a href="{local:doHtmlName('QQ',$qq/@id,$qqed,'.html')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:value-of select="$qqname[1]"/>
                                                                <xsl:text> </xsl:text>
                                                                <i>
                                                                    <xsl:text>(</xsl:text>
                                                                    <xsl:if test="$qq">
                                                                        <xsl:value-of select="$qq/(@id | @ref)"/>
                                                                        <xsl:text> / </xsl:text>
                                                                    </xsl:if>
                                                                    <xsl:choose>
                                                                        <xsl:when test="$qqed castable as xs:dateTime">
                                                                            <xsl:call-template name="showDate">
                                                                                <xsl:with-param name="date" select="$qqed"/>
                                                                            </xsl:call-template>
                                                                        </xsl:when>
                                                                        <xsl:otherwise>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                            </xsl:call-template>
                                                                        </xsl:otherwise>
                                                                    </xsl:choose>
                                                                    <xsl:text>)</xsl:text>
                                                                </i>
                                                            </xsl:attribute>
                                                            <xsl:copy-of select="$qqname"/>
                                                        </a>
                                                    </xsl:for-each>
                                                </td>
                                            </xsl:if>
                                        </tr>
                                    </xsl:for-each>
                                </tbody>
                            </table>
                        </xsl:otherwise>
                    </xsl:choose>
                </div>
                <!--  then show a summary of scenario table -->
                <h2>
                    <a name="labelScenarios"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'scenarioSummary'"/>
                    </xsl:call-template>
                </h2>
                <table width="50%" border="0" cellspacing="2" cellpadding="2">
                    <tr style="vertical-align: top;">
                        <td class="tabtab">
                            <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                <tr style="background-color: #CCFFCC;">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'scenarios'"/>
                                        </xsl:call-template>
                                    </td>
                                    <td colspan="2">
                                        <xsl:value-of select="count($allScenarios/scenarios/scenario)"/>
                                    </td>
                                </tr>
                                <tr style="background-color: #CCFFCC;">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'transactionGroups'"/>
                                        </xsl:call-template>
                                    </td>
                                    <td colspan="2">
                                        <xsl:value-of select="count($allScenarios/scenarios/scenario//transaction[@type = 'group'])"/>
                                    </td>
                                </tr>
                                <tr style="background-color: #CCFFCC;">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'transactions'"/>
                                        </xsl:call-template>
                                    </td>
                                    <td colspan="2">
                                        <xsl:value-of select="count($allScenarios/scenarios/scenario//transaction[not(@type = 'group')])"/>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                </table>
                <!-- create rule lists per scenario x -->
                <h2>
                    <a name="labelActors"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'transactionsPerActor'"/>
                    </xsl:call-template>
                </h2>
                <xsl:if test="count($allActors//actor) > 1">
                    <ol>
                        <xsl:for-each select="$allActors//actor">
                            <li>
                                <a href="#{local:doHtmlAnchor(@id, ())}">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="concat('actorType-', @type)"/>
                                    </xsl:call-template>
                                    <xsl:text>:&#160;</xsl:text>
                                    <xsl:call-template name="doName">
                                        <xsl:with-param name="ns" select="name"/>
                                    </xsl:call-template>
                                </a>
                            </li>
                        </xsl:for-each>
                    </ol>
                </xsl:if>
                <xsl:for-each select="$allActors//actor[@id]">
                    <xsl:variable name="actorid" select="@id"/>
                    <xsl:variable name="actorname">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="concat('actorType-', @type)"/>
                        </xsl:call-template>
                        <xsl:text>:&#160;</xsl:text>
                        <xsl:call-template name="doName">
                            <xsl:with-param name="ns" select="name"/>
                        </xsl:call-template>
                    </xsl:variable>
                    <a name="{local:doHtmlAnchor(@id, ())}"/>
                    <h3>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'transactionsForActor'"/>
                        </xsl:call-template>
                        <xsl:text> </xsl:text>
                        <i>
                            <xsl:value-of select="$actorname"/>
                        </i>
                    </h3>
                    <div class="tabtab">
                        <table class="highlight" width="100%" border="0" cellspacing="3" cellpadding="2">
                            <thead>
                                <tr>
                                    <th style="width: 10%;">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'actorRole'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th style="width: 30%;">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'Transaction'"/>
                                        </xsl:call-template>
                                    </th>
                                    <xsl:if test="$doModels">
                                        <th style="width: 150px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Model'"/>
                                            </xsl:call-template>
                                        </th>
                                    </xsl:if>
                                    <xsl:if test="$doTemplates">
                                        <th style="width: 20%;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Template'"/>
                                            </xsl:call-template>
                                        </th>
                                    </xsl:if>
                                    <xsl:if test="$doQuestionnaires">
                                        <th style="width: 20%;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Questionnaire'"/>
                                            </xsl:call-template>
                                        </th>
                                    </xsl:if>
                                    <th style="width: 2%;">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'Diagram'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th style="width: 2%">
                                        <xsl:text>XML</xsl:text>
                                    </th>
                                    <th style="width: 2%;">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'columnAllView'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th style="width: 2%;">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'columnCareView'"/>
                                        </xsl:call-template>
                                    </th>
                                </tr>
                            </thead>
                            <tbody>
                                <xsl:for-each select="$allScenarios/scenarios/scenario//transaction[@type = 'group'][transaction/actors/actor/@id = $actorid]">
                                    <tr class="trgroupline">
                                        <!-- Role -->
                                        <td style="text-align: left;">&#160;</td>
                                        <!-- Transaction name -->
                                        <td style="text-align: left;">
                                            <xsl:call-template name="showStatusDot">
                                                <xsl:with-param name="status" select="@statusCode"/>
                                                <xsl:with-param name="title">
                                                    <xsl:call-template name="getXFormsLabel">
                                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                    </xsl:call-template>
                                                </xsl:with-param>
                                            </xsl:call-template>
                                            <xsl:text>&#160;</xsl:text>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">doublearrow</xsl:with-param>
                                                <xsl:with-param name="tooltip">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'Group'"/>
                                                    </xsl:call-template>
                                                </xsl:with-param>
                                            </xsl:call-template>
                                            <xsl:text>&#160;</xsl:text>
                                            <i>
                                                <xsl:call-template name="doName">
                                                    <xsl:with-param name="ns" select="name"/>
                                                </xsl:call-template>
                                            </i>
                                        </td>
                                        <!-- Model -->
                                        <xsl:if test="$doModels">
                                            <td style="text-align: left;">&#160;</td>
                                        </xsl:if>
                                        <!-- Template -->
                                        <xsl:if test="$doTemplates">
                                            <td style="text-align: left;">&#160;</td>
                                        </xsl:if>
                                        <!-- Questionnaire -->
                                        <xsl:if test="$doQuestionnaires">
                                            <td style="text-align: left;">&#160;</td>
                                        </xsl:if>
                                        <!-- Diagram -->
                                        <td style="text-align: left;">&#160;</td>
                                        <!-- XML -->
                                        <td style="text-align: left;">&#160;</td>
                                        <!-- columnAllView -->
                                        <td style="text-align: left;">&#160;</td>
                                        <!-- columnCareViewView -->
                                        <td style="text-align: left;">&#160;</td>
                                    </tr>
                                    <xsl:for-each select="transaction[actors/actor/@id = $actorid]">
                                        <xsl:variable name="scid" select="ancestor::scenario/@id"/>
                                        <xsl:variable name="sced" select="ancestor::scenario/@effectiveDate"/>
                                        <xsl:variable name="trid" select="@id"/>
                                        <xsl:variable name="tred" select="@effectiveDate"/>
                                        <xsl:variable name="transactionAnchor" select="local:doHtmlAnchor($trid, $tred)"/>
                                        <tr class="trline">
                                            <!-- Role -->
                                            <td style="text-align: left;">
                                                <xsl:for-each select="actors/actor[@id = $actorid]">
                                                    <xsl:choose>
                                                        <xsl:when test="@role = 'sender' or @role = 'receiver'">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="concat('actorRole-', @role)"/>
                                                            </xsl:call-template>
                                                        </xsl:when>
                                                        <xsl:otherwise>
                                                            <xsl:value-of select="@role"/>
                                                        </xsl:otherwise>
                                                    </xsl:choose>
                                                    <xsl:if test="position() != last()">
                                                        <xsl:text> / </xsl:text>
                                                    </xsl:if>
                                                </xsl:for-each>
                                            </td>
                                            <!-- Transaction name -->
                                            <td style="text-align: left;">
                                                <span style=" margin-left: 20px;"/>
                                                <xsl:call-template name="showStatusDot">
                                                    <xsl:with-param name="status" select="@statusCode"/>
                                                    <xsl:with-param name="title">
                                                        <xsl:call-template name="getXFormsLabel">
                                                            <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                            <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                        </xsl:call-template>
                                                    </xsl:with-param>
                                                </xsl:call-template>
                                                <xsl:text>&#160;</xsl:text>
                                                <xsl:call-template name="showDirection">
                                                    <xsl:with-param name="dir" select="@type"/>
                                                </xsl:call-template>
                                                <xsl:text>&#160;</xsl:text>
                                                <xsl:call-template name="doName">
                                                    <xsl:with-param name="ns" select="name"/>
                                                </xsl:call-template>
                                            </td>
                                            <!-- Model -->
                                            <xsl:if test="$doModels">
                                                <td style="text-align: left;">
                                                    <xsl:value-of select="@model"/>
                                                </td>
                                            </xsl:if>
                                            <!-- Template -->
                                            <xsl:if test="$doTemplates">
                                                <td>
                                                    <xsl:for-each select="representingTemplate[@ref]">
                                                        <xsl:variable name="tmid" select="@ref"/>
                                                        <xsl:variable name="tmed" select="
                                                                if (@flexibility castable as xs:dateTime) then
                                                                    @flexibility
                                                                else
                                                                    ('dynamic')"/>
                                                        <xsl:variable name="rccontent" as="element()?">
                                                            <xsl:call-template name="getRulesetContent">
                                                                <xsl:with-param name="ruleset" select="$tmid"/>
                                                                <xsl:with-param name="flexibility" select="$tmed"/>
                                                                <xsl:with-param name="sofar" select="()"/>
                                                            </xsl:call-template>
                                                        </xsl:variable>
                                                        <xsl:variable name="tmname">
                                                            <xsl:choose>
                                                                <xsl:when test="$rccontent[@displayName]">
                                                                    <xsl:value-of select="$rccontent/@displayName"/>
                                                                </xsl:when>
                                                                <xsl:when test="$rccontent[@name]">
                                                                    <xsl:value-of select="$rccontent/@name"/>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:value-of select="$tmid"/>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </xsl:variable>
                                                        <a href="{local:doHtmlName('TM',$rccontent/@id,$tmed,'.html')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:value-of select="$tmname"/>
                                                                <xsl:text> </xsl:text>
                                                                <i>
                                                                    <xsl:text>(</xsl:text>
                                                                    <xsl:if test="$rccontent">
                                                                        <xsl:value-of select="$rccontent/(@id | @ref)"/>
                                                                        <xsl:text> / </xsl:text>
                                                                    </xsl:if>
                                                                    <xsl:choose>
                                                                        <xsl:when test="$tmed castable as xs:dateTime">
                                                                            <xsl:call-template name="showDate">
                                                                                <xsl:with-param name="date" select="$tmed"/>
                                                                            </xsl:call-template>
                                                                        </xsl:when>
                                                                        <xsl:otherwise>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                            </xsl:call-template>
                                                                        </xsl:otherwise>
                                                                    </xsl:choose>
                                                                    <xsl:text>)</xsl:text>
                                                                </i>
                                                            </xsl:attribute>
                                                            <xsl:value-of select="$tmname"/>
                                                        </a>
                                                    </xsl:for-each>
                                                </td>
                                            </xsl:if>
                                            <!-- Questionnaire -->
                                            <xsl:if test="$doQuestionnaires">
                                                <td>
                                                    <xsl:for-each select="representingTemplate[@representingQuestionnaire]">
                                                        <xsl:variable name="qqid" select="@representingQuestionnaire"/>
                                                        <xsl:variable name="qqed" select="@representingQuestionnaireFlexibility"/>
                                                        <xsl:variable name="qq" as="element()?">
                                                            <xsl:call-template name="getQuestionnaireContent">
                                                                <xsl:with-param name="qqid" select="$qqid"/>
                                                                <xsl:with-param name="qqed" select="$qqed"/>
                                                            </xsl:call-template>
                                                        </xsl:variable>
                                                        <xsl:variable name="qqname">
                                                            <xsl:call-template name="doName">
                                                                <xsl:with-param name="ns" select="$qq/name"/>
                                                            </xsl:call-template>
                                                        </xsl:variable>
                                                        <a href="{local:doHtmlName('QQ',$qq/@id,$qqed,'.html')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:value-of select="$qqname"/>
                                                                <xsl:text> </xsl:text>
                                                                <i>
                                                                    <xsl:text>(</xsl:text>
                                                                    <xsl:if test="$qq">
                                                                        <xsl:value-of select="$qq/(@id | @ref)"/>
                                                                        <xsl:text> / </xsl:text>
                                                                    </xsl:if>
                                                                    <xsl:choose>
                                                                        <xsl:when test="$qqed castable as xs:dateTime">
                                                                            <xsl:call-template name="showDate">
                                                                                <xsl:with-param name="date" select="$qqed"/>
                                                                            </xsl:call-template>
                                                                        </xsl:when>
                                                                        <xsl:otherwise>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                            </xsl:call-template>
                                                                        </xsl:otherwise>
                                                                    </xsl:choose>
                                                                    <xsl:text>)</xsl:text>
                                                                </i>
                                                            </xsl:attribute>
                                                            <xsl:copy-of select="$qqname"/>
                                                        </a>
                                                    </xsl:for-each>
                                                </td>
                                            </xsl:if>
                                            <!-- Diagram -->
                                            <td>
                                                <xsl:if test="representingTemplate/concept and string-length($projectRestUriDSDiagram) &gt; 0">
                                                    <a href="{$projectRestUriDSDiagram}&amp;transactionId={$trid}&amp;transactionEffectiveDate={encode-for-uri($tred)}&amp;format=svg" onclick="target='_blank';">link</a>
                                                </xsl:if>
                                            </td>
                                            <!-- XML -->
                                            <td>
                                                <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                <xsl:if test="representingTemplate/concept">
                                                    <xsl:if test="string-length($projectRestUriDS) &gt; 0">
                                                        <a href="{$projectRestUriDS}&amp;id={$trid}&amp;effectiveDate={encode-for-uri($tred)}&amp;format=xml" onclick="target='_blank';">xml</a>
                                                    </xsl:if>
                                                </xsl:if>
                                            </td>
                                            <!-- columnAllView -->
                                            <!-- FIXME? the ProjectIndex has a different understanding of "All" than this link has... -->
                                            <td>
                                                <xsl:choose>
                                                    <xsl:when test="self::scenario | self::transaction[@type[. = 'group'] | transaction]">
                                                        <a href="{local:doHtmlName('SC', $projectPrefix, $scid, $sced, $trid, $tred, (), (), '.html', 'false')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Details'"/>
                                                                </xsl:call-template>
                                                            </xsl:attribute>
                                                            <xsl:text>html</xsl:text>
                                                        </a>
                                                    </xsl:when>
                                                    <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                    <xsl:when test="representingTemplate/concept">
                                                        <a href="{local:doHtmlName('TR',$trid,$tred,'.html')}" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'columnAllView'"/>
                                                                </xsl:call-template>
                                                            </xsl:attribute>
                                                            <xsl:text>html</xsl:text>
                                                        </a>
                                                    </xsl:when>
                                                </xsl:choose>
                                            </td>
                                            <!-- columnCareViewView -->
                                            <!-- FIXME? the ProjectIndex offers 2 flavors of care view. We only offer one version here... -->
                                            <td>
                                                <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                <xsl:if test="representingTemplate/concept">
                                                    <xsl:if test="string-length($projectRestUriDS) &gt; 0">
                                                        <a href="{$projectRestUriDS}&amp;id={$trid}&amp;effectiveDate={encode-for-uri($tred)}&amp;format=html" onclick="target='_blank';">
                                                            <xsl:attribute name="title">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'columnCareView'"/>
                                                                </xsl:call-template>
                                                            </xsl:attribute>
                                                            <xsl:text>html</xsl:text>
                                                        </a>
                                                    </xsl:if>
                                                </xsl:if>
                                                <xsl:text> </xsl:text>
                                            </td>
                                        </tr>
                                    </xsl:for-each>
                                </xsl:for-each>
                            </tbody>
                        </table>
                    </div>
                </xsl:for-each>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabScenarios'"/>
            <xsl:with-param name="actualcontent" select="$content4"/>
        </xsl:call-template>
        <xsl:for-each select="//scenarios/scenario">
            <xsl:variable name="scid" select="ancestor-or-self::scenario/@id"/>
            <xsl:variable name="sced" select="ancestor-or-self::scenario/@effectiveDate"/>
            <xsl:call-template name="logMessage">
                <xsl:with-param name="level" select="$logDEBUG"/>
                <xsl:with-param name="msg">
                    <xsl:text>*** Creating HTML For Scenario "</xsl:text>
                    <xsl:call-template name="doName">
                        <xsl:with-param name="ns" select="name"/>
                    </xsl:call-template>
                    <xsl:text>" </xsl:text>
                    <xsl:value-of select="local:doHtmlName('SC', $scid, $sced, '.html')"/>
                </xsl:with-param>
            </xsl:call-template>
            <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('SC',$scid,$sced,'.html','true')}" format="xhtml">
                <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                    <head>
                        <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                        <title>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'columnScenario'"/>
                            </xsl:call-template>
                            <xsl:text> </xsl:text>
                            <xsl:call-template name="doName">
                                <xsl:with-param name="ns" select="name"/>
                            </xsl:call-template>
                            <xsl:text> - </xsl:text>
                            <xsl:value-of select="$scid"/>
                            <xsl:text> - </xsl:text>
                            <xsl:value-of select="$sced"/>
                        </title>
                        <xsl:call-template name="addAssetsHeader">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </head>
                    <body>
                        <div class="indexline">
                            <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToIndex'"/>
                                </xsl:call-template>
                            </a>
                            <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                            <a href="{local:doHtmlName('tabScenarios', $projectPrefix, (), (), $scid, $sced, (), (), '.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToScenarios'"/>
                                </xsl:call-template>
                            </a>
                        </div>
                        <h1>
                            <xsl:call-template name="showStatusDot">
                                <xsl:with-param name="status" select="@statusCode"/>
                                <xsl:with-param name="title">
                                    <xsl:call-template name="getXFormsLabel">
                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                    </xsl:call-template>
                                </xsl:with-param>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'columnScenario'"/>
                            </xsl:call-template>
                            <xsl:text> </xsl:text>
                            <i>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="name"/>
                                </xsl:call-template>
                            </i>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:value-of select="$scid"/>
                        </h1>
                        <div class="tabtab">
                            <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                <div>
                                    <button id="expandAll" type="button">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                        </xsl:call-template>
                                    </button>
                                    <button id="collapseAll" type="button">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                        </xsl:call-template>
                                    </button>
                                    <input id="nameSearch">
                                        <xsl:attribute name="placeholder">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'textSearch'"/>
                                            </xsl:call-template>
                                        </xsl:attribute>
                                    </input>
                                    <select id="hiddenColumns">
                                        <option value="title">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'showColumn'"/>
                                            </xsl:call-template>
                                        </option>
                                        <option value="columnDetails" disabled="disabled">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Details'"/>
                                            </xsl:call-template>
                                        </option>
                                    </select>
                                </div>
                            </xsl:if>
                            <table id="transactionTable" class="treetable" width="100%" border="0" cellspacing="0" cellpadding="5">
                                <tr>
                                    <th class="columnName">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'Name'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th class="columnDetails" colspan="2">
                                        <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Details'"/>
                                            </xsl:call-template>
                                            <xsl:text>&#160;</xsl:text>
                                            <span class="hideMe" type="button">[‑]</span>
                                        </xsl:if>
                                    </th>
                                </tr>
                                <xsl:call-template name="doScenarios"/>
                            </table>
                        </div>
                        <xsl:call-template name="addAssetsBottom">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </body>
                </html>
            </xsl:result-document>
        </xsl:for-each>
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logDEBUG"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating SVG</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <!-- Write functional and technical SVGs. Could not do this inside the variable because older versions of
            Saxon do not support switching output from within a variable -->
        <xsl:for-each select="$allSvg/transaction">
            <xsl:if test="*[1]">
                <xsl:result-document format="xml" href="{$theHtmlDir}{local:doHtmlName('TR',@id,@effectiveDate,'_functional.svg','true')}">
                    <xsl:copy-of select="*[1]" copy-namespaces="no"/>
                </xsl:result-document>
            </xsl:if>
            <xsl:if test="*[2]">
                <xsl:result-document format="xml" href="{$theHtmlDir}{local:doHtmlName('TR',@id,@effectiveDate,'_technical.svg','true')}">
                    <xsl:copy-of select="*[2]" copy-namespaces="no"/>
                </xsl:result-document>
            </xsl:if>
        </xsl:for-each>
    </xsl:template>
    <xsl:template name="doDatasetsTab">
        <xsl:variable name="sc-style" select="'border-top: 6px solid white;'"/>
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Data Sets Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="content3">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <h2>
                    <a name="labelDataSets"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabDataSetsTitleString'"/>
                    </xsl:call-template>
                </h2>
                <xsl:choose>
                    <xsl:when test="count(//datasets/dataset) = 0">
                        <table style="border: 0;">
                            <xsl:call-template name="doMessage">
                                <xsl:with-param name="level" select="'info'"/>
                                <xsl:with-param name="msg">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'issueNoDataSetsDefinedYet'"/>
                                    </xsl:call-template>
                                </xsl:with-param>
                            </xsl:call-template>
                        </table>
                    </xsl:when>
                    <xsl:otherwise>
                        <div class="tabtab">
                            <table class="highlight" width="100%" cellspacing="0" cellpadding="5">
                                <thead>
                                    <tr>
                                        <th style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Diagram'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="width: 50px;">XML</th>
                                        <th style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnAllView'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnCareView'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnName'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'effectiveDate'"/>
                                            </xsl:call-template>
                                        </th>
                                        <xsl:if test="//datasets/dataset[@expirationDate[not(. = '')]]">
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'expirationDate'"/>
                                                </xsl:call-template>
                                            </th>
                                        </xsl:if>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                            </xsl:call-template>
                                        </th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <xsl:for-each-group select="//datasets/dataset" group-by="string(@ident)">
                                        <xsl:sort select="current-grouping-key()" order="ascending"/>
                                        
                                        <xsl:for-each select="current-group()">
                                            <xsl:sort select="@effectiveDate" order="descending"/>
                                            <xsl:sort select="(name[@language = $defaultLanguage], name)[1]" order="ascending"/>
                                            <tr class="dsline">
                                                <td style="{$sc-style}">
                                                    <xsl:if test="string-length($projectRestUriDSDiagram) &gt; 0">
                                                        <a href="{$projectRestUriDSDiagram}&amp;datasetId={@id}&amp;datasetEffectiveDate={encode-for-uri(@effectiveDate)}&amp;format=svg" onclick="target='_blank';">link</a>
                                                    </xsl:if>
                                                </td>
                                                <td style="{$sc-style}">
                                                    <xsl:if test="string-length($projectRestUriDS) &gt; 0">
                                                        <a href="{$projectRestUriDS}&amp;id={@id}&amp;effectiveDate={encode-for-uri(@effectiveDate)}&amp;format=xml" onclick="target='_blank';">xml</a>
                                                    </xsl:if>
                                                </td>
                                                <td style="{$sc-style}">
                                                    <a href="{local:doHtmlName('DS',@id,@effectiveDate,'.html')}" onclick="target='_blank';">html</a>
                                                </td>
                                                <td style="{$sc-style}">
                                                    <xsl:if test="string-length($projectRestUriDS) &gt; 0">
                                                        <a href="{$projectRestUriDS}&amp;id={@id}&amp;effectiveDate={encode-for-uri(@effectiveDate)}&amp;format=html" onclick="target='_blank';">html</a>
                                                    </xsl:if>
                                                </td>
                                                <td style="{$sc-style}">
                                                    <xsl:variable name="theStatus">
                                                        <xsl:choose>
                                                            <!--<xsl:when test="@ident[not(. = $projectPrefix)]">
                                                                <xsl:value-of select="'ref'"/>
                                                            </xsl:when>-->
                                                            <xsl:when test="@statusCode">
                                                                <xsl:value-of select="@statusCode"/>
                                                            </xsl:when>
                                                            <xsl:otherwise>
                                                                <!-- 
                                                                    if any item is new draft - > data set is draft
                                                                    if any item is rejected cancelled deprecated - > data set is final
                                                                    otherwise the data set is final
                                                                -->
                                                                <xsl:variable name="ndcount" select="count(.//concept[@statusCode = 'new' or @statusCode = 'draft'])"/>
                                                                <xsl:choose>
                                                                    <xsl:when test="$ndcount &gt; 0">
                                                                        <xsl:text>draft</xsl:text>
                                                                    </xsl:when>
                                                                    <xsl:otherwise>
                                                                        <xsl:text>final</xsl:text>
                                                                    </xsl:otherwise>
                                                                </xsl:choose>
                                                            </xsl:otherwise>
                                                        </xsl:choose>
                                                    </xsl:variable>
                                                    <xsl:call-template name="showStatusDot">
                                                        <xsl:with-param name="status" select="$theStatus"/>
                                                        <xsl:with-param name="title">
                                                            <xsl:call-template name="getXFormsLabel">
                                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                <xsl:with-param name="simpleTypeValue" select="$theStatus"/>
                                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                            </xsl:call-template>
                                                        </xsl:with-param>
                                                    </xsl:call-template>
                                                    <xsl:text>&#160;</xsl:text>
                                                    <xsl:call-template name="doName">
                                                        <xsl:with-param name="ns" select="name"/>
                                                    </xsl:call-template>
                                                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                                </td>
                                                <td style="{$sc-style}">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="@effectiveDate"/>
                                                    </xsl:call-template>
                                                </td>
                                                <xsl:if test="ancestor::datasets/dataset[@expirationDate[not(. = '')]]">
                                                    <td style="{$sc-style}">
                                                        <xsl:call-template name="showDate">
                                                            <xsl:with-param name="date" select="@expirationDate"/>
                                                        </xsl:call-template>
                                                    </td>
                                                </xsl:if>
                                                <td style="{$sc-style}">
                                                    <xsl:value-of select="@versionLabel"/>
                                                </td>
                                            </tr>
                                        </xsl:for-each>
                                    </xsl:for-each-group>
                                </tbody>
                            </table>
                        </div>
                    </xsl:otherwise>
                </xsl:choose>
                <h2>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'dataSetDataSetsPerTransaction'"/>
                    </xsl:call-template>
                </h2>
                <xsl:choose>
                    <xsl:when test="count(//scenarios/scenario//transaction/representingTemplate[@sourceDataset]) = 0">
                        <table style="border: 0;">
                            <xsl:call-template name="doMessage">
                                <xsl:with-param name="level" select="'info'"/>
                                <xsl:with-param name="msg">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'issueNoTransactionsWithAnUnderlyingModelDefinedYet'"/>
                                    </xsl:call-template>
                                </xsl:with-param>
                            </xsl:call-template>
                        </table>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:variable name="doExpirationDate" select="exists(//scenarios/scenario//transaction[representingTemplate/@sourceDataset][@expirationDate])"/>
                        <div class="tabtab">
                            <table class="highlight" width="100%" cellspacing="0" cellpadding="5">
                                <thead>
                                    <tr>
                                        <th style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Diagram'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="width: 50px;">XML</th>
                                        <th style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnAllView'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="width: 50px;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnCareView'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnName'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'effectiveDate'"/>
                                            </xsl:call-template>
                                        </th>
                                        <xsl:if test="$doExpirationDate">
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'expirationDate'"/>
                                                </xsl:call-template>
                                            </th>
                                        </xsl:if>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnScenario'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'columnDataset'"/>
                                            </xsl:call-template>
                                        </th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <xsl:for-each select="//scenarios/scenario[.//transaction[representingTemplate/@sourceDataset]]">
                                        <xsl:sort select="@effectiveDate" order="descending"/>
                                        <xsl:variable name="scid" select="@id"/>
                                        <xsl:variable name="sced" select="@effectiveDate"/>
                                        <xsl:variable name="scenarioAnchor" select="local:doHtmlAnchor($scid, $sced)"/>
                                        <xsl:variable name="scstatus" select="@statusCode"/>
                                        <xsl:for-each select=".//transaction[representingTemplate/@sourceDataset]">
                                            <xsl:variable name="trid" select="@id"/>
                                            <!-- cache transaction/@effectiveDate. This is relatively new so might not be present -->
                                            <xsl:variable name="tred" select="@effectiveDate"/>
                                            <xsl:variable name="trsts" select="ancestor-or-self::*[1]/@statusCode"/>
                                            <xsl:variable name="dsid" select="representingTemplate/@sourceDataset"/>
                                            <xsl:variable name="dsed" select="representingTemplate/@sourceDatasetFlexibility"/>
                                            <xsl:variable name="reptc" select="representingTemplate"/>
                                            <xsl:variable name="dataset" select="if ($dsid) then local:getDataset($dsid, $dsed) else ()" as="element()?"/>
                                            <tr class="scline">
                                                <td style="{$sc-style}">
                                                    <xsl:if test="representingTemplate/concept and string-length($projectRestUriDSDiagram) &gt; 0">
                                                        <a href="{$projectRestUriDSDiagram}&amp;transactionId={@id}&amp;transactionEffectiveDate={encode-for-uri(@effectiveDate)}&amp;format=svg" onclick="target='_blank';">link</a>
                                                    </xsl:if>
                                                </td>
                                                <td style="{$sc-style}">
                                                    <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                    <xsl:if test="representingTemplate/concept">
                                                        <xsl:if test="string-length($projectRestUriDS) &gt; 0">
                                                            <a href="{$projectRestUriDS}&amp;id={@id}&amp;effectiveDate={encode-for-uri(@effectiveDate)}&amp;format=xml" onclick="target='_blank';">xml</a>
                                                        </xsl:if>
                                                    </xsl:if>
                                                </td>
                                                <!-- FIXME? the ProjectIndex has a different understanding of "All" than this link has... -->
                                                <td style="{$sc-style}">
                                                    <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                    <xsl:if test="representingTemplate/concept">
                                                        <a href="{local:doHtmlName('TR',$trid,$tred,'.html')}" onclick="target='_blank';">html</a>
                                                    </xsl:if>
                                                </td>
                                                <!-- FIXME? the ProjectIndex offers 2 flavors of care view. We only offer one version here... -->
                                                <td style="{$sc-style}">
                                                    <!-- Prevent link here so it saves the user pointless clicking into an HTML without content -->
                                                    <xsl:if test="representingTemplate/concept">
                                                        <xsl:if test="string-length($projectRestUriDS) &gt; 0">
                                                            <a href="{$projectRestUriDS}&amp;id={$trid}&amp;effectiveDate={encode-for-uri($tred)}&amp;format=html" onclick="target='_blank';">html</a>
                                                        </xsl:if>
                                                    </xsl:if>
                                                </td>
                                                <!-- Transaction -->
                                                <td style="{$sc-style}">
                                                    <xsl:call-template name="showStatusDot">
                                                        <xsl:with-param name="status" select="$trsts"/>
                                                        <xsl:with-param name="title">
                                                            <xsl:call-template name="getXFormsLabel">
                                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                <xsl:with-param name="simpleTypeValue" select="$trsts"/>
                                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                            </xsl:call-template>
                                                        </xsl:with-param>
                                                    </xsl:call-template>
                                                    <xsl:text> </xsl:text>
                                                    <xsl:call-template name="doName">
                                                        <xsl:with-param name="ns" select="name"/>
                                                    </xsl:call-template>
                                                    <!-- Add this here so it saves the user pointless clicking into an HTML with the same notice -->
                                                    <xsl:if test="not(representingTemplate/concept)">
                                                        <table style="border: 0; padding-left: 20px;">
                                                            <xsl:call-template name="doMessage">
                                                                <xsl:with-param name="level" select="'info'"/>
                                                                <xsl:with-param name="msg">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'issueNoUnderlyingModelWithThisScenarioTransactionDefinedYet'"/>
                                                                    </xsl:call-template>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                        </table>
                                                    </xsl:if>
                                                </td>
                                                <!-- effectiveDate -->
                                                <td style="{$sc-style}">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="@effectiveDate"/>
                                                    </xsl:call-template>
                                                </td>
                                                <!-- expirationDate -->
                                                <xsl:if test="$doExpirationDate">
                                                    <td style="{$sc-style}">
                                                        <xsl:call-template name="showDate">
                                                            <xsl:with-param name="date" select="@expirationDate"/>
                                                        </xsl:call-template>
                                                    </td>
                                                </xsl:if>
                                                <!-- versionLabel -->
                                                <td style="{$sc-style}">
                                                    <xsl:value-of select="@versionLabel"/>
                                                </td>
                                                <!-- Scenario -->
                                                <td style="{$sc-style}">
                                                    <xsl:call-template name="showStatusDot">
                                                        <xsl:with-param name="status" select="$scstatus"/>
                                                        <xsl:with-param name="title">
                                                            <xsl:call-template name="getXFormsLabel">
                                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                <xsl:with-param name="simpleTypeValue" select="$scstatus"/>
                                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                            </xsl:call-template>
                                                        </xsl:with-param>
                                                    </xsl:call-template>
                                                    <xsl:text> </xsl:text>
                                                    <a href="{local:doHtmlName('tabScenarios',$projectPrefix,(),(),$scid,$sced,(),(),'.html','false')}" onclick="target='_blank';">
                                                        <xsl:call-template name="doName">
                                                            <xsl:with-param name="ns" select="ancestor::scenario[1]/name"/>
                                                        </xsl:call-template>
                                                    </a>
                                                </td>
                                                <!-- Dataset -->
                                                <td style="{$sc-style}">
                                                    <xsl:if test="$dataset">
                                                        <xsl:variable name="dsStatus">
                                                            <xsl:choose>
                                                                <!--<xsl:when test="$dataset/@ident[not(. = $projectPrefix)]">
                                                                    <xsl:value-of select="'ref'"/>
                                                                </xsl:when>-->
                                                                <xsl:when test="$dataset/@statusCode">
                                                                    <xsl:value-of select="$dataset/@statusCode"/>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <!-- 
                                                                        if any item is new draft - > data set is draft
                                                                        if any item is rejected cancelled deprecated - > data set is final
                                                                        otherwise the data set is final
                                                                    -->
                                                                    <xsl:variable name="ndcount" select="count($dataset//concept[@statusCode = 'new' or @statusCode = 'draft'])"/>
                                                                    <xsl:choose>
                                                                        <xsl:when test="$ndcount &gt; 0">
                                                                            <xsl:text>draft</xsl:text>
                                                                        </xsl:when>
                                                                        <xsl:otherwise>
                                                                            <xsl:text>final</xsl:text>
                                                                        </xsl:otherwise>
                                                                    </xsl:choose>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </xsl:variable>
                                                        <xsl:call-template name="showStatusDot">
                                                            <xsl:with-param name="status" select="$dsStatus"/>
                                                            <xsl:with-param name="title">
                                                                <xsl:call-template name="getXFormsLabel">
                                                                    <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                    <xsl:with-param name="simpleTypeValue" select="$dsStatus"/>
                                                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                </xsl:call-template>
                                                            </xsl:with-param>
                                                        </xsl:call-template>
                                                        <xsl:text> </xsl:text>
                                                        <xsl:call-template name="doName">
                                                            <xsl:with-param name="ns" select="$dataset/name"/>
                                                        </xsl:call-template>
                                                        <xsl:if test="$dataset[@versionLabel]">
                                                            <xsl:text> </xsl:text>
                                                            <xsl:value-of select="$dataset/@versionLabel"/>
                                                        </xsl:if>
                                                    </xsl:if>
                                                </td>
                                            </tr>
                                        </xsl:for-each>
                                    </xsl:for-each>
                                </tbody>
                            </table>
                        </div>
                    </xsl:otherwise>
                </xsl:choose>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabDataSet'"/>
            <xsl:with-param name="actualcontent" select="$content3"/>
        </xsl:call-template>
        <!-- create all dataset html representations -->
        <xsl:for-each select="//datasets/dataset">
            <xsl:sort select="@effectiveDate" order="descending"/>
            <xsl:variable name="dsid" select="@id"/>
            <xsl:variable name="dsed" select="@effectiveDate"/>
            <xsl:variable name="isNewest" select="$dsed = max($allDatasets/dataset[(@id | @ref) = $dsid]/xs:dateTime(@effectiveDate))"/>
            <xsl:call-template name="logMessage">
                <xsl:with-param name="level" select="$logDEBUG"/>
                <xsl:with-param name="msg">
                    <xsl:text>*** HTML for dataset: name='</xsl:text>
                    <xsl:value-of select="name[1]"/>
                    <xsl:text>' id='</xsl:text>
                    <xsl:value-of select="$dsid"/>
                    <xsl:text>' effectiveDate='</xsl:text>
                    <xsl:value-of select="$dsed"/>
                    <xsl:text>'</xsl:text>
                </xsl:with-param>
            </xsl:call-template>
            <!-- Write the static variant for include/contains with static flexibility -->
            <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('DS',$dsid,$dsed,'.html','true')}" format="xhtml">
                <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                    <head>
                        <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                        <title>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Dataset'"/>
                            </xsl:call-template>
                            <xsl:text> </xsl:text>
                            <xsl:call-template name="doName">
                                <xsl:with-param name="ns" select="name"/>
                                <!--<xsl:with-param name="lang" select="$defaultLanguage"/>-->
                            </xsl:call-template>
                            <xsl:text> - </xsl:text>
                            <xsl:value-of select="$dsid"/>
                            <xsl:text> - </xsl:text>
                            <xsl:value-of select="$dsed"/>
                        </title>
                        <xsl:call-template name="addAssetsHeader">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </head>
                    <body>
                        <div class="indexline">
                            <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToIndex'"/>
                                </xsl:call-template>
                            </a>
                            <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                            <a href="{local:doHtmlName('tabDataSet',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToDatasets'"/>
                                </xsl:call-template>
                            </a>
                        </div>
                        <h1>
                            <xsl:variable name="dsStatus">
                                <xsl:choose>
                                    <!--<xsl:when test="@ident[not(. = $projectPrefix)]">
                                        <xsl:value-of select="'ref'"/>
                                    </xsl:when>-->
                                    <xsl:when test="@statusCode">
                                        <xsl:value-of select="@statusCode"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <!-- 
                                            if any item is new draft - > data set is draft
                                            if any item is rejected cancelled deprecated - > data set is final
                                            otherwise the data set is final
                                        -->
                                        <xsl:variable name="ndcount" select="count(.//concept[@statusCode = 'new' or @statusCode = 'draft'])"/>
                                        <xsl:choose>
                                            <xsl:when test="$ndcount &gt; 0">
                                                <xsl:text>draft</xsl:text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:text>final</xsl:text>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </xsl:variable>
                            <xsl:call-template name="showStatusDot">
                                <xsl:with-param name="status" select="$dsStatus"/>
                                <xsl:with-param name="title">
                                    <xsl:call-template name="getXFormsLabel">
                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                        <xsl:with-param name="simpleTypeValue" select="$dsStatus"/>
                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                    </xsl:call-template>
                                </xsl:with-param>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Dataset'"/>
                            </xsl:call-template>
                            <xsl:text> </xsl:text>
                            <i>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="name"/>
                                    <!--<xsl:with-param name="lang" select="$defaultLanguage"/>-->
                                </xsl:call-template>
                            </i>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:value-of select="$dsid"/>
                        </h1>
                        <table border="0" cellspacing="2" cellpadding="2" width="100%">
                            <tr style="vertical-align: top;">
                                <td class="tabtab">
                                    <xsl:call-template name="doDataset">
                                        <xsl:with-param name="nestingWithTables" select="true()"/>
                                        <xsl:with-param name="dsid" select="$dsid"/>
                                        <xsl:with-param name="dsed" select="$dsed"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                        </table>
                        <xsl:call-template name="addAssetsBottom">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </body>
                </html>
            </xsl:result-document>
            <xsl:if test="$isNewest = true()">
                <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('DS',$dsid,'dynamic','.html','true')}" format="xhtml">
                    <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                        <head>
                            <meta http-equiv="refresh" content="0; URL={local:doHtmlName('DS',$dsid,$dsed,'.html','true')}"/>
                            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                            <meta name="robots" content="noindex, nofollow"/>
                            <meta http-equiv="expires" content="0"/>
                            <!-- xhtml requirement -->
                            <title>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Dataset'"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="name"/>
                                </xsl:call-template>
                            </title>
                        </head>
                        <!-- xhtml requirement -->
                        <body/>
                    </html>
                </xsl:result-document>
            </xsl:if>
        </xsl:for-each>
        <!-- create all scenario html representations from a dataset view -->
        <xsl:for-each select="//scenarios/scenario[.//transaction[representingTemplate]]">
            <xsl:sort select="@effectiveDate" order="descending"/>
            <xsl:variable name="scstatus" select="@statusCode"/>
            <xsl:for-each select=".//transaction[representingTemplate]">
                <xsl:variable name="scid" select="ancestor::scenario/@id"/>
                <xsl:variable name="sced" select="ancestor::scenario/@effectiveDate"/>
                <xsl:variable name="trid" select="@id"/>
                <!-- cache transaction/@effectiveDate. This is relatively new so might not be present -->
                <xsl:variable name="tred" select="@effectiveDate"/>
                <xsl:variable name="dsid" select="representingTemplate/@sourceDataset"/>
                <xsl:variable name="dsed" select="representingTemplate/@sourceDatasetFlexibility"/>
                <xsl:variable name="reptc" select="representingTemplate"/>
                <xsl:variable name="dataset" select="if ($dsid) then local:getDataset($dsid, $dsed) else ()" as="element()?"/>
                <xsl:variable name="dsname" select="$dataset/name"/>
                <!-- create the data set filtered with concept mentioned in the representingTemplate only -->
                <xsl:variable name="filtereddataset" as="element()">
                    <xsl:variable name="tmp1" as="element()">
                        <tmp xmlns="">
                            <xsl:apply-templates select="$dataset/concept" mode="filter">
                                <xsl:with-param name="representingTemplate" select="$reptc"/>
                            </xsl:apply-templates>
                        </tmp>
                    </xsl:variable>
                    <filtereddataset xmlns="">
                        <xsl:copy-of select="@*"/>
                        <xsl:copy-of select="name | desc"/>
                        <xsl:apply-templates select="$tmp1/concept" mode="eliminatedHiddenConcepts"/>
                    </filtereddataset>
                </xsl:variable>
                <xsl:call-template name="logMessage">
                    <xsl:with-param name="level" select="$logDEBUG"/>
                    <xsl:with-param name="msg">
                        <xsl:text>*** HTML for dataset based on transaction: name='</xsl:text>
                        <xsl:value-of select="name[1]"/>
                        <xsl:text>' id='</xsl:text>
                        <xsl:value-of select="$trid"/>
                        <xsl:text>' effectiveDate='</xsl:text>
                        <xsl:value-of select="$tred"/>
                        <xsl:text>' dataset name='</xsl:text>
                        <xsl:value-of select="$dsname"/>
                        <xsl:if test="$dataset[@versionLabel]">
                            <xsl:text>' versionLabel='</xsl:text>
                            <xsl:value-of select="$dataset/@versionLabel"/>
                        </xsl:if>
                        <xsl:text>' id='</xsl:text>
                        <xsl:value-of select="$dsid"/>
                        <xsl:text>'</xsl:text>
                    </xsl:with-param>
                </xsl:call-template>
                <xsl:variable name="transactionAnchor" select="local:doHtmlAnchor($trid, $tred)"/>
                <!-- Write the static variant for include/contains with static flexibility -->
                <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('TR',$trid,$tred,'.html','true')}" format="xhtml">
                    <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                        <head>
                            <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                            <title>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'dataSetDataSetForTransaction'"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <xsl:value-of select="@id"/>
                                <xsl:text> - </xsl:text>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="name"/>
                                    <!--<xsl:with-param name="lang" select="$defaultLanguage"/>-->
                                </xsl:call-template>
                            </title>
                            <xsl:call-template name="addAssetsHeader">
                                <xsl:with-param name="doTreeTable" select="true()"/>
                            </xsl:call-template>
                        </head>
                        <body>
                            <div class="indexline">
                                <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'backToIndex'"/>
                                    </xsl:call-template>
                                </a>
                                <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                                <a href="{local:doHtmlName('tabScenarios', $projectPrefix, (), (), $trid, $tred, (), (), '.html', 'true')}">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'backToScenarios'"/>
                                    </xsl:call-template>
                                </a>
                                <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                                <a href="{local:doHtmlName('SC', $projectPrefix, $scid, $sced, $trid, $tred, (), (), '.html', 'false')}" onclick="target='_blank';">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'backToScenarioTransaction'"/>
                                    </xsl:call-template>
                                </a>
                            </div>
                            <h1>
                                <xsl:call-template name="showStatusDot">
                                    <xsl:with-param name="status" select="@statusCode"/>
                                    <xsl:with-param name="title">
                                        <xsl:call-template name="getXFormsLabel">
                                            <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                            <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                        </xsl:call-template>
                                    </xsl:with-param>
                                </xsl:call-template>
                                <xsl:text>&#160;</xsl:text>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Transaction'"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <i>
                                    <xsl:call-template name="doName">
                                        <xsl:with-param name="ns" select="name"/>
                                        <!--<xsl:with-param name="lang" select="$defaultLanguage"/>-->
                                    </xsl:call-template>
                                </i>
                                <xsl:text>&#160;</xsl:text>
                                <xsl:value-of select="$trid"/>
                            </h1>
                            <h2>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Dataset'"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <i>
                                    <xsl:call-template name="doName">
                                        <xsl:with-param name="ns" select="$dsname"/>
                                        <!--<xsl:with-param name="lang" select="$defaultLanguage"/>-->
                                    </xsl:call-template>
                                </i>
                                <xsl:text>&#160;</xsl:text>
                                <xsl:value-of select="$dsid"/>
                            </h2>
                            <xsl:choose>
                                <xsl:when test="count($filtereddataset/concept) = 0 or count(representingTemplate/concept) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoUnderlyingModelWithThisScenarioTransactionDefinedYet'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table border="0" cellspacing="2" cellpadding="2" width="100%">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <xsl:for-each select="$filtereddataset">
                                                    <xsl:call-template name="doDataset">
                                                        <xsl:with-param name="nestingWithTables" select="true()"/>
                                                        <xsl:with-param name="trid" select="$trid"/>
                                                        <xsl:with-param name="tred" select="$tred"/>
                                                    </xsl:call-template>
                                                </xsl:for-each>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                            <xsl:call-template name="addAssetsBottom">
                                <xsl:with-param name="doTreeTable" select="true()"/>
                            </xsl:call-template>
                        </body>
                    </html>
                </xsl:result-document>
            </xsl:for-each>
        </xsl:for-each>
    </xsl:template>
    <xsl:template name="doIdentificationTab">
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Identifiers Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="content4b">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <xsl:call-template name="doIdentifiers"/>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabIdentifiers'"/>
            <xsl:with-param name="actualcontent" select="$content4b"/>
        </xsl:call-template>
    </xsl:template>
    <xsl:template name="doTerminologyTab">
        <xsl:variable name="sc-style" select="'border-top: 6px solid white;'"/>
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Terminology Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="content5">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <h2>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabTerminologyTitleString'"/>
                    </xsl:call-template>
                </h2>
                <div id="Accordionvs" class="Accordion">
                    <xsl:choose>
                        <xsl:when test="(count($allValueSets/*/valueSet[@id]) + count($allCodeSystems/*/codeSystem[@id]) + count($allConceptMaps/*/conceptMap[@id])) = 0">
                            <table style="border: 0;">
                                <xsl:call-template name="doMessage">
                                    <xsl:with-param name="level" select="'info'"/>
                                    <xsl:with-param name="msg">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueNoCodesValueSetsDefinedYet'"/>
                                        </xsl:call-template>
                                    </xsl:with-param>
                                </xsl:call-template>
                            </table>
                        </xsl:when>
                        <xsl:otherwise>
                            <h3>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'valueSets'"/>
                                </xsl:call-template>
                            </h3>
                            <xsl:variable name="doExpirationDateVS" select="$allValueSets/*/valueSet[@id][@expirationDate[not(. = '')]]"/>
                            <div class="tabtab">
                                <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                    <div>
                                        <button id="expandAll" type="button">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                            </xsl:call-template>
                                        </button>
                                        <button id="collapseAll" type="button">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                            </xsl:call-template>
                                        </button>
                                        <input id="nameSearch">
                                            <xsl:attribute name="placeholder">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'textSearch'"/>
                                                </xsl:call-template>
                                            </xsl:attribute>
                                        </input>
                                    </div>
                                </xsl:if>
                                <table id="valuesetTable" class="treetable highlight" width="100%" border="0" cellspacing="0" cellpadding="5">
                                    <thead>
                                        <tr>
                                            <th class="columnAllView" style="width: 50px;">HTML</th>
                                            <th class="columnName">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'columnName'"/>
                                                </xsl:call-template>
                                            </th>
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'effectiveDate'"/>
                                                </xsl:call-template>
                                            </th>
                                            <xsl:if test="$doExpirationDateVS">
                                                <th>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'expirationDate'"/>
                                                    </xsl:call-template>
                                                </th>
                                            </xsl:if>
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                                </xsl:call-template>
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <!-- only first value set of most recent version -->
                                        <xsl:for-each-group select="$allValueSets/*/valueSet[@id]" group-by="@id">
                                            <xsl:sort select="lower-case(normalize-space((current-group()[1]/@displayName, current-group()[1]/@name)[1]))"/>
                                            <xsl:variable name="groupCount" select="count(current-group())"/>
                                            <xsl:for-each select="current-group()">
                                                <xsl:sort select="@effectiveDate" order="descending"/>
                                                <xsl:variable name="vsid" select="@id"/>
                                                <xsl:variable name="vsed" select="@effectiveDate"/>
                                                <xsl:variable name="pos" select="position()"/>
                                                <xsl:variable name="valuesetAnchor">
                                                    <xsl:text>#</xsl:text>
                                                    <xsl:value-of select="local:doHtmlAnchor($vsid, $vsed)"/>
                                                </xsl:variable>
                                                <xsl:variable name="node-id" select="generate-id(.)"/>
                                                <!-- if we have multiple versions of a template, group them under their name/id -->
                                                <xsl:if test="$pos = 1 and $groupCount gt 1">
                                                    <tr data-tt-id="{$node-id}-group" class="vsline">
                                                        <td style="{$sc-style}" class="columnAllView">
                                                            <xsl:text>&#160;</xsl:text>
                                                        </td>
                                                        <td style="{$sc-style}" class="columnName">
                                                            <xsl:call-template name="showStatusDot">
                                                                <xsl:with-param name="status" select="'ref'"/>
                                                                <xsl:with-param name="title">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'Group'"/>
                                                                    </xsl:call-template>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <xsl:value-of select="current-group()[1]/(@displayName, @name)[1]"/>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <xsl:value-of select="$vsid"/>
                                                        </td>
                                                        <td style="{$sc-style}">
                                                            <i>
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Multiple'"/>
                                                                </xsl:call-template>
                                                                <xsl:text> (</xsl:text>
                                                                <xsl:value-of select="$groupCount"/>
                                                                <xsl:text>)</xsl:text>
                                                            </i>
                                                        </td>
                                                        <xsl:if test="$doExpirationDateVS">
                                                            <td style="{$sc-style}">
                                                                <xsl:text>&#160;</xsl:text>
                                                            </td>
                                                        </xsl:if>
                                                        <td style="{$sc-style}">
                                                            <xsl:text>&#160;</xsl:text>
                                                        </td>
                                                    </tr>
                                                    <xsl:if test="current-group()[1]/desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                        <tr data-tt-id="{$node-id}-desc" data-tt-parent-id="{$node-id}-group">
                                                            <td class="columnAllView"/>
                                                            <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDateVS) then 4 else 3}">
                                                                <xsl:value-of>
                                                                    <xsl:call-template name="doDescription">
                                                                        <xsl:with-param name="ns" select="current-group()[1]/desc"/>
                                                                        <xsl:with-param name="shortDesc" select="true()"/>
                                                                    </xsl:call-template>
                                                                </xsl:value-of>
                                                            </td>
                                                        </tr>
                                                    </xsl:if>
                                                </xsl:if>
                                                <tr data-tt-id="{$node-id}" class="vsline">
                                                    <xsl:if test="$groupCount gt 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="concat(generate-id(current-group()[1]), '-group')"/>
                                                    </xsl:if>
                                                    <td style="{$sc-style}" class="columnAllView">
                                                        <a href="{local:doHtmlName('VS',@id,@effectiveDate,'.html')}" onclick="target='_blank';">html</a>
                                                    </td>
                                                    <td style="{$sc-style}" class="columnName">
                                                        <xsl:call-template name="showStatusDot">
                                                            <xsl:with-param name="status" select="@statusCode"/>
                                                            <xsl:with-param name="title">
                                                                <xsl:call-template name="getXFormsLabel">
                                                                    <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                    <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                </xsl:call-template>
                                                            </xsl:with-param>
                                                        </xsl:call-template>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="(@displayName, @name)[1]"/>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="$vsid"/>
                                                        <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                                    </td>
                                                    <td style="{$sc-style}">
                                                        <xsl:call-template name="showDate">
                                                            <xsl:with-param name="date" select="$vsed"/>
                                                        </xsl:call-template>
                                                    </td>
                                                    <xsl:if test="$doExpirationDateVS">
                                                        <td style="{$sc-style}">
                                                            <xsl:call-template name="showDate">
                                                                <xsl:with-param name="date" select="@expirationDate"/>
                                                            </xsl:call-template>
                                                        </td>
                                                    </xsl:if>
                                                    <td style="{$sc-style}">
                                                        <xsl:value-of select="@versionLabel"/>
                                                    </td>
                                                </tr>
                                                <xsl:if test="$groupCount = 1 and $pos = 1 and desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                    <tr data-tt-id="{$node-id}-desc" data-tt-parent-id="{$node-id}">
                                                        <td class="columnAllView"/>
                                                        <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDateVS) then 4 else 3}">
                                                            <xsl:value-of>
                                                                <xsl:call-template name="doDescription">
                                                                    <xsl:with-param name="ns" select="desc"/>
                                                                    <xsl:with-param name="shortDesc" select="true()"/>
                                                                </xsl:call-template>
                                                            </xsl:value-of>
                                                        </td>
                                                    </tr>
                                                </xsl:if>
                                            </xsl:for-each>
                                        </xsl:for-each-group>
                                    </tbody>
                                </table>
                            </div>
                            <h3>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'CodeSystems'"/>
                                </xsl:call-template>
                            </h3>
                            <xsl:variable name="doExpirationDateCS" select="$allCodeSystems/*/codeSystem[@id][@expirationDate[not(. = '')]]"/>
                            <div class="tabtab">
                                <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                    <div>
                                        <button id="expandAll" type="button">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                            </xsl:call-template>
                                        </button>
                                        <button id="collapseAll" type="button">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                            </xsl:call-template>
                                        </button>
                                        <input id="nameSearch">
                                            <xsl:attribute name="placeholder">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'textSearch'"/>
                                                </xsl:call-template>
                                            </xsl:attribute>
                                        </input>
                                    </div>
                                </xsl:if>
                                <table id="codesystemTable" class="treetable highlight" width="100%" border="0" cellspacing="0" cellpadding="5">
                                    <thead>
                                        <tr>
                                            <th class="columnAllView" style="width: 50px;">HTML</th>
                                            <th class="columnName">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'columnName'"/>
                                                </xsl:call-template>
                                            </th>
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'effectiveDate'"/>
                                                </xsl:call-template>
                                            </th>
                                            <xsl:if test="$doExpirationDateCS">
                                                <th>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'expirationDate'"/>
                                                    </xsl:call-template>
                                                </th>
                                            </xsl:if>
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                                </xsl:call-template>
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <!-- only first value set of most recent version -->
                                        <xsl:for-each-group select="$allCodeSystems/*/codeSystem[@id]" group-by="@id">
                                            <xsl:sort select="lower-case(normalize-space((current-group()[1]/@displayName, current-group()[1]/@name)[1]))"/>
                                            <xsl:variable name="groupCount" select="count(current-group())"/>
                                            <xsl:for-each select="current-group()">
                                                <xsl:sort select="@effectiveDate" order="descending"/>
                                                <xsl:variable name="csid" select="@id"/>
                                                <xsl:variable name="csed" select="@effectiveDate"/>
                                                <xsl:variable name="pos" select="position()"/>
                                                <xsl:variable name="codesystemAnchor">
                                                    <xsl:text>#</xsl:text>
                                                    <xsl:value-of select="local:doHtmlAnchor($csid, $csed)"/>
                                                </xsl:variable>
                                                <xsl:variable name="node-id" select="generate-id(.)"/>
                                                <!-- if we have multiple versions of a template, group them under their name/id -->
                                                <xsl:if test="$pos = 1 and $groupCount gt 1">
                                                    <tr data-tt-id="{$node-id}-group" class="vsline">
                                                        <td style="{$sc-style}" class="columnAllView">
                                                            <xsl:text>&#160;</xsl:text>
                                                        </td>
                                                        <td style="{$sc-style}" class="columnName">
                                                            <xsl:call-template name="showStatusDot">
                                                                <xsl:with-param name="status" select="'ref'"/>
                                                                <xsl:with-param name="title">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'Group'"/>
                                                                    </xsl:call-template>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <xsl:value-of select="current-group()[1]/(@displayName, @name)[1]"/>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <xsl:value-of select="$csid"/>
                                                        </td>
                                                        <td style="{$sc-style}">
                                                            <i>
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Multiple'"/>
                                                                </xsl:call-template>
                                                                <xsl:text> (</xsl:text>
                                                                <xsl:value-of select="$groupCount"/>
                                                                <xsl:text>)</xsl:text>
                                                            </i>
                                                        </td>
                                                        <xsl:if test="$doExpirationDateCS">
                                                            <td style="{$sc-style}">
                                                                <xsl:text>&#160;</xsl:text>
                                                            </td>
                                                        </xsl:if>
                                                        <td style="{$sc-style}">
                                                            <xsl:text>&#160;</xsl:text>
                                                        </td>
                                                    </tr>
                                                    <xsl:if test="current-group()[1]/desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                        <tr data-tt-id="{$node-id}-desc" data-tt-parent-id="{$node-id}-group">
                                                            <td class="columnAllView"/>
                                                            <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDateCS) then 4 else 3}">
                                                                <xsl:value-of>
                                                                    <xsl:call-template name="doDescription">
                                                                        <xsl:with-param name="ns" select="current-group()[1]/desc"/>
                                                                        <xsl:with-param name="shortDesc" select="true()"/>
                                                                    </xsl:call-template>
                                                                </xsl:value-of>
                                                            </td>
                                                        </tr>
                                                    </xsl:if>
                                                </xsl:if>
                                                <tr data-tt-id="{$node-id}" class="vsline">
                                                    <xsl:if test="$groupCount gt 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="concat(generate-id(current-group()[1]), '-group')"/>
                                                    </xsl:if>
                                                    <td style="{$sc-style}" class="columnAllView">
                                                        <a href="{local:doHtmlName('CS',@id,@effectiveDate,'.html')}" onclick="target='_blank';">html</a>
                                                    </td>
                                                    <td style="{$sc-style}" class="columnName">
                                                        <xsl:call-template name="showStatusDot">
                                                            <xsl:with-param name="status" select="@statusCode"/>
                                                            <xsl:with-param name="title">
                                                                <xsl:call-template name="getXFormsLabel">
                                                                    <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                    <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                </xsl:call-template>
                                                            </xsl:with-param>
                                                        </xsl:call-template>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="(@displayName, @name)[1]"/>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="$csid"/>
                                                        <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                                    </td>
                                                    <td style="{$sc-style}">
                                                        <xsl:call-template name="showDate">
                                                            <xsl:with-param name="date" select="$csed"/>
                                                        </xsl:call-template>
                                                    </td>
                                                    <xsl:if test="$doExpirationDateCS">
                                                        <td style="{$sc-style}">
                                                            <xsl:call-template name="showDate">
                                                                <xsl:with-param name="date" select="@expirationDate"/>
                                                            </xsl:call-template>
                                                        </td>
                                                    </xsl:if>
                                                    <td style="{$sc-style}">
                                                        <xsl:value-of select="@versionLabel"/>
                                                    </td>
                                                </tr>
                                                <xsl:if test="$groupCount = 1 and $pos = 1 and desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                    <tr data-tt-id="{$node-id}-desc" data-tt-parent-id="{$node-id}">
                                                        <td class="columnAllView"/>
                                                        <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDateCS) then 4 else 3}">
                                                            <xsl:value-of>
                                                                <xsl:call-template name="doDescription">
                                                                    <xsl:with-param name="ns" select="desc"/>
                                                                    <xsl:with-param name="shortDesc" select="true()"/>
                                                                </xsl:call-template>
                                                            </xsl:value-of>
                                                        </td>
                                                    </tr>
                                                </xsl:if>
                                            </xsl:for-each>
                                        </xsl:for-each-group>
                                    </tbody>
                                </table>
                            </div>
                            <h3>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'ConceptMaps'"/>
                                </xsl:call-template>
                            </h3>
                            <xsl:variable name="doExpirationDateMP" select="$allConceptMaps[@id][@expirationDate[not(. = '')]]"/>
                            <div class="tabtab">
                                <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                    <div>
                                        <button id="expandAll" type="button">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                            </xsl:call-template>
                                        </button>
                                        <button id="collapseAll" type="button">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                            </xsl:call-template>
                                        </button>
                                        <input id="nameSearch">
                                            <xsl:attribute name="placeholder">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'textSearch'"/>
                                                </xsl:call-template>
                                            </xsl:attribute>
                                        </input>
                                    </div>
                                </xsl:if>
                                <table id="conceptmapTable" class="treetable highlight" width="100%" border="0" cellspacing="0" cellpadding="5">
                                    <thead>
                                        <tr>
                                            <th class="columnAllView" style="width: 50px;">HTML</th>
                                            <th class="columnName">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'columnName'"/>
                                                </xsl:call-template>
                                            </th>
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'effectiveDate'"/>
                                                </xsl:call-template>
                                            </th>
                                            <xsl:if test="$doExpirationDateMP">
                                                <th>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'expirationDate'"/>
                                                    </xsl:call-template>
                                                </th>
                                            </xsl:if>
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                                </xsl:call-template>
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <!-- only first value set of most recent version -->
                                        <xsl:for-each-group select="$allConceptMaps[@id]" group-by="@id">
                                            <xsl:sort select="lower-case(normalize-space((current-group()[1]/@displayName, current-group()[1]/@name)[1]))"/>
                                            <xsl:variable name="groupCount" select="count(current-group())"/>
                                            <xsl:for-each select="current-group()">
                                                <xsl:sort select="@effectiveDate" order="descending"/>
                                                <xsl:variable name="mpid" select="@id"/>
                                                <xsl:variable name="mped" select="@effectiveDate"/>
                                                <xsl:variable name="pos" select="position()"/>
                                                <xsl:variable name="conceptmapAnchor">
                                                    <xsl:text>#</xsl:text>
                                                    <xsl:value-of select="local:doHtmlAnchor($mpid, $mped)"/>
                                                </xsl:variable>
                                                <xsl:variable name="node-id" select="generate-id(.)"/>
                                                <!-- if we have multiple versions of a template, group them under their name/id -->
                                                <xsl:if test="$pos = 1 and $groupCount gt 1">
                                                    <tr data-tt-id="{$node-id}-group" class="vsline">
                                                        <td style="{$sc-style}" class="columnAllView">
                                                            <xsl:text>&#160;</xsl:text>
                                                        </td>
                                                        <td style="{$sc-style}" class="columnName">
                                                            <xsl:call-template name="showStatusDot">
                                                                <xsl:with-param name="status" select="'ref'"/>
                                                                <xsl:with-param name="title">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'Group'"/>
                                                                    </xsl:call-template>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <xsl:value-of select="current-group()[1]/(@displayName, @name)[1]"/>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <xsl:value-of select="$mpid"/>
                                                        </td>
                                                        <td style="{$sc-style}">
                                                            <i>
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Multiple'"/>
                                                                </xsl:call-template>
                                                                <xsl:text> (</xsl:text>
                                                                <xsl:value-of select="$groupCount"/>
                                                                <xsl:text>)</xsl:text>
                                                            </i>
                                                        </td>
                                                        <xsl:if test="$doExpirationDateMP">
                                                            <td style="{$sc-style}">
                                                                <xsl:text>&#160;</xsl:text>
                                                            </td>
                                                        </xsl:if>
                                                        <td style="{$sc-style}">
                                                            <xsl:text>&#160;</xsl:text>
                                                        </td>
                                                    </tr>
                                                    <xsl:if test="current-group()[1]/desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                        <tr data-tt-id="{$node-id}-desc" data-tt-parent-id="{$node-id}-group">
                                                            <td class="columnAllView"/>
                                                            <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDateCS) then 4 else 3}">
                                                                <xsl:value-of>
                                                                    <xsl:call-template name="doDescription">
                                                                        <xsl:with-param name="ns" select="current-group()[1]/desc"/>
                                                                        <xsl:with-param name="shortDesc" select="true()"/>
                                                                    </xsl:call-template>
                                                                </xsl:value-of>
                                                            </td>
                                                        </tr>
                                                    </xsl:if>
                                                </xsl:if>
                                                <tr data-tt-id="{$node-id}" class="vsline">
                                                    <xsl:if test="$groupCount gt 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="concat(generate-id(current-group()[1]), '-group')"/>
                                                    </xsl:if>
                                                    <td style="{$sc-style}" class="columnAllView">
                                                        <a href="{local:doHtmlName('MP',@id,@effectiveDate,'.html')}" onclick="target='_blank';">html</a>
                                                    </td>
                                                    <td style="{$sc-style}" class="columnName">
                                                        <xsl:call-template name="showStatusDot">
                                                            <xsl:with-param name="status" select="@statusCode"/>
                                                            <xsl:with-param name="title">
                                                                <xsl:call-template name="getXFormsLabel">
                                                                    <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                    <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                </xsl:call-template>
                                                            </xsl:with-param>
                                                        </xsl:call-template>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="(@displayName, @name)[1]"/>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="$mpid"/>
                                                        <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                                    </td>
                                                    <td style="{$sc-style}">
                                                        <xsl:call-template name="showDate">
                                                            <xsl:with-param name="date" select="$mped"/>
                                                        </xsl:call-template>
                                                    </td>
                                                    <xsl:if test="$doExpirationDateMP">
                                                        <td style="{$sc-style}">
                                                            <xsl:call-template name="showDate">
                                                                <xsl:with-param name="date" select="@expirationDate"/>
                                                            </xsl:call-template>
                                                        </td>
                                                    </xsl:if>
                                                    <td style="{$sc-style}">
                                                        <xsl:value-of select="@versionLabel"/>
                                                    </td>
                                                </tr>
                                                <xsl:if test="$groupCount = 1 and $pos = 1 and desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                    <tr data-tt-id="{$node-id}-desc" data-tt-parent-id="{$node-id}">
                                                        <td class="columnAllView"/>
                                                        <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDateMP) then 4 else 3}">
                                                            <xsl:value-of>
                                                                <xsl:call-template name="doDescription">
                                                                    <xsl:with-param name="ns" select="desc"/>
                                                                    <xsl:with-param name="shortDesc" select="true()"/>
                                                                </xsl:call-template>
                                                            </xsl:value-of>
                                                        </td>
                                                    </tr>
                                                </xsl:if>
                                            </xsl:for-each>
                                        </xsl:for-each-group>
                                    </tbody>
                                </table>
                            </div>
                        </xsl:otherwise>
                    </xsl:choose>
                </div>
                <!-- 
                    then show a summary of scenario table
                -->
                <h2>
                    <a name="labelTerminology"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'terminologySummary'"/>
                    </xsl:call-template>
                </h2>
                <xsl:text>

</xsl:text>
                <table width="50%" border="0" cellspacing="2" cellpadding="2">
                    <tr style="vertical-align: top;">
                        <td class="tabtab">
                            <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                <tr style="background-color: #CCFFCC;">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'valueSets'"/>
                                        </xsl:call-template>
                                    </td>
                                    <td colspan="2">
                                        <xsl:value-of select="count($allValueSets/*/valueSet[@id])"/>
                                    </td>
                                </tr>
                                <tr style="background-color: #CCFFCC;">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'CodeSystems'"/>
                                        </xsl:call-template>
                                    </td>
                                    <td colspan="2">
                                        <xsl:value-of select="count($allCodeSystems/*/codeSystem[@id])"/>
                                    </td>
                                </tr>
                                <tr style="background-color: #CCFFCC;">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'ConceptMaps'"/>
                                        </xsl:call-template>
                                    </td>
                                    <td colspan="2">
                                        <xsl:value-of select="count($allConceptMaps[@id])"/>
                                    </td>
                                </tr>
                                <tr style="background-color: #CCFFCC;">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'terminologyAssociations'"/>
                                        </xsl:call-template>
                                    </td>
                                    <td colspan="2">
                                        <xsl:value-of select="count($allTerminologyAssociations/*/terminologyAssociation)"/>
                                    </td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                </table>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabTerminology'"/>
            <xsl:with-param name="actualcontent" select="$content5"/>
        </xsl:call-template>
        <!-- create all value set html representations -->
        <xsl:for-each-group select="$allValueSets/*/valueSet" group-by="concat((@id | @ref), '#', @effectiveDate)">
            <xsl:variable name="vsid" select="(@id | @ref)"/>
            <xsl:variable name="vsnm" select="@name"/>
            <xsl:variable name="vsed" select="@effectiveDate"/>
            <xsl:variable name="vsIsNewest" select="$vsed = max($allValueSets/*/valueSet[(@id | @ref) = $vsid]/xs:dateTime(@effectiveDate))"/>
            <xsl:call-template name="logMessage">
                <xsl:with-param name="level" select="$logDEBUG"/>
                <xsl:with-param name="msg">
                    <xsl:text>*** HTML for value set: name='</xsl:text>
                    <xsl:value-of select="$vsnm"/>
                    <xsl:text>' id='</xsl:text>
                    <xsl:value-of select="$vsid"/>
                    <xsl:text>' effectiveDate='</xsl:text>
                    <xsl:value-of select="$vsed"/>
                    <xsl:text>'</xsl:text>
                </xsl:with-param>
            </xsl:call-template>
            <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('VS',$vsid,$vsed,'.html','true')}" format="xhtml">
                <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                    <head>
                        <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                        <title>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'terminologyValueSetHeader'"/>
                                <xsl:with-param name="p1" select="$vsnm"/>
                            </xsl:call-template>
                        </title>
                        <!--<link href="{$theAssetsDir}decor.css" rel="stylesheet" type="text/css"/>
                            <link href="{$theAssetsDir}favicon.ico" rel="shortcut icon" type="image/x-icon"/>-->
                        <xsl:call-template name="addAssetsHeader">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </head>
                    <body>
                        <div class="indexline">
                            <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToIndex'"/>
                                </xsl:call-template>
                            </a>
                            <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                            <a href="{local:doHtmlName('tabTerminology',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToTerminology'"/>
                                </xsl:call-template>
                            </a>
                        </div>
                        <h1>
                            <xsl:call-template name="showStatusDot">
                                <xsl:with-param name="status" select="@statusCode"/>
                                <xsl:with-param name="title">
                                    <xsl:call-template name="getXFormsLabel">
                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                    </xsl:call-template>
                                </xsl:with-param>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'valueSet'"/>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <i>
                                <xsl:value-of select="
                                        if (@displayName) then
                                            @displayName
                                        else
                                            @name"/>
                            </i>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="showDate">
                                <xsl:with-param name="date" select="@effectiveDate"/>
                            </xsl:call-template>
                        </h1>
                        <!--<table style="border: 0;">
                                <tr>
                                    <td class="comment">
                                        <xsl:call-template name="doDescription">
                                            <xsl:with-param name="ns" select="desc"/>
                                            <xsl:with-param name="shortDesc" select="true()"/>
                                        </xsl:call-template>
                                    </td>
                                </tr>
                            </table>-->
                        <table width="100%" border="0">
                            <tr>
                                <td class="tabtab">
                                    <xsl:apply-templates select="."/>
                                </td>
                            </tr>
                        </table>
                    </body>
                </html>
            </xsl:result-document>
            <xsl:if test="$vsIsNewest = true()">
                <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('VS',$vsid,'dynamic','.html','true')}" format="xhtml">
                    <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                        <head>
                            <meta http-equiv="refresh" content="0; URL={local:doHtmlName('VS',$vsid,$vsed,'.html','true')}"/>
                            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                            <meta name="robots" content="noindex, nofollow"/>
                            <meta http-equiv="expires" content="0"/>
                            <!-- xhtml requirement -->
                            <title>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'terminologyValueSetHeader'"/>
                                    <xsl:with-param name="p1" select="$vsnm"/>
                                </xsl:call-template>
                            </title>
                        </head>
                        <!-- xhtml requirement -->
                        <body/>
                    </html>
                </xsl:result-document>
            </xsl:if>
        </xsl:for-each-group>
        <!-- create all code system html representations -->
        <xsl:for-each-group select="$allCodeSystems/*/codeSystem" group-by="concat((@id | @ref), '#', @effectiveDate)">
            <xsl:variable name="csid" select="(@id | @ref)"/>
            <xsl:variable name="csnm" select="@name"/>
            <xsl:variable name="csed" select="@effectiveDate"/>
            <xsl:variable name="csIsNewest" select="$csed = max($allCodeSystems/*/codeSystem[(@id | @ref) = $csid]/xs:dateTime(@effectiveDate))"/>
            <xsl:variable name="theStatusLabel">
                <xsl:variable name="t">
                    <xsl:call-template name="getXFormsLabel">
                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                    </xsl:call-template>
                </xsl:variable>
                <xsl:choose>
                    <xsl:when test="empty($t) or $t = @statusCode">
                        <xsl:call-template name="getXFormsLabel">
                            <xsl:with-param name="simpleTypeKey" select="'ReleaseStatusCodeLifeCycle'"/>
                            <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:copy-of select="$t"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:variable>
            <xsl:call-template name="logMessage">
                <xsl:with-param name="level" select="$logDEBUG"/>
                <xsl:with-param name="msg">
                    <xsl:text>*** HTML for code system: name='</xsl:text>
                    <xsl:value-of select="$csnm"/>
                    <xsl:text>' id='</xsl:text>
                    <xsl:value-of select="$csid"/>
                    <xsl:text>' effectiveDate='</xsl:text>
                    <xsl:value-of select="$csed"/>
                    <xsl:text>'</xsl:text>
                </xsl:with-param>
            </xsl:call-template>
            <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('CS',$csid,$csed,'.html','true')}" format="xhtml">
                <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                    <head>
                        <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                        <title>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'terminologyCodeSystemHeader'"/>
                                <xsl:with-param name="p1" select="$csnm"/>
                            </xsl:call-template>
                        </title>
                        <!--<link href="{$theAssetsDir}decor.css" rel="stylesheet" type="text/css"/>
                            <link href="{$theAssetsDir}favicon.ico" rel="shortcut icon" type="image/x-icon"/>-->
                        <xsl:call-template name="addAssetsHeader">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </head>
                    <body>
                        <div class="indexline">
                            <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToIndex'"/>
                                </xsl:call-template>
                            </a>
                            <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                            <a href="{local:doHtmlName('tabTerminology',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToTerminology'"/>
                                </xsl:call-template>
                            </a>
                        </div>
                        <h1>
                            <xsl:call-template name="showStatusDot">
                                <xsl:with-param name="status" select="@statusCode"/>
                                <xsl:with-param name="title" select="$theStatusLabel"/>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'CodeSystem'"/>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <i>
                                <xsl:value-of select="(@displayName, @name)[1]"/>
                            </i>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="showDate">
                                <xsl:with-param name="date" select="@effectiveDate"/>
                            </xsl:call-template>
                        </h1>
                        <!--<table style="border: 0;">
                                <tr>
                                    <td class="comment">
                                        <xsl:call-template name="doDescription">
                                            <xsl:with-param name="ns" select="desc"/>
                                            <xsl:with-param name="shortDesc" select="true()"/>
                                        </xsl:call-template>
                                    </td>
                                </tr>
                            </table>-->
                        <table width="100%" border="0">
                            <tr>
                                <td class="tabtab">
                                    <xsl:apply-templates select="."/>
                                </td>
                            </tr>
                        </table>
                    </body>
                </html>
            </xsl:result-document>
            <xsl:if test="$csIsNewest = true()">
                <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('CS',$csid,'dynamic','.html','true')}" format="xhtml">
                    <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                        <head>
                            <meta http-equiv="refresh" content="0; URL={local:doHtmlName('CS',$csid,$csed,'.html','true')}"/>
                            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                            <meta name="robots" content="noindex, nofollow"/>
                            <meta http-equiv="expires" content="0"/>
                            <!-- xhtml requirement -->
                            <title>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'terminologyCodeSystemHeader'"/>
                                    <xsl:with-param name="p1" select="$csnm"/>
                                </xsl:call-template>
                            </title>
                        </head>
                        <!-- xhtml requirement -->
                        <body/>
                    </html>
                </xsl:result-document>
            </xsl:if>
        </xsl:for-each-group>
        <!-- create all concept map html representations -->
        <xsl:for-each-group select="$allConceptMaps" group-by="concat((@id | @ref), '#', @effectiveDate)">
            <xsl:variable name="mpid" select="(@id | @ref)"/>
            <xsl:variable name="mpnm" select="@displayName"/>
            <xsl:variable name="mped" select="@effectiveDate"/>
            <xsl:variable name="csIsNewest" select="$mped = max($allConceptMaps[(@id | @ref) = $mpid]/xs:dateTime(@effectiveDate))"/>
            <xsl:variable name="theStatusLabel">
                <xsl:variable name="t">
                    <xsl:call-template name="getXFormsLabel">
                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                    </xsl:call-template>
                </xsl:variable>
                <xsl:choose>
                    <xsl:when test="empty($t) or $t = @statusCode">
                        <xsl:call-template name="getXFormsLabel">
                            <xsl:with-param name="simpleTypeKey" select="'ReleaseStatusCodeLifeCycle'"/>
                            <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:copy-of select="$t"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:variable>
            <xsl:call-template name="logMessage">
                <xsl:with-param name="level" select="$logDEBUG"/>
                <xsl:with-param name="msg">
                    <xsl:text>*** HTML for concept map: name='</xsl:text>
                    <xsl:value-of select="$mpnm"/>
                    <xsl:text>' id='</xsl:text>
                    <xsl:value-of select="$mpid"/>
                    <xsl:text>' effectiveDate='</xsl:text>
                    <xsl:value-of select="$mped"/>
                    <xsl:text>'</xsl:text>
                </xsl:with-param>
            </xsl:call-template>
            <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('MP',$mpid,$mped,'.html','true')}" format="xhtml">
                <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                    <head>
                        <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                        <title>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'terminologyConceptMapHeader'"/>
                                <xsl:with-param name="p1" select="$mpnm"/>
                            </xsl:call-template>
                        </title>
                        <!--<link href="{$theAssetsDir}decor.css" rel="stylesheet" type="text/css"/>
                            <link href="{$theAssetsDir}favicon.ico" rel="shortcut icon" type="image/x-icon"/>-->
                        <xsl:call-template name="addAssetsHeader">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </head>
                    <body>
                        <div class="indexline">
                            <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToIndex'"/>
                                </xsl:call-template>
                            </a>
                            <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                            <a href="{local:doHtmlName('tabTerminology',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToTerminology'"/>
                                </xsl:call-template>
                            </a>
                        </div>
                        <h1>
                            <xsl:call-template name="showStatusDot">
                                <xsl:with-param name="status" select="@statusCode"/>
                                <xsl:with-param name="title" select="$theStatusLabel"/>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'ConceptMap'"/>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <i>
                                <xsl:value-of select="(@displayName, @name)[1]"/>
                            </i>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="showDate">
                                <xsl:with-param name="date" select="@effectiveDate"/>
                            </xsl:call-template>
                        </h1>
                        <!--<table style="border: 0;">
                                <tr>
                                    <td class="comment">
                                        <xsl:call-template name="doDescription">
                                            <xsl:with-param name="ns" select="desc"/>
                                            <xsl:with-param name="shortDesc" select="true()"/>
                                        </xsl:call-template>
                                    </td>
                                </tr>
                            </table>-->
                        <table width="100%" border="0">
                            <tr>
                                <td class="tabtab">
                                    <xsl:apply-templates select="."/>
                                </td>
                            </tr>
                        </table>
                    </body>
                </html>
            </xsl:result-document>
            <xsl:if test="$csIsNewest = true()">
                <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('MP',$mpid,'dynamic','.html','true')}" format="xhtml">
                    <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                        <head>
                            <meta http-equiv="refresh" content="0; URL={local:doHtmlName('MP',$mpid,$mped,'.html','true')}"/>
                            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                            <meta name="robots" content="noindex, nofollow"/>
                            <meta http-equiv="expires" content="0"/>
                            <!-- xhtml requirement -->
                            <title>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'terminologyConceptMapHeader'"/>
                                    <xsl:with-param name="p1" select="$mpnm"/>
                                </xsl:call-template>
                            </title>
                        </head>
                        <!-- xhtml requirement -->
                        <body/>
                    </html>
                </xsl:result-document>
            </xsl:if>
        </xsl:for-each-group>
    </xsl:template>
    <xsl:template name="doRulesTab">
        <xsl:variable name="sc-style" select="'border-top: 6px solid white;'"/>
        <!-- certain columns are hidden initially by default when treetable is active.
            That doesn't work with classic so we don't hide them
        -->
        <xsl:variable name="coldisplay">
            <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                <xsl:text>display: none;</xsl:text>
            </xsl:if>
        </xsl:variable>
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Rules/Templates Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="representingTemplates" as="element()*">
            <xsl:for-each select="$allScenarios//representingTemplate[@ref]">
                <xsl:variable name="theTransaction" select="parent::transaction"/>
                <xsl:variable name="tmref" select="@ref"/>
                <!-- concat for backward compatibility -->
                <xsl:variable name="theTemplate">
                    <xsl:call-template name="getRulesetContent">
                        <xsl:with-param name="ruleset" select="$tmref"/>
                        <xsl:with-param name="flexibility" select="@flexibility"/>
                        <xsl:with-param name="sofar" select="()"/>
                    </xsl:call-template>
                </xsl:variable>
                <transaction xmlns="">
                    <xsl:copy-of select="$theTransaction/@*"/>
                    <xsl:copy-of select="$theTransaction/name"/>
                    <template>
                        <xsl:copy-of select="$theTemplate/template/@*"/>
                    </template>
                </transaction>
            </xsl:for-each>
        </xsl:variable>
        <xsl:variable name="content6">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <h2>
                    <a id="labelRules"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabRulesTitleString'"/>
                    </xsl:call-template>
                </h2>
                <div class="tabtab">
                    <xsl:variable name="doExpirationDate" select="$allTemplates/*/ref/template[@expirationDate[not(. = '')]]"/>
                    <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                        <div>
                            <button id="expandAll" type="button">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                </xsl:call-template>
                            </button>
                            <button id="collapseAll" type="button">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                </xsl:call-template>
                            </button>
                            <input id="nameSearch">
                                <xsl:attribute name="placeholder">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'textSearch'"/>
                                    </xsl:call-template>
                                </xsl:attribute>
                            </input>
                        </div>
                    </xsl:if>
                    <table id="transactionTable" class="treetable highlight" width="100%" border="0" cellspacing="0" cellpadding="5">
                        <thead>
                            <tr>
                                <!-- HTML -->
                                <th class="columnAllView" style="width: 50px;">HTML</th>
                                <!-- Name -->
                                <th>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'columnName'"/>
                                    </xsl:call-template>
                                </th>
                                <th style="width: 15em;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'effectiveDate'"/>
                                    </xsl:call-template>
                                </th>
                                <xsl:if test="$doExpirationDate">
                                    <th style="width: 15em;">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'expirationDate'"/>
                                        </xsl:call-template>
                                    </th>
                                </xsl:if>
                                <th style="width: 10em;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                    </xsl:call-template>
                                </th>
                            </tr>
                        </thead>
                        <tbody>
                            <xsl:variable name="node-id" select="'doclevel-templates'"/>
                            <!-- first show all document/transaction level templates -->
                            <!-- classification type row -->
                            <tr data-tt-id="{$node-id}" class="tmclassline">
                                <td class="columnName" style="border-top: 6px solid white;"/>
                                <td class="columnName" colspan="{if ($doExpirationDate) then 5 else 4}" style="border-top: 6px solid white;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tabRepresentingTemplatesString'"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <xsl:for-each select="$allTemplates/*/ref/template">
                                <xsl:sort select="if (@displayName) then lower-case(@displayName) else lower-case(@name)"/>
                                <xsl:variable name="tmid" select="@id"/>
                                <xsl:variable name="tmed" select="@effectiveDate"/>
                                <xsl:if test="$representingTemplates/template[@id = $tmid][@effectiveDate = $tmed]">
                                    <xsl:variable name="templateAnchor" select="local:doHtmlAnchor($tmid, $tmed)"/>
                                    <xsl:variable name="node-id-tm" select="generate-id(.)"/>
                                    <xsl:variable name="node-color" select="'#fff7f2'"/>
                                    <tr data-tt-id="{$node-id-tm}-reptemp" data-tt-parent-id="{$node-id}" class="tmtrline">
                                        <!-- HTML -->
                                        <td style="{$sc-style}">
                                            <a id="{$templateAnchor}"/>
                                            <a href="{local:doHtmlName('TM',$tmid,$tmed,'.html')}" onclick="target='_blank';">html</a>
                                        </td>
                                        <td class="columnName" style="{$sc-style}">
                                            <xsl:variable name="columnName">
                                                <xsl:call-template name="showStatusDot">
                                                    <xsl:with-param name="status" select="@statusCode"/>
                                                    <xsl:with-param name="title">
                                                        <xsl:call-template name="getXFormsLabel">
                                                            <xsl:with-param name="simpleTypeKey" select="'TemplateStatusCodeLifeCycle'"/>
                                                            <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                        </xsl:call-template>
                                                    </xsl:with-param>
                                                </xsl:call-template>
                                                <xsl:text>&#160;</xsl:text>
                                                <xsl:value-of select="if (@displayName) then @displayName else @name"/>
                                                <xsl:text>&#160;</xsl:text>
                                                <xsl:value-of select="@id"/>
                                                <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                            </xsl:variable>
                                            <xsl:choose>
                                                <xsl:when test="not($switchCreateTreeTableHtml = 'true')">
                                                    <table>
                                                        <tr>
                                                            <xsl:call-template name="doIndentLevel">
                                                                <xsl:with-param name="level" select="2"/>
                                                                <xsl:with-param name="icon" select="false()"/>
                                                            </xsl:call-template>
                                                            <td>
                                                                <xsl:copy-of select="$columnName"/>
                                                            </td>
                                                        </tr>
                                                    </table>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:copy-of select="$columnName"/>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </td>
                                        <td style="{$sc-style}">
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="@effectiveDate"/>
                                            </xsl:call-template>
                                        </td>
                                        <xsl:if test="$doExpirationDate">
                                            <td style="{$sc-style}">
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="@expirationDate"/>
                                                </xsl:call-template>
                                            </td>
                                        </xsl:if>
                                        <td style="{$sc-style}">
                                            <xsl:value-of select="@versionLabel"/>
                                        </td>
                                    </tr>
                                    <xsl:if test="desc[.//text()[string-length(normalize-space()) gt 0]]">
                                        <tr data-tt-id="{$node-id}-desc" data-tt-parent-id="{$node-id-tm}-reptemp">
                                            <td class="columnAllView"/>
                                            <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDate) then 4 else 3}">
                                                <xsl:value-of>
                                                    <xsl:call-template name="doDescription">
                                                        <xsl:with-param name="ns" select="desc"/>
                                                        <xsl:with-param name="shortDesc" select="true()"/>
                                                    </xsl:call-template>
                                                </xsl:value-of>
                                            </td>
                                        </tr>
                                    </xsl:if>
                                    <!-- Transaction details -->
                                    <tr data-tt-id="{$node-id-tm}-reptemp-{position()}" data-tt-parent-id="{$node-id-tm}-reptemp">
                                        <td class="columnDescription"/>
                                        <td class="columnDescription" colspan="{if ($doExpirationDate) then 5 else 4}" style="padding-left: 5em;">
                                            <xsl:variable name="doUnderlyingModel" select="$representingTemplates/template[@id = $tmid][@effectiveDate = $tmed][parent::transaction/@model]"/>
                                            <table class="tabtab" cellspacing="2" cellpadding="5">
                                                <tr class="headinglable">
                                                    <th>
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'Transaction'"/>
                                                        </xsl:call-template>
                                                    </th>
                                                    <th style="width: 160px;" colspan="2">
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'Direction'"/>
                                                        </xsl:call-template>
                                                    </th>
                                                    <xsl:if test="$doUnderlyingModel">
                                                        <th style="width: 160px;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'UnderlyingModel'"/>
                                                            </xsl:call-template>
                                                        </th>
                                                    </xsl:if>
                                                    <!--<th style="width: 160px;">
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'RuleName'"/>
                                                        </xsl:call-template>
                                                    </th>-->
                                                    <th>Schematron</th>
                                                </tr>
                                                <xsl:for-each select="$representingTemplates/template[@id = $tmid][@effectiveDate = $tmed]">
                                                    <tr>
                                                        <td>
                                                            <a href="{local:doHtmlName('tabScenarios',$projectPrefix,(),(),parent::transaction/@id, parent::transaction/@effectiveDate,(),(),'.html','false')}">
                                                                <xsl:call-template name="doName">
                                                                    <xsl:with-param name="ns" select="parent::transaction/name"/>
                                                                </xsl:call-template>
                                                            </a>
                                                        </td>
                                                        <td>
                                                            <xsl:call-template name="showDirection">
                                                                <xsl:with-param name="dir" select="parent::transaction/@type"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="concat('transactionDirection', parent::transaction/@type)"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <xsl:if test="$doUnderlyingModel">
                                                            <td>
                                                                <xsl:value-of select="parent::transaction/@model"/>
                                                            </td>
                                                        </xsl:if>
                                                        <!--<td>
                                                            <xsl:value-of select="parent::transaction/normalize-space(@label)"/>
                                                        </td>-->
                                                        <td>
                                                            <xsl:value-of select="concat($projectPrefix, parent::transaction/normalize-space(@label), '.sch')"/>
                                                        </td>
                                                    </tr>
                                                </xsl:for-each>
                                            </table>
                                        </td>
                                    </tr>
                                </xsl:if>
                            </xsl:for-each>
                            <!-- now show the rest of the templates -->
                            <xsl:variable name="classifications" select="$simpleTypes[@name = 'TemplateTypes']/*/xs:enumeration" as="element()+"/>
                            <xsl:for-each-group select="$allTemplates/*/ref/template" group-by="
                                    (if (classification/@type) then (
                                        count($classifications[@value = (current()/classification/@type)[1]]/preceding-sibling::*)
                                    ) else (
                                        count($classifications[@value = 'notype']/preceding-sibling::*)
                                    )) + 1">
                                <xsl:sort select="current-grouping-key()"/>
                                <xsl:variable name="classification" select="$classifications[position() = current-grouping-key()]/@value"/>
                                <xsl:variable name="node-id" select="concat('type-', $classification)"/>
                                <xsl:variable name="row-content" as="element()*">
                                    <xsl:for-each-group select="current-group()" group-by="@id">
                                        <xsl:sort select="lower-case(normalize-space((current-group()[1]/@displayName, current-group()[1]/@name)[1]))"/>
                                        <xsl:variable name="groupCount" select="count(current-group())"/>
                                        <xsl:for-each select="current-group()">
                                            <xsl:sort select="@effectiveDate" order="descending"/>
                                            <xsl:variable name="tmid" select="@id"/>
                                            <xsl:variable name="tmed" select="@effectiveDate"/>
                                            <xsl:variable name="pos" select="position()"/>
                                            
                                            <!--<xsl:if test="not($representingTemplates/template[@id = $tmid][@effectiveDate = $tmed])">-->
                                            <xsl:variable name="templateAnchor" select="local:doHtmlAnchor($tmid, $tmed)"/>
                                            <xsl:variable name="node-id-tm" select="generate-id(.)"/>
                                            <xsl:variable name="node-color" select="'#fff7f2'"/>
                                            <!-- if we have multiple versions of a template, group them under their name/id -->
                                            <xsl:if test="$pos = 1 and $groupCount gt 1">
                                                <tr data-tt-id="{$node-id-tm}-group" class="tmclassline">
                                                    <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                                                    
                                                    <td style="{$sc-style}" class="columnAllView">
                                                        <xsl:text>&#160;</xsl:text>
                                                    </td>
                                                    <td style="{$sc-style}" class="columnName">
                                                        <xsl:call-template name="showStatusDot">
                                                            <xsl:with-param name="status" select="'ref'"/>
                                                            <xsl:with-param name="title">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Group'"/>
                                                                </xsl:call-template>
                                                            </xsl:with-param>
                                                        </xsl:call-template>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="current-group()[1]/(@displayName, @name)[1]"/>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="$tmid"/>
                                                    </td>
                                                    <td style="{$sc-style}">
                                                        <i>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'Multiple'"/>
                                                            </xsl:call-template>
                                                            <xsl:text> (</xsl:text>
                                                            <xsl:value-of select="$groupCount"/>
                                                            <xsl:text>)</xsl:text>
                                                        </i>
                                                    </td>
                                                    <xsl:if test="$doExpirationDate">
                                                        <td style="{$sc-style}">
                                                            <xsl:text>&#160;</xsl:text>
                                                        </td>
                                                    </xsl:if>
                                                    <td style="{$sc-style}">
                                                        <xsl:text>&#160;</xsl:text>
                                                    </td>
                                                </tr>
                                                <xsl:if test="current-group()[1]/desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                    <tr data-tt-id="{$node-id-tm}-desc" data-tt-parent-id="{$node-id-tm}-group">
                                                        <td class="columnAllView"/>
                                                        <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDate) then 4 else 3}">
                                                            <xsl:value-of>
                                                                <xsl:call-template name="doDescription">
                                                                    <xsl:with-param name="ns" select="current-group()[1]/desc"/>
                                                                    <xsl:with-param name="shortDesc" select="true()"/>
                                                                </xsl:call-template>
                                                            </xsl:value-of>
                                                        </td>
                                                    </tr>
                                                </xsl:if>
                                            </xsl:if>
                                            <tr data-tt-id="{$node-id-tm}" class="tmline">
                                                <xsl:choose>
                                                    <xsl:when test="$groupCount gt 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="concat(generate-id(current-group()[1]), '-group')"/>
                                                    </xsl:when>
                                                    <xsl:when test="$pos = 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                                                    </xsl:when>
                                                    <xsl:when test="$pos > 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="generate-id(current-group()[1])"/>
                                                    </xsl:when>
                                                </xsl:choose>
                                                <td style="{$sc-style}" class="columnAllView">
                                                    <a href="{local:doHtmlName('TM',@id,@effectiveDate,'.html')}" onclick="target='_blank';">html</a>
                                                </td>
                                                <td style="{$sc-style}" class="columnName">
                                                    <xsl:call-template name="showStatusDot">
                                                        <xsl:with-param name="status" select="@statusCode"/>
                                                        <xsl:with-param name="title">
                                                            <xsl:call-template name="getXFormsLabel">
                                                                <xsl:with-param name="simpleTypeKey" select="'TemplateStatusCodeLifeCycle'"/>
                                                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                            </xsl:call-template>
                                                        </xsl:with-param>
                                                    </xsl:call-template>
                                                    <xsl:text>&#160;</xsl:text>
                                                    <xsl:value-of select="(@displayName, @name)[1]"/>
                                                    <xsl:text>&#160;</xsl:text>
                                                    <xsl:value-of select="$tmid"/>
                                                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                                </td>
                                                <td style="{$sc-style}">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="$tmed"/>
                                                    </xsl:call-template>
                                                </td>
                                                <xsl:if test="$doExpirationDate">
                                                    <td style="{$sc-style}">
                                                        <xsl:call-template name="showDate">
                                                            <xsl:with-param name="date" select="@expirationDate"/>
                                                        </xsl:call-template>
                                                    </td>
                                                </xsl:if>
                                                <td style="{$sc-style}">
                                                    <xsl:value-of select="@versionLabel"/>
                                                </td>
                                            </tr>
                                            <xsl:if test="$groupCount = 1 and $pos = 1 and desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                <tr data-tt-id="{$node-id-tm}-desc" data-tt-parent-id="{$node-id-tm}">
                                                    <td class="columnAllView"/>
                                                    <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDate) then 4 else 3}">
                                                        <xsl:value-of>
                                                            <xsl:call-template name="doDescription">
                                                                <xsl:with-param name="ns" select="desc"/>
                                                                <xsl:with-param name="shortDesc" select="true()"/>
                                                            </xsl:call-template>
                                                        </xsl:value-of>
                                                    </td>
                                                </tr>
                                            </xsl:if>
                                            <!--</xsl:if>-->
                                        </xsl:for-each>
                                    </xsl:for-each-group>
                                </xsl:variable>
                                <!-- classification type row -->
                                <xsl:if test="$row-content">
                                    <tr data-tt-id="{$node-id}" class="tmline">
                                        <td class="columnName" style="{$sc-style}"/>
                                        <td class="columnName" style="{$sc-style}" colspan="{if ($doExpirationDate) then 4 else 3}">
                                            <a id="{$classification}"/>
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'TemplateTypes'"/>
                                                <xsl:with-param name="simpleTypeValue" select="$classification"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                    <xsl:copy-of select="$row-content"/>
                                </xsl:if>
                            </xsl:for-each-group>
                        </tbody>
                    </table>
                </div>
                <!-- then show a summary of rules table -->
                <h2>
                    <a name="labelRulesSummary"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'templateSummary'"/>
                    </xsl:call-template>
                </h2>
                <table class="tabtab" style="width: 750px;" border="0" cellspacing="3" cellpadding="2">
                    <tr style="background-color: #CCFFCC;">
                        <td>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'templates'"/>
                            </xsl:call-template>
                        </td>
                        <td colspan="2">
                            <xsl:value-of select="count($allTemplates/*/ref/template)"/>
                        </td>
                    </tr>
                </table>
                <!-- create rule lists per scenario x -->
                <h2>
                    <a name="labelRulesPerScenario"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'templatesPerScenario'"/>
                    </xsl:call-template>
                </h2>
                <xsl:for-each select="$allScenarios/scenarios/scenario">
                    <xsl:variable name="scid" select="ancestor-or-self::scenario/@id"/>
                    <xsl:variable name="sced" select="ancestor-or-self::scenario/@effectiveDate"/>
                    <xsl:variable name="scenarioname">
                        <a href="{local:doHtmlName('tabScenarios', $projectPrefix, (), (), $scid, $sced, (), (),'.html', 'false')}">
                            <i>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="name"/>
                                </xsl:call-template>
                            </i>
                        </a>
                        <xsl:text> </xsl:text>
                        <xsl:call-template name="doShorthandId">
                            <xsl:with-param name="id" select="$scid"/>
                        </xsl:call-template>
                        <xsl:text> </xsl:text>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'xAsOfy'"/>
                        </xsl:call-template>
                        <xsl:call-template name="showDate">
                            <xsl:with-param name="date" select="$sced"/>
                        </xsl:call-template>
                    </xsl:variable>
                    <h3>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'templatesForScenario'"/>
                        </xsl:call-template>
                        <xsl:copy-of select="$scenarioname"/>
                    </h3>
                    <table class="tabtab" style="width: 750px;" border="0" cellspacing="3" cellpadding="2">
                        <tr>
                            <th style="width: 200px;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Transaction'"/>
                                </xsl:call-template>
                            </th>
                            <th style="width: 200px;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'representingTemplate'"/>
                                </xsl:call-template>
                            </th>
                            <th style="width: 200px;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Link'"/>
                                </xsl:call-template>
                            </th>
                            <th style="width: 150px;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'xAsOfy'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <xsl:choose>
                            <xsl:when test=".//representingTemplate[string-length(@ref) > 0]">
                                <xsl:for-each select=".//representingTemplate[string-length(@ref) > 0]">
                                    <xsl:variable name="trid" select="parent::transaction/@id"/>
                                    <xsl:variable name="tred" select="parent::transaction/@effectiveDate"/>
                                    <xsl:variable name="tmid" select="@ref"/>
                                    <xsl:variable name="tmflex" select="
                                            if (@flexibility) then
                                                (@flexibility)
                                            else
                                                ('dynamic')"/>
                                    <xsl:variable name="rccontent" as="element()*">
                                        <xsl:call-template name="getRulesetContent">
                                            <xsl:with-param name="ruleset" select="$tmid"/>
                                            <xsl:with-param name="flexibility" select="$tmflex"/>
                                            <xsl:with-param name="sofar" select="()"/>
                                        </xsl:call-template>
                                    </xsl:variable>
                                    <xsl:variable name="tmnm">
                                        <xsl:choose>
                                            <xsl:when test="string-length($rccontent/@displayName) &gt; 0">
                                                <xsl:value-of select="$rccontent/@displayName"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="$rccontent/@name"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <xsl:variable name="tmed" select="$rccontent/@effectiveDate"/>
                                    <tr>
                                        <td style="text-align: left;">
                                            <a href="{local:doHtmlName('tabScenarios', $projectPrefix, (), (), $trid, $tred, (), (), '.html', 'false')}">
                                                <xsl:call-template name="doName">
                                                    <xsl:with-param name="ns" select="parent::transaction/name"/>
                                                </xsl:call-template>
                                            </a>
                                        </td>
                                        <td style="text-align: left;">
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="$tmid"/>
                                            </xsl:call-template>
                                        </td>
                                        <td style="text-align: left;">
                                            <a href="{local:doHtmlName('TM', $rccontent/@id, $tmflex, '.html')}" onclick="target='_blank';">
                                                <xsl:value-of select="$tmnm"/>
                                            </a>
                                        </td>
                                        <td style="text-align: left;">
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$tmed"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                </xsl:for-each>
                            </xsl:when>
                            <xsl:otherwise>
                                <tr>
                                    <td style="text-align: left;" colspan="4">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'ruleNotDefinedYet'"/>
                                        </xsl:call-template>
                                    </td>
                                </tr>
                            </xsl:otherwise>
                        </xsl:choose>
                    </table>
                </xsl:for-each>
                <!-- create list of item labels in templates -->
                <h2>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'itemLabelsPerTemplate'"/>
                    </xsl:call-template>
                </h2>
                <!-- get all item labels in all registered templates and cache them -->
                <xsl:variable name="allitemlabels">
                    <labels xmlns="">
                        <xsl:for-each select="$allTemplates/*/ref/template">
                            <xsl:variable name="tid" select="@id"/>
                            <xsl:variable name="ted" select="@effectiveDate"/>
                            <xsl:variable name="tdn">
                                <xsl:choose>
                                    <xsl:when test="string-length(@displayName) &gt; 0">
                                        <xsl:value-of select="@displayName"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="@name"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </xsl:variable>
                            <xsl:for-each select=".//item[@label]">
                                <label>
                                    <xsl:attribute name="label">
                                        <xsl:value-of select="@label"/>
                                    </xsl:attribute>
                                    <xsl:attribute name="tid">
                                        <xsl:value-of select="$tid"/>
                                    </xsl:attribute>
                                    <xsl:attribute name="ted">
                                        <xsl:value-of select="$ted"/>
                                    </xsl:attribute>
                                    <xsl:attribute name="tdn">
                                        <xsl:value-of select="$tdn"/>
                                    </xsl:attribute>
                                </label>
                            </xsl:for-each>
                        </xsl:for-each>
                    </labels>
                </xsl:variable>
                <table class="tabtab" style="width: 750px;" border="0" cellspacing="3" cellpadding="2">
                    <tr>
                        <th>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'itemReference'"/>
                            </xsl:call-template>
                        </th>
                        <th colspan="2">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'templateId'"/>
                            </xsl:call-template>
                        </th>
                        <th>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'RuleName'"/>
                            </xsl:call-template>
                        </th>
                        <th>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Link'"/>
                            </xsl:call-template>
                        </th>
                        <th>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'xAsOfy'"/>
                            </xsl:call-template>
                        </th>
                    </tr>
                    <xsl:for-each select="$allitemlabels/*/label">
                        <xsl:sort select="@label"/>
                        <tr>
                            <td style="text-align: left;">
                                <xsl:value-of select="replace(string(@label), '-', '‑')"/>
                            </td>
                            <td style="text-align: left;">
                                <xsl:call-template name="showIcon">
                                    <xsl:with-param name="which">link11</xsl:with-param>
                                </xsl:call-template>
                            </td>
                            <td style="text-align: left;">
                                <xsl:call-template name="doShorthandId">
                                    <xsl:with-param name="id" select="@tid"/>
                                </xsl:call-template>
                            </td>
                            <td style="text-align: left;">
                                <i>
                                    <xsl:value-of select="@tdn"/>
                                </i>
                            </td>
                            <td style="text-align: left;">
                                <a href="{local:doHtmlName('TM',@tid,'dynamic','.html')}" onclick="target='_blank';">
                                    <xsl:value-of select="@tid"/>
                                </a>
                            </td>
                            <td style="text-align: left;">
                                <xsl:call-template name="showDate">
                                    <xsl:with-param name="date" select="@ted"/>
                                </xsl:call-template>
                            </td>
                        </tr>
                    </xsl:for-each>
                </table>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabRules'"/>
            <xsl:with-param name="actualcontent" select="$content6"/>
        </xsl:call-template>
        <!-- create all template html representations, all have the exact same contents, they are just written for referencing -->
        <xsl:for-each select="$allTemplates/*/ref/template">
            <xsl:variable name="tid" select="@id"/>
            <xsl:variable name="tnm" select="@name"/>
            <xsl:variable name="ted" select="@effectiveDate"/>
            <xsl:call-template name="logMessage">
                <xsl:with-param name="level" select="$logDEBUG"/>
                <xsl:with-param name="msg">
                    <xsl:text>*** HTML for template: name='</xsl:text>
                    <xsl:value-of select="$tnm"/>
                    <xsl:text>' id='</xsl:text>
                    <xsl:value-of select="$tid"/>
                    <xsl:text>' effectiveDate='</xsl:text>
                    <xsl:value-of select="$ted"/>
                    <xsl:text>'</xsl:text>
                </xsl:with-param>
            </xsl:call-template>
            <!-- Write the static variant for include/contains with static flexibility -->
            <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('TM',$tid,$ted,'.html','true')}" format="xhtml">
                <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                    <head>
                        <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                        <title>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Template'"/>
                            </xsl:call-template>
                            <xsl:value-of select="concat(@id, ' - ', @name)"/>
                        </title>
                        <xsl:call-template name="addAssetsHeader">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                        <!--link href="{$theAssetsDir}decor.css" rel="stylesheet" type="text/css"/-->
                    </head>
                    <body>
                        <div class="indexline">
                            <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToIndex'"/>
                                </xsl:call-template>
                            </a>
                            <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                            <a href="{local:doHtmlName('tabRules',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToRules'"/>
                                </xsl:call-template>
                            </a>
                        </div>
                        <h1>
                            <xsl:call-template name="showStatusDot">
                                <xsl:with-param name="status" select="@statusCode"/>
                                <xsl:with-param name="title">
                                    <xsl:choose>
                                        <xsl:when test="@ident[not(. = $projectPrefix)]">
                                            <xsl:value-of select="'ref'"/>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'TemplateStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </xsl:with-param>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Template'"/>
                            </xsl:call-template>
                            <!--xsl:value-of select="$tid"/-->
                            <!--xsl:if test="string-length(@displayName)>0 and (@name != @displayName)"-->
                            <xsl:text>&#160;</xsl:text>
                            <i>
                                <xsl:value-of select="
                                        if (string-length(@displayName) &gt; 0) then
                                            (@displayName)
                                        else
                                            (@name)"/>
                            </i>
                            <!--/xsl:if-->
                        </h1>
                        <!--h2>
                                <xsl:call-template name="showDate">
                                    <xsl:with-param name="date" select="@effectiveDate"/>
                                </xsl:call-template>
                            </h2-->
                        <table width="100%" border="0">
                            <tr>
                                <td class="tabtab">
                                    <xsl:apply-templates select=".">
                                        <xsl:with-param name="templatename" select="concat(@id, ' - ', @name)"/>
                                    </xsl:apply-templates>
                                </td>
                            </tr>
                        </table>
                        <xsl:call-template name="addAssetsBottom">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </body>
                </html>
            </xsl:result-document>
            <!-- Write the whole thing again in a 'DYNAMIC' file for references in include/contains -->
            <xsl:if test="../@newestForId = true()">
                <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('TM',$tid,'dynamic','.html','true')}" format="xhtml">
                    <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                        <head>
                            <meta http-equiv="refresh" content="0; URL={local:doHtmlName('TM',$tid,$ted,'.html','true')}"/>
                            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                            <meta name="robots" content="noindex, nofollow"/>
                            <meta http-equiv="expires" content="0"/>
                            <!-- xhtml requirement -->
                            <title>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Template'"/>
                                </xsl:call-template>
                                <xsl:value-of select="concat(@id, ' - ', @name)"/>
                            </title>
                        </head>
                        <!-- xhtml requirement -->
                        <body/>
                    </html>
                </xsl:result-document>
            </xsl:if>
        </xsl:for-each>
    </xsl:template>
    <xsl:template name="doQuestionnairesTab">
        <xsl:variable name="sc-style" select="'border-top: 6px solid white;'"/>
        <!-- certain columns are hidden initially by default when treetable is active.
            That doesn't work with classic so we don't hide them
        -->
        <xsl:variable name="coldisplay">
            <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                <xsl:text>display: none;</xsl:text>
            </xsl:if>
        </xsl:variable>
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Questionnaires Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="representingTemplates" as="element()*">
            <xsl:for-each select="$allScenarios//representingTemplate[@representingQuestionnaire]">
                <xsl:variable name="theTransaction" select="parent::transaction"/>
                <!-- concat for backward compatibility -->
                <xsl:variable name="qq" as="element()?">
                    <xsl:call-template name="getQuestionnaireContent">
                        <xsl:with-param name="qqid" select="@representingQuestionnaire"/>
                        <xsl:with-param name="qqed" select="@representingQuestionnaireFlexibility"/>
                    </xsl:call-template>
                </xsl:variable>
                <transaction xmlns="">
                    <xsl:copy-of select="$theTransaction/@*"/>
                    <xsl:copy-of select="$theTransaction/name"/>
                    <questionnaire>
                        <xsl:copy-of select="$qq/@*"/>
                    </questionnaire>
                </transaction>
            </xsl:for-each>
        </xsl:variable>
        <xsl:variable name="content6">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <h2>
                    <a id="labelRules"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabQuestionnairesTitleString'"/>
                    </xsl:call-template>
                </h2>
                <div class="tabtab">
                    <xsl:variable name="doExpirationDate" select="$allQuestionnaires[@expirationDate[not(. = '')]]"/>
                    <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                        <div>
                            <button id="expandAll" type="button">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                </xsl:call-template>
                            </button>
                            <button id="collapseAll" type="button">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                </xsl:call-template>
                            </button>
                            <input id="nameSearch">
                                <xsl:attribute name="placeholder">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'textSearch'"/>
                                    </xsl:call-template>
                                </xsl:attribute>
                            </input>
                        </div>
                    </xsl:if>
                    <table id="transactionTable" class="treetable highlight" width="100%" border="0" cellspacing="0" cellpadding="5">
                        <thead>
                            <tr>
                                <!-- HTML -->
                                <th class="columnAllView" style="width: 50px;">HTML</th>
                                <!-- Name -->
                                <th>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'columnName'"/>
                                    </xsl:call-template>
                                </th>
                                <th style="width: 15em;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'effectiveDate'"/>
                                    </xsl:call-template>
                                </th>
                                <xsl:if test="$doExpirationDate">
                                    <th style="width: 15em;">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'expirationDate'"/>
                                        </xsl:call-template>
                                    </th>
                                </xsl:if>
                                <th style="width: 10em;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'columnVersionLabel'"/>
                                    </xsl:call-template>
                                </th>
                            </tr>
                        </thead>
                        <tbody>
                            <xsl:variable name="node-id" select="'doclevel-templates'"/>
                            <!-- first show all document/transaction level templates -->
                            <!-- classification type row -->
                            <tr data-tt-id="{$node-id}" class="tmclassline">
                                <td class="columnName" style="border-top: 6px solid white;"/>
                                <td class="columnName" colspan="{if ($doExpirationDate) then 5 else 4}" style="border-top: 6px solid white;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tabRepresentingQuestionnairesString'"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <xsl:for-each select="$allQuestionnaires">
                                <xsl:sort select="name[1]"/>
                                <xsl:variable name="qqid" select="@id"/>
                                <xsl:variable name="qqed" select="@effectiveDate"/>
                                <xsl:if test="$representingTemplates/questionnaire[@id = $qqid][@effectiveDate = $qqed]">
                                    <xsl:variable name="templateAnchor" select="local:doHtmlAnchor($qqid, $qqed)"/>
                                    <xsl:variable name="node-id-tm" select="generate-id(.)"/>
                                    <xsl:variable name="node-color" select="'#fff7f2'"/>
                                    <tr data-tt-id="{$node-id-tm}-reptemp" data-tt-parent-id="{$node-id}" class="tmtrline">
                                        <!-- HTML -->
                                        <td style="{$sc-style}">
                                            <a id="{$templateAnchor}"/>
                                            <a href="{local:doHtmlName('QQ',$qqid,$qqed,'.html')}" onclick="target='_blank';">html</a>
                                        </td>
                                        <td class="columnName" style="{$sc-style}">
                                            <xsl:variable name="columnName">
                                                <xsl:call-template name="showStatusDot">
                                                    <xsl:with-param name="status" select="@statusCode"/>
                                                    <xsl:with-param name="title">
                                                        <xsl:call-template name="getXFormsLabel">
                                                            <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                            <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                        </xsl:call-template>
                                                    </xsl:with-param>
                                                </xsl:call-template>
                                                <xsl:text>&#160;</xsl:text>
                                                <xsl:value-of select="name[1]"/>
                                                <xsl:text>&#160;</xsl:text>
                                                <xsl:value-of select="@id"/>
                                                <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                            </xsl:variable>
                                            <xsl:choose>
                                                <xsl:when test="not($switchCreateTreeTableHtml = 'true')">
                                                    <table>
                                                        <tr>
                                                            <xsl:call-template name="doIndentLevel">
                                                                <xsl:with-param name="level" select="2"/>
                                                                <xsl:with-param name="icon" select="false()"/>
                                                            </xsl:call-template>
                                                            <td>
                                                                <xsl:copy-of select="$columnName"/>
                                                            </td>
                                                        </tr>
                                                    </table>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:copy-of select="$columnName"/>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </td>
                                        <td style="{$sc-style}">
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="@effectiveDate"/>
                                            </xsl:call-template>
                                        </td>
                                        <xsl:if test="$doExpirationDate">
                                            <td style="{$sc-style}">
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="@expirationDate"/>
                                                </xsl:call-template>
                                            </td>
                                        </xsl:if>
                                        <td style="{$sc-style}">
                                            <xsl:value-of select="@versionLabel"/>
                                        </td>
                                    </tr>
                                    <xsl:if test="desc[.//text()[string-length(normalize-space()) gt 0]]">
                                        <tr data-tt-id="{$node-id}-desc" data-tt-parent-id="{$node-id-tm}-reptemp">
                                            <td class="columnAllView"/>
                                            <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDate) then 4 else 3}">
                                                <xsl:value-of>
                                                    <xsl:call-template name="doDescription">
                                                        <xsl:with-param name="ns" select="desc"/>
                                                        <xsl:with-param name="shortDesc" select="true()"/>
                                                    </xsl:call-template>
                                                </xsl:value-of>
                                            </td>
                                        </tr>
                                    </xsl:if>
                                    <!-- Transaction details -->
                                    <tr data-tt-id="{$node-id-tm}-reptemp-{position()}" data-tt-parent-id="{$node-id-tm}-reptemp">
                                        <td class="columnDescription"/>
                                        <td class="columnDescription" colspan="{if ($doExpirationDate) then 5 else 4}" style="padding-left: 5em;">
                                            <xsl:variable name="doUnderlyingModel" select="$representingTemplates/questionnaire[@id = $qqid][@effectiveDate = $qqed][parent::transaction/@model]"/>
                                            <table class="tabtab" cellspacing="2" cellpadding="5">
                                                <tr class="headinglable">
                                                    <th>
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'Transaction'"/>
                                                        </xsl:call-template>
                                                    </th>
                                                    <th style="width: 160px;" colspan="2">
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'Direction'"/>
                                                        </xsl:call-template>
                                                    </th>
                                                    <xsl:if test="$doUnderlyingModel">
                                                        <th style="width: 160px;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'UnderlyingModel'"/>
                                                            </xsl:call-template>
                                                        </th>
                                                    </xsl:if>
                                                    <!--<th style="width: 160px;">
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'RuleName'"/>
                                                        </xsl:call-template>
                                                    </th>-->
                                                    <!--<th>Schematron</th>-->
                                                </tr>
                                                <xsl:for-each select="$representingTemplates/questionnaire[@id = $qqid][@effectiveDate = $qqed]">
                                                    <tr>
                                                        <td>
                                                            <a href="{local:doHtmlName('tabScenarios',$projectPrefix,(),(),parent::transaction/@id, parent::transaction/@effectiveDate,(),(),'.html','false')}">
                                                                <xsl:call-template name="doName">
                                                                    <xsl:with-param name="ns" select="parent::transaction/name"/>
                                                                </xsl:call-template>
                                                            </a>
                                                        </td>
                                                        <td>
                                                            <xsl:call-template name="showDirection">
                                                                <xsl:with-param name="dir" select="parent::transaction/@type"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="concat('transactionDirection', parent::transaction/@type)"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <xsl:if test="$doUnderlyingModel">
                                                            <td>
                                                                <xsl:value-of select="parent::transaction/@model"/>
                                                            </td>
                                                        </xsl:if>
                                                        <!--<td>
                                                            <xsl:value-of select="parent::transaction/normalize-space(@label)"/>
                                                        </td>-->
                                                        <!--<td>
                                                            <xsl:value-of select="concat($projectPrefix, parent::transaction/normalize-space(@label), '.sch')"/>
                                                        </td>-->
                                                    </tr>
                                                </xsl:for-each>
                                            </table>
                                        </td>
                                    </tr>
                                </xsl:if>
                            </xsl:for-each>
                            <!-- now show the rest of the templates -->
                            <xsl:variable name="classifications" select="($allQuestionnaires/classification/tag, 'notype')" as="xs:string+"/>
                            <xsl:for-each-group select="$allQuestionnaires" group-by="
                                    (if (classification/tag) then (
                                        index-of($classifications, (current()/classification/tag)[1])[1]
                                    ) else (
                                        index-of($classifications, 'notype')[1]
                                    ))">
                                <xsl:sort select="current-grouping-key()"/>
                                <xsl:variable name="classification" select="$classifications[position() = current-grouping-key()]"/>
                                <xsl:variable name="node-id" select="concat('type-', $classification)"/>
                                <xsl:variable name="row-content" as="element()*">
                                    <xsl:for-each-group select="current-group()" group-by="@id">
                                        <xsl:sort select="lower-case(normalize-space((current-group()/name)[1]))"/>
                                        <xsl:variable name="groupCount" select="count(current-group())"/>
                                        <xsl:for-each select="current-group()">
                                            <xsl:sort select="@effectiveDate" order="descending"/>
                                            <xsl:variable name="tmid" select="@id"/>
                                            <xsl:variable name="tmed" select="@effectiveDate"/>
                                            <xsl:variable name="pos" select="position()"/>
                                            
                                            <!--<xsl:if test="not($representingTemplates/template[@id = $tmid][@effectiveDate = $tmed])">-->
                                            <xsl:variable name="templateAnchor" select="local:doHtmlAnchor($tmid, $tmed)"/>
                                            <xsl:variable name="node-id-tm" select="generate-id(.)"/>
                                            <xsl:variable name="node-color" select="'#fff7f2'"/>
                                            <!-- if we have multiple versions of a template, group them under their name/id -->
                                            <xsl:if test="$pos = 1 and $groupCount gt 1">
                                                <tr data-tt-id="{$node-id-tm}-group" class="tmclassline">
                                                    <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                                                    
                                                    <td style="{$sc-style}" class="columnAllView">
                                                        <xsl:text>&#160;</xsl:text>
                                                    </td>
                                                    <td style="{$sc-style}" class="columnName">
                                                        <xsl:call-template name="showStatusDot">
                                                            <xsl:with-param name="status" select="'ref'"/>
                                                            <xsl:with-param name="title">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Group'"/>
                                                                </xsl:call-template>
                                                            </xsl:with-param>
                                                        </xsl:call-template>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="current-group()[1]/name[1]"/>
                                                        <xsl:text>&#160;</xsl:text>
                                                        <xsl:value-of select="$tmid"/>
                                                    </td>
                                                    <td style="{$sc-style}">
                                                        <i>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'Multiple'"/>
                                                            </xsl:call-template>
                                                            <xsl:text> (</xsl:text>
                                                            <xsl:value-of select="$groupCount"/>
                                                            <xsl:text>)</xsl:text>
                                                        </i>
                                                    </td>
                                                    <xsl:if test="$doExpirationDate">
                                                        <td style="{$sc-style}">
                                                            <xsl:text>&#160;</xsl:text>
                                                        </td>
                                                    </xsl:if>
                                                    <td style="{$sc-style}">
                                                        <xsl:text>&#160;</xsl:text>
                                                    </td>
                                                </tr>
                                                <xsl:if test="current-group()[1]/desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                    <tr data-tt-id="{$node-id-tm}-desc" data-tt-parent-id="{$node-id-tm}-group">
                                                        <td class="columnAllView"/>
                                                        <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDate) then 4 else 3}">
                                                            <xsl:value-of>
                                                                <xsl:call-template name="doDescription">
                                                                    <xsl:with-param name="ns" select="current-group()[1]/desc"/>
                                                                    <xsl:with-param name="shortDesc" select="true()"/>
                                                                </xsl:call-template>
                                                            </xsl:value-of>
                                                        </td>
                                                    </tr>
                                                </xsl:if>
                                            </xsl:if>
                                            <tr data-tt-id="{$node-id-tm}" class="tmline">
                                                <xsl:choose>
                                                    <xsl:when test="$groupCount gt 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="concat(generate-id(current-group()[1]), '-group')"/>
                                                    </xsl:when>
                                                    <xsl:when test="$pos = 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                                                    </xsl:when>
                                                    <xsl:when test="$pos > 1">
                                                        <xsl:attribute name="data-tt-parent-id" select="generate-id(current-group()[1])"/>
                                                    </xsl:when>
                                                </xsl:choose>
                                                <td style="{$sc-style}" class="columnAllView">
                                                    <a href="{local:doHtmlName('QQ',@id,@effectiveDate,'.html')}" onclick="target='_blank';">html</a>
                                                </td>
                                                <td style="{$sc-style}" class="columnName">
                                                    <xsl:call-template name="showStatusDot">
                                                        <xsl:with-param name="status" select="@statusCode"/>
                                                        <xsl:with-param name="title">
                                                            <xsl:call-template name="getXFormsLabel">
                                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                            </xsl:call-template>
                                                        </xsl:with-param>
                                                    </xsl:call-template>
                                                    <xsl:text>&#160;</xsl:text>
                                                    <xsl:value-of select="name[1]"/>
                                                    <xsl:text>&#160;</xsl:text>
                                                    <xsl:value-of select="$tmid"/>
                                                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                                </td>
                                                <td style="{$sc-style}">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="$tmed"/>
                                                    </xsl:call-template>
                                                </td>
                                                <xsl:if test="$doExpirationDate">
                                                    <td style="{$sc-style}">
                                                        <xsl:call-template name="showDate">
                                                            <xsl:with-param name="date" select="@expirationDate"/>
                                                        </xsl:call-template>
                                                    </td>
                                                </xsl:if>
                                                <td style="{$sc-style}">
                                                    <xsl:value-of select="@versionLabel"/>
                                                </td>
                                            </tr>
                                            <xsl:if test="$groupCount = 1 and $pos = 1 and desc[.//text()[string-length(normalize-space()) gt 0]]">
                                                <tr data-tt-id="{$node-id-tm}-desc" data-tt-parent-id="{$node-id-tm}">
                                                    <td class="columnAllView"/>
                                                    <td style="'padding-left: 2em;')" class="columnDesc comment" colspan="{if ($doExpirationDate) then 4 else 3}">
                                                        <xsl:value-of>
                                                            <xsl:call-template name="doDescription">
                                                                <xsl:with-param name="ns" select="desc"/>
                                                                <xsl:with-param name="shortDesc" select="true()"/>
                                                            </xsl:call-template>
                                                        </xsl:value-of>
                                                    </td>
                                                </tr>
                                            </xsl:if>
                                            <!--</xsl:if>-->
                                        </xsl:for-each>
                                    </xsl:for-each-group>
                                </xsl:variable>
                                <!-- classification type row -->
                                <xsl:if test="$row-content">
                                    <tr data-tt-id="{$node-id}" class="tmline">
                                        <td class="columnName" style="{$sc-style}"/>
                                        <td class="columnName" style="{$sc-style}" colspan="{if ($doExpirationDate) then 4 else 3}">
                                            <a id="{$classification}"/>
                                            <xsl:choose>
                                                <xsl:when test="$classification = 'notype'">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'questionnaireTypeNotSpecified'"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:value-of select="$classification"/>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </td>
                                    </tr>
                                    <xsl:copy-of select="$row-content"/>
                                </xsl:if>
                            </xsl:for-each-group>
                        </tbody>
                    </table>
                </div>
                <!-- then show a summary of rules table -->
                <h2>
                    <a name="labelRulesSummary"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'questionnaireSummary'"/>
                    </xsl:call-template>
                </h2>
                <table class="tabtab" style="width: 750px;" border="0" cellspacing="3" cellpadding="2">
                    <tr style="background-color: #CCFFCC;">
                        <td>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Questionnaires'"/>
                            </xsl:call-template>
                        </td>
                        <td colspan="2">
                            <xsl:value-of select="count($allQuestionnaires)"/>
                        </td>
                    </tr>
                </table>
                <!-- create rule lists per scenario x -->
                <h2>
                    <a name="labelRulesPerScenario"/>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'questionnairesPerScenario'"/>
                    </xsl:call-template>
                </h2>
                <xsl:for-each select="$allScenarios/scenarios/scenario">
                    <xsl:variable name="scid" select="ancestor-or-self::scenario/@id"/>
                    <xsl:variable name="sced" select="ancestor-or-self::scenario/@effectiveDate"/>
                    <xsl:variable name="scenarioname">
                        <a href="{local:doHtmlName('tabScenarios', $projectPrefix, (), (), $scid, $sced, (), (),'.html', 'false')}">
                            <i>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="name"/>
                                </xsl:call-template>
                            </i>
                        </a>
                        <xsl:text> </xsl:text>
                        <xsl:call-template name="doShorthandId">
                            <xsl:with-param name="id" select="$scid"/>
                        </xsl:call-template>
                        <xsl:text> </xsl:text>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'xAsOfy'"/>
                        </xsl:call-template>
                        <xsl:call-template name="showDate">
                            <xsl:with-param name="date" select="$sced"/>
                        </xsl:call-template>
                    </xsl:variable>
                    <h3>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'questionnairesForScenario'"/>
                        </xsl:call-template>
                        <xsl:copy-of select="$scenarioname"/>
                    </h3>
                    <table class="tabtab" style="width: 750px;" border="0" cellspacing="3" cellpadding="2">
                        <tr>
                            <th style="width: 200px;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Transaction'"/>
                                </xsl:call-template>
                            </th>
                            <th style="width: 200px;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'representingTemplate'"/>
                                </xsl:call-template>
                            </th>
                            <th style="width: 200px;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Link'"/>
                                </xsl:call-template>
                            </th>
                            <th style="width: 150px;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'xAsOfy'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <xsl:choose>
                            <xsl:when test=".//representingTemplate[string-length(@representingQuestionnaire) > 0]">
                                <xsl:for-each select=".//representingTemplate[string-length(@representingQuestionnaire) > 0]">
                                    <xsl:variable name="trid" select="parent::transaction/@id"/>
                                    <xsl:variable name="tred" select="parent::transaction/@effectiveDate"/>
                                    <xsl:variable name="qqid" select="@representingQuestionnaire"/>
                                    <xsl:variable name="qqed" select="(@representingQuestionnaireFlexibility, 'dynamic')[1]"/>
                                    <xsl:variable name="qq" as="element()*">
                                        <xsl:call-template name="getQuestionnaireContent">
                                            <xsl:with-param name="qqid" select="$qqid"/>
                                            <xsl:with-param name="qqed" select="$qqed"/>
                                        </xsl:call-template>
                                    </xsl:variable>
                                    <xsl:variable name="tmnm" select="$qq/name[1]"/>
                                    <xsl:variable name="tmed" select="$qq/@effectiveDate"/>
                                    <tr>
                                        <td style="text-align: left;">
                                            <a href="{local:doHtmlName('tabScenarios', $projectPrefix, (), (), $trid, $tred, (), (), '.html', 'false')}">
                                                <xsl:call-template name="doName">
                                                    <xsl:with-param name="ns" select="parent::transaction/name"/>
                                                </xsl:call-template>
                                            </a>
                                        </td>
                                        <td style="text-align: left;">
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="$qqid"/>
                                            </xsl:call-template>
                                        </td>
                                        <td style="text-align: left;">
                                            <a href="{local:doHtmlName('QQ', $qq/@id, $qqed, '.html')}" onclick="target='_blank';">
                                                <xsl:value-of select="$tmnm"/>
                                            </a>
                                        </td>
                                        <td style="text-align: left;">
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$tmed"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                </xsl:for-each>
                            </xsl:when>
                            <xsl:otherwise>
                                <tr>
                                    <td style="text-align: left;" colspan="4">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'questionnaireNotDefinedYet'"/>
                                        </xsl:call-template>
                                    </td>
                                </tr>
                            </xsl:otherwise>
                        </xsl:choose>
                    </table>
                </xsl:for-each>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabQuestionnaires'"/>
            <xsl:with-param name="actualcontent" select="$content6"/>
        </xsl:call-template>
        
        <!-- create all template html representations, all have the exact same contents, they are just written for referencing -->
        <xsl:for-each select="$allQuestionnaires[@id]">
            <xsl:variable name="tid" select="@id"/>
            <xsl:variable name="tnm">
                <xsl:call-template name="doName">
                    <xsl:with-param name="ns" select="name"/>
                </xsl:call-template>
            </xsl:variable>
            <xsl:variable name="ted" select="@effectiveDate"/>
            <xsl:call-template name="logMessage">
                <xsl:with-param name="level" select="$logDEBUG"/>
                <xsl:with-param name="msg">
                    <xsl:text>*** HTML for questionnaire: name='</xsl:text>
                    <xsl:value-of select="$tnm[1]"/>
                    <xsl:text>' id='</xsl:text>
                    <xsl:value-of select="$tid"/>
                    <xsl:text>' effectiveDate='</xsl:text>
                    <xsl:value-of select="$ted"/>
                    <xsl:text>'</xsl:text>
                </xsl:with-param>
            </xsl:call-template>
            <!-- Write the static variant for include/contains with static flexibility -->
            <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('QQ',$tid,$ted,'.html','true')}" format="xhtml">
                <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                    <head>
                        <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                        <title>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Questionnaire'"/>
                            </xsl:call-template>
                            <xsl:value-of select="concat(@id, ' - ', $tnm)"/>
                        </title>
                        <xsl:call-template name="addAssetsHeader">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                        <!--link href="{$theAssetsDir}decor.css" rel="stylesheet" type="text/css"/-->
                    </head>
                    <body>
                        <div class="indexline">
                            <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToIndex'"/>
                                </xsl:call-template>
                            </a>
                            <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                            <a href="{local:doHtmlName('tabRules',(),(),'.html', 'true')}">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'backToQuestionnaires'"/>
                                </xsl:call-template>
                            </a>
                        </div>
                        <h1>
                            <xsl:call-template name="showStatusDot">
                                <xsl:with-param name="status" select="@statusCode"/>
                                <xsl:with-param name="title">
                                    <xsl:choose>
                                        <xsl:when test="@ident[not(. = $projectPrefix)]">
                                            <xsl:value-of select="'ref'"/>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </xsl:with-param>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Questionnaire'"/>
                            </xsl:call-template>
                            <!--xsl:value-of select="$tid"/-->
                            <!--xsl:if test="string-length(@displayName)>0 and (@name != @displayName)"-->
                            <xsl:text>&#160;</xsl:text>
                            <i>
                                <xsl:copy-of select="$tnm"/>
                            </i>
                            <!--/xsl:if-->
                        </h1>
                        <!--h2>
                                <xsl:call-template name="showDate">
                                    <xsl:with-param name="date" select="@effectiveDate"/>
                                </xsl:call-template>
                            </h2-->
                        <table width="100%" border="0">
                            <tr>
                                <td class="tabtab">
                                    <xsl:apply-templates select=".">
                                        <xsl:with-param name="templatename" select="concat(@id, ' - ', @name)"/>
                                    </xsl:apply-templates>
                                </td>
                            </tr>
                        </table>
                        <xsl:call-template name="addAssetsBottom">
                            <xsl:with-param name="doTreeTable" select="true()"/>
                        </xsl:call-template>
                    </body>
                </html>
            </xsl:result-document>
            <!-- Write the whole thing again in a 'DYNAMIC' file for references in include/contains -->
            <xsl:if test="../@newestForId = true()">
                <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('QQ',$tid,'dynamic','.html','true')}" format="xhtml">
                    <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                        <head>
                            <meta http-equiv="refresh" content="0; URL={local:doHtmlName('QQ',$tid,$ted,'.html','true')}"/>
                            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                            <meta name="robots" content="noindex, nofollow"/>
                            <meta http-equiv="expires" content="0"/>
                            <!-- xhtml requirement -->
                            <title>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Questionnaire'"/>
                                </xsl:call-template>
                                <xsl:value-of select="concat(@id, ' - ')"/>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="name"/>
                                </xsl:call-template>
                            </title>
                        </head>
                        <!-- xhtml requirement -->
                        <body/>
                    </html>
                </xsl:result-document>
            </xsl:if>
        </xsl:for-each>
    </xsl:template>
    <xsl:template name="doIssuesTab">
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Issues Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="content7">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <xsl:choose>
                    <xsl:when test="count(//issues) = 0">
                        <!-- no issues at all -->
                        <h2>
                            <a name="labelIssues"/>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'tabIssuesTitleString'"/>
                            </xsl:call-template>
                        </h2>
                        <div id="Accordionis" class="Accordion">
                            <table style="border: 0;">
                                <xsl:call-template name="doMessage">
                                    <xsl:with-param name="level" select="'info'"/>
                                    <xsl:with-param name="msg">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueNoIssuesDocumentedYet'"/>
                                        </xsl:call-template>
                                    </xsl:with-param>
                                </xsl:call-template>
                            </table>
                        </div>
                    </xsl:when>
                    <xsl:otherwise>
                        <div id="Accordionis" class="Accordion">
                            <!-- show all issues sorted -->
                            <xsl:call-template name="doShowIssues"/>
                        </div>
                    </xsl:otherwise>
                </xsl:choose>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabIssues'"/>
            <xsl:with-param name="actualcontent" select="$content7"/>
        </xsl:call-template>
        <!-- create all issue html representation -->
        <xsl:for-each-group select="//issues/issue" group-by="@id">
            <xsl:variable name="isid" select="@id"/>
            <xsl:if test="string-length($isid) &gt; 0">
                <xsl:call-template name="logMessage">
                    <xsl:with-param name="level" select="$logDEBUG"/>
                    <xsl:with-param name="msg">
                        <xsl:text>*** HTML for issue: id='</xsl:text>
                        <xsl:value-of select="$isid"/>
                        <xsl:text>' name='</xsl:text>
                        <xsl:value-of select="@displayName"/>
                        <xsl:text>'</xsl:text>
                    </xsl:with-param>
                </xsl:call-template>
                <xsl:result-document href="{$theHtmlDir}{local:doHtmlName('IS',$isid,(),'.html','true')}" format="xhtml">
                    <html xml:lang="{substring($defaultLanguage,1,2)}" lang="{substring($defaultLanguage,1,2)}" xmlns="http://www.w3.org/1999/xhtml">
                        <head>
                            <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                            <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
                            <title>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'issuesHeader'"/>
                                    <xsl:with-param name="p1" select="$isid"/>
                                </xsl:call-template>
                            </title>
                            <!--<link href="{$theAssetsDir}decor.css" rel="stylesheet" type="text/css"/>
                                <link href="{$theAssetsDir}favicon.ico" rel="shortcut icon" type="image/x-icon"/>-->
                            <xsl:call-template name="addAssetsHeader">
                                <xsl:with-param name="doTreeTable" select="true()"/>
                            </xsl:call-template>
                        </head>
                        <body>
                            <div class="indexline">
                                <a href="{local:doHtmlName('tabFrontPage',(),(),'.html', 'true')}">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'backToIndex'"/>
                                    </xsl:call-template>
                                </a>
                                <xsl:text> &#160;&lt;&lt;&#160; </xsl:text>
                                <a href="{local:doHtmlName('tabIssues',(),(),'.html', 'true')}">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'backToIssues'"/>
                                    </xsl:call-template>
                                </a>
                            </div>
                            <xsl:apply-templates select="."/>
                        </body>
                    </html>
                </xsl:result-document>
            </xsl:if>
        </xsl:for-each-group>
    </xsl:template>
    <xsl:template name="doCompileTimeTab">
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Compilation Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="content8">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <h2>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabCompileTimeIssuesTitleString'"/>
                    </xsl:call-template>
                </h2>
                <div id="Accordionce" class="Accordion">
                    <!--h2>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'tabCompileTimeInfoTitleString'"/>
                        </xsl:call-template>
                    </h2-->
                    <!-- 
                        the follwing div id runtimezipavailable will be shown the download
                        button if a $theRuntimeDirZIP object is available for download
                        and nothing otherwise
                        2DO : show message that the runtime is not available yet.
                    -->
                    <!--div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <h3 class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');" id="runtimezipalert">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'tabCompileTimeInfoSchematronRuntime'"/>
                                </xsl:call-template>
                            </h3>
                        </div>
                        <div class="AccordionPanelContent" id="runtimezipavailable" zipfile="{$theRuntimeDirZIP}"-->
                    <!-- show the download button if there is a downloadable runtime zip -->
                    <!--
                        <xsl:call-template name="showIcon">
                            <xsl:with-param name="which">download</xsl:with-param>
                        </xsl:call-template>
                        <xsl:text> </xsl:text>
                        <a href="{$theRuntimeDirZIP}">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'tabCompileTimeInfoDownlaod'"/>
                            </xsl:call-template>
                        </a>
                    -->
                    <!--xsl:call-template name="showIcon">
                                <xsl:with-param name="which">download</xsl:with-param>
                            </xsl:call-template>
                            <xsl:text> </xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'tabCompileTimeInfoDownlaodNotAvailableHere'"/>
                            </xsl:call-template>

                        </div>
                    </div-->
                    <!--  List of referenced value sets that cannot be found - list already created, show it -->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="count($valueSetReferenceErrors/*/error) = 0">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">notice</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'compileTimeIssueReferencedValueSetButNotFound'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="count($valueSetReferenceErrors/*/error) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueAllReferencedValueSetsWereFound'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'compileTimeIssueReferencedValueSetButNotFoundNumber'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td>
                                                            <xsl:value-of select="count($valueSetReferenceErrors/*/error)"/>
                                                        </td>
                                                    </tr>
                                                    <xsl:for-each-group select="$valueSetReferenceErrors/*/error" group-by="concat(@id, @flexibility, @errortype)">
                                                        <xsl:variable name="id" select="@id"/>
                                                        <tr>
                                                            <td class="tabtab" colspan="2">
                                                                <table width="100%">
                                                                    <tr class="desclabel">
                                                                        <td>
                                                                            <strong>
                                                                                <xsl:value-of select="$id"/>
                                                                            </strong>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'xInBraces'"/>
                                                                                <xsl:with-param name="p1" select="@flexibility"/>
                                                                            </xsl:call-template>
                                                                        </td>
                                                                    </tr>
                                                                    <tr>
                                                                        <td style="vertical-align: top;">
                                                                            <xsl:variable name="compileTimeIssueKey">
                                                                                <xsl:choose>
                                                                                    <xsl:when test="@errortype = 'terminologyref'">compileTimeIssueReferencedFromTerminologyAssociation</xsl:when>
                                                                                    <xsl:when test="@errortype = 'valuesetref'">compileTimeIssueReferencedFromValueSetRef</xsl:when>
                                                                                    <xsl:when test="@errortype = 'templateref'">compileTimeIssueReferencedFromTemplate</xsl:when>
                                                                                    <xsl:otherwise>compileTimeIssueReferencedFromOther</xsl:otherwise>
                                                                                </xsl:choose>
                                                                            </xsl:variable>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="$compileTimeIssueKey"/>
                                                                            </xsl:call-template>
                                                                            <ul>
                                                                                <xsl:for-each select="current-group()">
                                                                                    <xsl:variable name="idEff" select="normalize-space(string-join(@from-id | @from-effectiveDate, ' '))"/>
                                                                                    <li>
                                                                                        <xsl:choose>
                                                                                            <xsl:when test="name and @from-id">
                                                                                                <xsl:value-of select="name[@language = $defaultLanguage]"/>
                                                                                                <xsl:call-template name="getMessage">
                                                                                                    <xsl:with-param name="key" select="'xInBraces'"/>
                                                                                                    <xsl:with-param name="p1" select="$idEff"/>
                                                                                                </xsl:call-template>
                                                                                            </xsl:when>
                                                                                            <xsl:otherwise>
                                                                                                <xsl:value-of select="name[@language = $defaultLanguage]"/>
                                                                                                <xsl:value-of select="$idEff"/>
                                                                                            </xsl:otherwise>
                                                                                        </xsl:choose>
                                                                                    </li>
                                                                                </xsl:for-each>
                                                                            </ul>
                                                                        </td>
                                                                    </tr>
                                                                </table>
                                                            </td>
                                                        </tr>
                                                    </xsl:for-each-group>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <!-- The arbitrary limitation on the combination of completeCodeSystem and concepts has been lifted -->
                    <!--<xsl:variable name="vsWithCodeSystemRefAndConcepts" select="count($allValueSets//valueSet[completeCodeSystem and conceptList/concept])"/>

                    <!-\- List them -\->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <h3 class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <xsl:choose>
                                    <xsl:when test="$vsWithCodeSystemRefAndConcepts=0">
                                        <xsl:call-template name="showIcon">
                                            <xsl:with-param name="which">info</xsl:with-param>
                                        </xsl:call-template>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:call-template name="showIcon">
                                            <xsl:with-param name="which">alert</xsl:with-param>
                                        </xsl:call-template>
                                    </xsl:otherwise>
                                </xsl:choose>
                                <xsl:text> </xsl:text>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'compileTimeIssueValueSetWithCodeSystemRefAndConcepts'"/>
                                </xsl:call-template>
                            </h3>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="$vsWithCodeSystemRefAndConcepts=0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoValueSetWithCodeSystemRefAndConceptsFound'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'compileTimeIssueValueSetWithCodeSystemRefAndConceptsNumber'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td>
                                                            <xsl:value-of select="$vsWithCodeSystemRefAndConcepts"/>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <td class="tabtab" colspan="2">
                                                            <ul>
                                                                <xsl:for-each select="$allValueSets//valueSet[completeCodeSystem and conceptList/concept]">
                                                                    <xsl:variable name="vsid" select="@id"/>
                                                                    <xsl:variable name="vseff" select="@effectiveDate"/>

                                                                    <li>
                                                                        <a href="{local:doHtmlName('VS',$vsid,$vseff,'.html')}">
                                                                            <xsl:value-of select="@name"/>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'xInBraces'"/>
                                                                                <xsl:with-param name="p1" select="$vsid"/>
                                                                            </xsl:call-template>
                                                                        </a>
                                                                    </li>
                                                                </xsl:for-each>
                                                            </ul>
                                                        </td>
                                                    </tr>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>-->
                    <xsl:variable name="missingTemplates">
                        <xsl:call-template name="getMissingTemplates"/>
                    </xsl:variable>
                    <!--  List of included templates without a corresponding or empty template - list already created, show it -->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="count($missingTemplates/*/error) = 0">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">alert</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'compileTimeIssueIncludedTemplatesWithoutCorrespondingOrEmptyTemplate'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="count($missingTemplates/*/error) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoIncludedTemplatesWithoutACorrespondingOrEmptyTemplateFound'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="80%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'compileTimeIssueIncludedTemplatesWithoutCorrespondingOrEmptyTemplateNumber'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td>
                                                            <xsl:value-of select="count(distinct-values($missingTemplates/*/error/concat(@ref, @flexibility, @empty)))"/>
                                                        </td>
                                                    </tr>
                                                    <xsl:for-each-group select="$missingTemplates/*/error" group-by="concat(@ref, @flexibility, @empty)">
                                                        <xsl:variable name="ref" select="current-group()[1]/@ref"/>
                                                        <xsl:variable name="flex" select="current-group()[1]/@flexibility"/>
                                                        <xsl:variable name="refdn" select="current-group()[1]/@refdisplayName"/>
                                                        <xsl:variable name="refstatus" select="current-group()[1]/@refstatusCode"/>
                                                        <xsl:variable name="empty" select="current-group()[1]/@empty"/>
                                                        <tr>
                                                            <td class="tabtab" colspan="2">
                                                                <table width="100%">
                                                                    <tr class="desclabel">
                                                                        <td>
                                                                            <xsl:if test="$refstatus">
                                                                                <xsl:call-template name="showStatusDot">
                                                                                    <xsl:with-param name="status" select="$refstatus"/>
                                                                                    <xsl:with-param name="title">
                                                                                        <xsl:call-template name="getXFormsLabel">
                                                                                            <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                                            <xsl:with-param name="simpleTypeValue" select="$refstatus"/>
                                                                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                                        </xsl:call-template>
                                                                                    </xsl:with-param>
                                                                                </xsl:call-template>
                                                                                <xsl:text> </xsl:text>
                                                                            </xsl:if>
                                                                            <strong>
                                                                                <xsl:value-of select="$ref"/>
                                                                            </strong>
                                                                            <xsl:text> </xsl:text>
                                                                            <xsl:if test="$refdn">
                                                                                <a href="{local:doHtmlName('TM',$ref,$flex,'.html')}">
                                                                                    <xsl:value-of select="$refdn"/>
                                                                                </a>
                                                                                <xsl:text> </xsl:text>
                                                                            </xsl:if>
                                                                            <xsl:choose>
                                                                                <xsl:when test="$flex castable as xs:dateTime">
                                                                                    <xsl:call-template name="showDate">
                                                                                        <xsl:with-param name="date" select="$flex"/>
                                                                                    </xsl:call-template>
                                                                                </xsl:when>
                                                                                <xsl:otherwise>
                                                                                    <xsl:call-template name="getMessage">
                                                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                                    </xsl:call-template>
                                                                                </xsl:otherwise>
                                                                            </xsl:choose>
                                                                            <xsl:text> </xsl:text>
                                                                            <xsl:if test="$empty">
                                                                                <xsl:call-template name="getMessage">
                                                                                    <xsl:with-param name="key" select="'registeredButEmpty'"/>
                                                                                </xsl:call-template>
                                                                            </xsl:if>
                                                                        </td>
                                                                    </tr>
                                                                    <tr>
                                                                        <td style="vertical-align: top;">
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'compileTimeIssueReferencedInTemplate'"/>
                                                                            </xsl:call-template>
                                                                            <ol>
                                                                                <xsl:for-each select="current-group()">
                                                                                    <li>
                                                                                        <xsl:if test="@statusCode">
                                                                                            <xsl:call-template name="showStatusDot">
                                                                                                <xsl:with-param name="status" select="@statusCode"/>
                                                                                                <xsl:with-param name="title">
                                                                                                    <xsl:call-template name="getXFormsLabel">
                                                                                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                                                    </xsl:call-template>
                                                                                                </xsl:with-param>
                                                                                            </xsl:call-template>
                                                                                            <xsl:text> </xsl:text>
                                                                                        </xsl:if>
                                                                                        <xsl:choose>
                                                                                            <xsl:when test="@id">
                                                                                                <a href="{local:doHtmlName('TM',@id,@effectiveDate,'.html')}">
                                                                                                    <xsl:value-of
                                                                                                        select="
                                                                                                            if (string-length(@displayName) &gt; 0) then
                                                                                                                @displayName
                                                                                                            else
                                                                                                                @name"
                                                                                                    />
                                                                                                </a>
                                                                                                <xsl:call-template name="getMessage">
                                                                                                    <xsl:with-param name="key" select="'xInBraces'"/>
                                                                                                    <xsl:with-param name="p1" select="@id"/>
                                                                                                </xsl:call-template>
                                                                                            </xsl:when>
                                                                                            <xsl:otherwise>
                                                                                                <xsl:value-of select="@name"/>
                                                                                                <xsl:text> </xsl:text>
                                                                                                <xsl:value-of select="@ref"/>
                                                                                            </xsl:otherwise>
                                                                                        </xsl:choose>
                                                                                    </li>
                                                                                </xsl:for-each>
                                                                            </ol>
                                                                        </td>
                                                                    </tr>
                                                                </table>
                                                            </td>
                                                        </tr>
                                                    </xsl:for-each-group>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <!-- create a list of ids with missing designation -->
                    <xsl:variable name="missingIds">
                        <ids xmlns="">
                            <xsl:for-each-group select="//@codeSystem[not(ancestor::example)]/tokenize(., '\|')" group-by=".">
                                <xsl:variable name="theOID" select="current-grouping-key()"/>
                                <xsl:if test="empty($allIDs//id[@root = $theOID] or $allDECOR//sourceCodeSystem[@id = $theOID] | $allDatasets//terminologyAssociation[@codeSystem = $theOID][@codeSystemName])">
                                    <missingId oid="{$theOID}" type="code system" count="1"/>
                                </xsl:if>
                            </xsl:for-each-group>
                            <xsl:for-each-group select="//@root[not(ancestor::example)]/tokenize(., '\|')" group-by=".">
                                <xsl:variable name="theOID" select="."/>
                                <xsl:if test="not($allIDs//id[@root = $theOID] or $allTemplates//template[@id = $theOID])">
                                    <missingId oid="{$theOID}" type="id root" count="1"/>
                                </xsl:if>
                            </xsl:for-each-group>
                        </ids>
                    </xsl:variable>
                    <!-- List them -->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="count($missingIds/*/missingId) = 0">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">notice</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'compileTimeIssueMissingIdentifierDescriptionsFromDesignations'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="count($missingIds/*/missingId) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoMissingIdentifierDescriptionsFromDesignations'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td style="text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'OID'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'Type'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td style="text-align: right;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'used'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                    </tr>
                                                    <xsl:for-each-group select="$missingIds/*/missingId" group-by="@oid">
                                                        <xsl:sort select="replace(replace(concat(@oid, '.'), '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1')"/>
                                                        <xsl:variable name="theOID" select="@oid"/>
                                                        <tr>
                                                            <td>
                                                                <xsl:value-of select="$theOID"/>
                                                            </td>
                                                            <td>
                                                                <xsl:value-of select="@type"/>
                                                            </td>
                                                            <td style="text-align: right;">
                                                                <xsl:value-of select="count($missingIds/*/missingId[@oid = $theOID])"/>
                                                                <xsl:text>x</xsl:text>
                                                            </td>
                                                        </tr>
                                                    </xsl:for-each-group>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <!--  List of issue element present in any of the templates, show them -->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="count($allTemplates//issue) = 0">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">notice</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'listOfIssuesInTemplateSources'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="count($allTemplates//issue) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoIssuesDocumentedInTemplateSources'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'listOfIssuesInTemplateSourcesNumber'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td>
                                                            <xsl:value-of select="count($allTemplates//issue)"/>
                                                        </td>
                                                    </tr>
                                                    <xsl:for-each select="$allTemplates//issue">
                                                        <tr style="vertical-align: top;">
                                                            <td colspan="2" class="tabtab">
                                                                <table border="0" cellspacing="2" cellpadding="2" width="100%">
                                                                    <tr>
                                                                        <td colspan="2" style="vertical-align: top;">
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'compileTimeIssueReferencedInTemplate'"/>
                                                                            </xsl:call-template>
                                                                            <xsl:for-each select="(ancestor-or-self::template)[last()]">
                                                                                <xsl:choose>
                                                                                    <xsl:when test="@name and @id">
                                                                                        <xsl:value-of select="@name"/>
                                                                                        <xsl:call-template name="getMessage">
                                                                                            <xsl:with-param name="key" select="'xInBraces'"/>
                                                                                            <xsl:with-param name="p1" select="@id"/>
                                                                                        </xsl:call-template>
                                                                                    </xsl:when>
                                                                                    <xsl:otherwise>
                                                                                        <xsl:value-of select="@name"/>
                                                                                        <xsl:value-of select="@id"/>
                                                                                    </xsl:otherwise>
                                                                                </xsl:choose>
                                                                            </xsl:for-each>
                                                                        </td>
                                                                    </tr>
                                                                    <tr>
                                                                        <td colspan="2" style="vertical-align: top;">
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'pathWithColon'"/>
                                                                            </xsl:call-template>
                                                                            <xsl:for-each select="ancestor-or-self::element">
                                                                                <xsl:value-of select="@name"/>
                                                                                <xsl:if test="position() != last()">
                                                                                    <xsl:text>/</xsl:text>
                                                                                </xsl:if>
                                                                            </xsl:for-each>
                                                                        </td>
                                                                    </tr>
                                                                    <xsl:for-each select="tracking">
                                                                        <tr class="desclabel">
                                                                            <td style="width: 30px;">
                                                                                <xsl:call-template name="showStatusDot">
                                                                                    <xsl:with-param name="status" select="@statusCode"/>
                                                                                    <xsl:with-param name="title">
                                                                                        <xsl:call-template name="getXFormsLabel">
                                                                                            <xsl:with-param name="simpleTypeKey" select="'IssueStatusCodeLifeCycle'"/>
                                                                                            <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                                        </xsl:call-template>
                                                                                    </xsl:with-param>
                                                                                </xsl:call-template>
                                                                            </td>
                                                                            <td style="vertical-align: top;">
                                                                                <xsl:call-template name="showDate">
                                                                                    <xsl:with-param name="date" select="@effectiveDate"/>
                                                                                </xsl:call-template>
                                                                                <xsl:text>&#160;</xsl:text>
                                                                                <xsl:for-each select="author">
                                                                                    <xsl:value-of select="."/>
                                                                                    <xsl:if test="position() != last()">
                                                                                        <xsl:text> / </xsl:text>
                                                                                    </xsl:if>
                                                                                </xsl:for-each>
                                                                            </td>
                                                                        </tr>
                                                                        <tr>
                                                                            <td/>
                                                                            <td>
                                                                                <xsl:call-template name="doDescription">
                                                                                    <xsl:with-param name="ns" select="desc"/>
                                                                                </xsl:call-template>
                                                                            </td>
                                                                        </tr>
                                                                    </xsl:for-each>
                                                                </table>
                                                            </td>
                                                        </tr>
                                                    </xsl:for-each>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <!-- create a list of duplicate ids of concepts in a single dataset -->
                    <xsl:variable name="duplicateKeyIds">
                        <duplicates xmlns="">
                            <xsl:for-each-group select="$allDatasets/dataset//concept[not(ancestor::conceptList | ancestor::history)]" group-by="concat(@id, @effectiveDate)">
                                <xsl:variable name="cnt" select="count(current-group())"/>
                                <xsl:if test="$cnt gt 1">
                                    <xsl:for-each select="current-group()">
                                        <xsl:variable name="deid" select="@id"/>
                                        <xsl:variable name="deed" select="@effectiveDate"/>
                                        <xsl:variable name="dsid" select="ancestor::dataset/@id"/>
                                        <xsl:variable name="dsed" select="ancestor::dataset/@effectiveDate"/>
                                        <dup id="{$deid}" effectiveDate="{$deed}" dataset="{$dsid}" datasetEffectiveDate="{$dsed}" count="{$cnt}"/>
                                    </xsl:for-each>
                                </xsl:if>
                            </xsl:for-each-group>
                        </duplicates>
                    </xsl:variable>
                    <!-- List them -->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="count($duplicateKeyIds/*/dup) = 0">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">notice</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'duplicateKeyIdsInDataConcepts'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="count($duplicateKeyIds/*/dup) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoDuplicateKeyIdsInDataConcepts'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td style="text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'DuplicateId'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'inDataSet'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td style="text-align: right;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'timesUsed'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                    </tr>
                                                    <xsl:for-each-group select="$duplicateKeyIds/*/dup" group-by="@id">
                                                        <xsl:sort select="replace(replace(concat(@id, '.'), '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1')"/>
                                                        <tr>
                                                            <td>
                                                                <xsl:value-of select="@id"/>
                                                            </td>
                                                            <td>
                                                                <xsl:value-of select="@dataset"/>
                                                            </td>
                                                            <td style="text-align: right;">
                                                                <xsl:value-of select="@count"/>
                                                                <xsl:text>x</xsl:text>
                                                            </td>
                                                        </tr>
                                                    </xsl:for-each-group>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <!-- create a list of data types used that are NOT in supported data types -->
                    <xsl:variable name="missingDatatypes">
                        <dts xmlns="">
                            <xsl:for-each-group select="//element/@datatype" group-by="concat(., local:getTemplateFormat(ancestor::template[1]))">
                                <xsl:variable name="theDT" select="."/>
                                <xsl:variable name="theDTtype" select="local:getTemplateFormat(current-group()[1]/ancestor::template[1])" as="xs:string"/>
                                <xsl:variable name="theDTsupported" select="exists($supportedDatatypes/dt[@type = $theDTtype][@name = $theDT])" as="xs:boolean"/>
                                <dt name="{$theDT}" supported="{$theDTsupported}" count="{count(current-group())}"/>
                            </xsl:for-each-group>
                            <xsl:for-each-group select="//attribute/@datatype" group-by="concat(., local:getTemplateFormat(ancestor::template[1]))">
                                <xsl:variable name="theDT" select="."/>
                                <xsl:variable name="theDTtype" select="local:getTemplateFormat(ancestor::template[1])" as="xs:string"/>
                                <xsl:variable name="theDTsupported" select="exists($supportedAtomicDatatypes/dt[@type = $theDTtype][@name = $theDT])" as="xs:boolean"/>
                                <dt name="{$theDT}" supported="{$theDTsupported}" count="{count(current-group())}"/>
                            </xsl:for-each-group>
                        </dts>
                    </xsl:variable>
                    <!-- List them -->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="count($missingDatatypes/*/dt[@supported = 'false']) = 0">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">alert</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'compileTimeIssueDatatypeFlavors'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="count($missingDatatypes/*/dt) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoDatatypesOrDatatypeFlavorsUsed'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td style="text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'compileTimeIssueDatatypeHeading'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'compileTimeIssueDatatypeDefined'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td style="text-align: right;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'compileTimeIssueDatatypeUsed'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                    </tr>
                                                    <xsl:for-each select="$missingDatatypes/*/dt">
                                                        <xsl:sort select="@name"/>
                                                        <xsl:variable name="theDT" select="@name"/>
                                                        <tr>
                                                            <xsl:if test="@supported != 'true'">
                                                                <xsl:attribute name="style" select="'background-color: #FFEEEE;'"/>
                                                            </xsl:if>
                                                            <td>
                                                                <xsl:choose>
                                                                    <xsl:when test="contains($theDT, '.')">
                                                                        <!-- indent a data type flavor -->
                                                                        <xsl:call-template name="repeatString">
                                                                            <xsl:with-param name="number" select="2"/>
                                                                            <xsl:with-param name="theString" select="'&#160;'"/>
                                                                        </xsl:call-template>
                                                                        <xsl:value-of select="$theDT"/>
                                                                    </xsl:when>
                                                                    <xsl:otherwise>
                                                                        <xsl:value-of select="$theDT"/>
                                                                    </xsl:otherwise>
                                                                </xsl:choose>
                                                            </td>
                                                            <td>
                                                                <xsl:choose>
                                                                    <xsl:when test="@supported = 'true'">
                                                                        <xsl:call-template name="getMessage">
                                                                            <xsl:with-param name="key" select="'yes'"/>
                                                                        </xsl:call-template>
                                                                    </xsl:when>
                                                                    <xsl:otherwise>
                                                                        <xsl:call-template name="getMessage">
                                                                            <xsl:with-param name="key" select="'no'"/>
                                                                        </xsl:call-template>
                                                                    </xsl:otherwise>
                                                                </xsl:choose>
                                                            </td>
                                                            <td style="text-align: right;">
                                                                <xsl:value-of select="@count"/>
                                                                <xsl:text>x</xsl:text>
                                                            </td>
                                                        </tr>
                                                    </xsl:for-each>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <!-- create a list of references thru @ref anywhere that are not resolvable within this DECOR project. 
                        - Contains/Inherit from different project will not resolve by design
                        - Partial compilations will not all resolve by design, hence we skip those that have precalculated attributes because they were apparently resolved in compilation
                    -->
                    <xsl:variable name="refs">
                        <refs xmlns="">
                            <xsl:for-each-group select="$allDECOR//*[@ref][empty(@datasetId)][not(
                                self::codeSystem | 
                                ancestor::history | 
                                self::inherit[not(@prefix = $projectPrefix)] | 
                                self::relationship[not(@prefix = $projectPrefix)] |
                                self::contains[not(@prefix = $projectPrefix)] |
                                self::concept[parent::templateAssociation][not(@ident = $projectPrefix)] | 
                                ancestor-or-self::*[@statusCode = ('cancelled', 'rejected')]
                            )]" group-by="concat(@ref, (@flexibility, @effectiveDate)[. castable as xs:dateTime][1])">
                                <xsl:variable name="theRef" select="current-group()[1]/@ref"/>
                                <xsl:variable name="theCount" select="count($allDECOR//*[@id = $theRef][not(ancestor::history)] | $allDECOR//*[@name = $theRef][not(ancestor::history)] | $allDECOR//*[@root = $theRef][not(ancestor::history)])"/>
                                <xsl:if test="$theCount = 0">
                                    <ref ref="{$theRef}" count="{$theCount}">
                                        <xsl:choose>
                                            <xsl:when test="matches(string($theRef), '[1-9][0-9]*(\.[0-9]+)*')">
                                                <xsl:attribute name="type" select="'oid'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:attribute name="type" select="'name'"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:for-each select="current-group()">
                                            <usage>
                                                <xsl:value-of select="string-join(ancestor-or-self::*/name(), '/')"/>
                                            </usage>
                                        </xsl:for-each>
                                    </ref>
                                </xsl:if>
                            </xsl:for-each-group>
                        </refs>
                    </xsl:variable>
                    <!-- List them -->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="empty($refs/*/ref)">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">notice</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'RefsWithoutId'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="empty($refs/*/ref)">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoIdForRef'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td style="text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'References'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                        <td>#</td>
                                                    </tr>
                                                    <xsl:for-each-group select="$refs/*/ref" group-by="@ref">
                                                        <xsl:sort select="replace(replace(concat(@ref, '.'), '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1')"/>
                                                        <tr>
                                                            <td>
                                                                <xsl:value-of select="@ref"/>
                                                                <div>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'pathWithColon'"/>
                                                                    </xsl:call-template>
                                                                    <xsl:for-each select="usage">
                                                                        <xsl:value-of select="."/>
                                                                        <xsl:if test="position() != last()">
                                                                            <br/>
                                                                        </xsl:if>
                                                                    </xsl:for-each>
                                                                </div>
                                                            </td>
                                                            <td>
                                                                <xsl:value-of select="@count"/>
                                                            </td>
                                                        </tr>
                                                    </xsl:for-each-group>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <!-- create a list of concept/@ref within a representingTemplate that do not belong to the data set stated in @sourceDataset -->
                    <xsl:variable name="conref" as="element(ref)*">
                        <xsl:for-each-group select="//representingTemplate[@sourceDataset][not(ancestor::*[@statusCode = ('cancelled', 'deprecated', 'rejected')])]" group-by="concat(@sourceDataset, @sourceDatasetFlexibility)">
                            <xsl:variable name="dsid" select="current-group()[1]/@sourceDataset"/>
                            <xsl:variable name="dsed" select="current-group()[1]/@sourceDatasetFlexibility"/>
                            <xsl:variable name="dataset" select="if ($dsid) then local:getDataset($dsid, $dsed) else ()" as="element()?"/>
                            <xsl:for-each select="current-group()">
                                <xsl:variable name="trid" select="ancestor::transaction[1]/@id"/>
                                <xsl:variable name="tred" select="ancestor::transaction[1]/@effectiveDate"/>
                                <xsl:variable name="thelabel" select="normalize-space(concat(@displayName, ' ', parent::transaction/name[@language = $defaultLanguage]))"/>
                                <xsl:for-each select="concept">
                                    <xsl:variable name="deid" select="@ref"/>
                                    <xsl:variable name="deed" select="@flexibility[not(. = 'dynamic')]"/>
                                    <xsl:variable name="concept" select="local:getConcept($deid, $deed, $dsid, $dsed)"/>
                                    <xsl:if test="empty($concept)">
                                        <xsl:variable name="alternativeDataset" as="element()*">
                                            <xsl:variable name="theConcept" select="$allDatasets//concept[@id = $deid]"/>
                                            <xsl:choose>
                                                <xsl:when test="$deed">
                                                    <xsl:copy-of select="$allDatasets//concept[@id = $deid][@effectiveDate = $deed]/ancestor::dataset"/>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:copy-of select="$allDatasets//concept[@id = $deid][@effectiveDate = max($theConcept/xs:dateTime(@effectiveDate))]/ancestor::dataset"/>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:variable>
                                        <ref ref="{$deid}" trid="{$trid}" tred="{$tred}" label="{$thelabel}" sourceDataset="{$dsid}" xmlns="">
                                            <xsl:if test="$deed">
                                                <xsl:attribute name="flexibility" select="$deed"/>
                                            </xsl:if>
                                            <xsl:if test="$dsed">
                                                <xsl:attribute name="sourceDatasetFlexibility" select="$dsed"/>
                                            </xsl:if>
                                            <xsl:if test="$dataset">
                                                <xsl:attribute name="datasetId" select="$dataset/@id"/>
                                                <xsl:attribute name="datasetEffectiveDate" select="$dataset/@effectiveDate"/>
                                                <xsl:attribute name="datasetName" select="($dataset/name[@language = $defaultLanguage], $dataset/name)[1]"/>
                                            </xsl:if>
                                            <xsl:if test="$alternativeDataset">
                                                <xsl:attribute name="locationId" select="$alternativeDataset/@id"/>
                                                <xsl:attribute name="locationEffectiveDate" select="$alternativeDataset/@effectiveDate"/>
                                                <xsl:attribute name="locationName" select="($alternativeDataset/name[@language = $defaultLanguage], $alternativeDataset/name)[1]"/>
                                            </xsl:if>
                                        </ref>
                                    </xsl:if>
                                </xsl:for-each>
                            </xsl:for-each>
                        </xsl:for-each-group>
                    </xsl:variable>
                    <!-- List them -->
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="count($conref) = 0">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">notice</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'RepTemConceptsNotInSourceDataSet'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="count($conref) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueNoRepTemConceptsNotInSourceDataSet'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td style="text-align: left;" colspan="2">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'RepTemConceptsNotInSourceDataSet'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                    </tr>
                                                    <xsl:for-each-group select="$conref" group-by="concat(@trid, @tred)">
                                                        <xsl:sort select="replace(replace(concat(current-grouping-key(), '.'), '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1')"/>
                                                        <xsl:variable name="trid" select="current-group()[1]/@trid"/>
                                                        <xsl:variable name="tred" select="current-group()[1]/@tred"/>
                                                        <xsl:variable name="trname" select="current-group()[1]/@label"/>
                                                        <xsl:variable name="dsid" select="current-group()[1]/@sourceDataset"/>
                                                        <xsl:variable name="dsed" select="current-group()[1]/@sourceDatasetFlexibility"/>
                                                        <xsl:variable name="dsname" select="current-group()[1]/@datasetName"/>
                                                        <tr style="background-color: #FFEEEE;">
                                                            <td style="text-align: left;" colspan="2">
                                                                <div>
                                                                    <b>
                                                                        <xsl:call-template name="getMessage">
                                                                            <xsl:with-param name="key" select="'Transaction'"/>
                                                                        </xsl:call-template>
                                                                        <xsl:text>: </xsl:text>
                                                                    </b>
                                                                    <xsl:value-of select="$trid"/>
                                                                    <xsl:text> </xsl:text>
                                                                    <i>
                                                                        <a href="{local:doHtmlName('TR', @trid, @tred, '.html')}" onclick="target='_blank';">
                                                                            <xsl:value-of select="normalize-space(@label)"/>
                                                                        </a>
                                                                    </i>
                                                                    <xsl:text> (</xsl:text>
                                                                    <xsl:value-of select="$tred"/>
                                                                    <xsl:text>)</xsl:text>
                                                                </div>
                                                                <div>
                                                                    <b>
                                                                        <xsl:call-template name="getMessage">
                                                                            <xsl:with-param name="key" select="'sourceDataSet'"/>
                                                                        </xsl:call-template>
                                                                        <xsl:text>: </xsl:text>
                                                                    </b>
                                                                    <xsl:value-of select="$dsid"/>
                                                                    <xsl:text> </xsl:text>
                                                                    <i>
                                                                        <xsl:value-of select="$dsname"/>
                                                                    </i>
                                                                    <xsl:text> (</xsl:text>
                                                                    <xsl:value-of select="$dsed"/>
                                                                    <xsl:text>)</xsl:text>
                                                                </div>
                                                            </td>
                                                        </tr>
                                                        <xsl:for-each select="current-group()">
                                                            <xsl:sort select="replace(replace(concat(@ref, '.'), '\.', '.0000000000'), '.0*([0-9]{9,})', '.$1')"/>
                                                            <tr>
                                                                <td>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'conceptIdLabel'"/>
                                                                    </xsl:call-template>
                                                                    <xsl:value-of select="@ref"/>
                                                                </td>
                                                                <td>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'compileTimeIsNotInSourceDataset'"/>
                                                                    </xsl:call-template>
                                                                </td>
                                                            </tr>
                                                            <tr>
                                                                <td>&#160;</td>
                                                                <td>
                                                                    <xsl:choose>
                                                                        <xsl:when test="@locationId">
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'compileTimeButIsInDataset'"/>
                                                                            </xsl:call-template>
                                                                            <xsl:text>: </xsl:text>
                                                                            <xsl:value-of select="@locationId"/>
                                                                            <xsl:text> </xsl:text>
                                                                            <i>
                                                                                <a href="{local:doHtmlName('DS', $projectPrefix, @locationId, @locationEffectiveDate, @ref, @flexibility, (), (), '.html', 'false')}" onclick="target='_blank';">
                                                                                    <xsl:value-of select="@locationName"/>
                                                                                </a>
                                                                            </i>
                                                                            <xsl:text> (</xsl:text>
                                                                            <xsl:value-of select="@locationEffectiveDate"/>
                                                                            <xsl:text>)</xsl:text>
                                                                        </xsl:when>
                                                                        <xsl:otherwise>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'compileTimeIsInNoDataset'"/>
                                                                            </xsl:call-template>
                                                                        </xsl:otherwise>
                                                                    </xsl:choose>
                                                                </td>
                                                            </tr>
                                                        </xsl:for-each>
                                                    </xsl:for-each-group>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:choose>
                                        <xsl:when test="count($allMissingLabels/*/missingCode) = 0">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">info</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">notice</xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>&#160;</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'compileTimeIssueReferencedLabelNotDefined'"/>
                                    </xsl:call-template>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <xsl:choose>
                                <xsl:when test="count($allMissingLabels/*/missingCode) = 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="level" select="'info'"/>
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'issueAllUsedIssueLabelWereDefined'"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table width="50%" border="0" cellspacing="2" cellpadding="2">
                                        <tr style="vertical-align: top;">
                                            <td class="tabtab">
                                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                                    <tr style="background-color: #FFCCCC;">
                                                        <td style="text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'compileTimeIssueReferencedLabelNotDefined'"/>
                                                            </xsl:call-template>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <td>
                                                            <table width="100%">
                                                                <thead>
                                                                    <tr>
                                                                        <th>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'definedCode'"/>
                                                                            </xsl:call-template>
                                                                        </th>
                                                                        <th>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'Name'"/>
                                                                            </xsl:call-template>
                                                                        </th>
                                                                        <th>
                                                                            <xsl:call-template name="getMessage">
                                                                                <xsl:with-param name="key" select="'Issue'"/>
                                                                            </xsl:call-template>
                                                                        </th>
                                                                    </tr>
                                                                </thead>
                                                                <tbody>
                                                                    <xsl:for-each select="$allMissingLabels/wrap/missingCode">
                                                                        <tr>
                                                                            <td>
                                                                                <xsl:value-of select="@labelCode"/>
                                                                            </td>
                                                                            <td>
                                                                                <xsl:value-of select="@refName"/>
                                                                            </td>
                                                                            <td>
                                                                                <xsl:call-template name="doShorthandId">
                                                                                    <xsl:with-param name="id" select="@ref"/>
                                                                                </xsl:call-template>
                                                                            </td>
                                                                        </tr>
                                                                    </xsl:for-each>
                                                                </tbody>
                                                            </table>
                                                        </td>
                                                    </tr>
                                                </table>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </div>
                    <div class="AccordionPanel">
                        <div class="AccordionPanelTab">
                            <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                <h3>
                                    <xsl:call-template name="showIcon">
                                        <xsl:with-param name="which">info</xsl:with-param>
                                    </xsl:call-template>
                                    <!-- !!!! don't delete or replace this div as it is used by ADRAM -->
                                    <div class="transformationProtocolIcon" style="display:none;"/>
                                    <xsl:text> </xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'transformationProtocol'"/>
                                    </xsl:call-template>
                                    <!-- !!!! don't delete or replace this div as it is used by ADRAM -->
                                    <div class="transformationProtocolSummary" style="display:none;"/>
                                </h3>
                            </div>
                        </div>
                        <xsl:text>

</xsl:text>
                        <div class="AccordionPanelContent">
                            <!-- !!!! don't delete or replace these div lines as it is used by ADRAM -->
                            <code>Protocol not registered. <!--
                                <div class="transformationProtocolContent" style="display:none;"/>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'transformationProtocolNotRegistered'"/>
                                </xsl:call-template>
                                <div class="transformationProtocolContent" style="display:none;"/>
                                --></code>
                        </div>
                    </div>
                    <!-- indicate filtered/partial publication if filters on -->
                    <xsl:if test="$filtersfileavailable = true()">
                        <div class="AccordionPanel">
                            <div class="AccordionPanelTab">
                                <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                                    <h3>
                                        <xsl:call-template name="showIcon">
                                            <xsl:with-param name="which">notice</xsl:with-param>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'partialPublication'"/>
                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                        </xsl:call-template>
                                    </h3>
                                </div>
                            </div>
                            <xsl:text>

</xsl:text>
                            <div class="AccordionPanelContent">
                                <xsl:call-template name="showIcon">
                                    <xsl:with-param name="which">partialpublication</xsl:with-param>
                                </xsl:call-template>
                                <xsl:text>&#160;</xsl:text>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'partialPublicationInfolist'"/>
                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                    <xsl:with-param name="p1" select="doc($filtersfile)/*/normalize-space(@label)"/>
                                </xsl:call-template>
                                <p/>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'partialPublicationArtefactlist'"/>
                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                </xsl:call-template>
                                <ul>
                                    <xsl:for-each select="doc($filtersfile)/filters/*[name() = ('dataset', 'scenario', 'transaction', 'valueSet', 'template')]">
                                        <xsl:variable name="artefactlabel">
                                            <xsl:choose>
                                                <xsl:when test="name() = 'dataset'">Dataset</xsl:when>
                                                <xsl:when test="name() = 'scenario'">Scenario</xsl:when>
                                                <xsl:when test="name() = 'transaction'">Transaction</xsl:when>
                                                <xsl:when test="name() = 'valueSet'">valueSet</xsl:when>
                                                <xsl:when test="name() = 'template'">Template</xsl:when>
                                            </xsl:choose>
                                        </xsl:variable>
                                        <li>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="$artefactlabel"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                            <xsl:text> </xsl:text>
                                            <xsl:value-of select="@ref"/>
                                            <xsl:if test="string-length(@name) > 0">
                                                <xsl:text> (</xsl:text>
                                                <xsl:value-of select="@name"/>
                                                <xsl:text>)</xsl:text>
                                            </xsl:if>
                                        </li>
                                    </xsl:for-each>
                                </ul>
                            </div>
                        </div>
                    </xsl:if>
                </div>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabCompileTime'"/>
            <xsl:with-param name="actualcontent" select="$content8"/>
        </xsl:call-template>
    </xsl:template>
    <xsl:template name="doLegalTab">
        <!-- a little milestoning -->
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logINFO"/>
            <xsl:with-param name="msg">
                <xsl:text>*** Creating HTML For Legal Tab</xsl:text>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:variable name="content9">
            <div class="TabbedPanelsContent" xmlns="http://www.w3.org/1999/xhtml">
                <table width="50%" border="0" cellspacing="20" cellpadding="20">
                    <tr>
                        <td>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'licenseNoticeGPL'"/>
                            </xsl:call-template>
                            <a href="http://www.gnu.org/licenses/gpl.html">http://www.gnu.org/licenses/gpl.html</a>
                        </td>
                    </tr>
                    <tr>
                        <td>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'furtherContentToBeDetermined_1'"/>
                            </xsl:call-template>
                            <a href="https://art-decor.org">art-decor.org</a>
                        </td>
                    </tr>
                    <tr>
                        <td>
                            <a href="http://www.axialis.com/free/icons">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Icons'"/>
                                </xsl:call-template>
                            </a>
                            <xsl:text> </xsl:text>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'by'"/>
                            </xsl:call-template>
                            <xsl:text> </xsl:text>
                            <a href="http://www.axialis.com">Axialis Team</a>
                        </td>
                    </tr>
                </table>
            </div>
        </xsl:variable>
        <xsl:call-template name="doTabs4ThisTab">
            <xsl:with-param name="actualtab" select="'tabLegal'"/>
            <xsl:with-param name="actualcontent" select="$content9"/>
        </xsl:call-template>
    </xsl:template>
    <xsl:template name="doProjectInfo">
        <!--h3>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabProjectInfoProject'"/>
                    </xsl:call-template>
                </h3>
                <strong>
                    <xsl:call-template name="doName">
                        <xsl:with-param name="ns" select="//project/name"/>
                    </xsl:call-template>
                </strong-->
        <!--h3>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabProjectInfoDefaultLanguage'"/>
                    </xsl:call-template>
                </h3>
                <xsl:value-of select="//project/@defaultLanguage"/-->
        <xsl:choose>
            <xsl:when test="count(//project/desc[string-length(.) &gt; 0])">
                <!--h3>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'tabProjectInfoDescription'"/>
                    </xsl:call-template>
                </h3-->
                <xsl:call-template name="doDescription">
                    <xsl:with-param name="ns" select="//project/desc[string-length(.) &gt; 0]"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <xsl:call-template name="getMessage">
                    <xsl:with-param name="key" select="'tabProjectInfoMissing'"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
        <xsl:for-each select="(//project/license)[1]">
            <h3 xmlns="http://www.w3.org/1999/xhtml">
                <xsl:call-template name="getMessage">
                    <xsl:with-param name="key" select="'tabProjectInfoLicense'"/>
                </xsl:call-template>
            </h3>
            <table style="border: 0;" xmlns="http://www.w3.org/1999/xhtml">
                <tr>
                    <td>
                        <xsl:variable name="licenseCode" select="@code"/>
                        <xsl:variable name="licenseDisplayName" select="@displayName"/>
                        <xsl:value-of select="$licenseCode"/>
                        <xsl:text> - </xsl:text>
                        <xsl:choose>
                            <xsl:when test="$licenseDisplayName">
                                <xsl:value-of select="$licenseDisplayName"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="getXFormsLabel">
                                    <xsl:with-param name="simpleTypeKey" select="'LicenseTypes'"/>
                                    <xsl:with-param name="simpleTypeValue" select="$licenseCode"/>
                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </td>
                </tr>
            </table>
        </xsl:for-each>
        <h3 xmlns="http://www.w3.org/1999/xhtml">
            <xsl:call-template name="getMessage">
                <xsl:with-param name="key" select="'tabProjectInfoAdditional'"/>
            </xsl:call-template>
        </h3>
        <table style="border: 0;" xmlns="http://www.w3.org/1999/xhtml">
            <tr>
                <td class="tabtab">
                    <table width="100%" border="0" cellspacing="3" cellpadding="2">
                        <xsl:if test="$projectIsRepository or $projectIsPrivate">
                            <tr>
                                <th style="text-align: left;" colspan="2">
                                    <xsl:call-template name="doPrivateRepositoryNotice"/>
                                </th>
                            </tr>
                        </xsl:if>
                        <tr class="headinglabel">
                            <th style="text-align: left;" colspan="2">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Properties'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <tr>
                            <td colspan="2">
                                <span style="margin-right: 20px">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Prefix'"/>
                                    </xsl:call-template>
                                    <xsl:text>: </xsl:text>
                                    <xsl:value-of select="$projectPrefix"/>
                                </span>
                                <span style="margin-right: 20px">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tabProjectInfoDefaultLanguage'"/>
                                    </xsl:call-template>
                                    <xsl:text>: </xsl:text>
                                    <xsl:value-of select="//project/@defaultLanguage"/>
                                </span>
                                <xsl:if test="$projectIsRepository">
                                    <span style="margin-right: 20px">
                                        <xsl:attribute name="title">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Is-repository-hint'"/>
                                            </xsl:call-template>
                                        </xsl:attribute>
                                        <input type="checkbox" disabled="disabled" value="{$projectIsRepository}" checked="{$projectIsRepository}">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">repository</xsl:with-param>
                                                <xsl:with-param name="style">margin: 0px 5px;</xsl:with-param>
                                            </xsl:call-template>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Is-repository'"/>
                                            </xsl:call-template>
                                        </input>
                                    </span>
                                </xsl:if>
                                <xsl:if test="$projectIsPrivate">
                                    <span style="margin-right: 20px">
                                        <xsl:attribute name="title">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Is-private-hint'"/>
                                            </xsl:call-template>
                                        </xsl:attribute>
                                        <input type="checkbox" disabled="disabled" value="{$projectIsPrivate}" checked="{$projectIsPrivate}">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Is-private'"/>
                                            </xsl:call-template>
                                        </input>
                                    </span>
                                </xsl:if>
                                <xsl:if test="$projectIsExperimental">
                                    <span>
                                        <xsl:attribute name="title">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Is-experimental-hint'"/>
                                            </xsl:call-template>
                                        </xsl:attribute>
                                        <input type="checkbox" disabled="disabled" value="{$projectIsExperimental}" checked="{$projectIsExperimental}">
                                            <xsl:call-template name="showIcon">
                                                <xsl:with-param name="which">experimental</xsl:with-param>
                                                <xsl:with-param name="style">margin: 0px 5px;</xsl:with-param>
                                            </xsl:call-template>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Is-experimental'"/>
                                            </xsl:call-template>
                                        </input>
                                    </span>
                                </xsl:if>
                            </td>
                        </tr>
                        <tr class="headinglabel">
                            <th style="text-align: left; width: 50%;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'publication-location'"/>
                                </xsl:call-template>
                            </th>
                            <th style="text-align: left; width: 50%;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'project-overview'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <tr>
                            <td>
                                <a href="{$seeURLprefix}">
                                    <xsl:value-of select="$seeURLprefix"/>
                                </a>
                            </td>
                            <td>
                                <a href="{$projectRestUri}">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'project-index'"/>
                                    </xsl:call-template>
                                </a>
                            </td>
                        </tr>
                        <tr class="headinglabel">
                            <th style="text-align: left;" colspan="2">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'tabProjectInfoDisclaimer'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <tr>
                            <td style="text-align: left;" colspan="2">
                                <xsl:value-of select="$disclaimer"/>
                            </td>
                        </tr>
                        <xsl:if test="//project/author">
                            <tr class="headinglabel">
                                <th style="text-align: left;" colspan="2">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tabProjectInfoAuthorList'"/>
                                    </xsl:call-template>
                                </th>
                            </tr>
                            <tr>
                                <td style="text-align: left;" colspan="2">
                                    <ul>
                                        <xsl:for-each select="//project/author">
                                            <li>
                                                <xsl:value-of select="."/>
                                            </li>
                                        </xsl:for-each>
                                    </ul>
                                </td>
                            </tr>
                        </xsl:if>
                    </table>
                </td>
            </tr>
        </table>
        <h3 xmlns="http://www.w3.org/1999/xhtml">
            <xsl:call-template name="getMessage">
                <xsl:with-param name="key" select="'tabProjectInfoVersionInfo'"/>
            </xsl:call-template>
        </h3>
        <table width="100%" border="0" xmlns="http://www.w3.org/1999/xhtml">
            <tr>
                <td class="tabtab">
                    <table width="100%" border="0" cellspacing="3" cellpadding="2">
                        <tr class="headinglabel">
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'tabProjectInfoAuthorDate'"/>
                                </xsl:call-template>
                            </th>
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'tabProjectInfoAuthorBy'"/>
                                </xsl:call-template>
                            </th>
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'tabProjectInfoAuthorDescription'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <xsl:for-each select="//project/version | //project/release">
                            <xsl:sort select="@date" order="descending"/>
                            <xsl:choose>
                                <xsl:when test="name() = 'release'">
                                    <tr>
                                        <xsl:if test="position() = 1">
                                            <xsl:attribute name="id">_current</xsl:attribute>
                                        </xsl:if>
                                        <th style="vertical-align: top;">
                                            <xsl:variable name="theAnchor" select="concat('_', replace(@date, '[^\d]', ''))"/>
                                            <xsl:attribute name="id" select="$theAnchor"/>
                                            <a href="#{$theAnchor}">
                                                <xsl:attribute name="title">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'direct-link-to-self'"/>
                                                    </xsl:call-template>
                                                </xsl:attribute>
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="@date"/>
                                                </xsl:call-template>
                                            </a>
                                        </th>
                                        <th style="vertical-align: top;">
                                            <xsl:value-of select="@by"/>
                                        </th>
                                        <td style="vertical-align: top;">
                                            <b>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'tabProjectInfoReleaseLabel'"/>
                                                </xsl:call-template>
                                                <xsl:if test="@versionLabel">
                                                    <xsl:text>: </xsl:text>
                                                    <xsl:value-of select="@versionLabel"/>
                                                </xsl:if>
                                            </b>
                                            <br/>
                                            <xsl:call-template name="doDescription">
                                                <xsl:with-param name="ns" select="desc | note"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                </xsl:when>
                                <xsl:otherwise>
                                    <tr>
                                        <td style="vertical-align: top;">
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="@date"/>
                                            </xsl:call-template>
                                        </td>
                                        <td style="vertical-align: top;">
                                            <xsl:value-of select="@by"/>
                                        </td>
                                        <td style="vertical-align: top;">
                                            <xsl:call-template name="doDescription">
                                                <xsl:with-param name="ns" select="desc"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:for-each>
                    </table>
                </td>
            </tr>
        </table>
    </xsl:template>
    <xsl:template name="doPrivateRepositoryNotice">
        <xsl:variable name="projectIsRepository" select="$allDECOR[string(@repository) = 'true']"/>
        <xsl:variable name="projectIsPrivate" select="$allDECOR[string(@private) = 'true']"/>
        <xsl:choose>
            <xsl:when test="$projectIsRepository and $projectIsPrivate">
                <xsl:call-template name="showIcon">
                    <xsl:with-param name="which">notice</xsl:with-param>
                </xsl:call-template>
                <xsl:call-template name="getMessage">
                    <xsl:with-param name="key" select="'noticeIsPrivateRepository'"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="$projectIsRepository">
                <xsl:call-template name="showIcon">
                    <xsl:with-param name="which">notice</xsl:with-param>
                </xsl:call-template>
                <xsl:call-template name="getMessage">
                    <xsl:with-param name="key" select="'noticeIsRepository'"/>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="$projectIsPrivate">
                <xsl:call-template name="showIcon">
                    <xsl:with-param name="which">notice</xsl:with-param>
                </xsl:call-template>
                <xsl:call-template name="getMessage">
                    <xsl:with-param name="key" select="'noticeIsPrivate'"/>
                </xsl:call-template>
            </xsl:when>
        </xsl:choose>
    </xsl:template>
    <xsl:template name="doDataset">
        <xsl:param name="nestingWithTables" select="true()"/>
        <xsl:param name="dsid" as="xs:string?"/>
        <xsl:param name="dsed" as="xs:string?"/>
        <xsl:param name="trid" as="xs:string?"/>
        <xsl:param name="tred" as="xs:string?"/>
        <xsl:param name="id" select="$dsid, $trid" as="xs:string?"/>
        <xsl:param name="effectiveDate" select="$dsed, $tred" as="xs:string?"/>
        <xsl:param name="versionLabel" select="@versionLabel" as="xs:string?"/>
        <xsl:param name="expirationDate" select="@expirationDate" as="xs:string?"/>
        <xsl:param name="statusCode" select="@statusCode" as="xs:string?"/>
        <!-- show data set meta data -->
        <table width="100%" border="0" cellspacing="3" cellpadding="2" xmlns="http://www.w3.org/1999/xhtml">
            <tr class="headinglabel">
                <th style="text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Name'"/>
                    </xsl:call-template>
                </th>
                <th style="text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Id'"/>
                    </xsl:call-template>
                </th>
            </tr>
            <tr>
                <td style="text-align: left;">
                    <xsl:call-template name="showStatusDot">
                        <xsl:with-param name="status" select="$statusCode"/>
                        <xsl:with-param name="title">
                            <xsl:call-template name="getXFormsLabel">
                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                <xsl:with-param name="simpleTypeValue" select="$statusCode"/>
                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                            </xsl:call-template>
                        </xsl:with-param>
                    </xsl:call-template>
                    <xsl:text>&#160;</xsl:text>
                    <xsl:call-template name="doName">
                        <xsl:with-param name="ns" select="name"/>
                    </xsl:call-template>
                    <xsl:if test="$versionLabel">
                        <xsl:text> (</xsl:text>
                        <xsl:value-of select="$versionLabel"/>
                        <xsl:text>)</xsl:text>
                    </xsl:if>
                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                </td>
                <td style="text-align: left; vertical-align: top;">
                    <xsl:value-of select="$id"/>
                </td>
            </tr>
            <tr class="headinglabel">
                <th style="text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'EffectiveDate'"/>
                    </xsl:call-template>
                </th>
                <th style="text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'ExpirationDate'"/>
                    </xsl:call-template>
                </th>
            </tr>
            <tr>
                <td style="text-align: left;">
                    <xsl:call-template name="showDate">
                        <xsl:with-param name="date" select="$effectiveDate"/>
                    </xsl:call-template>
                </td>
                <td style="text-align: left; vertical-align: top;">
                    <xsl:call-template name="showDate">
                        <xsl:with-param name="date" select="$expirationDate"/>
                    </xsl:call-template>
                </td>
            </tr>
            <tr class="headinglabel">
                <th style="text-align: left; vertical-align: top;" colspan="2">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Description'"/>
                    </xsl:call-template>
                </th>
            </tr>
            <tr>
                <td style="text-align: left;" colspan="2">
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="desc"/>
                    </xsl:call-template>
                </td>
            </tr>
            <!-- <relationship type="SPEC" ref="2.16.840.1.113883.2.4.3.11.60.40.1.10.1.1" flexibility="2015-04-01T00:00:00"/> -->
            <xsl:if test="property">
                <xsl:variable name="theProperties" as="element()">
                    <table class="zebra-table" width="100%">
                        <xsl:for-each select="property">
                            <xsl:variable name="ref" select="@ref"/>
                            <xsl:variable name="flex" select="@flexibility"/>
                            <tr>
                                <td style="text-align: left; background-color: #F6F3EE;">
                                    <xsl:value-of select="@name"/>
                                </td>
                                <td>
                                    <xsl:copy-of select="node()"/>
                                </td>
                            </tr>
                        </xsl:for-each>
                    </table>
                </xsl:variable>
                <tr>
                    <td colspan="2">
                        <table class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr class="desclabel" data-tt-id="property-{generate-id(.)}">
                                <td class="togglertreetable" style="height: 1.5em;">
                                    <span class="expcaption">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'Properties'"/>
                                        </xsl:call-template>
                                    </span>
                                </td>
                            </tr>
                            <tr data-tt-id="propertyBody-{generate-id(.)}" data-tt-parent-id="property-{generate-id(.)}">
                                <td>
                                    <xsl:copy-of select="$theProperties"/>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="relationship">
                <tr>
                    <th style="text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Relationship'"/>
                        </xsl:call-template>
                    </th>
                    <td>
                        <table class="zebra-table" width="100%">
                            <xsl:for-each select="relationship">
                                <xsl:variable name="ref" select="@ref"/>
                                <xsl:variable name="flex" select="@flexibility"/>
                                <tr>
                                    <td>
                                        <xsl:call-template name="getXFormsLabel">
                                            <xsl:with-param name="simpleTypeKey" select="'RelationshipTypes'"/>
                                            <xsl:with-param name="simpleTypeValue" select="@type"/>
                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                        </xsl:call-template>
                                    </td>
                                    <td>
                                        <xsl:variable name="dataset" select="
                                                if ($ref) then
                                                    local:getDataset($ref, $flex)
                                                else
                                                    ()" as="element(dataset)*"/>
                                        <xsl:variable name="dsed">
                                            <xsl:choose>
                                                <xsl:when test="$flex castable as xs:dateTime">
                                                    <xsl:value-of select="$flex"/>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:value-of select="$dataset[1]/@effectiveDate"/>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:variable>
                                        <xsl:variable name="dstext" as="xs:string*">
                                            <xsl:choose>
                                                <xsl:when test="$dataset[@iddisplay]">
                                                    <xsl:value-of select="($dataset/@iddisplay)[1]"/>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="doShorthandId">
                                                        <xsl:with-param name="id" select="$ref"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                            <xsl:text> - </xsl:text>
                                            <xsl:choose>
                                                <xsl:when test="matches(@flexibility, '^\d{4}')">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="$flex"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:variable>
                                        <xsl:choose>
                                            <xsl:when test="$dataset">
                                                <a href="{local:doHtmlName('DS',$dataset[1]/@id,$dsed,'.html')}" onclick="target='_blank';">
                                                    <xsl:call-template name="doName">
                                                        <xsl:with-param name="ns" select="$dataset[1]/name"/>
                                                    </xsl:call-template>
                                                </a>
                                                <i>
                                                    <xsl:text> (</xsl:text>
                                                    <xsl:value-of select="string-join($dstext, '')"/>
                                                    <xsl:text>)</xsl:text>
                                                </i>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="string-join($dstext, '')"/>
                                                <xsl:call-template name="logMessage">
                                                    <xsl:with-param name="level" select="$logDEBUG"/>
                                                    <xsl:with-param name="msg">
                                                        <xsl:text>+++ Could not find referenced dataset id='</xsl:text>
                                                        <xsl:value-of select="$ref"/>
                                                        <xsl:text>' referenced from relationship in dataset '</xsl:text>
                                                        <xsl:call-template name="doDescription">
                                                            <xsl:with-param name="ns" select="../name"/>
                                                        </xsl:call-template>
                                                        <xsl:text>' (</xsl:text>
                                                        <xsl:value-of select="../@id"/>
                                                        <xsl:text> - </xsl:text>
                                                        <xsl:value-of select="../@effectiveDate"/>
                                                        <xsl:text>). This might be due to it being in a different project/BBR or because the ref/effectiveDate is broken.</xsl:text>
                                                    </xsl:with-param>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </td>
                                </tr>
                            </xsl:for-each>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="source">
                <tr class="headinglabel">
                    <th style="text-align: left;" colspan="2">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Source'"/>
                        </xsl:call-template>
                    </th>
                </tr>
                <tr>
                    <td style="text-align: left;" colspan="2">
                        <xsl:value-of select="source"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="rationale">
                <tr class="headinglabel">
                    <th style="text-align: left;" colspan="2">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Rationale'"/>
                        </xsl:call-template>
                    </th>
                </tr>
                <tr>
                    <td style="text-align: left;" colspan="2">
                        <xsl:value-of select="rationale"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="operationalization">
                <tr class="headinglabel">
                    <th style="text-align: left;" colspan="2">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Operationalizations'"/>
                        </xsl:call-template>
                    </th>
                </tr>
                <tr>
                    <td style="text-align: left;" colspan="2">
                        <xsl:value-of select="operationalization"/>
                    </td>
                </tr>
            </xsl:if>
            <!--
            <tr class="headinglabel">
                <th style="text-align: left;" colspan="2">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'contentWithColon'"/>
                    </xsl:call-template>
                </th>
            </tr>
            -->
        </table>
        <xsl:if test="$switchCreateTreeTableHtml = 'true'">
            <div xmlns="http://www.w3.org/1999/xhtml">
                <button id="expandAll" type="button">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'buttonExpandAll'"/>
                    </xsl:call-template>
                </button>
                <button id="collapseAll" type="button">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                    </xsl:call-template>
                </button>
                <input id="nameSearch">
                    <xsl:attribute name="placeholder">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'textSearch'"/>
                        </xsl:call-template>
                    </xsl:attribute>
                </input>
                <select id="hiddenColumns">
                    <option value="title">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'showColumn'"/>
                        </xsl:call-template>
                    </option>
                    <option value="columnDetails" disabled="disabled">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Details'"/>
                        </xsl:call-template>
                    </option>
                </select>
            </div>
        </xsl:if>
        <table id="transactionTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2" xmlns="http://www.w3.org/1999/xhtml">
            <tr>
                <th class="columnName" style="text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Name'"/>
                    </xsl:call-template>
                </th>
                <th class="columnDetails" style="text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Details'"/>
                    </xsl:call-template>
                    <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                        <xsl:text>&#160;</xsl:text>
                        <span class="hideMe" type="button">[‑]</span>
                    </xsl:if>
                </th>
                <th class="columnNoPrint"/>
            </tr>
            <xsl:apply-templates select="concept" mode="dataset">
                <xsl:with-param name="level" select="1"/>
                <xsl:with-param name="dsid" select="$dsid"/>
                <xsl:with-param name="dsed" select="$dsed"/>
                <xsl:with-param name="trid" select="$trid"/>
                <xsl:with-param name="tred" select="$tred"/>
            </xsl:apply-templates>
        </table>
    </xsl:template>
    <xsl:template name="doScenarios">
        <xsl:variable name="node-id-sc" select="generate-id(.)"/>
        <!-- show data set meta data -->
        <tr data-tt-id="{$node-id-sc}" xmlns="http://www.w3.org/1999/xhtml">
            <td class="togglertreetable columnName" colspan="3">
                <xsl:call-template name="showStatusDot">
                    <xsl:with-param name="status" select="@statusCode"/>
                    <xsl:with-param name="title">
                        <xsl:call-template name="getXFormsLabel">
                            <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                            <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                        </xsl:call-template>
                    </xsl:with-param>
                </xsl:call-template>
                <xsl:text>&#160;</xsl:text>
                <xsl:call-template name="doName">
                    <xsl:with-param name="ns" select="name"/>
                </xsl:call-template>
                <div style="float: right;">
                    <xsl:call-template name="doShorthandId">
                        <xsl:with-param name="id" select="@id"/>
                    </xsl:call-template>
                </div>
            </td>
        </tr>
        <!-- ==== Generics for scenario ==== -->
        <!-- id / versionLabel / effectiveDate / expirationDate -->
        <tr data-tt-id="{$node-id-sc}-meta" data-tt-parent-id="{$node-id-sc}" xmlns="http://www.w3.org/1999/xhtml">
            <td class="columnDetails"/>
            <td class="columnDetails" colspan="2">
                <table border="0" cellspacing="0" cellpadding="0">
                    <tr>
                        <td class="strong" style="padding-left: 0;">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'EffectiveDate'"/>
                            </xsl:call-template>
                        </td>
                        <td>
                            <xsl:call-template name="showDate">
                                <xsl:with-param name="date" select="@effectiveDate"/>
                            </xsl:call-template>
                            <xsl:if test="@versionLabel">
                                <xsl:text> - </xsl:text>
                                <xsl:value-of select="@versionLabel"/>
                            </xsl:if>
                        </td>
                        <xsl:if test="@expirationDate">
                            <td class="strong">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'ExpirationDate'"/>
                                </xsl:call-template>
                            </td>
                            <td>
                                <xsl:call-template name="showDate">
                                    <xsl:with-param name="date" select="@expirationDate"/>
                                </xsl:call-template>
                            </td>
                        </xsl:if>
                    </tr>
                </table>
            </td>
        </tr>
        <xsl:if test="desc[.//text()[string-length(normalize-space()) gt 0]]">
            <tr data-tt-id="{$node-id-sc}-desc" data-tt-parent-id="{$node-id-sc}" xmlns="http://www.w3.org/1999/xhtml">
                <td class="columnDetails"/>
                <td class="columnDetails strong" style="vertical-align: top; width: 5%;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Description'"/>
                    </xsl:call-template>
                </td>
                <td class="columnDetails">
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="desc"/>
                    </xsl:call-template>
                </td>
            </tr>
        </xsl:if>
        <xsl:if test="trigger">
            <tr data-tt-id="{$node-id-sc}-trigger" data-tt-parent-id="{$node-id-sc}" xmlns="http://www.w3.org/1999/xhtml">
                <td class="columnDetails"/>
                <td class="columnDetails strong" style="vertical-align: top; width: 5%;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'scenarioTrigger'"/>
                    </xsl:call-template>
                </td>
                <td class="columnDetails">
                    <xsl:if test="trigger[@id]">
                        <div>
                            <span class="strong">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Id'"/>
                                </xsl:call-template>
                            </span>
                            <xsl:text>: </xsl:text>
                            <xsl:value-of select="trigger/@id[1]"/>
                        </div>
                    </xsl:if>
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="trigger"/>
                    </xsl:call-template>
                </td>
            </tr>
        </xsl:if>
        <xsl:if test="condition">
            <tr data-tt-id="{$node-id-sc}-condition" data-tt-parent-id="{$node-id-sc}" xmlns="http://www.w3.org/1999/xhtml">
                <td class="columnDetails"/>
                <td class="columnDetails strong" style="vertical-align: top; width: 5%;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'scenarioCondition'"/>
                    </xsl:call-template>
                </td>
                <td class="columnDetails">
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="condition"/>
                    </xsl:call-template>
                </td>
            </tr>
        </xsl:if>
        <xsl:apply-templates select="transaction"/>
    </xsl:template>
    <xsl:template name="doIdentifiers">
        <h2 xmlns="http://www.w3.org/1999/xhtml">
            <a name="labelIdentifiers"/>
            <xsl:call-template name="getMessage">
                <xsl:with-param name="key" select="'tabIdentifiersTitleString'"/>
            </xsl:call-template>
        </h2>
        <p xmlns="http://www.w3.org/1999/xhtml">
            <xsl:call-template name="getMessage">
                <xsl:with-param name="key" select="'tabIdentifiersRenderingRemark'"/>
            </xsl:call-template>
        </p>
        <div id="Accordionid" class="Accordion" xmlns="http://www.w3.org/1999/xhtml">
            <xsl:for-each select="$allIDs/id">
                <xsl:sort select="@root"/>
                <xsl:variable name="theRoot" select="@root"/>
                <div class="AccordionPanel">
                    <div class="AccordionPanelTab">
                        <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                            <h3>
                                <xsl:choose>
                                    <xsl:when test="count(designation[@language = $defaultLanguage]) &gt; 0">
                                        <i>
                                            <xsl:value-of select="designation[@language = $defaultLanguage]/@displayName"/>
                                        </i>
                                    </xsl:when>
                                    <!-- Do we have en-US at all? -->
                                    <xsl:when test="count(designation[@language = 'en-US']) &gt; 0">
                                        <i>
                                            <xsl:value-of select="designation[@language = 'en-US']/@displayName"/>
                                        </i>
                                    </xsl:when>
                                    <xsl:when test="count(designation[@language = 'en-EN']) &gt; 0">
                                        <i>
                                            <xsl:value-of select="designation[@language = 'en-EN']/@displayName"/>
                                        </i>
                                    </xsl:when>
                                </xsl:choose>
                                <xsl:text>&#160;</xsl:text>
                                <xsl:value-of select="$theRoot"/>
                            </h3>
                        </div>
                    </div>
                    <div class="AccordionPanelContent">
                        <table border="0" cellspacing="10">
                            <tr>
                                <td class="tabtab">
                                    <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                        <tr class="headinglabel" style="vertical-align: top;">
                                            <th style="text-align: left;">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'idLanguage'"/>
                                                </xsl:call-template>
                                            </th>
                                            <th style="text-align: left;">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'DisplayName'"/>
                                                </xsl:call-template>
                                            </th>
                                            <th style="vertical-align: top; text-align: left;">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'Description'"/>
                                                </xsl:call-template>
                                            </th>
                                            <xsl:if test="designation[@type | @preferredForLanguage]">
                                                <th style="text-align: left;">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'Type'"/>
                                                    </xsl:call-template>
                                                </th>
                                            </xsl:if>
                                        </tr>
                                        <xsl:for-each select="designation">
                                            <tr>
                                                <td>
                                                    <xsl:value-of select="@language"/>
                                                </td>
                                                <td>
                                                    <xsl:value-of select="@displayName"/>
                                                </td>
                                                <td>
                                                    <xsl:value-of select="text()"/>
                                                </td>
                                                <xsl:if test="../designation[@type | @preferredForLanguage]">
                                                    <td>
                                                        <xsl:choose>
                                                            <xsl:when test="@preferredForLanguage = 'true' or @type = 'preferred'">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'idPreferred'"/>
                                                                </xsl:call-template>
                                                            </xsl:when>
                                                            <xsl:when test="@type = 'synonym'">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'idSynonym'"/>
                                                                </xsl:call-template>
                                                            </xsl:when>
                                                            <xsl:when test="@type = 'abbreviation'">
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'idAbbreviation'"/>
                                                                </xsl:call-template>
                                                            </xsl:when>
                                                            <xsl:when test="not(@type)"/>
                                                            <xsl:otherwise>
                                                                <xsl:call-template name="logMessage">
                                                                    <xsl:with-param name="level" select="$logFATAL"/>
                                                                    <xsl:with-param name="msg">
                                                                        <xsl:text>+++ Internal XSL error: unsupported id[@root="</xsl:text>
                                                                        <xsl:value-of select="$theRoot"/>
                                                                        <xsl:text>"]/designation/@type="</xsl:text>
                                                                        <xsl:value-of select="@type"/>
                                                                        <xsl:text>"</xsl:text>
                                                                    </xsl:with-param>
                                                                </xsl:call-template>
                                                            </xsl:otherwise>
                                                        </xsl:choose>
                                                    </td>
                                                </xsl:if>
                                            </tr>
                                        </xsl:for-each>
                                    </table>
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
            </xsl:for-each>
        </div>
        <h2 xmlns="http://www.w3.org/1999/xhtml">
            <xsl:call-template name="getMessage">
                <xsl:with-param name="key" select="'listOfTemplateIdentifiersInThisProject'"/>
            </xsl:call-template>
        </h2>
        <table border="0" cellspacing="10" style="min-width: 80%" xmlns="http://www.w3.org/1999/xhtml">
            <tr>
                <td class="tabtab">
                    <table width="100%" border="0" cellspacing="3" cellpadding="2">
                        <tr class="headinglabel" style="vertical-align: top;">
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Id'"/>
                                </xsl:call-template>
                            </th>
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'DisplayName'"/>
                                </xsl:call-template>
                            </th>
                            <!--<th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Name'"/>
                                </xsl:call-template>
                            </th>-->
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'EffectiveDate'"/>
                                </xsl:call-template>
                            </th>
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Type'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <xsl:for-each-group select="$allTemplates/*/ref/template" group-by="(@id | @ref)">
                            <xsl:variable name="tmid" select="current-grouping-key()"/>
                            <xsl:variable name="tmnewest" select="(current-group()[@newestForId = 'true'], current-group())[1]"/>
                            <tr style="vertical-align: top;">
                                <td style="vertical-align: top;">
                                    <xsl:call-template name="doShorthandId">
                                        <xsl:with-param name="id" select="$tmid"/>
                                    </xsl:call-template>
                                </td>
                                <td style="vertical-align: top;">
                                    <xsl:value-of select="$tmnewest/(@displayName, @name)[1]"/>
                                </td>
                                <!--<td style="vertical-align: top;">
                                    <xsl:value-of select="@name"/>
                                </td>-->
                                <td style="vertical-align: top;">
                                    <xsl:for-each select="current-group()/@effectiveDate">
                                        <xsl:sort select="." order="descending"/>
                                        <div>
                                            <a href="{local:doHtmlName('TM', $tmid, ., '.html')}" onclick="target='_blank';">
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="."/>
                                                </xsl:call-template>
                                            </a>
                                        </div>
                                    </xsl:for-each>
                                </td>
                                <td style="vertical-align: top;">
                                    <xsl:value-of select="$tmnewest/classification/@type"/>
                                </td>
                            </tr>
                            <xsl:variable name="errmsg">
                                <xsl:for-each select="current-group()">
                                    <xsl:variable name="tmnm" select="@name"/>
                                    <xsl:variable name="tmed" select="@effectiveDate"/>
                                    
                                    <xsl:if test="count($allTemplates/*/ref/template[@id = $tmid][@effectiveDate = $tmed]) &gt; 1">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'possibleConflictingDuplicateIds'"/>
                                        </xsl:call-template>
                                    </xsl:if>
                                    <xsl:if test="count($allTemplates/*/ref/template[@name = $tmnm][not((@id | @ref) = $tmid)]) &gt; 1">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'possibleConflictingDuplicateNames'"/>
                                        </xsl:call-template>
                                    </xsl:if>
                                </xsl:for-each>
                            </xsl:variable>
                            <xsl:if test="not(empty($errmsg))">
                                <tr>
                                    <td> </td>
                                    <td colspan="3">
                                        <table style="border: 0;">
                                            <xsl:call-template name="doMessage">
                                                <xsl:with-param name="level" select="'warning'"/>
                                                <xsl:with-param name="msg" select="$errmsg"/>
                                            </xsl:call-template>
                                        </table>
                                    </td>
                                </tr>
                            </xsl:if>
                        </xsl:for-each-group>
                    </table>
                </td>
            </tr>
        </table>
        <h2 xmlns="http://www.w3.org/1999/xhtml">
            <xsl:call-template name="getMessage">
                <xsl:with-param name="key" select="'listOfValueSetIdentifiersInThisProject'"/>
            </xsl:call-template>
        </h2>
        <table border="0" cellspacing="10" style="min-width: 80%" xmlns="http://www.w3.org/1999/xhtml">
            <tr>
                <td class="tabtab">
                    <table width="100%" border="0" cellspacing="3" cellpadding="2">
                        <tr class="headinglabel" style="vertical-align: top;">
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Id'"/>
                                </xsl:call-template>
                            </th>
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'DisplayName'"/>
                                </xsl:call-template>
                            </th>
                            <!--<th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Name'"/>
                                </xsl:call-template>
                            </th>-->
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'EffectiveDate'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <xsl:for-each-group select="$allValueSets/*/valueSet" group-by="(@id | @ref)">
                            <xsl:variable name="vsid" select="current-grouping-key()"/>
                            <xsl:variable name="vsnewest" select="(current-group()[@effectiveDate = max(current-group()/xs:dateTime(@effectiveDate))], current-group())[1]"/>
                            <tr style="vertical-align: top;">
                                <td style="vertical-align: top;">
                                    <xsl:call-template name="doShorthandId">
                                        <xsl:with-param name="id" select="$vsid"/>
                                    </xsl:call-template>
                                </td>
                                <td style="vertical-align: top;">
                                    <xsl:value-of select="$vsnewest/(@displayName, @name)[1]"/>
                                </td>
                                <!--<td style="vertical-align: top;">
                                    <xsl:value-of select="@name"/>
                                </td>-->
                                <td style="vertical-align: top;">
                                    <xsl:for-each select="current-group()/@effectiveDate">
                                        <xsl:sort select="." order="descending"/>
                                        <div>
                                            <a href="{local:doHtmlName('VS', $vsid, ., '.html')}" onclick="target='_blank';">
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="."/>
                                                </xsl:call-template>
                                            </a>
                                        </div>
                                    </xsl:for-each>
                                </td>
                            </tr>
                            <xsl:variable name="errmsg" as="element()*">
                                <xsl:for-each select="current-group()">
                                    <xsl:variable name="vsnm" select="@name"/>
                                    <xsl:variable name="vsed" select="@effectiveDate"/>
                                    
                                    <xsl:if test="count($allValueSets/*/ref/valueSet[@id = $vsid][@effectiveDate = $vsed]) &gt; 1">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'possibleConflictingDuplicateIds'"/>
                                        </xsl:call-template>
                                    </xsl:if>
                                    <xsl:if test="count($allValueSets/*/ref/valueSet[@name = $vsnm][not((@id | @ref) = $vsid)]) &gt; 1">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'possibleConflictingDuplicateNames'"/>
                                        </xsl:call-template>
                                    </xsl:if>
                                </xsl:for-each>
                            </xsl:variable>
                            <xsl:if test="not(empty($errmsg))">
                                <tr>
                                    <td> </td>
                                    <td colspan="2">
                                        <table style="border: 0;">
                                            <xsl:call-template name="doMessage">
                                                <xsl:with-param name="level" select="'warning'"/>
                                                <xsl:with-param name="msg" select="$errmsg"/>
                                            </xsl:call-template>
                                        </table>
                                    </td>
                                </tr>
                            </xsl:if>
                        </xsl:for-each-group>
                    </table>
                </td>
            </tr>
        </table>
    </xsl:template>
    <xsl:template match="template" mode="showpreliminaries">
        <xsl:param name="underlyingModel"/>
        <xsl:param name="resultingRule"/>
        <xsl:param name="direction"/>
        <xsl:param name="onespacebetweenheaderparticlesonly" select="false()"/>
        <xsl:text>

</xsl:text>
        <!-- 
            create the template name description
            an @id and a @name is mandatory per template
            give human readable @displayName preference (if present) over pure @name
            if @name and @displayName are identical, print only once
        -->
        <xsl:variable name="templatename">
            <xsl:choose>
                <xsl:when test="string-length(@displayName) &gt; 0">
                    <xsl:value-of select="@displayName"/>
                </xsl:when>
                <xsl:when test="string-length(@name) &gt; 0">
                    <xsl:value-of select="@name"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'rulesNoNameOrDisplaynameDefinedYet'"/>
                    </xsl:call-template>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <!--div class="AccordionPanel">
            <div class="AccordionPanelTab"-->
        <div class="acco" onclick="window.open('{local:doHtmlName('TM',@id,@effectiveDate,'.html')}')" xmlns="http://www.w3.org/1999/xhtml">
            <h3>
                <xsl:call-template name="showStatusDot">
                    <xsl:with-param name="status" select="@statusCode"/>
                    <xsl:with-param name="title">
                        <xsl:choose>
                            <xsl:when test="@ident[not(. = $projectPrefix)]">
                                <xsl:value-of select="'ref'"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="getXFormsLabel">
                                    <xsl:with-param name="simpleTypeKey" select="'TemplateStatusCodeLifeCycle'"/>
                                    <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:with-param>
                </xsl:call-template>
                <xsl:text>&#160;</xsl:text>
                <xsl:copy-of select="$templatename"/>
                <xsl:if test="$onespacebetweenheaderparticlesonly = false()">
                    <!-- correction for printing re/ spaces between headline particles -->
                    <xsl:text>&#160;</xsl:text>
                </xsl:if>
                <xsl:text>&#160;</xsl:text>
                <xsl:value-of select="@id"/>
                <xsl:if test="$onespacebetweenheaderparticlesonly = false()">
                    <!-- correction for printing re/ spaces between headline particles -->
                    <xsl:text>&#160;</xsl:text>
                </xsl:if>
                <xsl:text>&#160;-&#160;</xsl:text>
                <xsl:call-template name="showDate">
                    <xsl:with-param name="date" select="@effectiveDate"/>
                </xsl:call-template>
                <xsl:if test="@versionLabel">
                    <xsl:text> (</xsl:text>
                    <xsl:value-of select="@versionLabel"/>
                    <xsl:text>)</xsl:text>
                </xsl:if>
                <xsl:apply-templates select="." mode="buildReferenceBox"/>
            </h3>
            <!-- in case this is a scenario transaction rule, show data -->
            <xsl:if test="string-length(concat($underlyingModel, $resultingRule)) &gt; 0">
                <table border="0" cellspacing="2" cellpadding="5" style="background-color: #FFFFFF;">
                    <tr class="headinglable">
                        <th colspan="2">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Direction'"/>
                            </xsl:call-template>
                        </th>
                        <th>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'UnderlyingModel'"/>
                            </xsl:call-template>
                        </th>
                        <th>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'RuleName'"/>
                            </xsl:call-template>
                        </th>
                        <th>Schematron</th>
                    </tr>
                    <tr>
                        <td>
                            <xsl:call-template name="showDirection">
                                <xsl:with-param name="dir" select="$direction"/>
                            </xsl:call-template>
                        </td>
                        <td>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="concat('transactionDirection', $direction)"/>
                            </xsl:call-template>
                        </td>
                        <td>
                            <xsl:value-of select="$underlyingModel"/>
                        </td>
                        <td>
                            <xsl:value-of select="$resultingRule"/>
                        </td>
                        <td>
                            <xsl:value-of select="concat($projectPrefix, $resultingRule, '.sch')"/>
                        </td>
                    </tr>
                </table>
            </xsl:if>
        </div>
        <!--/div>
            <xsl:text>

</xsl:text>
            <div class="AccordionPanelContent">
                <xsl:apply-templates select=".">
                    <xsl:with-param name="templatename" select="$templatename"/>
                </xsl:apply-templates>
            </div>
        </div-->
    </xsl:template>
    <xsl:template match="template">
        <xsl:param name="templatename"/>
        <xsl:param name="label" as="element()?"/>
        <!-- cache id, name, effectiveDate and calculate if we're the newest version of the template -->
        <xsl:variable name="tid" select="@id"/>
        <xsl:variable name="teff" select="@effectiveDate"/>
        <xsl:variable name="tname" select="@name"/>
        <xsl:variable name="tIsNewest" select="max($allTemplates/*/ref/template[@id = $tid]/xs:dateTime(@effectiveDate)) = $teff"/>
        <xsl:variable name="tmformat" select="local:getTemplateFormat(.)"/>
        <xsl:variable name="fWidth" select="'20em'"/>
        <!-- create item label to show -->
        <xsl:variable name="itemlabel" as="element()?">
            <xsl:choose>
                <xsl:when test="string-length(item[1]/@label) &gt; 0">
                    <!-- use current item/@label -->
                    <item xmlns="" label="{replace(string(item[1]/@label), '-', '‑')}">
                        <xsl:copy-of select="item[1]/desc"/>
                    </item>
                </xsl:when>
                <xsl:when test="$label[string-length(@label[1]) &gt; 0]">
                    <!-- use inherited label if present -->
                    <xsl:copy-of select="$label"/>
                </xsl:when>
                <xsl:otherwise>
                    <!-- use leave it empty -->
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <table width="100%" border="0" cellspacing="3" cellpadding="2" xmlns="http://www.w3.org/1999/xhtml">
            <!-- id / effectiveDate -->
            <tr style="vertical-align: top;">
                <!-- id -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Id'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:variable name="ident" select="(@ident, ../@ident)[1]"/>
                    <xsl:variable name="url" select="(@url, ../@url)[1]"/>
                    <xsl:choose>
                        <!-- In live ART-DECOR you can just go there, but in an offline publication, we cannot jump to a different publication location -->
                        <xsl:when test="$url and $ident[not(. = $projectPrefix)] and $seeThisUrlLocation = 'live'">
                            <a href="{local:doHtmlName('TM',$url,$ident,$tid,$teff,(),(),(),(),'.html','false',true())}" onclick="target='_blank';">
                                <xsl:value-of select="$tid"/>
                            </a>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$tid"/>
                        </xsl:otherwise>
                    </xsl:choose>
                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                </td>
                <!-- version information -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'effectiveDate'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:if test="$teff[string-length() &gt; 0]">
                        <!--<xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'templateValidFrom'"/>
                        </xsl:call-template>-->
                        <xsl:call-template name="showDate">
                            <xsl:with-param name="date" select="$teff"/>
                        </xsl:call-template>
                    </xsl:if>
                    <xsl:if test="@expirationDate[string-length() gt 0]">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'templateValidTo'"/>
                        </xsl:call-template>
                        <xsl:call-template name="showDate">
                            <xsl:with-param name="date" select="@expirationDate"/>
                        </xsl:call-template>
                    </xsl:if>
                    <xsl:variable name="otherVersions" select="$allTemplateRefs/*/template[@name = $tname or @id = $tid][@effectiveDate[not(. = $teff)]]" as="element()*"/>
                    <xsl:if test="$otherVersions">
                        <div>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'OtherVersionsWithThisId'"/>
                            </xsl:call-template>
                            <ul>
                                <xsl:for-each-group select="$otherVersions" group-by="concat(@id, @effectiveDate)">
                                    <xsl:sort select="@effectiveDate" order="descending"/>
                                    <xsl:for-each select="current-group()[1]">
                                        <li>
                                            <xsl:call-template name="showStatusDot">
                                                <xsl:with-param name="status" select="@statusCode"/>
                                                <xsl:with-param name="title">
                                                    <xsl:call-template name="getXFormsLabel">
                                                        <xsl:with-param name="simpleTypeKey" select="'TemplateStatusCodeLifeCycle'"/>
                                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                    </xsl:call-template>
                                                </xsl:with-param>
                                            </xsl:call-template>
                                            <xsl:text> </xsl:text>
                                            <a href="{local:doHtmlName('TM', @id, @effectiveDate, '.html')}" onclick="target='_blank';">
                                                <xsl:value-of select="(@displayName, @name)[1]"/>
                                            </a>
                                            <xsl:if test="@versionLabel">
                                                <xsl:value-of select="@versionLabel"/>
                                                <xsl:text> - </xsl:text>
                                            </xsl:if>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'xAsOfy'"/>
                                            </xsl:call-template>
                                            <xsl:text> </xsl:text>
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="@effectiveDate"/>
                                            </xsl:call-template>
                                            <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                        </li>
                                    </xsl:for-each>
                                </xsl:for-each-group>
                            </ul>
                        </div>
                    </xsl:if>
                </td>
            </tr>
            <!-- status / versionLabel -->
            <tr style="vertical-align: top;">
                <!-- status -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Status'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:call-template name="showStatusDot">
                        <xsl:with-param name="status" select="@statusCode"/>
                        <xsl:with-param name="title">
                            <xsl:call-template name="getXFormsLabel">
                                <xsl:with-param name="simpleTypeKey" select="'TemplateStatusCodeLifeCycle'"/>
                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                            </xsl:call-template>
                        </xsl:with-param>
                    </xsl:call-template>
                    <xsl:text> </xsl:text>
                    <xsl:call-template name="getXFormsLabel">
                        <xsl:with-param name="simpleTypeKey" select="'TemplateStatusCodeLifeCycle'"/>
                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                    </xsl:call-template>
                </td>
                <!-- versionLabel -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'columnVersionLabel'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:value-of select="@versionLabel"/>
                </td>
            </tr>
            <!-- name / displayName -->
            <tr style="vertical-align: top;">
                <!-- name -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Name'"/>
                    </xsl:call-template>
                    <!--<xsl:if test="count(context) = 0">
                        <!-\- no context means: internal template -\->
                        <xsl:text> (</xsl:text>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'internal'"/>
                        </xsl:call-template>
                        <xsl:text>)</xsl:text>
                    </xsl:if>-->
                </th>
                <td style="text-align: left;">
                    <xsl:value-of select="@name"/>
                </td>
                <!-- displayName -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'DisplayName'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:value-of select="@displayName"/>
                </td>
            </tr>
            <!-- description if given -->
            <xsl:if test="desc[not(. = '')]">
                <tr style="vertical-align: top;">
                    <xsl:choose>
                        <xsl:when test="desc[string-length(string-join(descendant-or-self::text(), ' ')) > 200]">
                            <td style="text-align: left;" colspan="4">
                                <xsl:variable name="elid" select="local:randomString2(., ())"/>
                                <!--<xsl:variable name="elidtoggler" select="concat('toggler-', $elid)"/>-->
                                <table id="templateDescTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                                    <tr class="desclabel" data-tt-id="templateDescHeader">
                                        <td class="togglertreetable" style="height: 1.5em;">
                                            <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                                <xsl:attribute name="class" select="'toggler'"/>
                                                <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                                            </xsl:if>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'templateDescription'"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                    <tr data-tt-id="templateDescBody" data-tt-parent-id="templateDescHeader">
                                        <td>
                                            <div id="{$elid}">
                                                <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                                    <xsl:attribute name="class" select="'toggle'"/>
                                                </xsl:if>
                                                <xsl:call-template name="doDescription">
                                                    <xsl:with-param name="ns" select="desc"/>
                                                </xsl:call-template>
                                            </div>
                                        </td>
                                    </tr>
                                </table>
                            </td>
                        </xsl:when>
                        <xsl:otherwise>
                            <th style="width: {$fWidth}; text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'templateDescription'"/>
                                </xsl:call-template>
                            </th>
                            <td style="text-align: left;" colspan="3">
                                <xsl:call-template name="doDescription">
                                    <xsl:with-param name="ns" select="desc"/>
                                </xsl:call-template>
                            </td>
                        </xsl:otherwise>
                    </xsl:choose>
                </tr>
            </xsl:if>
            <!-- context of this template -->
            <xsl:if test="context">
                <xsl:variable name="cid">
                    <xsl:choose>
                        <xsl:when test="context/@id = '*'">
                            <!-- use siblings of template's id -->
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'tmpContextSibling'"/>
                                <xsl:with-param name="p1" select="@id"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:when test="context/@id = '**'">
                            <!-- use parent of current template's id -->
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'tmpContextParent'"/>
                                <xsl:with-param name="p1" select="@id"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:when test="context/@path">
                            <xsl:variable name="cctxpth">
                                <xsl:call-template name="outputPath">
                                    <xsl:with-param name="pathname">
                                        <xsl:value-of select="context/@path"/>
                                    </xsl:with-param>
                                </xsl:call-template>
                            </xsl:variable>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'tmpContextPath'"/>
                                <xsl:with-param name="p1" select="$cctxpth"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="context/@id"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'context'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:value-of select="$cid"/>
                    </td>
                </tr>
            </xsl:if>
            <!-- item labels -->
            <xsl:if test="item/@label">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'itemReference'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:value-of select="replace(string(item/@label), '-', '‑')"/>
                        <xsl:if test="count(desc) &gt; 0">
                            <br/>
                        </xsl:if>
                        <xsl:call-template name="doDescription">
                            <xsl:with-param name="ns" select="item/desc"/>
                        </xsl:call-template>
                    </td>
                </tr>
            </xsl:if>
            <!-- classification -->
            <xsl:if test="classification">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Classification'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:for-each select="classification">
                            <xsl:call-template name="getXFormsLabel">
                                <xsl:with-param name="simpleTypeKey" select="'TemplateTypes'"/>
                                <xsl:with-param name="simpleTypeValue" select="@type"/>
                            </xsl:call-template>
                            <xsl:if test="position() != last()">
                                <br/>
                            </xsl:if>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:if>
            <!-- @closed -->
            <tr style="vertical-align: top;">
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'openClosedLabel'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;" colspan="3">
                    <xsl:choose>
                        <xsl:when test="string(@isClosed) = 'true'">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'templateIsClosed'"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'templateIsOpen'"/>
                            </xsl:call-template>
                        </xsl:otherwise>
                    </xsl:choose>
                </td>
            </tr>
            <!-- is associated with what concepts?
                
                <templateAssociation templateId="2.999.999.993.77.10.900100" effectiveDate="2012-04-10T00:00:00">
                    <concept ref="2.999.999.993.77.2.1.100060" effectiveDate="2012-04-10T00:00:00" elementId="2.999.999.993.77.9.900100.1"/>
                </templateAssociation>
                
            -->
            <xsl:variable name="associatedConcepts" select="$allTemplateAssociation/*/templateAssociation[@templateId = $tid][@effectiveDate = $teff]/concept | descendant::staticAssociations/origconcept" as="element()*"/>
            <xsl:variable name="renderedConcepts" as="element()*">
                <xsl:for-each-group select="$associatedConcepts" group-by="concat(@ref, @effectiveDate)">
                    <xsl:variable name="deid" select="current-group()[1]/@ref"/>
                    <xsl:variable name="deed" select="current-group()[1]/@effectiveDate"/>
                    <xsl:variable name="concept" select="(current-group()/self::origconcept[name], local:getConceptFlat($deid, $deed))[1]" as="element()?"/>
                    <xsl:variable name="destatus" select="(current-group()/@conceptStatusCode, $concept/@statusCode)[1]" as="item()?"/>
                    <xsl:variable name="deversionlabel" select="(current-group()/@conceptVersionLabel, $concept/@versionLabel)[1]"/>
                    <xsl:variable name="dataset" select="local:getDatasetForConcept($deid, $deed)"/>
                    <xsl:variable name="dsid" select="(current-group()/@datasetId, $dataset/@id)[1]"/>
                    <xsl:variable name="dsef" select="(current-group()/@datasetEffectiveDate, $dataset/@effectiveDate)[1]"/>
                    <xsl:variable name="dspath">
                        <xsl:choose>
                            <xsl:when test="@path">
                                <xsl:value-of select="current-group()[1]/@path"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:for-each select="$dataset//concept[(@id | @ref) = $concept/(@id | @ref)][(@effectiveDate | @flexibility) = $concept/(@effectiveDate | @flexibility)]/ancestor::concept">
                                    <xsl:call-template name="doName">
                                        <!-- will not exist on RetrieveTemplate expanded templates -->
                                        <xsl:with-param name="ns" select="local:getConceptFlat((@id | @ref), (@effectiveDate | @flexibility))/name"/>
                                    </xsl:call-template>
                                    <xsl:text> / </xsl:text>
                                </xsl:for-each>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsstatus" select="(current-group()/@datasetStatusCode, $dataset/@statusCode)[1]" as="item()?"/>
                    <xsl:variable name="dsname">
                        <xsl:choose>
                            <xsl:when test="current-group()[1]/datasetName">
                                <xsl:call-template name="doName">
                                    <!-- will not exist on RetrieveTemplate expanded templates -->
                                    <xsl:with-param name="ns" select="current-group()[1]/datasetName"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="@datasetName">
                                <xsl:value-of select="current-group()[1]/@datasetName"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="doName">
                                    <!-- will not exist on RetrieveTemplate expanded templates -->
                                    <xsl:with-param name="ns" select="$dataset/name"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsversionlabel" select="(current-group()/@datasetVersionLabel, $dataset/@versionLabel)[1]"/>
                    <xsl:variable name="deiddisplay">
                        <xsl:choose>
                            <xsl:when test="@refdisplay[not(. = '')]">
                                <xsl:value-of select="current-group()[1]/@refdisplay"/>
                            </xsl:when>
                            <xsl:when test="$concept[@iddisplay]">
                                <xsl:value-of select="($concept/@iddisplay)[1]"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="doShorthandId">
                                    <xsl:with-param name="id" select="$deid"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:if test="empty($concept)">
                        <xsl:call-template name="logMessage">
                            <xsl:with-param name="level" select="$logDEBUG"/>
                            <xsl:with-param name="msg">
                                <xsl:text>+++ Could not find concept "</xsl:text>
                                <xsl:value-of select="$deid"/>
                                <xsl:text>" effectiveDate "</xsl:text>
                                <xsl:value-of select="$deed"/>
                                <xsl:text>" associated with template id "</xsl:text>
                                <xsl:value-of select="$tid"/>
                                <xsl:text>" effectiveDate "</xsl:text>
                                <xsl:value-of select="$teff"/>
                                <xsl:text>" name "</xsl:text>
                                <xsl:value-of select="$tname"/>
                                <xsl:text> and element </xsl:text>
                                <xsl:value-of select="@elementId | @elementPath"/>
                                <xsl:text>. This may not be meaningful as all templateAssociations anywhere copied during compile phase, but datasets are limited to those pertaining to the project. This might place certain associations out of context.</xsl:text>
                            </xsl:with-param>
                        </xsl:call-template>
                        <!--<tr class="missingAssociation" xmlns="http://www.w3.org/1999/xhtml">
                            <td style="vertical-align: top;" colspan="3">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'theReferencedConceptAsOfCannotBeFound'"/>
                                    <xsl:with-param name="p1" select="$deid"/>
                                    <xsl:with-param name="p2">
                                        <xsl:call-template name="showDate">
                                            <xsl:with-param name="date" select="$deed"/>
                                        </xsl:call-template>
                                    </xsl:with-param>
                                </xsl:call-template>
                            </td>
                        </tr>-->
                    </xsl:if>
                    <xsl:for-each select="$concept">
                        <tr>
                            <td style="text-align: left; vertical-align: top; width: 1%;">
                                <xsl:choose>
                                    <xsl:when test="empty($dsid)">
                                        <xsl:copy-of select="$deiddisplay"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <a href="{local:doHtmlName('DS', $projectPrefix, $dsid, $dsef, $deid, $deed, (), (), '.html', 'false')}" onclick="target='_blank';">
                                            <xsl:copy-of select="$deiddisplay"/>
                                        </a>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </td>
                            <td style="vertical-align: top;">
                                <xsl:if test="string-length($dspath) > 0">
                                    <xsl:attribute name="title" select="$dspath"/>
                                </xsl:if>
                                <xsl:if test="string-length($destatus) > 0">
                                    <xsl:call-template name="showStatusDot">
                                        <xsl:with-param name="status" select="$destatus"/>
                                        <xsl:with-param name="title">
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="$destatus"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                    <xsl:text>&#160;</xsl:text>
                                </xsl:if>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="$concept/name"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <xsl:value-of select="$deversionlabel"/>
                            </td>
                            <td style="text-align: left; vertical-align: top;">
                                <xsl:if test="string-length($dsstatus) > 0">
                                    <xsl:call-template name="showStatusDot">
                                        <xsl:with-param name="status" select="$dsstatus"/>
                                        <xsl:with-param name="title">
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="$dsstatus"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                    <xsl:text>&#160;</xsl:text>
                                </xsl:if>
                                <xsl:copy-of select="$dsname"/>
                                <xsl:text> </xsl:text>
                                <xsl:value-of select="$dsversionlabel"/>
                                <xsl:if test="current-group()[1][not(@ident = $projectPrefix)]">
                                    <xsl:text> </xsl:text>
                                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                </xsl:if>
                            </td>
                        </tr>
                    </xsl:for-each>
                </xsl:for-each-group>
            </xsl:variable>
            <xsl:if test="$renderedConcepts[not(@class = 'missingAssociation')]">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'AssociatedWith'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:variable name="elid" select="local:randomString2(., ())"/>
                        <!--<xsl:variable name="elidtoggler" select="concat('toggler-', $elid)"/>-->
                        <table id="assocsTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr class="desclabel" data-tt-id="assocsHeader">
                                <td class="togglertreetable" style="height: 1.5em;">
                                    <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                        <xsl:attribute name="class" select="'toggler'"/>
                                        <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                                    </xsl:if>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'AssociatedWith'"/>
                                    </xsl:call-template>
                                    <xsl:text> </xsl:text>
                                    <xsl:variable name="assoccnt" select="count($renderedConcepts[not(@class = 'missingAssociation')])"/>
                                    <xsl:choose>
                                        <xsl:when test="$assoccnt = 1">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'numberOfConcept'"/>
                                                <xsl:with-param name="p1" select="$assoccnt"/>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'numberOfConcepts'"/>
                                                <xsl:with-param name="p1" select="$assoccnt"/>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <!-- 2018-03-27 Deactivated code because we will not be detecting those anymore due to changes in compilation. Activate logDEBUG to send messages to the log -->
                                    <!--<xsl:if test="$renderedConcepts[@class = 'missingAssociation']">
                                        <xsl:text>&#160;</xsl:text>
                                        <xsl:call-template name="showIcon">
                                            <xsl:with-param name="which">notice</xsl:with-param>
                                        </xsl:call-template>
                                    </xsl:if>-->
                                </td>
                            </tr>
                            <tr data-tt-id="assocsBody" data-tt-parent-id="assocsHeader">
                                <td>
                                    <div id="{$elid}">
                                        <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                            <xsl:attribute name="class" select="'toggle'"/>
                                        </xsl:if>
                                        <table width="100%" border="0" cellspacing="1" cellpadding="3">
                                            <tr>
                                                <th style="width: 20%; text-align: left;">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'Id'"/>
                                                    </xsl:call-template>
                                                </th>
                                                <th style="width: 60%; text-align: left;">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'Name'"/>
                                                    </xsl:call-template>
                                                </th>
                                                <th style="width: 20%; text-align: left;">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'Dataset'"/>
                                                    </xsl:call-template>
                                                </th>
                                            </tr>
                                            <xsl:copy-of select="$renderedConcepts[not(@class = 'missingAssociation')]"/>
                                        </table>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <!-- used in what other templates? -->
            <xsl:variable name="associatedTemplates" as="element()">
                <wrap xmlns="">
                    <xsl:choose>
                        <xsl:when test="ref | uses">
                            <!-- 
                                or in an already expanded template format use directly ref and uses
                                <ref type="contains" id="1.2.276.0.76.10.12345" name="ClinicalInformation" displayName="Klinische Parameter" effectiveDate="2014-11-01T00:00:00"/>
                                
                                ref types are 'include', 'contains', 'dependency'
                            -->
                            <!-- used by first -->
                            <!-- sometimes we reach a ref immediately (contains/include) and through another template (dependency). In this case, skip the dependency template -->
                            <xsl:variable name="refs" select="ref" as="element()*"/>
                            <xsl:for-each select="$refs">
                                <xsl:variable name="xti" select="@id"/>
                                <xsl:variable name="xte" select="@effectiveDate"/>
                                <xsl:variable name="flex" select="(@flexibility, 'dynamic')[1]"/>
                                <xsl:if test="not(@type = 'dependency') or not($refs[@id = $xti][@effectiveDate = $xte][not(@type = 'dependency')])">
                                    <ref flexibility="{$flex}">
                                        <xsl:copy-of select="@*"/>
                                    </ref>
                                </xsl:if>
                            </xsl:for-each>
                            <!-- now template uses -->
                            <xsl:for-each select="uses">
                                <xsl:variable name="flex" select="
                                        if (@flexibility) then
                                            @flexibility
                                        else
                                            ('dynamic')"/>
                                <uses flexibility="{$flex}">
                                    <xsl:copy-of select="@*"/>
                                </uses>
                            </xsl:for-each>
                        </xsl:when>
                        <xsl:otherwise>
                            <!-- used by not in wiki mode -->
                            <xsl:if test="not($adram = ('mediawiki', 'confluence', 'wordpress'))">
                                <!-- used by first -->
                                <xsl:for-each select="$allTemplates/*/ref[template//include[@ref = $tid or @ref = $tname][($tIsNewest and (not(@flexibility) or string(@flexibility) = 'dynamic')) or @flexibility = $teff]]">
                                    <xsl:variable name="xti" select="template/@id"/>
                                    <xsl:variable name="xtn" select="template/@name"/>
                                    <xsl:variable name="xte" select="template/@effectiveDate"/>
                                    <xsl:variable name="xdn" select="template/@displayName"/>
                                    <xsl:variable name="xin" select="max($allTemplates/*/ref/template[@id = $xti]/xs:dateTime(@effectiveDate)) = $xte"/>
                                    <xsl:variable name="xst" select="template/@statusCode"/>
                                    <xsl:variable name="xvl" select="template/@versionLabel"/>
                                    <ref type="include" newestForId="{$xin}">
                                        <xsl:copy-of select="template/(@* except (@type | @flexibility))"/>
                                    </ref>
                                    <!-- get Template Chain recursively, submit template id and name  -->
                                    <xsl:call-template name="getTemplateChain">
                                        <xsl:with-param name="yti" select="$xti"/>
                                        <xsl:with-param name="ytn" select="$xtn"/>
                                        <xsl:with-param name="yte" select="$xte"/>
                                        <xsl:with-param name="ydn" select="$xdn"/>
                                        <xsl:with-param name="yin" select="$xin"/>
                                        <xsl:with-param name="yst" select="$xst"/>
                                        <xsl:with-param name="yvl" select="$xvl"/>
                                        <xsl:with-param name="sofar" select="concat(':', $xti, ':', $xtn, ':')"/>
                                    </xsl:call-template>
                                </xsl:for-each>
                                <xsl:for-each select="$allTemplates/*/ref[template//*[@contains = $tid or @contains = $tname][($tIsNewest and (not(@flexibility) or string(@flexibility) = 'dynamic')) or @flexibility = $teff]]">
                                    <xsl:variable name="xti" select="template/@id"/>
                                    <xsl:variable name="xtn" select="template/@name"/>
                                    <xsl:variable name="xte" select="template/@effectiveDate"/>
                                    <xsl:variable name="xdn" select="template/@displayName"/>
                                    <xsl:variable name="xin" select="max($allTemplates/*/ref/template[@id = $xti]/xs:dateTime(@effectiveDate)) = $xte"/>
                                    <xsl:variable name="xst" select="template/@statusCode"/>
                                    <xsl:variable name="xvl" select="template/@versionLabel"/>
                                    <ref type="contains" newestForId="{$xin}">
                                        <xsl:copy-of select="template/(@* except (@type | @flexibility))"/>
                                    </ref>
                                    <!-- get Template Chain recursively, submit template id and name  -->
                                    <xsl:call-template name="getTemplateChain">
                                        <xsl:with-param name="yti" select="$xti"/>
                                        <xsl:with-param name="ytn" select="$xtn"/>
                                        <xsl:with-param name="yte" select="$xte"/>
                                        <xsl:with-param name="ydn" select="$xdn"/>
                                        <xsl:with-param name="yin" select="$xin"/>
                                        <xsl:with-param name="yst" select="$xst"/>
                                        <xsl:with-param name="yvl" select="$xvl"/>
                                        <xsl:with-param name="sofar" select="concat(':', $xti, ':', $xtn, ':')"/>
                                    </xsl:call-template>
                                </xsl:for-each>
                            </xsl:if>
                            <!-- now template uses -->
                            <xsl:for-each select=".//element[@contains] | .//include[@ref]">
                                <xsl:variable name="xid" select="concat(@contains, @ref)"/>
                                <xsl:variable name="flex" select="(@flexibility, 'dynamic')[1]"/>
                                <xsl:variable name="type">
                                    <xsl:choose>
                                        <xsl:when test="self::element">contains</xsl:when>
                                        <xsl:otherwise>include</xsl:otherwise>
                                    </xsl:choose>
                                </xsl:variable>
                                <xsl:variable name="effd">
                                    <xsl:choose>
                                        <xsl:when test="$flex castable as xs:dateTime">
                                            <xsl:value-of select="$flex"/>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:value-of select="string(max($allTemplates/*/ref[template[@id = $xid or @name = $xid]]/xs:dateTime(@effectiveDate)))"/>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </xsl:variable>
                                <xsl:variable name="tmpl" select="$allTemplates/*/ref[template[@id = $xid][@effectiveDate = $effd]] | $allTemplates/*/ref[template[@name = $xid][@effectiveDate = $effd]]"/>
                                <xsl:for-each select="$tmpl">
                                    <uses type="{$type}" flexibility="{$flex}">
                                        <xsl:copy-of select="template/(@* except (@type | @flexibility))"/>
                                    </uses>
                                </xsl:for-each>
                            </xsl:for-each>
                        </xsl:otherwise>
                    </xsl:choose>
                </wrap>
            </xsl:variable>
            <xsl:variable name="associatedTransactions" as="element()">
                <wrap xmlns="">
                    <xsl:choose>
                        <xsl:when test="representingTemplate">
                            <xsl:for-each select="representingTemplate">
                                <xsl:variable name="trid" select="@transactionId"/>
                                <xsl:variable name="tred" select="@transactionEffectiveDate"/>
                                <transaction id="{$trid}" effectiveDate="{$tred}">
                                    <xsl:copy-of select="@* except (@id | @effectiveDate)"/>
                                    <xsl:copy-of select="node()"/>
                                </transaction>
                            </xsl:for-each>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:for-each select="$allScenarios//transaction[representingTemplate/@ref]">
                                <xsl:choose>
                                    <xsl:when test="representingTemplate[@flexibility[not(. castable as xs:dateTime)]][$tIsNewest][@ref = ($tid, $tname)]">
                                        <xsl:copy-of select="."/>
                                    </xsl:when>
                                    <xsl:when test="representingTemplate[@flexibility[not(. castable as xs:dateTime)]][$tIsNewest][@ref = $associatedTemplates//ref[@newestForId = 'true']/(@id, @name)]">
                                        <xsl:copy-of select="."/>
                                    </xsl:when>
                                    <xsl:when test="representingTemplate[concat(@ref, @flexibility[. castable as xs:dateTime]) = (concat($tid, $teff), concat($tname, $teff))]">
                                        <xsl:copy-of select="."/>
                                    </xsl:when>
                                    <xsl:when test="representingTemplate[concat(@ref, @flexibility[. castable as xs:dateTime]) = ($associatedTemplates//ref/concat(@id, @ffectiveDate), $associatedTemplates//ref/concat(@name, @ffectiveDate))]">
                                        <xsl:copy-of select="."/>
                                    </xsl:when>
                                </xsl:choose>
                            </xsl:for-each>
                        </xsl:otherwise>
                    </xsl:choose>
                </wrap>
            </xsl:variable>
            <xsl:if test="count($associatedTemplates/*) > 0 or count($associatedTransactions/*) > 0">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:if test="not($adram = ('mediawiki', 'confluence', 'wordpress'))">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'UsedBy'"/>
                            </xsl:call-template>
                            <xsl:text> / </xsl:text>
                        </xsl:if>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Uses'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:variable name="elid" select="local:randomString2(., ())"/>
                        <table class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr class="desclabel" data-tt-id="usedHeader">
                                <td class="togglertreetable" style="height: 1.5em;">
                                    <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                        <xsl:attribute name="class" select="'toggler'"/>
                                        <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                                    </xsl:if>
                                    <xsl:variable name="trxcnt" select="count(distinct-values($associatedTransactions/transaction/concat(@id, @effectiveDate)))"/>
                                    <xsl:variable name="refcnt" select="count(distinct-values($associatedTemplates/ref/concat(@id, @effectiveDate)))"/>
                                    <xsl:variable name="usecnt" select="count(distinct-values($associatedTemplates/uses/concat(@id, @effectiveDate)))"/>
                                    <xsl:if test="not($adram = ('mediawiki', 'confluence', 'wordpress'))">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'UsedBy'"/>
                                        </xsl:call-template>
                                        <xsl:text> </xsl:text>
                                        <xsl:choose>
                                            <xsl:when test="$trxcnt = 1 and $refcnt = 1">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'numberOfTransactionTemplate'"/>
                                                    <xsl:with-param name="p1" select="$trxcnt"/>
                                                    <xsl:with-param name="p2" select="$refcnt"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:when test="$trxcnt != 1 and $refcnt = 1">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'numberOfTransactionsTemplate'"/>
                                                    <xsl:with-param name="p1" select="$trxcnt"/>
                                                    <xsl:with-param name="p2" select="$refcnt"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:when test="$trxcnt = 1 and $refcnt != 1">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'numberOfTransactionTemplates'"/>
                                                    <xsl:with-param name="p1" select="$trxcnt"/>
                                                    <xsl:with-param name="p2" select="$refcnt"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'numberOfTransactionsTemplates'"/>
                                                    <xsl:with-param name="p1" select="$trxcnt"/>
                                                    <xsl:with-param name="p2" select="$refcnt"/>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:text>, </xsl:text>
                                    </xsl:if>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Uses'"/>
                                    </xsl:call-template>
                                    <xsl:text> </xsl:text>
                                    <xsl:choose>
                                        <xsl:when test="$usecnt = 1">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'numberOfTemplate'"/>
                                                <xsl:with-param name="p1" select="$usecnt"/>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'numberOfTemplates'"/>
                                                <xsl:with-param name="p1" select="$usecnt"/>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </td>
                            </tr>
                            <tr data-tt-id="usedBody" data-tt-parent-id="usedHeader">
                                <td>
                                    <div id="{$elid}">
                                        <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                            <xsl:attribute name="class" select="'toggle'"/>
                                        </xsl:if>
                                        <table width="100%" border="0" cellspacing="1" cellpadding="3">
                                            <xsl:for-each select="$associatedTransactions/transaction, $associatedTemplates/ref, $associatedTemplates/uses">
                                                <xsl:variable name="xpfx" select="(@prefix, @ident, $projectPrefix)[1]"/>
                                                <xsl:variable name="xti" select="@id"/>
                                                <xsl:variable name="xtn" select="@name"/>
                                                <xsl:variable name="xty" select="@type"/>
                                                <xsl:variable name="xte" select="@effectiveDate"/>
                                                <xsl:variable name="xdn" select="@displayName"/>
                                                <xsl:variable name="xin" select="@newestForId"/>
                                                <xsl:variable name="xflx" select="@flexibility"/>
                                                <xsl:variable name="xsts" select="@statusCode"/>
                                                <xsl:variable name="xvsl" select="@versionLabel"/>
                                                <xsl:variable name="ename" select="name()"/>
                                                <xsl:if test="not(preceding-sibling::*[name() = $ename]) and not($ename = 'ref' and $associatedTransactions/transaction)">
                                                    <tr>
                                                        <th style="width: 20%; text-align: left;">
                                                            <xsl:choose>
                                                                <xsl:when test="$ename = ('transaction', 'ref')">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'UsedBy'"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'Uses'"/>
                                                                    </xsl:call-template>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </th>
                                                        <th style="width: 1%; text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'as'"/>
                                                            </xsl:call-template>
                                                        </th>
                                                        <th style="width: 60%; text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'Name'"/>
                                                            </xsl:call-template>
                                                        </th>
                                                        <th style="width: 19%; text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'Version'"/>
                                                            </xsl:call-template>
                                                        </th>
                                                    </tr>
                                                </xsl:if>
                                                <xsl:if test="not(preceding-sibling::*[name() = $ename][@id = $xti][@effectiveDate = $xte])">
                                                    <tr>
                                                        <td style="text-align: left;">
                                                            <xsl:choose>
                                                                <xsl:when test="$ename = 'transaction'">
                                                                    <a href="{local:doHtmlName('TR',$xpfx,$xti,$xte,(),(),(),(),'.html','false')}" onclick="target='_blank';">
                                                                        <xsl:call-template name="doShorthandId">
                                                                            <xsl:with-param name="id" select="$xti"/>
                                                                        </xsl:call-template>
                                                                    </a>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <a href="{local:doHtmlName('TM',$xpfx,$xti,$xte,(),(),(),(),'.html','false')}" onclick="target='_blank';">
                                                                        <xsl:value-of select="$xti"/>
                                                                    </a>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:choose>
                                                                <xsl:when test="$ename = 'transaction'">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'Transaction'"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:when test="$xty = 'contains'">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'tmpContains'"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:when test="$xty = 'include'">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'tmpInclude'"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:when test="$xty = 'dependency' or self::ref">
                                                                    <xsl:call-template name="showIcon">
                                                                        <xsl:with-param name="which">link</xsl:with-param>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                            </xsl:choose>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:if test="$xsts">
                                                                <xsl:call-template name="showStatusDot">
                                                                    <xsl:with-param name="status" select="$xsts"/>
                                                                    <xsl:with-param name="title">
                                                                        <xsl:call-template name="getXFormsLabel">
                                                                            <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                            <xsl:with-param name="simpleTypeValue" select="$xsts"/>
                                                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                        </xsl:call-template>
                                                                    </xsl:with-param>
                                                                </xsl:call-template>
                                                                <xsl:text> </xsl:text>
                                                            </xsl:if>
                                                            <xsl:choose>
                                                                <xsl:when test="self::transaction">
                                                                    <xsl:call-template name="doName">
                                                                        <xsl:with-param name="ns" select="name"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:when test="string-length($xdn) &gt; 0">
                                                                    <xsl:value-of select="$xdn"/>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:value-of select="$xtn"/>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                            <xsl:if test="string-length($xvsl)>0">
                                                                <xsl:text> (</xsl:text>
                                                                <xsl:value-of select="@versionLabel"/>
                                                                <xsl:text>)</xsl:text>
                                                            </xsl:if>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:choose>
                                                                <xsl:when test="$ename = 'uses' and matches($xflx, '^\d')">
                                                                    <xsl:call-template name="showDate">
                                                                        <xsl:with-param name="date" select="$xflx"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:when test="$ename = 'uses' and $xflx">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:call-template name="showDate">
                                                                        <xsl:with-param name="date" select="$xte"/>
                                                                    </xsl:call-template>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </td>
                                                    </tr>
                                                    <xsl:choose>
                                                        <xsl:when test="following-sibling::*[name() = $ename][@id = $xti][@effectiveDate = $xte][@circularForId]">
                                                            <tr>
                                                                <td style="text-align: left;"/>
                                                                <td style="text-align: left;" colspan="3">
                                                                    <xsl:call-template name="showIcon">
                                                                        <xsl:with-param name="which">notice</xsl:with-param>
                                                                    </xsl:call-template>
                                                                    <xsl:text>&#160;</xsl:text>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'tmpCircularReference'"/>
                                                                        <xsl:with-param name="p1">
                                                                            <xsl:call-template name="doShorthandId">
                                                                                <xsl:with-param name="id" select="$xti"/>
                                                                            </xsl:call-template>
                                                                        </xsl:with-param>
                                                                    </xsl:call-template>
                                                                </td>
                                                            </tr>
                                                        </xsl:when>
                                                        <xsl:when test="following-sibling::*[name() = $ename][@id = $xti][@effectiveDate = $xte][@circularForName]">
                                                            <tr>
                                                                <td style="text-align: left;"/>
                                                                <td style="text-align: left;" colspan="3">
                                                                    <xsl:call-template name="showIcon">
                                                                        <xsl:with-param name="which">notice</xsl:with-param>
                                                                    </xsl:call-template>
                                                                    <xsl:text>&#160;</xsl:text>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'tmpCircularReference'"/>
                                                                        <xsl:with-param name="p1">
                                                                            <xsl:call-template name="doShorthandId">
                                                                                <xsl:with-param name="id" select="$xtn"/>
                                                                            </xsl:call-template>
                                                                        </xsl:with-param>
                                                                    </xsl:call-template>
                                                                </td>
                                                            </tr>
                                                        </xsl:when>
                                                    </xsl:choose>
                                                </xsl:if>
                                            </xsl:for-each>
                                        </table>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <!-- relationship -->
            <!-- <relationship type="SPEC" model="" template="" flexibility=""/> -->
            <!-- 2DO model resolving comparable problem to transaction/@model -->
            <!-- 2DO implement links to templates/models if possible -->
            <xsl:if test="relationship">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Relationship'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:for-each select="relationship">
                            <xsl:call-template name="getXFormsLabel">
                                <xsl:with-param name="simpleTypeKey" select="'RelationshipTypes'"/>
                                <xsl:with-param name="simpleTypeValue" select="@type"/>
                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                            </xsl:call-template>
                            <xsl:text>: </xsl:text>
                            <xsl:choose>
                                <xsl:when test="@template">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tmpArtifactTypeTemplate'"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tmpArtifactTypeModel'"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                            <xsl:text> </xsl:text>
                            <xsl:choose>
                                <xsl:when test="@template and $seeThisUrlLocation = 'live'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tmpRelationshipLinked'"/>
                                        <xsl:with-param name="p1">
                                            <xsl:variable name="rccontent" as="element()*">
                                                <xsl:call-template name="getRulesetContent">
                                                    <xsl:with-param name="ruleset" select="@template"/>
                                                    <xsl:with-param name="flexibility" select="@flexibility"/>
                                                    <xsl:with-param name="sofar" select="()"/>
                                                </xsl:call-template>
                                            </xsl:variable>
                                            <xsl:value-of select="local:doHtmlName('TM', @url, @ident, @template, @flexibility, (), (), (), (), '.html', 'false', not(exists($rccontent)))"/>
                                        </xsl:with-param>
                                        <xsl:with-param name="p2" select="@template"/>
                                        <xsl:with-param name="p3" select="concat(@tmdisplayName, ' ')"/>
                                        <xsl:with-param name="p4">
                                            <xsl:choose>
                                                <xsl:when test="matches(@flexibility, '^\d{4}')">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="@flexibility"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tmpRelationship'"/>
                                        <xsl:with-param name="p1">
                                            <xsl:choose>
                                                <xsl:when test="@template">
                                                    <xsl:value-of select="@template"/>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:value-of select="@model"/>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:with-param>
                                        <xsl:with-param name="p2" select="concat(@tmdisplayName, ' ')"/>
                                        <xsl:with-param name="p3">
                                            <xsl:choose>
                                                <xsl:when test="matches(@flexibility, '^\d{4}')">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="@flexibility"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                            <xsl:apply-templates select="." mode="buildReferenceBox"/>
                            <xsl:if test="position() != last()">
                                <br/>
                            </xsl:if>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:if>
            <!-- template examples always first if any -->
            <xsl:apply-templates select="example" mode="templateparticle">
                <xsl:with-param name="parent-id" select="''"/>
                <xsl:with-param name="fWidth" select="$fWidth"/>
                <xsl:with-param name="sofar" select="()"/>
                <xsl:with-param name="templateFormat" select="$tmformat"/>
            </xsl:apply-templates>
            <!-- if this is a transsaction template, as links to the default instances is those have been generated -->
            <xsl:variable name="candidateTransactions" select="$allScenarios//representingTemplate[@ref = $tid]"/>
            <xsl:if test="$createDefaultInstancesForRepresentingTemplates=true() and $candidateTransactions">
                <!-- a little milestoning -->
                <xsl:call-template name="logMessage">
                    <xsl:with-param name="level" select="$logDEBUG"/>
                    <xsl:with-param name="msg">
                        <xsl:text>*** Creating links to default instances for representing templates</xsl:text>
                    </xsl:with-param>
                </xsl:call-template>
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'defaultInstance'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <table class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr class="desclabel" data-tt-id="instance-{generate-id(.)}">
                                <td class="togglertreetable" style="height: 1.5em;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'defaultInstance'"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <tr data-tt-id="instanceBody-{generate-id(.)}" data-tt-parent-id="instance-{generate-id(.)}">
                                <td>
                                    <table>
                                        <tr>
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'Scenario'"/>
                                                </xsl:call-template>
                                            </th>
                                            <th>
                                                <xsl:call-template name="showIcon">
                                                    <xsl:with-param name="which">doublearrow</xsl:with-param>
                                                    <xsl:with-param name="tooltip">
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'Group'"/>
                                                        </xsl:call-template>
                                                    </xsl:with-param>
                                                </xsl:call-template>
                                                <xsl:text>&#160;</xsl:text>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'Transaction'"/>
                                                </xsl:call-template>
                                            </th>
                                            <th>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'Transaction'"/>
                                                </xsl:call-template>
                                            </th>
                                            <th/>
                                        </tr>
                                        <xsl:for-each select="$candidateTransactions">
                                            <xsl:variable name="trid" select="parent::transaction/@id"/>
                                            <!-- cache transaction/@effectiveDate. This is relatively new so might not be present -->
                                            <xsl:variable name="treff" select="parent::transaction/@effectiveDate"/>
                                            <xsl:variable name="tflex" select="@flexibility"/>
                                            <xsl:variable name="rccontent" as="element(template)?">
                                                <xsl:call-template name="getRulesetContent">
                                                    <xsl:with-param name="ruleset" select="$tid"/>
                                                    <xsl:with-param name="flexibility" select="$tflex"/>
                                                    <xsl:with-param name="sofar" select="()"/>
                                                </xsl:call-template>
                                            </xsl:variable>
                                            <xsl:if test="$rccontent[@id = $tid][@effectiveDate = $teff]">
                                                <xsl:call-template name="logMessage">
                                                    <xsl:with-param name="level" select="$logDEBUG"/>
                                                    <xsl:with-param name="msg">
                                                        <xsl:text>*** Instance files HTML/XML for transaction: name='</xsl:text>
                                                        <xsl:value-of select="parent::transaction/name[1]"/>
                                                        <xsl:text>' id='</xsl:text>
                                                        <xsl:value-of select="$trid"/>
                                                        <xsl:text>' effectiveDate='</xsl:text>
                                                        <xsl:value-of select="parent::transaction/@effectiveDate"/>
                                                        <xsl:text>'</xsl:text>
                                                    </xsl:with-param>
                                                </xsl:call-template>
                                                <tr>
                                                    <td>
                                                        <xsl:for-each select="ancestor::scenario">
                                                            <xsl:call-template name="showStatusDot">
                                                                <xsl:with-param name="status" select="@statusCode"/>
                                                                <xsl:with-param name="title">
                                                                    <xsl:call-template name="getXFormsLabel">
                                                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                    </xsl:call-template>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <a href="{local:doHtmlName('SC', @id , @effectiveDate, '.html')}" onclick="target='_blank';" target="_blank">
                                                                <xsl:call-template name="doName">
                                                                    <xsl:with-param name="ns" select="name"/>
                                                                </xsl:call-template>
                                                                <xsl:if test="@versionLabel">
                                                                    <xsl:text> </xsl:text>
                                                                    <xsl:value-of select="@versionLabel"/>
                                                                </xsl:if>
                                                            </a>
                                                        </xsl:for-each>
                                                    </td>
                                                    <td>
                                                        <xsl:for-each select="parent::transaction/parent::transaction">
                                                            <xsl:call-template name="showStatusDot">
                                                                <xsl:with-param name="status" select="@statusCode"/>
                                                                <xsl:with-param name="title">
                                                                    <xsl:call-template name="getXFormsLabel">
                                                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                    </xsl:call-template>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <a href="{local:doHtmlName('TR', @id , @effectiveDate, '.html')}" onclick="target='_blank';" target="_blank">
                                                                <xsl:call-template name="doName">
                                                                    <xsl:with-param name="ns" select="name"/>
                                                                </xsl:call-template>
                                                                <xsl:if test="@versionLabel">
                                                                    <xsl:text> </xsl:text>
                                                                    <xsl:value-of select="@versionLabel"/>
                                                                </xsl:if>
                                                            </a>
                                                        </xsl:for-each>
                                                    </td>
                                                    <td>
                                                        <xsl:for-each select="parent::transaction">
                                                            <xsl:call-template name="showStatusDot">
                                                                <xsl:with-param name="status" select="@statusCode"/>
                                                                <xsl:with-param name="title">
                                                                    <xsl:call-template name="getXFormsLabel">
                                                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                    </xsl:call-template>
                                                                </xsl:with-param>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <xsl:call-template name="showDirection">
                                                                <xsl:with-param name="dir" select="@type"/>
                                                            </xsl:call-template>
                                                            <xsl:text>&#160;</xsl:text>
                                                            <a href="{local:doHtmlName('TR', @id , @effectiveDate, '.html')}" onclick="target='_blank';" target="_blank">
                                                                <xsl:call-template name="doName">
                                                                    <xsl:with-param name="ns" select="name"/>
                                                                </xsl:call-template>
                                                                <xsl:if test="@versionLabel">
                                                                    <xsl:text> </xsl:text>
                                                                    <xsl:value-of select="@versionLabel"/>
                                                                </xsl:if>
                                                            </a>
                                                        </xsl:for-each>
                                                    </td>
                                                    <td>
                                                        <div>
                                                            <a href="{local:doHtmlName('TR', $trid, $treff, '_instance.html', 'true')}" onclick="target='_blank';" target="_blank">
                                                                <xsl:text>HTML (</xsl:text>
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Transaction'"/>
                                                                </xsl:call-template>
                                                                <xsl:text>)</xsl:text>
                                                            </a>
                                                        </div>
                                                        <div>
                                                            <a href="{local:doHtmlName('DS', $trid, $treff, '_instance.html', 'true')}" onclick="target='_blank';" target="_blank">
                                                                <xsl:text>HTML (</xsl:text>
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'Dataset'"/>
                                                                </xsl:call-template>
                                                                <xsl:text>)</xsl:text>
                                                            </a>
                                                        </div>
                                                        <div>
                                                            <a href="{local:doHtmlName('TR', $trid, $treff, '_instance.xml', 'true')}" onclick="target='_blank';" target="_blank">
                                                                <xsl:text>XML</xsl:text>
                                                            </a>
                                                        </div>
                                                    </td>
                                                </tr>
                                            </xsl:if>
                                        </xsl:for-each>
                                    </table>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <!-- diff engine -->
            <xsl:if test="diff:delete[@name = ('example')]">
                <tr>
                    <td>
                        <table class="ad-diffblock-horizontal">
                            <tr class="ad-diff-red">
                                <td class="ad-diff-singlebox">
                                    <strike>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'Example'"/>
                                        </xsl:call-template>
                                    </strike>
                                </td>
                            </tr>
                        </table>
                    </td>
                    <td colspan="3"/>
                </tr>
            </xsl:if>
            <!-- issues -->
            <xsl:call-template name="check4Issue">
                <xsl:with-param name="id" select="@id"/>
                <xsl:with-param name="effectiveDate" select="@effectiveDate"/>
                <xsl:with-param name="colspans" select="3"/>
            </xsl:call-template>
            <!-- if there is at least one element or attribute or one choice or one include in a template, process it -->
            <xsl:choose>
                <xsl:when test="count(element | attribute | include | choice | assert | report | let | defineVariable | constraint | property) &gt; 0">
                    <tr style="vertical-align: top;">
                        <td colspan="4" class="tabtab">
                            <xsl:if test="$switchCreateTreeTableHtml = 'true'">
                                <div>
                                    <button id="expandAll" type="button">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                        </xsl:call-template>
                                    </button>
                                    <button id="collapseAll" type="button">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                        </xsl:call-template>
                                    </button>
                                    <input id="nameSearch">
                                        <xsl:attribute name="placeholder">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'textSearch'"/>
                                            </xsl:call-template>
                                        </xsl:attribute>
                                    </input>
                                    <!--<select id="hiddenColumns">
                                        <option value="title">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'showColumn'"/>
                                            </xsl:call-template>
                                        </option>
                                        <option value="columnDetails" disabled="disabled">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Details'"/>
                                            </xsl:call-template>
                                        </option>
                                    </select>-->
                                </div>
                            </xsl:if>
                            <table id="transactionTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                                <thead>
                                    <tr>
                                        <th style="width: 30%; text-align: left;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'itemLabel'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="width: 2em; text-align: left;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'datatypeLabel'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="width: 2em; text-align: left;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'cardinalityLabel'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="width: 2em; text-align: left;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'conformanceLabel'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="text-align: left;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'descLabel'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th style="width: 2em; text-align: left;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Label'"/>
                                            </xsl:call-template>
                                        </th>
                                    </tr>
                                </thead>
                                <tbody class="list">
                                    <xsl:apply-templates select="element | attribute | include | choice | assert | report | let | defineVariable | constraint | property" mode="templateparticle">
                                        <xsl:with-param name="level" select="0"/>
                                        <xsl:with-param name="label" select="$itemlabel"/>
                                        <xsl:with-param name="parent-id" select="''"/>
                                        <xsl:with-param name="sofar" select="concat($tid, '-', $teff)"/>
                                        <xsl:with-param name="templateFormat" select="$tmformat"/>
                                    </xsl:apply-templates>
                                </tbody>
                            </table>
                        </td>
                    </tr>
                </xsl:when>
                <xsl:otherwise>
                    <tr>
                        <td colspan="4">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'noElemsAttribInclChoiceDefined'"/>
                            </xsl:call-template>
                        </td>
                    </tr>
                </xsl:otherwise>
            </xsl:choose>
        </table>
    </xsl:template>
    <xsl:template name="getTemplateChain">
        <!-- 
            get template chain (dependencies), params are 
            current template id $xti and name $ytn
            and the chain so far in $sofar to detect circular references
            $sofar concats every id and name node visited so far surrounded by :
        -->
        <xsl:param name="yti"/>
        <xsl:param name="ytn"/>
        <xsl:param name="yte"/>
        <xsl:param name="ydn"/>
        <xsl:param name="yin"/>
        <xsl:param name="yst"/>
        <xsl:param name="yvl"/>
        <xsl:param name="sofar"/>
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logALL"/>
            <xsl:with-param name="msg">
                <xsl:text>R====</xsl:text>
                <xsl:value-of select="$yti"/>
                <xsl:text>/</xsl:text>
                <xsl:value-of select="$ytn"/>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:call-template name="logMessage">
            <xsl:with-param name="level" select="$logALL"/>
            <xsl:with-param name="msg">
                <xsl:text>SOFAR(</xsl:text>
                <xsl:value-of select="$sofar"/>
                <xsl:text>)   SUB(</xsl:text>
                <xsl:value-of select="substring-after($sofar, $yti)"/>
                <xsl:text>)   CIRC: </xsl:text>
                <xsl:value-of select="contains(substring-after($sofar, $yti), concat(':', $yti, ':'))"/>
                <xsl:text>&#160;</xsl:text>
                <xsl:value-of select="contains(substring-after($sofar, $ytn), concat(':', $ytn, ':'))"/>
            </xsl:with-param>
        </xsl:call-template>
        <xsl:choose>
            <xsl:when test="contains(substring-after($sofar, $yti), concat(':', $yti, ':'))">
                <!-- circular reference by id -->
                <ref xmlns="" type="dependency" id="{$yti}" name="{$ytn}" effectiveDate="{$yte}" displayName="{$ydn}" statusCode="{$yst}" versionLabel="{$yvl}" newestForId="{$yin}" circularForId="{true()}"/>
            </xsl:when>
            <xsl:when test="contains(substring-after($sofar, $ytn), concat(':', $ytn, ':'))">
                <!-- circular reference by name -->
                <ref xmlns="" type="dependency" id="{$yti}" name="{$ytn}" effectiveDate="{$yte}" displayName="{$ydn}" statusCode="{$yst}" versionLabel="{$yvl}" newestForId="{$yin}" circularForName="{true()}"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:for-each select="
                        $allTemplates/*/ref[template//include[@ref = $yti or @ref = $ytn][($yin and (not(@flexibility) or string(@flexibility) = 'dynamic')) or @flexibility = $yte]] |
                        $allTemplates/*/ref[template//*[@contains = $yti or @contains = $ytn][($yin and (not(@flexibility) or string(@flexibility) = 'dynamic')) or @flexibility = $yte]]">
                    <xsl:variable name="xti" select="template/@id"/>
                    <xsl:variable name="xtn" select="template/@name"/>
                    <xsl:variable name="xdn" select="template/@displayName"/>
                    <xsl:variable name="xte" select="template/@effectiveDate"/>
                    <xsl:variable name="xin" select="max($allTemplates/*/ref/template[@id = $xti]/xs:dateTime(@effectiveDate)) = $xte"/>
                    <xsl:variable name="xst" select="template/@statusCode"/>
                    <xsl:variable name="xvl" select="template/@versionLabel"/>
                    <ref xmlns="" type="dependency" newestForId="{$xin}">
                        <xsl:copy-of select="template/(@* except (@type | @flexibility))"/>
                        <xsl:copy-of select="template/classification"/>
                    </ref>
                    <xsl:call-template name="getTemplateChain">
                        <xsl:with-param name="yti" select="$xti"/>
                        <xsl:with-param name="ytn" select="$xtn"/>
                        <xsl:with-param name="yte" select="$xte"/>
                        <xsl:with-param name="ydn" select="$xdn"/>
                        <xsl:with-param name="yin" select="$xin"/>
                        <xsl:with-param name="yst" select="$xst"/>
                        <xsl:with-param name="yvl" select="$xvl"/>
                        <xsl:with-param name="sofar" select="concat($sofar, ':', $yti, ':', $ytn)"/>
                    </xsl:call-template>
                </xsl:for-each>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    <xsl:template match="*" mode="templateparticle">
        <xsl:param name="sofar" select="()"/>
        <xsl:text>OOPS: </xsl:text>
        <xsl:value-of select="name()"/>
    </xsl:template>
    <xsl:template match="questionnaire">
        <xsl:param name="templatename"/>
        <xsl:param name="label" as="element()?"/>
        <!-- cache id, name, effectiveDate and calculate if we're the newest version of the template -->
        <xsl:variable name="tid" select="@id"/>
        <xsl:variable name="teff" select="@effectiveDate"/>
        <xsl:variable name="tname" select="@name"/>
        <xsl:variable name="tIsNewest" select="max($allQuestionnaires[@id = $tid]/xs:dateTime(@effectiveDate)) = $teff"/>
        <xsl:variable name="fWidth" select="'20em'"/>
        <table width="100%" border="0" cellspacing="3" cellpadding="2" xmlns="http://www.w3.org/1999/xhtml">
            <!-- id / effectiveDate -->
            <tr style="vertical-align: top;">
                <!-- id -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Id'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:variable name="ident" select="(@ident, ../@ident)[1]"/>
                    <xsl:variable name="url" select="(@url, ../@url)[1]"/>
                    <xsl:choose>
                        <!-- In live ART-DECOR you can just go there, but in an offline publication, we cannot jump to a different publication location -->
                        <xsl:when test="$url and $ident[not(. = $projectPrefix)] and $seeThisUrlLocation = 'live'">
                            <a href="{local:doHtmlName('QQ',$url,$ident,$tid,$teff,(),(),(),(),'.html','false',true())}" onclick="target='_blank';">
                                <xsl:value-of select="$tid"/>
                            </a>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$tid"/>
                        </xsl:otherwise>
                    </xsl:choose>
                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                </td>
                <!-- version information -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'effectiveDate'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:if test="$teff[string-length() &gt; 0]">
                        <!--<xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'templateValidFrom'"/>
                        </xsl:call-template>-->
                        <xsl:call-template name="showDate">
                            <xsl:with-param name="date" select="$teff"/>
                        </xsl:call-template>
                    </xsl:if>
                    <xsl:if test="@expirationDate[string-length() gt 0]">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'templateValidTo'"/>
                        </xsl:call-template>
                        <xsl:call-template name="showDate">
                            <xsl:with-param name="date" select="@expirationDate"/>
                        </xsl:call-template>
                    </xsl:if>
                    <xsl:variable name="otherVersions" select="$allQuestionnaires[@id = $tid][@effectiveDate[not(. = $teff)]]" as="element()*"/>
                    <xsl:if test="$otherVersions">
                        <div>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'OtherVersionsWithThisId'"/>
                            </xsl:call-template>
                            <ul>
                                <xsl:for-each-group select="$otherVersions" group-by="concat(@id, @effectiveDate)">
                                    <xsl:sort select="@effectiveDate" order="descending"/>
                                    <xsl:for-each select="current-group()[1]">
                                        <li>
                                            <xsl:call-template name="showStatusDot">
                                                <xsl:with-param name="status" select="@statusCode"/>
                                                <xsl:with-param name="title">
                                                    <xsl:call-template name="getXFormsLabel">
                                                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                    </xsl:call-template>
                                                </xsl:with-param>
                                            </xsl:call-template>
                                            <xsl:text> </xsl:text>
                                            <a href="{local:doHtmlName('QQ', @id, @effectiveDate, '.html')}" onclick="target='_blank';">
                                                <xsl:value-of select="(@displayName, @name)[1]"/>
                                            </a>
                                            <xsl:if test="@versionLabel">
                                                <xsl:value-of select="@versionLabel"/>
                                                <xsl:text> - </xsl:text>
                                            </xsl:if>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'xAsOfy'"/>
                                            </xsl:call-template>
                                            <xsl:text> </xsl:text>
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="@effectiveDate"/>
                                            </xsl:call-template>
                                            <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                        </li>
                                    </xsl:for-each>
                                </xsl:for-each-group>
                            </ul>
                        </div>
                    </xsl:if>
                </td>
            </tr>
            <!-- status / versionLabel -->
            <tr style="vertical-align: top;">
                <!-- status -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Status'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:call-template name="showStatusDot">
                        <xsl:with-param name="status" select="@statusCode"/>
                        <xsl:with-param name="title">
                            <xsl:call-template name="getXFormsLabel">
                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                            </xsl:call-template>
                        </xsl:with-param>
                    </xsl:call-template>
                    <xsl:text> </xsl:text>
                    <xsl:call-template name="getXFormsLabel">
                        <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                        <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                    </xsl:call-template>
                </td>
                <!-- versionLabel -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'columnVersionLabel'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:value-of select="@versionLabel"/>
                </td>
            </tr>
            <!-- name / canonicalUri -->
            <tr style="vertical-align: top;">
                <!-- name -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Name'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:call-template name="doName">
                        <xsl:with-param name="ns" select="name"/>
                    </xsl:call-template>
                </td>
                <!-- canonicalUri -->
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'canonicalUri'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;">
                    <xsl:value-of select="@canonicalUri"/>
                </td>
            </tr>
            <!-- description if given -->
            <xsl:if test="desc[not(. = '')]">
                <tr style="vertical-align: top;">
                    <xsl:choose>
                        <xsl:when test="desc[string-length(string-join(descendant-or-self::text(), ' ')) > 200]">
                            <td style="text-align: left;" colspan="4">
                                <xsl:variable name="elid" select="local:randomString2(., ())"/>
                                <!--<xsl:variable name="elidtoggler" select="concat('toggler-', $elid)"/>-->
                                <table id="templateDescTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                                    <tr class="desclabel" data-tt-id="templateDescHeader">
                                        <td class="togglertreetable" style="height: 1.5em;">
                                            <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                                <xsl:attribute name="class" select="'toggler'"/>
                                                <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                                            </xsl:if>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'templateDescription'"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                    <tr data-tt-id="templateDescBody" data-tt-parent-id="templateDescHeader">
                                        <td>
                                            <div id="{$elid}">
                                                <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                                    <xsl:attribute name="class" select="'toggle'"/>
                                                </xsl:if>
                                                <xsl:call-template name="doDescription">
                                                    <xsl:with-param name="ns" select="desc"/>
                                                </xsl:call-template>
                                            </div>
                                        </td>
                                    </tr>
                                </table>
                            </td>
                        </xsl:when>
                        <xsl:otherwise>
                            <th style="width: {$fWidth}; text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'templateDescription'"/>
                                </xsl:call-template>
                            </th>
                            <td style="text-align: left;" colspan="3">
                                <xsl:call-template name="doDescription">
                                    <xsl:with-param name="ns" select="desc"/>
                                </xsl:call-template>
                            </td>
                        </xsl:otherwise>
                    </xsl:choose>
                </tr>
            </xsl:if>
            <xsl:if test="purpose[not(. = '')]">
                <tr style="vertical-align: top;">
                    <xsl:choose>
                        <xsl:when test="purpose[string-length(string-join(descendant-or-self::text(), ' ')) > 200]">
                            <td style="text-align: left;" colspan="4">
                                <xsl:variable name="elid" select="local:randomString2(., ())"/>
                                <!--<xsl:variable name="elidtoggler" select="concat('toggler-', $elid)"/>-->
                                <table id="templateDescTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                                    <tr class="desclabel" data-tt-id="templateDescHeader">
                                        <td class="togglertreetable" style="height: 1.5em;">
                                            <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                                <xsl:attribute name="class" select="'toggler'"/>
                                                <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                                            </xsl:if>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Purpose'"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                    <tr data-tt-id="templateDescBody" data-tt-parent-id="templateDescHeader">
                                        <td>
                                            <div id="{$elid}">
                                                <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                                    <xsl:attribute name="class" select="'toggle'"/>
                                                </xsl:if>
                                                <xsl:call-template name="doDescription">
                                                    <xsl:with-param name="ns" select="purpose"/>
                                                </xsl:call-template>
                                            </div>
                                        </td>
                                    </tr>
                                </table>
                            </td>
                        </xsl:when>
                        <xsl:otherwise>
                            <th style="width: {$fWidth}; text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Purpose'"/>
                                </xsl:call-template>
                            </th>
                            <td style="text-align: left;" colspan="3">
                                <xsl:call-template name="doDescription">
                                    <xsl:with-param name="ns" select="purpose"/>
                                </xsl:call-template>
                            </td>
                        </xsl:otherwise>
                    </xsl:choose>
                </tr>
            </xsl:if>
            <xsl:if test="copyright[not(. = '')]">
                <tr style="vertical-align: top;">
                    <xsl:choose>
                        <xsl:when test="copyright[string-length(string-join(descendant-or-self::text(), ' ')) > 200]">
                            <td style="text-align: left;" colspan="4">
                                <xsl:variable name="elid" select="local:randomString2(., ())"/>
                                <!--<xsl:variable name="elidtoggler" select="concat('toggler-', $elid)"/>-->
                                <table id="templateDescTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                                    <tr class="desclabel" data-tt-id="templateDescHeader">
                                        <td class="togglertreetable" style="height: 1.5em;">
                                            <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                                <xsl:attribute name="class" select="'toggler'"/>
                                                <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                                            </xsl:if>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Copyright'"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                    <tr data-tt-id="templateDescBody" data-tt-parent-id="templateDescHeader">
                                        <td>
                                            <div id="{$elid}">
                                                <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                                    <xsl:attribute name="class" select="'toggle'"/>
                                                </xsl:if>
                                                <xsl:call-template name="doDescription">
                                                    <xsl:with-param name="ns" select="copyright"/>
                                                </xsl:call-template>
                                            </div>
                                        </td>
                                    </tr>
                                </table>
                            </td>
                        </xsl:when>
                        <xsl:otherwise>
                            <th style="width: {$fWidth}; text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Copyright'"/>
                                </xsl:call-template>
                            </th>
                            <td style="text-align: left;" colspan="3">
                                <xsl:call-template name="doDescription">
                                    <xsl:with-param name="ns" select="copyright"/>
                                </xsl:call-template>
                            </td>
                        </xsl:otherwise>
                    </xsl:choose>
                </tr>
            </xsl:if>
            <xsl:if test="jurisdiction">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Jurisdiction'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:for-each select="jurisdiction">
                            <div>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'codeCodeSystemNameShallBeX-v2'"/>
                                    <xsl:with-param name="p1" select="@code"/>
                                    <xsl:with-param name="p3">
                                        <xsl:choose>
                                            <xsl:when test="@canonicalUri">
                                                <xsl:value-of select="concat(@codeSystem, ' (', @canonicalUri, ')')"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="@codeSystem"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:with-param>
                                </xsl:call-template>
                                <xsl:if test="@displayName">
                                    <xsl:text> - </xsl:text>
                                    <xsl:value-of select="@displayName"/>
                                </xsl:if>
                            </div>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:if>
            <xsl:apply-templates select="publishingAuthority | endorsingauthority">
                <xsl:with-param name="fWidth" select="$fWidth"/>
            </xsl:apply-templates>
            <!-- classification -->
            <xsl:if test="classification">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Classification'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:for-each select="classification/tag">
                            <xsl:value-of select="."/>
                            <xsl:if test="position() != last()">
                                <br/>
                            </xsl:if>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:if>
            <!-- is associated with what concepts?
                
                <templateAssociation templateId="2.999.999.993.77.10.900100" effectiveDate="2012-04-10T00:00:00">
                    <concept ref="2.999.999.993.77.2.1.100060" effectiveDate="2012-04-10T00:00:00" elementId="2.999.999.993.77.9.900100.1"/>
                </templateAssociation>
                
            -->
            <xsl:variable name="associatedConcepts" select="$allQuestionnaireAssociations[@questionnaireId = $tid][@questionnaireEffectiveDate = $teff]/concept" as="element()*"/>
            <xsl:variable name="renderedConcepts" as="element()*">
                <xsl:for-each-group select="$associatedConcepts" group-by="concat(@ref, @effectiveDate)">
                    <xsl:variable name="deid" select="current-group()[1]/@ref"/>
                    <xsl:variable name="deed" select="current-group()[1]/@effectiveDate"/>
                    <xsl:variable name="concept" select="(current-group()/self::origconcept[name], local:getConceptFlat($deid, $deed))[1]" as="element()?"/>
                    <xsl:variable name="destatus" select="(current-group()/@conceptStatusCode, $concept/@statusCode)[1]" as="item()?"/>
                    <xsl:variable name="deversionlabel" select="(current-group()/@conceptVersionLabel, $concept/@versionLabel)[1]"/>
                    <xsl:variable name="dataset" select="local:getDatasetForConcept($deid, $deed)"/>
                    <xsl:variable name="dsid" select="(current-group()/@datasetId, $dataset/@id)[1]"/>
                    <xsl:variable name="dsef" select="(current-group()/@datasetEffectiveDate, $dataset/@effectiveDate)[1]"/>
                    <xsl:variable name="dspath">
                        <xsl:choose>
                            <xsl:when test="@path">
                                <xsl:value-of select="current-group()[1]/@path"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:for-each select="$dataset//concept[(@id | @ref) = $concept/(@id | @ref)][(@effectiveDate | @flexibility) = $concept/(@effectiveDate | @flexibility)]/ancestor::concept">
                                    <xsl:call-template name="doName">
                                        <!-- will not exist on RetrieveTemplate expanded templates -->
                                        <xsl:with-param name="ns" select="local:getConceptFlat((@id | @ref), (@effectiveDate | @flexibility))/name"/>
                                    </xsl:call-template>
                                    <xsl:text> / </xsl:text>
                                </xsl:for-each>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsstatus" select="(current-group()/@datasetStatusCode, $dataset/@statusCode)[1]" as="item()?"/>
                    <xsl:variable name="dsname">
                        <xsl:choose>
                            <xsl:when test="current-group()[1]/datasetName">
                                <xsl:call-template name="doName">
                                    <!-- will not exist on RetrieveTemplate expanded templates -->
                                    <xsl:with-param name="ns" select="current-group()[1]/datasetName"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="@datasetName">
                                <xsl:value-of select="current-group()[1]/@datasetName"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="doName">
                                    <!-- will not exist on RetrieveTemplate expanded templates -->
                                    <xsl:with-param name="ns" select="$dataset/name"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsversionlabel" select="(current-group()/@datasetVersionLabel, $dataset/@versionLabel)[1]"/>
                    <xsl:variable name="deiddisplay">
                        <xsl:choose>
                            <xsl:when test="@refdisplay[not(. = '')]">
                                <xsl:value-of select="current-group()[1]/@refdisplay"/>
                            </xsl:when>
                            <xsl:when test="$concept[@iddisplay]">
                                <xsl:value-of select="($concept/@iddisplay)[1]"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="doShorthandId">
                                    <xsl:with-param name="id" select="$deid"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:if test="empty($concept)">
                        <xsl:call-template name="logMessage">
                            <xsl:with-param name="level" select="$logDEBUG"/>
                            <xsl:with-param name="msg">
                                <xsl:text>+++ Could not find concept "</xsl:text>
                                <xsl:value-of select="$deid"/>
                                <xsl:text>" effectiveDate "</xsl:text>
                                <xsl:value-of select="$deed"/>
                                <xsl:text>" associated with questionnaire id "</xsl:text>
                                <xsl:value-of select="$tid"/>
                                <xsl:text>" effectiveDate "</xsl:text>
                                <xsl:value-of select="$teff"/>
                                <xsl:text>" name "</xsl:text>
                                <xsl:value-of select="$tname"/>
                                <xsl:text> and item </xsl:text>
                                <xsl:value-of select="@elementId | @elementPath"/>
                                <xsl:text>. This may not be meaningful as all questionnaireAssociations anywhere copied during compile phase, but datasets are limited to those pertaining to the project. This might place certain associations out of context.</xsl:text>
                            </xsl:with-param>
                        </xsl:call-template>
                        <!--<tr class="missingAssociation" xmlns="http://www.w3.org/1999/xhtml">
                            <td style="vertical-align: top;" colspan="3">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'theReferencedConceptAsOfCannotBeFound'"/>
                                    <xsl:with-param name="p1" select="$deid"/>
                                    <xsl:with-param name="p2">
                                        <xsl:call-template name="showDate">
                                            <xsl:with-param name="date" select="$deed"/>
                                        </xsl:call-template>
                                    </xsl:with-param>
                                </xsl:call-template>
                            </td>
                        </tr>-->
                    </xsl:if>
                    <xsl:for-each select="$concept">
                        <tr>
                            <td style="text-align: left; vertical-align: top; width: 1%;">
                                <xsl:choose>
                                    <xsl:when test="empty($dsid)">
                                        <xsl:copy-of select="$deiddisplay"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <a href="{local:doHtmlName('DS', $projectPrefix, $dsid, $dsef, $deid, $deed, (), (), '.html', 'false')}" onclick="target='_blank';">
                                            <xsl:copy-of select="$deiddisplay"/>
                                        </a>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </td>
                            <td style="vertical-align: top;">
                                <xsl:if test="string-length($dspath) > 0">
                                    <xsl:attribute name="title" select="$dspath"/>
                                </xsl:if>
                                <xsl:if test="string-length($destatus) > 0">
                                    <xsl:call-template name="showStatusDot">
                                        <xsl:with-param name="status" select="$destatus"/>
                                        <xsl:with-param name="title">
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="$destatus"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                    <xsl:text>&#160;</xsl:text>
                                </xsl:if>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="$concept/name"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <xsl:value-of select="$deversionlabel"/>
                            </td>
                            <td style="text-align: left; vertical-align: top;">
                                <xsl:if test="string-length($dsstatus) > 0">
                                    <xsl:call-template name="showStatusDot">
                                        <xsl:with-param name="status" select="$dsstatus"/>
                                        <xsl:with-param name="title">
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="$dsstatus"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                    <xsl:text>&#160;</xsl:text>
                                </xsl:if>
                                <xsl:copy-of select="$dsname"/>
                                <xsl:text> </xsl:text>
                                <xsl:value-of select="$dsversionlabel"/>
                                <xsl:if test="current-group()[1][not(@ident = $projectPrefix)]">
                                    <xsl:text> </xsl:text>
                                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                </xsl:if>
                            </td>
                        </tr>
                    </xsl:for-each>
                </xsl:for-each-group>
            </xsl:variable>
            <xsl:if test="$renderedConcepts[not(@class = 'missingAssociation')]">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'AssociatedWith'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:variable name="elid" select="local:randomString2(., ())"/>
                        <!--<xsl:variable name="elidtoggler" select="concat('toggler-', $elid)"/>-->
                        <table id="assocsTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr class="desclabel" data-tt-id="assocsHeader">
                                <td class="togglertreetable" style="height: 1.5em;">
                                    <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                        <xsl:attribute name="class" select="'toggler'"/>
                                        <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                                    </xsl:if>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'AssociatedWith'"/>
                                    </xsl:call-template>
                                    <xsl:text> </xsl:text>
                                    <xsl:variable name="assoccnt" select="count($renderedConcepts[not(@class = 'missingAssociation')])"/>
                                    <xsl:choose>
                                        <xsl:when test="$assoccnt = 1">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'numberOfConcept'"/>
                                                <xsl:with-param name="p1" select="$assoccnt"/>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'numberOfConcepts'"/>
                                                <xsl:with-param name="p1" select="$assoccnt"/>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <!-- 2018-03-27 Deactivated code because we will not be detecting those anymore due to changes in compilation. Activate logDEBUG to send messages to the log -->
                                    <!--<xsl:if test="$renderedConcepts[@class = 'missingAssociation']">
                                        <xsl:text>&#160;</xsl:text>
                                        <xsl:call-template name="showIcon">
                                            <xsl:with-param name="which">notice</xsl:with-param>
                                        </xsl:call-template>
                                    </xsl:if>-->
                                </td>
                            </tr>
                            <tr data-tt-id="assocsBody" data-tt-parent-id="assocsHeader">
                                <td>
                                    <div id="{$elid}">
                                        <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                            <xsl:attribute name="class" select="'toggle'"/>
                                        </xsl:if>
                                        <table width="100%" border="0" cellspacing="1" cellpadding="3">
                                            <tr>
                                                <th style="width: 20%; text-align: left;">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'Id'"/>
                                                    </xsl:call-template>
                                                </th>
                                                <th style="width: 60%; text-align: left;">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'Name'"/>
                                                    </xsl:call-template>
                                                </th>
                                                <th style="width: 20%; text-align: left;">
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'Dataset'"/>
                                                    </xsl:call-template>
                                                </th>
                                            </tr>
                                            <xsl:copy-of select="$renderedConcepts[not(@class = 'missingAssociation')]"/>
                                        </table>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:variable name="associatedTransactions" select="
                $allScenarios//transaction[representingTemplate[@representingQuestionnaire = $tid][$tIsNewest][not(@representingQuestionnaireFlexibility castable as xs:dateTime)] |
                                           representingTemplate[@representingQuestionnaire = $tid][@representingQuestionnaireFlexibility = $teff]]" as="element()*"/>
            <xsl:if test="$associatedTransactions">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:if test="not($adram = ('mediawiki', 'confluence', 'wordpress'))">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'UsedBy'"/>
                            </xsl:call-template>
                        </xsl:if>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:variable name="elid" select="local:randomString2(., ())"/>
                        <table class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr class="desclabel" data-tt-id="usedHeader">
                                <td class="togglertreetable" style="height: 1.5em;">
                                    <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                        <xsl:attribute name="class" select="'toggler'"/>
                                        <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                                    </xsl:if>
                                    <xsl:variable name="trxcnt" select="count(distinct-values($associatedTransactions/concat(@id, @effectiveDate)))"/>
                                    <xsl:if test="not($adram = ('mediawiki', 'confluence', 'wordpress'))">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'UsedBy'"/>
                                        </xsl:call-template>
                                        <xsl:text> </xsl:text>
                                        <xsl:choose>
                                            <xsl:when test="$trxcnt = 1">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'numberOfTransaction'"/>
                                                    <xsl:with-param name="p1" select="$trxcnt"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'numberOfTransactions'"/>
                                                    <xsl:with-param name="p1" select="$trxcnt"/>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:if>
                                </td>
                            </tr>
                            <tr data-tt-id="usedBody" data-tt-parent-id="usedHeader">
                                <td>
                                    <div id="{$elid}">
                                        <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                            <xsl:attribute name="class" select="'toggle'"/>
                                        </xsl:if>
                                        <table width="100%" border="0" cellspacing="1" cellpadding="3">
                                            <xsl:for-each select="$associatedTransactions">
                                                <xsl:variable name="xpfx" select="(@prefix, @ident, $projectPrefix)[1]"/>
                                                <xsl:variable name="xti" select="@id"/>
                                                <xsl:variable name="xtn" select="@name"/>
                                                <xsl:variable name="xty" select="@type"/>
                                                <xsl:variable name="xte" select="@effectiveDate"/>
                                                <xsl:variable name="xdn" select="@displayName"/>
                                                <xsl:variable name="xin" select="@newestForId"/>
                                                <xsl:variable name="xflx" select="@flexibility"/>
                                                <xsl:variable name="xsts" select="@statusCode"/>
                                                <xsl:variable name="xvsl" select="@versionLabel"/>
                                                <xsl:variable name="ename" select="name()"/>
                                                <xsl:if test="not(preceding-sibling::*[name() = $ename]) and not($associatedTransactions)">
                                                    <tr>
                                                        <th style="width: 20%; text-align: left;">
                                                            <xsl:choose>
                                                                <xsl:when test="$ename = ('transaction', 'ref')">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'UsedBy'"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'Uses'"/>
                                                                    </xsl:call-template>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </th>
                                                        <th style="width: 1%; text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'as'"/>
                                                            </xsl:call-template>
                                                        </th>
                                                        <th style="width: 60%; text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'Name'"/>
                                                            </xsl:call-template>
                                                        </th>
                                                        <th style="width: 19%; text-align: left;">
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'Version'"/>
                                                            </xsl:call-template>
                                                        </th>
                                                    </tr>
                                                </xsl:if>
                                                <xsl:if test="not(preceding-sibling::*[name() = $ename][@id = $xti][@effectiveDate = $xte])">
                                                    <tr>
                                                        <td style="text-align: left;">
                                                            <xsl:choose>
                                                                <xsl:when test="$ename = 'transaction'">
                                                                    <a href="{local:doHtmlName('TR',$xpfx,$xti,$xte,(),(),(),(),'.html','false')}" onclick="target='_blank';">
                                                                        <xsl:call-template name="doShorthandId">
                                                                            <xsl:with-param name="id" select="$xti"/>
                                                                        </xsl:call-template>
                                                                    </a>
                                                                </xsl:when>
                                                            </xsl:choose>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:choose>
                                                                <xsl:when test="$ename = 'transaction'">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'Transaction'"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                            </xsl:choose>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:if test="$xsts">
                                                                <xsl:call-template name="showStatusDot">
                                                                    <xsl:with-param name="status" select="$xsts"/>
                                                                    <xsl:with-param name="title">
                                                                        <xsl:call-template name="getXFormsLabel">
                                                                            <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                                            <xsl:with-param name="simpleTypeValue" select="$xsts"/>
                                                                            <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                                        </xsl:call-template>
                                                                    </xsl:with-param>
                                                                </xsl:call-template>
                                                                <xsl:text> </xsl:text>
                                                            </xsl:if>
                                                            <xsl:choose>
                                                                <xsl:when test="self::transaction">
                                                                    <xsl:call-template name="doName">
                                                                        <xsl:with-param name="ns" select="name"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:when test="string-length($xdn) &gt; 0">
                                                                    <xsl:value-of select="$xdn"/>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:value-of select="$xtn"/>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                            <xsl:if test="string-length($xvsl)>0">
                                                                <xsl:text> (</xsl:text>
                                                                <xsl:value-of select="@versionLabel"/>
                                                                <xsl:text>)</xsl:text>
                                                            </xsl:if>
                                                        </td>
                                                        <td style="text-align: left;">
                                                            <xsl:choose>
                                                                <xsl:when test="$ename = 'uses' and matches($xflx, '^\d')">
                                                                    <xsl:call-template name="showDate">
                                                                        <xsl:with-param name="date" select="$xflx"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:when test="$ename = 'uses' and $xflx">
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                    </xsl:call-template>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:call-template name="showDate">
                                                                        <xsl:with-param name="date" select="$xte"/>
                                                                    </xsl:call-template>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </td>
                                                    </tr>
                                                    <xsl:choose>
                                                        <xsl:when test="following-sibling::*[name() = $ename][@id = $xti][@effectiveDate = $xte][@circularForId]">
                                                            <tr>
                                                                <td style="text-align: left;"/>
                                                                <td style="text-align: left;" colspan="3">
                                                                    <xsl:call-template name="showIcon">
                                                                        <xsl:with-param name="which">notice</xsl:with-param>
                                                                    </xsl:call-template>
                                                                    <xsl:text>&#160;</xsl:text>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'tmpCircularReference'"/>
                                                                        <xsl:with-param name="p1">
                                                                            <xsl:call-template name="doShorthandId">
                                                                                <xsl:with-param name="id" select="$xti"/>
                                                                            </xsl:call-template>
                                                                        </xsl:with-param>
                                                                    </xsl:call-template>
                                                                </td>
                                                            </tr>
                                                        </xsl:when>
                                                        <xsl:when test="following-sibling::*[name() = $ename][@id = $xti][@effectiveDate = $xte][@circularForName]">
                                                            <tr>
                                                                <td style="text-align: left;"/>
                                                                <td style="text-align: left;" colspan="3">
                                                                    <xsl:call-template name="showIcon">
                                                                        <xsl:with-param name="which">notice</xsl:with-param>
                                                                    </xsl:call-template>
                                                                    <xsl:text>&#160;</xsl:text>
                                                                    <xsl:call-template name="getMessage">
                                                                        <xsl:with-param name="key" select="'tmpCircularReference'"/>
                                                                        <xsl:with-param name="p1">
                                                                            <xsl:call-template name="doShorthandId">
                                                                                <xsl:with-param name="id" select="$xtn"/>
                                                                            </xsl:call-template>
                                                                        </xsl:with-param>
                                                                    </xsl:call-template>
                                                                </td>
                                                            </tr>
                                                        </xsl:when>
                                                    </xsl:choose>
                                                </xsl:if>
                                            </xsl:for-each>
                                        </table>
                                    </div>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <!-- relationship -->
            <!-- <relationship type="SPEC" model="" template="" flexibility=""/> -->
            <!-- 2DO model resolving comparable problem to transaction/@model -->
            <!-- 2DO implement links to templates/models if possible -->
            <xsl:if test="relationship">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Relationship'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:for-each select="relationship">
                            <xsl:variable name="reid" select="@ref"/>
                            <xsl:variable name="reed" select="@flexibility"/>
                            <xsl:variable name="rccontent" select="if (@type = 'DRIV') then $allScenarios//transaction[@id = $reid][@effectiveDate = $reed] else if (@type = 'ANSW') then $allQuestionnaires[@id = $reid][@effectiveDate = $reed] else ()" as="element()?"/>
                            <xsl:call-template name="getXFormsLabel">
                                <xsl:with-param name="simpleTypeKey" select="'QuestionnaireRelationshipTypes'"/>
                                <xsl:with-param name="simpleTypeValue" select="@type"/>
                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                            </xsl:call-template>
                            <xsl:text>: </xsl:text>
                            <xsl:choose>
                                <xsl:when test="@type = 'DRIV'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Transaction'"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:when test="@type= 'ANSW'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Questionnaire'"/>
                                    </xsl:call-template>
                                </xsl:when>
                            </xsl:choose>
                            <xsl:text> </xsl:text>
                            <xsl:choose>
                                <xsl:when test="@template and $seeThisUrlLocation = 'live'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tmpRelationshipLinked'"/>
                                        <xsl:with-param name="p1">
                                            <xsl:choose>
                                                <xsl:when test="@type = 'DRIV'">
                                                    <xsl:value-of select="local:doHtmlName('TR', @url, @ident, $reid, $reed, (), (), (), (), '.html', 'false', not(exists($rccontent)))"/>
                                                </xsl:when>
                                                <xsl:when test="@type= 'ANSW'">
                                                    <xsl:variable name="rccontent" select="$allQuestionnaires[@id = $reid][@effectiveDate = $reed]" as="element()*"/>
                                                    <xsl:value-of select="local:doHtmlName('QQ', @url, @ident, $reid, $reed, (), (), (), (), '.html', 'false', not(exists($rccontent)))"/>
                                                </xsl:when>
                                            </xsl:choose>
                                        </xsl:with-param>
                                        <xsl:with-param name="p2" select="$reid"/>
                                        <xsl:with-param name="p3" select="concat($rccontent/name[1], ' ')"/>
                                        <xsl:with-param name="p4">
                                            <xsl:choose>
                                                <xsl:when test="$reed castable as xs:dateTime">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="$reed"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'tmpRelationship'"/>
                                        <xsl:with-param name="p1" select="$reid"/>
                                        <xsl:with-param name="p2" select="concat($rccontent/name[1], ' ')"/>
                                        <xsl:with-param name="p3">
                                            <xsl:choose>
                                                <xsl:when test="$reed castable as xs:dateTime">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="$reed"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                            <xsl:apply-templates select="." mode="buildReferenceBox"/>
                            <xsl:if test="position() != last()">
                                <br/>
                            </xsl:if>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:if>
            <!-- issues -->
            <xsl:call-template name="check4Issue">
                <xsl:with-param name="id" select="@id"/>
                <xsl:with-param name="effectiveDate" select="@effectiveDate"/>
                <xsl:with-param name="colspans" select="3"/>
            </xsl:call-template>
            <xsl:if test="subjectType">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'SubjectType'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:value-of select="string-join(subjectType/@code, ', ')"/>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="code">
                <tr style="vertical-align: top;">
                    <th style="width: {$fWidth}; text-align: left;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Code'"/>
                        </xsl:call-template>
                    </th>
                    <td style="text-align: left;" colspan="3">
                        <xsl:for-each select="code">
                            <div>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'codeCodeSystemNameShallBeX-v2'"/>
                                    <xsl:with-param name="p1" select="@code"/>
                                    <xsl:with-param name="p3">
                                        <xsl:choose>
                                            <xsl:when test="@canonicalUri">
                                                <xsl:value-of select="concat(@codeSystem, ' (', @canonicalUri, ')')"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="@codeSystem"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:with-param>
                                </xsl:call-template>
                                <xsl:if test="@displayName">
                                    <xsl:text> - </xsl:text>
                                    <xsl:value-of select="@displayName"/>
                                </xsl:if>
                            </div>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:if>
            
            <tr style="vertical-align: top;">
                <th style="width: {$fWidth}; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Available formats'"/>
                    </xsl:call-template>
                </th>
                <td style="text-align: left;" colspan="3">
                    <xsl:for-each select="$projectRestUriFHIR">
                        <xsl:text>FHIR </xsl:text>
                        <xsl:value-of select="tokenize(., '/')[last() - 2]"/>
                        <xsl:text> </xsl:text>
                        <a href="{.}Questionnaire/{encode-for-uri($tid)}--{replace($teff, '\D', '')}?_format=xml">
                            <xsl:call-template name="showIcon">
                                <xsl:with-param name="which">document</xsl:with-param>
                            </xsl:call-template>
                            xml</a>
                        <xsl:text> </xsl:text>
                        <a href="{.}Questionnaire/{encode-for-uri($tid)}--{replace($teff, '\D', '')}?_format=json">
                            <xsl:call-template name="showIcon">
                                <xsl:with-param name="which">document</xsl:with-param>
                            </xsl:call-template>
                            json</a>
                    </xsl:for-each>
                </td>
            </tr>
            
            <!-- render qqitems -->
            <tr style="vertical-align: top;">
                <td style="text-align: left;" colspan="4">
                    <xsl:if test="$switchCreateTreeTableHtml = 'true-not-yet-used'">
                        <div xmlns="http://www.w3.org/1999/xhtml">
                            <button id="expandAll" type="button">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'buttonExpandAll'"/>
                                </xsl:call-template>
                            </button>
                            <button id="collapseAll" type="button">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'buttonCollapseAll'"/>
                                </xsl:call-template>
                            </button>
                        </div>
                    </xsl:if>
                    <table id="transactionTable" class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2" xmlns="http://www.w3.org/1999/xhtml">
                        <tr>
                            <th class="columnName" style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Name'"/>
                                </xsl:call-template>
                            </th>
                            <th class="columnDetails" style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Details'"/>
                                </xsl:call-template>
                            </th>
                            <th class="columnNoPrint">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Type'"/>
                                </xsl:call-template>
                            </th>
                        </tr>
                        <xsl:apply-templates select="item" mode="questionnaire">
                            <xsl:with-param name="level" select="1"/>
                            <xsl:with-param name="tid" select="$tid"/>
                            <xsl:with-param name="ted" select="$teff"/>
                        </xsl:apply-templates>
                    </table>
                </td>
            </tr>
        </table>
    </xsl:template>
    <xsl:template match="item" mode="questionnaire">
        <xsl:param name="level"/>
        <xsl:param name="tid"/>
        <xsl:param name="ted"/>
        <tr xmlns="http://www.w3.org/1999/xhtml">
            <td style="vertical-align: top;">
                <a id="_{$tid}-{replace($ted, '\D', '')}-{@id}"/>
                <table border="0" cellspacing="0" cellpadding="0">
                    <tr>
                        <xsl:call-template name="doIndentLevel">
                            <xsl:with-param name="level" select="$level"/>
                        </xsl:call-template>
                        <td style="vertical-align: top;">
                            <xsl:choose>
                                <xsl:when test="@type = 'group'">
                                    <xsl:call-template name="showIcon">
                                        <xsl:with-param name="which">folder</xsl:with-param>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="showStatusDot">
                                        <xsl:with-param name="status" select="'new'"/>
                                        <xsl:with-param name="title" select="''"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                        <td style="vertical-align: top;">
                            <xsl:call-template name="doDescription">
                                <xsl:with-param name="ns" select="text"/>
                                <xsl:with-param name="shortDesc" select="false()"/>
                            </xsl:call-template>
                        </td>
                    </tr>
                </table>
            </td>
            <td class="card" style="text-align: left;">
                <xsl:variable name="rq">
                    <xsl:choose>
                        <xsl:when test="@required='true'">1</xsl:when>
                        <xsl:otherwise>0</xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <xsl:variable name="rp">
                    <xsl:choose>
                        <xsl:when test="@repeats='true'">*</xsl:when>
                        <xsl:otherwise>1</xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <span>
                    <strong>
                        <xsl:value-of select="local:getCardConf($rq, $rp, (), ())"/>
                    </strong>
                </span>
            </td>
            <td style="vertical-align: top;">
                <xsl:value-of select="@type"/>
            </td>
        </tr>
        <xsl:if test="enableWhen">
            <tr xmlns="http://www.w3.org/1999/xhtml">
                <td/>
                <td colspan="2" style="vertical-align: top;">
                    <xsl:apply-templates select="." mode="enableWhen">
                        <xsl:with-param name="trid" select="$tid"/>
                        <xsl:with-param name="tred" select="$ted"/>
                    </xsl:apply-templates>
                </td>
            </tr>
        </xsl:if>
        <xsl:apply-templates select="item" mode="questionnaire">
            <xsl:with-param name="level" select="$level + 1"/>
            <xsl:with-param name="tid" select="$tid"/>
            <xsl:with-param name="ted" select="$ted"/>
        </xsl:apply-templates>
    </xsl:template>
    <xsl:template match="transaction">
        <xsl:variable name="node-id-parent" select="generate-id(parent::*)"/>
        <xsl:variable name="level" select="count(ancestor::scenario | ancestor::transaction) + 1"/>
        <xsl:variable name="trid" select="self::transaction/@id"/>
        <xsl:variable name="tred" select="self::transaction/@effectiveDate"/>
        <xsl:variable name="transactionAnchor" select="local:doHtmlAnchor($trid, $tred)"/>
        <xsl:variable name="node-id-transaction" select="generate-id(.)"/>
        <xsl:variable name="node-color" select="
                if (self::transaction[@type[. = 'group'] | transaction]) then
                    '#fff7f2'
                else
                    'white'"/>
        <!-- ==== Start of transaction (name, direction icon and id) ==== -->
        <tr data-tt-id="{$node-id-transaction}" data-tt-parent-id="{$node-id-parent}" xmlns="http://www.w3.org/1999/xhtml">
            <td class="columnName" style="background-color: {$node-color};" colspan="3">
                <a name="{$transactionAnchor}"/>
                <xsl:variable name="columnName">
                    <xsl:call-template name="showStatusDot">
                        <xsl:with-param name="status" select="@statusCode"/>
                        <xsl:with-param name="title">
                            <xsl:call-template name="getXFormsLabel">
                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                            </xsl:call-template>
                        </xsl:with-param>
                    </xsl:call-template>
                    <xsl:text>&#160;</xsl:text>
                    <xsl:choose>
                        <xsl:when test="self::transaction[@type[. = 'group'] | transaction]">
                            <xsl:call-template name="showIcon">
                                <xsl:with-param name="which">doublearrow</xsl:with-param>
                                <xsl:with-param name="tooltip">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Group'"/>
                                    </xsl:call-template>
                                </xsl:with-param>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                        </xsl:when>
                        <xsl:when test="self::transaction">
                            <xsl:call-template name="showDirection">
                                <xsl:with-param name="dir" select="@type"/>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                        </xsl:when>
                    </xsl:choose>
                    <xsl:call-template name="doName">
                        <xsl:with-param name="ns" select="name"/>
                    </xsl:call-template>
                </xsl:variable>
                <xsl:choose>
                    <xsl:when test="not($switchCreateTreeTableHtml = 'true')">
                        <table style="float: left;">
                            <tr>
                                <xsl:call-template name="doIndentLevel">
                                    <xsl:with-param name="level" select="$level"/>
                                    <xsl:with-param name="icon" select="false()"/>
                                </xsl:call-template>
                                <td>
                                    <xsl:copy-of select="$columnName"/>
                                </td>
                            </tr>
                        </table>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:copy-of select="$columnName"/>
                    </xsl:otherwise>
                </xsl:choose>
                <div style="float: right;">
                    <xsl:call-template name="doShorthandId">
                        <xsl:with-param name="id" select="$trid"/>
                    </xsl:call-template>
                    <!-- diagrams only for transation leaves -->
                    <xsl:if test="self::transaction[@type[. != 'group']] and string-length($projectRestUriDSDiagram) &gt; 0">
                        <a href="{$projectRestUriDSDiagram}&amp;transactionId={$trid}&amp;transactionEffectiveDate={encode-for-uri($tred)}&amp;format=svg" onclick="target='_blank';" style="margin-left: 5px;">
                            <xsl:attribute name="title">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'showDiagram'"/>
                                </xsl:call-template>
                            </xsl:attribute>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'showDiagram'"/>
                            </xsl:call-template>
                        </a>
                    </xsl:if>
                </div>
            </td>
        </tr>
        <!-- ==== Generics for any transaction type ==== -->
        <!-- id / versionLabel / effectiveDate / expirationDate -->
        <tr data-tt-id="{$node-id-transaction}-meta" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
            <td class="columnDetails"/>
            <td class="columnDetails" colspan="2">
                <table cellpadding="5">
                    <tr>
                        <td class="strong" style="padding-left: 0;">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'EffectiveDate'"/>
                            </xsl:call-template>
                        </td>
                        <td>
                            <xsl:call-template name="showDate">
                                <xsl:with-param name="date" select="$tred"/>
                            </xsl:call-template>
                            <xsl:if test="@versionLabel">
                                <xsl:text> - </xsl:text>
                                <xsl:value-of select="@versionLabel"/>
                            </xsl:if>
                        </td>
                        <xsl:choose>
                            <xsl:when test="@expirationDate">
                                <td class="strong">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'ExpirationDate'"/>
                                    </xsl:call-template>
                                </td>
                                <td>
                                    <xsl:call-template name="showDate">
                                        <xsl:with-param name="date" select="@expirationDate"/>
                                    </xsl:call-template>
                                </td>
                            </xsl:when>
                            <xsl:otherwise>
                                <td>&#160;</td>
                                <td>&#160;</td>
                            </xsl:otherwise>
                        </xsl:choose>
                    </tr>
                    <xsl:if test="@model | @label">
                        <tr>
                            <xsl:if test="@model">
                                <td class="strong" style="padding-left: 0;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Model'"/>
                                    </xsl:call-template>
                                </td>
                                <td colspan="{if (@label) then 1 else 3}">
                                    <xsl:value-of select="@model"/>
                                </td>
                            </xsl:if>
                            <xsl:if test="@label">
                                <td class="strong">
                                    <xsl:if test="not(@model)">
                                        <xsl:attribute name="style">padding-left: 0;</xsl:attribute>
                                    </xsl:if>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Label'"/>
                                    </xsl:call-template>
                                </td>
                                <td colspan="{if (@model) then 1 else 3}">
                                    <xsl:value-of select="normalize-space(@label)"/>
                                </td>
                            </xsl:if>
                        </tr>
                    </xsl:if>
                </table>
            </td>
        </tr>
        <!-- ==== Specifics of description (groups loop into content, other transactions get documented internally) ==== -->
        <xsl:if test="@type[. = 'group'] | transaction">
            <xsl:if test="$switchCreateDocSVG = true()">
                <xsl:variable name="tgId" select="@id"/>
                <!-- If we created at least one SVG, assume its the functional variant -->
                <xsl:if test="count($allSvg/transaction[@id = $tgId]/*) &gt; 0">
                    <tr data-tt-id="{$node-id-transaction}-svg" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
                        <td class="columnDetails"/>
                        <td class="columnDetails tabtab" colspan="2">
                            <img src="{local:doHtmlName('TR',@id,@effectiveDate,'_functional.svg')}">
                                <xsl:attribute name="alt">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'needBrowserWithSvgSupport'"/>
                                    </xsl:call-template>
                                </xsl:attribute>
                            </img>
                        </td>
                    </tr>
                </xsl:if>
            </xsl:if>
        </xsl:if>
        <xsl:if test="desc[.//text()[string-length(normalize-space()) gt 0]]">
            <tr data-tt-id="{$node-id-transaction}-desc" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
                <td class="columnDetails"/>
                <td class="columnDetails strong" style="vertical-align: top; width: 5%;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'Description'"/>
                    </xsl:call-template>
                </td>
                <td class="columnDetails">
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="desc"/>
                    </xsl:call-template>
                </td>
            </tr>
        </xsl:if>
        <xsl:if test="trigger">
            <tr data-tt-id="{$node-id-transaction}-trigger" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
                <td class="columnDetails"/>
                <td class="columnDetails strong" style="vertical-align: top; width: 5%;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'transactionTrigger'"/>
                    </xsl:call-template>
                </td>
                <td class="columnDetails">
                    <xsl:if test="trigger[@id]">
                        <div>
                            <span class="strong">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Id'"/>
                                </xsl:call-template>
                            </span>
                            <xsl:text>: </xsl:text>
                            <xsl:value-of select="trigger/@id[1]"/>
                        </div>
                    </xsl:if>
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="trigger"/>
                    </xsl:call-template>
                </td>
            </tr>
        </xsl:if>
        <xsl:if test="condition">
            <tr data-tt-id="{$node-id-transaction}-condition" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
                <td class="columnDetails"/>
                <td class="columnDetails strong" style="vertical-align: top; width: 5%;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'transactionCondition'"/>
                    </xsl:call-template>
                </td>
                <td class="columnDetails">
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="condition"/>
                    </xsl:call-template>
                </td>
            </tr>
        </xsl:if>
        <xsl:if test="dependencies">
            <tr data-tt-id="{$node-id-transaction}-dependencies" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
                <td class="columnDetails"/>
                <td class="columnDetails strong" style="vertical-align: top; width: 5%;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'transactionDependencies'"/>
                    </xsl:call-template>
                </td>
                <td class="columnDetails">
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="dependencies"/>
                    </xsl:call-template>
                </td>
            </tr>
        </xsl:if>
        <!-- ==== Specifics of description (groups loop into content, other transactions get documented internally) ==== -->
        <xsl:if test="@type[. = 'group'] | transaction">
            <xsl:apply-templates select="transaction"/>
        </xsl:if>
        <xsl:if test="not(@type[. = 'group'] | transaction)">
            <xsl:if test="actors/actor">
                <tr data-tt-id="{$node-id-transaction}-actor" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
                    <td class="columnDetails"/>
                    <td class="columnDetails strong" style="vertical-align: top; width: 5%;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Actor'"/>
                        </xsl:call-template>
                    </td>
                    <td class="columnDetails" style="vertical-align: top;">
                        <xsl:for-each select="actors/actor">
                            <xsl:variable name="aid" select="@id"/>
                            <xsl:variable name="theActor" select="$allActors//actor[@id = $aid]"/>
                            <div>
                                <xsl:attribute name="title">
                                    <xsl:call-template name="doDescription">
                                        <xsl:with-param name="ns" select="$theActor/desc"/>
                                    </xsl:call-template>
                                </xsl:attribute>
                                <xsl:choose>
                                    <xsl:when test="@role = ('sender', 'receiver')">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="concat('actorRole-', @role)"/>
                                        </xsl:call-template>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="@role"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                                <xsl:text> (</xsl:text>
                                <xsl:choose>
                                    <xsl:when test="$theActor/@type = ('device', 'person', 'organization')">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="concat('actorType-', $theActor/@type)"/>
                                        </xsl:call-template>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="$theActor/@type"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                                <xsl:text>) </xsl:text>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="$theActor/name"/>
                                </xsl:call-template>
                            </div>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:if>
            <xsl:for-each select="representingTemplate[@sourceDataset | @ref]">
                <xsl:variable name="dsid" select="@sourceDataset"/>
                <xsl:variable name="dsed" select="
                        if (@sourceDatasetFlexibility castable as xs:dateTime) then
                            @sourceDatasetFlexibility
                        else
                            'dynamic'"/>
                <xsl:variable name="dataset" select="
                        if ($dsid) then
                            local:getDataset($dsid, $dsed)
                        else
                            ()" as="element()?"/>
                <xsl:variable name="tmid" select="@ref"/>
                <xsl:variable name="tmed" select="
                        if (@flexibility castable as xs:dateTime) then
                            @flexibility
                        else
                            ('dynamic')"/>
                <xsl:variable name="rccontent" as="element()?">
                    <xsl:call-template name="getRulesetContent">
                        <xsl:with-param name="ruleset" select="$tmid"/>
                        <xsl:with-param name="flexibility" select="$tmed"/>
                        <xsl:with-param name="sofar" select="()"/>
                    </xsl:call-template>
                </xsl:variable>
                <xsl:variable name="tmname">
                    <xsl:choose>
                        <xsl:when test="$rccontent[@displayName]">
                            <xsl:value-of select="$rccontent/@displayName"/>
                        </xsl:when>
                        <xsl:when test="$rccontent[@name]">
                            <xsl:value-of select="$rccontent/@name"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$tmid"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <xsl:variable name="reptc" select="."/>
                <tr data-tt-id="{$node-id-transaction}-reptemp" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
                    <td class="columnDetails"/>
                    <td class="columnDetails strong" style="vertical-align: top;">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Associations'"/>
                        </xsl:call-template>
                    </td>
                    <td class="columnDetails">
                        <xsl:if test="$dsid">
                            <div>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'dataSetDataSetForTransaction'"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <a href="{local:doHtmlName('TR',$trid,$tred,'.html')}" onclick="target='_blank';">
                                    <xsl:call-template name="doName">
                                        <xsl:with-param name="ns" select="$dataset/name"/>
                                    </xsl:call-template>
                                    <xsl:if test="$dataset[@versionLabel]">
                                        <xsl:text> </xsl:text>
                                        <xsl:value-of select="$dataset/@versionLabel"/>
                                    </xsl:if>
                                    <i>
                                        <xsl:text>(</xsl:text>
                                        <xsl:value-of select="$dsid"/>
                                        <xsl:text> / </xsl:text>
                                        <xsl:choose>
                                            <xsl:when test="$dsed castable as xs:dateTime">
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="$dsed"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:text>)</xsl:text>
                                    </i>
                                </a>
                            </div>
                        </xsl:if>
                        <xsl:if test="$tmid">
                            <div>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Template'"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <a href="{local:doHtmlName('TM',$rccontent/(@id | @ref),$tmed,'.html')}" onclick="target='_blank';">
                                    <xsl:value-of select="$tmname"/>
                                    <xsl:text> </xsl:text>
                                    <i>
                                        <xsl:text>(</xsl:text>
                                        <xsl:if test="$rccontent">
                                            <xsl:value-of select="$rccontent/(@id | @ref)"/>
                                            <xsl:text> / </xsl:text>
                                        </xsl:if>
                                        <xsl:choose>
                                            <xsl:when test="$tmed castable as xs:dateTime">
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="$tmed"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                        <xsl:text>)</xsl:text>
                                    </i>
                                </a>
                            </div>
                            <xsl:if test="$createDefaultInstancesForRepresentingTemplates = true()">
                                <div>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'defaultInstance'"/>
                                    </xsl:call-template>
                                    <xsl:text> </xsl:text>
                                    <a href="{local:doHtmlName('TR',$trid,$tred,'_instance.html')}" onclick="target='_blank';">
                                        <xsl:value-of select="$tmname"/>
                                        <xsl:text> </xsl:text>
                                        <i>
                                            <xsl:text>(</xsl:text>
                                            <xsl:if test="$rccontent">
                                                <xsl:value-of select="$rccontent/(@id | @ref)"/>
                                                <xsl:text> / </xsl:text>
                                            </xsl:if>
                                            <xsl:choose>
                                                <xsl:when test="$tmed castable as xs:dateTime">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="$tmed"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                            <xsl:text>)</xsl:text>
                                        </i>
                                    </a>
                                </div>
                            </xsl:if>
                        </xsl:if>
                    </td>
                </tr>
                <xsl:if test="$dataset and concept">
                    <!-- prepare toggling -->
                    <xsl:variable name="elid" select="local:randomString2(., ())"/>
                    <tr data-tt-id="{$node-id-transaction}-concepts" data-tt-parent-id="{$node-id-transaction}" xmlns="http://www.w3.org/1999/xhtml">
                        <td class="columnDetails"/>
                        <td class="columnDetails" colspan="2" style="vertical-align: top;">
                            <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                <xsl:attribute name="class" select="'toggler'"/>
                                <xsl:attribute name="onclick" select="concat('return toggleZoom(''', $elid, ''',this, ''', $theAssetsDir, ''')')"/>
                            </xsl:if>
                            <span class="strong">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'ContainedConcepts'"/>
                                </xsl:call-template>
                            </span>
                        </td>
                    </tr>
                    <!-- create the data set filtered with concept mentioned in the representingTemplate only -->
                    <xsl:variable name="filtereddataset" as="element()">
                        <xsl:variable name="tmp1" as="element()">
                            <tmp xmlns="">
                                <xsl:apply-templates select="$dataset/concept" mode="filter">
                                    <xsl:with-param name="representingTemplate" select="$reptc"/>
                                </xsl:apply-templates>
                            </tmp>
                        </xsl:variable>
                        <filtereddataset xmlns="">
                            <xsl:apply-templates select="$tmp1/concept" mode="eliminatedHiddenConcepts"/>
                        </filtereddataset>
                    </xsl:variable>
                    <tr data-tt-id="{$node-id-transaction}-contents" data-tt-parent-id="{$node-id-transaction}-concepts" xmlns="http://www.w3.org/1999/xhtml">
                        <td class="columnDetails"/>
                        <td class="columnDetails tabtab" colspan="2" style="vertical-align: top;">
                            <div id="{$elid}">
                                <xsl:if test="not($switchCreateTreeTableHtml = 'true')">
                                    <xsl:attribute name="style" select="'display: none;'"/>
                                </xsl:if>
                                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                                    <tr>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Concept'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'cardinalityConformanceLabel'"/>
                                            </xsl:call-template>
                                        </th>
                                        <th>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'Id'"/>
                                            </xsl:call-template>
                                        </th>
                                    </tr>
                                    <xsl:apply-templates select="$filtereddataset/concept" mode="scenarioview">
                                        <xsl:with-param name="level" select="0"/>
                                        <xsl:with-param name="trid" select="$trid"/>
                                        <xsl:with-param name="tred" select="$tred"/>
                                    </xsl:apply-templates>
                                </table>
                            </div>
                        </td>
                    </tr>
                </xsl:if>
            </xsl:for-each>
        </xsl:if>
    </xsl:template>
    <xsl:template match="concept" mode="scenarioview">
        <xsl:param name="level"/>
        <xsl:param name="trid"/>
        <xsl:param name="tred"/>
        <xsl:variable name="conditionals" as="element()">
            <conditionals xmlns="">
                <!-- copy the extra pre-processed conditions -->
                <xsl:copy-of select="conditionalConcept"/>
            </conditionals>
        </xsl:variable>
        <tr xmlns="http://www.w3.org/1999/xhtml">
            <td style="vertical-align: top;">
                <a id="_{$trid}-{replace($tred, '\D', '')}-{@id}"/>
                <table border="0" cellspacing="0" cellpadding="0">
                    <tr>
                        <xsl:call-template name="doIndentLevel">
                            <xsl:with-param name="level" select="$level"/>
                        </xsl:call-template>
                        <td style="vertical-align: top;">
                            <xsl:choose>
                                <xsl:when test="@type = 'group'">
                                    <xsl:call-template name="showIcon">
                                        <xsl:with-param name="which">folder</xsl:with-param>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:when test="@type = 'item'">
                                    <xsl:call-template name="showStatusDot">
                                        <xsl:with-param name="status" select="@statusCode"/>
                                        <xsl:with-param name="title">
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                    <xsl:text>&#160;</xsl:text>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:value-of select="@type"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                        <td style="vertical-align: top;">
                            <xsl:choose>
                                <xsl:when test="@conformance = 'NP'">
                                    <strike>
                                        <xsl:call-template name="doName">
                                            <xsl:with-param name="ns" select="name"/>
                                        </xsl:call-template>
                                    </strike>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="doName">
                                        <xsl:with-param name="ns" select="name"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                    </tr>
                </table>
            </td>
            <td class="card" style="text-align: left;">
                <b>
                    <xsl:value-of select="local:getCardConf(., $defaultLanguage)"/>
                </b>
            </td>
            <td style="vertical-align: top;">
                <div style="float: right;">
                    <xsl:choose>
                        <xsl:when test="@iddisplay">
                            <xsl:value-of select="@iddisplay"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:call-template name="doShorthandId">
                                <xsl:with-param name="id" select="@id"/>
                            </xsl:call-template>
                        </xsl:otherwise>
                    </xsl:choose>
                    <xsl:if test="string-length($projectRestUriDSDiagram) &gt; 0">
                        <a href="{$projectRestUriDSDiagram}&amp;id={@id|@ref}&amp;effectiveDate{encode-for-uri(@effectiveDate|@flexibility)}&amp;transactionId={$trid}&amp;transactionEffectiveDate={encode-for-uri($tred)}&amp;format=svg" onclick="target='_blank';" style="margin-left: 5px;">
                            <xsl:attribute name="title">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'showDiagram'"/>
                                </xsl:call-template>
                            </xsl:attribute>
                            <xsl:text>link</xsl:text>
                        </a>
                    </xsl:if>
                </div>
            </td>
        </tr>
        <xsl:if test="desc[.//text()[string-length(normalize-space()) gt 0]] | operationalization">
            <tr xmlns="http://www.w3.org/1999/xhtml">
                <td style="vertical-align: top;"/>
                <td colspan="2">
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="desc"/>
                    </xsl:call-template>
                    <xsl:if test="count(operationalization) &gt; 0">
                        <xsl:value-of select="operationalization[1]"/>
                    </xsl:if>
                </td>
            </tr>
        </xsl:if>
        <xsl:if test="$conditionals[*]">
            <tr xmlns="http://www.w3.org/1999/xhtml">
                <td/>
                <td colspan="2" style="vertical-align: top;">
                    <xsl:apply-templates select="$conditionals"/>
                </td>
            </tr>
        </xsl:if>
        <xsl:if test="enableWhen">
            <tr xmlns="http://www.w3.org/1999/xhtml">
                <td/>
                <td colspan="2" style="vertical-align: top;">
                    <xsl:apply-templates select="." mode="enableWhen">
                        <xsl:with-param name="trid" select="$trid"/>
                        <xsl:with-param name="tred" select="$tred"/>
                    </xsl:apply-templates>
                </td>
            </tr>
        </xsl:if>
        <!--<tr>
            <td/>
            <td colspan="2" style="vertical-align: top; border-bottom: 1px solid #ECE9E4;">&#160;</td>
        </tr>-->
        <xsl:apply-templates select="concept" mode="scenarioview">
            <xsl:with-param name="level" select="$level + 1"/>
            <xsl:with-param name="trid" select="$trid"/>
            <xsl:with-param name="tred" select="$tred"/>
        </xsl:apply-templates>
    </xsl:template>
    <xsl:template match="conditionals">
        <xsl:if test="conditionalConcept">
            <table cellspacing="0" cellpadding="0" style="width: 100%; border: 0;" xmlns="http://www.w3.org/1999/xhtml">
                <tr>
                    <td class="tabtab">
                        <!---->
                        <table width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr>
                                <th style="width: 125px;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'cardinalityConformanceLabel'"/>
                                    </xsl:call-template>
                                </th>
                                <th style=" vertical-align: top;">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'conditionLabel'"/>
                                    </xsl:call-template>
                                </th>
                            </tr>
                            <xsl:for-each select="conditionalConcept">
                                <tr>
                                    <td class="card" style="width: 125px; text-align: left;">
                                        <xsl:value-of select="local:getCardConf(., $defaultLanguage)"/>
                                    </td>
                                    <td style=" vertical-align: top;">
                                        <xsl:choose>
                                            <xsl:when test="desc">
                                                <xsl:call-template name="doDescription">
                                                    <xsl:with-param name="ns" select="desc"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:when test="string-length(text()) = 0">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'otherwise'"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="text()"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </td>
                                </tr>
                            </xsl:for-each>
                        </table>
                        <!---->
                    </td>
                </tr>
            </table>
        </xsl:if>
    </xsl:template>
    <xsl:template match="concept" mode="enableWhen">
        <xsl:param name="trid" select="ancestor::transaction[1]/@id"/>
        <xsl:param name="tred" select="ancestor::transaction[1]/@effectiveDate"/>
        <xsl:if test="enableWhen">
            <table cellspacing="0" cellpadding="0" style="width: 100%; border: 0;" xmlns="http://www.w3.org/1999/xhtml">
                <tr>
                    <td class="tabtab">
                        <table width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr>
                                <th>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'enableWhen'"/>
                                    </xsl:call-template>
                                    <xsl:if test="count(enableWhen) gt 1">
                                        <xsl:text>: </xsl:text>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="concat('enableBehavior', (@enableBehavior, 'undefined')[1])"/>
                                        </xsl:call-template>
                                    </xsl:if>
                                </th>
                            </tr>
                            <xsl:for-each select="enableWhen">
                                <xsl:variable name="theReffedConcept" select="../../concept[@id = current()/@question]"/>
                                <tr>
                                    <td style=" vertical-align: top;">
                                        <xsl:text>"</xsl:text>
                                        <a href="#_{$trid}-{replace($tred, '\D', '')}-{@question}" title="{string-join($theReffedConcept/ancestor-or-self::concept/name[1], ' / '), ' - ',@question}"><xsl:value-of select="($theReffedConcept/name, @question)[1]"/></a>
                                        <xsl:text>" </xsl:text>
                                        <xsl:value-of select="@operator"/>
                                        <xsl:text> </xsl:text>
                                        <xsl:apply-templates select="answerBoolean | answerDecimal | answerInteger | answerDate | answerDateTime | answerTime | answerString | answerCoding | answerQuantity" mode="enableWhen"/>
                                    </td>
                                </tr>
                            </xsl:for-each>
                        </table>
                        <!---->
                    </td>
                </tr>
            </table>
        </xsl:if>
    </xsl:template>
    <xsl:template match="answerBoolean | answerDecimal | answerInteger | answerDate | answerDateTime | answerTime | answerString" mode="enableWhen">
        <xsl:value-of select="@value"/>
    </xsl:template>
    <xsl:template match="answerCoding" mode="enableWhen">
        <xsl:variable name="theCSN" as="xs:string*">
            <xsl:call-template name="getIDDisplayName">
                <xsl:with-param name="root" select="@codeSystem"/>
            </xsl:call-template>
        </xsl:variable>
        
        <xsl:value-of select="@code"/>
        <xsl:text> </xsl:text>
        <span title="{$theCSN}"><xsl:value-of select="@codeSystem"/></span>
        <xsl:text> </xsl:text>
        <xsl:value-of select="@displayName"/>
        <xsl:if test="string-length(@canonicalUri) &gt; 0">
            <xsl:text> - FHIR: </xsl:text>
            <i>
                <xsl:value-of select="@canonicalUri"/>
            </i>
        </xsl:if>
    </xsl:template>
    <xsl:template match="answerQuantity" mode="enableWhen">
        <xsl:variable name="theCSN" as="xs:string*">
            <xsl:call-template name="getIDDisplayName">
                <xsl:with-param name="root" select="@codeSystem"/>
            </xsl:call-template>
        </xsl:variable>
        <xsl:value-of select="@comparator"/>
        <xsl:text> </xsl:text>
        <xsl:value-of select="@value"/>
        <xsl:text> </xsl:text>
        <xsl:value-of select="@unit"/>
        <xsl:if test="@code">
            <xsl:text> (</xsl:text>
                <xsl:value-of select="@code"/>
                <xsl:text> </xsl:text>
                <span title="{$theCSN}"><xsl:value-of select="@codeSystem"/></span>
                <xsl:if test="string-length(@canonicalUri) &gt; 0">
                    <xsl:text> - FHIR: </xsl:text>
                    <i>
                        <xsl:value-of select="@canonicalUri"/>
                    </i>
                </xsl:if>
            <xsl:text>)</xsl:text>
        </xsl:if>
    </xsl:template>
    <xsl:template match="attribute" mode="templateparticle">
        <xsl:param name="level"/>
        <!-- item label -->
        <xsl:param name="label" as="element()?"/>
        <xsl:param name="parent-id" as="xs:string?" required="yes"/>
        <xsl:param name="sofar" select="()"/>
        <xsl:param name="templateFormat" as="xs:string?"/>
        <xsl:variable name="itemlabel" as="element()?">
            <xsl:choose>
                <xsl:when test="item[string-length(@label) &gt; 0]">
                    <!-- use current item/@label -->
                    <item xmlns="" label="{replace(string(item/@label), '-', '‑')}">
                        <xsl:copy-of select="item/desc"/>
                    </item>
                </xsl:when>
                <!--<xsl:when test="$label[string-length(@label)&gt;0]">
                    <!-\- use inherited label if present -\->
                    <xsl:copy-of select="$label"/>
                </xsl:when>-->
                <xsl:otherwise>
                    <!-- use leave it empty -->
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <!-- cache attribute isOptional, datatytpe, value, vocabulary -->
        <xsl:variable name="elmDatatype" select="parent::element/@datatype"/>
        <xsl:variable name="isOptional" select="@isOptional = 'true'" as="xs:boolean"/>
        <xsl:variable name="prohibited" select="@prohibited = 'true'" as="xs:boolean"/>
        <xsl:variable name="theDatatype" select="@datatype"/>
        <xsl:variable name="theStrength" select="@strength"/>
        <xsl:variable name="descriptions" select="desc"/>
        <xsl:variable name="vocabulary" select="vocabulary"/>
        <xsl:variable name="property" select="property"/>
        <xsl:for-each select="@name | @classCode | @contextConductionInd | @contextControlCode | @determinerCode | @extension | @independentInd | @institutionSpecified | @inversionInd | @mediaType | @moodCode | @negationInd | @nullFlavor | @operator | @qualifier | @representation | @root | @typeCode | @unit | @use">
            <!-- cache attribute name and value of attribute -->
            <xsl:variable name="theName" select="name()"/>
            <xsl:variable name="theAttName" select="
                    if (name() = 'name') then
                        (.)
                    else
                        (name())"/>
            <xsl:variable name="theAttValue" select="
                    if (name() = 'name') then
                        (../@value)
                    else
                        (.)"/>
            <xsl:variable name="node-id" select="concat(generate-id(.), '-', position())"/>
            <xsl:variable name="theDescription" as="item()*">
                <xsl:choose>
                    <xsl:when test="$prohibited">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'conformanceNotPresent'"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:when test="$theAttName = 'xsi:type' and (starts-with($elmDatatype, 'SD.TEXT') or starts-with($elmDatatype, 'StrucDoc.Text'))">
                        <!-- In DTr1, xsi:type for CDA Narrative Block is 'StrucDoc.Text', but DECOR carries 'SD.TEXT' -->
                        <xsl:value-of select="'StrucDoc.Text'"/>
                    </xsl:when>
                    <xsl:when test="$theAttName = 'xsi:type' and string-length($elmDatatype) > 0">
                        <!-- In DTr1, xsi:type should not be a flavorId, but the real datatype -->
                        <xsl:value-of select="tokenize($elmDatatype, '\.')[1]"/>
                    </xsl:when>
                    <xsl:when test="$theAttName = 'xsi:type' and string-length($theAttValue) > 0">
                        <!-- In DTr1, xsi:type should not be a flavorId, but the real datatype -->
                        <xsl:value-of select="tokenize($theAttValue, '\.')[1]"/>
                    </xsl:when>
                    <xsl:when test="contains($theAttValue, '|')">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'attribShallChoice'"/>
                            <xsl:with-param name="p1" select="$theAttName"/>
                        </xsl:call-template>
                        <ul xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:for-each select="tokenize($theAttValue, '\|')">
                                <li>
                                    <xsl:value-of select="."/>
                                </li>
                            </xsl:for-each>
                        </ul>
                    </xsl:when>
                    <xsl:when test="string-length($theAttValue) > 0">
                        <xsl:value-of select="$theAttValue"/>
                    </xsl:when>
                </xsl:choose>
            </xsl:variable>
            <tr style="vertical-align: top;" data-tt-id="{$node-id}" xmlns="http://www.w3.org/1999/xhtml">
                <xsl:if test="string-length($parent-id) > 0">
                    <xsl:attribute name="data-tt-parent-id" select="$parent-id"/>
                </xsl:if>
                <td class="columnName" style="vertical-align: top;">
                    <xsl:if test="../@id">
                        <a name="{local:doHtmlAnchor(../@id, ())}"/>
                    </xsl:if>
                    <xsl:choose>
                        <xsl:when test="$switchCreateTreeTableHtml = 'true'">
                            <xsl:call-template name="outputPath">
                                <xsl:with-param name="pathname" select="concat('@', $theAttName)"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <table cellpadding="1" style="border: 0;">
                                <tr>
                                    <xsl:call-template name="doIndentLevel">
                                        <xsl:with-param name="level" select="$level"/>
                                    </xsl:call-template>
                                    <td>
                                        <xsl:call-template name="outputPath">
                                            <xsl:with-param name="pathname" select="concat('@', $theAttName)"/>
                                        </xsl:call-template>
                                    </td>
                                </tr>
                            </table>
                        </xsl:otherwise>
                    </xsl:choose>
                </td>
                <td class="dt">
                    <!-- data type if given -->
                    <xsl:if test="string-length($theDatatype) &gt; 0">
                        <xsl:value-of select="$theDatatype"/>
                    </xsl:if>
                </td>
                <!-- cardinality 0..1 or 1..1 or not present -->
                <td class="card">
                    <xsl:choose>
                        <xsl:when test="$prohibited">
                            <xsl:text>0</xsl:text>
                        </xsl:when>
                        <xsl:when test="$isOptional">
                            <span>
                                <strong>
                                    <xsl:value-of select="local:getCardConf('0', '1', (), ())"/>
                                </strong>
                            </span>
                        </xsl:when>
                        <xsl:otherwise>
                            <span>
                                <strong>
                                    <xsl:value-of select="local:getCardConf('1', '1', (), ())"/>
                                </strong>
                            </span>
                        </xsl:otherwise>
                    </xsl:choose>
                </td>
                <!-- conformance -->
                <td>
                    <xsl:choose>
                        <!-- choice -->
                        <xsl:when test="$prohibited">
                            <xsl:text>NP</xsl:text>
                        </xsl:when>
                        <xsl:when test="$theAttName = 'xsi:type' and string-length($elmDatatype) &gt; 0">
                            <xsl:text>F</xsl:text>
                        </xsl:when>
                        <xsl:when test="not($isOptional) and string-length($theAttValue) = 0">
                            <xsl:text>R</xsl:text>
                        </xsl:when>
                        <xsl:when test="string-length($theAttValue) = 0">
                            <xsl:text>&#160;</xsl:text>
                        </xsl:when>
                        <xsl:when test="not(contains($theAttValue, '|')) or not($theName = 'name')">
                            <xsl:text>F</xsl:text>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:text>&#160;</xsl:text>
                        </xsl:otherwise>
                    </xsl:choose>
                </td>
                <td style="vertical-align: top;">
                    <xsl:if test="empty($itemlabel)">
                        <xsl:attribute name="colspan" select="2"/>
                    </xsl:if>
                    <xsl:choose>
                        <xsl:when test="empty($theDescription)">
                            <xsl:call-template name="doDescription">
                                <xsl:with-param name="ns" select="$descriptions"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:copy-of select="$theDescription"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </td>
                <!-- show item label -->
                <xsl:if test="$itemlabel[string-length(@label) &gt; 0]">
                    <td style="background-color: #FFEEEE;">
                        <span>
                            <xsl:attribute name="title">
                                <xsl:call-template name="doDescription">
                                    <xsl:with-param name="ns" select="$itemlabel/desc"/>
                                </xsl:call-template>
                            </xsl:attribute>
                            <!-- shorten the label if it does not contain any spaces -->
                            <xsl:choose>
                                <xsl:when test="matches($itemlabel/@label, '\s')">
                                    <xsl:value-of select="replace($itemlabel/@label, '-', '‑')"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="doLabelMiracle">
                                        <xsl:with-param name="labelcandidate" select="$itemlabel/@label"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </span>
                    </td>
                </xsl:if>
            </tr>
            <xsl:if test="not(empty($theDescription)) and $descriptions">
                <tr style="vertical-align: top;" data-tt-id="{$node-id}-desc" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                    <td style="background-color: white;" colspan="4">&#160;</td>
                    <td colspan="2">
                        <xsl:call-template name="doDescription">
                            <xsl:with-param name="ns" select="$descriptions"/>
                        </xsl:call-template>
                    </td>
                </tr>
            </xsl:if>
            <!-- do vocabulary or name+value of attribute -->
            <xsl:if test="$theName = 'name' and count($vocabulary) &gt; 0">
                <!-- only attributes with a name may have a vocabulary or property -->
                <tr style="vertical-align: top;" data-tt-id="{$node-id}-voc" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                    <td style="background-color: white;" colspan="1">&#160;</td>
                    <td class="conf">
                        <xsl:text>CONF</xsl:text>
                    </td>
                    <td colspan="4">
                        <table width="100%" border="0" cellspacing="2" cellpadding="2">
                            <xsl:for-each select="$vocabulary">
                                <xsl:call-template name="doVocabularyAttributes">
                                    <xsl:with-param name="targetAttributeName" select="$theAttName"/>
                                    <xsl:with-param name="theStrength" select="$theStrength"/>
                                    <xsl:with-param name="templateFormat" select="$templateFormat"/>
                                </xsl:call-template>
                                <xsl:if test="position() != last()">
                                    <tr style="vertical-align: top;">
                                        <td>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'orWord'"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                </xsl:if>
                            </xsl:for-each>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:apply-templates select="$property" mode="templateparticle">
                <xsl:with-param name="parent-id" select="$node-id"/>
                <xsl:with-param name="sofar" select="()"/>
            </xsl:apply-templates>
        </xsl:for-each>
        <xsl:apply-templates select=".[@id]" mode="doConnectedConcepts">
            <xsl:with-param name="parent-id" select="$parent-id"/>
        </xsl:apply-templates>
    </xsl:template>
    <xsl:template match="element | include | choice" mode="templateparticle">
        <xsl:param name="level"/>
        <xsl:param name="parent-id" as="xs:string?" required="yes"/>
        <!-- item label -->
        <xsl:param name="label" as="element()?"/>
        <xsl:param name="inheritedminimumMultiplicity"/>
        <xsl:param name="inheritedmaximumMultiplicity"/>
        <xsl:param name="inheritedConformance"/>
        <xsl:param name="inheritedIsMandatory"/>
        <xsl:param name="sofar" as="xs:string*" required="yes"/>
        <xsl:param name="templateFormat" as="xs:string?"/>
        <xsl:variable name="itemlabel" as="element()?">
            <xsl:choose>
                <xsl:when test="item[string-length(@label) &gt; 0]">
                    <!-- use current item/@label -->
                    <item xmlns="" label="{replace(string(item/@label), '-', '‑')}">
                        <xsl:copy-of select="item/desc"/>
                    </item>
                </xsl:when>
                <xsl:when test="$label[string-length(@label) &gt; 0]">
                    <!-- use inherited label if present -->
                    <xsl:copy-of select="$label"/>
                </xsl:when>
                <xsl:otherwise>
                    <!-- use leave it empty -->
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="node-id" select="generate-id(.)"/>
        <!-- solve references to concept defintions regarding multiplicity, conformance -->
        <!-- AH: deactivated. Not only does reference not exist anymore, it also disregards multiple connections through templateAssociation/concept -->
        <!--xsl:variable name="cid" select="references/@concept"/-->
        <xsl:variable name="cid" select="'bogus'"/>
        <xsl:variable name="tid" select="(ancestor-or-self::template/@id)[last()]"/>
        <xsl:variable name="tnm" select="(ancestor-or-self::template/@name)[last()]"/>
        <xsl:variable name="teff" select="(ancestor-or-self::template/@effectiveDate)[last()]"/>
        <xsl:variable name="templateIsNewest" select="$allTemplates/*/ref[template[@id = $tid][@effectiveDate = $teff]]/@newestForId"/>
        <xsl:variable name="rpt" select="$allScenarios//representingTemplate[@ref = $tid or @ref = $tnm][((not(@flexibility) or @flexibility = 'dynamic') and string($templateIsNewest[1]) = 'true') or @flexibility = $teff]"/>
        <!-- get the cardinalities conformances etc -->
        <xsl:variable name="minimumMultiplicity">
            <xsl:choose>
                <xsl:when test="string-length($inheritedminimumMultiplicity) &gt; 0">
                    <xsl:value-of select="$inheritedminimumMultiplicity"/>
                </xsl:when>
                <xsl:when test="string-length(@minimumMultiplicity) &gt; 0">
                    <xsl:value-of select="@minimumMultiplicity"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="()"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="maximumMultiplicity">
            <xsl:choose>
                <xsl:when test="string-length($inheritedmaximumMultiplicity) &gt; 0">
                    <xsl:value-of select="$inheritedmaximumMultiplicity"/>
                </xsl:when>
                <xsl:when test="string-length(@maximumMultiplicity) &gt; 0">
                    <xsl:value-of select="@maximumMultiplicity"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="()"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="isMandatory">
            <xsl:choose>
                <xsl:when test="string-length($inheritedIsMandatory) gt 0">
                    <xsl:value-of select="$inheritedIsMandatory"/>
                </xsl:when>
                <!--xsl:when test="string-length($rpt/concept[@id=$cid]/@isMandatory)>0">
                    <xsl:value-of select="$rpt/concept[@id=$cid]/@isMandatory"/>
                </xsl:when-->
                <xsl:when test="string-length(@isMandatory) &gt; 0">
                    <xsl:value-of select="@isMandatory"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="()"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="conformance">
            <xsl:choose>
                <xsl:when test="string-length($inheritedConformance) &gt; 0">
                    <xsl:value-of select="$inheritedConformance"/>
                </xsl:when>
                <!--xsl:when test="string-length($rpt/concept[@id=$cid]/@conformance)>0">
                    <xsl:value-of select="$rpt/concept[@id=$cid]/@conformance"/>
                </xsl:when-->
                <xsl:when test="string-length(@conformance) &gt; 0">
                    <xsl:value-of select="@conformance"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="()"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <!-- original multiplicity and conformance, 2DO only if different from inherited cardinality -->
        <xsl:variable name="cardFont">
            <xsl:choose>
                <!-- if calc cards has been shown already, orig cardinality is shown in grey font only... -->
                <!--xsl:when test="count($cid)>0">
                                <xsl:text>gray</xsl:text>
                            </xsl:when-->
                <xsl:when test="0 = 1"/>
                <!-- ...black otherwise -->
                <xsl:otherwise>
                    <xsl:text>color: black;</xsl:text>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <!-- get a possible include candidate -->
        <xsl:variable name="xref">
            <xsl:choose>
                <xsl:when test="self::element[@contains]">
                    <xsl:value-of select="@contains"/>
                </xsl:when>
                <xsl:when test="self::include">
                    <xsl:value-of select="@ref"/>
                </xsl:when>
                <xsl:otherwise>
                    <!-- deprecated -->
                    <xsl:value-of select="@include"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <!-- get the flexibility -->
        <xsl:variable name="xflex" select="@flexibility"/>
        <xsl:variable name="exptmpid" select="@tmid"/>
        <xsl:variable name="exptmpdisplayName" select="@tmdisplayName"/>
        <xsl:variable name="exptmpeffectiveDate" select="@tmeffectiveDate"/>
        <xsl:variable name="exptmpstatusCode" select="@tmstatusCode"/>
        <xsl:variable name="exptmname" select="@tmname"/>
        <xsl:variable name="rccontent" as="element()*">
            <xsl:choose>
                <!-- Template API expands includes inside the include (does not do this for contains) -->
                <xsl:when test="self::include[@linkedartefactmissing = 'false']">
                    <template id="{$exptmpid}" name="{$exptmname}" displayName="{$exptmpdisplayName}" effectiveDate="{$exptmpeffectiveDate}" statusCode="{$exptmpstatusCode}" xmlns="">
                        <xsl:copy-of select="*"/>
                    </template>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:call-template name="getRulesetContent">
                        <xsl:with-param name="ruleset" select="$xref"/>
                        <xsl:with-param name="flexibility" select="$xflex"/>
                        <xsl:with-param name="sofar" select="$sofar"/>
                    </xsl:call-template>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="theDatatype" select="@datatype"/>
        <xsl:variable name="theStrength" select="local:getBindingStrength(@strength)"/>
        <!-- make sure we have the id and not the name for file reference -->
        <xsl:variable name="xtid" select="
                if (string-length($exptmpid) &gt; 0) then
                    $exptmpid
                else
                    $rccontent/@id"/>
        <!-- get the template displayName or name, or the original reference as fallback -->
        <xsl:variable name="xtnm">
            <xsl:choose>
                <xsl:when test="$rccontent[@displayName]">
                    <xsl:value-of select="$rccontent/@displayName"/>
                </xsl:when>
                <xsl:when test="$rccontent/*[@name]">
                    <xsl:value-of select="$rccontent/@name"/>
                </xsl:when>
                <xsl:when test="$exptmpdisplayName">
                    <xsl:value-of select="$exptmpdisplayName"/>
                </xsl:when>
                <xsl:when test="$exptmname">
                    <xsl:value-of select="$exptmname"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$xref"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <!-- this is an include or @contains, show where from included (ref) and create a link to it -->
        <xsl:variable name="ahref">
            <xsl:choose>
                <!--<xsl:when test="string-length($exptmpid)&gt;0">
                    <!-\- never create a link on transformations based on an already expanded template format -\->
                </xsl:when>-->
                <xsl:when test="$xflex castable as xs:dateTime">
                    <xsl:value-of select="local:doHtmlName('TM', $xtid, $xflex, '.html')"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="local:doHtmlName('TM', $xtid, 'dynamic', '.html')"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <!-- 
            output 
            - on elements: name of element and card
            - include references: show from where included and original (black font) or overridden (grey font) cardinality
            - choices
        -->
        <!-- 
            this is an element with @name and @contains, show name and path (with possible where clause)
            or a include or a choice statement
        -->
        <xsl:choose>
            <xsl:when test="self::include">
                <tr class="include" data-tt-id="{$node-id}" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:if test="string-length($parent-id) > 0">
                        <xsl:attribute name="data-tt-parent-id" select="$parent-id"/>
                    </xsl:if>
                    <td style="font-style: italic;" colspan="2">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Included'"/>
                        </xsl:call-template>
                    </td>
                    <!-- show cardinality -->
                    <td class="card">
                        <xsl:choose>
                            <xsl:when test="diff:*[@name = ('maximumMultiplicity', 'minimumMultiplicity')]">
                                <table class="ad-diffblock-horizontal">
                                    <tr class="ad-diff-green">
                                        <td class="ad-diff-topbox">
                                            <span style="{$cardFont}">
                                                <strong>
                                                    <xsl:value-of select="local:getCardConf($minimumMultiplicity, $maximumMultiplicity, (), ())"/>
                                                </strong>
                                            </span>
                                        </td>
                                    </tr>
                                    <xsl:variable name="diffMinimumMultiplicity">
                                        <xsl:choose>
                                            <xsl:when test="diff:*[@name='minimumMultiplicity']/@before">
                                                <xsl:value-of select="diff:*[@name='minimumMultiplicity']/@before"/>
                                            </xsl:when>
                                            <xsl:when test="diff:*[@name = 'minimumMultiplicity']">
                                                <xsl:value-of select="'&omitted;'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="$minimumMultiplicity"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <xsl:variable name="diffMaximumMultiplicity">
                                        <xsl:choose>
                                            <xsl:when test="diff:*[@name='maximumMultiplicity']/@before">
                                                <xsl:value-of select="diff:*[@name='maximumMultiplicity']/@before"/>
                                            </xsl:when>
                                            <xsl:when test="diff:*[@name = 'maximumMultiplicity']">
                                                <xsl:value-of select="'&omitted;'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="$maximumMultiplicity"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <tr class="ad-diff-red">
                                        <td class="ad-diff-bottombox">
                                            <span style="{$cardFont};">
                                                <strong>
                                                    <xsl:value-of select="local:getCardConf($diffMinimumMultiplicity, $diffMaximumMultiplicity, (), ())"/>
                                                </strong>
                                            </span>
                                        </td>
                                    </tr>
                                </table>
                            </xsl:when>
                            <xsl:otherwise>
                                <span style="{$cardFont}">
                                    <strong>
                                        <xsl:value-of select="local:getCardConf($minimumMultiplicity, $maximumMultiplicity, (), ())"/>
                                    </strong>
                                </span>
                            </xsl:otherwise>
                        </xsl:choose>
                    </td>
                    <!-- show conformance -->
                    <td>
                        <xsl:choose>
                            <xsl:when test="diff:*[@name = ('conformance', 'isMandatory')]">
                                <table class="ad-diffblock-horizontal">
                                    <tr class="ad-diff-green">
                                        <td class="ad-diff-topbox">
                                            <span style="{$cardFont}">
                                                <strong>
                                                    <xsl:choose>
                                                        <xsl:when test="string($isMandatory) = 'true'">
                                                            <xsl:text>M</xsl:text>
                                                        </xsl:when>
                                                        <!-- 2DO conditional -->
                                                        <xsl:otherwise>
                                                            <xsl:value-of select="$conformance"/>
                                                        </xsl:otherwise>
                                                    </xsl:choose>
                                                </strong>
                                            </span>
                                        </td>
                                    </tr>
                                    <xsl:variable name="diffIsMandatory">
                                        <xsl:choose>
                                            <xsl:when test="diff:*[@name='isMandatory']/@before">
                                                <xsl:value-of select="diff:*[@name='isMandatory']/@before"/>
                                            </xsl:when>
                                            <xsl:when test="diff:*[@name = 'isMandatory']">
                                                <xsl:value-of select="'&omitted;'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="$isMandatory"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <xsl:variable name="diffConformance">
                                        <xsl:choose>
                                            <xsl:when test="diff:*[@name='conformance']/@before">
                                                <xsl:value-of select="diff:*[@name='conformance']/@before"/>
                                            </xsl:when>
                                            <xsl:when test="diff:*[@name = 'conformance']">
                                                <xsl:value-of select="'&omitted;'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="$conformance"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <tr class="ad-diff-red">
                                        <td class="ad-diff-bottombox">
                                            <span style="{$cardFont}; color:#EEE;">
                                                <strong>
                                                    <xsl:choose>
                                                        <xsl:when test="string($diffIsMandatory) = 'true'">
                                                            <xsl:text>M</xsl:text>
                                                        </xsl:when>
                                                        <xsl:when test="$diffIsMandatory = '&omitted;'">
                                                            <xsl:value-of select="'&omitted;'"/>
                                                        </xsl:when>
                                                        <!-- 2DO conditional -->
                                                        <xsl:otherwise>
                                                            <xsl:value-of select="$diffConformance"/>
                                                        </xsl:otherwise>
                                                    </xsl:choose>
                                                </strong>
                                            </span>
                                        </td>
                                    </tr>
                                </table>
                            </xsl:when>
                            <xsl:otherwise>
                                <span style="color: {$cardFont};">
                                    <strong>
                                        <xsl:choose>
                                            <xsl:when test="string($isMandatory) = 'true'">
                                                <xsl:text>M</xsl:text>
                                            </xsl:when>
                                            <!-- 2DO conditional -->
                                            <xsl:otherwise>
                                                <xsl:value-of select="$conformance"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </strong>
                                </span>
                            </xsl:otherwise>
                        </xsl:choose>
                    </td>
                    <td colspan="2">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'fromLabel'"/>
                        </xsl:call-template>
                        <!-- jump to tab panel with templates to the right vocab and open the accordion -->
                        <a href="{$ahref}" onclick="target='_blank';">
                            <xsl:value-of select="$xtid"/>
                        </a>
                        <xsl:text> </xsl:text>
                        <i>
                            <xsl:value-of select="$xtnm"/>
                        </i>
                        <xsl:text> (</xsl:text>
                        <xsl:choose>
                            <xsl:when test="matches($xflex, '^\d{4}')">
                                <xsl:call-template name="showDate">
                                    <xsl:with-param name="date" select="$xflex"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                        <xsl:text>)</xsl:text>
                        <xsl:if test="desc">
                            <br/>
                            <xsl:call-template name="doDescription">
                                <xsl:with-param name="ns" select="desc"/>
                            </xsl:call-template>
                        </xsl:if>
                    </td>
                </tr>
            </xsl:when>
            <xsl:when test="self::choice">
                <!-- this is a choice with -->
                <tr class="choice" data-tt-id="{$node-id}" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:if test="string-length($parent-id) > 0">
                        <xsl:attribute name="data-tt-parent-id" select="$parent-id"/>
                    </xsl:if>
                    <td style="font-style: italic;" colspan="2">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'choiceLabel'"/>
                        </xsl:call-template>
                    </td>
                    <!-- show cardinality -->
                    <td class="card">
                        <span style="{$cardFont}">
                            <strong>
                                <xsl:value-of select="local:getCardConf($minimumMultiplicity, $maximumMultiplicity, (), ())"/>
                            </strong>
                        </span>
                    </td>
                    <!-- show conformance -->
                    <td/>
                    <td colspan="2">
                        <xsl:if test="desc">
                            <div>
                                <xsl:call-template name="doDescription">
                                    <xsl:with-param name="ns" select="desc"/>
                                </xsl:call-template>
                            </div>
                        </xsl:if>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'choiceElements'"/>
                        </xsl:call-template>
                        <ul>
                            <xsl:for-each select="element | include">
                                <li>
                                    <xsl:choose>
                                        <xsl:when test="self::include | self::element[@contains]">
                                            <xsl:variable name="xref" select="@ref | @contains"/>
                                            <xsl:variable name="xflex">
                                                <xsl:choose>
                                                    <xsl:when test="matches(@flexibility, '^\d{4}')">
                                                        <xsl:value-of select="@flexibility"/>
                                                    </xsl:when>
                                                    <xsl:otherwise>dynamic</xsl:otherwise>
                                                </xsl:choose>
                                            </xsl:variable>
                                            <xsl:variable name="rccontent" as="element()*">
                                                <xsl:choose>
                                                    <!-- Template API expands includes inside the include (does not do this for contains) -->
                                                    <xsl:when test="self::include[@linkedartefactmissing = 'false']">
                                                        <template id="{@tmid}" name="{@tmname}" displayName="{@tmdisplayName}" effectiveDate="{@tmeffectiveDate}" statusCode="{@tmstatusCode}" xmlns="">
                                                            <xsl:copy-of select="*"/>
                                                        </template>
                                                    </xsl:when>
                                                    <xsl:otherwise>
                                                        <xsl:call-template name="getRulesetContent">
                                                            <xsl:with-param name="ruleset" select="$xref"/>
                                                            <xsl:with-param name="flexibility" select="@flexibility"/>
                                                            <xsl:with-param name="sofar" select="()"/>
                                                        </xsl:call-template>
                                                    </xsl:otherwise>
                                                </xsl:choose>
                                            </xsl:variable>
                                            <xsl:variable name="tmid" select="
                                                    if ($rccontent/@id) then
                                                        ($rccontent/@id)[1]
                                                    else
                                                        if (@tmid) then
                                                            (@tmid)
                                                        else
                                                            ($xref)"/>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="
                                                        if (self::include) then
                                                            'including_linked_1'
                                                        else
                                                            ('containment_linked_1')"/>
                                                <xsl:with-param name="p1">
                                                    <xsl:choose>
                                                        <xsl:when test="@name">
                                                            <xsl:value-of select="@name"/>
                                                        </xsl:when>
                                                        <xsl:when test="string-length(($rccontent/element/@name)[1]) &gt; 0">
                                                            <xsl:value-of select="($rccontent/element/@name)[1]"/>
                                                        </xsl:when>
                                                        <xsl:otherwise>
                                                            <!-- undetermined element -->
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'anElement'"/>
                                                            </xsl:call-template>
                                                        </xsl:otherwise>
                                                    </xsl:choose>
                                                </xsl:with-param>
                                            </xsl:call-template>
                                            <a href="{local:doHtmlName('TM',$tmid,$xflex,'.html')}" onclick="target='_blank';">
                                                <xsl:choose>
                                                    <xsl:when test="$rccontent/@id">
                                                        <xsl:value-of select="($rccontent/@id)[1]"/>
                                                    </xsl:when>
                                                    <xsl:otherwise>
                                                        <xsl:value-of select="$xref"/>
                                                    </xsl:otherwise>
                                                </xsl:choose>
                                            </a>
                                            <xsl:text> </xsl:text>
                                            <i>
                                                <xsl:choose>
                                                    <xsl:when test="$rccontent[@displayName]">
                                                        <xsl:value-of select="$rccontent/@displayName"/>
                                                    </xsl:when>
                                                    <xsl:when test="@tmdisplayName">
                                                        <xsl:value-of select="@tmdisplayName"/>
                                                    </xsl:when>
                                                    <xsl:when test="$rccontent[@name]">
                                                        <xsl:value-of select="$rccontent/@name"/>
                                                    </xsl:when>
                                                    <xsl:when test="@tmname">
                                                        <xsl:value-of select="@tmname"/>
                                                    </xsl:when>
                                                </xsl:choose>
                                            </i>
                                            <xsl:text> (</xsl:text>
                                            <xsl:choose>
                                                <xsl:when test="matches(@flexibility, '^\d{4}')">
                                                    <xsl:call-template name="showDate">
                                                        <xsl:with-param name="date" select="@flexibility"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="getMessage">
                                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                            <xsl:text>)</xsl:text>
                                        </xsl:when>
                                        <xsl:when test="self::element">
                                            <xsl:call-template name="getWherePathFromNodeset">
                                                <xsl:with-param name="rccontent" select="."/>
                                                <xsl:with-param name="sofar" select="()"/>
                                                <xsl:with-param name="templateFormat" select="$templateFormat"/>
                                            </xsl:call-template>
                                        </xsl:when>
                                    </xsl:choose>
                                </li>
                            </xsl:for-each>
                        </ul>
                    </td>
                </tr>
            </xsl:when>
            <xsl:otherwise>
                <!-- calculated where clauses are potentially hard to read. Only worry about where path if explicitly stated in the template -->
                <xsl:variable name="wherePath" select="@name" as="xs:string?">
                    <!--<xsl:choose>
                        <xsl:when test="@name and @contains">
                            <xsl:call-template name="getWherePathFromNodeset">
                                <xsl:with-param name="rccontent" select="."/>
                                <xsl:with-param name="sofar" select="()"/>
                                <xsl:with-param name="templateFormat" select="$templateFormat"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="@name"/>
                        </xsl:otherwise>
                    </xsl:choose>-->
                </xsl:variable>
                <xsl:variable name="hasawhere" select="contains($wherePath, '[')"/>
                <!-- this is a normal element -->
                <tr style="vertical-align: top; background-color: #FFEAEA;" data-tt-id="{$node-id}" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:if test="string-length($parent-id) > 0">
                        <xsl:attribute name="data-tt-parent-id" select="$parent-id"/>
                    </xsl:if>
                    <xsl:choose>
                        <xsl:when test="$switchCreateTreeTableHtml = 'true'">
                            <td class="columnName">
                                <xsl:if test="@id">
                                    <a name="{local:doHtmlAnchor(@id, ())}"/>
                                </xsl:if>
                                <xsl:choose>
                                    <xsl:when test="$hasawhere">
                                        <xsl:call-template name="outputPath">
                                            <xsl:with-param name="pathname" select="substring-before($wherePath, '[')"/>
                                        </xsl:call-template>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:call-template name="outputPath">
                                            <xsl:with-param name="pathname" select="$wherePath"/>
                                        </xsl:call-template>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </td>
                        </xsl:when>
                        <xsl:otherwise>
                            <td class="columnName">
                                <table cellpadding="1" style="border: none;">
                                    <xsl:attribute name="class">
                                        <xsl:text>ad-diffblock-horizontal </xsl:text>
                                        <xsl:if test="diff:insert[@type = 'element']">
                                            <xsl:text>ad-diff-xadded</xsl:text>
                                        </xsl:if>
                                        <xsl:if test="diff:delete[@type = 'element']">
                                            <xsl:text>ad-diff-xdeleted</xsl:text>
                                        </xsl:if>
                                    </xsl:attribute>
                                    <tr>
                                        <xsl:call-template name="doIndentLevel">
                                            <xsl:with-param name="level" select="$level"/>
                                        </xsl:call-template>
                                        <td>
                                            <xsl:choose>
                                                <xsl:when test="$hasawhere">
                                                    <xsl:call-template name="outputPath">
                                                        <xsl:with-param name="pathname" select="substring-before($wherePath, '[')"/>
                                                    </xsl:call-template>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:call-template name="outputPath">
                                                        <xsl:with-param name="pathname" select="$wherePath"/>
                                                    </xsl:call-template>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </td>
                                    </tr>
                                </table>
                            </td>
                        </xsl:otherwise>
                    </xsl:choose>
                    <!-- do datatype -->
                    <td class="dt">
                        <xsl:choose>
                            <xsl:when test="diff:*[@name = ('datatype', 'strength')]">
                                <table class="ad-diffblock-horizontal">
                                    <tr class="ad-diff-green">
                                        <td class="ad-diff-topbox">
                                            <strong>
                                                <xsl:choose>
                                                    <xsl:when test="not(@datatype)">
                                                        <xsl:value-of select="'&omitted;'"/>
                                                    </xsl:when>
                                                    <xsl:when test="$templateFormat = 'hl7v3xml1' and $supportedDatatypes/dt[@type = $templateFormat][@name = $theDatatype]">
                                                        <a href="https://docs.art-decor.org/documentation/datatypes/DTr1_{$theDatatype}" alt="">
                                                            <xsl:value-of select="$theDatatype"/>
                                                        </a>
                                                    </xsl:when>
                                                    <xsl:otherwise>
                                                        <xsl:value-of select="$theDatatype"/>
                                                    </xsl:otherwise>
                                                </xsl:choose>
                                            </strong>
                                            <xsl:if test="$theStrength">
                                                <xsl:text> </xsl:text>
                                                <a href="https://docs.art-decor.org/introduction/cda/#binding-strength" alt="">
                                                    <xsl:attribute name="title">
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'strengthIsX'"/>
                                                            <xsl:with-param name="p1" select="$theStrength"/>
                                                        </xsl:call-template>
                                                        <xsl:value-of select="local:showBindingStrength($theStrength, true())"/>
                                                    </xsl:attribute>
                                                    <xsl:value-of select="local:showBindingStrength($theStrength, false())"/>
                                                </a>
                                            </xsl:if>
                                        </td>
                                    </tr>
                                    <xsl:variable name="diffDatatype">
                                        <xsl:choose>
                                            <xsl:when test="diff:*[@name='datatype']/@before">
                                                <xsl:value-of select="diff:*[@name='datatype']/@before"/>
                                            </xsl:when>
                                            <xsl:when test="diff:*[@name = 'datatype']">
                                                <xsl:value-of select="'&omitted;'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="$theDatatype"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <xsl:variable name="diffStrength">
                                        <xsl:choose>
                                            <xsl:when test="diff:*[@name='strength']/@before">
                                                <xsl:value-of select="diff:*[@name='strength']/@before"/>
                                            </xsl:when>
                                            <xsl:when test="diff:*[@name = 'strength']">
                                                <xsl:value-of select="'&omitted;'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="local:getBindingStrength($theStrength)"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <tr class="ad-diff-red">
                                        <td class="ad-diff-bottombox">
                                            <strong>
                                                <xsl:choose>
                                                    <xsl:when test="$templateFormat = 'hl7v3xml1' and $supportedDatatypes/dt[@type = $templateFormat][@name = $diffDatatype]">
                                                        <a href="https://docs.art-decor.org/documentation/datatypes/DTr1_{$diffDatatype}" alt="">
                                                            <xsl:value-of select="$diffDatatype"/>
                                                        </a>
                                                    </xsl:when>
                                                    <xsl:otherwise>
                                                        <xsl:value-of select="$diffDatatype"/>
                                                    </xsl:otherwise>
                                                </xsl:choose>
                                            </strong>
                                            <xsl:if test="@strength">
                                                <xsl:text>&#160;</xsl:text>
                                                <a href="https://docs.art-decor.org/introduction/cda/#binding-strength" alt="">
                                                    <xsl:attribute name="title">
                                                        <xsl:call-template name="getMessage">
                                                            <xsl:with-param name="key" select="'strengthIsX'"/>
                                                            <xsl:with-param name="p1" select="$diffStrength"/>
                                                        </xsl:call-template>
                                                        <xsl:value-of select="local:showBindingStrength($diffStrength, true())"/>
                                                    </xsl:attribute>
                                                    <xsl:value-of select="local:showBindingStrength($diffStrength, false())"/>
                                                </a>
                                            </xsl:if>
                                        </td>
                                    </tr>
                                </table>
                            </xsl:when>
                            <xsl:otherwise>
                                <strong>
                                    <xsl:choose>
                                        <xsl:when test="$templateFormat = 'hl7v3xml1' and $supportedDatatypes/dt[@type = $templateFormat][@name = $theDatatype]">
                                            <a href="https://docs.art-decor.org/documentation/datatypes/DTr1_{$theDatatype}" alt="">
                                                <xsl:value-of select="$theDatatype"/>
                                            </a>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:value-of select="$theDatatype"/>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </strong>
                                <xsl:if test="$theStrength">
                                    <xsl:text>&#160;</xsl:text>
                                    <a href="https://docs.art-decor.org/introduction/cda/#binding-strength" alt="">
                                        <xsl:attribute name="title">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'strengthIsX'"/>
                                                <xsl:with-param name="p1" select="$theStrength"/>
                                            </xsl:call-template>
                                            <xsl:value-of select="local:showBindingStrength($theStrength, true())"/>
                                        </xsl:attribute>
                                        <xsl:value-of select="local:showBindingStrength($theStrength, false())"/>
                                    </a>
                                </xsl:if>
                            </xsl:otherwise>
                        </xsl:choose>
                    </td>
                    <!-- show cardinality -->
                    <td class="card">
                        <span style="{$cardFont}">
                            <strong>
                                <xsl:value-of select="local:getCardConf($minimumMultiplicity, $maximumMultiplicity, (), ())"/>
                            </strong>
                        </span>
                    </td>
                    <!-- show conformance -->
                    <td>
                        <span style="color: {$cardFont}">
                            <strong>
                                <xsl:choose>
                                    <xsl:when test="string($isMandatory) = 'true'">
                                        <xsl:text>M</xsl:text>
                                    </xsl:when>
                                    <!-- 2DO conditional -->
                                    <xsl:otherwise>
                                        <xsl:value-of select="$conformance"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </strong>
                        </span>
                    </td>
                    <!-- show description or contains content -->
                    <td>
                        <xsl:choose>
                            <xsl:when test="desc/diff:update[@type = 'elementcontent']/desc">
                                <table class="ad-diffblock-horizontal">
                                    <tr class="ad-diff-green">
                                        <td class="ad-diff-topbox">
                                            <xsl:call-template name="doDescription">
                                                <xsl:with-param name="ns" select="desc"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                    <tr class="ad-diff-red">
                                        <td class="ad-diff-bottombox">
                                            <xsl:call-template name="doDescription">
                                                <xsl:with-param name="ns" select="desc/diff:update[@type = 'elementcontent']/desc"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                </table>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="doDescription">
                                    <xsl:with-param name="ns" select="desc"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                        <!-- show error message if minimumMultiplicity=0 and isMandatory=true -->
                        <xsl:if test="$minimumMultiplicity = '0' and $isMandatory = 'true'">
                            <p>
                                <table style="border: 0;">
                                    <xsl:call-template name="doMessage">
                                        <xsl:with-param name="level" select="'warning'"/>
                                        <xsl:with-param name="msg">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'issueIfIsMandatoryTrueMinimumMultiplicityGreaterThanZero'"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </table>
                            </p>
                        </xsl:if>
                        <!-- put containment information directly here instead of in an own row (ref old-style ART template viewer -->
                        <xsl:if test="@contains">
                            <xsl:if test="desc">
                                <br/>
                            </xsl:if>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Contains'"/>
                            </xsl:call-template>
                            <xsl:text> </xsl:text>
                            <xsl:choose>
                                <xsl:when test="$rccontent[@id] or @linkedartefactmissing = 'false'">
                                    <a href="{$ahref}" onclick="target='_blank';">
                                        <xsl:value-of select="$xtid"/>
                                    </a>
                                    <xsl:text> </xsl:text>
                                    <i>
                                        <xsl:value-of select="$xtnm"/>
                                    </i>
                                    <xsl:text> (</xsl:text>
                                    <xsl:choose>
                                        <xsl:when test="matches($xflex, '^\d{4}')">
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$xflex"/>
                                            </xsl:call-template>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                            </xsl:call-template>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                    <xsl:text>)</xsl:text>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'cannotFindRuleSet'"/>
                                                    <xsl:with-param name="p1" select="$xref"/>
                                                    <xsl:with-param name="p2">
                                                        <xsl:choose>
                                                            <xsl:when test="$xflex castable as xs:dateTime">
                                                                <xsl:value-of select="$xflex"/>
                                                            </xsl:when>
                                                            <xsl:otherwise>
                                                                <xsl:call-template name="getMessage">
                                                                    <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                                </xsl:call-template>
                                                            </xsl:otherwise>
                                                        </xsl:choose>
                                                    </xsl:with-param>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                            <xsl:if test="diff:update[@name = 'contains']">
                                <table class="ad-diffblock-horizontal">
                                    <tr class="ad-diff-green">
                                        <td class="ad-diff-topbox">
                                            <xsl:value-of select="$xtid"/>
                                        </td>
                                    </tr>
                                    <tr class="ad-diff-red">
                                        <td class="ad-diff-bottombox">
                                            <xsl:value-of select="diff:update[@name='contains']/@before"/>
                                        </td>
                                    </tr>
                                </table>
                            </xsl:if>
                        </xsl:if>
                    </td>
                    <!-- show item label -->
                    <td style="background-color: #FFEEEE;">
                        <xsl:if test="$itemlabel[string-length(@label) &gt; 0]">
                            <span>
                                <xsl:attribute name="title">
                                    <xsl:call-template name="doDescription">
                                        <xsl:with-param name="ns" select="$itemlabel/desc"/>
                                    </xsl:call-template>
                                </xsl:attribute>
                                <!-- shorten the label if it does not contain any spaces -->
                                <xsl:choose>
                                    <xsl:when test="matches($itemlabel/@label, '\s')">
                                        <xsl:value-of select="replace($itemlabel/@label, '-', '‑')"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:call-template name="doLabelMiracle">
                                            <xsl:with-param name="labelcandidate" select="$itemlabel/@label"/>
                                        </xsl:call-template>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </span>
                            <!--
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'xInBraces'"/>
                                <xsl:with-param name="p1" select="$itemlabel"/>
                            </xsl:call-template>
                            -->
                        </xsl:if>
                    </td>
                </tr>
                <xsl:if test="$hasawhere">
                    <tr style="vertical-align: top; background-color: #FFEAEA;" data-tt-id="{$node-id}-where" data-tt-parent-id="{$node-id}" xmlns="http://www.w3.org/1999/xhtml">
                        <td colspan="5" style="background-color: #FFEAEA;">
                            <xsl:call-template name="outputPath">
                                <xsl:with-param name="pathname" select="concat('[', substring-after($wherePath, '['))"/>
                            </xsl:call-template>
                            <!--<table cellpadding="1" style="border: none;">
                                <tr>
                                    <xsl:call-template name="doIndentLevel">
                                        <xsl:with-param name="level" select="$level + 1"/>
                                        <xsl:with-param name="icon" select="false()"/>
                                    </xsl:call-template>
                                    <td>
                                        <xsl:call-template name="outputPath">
                                            <xsl:with-param name="pathname" select="concat('[', substring-after($wherePath, '['))"/>
                                        </xsl:call-template>
                                    </td>
                                </tr>
                            </table>-->
                        </td>
                        <td/>
                    </tr>
                </xsl:if>
            </xsl:otherwise>
        </xsl:choose>
        <!-- do target id (element id as a target for a concept) -->
        <xsl:apply-templates select=".[@id]" mode="doConnectedConcepts">
            <xsl:with-param name="parent-id" select="$node-id"/>
        </xsl:apply-templates>
        <!-- 
            do all subsequent attributes first. 
            If this is an expanded template we could encounter the template contents inline.
             Simple parsing it here would result in duplicates
        -->
        <xsl:apply-templates select="attribute[not(parent::include)]" mode="templateparticle">
            <xsl:with-param name="level" select="$level + 1"/>
            <xsl:with-param name="parent-id" select="$node-id"/>
            <xsl:with-param name="sofar" select="()"/>
            <xsl:with-param name="templateFormat" select="$templateFormat"/>
        </xsl:apply-templates>
        <!-- 
            do constraints.
            If this is an expanded template we could encounter the template contents inline.
            Simple parsing it here would result in duplicates
        -->
        <xsl:apply-templates select="constraint[not(parent::include) or string-length($adram)>0]" mode="templateparticle">
            <xsl:with-param name="parent-id" select="$node-id"/>
            <xsl:with-param name="sofar" select="()"/>
            <xsl:with-param name="templateFormat" select="$templateFormat"/>
        </xsl:apply-templates>
        <!-- do vocabularies -->
        <xsl:choose>
            <xsl:when test="$templateFormat = 'hl7v3xml1' and count(vocabulary) = 1 and not(vocabulary[@valueSet or @domain])">
                <xsl:variable name="theElementIsMandatory" as="xs:boolean">
                    <xsl:choose>
                        <xsl:when test="$isMandatory = 'true'">
                            <xsl:value-of select="true()"/>
                        </xsl:when>
                        <!-- If nullFlavor is prohibited or some attribute (not xsi:type or xsi:nil) is required then the element is effectively mandatory -->
                        <xsl:when test="
                                .[@isMandatory = 'true'] |
                                attribute[@name = 'nullFlavor'][@prohibited = 'true'][not(@value)] |
                                attribute[not(@name = ('nullFlavor', 'xsi:type') or @nullFlavor)][not(@prohibited = 'true')][not(@isOptional = 'true')]">
                            <xsl:value-of select="true()"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="false()"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <xsl:variable name="theCode" select="vocabulary/@code"/>
                <xsl:variable name="theCodeSystem" select="vocabulary/@codeSystem"/>
                <xsl:variable name="theDisplayName">
                    <xsl:choose>
                        <xsl:when test="@displayName">
                            <xsl:value-of select="@displayName"/>
                        </xsl:when>
                        <xsl:when test="@code and @codeSystem">
                            <xsl:value-of select="($allValueSets/*/valueSet/conceptList/concept[@code = $theCode and @codeSystem = $theCodeSystem]/@displayName)[1]"/>
                        </xsl:when>
                    </xsl:choose>
                </xsl:variable>
                <xsl:variable name="vocattrcount" select="count(vocabulary/@*)"/>
                <xsl:for-each select="vocabulary/@*">
                    <!-- cache attribute name and value of attribute -->
                    <xsl:variable name="theAttName" select="name(.)"/>
                    <xsl:variable name="theAttValue" select="."/>
                    <tr style="vertical-align: top;" data-tt-id="{generate-id(.)}-voc-{position()}" xmlns="http://www.w3.org/1999/xhtml">
                        <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                        <td>
                            <xsl:choose>
                                <xsl:when test="$switchCreateTreeTableHtml = 'true'">
                                    <xsl:call-template name="outputPath">
                                        <xsl:with-param name="pathname" select="concat('@', $theAttName)"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <table cellpadding="1" style="border: 0;">
                                        <tr>
                                            <xsl:call-template name="doIndentLevel">
                                                <xsl:with-param name="level" select="$level + 1"/>
                                            </xsl:call-template>
                                            <td>
                                                <!--<xsl:text>@</xsl:text>-->
                                                <xsl:call-template name="outputPath">
                                                    <xsl:with-param name="pathname" select="concat('@', $theAttName)"/>
                                                </xsl:call-template>
                                            </td>
                                        </tr>
                                    </table>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                        <xsl:choose>
                            <xsl:when test="position() = 1">
                                <td rowspan="{$vocattrcount}" class="conf">
                                    <xsl:text>CONF</xsl:text>
                                </td>
                            </xsl:when>
                        </xsl:choose>
                        <!-- cardinality is 1..1 when theElementIsMandatory = true() -->
                        <td class="card">
                            <span style="{$cardFont}">
                                <strong>
                                    <xsl:choose>
                                        <xsl:when test="$theElementIsMandatory">
                                            <xsl:value-of select="local:getCardConf('1', '1', (), ())"/>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:value-of select="local:getCardConf('0', '1', (), ())"/>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </strong>
                            </span>
                        </td>
                        <!-- conformance -->
                        <td>
                            <strong>
                                <xsl:choose>
                                    <!-- choice -->
                                    <xsl:when test="contains($theAttValue, '|')">
                                        <xsl:text>&#160;</xsl:text>
                                    </xsl:when>
                                    <xsl:when test="string-length($theAttValue) &gt; 0">
                                        <xsl:text>F</xsl:text>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:text>&#160;</xsl:text>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </strong>
                        </td>
                        <td colspan="2">
                            <xsl:choose>
                                <!-- choice -->
                                <xsl:when test="contains($theAttValue, '|')">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'attribShallChoice'"/>
                                        <xsl:with-param name="p1" select="$theAttName"/>
                                    </xsl:call-template>
                                    <ul>
                                        <xsl:for-each select="tokenize($theAttValue, '\|')">
                                            <li>
                                                <xsl:value-of select="."/>
                                            </li>
                                        </xsl:for-each>
                                    </ul>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:value-of select="$theAttValue"/>
                                    <xsl:if test="string-length($theDisplayName) &gt; 0 and name(.) = 'code'">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'xInBraces'"/>
                                            <xsl:with-param name="p1" select="$theDisplayName"/>
                                        </xsl:call-template>
                                    </xsl:if>
                                </xsl:otherwise>
                            </xsl:choose>
                            <xsl:if test="$theAttName = 'codeSystem'">
                                <xsl:variable name="theName">
                                    <xsl:call-template name="getIDDisplayName">
                                        <xsl:with-param name="root" select="$theCodeSystem"/>
                                    </xsl:call-template>
                                </xsl:variable>
                                <xsl:if test="string-length($theName) &gt; 0">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'xInBraces'"/>
                                        <xsl:with-param name="p1" select="$theName"/>
                                    </xsl:call-template>
                                </xsl:if>
                            </xsl:if>
                        </td>
                    </tr>
                </xsl:for-each>
            </xsl:when>
            <xsl:when test="vocabulary">
                <tr style="vertical-align: top;" data-tt-id="{generate-id(.)}-voc" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                    <td style="background-color: white;" colspan="1">&#160;</td>
                    <td class="conf">
                        <xsl:text>CONF</xsl:text>
                    </td>
                    <td colspan="4">
                        <table width="100%" border="0" cellspacing="2" cellpadding="2">
                            <xsl:for-each select="vocabulary">
                                <xsl:call-template name="doVocabularyAttributes">
                                    <xsl:with-param name="targetAttributeName" select="'code'"/>
                                    <xsl:with-param name="theStrength" select="$theStrength"/>
                                    <xsl:with-param name="templateFormat" select="$templateFormat"/>
                                </xsl:call-template>
                                <xsl:if test="position() != last()">
                                    <tr style="vertical-align: top;">
                                        <td>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'orWord'"/>
                                            </xsl:call-template>
                                        </td>
                                    </tr>
                                </xsl:if>
                            </xsl:for-each>
                        </table>
                    </td>
                </tr>
            </xsl:when>
        </xsl:choose>
        <!-- do properties. If this is an expanded template we could encounter the template contents inline. Parsing it here would result in duplicates -->
        <xsl:apply-templates select="property[not(parent::include)]" mode="templateparticle">
            <xsl:with-param name="parent-id" select="$node-id"/>
            <xsl:with-param name="sofar" select="()"/>
            <xsl:with-param name="templateFormat" select="$templateFormat"/>
        </xsl:apply-templates>
        <!-- do text -->
        <xsl:if test="text">
            <tr style="vertical-align: top; background-color: #FFEEEE;" data-tt-id="{generate-id(.)}-text" xmlns="http://www.w3.org/1999/xhtml">
                <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                <td style="background-color: white;" colspan="1">&#160;</td>
                <td class="conf">
                    <xsl:text>CONF</xsl:text>
                </td>
                <td colspan="4" style="background-color: white;">
                    <table width="100%" border="0" cellspacing="2" cellpadding="2">
                        <xsl:for-each select="text">
                            <tr style="vertical-align: top;">
                                <td>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'elementContentShallBe'"/>
                                        <xsl:with-param name="p1" select="text()"/>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <xsl:if test="position() != last()">
                                <tr style="vertical-align: top;">
                                    <td>
                                        <xsl:text>-</xsl:text>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'orWord'"/>
                                        </xsl:call-template>
                                        <xsl:text>-</xsl:text>
                                    </td>
                                </tr>
                            </xsl:if>
                        </xsl:for-each>
                    </table>
                </td>
            </tr>
        </xsl:if>
        <!-- @closed, only displaying closed, to avoid clutter... -->
        <xsl:if test="string(@isClosed) = 'true'">
            <tr style="vertical-align: top;" data-tt-id="{generate-id(.)}-closed" xmlns="http://www.w3.org/1999/xhtml">
                <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                <td style="background-color: white;" colspan="1">&#160;</td>
                <td style="vertical-align: top; text-align: left;">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'openClosedLabel'"/>
                    </xsl:call-template>
                </td>
                <td colspan="4" class="tabtab">
                    <xsl:choose>
                        <xsl:when test="string(@isClosed) = 'true'">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'elementIsClosed'"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'elementIsOpen'"/>
                            </xsl:call-template>
                        </xsl:otherwise>
                    </xsl:choose>
                </td>
            </tr>
        </xsl:if>
        <xsl:apply-templates select="example" mode="templateparticle">
            <xsl:with-param name="parent-id" select="$node-id"/>
            <xsl:with-param name="sofar" select="()"/>
            <xsl:with-param name="templateFormat" select="$templateFormat"/>
        </xsl:apply-templates>
        <xsl:if test="diff:delete[@name = ('example')]">
            <tr xmlns="http://www.w3.org/1999/xhtml">
                <td/>
                <td>
                    <table class="ad-diffblock-horizontal">
                        <tr class="ad-diff-red">
                            <td class="ad-diff-singlebox">
                                <strike>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'Example'"/>
                                    </xsl:call-template>
                                </strike>
                            </td>
                        </tr>
                    </table>
                </td>
                <td colspan="2"/>
            </tr>
        </xsl:if>
        <xsl:choose>
            <xsl:when test="self::include">
                <xsl:choose>
                    <xsl:when test="$sofar[. = concat($rccontent/@id, '-', $rccontent/@effectiveDate)]">
                        <tr style="vertical-align: top;" data-tt-id="{generate-id(.)}-incmissing" xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                            <td style="background-color: white;" colspan="1">&#160;</td>
                            <td style="background-color: #FFEEEE;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'tmpInclude'"/>
                                </xsl:call-template>
                            </td>
                            <td style="text-align: left;" colspan="4">
                                <xsl:call-template name="showIcon">
                                    <xsl:with-param name="which">notice</xsl:with-param>
                                </xsl:call-template>
                                <xsl:text>&#160;</xsl:text>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'tmpCircularReferenceNotExpanding'"/>
                                    <xsl:with-param name="p1" select="$rccontent/@id"/>
                                </xsl:call-template>
                            </td>
                        </tr>
                    </xsl:when>
                    <xsl:when test="$rccontent[@id]">
                        <!--<xsl:apply-templates select="$rccontent/*/(element | assert | report | defineVariable | include | choice)" mode="templateparticle">
                            <xsl:with-param name="sofar" select="()"/>
                            <xsl:with-param name="templateFormat" select="$templateFormat"/>
                        </xsl:apply-templates>-->
                        <!-- get a possible new item label -->
                        <xsl:variable name="newitemlabel" as="element()?">
                            <xsl:choose>
                                <xsl:when test="$rccontent/item[string-length(@label) &gt; 0]">
                                    <!-- use current item/@label -->
                                    <item xmlns="" label="{replace(string($rccontent/item/@label), '-', '‑')}">
                                        <xsl:copy-of select="$rccontent/item/desc"/>
                                    </item>
                                </xsl:when>
                                <xsl:when test="$label[string-length(@label) &gt; 0]">
                                    <!-- use inherited label if present -->
                                    <xsl:copy-of select="$label"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <!-- use leave it empty -->
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:variable>
                        <xsl:apply-templates select="$rccontent/(element | attribute | include | choice | assert | report | let | defineVariable | constraint | property)" mode="templateparticle">
                            <xsl:with-param name="level" select="$level"/>
                            <xsl:with-param name="label" select="$newitemlabel"/>
                            <xsl:with-param name="inheritedminimumMultiplicity" select="$minimumMultiplicity"/>
                            <xsl:with-param name="inheritedmaximumMultiplicity" select="$maximumMultiplicity"/>
                            <xsl:with-param name="inheritedConformance" select="$conformance"/>
                            <xsl:with-param name="inheritedIsMandatory" select="$isMandatory"/>
                            <xsl:with-param name="parent-id" select="$node-id"/>
                            <xsl:with-param name="sofar" select="$sofar, concat($rccontent/@id, '-', $rccontent/@effectiveDate)"/>
                            <xsl:with-param name="templateFormat" select="$templateFormat"/>
                        </xsl:apply-templates>
                    </xsl:when>
                    <xsl:otherwise>
                        <tr style="vertical-align: top;" data-tt-id="{generate-id(.)}-incmissing" xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:attribute name="data-tt-parent-id" select="$node-id"/>
                            <td style="background-color: white;" colspan="1">&#160;</td>
                            <td style="background-color: #FFEEEE;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'tmpInclude'"/>
                                </xsl:call-template>
                            </td>
                            <td colspan="4">
                                <table style="border: 0;">
                                    <xsl:call-template name="doMessage">
                                        <xsl:with-param name="msg">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'cannotFindRuleSet'"/>
                                                <xsl:with-param name="p1" select="$xref"/>
                                                <xsl:with-param name="p2">
                                                    <xsl:choose>
                                                        <xsl:when test="$xflex castable as xs:dateTime">
                                                            <xsl:value-of select="$xflex"/>
                                                        </xsl:when>
                                                        <xsl:otherwise>
                                                            <xsl:call-template name="getMessage">
                                                                <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                                            </xsl:call-template>
                                                        </xsl:otherwise>
                                                    </xsl:choose>
                                                </xsl:with-param>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </table>
                            </td>
                        </tr>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:when>
            <xsl:otherwise>
                <xsl:apply-templates select="element | include | choice | assert | report | let | defineVariable" mode="templateparticle">
                    <xsl:with-param name="level" select="$level + 1"/>
                    <xsl:with-param name="label" select="$itemlabel"/>
                    <xsl:with-param name="parent-id" select="$node-id"/>
                    <xsl:with-param name="sofar" select="$sofar"/>
                    <xsl:with-param name="templateFormat" select="$templateFormat"/>
                </xsl:apply-templates>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    <!-- Only render constraint in the requested language. Revert to first other language if missing -->
    <xsl:template match="constraint" mode="templateparticle">
        <xsl:param name="parent-id" as="xs:string?" required="yes"/>
        <xsl:if test=".[@language = $defaultLanguage] or (not(preceding-sibling::constraint) and not(../constraint[@language = $defaultLanguage]))">
            <tr style="vertical-align: top; background-color: #FAFAD2;" data-tt-id="{generate-id(.)}-constr" xmlns="http://www.w3.org/1999/xhtml">
                <xsl:if test="string-length($parent-id) > 0">
                    <xsl:attribute name="data-tt-parent-id" select="$parent-id"/>
                </xsl:if>
                <td style="background-color: white;" colspan="1">&#160;</td>
                <td class="conf">
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'constraintLabel'"/>
                    </xsl:call-template>
                </td>
                <td style="background-color: #FAFAD2;" colspan="4">
                    <xsl:call-template name="doDescription">
                        <xsl:with-param name="ns" select="."/>
                    </xsl:call-template>
                </td>
            </tr>
        </xsl:if>
    </xsl:template>
    <xsl:template match="property" mode="templateparticle">
        <xsl:param name="parent-id" as="xs:string?" required="yes"/>
        <tr style="vertical-align: top;" data-tt-id="{generate-id(.)}-prop" xmlns="http://www.w3.org/1999/xhtml">
            <xsl:if test="string-length($parent-id) > 0">
                <xsl:attribute name="data-tt-parent-id" select="$parent-id"/>
            </xsl:if>
            <td style="background-color: white;" colspan="1">&#160;</td>
            <td class="conf">
                <xsl:text>CONF</xsl:text>
            </td>
            <td colspan="4">
                <table width="100%" border="0" cellspacing="2" cellpadding="2">
                    <xsl:for-each select="@*">
                        <xsl:variable name="tdcontent">
                            <xsl:choose>
                                <xsl:when test="name(.) = 'value'">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'attribIs'"/>
                                            <xsl:with-param name="p1" select="concat('@', name(.))"/>
                                            <xsl:with-param name="p2" select="."/>
                                        </xsl:call-template>
                                    </td>
                                </xsl:when>
                                <xsl:when test="name(.) = 'unit'">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'attribIs'"/>
                                            <xsl:with-param name="p1" select="concat('@', name(.))"/>
                                            <xsl:with-param name="p2" select="."/>
                                        </xsl:call-template>
                                    </td>
                                </xsl:when>
                                <xsl:when test="name(.) = 'currency'">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'attribIs'"/>
                                            <xsl:with-param name="p1" select="name(.)"/>
                                            <xsl:with-param name="p2" select="."/>
                                        </xsl:call-template>
                                    </td>
                                </xsl:when>
                                <xsl:when test="name(.) = 'fractionDigits'">
                                    <td>
                                        <xsl:choose>
                                            <xsl:when test="string-length(.) = 0">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'attribFracDigits'"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:when test="ends-with(., '!$')">
                                                <!-- exact fraction digits -->
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'attribFracDigitsExact'"/>
                                                    <xsl:with-param name="p1" select="substring-before(., '!')"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:when test="ends-with(., '.')">
                                                <!-- max fraction digits -->
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'attribFracDigitsMax'"/>
                                                    <xsl:with-param name="p1" select="substring-before(., '.')"/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <!-- minimum no of digits -->
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'attribFracDigitsMin'"/>
                                                    <xsl:with-param name="p1" select="."/>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </td>
                                </xsl:when>
                                <xsl:when test="name(.) = 'minInclude'">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'attribMinIncludeIs'"/>
                                            <xsl:with-param name="p1" select="name(.)"/>
                                            <xsl:with-param name="p2" select="."/>
                                        </xsl:call-template>
                                    </td>
                                </xsl:when>
                                <xsl:when test="name(.) = 'maxInclude'">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'attribMaxIncludeIs'"/>
                                            <xsl:with-param name="p1" select="name(.)"/>
                                            <xsl:with-param name="p2" select="."/>
                                        </xsl:call-template>
                                    </td>
                                </xsl:when>
                                <xsl:when test="name(.) = 'minLength'">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'attribMinLengthIs'"/>
                                            <xsl:with-param name="p1" select="name(.)"/>
                                            <xsl:with-param name="p2" select="."/>
                                        </xsl:call-template>
                                    </td>
                                </xsl:when>
                                <xsl:when test="name(.) = 'maxLength'">
                                    <td>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'attribMaxLengthIs'"/>
                                            <xsl:with-param name="p1" select="name(.)"/>
                                            <xsl:with-param name="p2" select="."/>
                                        </xsl:call-template>
                                    </td>
                                </xsl:when>
                                <xsl:otherwise>
                                    <td>?</td>
                                    <!-- Don't silently fail this...question marks hard to spot in large specifications -->
                                    <xsl:call-template name="logMessage">
                                        <xsl:with-param name="level" select="$logERROR"/>
                                        <xsl:with-param name="msg">
                                            <xsl:text>+++ Found unrecognized property attribute @</xsl:text>
                                            <xsl:value-of select="name(.)"/>
                                            <xsl:text>="</xsl:text>
                                            <xsl:value-of select="."/>
                                            <xsl:text>" in template id "</xsl:text>
                                            <xsl:value-of select="ancestor::template/@id"/>
                                            <xsl:text>"</xsl:text>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:variable>
                        <xsl:if test="count($tdcontent) &gt; 0">
                            <tr style="vertical-align: top;">
                                <xsl:copy-of select="$tdcontent"/>
                            </tr>
                        </xsl:if>
                        <xsl:if test="position() != last()">
                            <tr style="vertical-align: top;">
                                <td>
                                    <xsl:text>-</xsl:text>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'andWord'"/>
                                    </xsl:call-template>
                                    <xsl:text>-</xsl:text>
                                </td>
                            </tr>
                        </xsl:if>
                    </xsl:for-each>
                </table>
            </td>
        </tr>
    </xsl:template>
    <!-- Render template associations -->
    <xsl:template match="*" mode="doConnectedConcepts">
        <xsl:param name="parent-id" as="xs:string?" required="yes"/>
        <xsl:param name="tid" select="(ancestor-or-self::template/@id)[last()]"/>
        <xsl:param name="tnm" select="(ancestor-or-self::template/@name)[last()]"/>
        <xsl:param name="teff" select="(ancestor-or-self::template/@effectiveDate)[last()]"/>
        <!-- do target id (attribute/element id as a target for a concept) -->
        <xsl:if test="@id">
            <xsl:variable name="theId" select="@id"/>
            <!-- As discussed: this is purely of technical 'under water' nature and holds no information to anyone except perhaps the developer/author of the Decor file -->
            <!--tr>
                <td style="background-color: white;"> </td>
                <td>
                    <xsl:call-template name="getMessage">
                        <xsl:with-param name="key" select="'idAsTarget'"/>
                    </xsl:call-template>
                </td>
                <td colspan="4">
                    <xsl:value-of select="$theId"/>
                </td>
            </tr-->
            <!--
                <origconcept datasetId="2.16.840.1.113883.2.4.3.11.60.20.77.1.2" datasetEffectiveDate="2013-12-01T00:00:00" 
                            ref="2.16.840.1.113883.2.4.3.11.60.20.77.2.2.9528" effectiveDate="2013-12-01T00:00:00" 
                            elementId="2.16.840.1.113883.2.4.3.11.60.20.77.9.135" path="Voorschrift / Patientgegevens" 
                            refdisplay="mp-dataelement800-9528">
                    <concept id="2.16.840.1.113883.2.4.3.11.60.20.77.2.2.9528" effectiveDate="2013-12-01T00:00:00">
                        <name language="nl-NL">Geboortegegevens</name>
                        <name language="en-US">Birth data</name>
                        <desc language="nl-NL">Algemene gegevens over de geboorte van de persoon</desc>
                        <desc language="en-US">General data on the persons' birth</desc>
                    </concept>
                </origconcept>
            -->
            <xsl:variable name="associatedConcepts" select="$allTemplateAssociation/*/templateAssociation[@templateId = $tid][@effectiveDate = $teff]/concept[@elementId = $theId] | ancestor::include[1]/staticAssociations/origconcept[@elementId = $theId] | .[not(ancestor::include)]/ancestor::template/staticAssociations/origconcept[@elementId = $theId]" as="element()*"/>
            <!-- do template associations -->
            <xsl:variable name="renderedConcepts" as="element()*">
                <xsl:for-each-group select="$associatedConcepts" group-by="concat(@ref, @effectiveDate)">
                    <xsl:variable name="deid" select="current-group()[1]/@ref"/>
                    <xsl:variable name="deed" select="current-group()[1]/@effectiveDate"/>
                    <xsl:variable name="concept" select="
                            if (current-group()[self::origconcept]) then
                                (current-group()/self::origconcept[name])[1]
                            else
                                local:getConceptFlat($deid, $deed)" as="element()?"/>
                    <xsl:variable name="dataset" select="local:getDatasetForConcept($deid, $deed)"/>
                    <xsl:variable name="destatus" as="item()?">
                        <xsl:choose>
                            <xsl:when test="@conceptStatusCode">
                                <xsl:value-of select="current-group()[1]/@conceptStatusCode"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$concept/@statusCode"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="deversionlabel">
                        <xsl:choose>
                            <xsl:when test="@conceptVersionLabel">
                                <xsl:value-of select="current-group()[1]/@conceptVersionLabel"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$concept/@versionLabel"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsid">
                        <xsl:choose>
                            <xsl:when test="@datasetId">
                                <xsl:value-of select="current-group()[1]/@datasetId"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$dataset/@id"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsef">
                        <xsl:choose>
                            <xsl:when test="@datasetEffectiveDate">
                                <xsl:value-of select="current-group()[1]/@datasetEffectiveDate"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$dataset/@effectiveDate"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dspath">
                        <xsl:choose>
                            <xsl:when test="@path">
                                <xsl:value-of select="current-group()[1]/@path"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:for-each select="$dataset//concept[@id = $concept/(@id | @ref)][@effectiveDate = $concept/(@effectiveDate | @flexibility)]/ancestor::concept">
                                    <xsl:call-template name="doName">
                                        <!-- will not exist on RetrieveTemplate expanded templates -->
                                        <xsl:with-param name="ns" select="local:getConceptFlat(@id, @effectiveDate)/name"/>
                                    </xsl:call-template>
                                    <xsl:text> / </xsl:text>
                                </xsl:for-each>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsname">
                        <xsl:choose>
                            <xsl:when test="current-group()[1]/datasetName">
                                <xsl:call-template name="doName">
                                    <!-- will not exist on RetrieveTemplate expanded templates -->
                                    <xsl:with-param name="ns" select="current-group()[1]/datasetName"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:when test="@datasetName">
                                <xsl:value-of select="current-group()[1]/@datasetName"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="doName">
                                    <!-- will not exist on RetrieveTemplate expanded templates -->
                                    <xsl:with-param name="ns" select="$dataset/name"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsstatus" as="item()?">
                        <xsl:choose>
                            <xsl:when test="@datasetStatusCode">
                                <xsl:value-of select="current-group()[1]/@datasetStatusCode"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$dataset/@statusCode"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="dsversionlabel">
                        <xsl:choose>
                            <xsl:when test="@datasetVersionLabel">
                                <xsl:value-of select="current-group()[1]/@datasetVersionLabel"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$dataset/@versionLabel"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="deiddisplay">
                        <xsl:choose>
                            <xsl:when test="@refdisplay[not(. = '')]">
                                <xsl:value-of select="current-group()[1]/@refdisplay"/>
                            </xsl:when>
                            <xsl:when test="$concept[@iddisplay]">
                                <xsl:value-of select="($concept/@iddisplay)[1]"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="doShorthandId">
                                    <xsl:with-param name="id" select="$deid"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:variable name="outerposition" select="position()"/>
                    <!--<xsl:if test="empty($concept)">
                        <tr style="background-color: #F4FFF4;" class="missingAssociation">
                            <td style="vertical-align: top; width: 1%;">
                                <xsl:call-template name="showIcon">
                                    <xsl:with-param name="which">target</xsl:with-param>
                                    <xsl:with-param name="tooltip">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'targetOfConceptIds'"/>
                                        </xsl:call-template>
                                    </xsl:with-param>
                                </xsl:call-template>
                            </td>
                            <td style="width: 25%; vertical-align: top;" colspan="3">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'theReferencedConceptAsOfCannotBeFound'"/>
                                    <xsl:with-param name="p1" select="$deid"/>
                                    <xsl:with-param name="p2">
                                        <xsl:call-template name="showDate">
                                            <xsl:with-param name="date" select="$deed"/>
                                        </xsl:call-template>
                                    </xsl:with-param>
                                </xsl:call-template>
                            </td>
                        </tr>
                    </xsl:if>-->
                    <xsl:for-each select="$concept">
                        <tr style="background-color: #F4FFF4;">
                            <td style="width: 25%; vertical-align: top;">
                                <a href="{local:doHtmlName('DS', $projectPrefix, $dsid, $dsef, $deid, $deed, (), (), '.html', 'false')}" onclick="target='_blank';">
                                    <xsl:copy-of select="$deiddisplay"/>
                                </a>
                            </td>
                            <td style="vertical-align: top;">
                                <xsl:if test="string-length($dspath) > 0">
                                    <xsl:attribute name="title" select="$dspath"/>
                                </xsl:if>
                                <xsl:if test="string-length($destatus) > 0">
                                    <xsl:call-template name="showStatusDot">
                                        <xsl:with-param name="status" select="$destatus"/>
                                        <xsl:with-param name="title">
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="$destatus"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                    <xsl:text>&#160;</xsl:text>
                                </xsl:if>
                                <xsl:call-template name="doName">
                                    <xsl:with-param name="ns" select="$concept/name"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <xsl:value-of select="$deversionlabel"/>
                            </td>
                            <td style="width: 35%; vertical-align: top;">
                                <xsl:if test="string-length($dsstatus) > 0">
                                    <xsl:call-template name="showStatusDot">
                                        <xsl:with-param name="status" select="$dsstatus"/>
                                        <xsl:with-param name="title">
                                            <xsl:call-template name="getXFormsLabel">
                                                <xsl:with-param name="simpleTypeKey" select="'ItemStatusCodeLifeCycle'"/>
                                                <xsl:with-param name="simpleTypeValue" select="$dsstatus"/>
                                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                    <xsl:text>&#160;</xsl:text>
                                </xsl:if>
                                <xsl:copy-of select="$dsname"/>
                                <xsl:text> </xsl:text>
                                <xsl:value-of select="$dsversionlabel"/>
                                <xsl:if test="current-group()[1][not(@ident = $projectPrefix)]">
                                    <xsl:text> </xsl:text>
                                    <xsl:apply-templates select="." mode="buildReferenceBox"/>
                                </xsl:if>
                            </td>
                        </tr>
                    </xsl:for-each>
                </xsl:for-each-group>
            </xsl:variable>
            
            <xsl:if test="$renderedConcepts[not(@class = 'missingAssociation')]">
                <tr style="vertical-align: top;" data-tt-id="{generate-id(.)}-assoc" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:if test="string-length($parent-id) > 0">
                        <xsl:attribute name="data-tt-parent-id" select="$parent-id"/>
                    </xsl:if>
                    <td style="background-color: white;">&#160;</td>
                    <td colspan="5" class="tabtab" style="background-color: #F4FFF4;">
                        <!--<xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'targetOfConceptIds'"/>
                        </xsl:call-template>-->
                        <table width="100%" border="0" cellspacing="2" cellpadding="2">
                            <tr>
                                <td style="vertical-align: top; width: 20px" rowspan="{count($renderedConcepts[not(@class = 'missingAssociation')]) + 1}">
                                    <xsl:call-template name="showIcon">
                                        <xsl:with-param name="which">target</xsl:with-param>
                                        <xsl:with-param name="tooltip">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'targetOfConceptIds'"/>
                                            </xsl:call-template>
                                        </xsl:with-param>
                                    </xsl:call-template>
                                </td>
                            </tr>
                            <xsl:copy-of select="$renderedConcepts[not(@class = 'missingAssociation')]"/>
                        </table>
                    </td>
                </tr>
            </xsl:if>
        </xsl:if>
    </xsl:template>
    <xsl:template name="doVocabularyAttributes">
        <xsl:param name="targetAttributeName"/>
        <xsl:param name="theStrength"/>
        <xsl:param name="templateFormat" as="xs:string?"/>
        <xsl:variable name="xvsref" select="@valueSet"/>
        <xsl:variable name="xvsflex" select="
                if (@flexibility) then
                    (@flexibility)
                else
                    ('dynamic')"/>
        <xsl:variable name="xvs">
            <xsl:call-template name="getValueset">
                <xsl:with-param name="reference" select="$xvsref"/>
                <xsl:with-param name="flexibility" select="$xvsflex"/>
            </xsl:call-template>
        </xsl:variable>
        <!-- possible candidates for information in already expanded template format -->
        <xsl:variable name="expvsid" select="@vsid"/>
        <xsl:variable name="expvsdisplayName" select="@vsdisplayName"/>
        <xsl:variable name="expvsname" select="@vsname"/>
        <xsl:for-each select="@*">
            <xsl:variable name="tdcontent" as="element()*">
                <xsl:choose>
                    <xsl:when test="name(.) = 'valueSet'">
                        <xsl:variable name="xvsid">
                            <xsl:choose>
                                <xsl:when test="string-length($expvsid) = 0">
                                    <xsl:value-of select="$xvsref"/>
                                </xsl:when>
                                <xsl:when test="string-length($expvsid) &gt; 0">
                                    <xsl:value-of select="$expvsid"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:value-of select="($xvs/valueSet)[1]/@id"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:variable>
                        <xsl:variable name="xvsname">
                            <xsl:choose>
                                <xsl:when test="string-length(($xvs/valueSet)[1]/@displayName) &gt; 0">
                                    <xsl:value-of select="($xvs/valueSet)[1]/@displayName"/>
                                </xsl:when>
                                <xsl:when test="$expvsdisplayName">
                                    <xsl:value-of select="$expvsdisplayName"/>
                                </xsl:when>
                                <xsl:when test="$expvsname">
                                    <xsl:value-of select="$expvsname"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:value-of select="($xvs/valueSet)[1]/@name"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:variable>
                        <xsl:variable name="ahref">
                            <xsl:choose>
                                <xsl:when test="string-length($xvsid) = 0">
                                    <xsl:value-of select="''"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:value-of select="local:doHtmlName('VS', $xvsid, $xvsflex, '.html')"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:variable>
                        <td xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:variable name="vs" select="."/>
                            <xsl:choose>
                                <xsl:when test="$theStrength = 'preferred'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeFromValueSetPreferred'"/>
                                        <xsl:with-param name="p1" select="$targetAttributeName"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:when test="$theStrength = 'example'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeFromValueSetExample'"/>
                                        <xsl:with-param name="p1" select="$targetAttributeName"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:when test="$theStrength = ('CWE', 'extensible')">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeShouldBeFromValueSet'"/>
                                        <xsl:with-param name="p1" select="$targetAttributeName"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeShallBeFromValueSet'"/>
                                        <xsl:with-param name="p1" select="$targetAttributeName"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                            <!-- link to vocab html, if any -->
                            <xsl:choose>
                                <xsl:when test="string-length($ahref) &gt; 0">
                                    <a href="{$ahref}" onclick="target='_blank';">
                                        <xsl:value-of select="$xvsid"/>
                                    </a>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:value-of select="$xvsid"/>
                                </xsl:otherwise>
                            </xsl:choose>
                            <xsl:text> </xsl:text>
                            <!--xsl:if test="string-length(parent::*/@flexibility)>0">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'flexibilityLabel'"/>
                                </xsl:call-template>
                                <xsl:text> </xsl:text>
                                <xsl:value-of select="parent::*/@flexibility"/>
                            </xsl:if-->
                            <xsl:if test="string-length($xvsname) &gt; 0">
                                <i>
                                    <xsl:value-of select="$xvsname"/>
                                    <xsl:text>&#160;</xsl:text>
                                </i>
                            </xsl:if>
                            <xsl:text>(</xsl:text>
                            <xsl:choose>
                                <xsl:when test="matches($xvsflex, '^\d{4}')">
                                    <xsl:call-template name="showDate">
                                        <xsl:with-param name="date" select="$xvsflex"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'associationDYNAMIC'"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                            <xsl:text>)</xsl:text>
                            <!-- show "value set not found" message if not found -->
                            <xsl:if test="false()">
                                <!-- outcommented temporarily KH 20180418 -->
                                <xsl:if test="count($valueSetReferenceErrors/*/error[@id = $vs]) &gt; 0">
                                    <table style="border: 0;">
                                        <xsl:call-template name="doMessage">
                                            <xsl:with-param name="msg">
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'CannotFindValueSet'"/>
                                                    <xsl:with-param name="p1" select="$vs"/>
                                                </xsl:call-template>
                                            </xsl:with-param>
                                        </xsl:call-template>
                                    </table>
                                </xsl:if>
                            </xsl:if>
                        </td>
                    </xsl:when>
                    <xsl:when test="name(.) = 'code'">
                        <td xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:choose>
                                <xsl:when test="contains($templateFormat, 'hl7v2')">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeorsoShallBeX-v2'"/>
                                        <xsl:with-param name="p1" select="'Identifier'"/>
                                        <xsl:with-param name="p2" select="."/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeorsoShallBeX'"/>
                                        <xsl:with-param name="p1" select="$targetAttributeName"/>
                                        <xsl:with-param name="p2" select="."/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                    </xsl:when>
                    <xsl:when test="name(.) = 'codeSystem' and not(contains($templateFormat, 'hl7v2') and ../@codeSystemName)">
                        <td xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:choose>
                                <xsl:when test="contains($templateFormat, 'hl7v2')">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeorsoShallBeX-v2'"/>
                                        <xsl:with-param name="p1" select="'Name of Coding System'"/>
                                        <xsl:with-param name="p2" select="."/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeSystemShallBeX'"/>
                                        <xsl:with-param name="p1" select="."/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                    </xsl:when>
                    <xsl:when test="name(.) = 'codeSystem'"/>
                    <xsl:when test="name(.) = 'displayName'">
                        <td xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:choose>
                                <xsl:when test="contains($templateFormat, 'hl7v2')">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeorsoShallBeX-v2'"/>
                                        <xsl:with-param name="p1" select="'Text'"/>
                                        <xsl:with-param name="p2" select="."/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'displayNameShallBeX'"/>
                                        <xsl:with-param name="p1" select="."/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                    </xsl:when>
                    <xsl:when test="name(.) = 'codeSystemName'">
                        <td xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:choose>
                                <xsl:when test="contains($templateFormat, 'hl7v2')">
                                    <!-- Assumption: @codeSystemName contains the V2 code system mnemonic -->
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeorsoShallBeX-v2'"/>
                                        <xsl:with-param name="p1" select="'Name of Coding System'"/>
                                        <xsl:with-param name="p2" select="."/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeSystemNameShallBeX'"/>
                                        <xsl:with-param name="p1" select="."/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                    </xsl:when>
                    <xsl:when test="name(.) = 'codeSystemVersion'">
                        <td xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:choose>
                                <xsl:when test="contains($templateFormat, 'hl7v2')">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeorsoShallBeX-v2'"/>
                                        <xsl:with-param name="p1" select="'Coding System Version ID'"/>
                                        <xsl:with-param name="p2" select="."/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'codeSystemVersionShallBeX'"/>
                                        <xsl:with-param name="p1" select="."/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                    </xsl:when>
                    <xsl:when test="name(.) = 'domain'">
                        <td xmlns="http://www.w3.org/1999/xhtml">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'domainIsX'"/>
                                <xsl:with-param name="p1" select="."/>
                            </xsl:call-template>
                        </td>
                    </xsl:when>
                    <!--<xsl:when test="name(.)='strength'">
                        <td>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'strengthIsX'"/>
                                <xsl:with-param name="p1" select="."/>
                            </xsl:call-template>
                        </td>
                    </xsl:when>-->
                    <xsl:when test="name(.) = 'flexibility'">
                        <!-- Skip. Is handled within other when leaves -->
                    </xsl:when>
                    <xsl:when test="name(.) = ('vsid', 'vsname', 'vsdisplayName', 'vseffectiveDate', 'vsstatusCode', 'linkedartefactmissing')">
                        <!-- Relax and skip: as this may be included in already exapnded template representations -->
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:call-template name="logMessage">
                            <xsl:with-param name="level" select="$logERROR"/>
                            <xsl:with-param name="msg">
                                <xsl:text>+++ Found unknown vocabulary attribute </xsl:text>
                                <xsl:value-of select="name(.)"/>
                                <xsl:text>="</xsl:text>
                                <xsl:value-of select="."/>
                                <xsl:text>" template id "</xsl:text>
                                <xsl:value-of select="ancestor::template/@id"/>
                                <xsl:text>"</xsl:text>
                            </xsl:with-param>
                        </xsl:call-template>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:variable>
            <xsl:if test="count($tdcontent) &gt; 0">
                <tr style="vertical-align: top;" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:copy-of select="$tdcontent"/>
                </tr>
            </xsl:if>
        </xsl:for-each>
    </xsl:template>
    <xsl:template match="example" mode="templateparticle">
        <xsl:param name="parent-id" as="xs:string?" required="yes"/>
        <xsl:param name="fWidth" as="xs:string" select="'10em'"/>
        <!-- 
            show example pretty printed
            if parent is template then different td's are used compared to in-element examples
        -->
        <xsl:variable name="expclass">
            <xsl:choose>
                <xsl:when test="@type = 'valid'">
                    <!-- a valid example, render it green -->
                    <xsl:text>explabelgreen</xsl:text>
                </xsl:when>
                <xsl:when test="@type = 'error'">
                    <!-- an invalid example, render it red -->
                    <xsl:text>explabelred</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                    <!-- normal rendering otherwise -->
                    <xsl:text>explabelblue</xsl:text>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="difftrclass">
            <xsl:choose>
                <xsl:when test="diff:insert[@name = ('example')]">
                    <xsl:text>ad-diff-green</xsl:text>
                </xsl:when>
                <xsl:when test="diff:delete[@name = ('example')]">
                    <xsl:text>ad-diff-red</xsl:text>
                </xsl:when>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="difftdclass">
            <xsl:if test="diff:*[@name = ('example')]">
                <xsl:text>ad-diff-singlebox</xsl:text>
            </xsl:if>
        </xsl:variable>
        <xsl:choose>
            <xsl:when test="parent::template">
                <tr class="explabel {$difftrclass}" style="vertical-align: top;" xmlns="http://www.w3.org/1999/xhtml">
                    <th class="{$difftdclass}" style="width: {$fWidth}; text-align: left;">
                        <xsl:choose>
                            <xsl:when test="@type = 'error'">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'ExampleInvalid'"/>
                                </xsl:call-template>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Example'"/>
                                </xsl:call-template>
                            </xsl:otherwise>
                        </xsl:choose>
                    </th>
                    <td class="{$expclass}" colspan="3">
                        <table class="treetable" width="100%" border="0" cellspacing="3" cellpadding="2">
                            <tr class="desclabel" data-tt-id="example-{generate-id(.)}">
                                <td class="togglertreetable" style="height: 1.5em;">
                                    <span class="expcaption">
                                        <xsl:choose>
                                            <xsl:when test="@caption">
                                                <xsl:value-of select="@caption"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:call-template name="getMessage">
                                                    <xsl:with-param name="key" select="'Example'"/>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </span>
                                </td>
                            </tr>
                            <tr data-tt-id="exampleBody-{generate-id(.)}" data-tt-parent-id="example-{generate-id(.)}">
                                <td>
                                    <span style="font-family: Courier, 'Courier New', monospace; ">
                                        <xsl:apply-templates mode="explrender"/>
                                    </span>
                                </td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </xsl:when>
            <xsl:otherwise>
                <tr class="explabel {$difftrclass}" style="vertical-align: top;" data-tt-id="{generate-id(.)}-exmpl" xmlns="http://www.w3.org/1999/xhtml">
                    <xsl:if test="string-length($parent-id) > 0">
                        <xsl:attribute name="data-tt-parent-id" select="$parent-id"/>
                    </xsl:if>
                    <td style="background-color: white;" colspan="1">&#160;</td>
                    <td class="{$difftdclass}">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'Example'"/>
                        </xsl:call-template>
                    </td>
                    <td class="{$expclass}" colspan="4">
                        <xsl:if test="@caption">
                            <div class="expcaption">
                                <xsl:value-of select="@caption"/>
                            </div>
                        </xsl:if>
                        <span style="font-family: Courier, 'Courier New', monospace; ">
                            <xsl:apply-templates mode="explrender"/>
                        </span>
                    </td>
                </tr>
            </xsl:otherwise>
        </xsl:choose>
        <!-- -->
    </xsl:template>
    <xsl:template name="doIndentLevel">
        <xsl:param name="level"/>
        <xsl:param name="icon" select="true()" as="xs:boolean"/>
        <xsl:for-each select="1 to $level - 1">
            <td style="vertical-align: top;" xmlns="http://www.w3.org/1999/xhtml">
                <xsl:call-template name="showIcon">
                    <xsl:with-param name="which">treeblank</xsl:with-param>
                </xsl:call-template>
            </td>
        </xsl:for-each>
        <xsl:if test="$level &gt; 0 and $icon">
            <td style="vertical-align: top;" xmlns="http://www.w3.org/1999/xhtml">
                <xsl:call-template name="showIcon">
                    <xsl:with-param name="which">treetree</xsl:with-param>
                </xsl:call-template>
            </td>
        </xsl:if>
    </xsl:template>
    <xsl:template name="doShowIssues">
        <!-- correction for printing, no extended h3 tag when printing, some more when HTML rendering -->
        <xsl:param name="extendedh3" select="true()"/>
        <!-- 
            create a new nodeset with
            all issues that are not closed and not cancelled wrapped in <open>
            and all other issues wrapped in <closed>
        -->
        <xsl:variable name="newnodeset">
            <issues xmlns="">
                <xsl:for-each select="$allIssues/issue">
                    <xsl:variable name="startDate" select="min((tracking | assignment)/xs:dateTime(@effectiveDate))"/>
                    <xsl:variable name="newestTrackingOrAssignment" select="max((tracking | assignment)/xs:dateTime(@effectiveDate))"/>
                    <xsl:variable name="currentStatus" select="((tracking | assignment)[@statusCode][@effectiveDate = max(../(tracking | assignment)[@statusCode]/xs:dateTime(@effectiveDate))]/@statusCode)[1]"/>
                    <xsl:variable name="newestTracking" select="(tracking[@effectiveDate = max(../tracking/xs:dateTime(@effectiveDate))])[1]"/>
                    <xsl:variable name="newestAssignment" select="(assignment[@effectiveDate = max(../assignment/xs:dateTime(@effectiveDate))])[1]"/>
                    <xsl:variable name="assignedTo" select="$newestAssignment/@to"/>
                    <xsl:variable name="assignedName" select="$newestAssignment/@name"/>
                    <xsl:variable name="elmName">
                        <xsl:choose>
                            <xsl:when test="$currentStatus = ('closed', 'cancelled')">closed</xsl:when>
                            <xsl:otherwise>open</xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:element name="{$elmName}">
                        <xsl:attribute name="currentStatus" select="$currentStatus"/>
                        <xsl:attribute name="startDate" select="$startDate"/>
                        <xsl:attribute name="newestTrackingOrAssignment" select="$newestTrackingOrAssignment"/>
                        <xsl:if test="exists($newestTracking)">
                            <xsl:attribute name="newestTracking" select="$newestTracking/@effectiveDate"/>
                        </xsl:if>
                        <xsl:if test="exists($newestAssignment)">
                            <xsl:attribute name="newestAssignment" select="$newestAssignment/@effectiveDate"/>
                            <xsl:attribute name="assignedTo" select="$assignedTo"/>
                            <xsl:attribute name="assignedName" select="$assignedName"/>
                        </xsl:if>
                        <xsl:copy-of select="."/>
                    </xsl:element>
                </xsl:for-each>
            </issues>
        </xsl:variable>
        <!-- 
            emit issues
            grouped by assigned vs not assigned (existence of assignment)
            sorted by assignee (@to)
            sorted by status code (all trackings)
            1 new open
            2 inprogress feedback 
            3 closed rejected deferred cancelled
            sorted by tracking[1]/@effectiveDate
            sorted by @displayName
        -->
        <table width="100%" border="0" cellspacing="0" cellpadding="0" xmlns="http://www.w3.org/1999/xhtml">
            <!-- issuesAlreadyAssigned -->
            <tr>
                <td colspan="2">
                    <h2>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'issuesAlreadyAssigned'"/>
                        </xsl:call-template>
                    </h2>
                </td>
            </tr>
            <xsl:for-each-group select="$newnodeset/*/open[@assignedTo]" group-by="@assignedTo">
                <tr>
                    <td colspan="2">
                        <h3>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'actuallyAssignedTo'"/>
                                <xsl:with-param name="p1" select="@assignedTo"/>
                                <xsl:with-param name="p2" select="@assignedName"/>
                            </xsl:call-template>
                        </h3>
                    </td>
                </tr>
                <tr>
                    <td colspan="2">
                        <xsl:for-each select="current-group()/issue">
                            <xsl:sort select="parent::*/@currentStatus"/>
                            <xsl:sort select="parent::*/@startDate"/>
                            <xsl:sort select="@displayName"/>
                            <xsl:apply-templates select=".">
                                <xsl:with-param name="extendedh3" select="$extendedh3"/>
                            </xsl:apply-templates>
                        </xsl:for-each>
                    </td>
                </tr>
            </xsl:for-each-group>
            <!-- issuesNotYetAssigned -->
            <tr>
                <td colspan="2">
                    <h2>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'issuesNotYetAssigned'"/>
                        </xsl:call-template>
                    </h2>
                </td>
            </tr>
            <tr>
                <td colspan="2">
                    <xsl:for-each select="$newnodeset/*/open[not(@assignedTo)]/issue">
                        <xsl:sort select="parent::*/@currentStatus"/>
                        <xsl:sort select="parent::*/@startDate"/>
                        <xsl:sort select="@displayName"/>
                        <xsl:apply-templates select=".">
                            <xsl:with-param name="extendedh3" select="$extendedh3"/>
                        </xsl:apply-templates>
                    </xsl:for-each>
                </td>
            </tr>
            <!-- issuesNotActive -->
            <tr>
                <td colspan="2">
                    <h2>
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'issuesNotActive'"/>
                        </xsl:call-template>
                    </h2>
                </td>
            </tr>
            <tr>
                <td colspan="2">
                    <xsl:for-each select="$newnodeset/*/closed/issue">
                        <xsl:sort select="parent::*/@currentStatus"/>
                        <xsl:sort select="@displayName"/>
                        <xsl:apply-templates select=".">
                            <xsl:with-param name="extendedh3" select="$extendedh3"/>
                        </xsl:apply-templates>
                    </xsl:for-each>
                </td>
            </tr>
        </table>
        <table width="100%" border="0" cellspacing="3" cellpadding="2" xmlns="http://www.w3.org/1999/xhtml">
            <xsl:if test="$allIssues/labels/label">
                <tr>
                    <td colspan="2">
                        <h2>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'definedLabels'"/>
                            </xsl:call-template>
                        </h2>
                        <table width="100%">
                            <thead>
                                <tr>
                                    <th>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'definedColor'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'definedCode'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'definedName'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'Description'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'IsUsed'"/>
                                        </xsl:call-template>
                                    </th>
                                </tr>
                            </thead>
                            <tbody>
                                <xsl:for-each select="$allIssues/labels/label">
                                    <xsl:sort select="@code"/>
                                    <xsl:apply-templates select="." mode="definition"/>
                                </xsl:for-each>
                            </tbody>
                        </table>
                    </td>
                </tr>
            </xsl:if>
            <xsl:if test="$allMissingLabels/wrap/missingCode">
                <tr>
                    <td colspan="2">
                        <h2>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'usedButUndefinedLabels'"/>
                            </xsl:call-template>
                        </h2>
                        <table width="100%">
                            <thead>
                                <tr>
                                    <th>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'definedCode'"/>
                                        </xsl:call-template>
                                    </th>
                                    <th>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'Issue'"/>
                                        </xsl:call-template>
                                    </th>
                                </tr>
                            </thead>
                            <tbody>
                                <xsl:for-each select="$allMissingLabels/wrap/missingCode">
                                    <tr>
                                        <td>
                                            <xsl:value-of select="@labelCode"/>
                                        </td>
                                        <td>
                                            <xsl:text>(</xsl:text>
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="@ref"/>
                                            </xsl:call-template>
                                            <xsl:text>) </xsl:text>
                                            <xsl:value-of select="@refName"/>
                                        </td>
                                    </tr>
                                </xsl:for-each>
                            </tbody>
                        </table>
                    </td>
                </tr>
            </xsl:if>
        </table>
    </xsl:template>
    <xsl:template match="issue">
        <!--
            correction for printing, no extended h3 tag when printing 
            new 2014-03-13: don't show issue tracking or assignments with content, only headings and link to live issue in ART.
        -->
        <xsl:param name="extendedh3" select="true()"/>
        <xsl:variable name="issueId" select="@id"/>
        <xsl:variable name="newestTracking" select="max(tracking/xs:dateTime(@effectiveDate))"/>
        <xsl:variable name="newestAssignment" select="max(assignment/xs:dateTime(@effectiveDate))"/>
        <xsl:variable name="newestTrackingOrAssignment" select="max((tracking | assignment)/xs:dateTime(@effectiveDate))"/>
        <xsl:variable name="currentStatus" select="((tracking | assignment)[@statusCode][@effectiveDate = max(../(tracking | assignment)[@statusCode]/xs:dateTime(@effectiveDate))]/@statusCode)[1]"/>
        <xsl:variable name="issuename">
            <xsl:choose>
                <xsl:when test="string-length(@displayName) &gt; 0">
                    <xsl:value-of select="@displayName"/>
                </xsl:when>
                <xsl:otherwise>
                    <span style="color: grey;" xmlns="http://www.w3.org/1999/xhtml">
                        <xsl:call-template name="getMessage">
                            <xsl:with-param name="key" select="'notLabeled'"/>
                        </xsl:call-template>
                    </span>
                </xsl:otherwise>
            </xsl:choose>
            <!--
                <xsl:text> (</xsl:text>
                <xsl:value-of select="@type"/>
                <xsl:text>)</xsl:text>
            -->
        </xsl:variable>
        <a xmlns="http://www.w3.org/1999/xhtml">
            <xsl:attribute name="name">
                <xsl:call-template name="doShorthandId">
                    <xsl:with-param name="id" select="$issueId"/>
                </xsl:call-template>
            </xsl:attribute>
        </a>
        <div class="AccordionPanel" xmlns="http://www.w3.org/1999/xhtml">
            <div class="AccordionPanelTab">
                <div class="acco" onclick="toggleZoomImg(this,'zoomout','{$theAssetsDir}');">
                    <h3>
                        <xsl:variable name="theStatus" select="(tracking[@effectiveDate = $newestTracking]/@statusCode)[last()]"/>
                        <xsl:call-template name="showStatusDot">
                            <xsl:with-param name="status" select="$theStatus"/>
                            <xsl:with-param name="title">
                                <xsl:call-template name="getXFormsLabel">
                                    <xsl:with-param name="simpleTypeKey" select="'IssueStatusCodeLifeCycle'"/>
                                    <xsl:with-param name="simpleTypeValue" select="$theStatus"/>
                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                </xsl:call-template>
                            </xsl:with-param>
                        </xsl:call-template>
                        <xsl:text>&#160;(</xsl:text>
                        <xsl:call-template name="doShorthandId">
                            <xsl:with-param name="id" select="$issueId"/>
                        </xsl:call-template>
                        <xsl:text>)&#160;</xsl:text>
                        <xsl:copy-of select="$issuename"/>
                    </h3>
                    <xsl:if test="$extendedh3 = true()">
                        <!-- is this issue already assigned to someone, if so, get the last assignment -->
                        <xsl:variable name="assignedTo">
                            <xsl:choose>
                                <xsl:when test="string-length(assignment[@effectiveDate = $newestAssignment]/@name) &gt; 0">
                                    <xsl:value-of select="assignment[@effectiveDate = $newestAssignment]/@name"/>
                                </xsl:when>
                                <xsl:when test="string-length(assignment[@effectiveDate = $newestAssignment]/@to) &gt; 0">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'entityNumber'"/>
                                        <xsl:with-param name="p1" select="assignment[@effectiveDate = $newestAssignment]/@to"/>
                                    </xsl:call-template>
                                </xsl:when>
                            </xsl:choose>
                        </xsl:variable>
                        <div class="comment">
                            <xsl:call-template name="showIssueType">
                                <xsl:with-param name="it" select="@type"/>
                            </xsl:call-template>
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'issueIdX'"/>
                                <xsl:with-param name="p1" select="$issueId"/>
                            </xsl:call-template>
                            <br/>
                            <xsl:choose>
                                <xsl:when test="string-length($assignedTo) = 0">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'notAssignedYet'"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'lastAssignedTo'"/>
                                        <xsl:with-param name="p1" select="$assignedTo"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </div>
                    </xsl:if>
                </div>
            </div>
            <div class="AccordionPanelContent">
                <table width="100%" border="0" cellspacing="3" cellpadding="2">
                    <tr>
                        <th style="border:1px solid #C0C0C0; width: 107pt; text-align: left;">
                            <strong>
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Issue'"/>
                                </xsl:call-template>
                            </strong>
                        </th>
                        <td style="background-color: #F6F3EE;" colspan="2">
                            <xsl:copy-of select="$issuename"/>
                        </td>
                    </tr>
                    <tr style="vertical-align: top;">
                        <th style="text-align: left;">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Id'"/>
                            </xsl:call-template>
                        </th>
                        <td style="background-color: #F6F3EE;" colspan="2">
                            <xsl:call-template name="doShorthandId">
                                <xsl:with-param name="id" select="$issueId"/>
                            </xsl:call-template>
                        </td>
                    </tr>
                    <tr style="vertical-align: top;">
                        <th style="text-align: left;">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Type'"/>
                            </xsl:call-template>
                        </th>
                        <td style="background-color: #F6F3EE;" colspan="2">
                            <xsl:call-template name="showIssueType">
                                <xsl:with-param name="it" select="@type"/>
                            </xsl:call-template>
                        </td>
                    </tr>
                    <tr style="vertical-align: top;">
                        <th style="text-align: left;">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Status'"/>
                            </xsl:call-template>
                        </th>
                        <td style="background-color: #F6F3EE;" colspan="2">
                            <xsl:call-template name="showStatusDot">
                                <xsl:with-param name="status" select="$currentStatus"/>
                                <xsl:with-param name="title">
                                    <xsl:call-template name="getXFormsLabel">
                                        <xsl:with-param name="simpleTypeKey" select="'IssueStatusCodeLifeCycle'"/>
                                        <xsl:with-param name="simpleTypeValue" select="$currentStatus"/>
                                        <xsl:with-param name="lang" select="$defaultLanguage"/>
                                    </xsl:call-template>
                                </xsl:with-param>
                            </xsl:call-template>
                            <xsl:text>&#160;</xsl:text>
                            <xsl:call-template name="getXFormsLabel">
                                <xsl:with-param name="simpleTypeKey" select="'IssueStatusCodeLifeCycle'"/>
                                <xsl:with-param name="simpleTypeValue" select="$currentStatus"/>
                                <xsl:with-param name="lang" select="$defaultLanguage"/>
                            </xsl:call-template>
                            <xsl:if test="count(assignment) &gt; 0">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'commaAssigned'"/>
                                </xsl:call-template>
                            </xsl:if>
                        </td>
                    </tr>
                    <tr style="vertical-align: top;">
                        <th style="text-align: left;">
                            <xsl:call-template name="getMessage">
                                <xsl:with-param name="key" select="'Priority'"/>
                            </xsl:call-template>
                        </th>
                        <td style="background-color: #F6F3EE;" colspan="2">
                            <xsl:variable name="iprio" select="tracking[@effectiveDate = $newestTracking]/@priority"/>
                            <xsl:choose>
                                <xsl:when test="$iprio = 'HH'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'highest'"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:when test="$iprio = 'H'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'high'"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:when test="$iprio = 'L'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'low'"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:when test="$iprio = 'LL'">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'lowest'"/>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'normal'"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </td>
                    </tr>
                    <xsl:if test="((tracking | assignment)[@effectiveDate = $newestTrackingOrAssignment]/@labels)[last()] != ''">
                        <tr style="vertical-align: top;">
                            <th style="text-align: left;">
                                <xsl:call-template name="getMessage">
                                    <xsl:with-param name="key" select="'Labels'"/>
                                </xsl:call-template>
                            </th>
                            <td style="background-color: #F6F3EE;" colspan="2">
                                <xsl:call-template name="showLabels">
                                    <xsl:with-param name="labels" select="((tracking | assignment)[@effectiveDate = $newestTrackingOrAssignment]/@labels)[last()]"/>
                                </xsl:call-template>
                            </td>
                        </tr>
                    </xsl:if>
                    <xsl:for-each select="object">
                        <xsl:variable name="objectId" select="@id" as="xs:string"/>
                        <xsl:variable name="objectDate" select="@effectiveDate" as="xs:string?"/>
                        <xsl:variable name="object" select="($allDECOR//*[not(self::object)][@id = $objectId][@effectiveDate = $objectDate or string-length($objectDate) = 0])[1]" as="element()?"/>
                        <tr style="vertical-align: top;">
                            <th style="vertical-align: top; text-align: left;">
                                <xsl:if test="position() = 1">
                                    <xsl:call-template name="getMessage">
                                        <xsl:with-param name="key" select="'objectsLabel'"/>
                                    </xsl:call-template>
                                </xsl:if>
                            </th>
                            <td style="background-color: #F6F3EE;" colspan="2">
                                <xsl:choose>
                                    <xsl:when test="empty($object)">
                                        <span style="color: red; font-style: italic;">
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'compileTimeObjectReferenceTargetMissing'"/>
                                            </xsl:call-template>
                                        </span>
                                        <xsl:text>&#160;</xsl:text>
                                        <xsl:call-template name="doShorthandId">
                                            <xsl:with-param name="id" select="$objectId"/>
                                        </xsl:call-template>
                                        <xsl:if test="$objectDate castable as xs:dateTime">
                                            <xsl:text>&#160;(</xsl:text>
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$objectDate"/>
                                            </xsl:call-template>
                                            <xsl:text>)</xsl:text>
                                        </xsl:if>
                                        <xsl:if test="@name | @displayName">
                                            <xsl:text> </xsl:text>
                                            <xsl:value-of select="(@displayName, @name)[1]"/>
                                        </xsl:if>
                                    </xsl:when>
                                    <xsl:when test="@type = 'VS'">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueOnVS'"/>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <a href="{local:doHtmlName('VS', $objectId, ($objectDate[. castable as xs:dateTime], 'dynamic')[1], '.html')}" onclick="target='_blank';">
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="$object/@id"/>
                                            </xsl:call-template>
                                            <xsl:if test="$objectDate">
                                                <xsl:text>&#160;(</xsl:text>
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="$objectDate"/>
                                                </xsl:call-template>
                                                <xsl:text>)</xsl:text>
                                            </xsl:if>
                                        </a>
                                    </xsl:when>
                                    <xsl:when test="@type = 'CS'">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueOnCS'"/>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <xsl:call-template name="doShorthandId">
                                            <xsl:with-param name="id" select="$object/@id"/>
                                        </xsl:call-template>
                                        <xsl:if test="$objectDate">
                                            <xsl:text>&#160;(</xsl:text>
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$objectDate"/>
                                            </xsl:call-template>
                                            <xsl:text>)</xsl:text>
                                        </xsl:if>
                                    </xsl:when>
                                    <xsl:when test="@type = 'DE'">
                                        <xsl:variable name="dsid" select="$object/ancestor::dataset/@id"/>
                                        <xsl:variable name="dseff" select="$object/ancestor::dataset/@effectiveDate"/>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueOnDE'"/>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <a href="{local:doHtmlName('DS', $projectPrefix, $dsid, $dseff, $objectId, $objectDate, (), (), '.html', 'false')}" onclick="target='_blank';">
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="$objectId"/>
                                            </xsl:call-template>
                                            <xsl:text>&#160;</xsl:text>
                                            <xsl:text>"</xsl:text>
                                            <xsl:call-template name="doName">
                                                <xsl:with-param name="ns" select="$object/name"/>
                                            </xsl:call-template>
                                            <xsl:text>"</xsl:text>
                                        </a>
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'fromDatasetLabel'"/>
                                        </xsl:call-template>
                                        <xsl:text>"</xsl:text>
                                        <xsl:call-template name="doName">
                                            <xsl:with-param name="ns" select="$object/ancestor::dataset/name"/>
                                        </xsl:call-template>
                                        <xsl:text>"</xsl:text>
                                        <xsl:text>&#160;</xsl:text>
                                        <xsl:call-template name="showDate">
                                            <xsl:with-param name="date" select="$object/ancestor::dataset/@effectiveDate"/>
                                        </xsl:call-template>
                                        <xsl:if test="count($object/ancestor::concept) &gt; 0">
                                            <br/>
                                            <xsl:call-template name="getMessage">
                                                <xsl:with-param name="key" select="'pathToElement'"/>
                                            </xsl:call-template>
                                            <xsl:text>: </xsl:text>
                                            <xsl:for-each select="$object/ancestor::concept">
                                                <xsl:call-template name="doName">
                                                    <xsl:with-param name="ns" select="name"/>
                                                </xsl:call-template>
                                                <xsl:if test="position() != last()">
                                                    <xsl:text> / </xsl:text>
                                                </xsl:if>
                                            </xsl:for-each>
                                        </xsl:if>
                                    </xsl:when>
                                    <xsl:when test="@type = 'DS'">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueOnDS'"/>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <a href="{local:doHtmlName('DS',$objectId,$objectDate,'.html')}">
                                            <xsl:call-template name="doName">
                                                <xsl:with-param name="ns" select="$object/name"/>
                                            </xsl:call-template>
                                            <xsl:text>&#160;</xsl:text>
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="$objectId"/>
                                            </xsl:call-template>
                                            <xsl:if test="$objectDate">
                                                <xsl:text>&#160;(</xsl:text>
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="$objectDate"/>
                                                </xsl:call-template>
                                                <xsl:text>)</xsl:text>
                                            </xsl:if>
                                        </a>
                                    </xsl:when>
                                    <xsl:when test="@type = 'TM'">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueOnTM'"/>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <a href="{local:doHtmlName('TM', $objectId, ($objectDate[. castable as xs:dateTime], 'dynamic')[1], '.html')}" onclick="target='_blank';">
                                            <xsl:value-of select="($object/@displayName, $object/@name)[1]"/>
                                            <xsl:text>&#160;</xsl:text>
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="$objectId"/>
                                            </xsl:call-template>
                                            <xsl:if test="$objectDate">
                                                <xsl:text>&#160;(</xsl:text>
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="$objectDate"/>
                                                </xsl:call-template>
                                                <xsl:text>)</xsl:text>
                                            </xsl:if>
                                        </a>
                                    </xsl:when>
                                    <!-- Type is EL is BS. Template element/attribute ids are only unique within the template, 
                                        so any id could return multiple template elements/attributes -->
                                    <xsl:when test="@type = 'EL'">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueOnEL'"/>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <xsl:if test="exists($object/@name)">
                                            <xsl:value-of select="$object/@name"/>
                                            <xsl:text>&#160;</xsl:text>
                                        </xsl:if>
                                        <xsl:call-template name="doShorthandId">
                                            <xsl:with-param name="id" select="$objectId"/>
                                        </xsl:call-template>
                                        <xsl:if test="$objectDate">
                                            <xsl:text>&#160;(</xsl:text>
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$objectDate"/>
                                            </xsl:call-template>
                                            <xsl:text>)</xsl:text>
                                        </xsl:if>
                                    </xsl:when>
                                    <xsl:when test="@type = ('SC', 'TR')">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueOnSC'"/>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <a href="{local:doHtmlName('tabScenarios', (), (), '.html', 'true')}">
                                            <xsl:call-template name="doName">
                                                <xsl:with-param name="ns" select="$object/name"/>
                                            </xsl:call-template>
                                            <xsl:text>&#160;</xsl:text>
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="$objectId"/>
                                            </xsl:call-template>
                                            <xsl:if test="$objectDate castable as xs:dateTime">
                                                <xsl:text>&#160;(</xsl:text>
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="$objectDate"/>
                                                </xsl:call-template>
                                                <xsl:text>)</xsl:text>
                                            </xsl:if>
                                        </a>
                                    </xsl:when>
                                    <xsl:when test="@type = 'IS'">
                                        <xsl:call-template name="getMessage">
                                            <xsl:with-param name="key" select="'issueOnIS'"/>
                                        </xsl:call-template>
                                        <xsl:text>&#160;</xsl:text>
                                        <a href="{local:doHtmlName('tabIssues', $projectPrefix, (), (), $objectId, $objectDate, (), (), '.html', 'true')}" onclick="target='_blank';">
                                            <xsl:if test="exists($object/@displayName)">
                                                <xsl:value-of select="$object/@displayName"/>
                                                <xsl:text>&#160;</xsl:text>
                                            </xsl:if>
                                            <xsl:call-template name="doShorthandId">
                                                <xsl:with-param name="id" select="$objectId"/>
                                            </xsl:call-template>
                                            <xsl:if test="$objectDate">
                                                <xsl:text>&#160;(</xsl:text>
                                                <xsl:call-template name="showDate">
                                                    <xsl:with-param name="date" select="$objectDate"/>
                                                </xsl:call-template>
                                                <xsl:text>)</xsl:text>
                                            </xsl:if>
                                        </a>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="@type"/>
                                        <xsl:text>&#160;</xsl:text>
                                        <xsl:choose>
                                            <xsl:when test="exists($object/name)">
                                                <xsl:call-template name="doName">
                                                    <xsl:with-param name="ns" select="$object/name"/>
                                                </xsl:call-template>
                                                <xsl:text>&#160;</xsl:text>
                                            </xsl:when>
                                            <xsl:when test="exists($object/@displayName)">
                                                <xsl:value-of select="$object/@displayName"/>
                                                <xsl:text>&#160;</xsl:text>
                                            </xsl:when>
                                            <xsl:when test="exists($object/@name)">
                                                <xsl:value-of select="$object/@name"/>
                                                <xsl:text>&#160;</xsl:text>
                                            </xsl:when>
                                        </xsl:choose>
                                        <xsl:call-template name="doShorthandId">
                                            <xsl:with-param name="id" select="$objectId"/>
                                        </xsl:call-template>
                                        <xsl:if test="$objectDate castable as xs:dateTime">
                                            <xsl:text>&#160;(</xsl:text>
                                            <xsl:call-template name="showDate">
                                                <xsl:with-param name="date" select="$objectDate"/>
                                            </xsl:call-template>
                                            <xsl:text>)</xsl:text>
                                        </xsl:if>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </td>
                        </tr>
                    </xsl:for-each>
                    <!-- new 2014-03-13: don't show tracking|assignment with content, only live link to ART -->
                    <!--<xsl:for-each select="tracking | assignment">
                        <xsl:sort select="xs:dateTime(@effectiveDate)" order="descending"/>
                        <xsl:choose>
                            <xsl:when test="name() = 'tracking'">
                                <tr style="vertical-align: top;">
                                    <th style="width: 137px; text-align: center;">
                                        <xsl:call-template name="showIcon">
                                            <xsl:with-param name="which">tracking</xsl:with-param>
                                        </xsl:call-template>
                                    </th>
                                    <th style="text-align: left;" colspan="2">
                                        <xsl:call-template name="showStatusDot">
                                            <xsl:with-param name="status" select="@statusCode"/>
                                            <xsl:with-param name="title">
                                                <xsl:call-template name="getXFormsLabel">
                                                    <xsl:with-param name="simpleTypeKey" select="'IssueStatusCodeLifeCycle'"/>
                                                    <xsl:with-param name="simpleTypeValue" select="@statusCode"/>
                                                    <xsl:with-param name="lang" select="$defaultLanguage"/>
                                                </xsl:call-template>
                                            </xsl:with-