xquery version "1.0";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright
    
    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.
    
    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.
    
    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
    
:)

import module namespace get         = "http://art-decor.org/ns/art-decor-settings" at "../../../art/modules/art-decor-settings.xqm";
import module namespace adserver    = "http://art-decor.org/ns/art-decor-server" at "../../../art/api/api-server-settings.xqm";
import module namespace art         = "http://art-decor.org/ns/art" at "../../../art/modules/art-decor.xqm";

declare namespace svg       = "http://www.w3.org/2000/svg";
declare variable $artDeepLinkServices   := adserver:getServerURLServices();
declare variable $useLocalAssets        := if (request:exists()) then request:get-parameter('useLocalAssets','false') else 'false';
declare variable $download              := if (request:exists()) then request:get-parameter('download','false') else '';
(:When called with localAssets=true then we need relative local paths, else 
    we need our server services URL, but with matching scheme (http or https)
    If the scheme is https and servicesUrl is http this leads to a security 
    problem in some browsers
:)
declare variable $resourcePath          := if ($download='true') then ('https://assets.art-decor.org/ADAR/rv/assets') else if ($useLocalAssets = 'true') then ('../assets') else ('../core/assets');

declare variable $levelSpacing          := 45;

declare variable $withInteractiveCode   := if (request:exists()) then request:get-parameter('interactive','true') else 'true';


(:
    function takes a DECOR concept hierarchy as argument and returns
    the concept hierarchie as a set of nested svg:g elements 
:)
declare %private function local:conceptClassbox($concept as element(), $language as xs:string, $filterStatuses as xs:string*) as element() {
    let $id                 := if ($concept/@ref) then $concept/@ref else $concept/@id
    let $effectiveDate      := if ($concept/@ref) then $concept/@flexibility else $concept/@effectiveDate
    let $name               := $concept/name[@language=$language][1]
    let $maxConceptName     := max($concept/(name[@language=$language][1]|concept/name[@language=$language])/string-length(normalize-space(.)))
    
    let $minClassboxWidth   := 200
    (:
        the width of the 'right hand side'. 
        for transaction based concepts this is valueDomain/@type + card/conf. 
        for dataset based concepts this valueDomain/@type 
    :)
    let $typeCardConfWidth  := if ($concept/ancestor-or-self::dataset[@transactionId]) then (130) else (80)
    let $classboxWidth      :=
        if ($typeCardConfWidth + $maxConceptName*6 > $minClassboxWidth) then
            $typeCardConfWidth + $maxConceptName*6
        else (
            $minClassboxWidth
        )
    let $classboxHeight     := 60 + count($concept/concept[@type='item']) *15
    
    let $detailUri          :=
        for $p in request:get-parameter-names()[not(.=('id','effectiveDate'))] return 
        for $pval in request:get-parameter($p,()) return concat($p,'=',$pval)
    
    return
        (: group for the classbox and children :)
        element {QName('http://www.w3.org/2000/svg','g')} 
        {
            attribute id {concat('_',util:uuid())}
            ,
            (: blue-ish rectangle in the classbox :)
            element {QName('http://www.w3.org/2000/svg','rect')}
            {
                attribute x {0},
                attribute y {0},
                attribute height {$classboxHeight},
                attribute width {$classboxWidth},
                attribute filter {"url(#shadow)"}, 
                (:attribute id {$id},:)
                if ($concept/self::dataset) then (
                    attribute class {'class-box'}
                ) else (
                    attribute class {'class-box class-box-hover'},
                    if ($withInteractiveCode = 'true') then
                        attribute onclick {concat('javascript:location.href=window.location.pathname+''?id=',$id,'&amp;effectiveDate=',$effectiveDate,'&amp;',string-join($detailUri,'&amp;'),'''')}
                    else ()
                )
            }
            ,
            (: line between classbox name and classbox content :)
            element {QName('http://www.w3.org/2000/svg','path')}
            {
                attribute style {'fill:none;stroke:#000000;stroke-width:0.5px;stroke-linecap:butt;stroke-linejoin:miter;stroke-opacity:1'},
                attribute d {concat('m 5, 30  h ',$classboxWidth -10)}
            }
            ,
            (: classbox name (card/conf) :)
            element {QName('http://www.w3.org/2000/svg','text')}
            {
                attribute x {$classboxWidth div 2},
                attribute y {20},
                if ($concept/inherit) then
                    attribute class {'inherit-bold', concat('node-s', $concept/@statusCode)}
                else if ($concept/self::dataset) then
                    attribute class {'dataset-bold', concat('node-s', $concept/@statusCode)}
                else (
                    attribute class {'normal-bold', concat('node-s', $concept/@statusCode)}
                )
                ,
                let $conceptName := normalize-space(string-join($name/text(),''))
                return if (string-length($conceptName)=0) then '&#160;' else $conceptName
                ,
                if ($concept/ancestor::dataset[@transactionId]) then (
                    element {QName('http://www.w3.org/2000/svg','tspan')}
                    {
                        attribute x {$classboxWidth - 5},
                        attribute dy {0},
                        attribute style {'text-anchor:end;font-style:italic;'},
                        ' ',
                        if ($concept/@conformance=('NP','C')) then (
                            string($concept/@conformance)
                        )
                        else (
                            if ($concept/@minimumMultiplicity or $concept/@maximumMultiplicity) then (
                                concat($concept/@minimumMultiplicity,'..',$concept/@maximumMultiplicity)
                            ) else ()
                            ,
                            if ($concept/@isMandatory='true') then 'M' else string($concept/@conformance)
                        )
                    }
                ) else ()
                ,
                element {QName('http://www.w3.org/2000/svg','tspan')}
                {
                    attribute x {5},
                    attribute dy {15},
                    '&#160;'
                }
            }
            ,
            (: classbox content :)
            element {QName('http://www.w3.org/2000/svg','text')}
            {
                attribute x {5},
                attribute y {40},
                attribute class {'normal-start'},
                for $c in $concept/concept[@type='item'][not(@statusCode = $filterStatuses)]
                return
                    element {QName('http://www.w3.org/2000/svg','tspan')}
                    {
                        attribute x {5},
                        attribute dy {15},
                        attribute class {
                            if ($c/inherit) then ('inherit') else (),
                            concat('node-s', $concept/@statusCode)
                        }
                        ,
                        let $conceptName := normalize-space(string-join($c/name[@language=$language],''))
                        return if (string-length($conceptName)=0) then '&#160;' else $conceptName
                        ,
                        element {QName('http://www.w3.org/2000/svg','tspan')}
                        {
                            attribute x {$classboxWidth - 5},
                            attribute dy {0},
                            attribute style {'text-anchor:end;'}
                            ,
                            let $conceptType := normalize-space(string-join($c/valueDomain/@type,''))
                            return if (string-length($conceptType)=0) then '&#160;' else $conceptType
                            ,
                            element {QName('http://www.w3.org/2000/svg','tspan')}
                            {
                                attribute dy {0},
                                attribute style {'font-style:italic;'},
                                ' ',
                                if ($c/@conformance=('NP','C')) then (
                                    string($c/@conformance)
                                )
                                else (
                                    if ($c/@minimumMultiplicity or $c/@maximumMultiplicity) then (
                                        concat($c/@minimumMultiplicity,'..',$c/@maximumMultiplicity)
                                    ) else ()
                                    ,
                                    if ($c/@isMandatory='true') then 'M' else string($c/@conformance)
                                )
                            }
                        }
                    }
                ,
                element {QName('http://www.w3.org/2000/svg','tspan')}
                {
                    attribute x {5},
                    attribute dy {15},
                    '&#160;'
                }
            }
            ,
            (: get classboxes for child concepts :)
            for $g in $concept/concept[@type='group'][not(@statusCode = $filterStatuses)]
            return
            local:conceptClassbox($g,$language,$filterStatuses)
        }
};

declare %private function local:reverseScan($classes as element(), $startDepth as item()) as element() {
    let $current := $classes//svg:g[count(ancestor::svg:g)=$startDepth]
    let $spacing := 10
    return
    <level depth="{$startDepth}" count="{count($current)}" maxHeight="{max($current/svg:rect/@height)}" width="{sum($current/svg:rect/@width)}">
    {
        for $item in $current
        return
            <class id="{$item/@id}" parentId="{$item/parent::svg:g/@id}" height="{$item/svg:rect/@height}" width="{$item/svg:rect/@width + ((count($current)-1) * $spacing)}"/>
    }
    {
        if ($startDepth > 0) then
            local:reverseScan($classes,$startDepth -1)
        else()
    }
    </level>
};

declare %private function local:procesScan($scan as element()) as element() {
    let $level := $scan
    return
    <level depth="{$level/@depth}" count="{$level/@count}" maxHeight="{$level/@maxHeight}" width="{$level/@width}">
    {
        for $class in $level/class
        let $id := $class/@id
        return
            <class id="{$class/@id}" parentId="{$class/@parentId}" height="{$class/@height}" width="{if (sum($class/preceding::*[@parentId=$id]/@width)>$class/@width) then sum($class/preceding::*[@parentId=$id]/@width) else ($class/@width)}"/>
    }
    {
        if ($level/level) then
            local:procesScan($level/level)
        else()
    }
    </level>
};

declare %private function local:positionClasses($classes as element(), $processedScan as element(), $de-parent-id as xs:string?, $de-parent-ed as xs:string?, $atTopLevel as xs:boolean) as element() {
    let $id             := $classes/@id
    
    let $parentId       := $processedScan//class[@id=$id]/@parentId
    let $parentWidth    := $classes/ancestor::svg:g[@id=$parentId]/svg:rect/@width div 2
    let $parentHeight   := $classes/ancestor::svg:g[@id=$parentId]/svg:rect/@height
    let $level          := $processedScan//class[@id=$id]/parent::level/@depth
    let $xShift         := 
        if ($level=0) then
            ($processedScan//class[@id=$id]/@width div 2) - ($classes/svg:rect/@width div 2)
        else( 
            ($processedScan//class[@id=$parentId]/@width div -2) + ($processedScan//class[@id=$id]/@width div 2) - ($classes/svg:rect/@width div 2) + $parentWidth + sum($processedScan//class[@id=$id]/preceding-sibling::class[@parentId=$parentId]/@width)
        )
    let $xPath          :=
        if ($level>0) then
            $xShift + (($classes/svg:rect/@width div 2) - $parentWidth)
        else()
    let $yShift         :=
        if ($level=0) then
            $levelSpacing
        else(
            $processedScan//level[@depth=$level -1]/@maxHeight + $levelSpacing
        )
    let $startnode      :=
        if ($level=0 and not($atTopLevel)) then (
            local:getCircle($de-parent-id, $de-parent-ed, ($classes/svg:rect/@width div 2),($levelSpacing * -1) + 20)
        ) else()
    let $path           :=
        if ($level>0 or ($level=0 and not($atTopLevel))) then
            element {QName('http://www.w3.org/2000/svg','path')}
            {
                attribute style {'fill:none;stroke:#000000;stroke-width:1px;'},
                attribute d {concat('m ',($classes/svg:rect/@width div 2),', 0 ','v -',($levelSpacing div 3),
                    let $hVal := $xPath * -1
                    return if ($hVal castable as xs:integer) then (concat(' h ',$hVal)) else ()
                    ,
                    let $vVal := ($yShift - $parentHeight - ($levelSpacing div 3)) * -1 
                    return if ($vVal castable as xs:integer) then (concat(' v ',$vVal)) else ())}
            }
        else()
    return
    element {QName('http://www.w3.org/2000/svg','g')}
    {
        attribute id {$id},
        attribute transform {concat('translate(',$xShift,',',$yShift,')')},
        $classes/(@* except @id|@transform),
        $classes/svg:rect,
        $classes/svg:path,
        $classes/svg:text,
        $classes/svg:a,
        $startnode,
        $path,
        for $grp in $classes/svg:g
        return
        local:positionClasses($grp, $processedScan, $de-parent-id, $de-parent-ed, $atTopLevel)
    }
};

declare %private function local:getCircle($de-parent-id as xs:string?, $de-parent-ed as xs:string?, $cx as xs:integer, $cy as xs:integer) as element() {
    let $detailUri          :=
        for $p in request:get-parameter-names()[not(.=('id','effectiveDate'))] return 
        for $pval in request:get-parameter($p,()) return concat($p,'=',$pval)
        
    let $detailUri          :=
        if ($de-parent-id) then (
            concat('javascript:location.href=window.location.pathname+''?id=',$de-parent-id,'&amp;effectiveDate=',$de-parent-ed,'&amp;',string-join($detailUri,'&amp;'),'''')
        )
        else (
            concat('javascript:location.href=window.location.pathname+''?',string-join($detailUri,'&amp;'),'''')
        )
    return
    element {QName('http://www.w3.org/2000/svg','circle')}
    {
        attribute cx {$cx},
        attribute cy {$cy},
        attribute r {'10'},
        attribute class {'class-box class-box-hover'},
        if ($withInteractiveCode = 'true') then
            attribute onclick {$detailUri}
        else ()
    }
};

let $format                     := if (request:exists()) then request:get-parameter('format','svg')[string-length()>0] else ('svg')
let $version                    := if (request:exists()) then request:get-parameter('version',())[string-length()>0] else ()
let $datasetId                  := if (request:exists()) then request:get-parameter('datasetId',())[string-length()>0] else ()
let $datasetEffectiveDate       := if (request:exists()) then request:get-parameter('datasetEffectiveDate',())[string-length()>0] else ()
let $transactionId              := if (request:exists()) then request:get-parameter('transactionId',())[string-length()>0] else ()
let $transactionEffectiveDate   := if (request:exists()) then request:get-parameter('transactionEffectiveDate',())[string-length()>0] else ()
let $conceptId                  := if (request:exists()) then request:get-parameter('id',())[string-length()>0] else ()
let $conceptEffectiveDate       := if (request:exists()) then request:get-parameter('effectiveDate',())[string-length()>0] else ()
let $language                   := if (request:exists()) then request:get-parameter('language',())[string-length()>0] else ()
let $filterStatuses             := 
    if ($datasetId[string-length()>0]) then 
        if (request:exists()) then 
            for $p in request:get-parameter('filter',())[string-length()>0]
            return
                tokenize($p, '\s')
        else (
            'cancelled', 'rejected', 'deprecated'
        )
    else ()
    
let $transactionOrDataset       := 
    if ($transactionId[string-length()>0]) then art:getTransaction($transactionId, $transactionEffectiveDate, $version, $language) else 
    if ($datasetId[string-length()>0]) then art:getDataset($datasetId, $datasetEffectiveDate, $version, $language) else ()

let $language                   := 
    if (empty($language)) then
        ($transactionOrDataset/ancestor::decor/project/@defaultLanguage)[1]
    else (
        $language[1]
    )

let $transactionOrDataset   :=
    if ($version castable as xs:dateTime and string-length($language)>0) then (
        $transactionOrDataset[ancestor::decor/@language=$language][1]
    ) else (
        $transactionOrDataset[1]
    )
let $concept                := 
    if ($version castable as xs:dateTime) then (
        let $datasets   := 
            if ($transactionOrDataset[self::dataset]) then ($transactionOrDataset) else (
                let $x  := $get:colDecorVersion//transactionDatasets[@versionDate = $version][@language = $language]//dataset[@transactionId=$transactionOrDataset/@id]
                return
                if ($transactionEffectiveDate castable as xs:dateTime) then $x[@transactionEffectiveDate=$transactionEffectiveDate] else ($x[@transactionEffectiveDate=string(max($x/xs:dateTime(@transactionEffectiveDate)))])
            )
        return
            if (empty($conceptId)) then $datasets else ($datasets//concept[@id = $conceptId])
    ) else if ($transactionOrDataset) then (
        let $datasets   := art:getFullDatasetTree($transactionOrDataset, $conceptId, $conceptEffectiveDate, $language, (), false(), ())
        return
            if (empty($conceptId)) then $datasets else ($datasets/concept[1])
    ) else ()

let $dconcept                   := if ($conceptId[string-length()>0]) then art:getConcept($conceptId, $conceptEffectiveDate, $version) else ()
let $atTopLevel                 := not($concept/name()='concept')
let $conceptParentId            := if ($dconcept) then $dconcept/parent::concept/@id else ()
let $conceptParentEd            := if ($dconcept) then $dconcept/parent::concept/@effectiveDate else ()
let $conceptTransactId          := if ($transactionId[string-length()>0]) then $transactionId else ($datasetId)
let $conceptTransactEd          := if ($transactionId[string-length()>0]) then $transactionEffectiveDate else ($datasetEffectiveDate)

return
if ($format='xml') then (
    $concept
) else
if ($concept[not(@statusCode = $filterStatuses)]) then (
    let $classes            := <classes>{local:conceptClassbox($concept[not(@statusCode = $filterStatuses)][1],$language,$filterStatuses)}</classes>
    (:let $depth            := max(count($classes//svg:g/ancestor::svg:g)):)
    
    let $depths             := 
        for $group in $classes//svg:g
        return
            count($group/ancestor::svg:g)
    let $depth              := max($depths)
    
    let $scan               := local:reverseScan($classes,$depth)
    let $process            := 
        for $i in (1 to $depth)
        return
            if ($i=1 and $depth>1) then
                <step>let {concat('$step',$i,':= local:procesScan($scan)')}</step>
            else if ($i=$depth and $depth=1) then
                <step>let {concat('$step',$i,':= local:procesScan($scan)')}</step>|
                <step>return {concat('$step',$i)}</step>
            else if ($i=$depth and $depth>1) then
                <step>let {concat('$step',$i,':= local:procesScan($step',$i -1,')')}</step>|
                <step>return {concat('$step',$i)}</step>
            else(<step>let {concat('$step',$i,':= local:procesScan($step',$i -1,')')}</step>)
    
    let $processed          := util:eval(string-join($process,' '))
    (:let $processed :=	util:eval(string-join($process//step,' '),xs:boolean('false'),$scan):)
    
    let $width              :=  
        if (count($classes//svg:g)=1) then
            $classes//svg:rect[1]/@width
        else ($processed//level[@depth=0]/class/@width)
    
    let $height :=
        if (count($classes//svg:g)=1) then
            $classes//svg:rect[1]/@height + $levelSpacing 
        else (
            sum($processed//@maxHeight) + $levelSpacing + $levelSpacing * $depth
        )
    return (
      if (request:exists()) then (response:set-status-code(200), response:set-header('Content-Type','image/svg+xml'), response:set-header('X-Robots-Tag', 'noindex')) else ()
      ,
      <svg xmlns="http://www.w3.org/2000/svg" version="1.1" height="{$height+5}" width="{$width+5}">
          <!--
          <link href='http://fonts.googleapis.com/css?family=Open+Sans:300italic,400italic,700italic,700,300,400' rel='stylesheet' type='text/css'/>
          -->
          <style type="text/css"><![CDATA[
              body {
                  font-family: Verdana, Arial, sans-serif;
              }
          ]]>
          </style>
          <style type="text/css"><![CDATA[
              .dataset-bold {
                  font-family: Verdana, Arial, sans-serif;
                  font-size:11px;
                  font-weight:700;
                  font-style: italic;
                  text-align:start;
                  line-height:125%;
                  text-anchor:middle;
                  fill:#000000;
                  fill-opacity:1;
                  stroke:none;
              }
          ]]>
          </style>
          <style type="text/css"><![CDATA[
              .normal-bold {
                  font-family: Verdana, Arial, sans-serif;
                  font-size:11px;
                  font-weight:700;
                  text-align:start;
                  line-height:125%;
                  text-anchor:middle;
                  fill:#000000;
                  fill-opacity:1;
                  stroke:none;
              }
          ]]>
          </style>
          <style type="text/css"><![CDATA[
              .inherit-bold {
                  font-family: Verdana, Arial, sans-serif;
                  font-size:11px;
                  font-weight:700;
                  text-align:start;
                  line-height:125%;
                  text-anchor:middle;
                  fill:#4a12eb;
                  fill-opacity:1;
                  stroke:none;
              }
              .inherit {
                  fill:#4a12eb;
                  fill-opacity:1;
              }
          ]]>
          </style>
          <style type="text/css"><![CDATA[
              .normal-start {
                  font-family: Verdana, Arial, sans-serif;
                  font-weight:300;
                  font-size:11px;
                  text-align:start;
                  font-weight:normal;
                  text-anchor:start;
                  fill:#000000;
                  fill-opacity:1;
                  stroke:none;
              }
          ]]>
          </style>
          <style type="text/css"><![CDATA[
              .class-box {
                  fill: #d7ecff;
                  fill-opacity:1;
                  fill-rule:evenodd;
                  stroke:#000000;
                  stroke-width:0.2;
                  stroke-linecap:butt;
                  stroke-linejoin:miter;
                  stroke-miterlimit:4;
                  stroke-opacity:1;
                  stroke-dasharray:none
              }
              .class-box-hover {
                  cursor: pointer;
              }
              .class-box-hover:hover {
                  fill-opacity:0.5;
              }
          ]]>
          </style>
          <defs>
              <filter id="shadow" height="200%" width="200%" y="0" x="0">
                  <feOffset result="offOut" dx="3" dy="3" in="SourceGraphic"/>
                  <feColorMatrix result="matrixOut" values="0.2 0 0 0 0 0 0.2 0 0 0 0 0 0.2 0 0 0 0 0 1 0" type="matrix" in="offOut"/>
                  <feGaussianBlur result="blurOut" stdDeviation="2" in="matrixOut"/>
                  <feBlend in2="blurOut" in="SourceGraphic" mode="normal"/>
              </filter>
          </defs>
      
          <rect style="fill:#ffffff;fill-opacity:1;stroke:#000000;stroke-width:0" width="{$width}" height="{$height+5}" x="0" y="0">
              <desc>Background rectangle in white to avoid transparency.</desc>
          </rect>
          {
              if ($classes/svg:g/svg:g) then
                  local:positionClasses($classes/svg:g, $processed, $conceptParentId, $conceptParentEd, $atTopLevel)
              else (
              (:$classes/*:)
                  element {QName('http://www.w3.org/2000/svg','g')}
                  {
                      attribute transform {concat('translate(',0,',',$levelSpacing,')')},
                      $classes/svg:g/(@* except @transform),
                      $classes/svg:g/svg:*,
                      if (not($atTopLevel)) then (
                          local:getCircle($conceptParentId, $conceptParentEd, ($classes//svg:rect[1]/@width div 2), -25),
                          element {QName('http://www.w3.org/2000/svg','path')}
                          {
                              attribute style {'fill:none;stroke:#000000;stroke-width:1px'},
                              attribute d {concat('m ',($classes//svg:rect[1]/@width div 2),', 0 ','v -',($levelSpacing div 3))}
                          }
                      ) else ()
                  }
              )
          }
      </svg>
  )
) else (
    if (request:exists()) then (response:set-status-code(404), response:set-header('Content-Type','text/html'), response:set-header('X-Robots-Tag', 'noindex')) else ()
    ,
        <html>
        <head><title>Not Found</title></head>
        <body>
        <h3>Not Found</h3>
        <div>Could not find requested content with these parameters:</div>
        <ul>
            <li>version="{$version}"</li>
            <li>datasetId="{$datasetId}"</li>
            <li>datasetEffectiveDate="{$datasetEffectiveDate}"</li>
            <li>transactionId="{$transactionId}"</li>
            <li>transactionEffectiveDate="{$transactionEffectiveDate}"</li>
            <li>id="{$conceptId}"</li>
            <li>effectiveDate="{$conceptEffectiveDate}"</li>
            <li>language="{$language}"</li>
        </ul>
        </body>
        </html>
)