xquery version "3.1";
import module namespace adserver       = "http://art-decor.org/ns/art-decor-server" at "../art/api/api-server-settings.xqm";
declare variable $exist:path external;
declare variable $exist:prefix external;
declare variable $exist:resource external;
declare variable $exist:controller external;

let $supported-packages                 := xmldb:get-child-collections(concat(repo:get-root(), '/fhir'))
let $defaultFhirVersion                 := try { adserver:getServerFhirDefaultVersion() } catch * {()}
(:  Input   Accept: application/fhir+json;fhir-version=X
    Output  X
    
    X could be a specific version, 'any version' (*), and a comma separated list of those. first in list has higher priority than last
:)
let $acceptHeader                       := request:get-header('Accept')
let $requestedFhirVersions              := 
    for $v in tokenize(replace(normalize-space(tokenize($acceptHeader, ';')[matches(., 'fhirVersion=')][1]), '^fhirVersion=(.*)$', '$1'), ',')
    return if (string-length($v) gt 0) then normalize-space($v) else ()
    
let $requestedAndSupportedFhirVersions  := (
    if (empty($requestedFhirVersions)) then $defaultFhirVersion else ( 
        for $v in $requestedFhirVersions return 
        if ($v = $supported-packages) then $v else if ($v = '*') then $defaultFhirVersion else ()
    )
)
let $versionsOperation                 := starts-with($exist:path, '/$versions')

return
    (:if (1=1) then <path hasVersions="{$versionsOperation}">{$exist:path}</path> else:)
    if ($versionsOperation) then (
        if (contains($acceptHeader, 'application/fhir+json') or contains($acceptHeader, 'application/json+fhir')) then (
            response:set-header('Content-Type', 'application/fhir+json'),
            '{',
                '"resourceType": "Parameters", ',
                    '"parameter": [',
                        string-join (
                        for $v in $supported-packages
                        return (
                            string-join(('{',
                            '    "name": "version",',
                            '    "valueString": ', concat('"', $v, '"'),
                            '}'), ' ')
                        ), ', '),
                    ']',
            '}'
        )
        else
        if (contains($acceptHeader, 'application/fhir+xml') or contains($acceptHeader, 'application/xml+fhir')) then (
            response:set-header('Content-Type', 'application/fhir+xml'),
            <Parameters xmlns="http://hl7.org/fhir">
            {
                for $v in $supported-packages
                return
                <parameter>
                    <name value="version"/>
                    <valueString value="{$v}"/>
                </parameter>
            }
            </Parameters>
        )
        else
        if (contains($acceptHeader, 'json')) then (
            response:set-header('Content-Type', 'application/json'),
            '{ "versions": [', string-join(for $v in $supported-packages return concat('"',$v,'"'), ','), ']}'
        )
        else (
            <versions>{for $v in $supported-packages return <version>{$v}</version>}</versions>
        )
    )
    else
    (: we used to have a /fhir/dstu2/ and a /fhir/stu3 endpoint. we now have /fhir/1.0/, fhir/3.0/, fhir/4.0/ endpoint. reroute that first :)
    if (matches($exist:path, '^/dstu2/public') and $supported-packages[. = '1.0']) then (
        response:redirect-to(xs:anyURI(string-join((concat(adserver:getServerURLFhirServices(), replace($exist:path, '/dstu2/', '')), request:get-query-string()), '?')))
    )
    else
    if (matches($exist:path, '^/stu3/public') and $supported-packages[. = '3.0']) then (
        response:redirect-to(xs:anyURI(string-join((concat(adserver:getServerURLFhirServices(), replace($exist:path, '/stu3/', '')), request:get-query-string()), '?')))
    )
    else
    (: if we get /<...>/public then it must be because the requested FHIR endpint is not supported, otherwise you would be sent to that controller :)
    if (matches($exist:path, '^/[^/]+/public')) then (
        response:set-status-code(400),
        <OperationOutcome xmlns="http://hl7.org/fhir">
            <issue>
                <severity value="fatal"/>
                <code value="exception"/>
                <details>
                    <text value="There is no HL7® FHIR® service installed on this endpoint that matches the version(s) you requested ('{replace($exist:path, '^/([^/]+).*$', '$1')}'). Please contact the administrator for more information. The following HL7® FHIR® service endpoints are available: {
                        for $endpoint in $supported-packages
                        return concat($endpoint, '/public/')
                    }"/>
                </details>
            </issue>
        </OperationOutcome>
    )
    else
    (: in the beginning we only had a /fhir/ endpoint. we now have a versioned endpoint. Reroute there if possible, 
       or suggest alternative endpoints. If supported-packages is empty then no FHIR server lives here.
    :)
    if ($supported-packages[. = $requestedAndSupportedFhirVersions]) then (
        (: if we get a path like /public/.. or /demo1-/..., then all we need to do is insert the right version for rerouting.
           otherwise we need to add /public to the path too.
           
           Note that this assumes that all FHIR resources start with a capital, and all ART-DECOR prefixes and registry names start with anything else
        :)
        if (matches($exist:path, '^/[^A-Z]')) then 
            response:redirect-to(xs:anyURI(string-join((concat(adserver:getServerURLFhirServices(), $requestedAndSupportedFhirVersions[1], $exist:path), request:get-query-string()), '?')))
        else (
            response:redirect-to(xs:anyURI(string-join((concat(adserver:getServerURLFhirServices(), $requestedAndSupportedFhirVersions[1], '/public', $exist:path), request:get-query-string()), '?')))
        )
    )
    else
    if (empty($supported-packages)) then (
        response:set-status-code(400),
        <OperationOutcome xmlns="http://hl7.org/fhir">
            <issue>
                <severity value="fatal"/>
                <code value="exception"/>
                <details>
                    <text value="There is no HL7® FHIR® service installed on this endpoint. Please contact the administrator for more information."/>
                </details>
            </issue>
        </OperationOutcome>
    )
    else
    if (string-length($defaultFhirVersion) = 0) then (
        response:set-status-code(500),
        <OperationOutcome xmlns="http://hl7.org/fhir">
            <issue>
                <severity value="fatal"/>
                <code value="exception"/>
                <details>
                    <text value="The default endpoint has not been configured. Please ask your server administrator to set this up. The following HL7® FHIR® service endpoints are available: {
                        for $endpoint in $supported-packages
                        return concat($endpoint, '/public/')
                    }"/>
                </details>
            </issue>
        </OperationOutcome>
    )
    else 
    if (empty($requestedFhirVersions)) then (
        response:set-status-code(500),
        <OperationOutcome xmlns="http://hl7.org/fhir">
            <issue>
                <severity value="fatal"/>
                <code value="exception"/>
                <details>
                    <text value="The default endpoint '{$defaultFhirVersion}' does not exist. Please ask your server administrator to set this up. The following HL7® FHIR® service endpoints are available: {
                        for $endpoint in $supported-packages
                        return concat($endpoint, '/public/')
                    }"/>
                </details>
            </issue>
        </OperationOutcome>
    )
    else (
        response:set-status-code(400),
        <OperationOutcome xmlns="http://hl7.org/fhir">
            <issue>
                <severity value="fatal"/>
                <code value="exception"/>
                <details>
                    <text value="There is no HL7® FHIR® service installed on this endpoint that matches the version(s) you requested ('{$requestedFhirVersions}'). Please contact the administrator for more information. The following HL7® FHIR® service endpoints are available: {
                        for $endpoint in $supported-packages
                        return concat($endpoint, '/public/')
                    }"/>
                </details>
            </issue>
        </OperationOutcome>
    )