xquery version "3.1";
(:
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools
    see https://art-decor.org/mediawiki/index.php?title=Copyright
    
    This program is free software; you can redistribute it and/or modify it under the terms of the
    GNU Lesser General Public License as published by the Free Software Foundation; either version
    2.1 of the License, or (at your option) any later version.
    
    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
    See the GNU Lesser General Public License for more details.
    
    The full text of the license is available at http://www.gnu.org/copyleft/lesser.html
:)
module namespace adfhirq           = "http://art-decor.org/ns/fhir/4.0/questionnaire";

import module namespace adfhir      = "http://art-decor.org/ns/fhir/4.0" at "api-fhir.xqm";
import module namespace adserver    = "http://art-decor.org/ns/art-decor-server" at "../../../art/api/api-server-settings.xqm";
import module namespace getf        = "http://art-decor.org/ns/fhir-settings" at "fhir-settings.xqm";
import module namespace art         = "http://art-decor.org/ns/art" at "../../../art/modules/art-decor.xqm";
import module namespace get         = "http://art-decor.org/ns/art-decor-settings" at "../../../art/modules/art-decor-settings.xqm";

declare namespace f                 = "http://hl7.org/fhir";
declare namespace error             = "http://art-decor.org/ns/fhir/error";
declare namespace json              = "http://www.json.org";
declare namespace xhtml             = "http://www.w3.org/1999/xhtml";

declare %private variable $adfhirq:type                 := 'Questionnaire';
declare %private variable $adfhirq:inactiveStatusCodes  := ('cancelled','rejected','deprecated');
declare %private variable $adfhirq:UCUMCODES            := if (doc-available(concat($get:strDecorCore, '/DECOR-ucum.xml'))) then doc(concat($get:strDecorCore, '/DECOR-ucum.xml'))/* else ();

declare function adfhirq:convertDecorDatasetOrTransaction2FHIRQuestionnaire($id as xs:string, $effectiveDate as xs:string?, $version as xs:string?, $language as xs:string?, $fhirLinkItemStyle as xs:string?) as element(f:Questionnaire)? {
    let $transactionOrDataset   := 
        if (string-length($id) > 0) then
            art:getDataset($id, $effectiveDate, $version, $language) | art:getTransaction($id, $effectiveDate, $version, $language)
        else ()
    
    let $language               := if (empty($language) or ($language = '*')) then $transactionOrDataset/ancestor::decor/project/@defaultLanguage else ($language)
    (: fix for de-CH :)
    let $language               := if ($language='de-CH') then 'de-DE' else $language
    
    let $fullDatasetTree        := 
        if ($version castable as xs:dateTime) then (
            let $datasets   := 
                if ($transactionOrDataset[self::dataset]) then ($transactionOrDataset) else (
                    if ($language = '*') then
                        $get:colDecorVersion//transactionDatasets[@versionDate = $version]//dataset[@transactionId = $transactionOrDataset/@id][@transactionEffectiveDate = $transactionOrDataset/@effectiveDate]
                    else (
                        $get:colDecorVersion//transactionDatasets[@versionDate = $version][@language = $language]//dataset[@transactionId = $transactionOrDataset/@id][@transactionEffectiveDate = $transactionOrDataset/@effectiveDate]
                    )
                )
            
            return $datasets[1]
        ) else if ($transactionOrDataset) then (
            art:getFullDatasetTree($transactionOrDataset, (), (), $language, (), false(), ())
        ) else ()
        
    let $result                 := 
        if ($fullDatasetTree) then (
            (:let $xsl            := doc('../resources/stylesheets/tr2quest.xsl')
            return transform:transform($fullDatasetTree, $xsl, ()):)
            let $oidMap           := 
                for $cs in $fullDatasetTree//@codeSystem
                let $csid := $cs
                group by $csid
                return map:entry($csid, art:getCanonicalUriForOID($csid, (), $transactionOrDataset/ancestor::decor, $get:strKeyCanonicalUriPrefdR4))
            return
            adfhirq:decorTransaction2Questionnaire($fullDatasetTree,
                map { 
                    "fhirLinkItemStyle": $fhirLinkItemStyle, 
                    "fhirCanonicalBase": adserver:getServerURLFhirServices(), 
                    "publisher": ($transactionOrDataset/publishingAuthority/@name, $transactionOrDataset/ancestor::decor/project/copyright/@by, "ART-DECOR Expert Group")[1], 
                    "language":  ($language, $transactionOrDataset/ancestor::decor/project/@defaultLanguage)[1],
                    "oids": map:merge($oidMap)
                }
            )
        )
        else ()
    return $result
};

declare function adfhirq:convertDecorQuestionnaire2FHIRQuestionnaire($q as element(questionnaire), $params as map(*)?) as element(f:Questionnaire) {
    let $language               := ($params?language, $q/*:name[@language = 'en-US']/@language, $q/*:name/@language)[not(. = '*')][1]
    let $projectPrefix          := ($params?projectPrefix, $q/ancestor::decor/project/@prefix)[1]
    let $fhirCanonicalBase      := $params?fhirCanonicalBase
    let $publisher              := ($q/*:publishingAuthority, $params?publisher)[1]
    let $fhirId                 := concat($q/@id, '--', replace($q/@effectiveDate,'\D',''))
    let $oidMap                 := map:merge(
        for $cs in $q//*[empty(@canonicalUri)]/@codeSystem
        let $csid := $cs
        group by $csid
        return map:entry($csid, art:getCanonicalUriForOID($csid, (), $q/ancestor::decor, $get:strKeyCanonicalUriPrefdR4))
    )
    return
    <Questionnaire xmlns="http://hl7.org/fhir">
        <id value="{$fhirId}"/>
    {
        if ($q/@lastModifiedDate or $projectPrefix) then
            <meta>
            {
                if ($q/@lastModifiedDate) then
                    <lastUpdated value="{$q/@lastModifiedDate || 'Z'}"/>
                else (),
                if ($projectPrefix) then 
                    <source value="{$fhirCanonicalBase || $getf:strFhirVersionShort || '/' || $projectPrefix}"/>
                else ()
            }
            </meta>
        else ()
    }
        <language value="{$language}"/>
    {
        adfhir:item2fhirNamespace($q/*:contained[*/*:id/concat('#', @value) = $q/*:item//@ref])
    }
        <url value="{$q/@canonicalUri}"/>
        <identifier>
            <system value="urn:ietf:rfc:3986"/>
            <value value="urn:oid:{$q/@id}"/>
        </identifier>
    {
        if ($q/@versionLabel) then <version value="{$q/@versionLabel}"/> else (),
        adfhir:decorName2fhirName($q/*:name, $language),
        adfhir:decorName2fhirTitle($q/*:name, $language),
        (: derivedFrom - could come from relationship to another Questionnaire? :)
        adfhirq:decorStatusCode2fhirQuestionnaireStatus($q/@statusCode),
        <experimental value="{$q/@experimental = 'true'}"/>,
        adfhirq:decorSubjectType2fhirSubjectType($q/*:subjectType),
        <date value="{$q/@effectiveDate || 'Z'}"/>,
        adfhir:decorPublishingAuthority2fhirPublisher($publisher),
        adfhir:decorPublishingAuthority2fhirContact($publisher),
        adfhir:decorDesc2fhirDescription($q/*:desc, $language),
        (: useContext - not supported :)
        adfhir:decorJurisdiction2fhirJurisdiction($q/*:jurisdiction, $oidMap),
        adfhir:decorPurpose2fhirPurpose($q/*:purpose, $language),
        adfhir:decorCopyright2fhirCopyright($q/*:copyright, $language),
        if ($q/@officialReleaseDate) then
            <approvalDate value="{substring-before($q/@officialReleaseDate, 'T')}"/>
        else (),
        (: lastReviewDate - not supported :)
        <effectivePeriod>
            <start value="{$q/@effectiveDate || 'Z'}"/>
        {
            if ($q/@expirationDate) then
                <end value="{$q/@expirationDate || 'Z'}"/>
            else ()
        }
        </effectivePeriod>,
        adfhirq:decorCode2fhirCode($q/*:code, $oidMap),
        adfhirq:decorItem2fhirItem($q/*:item, $language, $oidMap)
    }
    </Questionnaire>
};

declare %private function adfhirq:decorTransaction2Questionnaire($transaction as element(dataset), $params as map(*)?) as element(f:Questionnaire) {
    let $language               := ($params?language, $transaction/*:name[@language = 'en-US']/@language, $transaction/*:name/@language)[1]
    let $fhirLinkItemStyle      := $params?fhirLinkItemStyle
    let $fhirCanonicalBase      := $params?fhirCanonicalBase
    let $publisher              := $params?publisher
    let $transactionId          := $transaction/@transactionId
    let $fhirId                 := 
        if ($transaction/@transactionId) then 
            concat($transaction/@transactionId,'--',replace($transaction/@transactionEffectiveDate,'\D','')) 
        else (
            concat($transaction/@id,'--',replace($transaction/@effectiveDate,'\D',''))
        )
    let $targets                := if (map:keys($params) = 'targets') then map:get($params, 'targets') else $transaction/*:concept/@id
    
    let $check                  :=
        if (empty($targets)) then
            error(xs:QName('adfhirq:NOCONCEPTS'), 'ERROR: no targets for this transaction')
        else ()

    return
    <Questionnaire xmlns="http://hl7.org/fhir">
        <id value="{$fhirId}"/>
        <language value="{$transaction/*:name[1]/@language}"/>
        <text>
            <status value="generated"/>
            <div xmlns="http://www.w3.org/1999/xhtml">Questionnaire for {data($transaction/*:name[1])}</div>
        </text>
        <url value="{$fhirCanonicalBase}Questionnaire/{$fhirId}"/>
        {
            if ($transaction/@ersionLabel) then 
                <version value="{$transaction/@ersionLabel}"/> 
            else (),
            adfhir:decorName2fhirName($transaction/*:name, $language),
            adfhir:decorName2fhirTitle($transaction/*:name, $language)
        }
        <status value="draft"/>
        <subjectType value="Patient"/>
        <date value="{substring(string(current-date()), 1, 10)}"/>
    {
        if (empty($publisher)) then () else <publisher value="{$publisher}"/>
    }
    {
        for $c in $transaction/*:concept[@id = $targets] return adfhirq:decorConcept2QuestionnaireItem($c, $params)
    }
    </Questionnaire>
};

declare function adfhirq:decorSubjectType2fhirSubjectType($in as element(subjectType)?) as element(f:subjectType)* {
    for $s in $in
    return
        <subjectType xmlns="http://hl7.org/fhir" value="{$s/@code}"/>
};
declare function adfhirq:decorCode2fhirCode($in as element(code)*, $oidMap as map(*)?) as element(f:code)* {
    for $j in $in[@code][@codeSystem | @canonicalUri]
    let $system   := 
        if ($j/@canonicalUri) then 
            $j/@canonicalUri 
        else 
        if ($j/@codeSystem) then
            map:get($oidMap, $j/@codeSystem)
        else ()
    return
    <code xmlns="http://hl7.org/fhir">
        <system value="{$system}"/>
        <code value="{$j/@code}"/>
    {
        if ($j[@displayName]) then
            <display value="{replace($j/@displayName, '(^\s+)|(\s+$)', '')}"/>
        else ()
    }
    </code>
};
declare function adfhirq:decorItem2fhirItem($in as element(item)*, $language as xs:string, $oidMap as map(*)?) as element(f:item)* {
    for $i in $in
    return
        <item xmlns="http://hl7.org/fhir">
        {
            if ($i/*:designNote) then
                <extension url="http://hl7.org/fhir/StructureDefinition/designNote">
                    <valueMarkdown value="{($i/*:designNote[@language = $language], $i/*:designNote)[1]}"/>
                </extension>
            else ()
            ,
            for $t in $i/*:terminologyServer
            return
                <extension url="http://hl7.org/fhir/uv/sdc/StructureDefinition/sdc-questionnaire-preferredTerminologyServer">
                    <valueUrl value="{$t/@ref}"/>
                </extension>
            ,
            for $u in $i/*:unitOption[@code]
            let $system   := 
                if ($u/@canonicalUri) then 
                    $u/@canonicalUri 
                else 
                if ($u/@codeSystem) then
                    map:get($oidMap, $u/@codeSystem)
                else ()
            return
                <extension url="http://hl7.org/fhir/StructureDefinition/questionnaire-unitOption">
                    <valueCoding>
                        <system value="{$system}"/>
                        <code value="{$u/@code}"/>
                    {
                        if ($u[@displayName]) then
                            <display value="{replace($u/@displayName, '(^\s+)|(\s+$)', '')}"/>
                        else ()
                    }
                    </valueCoding>
                </extension>
            ,
            for $u in $i/*:itemControl
            let $system   := 
                if ($u/@canonicalUri) then 
                    $u/@canonicalUri 
                else 
                if ($u/@codeSystem) then
                    map:get($oidMap, $u/@codeSystem)
                else ()
            return
                <extension url="http://hl7.org/fhir/StructureDefinition/questionnaire-itemControl">
                    <valueCodeableConcept>
                    {
                        if ($u[@code]) then
                            <coding>
                                <system value="{$system}"/>
                                <code value="{$u/@code}"/>
                            {
                                if ($u[@displayName]) then
                                    <display value="{replace($u/@displayName, '(^\s+)|(\s+$)', '')}"/>
                                else ()
                            }
                            </coding>
                        else (
                            <text value="{replace($u/@displayName, '(^\s+)|(\s+$)', '')}"/>
                        )
                    }
                    </valueCodeableConcept>
                </extension>
            ,
            if ($i/@minValue) then
                <extension url="http://hl7.org/fhir/StructureDefinition/minValue">
                {
                    typeswitch ($i/@minValue)
                    case xs:date return <valueDate value="{$i/@minValue}"/>
                    case xs:dateTime return <valueDateTime value="{$i/@minValue}"/>
                    case xs:time return <valueTime value="{$i/@minValue}"/>
                    default return
                        if (matches($i/@minValue, '\.')) then
                            <valueDecimal value="{$i/@minValue}"/>
                        else (
                            <valueInteger value="{$i/@minValue}"/>
                        )
                }   
                </extension>
            else ()
            ,
            if ($i/@maxValue) then
                <extension url="http://hl7.org/fhir/StructureDefinition/maxValue">
                {
                    typeswitch ($i/@maxValue)
                    case xs:date return <valueDate value="{$i/@maxValue}"/>
                    case xs:dateTime return <valueDateTime value="{$i/@maxValue}"/>
                    case xs:time return <valueTime value="{$i/@maxValue}"/>
                    default return
                        if (matches($i/@maxValue, '\.')) then
                            <valueDecimal value="{$i/@maxValue}"/>
                        else (
                            <valueInteger value="{$i/@maxValue}"/>
                        )
                }   
                </extension>
            else ()
            ,
            for $iec in $i/*:itemExtractionContext
            return
                <extension url="http://hl7.org/fhir/uv/sdc/StructureDefinition/sdc-questionnaire-itemExtractionContext">
                {
                    if ($iec/@code) then
                        <valueCode value="{$iec/@code}"/>
                    else
                    if ($iec/*:expression) then
                        <valueExpression>
                        {
                            for $att in $iec/*:expression/(@description, @name, @language, @expression, @reference)
                            return
                                element {name($att)} {attribute value {$att}}
                        }
                        </valueExpression>
                    else ()
                }   
                </extension>
        }
        {
            if ($i/@linkId) then <linkId value="{$i/@linkId}"/> else (),
            if ($i/@definition) then <definition value="{$i/@definition}"/> else ()
            ,
            for $u in $i/*:code
            let $system   := 
                if ($u/@canonicalUri) then 
                    $u/@canonicalUri 
                else 
                if ($u/@codeSystem) then
                    map:get($oidMap, $u/@codeSystem)
                else ()
            return
                <code>
                    <system value="{$system}"/>
                    <code value="{$u/@code}"/>
                {
                    if ($u[@displayName]) then
                        <display value="{replace($u/@displayName, '(^\s+)|(\s+$)', '')}"/>
                    else ()
                }
                </code>
            ,
            if ($i/@prefix) then <prefix value="{$i/@prefix}"/> else ()
            ,
            let $default    := ($i/*:text[@language = $language], $i/*:text)[1]
            return
                if ($default) then 
                    <text xmlns="http://hl7.org/fhir" value="{$default}">
                    {
                        for $n in $i/*:text[not(@language = $default/@language)]
                        return
                            <extension url="http://hl7.org/fhir/StructureDefinition/translation">
                                <extension url="lang">
                                    <valueCode value="{$n/@language}"/>
                                </extension>
                                <extension url="content">
                                    <valueString value="{$n}"/>
                                </extension>
                            </extension>
                    }
                    </text>
                else ()
            ,
            for $att in $i/(@type, @required, @repeats, @readOnly, @maxLength)
            return
                element {name($att)} {attribute value {$att}}
            ,
            if ($i/*:answerValueSet) then
                <answerValueSet>
                    <reference value="{$i/*:answerValueSet/@ref}"/>
                </answerValueSet>
            else ()
        }
        {
            for $ao in $i/*:answerOption
            let $option   := ($ao/(*:valueInteger|*:valueDate|*:valueTime|*:valueString|*:valueCoding))[1]
            let $system   := 
                if ($option/@canonicalUri) then 
                    $option/@canonicalUri 
                else 
                if ($option/@codeSystem) then
                    map:get($oidMap, $option/@codeSystem)
                else ()
            return
                <answerOption>
                {
                    if ($option[local-name() = ('valueInteger', 'valueDate', 'valueTime', 'valueString')]) then
                        element {name($option)} {attribute value {$option/@value}}
                    else (
                        <valueCoding>
                            <system value="{$system}"/>
                            <code value="{$option/@code}"/>
                        {
                            if ($option[@displayName]) then
                                <display value="{replace($option/@displayName, '(^\s+)|(\s+$)', '')}"/>
                            else ()
                        }
                        </valueCoding>
                    )
                }
                </answerOption>
        }
        {
            adfhirq:decorItem2fhirItem($i/*:item, $language, $oidMap)
        }
        </item>
};

declare %private function adfhirq:decorConcept2QuestionnaireItem($concept as element(concept), $params as map(*)?) as element(f:item)? {
    let $language               := map:get($params, 'language')
    let $fhirLinkItemStyle      := map:get($params, 'fhirLinkItemStyle')
    let $fhirCanonicalBase      := map:get($params, 'fhirCanonicalBase')
    let $publisher              := map:get($params, 'publisher')
    let $oidMap                 := map:get($params, 'oids')
    let $targetId               := 
        if ($fhirLinkItemStyle='idDisplay') 
        then $concept/@iddisplay 
        else if ($fhirLinkItemStyle='oid')
        then $concept/@id 
        else $concept/concat(@id, '--', replace(@effectiveDate, '\D', ''))
    let $conceptId              := $concept/@id
    
    let $conceptDescription     := $concept/*:desc[.//text()[not(normalize-space() = '')]]
    let $conceptDescriptionText := replace(replace(string-join($conceptDescription[1], '\n'), '&#13;?&#10;\s+' , '&#10;'), '^\s+|\s+$', '')
    let $conceptDescriptionHtml := if ($conceptDescription) then replace(replace(art:serializeNode($conceptDescription[1]), '&#13;?&#10;\s+' , '&#10;'), '^\s+|\s+$', '') else ()
    let $definitionUri          := 
        $fhirCanonicalBase || 'StructureDefinition/' || $concept/ancestor::*:dataset/@transactionId || '--' || replace($concept/ancestor::*:dataset/@transactionEffectiveDate, '\D', '') || 
                                                 '#' || $concept/@id                                || '--' || replace($concept/@effectiveDate, '\D', '')

    let $itemText               := ($concept/*:operationalization, $concept/*:name)[1]
    (:let $itemText               :=
        if ($itemText[count(*) = 1][*:div][normalize-space(string-join(text(), '')) = '']) then
            md:html2markdown($itemText/*:div/node())
        else (
            md:html2markdown($itemText/node())
        ):)
    (:let $itemText               :=
        if ($itemText[count(*) = 1][*:div][normalize-space(string-join(text(), '')) = '']) then
            art:serializeNode($itemText/*:div)
        else (
            art:serializeNode($itemText)
        ):)
    let $itemText               :=
        if ($itemText[count(*) = 1][*:div][normalize-space(string-join(text(), '')) = '']) then
            $itemText/*:div
        else (
            $itemText/node()
        )
    
    let $defaultValue           := ($concept/*:valueDomain/*:property/@default)[1]
    (: In addition, the following extensions MUST be supported: minValue, maxValue, minLength, maxDecimalPlaces, unit :)
    let $minInclude             := min($concept/*:valueDomain/*:property[@minInclude castable as xs:decimal]/xs:decimal(@minInclude))
    let $maxInclude             := max($concept/*:valueDomain/*:property[@maxInclude castable as xs:decimal]/xs:decimal(@maxInclude))
    let $minLength              := min($concept/*:valueDomain/*:property[@minLength castable as xs:integer]/xs:integer(@minLength))
    let $maxDecimalPlaces       := 
        max(
            for $t in $concept/*:valueDomain/*:property/@fractionDigits
            let $md   := replace($t, '[^\d]', '')[not(. = '')]
            return
                if ($md castable as xs:integer) then xs:integer($md) else ()
        )
    return
    switch ($concept/@type)
    case 'group' return (
        <item xmlns="http://hl7.org/fhir">
            <linkId value="{$targetId}"/>
        {
            if ($conceptDescription) then
                (:<definition value="{md:html2markdown(art:parseNode($conceptDescription[1])/node())}"/>:)
                <definition value="{$definitionUri}"/>
            else ()
        }
            <text value="{replace(string-join($itemText, ''), '^\s+|\s+$', '')}"/>
        {
            for $assoc in $concept/*:terminologyAssociation[@conceptId = $conceptId]
            return
                <code>
                    <system value="{map:get($oidMap, $assoc/@codeSystem)}"/>
                    <code value="{$assoc/@code}"/>
                {
                    if (empty($assoc/@displayName)) then () else ( 
                        <display value="{$assoc/@displayName}"/>
                    )
                }
                </code>
        }
            <type value="group"/>
            <required value="{$concept/@minimumMultiplicity and $concept/@minimumMultiplicity != '0'}"/>
            <repeats value="{not($concept/@maximumMultiplicity = '0' or $concept/@maximumMultiplicity = '1')}"/>
        {
            if ($conceptDescription) then
                (: Note: datatype string does not officially allow xhtmland will trigger a datatype error, however ... it is what the core extension is defined with
                https://chat.fhir.org/#narrow/stream/179166-implementers/topic/Questionnaire.20extension.20rendering-xhtml.20with.20datatype.20string :)
                <item>
                    <extension url="http://hl7.org/fhir/StructureDefinition/questionnaire-itemControl">
                        <valueCodeableConcept>
                            <coding>
                                <system value="http://hl7.org/fhir/questionnaire-item-control"/>
                                <code value="help"/>
                                <display value="Help-Button"/>
                            </coding>
                            <text value="Help-Button"/>
                        </valueCodeableConcept>
                    </extension>
                    <linkId value="{$targetId}-help"/>
                    <text value="{$conceptDescriptionText}">
                        <extension url="http://hl7.org/fhir/StructureDefinition/rendering-xhtml">
                            <valueString value="{$conceptDescriptionHtml}"/>
                        </extension>
                    </text>
                    <type value="display"/>
                </item>
            else ()
        }
        {
            for $c in $concept/*:concept return adfhirq:decorConcept2QuestionnaireItem($c, $params)
        }
        </item>
    )
    case 'item' return (
        <item xmlns="http://hl7.org/fhir">
        {
            if (empty($minInclude)) then () else (
                <extension url="http://hl7.org/fhir/StructureDefinition/minValue">
                {
                    if ($minInclude castable as xs:integer) then
                        <valueInteger value="{xs:integer($minInclude)}"/>
                    else (
                        <valueDecimal value="{$minInclude}"/>
                    )
                }
                </extension>
            )
        }
        {
            if (empty($maxInclude)) then () else (
                <extension url="http://hl7.org/fhir/StructureDefinition/maxValue">
                {
                    if ($maxInclude castable as xs:integer) then
                        <valueInteger value="{xs:integer($maxInclude)}"/>
                    else (
                        <valueDecimal value="{$maxInclude}"/>
                    )
                }
                </extension>
            )
        }
        {
            if (empty($minLength)) then () else (
                <extension url="http://hl7.org/fhir/StructureDefinition/minLength">
                    <valueInteger value="{xs:integer($minLength)}"/>
                </extension>
            )
        }
        {
            if ($concept/*:valueDomain/@type = 'quantity') then
                for $unit in $concept/*:valueDomain/*:property/@unit
                let $unit     := normalize-space($unit)
                let $ucumUnit := $adfhirq:UCUMCODES/*:ucum[@unit = $unit]
                return
                    <extension url="http://hl7.org/fhir/StructureDefinition/questionnaire-unitOption">
                        <valueCoding>
                        {
                            if (empty($ucumUnit)) then () else (
                                <system value="http://unitsofmeasure.org"/>
                            )
                        }
                            <code value="{$unit}"/>
                            <display value="{($ucumUnit/@displayName[not(. = '')], $unit)[1]}"/>
                        </valueCoding>
                    </extension>
            else ()
        }
        {
            if (empty($maxDecimalPlaces)) then () else (
                <extension url="http://hl7.org/fhir/StructureDefinition/maxDecimalPlaces">
                    <valueInteger value="{$maxDecimalPlaces}"/>
                </extension>
            )
        }
            <linkId value="{$targetId}"/>
        {
            if ($conceptDescription) then
                (:<definition value="{md:html2markdown(art:parseNode($conceptDescription[1])/node())}"/>:)
                <definition value="{$definitionUri}"/>
            else ()
        }
        {
            for $assoc in $concept/*:terminologyAssociation[@conceptId = $conceptId]
            return
                <code>
                    <system value="{map:get($oidMap, $assoc/@codeSystem)}"/>
                    <code value="{$assoc/@code}"/>
                {
                    if (empty($assoc/@displayName)) then () else ( 
                        <display value="{$assoc/@displayName}"/>
                    )
                }
                </code>
        }
            <text value="{replace(string-join($itemText, ''), '^\s+|\s+$', '')}">
            {
                if ($itemText[*]) then
                    <extension url="http://hl7.org/fhir/StructureDefinition/rendering-markdown">
                        <valueMarkdown value="{serialize($itemText)}"/>
                    </extension>
                else ()
            }
            </text>
        {
            let $itemType     := adfhirq:decor2questionnaireType($concept)
            
            return (
                <type value="{$itemType}"/>,
                if ($itemType = 'display') then () else (
                    <required value="{$concept/@minimumMultiplicity = '1'}"/> |
                    <repeats value="{not($concept/@maximumMultiplicity = '0' or $concept/@maximumMultiplicity = '1')}"/>
                )
            )
        }
        {
            if ($concept/*:valueDomain/*:property/@fixed='true') then 
                <readOnly value="true"/>
            else ()
        }
        {
            if ($concept/*:valueDomain/*:property[@maxLength castable as xs:integer]) then
                <maxLength value="{max($concept/*:valueDomain/*:property[@maxLength castable as xs:integer]/xs:integer(@maxLength))}"/>
            else ()
        }
        {
            for $vs in $concept/*:valueSet[*:completeCodeSystem | *:conceptList/*:include | *:conceptList/*:exclude]
            return
                <answerValueSet value="{
                  if ($vs[@canonicalUri]) then $vs/@canonicalUri else (
                      concat(adserver:getServerURLFhirServices(),'ValueSet/',$vs/@id, '--', replace($vs/@effectiveDate,'[^\d]',''))
                  )
                }"/>
        }
        {
            let $codedConcepts  :=
                if ($concept/*:valueDomain/*:conceptList/*:concept) then (
                    for $c in $concept/*:valueDomain/*:conceptList/*:concept
                    let $associations := $concept/*:terminologyAssociation[@conceptId = $c/@id]
                    return
                        for $ta in $associations order by $ta/@codeSystem return $ta
                )
                else ()
            let $codedConcepts   :=
                if ($codedConcepts) then $codedConcepts else (
                    $concept/*:valueSet[empty(*:completeCodeSystem | *:conceptList/*:include | *:conceptList/*:exclude)]/*:conceptList/(*:concept | *:exception)
                )
            for $c in $codedConcepts
            return
                <answerOption>
                {
                    (: Ordinal Value: A numeric value that allows the comparison (less than, greater than) or other numerical manipulation of a concept 
                        (e.g. Adding up components of a score). Scores are usually a whole number, but occasionally decimals are encountered in scores. :)
                    if ($c/@ordinal) then 
                        <extension url="http://hl7.org/fhir/StructureDefinition/ordinalValue">
                            <valueDecimal value="{$c/@ordinal}"/>
                        </extension>
                    else ()
                }
                    <valueCoding>
                        <system value="{map:get($oidMap, $c/@codeSystem)}"/>
                        <code value="{$c/@code}"/>
                        <display value="{$c/@displayName}"/>
                    </valueCoding>
                {
                    if ($c/@code = $defaultValue) then
                        <initialSelected value="true"/>
                    else ()
                }
                </answerOption>
        }
        {
            if ($defaultValue) then 
                (: For code, Questionnaire expect Coding, DECOR default is just a simple string :)
                if ($concept/*:valueDomain/@type != 'code') then
                    <initial>
                    {
                        element {adfhirq:decor2questionnaireAnswerType($concept)} {
                            attribute value {$defaultValue}
                        }
                    }
                    </initial>
                else ()
            else ()
        }
        {
            (: Note: datatype string does not officially allow xhtmland will trigger a datatype error, however ... it is what the core extension is defined with
                https://chat.fhir.org/#narrow/stream/179166-implementers/topic/Questionnaire.20extension.20rendering-xhtml.20with.20datatype.20string :)
            if ($conceptDescription) then
                <item>
                    <extension url="http://hl7.org/fhir/StructureDefinition/questionnaire-itemControl">
                        <valueCodeableConcept>
                            <coding>
                                <system value="http://hl7.org/fhir/questionnaire-item-control"/>
                                <code value="help"/>
                                <display value="Help-Button"/>
                            </coding>
                            <text value="Help-Button"/>
                        </valueCodeableConcept>
                    </extension>
                    <linkId value="{$targetId}-help"/>
                    <text value="{$conceptDescriptionText}">
                        <extension url="http://hl7.org/fhir/StructureDefinition/rendering-xhtml">
                            <valueString value="{$conceptDescriptionHtml}"/>
                        </extension>
                    </text>
                    <type value="display"/>
                </item>
            else ()
        }
        </item>
    )
    default return (
        error(xs:QName('adfhirq:UNKNOWNCONCEPTTYPE'), concat('ERROR: Unknown type of concept "', $concept/@type, '" expected item or group'))
    )
};

declare %private function adfhirq:decor2questionnaireType($concept as element(concept)) as xs:string {
    let $valueDomainType    := $concept/valueDomain/@type
    let $conceptItemType    := $concept/property[@name = 'adfhirq::ItemType']
    let $conceptItem        := $conceptItemType/normalize-space(string-join(.//text()))
    return
    (: allow override of all other options by explicitly setting the questionnaire item type value.
        https://hl7.org/fhir/R4/valueset-item-type.html
    :)
    if ($conceptItemType) then 
        switch (lower-case($conceptItem))
        case 'group'
        case 'display'
        case 'question'
        case 'boolean'
        case 'decimal'
        case 'integer'
        case 'date'
        case 'time'
        case 'string'
        case 'text'
        case 'url'
        case 'choice'
        case 'open-choice'
        case 'attachment'
        case 'reference'
        case 'quantity' return $conceptItem
        case 'datetime' return 'dateTime'
        default return error(xs:QName('adfhirq:IllegalItemType'), concat('Concept id=', $concept/@id, ' effectiveDate=', $concept/@effectiveDate, ' ', $concept/name[1], ' defines unknown property adfhirq::ItemType value: ''', $conceptItem, ''''))
    else
    (: Input param is a concept, since we need to look at group :)
    if ($concept/@type = 'group') then 'group' else
    if ($valueDomainType = 'boolean') then $valueDomainType else
    if ($valueDomainType = 'date') then $valueDomainType else
    if ($valueDomainType = 'decimal') then $valueDomainType else
    if ($valueDomainType = 'quantity') then $valueDomainType else
    if ($valueDomainType = 'text') then $valueDomainType else
    if ($valueDomainType = 'time') then $valueDomainType else
    if ($valueDomainType = 'count') then 'integer' else
    if ($valueDomainType = 'datetime') then 'dateTime' else
    if ($valueDomainType = 'duration') then 'quantity' else
    if ($valueDomainType = 'currency') then 'quantity' else
    if ($valueDomainType = 'identifier') then 'string' else
    if ($valueDomainType = 'ordinal') then 'choice' else
    if ($valueDomainType = 'blob') then 'attachment' else
    if ($valueDomainType = 'code') then (
        if ($concept/*:valueSet) then
            if ($concept/*:terminologyAssociation[@strength[not(. = 'required')]]) then
                'open-choice'
            else
            if ($concept/*:valueSet[*:completeCodeSystem | *:conceptList/*:include | *:conceptList/*:exclude]) then
                'open-choice'
            else
            if ($concept/*:valueSet/*:conceptList/*[@code = 'OTH'][@codeSystem = '2.16.840.1.113883.65.1008']) then
                'open-choice'
            else (
                'choice'
            )
        else (
            'string'
        )
    )
    else (
        'string'
    )
};
declare %private function adfhirq:decor2questionnaireAnswerType($concept as element(concept)) as xs:string {
    (: Input param is a concept, since we need to look at group :)
    switch ($concept/valueDomain/@type)
    case 'count' return 'valueInteger'
    case 'date' return 'valueDate'
    case 'time'  return 'valueTime'
    case 'code' return 'valueCoding'
    case 'blob' return 'valueReference'
    default return 'valueString'
};
declare %private function adfhirq:decorStatusCode2fhirQuestionnaireStatus($status as xs:string) as element(f:status) {
    switch ($status)
    case 'new'
    case 'draft'
    case 'pending' return <status xmlns="http://hl7.org/fhir" value="draft"/>
    case 'final' return <status xmlns="http://hl7.org/fhir" value="active"/>
    case 'rejected'
    case 'cancelled'
    case 'deprecated' return <status xmlns="http://hl7.org/fhir" value="retired"/>
    default return <status xmlns="http://hl7.org/fhir" value="draft"/>
};
declare function adfhirq:fhirQuestionnaireStatusToDecorStatus($status as xs:string) as xs:string {
    switch ($status)
    case 'active' return 'final'
    case 'retired' return 'deprecated'
    default return 'draft'
};