xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace adfhirig           = "http://art-decor.org/ns/fhir/4.0/implementationguide";

import module namespace adfhir      = "http://art-decor.org/ns/fhir/4.0" at "api-fhir.xqm";
import module namespace getf        = "http://art-decor.org/ns/fhir-settings" at "fhir-settings.xqm";
import module namespace utillib     = "http://art-decor.org/ns/api/util" at "../../../api/modules/library/util-lib.xqm";
(:import module namespace setlib      = "http://art-decor.org/ns/api/settings" at "../../../api/modules/library/settings-lib.xqm";:)
import module namespace comp        = "http://art-decor.org/ns/art-decor-compile" at "../../../art/api/api-decor-compile.xqm";
import module namespace adfhirsd    = "http://art-decor.org/ns/fhir/4.0/structuredefinition" at "../api/api-fhir-structuredefinition.xqm";
import module namespace adfhirvs    = "http://art-decor.org/ns/fhir/4.0/valueset" at "../api/api-fhir-valueset.xqm";
import module namespace markdown    = "http://art-decor.org/ns/api/markdown" at "../../../api/modules/library/markdown-lib.xqm";
declare namespace f                 = "http://hl7.org/fhir";
declare namespace error             = "http://art-decor.org/ns/fhir/error";

declare %private variable $adfhirig:type            := 'ImplementationGuide';

declare function adfhirig:compileIGresources() as element() {
(: 
 : function to compile the artifacts for an IG
 : 
 : artifacts stored in a subdirectory 'reosurces' of the IG folder
 : - in the original decor format,
 :   prefixed tmp- (template) and vs- (value set) with id and effective time as file name,
 :   e.g. tmp-2.16.840.1.113883.10.22.4.9--20170302000000.xml
 :   - these may be rendered through the normal DECOR2HTML process
 : - in FHIR format, converted from the original decor format artifacts,
 :   prefixed fsd- (structure definition) and fvs- (value set) with id and effective time as file name,
 :   e.g. fvs-2.16.840.1.113883.11.21.1--20150429000000.xml
 :   - these may be rendered through a FHIR IG for CDA process
 : 
 : In addition all artifacts as zipped into a {uuid}.zip file,
 : e.g. 4746b078-e019-40a0-86b0-78e019b0a0eb.zip in the subdirectory 'reosurces'
 : 
 : A list.xml file gets 
 : - all information about the count of the artifacts
 : - the collection path to the zip file
 : - the list of all templates and value sets in original format as template and valueSet
 : - the (hopefully) succesfully converted FHIR artifacts as structuredefinition and valueSet
 : 
 : <resources artifacts="2" zip=".../resources/4746b078-e019-40a0-86b0-78e019b0a0eb.zip">
 :  <template id="..." name="tmp-...xml" statusCode="pending" count="1" file="..." path="..."/>
 :  <fsd id="..." name="fsd-...xml" statusCode="pending" count="1" file="..." path="..."/>
 :  <valueSet id="..." name="vs-...xml" statusCode="final" count="1" file="..." path="..."/>
 :  <fvs id="..." name="fvs-...xml" statusCode="final" count="1" file="..." path="..."/>
 : ...
 : </resources>
 : 
 : 
 : 
 :)
 

(: request parameters :)
let $igCollection := request:get-attribute('ig.collection')

(: home of the IG to process :)
let $homie := $igCollection
let $homie := '/db/apps/decor/implementationguides/hl7ips/ig-20230719172642' (:***TEST***:)

(:  use IG, see its resource references :)
let $ig := (collection($homie)//implementationGuide)[1]
let $igid := $ig/@id

(: make temp collection :)
let $resourcecollection := xmldb:create-collection($homie, "resources")

(: compile project :)
let $request        := collection($homie)//compile-request[1]
let $projectPrefix  := $request/@for[not(. = '*')]
let $decor          := if (empty($projectPrefix)) then () else utillib:getDecorByPrefix($projectPrefix)
let $development    := $request/@development = 'true'
let $timeStamp      := if ($development) then 'development' else $request/@on
let $filters        := $request/filters

let $templates      := $ig//definition/resource[@artefact='TM']    (: all templates referenced in definition :)
let $valuesets      := $ig//definition/resource[@artefact='VS']    (: all value sets referenced in definition :)
let $pages          := $ig//definition/page                        (: this is the index page and all others as subpages :)

let $resultsphase1  :=
    <result>
    {
        (:
            store the original DECOR IG
        :)
        let $igna   := 'decor-implementationguide.xml'
        let $stored := 
          if (empty($ig))
          then 
              concat('+++ No implementation guide found with ID ', $igid)
          else (
              (: store original decor format :)
              xmldb:store($resourcecollection, $igna, $ig)
          )
        return
            <implementationguide type="DECOR" file="{$igna}" path="{$stored}"/>
        ,
        (:
            create the FHIR IG (if requested) with special parameters
        :)
        let $figna  := 'fhir-implementationguide.xml'
        let $fpara  := ()
        let $fig    := adfhirig:createFhirIg($ig, $fpara)
        let $stored := 
          if (empty($fig))
          then 
              concat('+++ No FHIR implementation guide created based on IG ID ', $igid)
          else (
              (: store original decor format :)
              xmldb:store($resourcecollection, $figna, $fig)
          )
        return
            <implementationguide type="FHIR" file="{$figna}" path="{$stored}"/>
        ,
        (:
            store tree of pages as a separate file, although also contained in the original DECOR IG
        :)
        let $pfna := 'pages.xml'
        let $stored := 
          if (empty($pages))
          then 
              concat('+++ No text pages found in IG ', $igid)
          else (
              (: store original decor format :)
              xmldb:store($resourcecollection, $pfna, $pages)
          )
        return
            <pages count="{count($pages)}" file="{$pfna}" path="{$stored}"/>
    }
    </result>

(:  create compiled project for further processing :)
let $filters        := comp:getCompilationFilters($decor, (), ())
let $filters        := comp:getFinalCompilationFilters($decor, $filters)
let $project        := comp:compileDecor($decor, '*', $timeStamp, $filters, false(), false())
(: store compiled project :)
let $cpna           := 'compiled.xml'
let $stored         := xmldb:store($resourcecollection, $cpna, $project)

let $results        :=
    <result>
    {
        (: all of before :)
        $resultsphase1/*,

        (: all templates referenced in resources :)
        for $t in $templates (:***TEST predicate [@id='2.16.840.1.113883.10.22.4.9'] ***:)
        let $id := $t/@id
        let $me := $t/@effectiveDate
        (:let $me := max($t/xs:dateTime(@effectiveDate)):)
        (:  make up the file names :)
        let $dfna := concat('tmp-', $id, '--', replace($me,'[^\d]',''), '.xml')
        let $ffna := concat('fsd-', $id, '--', replace($me,'[^\d]',''), '.xml')
        let $sa := $t/@statusCode
        (: get artifact with certain status codes only :)
        let $et := ($project//template[@id=$id][@effectiveDate=$me][@statusCode=('draft','active','pending','review')])[1]
        let $dn := $et/@displayName
        (:  conversion to FHIR :)
        let $re :=
            if (count($et) eq 1)
            then
                try {
                    adfhirsd:convertDecorTemplate2FHIRStructureDefinition($et)
                   
                } catch * {
                    <errors>
                        <error>Caught error {$err:code}: {$err:description}</error>
                    </errors>
                }
            else
                <errors>
                    <error>Error: unexpected cardinality of artifact {$id/text()} - {count($et)} found, 1 expected</error>
                </errors>
        (: 
            build result:
            1. if artifact exist, store its original...
        :)
        let $stored := 
            if (empty($et))
            then 
                concat('+++ No artifact found with id ', $id, ' as of ', $me) 
            else (
                xmldb:store($resourcecollection, $dfna, $et[1])
            )
        (:
            2. store converted format if successfully converted or give back the caught error message
        :)
        let $storef := 
            if (empty($re) or $re/error)
            then
                concat('+++ Conversion failed for artifact with id ', $id, ' as of ', $me, ' - ', $re/node()) 
            else (
                xmldb:store($resourcecollection, $ffna, $re)
            )
        return
            (
                <template id="{$t/@id}" displayName="{$dn}" statusCode="{$sa}" count="{count($et)}" file="{$dfna}">
                {
                    if ($stored[starts-with(., '+++ ')]) 
                    then attribute error {$stored}
                    else attribute path {$stored}
                }
                </template>,
                <fsd id="{$t/@id}" displayName="{$dn}" statusCode="{$sa}" count="{count($re)}" file="{$ffna}">
                {
                    if ($storef[starts-with(., '+++ ')]) 
                    then attribute error {$storef}
                    else attribute path {$storef}
                }
                </fsd>
            )   
        ,
        (: all value sets referenced in resources :)
        for $t in $valuesets (:***TEST predicate [@id='2.16.840.1.113883.11.21.1'] ***:)
        let $id := $t/@id
        let $me := $t/@effectiveDate
        (:  make up the file names :)
        let $dfna := concat('vs-', $id, '--', replace($me,'[^\d]',''), '.xml')
        let $ffna := concat('fvs-', $id, '--', replace($me,'[^\d]',''), '.xml')
        let $sa := $t/@statusCode
        (: get artifacts with certain status codes only :)
        let $et := ($project//valueSet[@id=$id][@effectiveDate=$me][@statusCode=('draft','final','pending','review')])[1]
        let $dn := $et/@displayName
        (:  conversion to FHIR :)
        let $re :=
            if (count($et) eq 1)
            then
                try {
                    adfhirvs:convertDecorValueSet2FHIRValueSet($et, $project)
                } catch * {
                    <errors>
                        <error>Caught error {$err:code}: {$err:description}</error>
                    </errors>
                }
            else 
                <errors>
                    <error>Error: unexpected cardinality of artifact {$id/text()} - {count($et)} found, 1 expected</error>
                </errors>
        (: 
            build result:
            1. if artifact exist, store its original...
        :)
        let $stored := 
            if (empty($et))
            then
                concat('+++ No artifact found with id ', $id, ' as of ', $me) 
            else (
                xmldb:store($resourcecollection, $dfna, $et[1])
            )
        (:
            2. store converted format if successfully converted or give back the caught error message
        :)
        let $storef := 
            if (empty($re) or $re/error)
            then
                concat('+++ Conversion failed for artifact with id ', $id, ' as of ', $me, ' - ', $re/node())  
            else (
              (: store converted FHIR format :)
              xmldb:store($resourcecollection, $ffna, $re)
            )
        return
            (
                <valueSet id="{$t/@id}" displayName="{$dn}" statusCode="{$sa}" count="{count($et)}" file="{$dfna}">
                {
                    if ($stored[starts-with(., '+++ ')]) 
                    then attribute error {$stored}
                    else attribute path {$stored}
                }
                </valueSet>,
                <fvs id="{$t/@id}" displayName="{$dn}" statusCode="{$sa}" count="{count($re)}" file="{$ffna}">
                {
                    if ($storef[starts-with(., '+++ ')]) 
                    then attribute error {$storef}
                    else attribute path {$storef}
                }
                </fvs>
            )
            
    }
    </result>
    
    (: construct zip-bound <entry> elements for the documents in the resource collection :)
    let $entries := collection($resourcecollection) !
        <entry name="{util:document-name(.)}" type="xml" method="store">
        {
            serialize(., map { "method": "xml" })
        }
        </entry>
    
    (: compress the entries, store later to db :)
    let $zc := compression:zip($entries, false())
    
    (: remove entries, only list and zip will remain :)
    let $dele := collection($resourcecollection) ! xmldb:remove($resourcecollection, util:document-name(.))
    
    (: converted artifact summary zip file :)
    let $uuid := util:uuid()
    let $zipfile := concat($uuid, '.zip')

    (: store the zip :)
    let $sz := xmldb:store($resourcecollection, $zipfile, $zc)
    
    (: create the list file, store to db :)
    let $list := 
        <resources>
        {
            attribute implementationguide { $igid },
            attribute artifacts { count($results/*) },
            attribute errors { count($results//*/@error) },
            attribute zip { $sz },
            $results/*
        }
        </resources>
    let $sl := xmldb:store($resourcecollection, 'list.xml', $list)
    
    (: finally return results that are also in list.xml :)
    return
        $list
};

declare %private function adfhirig:createFhirIg($decorig as node(), $parameters as node()) as element() {

(:
<implementationGuide
  id="2.16.840.1.113883.3.1937.777.13.29.1"
  effectiveDate="2023-07-19T17:26:42"
  name="hl7ips-20230719172642-implementationguide"
  statusCode="draft"
  lastModifiedDate="2023-08-23T10:32:04"
  versionLabel="2023"
  projectId="2.16.840.1.113883.3.1937.777.13">
    <title language="en-US" lastTranslated="2023-07-19T17:27:04">International Patient Summary IPS</title>

:)

let $result :=
    <ImplementationGuide xmlns="http://hl7.org/fhir">
        <id value="{$decorig/@name}"/>
        <url value="{$decorig/@id}"/>
        <version value="{$decorig/@versionLabel}"/>
        <name value="{$decorig/@name}"/>
        <title value="{$decorig/title/text()}"/>
        <status value="{$decorig/@statusCode}"/>
        
        <publisher value="Health Level Seven"/>
        
        <contact>
            <name value="HL7 International - International Patient Summary"/>
            <telecom>
                <system value="url"/>
                <value value="http://www.hl7.org/Special/committees/structure"/>
            </telecom>
        </contact>
        
        <packageId value="hl7.cda.uv.ips"/>
        
        <fhirVersion value="5.0.0"/>
        
        <dependsOn id="cda">
            <uri value="http://hl7.org/fhir/cda/ImplementationGuide/hl7.fhir.cda"/>
            <packageId value="hl7.fhir.cda"/>
            <version value="dev"/>
        </dependsOn>
        
        <definition>
            <grouping id="document">
                <name value="Document Templates"/>
                <description value="Document-level templates ..."/>
            </grouping>
            <grouping id="header">
                <name value="Header Templates"/>
                <description value="Header-level Templates ..."/>
            </grouping>
            <grouping id="section">
                <name value="Section Templates"/>
                <description value="Section-level Templates ..."/>
            </grouping>
            <grouping id="entry">
                <name value="Entry Templates"/>
                <description value="Entry-level Templates"/>
            </grouping>
            <grouping id="other">
                <name value="Other Templates"/>
                <description value="Other templates ..."/>
            </grouping>
            <resource>
                <reference>
                    <reference value="StructureDefinition/2.16.840.1.113883.10.22.1.1--20200714160821"/>
                    <display value="HL7-IPS"/>
                </reference>
                <name value="HL7-IPS"/>
                <exampleBoolean value="false"/>
                <groupingId value="document"/>
            </resource>
            <page>
                <sourceUrl value="index.html"/>
                <name value="index.html"/>
                <title value="IG Home Page"/>
                <generation value="markdown"/>
                <page>
                    <sourceUrl value="introduction.html"/>
                    <name value="introduction.html"/>
                    <title value="Introduction"/>
                    <generation value="markdown"/>
                </page>
                <page>
                    <sourceUrl value="background.html"/>
                    <name value="background.html"/>
                    <title value="Technical Background"/>
                    <generation value="markdown"/>
                </page>
                <page>
                    <sourceUrl value="requirements.html"/>
                    <name value="requirements.html"/>
                    <title value="Functional Requirements"/>
                    <generation value="markdown"/>
                </page>
                <page>
                    <sourceUrl value="conventions.html"/>
                    <name value="conventions.html"/>
                    <title value="Design Conventions and Principles"/>
                    <generation value="markdown"/>
                </page>
                <page>
                    <sourceUrl value="artifacts.html"/>
                    <name value="artifacts.html"/>
                    <title value="Artifact Index"/>
                    <generation value="markdown"/>
                </page>
                <page>
                    <sourceUrl value="appendix.html"/>
                    <name value="appendix.html"/>
                    <title value="Appendix (Informative)"/>
                    <generation value="markdown"/>
                </page>
            </page>
            <parameter>
                <code>
                   <system value="http://hl7.org/fhir/tools/CodeSystem/ig-parameters"/>
                   <code value="releaselabel"/>
                </code>
                <value value="CI Build"/>
            </parameter>
            <parameter>
                <code>
                  <system value="http://hl7.org/fhir/tools/CodeSystem/ig-parameters"/>
                  <code value="copyrightyear"/>
                </code>
                <value value="2022+"/>
            </parameter>
        </definition>
    </ImplementationGuide>

return $result

};