xquery version "3.0";

module namespace labpanels = "http://art-decor.org/ns/labpanels";
import module namespace labterm = "http://art-decor.org/ns/labterm" at "../api/api-labterm.xqm";

declare function labpanels:makeLocalPanelConcept($concept as node()) as node() {
    <concept panelMember="original">
        {
            $concept/@*, 
            $concept/(* except concept),
            for $child in $concept/concept return labpanels:makeLocalPanelConcept($child)
        }
    </concept>
};

declare function labpanels:addLocalPanel($panelLoincId as xs:string) as xs:string {
    let $concept := labterm:getLabConceptById($panelLoincId)//lab_concept
    let $id := $concept/concept/@loinc_num
    (: Look for top-level panels to avoid catching child panels :)
    let $panel := collection($labterm:strLoincPanels)//concept[Loinc=$id][1]
    let $check :=
        if (not($concept)) then error(QName('http://art-decor.org/ns/error', 'NotAConcept'), concat('LOINC ', $panelLoincId, ' is not in Labcodeset'))
        else if (not($concept/@status=('draft', 'update'))) then error(QName('http://art-decor.org/ns/error', 'StatusError'), concat('LOINC ', $panelLoincId, " has status '", $concept/@status/string(), "' and can't be changed"))
        else if (not($panel)) then error(QName('http://art-decor.org/ns/error', 'NoLoincPanelConcept'), concat('No LOINC Panel ', $panelLoincId, ' found'))
        else if (not($concept/@status = 'draft' or $concept/@status = 'update')) then error(QName('http://art-decor.org/ns/error', 'PanelNotEditable'), concat('LOINC ', $panelLoincId, ' must be draft or update to edit it'))
        else if (not($concept/concept[elem[@name='PanelType']='Panel'])) then error(QName('http://art-decor.org/ns/error', 'NotAPanel'), concat('LOINC ', $panelLoincId, ' is not a panel'))
        else if (collection($labterm:strLocalPanels)//concept[Loinc=$id]) then error(QName('http://art-decor.org/ns/error', 'LocalPanelExist'), concat('A local panel ', $panelLoincId, ' already exists'))
        else ()
    let $localPanel := labpanels:makeLocalPanelConcept($panel)
    let $resource := xmldb:store($labterm:strLocalPanels, concat('local_panel_', $id, '.xml'), $localPanel)
    let $do := sm:chgrp($resource, 'lab') 
    return <result status="OK"/>
};

declare function labpanels:removeLocalPanel($panelLoincId as xs:string) as xs:string? {
    let $panel := $labterm:localPanels//concept[Loinc=$panelLoincId]
    let $concept := labterm:getOnlyLabConceptById($panelLoincId)
    let $check :=
        if (not($panel)) then error(QName('http://art-decor.org/ns/error', 'NotALocalPanel'), concat('LOINC ', $panelLoincId, ' is not a local panel'))
        else if (not($concept)) then error(QName('http://art-decor.org/ns/error', 'NotAConcept'), concat('LOINC ', $panelLoincId, ' is not in the Labcodeset'))
        else if (not($concept/lab_concept/@status=('draft', 'update'))) then error(QName('http://art-decor.org/ns/error', 'StatusError'), concat('LOINC ', $panelLoincId, " has status '", $concept/lab_concept/@status/string(), "' and can't be changed"))
        else ()
    let $do := xmldb:remove($labterm:strLocalPanels, concat('local_panel_', $panelLoincId, '.xml'))
    return <result status="OK"/>
};

declare function labpanels:replaceInLocalPanel($panelLoincId as xs:string, $oldLoincId as xs:string, $newLoincId as xs:string) as node() {
    let $newConcept := labterm:getLabConceptById($newLoincId)//lab_concept
    let $panelConcept := labterm:getLabConceptById($panelLoincId)//lab_concept
    let $localPanel := collection($labterm:strLocalPanels)//concept[Loinc=$panelLoincId][1]
    let $check :=
        if (not($newConcept)) then error(QName('http://art-decor.org/ns/error', 'NotAConcept'), concat('LOINC ', $newLoincId, ' is not in Labcodeset'))
        else if (not($localPanel/concept[Loinc=$oldLoincId])) then error(QName('http://art-decor.org/ns/error', 'NotAMember'), concat('LOINC ', $oldLoincId, ' is not a member of panel', $panelLoincId))
        else if (not($panelConcept/@status = 'draft' or $panelConcept/@status = 'update')) then error(QName('http://art-decor.org/ns/error', 'PanelNotEditable'), concat('LOINC ', $panelLoincId, ' must be draft or update to edit it'))
        else if (not($panelConcept/concept[elem[@name='PanelType']='Panel'])) then error(QName('http://art-decor.org/ns/error', 'NotAPanel'), concat('LOINC ', $panelLoincId, ' is not a panel'))
        else ()
    let $oldPanelData := $localPanel/concept[Loinc=$oldLoincId]
    let $xml :=
        <concept panelMember="added">
            {$oldPanelData/ParentId, $oldPanelData/ParentLoinc, $oldPanelData/ParentName, $oldPanelData/ParentId, $oldPanelData/ParentId, $oldPanelData/ID, $oldPanelData/SEQUENCE}
            <Loinc>{$newConcept/concept/@loinc_num/string()}</Loinc>
            <LoincName>{$newConcept/concept/shortName/string()}</LoincName>
            <ObservationRequiredInPanel>O</ObservationRequiredInPanel>
        </concept>
    let $update := update insert $xml following $oldPanelData
    let $update := update replace $oldPanelData/@panelMember with 'removed'
    return $xml
};

declare function labpanels:changeLocalPanelMembers($panelLoincId as xs:string, $loincId as xs:string, $membership as xs:string) as node() {
    let $panel := $labterm:localPanels//concept[Loinc=$panelLoincId]
    let $panelConcept := labterm:getLabConceptById($panelLoincId)
    let $concept := labpanels:getLocalPanelMemberById($panelLoincId, $loincId)
    let $check :=
        if (not($panel)) then error(QName('http://art-decor.org/ns/error', 'NotALocalPanel'), concat('LOINC ', $panelLoincId, ' is not a local panel'))
        else if (not($concept)) then error(QName('http://art-decor.org/ns/error', 'NotAConcept'), concat('LOINC ', $panelLoincId, ' is not in the Labcodeset'))
        else if (not($panelConcept/lab_concept/@status=('draft','update'))) then error(QName('http://art-decor.org/ns/error', 'StatusError'), concat('LOINC ', $panelLoincId, " has status '", $panelConcept/lab_concept/@status, "' and can't be changed"))
        else ()
    let $update := 
        if ($concept/lab_concept/@panelMember="added")
        then update delete $labterm:localPanels/concept[Loinc=$panelLoincId]//concept[Loinc=$loincId]
        else update replace $labterm:localPanels/concept[Loinc=$panelLoincId]//concept[Loinc=$loincId]/@panelMember with $membership
    return <result status="OK"/>
};

declare function labpanels:findMassVsSubstanceConcepts($loincId as xs:string) as element()* {
    let $ori := $labterm:loincConcepts/concept[@loinc_num=$loincId]
    (: mass vs substance alternatieven :)
    let $targets := $labterm:labConcepts/lab_concept/concept[component=$ori/component][timing=$ori/timing][system=$ori/system][scale=$ori/scale][(not(method) and not($ori/method)) or (method=$ori/method)][property!=$ori/property]
    return $targets
};

(:~ Get panel node with LOINC concepts:)
declare function labpanels:getLabPanelWithConcepts($panelChild as node(), $language as xs:string, $level as xs:string) as node()* {
    let $loincId        := $panelChild/Loinc/string()
    let $loincConcept   := collection($labterm:strLoincData)//concept[@loinc_num=$loincId]
    let $labConcept     := labterm:getOnlyLabConceptById($loincId)/lab_concept
    let $result :=
        <lab_concept level="{$level}">
            {
            $panelChild/@*,
            if ($labConcept) then $labConcept/@* else attribute status {'potential'}
            ,
            $panelChild/(* except concept)
            ,
            if ($labConcept) 
            then $labConcept/* 
            else
                <concept>
                    {$loincConcept/@*, $loincConcept/(* except concept), $loincConcept/concept[@language=$language]}
                </concept>
            }
        </lab_concept>
    (: NL LongName has already been added by getOnlyLabConceptById for lab concepts, so we need it only for LOINC concepts :)
    let $result := if ($labConcept) then $result else labterm:addLongNameNL($result)
    let $level          := concat($level, '--') 
    return 
        (
        $result,
        for $child in $panelChild/concept
        order by xs:integer($child/SEQUENCE) ascending
        return labpanels:getLabPanelWithConcepts($child, $language, $level)
        )

};

(: Get the panel as a tree.
For panels, we will always search loinc, since each panel member MUST be present in Labcodeset if the panel is present.
:)
declare function labpanels:getLabPanelTreeById($loincId as xs:string, $language as xs:string) as element()? {
    let $loincConcept := collection($labterm:strLoincData)//concept[@loinc_num=$loincId][1]
    (: There may be multiple panels with the same Loinc, i.e. organizers which appear in multiple panels. They should all be the same, so just pick the first :)
    let $localPanel := collection($labterm:strLocalPanels)/concept[Loinc=$loincId][1]
    let $panelTree  := 
        if ($localPanel) then <panel type="localpanel">{$localPanel}</panel> 
        else if($loincConcept/elem[@name='PanelType']) then <panel type="panel">{collection($labterm:strLoincPanels)//concept[Loinc=$loincId][1]}</panel> else ()
    return $panelTree
};

(: Get the panel as a tree for publication (no "removed" concepts :)
declare function labpanels:getFinalLabPanelConcept($concept as element()) as element()? {
    if ($concept/@panelMember = 'removed')
    then ()
    else
        <concept>{$concept/@*, $concept/(* except concept), for $child in $concept/concept return labpanels:getFinalLabPanelConcept($child)}</concept>
};

(: Get the panel as a tree for publication (no "removed" concepts :)
declare function labpanels:getFinalLabPanelTreeById($loincId as xs:string, $language as xs:string) as element()? {
    let $panel := labpanels:getLabPanelTreeById($loincId, $language)
    return <panel>{$panel/@*, $panel/(* except concept), for $child in $panel/concept return labpanels:getFinalLabPanelConcept($child)}</panel>
};

(: Get panel by id.
This function returns a flat list with @level, not a tree. Easier for display in front-end. :)
declare function labpanels:getLabPanelById($loincId as xs:string, $language as xs:string) as element()? {
    let $panelTree  := 
        if (labterm:isLabUser()) 
        then labpanels:getLabPanelTreeById($loincId, $language)
    (: Guest users only see original + added concepts :)
        else labpanels:getFinalLabPanelTreeById($loincId, $language)
    let $results    :=
        if (not($panelTree)) then error(QName('http://art-decor.org/ns/error', 'NoLoincPanelConcept'), concat('No LOINC Panel ', $loincId, ' found')) 
        else labpanels:getLabPanelWithConcepts($panelTree/concept, $language, '')
    (: We return a list with panel items. The first one is the panel itself, the next ones are ordered by Sequence (item in panel), and have a 'level' attribute :)
    return <result count="{count($results)}" search="{$loincId}" mode="{$panelTree/@type}" current-user="{labterm:get-current-user()}">{$results}</result>
};

declare function labpanels:getLocalPanelMemberById($panelLoincId as xs:string, $loincId as xs:string) as element()? {
    let $localPanel := $labterm:localPanels/concept[Loinc=$panelLoincId]
    let $check := if (not($localPanel)) then error(QName('http://art-decor.org/ns/error', 'NoLocalPanel'), concat('LOINC  ', $panelLoincId, '  is not a local panel ')) else ()
    let $panelMember := $localPanel//concept[Loinc=$loincId]
    let $level := replace(string-join($panelMember/ancestor::concept/local-name()), 'concept', '--')
    let $result :=
        if (not($panelMember)) then error(QName('http://art-decor.org/ns/error', 'NoLocalPanelMember'), concat('LOINC  ', $loincId, '  is not a member of local panel ', $panelLoincId)) 
        else labpanels:getLabPanelWithConcepts($panelMember, 'nl-NL', $level)
    return <result status="OK">{$result}</result>
};

(: Get Panel parents IF this concept is child of such a panel :)
declare function labpanels:getActiveLabPanelParents($loincId as xs:string) as xs:string* {
    (: Get all top-level concepts in LOINC Panels who have a descendant with $loincId :)
    let $panelParents   := collection($labterm:strLoincPanels)//Panels/concept[descendant::concept[Loinc=$loincId]]/Loinc/string()
    let $activeLabParents     := $labterm:labConcepts//concept[@loinc_num=$panelParents][../@status='active']/@loinc_num
    return $activeLabParents
};
