xquery version "3.0";

module namespace labterm            = "http://art-decor.org/ns/labterm";
import module namespace snomed     = "http://art-decor.org/ns/terminology/snomed" at "../../terminology/snomed/api/api-snomed.xqm";
import module namespace adloinc     = "http://art-decor.org/ns/terminology/loinc" at "../../terminology/loinc/api/api-loinc.xqm";
import module namespace nlLongName  = "http://art-decor.org/ns/nlLongName" at "api-nl-longname.xqm";
import module namespace labsearch   = "http://art-decor.org/ns/labsearch" at "../api/api-labsearch.xqm";
import module namespace labpanels   = "http://art-decor.org/ns/labpanels" at "../api/api-labpanels.xqm";
import module namespace console     = "http://exist-db.org/xquery/console";

declare variable $labterm:root              := repo:get-root();
declare variable $labterm:strTerminologyData := concat($labterm:root,'terminology-data');
declare variable $labterm:strLoincData      := concat($labterm:strTerminologyData,'/loinc-data/data/');
declare variable $labterm:strLoincPanels    := concat($labterm:strTerminologyData,'/loinc-data/panels/');
declare variable $labterm:strLab            := concat($labterm:root,'lab');
declare variable $labterm:strLabData        := concat($labterm:root,'lab-data/data');
declare variable $labterm:labConcepts       := collection(concat($labterm:strLabData, '/lab_concepts'));
declare variable $labterm:strLocalPanels    := concat($labterm:strLabData, '/local_panels');
declare variable $labterm:localPanels       := collection($labterm:strLocalPanels);
declare variable $labterm:strLabDataLog     := concat($labterm:root,'lab-data/log');
declare variable $labterm:LabDataLog        := collection($labterm:strLabDataLog);
declare variable $labterm:strLabPublications    := concat($labterm:root,'lab-data/publications');
declare variable $labterm:labMaterials      := doc(concat($labterm:strLabData, '/materials.xml'))/materials;
declare variable $labterm:labLoincSnomedMapping := doc(concat($labterm:strLabData, '/loincsystem-to-snomed.xml'))/map;
declare variable $labterm:labMethods        := doc(concat($labterm:strLabData, '/methods.xml'))/methods;
declare variable $labterm:labUnits          := doc(concat($labterm:strLabData, '/units.xml'))/units;
declare variable $labterm:labOrdinals       := doc(concat($labterm:strLabData, '/ordinals.xml'))/ordinals;
declare variable $labterm:labNominals       := doc(concat($labterm:strLabData, '/nominals.xml'))/nominals;
declare variable $labterm:loincConcepts     := doc(concat($labterm:root, 'terminology-data/loinc-data/data/universal/', 'LOINC_DB_with_LinguisticVariants.xml'))/loinc_db;
declare variable $labterm:loincPrerelease   := doc(concat($labterm:strLabData, '/prerelease.xml'));
declare variable $labterm:strLabConcept     := concat($labterm:strLabData,'/lab_concepts');

(:~ Get current user name. effective user name if available or real user name otherwise. The function labterm:get-current-user() was removed 
:   in eXist-db 5.0 and replaced with sm:id(). By centralizing the new slightly more complicated way of doing this, you can avoid making 
:   mistakes throughout the code base
:   @return user name string.
:   @since 2019-11-11
:)
declare function labterm:get-current-user() as xs:string? {
    sm:id()//sm:real/sm:username/string()
};
(:~
This function gets the errors for a concept.

Update the errors table when code changes at:
https://informatiestandaarden.nictiz.nl/wiki/Landingspagina_Labcodeset#Errors
:)
declare function labterm:getErrors($labConcept as node(), $language as xs:string) as node()* {
    let $loinc_num      := $labConcept/concept/@loinc_num/string()
    let $loincConcept   := $labterm:loincConcepts//concept[@loinc_num = $loinc_num]
    let $errors := ()

    let $errors :=
        if($loincConcept or $labConcept/concept/@status="PRERELEASE")
        then $errors
        else ($errors, <error src="checkConcept" code="MISSING" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} is not present in LOINC</error>)
    let $errors :=
        if($loincConcept and $loincConcept[@status = 'ACTIVE']) 
        then $errors
        (: Warning, status TRIAL is not an error in itself :)
        else if($loincConcept and $loincConcept[@status = 'TRIAL']) 
        then ($errors, <warning src="checkConcept" code="TRIAL" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} has LOINC status 'TRIAL'</warning>)
        else if($labConcept/concept/@status="PRERELEASE")
        then $errors
        else if($labConcept/@status = 'retired') 
        then $errors
        else ($errors, <error src="checkConcept" code="STATUS" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} has status {$loincConcept/@status/string()} and lab concept has status {$labConcept/@status/string()}</error>)
    let $errors :=
        (: See BITS LCB-117, check on OrdQn is disabled for the time being :)
        (: Scale Qn must have a unit, except panel types. :)
        if($labConcept[concept/scale!='Qn' or units/unit or concept/elem/@name='PanelType'])
        then $errors
        else ($errors, <error src="checkConcept" code="NOUNIT" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} with scale {$labConcept/concept/scale/text()} has no unit</error>)
    let $errors :=
        ($errors,
        for $unitRef in $labConcept//unit/@ref
        return 
            if ($labterm:labUnits/unit[@id=$unitRef])
            then ()
            else ($errors, <error src="checkConcept" code="UNITREF" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} has unit ref {$unitRef/string()} which is missing in units.</error>)
        )
    let $errors :=
        ($errors,
        for $materialRef in $labConcept//material/@ref
        return 
            if ($labterm:labMaterials//material[@id=$materialRef])
            then ()
            else ($errors, <error src="checkConcept" code="MATERIALREF" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} has material ref {$materialRef/string()} which is missing in materials.</error>)
        )
    let $errors :=
        ($errors,
        for $methodRef in $labConcept//method/@ref
        return 
            if ($labterm:labMethods//method[@id=$methodRef])
            then ()
            else ($errors, <error src="checkConcept" code="METHODREF" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} has method ref {$methodRef/string()} which is missing in methods.</error>)
        )
    let $errors :=
        ($errors,
        for $valueSetRef in $labConcept//valueSet/@ref
        return 
            if ($labterm:labOrdinals//valueSet[@id=$valueSetRef])
            then ()
            else ($errors, <error src="checkConcept" code="VALUESETREF" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} has valueSet ref {$valueSetRef/string()} which is missing in valueSets.</error>)
        )
    let $errors :=
        if($labConcept//concept[@language=$language]/component) 
        then $errors
        else ($errors, <warning src="checkConcept" code="NOCOMP" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} has no component in {$language}</warning>)
    let $errors :=
        if($labConcept/concept/@status != "PRERELEASE")
        then $errors
        else ($errors, <warning src="checkConcept" code="PRERELEASE" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} has status PRERELEASE</warning>)
    let $errors :=
        if ($labterm:labLoincSnomedMapping//concept[LOINCSystem=$labConcept/concept/system/string()])
        then $errors
        else ($errors, <error src="checkConcept" code="NOMAPPING" dateTime="{fn:current-dateTime()}">No LOINC-Snomed mapping for {$loinc_num} for system {$labConcept/concept/system/string()}</error>)
    let $panel_check :=
        if ($labConcept/concept/elem/@name='PanelType')
        then
            let $panelTree  := labpanels:getLabPanelById($loinc_num, 'nl-NL')
            let $check :=  
                if (not($panelTree))
                then <error src="checkConcept" code="NOPANEL" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} is of type panel, but panel cannot be found</error>
                else if ($panelTree//lab_concept[not(@panelMember='removed') and @status = 'potential'])
                then <error src="checkConcept" code="PANELMEMBER" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} is a panel and some panel members are not part of the Labcodeset</error>
                (: panel members should be active, except for the panel itself, and removed members :)
                else if ($panelTree//lab_concept[not(concept[@loinc_num=$loinc_num])][not(@panelMember='removed')][@status != 'active'])
                then <error src="checkConcept" code="PANELSTATUS" dateTime="{fn:current-dateTime()}">LOINC concept {$loinc_num} is a panel with status 'active' and some panel members are not 'active'</error>
                else 'OK'
            return $check
        else
            'OK'
    let $errors :=
        if($panel_check = 'OK') 
        then $errors
        else ($errors, $panel_check)

    return $errors
};

(:~
This function will check a concept for errors.
- will insert <errors> element with errors when there are errors
- will delete <errors> if no errors are found
An errors element may contain multiple errors.

If lab_concept/@status='active' status will be set to 'update' if there are errors.
:)
declare function labterm:checkConcept($loinc_num as xs:string?, $language as xs:string) as node() {
    let $labConcept     := $labterm:labConcepts/lab_concept[concept[@loinc_num = $loinc_num]]
    let $errors         := labterm:getErrors($labConcept, $language)
    let $do := update delete $labConcept/errors
    let $do := 
        if ($errors) 
        then (
            update insert <errors loinc_num="{$loinc_num}">{$errors}</errors> into $labConcept
            ,
            (: no status change for warnings :)
            if ($labConcept/@status = 'active' and count($errors[local-name() = 'error']) >= 1) 
            then (
                let $logdata := <statusChange object="concept" statusCode="update" effectiveTime="{substring(xs:string(current-dateTime()), 1, 19)}" user="system" loincId="{$loinc_num}">{$labConcept}</statusChange>
                return 
                    (
                    labterm:appendLog($logdata),
            update replace $labConcept/@status with 'update', 
                    update replace $labConcept/@user with 'system'
                    )
            )
            else ()
        ) else ()
    return if ($errors) then <result loinc_num="{$loinc_num}" status="ERRORS">{$errors}</result> else <result loinc_num="{$loinc_num}" status="OK"/>
};

declare function labterm:checkAllConcept($statusList as xs:string*) as node()* {
    let $concepts := if ($statusList) then $labterm:labConcepts/lab_concept[@status=$statusList] else $labterm:labConcepts/lab_concept
    let $panels := $concepts[concept/elem[@name='PanelType']]
    let $nonpanels := $concepts[not(concept/elem[@name='PanelType'])]
    return
    <lab_concepts> {
        for $labConcept in $nonpanels
        return labterm:checkConcept($labConcept/concept/@loinc_num, 'nl-NL')
        ,
        for $labConcept in $panels
        return labterm:checkConcept($labConcept/concept/@loinc_num, 'nl-NL')
        }
    </lab_concepts>
};

(: This function will append the log in /lab-data/log/{$logFileName}.xml if the file is already present. Otherwise will create the file and then write the log provided as parameter :)
declare function labterm:appendLog($logData as node()) as xs:boolean {
    let $logFileDir := concat($labterm:strLabDataLog,  '/')
    let $logFileName := concat('transactions-',substring(xs:string(current-dateTime()), 1, 10),'.xml')
    let $createLogFile :=
    if (not(doc-available(concat($logFileDir,$logFileName)))) then (
        xmldb:store($logFileDir, $logFileName, <logs/>),
        sm:chmod(xs:anyURI(concat($logFileDir,$logFileName)),'rwxrwxr--'),
        sm:clear-acl(xs:anyURI(concat($logFileDir,$logFileName)))
    )
    else()
    let $log := doc(concat($logFileDir, $logFileName))/logs   
    let $appendLog :=  
            if($log and sm:has-access(xs:anyURI(concat($logFileDir, $logFileName)),'w')) then
                update insert $logData into $log
            else()    
    return true()    
};
(: This function adds active concepts in publication file and updates the reference of given node with actual node from the database. :)
declare function labterm:updatePublication($publicationFilePath as xs:string, $currentDate as xs:string) as element()? {
    let $publication := doc($publicationFilePath)/publication
    let $update := update replace $publication/@effectiveDate with $currentDate
    let $update := 
        (: getOnlyLabCoonceptById will get the last active concept, if any, for update and pending :)
        for $lab_concept in $labterm:labConcepts/lab_concept[@status = ('update', 'pending', 'active', 'retired')]
        let $content := labterm:getOnlyLabConceptById($lab_concept/concept/@loinc_num, true())/lab_concept 
        return if ($content) then update insert $content into $publication/lab_concepts else () 
    let $update := update insert $labterm:labMaterials/.. into $publication
    let $update := update insert $labterm:labMethods/.. into $publication
    let $update := update insert $labterm:labUnits/.. into $publication
    let $update := update insert $labterm:labOrdinals/.. into $publication
    let $update := update insert $labterm:labNominals/.. into $publication
    let $panels := 
    <panels>{
        for $concept in $publication//lab_concept/concept[elem[@name="PanelType"]='Panel']
        return labpanels:getFinalLabPanelTreeById($concept/@loinc_num, 'nl=NL')
    }</panels>
    let $update := update insert $panels into $publication
    return $update
};

(: This function changes the status of the concept having concept id passed in second parameter with status passed in first paramter.  :)
declare function labterm:changeStatus($status as xs:string, $loincId as xs:string) as node() {
    let $check := labterm:checkConcept($loincId, 'nl-NL')
    let $concept := labterm:getOnlyLabConceptById($loincId)/lab_concept
    let $oldStatus := $concept/@status/string()
    
    let $allowed := 
        if (not($oldStatus)) then concat('No concept ', $loincId)
        else if ($oldStatus = 'initial' and ($status = 'draft' or $status = 'rejected')) then 'OK'
        else if ($oldStatus = 'draft' and ($status = 'pending' or $status = 'active' or $status = 'rejected')) then 'OK'
        else if ($oldStatus = 'pending' and ($status = 'update' or $status = 'active')) then 'OK'
        else if ($oldStatus = 'update' and ($status = 'pending' or $status = 'active' or $status = 'retired')) then 'OK'
        else if ($oldStatus = 'active' and ($status = 'update' or $status = 'retired')) then 'OK'
        else if ($oldStatus = 'retired' and $status = 'draft' and labterm:isLabAdmin()) then 'OK'
        else if ($oldStatus = 'retired' and $status = 'update' and labterm:isLabAdmin()) then 'OK'
        else concat('You cannot change ', $oldStatus, ' to ', $status, ' for concept ', $loincId)
    let $allowed := 
        if ($allowed != 'OK') then $allowed
        else 
            (: Admin may make concepts active if only error is no translation :)
            if ($status != 'active')
            then $allowed
            else if (not($concept//error))
            then $allowed
            else concat('You cannot make concept ', $loincId, ' active when it has errors.')
    (: For panels, only allow active when all in panel are active too :)
    let $panelAllowed :=
        if ($status != 'active' or not($concept/concept/elem[@name='PanelType']))
        then true()
        else
            let $panelTree  := labpanels:getLabPanelById($loincId, 'nl-NL')
            return 
                (: Check if other concepts than the current are not active :)
                if ($panelTree//lab_concept[concept/@loinc_num!=$loincId][(@status != 'active' and @panelMember != 'removed')])
                then false()
                else true()
    (: If there are active Panel parents, do not allow making this concept non-active :)
    let $panelParentsOk :=
        if ($status != 'active' and count(labpanels:getActiveLabPanelParents($loincId)) > 0)
        then false()
        else true()
    let $update := 
        if ($allowed = 'OK' and $panelAllowed and $panelParentsOk) then 
            if (labterm:updateStatus($status, $loincId)) then 'OK' else 'Unexpected error'
        else 
            if (not($panelAllowed))
            then 'Status change for panel is not allowed, not all members are active.'
            else if (not($panelParentsOk))
            then concat('Changing a concept is not allowed if it has an active parent (see parent: ', string-join(labpanels:getActiveLabPanelParents($loincId), ' '), ') ')
            else if ($allowed != 'OK') 
            then $allowed 
            else 'Status change is not allowed.'
    let $concept := labterm:getOnlyLabConceptById($loincId)/lab_concept
    let $result := <result status="{if ($update = 'OK') then 'OK' else 'NOK'}" message="{$update}">{$concept}</result>
    return $result
};

(: This function checks if the current user is of lab group or not. :)
declare function labterm:isLabUser() as xs:boolean {
    let $user := labterm:get-current-user()
    let $groups := sm:get-user-groups($user)
    return if ($groups= 'lab') then true() else false()
};

(: This function checks if the current user is of lab group or not. :)
declare function labterm:isLabAdmin() as xs:boolean {
    let $user := labterm:get-current-user()
    let $groups := sm:get-user-groups($user)
    return if ($groups= 'lab-admin') then true() else false()
};

(: This function changes the status of the concept and deletes the concept from the database if the new status is rejected :)
declare function labterm:updateStatus($status as xs:string, $loincId as xs:string) as xs:boolean { 
    let $update := 
        if($status = 'rejected') then (
            let $deleteResource :=  
                if(doc-available(concat($labterm:strLabConcept,'/lab_concept_', $loincId, '.xml'))) 
                then (xmldb:remove($labterm:strLabConcept, concat('lab_concept_', $loincId, '.xml'))) 
                else (error(QName('http://art-decor.org/ns/error', 'ConceptDoesNotExist'), concat('Concept ', $loincId, ' is not present.'))
            )
            return true()
        ) else 
            let $labConcept := $labterm:labConcepts//@loinc_num[.=$loincId]/ancestor::lab_concept
            let $logdata := <statusChange object="concept" statusCode="{$status}" effectiveTime="{substring(xs:string(current-dateTime()), 1, 19)}" user="{labterm:get-current-user()}" loincId="{$loincId}">{$labConcept}</statusChange>
            let $logdata := labterm:appendLog($logdata)
            let $update := 
                if ($labConcept/@previousStatus) then
                    update replace $labConcept/@previousStatus with $labConcept/@status
            else update insert attribute {'previousStatus'} {$labConcept/@status} into $labConcept
            let $update := 
                if ($labConcept/@previousUser) then
                    update replace $labConcept/@previousUser with $labConcept/@user
            else update insert attribute {'previousUser'} {$labConcept/@user} into $labConcept
            let $update := 
                if ($labConcept/@last_update) then
                    update replace $labConcept/@last_update with current-dateTime()
                    else update insert attribute {'last_update'} {current-dateTime()} into $labConcept
            let $update := update replace $labConcept/@status with $status
            let $update := update replace $labConcept/@user with labterm:get-current-user()
            return true()
        
        return true()
};

declare function labterm:addLabConcept($loincId as xs:string, $language as xs:string) as node() {
    let $concept            := labterm:getLabConceptById($loincId, $language, true(), true(), '')/lab_concept
    let $concept            := if ($concept/concept) then $concept else labterm:getPrereleaseConcept($loincId)
    let $conceptFilePath    := concat($labterm:strLabConcept, '/lab_concept_', $loincId, '.xml')
    (: If document for that loinc is already present then it's an error. :)
    let $checkFileForLabConcept := 
        if (doc-available($conceptFilePath)) then (
            error(QName('http://art-decor.org/ns/error', 'ConceptAlreadyExists'), concat('Concept ', $loincId, ' already exists.'))
        ) else ()
    
    (: For panels, create all concepts which do not exist in lab concepts :)
    let $doPanel :=     
        if (not($concept/concept/elem[@name='PanelType']))
        then ()
        else
            let $panelTree  := collection($labterm:strLoincPanels)//concept[Loinc=$loincId]
            for $concept in $panelTree/concept
            let $panelConcept := labterm:getOnlyLabConceptById($concept/Loinc/string())
            return 
                if ($panelConcept/lab_concept) then ()
                else
                labterm:addLabConcept($concept/Loinc/string(), $language)
    let $labConcept := 
        <lab_concept status="draft" user="{labterm:get-current-user()}" last_update="{current-dateTime()}">
            {$concept/(@* except @status), $concept/*}
        </lab_concept>
    let $errors := labterm:getErrors($labConcept, $language)
    let $labConcept := 
        <lab_concept>
            {$labConcept/@*, $labConcept/*, if ($errors) then <errors>{$errors}</errors> else ()}
        </lab_concept>
    let $createFileForLabConcept    := xmldb:store($labterm:strLabConcept, concat('/lab_concept_', $loincId, '.xml'), $labConcept)
    let $perms                      := sm:chmod(xs:anyURI($conceptFilePath), 'rwxrwxr--')
    let $perms                      := sm:chgrp(xs:anyURI($conceptFilePath), 'lab')
    let $perms                      := sm:clear-acl(xs:anyURI($conceptFilePath))
    let $result := <result status="OK">{$labConcept}</result>
    let $logdata := <statusChange object="concept" statusCode="draft" effectiveTime="{substring(xs:string(current-dateTime()), 1, 19)}" current-user="{labterm:get-current-user()}" loincId="{$loincId}"/>
    let $logdata := labterm:appendLog($logdata)
    return $result

};

declare function labterm:saveUnit($unit as node()) as xs:boolean {
    if ($unit/rm = '') 
    then error(QName('http://art-decor.org/ns/error', 'UnitEmpty'), 'Unit cannot be empty.')
    else if ($unit/name = '') 
    then error(QName('http://art-decor.org/ns/error', 'NameEmpty'), 'Name cannot be empty.')
    else if ($unit/nlname = '') 
    then error(QName('http://art-decor.org/ns/error', 'NlNameEmpty'), 'NL name cannot be empty.')
    else
        if ($labterm:labUnits//unit[@id=$unit/@id])
        then
            let $do := update replace $labterm:labUnits//unit[@id=$unit/@id] with <unit id="{$unit/@id}" status="active">{$unit/*}</unit>
            return true()
        else
            if ($labterm:labUnits//unit[rm=$unit/rm])
            then error(QName('http://art-decor.org/ns/error', 'UnitExists'), 'Unit already exists.')
            else 
            let $newId := max($labterm:labUnits//@id) + 1
            let $do := update insert <unit id="{$newId}" status="active">{$unit/*}</unit> into $labterm:labUnits
            return true()
};

declare function labterm:addMaterial($materialCode as xs:string) as node() {
    let $materialAlreadyExist := $labterm:labMaterials//material[@code = $materialCode]
    let $maxId := max($labterm:labMaterials//material/@id/xs:integer(string())) + 1
    let $concept := snomed:getRawConcept($materialCode)
    (: Name, NL if available :)
    let $name       := if ($concept/desc[@type='fsn'][@active='1'][@language='nl']) then $concept/desc[@type='fsn'][@active='1'][@language='nl'] else $concept/desc[@type='fsn'][@active='1'][1]
    (: Get the concepts, only active :)
    let $substance  := $concept/grp/src[@typeId="370133003"][@active='1'][1]
    let $topo       := $concept/grp/src[@typeId="118169006"][@active='1'][1]
    let $morph      := $concept/grp/src[@typeId="118168003"][@active='1'][1]
    let $ident      := $concept/grp/src[@typeId="118171007"][@active='1'][1]
    let $proc       := $concept/grp/src[@typeId="118170006"][@active='1'][1]
    
    let $newMaterial := 
        <material id="{$maxId}" code="{$materialCode}" displayName="{($concept/desc[@type='fsn'][@active='1'])[1]/string()}" status="draft">
            <name>{$name/string()}</name>
            <root code="123038009" displayName="Specimen"/>
            { 
            if ($substance) then <substance code="{$substance/@destinationId}" displayName="{$substance/string()}">{$substance/@destinationId/string()}</substance> else (),
            if ($topo) then <topo code="{$topo/@destinationId}" displayName="{$topo/string()}">{$topo/@destinationId/string()}</topo> else (),
            if ($morph) then <morph code="{$morph/@destinationId}" displayName="{$morph/string()}">{$morph/@destinationId/string()}</morph> else (),
            if ($ident) then <ident code="{$ident/@destinationId}" displayName="{$ident/string()}">{$ident/@destinationId/string()}</ident> else (),
            if ($proc) then <proc code="{$proc/@destinationId}" displayName="{$proc/string()}">{$proc/@destinationId/string()}</proc> else ()
            }
        </material>
    
    let $do := 
        if($materialAlreadyExist) 
        then error(QName('http://art-decor.org/ns/error', 'MaterialExists'), 'Material already exists.')
        else update insert $newMaterial into $labterm:labMaterials
    return <result status="OK"/>
};

declare function labterm:addMethod($methodCode as xs:string) as node() {
    let $methodAlreadyExist := $labterm:labMethods//Method[@code = $methodCode]
    let $maxId := max($labterm:labMethods//method/@id/xs:integer(string())) + 1
    let $concept := snomed:getRawConcept($methodCode)
    (: Name, NL if available :)
    let $name       := if ($concept/desc[@type='fsn'][@active='1'][@language='nl']) then $concept/desc[@type='fsn'][@active='1'][@language='nl'] else $concept/desc[@type='fsn'][@active='1'][1]
    (: Get the concepts, only active :)
    
    let $newMethod := 
        <method id="{$maxId}" code="{$methodCode}" displayName="{($concept/desc[@type='fsn'][@active='1'])[1]/string()}" status="draft">
            <name>{$name/string()}</name>
        </method>
    
    let $do := 
        if($methodAlreadyExist) 
        then error(QName('http://art-decor.org/ns/error', 'MethodExists'), 'Method already exists.')
        else update insert $newMethod into $labterm:labMethods
    return <result status="OK"/>
};

declare function labterm:removeUnit($unit as node()) as xs:boolean {
    if ($labterm:labConcepts//unit[@ref=$unit/@id])
    then error(QName('http://art-decor.org/ns/error', 'UnitInUse'), 'Unit is in use by concepts.')
    else if (not(labterm:isLabAdmin()))
    then error(QName('http://art-decor.org/ns/error', 'UserIsNotAdmin'), 'Only admin users can remove units.')
    else
        let $do := update value $labterm:labUnits//unit[@id=$unit/@id]/@status with "retired"
        return true()
};

declare function labterm:updateComment($loincId as xs:string, $comment as xs:string, $language as xs:string) as xs:boolean {
    let $oldConcept := $labterm:labConcepts/lab_concept[concept[@loinc_num=$loincId]]
    let $commentElement := <comment>{$comment}</comment>
    let $update := 
        if($oldConcept) then
            if ($oldConcept/comment)
            then
 if (string-length($comment) = 0)
                then update delete $oldConcept/comment
                else update replace $oldConcept/comment with $commentElement
            else
                update insert $commentElement into $oldConcept
        else (
            error(QName('http://art-decor.org/ns/error', 'NoConceptPresent'), 'No concept present for the given language.')
        )       
    return true()
};

declare function labterm:updateReleaseNote($loincId as xs:string, $note as xs:string, $language as xs:string) as xs:boolean {
    let $oldConcept := $labterm:labConcepts/lab_concept[concept[@loinc_num=$loincId]]
    let $commentElement := <releasenote>{$note}</releasenote>
    let $update := 
        if($oldConcept) then
            if ($oldConcept/releasenote)
            then 
                if (string-length($note) = 0)
                then update delete $oldConcept/releasenote
                else update replace $oldConcept/releasenote with $commentElement
            else
                update insert $commentElement into $oldConcept
        else (
            error(QName('http://art-decor.org/ns/error', 'NoConceptPresent'), 'No concept present for the given language.')
        )       
    return true()
};

declare function labterm:updateRetiredReason($loincId as xs:string, $note as xs:string) as xs:boolean {
    let $oldConcept := $labterm:labConcepts/lab_concept[concept[@loinc_num=$loincId]]
    let $commentElement := <retired-reason>{$note}</retired-reason>
    let $update := 
        if($oldConcept) then
            if ($oldConcept/retired-reason)
            then 
                if (string-length($note) = 0)
                then update delete $oldConcept/retired-reason
                else update replace $oldConcept/retired-reason with $commentElement
            else
                update insert $commentElement into $oldConcept
        else (
            error(QName('http://art-decor.org/ns/error', 'NoConceptPresent'), 'No concept present for the given language.')
        )       
    return true()
};

declare function labterm:updateRetiredReplacement($loincId as xs:string, $replacements as xs:string) as xs:boolean {
    let $oldConcept := $labterm:labConcepts/lab_concept[concept[@loinc_num=$loincId]]
    let $commentElement := <retired-replacement>{$replacements}</retired-replacement>
    let $check :=
        for $replacement in tokenize($replacements, ',')
        let $concept := $labterm:labConcepts//lab_concept[concept/@loinc_num=normalize-space($replacement)]
        return 
            if (not($concept))
            then error(QName('http://art-decor.org/ns/error', 'ReplacementDoesNotExist'), concat('Replacement ', $replacement, ' is not part of the Labcodeset'))        
            else if (not($concept/@status = 'active'))
            then error(QName('http://art-decor.org/ns/error', 'ReplacementNotActive'), concat('Replacement ', $replacement, ' is not active'))
            else ()
    let $update := 
        if($oldConcept) then
            if ($oldConcept/retired-replacement)
            then 
                if (string-length($replacements) = 0)
                then update delete $oldConcept/retired-replacement
                else update replace $oldConcept/retired-replacement with $commentElement
            else
                update insert $commentElement into $oldConcept
        else (
            error(QName('http://art-decor.org/ns/error', 'NoConceptPresent'), 'No concept present for the given language.')
        )       
    return true()
};

declare function labterm:addLongNameNL($labconcept as element()?) as element()? {
    (:  Not for T-concepts etc, and not if NL longName already present.
        longNAme is added for LOINC concepts which are not in LCS yet.
    :)
    if (not($labconcept/concept/@loinc_num) or ($labconcept/concept/concept[@language='nl-NL']/longName))
    then $labconcept
    else
        element lab_concept {
            $labconcept/@*,
            element concept {
                $labconcept/concept/@*,
                $labconcept/concept/(* except concept),
                element concept {
                    attribute language {'nl-NL'},
                    $labconcept/concept/concept/*,
                    element longName {nlLongName:getLongName($labconcept)}
                    }
                },
                $labconcept/(* except concept)
            }
};

(: Add materials for non-XXX concepts :)
declare function labterm:addMaterialsToConcept($labconcept as element()?) as element()? {
    if ($labconcept/materials/material) then $labconcept else
    element lab_concept {
        $labconcept/@*,
        $labconcept/(* except materials),
        (: Add materials from mapping :)
        <materials>{
            for $material in $labterm:labLoincSnomedMapping//concept[LOINCSystem=$labconcept//concept[@loinc_num]/system]
            return <material ref="{$material/materialId/string()}" status="active"/>
        }</materials>
    }
};

(: Function which returns just LCS concepts, no LOINC concepts :) 
declare function labterm:getOnlyLabConceptById($loincId as xs:string) as element()? {
    labterm:getOnlyLabConceptById($loincId, false())
};

(: 
Function which returns just LCS concepts, no LOINC concepts.
If lastActive, then get the most recent active concept for update, pending, if not lastActive, then the concept as is. 
Wrap, add materials.
:)
declare function labterm:getOnlyLabConceptById($loincId as xs:string, $lastActive as xs:boolean) as element()? {
    let $labConcept := $labterm:labConcepts//concept[@loinc_num=$loincId]/ancestor::lab_concept
    let $labConcept := 
            if (not($lastActive)) then $labConcept
            else if ($labConcept[@status = ('active', 'retired')]) then $labConcept
            else if ($labConcept[@status = ('update', 'pending')]) then labterm:getLastActiveConcept($loincId)
            else ()
    return  <result count="{count($labConcept)}" search="{$loincId}" current-user="{labterm:get-current-user()}">{if ($labConcept) then labterm:addMaterialsToConcept(labterm:addLongNameNL($labConcept)) else ()}</result>
};



declare function labterm:getLabConceptById($loincId as xs:string) as element()? {
    labterm:getLabConceptById($loincId, 'nl-NL', false(), false(), '')
};

(: Function returns LCS concepts when there is one, otherwise LOINC concept :)
declare function labterm:getLabConceptById($loincId as xs:string, $language as xs:string, $searchloinc as xs:boolean, $searchprerelease as xs:boolean, $statusSearch as xs:string) as element()? {
    let $labConcept :=  $labterm:labConcepts//concept[@loinc_num=$loincId]/ancestor::lab_concept
    let $labConcept :=  
        if (string-length($statusSearch) > 1) 
                then $labConcept[@status = tokenize($statusSearch,',')]
                else $labConcept
    let $loincConcept := if (not($searchloinc)) then () else collection($labterm:strLoincData)//concept[@loinc_num=$loincId][1]
    let $loincConcept := if (not($loincConcept)) then() else 
        <lab_concept status="potential">
            <concept>
                {$loincConcept/@*, $loincConcept/(* except concept), $loincConcept/concept[@language=$language]}
            </concept>
        </lab_concept>
    let $preleaseConcept := if (not($searchprerelease)) then () else labterm:getPrereleaseConcept($loincId)        
    let $result := if ($labConcept) then $labConcept else if ($loincConcept) then $loincConcept else if ($preleaseConcept) then $preleaseConcept else ()
    let $result := labterm:addMaterialsToConcept(labterm:addLongNameNL($result))
    return  <result count="{count($result)}" search="{$loincId}" current-user="{labterm:get-current-user()}">{$result}</result>
};

(: Access function for labConcepts, sends query to the appropriate function.
This is the function called by modules and XForms for concept search.
Helpers sometimes call the other :)
declare function labterm:getLabConcepts($search as xs:string, $language as xs:string, $searchloinc as xs:boolean, $searchprerelease as xs:boolean, $statusSearch as xs:string, $user as xs:string, $show as xs:integer?, $onlyErrors as xs:boolean?, $onlyComments as xs:boolean?, $property as xs:string?, $timing as xs:string?, $system as xs:string?, $scale as xs:string?, $class as xs:string?) as element()* {
    if (fn:normalize-space($search)) then
        (: Allow whitespace before and after loinc_num :)
        if (fn:matches($search, '^\s*[0-9]+\-[0-9]\s*$')) then 
            let $result := labterm:getLabConceptById(fn:normalize-space($search), $language, $searchloinc, $searchprerelease, $statusSearch)
            (: If the concept is a panel, always retrieve whole panel except for prereleases which do not support it :)
            return if ($result//concept[1]/elem[@name='PanelType']='Panel' and $searchprerelease = false()) 
                then labpanels:getLabPanelById(fn:normalize-space($search), $language)
                else $result
        else 
            labterm:getLabConceptBySearchTerm($search, $language, $searchloinc, $searchprerelease, $statusSearch, $user, $show, $onlyErrors, $onlyComments, $property, $timing, $system, $scale, $class) 
    else 
        (: If empty search, return nothing :)
        <result count="0" search=""/>
};

declare function labterm:getPrereleaseConcept($id) as element()* {
    let $concept := $labterm:loincPrerelease//concept[LOINC=$id]
    let $labconcept := $labterm:labConcepts//lab_concept[concept/@loinc_num=$id]
    return if ($labconcept or $concept) then 
        <lab_concept status="{if ($labconcept) then $labconcept/@status else 'potential'}">
            <concept loinc_num="{$concept/LOINC}" status="PRERELEASE">
                <elem name="LOINC_NUM">{$concept/LOINC/string()}</elem>
                <component name="COMPONENT">{$concept/component/string()}</component>
                <property name="PROPERTY">{$concept/property/string()}</property>
                <timing name="TIME_ASPCT">{$concept/timing/string()}</timing>
                <system name="SYSTEM">{$concept/system/string()}</system>
                <scale name="SCALE_TYP">{$concept/scale/string()}</scale>
                <method name="METHOD_TYP">{$concept/method/string()}</method>
                <elem name="STATUS">PRERELEASE</elem>
                <shortName name="SHORTNAME">{$concept/shortName/string()}</shortName>
                <longName name="LONG_COMMON_NAME">{$concept/longName/string()}</longName>
            </concept>
        </lab_concept>
    else ()
};

(:
Input:
$search:        Search string
$language:      Will search concepts in this language, plus the English containing concept
$searchloinc:   Will search for concepts in LOINC too. Returns those wrapped in <lab_concept status="potential">.
                No double loinc_num's are returned, i.e. you either get a labcodeset concept or a loinc concept, never both.
$statusSearch:  Comma-separated list of statuses to include (i.e. 'draft, update,')
:)
declare function labterm:getLabConceptBySearchTerm($search as xs:string) as element()* {
    labterm:getLabConceptBySearchTerm($search, 'nl-NL', false(), false(), '', '', (), (), (), (), (), (), (), ())
};

declare function labterm:getLabConceptBySearchTerm($search as xs:string, $language as xs:string, $searchloinc as xs:boolean, $searchprerelease as xs:boolean, $statusSearch as xs:string, $user as xs:string, $show as xs:integer?, $onlyErrors as xs:boolean?, $onlyComments as xs:boolean?, $property as xs:string?, $timing as xs:string?, $system as xs:string?, $scale as xs:string?, $class as xs:string?) as element()* {
    let $labresults :=  
	    if ($searchloinc or $searchprerelease)
	    then labsearch:textOnlyQuery($search)//lab_concept
	    else labsearch:query($search)//lab_concept
    let $labresults := if($onlyErrors) then $labresults[errors/(error | warning)] else $labresults
    let $labresults := if($onlyComments) then $labresults[comment] else $labresults
    let $labresults := if(string-length($user) > 1) then 
        if ($user = '--all--') then $labresults
        else if ($user = '--empty--') then $labresults[@user = '' or not(@user)] 
        else $labresults[@user = $user]
        else $labresults
    let $labresults := 
        if($property) 
        then $labresults[.//property=$property] 
        else $labresults
    let $labresults := 
        if($timing) 
        then $labresults[.//timing=$timing] 
        else $labresults
    let $labresults := 
        if($system) 
        then $labresults[.//system=$system] 
        else $labresults
    let $labresults := 
        if($scale) 
        then $labresults[.//scale=$scale] 
        else $labresults
    let $labresults := 
        if($class) 
        then $labresults[.//class=$class] 
        else $labresults
    let $labcount := count($labresults)
    let $loincresults := 
        if (not($searchloinc)) then () 
        else if ($search = '*') then () else
        for $loincConcept in adloinc:searchConcept($search, 1, (), (), (), $language)/concept
        let $labConcept := $labresults//@loinc_num[.=$loincConcept/@loinc_num]/ancestor::lab_concept
        return 
            if ($labConcept) then $labConcept
            else 
                let $loincConcept :=
                    <lab_concept status="potential">
                        <concept>
                            {$loincConcept/@*, $loincConcept/(* except concept), $loincConcept/concept[@language=$language]}
                        </concept>
                    </lab_concept>
                return $loincConcept
    let $loinccount := count($loincresults)
    let $prereleaseresults := 
        if (not($searchprerelease)) then () 
        else 
        for $preConcept in labsearch:prereleaseQuery($search)
        return labterm:getPrereleaseConcept($preConcept/LOINC/string())
    let $results := 
        if ($searchloinc) then $loincresults 
        else if ($searchprerelease) then $prereleaseresults 
        else $labresults
    let $results := 
    for $result in $results
            order by $result/concept/longName ascending
        return 
            if (labterm:isLabUser()) then $result
            else if ($result[lower-case(@status) = ('active', 'retired')]) then $result
            else if ($result[lower-case(@status) = ('update', 'pending')]) then labterm:getLastActiveConcept($result/concept/@loinc_num)
    else ()
    let $results := if(string-length($statusSearch) > 1) then $results[@status = tokenize($statusSearch,',')] else $results
    let $showResults := if ($show) then $results[position() <= $show] else $results
    let $showResults := for $result in $showResults return labterm:addMaterialsToConcept(labterm:addLongNameNL($result))
    return 
        <result count="{count($showResults)}" total="{if ($searchloinc) then $loinccount else if ($searchprerelease) then count($prereleaseresults) else $labcount}" search="{$search}" current-user="{labterm:get-current-user()}" labuser="{labterm:isLabUser()}">
            {$showResults}
        </result>
};

(: Get the last active version of a concept from the logs. 

Use for 'update' or 'pending' concepts. 
For 'active' or 'retired' concepts, you get the concept itself.
For 'pending' concepts w/o previous active one, empty sequence. 
For other status (i.e. 'draft'), empty sequence. 
:) 
declare function labterm:getLastActiveConcept($loincId as xs:string) as node()? {
    let $labConcept := $labterm:labConcepts/lab_concept[concept/@loinc_num=$loincId]
    return 
        if ($labConcept[@status=('update' , 'pending')])
        then 
            let $logItems := 
                for $logLine in $labterm:LabDataLog//lab_concept[concept/@loinc_num=$loincId][@status='active']/ancestor::statusChange
                order by $logLine/@effectiveTime ascending
                return $logLine
            return ($logItems[last()])/lab_concept
        else if ($labConcept[@status=('active' , 'retired')])
        then $labConcept
        else ()
};